/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <inttypes.h>
#include <stdarg.h>

#define NVARS	64

struct jit_inst {
	enum {
		JIT_BC_LD_IU64_CONST,

		JIT_BC_LD_I8_GLOBAL,
		JIT_BC_LD_U8_GLOBAL,
		JIT_BC_LD_I16_GLOBAL,
		JIT_BC_LD_U16_GLOBAL,
		JIT_BC_LD_I32_GLOBAL,
		JIT_BC_LD_U32_GLOBAL,
		JIT_BC_LD_IU64_GLOBAL,

		JIT_BC_LD_PARAM,

		JIT_BC_LD_I8_LOCAL,
		JIT_BC_LD_U8_LOCAL,
		JIT_BC_LD_I16_LOCAL,
		JIT_BC_LD_U16_LOCAL,
		JIT_BC_LD_I32_LOCAL,
		JIT_BC_LD_U32_LOCAL,
		JIT_BC_LD_IU64_LOCAL,

		JIT_BC_LD_I8_STAR,
		JIT_BC_LD_U8_STAR,
		JIT_BC_LD_I16_STAR,
		JIT_BC_LD_U16_STAR,
		JIT_BC_LD_I32_STAR,
		JIT_BC_LD_U32_STAR,
		JIT_BC_LD_IU64_STAR,

		JIT_BC_ST_IU8_GLOBAL,
		JIT_BC_ST_IU16_GLOBAL,
		JIT_BC_ST_IU32_GLOBAL,
		JIT_BC_ST_IU64_GLOBAL,

		JIT_BC_ST_IU8_LOCAL,
		JIT_BC_ST_IU16_LOCAL,
		JIT_BC_ST_IU32_LOCAL,
		JIT_BC_ST_IU64_LOCAL,

		JIT_BC_ST_IU8_STAR,
		JIT_BC_ST_IU16_STAR,
		JIT_BC_ST_IU32_STAR,
		JIT_BC_ST_IU64_STAR,

		JIT_BC_CONV_S8,
		JIT_BC_CONV_U8,
		JIT_BC_CONV_S16,
		JIT_BC_CONV_U16,
		JIT_BC_CONV_S32,
		JIT_BC_CONV_U32,

		JIT_BC_NOT,
		JIT_BC_INV,
		JIT_BC_NEG,

		JIT_BC_ADD,
		JIT_BC_SUB,
		JIT_BC_MUL,
		JIT_BC_DIV,
		JIT_BC_REM,
		JIT_BC_AND,
		JIT_BC_OR,
		JIT_BC_XOR,
		JIT_BC_LSL,
		JIT_BC_ASR,
		JIT_BC_LSR,

		JIT_BC_EQ,
		JIT_BC_NEQ,
		JIT_BC_B,
		JIT_BC_AE,

		JIT_BC_GOTO,
		JIT_BC_IFGOTO,
		JIT_BC_IFNOTGOTO,

		JIT_BC_CALL,

		JIT_BC_RET,
	} type;
	unsigned int rd;
	unsigned int rs0;
	unsigned int rs1;
	uint64_t val;
	void *addr;
	unsigned long offset;
	struct jit_inst *jmp;
};

static struct jit_inst jit_buf[1024*1024];
static struct jit_inst *jit_last = jit_buf;


static void
jit__patch(void *jmp, void *label)
{
	((struct jit_inst *) jmp)->jmp = (struct jit_inst *) label;
}

static void 
jit__jmp(struct jit_label *label, void *jmp)
{
	int i;

	for (i = 0; ; i++) {
		assert(i < sizeof(label->jmp) / sizeof(label->jmp[0]));
		if (! label->jmp[i]) {
			label->jmp[i] = jmp; 

			if (label->label) {
				jit__patch(label->jmp[i], label->label);
			}
			break;
		}
	}
}

static void
jit__eval(struct jit_func *func, int reg, struct jit_expr *expr)
{
	struct jit_inst *inst;

	assert(reg < 16); /* FIXME */

	switch (expr->type) {
	case JIT_CONST:
		inst = func->tail;
		inst->type = JIT_BC_LD_IU64_CONST;
		inst->rd = reg;
		inst->val = expr->val;
		inst++;
		func->tail = inst;
		break;

	case JIT_GLOBAL:
		inst = func->tail;
		switch (expr->size) {
		case -1:
			inst->type = JIT_BC_LD_I8_GLOBAL;
			break;
		case 1:
			inst->type = JIT_BC_LD_U8_GLOBAL;
			break;
		case -2:
			inst->type = JIT_BC_LD_I16_GLOBAL;
			break;
		case 2:
			inst->type = JIT_BC_LD_U16_GLOBAL;
			break;
		case -4:
			inst->type = JIT_BC_LD_I32_GLOBAL;
			break;
		case 4:
			inst->type = JIT_BC_LD_U32_GLOBAL;
			break;
		case 8:
			inst->type = JIT_BC_LD_IU64_GLOBAL;
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		inst->rd = reg;
		inst->addr = expr->addr;
		inst++;
		func->tail = inst;
		break;

	case JIT_PARAM:
		/* FIXME */
		inst = func->tail;
		inst->type = JIT_BC_LD_PARAM;
		inst->rd = reg;
		inst++;
		func->tail = inst;
		break;

	case JIT_LOCAL:
		inst = func->tail;
		switch (expr->size) {
		case -1:
			inst->type = JIT_BC_LD_I8_LOCAL;
			break;
		case 1:
			inst->type = JIT_BC_LD_U8_LOCAL;
			break;
		case -2:
			inst->type = JIT_BC_LD_I16_LOCAL;
			break;
		case 2:
			inst->type = JIT_BC_LD_U16_LOCAL;
			break;
		case -4:
			inst->type = JIT_BC_LD_I32_LOCAL;
			break;
		case 4:
			inst->type = JIT_BC_LD_U32_LOCAL;
			break;
		case 8:
			inst->type = JIT_BC_LD_IU64_LOCAL;
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		inst->rd = reg;
		inst->offset = expr->offset;
		inst++;
		func->tail = inst;
		break;

	case JIT_STAR:
		jit__eval(func, reg, expr->op0);

		inst = func->tail;
		switch (expr->size) {
		case -1:
			inst->type = JIT_BC_LD_I8_STAR;
			break;
		case 1:
			inst->type = JIT_BC_LD_U8_STAR;
			break;
		case -2:
			inst->type = JIT_BC_LD_I16_STAR;
			break;
		case 2:
			inst->type = JIT_BC_LD_U16_STAR;
			break;
		case -4:
			inst->type = JIT_BC_LD_I32_STAR;
			break;
		case 4:
			inst->type = JIT_BC_LD_U32_STAR;
			break;
		case -8:
		case 8:
			inst->type = JIT_BC_LD_IU64_STAR;
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		inst->rd = reg;
		inst->rs0 = reg;
		inst++;
		func->tail = inst;
		break;

	case JIT_ADDR:
		switch (expr->op0->type) {
		case JIT_GLOBAL:
			assert(0); /* FIXME */
		case JIT_PARAM:
			assert(0); /* FIXME */
		case JIT_LOCAL:
			assert(0); /* FIXME */
		default:
			assert(0); /* Mustn't happen. */
		}
		break;

	case JIT_CONV:
		jit__eval(func, reg, expr->op0);

		inst = func->tail;
		switch (expr->sign * expr->size) {
		case -1:
			inst->type = JIT_BC_CONV_S8;
			break;
		case 1:
			inst->type = JIT_BC_CONV_U8;
			break;
		case -2:
			inst->type = JIT_BC_CONV_S16;
			break;
		case 2:
			inst->type = JIT_BC_CONV_U16;
			break;
		case -4:
			inst->type = JIT_BC_CONV_S32;
			break;
		case 4:
			inst->type = JIT_BC_CONV_U32;
			break;
		case 8:
		default:
			assert(0); /* Mustn't happen. */
		}
		inst->rd = reg;
		inst->rs0 = reg;
		inst++;
		func->tail = inst;
		break;

	case JIT_NOT:
		jit__eval(func, reg + 0, expr->op0);

		inst = func->tail;
		/* not %r0, %r0 */
		inst->type = JIT_BC_NOT;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst++;
		func->tail = inst;
		break;

	case JIT_INV:
		jit__eval(func, reg + 0, expr->op0);

		inst = func->tail;
		/* inv %r0, %r0 */
		inst->type = JIT_BC_INV;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst++;
		func->tail = inst;
		break;

	case JIT_NEG:
		jit__eval(func, reg + 0, expr->op0);

		inst = func->tail;
		/* neg %r0, %r0 */
		inst->type = JIT_BC_NEG;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst++;
		func->tail = inst;
		break;

	case JIT_ADD:
	case JIT_SUB:
	case JIT_MUL:
	case JIT_DIV:
	case JIT_REM:
	case JIT_AND:
	case JIT_OR:
	case JIT_XOR:
	case JIT_LSL:
	case JIT_ASR:
	case JIT_LSR:
		jit__eval(func, reg + 0, expr->op0);
		jit__eval(func, reg + 1, expr->op1);

		inst = func->tail;
		switch (expr->type) {
		case JIT_ADD:
			/* add %r0, %r0, %r1 */
			inst->type = JIT_BC_ADD;
			break;
		case JIT_SUB:
			/* sub %r0, %r0, %r1 */
			inst->type = JIT_BC_SUB;
			break;
		case JIT_MUL:
			/* mul %r0, %r0, %r1 */
			inst->type = JIT_BC_MUL;
			break;
		case JIT_DIV:
			/* div %r0, %r0, %r1 */
			inst->type = JIT_BC_DIV;
			break;
		case JIT_REM:
			/* rem %r0, %r0, %r1 */
			inst->type = JIT_BC_REM;
			break;
		case JIT_AND:
			/* and %r0, %r0, %r1 */
			inst->type = JIT_BC_AND;
			break;
		case JIT_OR:
			/* or %r0, %r0, %r1 */
			inst->type = JIT_BC_OR;
			break;
		case JIT_XOR:
			/* xor %r0, %r0, %r1 */
			inst->type = JIT_BC_XOR;
			break;
		case JIT_LSL:
			/* lsl %r0, %r0, %r1 */
			inst->type = JIT_BC_LSL;
			break;
		case JIT_ASR:
			/* asr %r0, %r0, %r1 */
			inst->type = JIT_BC_ASR;
			break;
		case JIT_LSR:
			/* lsr %r0, %r0, %r1 */
			inst->type = JIT_BC_LSR;
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst->rs1 = reg + 1;
		inst++;
		func->tail = inst;
		break;

	case JIT_EQ:
		jit__eval(func, reg + 0, expr->op0);
		jit__eval(func, reg + 1, expr->op1);

		inst = func->tail;
		/* eq %r0, %r0, %r1 */
		inst->type = JIT_BC_EQ;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst->rs1 = reg + 1;
		inst++;
		func->tail = inst;
		break;

	case JIT_NEQ:
		jit__eval(func, reg + 0, expr->op0);
		jit__eval(func, reg + 1, expr->op1);

		inst = func->tail;
		/* neq %r0, %r0, %r1 */
		inst->type = JIT_BC_NEQ;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst->rs1 = reg + 1;
		inst++;
		func->tail = inst;
		break;

	case JIT_B:
		jit__eval(func, reg + 0, expr->op0);
		jit__eval(func, reg + 1, expr->op1);

		inst = func->tail;
		/* b %r0, %r0, %r1 */
		inst->type = JIT_BC_B;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst->rs1 = reg + 1;
		inst++;
		func->tail = inst;
		break;

	case JIT_AE:
		jit__eval(func, reg + 0, expr->op0);
		jit__eval(func, reg + 1, expr->op1);

		inst = func->tail;
		/* ae %r0, %r0, %r1 */
		inst->type = JIT_BC_AE;
		inst->rd = reg + 0;
		inst->rs0 = reg + 0;
		inst->rs1 = reg + 1;
		inst++;
		func->tail = inst;
		break;
	}
}

/*
 * Store value in %reg to variable.
 */
static void
jit_st(struct jit_func *func, struct jit_expr *dst, unsigned int reg)
{
	struct jit_inst *inst;

	switch (dst->type) {
	case JIT_CONST:
		assert(0); /* Mustn't happen. */

	case JIT_GLOBAL:
		inst = func->tail;
		switch (dst->size) {
		case 1:
			inst->type = JIT_BC_ST_IU8_GLOBAL;
			break;
		case 2:
			inst->type = JIT_BC_ST_IU16_GLOBAL;
			break;
		case 4:
			inst->type = JIT_BC_ST_IU32_GLOBAL;
			break;
		case 8:
			inst->type = JIT_BC_ST_IU64_GLOBAL;
			break;
		}
		inst->rs0 = reg;
		inst->addr = dst->addr;
		inst++;
		func->tail = inst;
		break;

	case JIT_PARAM:
		assert(0); /* Mustn't happen. */

	case JIT_LOCAL:
		inst = func->tail;
		switch (dst->size) {
		case 1:
			inst->type = JIT_BC_ST_IU8_LOCAL;
			break;
		case 2:
			inst->type = JIT_BC_ST_IU16_LOCAL;
			break;
		case 4:
			inst->type = JIT_BC_ST_IU32_LOCAL;
			break;
		case 8:
			inst->type = JIT_BC_ST_IU64_LOCAL;
			break;
		}
		inst->rs0 = reg;
		inst->offset = dst->offset;
		inst++;
		func->tail = inst;
		break;

	case JIT_STAR:
		jit__eval(func, reg + 1, dst->op0);

		inst = func->tail;
		switch (dst->size) {
		case 1:
			inst->type = JIT_BC_ST_IU8_STAR;
			break;
		case 2:
			inst->type = JIT_BC_ST_IU16_STAR;
			break;
		case 4:
			inst->type = JIT_BC_ST_IU32_STAR;
			break;
		case 8:
			inst->type = JIT_BC_ST_IU64_STAR;
			break;
		}
		inst->rs0 = reg + 1;
		inst->rs1 = reg + 0;
		inst++;
		func->tail = inst;
		break;

	case JIT_ADDR:
	case JIT_CONV:
	case JIT_NOT:
	case JIT_INV:
	case JIT_NEG:
	case JIT_ADD:
	case JIT_SUB:
	case JIT_MUL:
	case JIT_DIV:
	case JIT_REM:
	case JIT_AND:
	case JIT_OR:
	case JIT_XOR:
	case JIT_LSL:
	case JIT_ASR:
	case JIT_LSR:
	case JIT_EQ:
	case JIT_NEQ:
	case JIT_B:
	case JIT_AE:
		assert(0); /* Mustn't happen. */
	}
}

static void
jit_bc_assign(struct jit_func *func, struct jit_expr *res, struct jit_expr *op0)
{
	/* Copy value from op0 to %r0. */
	jit__eval(func, 0, op0);

	/* Copy value from %r0 to res */
	jit_st(func, res, 0);
}

static void
jit_bc_goto(struct jit_func *func, struct jit_label *label)
{
	struct jit_inst *inst;

	inst = func->tail;
	/* jmp label */
	inst->type = JIT_BC_GOTO;
	inst->jmp = NULL; /* Will be patched later. */
	jit__jmp(label, inst);
	inst++;
	func->tail = inst;
}

static void
jit_bc_ifgoto(struct jit_func *func, struct jit_expr *cond, struct jit_label *label)
{
	struct jit_inst *inst;

	switch (cond->type) {
	case JIT_NOT:
		/*
		 * if (cond) goto label;
		 * ->
		 * if (! cond->op0) goto label;
		 */
		jit__eval(func, 0, cond->op0);

		inst = func->tail;
		/* jne label */
		inst->type = JIT_BC_IFNOTGOTO;
		inst->rs0 = 0;
		inst->jmp = NULL; /* Will be patched later. */
		jit__jmp(label, inst);
		inst++;
		func->tail = inst;
		break;

	default:
		/* Copy value from cond to %r0. */
		jit__eval(func, 0, cond);

		inst = func->tail;
		/* jne label */
		inst->type = JIT_BC_IFGOTO;
		inst->rs0 = 0;
		inst->jmp = NULL; /* Will be patched later. */
		jit__jmp(label, inst);
		inst++;
		func->tail = inst;
		break;
	}
}

static void
jit_x86_64_switchgoto(
	struct jit_func *func,
	struct jit_expr *expr,
	struct jit_label *def,
	int val[],
	struct jit_label *lab[]
)
{
	struct jit_inst *inst;
	unsigned int n;

	/* Copy value from expr to %r0. */
	jit__eval(func, 0, expr);

	inst = func->tail;
	for (n = 0; lab[n]; n++) {
		/* mov $val[n], %r1 */
		inst->type = JIT_BC_LD_IU64_CONST;
		inst->rd = 1;
		inst->val = val[n];
		inst++;
		/* eq %r0, %r1, %r1 */
		inst->type = JIT_BC_EQ;
		inst->rd = 1;
		inst->rs0 = 0;
		inst->rs1 = 1;
		inst++;
		/* jne %r0, lab[n] */
		inst->type = JIT_BC_IFGOTO;
		inst->rs0 = 1;
		inst->jmp = NULL; /* Will be patched later. */
		jit__jmp(lab[n], inst);
		inst++;
	}
	/* goto def */
	inst->type = JIT_BC_GOTO;
	inst->jmp = NULL; /* Will be patched later. */
	jit__jmp(def, inst);
	inst++;
	func->tail = inst;
}

static void
jit_bc_label(struct jit_func *func, struct jit_label *label)
{
	unsigned int i;

	label->label = func->tail;

	for (i = 0;
	    i < sizeof(label->jmp) / sizeof(label->jmp[0]) && label->jmp[i];
	    i++) {
		jit__patch(label->jmp[i], label->label);
	}
}

static void
jit_bc_call(struct jit_func *func, struct jit_expr *res, void *_ptr, ...)
{
	char *ptr = _ptr;
	va_list arglist;
	struct jit_expr *arg;
	unsigned int n;
	struct jit_inst *inst;

	/*
	 * Copy parameter to registers.
	 */
	va_start(arglist, _ptr);

	for (n = 0; ; n++) {
		arg = va_arg(arglist, struct jit_expr *);
		if (arg == NULL) {
			break;
		}

		assert(n < 6);
		jit__eval(func, n, arg);
	}

	va_end(arglist);

	/*
	 * Call function.
	 */
	inst = func->tail;
	/* call func */
	inst->type = JIT_BC_CALL;
	inst->addr = ptr;
	inst++;
	func->tail = inst;

	/*
	 * Copy result to register.
	 */
	if (res != NULL) {
		jit_st(func, res, 0);
	}
}

void
jit__compile(struct jit_func *func)
{
	int i;
	struct jit_expr *l;
	struct jit_inst *inst;
	struct jit_stmt *stmt;

	i = NVARS; /* See below. */
	for (l = func->local_last; l; l = l->prev) {
		l->offset = --i;
	}

	func->head = jit_last;
	func->tail = jit_last;

	for (stmt = func->stmt_first; stmt; stmt = stmt->next) {
		switch (stmt->type) {
		case JIT_STMT_ASSIGN:
			jit_bc_assign(func,
					stmt->assign.dst, stmt->assign.src);
			break;
		case JIT_STMT_GOTO:
			jit_bc_goto(func,
					stmt->goto_label);
			break;
		case JIT_STMT_IFGOTO:
			jit_bc_ifgoto(func,
					stmt->ifgoto.cond, stmt->ifgoto.label);
			break;
		case JIT_STMT_SWITCHGOTO:
			jit_x86_64_switchgoto(func,
					stmt->switchgoto.expr,
					stmt->switchgoto.def,
					stmt->switchgoto.val,
					stmt->switchgoto.lab);
			break;
		case JIT_STMT_LABEL:
			jit_bc_label(func, stmt->label);
			break;
		case JIT_STMT_CALL:
			jit_bc_call(func, stmt->call.dst, stmt->call.func,
					stmt->call.arg[0], stmt->call.arg[1],
					stmt->call.arg[2], stmt->call.arg[3],
					stmt->call.arg[4], stmt->call.arg[5],
					NULL);
			break;
		}
	}

	inst = func->tail;
	inst->type = JIT_BC_RET;
	inst++;
	func->tail = inst;

	jit_last = func->tail;
}

void
jit_exec(void *_func, void *_cpssp)
{
	struct jit_inst *func = _func;
	char *cpssp = _cpssp;
	uint64_t op[NVARS];
	uint64_t (*f)(uint64_t, uint64_t, uint64_t,
			uint64_t, uint64_t, uint64_t);

	while (func->type != JIT_BC_RET) {
		switch (func->type) {
		case JIT_BC_LD_IU64_CONST:
			op[func->rd] = func->val;
			func++;
			break;

		case JIT_BC_LD_I8_GLOBAL:
			op[func->rd] = (int64_t) *(int8_t *) func->addr;
			func++;
			break;
		case JIT_BC_LD_U8_GLOBAL:
			op[func->rd] = (uint64_t) *(uint8_t *) func->addr;
			func++;
			break;
		case JIT_BC_LD_I16_GLOBAL:
			op[func->rd] = (int64_t) *(int16_t *) func->addr;
			func++;
			break;
		case JIT_BC_LD_U16_GLOBAL:
			op[func->rd] = (uint64_t) *(uint16_t *) func->addr;
			func++;
			break;
		case JIT_BC_LD_I32_GLOBAL:
			op[func->rd] = (int64_t) *(int32_t *) func->addr;
			func++;
			break;
		case JIT_BC_LD_U32_GLOBAL:
			op[func->rd] = (uint64_t) *(uint32_t *) func->addr;
			func++;
			break;
		case JIT_BC_LD_IU64_GLOBAL:
			op[func->rd] = *(uint64_t *) func->addr;
			func++;
			break;

		case JIT_BC_LD_I8_LOCAL:
			op[func->rd] = (int64_t) (int8_t) op[func->offset];
			func++;
			break;
		case JIT_BC_LD_U8_LOCAL:
			op[func->rd] = (uint64_t) (uint8_t) op[func->offset];
			func++;
			break;
		case JIT_BC_LD_I16_LOCAL:
			op[func->rd] = (int64_t) (int16_t) op[func->offset];
			func++;
			break;
		case JIT_BC_LD_U16_LOCAL:
			op[func->rd] = (uint64_t) (uint16_t) op[func->offset];
			func++;
			break;
		case JIT_BC_LD_I32_LOCAL:
			op[func->rd] = (int64_t) (int32_t) op[func->offset];
			func++;
			break;
		case JIT_BC_LD_U32_LOCAL:
			op[func->rd] = (uint64_t) (uint32_t) op[func->offset];
			func++;
			break;
		case JIT_BC_LD_IU64_LOCAL:
			op[func->rd] = op[func->offset];
			func++;
			break;

		case JIT_BC_LD_I8_STAR:
			op[func->rd] = (int64_t) *(int8_t *) op[func->rs0];
			func++;
			break;
		case JIT_BC_LD_U8_STAR:
			op[func->rd] = (uint64_t) *(uint8_t *) op[func->rs0];
			func++;
			break;
		case JIT_BC_LD_I16_STAR:
			op[func->rd] = (int64_t) *(int16_t *) op[func->rs0];
			func++;
			break;
		case JIT_BC_LD_U16_STAR:
			op[func->rd] = (uint64_t) *(uint16_t *) op[func->rs0];
			func++;
			break;
		case JIT_BC_LD_I32_STAR:
			op[func->rd] = (int64_t) *(int32_t *) op[func->rs0];
			func++;
			break;
		case JIT_BC_LD_U32_STAR:
			op[func->rd] = (uint64_t) *(uint32_t *) op[func->rs0];
			func++;
			break;
		case JIT_BC_LD_IU64_STAR:
			op[func->rd] = *(uint64_t *) op[func->rs0];
			func++;
			break;

		case JIT_BC_LD_PARAM:
			/* FIXME */
			op[func->rd] = (uint64_t) cpssp;
			func++;
			break;

		case JIT_BC_ST_IU8_GLOBAL:
			*(uint8_t *) func->addr = (uint8_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_ST_IU16_GLOBAL:
			*(uint16_t *) func->addr = (uint16_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_ST_IU32_GLOBAL:
			*(uint32_t *) func->addr = (uint32_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_ST_IU64_GLOBAL:
			*(uint64_t *) func->addr = op[func->rs0];
			func++;
			break;

		case JIT_BC_ST_IU8_LOCAL:
			op[func->offset] = (uint8_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_ST_IU16_LOCAL:
			op[func->offset] = (uint16_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_ST_IU32_LOCAL:
			op[func->offset] = (uint32_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_ST_IU64_LOCAL:
			op[func->offset] = op[func->rs0];
			func++;
			break;

		case JIT_BC_ST_IU8_STAR:
			*(uint8_t *) op[func->rs0] = (uint8_t) op[func->rs1];
			func++;
			break;
		case JIT_BC_ST_IU16_STAR:
			*(uint16_t *) op[func->rs0] = (uint16_t) op[func->rs1];
			func++;
			break;
		case JIT_BC_ST_IU32_STAR:
			*(uint32_t *) op[func->rs0] = (uint32_t) op[func->rs1];
			func++;
			break;
		case JIT_BC_ST_IU64_STAR:
			*(uint64_t *) op[func->rs0] = op[func->rs1];
			func++;
			break;

		case JIT_BC_CONV_S8:
			op[func->rd] = (int8_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_CONV_U8:
			op[func->rd] = (uint8_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_CONV_S16:
			op[func->rd] = (int16_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_CONV_U16:
			op[func->rd] = (uint16_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_CONV_S32:
			op[func->rd] = (int32_t) op[func->rs0];
			func++;
			break;
		case JIT_BC_CONV_U32:
			op[func->rd] = (uint32_t) op[func->rs0];
			func++;
			break;

		case JIT_BC_NOT:
			op[func->rd] = ! op[func->rs0];
			func++;
			break;
		case JIT_BC_INV:
			op[func->rd] = ~op[func->rs0];
			func++;
			break;
		case JIT_BC_NEG:
			op[func->rd] = -op[func->rs0];
			func++;
			break;

		case JIT_BC_ADD:
			op[func->rd] = op[func->rs0] + op[func->rs1];
			func++;
			break;
		case JIT_BC_SUB:
			op[func->rd] = op[func->rs0] - op[func->rs1];
			func++;
			break;
		case JIT_BC_MUL:
			op[func->rd] = op[func->rs0] * op[func->rs1];
			func++;
			break;
		case JIT_BC_DIV:
			op[func->rd] = op[func->rs0] / op[func->rs1];
			func++;
			break;
		case JIT_BC_REM:
			op[func->rd] = op[func->rs0] % op[func->rs1];
			func++;
			break;
		case JIT_BC_AND:
			op[func->rd] = op[func->rs0] & op[func->rs1];
			func++;
			break;
		case JIT_BC_OR:
			op[func->rd] = op[func->rs0] | op[func->rs1];
			func++;
			break;
		case JIT_BC_XOR:
			op[func->rd] = op[func->rs0] ^ op[func->rs1];
			func++;
			break;
		case JIT_BC_LSL:
			op[func->rd] = op[func->rs0] << op[func->rs1];
			func++;
			break;
		case JIT_BC_ASR:
			op[func->rd] = (int64_t) op[func->rs0] >> op[func->rs1];
			func++;
			break;
		case JIT_BC_LSR:
			op[func->rd] = op[func->rs0] >> op[func->rs1];
			func++;
			break;

		case JIT_BC_EQ:
			op[func->rd] = op[func->rs0] == op[func->rs1];
			func++;
			break;
		case JIT_BC_NEQ:
			op[func->rd] = op[func->rs0] != op[func->rs1];
			func++;
			break;
		case JIT_BC_B:
			op[func->rd] = op[func->rs0] < op[func->rs1];
			func++;
			break;
		case JIT_BC_AE:
			op[func->rd] = op[func->rs0] >= op[func->rs1];
			func++;
			break;

		case JIT_BC_GOTO:
			func = func->jmp;
			break;
		case JIT_BC_IFGOTO:
			if (op[func->rs0]) {
				func = func->jmp;
			} else {
				func++;
			}
			break;
		case JIT_BC_IFNOTGOTO:
			if (! op[func->rs0]) {
				func = func->jmp;
			} else {
				func++;
			}
			break;

		case JIT_BC_CALL:
			f = (uint64_t (*)(uint64_t, uint64_t, uint64_t,
				uint64_t, uint64_t, uint64_t)) func->addr;
			op[0] = (*f)(op[0], op[1], op[2], op[3], op[4], op[5]);
			func++;
			break;

		case JIT_BC_RET:
			assert(0); /* Cannot happen. */
		}
	}
}

void
jit_comp_reset(struct jit_comp *jit)
{
}

void
jit_comp_create(struct jit_comp *jit)
{
}

void
jit_comp_destroy(struct jit_comp *jit)
{
}
