//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.plcgen.conversion.expressions;

import static org.eclipse.escet.cif.common.CifTypeUtils.normalizeType;
import static org.eclipse.escet.cif.common.CifValueUtils.flattenBinExpr;
import static org.eclipse.escet.cif.common.CifValueUtils.getTupleProjIndex;
import static org.eclipse.escet.cif.metamodel.cif.expressions.BinaryOperator.DISJUNCTION;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newIntType;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newRealType;
import static org.eclipse.escet.common.java.Lists.copy;
import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Maps.map;
import static org.eclipse.escet.common.java.Sets.set;

import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.common.RangeCompat;
import org.eclipse.escet.cif.metamodel.cif.expressions.AlgVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryOperator;
import org.eclipse.escet.cif.metamodel.cif.expressions.BoolExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.CastExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ConstantExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ContVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DictExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ElifExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.EnumLiteralExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionCallExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IfExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IntExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ListExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.LocationExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ProjectionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.RealExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ReceivedExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.SetExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.SliceExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.StdLibFunction;
import org.eclipse.escet.cif.metamodel.cif.expressions.StdLibFunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.StringExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TimeExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TupleExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.UnaryExpression;
import org.eclipse.escet.cif.metamodel.cif.types.BoolType;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.metamodel.cif.types.EnumType;
import org.eclipse.escet.cif.metamodel.cif.types.IntType;
import org.eclipse.escet.cif.metamodel.cif.types.ListType;
import org.eclipse.escet.cif.metamodel.cif.types.RealType;
import org.eclipse.escet.cif.metamodel.cif.types.TupleType;
import org.eclipse.escet.cif.plcgen.conversion.PlcFunctionAppls;
import org.eclipse.escet.cif.plcgen.generators.NameGenerator;
import org.eclipse.escet.cif.plcgen.generators.PlcVariablePurpose;
import org.eclipse.escet.cif.plcgen.generators.TypeGenerator;
import org.eclipse.escet.cif.plcgen.generators.names.NameScope;
import org.eclipse.escet.cif.plcgen.model.declarations.PlcBasicVariable;
import org.eclipse.escet.cif.plcgen.model.declarations.PlcDataVariable;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcBoolLiteral;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcExpression;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcIntLiteral;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcRealLiteral;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcVarExpression;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcVarExpression.PlcArrayProjection;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcVarExpression.PlcProjection;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcVarExpression.PlcStructProjection;
import org.eclipse.escet.cif.plcgen.model.functions.PlcFuncOperation;
import org.eclipse.escet.cif.plcgen.model.statements.PlcAssignmentStatement;
import org.eclipse.escet.cif.plcgen.model.statements.PlcCommentBlock;
import org.eclipse.escet.cif.plcgen.model.statements.PlcCommentLine;
import org.eclipse.escet.cif.plcgen.model.statements.PlcSelectionStatement;
import org.eclipse.escet.cif.plcgen.model.statements.PlcSelectionStatement.PlcSelectChoice;
import org.eclipse.escet.cif.plcgen.model.statements.PlcStatement;
import org.eclipse.escet.cif.plcgen.model.types.PlcElementaryType;
import org.eclipse.escet.cif.plcgen.model.types.PlcStructType;
import org.eclipse.escet.cif.plcgen.model.types.PlcType;
import org.eclipse.escet.cif.plcgen.targets.PlcTarget;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.BitSetIterator;

/** Converter of CIF expressions to PLC expressions and statements. */
public class ExprGenerator {
    /** An integer CIF type, used for type conversions. */
    private static final CifType CIF_INT_TYPE = newIntType();

    /** A real CIF type, used for type conversions. */
    private static final CifType CIF_REAL_TYPE = newRealType();

    /** Name scope to create new local variables. */
    private final NameScope localNameScope = new NameScope();

    /** Local and scratch variables of the generator. */
    private final List<PlcDataVariable> variables = list();

    /** Map of variable names to their {@link #variables} index. */
    private final Map<String, Integer> varNameToVarIndex = map();

    /** Indices set of scratch variables in {@link #variables}. */
    private final BitSet variableIsScratch = new BitSet();

    /** Indices set of scratch variables in {@link #variables} that can be handed out. */
    private final BitSet variableIsAvailable = new BitSet();

    /** PLC target to generate code for. */
    final PlcTarget target;

    /** Generator for creating clash-free names in the generated code. */
    private final NameGenerator nameGenerator;

    /** Generator for converting CIF types to PLC types. */
    private final TypeGenerator typeGenerator;

    /** Bottom level access to PLC equivalents of CIF data for the scope of the expression generator. */
    private final CifDataProvider scopeCifDataProvider;

    /**
     * Access to PLC equivalents of CIF data, used in expression conversions. Used when resolving CIF data references in
     * the {@link #convertVariableAddressable}, {@link #convertProjectedAddressable} and {@link #convertValue}
     * functions.
     *
     * <p>
     * May be the same as {@link #scopeCifDataProvider}, or access may have been altered using
     * {@link #getScopeCifDataProvider}, adding one or more {@link CifDataProvider}s on top of it, and setting it using
     * {@link #setCurrentCifDataProvider}.
     * </p>
     */
    private CifDataProvider currentCifDataProvider;

    /** PLC function applications of the target. */
    private final PlcFunctionAppls funcAppls;

    /**
     * PLC variable that contains the value sent over the channel. Should be {@code null} if not available (not
     * communicating over a channel or the channel has a void type).
     */
    private PlcBasicVariable channelValueVariable = null;

    /**
     * Constructor of the {@link ExprGenerator} class.
     *
     * @param target PLC target to generate code for.
     * @param nameGenerator Generator for creating clash-free names in the generated code.
     * @param typeGenerator Generator for converting CIF types to PLC types.
     * @param cifDataProvider Access to PLC equivalents of CIF data.
     */
    public ExprGenerator(PlcTarget target, NameGenerator nameGenerator, TypeGenerator typeGenerator,
            CifDataProvider cifDataProvider)
    {
        this.target = target;
        this.nameGenerator = nameGenerator;
        this.typeGenerator = typeGenerator;
        this.scopeCifDataProvider = cifDataProvider;
        this.currentCifDataProvider = cifDataProvider;
        this.funcAppls = new PlcFunctionAppls(target);
    }

    /**
     * Obtain a (local) scratch variable. Its name starts with the provided prefix, and it will have a PLC type that
     * matches with the provided CIF type.
     *
     * @param prefix Initial part of the name of the variable.
     * @param cifType CIF type to convert to a PLC type.
     * @return The created variable.
     */
    public PlcBasicVariable getScratchVariable(String prefix, CifType cifType) {
        PlcType plcType = typeGenerator.convertType(cifType);
        return getScratchVariable(prefix, plcType);
    }

    /**
     * Obtain a (local) scratch variable. Its name starts with the provided prefix, and it will have the provided type.
     *
     * @param prefix Initial part of the name of the variable.
     * @param plcType Type of the returned variable.
     * @return The created variable.
     */
    public PlcBasicVariable getScratchVariable(String prefix, PlcType plcType) {
        // 1. Attempt to find a scratch variable that can be used.
        for (int idx: new BitSetIterator(variableIsAvailable)) {
            PlcBasicVariable var = variables.get(idx);
            if (plcType.equals(var.type) && var.varName.startsWith(prefix)) {
                variableIsAvailable.clear(idx);
                return var;
            }
        }

        // 2. Make a new variable.
        return createVariable(prefix, plcType, null, null, true);
    }

    /**
     * Construct a local variable to use in the generated code.
     *
     * @param prefix Initial part of the name of the variable.
     * @param plcType Type of the returned variable.
     * @return The created variable.
     */
    public PlcDataVariable makeLocalVariable(String prefix, PlcType plcType) {
        return createVariable(prefix, plcType, null, null, false);
    }

    /**
     * Construct a local variable to use in the generated code.
     *
     * @param prefix Initial part of the name of the variable.
     * @param plcType Type of the returned variable.
     * @param address The address of the variable, or {@code null} if not specified.
     * @param value The initial value of the variable, or {@code null} if not specified.
     * @return The created variable.
     */
    public PlcDataVariable makeLocalVariable(String prefix, PlcType plcType, String address, PlcExpression value) {
        return createVariable(prefix, plcType, address, value, false);
    }

    /**
     * Construct a new local variable and add it to the variable administration.
     *
     * @param prefix Initial part of the name of the variable.
     * @param plcType Type of the returned variable.
     * @param address The address of the variable, or {@code null} if not specified.
     * @param value The initial value of the variable, or {@code null} if not specified.
     * @param isScratchVar Whether the variable is a scratch variable.
     * @return The created variable.
     * @note The new variable is not marked as available.
     */
    private PlcDataVariable createVariable(String prefix, PlcType plcType, String address, PlcExpression value,
            boolean isScratchVar)
    {
        String name = nameGenerator.generateLocalName(prefix, localNameScope);
        String targetText = target.getUsageVariableText(PlcVariablePurpose.LOCAL_VAR, name);
        PlcDataVariable newVar = new PlcDataVariable(targetText, name, plcType, address, value);
        addLocalVariable(newVar, isScratchVar);
        return newVar;
    }

    /**
     * Add a variable to the scope.
     *
     * @param variable Variable to add.
     * @param isScratchVar Whether the variable is a scratch variable.
     */
    public void addLocalVariable(PlcDataVariable variable, boolean isScratchVar) {
        int newVarIndex = variables.size();
        variables.add(variable);
        varNameToVarIndex.put(variable.varName, newVarIndex);
        if (isScratchVar) {
            variableIsScratch.set(newVarIndex);
        }
    }

    /**
     * Give variables back to the generator for future re-use. Returning non-scratch variables is allowed but they are
     * ignored.
     *
     * <p>
     * Intended to be used by {@link ExprValueResult} instances.
     * </p>
     *
     * @param variables Variables being returned.
     */
    public void releaseScratchVariables(Collection<PlcBasicVariable> variables) {
        for (PlcBasicVariable var: variables) {
            releaseScratchVariable(var);
        }
    }

    /**
     * Give a variable back to the generator for future re-use. Returning a non-scratch variable is allowed but it is
     * ignored.
     *
     * <p>
     * Intended to be used by {@link ExprValueResult} instances.
     * </p>
     *
     * @param variable Variable being returned.
     */
    public void releaseScratchVariable(PlcBasicVariable variable) {
        Integer idx = varNameToVarIndex.get(variable.varName);
        if (idx == null || !variableIsScratch.get(idx)) {
            return;
        }
        variableIsAvailable.set(idx);
    }

    /**
     * Obtain the scratch variables created in the expression generator.
     *
     * @return The created scratch variables of the expression generator.
     */
    public List<PlcDataVariable> getCreatedScratchVariables() {
        List<PlcDataVariable> scratchVars = listc(variableIsScratch.cardinality());
        for (int idx: new BitSetIterator(variableIsScratch)) {
            scratchVars.add(variables.get(idx));
        }

        // Sort variables on name.
        Collections.sort(scratchVars, Comparator.comparing(v -> v.varName));
        return scratchVars;
    }

    /**
     * Returns the CIF data provider from the scope in which this expression generator is used.
     *
     * <p>
     * Use this scope CIF data provider only to create new data providers on top of the scope CIF data provider. Such
     * new data providers can be set with {@link #setCurrentCifDataProvider}. To convert values and addressables, use
     * {@link #convertValue}, {@link #convertVariableAddressable} and {@link #convertProjectedAddressable}.
     * </p>
     *
     * @return The CIF data provider from the scope in which this expression generator is used.
     */
    public CifDataProvider getScopeCifDataProvider() {
        return scopeCifDataProvider;
    }

    /**
     * Change the access to variables in this scope.
     *
     * @param newCifDataProvider New CIF data provider to use. If {@code null}, the scope data provider is used instead.
     */
    public void setCurrentCifDataProvider(CifDataProvider newCifDataProvider) {
        currentCifDataProvider = (newCifDataProvider == null) ? scopeCifDataProvider : newCifDataProvider;
    }

    /**
     * Set the PLC variable to use for retrieving the communicated channel value at the receiver. Should be set to
     * {@code null} if no value should be communicated or outside channel value communication context.
     *
     * @param channelValueVariable PLC variable to use for obtaining the communicated channel value. Set to {@code null}
     *     if the variable is not valid (not doing communication or a void channel).
     */
    public void setChannelValueVariable(PlcBasicVariable channelValueVariable) {
        this.channelValueVariable = channelValueVariable;
    }

    /**
     * Convert a CIF variable expression to a combination of a PLC write-only expression, used variables, and
     * statements.
     *
     * @param expr CIF expression to convert. Must be a {@link DiscVariableExpression} or a
     *     {@link ContVariableExpression}.
     * @return The converted expression.
     */
    public ExprAddressableResult convertVariableAddressable(Expression expr) {
        if (expr instanceof DiscVariableExpression de) {
            // TODO This may not work for user-defined internal function parameters and local variables.
            return new ExprAddressableResult(de.getVariable(), this)
                    .setValue(currentCifDataProvider.getAddressableForDiscVar(de.getVariable()));
        } else if (expr instanceof ContVariableExpression ce) {
            // Writable continuous variable is always the value (and never the derivative).
            return new ExprAddressableResult(ce.getVariable(), this)
                    .setValue(currentCifDataProvider.getAddressableForContvar(ce.getVariable()));
        }
        // Intentionally leaving out writing to an input variable, as such expressions should not exist in CIF.
        throw new RuntimeException("Unexpected expr: " + expr);
    }

    /**
     * Convert a CIF expression to a combination of a PLC read-only expression, used variables, and statements.
     *
     * @param expr CIF expression to convert.
     * @return The converted expression.
     */
    public ExprValueResult convertValue(Expression expr) {
        if (expr instanceof BoolExpression be) {
            return new ExprValueResult(this).setValue(new PlcBoolLiteral(be.isValue()));
        } else if (expr instanceof IntExpression ie) {
            return new ExprValueResult(this).setValue(target.makeStdInteger(ie.getValue()));
        } else if (expr instanceof RealExpression re) {
            return new ExprValueResult(this).setValue(target.makeStdReal(re.getValue()));
        } else if (expr instanceof StringExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof TimeExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof CastExpression ce) {
            return convertCastExpr(ce);
        } else if (expr instanceof UnaryExpression ue) {
            return convertUnaryExpr(ue);
        } else if (expr instanceof BinaryExpression be) {
            return convertBinaryExpr(be);
        } else if (expr instanceof IfExpression ife) {
            return convertIfExpr(ife);
        } else if (expr instanceof ProjectionExpression pe) {
            return convertProjectionValue(pe);
        } else if (expr instanceof SliceExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof FunctionCallExpression fce) {
            return convertFuncCallExpr(fce);
        } else if (expr instanceof ListExpression le) {
            return convertArrayExpr(le);
        } else if (expr instanceof SetExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof TupleExpression te) {
            return convertTupleExpr(te);
        } else if (expr instanceof DictExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof ConstantExpression ce) {
            // Pull supported constants from the constants table, and eliminate all other constants.
            if (target.supportsConstant(ce.getConstant())) {
                return new ExprValueResult(this).setValue(currentCifDataProvider.getValueForConstant(ce.getConstant()));
            } else {
                return convertValue(ce.getConstant().getValue());
            }
        } else if (expr instanceof DiscVariableExpression de) {
            // TODO This may not work for user-defined internal function parameters and local variables.
            return new ExprValueResult(this).setValue(currentCifDataProvider.getValueForDiscVar(de.getVariable()));
        } else if (expr instanceof AlgVariableExpression ae) {
            // TODO: Decide how to deal with algebraic variables.
            return convertValue(ae.getVariable().getValue()); // Convert its definition.
        } else if (expr instanceof ContVariableExpression ce) {
            return new ExprValueResult(this)
                    .setValue(currentCifDataProvider.getValueForContvar(ce.getVariable(), ce.isDerivative()));
        } else if (expr instanceof LocationExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof EnumLiteralExpression eLitExpr) {
            return new ExprValueResult(this).setValue(typeGenerator.convertEnumLiteral(eLitExpr.getLiteral()));
        } else if (expr instanceof FunctionExpression) {
            throw new RuntimeException("Precondition violation.");
        } else if (expr instanceof InputVariableExpression ie) {
            return new ExprValueResult(this).setValue(currentCifDataProvider.getValueForInputVar(ie.getVariable()));
        } else if (expr instanceof ReceivedExpression) {
            Assert.notNull(channelValueVariable);
            return new ExprValueResult(this).setValue(new PlcVarExpression(channelValueVariable));
        }
        throw new RuntimeException("Unexpected expr: " + expr);
    }

    /**
     * Convert a cast expression.
     *
     * @param castExpr Expression to convert.
     * @return The generated result.
     */
    private ExprValueResult convertCastExpr(CastExpression castExpr) {
        ExprValueResult result = convertValue(castExpr.getChild());
        CifType ctype = normalizeType(castExpr.getChild().getType());
        CifType rtype = normalizeType(castExpr.getType());
        if (ctype instanceof IntType && rtype instanceof RealType) {
            return result.setValue(funcAppls.castFuncAppl(result.value, target.getStdRealType()));
        }
        if (CifTypeUtils.checkTypeCompat(ctype, rtype, RangeCompat.EQUAL)) {
            // Ignore cast expression.
            return result;
        }

        throw new RuntimeException("Precondition violation.");
    }

    /**
     * Convert a unary operator expression.
     *
     * @param unaryExpr Expression to convert.
     * @return The generated result.
     */
    private ExprValueResult convertUnaryExpr(UnaryExpression unaryExpr) {
        ExprValueResult result = convertValue(unaryExpr.getChild());
        switch (unaryExpr.getOperator()) {
            case INVERSE:
                return result.setValue(funcAppls.complementFuncAppl(result.value));

            case NEGATE:
                return result.setValue(funcAppls.negateFuncAppl(result.value));

            case PLUS:
                return result;

            case SAMPLE:
                throw new RuntimeException("Precondition violation.");

            default:
                throw new RuntimeException("Unknown unop: " + unaryExpr.getOperator());
        }
    }

    /**
     * Convert a binary operator expression.
     *
     * @param binExpr Binary expression to convert.
     * @return The generated result.
     */
    private ExprValueResult convertBinaryExpr(BinaryExpression binExpr) {
        CifType ltype = normalizeType(binExpr.getLeft().getType());
        CifType rtype = normalizeType(binExpr.getRight().getType());

        ExprValueResult leftResult = convertValue(binExpr.getLeft());
        ExprValueResult rightResult = convertValue(binExpr.getRight());
        ExprValueResult result = new ExprValueResult(this, leftResult, rightResult);
        switch (binExpr.getOperator()) {
            case IMPLICATION: {
                ExprValueResult[] exprValueResults = new ExprValueResult[] {leftResult, rightResult};

                // (not left-value) or right-value.
                PlcExpression[] values = new PlcExpression[] {
                        funcAppls.complementFuncAppl(leftResult.value),
                        rightResult.value
                };
                return generateShortCircuitConds(BinaryOperator.DISJUNCTION, exprValueResults, values);
            }
            case BI_CONDITIONAL:
                return result.setValue(funcAppls.equalFuncAppl(leftResult.value, rightResult.value));

            case DISJUNCTION:
            case CONJUNCTION:
                if (ltype instanceof BoolType) {
                    return convertFlattenedExpr(binExpr);
                }
                throw new RuntimeException("Precondition violation.");

            case LESS_THAN: {
                PlcExpression leftSide = unifyTypeOfExpr(leftResult.value, ltype, rtype);
                PlcExpression rightSide = unifyTypeOfExpr(rightResult.value, rtype, ltype);
                return result.setValue(funcAppls.lessThanFuncAppl(leftSide, rightSide));
            }

            case LESS_EQUAL: {
                PlcExpression leftSide = unifyTypeOfExpr(leftResult.value, ltype, rtype);
                PlcExpression rightSide = unifyTypeOfExpr(rightResult.value, rtype, ltype);
                return result.setValue(funcAppls.lessEqualFuncAppl(leftSide, rightSide));
            }

            case GREATER_THAN: {
                PlcExpression leftSide = unifyTypeOfExpr(leftResult.value, ltype, rtype);
                PlcExpression rightSide = unifyTypeOfExpr(rightResult.value, rtype, ltype);
                return result.setValue(funcAppls.greaterThanFuncAppl(leftSide, rightSide));
            }

            case GREATER_EQUAL: {
                PlcExpression leftSide = unifyTypeOfExpr(leftResult.value, ltype, rtype);
                PlcExpression rightSide = unifyTypeOfExpr(rightResult.value, rtype, ltype);
                return result.setValue(funcAppls.greaterEqualFuncAppl(leftSide, rightSide));
            }

            case EQUAL:
                // Comparing structure types is not allowed in IEC 61131-3,
                // and thus equality on tuples can't be supported directly.
                // We could always create code for it though.
                if (ltype instanceof BoolType || ltype instanceof IntType || ltype instanceof RealType
                        || ltype instanceof EnumType)
                {
                    return result.setValue(funcAppls.equalFuncAppl(leftResult.value, rightResult.value));
                }

                throw new RuntimeException("Precondition violation.");

            case UNEQUAL:
                // Comparing structure types is not allowed in IEC 61131-3,
                // and thus equality on tuples can't be supported directly.
                // We could always create code for it though.
                if (ltype instanceof BoolType || ltype instanceof IntType || ltype instanceof RealType
                        || ltype instanceof EnumType)
                {
                    return result.setValue(funcAppls.unEqualFuncAppl(leftResult.value, rightResult.value));
                }

                throw new RuntimeException("Precondition violation.");

            case ADDITION:
                if (ltype instanceof IntType || ltype instanceof RealType) {
                    return convertFlattenedExpr(binExpr);
                }

                throw new RuntimeException("Precondition violation.");

            case SUBTRACTION:
                if (ltype instanceof IntType || ltype instanceof RealType) {
                    PlcExpression leftSide = unifyTypeOfExpr(leftResult.value, ltype, rtype);
                    PlcExpression rightSide = unifyTypeOfExpr(rightResult.value, rtype, ltype);
                    return result.setValue(funcAppls.subtractFuncAppl(leftSide, rightSide));
                }

                throw new RuntimeException("Precondition violation.");

            case MULTIPLICATION:
                return convertFlattenedExpr(binExpr);

            case DIVISION: {
                PlcExpression leftSide = unifyTypeOfExpr(leftResult.value, ltype, CIF_REAL_TYPE);
                PlcExpression rightSide = unifyTypeOfExpr(rightResult.value, rtype, CIF_REAL_TYPE);
                return result.setValue(funcAppls.divideFuncAppl(leftSide, rightSide));
            }

            case INTEGER_DIVISION:
                // Truncated towards zero in both CIF and IEC 61131-3.
                return result.setValue(funcAppls.divideFuncAppl(leftResult.value, rightResult.value));

            case MODULUS:
                // Note that in CIF division by zero is an error, while
                // in IEC 61131-3 it results in zero.
                return result.setValue(funcAppls.moduloFuncAppl(leftResult.value, rightResult.value));

            case ELEMENT_OF:
                throw new RuntimeException("Precondition violation.");

            case SUBSET:
                throw new RuntimeException("Precondition violation.");

            default:
                throw new RuntimeException("Unknown binary expression operator: " + binExpr.getOperator());
        }
    }

    /**
     * Flatten the binary expression on its operator, convert the collection children, and combine the children into an
     * n-ary PLC function.
     *
     * @param binExpr Binary expression to flatten and convert. Must be a disjunction, conjunction, addition, or
     *     multiplication expression.
     * @return The converted expression.
     */
    private ExprValueResult convertFlattenedExpr(BinaryExpression binExpr) {
        // Configure some variables to guide the conversion.
        boolean unifyTypes;
        switch (binExpr.getOperator()) {
            case DISJUNCTION:
                unifyTypes = false;
                break;
            case CONJUNCTION:
                unifyTypes = false;
                break;
            case ADDITION:
                unifyTypes = true;
                break;
            case MULTIPLICATION:
                unifyTypes = true;
                break;
            default:
                throw new RuntimeException("Unexpected flattened binary expression operator: " + binExpr.getOperator());
        }

        // Collect the child expressions and compute a unified type if needed.
        List<Expression> exprs = flattenBinExpr(List.of(binExpr), binExpr.getOperator());
        CifType unifiedType;
        if (unifyTypes) {
            unifiedType = CIF_INT_TYPE;
            for (Expression expr: exprs) {
                if (normalizeType(expr.getType()) instanceof RealType) {
                    unifiedType = CIF_REAL_TYPE;
                    break;
                }
            }
        } else {
            unifiedType = null;
        }

        // Convert each child expression, and collect the child results as preparation to their merge. Also collect the
        // child result expressions separately as they need to be applied to the N-ary function.
        ExprGenResult<?, ?>[] exprValueResults = new ExprGenResult<?, ?>[exprs.size()];
        PlcExpression[] values = new PlcExpression[exprs.size()];
        int i = 0;
        for (Expression expr: exprs) {
            ExprValueResult exprValueResult = convertValue(expr);
            exprValueResults[i] = exprValueResult;
            if (unifyTypes) {
                values[i] = unifyTypeOfExpr(exprValueResult.value, normalizeType(expr.getType()), unifiedType);
            } else {
                values[i] = exprValueResult.value;
            }
            i++;
        }

        // Create the final result and give it to the caller.
        switch (binExpr.getOperator()) {
            case CONJUNCTION:
            case DISJUNCTION: {
                return generateShortCircuitConds(binExpr.getOperator(), exprValueResults, values);
            }
            case ADDITION: {
                ExprValueResult exprValueResult = new ExprValueResult(this, exprValueResults);
                return exprValueResult.setValue(funcAppls.addFuncAppl(values));
            }
            case MULTIPLICATION: {
                ExprValueResult exprValueResult = new ExprValueResult(this, exprValueResults);
                return exprValueResult.setValue(funcAppls.multiplyFuncAppl(values));
            }
            default:
                throw new RuntimeException("Unexpected flattened binary expression operator: " + binExpr.getOperator());
        }
    }

    /**
     * Construct code that computes the value of boolean conditions using short circuit semantics.
     *
     * <p>
     * If available for the target, it generates function applications. Otherwise code is generated that explicitly
     * implements short-circuit evaluation.
     * </p>
     *
     * @param binOp Operation to use to combine the given sub-expressions.
     * @param exprValueResults Expression generator results for all the sub-expressions in the source expression. Note
     *     that the 'value' part in the results should not be used any more, use the 'values' array instead.
     * @param values Type-aligned values that must be combined in the generated code.
     * @return The expression result that computes the condition using short-circuit semantics.
     */
    private ExprValueResult generateShortCircuitConds(BinaryOperator binOp, ExprGenResult<?, ?>[] exprValueResults,
            PlcExpression[] values)
    {
        // If there is just one sub-expression, there is no need to handle short circuit semantics.
        if (exprValueResults.length == 1) {
            return new ExprValueResult(this, exprValueResults[0]).setValue(values[0]);
        }

        // If the target has a function with short circuit semantics, use it.
        switch (binOp) {
            case CONJUNCTION:
                if (target.supportsOperation(PlcFuncOperation.AND_SHORT_CIRCUIT_OP, values.length)) {
                    ExprValueResult exprValueResult = new ExprValueResult(this, exprValueResults);
                    return exprValueResult.setValue(funcAppls.andFuncAppl(true, values));
                }
                break;
            case DISJUNCTION:
                if (target.supportsOperation(PlcFuncOperation.OR_SHORT_CIRCUIT_OP, values.length)) {
                    ExprValueResult exprValueResult = new ExprValueResult(this, exprValueResults);
                    return exprValueResult.setValue(funcAppls.orFuncAppl(true, values));
                }
                break;
            default:
                throw new RuntimeException("Unexpected flattened binary expression operator: " + binOp);
        }

        // Fallback for targets that don't have an operation with short-circuit semantics.

        // Construct an IF ... END_IF statement sequence that checks the sub-expression values, and derives the
        // boolean result value.
        ExprValueResult result = new ExprValueResult(this);
        PlcBasicVariable resultVariable = getScratchVariable("b", PlcElementaryType.BOOL_TYPE);
        for (int i = 0; i < exprValueResults.length; i++) {
            ExprGenResult<?, ?> genResult = exprValueResults[i];

            // Compute the sub-expression and assign it to the result variable.
            List<PlcStatement> thenStats = list();
            thenStats.addAll(genResult.code);
            result.codeVariables.addAll(genResult.codeVariables);
            thenStats.add(new PlcAssignmentStatement(resultVariable, values[i]));
            result.codeVariables.addAll(genResult.valueVariables);

            // Add the constructed assignment to the computation.
            if (i == 0) {
                // The first sub-expression directly initializes the result variable.
                result.code.addAll(thenStats);
            } else {
                // For all other sub-expressions, only evaluate the sub-expression if the result is not decided yet.
                PlcExpression condition = new PlcVarExpression(resultVariable);
                condition = (binOp == DISJUNCTION) ? funcAppls.complementFuncAppl(condition) : condition;
                result.code.add(new PlcSelectionStatement(new PlcSelectChoice(condition, thenStats)));
            }
        }

        // Set the variable as result expression, and return the constructed code.
        result.value = new PlcVarExpression(resultVariable);
        result.valueVariables.add(resultVariable);
        return result;
    }

    /**
     * Convert an 'if' expression to PLC code.
     *
     * @param ifExpr Expression to convert.
     * @return The converted expression.
     */
    private ExprValueResult convertIfExpr(IfExpression ifExpr) {
        ExprValueResult result = new ExprValueResult(this);
        PlcType resultValueType = typeGenerator.convertType(ifExpr.getType());
        PlcBasicVariable resultVar = getScratchVariable("ifResult", resultValueType);
        result.valueVariables.add(resultVar);
        result.setValue(new PlcVarExpression(resultVar));

        PlcSelectionStatement selStat = null;
        selStat = addBranch(ifExpr.getGuards(), generateThenStatement(resultVar, ifExpr.getThen()), selStat,
                result.code);

        for (ElifExpression elif: ifExpr.getElifs()) {
            selStat = addBranch(elif.getGuards(), generateThenStatement(resultVar, elif.getThen()), selStat,
                    result.code);
        }
        addBranch(null, generateThenStatement(resultVar, ifExpr.getElse()), selStat, result.code);
        return result;
    }

    /**
     * Construct a callback function for generating code to assign the result value to the result variable.
     *
     * @param resultVar Result variable to assign.
     * @param resultValue Result value to assign to the variable.
     * @return Callback function that produces the code with the assignment.
     */
    private Supplier<List<PlcStatement>> generateThenStatement(PlcBasicVariable resultVar, Expression resultValue) {
        return () -> {
            List<PlcStatement> statements = list();
            ExprValueResult retValueResult = convertValue(resultValue);
            statements.addAll(retValueResult.code);
            statements.add(new PlcAssignmentStatement(new PlcVarExpression(resultVar), retValueResult.value));
            releaseScratchVariables(retValueResult.codeVariables);
            releaseScratchVariables(retValueResult.valueVariables);
            return statements;
        };
    }

    /**
     * Append an {@code IF} branch to a selection statement in the PLC code.
     *
     * <p>
     * Conceptually this function appends a <pre>ELSE IF guards THEN ....</pre> branch to the selection statement in
     * {@code selStat}. The {@code guards} variable also controls whether there is a condition at all to test and
     * {@code selStat} controls whether the first branch is created.
     * </p>
     * <p>
     * The difficulty here is that the converted {@code guards} may have generated code attached which must be executed
     * before evaluating the guards condition. The PLC {@code IF} statement does not support that.
     * </p>
     * <p>
     * Therefore in such a case the current {@code selStat} cannot be extended with another {@code IF} branch. Instead,
     * the code attached to the converted guards must be put in its {@code ELSE} branch so it can be executed. Below
     * that code, a new selection statement must be started to evaluate the guards and possibly perform the assignment.
     * That is, it generates <pre> ELSE
     *     // Code to perform before evaluating the guards.
     *     IF guard-expr THEN ... // Statements for the new branch.
     *     ... // Possibly more branches will be added.
     *     END_IF
     * END_IF</pre> where the top {@code ELSE} and bottom {@code END_IF} are part of the supplied {@code selStat}.
     * </p>
     * <p>
     * In addition in that case, next branches must now be added to this new selection statement. The returned value
     * thus changes to the new selection statement.
     * </p>
     *
     * @param guards CIF expressions that must hold to select the branch. Is {@code null} for the final 'else' branch.
     * @param genThenStats Code generator for the statements in the added branch.
     * @param selStat Selection statement returned the previous time, or {@code null} if no selection statement has been
     *     created yet.
     * @param rootCode Code block for storing the entire generated PLC {@code IF} statement.
     * @return The last used selection statement after adding the branch.
     */
    public PlcSelectionStatement addBranch(List<Expression> guards, Supplier<List<PlcStatement>> genThenStats,
            PlcSelectionStatement selStat, List<PlcStatement> rootCode)
    {
        // Convert the guard conditions and drop into the add selection statement branch function below.
        List<ExprValueResult> convertedGuards;
        if (guards == null) {
            convertedGuards = null;
        } else if (guards.isEmpty()) {
            convertedGuards = List.of(new ExprValueResult(this).setValue(new PlcBoolLiteral(true)));
        } else {
            convertedGuards = listc(guards.size());
            for (Expression guard: guards) {
                convertedGuards.add(convertValue(guard));
            }
        }
        return addPlcBranch(convertedGuards, genThenStats, selStat, rootCode);
    }

    /**
     * Append an {@code IF} branch to a selection statement in the PLC code.
     *
     * <p>
     * Conceptually this function appends a <pre>ELSE IF guards THEN ....</pre> branch to the selection statement in
     * {@code selStat}. The {@code guards} variable also controls whether there is a condition at all to test and
     * {@code selStat} controls whether the first branch is created.
     * </p>
     * <p>
     * The difficulty here is that the converted {@code guards} may have generated code attached which must be executed
     * before evaluating the guards condition. The PLC {@code IF} statement does not support that.
     * </p>
     * <p>
     * Therefore in such a case the current {@code selStat} cannot be extended with another {@code IF} branch. Instead,
     * the code attached to the converted guards must be put in its {@code ELSE} branch so it can be executed. Below
     * that code, a new selection statement must be started to evaluate the guards and possibly perform the assignment.
     * That is, it generates <pre> ELSE
     *     // Code to perform before evaluating the guards.
     *     IF guard-expr THEN ... // Statements for the new branch.
     *     ... // Possibly more branches will be added.
     *     END_IF
     * END_IF</pre> where the top {@code ELSE} and bottom {@code END_IF} are part of the supplied {@code selStat}.
     * </p>
     * <p>
     * In addition in that case, next branches must now be added to this new selection statement. The returned value
     * thus changes to the new selection statement.
     * </p>
     *
     * @param plcGuards PLC expressions that must hold to select the branch. Is {@code null} for the final 'else'
     *     branch.
     * @param genThenStats Code generator for the statements in the added branch.
     * @param selStat Selection statement returned the previous time, or {@code null} if no selection statement has been
     *     created yet.
     * @param rootCode Code block for storing the entire generated PLC {@code IF} statement.
     * @return The last used selection statement after adding the branch.
     */
    public PlcSelectionStatement addPlcBranch(List<ExprValueResult> plcGuards,
            Supplier<List<PlcStatement>> genThenStats, PlcSelectionStatement selStat, List<PlcStatement> rootCode)
    {
        // Place to store generated guard condition code. If no guards are present (that is, it's the final 'else'
        // branch), the 'then' statements are put in the ELSE branch.
        List<PlcStatement> codeStorage = (selStat != null) ? selStat.elseStats : rootCode;

        if (plcGuards != null) {
            // Convert the guard conditions. Copy any generated code into storage, collect the used variables and the
            // converted expression for the final N-ary AND.
            PlcExpression[] grdValues = new PlcExpression[plcGuards.size()];
            boolean seenGuardCode = false;
            Set<PlcBasicVariable> grdVariables = set();

            // For all guard expressions, convert them and store their output.
            int grdNum = 0;
            for (ExprValueResult grdResult: plcGuards) {
                if (grdResult.hasCode()) {
                    seenGuardCode = true;
                    codeStorage.addAll(grdResult.code);
                    grdVariables.addAll(grdResult.codeVariables);
                }
                grdVariables.addAll(grdResult.valueVariables);
                grdValues[grdNum] = grdResult.value;
                grdNum++;
            }

            // If there is no previous selection statement or we added code to the 'else' branch of it, the previous
            // selection statement cannot be used for this branch. Append a new selection statement to the code block
            // in that case.
            if (selStat == null || seenGuardCode) {
                selStat = new PlcSelectionStatement();
                codeStorage.add(selStat);
            }

            // Add a new branch in the previous selection statement or in the just appended new selection statement.
            PlcSelectChoice choice;
            if (grdNum == 1) {
                choice = new PlcSelectChoice(grdValues[0], list());
            } else {
                choice = new PlcSelectChoice(funcAppls.andFuncAppl(false, grdValues), list());
            }
            selStat.condChoices.add(choice);
            releaseScratchVariables(grdVariables);

            // The 'then' statements of that choice are now the spot to write the 'then' code + value.
            codeStorage = choice.thenStats;
        }
        // else there is no guard and 'codeStorage' already points at the right spot for writing the final 'else' code +
        // value.

        // Add the statements in the branch.
        codeStorage.addAll(genThenStats.get());

        return selStat;
    }

    /**
     * Convert projection expressions to a write-only PLC expression.
     *
     * @param expr Projection expression to convert.
     * @return The converted expression.
     */
    public ExprAddressableResult convertProjectedAddressable(Expression expr) {
        // Unwrap and store the nested projections, last projection at index 0.
        List<ProjectionExpression> projections = list();
        while (expr instanceof ProjectionExpression proj) {
            projections.add(proj);
            expr = proj.getChild();
        }
        Assert.check(!projections.isEmpty());

        // Convert the projection root value and make it usable for the PLC.
        ExprAddressableResult exprResult = convertVariableAddressable(expr);

        // Build new PLC projections expressions with the parent variable and the collected projections.
        PlcVarExpression varExpr = new PlcVarExpression(exprResult.value.variable,
                convertAddProjections(projections, copy(exprResult.value.projections), exprResult));
        exprResult.setValue(varExpr);
        return exprResult;
    }

    /**
     * Convert projection expressions to a read-only PLC expression.
     *
     * @param expr Projection expression to convert.
     * @return The converted expression.
     */
    private ExprValueResult convertProjectionValue(Expression expr) {
        // Unwrap and store the nested projections, last projection at index 0.
        List<ProjectionExpression> projections = list();
        while (expr instanceof ProjectionExpression proj) {
            projections.add(proj);
            expr = proj.getChild();
        }
        Assert.check(!projections.isEmpty());

        // Convert the projection root value and make it usable for the PLC.
        ExprValueResult exprResult = convertValue(expr);

        // Setup the result of this method and prepare it for stacking the above collected projections on top of it.
        if (exprResult.value instanceof PlcVarExpression parentVarExpr) {
            // We received a variable to project at, grab the result to add more projections.

            // Build a new PLC projections expressions with the parent variable and the collected projections.
            PlcVarExpression varExpr = new PlcVarExpression(parentVarExpr.variable,
                    convertAddProjections(projections, copy(parentVarExpr.projections), exprResult));
            exprResult.setValue(varExpr);
            return exprResult;
        } else {
            // We got something different than a single variable. Assume the worst and use a new variable.
            PlcType plcType = typeGenerator.convertType(expr.getType());
            PlcBasicVariable projectVar = getScratchVariable("project", plcType);

            // Construct a new result, add the parent result, and append "projectVar := <root-value expression>;" to the
            // code to get the parent result in the new variable.
            ExprValueResult convertResult = new ExprValueResult(this, exprResult);
            PlcVarExpression varExpr = new PlcVarExpression(projectVar);
            convertResult.code.add(new PlcAssignmentStatement(varExpr, convertResult.value));
            convertResult.codeVariables.addAll(exprResult.valueVariables); // Parent value is now in code.

            // Convert the CIF projections that were on top of the projection root value and apply them to the new
            // variable.
            convertResult.setValue(
                    new PlcVarExpression(projectVar, convertAddProjections(projections, list(), convertResult)));
            convertResult.valueVariables.add(projectVar);
            return convertResult;
        }
    }

    /**
     * Convert CIF projections to PLC projections while reversing order and add them after the supplied PLC projections.
     *
     * @param cifProjections CIF projections to convert, in reverse order. Last projection to apply should be at index
     *     {@code 0}.
     * @param plcProjections Storage of converted CIF projections. Is extended in-place.
     * @param convertResult Storage of expression generator results from CIF array index expressions.
     * @return The updated list PLC projections.
     */
    private List<PlcProjection> convertAddProjections(List<ProjectionExpression> cifProjections,
            List<PlcProjection> plcProjections, ExprGenResult<?, ?> convertResult)
    {
        for (int i = cifProjections.size() - 1; i >= 0; i--) {
            ProjectionExpression cifProjection = cifProjections.get(i);
            CifType unProjectedType = normalizeType(cifProjection.getChild().getType());
            Expression cifIndexExpr = cifProjection.getIndex();

            if (unProjectedType instanceof ListType lt) {
                // Convert the index.
                ExprValueResult indexResult = convertValue(cifIndexExpr);
                convertResult.mergeCodeAndVariables(indexResult);

                PlcExpression normalizedIndex = funcAppls.normalizeArrayIndex(indexResult.value, lt.getUpper());
                plcProjections.add(new PlcArrayProjection(normalizedIndex));
            } else if (unProjectedType instanceof TupleType tt) {
                int fieldIndex = getTupleProjIndex(cifProjection);

                PlcStructType structType = typeGenerator.convertTupleType(tt);
                plcProjections.add(new PlcStructProjection(structType.fields.get(fieldIndex).fieldName));
            } else {
                throw new AssertionError("Unexpected unprojected type \"" + unProjectedType + "\" found.");
            }
        }
        return plcProjections;
    }

    /**
     * Convert a function call.
     *
     * @param funcCallExpr Expression performing the call.
     * @return The converted expression.
     */
    private ExprValueResult convertFuncCallExpr(FunctionCallExpression funcCallExpr) {
        // Convert all arguments of the call.
        List<ExprValueResult> argumentResults = listc(funcCallExpr.getArguments().size());
        for (Expression arg: funcCallExpr.getArguments()) {
            argumentResults.add(convertValue(arg));
        }

        // Dispatch call construction based on the function being called.
        Expression fexpr = funcCallExpr.getFunction();
        if (fexpr instanceof StdLibFunctionExpression) {
            return convertStdlibExpr(funcCallExpr, argumentResults);
        }
        // TODO: Implement function calls to internal user-defined functions.
        throw new RuntimeException("Calls to internal user-defined functions are not implemented yet.");
    }

    /**
     * Convert a call to a standard library function.
     *
     * @param stdlibCallExpr The performed call to convert.
     * @param argumentResults Already converted argument values of the call.
     * @return The converted expression.
     */
    private ExprValueResult convertStdlibExpr(FunctionCallExpression stdlibCallExpr,
            List<ExprValueResult> argumentResults)
    {
        List<Expression> arguments = stdlibCallExpr.getArguments();
        StdLibFunction stdlib = ((StdLibFunctionExpression)stdlibCallExpr.getFunction()).getFunction();
        switch (stdlib) {
            case ABS: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.absFuncAppl(arg1.value));
            }

            case CEIL: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return cifCeil(arg1);
            }

            case DELETE:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");

            case EMPTY:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");

            case EXP: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.expFuncAppl(arg1.value));
            }

            case FLOOR: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return cifFloor(arg1);
            }

            case FORMAT:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");

            case LN: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.lnFuncAppl(arg1.value));
            }

            case LOG: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);

                if (!target.supportsOperation(PlcFuncOperation.STDLIB_LOG, argumentResults.size())) {
                    // Fallback to log10(x) = ln(x) / ln(10).
                    PlcExpression lnX = funcAppls.lnFuncAppl(arg1.value);
                    PlcExpression ln10 = funcAppls.lnFuncAppl(target.makeStdReal("10.0"));
                    return arg1.setValue(funcAppls.divideFuncAppl(lnX, ln10));
                }
                return arg1.setValue(funcAppls.logFuncAppl(arg1.value));
            }

            case MAXIMUM:
            case MINIMUM: {
                // TODO Both MIN and MAX are n-ary functions in the PLC (just like disjunction, add, or conjunction).
                Assert.check(argumentResults.size() == 2);
                CifType ltype = normalizeType(arguments.get(0).getType());
                CifType rtype = normalizeType(arguments.get(1).getType());
                PlcExpression leftSide = unifyTypeOfExpr(argumentResults.get(0).value, ltype, rtype);
                PlcExpression rightSide = unifyTypeOfExpr(argumentResults.get(1).value, rtype, ltype);

                ExprValueResult result = new ExprValueResult(this, argumentResults.get(0), argumentResults.get(1));
                if (stdlib == StdLibFunction.MAXIMUM) {
                    return result.setValue(funcAppls.maxFuncAppl(leftSide, rightSide));
                } else {
                    return result.setValue(funcAppls.minFuncAppl(leftSide, rightSide));
                }
            }

            case POP:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");

            case ROUND: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);

                // arg1 + 0.5
                ExprValueResult exprResult = new ExprValueResult(this, arg1);
                exprResult.code.add(new PlcCommentLine("CIFround(arg) = CIFfloor(arg + 0.5)"));
                PlcExpression aHalf = target.makeRealValue("0.5", arg1.value.type);
                exprResult.value = funcAppls.addFuncAppl(arg1.value, aHalf);

                // cifFloor(arg1 + 0.5);
                return cifFloor(exprResult);
            }

            case SCALE: {
                // Check and get the arguments, construct an expression value result to fill.
                Assert.check(argumentResults.size() == 5);
                ExprValueResult valueResult = argumentResults.get(0);
                ExprValueResult inMinResult = argumentResults.get(1);
                ExprValueResult inMaxResult = argumentResults.get(2);
                ExprValueResult outMinResult = argumentResults.get(3);
                ExprValueResult outMaxResult = argumentResults.get(4);
                ExprValueResult exprResult = new ExprValueResult(this,
                        valueResult, inMinResult, inMaxResult, outMinResult, outMaxResult);

                // Explain what 'scale' is doing.
                exprResult.code.add(new PlcCommentBlock(0, List.of(
                        "CIFscale: Input value 'inMin' is scaled to output value 'outMin'. Input value",
                        "'inMax' is scaled to output value 'outMax'. All other values are scaled by",
                        "interpolating or extrapolating from those value pairs.")));

                // Convert the input values to real-typed variables.
                PlcExpression value, inMin, inMax, outMin, outMax;
                value = new PlcVarExpression(ensureFreshRealVariable("value", valueResult.value, exprResult));
                inMin = new PlcVarExpression(ensureFreshRealVariable("inMin", inMinResult.value, exprResult));
                inMax = new PlcVarExpression(ensureFreshRealVariable("inMax", inMaxResult.value, exprResult));
                outMin = new PlcVarExpression(ensureFreshRealVariable("outMin", outMinResult.value, exprResult));
                outMax = new PlcVarExpression(ensureFreshRealVariable("outMax", outMaxResult.value, exprResult));

                // scale(v, inmin, inmax, outmin, outmax)
                //   = outmin + fraction * (outmax − outmin)
                //     with fraction = (v − inmin)/(inmax − inmin)
                //
                // value - inMin
                PlcExpression valueSubInMin = funcAppls.subtractFuncAppl(value, inMin);
                // inMax - inMin
                PlcExpression inMaxSubInMin = funcAppls.subtractFuncAppl(inMax, inMin);
                // (v − inmin)/(inmax − inmin)
                PlcExpression fraction = funcAppls.divideFuncAppl(valueSubInMin, inMaxSubInMin);
                // outMax - outMin
                PlcExpression outRange = funcAppls.subtractFuncAppl(outMax, outMin);
                // fraction * (outmax − outmin)
                PlcExpression offset = funcAppls.multiplyFuncAppl(fraction, outRange);
                // outmin + fraction * (outmax − outmin)
                exprResult.value = funcAppls.addFuncAppl(outMin, offset);
                return exprResult;
            }

            case SIGN: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);

                PlcExpression argValue = arg1.value;
                // 0 for the input type.
                PlcExpression inputZero = PlcElementaryType.isIntType(argValue.type)
                        ? new PlcIntLiteral(0, argValue.type)
                        : new PlcRealLiteral("0.0", argValue.type);
                // arg1 < 0
                PlcExpression lessZero = funcAppls.lessThanFuncAppl(argValue, inputZero);
                // arg1 > 0
                PlcExpression greaterZero = funcAppls.greaterThanFuncAppl(argValue, inputZero);

                // +1 (int)
                PlcType resultType = target.getStdIntegerType();
                PlcExpression plusOne = new PlcIntLiteral(1, resultType);
                // -1 (int)
                PlcExpression minusOne = funcAppls.negateFuncAppl(new PlcIntLiteral(1, resultType));
                // 0 (int)
                PlcExpression intZero = new PlcIntLiteral(0, resultType);

                // SEL(arg1 < 0, 0, -1)
                PlcExpression selectLess = funcAppls.selFuncAppl(lessZero, intZero, minusOne);
                // SEL(arg1 > 0, SEL(arg1 < 0, 0, -1), 1)
                PlcExpression selectGreater = funcAppls.selFuncAppl(greaterZero, selectLess, plusOne);

                ExprValueResult exprResult = new ExprValueResult(this, arg1);

                exprResult.code.add(new PlcCommentBlock(0, List.of(
                        "CIFsign(x) = -1 if x < 0",
                        "              0 if x = 0",
                        "             +1 if x > 0")));
                return exprResult.setValue(selectGreater);
            }

            case SIZE:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");

            case SQRT: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.sqrtFuncAppl(arg1.value));
            }

            case ACOS: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.acosFuncAppl(arg1.value));
            }

            case ASIN: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.asinFuncAppl(arg1.value));
            }

            case ATAN: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.atanFuncAppl(arg1.value));
            }

            case COS: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.cosFuncAppl(arg1.value));
            }

            case SIN: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.sinFuncAppl(arg1.value));
            }

            case TAN: {
                Assert.check(argumentResults.size() == 1);
                ExprValueResult arg1 = argumentResults.get(0);
                return arg1.setValue(funcAppls.tanFuncAppl(arg1.value));
            }

            case ACOSH:
            case ASINH:
            case ATANH:
            case CBRT:
            case COSH:
            case POWER:
            case SINH:
            case TANH:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");

            case BERNOULLI:
            case BETA:
            case BINOMIAL:
            case CONSTANT:
            case ERLANG:
            case EXPONENTIAL:
            case GAMMA:
            case GEOMETRIC:
            case LOG_NORMAL:
            case NORMAL:
            case POISSON:
            case RANDOM:
            case TRIANGLE:
            case UNIFORM:
            case WEIBULL:
                // Unsupported.
                throw new RuntimeException("Precondition violation.");
        }
        throw new RuntimeException("Unexpected standard library function: " + stdlib);
    }

    /**
     * Ensure that the provided value is or becomes of the standard real type, and is stored in a variable that is used
     * in the result value.
     *
     * @param name Suggested name for the variable.
     * @param value Value to examine and/or convert.
     * @param exprResult Storage of created code and variables.
     * @return The assigned fresh variable.
     */
    private PlcBasicVariable ensureFreshRealVariable(String name, PlcExpression value, ExprValueResult exprResult) {
        PlcElementaryType stdRealType = target.getStdRealType();

        // Create a variable to store the value.
        PlcBasicVariable variable = getScratchVariable(name, stdRealType);
        exprResult.valueVariables.add(variable);

        // Construct an assignment with or without cast.
        if (value.type.equals(stdRealType)) {
            exprResult.code.add(new PlcAssignmentStatement(variable, value));
        } else {
            // The value needs a cast.
            exprResult.code.add(new PlcAssignmentStatement(variable, funcAppls.castFuncAppl(value, stdRealType)));
        }
        return variable;
    }

    /**
     * Generate statements that compute the CIF floor function.
     *
     * @param argument Real valued input value to floor.
     * @return The floor computation.
     */
    private ExprValueResult cifFloor(ExprValueResult argument) {
        PlcElementaryType argumentType = (PlcElementaryType)argument.value.type;
        Assert.check(PlcElementaryType.isRealType(argumentType));
        ExprValueResult exprResult = new ExprValueResult(this, argument);

        // Explain what is being computed.
        exprResult.code.add(new PlcCommentLine("CIFfloor: Round down towards -infinity."));

        // "int" trunced := TRUNC(x);
        PlcBasicVariable trunced = getScratchVariable("trunced", CIF_INT_TYPE);
        PlcElementaryType truncedType = (PlcElementaryType)trunced.type;
        exprResult.valueVariables.add(trunced);
        exprResult.code.add(new PlcAssignmentStatement(trunced, funcAppls.truncFuncAppl(argument.value, truncedType)));

        // SEL("int"_TO_"real"(trunced) <> x AND x < 0 AND x > "min-int", trunced, trunced - 1)
        exprResult.code.add(new PlcCommentBlock(0, List.of(
                "If trunced = inputValue, the inputValue was already floored.",
                "If inputValue >= 0, TRUNC already rounded towards -infinity.",
                "If inputValue <= minimal " + truncedType.name
                        + " value, TRUNC already rounded towards -infinity as far as possible.")));

        PlcExpression truncedAsReal = funcAppls.castFuncAppl(new PlcVarExpression(trunced), argumentType);
        PlcExpression unEqual = funcAppls.unEqualFuncAppl(truncedAsReal, argument.value);
        PlcExpression below0 = funcAppls.lessThanFuncAppl(argument.value, target.makeRealValue("0.0", argumentType));
        String minValueAsReal = String.valueOf(PlcElementaryType.getMinIntValue(truncedType)) + ".0";
        PlcExpression aboveMinInt = funcAppls.greaterThanFuncAppl(argument.value,
                target.makeRealValue(minValueAsReal, argumentType));
        PlcExpression cond = funcAppls.andFuncAppl(false, unEqual, below0, aboveMinInt);
        PlcExpression truncedSub1 = funcAppls.subtractFuncAppl(new PlcVarExpression(trunced),
                new PlcIntLiteral(1, target.getStdIntegerType()));
        exprResult.value = funcAppls.selFuncAppl(cond, new PlcVarExpression(trunced), truncedSub1);
        return exprResult;
    }

    /**
     * Generate statements that compute the CIF ceil function.
     *
     * @param argument Real valued input value to ceil.
     * @return The ceil computation.
     */
    private ExprValueResult cifCeil(ExprValueResult argument) {
        PlcElementaryType argumentType = (PlcElementaryType)argument.value.type;
        Assert.check(PlcElementaryType.isRealType(argumentType));
        ExprValueResult exprResult = new ExprValueResult(this, argument);

        // Explain what is being computed.
        exprResult.code.add(new PlcCommentLine("CIFceil: Round up towards +infinity."));

        // "int" trunced := TRUNC(x);
        PlcBasicVariable trunced = getScratchVariable("trunced", CIF_INT_TYPE);
        PlcElementaryType truncedType = (PlcElementaryType)trunced.type;
        exprResult.valueVariables.add(trunced);
        exprResult.code.add(new PlcAssignmentStatement(trunced, funcAppls.truncFuncAppl(argument.value, truncedType)));

        // SEL("int"_TO_"real"(trunced) <> x AND x > 0 AND x < "max-int", trunced, trunced + 1)
        exprResult.code.add(new PlcCommentBlock(0, List.of(
                "If trunced = inputValue, the inputValue was already ceiled.",
                "If inputValue <= 0, TRUNC already rounded towards +infinity.",
                "If inputValue >= maximal " + truncedType.name
                        + " value, TRUNC already rounded towards +infinity as far as possible.")));

        PlcExpression truncedAsReal = funcAppls.castFuncAppl(new PlcVarExpression(trunced), argumentType);
        PlcExpression unEqual = funcAppls.unEqualFuncAppl(truncedAsReal, argument.value);
        PlcExpression above0 = funcAppls.greaterThanFuncAppl(argument.value, target.makeRealValue("0.0", argumentType));
        String maxValAsReal = String.valueOf(PlcElementaryType.getMaxIntValue(truncedType)) + ".0";
        PlcExpression belowMaxInt = funcAppls.lessThanFuncAppl(argument.value,
                target.makeRealValue(maxValAsReal, argumentType));
        PlcExpression cond = funcAppls.andFuncAppl(false, unEqual, above0, belowMaxInt);
        PlcExpression truncedAdd1 = funcAppls.addFuncAppl(new PlcVarExpression(trunced),
                new PlcIntLiteral(1, target.getStdIntegerType()));
        exprResult.value = funcAppls.selFuncAppl(cond, new PlcVarExpression(trunced), truncedAdd1);
        return exprResult;
    }

    /**
     * Convert an array literal expression to PLC code.
     *
     * @param listExpr Expression to convert.
     * @return The converted expression.
     */
    private ExprValueResult convertArrayExpr(ListExpression listExpr) {
        PlcType listType = typeGenerator.convertType(listExpr.getType());
        PlcBasicVariable arrayVar = getScratchVariable("litArray", listType);

        ExprValueResult result = new ExprValueResult(this);
        int idx = 0;
        for (Expression e: listExpr.getElements()) {
            ExprValueResult childResult = convertValue(e);
            // Add child computation to the result, return the scratch variables of it.
            result.mergeCode(childResult);
            releaseScratchVariables(childResult.codeVariables);

            // Construct assignment.
            PlcArrayProjection arrayProj = new PlcArrayProjection(target.makeStdInteger(idx));
            PlcVarExpression lhs = new PlcVarExpression(arrayVar, List.of(arrayProj));
            PlcAssignmentStatement assignment = new PlcAssignmentStatement(lhs, childResult.value);
            idx++;

            // Add statement to the result.
            result.code.add(assignment);
            releaseScratchVariables(childResult.valueVariables);
        }
        result.valueVariables.add(arrayVar);
        return result.setValue(new PlcVarExpression(arrayVar));
    }

    /**
     * Convert a tuple literal expression to PLC code.
     *
     * @param tupleExpr Expression to convert.
     * @return The converted expression.
     */
    private ExprValueResult convertTupleExpr(TupleExpression tupleExpr) {
        // Construct the destination variable.
        TupleType tupleType = (TupleType)normalizeType(tupleExpr.getType());
        PlcStructType structType = typeGenerator.convertTupleType(tupleType);
        PlcBasicVariable structVar = getScratchVariable("litStruct", structType);

        // Convert the values of the tuple expression and assign them to fields of the destination variable.
        ExprValueResult result = new ExprValueResult(this);
        int idx = 0;
        for (Expression e: tupleExpr.getFields()) {
            ExprValueResult childResult = convertValue(e);
            // Add child computation to the result, release the scratch variables of it.
            result.mergeCode(childResult);
            releaseScratchVariables(childResult.codeVariables);

            // Construct assignment.
            PlcStructProjection structProj = new PlcStructProjection(structType.fields.get(idx).fieldName);
            PlcVarExpression lhs = new PlcVarExpression(structVar, List.of(structProj));
            PlcAssignmentStatement assignment = new PlcAssignmentStatement(lhs, childResult.value);
            idx++;

            // Add statement to the result.
            result.code.add(assignment);
            releaseScratchVariables(childResult.valueVariables);
        }
        result.valueVariables.add(structVar);
        return result.setValue(new PlcVarExpression(structVar));
    }

    /**
     * If necessary, adapt the result value of {@code subExpr} such that it becomes compatible with {@code otherType}.
     *
     * @param expr Expression to make compatible with {@code otherType}.
     * @param myType Type of the expression, must be either {@link IntType} or {@link RealType}.
     * @param otherType Type to unify to, must be either {@link IntType} or {@link RealType}.
     * @return An expression like {@code expr} with compatible type to {@code otherType}.
     */
    private PlcExpression unifyTypeOfExpr(PlcExpression expr, CifType myType, CifType otherType) {
        if (myType instanceof IntType && otherType instanceof RealType) {
            return funcAppls.castFuncAppl(expr, target.getStdRealType());
        }
        return expr;
    }
}
