"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCodeLensModel = exports.CodeLensModel = void 0;
const cancellation_1 = require("../../../../base/common/cancellation");
const errors_1 = require("../../../../base/common/errors");
const lifecycle_1 = require("../../../../base/common/lifecycle");
const types_1 = require("../../../../base/common/types");
const uri_1 = require("../../../../base/common/uri");
const model_1 = require("../../../common/services/model");
const commands_1 = require("../../../../platform/commands/common/commands");
const languageFeatures_1 = require("../../../common/services/languageFeatures");
class CodeLensModel {
    constructor() {
        this.lenses = [];
        this._disposables = new lifecycle_1.DisposableStore();
    }
    dispose() {
        this._disposables.dispose();
    }
    get isDisposed() {
        return this._disposables.isDisposed;
    }
    add(list, provider) {
        this._disposables.add(list);
        for (const symbol of list.lenses) {
            this.lenses.push({ symbol, provider });
        }
    }
}
exports.CodeLensModel = CodeLensModel;
function getCodeLensModel(registry, model, token) {
    return __awaiter(this, void 0, void 0, function* () {
        const provider = registry.ordered(model);
        const providerRanks = new Map();
        const result = new CodeLensModel();
        const promises = provider.map((provider, i) => __awaiter(this, void 0, void 0, function* () {
            providerRanks.set(provider, i);
            try {
                const list = yield Promise.resolve(provider.provideCodeLenses(model, token));
                if (list) {
                    result.add(list, provider);
                }
            }
            catch (err) {
                (0, errors_1.onUnexpectedExternalError)(err);
            }
        }));
        yield Promise.all(promises);
        result.lenses = result.lenses.sort((a, b) => {
            // sort by lineNumber, provider-rank, and column
            if (a.symbol.range.startLineNumber < b.symbol.range.startLineNumber) {
                return -1;
            }
            else if (a.symbol.range.startLineNumber > b.symbol.range.startLineNumber) {
                return 1;
            }
            else if ((providerRanks.get(a.provider)) < (providerRanks.get(b.provider))) {
                return -1;
            }
            else if ((providerRanks.get(a.provider)) > (providerRanks.get(b.provider))) {
                return 1;
            }
            else if (a.symbol.range.startColumn < b.symbol.range.startColumn) {
                return -1;
            }
            else if (a.symbol.range.startColumn > b.symbol.range.startColumn) {
                return 1;
            }
            else {
                return 0;
            }
        });
        return result;
    });
}
exports.getCodeLensModel = getCodeLensModel;
commands_1.CommandsRegistry.registerCommand('_executeCodeLensProvider', function (accessor, ...args) {
    let [uri, itemResolveCount] = args;
    (0, types_1.assertType)(uri_1.URI.isUri(uri));
    (0, types_1.assertType)(typeof itemResolveCount === 'number' || !itemResolveCount);
    const { codeLensProvider } = accessor.get(languageFeatures_1.ILanguageFeaturesService);
    const model = accessor.get(model_1.IModelService).getModel(uri);
    if (!model) {
        throw (0, errors_1.illegalArgument)();
    }
    const result = [];
    const disposables = new lifecycle_1.DisposableStore();
    return getCodeLensModel(codeLensProvider, model, cancellation_1.CancellationToken.None).then(value => {
        disposables.add(value);
        const resolve = [];
        for (const item of value.lenses) {
            if (itemResolveCount === undefined || itemResolveCount === null || Boolean(item.symbol.command)) {
                result.push(item.symbol);
            }
            else if (itemResolveCount-- > 0 && item.provider.resolveCodeLens) {
                resolve.push(Promise.resolve(item.provider.resolveCodeLens(model, item.symbol, cancellation_1.CancellationToken.None)).then(symbol => result.push(symbol || item.symbol)));
            }
        }
        return Promise.all(resolve);
    }).then(() => {
        return result;
    }).finally(() => {
        // make sure to return results, then (on next tick)
        // dispose the results
        setTimeout(() => disposables.dispose(), 100);
    });
});
//# sourceMappingURL=codelens.js.map