/**
 * Copyright (c) 2010-2017, Dénes Harmath, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.testing.core.coverage;

import com.google.common.collect.Iterables;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * The coverage of a PSystem element is measured within a specific scope.
 * This tuple identifies a single coverage measurement.
 */
@Data
@SuppressWarnings("all")
public class CoverageContext<T extends Object> {
  private final T element;

  private final Set<URI> scope;

  public static <T extends Object> CoverageContext<T> create(final T element, final EMFScope scope) {
    Set<URI> _uris = CoverageContext.getUris(scope);
    return new CoverageContext<T>(element, _uris);
  }

  private static Set<URI> getUris(final EMFScope scope) {
    final Function1<Notifier, Collection<URI>> _function = (Notifier it) -> {
      Collection<URI> _switchResult = null;
      boolean _matched = false;
      if (it instanceof Resource) {
        _matched=true;
        URI _uRI = ((Resource)it).getURI();
        _switchResult = Collections.<URI>unmodifiableSet(CollectionLiterals.<URI>newHashSet(_uRI));
      }
      if (!_matched) {
        if (it instanceof ResourceSet) {
          _matched=true;
          final Function1<Resource, URI> _function_1 = (Resource it_1) -> {
            return it_1.getURI();
          };
          _switchResult = ListExtensions.<Resource, URI>map(((ResourceSet)it).getResources(), _function_1);
        }
      }
      if (!_matched) {
        if (it instanceof EObject) {
          _matched=true;
          URI _uRI = EcoreUtil.getURI(((EObject)it));
          _switchResult = Collections.<URI>unmodifiableSet(CollectionLiterals.<URI>newHashSet(_uRI));
        }
      }
      if (!_matched) {
        _switchResult = Collections.<URI>unmodifiableSet(CollectionLiterals.<URI>newHashSet());
      }
      return _switchResult;
    };
    return IterableExtensions.<URI>toSet(Iterables.<URI>concat(IterableExtensions.map(scope.getScopeRoots(), _function)));
  }

  public CoverageContext(final T element, final Set<URI> scope) {
    super();
    this.element = element;
    this.scope = scope;
  }

  @Override
  @Pure
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.element== null) ? 0 : this.element.hashCode());
    return prime * result + ((this.scope== null) ? 0 : this.scope.hashCode());
  }

  @Override
  @Pure
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    CoverageContext<?> other = (CoverageContext<?>) obj;
    if (this.element == null) {
      if (other.element != null)
        return false;
    } else if (!this.element.equals(other.element))
      return false;
    if (this.scope == null) {
      if (other.scope != null)
        return false;
    } else if (!this.scope.equals(other.scope))
      return false;
    return true;
  }

  @Override
  @Pure
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("element", this.element);
    b.add("scope", this.scope);
    return b.toString();
  }

  @Pure
  public T getElement() {
    return this.element;
  }

  @Pure
  public Set<URI> getScope() {
    return this.scope;
  }
}
