"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveConfig = void 0;
const language_core_1 = require("@vue/language-core");
const shared_1 = require("@vue/shared");
const nameCasing_1 = require("./ideFeatures/nameCasing");
const types_1 = require("./types");
// volar services
const CssService = require("volar-service-css");
const EmmetService = require("volar-service-emmet");
const HtmlService = require("volar-service-html");
const JsonService = require("volar-service-json");
const PugService = require("volar-service-pug");
const PugFormatService = require("volar-service-pug-beautify");
const TsService = require("volar-service-typescript");
const TsTqService = require("volar-service-typescript-twoslash-queries");
// our services
const VueService = require("./plugins/vue");
const AutoDotValueService = require("./plugins/vue-autoinsert-dotvalue");
const AutoWrapParenthesesService = require("./plugins/vue-autoinsert-parentheses");
const AutoAddSpaceService = require("./plugins/vue-autoinsert-space");
const ReferencesCodeLensService = require("./plugins/vue-codelens-references");
const DirectiveCommentsService = require("./plugins/vue-directive-comments");
const ExtractComponentService = require("./plugins/vue-extract-file");
const VueTemplateLanguageService = require("./plugins/vue-template");
const ToggleVBindService = require("./plugins/vue-toggle-v-bind-codeaction");
const VueTqService = require("./plugins/vue-twoslash-queries");
const VisualizeHiddenCallbackParamService = require("./plugins/vue-visualize-hidden-callback-param");
function resolveConfig(config, compilerOptions = {}, vueCompilerOptions = {}, ts = require('typescript'), codegenStack = false) {
    const resolvedVueCompilerOptions = (0, language_core_1.resolveVueCompilerOptions)(vueCompilerOptions);
    const vueLanguageModules = (0, language_core_1.createLanguages)(compilerOptions, resolvedVueCompilerOptions, ts, codegenStack);
    config.languages = Object.assign({}, vueLanguageModules, config.languages);
    config.services = resolvePlugins(config.services, resolvedVueCompilerOptions);
    return config;
}
exports.resolveConfig = resolveConfig;
function resolvePlugins(services, vueCompilerOptions) {
    const originalTsPlugin = services?.typescript ?? TsService.create();
    services ??= {};
    services.typescript = (ctx, modules) => {
        const base = typeof originalTsPlugin === 'function' ? originalTsPlugin(ctx, modules) : originalTsPlugin;
        if (!ctx || !modules?.typescript)
            return base;
        const ts = modules.typescript;
        return {
            ...base,
            async provideCompletionItems(document, position, context, item) {
                const result = await base.provideCompletionItems?.(document, position, context, item);
                if (result) {
                    // filter __VLS_
                    result.items = result.items.filter(item => item.label.indexOf('__VLS_') === -1
                        && (!item.labelDetails?.description || item.labelDetails.description.indexOf('__VLS_') === -1));
                    // handle component auto-import patch
                    let casing;
                    for (const [_, map] of ctx.documents.getMapsByVirtualFileUri(document.uri)) {
                        const virtualFile = ctx.documents.getSourceByUri(map.sourceFileDocument.uri)?.root;
                        if (virtualFile instanceof language_core_1.VueFile) {
                            const isAutoImport = !!map.toSourcePosition(position, data => typeof data.completion === 'object' && !!data.completion.autoImportOnly);
                            if (isAutoImport) {
                                const source = ctx.documents.getVirtualFileByUri(document.uri)[1];
                                for (const item of result.items) {
                                    item.data.__isComponentAutoImport = true;
                                }
                                // fix #2458
                                if (source) {
                                    casing ??= await (0, nameCasing_1.getNameCasing)(ts, ctx, ctx.env.fileNameToUri(source.fileName), vueCompilerOptions);
                                    if (casing.tag === types_1.TagNameCasing.Kebab) {
                                        for (const item of result.items) {
                                            item.filterText = (0, language_core_1.hyphenateTag)(item.filterText ?? item.label);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                return result;
            },
            async resolveCompletionItem(item, token) {
                item = await base.resolveCompletionItem?.(item, token) ?? item;
                const itemData = item.data;
                let newName;
                if (itemData?.uri && item.additionalTextEdits) {
                    patchAdditionalTextEdits(itemData.uri, item.additionalTextEdits);
                }
                for (const ext of vueCompilerOptions.extensions) {
                    const suffix = (0, shared_1.capitalize)(ext.substring('.'.length)); // .vue -> Vue
                    if (itemData?.uri
                        && item.textEdit?.newText.endsWith(suffix)
                        && item.additionalTextEdits?.length === 1 && item.additionalTextEdits[0].newText.indexOf('import ' + item.textEdit.newText + ' from ') >= 0
                        && (await ctx.env.getConfiguration?.('vue.complete.normalizeComponentImportName') ?? true)) {
                        newName = item.textEdit.newText.slice(0, -suffix.length);
                        newName = newName[0].toUpperCase() + newName.substring(1);
                        if (newName === 'Index') {
                            const tsItem = item.data.originalItem;
                            if (tsItem.source) {
                                const dirs = tsItem.source.split('/');
                                if (dirs.length >= 3) {
                                    newName = dirs[dirs.length - 2];
                                    newName = newName[0].toUpperCase() + newName.substring(1);
                                }
                            }
                        }
                        item.additionalTextEdits[0].newText = item.additionalTextEdits[0].newText.replace('import ' + item.textEdit.newText + ' from ', 'import ' + newName + ' from ');
                        item.textEdit.newText = newName;
                        const source = ctx.documents.getVirtualFileByUri(itemData.uri)[1];
                        if (source) {
                            const casing = await (0, nameCasing_1.getNameCasing)(ts, ctx, ctx.env.fileNameToUri(source.fileName), vueCompilerOptions);
                            if (casing.tag === types_1.TagNameCasing.Kebab) {
                                item.textEdit.newText = (0, language_core_1.hyphenateTag)(item.textEdit.newText);
                            }
                        }
                    }
                    else if (item.textEdit?.newText && new RegExp(`import \\w*${suffix}\\$1 from [\\S\\s]*`).test(item.textEdit.newText)) {
                        // https://github.com/vuejs/language-tools/issues/2286
                        item.textEdit.newText = item.textEdit.newText.replace(`${suffix}$1`, '$1');
                    }
                }
                const data = item.data;
                if (item.data?.__isComponentAutoImport && data && item.additionalTextEdits?.length && item.textEdit && itemData?.uri) {
                    const fileName = ctx.env.uriToFileName(itemData.uri);
                    const langaugeService = ctx.inject('typescript/languageService');
                    const [virtualFile] = ctx.virtualFiles.getVirtualFile(fileName);
                    const ast = langaugeService.getProgram()?.getSourceFile(fileName);
                    const exportDefault = ast ? language_core_1.scriptRanges.parseScriptRanges(ts, ast, false, true).exportDefault : undefined;
                    if (virtualFile && ast && exportDefault) {
                        const componentName = newName ?? item.textEdit.newText;
                        const optionEdit = ExtractComponentService.createAddComponentToOptionEdit(ts, ast, componentName);
                        if (optionEdit) {
                            const textDoc = ctx.documents.getDocumentByFileName(virtualFile.snapshot, virtualFile.fileName);
                            item.additionalTextEdits.push({
                                range: {
                                    start: textDoc.positionAt(optionEdit.range.start),
                                    end: textDoc.positionAt(optionEdit.range.end),
                                },
                                newText: optionEdit.newText,
                            });
                        }
                    }
                }
                return item;
            },
            async provideCodeActions(document, range, context, token) {
                const result = await base.provideCodeActions?.(document, range, context, token);
                return result?.filter(codeAction => codeAction.title.indexOf('__VLS_') === -1);
            },
            async resolveCodeAction(item, token) {
                const result = await base.resolveCodeAction?.(item, token) ?? item;
                if (result?.edit?.changes) {
                    for (const uri in result.edit.changes) {
                        const edits = result.edit.changes[uri];
                        if (edits) {
                            patchAdditionalTextEdits(uri, edits);
                        }
                    }
                }
                if (result?.edit?.documentChanges) {
                    for (const documentChange of result.edit.documentChanges) {
                        if ('textDocument' in documentChange) {
                            patchAdditionalTextEdits(documentChange.textDocument.uri, documentChange.edits);
                        }
                    }
                }
                return result;
            },
            async provideSemanticDiagnostics(document, token) {
                const result = await base.provideSemanticDiagnostics?.(document, token);
                return result?.map(diagnostic => {
                    if (diagnostic.source === 'ts'
                        && diagnostic.code === 2578 /* Unused '@ts-expect-error' directive. */
                        && document.getText(diagnostic.range) === '// @ts-expect-error __VLS_TS_EXPECT_ERROR') {
                        diagnostic.source = 'vue';
                        diagnostic.code = 'ts-2578';
                        diagnostic.message = diagnostic.message.replace(/@ts-expect-error/g, '@vue-expect-error');
                    }
                    return diagnostic;
                });
            },
        };
    };
    services.html ??= VueTemplateLanguageService.create({
        baseService: HtmlService.create(),
        getScanner: (htmlService, document) => {
            return htmlService.provide['html/languageService']().createScanner(document.getText());
        },
        updateCustomData(htmlService, extraData) {
            htmlService.provide['html/updateCustomData'](extraData);
        },
        isSupportedDocument: (document) => document.languageId === 'html',
        vueCompilerOptions,
    });
    services.pug ??= VueTemplateLanguageService.create({
        baseService: PugService.create(),
        getScanner: (pugService, document) => {
            const pugDocument = pugService.provide['pug/pugDocument'](document);
            if (pugDocument) {
                return pugService.provide['pug/languageService']().createScanner(pugDocument);
            }
        },
        updateCustomData(pugService, extraData) {
            pugService.provide['pug/updateCustomData'](extraData);
        },
        isSupportedDocument: (document) => document.languageId === 'jade',
        vueCompilerOptions,
    });
    services.vue ??= VueService.create();
    services.css ??= CssService.create();
    services['pug-beautify'] ??= PugFormatService.create();
    services.json ??= JsonService.create();
    services['typescript/twoslash-queries'] ??= TsTqService.create();
    services['vue/referencesCodeLens'] ??= ReferencesCodeLensService.create();
    services['vue/autoInsertDotValue'] ??= AutoDotValueService.create();
    services['vue/twoslash-queries'] ??= VueTqService.create();
    services['vue/autoInsertParentheses'] ??= AutoWrapParenthesesService.create();
    services['vue/autoInsertSpaces'] ??= AutoAddSpaceService.create();
    services['vue/visualizeHiddenCallbackParam'] ??= VisualizeHiddenCallbackParamService.create();
    services['vue/directiveComments'] ??= DirectiveCommentsService.create();
    services['vue/extractComponent'] ??= ExtractComponentService.create();
    services['vue/toggleVBind'] ??= ToggleVBindService.create();
    services.emmet ??= EmmetService.create();
    return services;
}
// fix https://github.com/vuejs/language-tools/issues/916
function patchAdditionalTextEdits(uri, edits) {
    if (uri.endsWith('.vue.js')
        || uri.endsWith('.vue.ts')
        || uri.endsWith('.vue.jsx')
        || uri.endsWith('.vue.tsx')) {
        for (const edit of edits) {
            if (edit.range.start.line === 0
                && edit.range.start.character === 0
                && edit.range.end.line === 0
                && edit.range.end.character === 0) {
                edit.newText = '\n' + edit.newText;
            }
        }
    }
}
//# sourceMappingURL=languageService.js.map