//**************************************************************************
//*                     This file is part of the                           *
//*              LAME MP3 encoder output DLL plug-in for                   *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//MP3 encoding with LAME library

#include "mpxplay.h"
#include "newfunc\newfunc.h"
#include "control\control.h"
#include "lib_lame\lame.h"
#include <string.h>
#include <malloc.h>

typedef struct mp3_main_info{
 lame_global_flags *gfp;
 int filehand;

 unsigned int channels;
 long samplesInput;
 long maxBytesOutput;

 float **pcmbuff;
 unsigned char *bitbuff;

 lame_file_callback_t lamecbf;
}mp3_main_info;

static int opt_bitrate,opt_channelmode=-1,opt_quality=-1;
static int opt_id3v1,opt_id3v2;
static vbr_mode opt_vbrmode;

static topt lameenc_opts[] = {
{"lame_b"  ,ARG1|ARG_NUM,      &opt_bitrate      ,0,0},
{"lame_q"  ,ARG1|ARG_NUM,      &opt_quality      ,0,0},
{"lame_v"  ,0,                 &opt_vbrmode      ,vbr_default,0},
{"lame_abr",ARG2|ARG_NUM,      &opt_vbrmode      ,vbr_abr    ,&opt_bitrate},
{"lame_m"  ,ARG1|ARG_NUM,      &opt_channelmode  ,0,0},
{"lame_id3v1",0,               &opt_id3v1        ,1,0},
{"lame_id3v2",0,               &opt_id3v2        ,1,0},
{NULL    ,0,0,0,0}
};

struct mpxplay_resource_s *mrs;

static void lameenc_debugf(const char *format, va_list ap)
{
 //va_list  argptr;
 char sout[500];

 //va_start ( argptr, format );

 (void)vsprintf (sout, format, ap );

 //va_end   ( argptr );

 //sprintf(sout,format,ap);
 mrs->pds_textdisplay_printf(sout);
 //mrs->pds_textdisplay_printf("----------");
 //return;
}

static uint32_t lameenc_fread(void *user_data,void *buffer,uint32_t length)
{
 return mrs->pds_dos_read((int)user_data,buffer,length);

}
static int32_t lameenc_fwrite(void *user_data, void *buffer, uint32_t length)
{
 return mrs->pds_dos_write((int)user_data,buffer,length);
}

static int32_t lameenc_fseek(void *user_data, long position, int whence)
{
 return mrs->pds_lseek((int)user_data,position,whence);
}

static int32_t lameenc_ftell(void *user_data)
{
 return mrs->pds_tell((int)user_data);
}

static uint32_t lameenc_ftruncate(void *user_data)
{
 return mrs->pds_chsize((int)user_data,lameenc_ftell(user_data));
}

static void LAME_config_id3tags(struct mp3_main_info *mp3i,struct audio_info *aui)
{
 struct playlist_entry_info *pei=aui->mvp->pei0;

 id3tag_init(mp3i->gfp);

 if(opt_id3v2){
  if(opt_id3v1)
   id3tag_add_v2(mp3i->gfp);
  else
   id3tag_v2_only(mp3i->gfp);
 }else
  id3tag_v1_only(mp3i->gfp);

 if(pei->id3info[I3I_ARTIST])  id3tag_set_artist(mp3i->gfp, pei->id3info[I3I_ARTIST]);
 if(pei->id3info[I3I_TITLE])   id3tag_set_title(mp3i->gfp,  pei->id3info[I3I_TITLE]);
 if(pei->id3info[I3I_ALBUM])   id3tag_set_album(mp3i->gfp,  pei->id3info[I3I_ALBUM]);
 if(pei->id3info[I3I_YEAR])    id3tag_set_year(mp3i->gfp,   pei->id3info[I3I_YEAR]);
 if(pei->id3info[I3I_GENRE])   id3tag_set_genre(mp3i->gfp,  pei->id3info[I3I_GENRE]);
 if(pei->id3info[I3I_COMMENT]) id3tag_set_comment(mp3i->gfp,pei->id3info[I3I_COMMENT]);
 if(pei->id3info[I3I_TRACKNUM])id3tag_set_track(mp3i->gfp,  pei->id3info[I3I_TRACKNUM]);
}

static int LAME_encoder_open(struct audio_info *aui)
{
 struct mp3_main_info *mp3i;
 unsigned int i;
 char fullname[300];

 aui->card_wave_id=0x0003; // 32-bit float pcm needed
 aui->bits_card=16;        // range: -32768.0 to +32767.0

 if(sizeof(MPXPLAY_PCMOUT_FLOAT_T)!=sizeof(float)){
  mrs->pds_textdisplay_printf("Mpxplay/lame float mismatch!");
  return -1;
 }

 aui->card_private_data=mp3i=calloc(1,sizeof(struct mp3_main_info));
 if(mp3i==NULL)
  return 0;

 mp3i->gfp=lame_init();
 if(!mp3i->gfp){
  mrs->pds_textdisplay_printf("LAMEenc error: cannot initialize encoder!");
  return -1;
 }

 if(aui->card_controlbits&AUINFOS_CARDCNTRLBIT_BITSTREAMOUT){
  funcbit_enable(aui->card_infobits,AUINFOS_CARDINFOBIT_BITSTREAMOUT);
  lame_set_bWriteVbrTag(mp3i->gfp,0);
 }

 lame_set_errorf(mp3i->gfp,&lameenc_debugf);
 lame_set_debugf(mp3i->gfp,&lameenc_debugf);
 lame_set_msgf(mp3i->gfp,&lameenc_debugf);

 LAME_config_id3tags(mp3i,aui);

 if(aui->chan_card>2)
  aui->chan_card=2;
 if(aui->freq_card>48000)
  aui->freq_card=48000;

 lame_set_num_channels(mp3i->gfp,aui->chan_card);
 lame_set_in_samplerate(mp3i->gfp,aui->freq_card);
 if(opt_channelmode>=0)
  if(lame_set_mode(mp3i->gfp,opt_channelmode)<0){
   mrs->pds_textdisplay_printf("LAMEenc error: invalid channelmode! (valid:0-3)");
   return -1;
  }

 if(opt_quality>9)
  opt_quality=9;

 if(opt_vbrmode!=vbr_off){ // VBR,ABR
  if(lame_set_VBR(mp3i->gfp,opt_vbrmode)<0){
   mrs->pds_textdisplay_printf("LAMEenc error: VBR init error!");
   return -1;
  }
  if(opt_bitrate){
   if(lame_set_VBR_mean_bitrate_kbps(mp3i->gfp,opt_bitrate)<0){
    mrs->pds_textdisplay_printf("LAMEenc error: invalid bitrate (valid:8-320)!");
    return -1;
   }
  }else{
   if(opt_quality)
    lame_set_VBR_q(mp3i->gfp,opt_quality);
  }
 }else{ // CBR
  if(opt_bitrate)
   if(lame_set_brate(mp3i->gfp,opt_bitrate)<0){
    mrs->pds_textdisplay_printf("LAMEenc error: invalid bitrate (valid:8-320)!");
    return -1;
   }

  if(opt_quality>=0)
   lame_set_quality(mp3i->gfp,opt_quality);
 }

 if(lame_init_params(mp3i->gfp)<0){
  mrs->pds_textdisplay_printf("LAMEenc error: invalid parameters (bitrate or chanmode)!");
  return -1;
 }

 mp3i->channels=aui->chan_card;

 mp3i->samplesInput=576;
 mp3i->maxBytesOutput=2*mp3i->samplesInput+7200;

 mp3i->pcmbuff=(float **)calloc(mp3i->channels,sizeof(float *));
 if(!mp3i->pcmbuff)
  return -1;
 for(i=0;i<mp3i->channels;i++)
  mp3i->pcmbuff[i]=(float *)malloc(3*mp3i->samplesInput*sizeof(float));

 if(!mp3i->pcmbuff)
  return -1;
 mp3i->bitbuff=(unsigned char*)malloc(mp3i->maxBytesOutput*sizeof(unsigned char));
 if(!mp3i->bitbuff)
  return -1;

 mrs->pds_getfilename_noext_from_fullname(fullname,aui->mvp->pei0->filename);
 mrs->pds_strcat(fullname,".MP3");

 if(mrs->pds_stricmp(fullname,aui->mvp->pei0->filename)!=0) // input and output filename must be different
  mp3i->filehand=mrs->pds_open_create(fullname,O_RDWR|O_BINARY);
 if(!mp3i->filehand){
  mrs->pds_textdisplay_printf("LAMEenc error: cannot open output file!");
  return -1;
 }

 mp3i->lamecbf.read =&lameenc_fread;
 mp3i->lamecbf.write=&lameenc_fwrite;
 mp3i->lamecbf.seek =&lameenc_fseek;
 mp3i->lamecbf.tell =&lameenc_ftell;
 mp3i->lamecbf.truncate=&lameenc_ftruncate;
 mp3i->lamecbf.user_data=(void *)mp3i->filehand;

 if(aui->card_infobits&AUINFOS_CARDINFOBIT_BITSTREAMOUT){
  int bytesWritten=lame_encode_buffer_float(mp3i->gfp,mp3i->pcmbuff[0],mp3i->pcmbuff[1],
                                            0,mp3i->bitbuff,mp3i->maxBytesOutput);

  if(bytesWritten>0)
   mrs->pds_dos_write(mp3i->filehand,mp3i->bitbuff,bytesWritten);
 }

 //lame_print_config(mp3i->gfp);

 return 0;
}

//-------------------------------------------------------------------------

static void LAME_writedata(struct audio_info *aui,char *pcm_sample,unsigned int outbytes)
{
 struct mp3_main_info *mp3i=aui->card_private_data;
 float *pcmf=(float *)pcm_sample;
 unsigned int i,n,samplenum;
 int bytesWritten;

 if(!mp3i)
  return;

 if(aui->card_infobits&AUINFOS_CARDINFOBIT_BITSTREAMOUT){

  mrs->pds_dos_write(mp3i->filehand,pcm_sample,outbytes);

 }else{

  samplenum=outbytes/mp3i->channels/sizeof(float);

  for(i=0;i<mp3i->channels;i++)
   for(n=0;n<samplenum;n++)
    mp3i->pcmbuff[i][n]=pcmf[n*mp3i->channels+i];

  bytesWritten=lame_encode_buffer_float(mp3i->gfp,mp3i->pcmbuff[0],mp3i->pcmbuff[1],
                                       samplenum,mp3i->bitbuff,mp3i->maxBytesOutput);

  if(bytesWritten>0)
   mrs->pds_dos_write(mp3i->filehand,mp3i->bitbuff,bytesWritten);
 }

 return;
}

static unsigned int LAME_bufpos(struct audio_info *aui)
{
 struct mp3_main_info *mp3i=aui->card_private_data;

 if(!mp3i)
  return (2*576*sizeof(float));

 return (mp3i->channels*mp3i->samplesInput*sizeof(float)); // bytenum request
}

static void LAME_close(struct audio_info *aui)
{
 struct mp3_main_info *mp3i=aui->card_private_data;

 if(mp3i){
  if(mp3i->gfp){
   if(mp3i->bitbuff && mp3i->filehand){
    int bytesWritten=lame_encode_flush(mp3i->gfp,mp3i->bitbuff,mp3i->maxBytesOutput);

    if(bytesWritten>0)
     mrs->pds_dos_write(mp3i->filehand,mp3i->bitbuff,bytesWritten);

    lame_mp3_tags_fid_cbf(mp3i->gfp,&mp3i->lamecbf);
     //lame_mp3_tags_fid(mp3i->gfp,NULL);
   }
   lame_close(mp3i->gfp);
  }

  if(mp3i->filehand)
   mrs->pds_close(mp3i->filehand);
  if(mp3i->bitbuff)
   free(mp3i->bitbuff);

  if(mp3i->pcmbuff){
   unsigned int i;
   for(i=0;i<mp3i->channels;i++)
    if(mp3i->pcmbuff[i])
     free(mp3i->pcmbuff[i]);
   free(mp3i->pcmbuff);
  }

  free(aui->card_private_data);
  aui->card_private_data=NULL;
 }
}

//--------------------------------------------------------------------
static int LAME_init(struct audio_info *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 return 1;
}

static int LAME_detect(struct audio_info *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 return 1;
}

static void LAME_card_info(struct audio_info *aui)
{
 mrs->pds_textdisplay_printf("LAME : MP3 audio encoder output plugin (disk writer)(DLL)");
}

static void LAME_setrate(struct audio_info *aui)
{
 LAME_close(aui);
 if(LAME_encoder_open(aui)!=0)
  LAME_close(aui);
}

one_sndcard_info LAME_sndcard_info={
 "MP3",
 SNDCARD_FLAGS_DISKWRITER|SNDCARD_CARDBUF_SPACE,

 &LAME_init,      // card_init
 &LAME_detect,    // card_detect
 &LAME_card_info, // card_info
 NULL,            // card_start
 NULL,            // card_stop
 &LAME_close,     // card_close
 &LAME_setrate,   // card_setrate

 NULL,            // cardbuf_init
 &LAME_writedata, // cardbuf_writedata
 &LAME_bufpos,    // cardbuf_pos
 NULL,            // cardbuf_clear
 NULL,            // cardbuf_int_monitor
 NULL,            // irq_routine

 NULL,            // card_writemixer
 NULL,            // card_readmixer
 NULL             // card_mixerchans
};

static mpxplay_module_entry_s lameenc_output_module_entry={
 MPXPLAY_DLLMODULETYPE_AUCARD,
 0,
 "LAME",
 MPXPLAY_DLLMODULEVER_AUCARD,
 (void *)&LAME_sndcard_info
};

static mpxplay_module_entry_s lameenc_cmdline_module_entry={
 MPXPLAY_DLLMODULETYPE_CONTROL_CMDLINE,
 0,
 NULL,
 MPXPLAY_DLLMODULEVER_CONTROL_CMDLINE,
 (void *)lameenc_opts
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &lameenc_output_module_entry,
  &lameenc_cmdline_module_entry,
  NULL
 }
};

extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin);
 return (long)(&mpxplay_dll_entry_structure);
}
