/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: basic codebook pack/unpack/code/decode operations
 last mod: $Id: codebook.c,v 1.39 2002/08/31 00:00:00 PDSoft Exp $

 ********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ogg.h"
#include "codec.h"
#include "codebook.h"
#include "scales.h"
#include "misc.h"
#include "os.h"

/* packs the given codebook into the bitstream **************************/

int vorbis_staticbook_pack(const static_codebook *c,oggpack_buffer *opb){
  long i,j;
  int ordered=0;

  /* first the basic parameters */
  oggpack_write(opb,0x564342,24);
  oggpack_write(opb,c->dim,16);
  oggpack_write(opb,c->entries,24);

  /* pack the codewords.  There are two packings; length ordered and
     length random.  Decide between the two now. */
  
  for(i=1;i<c->entries;i++)
    if(c->lengthlist[i-1]==0 || c->lengthlist[i]<c->lengthlist[i-1])break;
  if(i==c->entries)ordered=1;
  
  if(ordered){
    /* length ordered.  We only need to say how many codewords of
       each length.  The actual codewords are generated
       deterministically */

    long count=0;
    oggpack_write(opb,1,1);  /* ordered */
    oggpack_write(opb,c->lengthlist[0]-1,5); /* 1 to 32 */

    for(i=1;i<c->entries;i++){
      long this=c->lengthlist[i];
      long last=c->lengthlist[i-1];
      if(this>last){
	for(j=last;j<this;j++){
	  oggpack_write(opb,i-count,_ilog(c->entries-count));
	  count=i;
	}
      }
    }
    oggpack_write(opb,i-count,_ilog(c->entries-count));
    
  }else{
    /* length random.  Again, we don't code the codeword itself, just
       the length.  This time, though, we have to encode each length */
    oggpack_write(opb,0,1);   /* unordered */
    
    /* algortihmic mapping has use for 'unused entries', which we tag
       here.  The algorithmic mapping happens as usual, but the unused
       entry has no codeword. */
    for(i=0;i<c->entries;i++)
      if(c->lengthlist[i]==0)break;

    if(i==c->entries){
      oggpack_write(opb,0,1); /* no unused entries */
      for(i=0;i<c->entries;i++)
	oggpack_write(opb,c->lengthlist[i]-1,5);
    }else{
      oggpack_write(opb,1,1); /* we have unused entries; thus we tag */
      for(i=0;i<c->entries;i++){
	if(c->lengthlist[i]==0){
	  oggpack_write(opb,0,1);
	}else{
	  oggpack_write(opb,1,1);
	  oggpack_write(opb,c->lengthlist[i]-1,5);
	}
      }
    }
  }

  /* is the entry number the desired return value, or do we have a
     mapping? If we have a mapping, what type? */
  oggpack_write(opb,c->maptype,4);
  switch(c->maptype){
  case 0:
    /* no mapping */
    break;
  case 1:case 2:
    /* implicitly populated value mapping */
    /* explicitly populated value mapping */
    
    if(!c->quantlist){
      /* no quantlist?  error */
      return(-1);
    }
    
    /* values that define the dequantization */
    oggpack_write(opb,c->q_min,32);
    oggpack_write(opb,c->q_delta,32);
    oggpack_write(opb,c->q_quant-1,4);
    oggpack_write(opb,c->q_sequencep,1);
    
    {
      int quantvals;
      switch(c->maptype){
      case 1:
	/* a single column of (c->entries/c->dim) quantized values for
	   building a full value list algorithmically (square lattice) */
	quantvals=_book_maptype1_quantvals(c);
	break;
      case 2:
	/* every value (c->entries*c->dim total) specified explicitly */
	quantvals=c->entries*c->dim;
	break;
      default: /* NOT_REACHABLE */
	quantvals=-1;
      }

      /* quantized values */
      for(i=0;i<quantvals;i++)
	oggpack_write(opb,labs(c->quantlist[i]),c->q_quant);

    }
    break;
  default:
    /* error case; we don't have any other map types now */
    return(-1);
  }

  return(0);
}

/* unpacks a codebook from the packet buffer into the codebook struct,
   readies the codebook auxiliary structures for decode *************/
int vorbis_staticbook_unpack(oggpack_buffer *opb,static_codebook *s){
  long i,j;
  memset(s,0,sizeof(*s));
  s->allocedp=1;

  /* make sure alignment is correct */
  if(oggpack_read(opb,24)!=0x564342)goto _eofout;

  /* first the basic parameters */
  s->dim=oggpack_read(opb,16);
  s->entries=oggpack_read(opb,24);
  if(s->entries==-1)goto _eofout;

  /* codeword ordering.... length ordered or unordered? */
  switch((int)oggpack_read(opb,1)){
  case 0:
    /* unordered */
    s->lengthlist=_ogg_malloc(sizeof(*s->lengthlist)*s->entries);

    /* allocated but unused entries? */
    if(oggpack_read(opb,1)){
      /* yes, unused entries */

      for(i=0;i<s->entries;i++){
	if(oggpack_read(opb,1)){
	  long num=oggpack_read(opb,5);
	  if(num==-1)goto _eofout;
	  s->lengthlist[i]=num+1;
	}else
	  s->lengthlist[i]=0;
      }
    }else{
      /* all entries used; no tagging */
      for(i=0;i<s->entries;i++){
	long num=oggpack_read(opb,5);
	if(num==-1)goto _eofout;
	s->lengthlist[i]=num+1;
      }
    }
    
    break;
  case 1:
    /* ordered */
    {
      long length=oggpack_read(opb,5)+1;
      s->lengthlist=_ogg_malloc(sizeof(*s->lengthlist)*s->entries);

      for(i=0;i<s->entries;){
	long num=oggpack_read(opb,_ilog(s->entries-i));
	if(num==-1)goto _eofout;
	for(j=0;j<num && i<s->entries;j++,i++)
	  s->lengthlist[i]=length;
	length++;
      }
    }
    break;
  default:
    /* EOF */
    return(-1);
  }
  
  /* Do we have a mapping to unpack? */
  switch((s->maptype=oggpack_read(opb,4))){
  case 0:
    /* no mapping */
    break;
  case 1: case 2:
    /* implicitly populated value mapping */
    /* explicitly populated value mapping */

    s->q_min=oggpack_read(opb,32);
    s->q_delta=oggpack_read(opb,32);
    s->q_quant=oggpack_read(opb,4)+1;
    s->q_sequencep=oggpack_read(opb,1);

    {
      int quantvals=0;
      switch(s->maptype){
      case 1:
	quantvals=_book_maptype1_quantvals(s);
	break;
      case 2:
	quantvals=s->entries*s->dim;
	break;
      }
      
      /* quantized values */
      s->quantlist=_ogg_malloc(sizeof(*s->quantlist)*quantvals);
      for(i=0;i<quantvals;i++)
	s->quantlist[i]=oggpack_read(opb,s->q_quant);
      
      if(quantvals&&s->quantlist[quantvals-1]==-1)goto _eofout;
    }
    break;
  default:
    goto _errout;
  }

  /* all set */
  return(0);
  
 _errout:
 _eofout:
  vorbis_staticbook_clear(s);
  return(-1); 
}

/* returns the number of bits ************************************************/
int vorbis_book_encode(codebook *book, int a, oggpack_buffer *b){
  oggpack_write(b,book->codelist[a],book->c->lengthlist[a]);
  return(book->c->lengthlist[a]);
}

/* returns the number of bits and *modifies a* to the quantization value *****/
int vorbis_book_encodev(codebook *book,int best,float *a,oggpack_buffer *b){
  int k,dim=book->dim;
  for(k=0;k<dim;k++)
    a[k]=(book->valuelist+best*dim)[k];
  return(vorbis_book_encode(book,best,b));
}

/********************************************************************/

#define CODEBOOK_ASM 1

#if defined(CODEBOOK_ASM) && (defined(__WATCOMC__) || defined(_WIN32))

#ifdef __WATCOMC__

long vorbis_book_inline_decode(const decode_aux *t,oggpack_buffer *b);
//for max. t->tab_maxlen+25 huffbits
#pragma aux vorbis_book_inline_decode=\
 "mov edx,dword ptr 4[esi]"\
 "sub edx,dword ptr [esi]"\
 "cmp edx,dword ptr [edi]"\
 "jle vbd_notab_maxlen"\
  "mov eax,dword ptr [esi]"\
  "mov ecx,eax"\
  "shr eax,3"\
  "add eax,dword ptr 8[esi]"\
  "and ecx,7"\
  "mov eax,dword ptr [eax]"\
  "shr eax,cl"\
  "mov ecx,dword ptr [edi]"\
  "and eax,mask[ecx*4]"\
  "mov ecx,dword ptr 8[edi]"\
  "movzx ebx,byte ptr [ecx+eax]"\
  "mov ecx,dword ptr 4[edi]"\
  "movsx eax,word ptr [ecx+eax*2]"\
  "add dword ptr [esi],ebx"\
  "test eax,eax"\
  "jle vbd_end_neg"\
  "sub edx,ebx"\
  "jmp vbd_flipflop"\
 "vbd_notab_maxlen:"\
  "test edx,edx"\
  "jg vbd_tab"\
  "mov eax,0xffffffff"\
  "jmp vbd_end"\
  "vbd_tab:"\
  "xor eax,eax"\
 "vbd_flipflop:"\
 "mov ebx,dword ptr [esi]"\
 "mov ecx,ebx"\
 "shr ebx,3"\
 "add ebx,dword ptr 8[esi]"\
 "and ecx,7"\
 "mov ebx,dword ptr [ebx]"\
 "shr ebx,cl"\
 "mov ecx,dword ptr 12[edi]"\
 "mov edi,dword ptr 16[edi]"\
 "vbd_flip_back:"\
  "shr ebx,1"\
  "jnc vbd_ptr0"\
   "movsx eax,word ptr [edi+eax*2]"\
  "jmp vbd_ptr_end"\
  "vbd_ptr0:"\
   "movsx eax,word ptr [ecx+eax*2]"\
  "vbd_ptr_end:"\
  "dec edx"\
  "jz vbd_flip_end"\
  "test eax,eax"\
 "jg vbd_flip_back"\
 "vbd_flip_end:"\
 "mov ebx,dword ptr 4[esi]"\
 "sub ebx,edx"\
 "mov dword ptr [esi],ebx"\
 "vbd_end_neg:neg eax"\
 "vbd_end:"\
parm[edi][esi] value[eax] modify[eax ebx ecx edx edi esi];

#else

#ifdef _WIN32

static __inline long vorbis_book_inline_decode(const decode_aux *tree,oggpack_buffer *book)
{
 long entry;
 __asm{
 mov edi,dword ptr tree
 mov esi,dword ptr book
 mov edx,dword ptr 4[esi]
 sub edx,dword ptr [esi]
 cmp edx,dword ptr [edi]
 jle vbd_notab_maxlen
  mov eax,dword ptr [esi]
  mov ecx,eax
  shr eax,3
  add eax,dword ptr 8[esi]
  and ecx,7
  mov eax,dword ptr [eax]
  shr eax,cl
  mov ecx,dword ptr [edi]
  and eax,mask[ecx*4]
  mov ecx,dword ptr 8[edi]
  movzx ebx,byte ptr [ecx+eax]
  mov ecx,dword ptr 4[edi]
  movsx eax,word ptr [ecx+eax*2]
  add dword ptr [esi],ebx
  test eax,eax
  jle vbd_end_neg
  sub edx,ebx
  jmp vbd_flipflop
 vbd_notab_maxlen:
  test edx,edx
  jg vbd_tab
  mov eax,0xffffffff
  jmp vbd_end
  vbd_tab:
  xor eax,eax
 vbd_flipflop:
 mov ebx,dword ptr [esi]
 mov ecx,ebx
 shr ebx,3
 add ebx,dword ptr 8[esi]
 and ecx,7
 mov ebx,dword ptr [ebx]
 shr ebx,cl
 mov ecx,dword ptr 12[edi]
 mov edi,dword ptr 16[edi]
 vbd_flip_back:
  shr ebx,1
  jnc vbd_ptr0
   movsx eax,word ptr [edi+eax*2]
  jmp vbd_ptr_end
  vbd_ptr0:
   movsx eax,word ptr [ecx+eax*2]
  vbd_ptr_end:
  dec edx
  jz vbd_flip_end
  test eax,eax
 jg vbd_flip_back
 vbd_flip_end:
 mov ebx,dword ptr 4[esi]
 sub ebx,edx
 mov dword ptr [esi],ebx
 vbd_end_neg:neg eax
 vbd_end:mov dword ptr entry,eax
 }
 return entry;
}

#endif

#endif

long vorbis_book_decode(codebook *book,oggpack_buffer *b)
{
 return vorbis_book_inline_decode(book->decode_tree,b);
}

#define DECL_BOOKDECTREE const decode_aux *bookdectree=book->decode_tree

#else

long vorbis_book_decode(codebook *book,oggpack_buffer *b)
{
 decode_aux *t=book->decode_tree;
 unsigned long bitstore;
 long ptr,leftbits;
 short *ptr0p,*ptr1p;

 leftbits=oggpack_leftbits(b);
 if(leftbits>t->tab_maxlen){
  unsigned int lok=oggpack_look(b,t->tab_maxlen),adv;
  ptr=t->tab_ptr[lok];
  adv=t->tab_codelen[lok];
  oggpack_adv(b,adv);
  if(ptr<=0)
   return (-ptr);
  leftbits-=adv;
 }else{
  if(leftbits<=0)
   return -1;
  ptr=0;
 }

 // I assume that the huffbits are never more than t->tab_maxlen+32
 bitstore=oggpack_look(b,(leftbits>32)? 32:leftbits);
 ptr0p=t->ptr0;
 ptr1p=t->ptr1;
 do{
  if(bitstore&1)
   ptr=ptr1p[ptr];
  else
   ptr=ptr0p[ptr];
  bitstore>>=1;
  if(!(--leftbits))
   break;
 }while(ptr>0);

 oggpack_setleft(b,leftbits);

 return(-ptr);
}

/*long vorbis_book_decode(codebook *book,oggpack_buffer *b)
{
 decode_aux *t=book->decode_tree;
 long ptr;

 if(oggpack_leftbits(b)>=t->tabn){
  unsigned int lok=oggpack_look(b,t->tabn);
  ptr=t->tab[lok];
  oggpack_adv(b,t->tabl[lok]);
  if(ptr<=0)
   return -ptr;
 }else
  ptr=0;

 if(oggpack_leftbits(b)>=24){
  unsigned long bitstore=oggpack_look(b,32);
  short *ptr0p=t->ptr0,*ptr1p=t->ptr1;
  unsigned int bitused=0;

  do{
   if(bitstore&1)
    ptr=ptr1p[ptr];
   else
    ptr=ptr0p[ptr];
   bitstore>>=1;
   bitused++;        // I assume it's never more than 32
  }while(ptr>0);
  oggpack_adv(b,bitused);
 }else{
  short *ptr0p=t->ptr0,*ptr1p=t->ptr1;
  do{
   switch(oggpack_read1(b)){
    case -1:return(-1);
    case 0:ptr=ptr0p[ptr];break;
    case 1:ptr=ptr1p[ptr];break;
   }
  }while(ptr>0);
 }

 if(ptr>0)
  return -1;

 return(-ptr);
}*/

#define vorbis_book_inline_decode(t,b) vorbis_book_decode(t,b)

#define DECL_BOOKDECTREE codebook *bookdectree=book

#endif

long vorbis_book_decodevs_add(codebook *book,float *a0,oggpack_buffer *b,int n)
{
 const float *bookvallist=book->valuelist;
 //const decode_aux *bookdectree=book->decode_tree;
 DECL_BOOKDECTREE;
 const unsigned int bookdim=book->dim;
 const unsigned int step=n/bookdim;
 unsigned int i,j;

 i=step;
 do{
  float *t;
  float *a;
  long entry=vorbis_book_inline_decode(bookdectree,b);
  if(entry<0)
   return(-1);
  t=(float *)(bookvallist+entry*bookdim);
  a=a0++;
  j=bookdim;
  do{
   a[0] += *t++;
   a+=step;
  }while(--j);
 }while(--i);
 return(0);
}

long vorbis_book_decodev_add(codebook *book,float *a,oggpack_buffer *b,int n)
{
 const float *bookvallist=book->valuelist;
 //const decode_aux *bookdectree=book->decode_tree;
 DECL_BOOKDECTREE;
 const unsigned int bookdim=book->dim;

 if(bookdim>8){
  do{
   float *t;
   long entry=vorbis_book_inline_decode(bookdectree,b),j;
   if(entry<0)
    return(-1);
   t=(float *)(bookvallist+(entry*bookdim));
   j=bookdim;
   n-=j;
   do{
    *a++ += *t++;
   }while(--j);
  }while(n>0);
 }else{
  do{
   float *t;
   long entry=vorbis_book_inline_decode(bookdectree,b);
   if(entry<0)
    return(-1);
   t=(float *)(bookvallist+(entry*bookdim));
   switch(bookdim){
    case 8:a[7] += t[7];
    case 7:a[6] += t[6];
    case 6:a[5] += t[5];
    case 5:a[4] += t[4];
    case 4:a[3] += t[3];
    case 3:a[2] += t[2];
    case 2:a[1] += t[1];
    case 1:a[0] += t[0];
    case 0:break;
   }
   a+=bookdim;
   n-=bookdim;
  }while(n>0);
 }
 return(0);
}

long vorbis_book_decodev_set(codebook *book,float *a,oggpack_buffer *b,int n)
{
 const float *bookvallist=book->valuelist;
 //const decode_aux *bookdectree=book->decode_tree;
 DECL_BOOKDECTREE;
 const unsigned int bookdim=book->dim;

 if(bookdim>8){
  do{
   float *t;
   long entry=vorbis_book_inline_decode(bookdectree,b),j;
   if(entry<0)
    return(-1);
   t=(float *)(bookvallist+(entry*bookdim));
   j=bookdim;
   n-=j;
   do{
    *a++ = *t++;
   }while(--j);
  }while(n>0);
 }else{
  do{
   float *t;
   long entry=vorbis_book_inline_decode(bookdectree,b);
   if(entry<0)
    return(-1);
   t=(float *)(bookvallist+(entry*bookdim));
   switch(bookdim){
    case 8:a[7] = t[7];
    case 7:a[6] = t[6];
    case 6:a[5] = t[5];
    case 5:a[4] = t[4];
    case 4:a[3] = t[3];
    case 3:a[2] = t[2];
    case 2:a[1] = t[1];
    case 1:a[0] = t[0];
    case 0:break;
   }
   a+=bookdim;
   n-=bookdim;
  }while(n>0);
 }
 return(0);
}

long vorbis_book_decodevv_add(codebook *book,float **a,long offset,int ch,
			      oggpack_buffer *b,int n)
{
 const unsigned int bookdim=book->dim;
 const float *bookvallist=book->valuelist;
 //const decode_aux *bookdectree=book->decode_tree;
 DECL_BOOKDECTREE;
 unsigned int i;

 if((ch==2) && (bookdim<=8) && !(bookdim&1)){
  float *al=a[0]+(offset>>1),*ar=a[1]+(offset>>1);
  const unsigned int bd2=bookdim>>1;
  do{
   float *t;
   long entry=vorbis_book_inline_decode(bookdectree,b);
   if(entry<0)
    return(-1);
   t=(float *)(bookvallist+(entry*bookdim));
   switch(bookdim){
    case 8:ar[3] += t[7];
    case 7:al[3] += t[6];
    case 6:ar[2] += t[5];
    case 5:al[2] += t[4];
    case 4:ar[1] += t[3];
    case 3:al[1] += t[2];
    case 2:ar[0] += t[1];
    case 1:al[0] += t[0];
    case 0:break;
   }
   al+=bd2;
   ar+=bd2;
   n-=bookdim;
  }while(n>0);
 }else{
  unsigned int chptr=0;
  for(i=offset/ch;i<(offset+n)/ch;){
   long entry = vorbis_book_inline_decode(bookdectree,b);
   if(entry<0)
    return(-1);
   {
    const float *t = bookvallist+entry*bookdim;
    unsigned int j=bookdim;
    do{
     a[chptr++][i] += *t++;
     if(chptr==ch){
      chptr=0;
      i++;
     }
    }while(--j);
   }
  }
 }
 return(0);
}
