//**************************************************************************
//*                     This file is part of the                           *
//*             AudioCV - a general audio converter program                *
//*                  The source code of AudioCV is                         *
//*          (c) copyright 2001-2004 by PDSoft (Attila Padar)              *
//*                    http://mpxplay.cjb.net                              *
//* email: mpxplay@freemail.hu (please write AudioCV in the subject field) *
//**************************************************************************
//OGG file handling

#include "audiocv.h"
#include "libs\vorbis\os_types.h"
#include "libs\vorbis\codec.h"
#include "libs\vorbis\vorbisen.h"
#include "utf8.h"

typedef float OGG_DOUBLE_T;
typedef float OGG_FLOAT_T;
#define FLOAT_BITS (sizeof(OGG_DOUBLE_T)*8)
#define OGG_SYNC_BUFFER_SIZE 4096

typedef struct ogg_main_info {
 ogg_sync_state   oy;
 ogg_stream_state os;
 ogg_page         og;
 ogg_packet       op;

 vorbis_info      vi;
 vorbis_comment   vc;
 vorbis_dsp_state vd;
 vorbis_block     vb;
 int current_decoder_part;
} ogg_main_info;

extern long acv_filelen(FILE *);

extern int dispquiet;
extern int opt_tagenc_utf8;
extern int opt_channel_coupling,opt_const_bitrate;
extern float opt_lowpass_khz,opt_vbr_quality;

//-------------------------------------------------------------------------
       int  ogg_init_encoder(acv_fileinfo *);
static void ogg_comment_set(acv_fileinfo *,vorbis_comment *);

static int  ogg_write_header(acv_fileinfo *);
       int  ogg_encode(acv_fileinfo *);
       int  ogg_update_header(acv_fileinfo *);
       void ogg_close_encoder(acv_fileinfo *);
//-------------------------------------------------------------------------
       int  ogg_init_decoder(acv_fileinfo *);
       void ogg_decode(acv_fileinfo *,int);
       void ogg_close_decoder(acv_fileinfo *);

static int refill_sync_buff(acv_fileinfo *);
static int check_ogg_header(acv_fileinfo *);
static int read_ogg_comment_book(acv_fileinfo *);
static ogg_int64_t ogg_get_pcmlength(acv_fileinfo *);

static int read_ogg_frame(acv_fileinfo *);
static int decode_ogg_frame(acv_fileinfo *);
static int get_ogg_outdata(acv_fileinfo *,int);
//**************************************************************************
//Ogg encoder routines
int ogg_init_encoder(acv_fileinfo *af_out)
{
 int error;
 struct ogg_main_info *omip;
 double newkhz;

 /*if(af_out->freq<8000 || af_out->freq>48000){
  fprintf(stderr,"Cannot encode on %d Hz! Ogg encoding frequency must be between 8 and 48kHz! (use -of 44100)\n",af_out->freq);
  return -1;
 }*/
 af_out->decoder_info=omip=malloc(sizeof(struct ogg_main_info));
 if(omip==NULL)
  return 0;
 memset(omip,0,sizeof(struct ogg_main_info));
 if(!af_out->bitrate)
  af_out->bitrate=128000;

 vorbis_info_init(&omip->vi);

 if(opt_channel_coupling) // we must to set this before the bitrate management
  vorbis_encode_ctl(&omip->vi,OV_ECTL_COUPLE_SET,&opt_channel_coupling);

 if((opt_vbr_quality>=0.0f) && (opt_vbr_quality<=1.0f))
  error=vorbis_encode_setup_vbr(&omip->vi,af_out->channels,af_out->freq,opt_vbr_quality);
 else
  error=vorbis_encode_setup_managed(&omip->vi,af_out->channels,af_out->freq,-1,af_out->bitrate,-1);

 if(error!=0){
  fprintf(stderr,"Couldn't initialize encoder! (invalid freq or bitrate (-br) or -ocp required?)\n");
  return error;
 }

 if(!opt_const_bitrate)
  vorbis_encode_ctl(&omip->vi,OV_ECTL_RATEMANAGE2_SET,NULL);

 if(opt_lowpass_khz){
  newkhz=(double)opt_lowpass_khz;
  vorbis_encode_ctl(&omip->vi,OV_ECTL_LOWPASS_SET,&newkhz);
 }

 error=vorbis_encode_setup_init(&omip->vi);
 if(error!=0){
  fprintf(stderr,"Couldn't initialize Ogg encoder! (invalid settings?)\n");
  return error;
 }

 ogg_comment_set(af_out,&omip->vc);
 vorbis_analysis_init(&omip->vd,&omip->vi);
 vorbis_block_init(&omip->vd,&omip->vb);

 error=ogg_write_header(af_out);
 if(error!=0){
  fprintf(stderr,"Error writing Ogg header! (disk writable?)\n");
  return error;
 }
 af_out->databits=af_out->scalebits=BITTYPE_UNSCALED;
 af_out->filemode|=CF_FLOAT;
 return error;
}

static void acv_ogg_add_tag(vorbis_comment *vc,char *tagtype,char *taginfo)
{
 char *utf8=NULL;
 if(opt_tagenc_utf8){
  if(utf8_encode(taginfo,&utf8)>=0){
   vorbis_comment_add_tag(vc,tagtype,utf8);
  }else{
   if(!dispquiet)
    fprintf(stderr,"Warning: Couldn't convert tag to UTF8 style... added without conversion!\n");
   vorbis_comment_add_tag(vc,tagtype,taginfo);
  }
  if(utf8)
   free(utf8);
 }else{
  vorbis_comment_add_tag(vc,tagtype,taginfo);
 }
}

static void ogg_comment_set(acv_fileinfo *af_out,vorbis_comment *vc)
{
 char strtmp[128];
 vorbis_comment_init(vc);

 if(af_out->tag_artist)  acv_ogg_add_tag(vc,"ARTIST", af_out->tag_artist);
 if(af_out->tag_title)   acv_ogg_add_tag(vc,"TITLE",  af_out->tag_title);
 if(af_out->tag_album)   acv_ogg_add_tag(vc,"ALBUM",  af_out->tag_album);
 if(af_out->tag_date)    acv_ogg_add_tag(vc,"DATE",   af_out->tag_date);
 if(af_out->tag_genre)   acv_ogg_add_tag(vc,"GENRE",  af_out->tag_genre);
 if(af_out->tag_comment) acv_ogg_add_tag(vc,"COMMENT",af_out->tag_comment);

 sprintf(strtmp,"AudioCV v%s/%d for %s by PDSoft",PRG_VERSION,FLOAT_BITS,OS_TYPE);
 acv_ogg_add_tag(vc,"ENCODER",strtmp);
}

//-------------------------------------------------------------------------

static int ogg_write_header(acv_fileinfo *af_out)
{
 struct ogg_main_info *omip=af_out->decoder_info;
 ogg_packet header;
 ogg_packet header_comm;
 ogg_packet header_code;

 //srand(time(NULL));
 //ogg_stream_init(os,rand());
 ogg_stream_init(&omip->os,16); // for binary comparisons (tests)
 vorbis_analysis_headerout(&omip->vd,&omip->vc,&header,&header_comm,&header_code);
 ogg_stream_packetin(&omip->os,&header);
 ogg_stream_packetin(&omip->os,&header_comm);
 ogg_stream_packetin(&omip->os,&header_code);

 while(1){
  int result=ogg_stream_flush(&omip->os,&omip->og);
  if(result==0)
   break;
  fwrite(omip->og.header,1,omip->og.header_len,af_out->fp);
  fwrite(omip->og.body,1,omip->og.body_len,af_out->fp);
 }
 return 0;
}

int ogg_encode(acv_fileinfo *af_out)
{
 int eos=0;
 struct ogg_main_info *omip=af_out->decoder_info;
 if(af_out->blocksamplenum==0){
  vorbis_analysis_wrote(&omip->vd,0);
 }else{
  OGG_DOUBLE_T **buffer=vorbis_analysis_buffer(&omip->vd,PCM_INSAMPLES);
  unsigned int i,ch,outchs=af_out->channels;
  unsigned int bsnperch=af_out->blocksamplenum/outchs;
  for(ch=0;ch<outchs;ch++){
   PCM_CV_TYPE_F *inbuf=((PCM_CV_TYPE_F *)(af_out->buffer))+ch;
   OGG_DOUBLE_T *outbuf=buffer[ch];
   for(i=0;i<bsnperch;i++){
    PCM_CV_TYPE_F insamp=inbuf[0];
    outbuf[0]=(OGG_DOUBLE_T)insamp;
    inbuf+=outchs;
    outbuf++;
   }
  }
  vorbis_analysis_wrote(&omip->vd,bsnperch);
 }

 //RC3 & 1.0
 while(vorbis_analysis_blockout(&omip->vd,&omip->vb)==1){
  vorbis_analysis(&omip->vb,NULL);
  vorbis_bitrate_addblock(&omip->vb);

  while(vorbis_bitrate_flushpacket(&omip->vd,&omip->op)){
   ogg_stream_packetin(&omip->os,&omip->op);
   do{
    int writelen;
    int result=ogg_stream_pageout(&omip->os,&omip->og);
    if(result==0)
     break;

    writelen=(long)fwrite(omip->og.header,1,omip->og.header_len,af_out->fp);
    if(writelen!=omip->og.header_len)
     return ACV_ERROR_FILE_CANTWRITE;

    writelen=(long)fwrite(omip->og.body,1,omip->og.body_len,af_out->fp);
    if(writelen!=omip->og.body_len)
     return ACV_ERROR_FILE_CANTWRITE;

    if(ogg_page_eos(&omip->og))
     eos=1;
   }while(!eos);
  }
 }

 return 0;
}

int ogg_update_header(acv_fileinfo *af_out)
{
 //struct ogg_main_info *omip=af_out->decoder_info;
 long outtimesec=af_out->allsamplenum/(af_out->channels*af_out->freq);
 long outdatalen=acv_filelen(af_out->fp) - af_out->headerlen;
 int error=0;
 //if(outtimesec>20){
  //af_out->bitrate=omip->vi.bitrate_nominal=outdatalen*8/outtimesec;
 if(outtimesec)
  af_out->bitrate=outdatalen*8/outtimesec;
  //error=ogg_write_header(af_out);
 //}else
  af_out->headerlen=0;

 return error;
}

void ogg_close_encoder(acv_fileinfo *af_out)
{
 struct ogg_main_info *omip;

 if(af_out->decoder_info){
  omip=af_out->decoder_info;

  ogg_stream_clear(&omip->os);
  vorbis_block_clear(&omip->vb);
  vorbis_dsp_clear(&omip->vd);
  vorbis_comment_clear(&omip->vc);
  vorbis_info_clear(&omip->vi);

  free(af_out->decoder_info);
  af_out->decoder_info=NULL;
 }
}

//***************************************************************************
// Ogg decoder routines

int ogg_init_decoder(acv_fileinfo *af_in)
{
 struct ogg_main_info *omip;
 af_in->decoder_info=omip=malloc(sizeof(struct ogg_main_info));
 if(omip==NULL)
  return 0;
 memset(omip,0,sizeof(struct ogg_main_info));
 fseek(af_in->fp,0,SEEK_SET);

 af_in->filetype=FT_OGG;

 if(!check_ogg_header(af_in))      // unpack info
  return -1;
 if(read_ogg_comment_book(af_in)<0) // unpack comment
  return -1;
 if(read_ogg_comment_book(af_in)<0) // unpack book
  return -1;

 vorbis_synthesis_init(&omip->vd,&omip->vi);
 if(vorbis_block_init(&omip->vd,&omip->vb)<0)
  return -1;

 af_in->allsamplenum  =(unsigned long)(ogg_get_pcmlength(af_in))*omip->vi.channels;
 if(!af_in->allsamplenum){
  if(!omip->vi.bitrate_nominal)
   return -1;
  af_in->headerlen=ftell(af_in->fp)-(omip->oy.fill-omip->oy.returned);
  af_in->allsamplenum=(unsigned long)((float)(acv_filelen(af_in->fp)-af_in->headerlen)
		      *(float)omip->vi.rate*16.0f*(float)omip->vi.channels
		      /(float)omip->vi.bitrate_nominal
		      /2.0f);
 }

 af_in->channels      =omip->vi.channels;   // number of channels
 af_in->freq          =omip->vi.rate;       // frequency;
 af_in->bitrate       =omip->vi.bitrate_nominal;
 af_in->databits =BITTYPE_UNDEFINED;
 af_in->scalebits=BITTYPE_UNDEFINED;
 af_in->filemode|=CF_FLOAT;

 // extract artist/title/etc. infos
 {
  char *comment_types[6]={"title","artist","album","date","comment","genre"};
  char **commentsp;
  char **af_ptrs[6];
  int i;

  af_ptrs[0]=&af_in->tag_title;
  af_ptrs[1]=&af_in->tag_artist;
  af_ptrs[2]=&af_in->tag_album;
  af_ptrs[3]=&af_in->tag_date;
  af_ptrs[4]=&af_in->tag_comment;
  af_ptrs[5]=&af_in->tag_genre;

  commentsp=omip->vc.user_comments;
  if(commentsp==NULL)
   return 0;
  while(*commentsp){
   char *p=strchr(*commentsp,'=');
   if(p!=NULL){
    *p++=0;
    for(i=0;i<6;i++){
     if(stricmp(*commentsp,comment_types[i])==0){
      *af_ptrs[i]=p;
      break;
     }
    }
   }
   commentsp++;
  }
 }

 return 0;
}

void ogg_decode(acv_fileinfo *af_in,int samplenum)
{
 struct ogg_main_info *omip=af_in->decoder_info;
 int eos=0,outsamples=0;
 while(!eos && !outsamples){
  switch(omip->current_decoder_part){
   case 0:if(read_ogg_frame(af_in))
	   omip->current_decoder_part=1;
	  else{
	   outsamples=get_ogg_outdata(af_in,samplenum);
	   eos=1;
	   break;
	  }
   case 1:if(decode_ogg_frame(af_in))
	   omip->current_decoder_part=2;
	  else{
	   omip->current_decoder_part=0;
	   break;
	  }
   case 2:outsamples=get_ogg_outdata(af_in,samplenum);
	  if(!outsamples)
	   omip->current_decoder_part=1;
	  break;
  }
 }
 af_in->blocksamplenum=outsamples;
}

void ogg_close_decoder(acv_fileinfo *af_in)
{
 struct ogg_main_info *omip;

 if(af_in->decoder_info){
  omip=af_in->decoder_info;

  vorbis_block_clear(&(omip->vb));
  ogg_stream_clear(&omip->os);
  vorbis_dsp_clear(&omip->vd);
  vorbis_info_clear(&omip->vi);
  ogg_sync_clear(&omip->oy);
  free(af_in->decoder_info);
  af_in->decoder_info=NULL;
 }
}

//------------------------------------------------------------------------

static int refill_sync_buff(acv_fileinfo *af_in)
{
 int bytes;
 char *buffer;
 struct ogg_main_info *omip=af_in->decoder_info;
 buffer=ogg_sync_buffer(&omip->oy,OGG_SYNC_BUFFER_SIZE);
 bytes=fread(buffer,1,OGG_SYNC_BUFFER_SIZE,af_in->fp);
 ogg_sync_wrote(&omip->oy,bytes);
 return bytes;
}

static int check_ogg_header(acv_fileinfo *af_in)
{
 struct ogg_main_info *omip=af_in->decoder_info;

 ogg_sync_init(&omip->oy);

 if(refill_sync_buff(af_in)<OGG_SYNC_BUFFER_SIZE) // too short header
  return 0;

 if(ogg_sync_pageout(&omip->oy,&omip->og)!=1)   // Ogg header not found
  return 0;

 ogg_stream_init(&omip->os,ogg_page_serialno(&omip->og));
 vorbis_info_init(&omip->vi);
 vorbis_comment_init(&omip->vc);

 if(ogg_stream_pagein(&omip->os,&omip->og)<0)
  return 0;
 if(ogg_stream_packetout(&omip->os,&omip->op)!=1)
  return 0;
 if(vorbis_synthesis_headerin(&omip->vi,&omip->vc,&omip->op)<0)  // unpack_info
  return 0;

 return 1;
}

static int read_ogg_comment_book(acv_fileinfo *af_in)
{
 int result;
 struct ogg_main_info *omip=af_in->decoder_info;
 while(1){
  result=ogg_sync_pageout(&omip->oy,&omip->og);
  if(result==1){
   ogg_stream_pagein(&omip->os,&omip->og);
   result=ogg_stream_packetout(&omip->os,&omip->op);
   if(result<0)
    return 0;
   if(result>0)
    return vorbis_synthesis_headerin(&omip->vi,&omip->vc,&omip->op);
  }
  if(result==0)
   if(refill_sync_buff(af_in)==0)
    return 0;
 }
 return 1;
}

#define OGGID_OggS 0x5367674f   // 'SggO' (OggS)
#define OGG_CHUNKSIZE 8192
#define OGG_CHUNKNUM  8

static ogg_int64_t ogg_get_pcmlength(acv_fileinfo *af_in)
{
 static ogg_sync_state oy;
 static ogg_page og;
 ogg_int64_t pcmlen=0;
 char *ptr,*buffer;
 long newfilepos=0,oldfilepos=ftell(af_in->fp);
 long filelen=acv_filelen(af_in->fp);
 long bytes,oyret;
 int retry1=5,retry2;

 if(!filelen)
  return 0;
 ogg_sync_init(&oy);
 ogg_sync_buffer(&oy,OGG_CHUNKSIZE*OGG_CHUNKNUM);
 if(oy.data==NULL)
  return 0;
 while(retry1--){
  retry2=30;
  memset((void *)(&og),0,sizeof(ogg_page));
  oy.returned=oy.fill=OGG_CHUNKSIZE*OGG_CHUNKNUM;
  newfilepos=0;
  do{
   if((newfilepos-OGG_CHUNKSIZE)>(-filelen)){
    bytes=OGG_CHUNKSIZE;
    newfilepos-=OGG_CHUNKSIZE;
   }else{
    bytes=filelen+newfilepos;
    newfilepos=-(filelen);
   }
   fseek(af_in->fp,newfilepos,SEEK_END);
   ptr=oy.data+oy.returned;
   buffer=ptr-bytes;
   bytes=fread(buffer,1,bytes,af_in->fp);
   if(bytes<1)
    break;
   while((bytes>0) && (pcmlen<1)){
    while((bytes>0) && *((unsigned long *)(ptr))!=OGGID_OggS){
     ptr--;
     bytes--;
     oy.returned--;
    }
    oyret=oy.returned;
    if(*((unsigned long *)(ptr))==OGGID_OggS){
     if(ogg_sync_pageout(&oy,&og)>0)
      pcmlen=ogg_page_granulepos(&og);
     if(pcmlen<1)
      if(bytes>0){
       ptr--;
       bytes--;
       oyret--;
      }
    }
    oy.headerbytes=0;
    oy.bodybytes=0;
    oy.unsynced=0;
    oy.returned=oyret;
   }
  }while((pcmlen<1) && (oy.returned>0) && (retry2--));
 }
 ogg_sync_clear(&oy);
 fseek(af_in->fp,oldfilepos,SEEK_SET);
 return pcmlen;
}

//---------------------------------------------------------------------------

static int read_ogg_frame(acv_fileinfo *af_in)
{
 int result,counter=50;
 struct ogg_main_info *omip=af_in->decoder_info;
 do{
  result=ogg_sync_pageout(&omip->oy,&omip->og);
  if(result>0)
   ogg_stream_pagein(&omip->os,&omip->og);
  if(result==0)
   if(!refill_sync_buff(af_in))
    return 0;
 }while(result<=0 && counter--);
 if(result<=0)
  return 0;
 return 1;
}

static int decode_ogg_frame(acv_fileinfo *af_in)
{
 struct ogg_main_info *omip=af_in->decoder_info;
 if(ogg_stream_packetout(&omip->os,&omip->op)>0)
  if(vorbis_synthesis(&omip->vb,&omip->op)==0){
   vorbis_synthesis_blockin(&omip->vd,&omip->vb);
   return 1;
  }
 return 0;
}

static int get_ogg_outdata(acv_fileinfo *af_in,int samplenum)
{
 struct ogg_main_info *omip=af_in->decoder_info;
 int i,j,vich=af_in->channels,bufsamples,outsamples=0;
 OGG_DOUBLE_T **pcm;
 if((bufsamples=vorbis_synthesis_pcmout(&omip->vd,&pcm))>0){
  outsamples=(samplenum<bufsamples)? samplenum:bufsamples;
  for(i=0;i<vich;i++){
   PCM_CV_TYPE_F *outptr=((PCM_CV_TYPE_F *)(af_in->buffer))+i;
   OGG_DOUBLE_T *inptr=pcm[i];
   for(j=0;j<outsamples;j++){
    OGG_DOUBLE_T insamp=*inptr;
    *outptr=(PCM_CV_TYPE_F)insamp;
    inptr++;
    outptr+=vich;
   }
  }
 }
 vorbis_synthesis_read(&omip->vd,outsamples);
 outsamples*=vich;
 return outsamples;
}
