//**************************************************************************
//*                     This file is part of the                           *
//*               AAC encoder output DLL plug-in for                       *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2007 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//AAC encoding with FAAC library

#include "mpxplay.h"
#include "faac.h"
#include "newfunc\newfunc.h"
#include "control\control.h"
//#include "utf8.h"
#include <string.h>
#include <malloc.h>

#define DEFAULT_TNS     0

enum stream_format {
  RAW_STREAM = 0,
  ADTS_STREAM = 1,
};

enum container_format {
  NO_CONTAINER,
  MP4_CONTAINER,
};

enum flags {
  SHORTCTL_FLAG = 300,
  TNS_FLAG = 301,
  NO_TNS_FLAG = 302,
  MPEGVERS_FLAG = 303,
  OBJTYPE_FLAG = 304,
  NO_MIDSIDE_FLAG = 306,

#ifdef HAVE_LIBMP4V2
  ARTIST_FLAG = 320,
  TITLE_FLAG = 321,
  GENRE_FLAG = 322,
  ALBUM_FLAG = 323,
  COMPILATION_FLAG = 324,
  TRACK_FLAG = 325,
  DISC_FLAG = 326,
  YEAR_FLAG = 327,
  COVER_ART_FLAG = 328,
  COMMENT_FLAG = 329,
  WRITER_FLAG = 330,
#endif
};

typedef struct aac_main_info{
 int filehand;
 faacEncHandle hEncoder;
 unsigned long samplesInput, maxBytesOutput;
 faacEncConfigurationPtr myFormat;
 float *pcmbuff;
 unsigned int pcmbuf_samples;
 unsigned char *bitbuf;
}aac_main_info;

static int opt_bitrate,opt_freq_cutoff=-1,opt_quantqual;
static int opt_usemidside=1,opt_usetns=DEFAULT_TNS;
static int opt_streamtype=ADTS_STREAM,opt_container=NO_CONTAINER;
static char *opt_objtype_str;
//static int   opt_utf8_disable;
//static char *opt_charset;

static topt aacenc_opts[] = {
{"faac_qq" ,ARG1|ARG_NUM,      &opt_quantqual    ,0,0},
{"faac_br" ,ARG1|ARG_NUM,      &opt_bitrate      ,0,0},
{"faac_co" ,ARG1|ARG_NUM,      &opt_freq_cutoff  ,0,0},
{"faac_raw",0,                 &opt_streamtype   ,RAW_STREAM,0},
{"faac_obj",ARG1|ARG_CHAR,     &opt_objtype_str  ,0,0},
{"faac_tns",0,                 &opt_usetns       ,1,0},
{"faac_nms",0,                 &opt_usemidside   ,0,0},
//{"faac_mp4",0,                 &opt_container    ,MP4_CONTAINER,0},
//{"o_tn8" ,0,                 &opt_utf8_disable    ,1,0},
//{"o_tecp",ARG2|ARG_CHAR,     &opt_utf8_disable    ,0,&opt_charset},
{NULL    ,0,0,0,0}
};

#ifdef __DOS__
struct mpxplay_resource_s *mrs;
#define FAAC_PRINTF(s) mrs->pds_textdisplay_printf(s)
#else
#include <share.h>
void FAAC_PRINTF(char *s)
{
 printf("%s\n",s);
}
#endif

static unsigned int aec_ready;

static void aac_encoder_config(void)
{
 if(!aec_ready){
  opt_bitrate*=1000;
  aec_ready=1;
 }
}

static int aac_encoder_open(struct mpxplay_audioout_info_s *aui)
{
 struct aac_main_info *amip;
 unsigned int mpegVersion = MPEG2,objectType;
 int cutOff;
 char fullname[300];

 if(!aui->pei){
  FAAC_PRINTF("aacenc error: no aui->pei!");
  return -1;
 }

 if(sizeof(MPXPLAY_PCMOUT_FLOAT_T)!=sizeof(float)){
  FAAC_PRINTF("Mpxplay/faac float mismatch!");
  return -1;
 }

 aui->card_private_data=amip=malloc(sizeof(struct aac_main_info));
 if(amip==NULL)
  return 0;
 memset(amip,0,sizeof(struct aac_main_info));

 pds_getfilename_noext_from_fullname(fullname,aui->pei->filename);
 pds_strcat(fullname,".AAC");

 if(pds_stricmp(fullname,aui->pei->filename)!=0) // input and output filename must be different
#ifdef __DOS__
  amip->filehand=mrs->pds_open_create(fullname,O_RDWR|O_BINARY);
#else
  amip->filehand=sopen(fullname,O_RDWR|O_BINARY|O_CREAT|O_TRUNC,SH_DENYWR,S_IREAD|S_IWRITE);
#endif
 if(!amip->filehand){
  FAAC_PRINTF("aacenc error: Cannot open output file!");
  return -1;
 }

 amip->hEncoder=faacEncOpen(aui->freq_card,aui->chan_card,&amip->samplesInput,&amip->maxBytesOutput);
 if(!amip->hEncoder){
  FAAC_PRINTF("aacenc error: Cannot initialize encoder!");
  return -1;
 }

 if(opt_container == MP4_CONTAINER){
  mpegVersion = MPEG4;
  opt_streamtype = RAW_STREAM;
 }

 amip->pcmbuff=(float *)malloc(3*amip->samplesInput*sizeof(float));
 if(!amip->pcmbuff)
  return -1;
 amip->bitbuf=(unsigned char*)malloc(amip->maxBytesOutput*sizeof(unsigned char));
 if(!amip->bitbuf)
  return -1;

 cutOff=opt_freq_cutoff;

 if(cutOff <= 0){
  if(cutOff < 0) // default
   cutOff = 0;
  else // disabled
   cutOff = aui->freq_card / 2;
 }
 if(cutOff > (aui->freq_card / 2))
  cutOff = aui->freq_card / 2;

 objectType=LOW;
 if(opt_objtype_str){
  if(pds_stricmp(opt_objtype_str,"Main")==0)
   objectType=MAIN;
  else
   if(pds_stricmp(opt_objtype_str,"LTP")==0){
    objectType=LTP;
    mpegVersion=MPEG4;
   }
 }

 amip->myFormat = faacEncGetCurrentConfiguration(amip->hEncoder);
 amip->myFormat->aacObjectType = objectType;
 amip->myFormat->mpegVersion = mpegVersion;
 amip->myFormat->useTns = opt_usetns;
 amip->myFormat->shortctl = SHORTCTL_NORMAL;
 if(aui->chan_card >= 6)
  amip->myFormat->useLfe = 1;
 amip->myFormat->allowMidside = opt_usemidside;
 if(opt_bitrate)
  amip->myFormat->bitRate = opt_bitrate / aui->chan_card;
 amip->myFormat->bandWidth = cutOff;
 if(opt_quantqual > 0)
  amip->myFormat->quantqual = opt_quantqual;
 amip->myFormat->outputFormat = opt_streamtype;
 amip->myFormat->inputFormat = FAAC_INPUT_FLOAT;
 if(!faacEncSetConfiguration(amip->hEncoder, amip->myFormat)){
  FAAC_PRINTF("aacenc error: Unsupported output format!");
  return -1;
 }

 aui->card_wave_id=0x0003; // 32-bit float pcm needed
 aui->bits_card=16;        // range: -32768.0 to +32767.0
 return 0;
}

//-------------------------------------------------------------------------

static void AAC_writedata(struct mpxplay_audioout_info_s *aui,char *pcm_sample,unsigned long outbytes)
{
 struct aac_main_info *amip=aui->card_private_data;
 int bytesWritten;

 if(!amip)
  return;

 memcpy((void *)&amip->pcmbuff[amip->pcmbuf_samples],(void *)pcm_sample,outbytes);

 amip->pcmbuf_samples+=outbytes/sizeof(float);

 if(amip->pcmbuf_samples<amip->samplesInput)
  return;

 bytesWritten = faacEncEncode(amip->hEncoder,
                (int32_t *)amip->pcmbuff,
                amip->samplesInput,
                amip->bitbuf,
                amip->maxBytesOutput);

 if(bytesWritten>0)
#ifdef __DOS__
  mrs->pds_dos_write(amip->filehand,amip->bitbuf,bytesWritten);
#else
  write(amip->filehand,amip->bitbuf,bytesWritten);
#endif

 memcpy((void*)amip->pcmbuff,(void *)&amip->pcmbuff[amip->samplesInput],(amip->pcmbuf_samples-amip->samplesInput)*sizeof(float));
 amip->pcmbuf_samples-=amip->samplesInput;

 return;
}

static long AAC_bufpos(struct mpxplay_audioout_info_s *aui)
{
 struct aac_main_info *amip=aui->card_private_data;

 if(!amip)
  return (1024*sizeof(float));

 return (amip->samplesInput*sizeof(float)); // bytenum request
}

static void AAC_close(struct mpxplay_audioout_info_s *aui)
{
 struct aac_main_info *amip;

 if(aui->card_private_data){
  amip=aui->card_private_data;

  if(amip->hEncoder) faacEncClose(amip->hEncoder);
#ifdef __DOS__
  if(amip->filehand) mrs->pds_close(amip->filehand);
#else
  if(amip->filehand) close(amip->filehand);
#endif
  if(amip->pcmbuff)  free(amip->pcmbuff);
  if(amip->bitbuf)   free(amip->bitbuf);

  free(aui->card_private_data);
  aui->card_private_data=NULL;
 }
}

//--------------------------------------------------------------------
static int AAC_init(struct mpxplay_audioout_info_s *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 aac_encoder_config();
 return 1;
}

static int AAC_detect(struct mpxplay_audioout_info_s *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 aac_encoder_config();
 return 1;
}

static void AAC_card_info(struct mpxplay_audioout_info_s *aui)
{
 FAAC_PRINTF("------------------------------------------------------------------");
 FAAC_PRINTF("FAAC : AAC audio encoder output plugin (disk writer) v1.54");
 FAAC_PRINTF("");
 FAAC_PRINTF("Available options (use in the command line of Mpxplay):");
 FAAC_PRINTF(" -faac_qq NUM : quant quality (10-500)");
 FAAC_PRINTF(" -faac_br NUM : ABR bitrate (64-170)");
 FAAC_PRINTF(" -faac_co NUM : cut-off frequency (4000-22050)");
 FAAC_PRINTF(" -faac_raw    : write RAW format (useless)");
 FAAC_PRINTF(" -faac_obj OBJ: select object type (LC (default),Main,LTP)");
 FAAC_PRINTF(" -faac_tns    : enable tns coding");
 FAAC_PRINTF(" -faac_nms    : no mid/side coding");
 //FAAC_PRINTF(" -faac_mp4    : put aac stream in MP4 container");
 FAAC_PRINTF("------------------------------------------------------------------");
}

static void AAC_setrate(struct mpxplay_audioout_info_s *aui)
{
 AAC_close(aui);
 if(aac_encoder_open(aui)!=0)
  AAC_close(aui);
}

one_sndcard_info AAC_sndcard_info={
 "AAC",
 SNDCARD_FLAGS_DISKWRITER|SNDCARD_CARDBUF_SPACE,

 NULL,            // card_config
 &AAC_init,       // card_init
 &AAC_detect,     // card_detect
 &AAC_card_info,  // card_info
 NULL,            // card_start
 NULL,            // card_stop
 &AAC_close,      // card_close
 &AAC_setrate,    // card_setrate

 &AAC_writedata,  // cardbuf_writedata
 &AAC_bufpos,     // cardbuf_pos
 NULL,            // cardbuf_clear
 NULL,            // cardbuf_int_monitor
 NULL,            // irq_routine

 NULL,            // card_writemixer
 NULL,            // card_readmixer
 NULL             // card_mixerchans
};

static mpxplay_module_entry_s aacenc_output_module_entry={
 MPXPLAY_DLLMODULETYPE_AUCARD,
 0,
 "FAAC",
 MPXPLAY_DLLMODULEVER_AUCARD,
 (void *)&AAC_sndcard_info
};

static mpxplay_module_entry_s aacenc_cmdline_module_entry={
 MPXPLAY_DLLMODULETYPE_CONTROL_CMDLINE,
 0,
 NULL,
 MPXPLAY_DLLMODULEVER_CONTROL_CMDLINE,
 (void *)aacenc_opts
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &aacenc_output_module_entry,
  &aacenc_cmdline_module_entry,
  NULL
 }
};

#ifdef __DOS__

extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin);
 return (long)(&mpxplay_dll_entry_structure);
}

#else

int matherr(struct _exception *a)
{
 a->retval=1.0;
 return 1;
}

__declspec( dllexport ) mpxplay_dll_entry_s *mpxplay_dll_entrypoint(void)
{
 return (&mpxplay_dll_entry_structure);
}

#endif
