/*
** .WV input plug-in for WavPack
** Copyright (c) 2000 - 2003, Conifer Software, All Rights Reserved
*/

#include <windows.h>
#include <stdio.h>
#include <mmreg.h>
#include <msacm.h>
#include <math.h>

#include "in2.h"
#include "wavpack.h"
#include "resource.h"

static HANDLE open_wavpack_file (char *fn, WaveHeader *wavhdr,
    WavpackHeader *wvpkhdr, M_Tag *m_tag, char *error, HANDLE *in2file);

static void kill_wavpack_index (void);

static int unpack_wavpack_data (char *buffer, WavpackHeader *wphdr, HANDLE infile, HANDLE in2file,
    int *sample_index, int sample_count, int *bytes_used);

static int wavpack_seek (WavpackHeader *wphdr, int *sample_index, int desired_index);
static void close_wavpack_file (HANDLE infile, HANDLE in2file, M_Tag *m_tag);
static float calculate_gain (M_Tag *m_tag, int *pSoftClip);

//#define DEBUG_CONSOLE

// use CRT. Good. Useful. Portable.
BOOL WINAPI DllMain (HANDLE hInst, ULONG ul_reason_for_call, LPVOID lpReserved)
{
    return TRUE;
}

// post this to the main window at end of file (after playback as stopped)
#define WM_WA_MPEG_EOF WM_USER+2

#define MAX_NCH 2
#define MAX_BPS 32

#define BPS (WavHdr.BitsPerSample > 16 ? 32 : 16)
#define NCH (WavHdr.NumChannels)

In_Module mod;		// the output module (declared near the bottom of this file)
char lastfn[MAX_PATH];	// currently playing file (used for getting info on the current file)
int decode_pos_ms;	// current decoding position, in milliseconds
int decode_pos_sample;	// current decoding position, in samples
int paused;		// are we paused?
int seek_needed;	// if != -1, it is the point that the decode thread should seek to, in ms.
char sample_buffer[576*MAX_NCH*(MAX_BPS/8)*2];	// sample buffer

#define IGNORE_WVC		0x1
#define REPLAYGAIN_TRACK	0x2
#define REPLAYGAIN_ALBUM	0x4
#define SOFTEN_CLIPPING		0x8
#define PREVENT_CLIPPING	0x10

int config_bits;	// all configuration goes here

WaveHeader WavHdr;	// standard wav header for sample rate, etc.
WavpackHeader WvpkHdr;	// WavPack header for version, mode, etc.
M_Tag m_tag;		// ID3v1 or APEv2 Tag (if present)
float play_gain;	// playback gain (for replaygain support)
int soft_clipping;	// soft clipping active for playback

HANDLE input_file=INVALID_HANDLE_VALUE;	// input file handle
HANDLE wvc_file=INVALID_HANDLE_VALUE;	// wvc file handle

int killDecodeThread=0;				// the kill switch for the decode thread
HANDLE thread_handle=INVALID_HANDLE_VALUE;	// the handle to the decode thread

DWORD WINAPI __stdcall DecodeThread(void *b);	// the decode thread procedure

static BOOL CALLBACK WavPackDlgProc (HWND, UINT, WPARAM, LPARAM);

void config (HWND hwndParent)
{
    char dllname [512];
    int temp_config;
    HMODULE module;
    HANDLE confile;
    DWORD result;

    module = GetModuleHandle ("in_wv.dll");
    temp_config = DialogBoxParam (module, "WinAmp", hwndParent, WavPackDlgProc, config_bits);

    if (temp_config == config_bits || (temp_config & 0xffffffe0) ||
	(temp_config & 6) == 6 || (temp_config & 0x18) == 0x18)
	    return;

    config_bits = temp_config;

    if (module && GetModuleFileName (module, dllname, sizeof (dllname))) {
	dllname [strlen (dllname) - 2] = 'a';
	dllname [strlen (dllname) - 1] = 't';

	confile = CreateFile (dllname, GENERIC_WRITE, 0, NULL,
	    CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

	if (confile == INVALID_HANDLE_VALUE)
	    return;

	WriteFile (confile, &config_bits, sizeof (config_bits), &result, NULL);
	CloseHandle (confile);
    }
}

BOOL CALLBACK WavPackDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    static int local_config;
    char str [80];
    int i;

    switch (message) {
	case WM_INITDIALOG:
	    local_config = lParam;

	    CheckDlgButton (hDlg, IDC_USEWVC, ~local_config & IGNORE_WVC);

	    SendDlgItemMessage (hDlg, IDC_REPLAYGAIN, CB_ADDSTRING, 0, (long) "disabled");
	    SendDlgItemMessage (hDlg, IDC_REPLAYGAIN, CB_ADDSTRING, 0, (long) "use track gain");
	    SendDlgItemMessage (hDlg, IDC_REPLAYGAIN, CB_ADDSTRING, 0, (long) "use album gain");
	    SendDlgItemMessage (hDlg, IDC_REPLAYGAIN, CB_SETCURSEL, (local_config >> 1) & 3, 0);

	    SendDlgItemMessage (hDlg, IDC_CLIPPING, CB_ADDSTRING, 0, (long) "just clip peaks");
	    SendDlgItemMessage (hDlg, IDC_CLIPPING, CB_ADDSTRING, 0, (long) "softly clip peaks");
	    SendDlgItemMessage (hDlg, IDC_CLIPPING, CB_ADDSTRING, 0, (long) "scale track to prevent clips");
	    SendDlgItemMessage (hDlg, IDC_CLIPPING, CB_SETCURSEL, (local_config >> 3) & 3, 0);

	    if (!(local_config & (REPLAYGAIN_TRACK | REPLAYGAIN_ALBUM))) {
		EnableWindow (GetDlgItem (hDlg, IDC_CLIPPING_TEXT), FALSE);
		EnableWindow (GetDlgItem (hDlg, IDC_CLIPPING), FALSE);
	    }

	    SetFocus (GetDlgItem (hDlg, IDC_USEWVC));
	    return FALSE;

	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case IDC_REPLAYGAIN:
		    if (SendDlgItemMessage (hDlg, IDC_REPLAYGAIN, CB_GETCURSEL, 0, 0)) {
			EnableWindow (GetDlgItem (hDlg, IDC_CLIPPING_TEXT), TRUE);
			EnableWindow (GetDlgItem (hDlg, IDC_CLIPPING), TRUE);
		    }
		    else {
			EnableWindow (GetDlgItem (hDlg, IDC_CLIPPING_TEXT), FALSE);
			EnableWindow (GetDlgItem (hDlg, IDC_CLIPPING), FALSE);
		    }

		    break;

		case IDOK:
		    local_config = 0;

		    if (!IsDlgButtonChecked (hDlg, IDC_USEWVC))
			local_config |= IGNORE_WVC;

		    local_config |= SendDlgItemMessage (hDlg, IDC_REPLAYGAIN, CB_GETCURSEL, 0, 0) << 1;
		    local_config |= SendDlgItemMessage (hDlg, IDC_CLIPPING, CB_GETCURSEL, 0, 0) << 3;

		case IDCANCEL:
		    EndDialog (hDlg, local_config);
		    return TRUE;
	    }

	    break;
    }

    return FALSE;
}

void about (HWND hwndParent)
{
    MessageBox (hwndParent,"WavPack Player Version 1.8.1 \nCopyright (c) 2003 Conifer Software","About WavPack Player",MB_OK);
}

void init() { /* any one-time initialization goes here (configuration reading, etc) */
    char dllname [512];
    HMODULE module;
    HANDLE confile;
    DWORD result;

    module = GetModuleHandle ("in_wv.dll");
    config_bits = 0;

    if (module && GetModuleFileName (module, dllname, sizeof (dllname))) {
	dllname [strlen (dllname) - 2] = 'a';
	dllname [strlen (dllname) - 1] = 't';

	confile = CreateFile (dllname, GENERIC_READ, FILE_SHARE_READ, NULL,
	    OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

	if (confile == INVALID_HANDLE_VALUE)
	    return;

	if (!ReadFile (confile, &config_bits, sizeof (config_bits), &result, NULL) ||
	    result != sizeof (config_bits))
		config_bits = 0;

	CloseHandle (confile);
    }
}

#ifdef DEBUG_CONSOLE

HANDLE debug_console=INVALID_HANDLE_VALUE;	// debug console

void debug_write (char *str)
{
    static int cant_debug;

    if (cant_debug)
	return;

    if (debug_console == INVALID_HANDLE_VALUE) {
	AllocConsole ();

#if 1
	debug_console = GetStdHandle (STD_OUTPUT_HANDLE);
#else
	debug_console = CreateConsoleScreenBuffer (GENERIC_WRITE, FILE_SHARE_WRITE,
	    NULL, CONSOLE_TEXTMODE_BUFFER, NULL);
#endif

	if (debug_console == INVALID_HANDLE_VALUE) {
	    MessageBox(NULL, "Can't get a console handle", "WavPack",MB_OK);
	    cant_debug = 1;
	    return;
	}
	else if (!SetConsoleActiveScreenBuffer (debug_console)) {
	    MessageBox(NULL, "Can't activate console buffer", "WavPack",MB_OK);
	    cant_debug = 1;
	    return;
	}
    }

    WriteConsole (debug_console, str, strlen (str), NULL, NULL);
}

#endif

void quit() { /* one-time deinit, such as memory freeing */
#ifdef DEBUG_CONSOLE
    if (debug_console != INVALID_HANDLE_VALUE) {
	FreeConsole ();

	if (debug_console != GetStdHandle (STD_OUTPUT_HANDLE))
	    CloseHandle (debug_console);

	debug_console = INVALID_HANDLE_VALUE;
    }
#endif
}

int isourfile(char *fn)
{
    return 0;
} 
// used for detecting URL streams.. unused here. strncmp(fn,"http://",7) to detect HTTP streams, etc

int play (char *fn) 
{ 
    WavpackHeader last_wvpkhdr = WvpkHdr;
    char error [128];
    int maxlatency;
    int thread_id;

#ifdef DEBUG_CONSOLE
    sprintf (error, "play (%s)\n", fn);
    debug_write (error);
#endif

    wvc_file = INVALID_HANDLE_VALUE;

    input_file = open_wavpack_file (fn, &WavHdr, &WvpkHdr, &m_tag, error,
	(config_bits & IGNORE_WVC) ? NULL : &wvc_file);

    if (input_file == INVALID_HANDLE_VALUE) {	// error opening file
	MessageBox (NULL, error, "WavPack Player", MB_OK);
	return 1;
    }

    play_gain = calculate_gain (&m_tag, &soft_clipping);

    // If this file is different than the previous, or if it's the same file
    // but the WVC_FLAG has changed because the user changed the configuration
    // or deleted the .wvc file, kill our index table.

    if (memcmp (&last_wvpkhdr, &WvpkHdr, sizeof (WavpackHeader)) || strcmp (lastfn, fn)) {
	kill_wavpack_index ();
	strcpy (lastfn, fn);
    }

    paused = 0;
    decode_pos_ms = decode_pos_sample = 0;
    seek_needed = -1;

    maxlatency = mod.outMod->Open (WavHdr.SampleRate, NCH, BPS, -1, -1);

    if (maxlatency < 0) { // error opening device
	close_wavpack_file (input_file, wvc_file, &m_tag);
	input_file = INVALID_HANDLE_VALUE;
	return 1;
    }

    // dividing by 1000 for the first parameter of setinfo makes it
    // display 'H'... for hundred.. i.e. 14H Kbps.

    mod.SetInfo (0, (WavHdr.SampleRate + 500) / 1000, NCH, 1);

    // initialize vis stuff

    mod.SAVSAInit (maxlatency, WavHdr.SampleRate);
    mod.VSASetInfo (WavHdr.SampleRate, NCH);

    mod.outMod->SetVolume (-666);	// set the output plug-ins default volume

    killDecodeThread=0;

    thread_handle = (HANDLE) CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) DecodeThread,
	(void *) &killDecodeThread, 0, &thread_id);

    if (SetThreadPriority (thread_handle, THREAD_PRIORITY_HIGHEST) == 0) {
	close_wavpack_file (input_file, wvc_file, &m_tag);
	input_file = INVALID_HANDLE_VALUE;
	return 1;
    }

    return 0; 
}

void pause ()
{
#ifdef DEBUG_CONSOLE
    debug_write ("pause ()\n");
#endif

    paused = 1;
    mod.outMod->Pause (1);
}

void unpause ()
{
#ifdef DEBUG_CONSOLE
    debug_write ("unpause ()\n");
#endif

    paused = 0;
    mod.outMod->Pause (0);
}

int ispaused ()
{
    return paused;
}

void stop()
{ 
#ifdef DEBUG_CONSOLE
    debug_write ("stop ()\n");
#endif

    if (thread_handle != INVALID_HANDLE_VALUE) {

	killDecodeThread = 1;

	if (WaitForSingleObject (thread_handle, INFINITE) == WAIT_TIMEOUT) {
	    MessageBox(mod.hMainWindow,"error asking thread to die!\n", "error killing decode thread", 0);
	    TerminateThread(thread_handle,0);
	}

	CloseHandle (thread_handle);
	thread_handle = INVALID_HANDLE_VALUE;
    }

    if (input_file != INVALID_HANDLE_VALUE) {
	close_wavpack_file (input_file, wvc_file, &m_tag);
	input_file=INVALID_HANDLE_VALUE;
    }

    mod.outMod->Close ();
    mod.SAVSADeInit ();
}

int getlength()
{ 
    return (int)(WvpkHdr.total_samples * 1000.0 / WavHdr.SampleRate); 
}

int getoutputtime()
{ 
    return decode_pos_ms + (mod.outMod->GetOutputTime () - mod.outMod->GetWrittenTime ());
}

void setoutputtime (int time_in_ms)
{ 
#ifdef DEBUG_CONSOLE
    char str [40];
    sprintf (str, "setoutputtime (%d)\n", time_in_ms);
    debug_write (str);
#endif

    seek_needed = time_in_ms; 
}

void setvolume (int volume)
{
    mod.outMod->SetVolume (volume);
}

void setpan (int pan)
{
    mod.outMod->SetPan(pan);
}

int infoDlg (char *fn, HWND hwnd)
{
    WavpackHeader wvpkhdr;
    WaveHeader wavhdr;
    char string [1024];
    HANDLE hFile, hFile2;
    M_Tag m_ltag;

    hFile2 = INVALID_HANDLE_VALUE;

    hFile = open_wavpack_file (fn, &wavhdr, &wvpkhdr, &m_ltag, string,
	(config_bits & IGNORE_WVC) ? NULL : &hFile2);

    if (hFile != INVALID_HANDLE_VALUE) {
	int original_bytes = wvpkhdr.total_samples * wavhdr.NumChannels * ((wvpkhdr.flags & BYTES_3) ? 3 : 2);
	int compressed_bytes = GetFileSize (hFile, NULL) - SetFilePointer (hFile, 0, NULL, FILE_CURRENT);
	int length_in_ms = (int)(wvpkhdr.total_samples * 1000.0 / wavhdr.SampleRate);
	double ratio;

	if (wvpkhdr.flags & WVC_FLAG)
	    compressed_bytes += GetFileSize (hFile2, NULL);
	    
	ratio = (double) original_bytes / (compressed_bytes + 1);

	sprintf (string, "Source:  %d-bit %s at %d Hz \n",
	    wavhdr.BitsPerSample, (wavhdr.NumChannels == 2 ? "stereo" : "mono"), wavhdr.SampleRate);

	if (wvpkhdr.bits) {
	    if (wvpkhdr.flags & NEW_HIGH_FLAG)
		sprintf (string + strlen (string), "Compression:  hybrid %s%s\n",
		    (wvpkhdr.flags & WVC_FLAG) ? "lossless" : "lossy",
		    (wvpkhdr.flags & EXTREME_DECORR) ? " (high)" : "");
	    else if (!(wvpkhdr.flags & (HIGH_FLAG | FAST_FLAG)))
		sprintf (string + strlen (string), "Compression:  %d-bit lossy \n", wvpkhdr.bits + 3);
	    else
		sprintf (string + strlen (string), "Compression:  %d-bit lossy (%s mode) \n", wvpkhdr.bits + 3,
		    (wvpkhdr.flags & HIGH_FLAG) ? "high" : "fast");
	}
	else {
	    if (!(wvpkhdr.flags & HIGH_FLAG))
		sprintf (string + strlen (string), "Compression:  lossless (fast mode)\n");
	    else if (wvpkhdr.flags & EXTREME_DECORR)
		sprintf (string + strlen (string), "Compression:  lossless (high mode)\n");
	    else
		sprintf (string + strlen (string), "Compression:  lossless\n");
	}

	sprintf (string + strlen (string), "Average bitrate:  %d kbps \n", (int) floor (compressed_bytes * 8.0 / length_in_ms + 0.5));
	sprintf (string + strlen (string), "Overall ratio:  %.2f to 1 \n", ratio);

	if (ValidTag (&m_ltag)) {
	    char value [64];

	    if (config_bits & (REPLAYGAIN_TRACK | REPLAYGAIN_ALBUM)) {
		int local_clipping;
		float local_gain;

		local_gain = calculate_gain (&m_ltag, &local_clipping);

		if (local_gain != 1.0)
		    sprintf (string + strlen (string), "Gain:  %+.2f dB %s\n",
			log10 (local_gain) * 20.0, local_clipping ? "(w/soft clipping)" : "");
	    }

	    if (GetTagItem (&m_ltag, "title", value, sizeof (value)))
		sprintf (string + strlen (string), "\nTitle:  %s", value);

	    if (GetTagItem (&m_ltag, "artist", value, sizeof (value)))
		sprintf (string + strlen (string), "\nArtist:  %s", value);

	    if (GetTagItem (&m_ltag, "album", value, sizeof (value)))
		sprintf (string + strlen (string), "\nAlbum:  %s", value);

	    if (GetTagItem (&m_ltag, "comment", value, sizeof (value)))
		sprintf (string + strlen (string), "\nComment:  %s", value);

	    if (GetTagItem (&m_ltag, "year", value, sizeof (value)))
		sprintf (string + strlen (string), "\nYear:  %s", value);

	    strcat (string, "\n");
	}

	MessageBox (hwnd, string, "WavPack File Info Box", MB_OK);
	close_wavpack_file (hFile, hFile2, &m_ltag);
    }
    else
	MessageBox (hwnd, string, "WavPack Player", MB_OK);

    return 0;
}

void getfileinfo (char *filename, char *title, int *length_in_ms)
{
    if (!filename || !*filename) {	// currently playing file

	if (length_in_ms)
	    *length_in_ms = getlength ();

	if (title) {
	    if (ValidTag (&m_tag) && GetTagItem (&m_tag, "title", NULL, 0)) {
		char art [64], ttl [64];

		GetTagItem (&m_tag, "title", ttl, sizeof (ttl));

		if (GetTagItem (&m_tag, "artist", art, sizeof (art)))
		    sprintf (title, "%s - %s", art, ttl);
		else
		    strcpy (title, ttl);
	    }
	    else {
		char *p = lastfn + strlen (lastfn);

		while (*p != '\\' && p >= lastfn)
		    p--;

		strcpy(title,++p);
	    }
	}
    }
    else {	// some other file
	WavpackHeader wvpkhdr;
	WaveHeader wavhdr;
	char error [128];
	M_Tag m_ltag;
	HANDLE hFile;

	if (length_in_ms)
	    *length_in_ms = -1000;

	if (title)
	    *title = 0;

	hFile = open_wavpack_file (filename, &wavhdr, &wvpkhdr, &m_ltag, error, NULL);

	if (hFile != INVALID_HANDLE_VALUE) {
	    if (length_in_ms)
		*length_in_ms = (int)(wvpkhdr.total_samples * 1000.0 / wavhdr.SampleRate);

	    if (title && ValidTag (&m_ltag) && GetTagItem (&m_ltag, "title", NULL, 0)) {
		char art [64], ttl [64];

		GetTagItem (&m_ltag, "title", ttl, sizeof (ttl));

		if (GetTagItem (&m_ltag, "artist", art, sizeof (art)))
		    sprintf (title, "%s - %s", art, ttl);
		else
		    strcpy (title, ttl);
	    }

	    close_wavpack_file (hFile, INVALID_HANDLE_VALUE, &m_ltag);
	}
	else
	    MessageBox (NULL, error, "WavPack Player", MB_OK);

	if (title && !*title) {
	    char *p = filename + strlen (filename);

	    while (*p != '\\' && p >= filename) p--;
	    strcpy(title,++p);
	}
    }
}

void eq_set (int on, char data [10], int preamp) 
{ 
	// most plug-ins can't even do an EQ anyhow.. I'm working on writing
	// a generic PCM EQ, but it looks like it'll be a little too CPU 
	// consuming to be useful :)
}

DWORD WINAPI __stdcall DecodeThread (void *b)
{
    int ave_bits = 0, used_bytes = 0, done = 0;

    while (!*((int *)b) ) {

	if (seek_needed != -1) {
	    int seek_position = seek_needed;
	    int bc = 0;

	    seek_needed = -1;

	    if (seek_position > getlength () - 1000 && getlength () > 1000)
		seek_position = getlength () - 1000; // don't seek to last second

	    mod.outMod->Flush (decode_pos_ms = seek_position);
	    wavpack_seek (&WvpkHdr, &decode_pos_sample, (int)(WavHdr.SampleRate / 1000.0 * seek_position));
	    decode_pos_ms = (int)(decode_pos_sample * 1000.0 / WavHdr.SampleRate);
	    
	    while (!*((int *)b) && decode_pos_ms < seek_position && seek_needed == -1) {

		int tsamples = unpack_wavpack_data (sample_buffer, &WvpkHdr, input_file, wvc_file, &decode_pos_sample, 576, &used_bytes) * NCH;
		int tbytes = tsamples * (BPS/8);

		if (!tsamples) {
		    done = 1;
		    break;
		}

		if (!(bc++ & 8) && mod.outMod->CanWrite() >= tbytes) {

		    if (BPS == 16) {
			short *atten_ptr = (short *) sample_buffer;
			int atten_cnt = tsamples;
		
			if (play_gain == 1.0)
			    while (atten_cnt--)
				*atten_ptr++ >>= 2;
			else
			    while (atten_cnt--)
				*atten_ptr++ *= play_gain * 0.25;
		    }
		    else {
			int *atten_ptr = (int *) sample_buffer;
			int atten_cnt = tsamples;
			int sc = tsamples;
			char *dst, *src;

			dst = sample_buffer + (sc * 4);
			src = sample_buffer + (sc * 3);

			while (sc--) {
			    *--dst = *--src;
			    *--dst = *--src;
			    *--dst = *--src;
			    *--dst = 0;
			}
		
			if (play_gain == 1.0)
			    while (atten_cnt--)
				*atten_ptr++ >>= 2;
			else
			    while (atten_cnt--)
				*atten_ptr++ *= play_gain * 0.25;
		    }

		    mod.outMod->Write (sample_buffer, tbytes);
		}

		decode_pos_ms = (int)(decode_pos_sample * 1000.0 / WavHdr.SampleRate);
	    }

	    mod.outMod->Flush (decode_pos_ms);
	    continue;
	}

	if (done) {
	    mod.outMod->CanWrite ();

	    if (!mod.outMod->IsPlaying ()) {
		PostMessage (mod.hMainWindow, WM_WA_MPEG_EOF, 0, 0);
		return 0;
	    }

	    Sleep (10);
	}
	else if (mod.outMod->CanWrite() >= ((576 * NCH * (BPS / 8)) << (mod.dsp_isactive () ? 1 : 0))) {
	    int tsamples = unpack_wavpack_data (sample_buffer, &WvpkHdr, input_file, wvc_file, &decode_pos_sample, 576, &used_bytes) * NCH;
	    int tbytes = tsamples * (BPS/8);

	    ave_bits = ave_bits - ((ave_bits + 4) >> 3) + used_bytes;

	    if (!tsamples)
		done = 1;
	    else {
		// If we are handling 32-bit data, we must convert from the
		// 24-bit native format used by WavPack.

		if (BPS == 32) {
		    int sc = tsamples;
		    char *dst, *src;

		    dst = sample_buffer + (sc * 4);
		    src = sample_buffer + (sc * 3);

		    while (sc--) {
			 *--dst = *--src;
			 *--dst = *--src;
			 *--dst = *--src;
			 *--dst = 0;
		    }

		    // check for and handle required gain for 24-bit data

		    if (play_gain != 1.0) {
			int *gain_ptr = (int *) sample_buffer;
			int gain_cnt = tsamples;

			if (play_gain > 1.0)
			    while (gain_cnt--) {
				float value = *gain_ptr * play_gain * (1/256.0);

				if (soft_clipping && value > 6291456.0)
				    *gain_ptr = 8388608.0 - (4398046511104.0 / (value - 4194304.0));
				else if (soft_clipping && value < -6291456.0)
				    *gain_ptr = -8388608.0 - (4398046511104.0 / (value + 4194304.0));
				else if (value > 8388607.0)
				    *gain_ptr = 8388607;
				else if (value < -8388608.0)
				    *gain_ptr = -8388608;
				else
				    *gain_ptr = floor (value + 0.5);

				*gain_ptr++ <<= 8;
			    }
			else
			    while (gain_cnt--) {
				float value = *gain_ptr * play_gain * (1/256.0);
				*gain_ptr++ = (int) floor (value + 0.5) << 8;
			    }
		    }
		}
		else if (play_gain != 1.0) {
		    short *gain_ptr = (short *) sample_buffer;
		    int gain_cnt = tsamples;

		    if (play_gain > 1.0)
			while (gain_cnt--) {
			    float value = *gain_ptr * play_gain;

			    if (soft_clipping && value > 24576.0)
				*gain_ptr++ = 32768.0 - (67108864.0 / (value - 16384.0));
			    else if (soft_clipping && value < -24576.0)
				*gain_ptr++ = -32768.0 - (67108864.0 / (value + 16384.0));
			    else if (value > 32767.0)
				*gain_ptr++ = 32767;
			    else if (value < -32768.0)
				*gain_ptr++ = -32768;
			    else
				*gain_ptr++ = floor (value + 0.5);
			}
		    else
			while (gain_cnt--) {
			    float value = *gain_ptr * play_gain;
			    *gain_ptr++ = floor (value + 0.5);
			}
		}

		mod.SAAddPCMData ((char *) sample_buffer, NCH, BPS, decode_pos_ms);
		mod.VSAAddPCMData ((char *) sample_buffer, NCH, BPS, decode_pos_ms);
		decode_pos_ms = (int)(decode_pos_sample * 1000.0 / WavHdr.SampleRate);

		if (mod.dsp_isactive())
		    tbytes = mod.dsp_dosamples ((short *) sample_buffer,
			tsamples / NCH, BPS, NCH, WavHdr.SampleRate) * (NCH * (BPS/8));

		mod.outMod->Write (sample_buffer, tbytes);
	    }
	}
	else {
	    mod.SetInfo (ave_bits * (WavHdr.SampleRate / 100) / 5760, -1, -1, 1);
	    Sleep(20);
	}
    }

    return 0;
}



In_Module mod = 
{
    IN_VER,
    "WavPack Player v1.8.1 "

#ifdef __alpha
    "(AXP)"
#else
    "(x86)"
#endif
    ,
    0,		// hMainWindow
    0,		// hDllInstance
    "WV\0WavPack File (*.WV)\0"
    ,
    1,		// is_seekable
    1,		// uses output
    config,
    about,
    init,
    quit,
    getfileinfo,
    infoDlg,
    isourfile,
    play,
    pause,
    unpause,
    ispaused,
    stop,
    getlength,
    getoutputtime,
    setoutputtime,
    setvolume,
    setpan,
    0,0,0,0,0,0,0,0,0,	// vis stuff
    0,0,		// dsp
    eq_set,
    NULL,		// setinfo
    0			// out_mod
};

__declspec (dllexport) In_Module * winampGetInModule2 ()
{
    return &mod;
}

// These routines provide the interface between the plugin code and the
// standard WavPack library.

// This array keeps track of up to 256 index points during playback for rapid
// seeking. The size of the unpack_data field is simply the actual largest
// size currently required (plus some margin). In a perfect world we would
// allocate this according to the amount actually required, or at least make
// a call to unpack_size() to make sure we have hardcoded enough. But, of
// course, ...

static struct index_point {
    int saved, sample_index;
    char unpack_data [768];
} index_points [256];

//////////////////////////////////////////////////////////////////////////////
// This function attempts to open the specified WavPack file. The caller    //
// must provide allocated wavefile, WavPack, and tag structures that will   //
// be filled in with all appropriate information (assuming the process does //
// not fail). If the function fails then an appropriate message for the     //
// user is stored at "error" and INVALID_HANDLE_VALUE is returned, other-   //
// wise the HANDLE for the file is returned (and the file is prepared for   //
// playback in case that is desired). If the "in2file" handle pointer is    //
// not NULL and the specified file could have a "correction" file connected //
// with it, an attempt is made to open the same name with a .wvc extension. //
// If this is successful, then that file's handle is stored at "in2file",   //
// otherwise INVALID_HANDLE_VALUE is stored.                                //
//////////////////////////////////////////////////////////////////////////////

static HANDLE open_wavpack_file (char *fn, WaveHeader *wavhdr, WavpackHeader *wvpkhdr,
    M_Tag *m_tag, char *error, HANDLE *in2file)
{
    long total_samples, total_header_bytes;
    RiffChunkHeader RiffChunkHeader;
    ChunkHeader ChunkHeader;
    HANDLE infile;
    int result;		    

    infile = CreateFile (fn, GENERIC_READ, FILE_SHARE_READ, NULL,
	OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

    if (infile == INVALID_HANDLE_VALUE) {
	strcpy (error, "Can't Open File!");
	return infile;
    }

    if (m_tag) {
	LoadTag (m_tag, infile);
	SetFilePointer (infile, 0, NULL, FILE_BEGIN);
    }

    if (ReadFile (infile, &RiffChunkHeader, sizeof (RiffChunkHeader), &result, NULL) && result == sizeof (RiffChunkHeader) &&
	!strncmp (RiffChunkHeader.ckID, "RIFF", 4) && !strncmp (RiffChunkHeader.formType, "WAVE", 4)) {

	    while (1) {

		if (!ReadFile (infile, &ChunkHeader, sizeof (ChunkHeader), &result, NULL) || result != sizeof (ChunkHeader)) {
		    strcpy (error, "Not A Valid WavPack File!");
		    CloseHandle (infile);
		    return INVALID_HANDLE_VALUE;
		}

		if (!strncmp (ChunkHeader.ckID, "fmt ", 4)) {

		    if (ChunkHeader.ckSize < sizeof (WaveHeader) ||
			!ReadFile (infile, wavhdr, sizeof (WaveHeader), &result, NULL) || result != sizeof (WaveHeader)) {
			    strcpy (error, "Not A Valid WavPack File!");
			    CloseHandle (infile);
			    return INVALID_HANDLE_VALUE;
		    }

		    if (ChunkHeader.ckSize > sizeof (WaveHeader))
			SetFilePointer (infile, (ChunkHeader.ckSize + 1 - sizeof (WaveHeader)) & ~1L, NULL, FILE_CURRENT);
		}
		else if (!strncmp (ChunkHeader.ckID, "data", 4)) {
		    total_samples = ChunkHeader.ckSize / wavhdr->NumChannels / 2;
		    break;
		}
		else
		    SetFilePointer (infile, (ChunkHeader.ckSize + 1) & ~1L, NULL, FILE_CURRENT);
	    }

	    total_header_bytes = SetFilePointer (infile, 0, NULL, FILE_CURRENT);
    }
    else {
	strcpy (error, "Can't Handle Raw WavPack Files!");
	CloseHandle (infile);
	return INVALID_HANDLE_VALUE;
    }

    if (!ReadFile (infile, wvpkhdr, sizeof (WavpackHeader), &result, NULL) || result != sizeof (WavpackHeader) ||
	strncmp (wvpkhdr->ckID, "wvpk", 4)) {
	    strcpy (error, "Not A Valid WavPack File!");
	    CloseHandle (infile);
	    return INVALID_HANDLE_VALUE;
    }

    if (wvpkhdr->version < 1 || wvpkhdr->version > 3) {
	strcpy (error, "Incompatible WavPack File!");
	CloseHandle (infile);
	return INVALID_HANDLE_VALUE;
    }

    if (wvpkhdr->version == 3) {

	if (wvpkhdr->flags & EXTREME_DECORR) {

	    if ((wvpkhdr->flags & NOT_STORED_FLAGS) ||
		((wvpkhdr->bits) &&
		(((wvpkhdr->flags & NEW_HIGH_FLAG) &&
		(wvpkhdr->flags & (FAST_FLAG | HIGH_FLAG))) ||
		(wvpkhdr->flags & CROSS_DECORR)))) {
		    strcpy (error, "Incompatible WavPack File!");
		    CloseHandle (infile);
		    return INVALID_HANDLE_VALUE;
	    }

	    if (wvpkhdr->flags & CANCEL_EXTREME)
		wvpkhdr->flags &= ~(EXTREME_DECORR | CANCEL_EXTREME);
	}
	else
	    wvpkhdr->flags &= ~CROSS_DECORR;
    }

    SetFilePointer (infile, total_header_bytes, NULL, FILE_BEGIN);

    switch (wvpkhdr->version) {

	case 1:
	    wvpkhdr->bits = 0;
	    SetFilePointer (infile, -2, NULL, FILE_CURRENT);

	case 2:
	    wvpkhdr->total_samples = total_samples;
	    wvpkhdr->flags = wavhdr->NumChannels == 1 ? MONO_FLAG : 0;
	    wvpkhdr->shift = 16 - wavhdr->BitsPerSample;
	    SetFilePointer (infile, 12, NULL, FILE_CURRENT);
	    break;

	case 3:
	    SetFilePointer (infile, sizeof (WavpackHeader), NULL, FILE_CURRENT);
	    break;
    }

    if (in2file && wvpkhdr->version == 3 && wvpkhdr->bits && (wvpkhdr->flags & NEW_HIGH_FLAG)) {
	char *in2fn = malloc (strlen (fn) + 10);

	strcpy (in2fn, fn);
	strcat (in2fn, "c");

	*in2file = CreateFile (in2fn, GENERIC_READ, FILE_SHARE_READ, NULL,
	    OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

	free (in2fn);
    }
    else if (in2file)
	*in2file = INVALID_HANDLE_VALUE;

    if (in2file && *in2file != INVALID_HANDLE_VALUE)
	wvpkhdr->flags |= WVC_FLAG;
    else
	wvpkhdr->flags &= ~WVC_FLAG;

    return infile;
}

//////////////////////////////////////////////////////////////////////////////
// Force any existing index points to be discarded. This should be done if  //
// a new file is to be played.                                              //
//////////////////////////////////////////////////////////////////////////////

static void kill_wavpack_index (void)
{
    memset (index_points, 0, sizeof (index_points));
}

// The WavPack libraries use this single context structure instead of having
// any static variables. Because this application is not threadsafe, we can
// simply have a single static copy of this for any playing file.

WavpackContext wpc;

//////////////////////////////////////////////////////////////////////////////
// Unpack the specified number of audio samples from the specified WavPack  //
// file(s) into the specified buffer. The "sample_index" value must be 0    //
// the first time this routine is called for a given file and is updated by //
// this routine according to the number of samples unpacked. This value     //
// should NOT be changed by the caller; use wavpack_seek() instead to index //
// within the file. The "bytes_used" value is not precise (and sometimes    //
// will not get written at all) but is just used for bitrate displays. The  //
// return value is the number of samples stored which may be from 16 to 48  //
// bits each (for 16-bit mono to 24-bit stereo).                            //
//////////////////////////////////////////////////////////////////////////////

static int unpack_wavpack_data (char *buffer, WavpackHeader *wphdr, HANDLE infile, HANDLE in2file,
    int *sample_index, int sample_count, int *bytes_used)
{
    int points_index = *sample_index / ((wphdr->total_samples >> 8) + 1);
    char *start_pos, *start_pos2;
    int result = 0;

    if (!wphdr->total_samples)
	return result;

    if (!*sample_index) {

	wpc.wphdr = wphdr;
	wpc.inbits.bufsiz = wpc.in2bits.bufsiz = 512 * 1024;

	if (bs_open_read (&wpc.inbits, infile))
	    return result;

	if ((wphdr->flags & WVC_FLAG) && bs_open_read (&wpc.in2bits, in2file))
	    return result;

	unpack_init (&wpc);
    }

    if (!index_points [points_index].saved) {
	index_points [points_index].sample_index = *sample_index;
	unpack_save (&wpc, index_points [points_index].unpack_data);
	index_points [points_index].saved = TRUE;
    }

    // In a perfect world we would not be poking our nose into these structures
    // to find out how much data is being used. Again, however...

    start_pos = wpc.inbits.ptr;
    start_pos2 = wpc.in2bits.ptr;

    if (sample_count > wphdr->total_samples - *sample_index)
	sample_count = wphdr->total_samples - *sample_index;

    result = unpack_samples (&wpc, buffer, sample_count);

    if (bytes_used) {
	if (wphdr->flags & WVC_FLAG) {
	    if (wpc.inbits.ptr > start_pos && wpc.in2bits.ptr > start_pos2)
		*bytes_used = (wpc.inbits.ptr - start_pos) + (wpc.in2bits.ptr - start_pos2);
	}
	else if (wpc.inbits.ptr > start_pos)
	    *bytes_used = wpc.inbits.ptr - start_pos;
    }
	
    if ((*sample_index += result) == wphdr->total_samples) {
#if 0
	if (crc != ((wphdr->flags & WVC_FLAG) ? wphdr->crc2 : wphdr->crc))
	    MessageBox (NULL, "CRC Error!", "WavPack Player", MB_OK);
#if 0
	else
	    MessageBox (NULL, "CRC Correct", "WavPack Player", MB_OK);
#endif
#endif
    }

    return result;
}

//////////////////////////////////////////////////////////////////////////////
// Attempt to seek to the desired location. If the specified location has   //
// not been stored in the index_points array, then this routine will fail   //
// and return FALSE. In this case the caller will simply have to manually   //
// seek to the desired position by unpacking all intervening data. If the   //
// routine does not fail then the "sample_index" value is set to the actual //
// position exactly seeked to (which will be <= to the desired position)    //
// and then it will be possible to unpack from that position.               //
//////////////////////////////////////////////////////////////////////////////

static int wavpack_seek (WavpackHeader *wphdr, int *sample_index, int desired_index)
{
    int points_index = desired_index / ((wphdr->total_samples >> 8) + 1);

    while (points_index)
	if (index_points [points_index].saved &&
	    index_points [points_index].sample_index <= desired_index)
	        break;
	else
	    points_index--;

    if (!index_points [points_index].saved)
	return FALSE;

    if (index_points [points_index].sample_index <= *sample_index &&
	*sample_index <= desired_index)
	    return FALSE;

    *sample_index = index_points [points_index].sample_index;
    unpack_restore (&wpc, index_points [points_index].unpack_data, TRUE);
    return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
// This function uses the ReplayGain mode selected by the user and the info //
// stored in the specified tag to determine the gain value used to play the //
// file and whether "soft clipping" is required. Note that the gain is in   //
// voltage scaling (not dB), so a value of 1.0 (not 0.0) is unity gain.     //
//////////////////////////////////////////////////////////////////////////////

static float calculate_gain (M_Tag *m_tag, int *pSoftClip)
{
    *pSoftClip = FALSE;

    if (config_bits & (REPLAYGAIN_TRACK | REPLAYGAIN_ALBUM)) {
	float gain_value = 0.0, peak_value = 1.0;
	char value [32];

	if ((config_bits & REPLAYGAIN_ALBUM) && GetTagItem (m_tag, "replaygain_album_gain", value, sizeof (value))) {
	    gain_value = atof (value);

	    if (GetTagItem (m_tag, "replaygain_album_peak", value, sizeof (value)))
		peak_value = atof (value);
	}
	else if (GetTagItem (m_tag, "replaygain_track_gain", value, sizeof (value))) {
	    gain_value = atof (value);

	    if (GetTagItem (m_tag, "replaygain_track_peak", value, sizeof (value)))
		peak_value = atof (value);
	}
	else
	    return 1.0;

	// convert gain from dB to voltage (with +/- 20 dB limit)

	if (gain_value > 20.0)
	    gain_value = 10.0;
	else if (gain_value < -20.0)
	    gain_value = 0.1;
	else
	    gain_value = pow (10.0, gain_value / 20.0);

	if (peak_value * gain_value > 1.0) {
	    if (config_bits & PREVENT_CLIPPING)
		gain_value = 1.0 / peak_value;
	    else if (config_bits & SOFTEN_CLIPPING)
		*pSoftClip = TRUE;
	}

	return gain_value;
    }
    else
	return 1.0;
}

//////////////////////////////////////////////////////////////////////////////
// Close the specified WavPack file(s) and release any resources used. It's //
// a little kludgy, but we compare the file handles to make sure that we    //
// don't free the bitstream buffers that we might currently be using to     //
// play another file. Okay, it's really kludgy.                             //
//////////////////////////////////////////////////////////////////////////////

static void close_wavpack_file (HANDLE infile, HANDLE in2file, M_Tag *m_tag)
{
    if (infile != INVALID_HANDLE_VALUE) {
	
	if (wpc.inbits.file == infile)
	    bs_close_read (&wpc.inbits);

	CloseHandle (infile);
    }

    if (in2file != INVALID_HANDLE_VALUE) {
	
	if (wpc.in2bits.file == in2file)
	    bs_close_read (&wpc.in2bits);

	CloseHandle (in2file);
    }

    FreeTag (m_tag);
}
