////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2002 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// bits.c

// This module provides utilities to support the BitStream structure which is
// used to read and write all WavPack audio data streams. Each function is
// provided in both a single-threaded and multi-threaded version, the latter
// providing the option of overlapped file I/O with systems that do not provide
// this as intrinsic (i.e. Win95/98). For some reason this actually hurts
// performance on NT based systems and should not be used. This does not apply
// for bs_restore() which is not usable with overlapped operation.

#include "wavpack.h"

#include <stdlib.h>
#include <process.h>
#include <string.h>

#ifndef __WIN32__
#include <io.h>
#endif

#ifdef UNPACK

#ifdef __MT__

static int read_pending;

//////////////////////////////////////////////////////////////////////////////
// Open the specified BitStream and associate with the specified file. If   //
// the "overlap" argument is TRUE, then the stream is opened to perform     //
// overlapping reads automatically. The "bufsiz" field of the structure     //
// must be preset with the desired buffer size and the file's read pointer  //
// must be set to where the desired bit data is located. Note that the      //
// actual file read does not occur until the first bit is requested. A      //
// return value of TRUE indicates an error in allocating buffer space.      //
//////////////////////////////////////////////////////////////////////////////

int bs_open_read (Bitstream *bs, HANDLE file, int overlap)
{
    if (!bs->buf)
	bs->buf = (uchar *) malloc (bs->bufsiz);

    if (overlap) {
	if (!bs->altbuf)
	    bs->altbuf = (uchar *) malloc (bs->bufsiz);

	bs->error = (bs->buf && bs->altbuf) ? 0 : 1;
    }
    else {
	if (bs->altbuf)
	    free (bs->altbuf);

	bs->altbuf = NULL;
	bs->error = (bs->buf) ? 0 : 1;
    }

    bs->end = bs->buf + bs->bufsiz;
    bs->ptr = bs->end - 1;
    bs->file = file;
    bs->sr = bs->bc = 0;
    read_pending = 0;
    return bs->error;
}

//////////////////////////////////////////////////////////////////////////////
// This function is normally only called from the getbit() and getbits()    //
// macros when the BitStream has been exhausted adn more data is required.  //
//////////////////////////////////////////////////////////////////////////////

void bs_read (Bitstream *bs)
{
    if (bs->altbuf) {
	uint bytes_read;
	uchar *tmp;

	if (!read_pending) {
	    start_read (bs->file, bs->buf, bs->bufsiz);
	    bytes_read = finish_read ();
	    bs->end = bs->buf + bytes_read;
	    start_read (bs->file, bs->altbuf, bs->bufsiz);
	    read_pending = 1;
	}
	else {
	    tmp = bs->buf;
	    bs->buf = bs->altbuf;
	    bs->altbuf = tmp;
	    bytes_read = finish_read ();
	    bs->end = bs->buf + bytes_read;
	    start_read (bs->file, bs->altbuf, bs->bufsiz);
	}
    }
    else {
	ulong bytes_read;

	DoReadFile (bs->file, bs->buf, bs->bufsiz, &bytes_read, NULL);
	bs->end = bs->buf + bytes_read;
    }

    if (bs->end == bs->buf) {
	memset (bs->buf, -1, bs->bufsiz);
	bs->end += bs->bufsiz;
    }

    bs->ptr = bs->buf;
}

//////////////////////////////////////////////////////////////////////////////
// This function is called to release any resources used by the BitStream   //
// and position the file pointer to the first byte past the read bits.      //
//////////////////////////////////////////////////////////////////////////////

void bs_close_read (Bitstream *bs)
{
    if (bs->altbuf && read_pending)
	finish_read ();

    if (bs->bc < 8)
	bs->ptr++;

    if (bs->ptr < bs->end &&
#ifdef __WIN32__
	SetFilePointer (bs->file, bs->ptr - bs->end, NULL, FILE_CURRENT) == INVALID_FILE_SIZE)
#else
	lseek (bs->file, bs->ptr - bs->end, SEEK_CUR) == INVALID_FILE_SIZE)
#endif
	    bs->error = 1;

    if (bs->buf) {
	free (bs->buf);
	bs->buf = NULL;
    }

    if (bs->altbuf) {
	free (bs->altbuf);
	bs->altbuf = NULL;
    }
}

#else

//////////////////////////////////////////////////////////////////////////////
// Open the specified BitStream and associate with the specified file. The  //
// "bufsiz" field of the structure must be preset with the desired buffer   //
// size and the file's read pointer must be set to where the desired bit    //
// data is located.  A return value of TRUE indicates an error in           //
// allocating buffer space.                                                 //
//////////////////////////////////////////////////////////////////////////////

int bs_open_read (Bitstream *bs, HANDLE file)
{
#ifdef __WIN32__
    bs->fpos = SetFilePointer ((bs->file = file), 0, NULL, FILE_CURRENT);
#else
    bs->fpos = tell (bs->file = file);
#endif

    if (!bs->buf)
	bs->buf = (uchar *) malloc (bs->bufsiz);

    bs->end = bs->buf + bs->bufsiz;
    bs->ptr = bs->end - 1;
    bs->file = file;
    bs->sr = bs->bc = 0;
    bs->error = bs->buf ? 0 : 1;
    return bs->error;
}

//////////////////////////////////////////////////////////////////////////////
// This function is normally only called from the getbit() and getbits()    //
// macros when the BitStream has been exhausted and more data is required.  //
//////////////////////////////////////////////////////////////////////////////

void bs_read (Bitstream *bs)
{
    ulong bytes_read;

    DoReadFile (bs->file, bs->buf, bs->bufsiz, &bytes_read, NULL);
    bs->end = bs->buf + bytes_read;
    bs->fpos += bytes_read;

    if (bs->end == bs->buf) {
	memset (bs->buf, -1, bs->bufsiz);
	bs->end += bs->bufsiz;
    }

    bs->ptr = bs->buf;
}

#ifdef SEEKING

//////////////////////////////////////////////////////////////////////////////
// This function is called after a call to unpack_restore() has restored    //
// the BitStream structure to a previous state and causes any required data //
// to be read from the file. This function is NOT supported for overlapped  //
// operation.                                                               //
//////////////////////////////////////////////////////////////////////////////

void bs_restore (Bitstream *bs)
{
    ulong bytes_to_read = bs->end - bs->ptr - 1, bytes_read;

#ifdef __WIN32__
    SetFilePointer (bs->file, bs->fpos - bytes_to_read, NULL, FILE_BEGIN);
#else
    lseek (bs->file, bs->fpos - bytes_to_read, SEEK_SET);
#endif

    if (bytes_to_read > 0) {

	DoReadFile (bs->file, bs->ptr + 1, bytes_to_read, &bytes_read, NULL);

	if (bytes_to_read != bytes_read)
	    bs->end = bs->ptr + 1 + bytes_read;
    }
}

#endif

//////////////////////////////////////////////////////////////////////////////
// This function is called to release any resources used by the BitStream   //
// and position the file pointer to the first byte past the read bits.      //
//////////////////////////////////////////////////////////////////////////////

void bs_close_read (Bitstream *bs)
{
    if (bs->bc < 8)
	bs->ptr++;

    if (bs->ptr < bs->end &&
#ifdef __WIN32__
	SetFilePointer (bs->file, bs->ptr - bs->end, NULL, FILE_CURRENT) == INVALID_FILE_SIZE)
#else
	lseek (bs->file, bs->ptr - bs->end, SEEK_CUR) == -1)
#endif
	    bs->error = 1;

    if (bs->buf) {
	free (bs->buf);
	bs->buf = NULL;
    }
}

#endif

#endif

#ifdef PACK

#ifdef NO_BS_WRITE
ulong bytes_not_written;
#endif

#ifdef __MT__

//////////////////////////////////////////////////////////////////////////////
// Open the specified BitStream and associate with the specified file. If   //
// the "overlap" argument is TRUE, then the stream is opened to perform     //
// overlapping writes automatically. The "bufsiz" field of the structure    //
// must be preset with the desired buffer size and the file's write pointer //
// must be set to where the bit data should be written. A return value of   //
// TRUE indicates an error in allocating buffer space.                      //
//////////////////////////////////////////////////////////////////////////////

int bs_open_write (Bitstream *bs, HANDLE file, int overlap)
{
    if (!bs->buf)
	bs->buf = (uchar *) malloc (bs->bufsiz);

    bs->ptr = bs->buf;

#ifdef NO_BS_WRITE
    overlap = FALSE;
#endif

    if (overlap) {
	if (!bs->altbuf)
	    bs->altbuf = (uchar *) malloc (bs->bufsiz);

	bs->error = (bs->buf && bs->altbuf) ? 0 : 1;
    }
    else {
	if (bs->altbuf)
	    free (bs->altbuf);

	bs->altbuf = NULL;
	bs->error = (bs->buf) ? 0 : 1;
    }

    bs->end = bs->buf + bs->bufsiz;
    bs->file = file;
    bs->sr = bs->bc = 0;
    return bs->error;
}

//////////////////////////////////////////////////////////////////////////////
// This function is normally only called from the putbit() and putbits()    //
// macros when the buffer is full and must be written to the actual file.   //
// If there is an error writing the data (disk full, probably) then the     //
// "error" field of the structure is set to TRUE.                           //
//////////////////////////////////////////////////////////////////////////////

void bs_write (Bitstream *bs)
{
    if (bs->altbuf) {
	static uint bytes_to_write;
	char *btmp;

	if (bytes_to_write)
	    if (bytes_to_write != finish_write ())
		bs->error = 1;

	bytes_to_write = bs->ptr - bs->buf;

	if (bytes_to_write)
	    start_write (bs->file, bs->buf, bytes_to_write);

	btmp = bs->buf;
	bs->buf = bs->altbuf;
	bs->altbuf = btmp;
    }
    else {
	ulong bytes_written;

#ifdef NO_BS_WRITE
	bytes_not_written += bs->ptr - bs->buf;
#else
	if (!DoWriteFile (bs->file, bs->buf, bs->ptr - bs->buf, &bytes_written, NULL) ||
	    bytes_written != bs->ptr - bs->buf)
		bs->error = 1;
#endif
    }

    bs->ptr = bs->buf;
    bs->end = bs->buf + bs->bufsiz;
}

//////////////////////////////////////////////////////////////////////////////
// This function forces a flushing write of the specified BitStream,        //
// positions the write pointer past the bit data written, and releases any  //
// resources used by the BitStream. The "error" field should be checked     //
// after this to make sure all data was properly written.                   //
//////////////////////////////////////////////////////////////////////////////

void bs_close_write (Bitstream *bs)
{
    while (bs->bc) putbit_1 (bs);
    bs_write (bs);

    if (bs->altbuf) {
	bs_write (bs);
	free (bs->altbuf);
	bs->altbuf = NULL;
    }

    if (bs->buf) {
	free (bs->buf);
	bs->buf = NULL;
    }
}

#else

//////////////////////////////////////////////////////////////////////////////
// Open the specified BitStream and associate with the specified file. The  //
// "bufsiz" field of the structure must be preset with the desired buffer   //
// size and the file's write pointer must be set to where the bit data      //
// should be written. A return value of TRUE indicates an error in          //
// allocating buffer space.                                                 //
//////////////////////////////////////////////////////////////////////////////

int bs_open_write (Bitstream *bs, HANDLE file)
{
    if (!bs->buf)
	bs->buf = (uchar *) malloc (bs->bufsiz);

    bs->ptr = bs->buf;
    bs->end = bs->buf + bs->bufsiz;
    bs->file = file;
    bs->sr = bs->bc = 0;
    bs->error = bs->buf ? 0 : 1;
    return bs->error;
}

//////////////////////////////////////////////////////////////////////////////
// This function is normally only called from the putbit() and putbits()    //
// macros when the buffer is full and must be written to the actual file.   //
// If there is an error writing the data (disk full, probably) then the     //
// "error" field of the structure is set to TRUE.                           //
//////////////////////////////////////////////////////////////////////////////

void bs_write (Bitstream *bs)
{
    ulong bytes_written;

#ifdef NO_BS_WRITE
    bytes_not_written += bs->ptr - bs->buf;
#else
    if (!DoWriteFile (bs->file, bs->buf, bs->ptr - bs->buf, &bytes_written, NULL) ||
	bytes_written != bs->ptr - bs->buf)
	    bs->error = 1;
#endif

    bs->ptr = bs->buf;
}

//////////////////////////////////////////////////////////////////////////////
// This function forces a flushing write of the specified BitStream,        //
// positions the write pointer past the bit data written, and releases any  //
// resources used by the BitStream. The "error" field should be checked     //
// after this to make sure all data was properly written.                   //
//////////////////////////////////////////////////////////////////////////////

void bs_close_write (Bitstream *bs)
{
    while (bs->bc) putbit_1 (bs);
    bs_write (bs);

    if (bs->buf) {
	free (bs->buf);
	bs->buf = NULL;
    }
}

#endif

//////////////////////////////////////////////////////////////////////////////
// This function returns the error status of the specified Bitstream. A     //
// value of TRUE would occur if the specified buffer size could not be      //
// allocated or if a write operation was not successful (i.e. disk full).   //
//////////////////////////////////////////////////////////////////////////////

int bs_error (Bitstream *bs)
{
    return bs->error;
}

#endif

//////////////////////////////////////////////////////////////////////////////
// This set of functions provides a little compatibility between standard   //
// ANSI file I/O routines and their Win32 equivalents. There is also an     //
// enhancement to DoReadFile() and DoWriteFile() in that they now will not  //
// return until they have processed the entire specified error, which I     //
// found that they will sometimes do when processing pipes.                 //
//////////////////////////////////////////////////////////////////////////////

#ifdef __WIN32__

int DoReadFile (HANDLE hFile, void *lpBuffer, ulong nNumberOfBytesToRead,
    ulong *lpNumberOfBytesRead, ...)
{
    ulong bcount;
    int retres;

    *lpNumberOfBytesRead = 0;

    while (nNumberOfBytesToRead) {
	retres = ReadFile (hFile, (uchar *) lpBuffer + *lpNumberOfBytesRead, nNumberOfBytesToRead, &bcount, NULL);

	if (retres && bcount) {
	    *lpNumberOfBytesRead += bcount;
	    nNumberOfBytesToRead -= bcount;
	}
	else
	    break;
    }

    return retres;
}

int DoWriteFile (HANDLE hFile, void *lpBuffer, ulong nNumberOfBytesToWrite,
    ulong *lpNumberOfBytesWritten, ...)
{
    ulong bcount;
    int retres;

    *lpNumberOfBytesWritten = 0;

    while (nNumberOfBytesToWrite) {
	retres = WriteFile (hFile, (uchar *) lpBuffer + *lpNumberOfBytesWritten, nNumberOfBytesToWrite, &bcount, NULL);

	if (retres && bcount) {
	    *lpNumberOfBytesWritten += bcount;
	    nNumberOfBytesToWrite -= bcount;
	}
	else
	    break;
    }

    return retres;
}

int DoCloseHandle (HANDLE hFile)
{
    return CloseHandle (hFile);
}

int DoTruncateFile (HANDLE hFile)
{
    return SetFilePointer (hFile, 0, NULL, FILE_BEGIN) != INVALID_FILE_SIZE &&
	SetEndOfFile (hFile);
}

int DoDeleteFile (char *filename)
{
    return DeleteFile (filename);
}

#else

int DoReadFile (HANDLE hFile, void *lpBuffer, ulong nNumberOfBytesToRead,
    ulong *lpNumberOfBytesRead, ...)
{
    ulong bcount;

    *lpNumberOfBytesRead = 0;

    while (nNumberOfBytesToRead) {
	bcount = read (hFile, (uchar *) lpBuffer + *lpNumberOfBytesRead, nNumberOfBytesToRead);

	if (bcount && bcount != (ulong) -1) {
	    *lpNumberOfBytesRead += bcount;
	    nNumberOfBytesToRead -= bcount;
	}
	else
	    break;
    }

    return bcount != (ulong) -1;
}

int DoWriteFile (HANDLE hFile, void *lpBuffer, ulong nNumberOfBytesToWrite,
    ulong *lpNumberOfBytesWritten, ...)
{
    ulong bcount;

    *lpNumberOfBytesWritten = 0;

    while (nNumberOfBytesToWrite) {
	bcount = write (hFile, (uchar *) lpBuffer + *lpNumberOfBytesWritten, nNumberOfBytesToWrite);

	if (bcount && bcount != (ulong) -1) {
	    *lpNumberOfBytesWritten += bcount;
	    nNumberOfBytesToWrite -= bcount;
	}
	else
	    break;
    }

    return bcount != (ulong) -1;
}

int DoCloseHandle (HANDLE hFile)
{
    return !close (hFile);
}

int DoTruncateFile (HANDLE hFile)
{
    return !chsize (hFile, 0);
}

int DoDeleteFile (char *filename)
{
    return !remove (filename);
}

#endif

//////////////////////////////////////////////////////////////////////////////
// This set of functions provides support for overlapped read and write     //
// operations in Win95/98 where it is not intrinsic. This results in a      //
// significant performance enhancement on those platforms, but because it   //
// does not seem to offer any improvement in NT or Win2K it is not used on  //
// those platforms and will probably be obsoleted one of these days. Also   //
// note that because of all these static variables, these functions can     //
// only be used on one file and one I/O operation at a time.                //
//////////////////////////////////////////////////////////////////////////////

#ifdef __MT__

static HANDLE read_file, read_start_event, read_done_event;
static int read_init_flag, reading_flag;
static ulong bytes_to_read, bytes_read;
static void *read_buffer;

#ifdef __BORLANDC__
static void __cdecl read_thread ()
#else
static void __cdecl read_thread (void *p)
#endif
{
    while (1) {
	WaitForSingleObject (read_start_event, INFINITE);
	DoReadFile (read_file, read_buffer, bytes_to_read, &bytes_read);
	SetEvent (read_done_event);
    }
}

void start_read (HANDLE file, void *buff, ulong len)
{
    if (reading_flag) {
	reading_flag = 0;
	return;
    }

    if (!read_init_flag) {
	read_start_event = CreateEvent (NULL, FALSE, FALSE, NULL);
	read_done_event = CreateEvent (NULL, FALSE, FALSE, NULL);
	_beginthread (read_thread, 0, NULL);
	read_init_flag = 1;
    }

    read_file = file;
    read_buffer = buff;
    bytes_to_read = len;
    SetEvent (read_start_event);
    reading_flag = 1;
}

ulong finish_read (void)
{
    if (!read_init_flag || !reading_flag)
	return 0;

    WaitForSingleObject (read_done_event, INFINITE);
    reading_flag = 0;
    return bytes_read;
}

static int write_init_flag, writing_flag;
static HANDLE write_file, write_start_event, write_done_event;
static ulong bytes_to_write, bytes_written;
static void *write_buffer;

#ifdef __BORLANDC__
static void __cdecl write_thread ()
#else
static void __cdecl write_thread (void *p)
#endif
{
    while (1) {
	WaitForSingleObject (write_start_event, INFINITE);
	DoWriteFile (write_file, write_buffer, bytes_to_write, &bytes_written, NULL);
	SetEvent (write_done_event);
    }
}

void start_write (HANDLE file, void *buff, ulong len)
{
    if (writing_flag) {
	writing_flag = 0;
	return;
    }

    if (!write_init_flag) {
	write_start_event = CreateEvent (NULL, FALSE, FALSE, NULL);
	write_done_event = CreateEvent (NULL, FALSE, FALSE, NULL);
	_beginthread (write_thread, 0, NULL);
	write_init_flag = 1;
    }

    write_file = file;
    write_buffer = buff;
    bytes_to_write = len;
    SetEvent (write_start_event);
    writing_flag = 1;
}

ulong finish_write (void)
{
    if (!write_init_flag || !writing_flag)
	return 0;

    WaitForSingleObject (write_done_event, INFINITE);
    writing_flag = 0;
    return bytes_written;
}

#endif
