////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2003 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// cool_wv.c

// This is the main module for the WavPack file filter for CoolEdit

// Version 1.0 - May 31, 2003
// Version 1.1 - June 5, 2003  (fixed bug in 20-bit headers)

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "filters.h" 
#include "wavpack.h"
#include "resource.h"


// These masks reference the bits in the 32-bit configuration value kept by
// CoolEdit which is essentially used to hold the "flags" and "bits" fields
// of the WavpackHeader:

#define CONFIG_FLAGS 0xffff		// these bits come from "flags" field
#define CONFIG_BITS 0x7fff0000		// these bits come from "bits" field
					//  and indicate hybrid files
#define CONFIG_UPDATED 0x80000000	// configuration is not "raw" from file

// These two bits are placed into currently unstored positions in the "flags"
// field on the WavpackHeader (but should not be written to WavPack header!)

#define FLOAT_NOISESHAPE 8		// noise-shaping when converting floats
#define FLOAT_DITHER 0x200		// dither added when converting floats

// In addition, a new combination has been given meaning to the BYTES_3 and
// OVER_20 bits, and this new combination should not be written either:

// BYTES_3  OVER_20  MEANING
// ------------------------------------------------------------------------
//    0        0     16-bit data written to 16-bit file
//    1        0     float data written to 20-bit file
//    1        1     float data written to 24-bit file
//    0        1     float data written to 16-bit file (don't store!)


#define CHUNKSIZE 16384			// let CoolEdit decide chunk size


// These two structures hold all information needed during reading or writing
// of WavPack files:

typedef struct {
    HANDLE infile, in2file;
    RiffChunkHeader listhdr;
    ulong samples_remaining, special_bytes;
    WavpackHeader wphdr;
    WavpackContext wpc;
    WaveHeader wavhdr;
} INPUT;

typedef struct {
    HANDLE outfile, out2file;
    ulong wavpack_header_pos, special_bytes;
    RiffChunkHeader riffhdr, listhdr;
    WavpackHeader wphdr;
    WavpackContext wpc;
    WaveHeader wavhdr;
    float error [2];
    long random;
} OUTPUT;


//////////////////////////////////////////////////////////////////////////////
// Let CoolEdit know who we are and what we can do.                         //
//////////////////////////////////////////////////////////////////////////////

short PASCAL QueryCoolFilter (COOLQUERY *lpcq)
{   
    strcpy (lpcq->szName, "WavPack");
    strcpy (lpcq->szCopyright, "WavPack Hybrid Audio Compression");
    strcpy (lpcq->szExt, "WV");

    lpcq->lChunkSize = CHUNKSIZE;

    lpcq->dwFlags = QF_CANLOAD | QF_CANSAVE | QF_RATEADJUSTABLE | QF_CANDO32BITFLOATS |
	QF_HASOPTIONSBOX | QF_READSPECIALLAST | QF_WRITESPECIALLAST;

    lpcq->Mono8 = lpcq->Stereo8 = 0;
    lpcq->Mono12 = lpcq->Stereo12 = 0;
    lpcq->Mono16 = lpcq->Stereo16 = 0x3ff;
    lpcq->Mono24 = lpcq->Stereo24 = 0;
    lpcq->Mono32 = lpcq->Stereo32 = 0x3ff;

    lpcq->Quad32 = 0;
    lpcq->Quad16 = 0;
    lpcq->Quad8 = 0;

    return C_VALIDLIBRARY;
}


//////////////////////////////////////////////////////////////////////////////
// Since WavPack extensions are unique checking that is good enough for now //
//////////////////////////////////////////////////////////////////////////////

BOOL PASCAL FilterUnderstandsFormat (LPSTR lpszFilename)
{
    if (!lpszFilename || !*lpszFilename)
	return 0;

    return strstr (lpszFilename, ".WV") || strstr (lpszFilename, ".wv");
}


//////////////////////////////////////////////////////////////////////////////
// We can handle everything except 8-bit data or more than 2 channels.      //
//////////////////////////////////////////////////////////////////////////////

void PASCAL GetSuggestedSampleType (long *lplSamprate, WORD *lpwBitsPerSample, WORD *lpwChannels)
{
    *lplSamprate = 0;

    if (*lpwChannels > 2)
	*lpwChannels = 2;

    *lpwBitsPerSample = (*lpwBitsPerSample > 16) ? 32 : 16;
}


//////////////////////////////////////////////////////////////////////////////
// Open file for output. Because normal WavPack files contain verbatum RIFF //
// headers, we must create them here from scratch.                          //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterGetOptions (HWND, HINSTANCE, long, WORD, WORD, DWORD);

HANDLE PASCAL OpenFilterOutput (LPSTR lpszFilename, long lSamprate,
    WORD wBitsPerSample, WORD wChannels, long lSize, long *lplChunkSize, DWORD dwOptions)
{
    ChunkHeader FormatChunkHeader, DataChunkHeader;
    ulong bcount;
    OUTPUT *out;

    // use FilterGetOptions() (without a dialog) to process the (possibly) raw
    // config word into something we can use directly.

    dwOptions = FilterGetOptions (NULL, NULL, lSamprate, wChannels, wBitsPerSample, dwOptions);

    // if starting with float data, we must convert it to 16, 20, or 24-bit,
    // otherwise 4 config bits are not used so clear them now

    if (wBitsPerSample == 32) {
	if (dwOptions & BYTES_3) {
	    wBitsPerSample = (dwOptions & OVER_20) ? 24 : 20;
	    lSize = lSize / 4 * 3;
	}
	else {
	    wBitsPerSample = 16;
	    lSize /= 2;
	}
    }
    else
	dwOptions &= ~(BYTES_3 | OVER_20 | FLOAT_NOISESHAPE | FLOAT_DITHER);

    // if not hybrid, clear WVC_FLAG now

    if (!(dwOptions & CONFIG_BITS))
	dwOptions &= ~WVC_FLAG;

    *lplChunkSize = CHUNKSIZE;

    if ((out = malloc (sizeof (OUTPUT))) == NULL)
	return 0;

    CLEAR (*out);
    out->wpc.wphdr = &out->wphdr;

    // create all the various headers

    strncpy (out->riffhdr.ckID, "RIFF", sizeof (out->riffhdr.ckID));
    out->riffhdr.ckSize = lSize + sizeof (ChunkHeader) * 2 + sizeof (WaveHeader) + 4;
    strncpy (out->riffhdr.formType, "WAVE", sizeof (out->riffhdr.formType));

    strncpy (FormatChunkHeader.ckID, "fmt ", sizeof (FormatChunkHeader.ckID));
    FormatChunkHeader.ckSize = sizeof (WaveHeader);

    out->wavhdr.FormatTag = 1;
    out->wavhdr.NumChannels = wChannels;
    out->wavhdr.SampleRate = lSamprate;
    out->wavhdr.BitsPerSecond = lSamprate * wChannels * (wBitsPerSample > 16 ? 3 : 2);
    out->wavhdr.BlockAlign = (wBitsPerSample > 16 ? 3 : 2) * wChannels;
    out->wavhdr.BitsPerSample = wBitsPerSample;

    strncpy (DataChunkHeader.ckID, "data", sizeof (DataChunkHeader.ckID));
    DataChunkHeader.ckSize = lSize; 

    strncpy (out->wphdr.ckID, "wvpk", sizeof (out->wphdr.ckID));
    strcpy (out->wphdr.extension, "wav");
    out->wphdr.ckSize = sizeof (WavpackHeader) - 8;
    out->wphdr.version = 3;
    out->wphdr.flags = dwOptions & CONFIG_FLAGS;
    out->wphdr.bits = (dwOptions & CONFIG_BITS) >> 16;

    if (wBitsPerSample == 16)
	out->wphdr.shift = 0;
    else if (wBitsPerSample == 24 && !out->wphdr.bits)
	out->wphdr.shift = 4;
    else
	out->wphdr.shift = 24 - wBitsPerSample;

    // open output file for writing

    if ((out->outfile = CreateFile (lpszFilename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
	CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
//	    error_line ("can't create file %s!", outfilename);
	    free (out);
	    return 0;
    }

    // write the RIFF chunks up to just before the data starts

    if (!DoWriteFile (out->outfile, &out->riffhdr, sizeof (out->riffhdr), &bcount, NULL) || bcount != sizeof (out->riffhdr) ||
	!DoWriteFile (out->outfile, &FormatChunkHeader, sizeof (FormatChunkHeader), &bcount, NULL) || bcount != sizeof (FormatChunkHeader) ||
	!DoWriteFile (out->outfile, &out->wavhdr, sizeof (WaveHeader), &bcount, NULL) || bcount != sizeof (WaveHeader) ||
	!DoWriteFile (out->outfile, &DataChunkHeader, sizeof (DataChunkHeader), &bcount, NULL) || bcount != sizeof (DataChunkHeader)) {
	    DoCloseHandle (out->outfile);
	    DoDeleteFile (lpszFilename);
	    free (out);
	    return 0;
    }

    // remember where the WavpackHeader goes and write it

    out->wavpack_header_pos = SetFilePointer (out->outfile, 0, NULL, FILE_CURRENT);

    if (!DoWriteFile (out->outfile, &out->wphdr, sizeof (WavpackHeader), &bcount, NULL) || bcount != sizeof (WavpackHeader)) {
	DoCloseHandle (out->outfile);
	DoDeleteFile (lpszFilename);
	free (out);
	return 0;
    }

    // if we are creating a "correction" file also, get that ready now

    if (out->wphdr.flags & WVC_FLAG) {
	char *out2filename = malloc (strlen (lpszFilename) + 10);

	strcpy (out2filename, lpszFilename);

	strcat (out2filename, (islower (*(out2filename + strlen (out2filename) - 1))) ?
	    "c" : "C");

	if ((out->out2file = CreateFile (out2filename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
	    CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
		free (out2filename);
//		error_line ("can't create file %s!", out2filename);
		DoCloseHandle (out->outfile);
		DoDeleteFile (lpszFilename);
		free (out);
		return 0;
	}

	free (out2filename);
    }

    return out;
}


//////////////////////////////////////////////////////////////////////////////
// Write data to Wavpack file.                                              //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL WriteFilterOutput (HANDLE hOutput, BYTE *lpbData, long lBytes)
{
    OUTPUT *out = hOutput;

    if (out && lBytes) {
	int flags = out->wphdr.flags, random = out->random;
	ulong lSamples = lBytes / ((flags & (BYTES_3 | OVER_20)) ? 4 : 2) / ((flags & MONO_FLAG) ? 1 : 2);

	// if we have not written any samples yet, we need to open the
	// bitstreams and initialize some other things

	if (!out->wphdr.total_samples) {
	    out->wpc.outbits.bufsiz = out->wpc.out2bits.bufsiz = 1024 * 1024;

	    if (bs_open_write (&out->wpc.outbits, out->outfile))
		return 0;

	    if ((flags & WVC_FLAG) && bs_open_write (&out->wpc.out2bits, out->out2file))
		return 0;

	    pack_init (&out->wpc);
	    out->error [0] = out->error [1] = 0.0;
	    random = 1234567890;
	}

	// if BYTES_3 is set, then we are storing float data as either 20-bit
	// or 24-bit ints (as indicated by OVER_20) and this must be converted
	// "in-place" here

	if (flags & BYTES_3) {
	    int sc = lBytes / ((flags & MONO_FLAG) ? 4 : 8);
	    float fMult, fMin, fMax, fDither = 0.5;
	    char *dst = lpbData, *src = lpbData;
	    long temp;

	    fMult = (flags & OVER_20) ? 256.0 : 16.0;
	    fMin = (flags & OVER_20) ? -8388608.0 : -524288.0;
	    fMax = (flags & OVER_20) ? 8388607.0 : 524287.0;

	    while (sc--) {
		* (float*) src *= fMult;

		if (flags & FLOAT_NOISESHAPE)
		    * (float*) src -= out->error [0];

		if (flags & FLOAT_DITHER) {
		    temp = (random = (((random << 4) - random) ^ 1)) >> 1;
		    temp += (random = (((random << 4) - random) ^ 1)) >> 1;
		    fDither = temp * (1/2147483648.0) + 0.5;
		}

		if (* (float*) src > fMax)
		    out->error [0] = (temp = fMax) - * (float*) src;
		else if (* (float*) src < fMin)
		    out->error [0] = (temp = fMin) - * (float*) src;
		else
		    out->error [0] = (temp = floor (* (float*) src + fDither)) - * (float*) src;

		* (long*) src = (flags & OVER_20) ? temp << 8 : temp << 12;

		*dst++ = *++src;
		*dst++ = *++src;
		*dst++ = *++src;
		++src;

		if (flags & MONO_FLAG)
		    continue;

		* (float*) src *= fMult;

		if (flags & FLOAT_NOISESHAPE)
		    * (float*) src -= out->error [1];

		if (flags & FLOAT_DITHER) {
		    temp = (random = (((random << 4) - random) ^ 1)) >> 1;
		    temp += (random = (((random << 4) - random) ^ 1)) >> 1;
		    fDither = temp * (1/2147483648.0) + 0.5;
		}

		if (* (float*) src > fMax)
		    out->error [1] = (temp = fMax) - * (float*) src;
		else if (* (float*) src < fMin)
		    out->error [1] = (temp = fMin) - * (float*) src;
		else
		    out->error [1] = (temp = floor (* (float*) src + fDither)) - * (float*) src;

		* (long*) src = (flags & OVER_20) ? temp << 8 : temp << 12;

		*dst++ = *++src;
		*dst++ = *++src;
		*dst++ = *++src;
		++src;
	    }
	}
	else if (flags & OVER_20) {
	    int sc = lBytes / ((flags & MONO_FLAG) ? 4 : 8);
	    short *dst = (short *) lpbData;
	    float *src = (float *) lpbData;
	    float fSam, fDither = 0.5;
	    long temp;

	    // If BYTES_3 is not set but OVER_20 is set, then this is the
	    // special case where we are storing float data as 16-bit ints.
	    // In this case, we do NOT want to pass the OVER_20 bit to
	    // pack_samples(); instead we just convert the data "in-place".

	    out->wphdr.flags &= ~OVER_20;

	    while (sc--) {
		fSam = *src++;

		if (flags & FLOAT_NOISESHAPE)
		    fSam -= out->error [0];

		if (flags & FLOAT_DITHER) {
		    temp = (random = (((random << 4) - random) ^ 1)) >> 1;
		    temp += (random = (((random << 4) - random) ^ 1)) >> 1;
		    fDither = temp * (1/2147483648.0) + 0.5;
		}

		if (fSam > 32767.0)
		    out->error [0] = (*dst++ = 32767) - fSam;
		else if (fSam < -32768.0)
		    out->error [0] = (*dst++ = -32768) - fSam;
		else
		    out->error [0] = (*dst++ = floor (fSam + fDither)) - fSam;

		if (flags & MONO_FLAG)
		    continue;

		fSam = *src++;

		if (flags & FLOAT_NOISESHAPE)
		    fSam -= out->error [1];

		if (flags & FLOAT_DITHER) {
		    temp = (random = (((random << 4) - random) ^ 1)) >> 1;
		    temp += (random = (((random << 4) - random) ^ 1)) >> 1;
		    fDither = temp * (1/2147483648.0) + 0.5;
		}

		if (fSam > 32767.0)
		    out->error [1] = (*dst++ = 32767) - fSam;
		else if (fSam < -32768.0)
		    out->error [1] = (*dst++ = -32768) - fSam;
		else
		    out->error [1] = (*dst++ = floor (fSam + fDither)) - fSam;
	    }
	}

	// we must not pass FLOAT_NOISESHAPE or FLOAT_DITHER to pack_samples(),
	// so we clear those bits now and restore the flags field afterward

	out->wphdr.flags &= ~(FLOAT_NOISESHAPE | FLOAT_DITHER);
	pack_samples (&out->wpc, lpbData, lSamples);
	out->wphdr.total_samples += lSamples;
	out->wphdr.flags = flags;
	out->random = random;

	// a bitstream error probably means that we ran out of disk space

	if (bs_error (&out->wpc.outbits) || ((out->wphdr.flags & WVC_FLAG) && bs_error (&out->wpc.out2bits)))
	    return 0;
	else
	    return lBytes;
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Close the WavPack file and free the OUTPUT structure. We have to seek    //
// back to the beginning of the file to rewrite the WavpackHeader with the  //
// updated crc(s) and possibly rewrite the RIFF header if we wrote some     //
// extra RIFF data.                                                         //
//////////////////////////////////////////////////////////////////////////////

void PASCAL CloseFilterOutput (HANDLE hOutput)
{
    OUTPUT *out = hOutput;
    ulong bcount;

    if (out) {
	// if we wrote any special RIFF bytes, then we have already done this

	if (!out->special_bytes && out->wphdr.total_samples) {
	    flush_word1 (&out->wpc);
	    bs_close_write (&out->wpc.outbits);
	}

	if (out->wphdr.flags & WVC_FLAG)
	    bs_close_write (&out->wpc.out2bits);

	if (bs_error (&out->wpc.outbits) || ((out->wphdr.flags & WVC_FLAG) && bs_error (&out->wpc.out2bits)))
	    MessageBox (NULL, "File not completed! Probably out of disk space!", "WavPack", MB_OK);

	if ((out->wphdr.flags & WVC_FLAG) && !DoCloseHandle (out->out2file))
	    MessageBox (NULL, "Could not close correction file!", "WavPack", MB_OK);

	// if we wrote special RIFF bytes, update RIFF header and rewrite

	if (out->special_bytes) {
	    out->riffhdr.ckSize += out->special_bytes;
	    SetFilePointer (out->outfile, 0, NULL, FILE_BEGIN);

	    if (!DoWriteFile (out->outfile, &out->riffhdr, sizeof (out->riffhdr), &bcount, NULL) ||
		bcount != sizeof (out->riffhdr))
		    MessageBox (NULL, "File not completed! Probably out of disk space!", "WavPack", MB_OK);
	}

	SetFilePointer (out->outfile, out->wavpack_header_pos, NULL, FILE_BEGIN);

	// Because I ran out of flag bits in the WavPack header, an amazingly ugly
	// kludge was forced upon me! This code takes care of preparing the flags
	// field to be written to the file. When the file is read, this is undone.

	if ((out->wphdr.flags & CROSS_DECORR) && !(out->wphdr.flags & EXTREME_DECORR))
	    out->wphdr.flags |= EXTREME_DECORR | CANCEL_EXTREME;

	out->wphdr.flags &= STORED_FLAGS;

	// here's that special case again of writing float data to 16-bit file

	if ((out->wphdr.flags & OVER_20) && !(out->wphdr.flags & BYTES_3))
	    out->wphdr.flags &= ~OVER_20;

	// write WavPack header with updated flags field and CRC(s)

	if (!DoWriteFile (out->outfile, &out->wphdr, sizeof (WavpackHeader), &bcount, NULL) ||
	    bcount != sizeof (WavpackHeader) ||	!DoCloseHandle (out->outfile))
		MessageBox (NULL, "Could not close file!", "WavPack", MB_OK);

	free (out);
    }
}


//////////////////////////////////////////////////////////////////////////////
// Open file for input and store information retrieved from file. For "raw" //
// WavPack files we don't know the sample rate so we set this to zero and   //
// CoolEdit will query the all-knowing user.                                //
//////////////////////////////////////////////////////////////////////////////

HANDLE PASCAL OpenFilterInput (LPSTR lpszFilename, long *lplSamprate,
    WORD *lpwBitsPerSample, WORD *lpwChannels, HWND hWnd, long *lplChunkSize)
{
    long total_samples, total_header_bytes;
    RiffChunkHeader RiffChunkHeader;
    ChunkHeader ChunkHeader;
    ulong bcount;
    INPUT *in;

    *lplChunkSize = CHUNKSIZE;

    if ((in = malloc (sizeof (INPUT))) == NULL)
	return 0;

    CLEAR (*in);
    in->wpc.wphdr = &in->wphdr;

    // open the source file for reading

    if ((in->infile = CreateFile (lpszFilename, GENERIC_READ, FILE_SHARE_READ, NULL,
	OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
//	    error_line ("can't open file %s!", lpszFilename);
	    free (in);
	    return 0;
    }

    // If the first chunk is a wave RIFF header, then read the various chunks
    // until we get to the "data" chunk (and WavPack header should follow). If
    // the first chunk is not a RIFF, then we assume a "raw" WavPack file and
    // the WavPack header must be first. In either event, "total_header_bytes"
    // will end up set to the file offset of the WavpackHeader.

    if (DoReadFile (in->infile, &RiffChunkHeader, sizeof (RiffChunkHeader), &bcount, NULL) &&
	bcount == sizeof (RiffChunkHeader) && !strncmp (RiffChunkHeader.ckID, "RIFF", 4) &&
	!strncmp (RiffChunkHeader.formType, "WAVE", 4)) {

	    in->special_bytes = RiffChunkHeader.ckSize - 4;

	    while (1) {

		if (!DoReadFile (in->infile, &ChunkHeader, sizeof (ChunkHeader), &bcount, NULL) ||
		    bcount != sizeof (ChunkHeader)) {
//			error_line ("%s is not a valid WavPack file!", lpszFilename);
			DoCloseHandle (in->infile);
			free (in);
			return 0;
		}

		in->special_bytes -= ChunkHeader.ckSize + 8;

		if (!strncmp (ChunkHeader.ckID, "fmt ", 4)) {

		    if (ChunkHeader.ckSize < sizeof (WaveHeader) ||
			!DoReadFile (in->infile, &in->wavhdr, sizeof (WaveHeader), &bcount, NULL) ||
			bcount != sizeof (WaveHeader)) {
//			    error_line ("%s is not a valid WavPack file!", lpszFilename);
			    DoCloseHandle (in->infile);
			    free (in);
			    return 0;
		    }

		    *lplSamprate = in->wavhdr.SampleRate;

		    if (ChunkHeader.ckSize > sizeof (WaveHeader))
			SetFilePointer (in->infile, (ChunkHeader.ckSize + 1 - sizeof (WaveHeader)) & ~1L, NULL, FILE_CURRENT);
		}
		else if (!strncmp (ChunkHeader.ckID, "data", 4)) {
#ifdef VER2
		    total_samples = ChunkHeader.ckSize / in->wavhdr.NumChannels /
			((in->wavhdr.BitsPerSample > 16) ? 3 : 2);
#endif
		    break;
		}
		else
		    SetFilePointer (in->infile, (ChunkHeader.ckSize + 1) & ~1L, NULL, FILE_CURRENT);
	    }

	    total_header_bytes = SetFilePointer (in->infile, 0, NULL, FILE_CURRENT);
    }
    else {
	SetFilePointer (in->infile, 0, NULL, FILE_BEGIN);
	in->wavhdr.SampleRate = 44100;
	*lplSamprate = 0;
	total_header_bytes = 0;
    }

    // read WavPack header and make sure this is a version we know about

    if (!DoReadFile (in->infile, &in->wphdr, sizeof (WavpackHeader), &bcount, NULL) ||
	bcount != sizeof (WavpackHeader) || strncmp (in->wphdr.ckID, "wvpk", 4)) {
//	    error_line ("%s is not a valid WavPack file!", lpszFilename);
	    DoCloseHandle (in->infile);
	    free (in);
	    return 0;
    }

    if (in->wphdr.version < 1 || in->wphdr.version > 3) {
//	error_line ("not compatible with this version of WavPack file!");
	DoCloseHandle (in->infile);
	free (in);
	return 0;
    }

    // Because I ran out of flag bits in the WavPack header, an amazingly ugly
    // kludge was forced upon me! This code takes care of preparing the flags
    // field for internal use and checking for unknown formats we can't decode

    if (in->wphdr.version == 3) {

	if (in->wphdr.flags & EXTREME_DECORR) {

	    if ((in->wphdr.flags & NOT_STORED_FLAGS) ||
		((in->wphdr.bits) &&
		(((in->wphdr.flags & NEW_HIGH_FLAG) &&
		(in->wphdr.flags & (FAST_FLAG | HIGH_FLAG))) ||
		(in->wphdr.flags & CROSS_DECORR)))) {
//		    error_line ("not compatible with this version of WavPack file!");
		    DoCloseHandle (in->infile);
		    free (in);
		    return 0;
	    }

	    if (in->wphdr.flags & CANCEL_EXTREME)
		in->wphdr.flags &= ~(EXTREME_DECORR | CANCEL_EXTREME);
	}
	else
	    in->wphdr.flags &= ~CROSS_DECORR;
    }

    // check to see if we should look for a "correction" file, and if so try
    // to open it for reading, then set WVC_FLAG accordingly

    if (in->wphdr.version == 3 && in->wphdr.bits && (in->wphdr.flags & NEW_HIGH_FLAG)
	&& in->wphdr.crc != in->wphdr.crc2) {
	    char *in2filename = malloc (strlen (lpszFilename) + 10);

	    strcpy (in2filename, lpszFilename);

	    strcat (in2filename, (islower (*(in2filename + strlen (in2filename) - 1))) ?
		"c" : "C");

	    in->in2file = CreateFile (in2filename, GENERIC_READ, FILE_SHARE_READ, NULL,
		OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

	    free (in2filename);
    }
    else
	in->in2file = INVALID_HANDLE_VALUE;

    if (in->in2file == INVALID_HANDLE_VALUE)
	in->wphdr.flags &= ~WVC_FLAG;
    else
	in->wphdr.flags |= WVC_FLAG;

    SetFilePointer (in->infile, total_header_bytes, NULL, FILE_BEGIN);

    // switch on WavPack version to handle special requirements of versions
    // before 3.0 that had smaller headers

    switch (in->wphdr.version) {

#ifdef VER2
	case 1:
	    in->wphdr.bits = 0;
	    SetFilePointer (in->infile, -2, NULL, FILE_CURRENT);

	case 2:
	    in->wphdr.total_samples = total_samples;
	    in->wphdr.flags = in->wavhdr.NumChannels == 1 ? MONO_FLAG : 0;
	    in->wphdr.shift = 16 - in->wavhdr.BitsPerSample;
	    SetFilePointer (in->infile, 12, NULL, FILE_CURRENT);
	    break;
#endif

	case 3:
	    SetFilePointer (in->infile, sizeof (WavpackHeader), NULL, FILE_CURRENT);
    }

    *lpwChannels = (in->wphdr.flags & MONO_FLAG) ? 1 : 2;
    *lpwBitsPerSample = (in->wphdr.flags & BYTES_3) ? 32 : 16;
    in->samples_remaining = in->wphdr.total_samples;

    return in;
}


//////////////////////////////////////////////////////////////////////////////
// Return the audio data size of the open file, 20/24-bit files are floats. //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterGetFileSize (HANDLE hInput)
{
    INPUT *in = hInput;

    if (in) {
	int flags = in->wphdr.flags;
	return in->wphdr.total_samples * ((flags & BYTES_3) ? 4 : 2) * ((flags & MONO_FLAG) ? 1 : 2);
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Read the specified number of audio data bytes from the open file.        //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL ReadFilterInput (HANDLE hInput, BYTE *lpbData, long lBytes)
{
    INPUT *in = hInput;
    ulong lSamples;

    if (in && lBytes) {
	int flags = in->wphdr.flags;

	// compute actual number of samples and clip for past EOF

	lSamples = lBytes / ((flags & BYTES_3) ? 4 : 2) / ((flags & MONO_FLAG) ? 1 : 2);

	if (lSamples > in->samples_remaining)
	    lSamples = in->samples_remaining;

	if (!lSamples)
	    return 0;

	// if we have not read any samples yet, we must initialize the
	// bitstreams and the unpacking code

	if (in->samples_remaining == in->wphdr.total_samples) {

	    in->wpc.inbits.bufsiz = in->wpc.in2bits.bufsiz = 1024 * 1024;

	    if (bs_open_read (&in->wpc.inbits, in->infile))
		return 0;

	    if ((flags & WVC_FLAG) && bs_open_read (&in->wpc.in2bits, in->in2file))
		return 0;

	    unpack_init (&in->wpc);
	}

	lSamples = unpack_samples (&in->wpc, lpbData, lSamples);

	// if we are reading 20 or 24-bit files, we must expand them "in-place"
	// to the CoolEdit internal 32-bit float format

	if (flags & BYTES_3) {
	    int sc = (flags & MONO_FLAG) ? lSamples : lSamples * 2;
	    char *dst, *src;

	    dst = lpbData + (sc * 4);
	    src = lpbData + (sc * 3);

	    while (sc--) {
		*--dst = *--src;
		*--dst = *--src;
		*--dst = *--src;
		*--dst = 0;
		* (float*) dst = * (long*) dst / 65536.0;
	    }
	}

	// if we are all done with file then we close the bitstreams here

	if (!(in->samples_remaining -= lSamples)) {
	    bs_close_read (&in->wpc.inbits);

	    if (flags & WVC_FLAG)
		bs_close_read (&in->wpc.in2bits);
	}

	return lSamples * ((flags & BYTES_3) ? 4 : 2) * ((flags & MONO_FLAG) ? 1 : 2);
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Close input file and release INPUT structure.                            //
//////////////////////////////////////////////////////////////////////////////

void PASCAL CloseFilterInput (HANDLE hInput)
{
    INPUT *in = hInput;

    if (in) {
	int flags = in->wphdr.flags;

	// if we did not read the whole file (but we read some), close the
	// bitstreams here

	if (in->samples_remaining && in->samples_remaining != in->wphdr.total_samples) {
	    bs_close_read (&in->wpc.inbits);

	    if (flags & WVC_FLAG)
		bs_close_read (&in->wpc.in2bits);
	}

	if (flags & WVC_FLAG)
	    DoCloseHandle (in->in2file);

	DoCloseHandle (in->infile);

	// if we read the whole, but the CRC does not match report a warning
	// to the user that the file may be corrupt

	if (in->wphdr.version == 3 && in->wphdr.total_samples && !in->samples_remaining &&
	    unpack_crc (&in->wpc) != ((flags & WVC_FLAG) ? in->wphdr.crc2 : in->wphdr.crc))
		MessageBox (NULL, "Warning: CRC Error!", "WavPack", MB_OK);

	free (in);
    }
}


static BOOL CALLBACK WavPackDlgProc (HWND, UINT, WPARAM, LPARAM);
static int std_bitrate (int bitrate);

// these variables are use to communicate with the dialog routines

static int iCurrentMode;	// lossless [std/high/fast], bybrid [high]
static int iCurrentFloatBits;	// 16, 20, or 24 bit conversion from float
static int iCurrentFlags;	// WVC_FLAG, FLOAT_NOISESHAPE, & FLOAT_DITHER
static int iCurrentBitrate, iMinBitrate; // kbps

//////////////////////////////////////////////////////////////////////////////
// Query user for WavPack file writing options. This also has the function  //
// of converting the 32-bit configuration word that CoolEdit uses from a    //
// possibly "raw" form (read directly from an input file) into a form that  //
// we can use for writing. This is done because some WavPack file formats   //
// can no longer be written (like the old lossy mode) and because some      //
// things we are not supporting here (like hybrid joint stereo and very     //
// fast mode) and some things are simply not knowable from reading a file   //
// (like the dither and noiseshaping modes used when converting floats).    //
// This function can also be used just to convert from "raw" to "usable"    //
// configuration by just calling it with NULL window and instance args.     //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterGetOptions (HWND hWnd, HINSTANCE hInst, long lSamprate, WORD wChannels, WORD wBitsPerSample, DWORD dwOptions)
{
    int bits;

    // this kludge ignores config bytes generated (incorrectly) by a very
    // early version of the filter

    if ((dwOptions & 0xffff8000) == 0xffff8000)
	dwOptions = 0;

    iCurrentFlags = 0;

    // generate the minimum and default bitrates for hybrid modes (and make
    // the default a "standard" bitrate) but the default could be overwritten
    // if we have actual information from a hybrid (or old-style lossy) file

    iCurrentBitrate = floor (lSamprate / 1000.0 * 4 * wChannels + 0.5);
    iMinBitrate = floor (lSamprate / 1000.0 * 3 * wChannels + 0.5);

    while (!std_bitrate (iCurrentBitrate))
	--iCurrentBitrate;

    if (dwOptions & CONFIG_BITS) {	// handle lossy/hybrid modes
	bits = (dwOptions & CONFIG_BITS) >> 16;

	if (dwOptions & NEW_HIGH_FLAG) {
	    iCurrentBitrate = floor ((float) bits * lSamprate / 256000.0 + 0.5);
	    iCurrentMode = (dwOptions & EXTREME_DECORR) ? IDC_HYBRID_HIGH : IDC_HYBRID;
	}
	else {
	    iCurrentBitrate = floor ((bits + 2.6836) * lSamprate / 1000.0 * wChannels + 0.5);
	    iCurrentMode = (dwOptions & HIGH_FLAG) ? IDC_HYBRID_HIGH : IDC_HYBRID;
	}
    }
    else if (dwOptions) {		// else handle pure lossless modes
	if (!(dwOptions & HIGH_FLAG))
	    iCurrentMode = IDC_LOSSLESS_FAST;
	else if (dwOptions & EXTREME_DECORR)
	    iCurrentMode = IDC_LOSSLESS_HIGH;
	else
	    iCurrentMode = IDC_LOSSLESS;
    }
    else				// else default to pure lossless
	iCurrentMode = IDC_LOSSLESS;

    // if the last file read was 20-bit or 24-bit, we want to use that as the
    // default conversion mode, but we do NOT want to set 16-bit conversions
    // just because the last file was 16-bit (this must be set explicitly)

    if (dwOptions & BYTES_3)
	iCurrentFloatBits = (dwOptions & OVER_20) ? IDC_FLOAT24 : IDC_FLOAT20;
    else
	iCurrentFloatBits = (dwOptions & OVER_20) ? IDC_FLOAT16 : IDC_FLOAT24;

    // since we cannot determine the float conversion options of noiseshaping
    // and dither from files that we read, we must default these unless they
    // have been explicitly set by the user

    if (dwOptions & CONFIG_UPDATED)
	iCurrentFlags |= dwOptions & (FLOAT_NOISESHAPE | FLOAT_DITHER);
    else
	iCurrentFlags |= FLOAT_NOISESHAPE;

    if (dwOptions & WVC_FLAG)
	iCurrentFlags |= WVC_FLAG;

    // If the instance and window args are not NULL, execute the dialog box
    // and return the unchanged configuration if the user hit "cancel". The
    // optional parameter that can be passed to the dialog is used to indicate
    // whether the float conversion options should be disabled.

    if (hInst && hWnd && !DialogBoxParam (hInst, "CoolEdit", hWnd, WavPackDlgProc, wBitsPerSample == 16))
	return dwOptions;

    // now generate a configuration that is usable for writing from the options
    // that the user selected (or at least approved of)

    dwOptions = CONFIG_UPDATED;

    switch (iCurrentMode) {
	case IDC_LOSSLESS_HIGH:
	    dwOptions |= EXTREME_DECORR;

	case IDC_LOSSLESS:
	    dwOptions |= HIGH_FLAG | NEW_HIGH_FLAG | CROSS_DECORR | NEW_DECORR_FLAG | JOINT_STEREO;
	    break;

	case IDC_HYBRID_HIGH:
	    dwOptions |= EXTREME_DECORR;

	case IDC_HYBRID:
	    dwOptions |= NEW_HIGH_FLAG | NEW_DECORR_FLAG;

	    if (lSamprate >= 64000)
		dwOptions |= LOSSY_SHAPE;

	    bits = floor ((float) iCurrentBitrate / lSamprate * 256000.0 + 0.5);
	    dwOptions |= bits <= (64 << 8) ? (bits << 16) : (64 << 24);
    }

    if (wChannels == 1) {
	dwOptions &= ~(JOINT_STEREO | CROSS_DECORR);
	dwOptions |= MONO_FLAG;
    }

    if (iCurrentFloatBits == IDC_FLOAT16)
	dwOptions |= OVER_20;
    else if (iCurrentFloatBits == IDC_FLOAT24)
	dwOptions |= (BYTES_3 | OVER_20);
    else
	dwOptions |= BYTES_3;

    dwOptions |= iCurrentFlags & (FLOAT_NOISESHAPE | FLOAT_DITHER | WVC_FLAG);

    return dwOptions;
}


//////////////////////////////////////////////////////////////////////////////
// Standard callback for handling dialogs. The optional parameter is used   //
// to indicate whether the float conversion options should be disabled and  //
// the return value indicates whether the user chose OK or CANCEL.          //
//////////////////////////////////////////////////////////////////////////////

static BOOL CALLBACK WavPackDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    char str [80];
    int i;

    switch (message) {
	case WM_INITDIALOG:
	    for (i = IDC_BITRATE; i <= IDC_CORRECTION; ++i)
		EnableWindow (GetDlgItem (hDlg, i), iCurrentMode >= IDC_HYBRID);

	    if (lParam)
		for (i = IDC_FLOAT16; i <= IDC_DITHER; ++i)
		    EnableWindow (GetDlgItem (hDlg, i), 0);

	    CheckRadioButton (hDlg, IDC_LOSSLESS, IDC_HYBRID_HIGH, iCurrentMode);
	    CheckRadioButton (hDlg, IDC_FLOAT16, IDC_FLOAT24, iCurrentFloatBits);

	    SendDlgItemMessage (hDlg, IDC_BITRATE, CB_LIMITTEXT, 4, 0);

	    for (i = iMinBitrate; i <= iMinBitrate * 4; ++i)
		if (i == iMinBitrate || std_bitrate (i)) {
		    sprintf (str, "%d", i);
		    SendDlgItemMessage (hDlg, IDC_BITRATE, CB_ADDSTRING, 0, (long) str);
		}

	    sprintf (str, "%d", iCurrentBitrate);
	    SetDlgItemText (hDlg, IDC_BITRATE, str);

	    CheckDlgButton (hDlg, IDC_CORRECTION, iCurrentFlags & WVC_FLAG);
	    CheckDlgButton (hDlg, IDC_NOISESHAPE, iCurrentFlags & FLOAT_NOISESHAPE);
	    CheckDlgButton (hDlg, IDC_DITHER, iCurrentFlags & FLOAT_DITHER);

	    SetFocus (GetDlgItem (hDlg, iCurrentMode));

	    return FALSE;

	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case IDC_LOSSLESS: case IDC_LOSSLESS_HIGH: case IDC_LOSSLESS_FAST:
		    for (i = IDC_BITRATE; i <= IDC_CORRECTION; ++i)
			EnableWindow (GetDlgItem (hDlg, i), FALSE);

		    break;

		case IDC_HYBRID: case IDC_HYBRID_HIGH:
		    for (i = IDC_BITRATE; i <= IDC_CORRECTION; ++i)
			EnableWindow (GetDlgItem (hDlg, i), TRUE);

		    break;

		case IDOK:
		    for (i = IDC_LOSSLESS; i <= IDC_HYBRID_HIGH; ++i)
			if (IsDlgButtonChecked (hDlg, i)) {
			    iCurrentMode = i;
			    break;
			}

		    for (i = IDC_FLOAT16; i <= IDC_FLOAT24; ++i)
			if (IsDlgButtonChecked (hDlg, i)) {
			    iCurrentFloatBits = i;
			    break;
			}

		    iCurrentFlags = 0;

		    if (IsDlgButtonChecked (hDlg, IDC_CORRECTION))
			iCurrentFlags |= WVC_FLAG;

		    if (IsDlgButtonChecked (hDlg, IDC_NOISESHAPE))
			iCurrentFlags |= FLOAT_NOISESHAPE;

		    if (IsDlgButtonChecked (hDlg, IDC_DITHER))
			iCurrentFlags |= FLOAT_DITHER;

		    GetWindowText (GetDlgItem (hDlg, IDC_BITRATE), str, sizeof (str));

		    if (atol (str) && atol (str) <= 9999)
			iCurrentBitrate = atol (str) < iMinBitrate ? iMinBitrate : atol (str);

		    EndDialog (hDlg, TRUE);
		    return TRUE;

		case IDCANCEL:
		    EndDialog (hDlg, FALSE);
		    return TRUE;
	    }

	    break;
    }

    return FALSE;
}


//////////////////////////////////////////////////////////////////////////////
// Function to determine whether a given bitrate is a "standard".           //
//////////////////////////////////////////////////////////////////////////////

static int std_bitrate (int bitrate)
{
    int sd, sc = 0;

    for (sd = bitrate; sd > 7; sd >>= 1)
	sc++;

    return bitrate == sd << sc;
}


//////////////////////////////////////////////////////////////////////////////
// Return a "raw" configuration value for the open file. Unfortunately,     //
// zero would be a valid value but we can't return that so we have to fudge //
// that case. Having the CONFIG_UPDATED bit clear indicates that this has   //
// not passed through a config dialog and is NOT suitable for writing.      //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterOptions (HANDLE hInput)
{
    INPUT *in = hInput;

    if (in) {
	DWORD opts = (ushort) in->wphdr.flags | ((DWORD) in->wphdr.bits << 16);

	opts &= ~CONFIG_UPDATED;	// configuration is not "updated"

	if (!opts)			// don't allow zero!
	    opts |= CALC_NOISE;

	return opts;
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Return a string for displaying the configuration of the open WavPack     //
// file. Note that we can display modes that we can no longer write like    //
// the old lossy mode. We return the "raw" configuration word.              //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterOptionsString (HANDLE hInput, LPSTR lpszString)
{
    INPUT *in = hInput;

    if (in && lpszString) {
	float time_secs = (float) in->wphdr.total_samples / in->wavhdr.SampleRate;
	uint compressed_bytes = GetFileSize (in->infile, NULL);
	DWORD opts = FilterOptions (hInput);

	if (in->wphdr.flags & WVC_FLAG)
	    compressed_bytes += GetFileSize (in->in2file, NULL);

	sprintf (lpszString, "WavPack ");

	if (in->wphdr.bits) {
	    if (in->wphdr.flags & NEW_HIGH_FLAG)
		sprintf (lpszString + strlen (lpszString), "Hybrid %s%s Mode ",
		    (in->wphdr.flags & WVC_FLAG) ? "Lossless" : "Lossy",
		    (in->wphdr.flags & EXTREME_DECORR) ? " High" : "");
	    else if (!(in->wphdr.flags & (HIGH_FLAG | FAST_FLAG)))
		sprintf (lpszString + strlen (lpszString), "%d-bit Lossy Mode ", in->wphdr.bits + 3);
	    else
		sprintf (lpszString + strlen (lpszString), "%d-bit Lossy %s Mode ", in->wphdr.bits + 3,
		    (in->wphdr.flags & HIGH_FLAG) ? "High" : "Fast");
	}
	else {
	    if (!(in->wphdr.flags & HIGH_FLAG))
		sprintf (lpszString + strlen (lpszString), "Lossless Fast Mode ");
	    else if (in->wphdr.flags & EXTREME_DECORR)
		sprintf (lpszString + strlen (lpszString), "Lossless High Mode ");
	    else
		sprintf (lpszString + strlen (lpszString), "Lossless Mode ");
	}

	if (in->wphdr.flags & RAW_FLAG)
	    sprintf (lpszString + strlen (lpszString), "(Raw)");
	else
	    sprintf (lpszString + strlen (lpszString), "(%d kbps)", (int) floor (compressed_bytes / time_secs / 125.0 + 0.5));

	return opts;
    }

    return 0;
}


static DWORD write_special_riff_chunk (OUTPUT *out, LPCSTR szListType, ChunkHeader *ckHdr, char *pData);

//////////////////////////////////////////////////////////////////////////////
// Write special data into WavPack file. All special data types basically   //
// mirror the equivalent RIFF chunks, so all we really have to do is        //
// convert those that aren't exactly the same and then generate RIFF chucks //
// and write those to the file (because WavPack stores extra RIFF chunks    //
// verbatim). Because this is very CoolEdit specific, I will not comment it //
// in detail.                                                               //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterWriteSpecialData (HANDLE hOutput, LPCSTR szListType,
    LPCSTR szType, char *pData, DWORD dwSize)
{
    ChunkHeader ChunkHeader;
    OUTPUT *out = hOutput;
    char str [80];
    DWORD bcount;

    if (out) {
	if (!strncmp (szListType, "INFO", 4)) {

	    strncpy (ChunkHeader.ckID, szType, 4);
	    ChunkHeader.ckSize = dwSize;
	    return write_special_riff_chunk (out, szListType, &ChunkHeader, pData);
	}
	else if (!strncmp (szListType, "adtl", 4)) {
	    if (!strncmp (szType, "ltxt", 4)) {
		char *cp = pData, *ep = pData + dwSize;

		while (ep - cp > 4) {
		    DWORD length = * (DWORD *) cp;
		    char *pBuffer = malloc (length + 4);

		    memset (pBuffer, 0, length + 4);
		    memcpy (pBuffer, cp + 4, 12);

		    if (length > 16)
			memcpy (pBuffer + 20, cp + 16, length - 16);

		    strncpy (ChunkHeader.ckID, szType, 4);
		    ChunkHeader.ckSize = length + 4;
		    write_special_riff_chunk (out, szListType, &ChunkHeader, pBuffer);
		    free (pBuffer);
		    cp += length;
		}
	    }
	    else if (!strncmp (szType, "labl", 4) || !strncmp (szType, "note", 4)) {
		char *cp = pData, *ep = pData + dwSize;

		while (ep - cp > 4) {
		    strncpy (ChunkHeader.ckID, szType, 4);
		    ChunkHeader.ckSize = strlen (cp + 4) + 5;
		    write_special_riff_chunk (out, szListType, &ChunkHeader, cp);
		    cp += ChunkHeader.ckSize;
		}
	    }
	    else
		return 0;
	}
	else if (!strncmp (szListType, "WAVE", 4)) {
	    if (!strncmp (szType, "cue ", 4)) {
		DWORD num_cues = dwSize / 8, *dwPtr = (DWORD *) pData;
		struct cue_type *pCue;
		char *pBuffer;

		strncpy (ChunkHeader.ckID, szType, 4);
		ChunkHeader.ckSize = num_cues * sizeof (*pCue) + sizeof (DWORD);
		pBuffer = malloc (ChunkHeader.ckSize);
		* (DWORD*) pBuffer = num_cues;
		pCue = (struct cue_type *) (pBuffer + sizeof (DWORD));

		while (num_cues--) {
		    CLEAR (*pCue);
		    strncpy ((char *) &pCue->fccChunk, "data", 4);
		    pCue->dwName = *dwPtr++;
		    pCue->dwPosition = pCue->dwSampleOffset = *dwPtr++;
		    pCue++;
		}

		write_special_riff_chunk (out, szListType, &ChunkHeader, pBuffer);
		free (pBuffer);
		return 1;
	    }
	    else if (!strncmp (szType, "plst", 4)) {
		DWORD num_plays = dwSize / 16, *dwPtr = (DWORD *) pData;
		struct play_type *pPlay;
		char *pBuffer;

		strncpy (ChunkHeader.ckID, szType, 4);
		ChunkHeader.ckSize = num_plays * sizeof (*pPlay) + sizeof (num_plays);
		pBuffer = malloc (ChunkHeader.ckSize);
		* (DWORD*) pBuffer = num_plays;
		pPlay = (struct play_type *) (pBuffer + sizeof (DWORD));

		while (num_plays--) {
		    CLEAR (*pPlay);
		    pPlay->dwName = *dwPtr++;
		    pPlay->dwLength = *dwPtr++;
		    pPlay->dwLoops = *dwPtr++;
		    dwPtr++;	// I don't know what to do with "dwMode" (?)
		    pPlay++;
		}

		write_special_riff_chunk (out, szListType, &ChunkHeader, pBuffer);
		free (pBuffer);
		return 1;
	    }

	    strncpy (ChunkHeader.ckID, szType, 4);
	    ChunkHeader.ckSize = dwSize;
	    return write_special_riff_chunk (out, szListType, &ChunkHeader, pData);
	}
	else
	    return 0;
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Write the specified RIFF chunk to the end of the WavPack file. The only  //
// tricky part is that chunks that are not top level types should be        //
// combined into LIST chunks. When we are done we must also update the RIFF //
// chunk header at the very beginning of the file.                          //
//////////////////////////////////////////////////////////////////////////////

static DWORD write_special_riff_chunk (OUTPUT *out, LPCSTR szListType, ChunkHeader *ckHdr, char *pData)
{
    DWORD bcount, temp;

    // if this is the first special RIFF chunk then we must flush out all the
    // audio data and write the 16 bytes of "ones" that indicates to the
    // decoder that special RIFF data is appended to the WavPack file.

    if (!out->special_bytes) {
	char ones [16];

	if (out->wphdr.total_samples) {
	    flush_word1 (&out->wpc);
	    bs_close_write (&out->wpc.outbits);
	}

	memset (ones, -1, sizeof (ones));
	DoWriteFile (out->outfile, ones, sizeof (ones), &bcount, NULL);
    }

    if (strncmp (szListType, "WAVE", 4)) {
	if (!out->listhdr.ckSize || strncmp (out->listhdr.formType, szListType, 4)) {
	    out->special_bytes += sizeof (out->listhdr);
	    strncpy (out->listhdr.formType, szListType, 4);
	    strncpy (out->listhdr.ckID, "LIST", 4);
	    out->listhdr.ckSize = 4;
	}
	else
	    SetFilePointer (out->outfile, -(8 + out->listhdr.ckSize), NULL, FILE_CURRENT);

	temp = out->listhdr.ckSize - 4;
	out->listhdr.ckSize += sizeof (ChunkHeader) + ((ckHdr->ckSize + 1) & ~1);
	DoWriteFile (out->outfile, &out->listhdr, sizeof (out->listhdr), &bcount, NULL);
	SetFilePointer (out->outfile, temp, NULL, FILE_CURRENT);
    }
    else
	out->listhdr.ckSize = 0;

    DoWriteFile (out->outfile, ckHdr, sizeof (ChunkHeader), &bcount, NULL);

    if (ckHdr->ckSize) {
	DoWriteFile (out->outfile, pData, ckHdr->ckSize, &bcount, NULL);

	if (ckHdr->ckSize & 1)
	    DoWriteFile (out->outfile, "\0", 1, &bcount, NULL);
    }

    out->special_bytes += sizeof (ChunkHeader) + ((ckHdr->ckSize + 1) & ~1);
    return 1;
}


//////////////////////////////////////////////////////////////////////////////
// Read next special data from WavPack file. Special data types basically   //
// mirror the equivalent RIFF chunks, so all we really have to do is        //
// convert those that aren't exactly the same and then pass them back to    //
// CoolEdit. Because this is very CoolEdit specific, I will not comment it  //
// in detail.                                                               //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterGetNextSpecialData (HANDLE hInput, SPECIALDATA *psp)
{
    ChunkHeader ChunkHeader;
    INPUT *in = hInput;
    DWORD bcount;

    while (in && in->special_bytes) {
	char * pData;

	if (in->special_bytes < sizeof (ChunkHeader) ||
	    !DoReadFile (in->infile, &ChunkHeader, sizeof (ChunkHeader), &bcount, NULL) ||
	    bcount != sizeof (ChunkHeader))
		return 0;

	in->special_bytes -= bcount;

	if (!strncmp (ChunkHeader.ckID, "LIST", 4)) {
	    memcpy (&in->listhdr, &ChunkHeader, sizeof (ChunkHeader));

	    if (in->special_bytes < sizeof (in->listhdr.formType) || 
		!DoReadFile (in->infile, &in->listhdr.formType, sizeof (in->listhdr.formType), &bcount, NULL) ||
		bcount != sizeof (in->listhdr.formType))
		    return 0;

	    in->special_bytes -= bcount;

	    if ((in->listhdr.ckSize -= sizeof (in->listhdr.formType)) < 0)
		in->listhdr.ckSize = 0;

	    continue;
	}

	if (in->listhdr.ckSize) {
	    if ((in->listhdr.ckSize -= sizeof (ChunkHeader)) < 0)
		in->listhdr.ckSize = 0;

	    strncpy (psp->szListType, in->listhdr.formType, 4);
	    psp->szListType [4] = 0;
	}
	else
	    strcpy (psp->szListType, "WAVE");

	strncpy (psp->szType, ChunkHeader.ckID, 4);
	psp->szType [4] = 0;
	psp->dwSize = ChunkHeader.ckSize;
	psp->dwExtra = 1;
	psp->hSpecialData = NULL;
	psp->hData = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, psp->dwSize + 1);
	pData = (char *) GlobalLock (psp->hData);

	if (in->special_bytes < ((psp->dwSize + 1) & ~1) ||
	    !DoReadFile (in->infile, pData, (psp->dwSize + 1) & ~1, &bcount, NULL) ||
	    bcount != ((psp->dwSize + 1) & ~1)) {
		GlobalUnlock (pData);
		GlobalFree (psp->hData);
		return 0;
	}

	in->special_bytes -= bcount;

	if (in->listhdr.ckSize)
	    if ((in->listhdr.ckSize -= (ChunkHeader.ckSize + 1) & ~1) < 0)
		in->listhdr.ckSize = 0;

	if (!strncmp (ChunkHeader.ckID, "cue ", 4)) {
	    int num_cues = (psp->dwExtra = * (DWORD *) pData);
	    struct cue_type *pCue = (struct cue_type *)(pData + sizeof (DWORD));
	    HANDLE hxData = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, num_cues * 8);
	    DWORD *pdwData = (DWORD *) GlobalLock (hxData);

	    while (num_cues--) {
		*pdwData++ = pCue->dwName;
		*pdwData++ = pCue->dwPosition;
		pCue++;
	    }

	    GlobalUnlock (pData);
	    GlobalFree (psp->hData);
	    GlobalUnlock (pdwData);
	    psp->hData = hxData;
	    return 1;
	}
	else if (!strncmp (ChunkHeader.ckID, "plst", 4)) {
	    int num_plays = (psp->dwExtra = * (DWORD *) pData);
	    struct play_type *pCue = (struct play_type *)(pData + sizeof (DWORD));
	    HANDLE hxData = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, num_plays * 16);
	    DWORD *pdwData = (DWORD *) GlobalLock (hxData);

	    while (num_plays--) {
		*pdwData++ = pCue->dwName;
		*pdwData++ = pCue->dwLength;
		*pdwData++ = pCue->dwLoops;
		*pdwData++ = 0;
		pCue++;
	    }

	    GlobalUnlock (pData);
	    GlobalFree (psp->hData);
	    GlobalUnlock (pdwData);
	    psp->hData = hxData;
	    return 1;
	}
	else if (!strncmp (ChunkHeader.ckID, "ltxt", 4)) {
	    HANDLE hxData = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, psp->dwSize - 4);
	    char *pxData = GlobalLock (hxData);

	    memset (pxData, 0, psp->dwSize - 4);
	    * (DWORD *) pxData = psp->dwSize - 4;
	    memcpy (pxData + 4, pData, 12);

	    if (psp->dwSize > 20)
		memcpy (pxData + 16, pData + 20, psp->dwSize - 20);

	    GlobalUnlock (pData);
	    GlobalFree (psp->hData);
	    GlobalUnlock (pxData);
	    psp->hData = hxData;
	    return 1;
	}

	GlobalUnlock (pData);
	return 1;
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Attempt to read special RIFF data from the open WavPack file. If the     //
// audio data was not completely read then we can't do this (cause we're    //
// lazy). Once we determine that extra RIFF chunks are available, we use    //
// FilterGetNextSpecialData() to read them and return them to CoolEdit.     //
//////////////////////////////////////////////////////////////////////////////

DWORD PASCAL FilterGetFirstSpecialData (HANDLE hInput, SPECIALDATA *psp)
{
    INPUT *in = hInput;
    DWORD bcount;

    if (in) {
	char c, ones [16];

	if (in->samples_remaining)
	    return 0;

	if (!DoReadFile (in->infile, &ones, sizeof (ones), &bcount, NULL) || bcount != sizeof (ones))
	    return 0;

	for (c = 0; c < sizeof (ones) && !~ones [c]; ++c);

	if (c < sizeof (ones))
	    return 0;

	return FilterGetNextSpecialData (hInput, psp);
    }

    return 0;
}
