////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2003 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// tags.c

// This module handles reading ID3v1 and APEv2 tags.

#include "wavpack.h"

#include <stdlib.h>

//////////////////////////////////////////////////////////////////////////////
// This function attempts to load an ID3v1 or APEv2 tag from the specified  //
// file into the specified M_Tag structure. The ID3 tag fits in completely, //
// but an APEv2 tag is variable length and so space must be allocated here  //
// to accomodate the data, and this will need to be freed later. A return   //
// value of TRUE indicates a valid tag was found and loaded. Note that the  //
// file pointer is undefined when this function exits.                      //
//////////////////////////////////////////////////////////////////////////////

int LoadTag (M_Tag *m_tag, HANDLE infile)
{
    ulong bcount;

    CLEAR (*m_tag);

    // First, attempt to find an APEv2 tag...

    SetFilePointer (infile, -sizeof (APE_Tag_Hdr), NULL, FILE_END);

    if (ReadFile (infile, &m_tag->ape_tag_hdr, sizeof (APE_Tag_Hdr), &bcount, NULL) &&
	bcount == sizeof (APE_Tag_Hdr) && !strncmp (m_tag->ape_tag_hdr.ID, "APETAGEX", 8) &&
	m_tag->ape_tag_hdr.version == 2000 && m_tag->ape_tag_hdr.item_count &&
	m_tag->ape_tag_hdr.length > sizeof (m_tag->ape_tag_hdr) &&
	m_tag->ape_tag_hdr.length < (1024 * 1024) &&
	(m_tag->ape_tag_data = malloc (m_tag->ape_tag_hdr.length)) != NULL) {

	    memset (m_tag->ape_tag_data, 0, m_tag->ape_tag_hdr.length);
	    SetFilePointer (infile, -m_tag->ape_tag_hdr.length, NULL, FILE_END);

	    if (!ReadFile (infile, m_tag->ape_tag_data, m_tag->ape_tag_hdr.length - sizeof (APE_Tag_Hdr), &bcount, NULL) ||
		bcount != m_tag->ape_tag_hdr.length - sizeof (APE_Tag_Hdr)) {
		    free (m_tag->ape_tag_data);
		    CLEAR (*m_tag);
		    return FALSE;
	    }
	    else
		return TRUE;
    }

    // ...if not, try a ID3v1 tag

    SetFilePointer (infile, -sizeof (ID3_Tag), NULL, FILE_END);

    if (ReadFile (infile, &m_tag->id3_tag, sizeof (ID3_Tag), &bcount, NULL) &&
	bcount == sizeof (ID3_Tag) && !strncmp (m_tag->id3_tag.tag_id, "TAG", 3))
	    return TRUE;
    else {
	CLEAR (*m_tag);
	return FALSE;
    }
}

static void tagcpy (char *dest, char *src, int tag_size)
{
    char *s1 = src, *s2 = src + tag_size - 1;

    while (s1 <= s2)
	if (*s1 == ' ')
	    ++s1;
	else if (!*s2 || *s2 == ' ')
	    --s2;
	else
	    break;

    while (*s1 && s1 <= s2)
	*dest++ = *s1++;

    *dest = 0;
}

//////////////////////////////////////////////////////////////////////////////
// Attempt to get the specified item from the specified ID3v1 or APEv2 tag. //
// The "size" parameter specifies the amount of space available at "value", //
// if the desired item will not fit in this space then ellipses (...) will  //
// be appended and the string terminated. Only text data are supported.     //
//////////////////////////////////////////////////////////////////////////////

int GetTagItem (M_Tag *m_tag, const char *item, char *value, int size)
{
    char lvalue [1024];

    lvalue [0] = 0;

    if (value)
	*value = 0;

    if (m_tag->ape_tag_hdr.ID [0] == 'A') {
	char *p = m_tag->ape_tag_data;
	char *q = p + m_tag->ape_tag_hdr.length - sizeof (APE_Tag_Hdr);
	int i;

	for (i = 0; i < m_tag->ape_tag_hdr.item_count; ++i) {
	    int vsize = * ((long *) p)++;
	    int flags = * ((long *) p)++;
	    int isize = strlen (p);

	    if (p + isize + vsize + 1 > q)
		break;

	    if (isize && vsize && vsize < sizeof (lvalue) && !stricmp (item, p) && !(flags & 6)) {
		strncpy (lvalue, p + isize + 1, vsize);
		lvalue [vsize] = 0;
		break;
	    }
	    else
		p += isize + vsize + 1;
	}
    }
    else if (m_tag->id3_tag.tag_id [0] == 'T') {
	if (!stricmp (item, "title"))
	    tagcpy (lvalue, m_tag->id3_tag.title, sizeof (m_tag->id3_tag.title));
	else if (!stricmp (item, "artist"))
	    tagcpy (lvalue, m_tag->id3_tag.artist, sizeof (m_tag->id3_tag.artist));
	else if (!stricmp (item, "album"))
	    tagcpy (lvalue, m_tag->id3_tag.album, sizeof (m_tag->id3_tag.album));
	else if (!stricmp (item, "year"))
	    tagcpy (lvalue, m_tag->id3_tag.year, sizeof (m_tag->id3_tag.year));
	else if (!stricmp (item, "comment"))
	    tagcpy (lvalue, m_tag->id3_tag.comment, sizeof (m_tag->id3_tag.title));
    }
    else
	return FALSE;

    if (lvalue [0]) {
	if (value && size >= 4) {
	    if (strlen (lvalue) >= size) {
		lvalue [size - 4] = lvalue [size - 3] = lvalue [size - 2] = '.';
		lvalue [size - 1] = 0;
	    }

	    strcpy (value, lvalue);
	}

	return TRUE;
    }
    else
	return FALSE;
}

//////////////////////////////////////////////////////////////////////////////
// Return TRUE is a valid ID3v1 or APEv2 tag has been loaded.               //
//////////////////////////////////////////////////////////////////////////////

int ValidTag (M_Tag *m_tag)
{
    return (m_tag->ape_tag_hdr.ID [0] == 'A' || m_tag->id3_tag.tag_id [0] == 'T');
}

//////////////////////////////////////////////////////////////////////////////
// Free the data for any APEv2 tag that was allocated.                      //
//////////////////////////////////////////////////////////////////////////////

void FreeTag (M_Tag *m_tag)
{
    if (m_tag->ape_tag_data) {
	free (m_tag->ape_tag_data);
	m_tag->ape_tag_data = 0;
    }
}
