////////////////////////////////////////////////////////////////////////////
//			     **** WAVPACK ****				  //
//		    Hybrid Lossless Wavefile Compressor			  //
//		Copyright (c) 1998 - 2002 Conifer Software.		  //
//			    All Rights Reserved.			  //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// unpack.c

// This module actually handles the decompression of the audio data, except
// for the entropy decoding which is handled by the words? modules. For
// maximum efficiency, the conversion is isolated to tight loops that handle
// an entire buffer. The actual bitstreams are "inbits" for the main WavPack
// file and "in2bits" for the .wvc file (if present) and these must be
// initialized with bs_open_read() before unpack_samples() is called.
//
// Functions are provided to save (and restore) the complete unpacking context
// to (and from) another buffer. This may be used (as is done in the winamp
// plugin) to store index points during decoding for rapid seeking during
// playback. The exact format of these context images are not as small as
// possible (although they are much smaller than simply storing everything)
// and are not guaranteed to remain the same from different versions (or even
// compiles). Therefore, they should only be used temporarily and not stored
// to HD.
//
// The "COMPACT" define specifies an alternate version for the default lossless
// mode that uses less inline code but is somewhat slower. Use at your own
// discretion.

#include "wavpack.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#ifndef __WIN32__
#include <io.h>
#endif


//////////////////////////////// local macros /////////////////////////////////

#define PUT_24(ptr,val) (*((uchar*)ptr)=val,*(short*)((uchar*)(ptr)+1)=(val)>>8)

#define apply_weight(bits, weight, sample) ((weight * sample + (1 << (bits - 1))) >> bits)

#define update_weight(bits, weight, source, result) \
    if (source && result) { \
	if ((source ^ result) >= 0) { if (weight++ == (1 << bits)) weight--; } \
	else if (weight-- == min_weight) weight++; \
    }

#define apply_weight24(weight, sample) ((sample >= 0x800000 || sample < -0x800000) ? ((long)floor(((double) weight * sample + 128.0) / 256.0)) : ((weight * sample + 128) >> 8))

#define update_weight2(weight, source, result) \
    if (source && result) { \
	if ((source ^ result) >= 0) { if (weight++ == 256) weight--; } \
	else if (weight-- == min_weight) weight++; \
    }

//////////////////////////////// local tables ///////////////////////////////

// These three tables specify the characteristics of the decorrelation filters.
// Each term represents one layer of the sequential filter, where positive
// values indicate the relative sample involved from the same channel (1=prev)
// while -1 and -2 indicate cross channel decorrelation (in stereo only). The
// "simple_terms" table is no longer used for writing, but is kept for older
// file decoding.

static const char extreme_terms [] = { 1,1,1,2,4,-1,1,2,3,6,-2,8,5,7,4,1,2,3 };
static const char default_terms [] = { 1,1,1,-1,2,1,-2 };
static const char simple_terms []  = { 1,1,1,1 };

//////////////////////////////////////////////////////////////////////////////
// This function initializes everything required to unpack WavPack          //
// bitstreams and must be called before any unpacking is performed. Note    //
// that the (WavpackHeader *) in the WavpackContext struct must be valid.   //
//////////////////////////////////////////////////////////////////////////////

void unpack_init (WavpackContext *wpc)
{
    int flags = wpc->wphdr->flags;
    struct decorr_pass *dpp;
    int ti;

    CLEAR (wpc->decorr_passes);
    CLEAR (wpc->dc);

    if (flags & EXTREME_DECORR) {
	for (dpp = wpc->decorr_passes, ti = 0; ti < sizeof (extreme_terms); ti++)
	    if (extreme_terms [sizeof (extreme_terms) - ti - 1] > 0 || (flags & CROSS_DECORR))
		dpp++->term = extreme_terms [sizeof (extreme_terms) - ti - 1];
    }
    else if (flags & NEW_DECORR_FLAG) {
	for (dpp = wpc->decorr_passes, ti = 0; ti < sizeof (default_terms); ti++)
	    if (default_terms [sizeof (default_terms) - ti - 1] > 0 || (flags & CROSS_DECORR))
		dpp++->term = default_terms [sizeof (default_terms) - ti - 1];
    }
    else
	for (dpp = wpc->decorr_passes, ti = 0; ti < sizeof (simple_terms); dpp++, ti++)
	    dpp->term = simple_terms [sizeof (simple_terms) - ti - 1];

    init_word4 (wpc);
    init_word3 (wpc);
    init_word1 (wpc);

#ifdef VER2
    init_word2 (wpc);
#endif
}

#ifdef SEEKING

//////////////////////////////////////////////////////////////////////////////
// This function returns the size (in bytes) required to save the unpacking //
// context. Note that the (WavpackHeader *) in the WavpackContext struct    //
// must be valid.                                                           //
//////////////////////////////////////////////////////////////////////////////

int unpack_size (WavpackContext *wpc)
{
    int flags = wpc->wphdr->flags, byte_sum = 0;
    struct decorr_pass *dpp;

    byte_sum += sizeof (wpc->inbits);

    if (flags & WVC_FLAG)
	byte_sum += sizeof (wpc->in2bits);

    if (wpc->wphdr->version == 3) {
	if (wpc->wphdr->bits)
	    byte_sum += sizeof (wpc->w4);
	else
	    byte_sum += sizeof (wpc->w1);

	byte_sum += sizeof (wpc->w3) + sizeof (wpc->dc.crc);
    }
    else
	byte_sum += sizeof (wpc->w2);

    if (wpc->wphdr->bits)
	byte_sum += sizeof (wpc->dc.error);
    else
	byte_sum += sizeof (wpc->dc.sum_level) + sizeof (wpc->dc.left_level) +
	    sizeof (wpc->dc.right_level) + sizeof (wpc->dc.diff_level);

    if (flags & OVER_20)
	byte_sum += sizeof (wpc->dc.last_extra_bits) + sizeof (wpc->dc.extra_bits_count);

    if (!(flags & EXTREME_DECORR)) {
	byte_sum += sizeof (wpc->dc.sample);
	byte_sum += sizeof (wpc->dc.weight);

#ifndef COMPACT
	byte_sum += sizeof (wpc->dc.dsample);
	byte_sum += sizeof (wpc->dc.csample);
	byte_sum += sizeof (wpc->dc.cweight);
#endif
    }

    if (flags & (HIGH_FLAG | NEW_HIGH_FLAG))
	for (dpp = wpc->decorr_passes;; dpp++)
	    if (dpp->term > 0) {
		byte_sum += sizeof (dpp->samples_A [0]) * dpp->term;
		byte_sum += sizeof (dpp->weight_A);

		if (!(flags & MONO_FLAG)) {
		    byte_sum += sizeof (dpp->samples_B [0]) * dpp->term;
		    byte_sum += sizeof (dpp->weight_B);
		}
	    }
	    else if (dpp->term < 0) {
		byte_sum += sizeof (dpp->samples_A [0]) + sizeof (dpp->samples_B [0]);
		byte_sum += sizeof (dpp->weight_A) + sizeof (dpp->weight_B);
	    }
	    else
		break;

    return byte_sum;
}

//////////////////////////////////////////////////////////////////////////////
// This function saves the unpacking context at the specified pointer and   //
// returns the updated pointer. The actual amount of data required can be   //
// determined beforehand by calling unpack_size() but must be allocated by  //
// the caller.                                                              //
//////////////////////////////////////////////////////////////////////////////

void *unpack_save (WavpackContext *wpc, void *destin)
{
    int flags = wpc->wphdr->flags;
    struct decorr_pass *dpp;

    SAVE (destin, wpc->inbits);

    if (flags & WVC_FLAG)
	SAVE (destin, wpc->in2bits);

    if (wpc->wphdr->version == 3) {
	if (wpc->wphdr->bits) {
	    SAVE (destin, wpc->w4);
	}
	else {
	    SAVE (destin, wpc->w1);
	}

	SAVE (destin, wpc->w3);
	SAVE (destin, wpc->dc.crc);
    }
    else
	SAVE (destin, wpc->w2);

    if (wpc->wphdr->bits) {
	SAVE (destin, wpc->dc.error);
    }
    else {
	SAVE (destin, wpc->dc.sum_level);
	SAVE (destin, wpc->dc.left_level);
	SAVE (destin, wpc->dc.right_level);
	SAVE (destin, wpc->dc.diff_level);
    }

    if (flags & OVER_20) {
	SAVE (destin, wpc->dc.last_extra_bits);
	SAVE (destin, wpc->dc.extra_bits_count);
    }

    if (!(flags & EXTREME_DECORR)) {
	SAVE (destin, wpc->dc.sample);
	SAVE (destin, wpc->dc.weight);

#ifndef COMPACT
	SAVE (destin, wpc->dc.dsample);
	SAVE (destin, wpc->dc.csample);
	SAVE (destin, wpc->dc.cweight);
#endif
    }

    if (flags & (HIGH_FLAG | NEW_HIGH_FLAG))
	for (dpp = wpc->decorr_passes;; dpp++)
	    if (dpp->term > 0) {
		int count = dpp->term;
		int index = wpc->dc.m;

		SAVE (destin, dpp->weight_A);

		while (count--) {
		    SAVE (destin, dpp->samples_A [index]);
		    index = (index + 1) & (MAX_TERM - 1);
		}

		if (!(flags & MONO_FLAG)) {
		    count = dpp->term;
		    index = wpc->dc.m;

		    SAVE (destin, dpp->weight_B);

		    while (count--) {
			SAVE (destin, dpp->samples_B [index]);
			index = (index + 1) & (MAX_TERM - 1);
		    }
		}
	    }
	    else if (dpp->term < 0) {
		SAVE (destin, dpp->weight_A);
		SAVE (destin, dpp->weight_B);
		SAVE (destin, dpp->samples_A [0]);
		SAVE (destin, dpp->samples_B [0]);
	    }
	    else
		break;

    return destin;
}

//////////////////////////////////////////////////////////////////////////////
// This function restores the unpacking context from the specified pointer  //
// and returns the updated pointer. After this call, unpack_samples() will  //
// continue where it left off immediately before unpack_save() was called.  //
// If the WavPack files and bitstreams might have been closed and reopened, //
// then the "keep_resources" flag should be set to avoid using the "old"    //
// resources that were originally saved (and are probably now invalid).     //
//////////////////////////////////////////////////////////////////////////////

void *unpack_restore (WavpackContext *wpc, void *source, int keep_resources)
{
    int flags = wpc->wphdr->flags;
    struct decorr_pass *dpp;
    HANDLE temp_file;
    uchar *temp_buf;

    unpack_init (wpc);
    temp_file = wpc->inbits.file;
    temp_buf = wpc->inbits.buf;
    RESTORE (wpc->inbits, source);

    if (keep_resources) {
	wpc->inbits.file = temp_file;
	wpc->inbits.ptr += temp_buf - wpc->inbits.buf;
	wpc->inbits.end += temp_buf - wpc->inbits.buf;
	wpc->inbits.buf = temp_buf;
    }

    bs_restore (&wpc->inbits);

    if (flags & WVC_FLAG) {
	temp_file = wpc->in2bits.file;
	temp_buf = wpc->in2bits.buf;
	RESTORE (wpc->in2bits, source);

	if (keep_resources) {
	    wpc->in2bits.file = temp_file;
	    wpc->in2bits.ptr += temp_buf - wpc->in2bits.buf;
	    wpc->in2bits.end += temp_buf - wpc->in2bits.buf;
	    wpc->in2bits.buf = temp_buf;
	}

	bs_restore (&wpc->in2bits);
    }

    if (wpc->wphdr->version == 3) {
	if (wpc->wphdr->bits) {
	    RESTORE (wpc->w4, source);
	}
	else {
	    RESTORE (wpc->w1, source);
	}

	RESTORE (wpc->w3, source);
	RESTORE (wpc->dc.crc, source);
    }
    else
	RESTORE (wpc->w2, source);

    if (wpc->wphdr->bits) {
	RESTORE (wpc->dc.error, source);
    }
    else {
	RESTORE (wpc->dc.sum_level, source);
	RESTORE (wpc->dc.left_level, source);
	RESTORE (wpc->dc.right_level, source);
	RESTORE (wpc->dc.diff_level, source);
    }

    if (flags & OVER_20) {
	RESTORE (wpc->dc.last_extra_bits, source);
	RESTORE (wpc->dc.extra_bits_count, source);
    }

    if (!(flags & EXTREME_DECORR)) {
	RESTORE (wpc->dc.sample, source);
	RESTORE (wpc->dc.weight, source);

#ifndef COMPACT
	RESTORE (wpc->dc.dsample, source);
	RESTORE (wpc->dc.csample, source);
	RESTORE (wpc->dc.cweight, source);
#endif
    }

    if (flags & (HIGH_FLAG | NEW_HIGH_FLAG))
	for (dpp = wpc->decorr_passes;; dpp++)
	    if (dpp->term > 0) {
		int count = dpp->term;
		int index = wpc->dc.m;

		RESTORE (dpp->weight_A, source);

		while (count--) {
		    RESTORE (dpp->samples_A [index], source);
		    index = (index + 1) & (MAX_TERM - 1);
		}

		if (!(flags & MONO_FLAG)) {
		    count = dpp->term;
		    index = wpc->dc.m;

		    RESTORE (dpp->weight_B, source);

		    while (count--) {
			RESTORE (dpp->samples_B [index], source);
			index = (index + 1) & (MAX_TERM - 1);
		    }
		}
	    }
	    else if (dpp->term < 0) {
		RESTORE (dpp->weight_A, source);
		RESTORE (dpp->weight_B, source);
		RESTORE (dpp->samples_A [0], source);
		RESTORE (dpp->samples_B [0], source);
	    }
	    else
		break;

    return source;
}

#endif


///////////////////////////////////////////////////////////////////////////////
// This monster actually unpacks the WavPack bitstream(s) into the specified //
// buffer as either 16-bit or 24-bit values. The function unpack_init() must //
// have been called and the bitstreams must have already been opened with    //
// bs_open_read(). For maximum clarity, the function is broken up into       //
// segments that handle various modes. This makes for a few extra infrequent //
// flag checks, but makes the code easier to follow because the nesting does //
// not become so deep. For maximum efficiency, the conversion is isolated to //
// tight loops that handle an entire buffer.                                 //
///////////////////////////////////////////////////////////////////////////////

#define inbits (&wpc->inbits)

long unpack_samples (WavpackContext *wpc, void *buffer, uint sample_count)
{
    int shift = wpc->wphdr->shift, flags = wpc->wphdr->flags, min_weight = 0, m = wpc->dc.m;
    long min_value, max_value, min_shifted, max_shifted;
    long correction [2], crc = wpc->dc.crc;
    struct decorr_pass *dpp;
    long read_word;
    uchar *bptr;
    uint i, j;

#ifdef COMPACT
    long sample [2] [2];
    int weight [2] [1];
#else
    long sample [2] [5], dsample [2], csample [2];
    int weight [2] [5], cweight [4];
#endif

#ifdef COMPACT
    memcpy (sample, wpc->dc.sample, sizeof (sample));
    memcpy (weight, wpc->dc.weight, sizeof (weight));
#else
    memcpy (sample, wpc->dc.sample, sizeof (sample));
    memcpy (dsample, wpc->dc.dsample, sizeof (dsample));
    memcpy (csample, wpc->dc.csample, sizeof (csample));
    memcpy (weight, wpc->dc.weight, sizeof (weight));
    memcpy (cweight, wpc->dc.cweight, sizeof (cweight));
#endif

    if (wpc->wphdr->bits) {
	if (flags & (NEW_DECORR_FLAG | EXTREME_DECORR))
	    min_weight = -256;
    }
    else
	if (flags & NEW_DECORR_FLAG)
	    min_weight = (flags & EXTREME_DECORR) ? -512 : -256;

    if (flags & BYTES_3) {
	min_shifted = (min_value = -8388608 >> shift) << shift;
	max_shifted = (max_value = 8388607 >> shift) << shift;
    }
    else {
	min_value = -32768 >> shift;
	max_value = 32767 >> shift;
    }

    ///////////////// handle version 3 lossless mono data /////////////////////

    if (wpc->wphdr->version == 3 && !wpc->wphdr->bits && (flags & MONO_FLAG)) {
	if (flags & FAST_FLAG) {
	    if (flags & OVER_20)
		for (bptr = buffer, i = 0; i < sample_count; bptr += 3, ++i) {
		    long temp;

		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    sample [0] [0] += sample [0] [1] += read_word;
		    getbits (&temp, 4, inbits);
		    crc = crc * 3 + (temp = (temp & 0xf) + (sample [0] [0] << 4));
		    PUT_24 (bptr, temp);
		}
	    else if (flags & BYTES_3)
		for (bptr = buffer, i = 0; i < sample_count; bptr += 3, ++i) {
		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    crc = crc * 3 + (sample [0] [0] += sample [0] [1] += read_word);
		    PUT_24 (bptr, sample [0] [0] << shift);
		}
	    else
		for (bptr = buffer, i = 0; i < sample_count; ++i) {

		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    crc = crc * 3 + (*((short*) bptr)++ = sample [0] [0] += sample [0] [1] += read_word);
		}
	}
	else if (flags & HIGH_FLAG)
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long temp;

		if (flags & NEW_HIGH_FLAG) {
		    if ((read_word = get_word1 (wpc, 0)) == WORD_EOF)
			break;
		}
		else {
		    if ((read_word = get_old_word1 (wpc, 0)) == WORD_EOF)
			break;
		}

		if (flags & EXTREME_DECORR) {
		    for (dpp = wpc->decorr_passes; dpp->term; dpp++) {
			long sam = dpp->samples_A [m];

			temp = apply_weight (9, dpp->weight_A, sam) + read_word;
			update_weight (9, dpp->weight_A, sam, read_word);
			dpp->samples_A [(m + dpp->term) & (MAX_TERM - 1)] = read_word = temp;
		    }

		    m = (m + 1) & (MAX_TERM - 1);
		}
		else {
#ifdef COMPACT
		    for (dpp = wpc->decorr_passes; dpp->term; dpp++) {
			long sam = dpp->samples_A [m];

			temp = apply_weight (8, dpp->weight_A, sam) + read_word;
			update_weight (8, dpp->weight_A, sam, read_word);
			dpp->samples_A [(m + dpp->term) & (MAX_TERM - 1)] = read_word = temp;
		    }

		    m = (m + 1) & (MAX_TERM - 1);
#else
		    temp = apply_weight (8, weight [0] [0], sample [0] [0]) + read_word;
		    update_weight (8, weight [0] [0], sample [0] [0], read_word);

		    if (flags & NEW_DECORR_FLAG) {
			temp = apply_weight (8, weight [0] [1], sample [0] [1]) + (sample [0] [0] = temp);
			update_weight (8, weight [0] [1], sample [0] [1], sample [0] [0]);
			sample [0] [1] = dsample [0];
			temp = apply_weight (8, weight [0] [2], sample [0] [2]) + (dsample [0] = temp);
			update_weight (8, weight [0] [2], sample [0] [2], dsample [0]);
		    }
		    else {
			temp = apply_weight (8, weight [0] [2], sample [0] [2]) + (sample [0] [0] = temp);
			update_weight (8, weight [0] [2], sample [0] [2], sample [0] [0]);
		    }

		    temp = apply_weight (8, weight [0] [3], sample [0] [3]) + (sample [0] [2] = temp);
		    update_weight (8, weight [0] [3], sample [0] [3], sample [0] [2]);
		    temp = apply_weight (8, weight [0] [4], sample [0] [4]) + (sample [0] [3] = temp);
		    update_weight (8, weight [0] [4], sample [0] [4], sample [0] [3]);
		    sample [0] [4] = read_word = temp;
#endif
		}

		if (flags & OVER_20) {
		    if (wpc->dc.extra_bits_count < 8 || !getbit (inbits)) {
			getbits (&temp, 4, inbits);

			if ((temp &= 0xf) != wpc->dc.last_extra_bits) {
			    wpc->dc.last_extra_bits = temp;
			    wpc->dc.extra_bits_count = 0;
			}
			else
			    ++wpc->dc.extra_bits_count;
		    }

		    crc = crc * 3 + (temp = wpc->dc.last_extra_bits + (read_word << 4));
		    PUT_24 (bptr, temp);
		    bptr += 3;
		}
		else if (flags & BYTES_3) {
		    crc = crc * 3 + read_word;
		    PUT_24 (bptr, read_word << shift);
		    bptr += 3;
		}
		else
		    crc = crc * 3 + (*((short*) bptr)++ = read_word);
	    }
	else
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {

		long temp;

		if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
		    break;

		temp = sample [0] [0] + ((sample [0] [1] * weight [0] [0] + 128) >> 8) + read_word;

		if ((sample [0] [1] >= 0) == (read_word > 0)) {
		    if (weight [0] [0]++ == 256)
			weight [0] [0]--;
		}
		else if (weight [0] [0]-- == 0)
		    weight [0] [0]++;

		sample [0] [0] += (sample [0] [1] = temp - sample [0] [0]);

		if (flags & OVER_20) {
		    if (wpc->dc.extra_bits_count < 8 || !getbit (inbits)) {
			getbits (&temp, 4, inbits);

			if ((temp &= 0xf) != wpc->dc.last_extra_bits) {
			    wpc->dc.last_extra_bits = temp;
			    wpc->dc.extra_bits_count = 0;
			}
			else
			    ++wpc->dc.extra_bits_count;
		    }

		    crc = crc * 3 + (temp = wpc->dc.last_extra_bits + (sample [0] [0] << 4));
		    PUT_24 (bptr, temp);
		    bptr += 3;
		}
		else if (flags & BYTES_3) {
		    crc = crc * 3 + sample [0] [0];
		    PUT_24 (bptr, sample [0] [0] << shift);
		    bptr += 3;
		}
		else
		    crc = crc * 3 + (*((short*) bptr)++ = sample [0] [0]);
	    }

	if (shift && !(flags & BYTES_3))
	    for (bptr = buffer, j = 0; j < i; ++j)
		*((short*) bptr)++ <<= shift;
    }

    //////////////// handle version 3 lossless stereo data ////////////////////

    else if (wpc->wphdr->version == 3 && !wpc->wphdr->bits && !(flags & MONO_FLAG)) {
	long left_level = wpc->dc.left_level, right_level = wpc->dc.right_level;
	long sum_level = wpc->dc.sum_level, diff_level = wpc->dc.diff_level;

	if (flags & FAST_FLAG) {
	    if (flags & OVER_20)
		for (bptr = buffer, i = 0; i < sample_count; ++i) {
		    long sum, diff, temp, lbuf, rbuf;

		    read_word = get_word3 (wpc, 0);

		    if (read_word == WORD_EOF)
			break;

		    sum = (read_word << 1) | ((diff = get_word3 (wpc, 1)) & 1);
		    sample [0] [0] += sample [0] [1] += ((sum + diff) >> 1);
		    sample [1] [0] += sample [1] [1] += ((sum - diff) >> 1);
		    getbits (&temp, 8, inbits);
		    crc = crc * 3 + (lbuf = (sample [0] [0] << 4) + ((temp >> 4) & 0xf));
		    crc = crc * 3 + (rbuf = (sample [1] [0] << 4) + (temp & 0xf));
		    PUT_24 (bptr, lbuf);
		    bptr += 3;
		    PUT_24 (bptr, rbuf);
		    bptr += 3;
		}
	    else if (flags & BYTES_3)
		for (bptr = buffer, i = 0; i < sample_count; ++i) {
		    long sum, diff;

		    read_word = get_word3 (wpc, 0);

		    if (read_word == WORD_EOF)
			break;

		    sum = (read_word << 1) | ((diff = get_word3 (wpc, 1)) & 1);
		    sample [0] [1] += ((sum + diff) >> 1);
		    sample [1] [1] += ((sum - diff) >> 1);
		    crc = crc * 3 + (sample [0] [0] += sample [0] [1]);
		    crc = crc * 3 + (sample [1] [0] += sample [1] [1]);
		    PUT_24 (bptr, sample [0] [0] << shift);
		    bptr += 3;
		    PUT_24 (bptr, sample [1] [0] << shift);
		    bptr += 3;
		}
	    else
		for (bptr = buffer, i = 0; i < sample_count; ++i) {
		    long sum, diff;

		    read_word = get_word3 (wpc, 0);

		    if (read_word == WORD_EOF)
			break;

		    sum = (read_word << 1) | ((diff = get_word3 (wpc, 1)) & 1);
		    sample [0] [1] += ((sum + diff) >> 1);
		    sample [1] [1] += ((sum - diff) >> 1);
		    crc = crc * 3 + (*((short*) bptr)++ = sample [0] [0] += sample [0] [1]);
		    crc = crc * 3 + (*((short*) bptr)++ = sample [1] [0] += sample [1] [1]);
		}
	}
	else if (flags & HIGH_FLAG) {
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long sum, left, right, diff, left2, right2, extra_bits, next_word;

		if (flags & CROSS_DECORR) {
		    left = get_word1 (wpc, 0);

		    if (left == WORD_EOF)
			break;

		    right = get_word1 (wpc, 1);
		}
		else {
		    if (flags & NEW_HIGH_FLAG) {
			read_word = get_word1 (wpc, 0);

			if (read_word == WORD_EOF)
			    break;

			next_word = get_word1 (wpc, 1);

			if (right_level > left_level) {
			    if (left_level + right_level < sum_level + diff_level && right_level < diff_level) {
				sum = (right = read_word) + (left = next_word);
				diff = left - right;
			    }
			    else {
				diff = read_word;

				if (sum_level < left_level) {
				    sum = (next_word << 1) | (diff & 1);
				    left = (sum + diff) >> 1;
				    right = (sum - diff) >> 1;
				}
				else
				    sum = left + (right = (left = next_word) - diff);
			    }
			}
			else {
			    if (left_level + right_level < sum_level + diff_level && left_level < diff_level) {
				sum = (left = read_word) + (right = next_word);
				diff = left - right;
			    }
			    else {
				diff = read_word;

				if (sum_level < right_level) {
				    sum = (next_word << 1) | (diff & 1);
				    left = (sum + diff) >> 1;
				    right = (sum - diff) >> 1;
				}
				else
				    sum = (left = diff + (right = next_word)) + right;
			    }
			}
		    }
		    else {
			read_word = get_old_word1 (wpc, 0);

			if (read_word == WORD_EOF)
			    break;

			next_word = get_old_word1 (wpc, 1);

			if (sum_level <= right_level && sum_level <= left_level) {
			    sum = (next_word << 1) | (read_word & 1);
			    left = (sum + read_word) >> 1;
			    right = (sum - read_word) >> 1;
			}
			else if (left_level <= right_level)
			    sum = left + (right = (left = next_word) - read_word);
			else
			    sum = right + (left = read_word + (right = next_word));

			diff = left - right;
		    }

		    sum_level = sum_level - (sum_level >> 8) + labs (sum >> 1);
		    left_level = left_level - (left_level >> 8) + labs (left);
		    right_level = right_level - (right_level >> 8) + labs (right);
		    diff_level = diff_level - (diff_level >> 8) + labs (diff);

		    if (flags & JOINT_STEREO) {
			left = diff;
			right = sum >> 1;
		    }
		}

		if (flags & EXTREME_DECORR) {
		    for (dpp = wpc->decorr_passes;; dpp++)
			if (dpp->term > 0) {
			    long sam_A = dpp->samples_A [m], sam_B = dpp->samples_B [m];
			    int k = (m + dpp->term) & (MAX_TERM - 1);

			    left2 = apply_weight (9, dpp->weight_A, sam_A) + left;
			    right2 = apply_weight (9, dpp->weight_B, sam_B) + right;

			    update_weight (9, dpp->weight_A, sam_A, left);
			    update_weight (9, dpp->weight_B, sam_B, right);

			    dpp->samples_A [k] = left = left2;
			    dpp->samples_B [k] = right = right2;
			}
			else if (dpp->term == -1) {
			    left2 = left + apply_weight (9, dpp->weight_A, dpp->samples_A [0]);
			    update_weight (9, dpp->weight_A, dpp->samples_A [0], left);
			    left = left2;
			    right2 = right + apply_weight (9, dpp->weight_B, left);
			    update_weight (9, dpp->weight_B, left, right);
			    dpp->samples_A [0] = right = right2;
			}
			else if (dpp->term == -2) {
			    right2 = right + apply_weight (9, dpp->weight_A, dpp->samples_A [0]);
			    update_weight (9, dpp->weight_A, dpp->samples_A [0], right);
			    right = right2;
			    left2 = left + apply_weight (9, dpp->weight_B, right);
			    update_weight (9, dpp->weight_B, right, left);
			    dpp->samples_A [0] = left = left2;
			}
			else
			    break;

		    m = (m + 1) & (MAX_TERM - 1);
		}
		else {
#ifdef COMPACT
		    for (dpp = wpc->decorr_passes;; dpp++)
			if (dpp->term > 0) {
			    long sam_A = dpp->samples_A [m], sam_B = dpp->samples_B [m];
			    int k = (m + dpp->term) & (MAX_TERM - 1);

			    left2 = apply_weight (8, dpp->weight_A, sam_A) + left;
			    right2 = apply_weight (8, dpp->weight_B, sam_B) + right;

			    update_weight (8, dpp->weight_A, sam_A, left);
			    update_weight (8, dpp->weight_B, sam_B, right);

			    dpp->samples_A [k] = left = left2;
			    dpp->samples_B [k] = right = right2;
			}
			else if (dpp->term == -1) {
			    left2 = left + apply_weight (8, dpp->weight_A, dpp->samples_A [0]);
			    update_weight (8, dpp->weight_A, dpp->samples_A [0], left);
			    left = left2;
			    right2 = right + apply_weight (8, dpp->weight_B, left);
			    update_weight (8, dpp->weight_B, left, right);
			    dpp->samples_A [0] = right = right2;
			}
			else if (dpp->term == -2) {
			    right2 = right + apply_weight (8, dpp->weight_A, dpp->samples_A [0]);
			    update_weight (8, dpp->weight_A, dpp->samples_A [0], right);
			    right = right2;
			    left2 = left + apply_weight (8, dpp->weight_B, right);
			    update_weight (8, dpp->weight_B, right, left);
			    dpp->samples_A [0] = left = left2;
			}
			else
			    break;

		    m = (m + 1) & (MAX_TERM - 1);
#else
		    if (flags & CROSS_DECORR) {
			right2 = right + apply_weight (8, cweight [0], csample [0]);
			update_weight (8, cweight [0], csample [0], right);
			right = right2;
			left2 = left + apply_weight (8, cweight [1], right);
			update_weight (8, cweight [1], right, left);
			csample [0] = left = left2;
		    }

		    left2 = apply_weight (8, weight [0] [0], sample [0] [0]) + left;
		    right2 = apply_weight (8, weight [1] [0], sample [1] [0]) + right;

		    update_weight (8, weight [0] [0], sample [0] [0], left);
		    update_weight (8, weight [1] [0], sample [1] [0], right);

		    if (flags & NEW_DECORR_FLAG) {
			left = apply_weight (8, weight [0] [1], sample [0] [1]) + (sample [0] [0] = left2);
			right = apply_weight (8, weight [1] [1], sample [1] [1]) + (sample [1] [0] = right2);

			update_weight (8, weight [0] [1], sample [0] [1], sample [0] [0]);
			update_weight (8, weight [1] [1], sample [1] [1], sample [1] [0]);

			sample [0] [1] = dsample [0];
			sample [1] [1] = dsample [1];
			dsample [0] = left;
			dsample [1] = right;

			if (flags & CROSS_DECORR) {
			    left2 = left + apply_weight (8, cweight [2], csample [1]);
			    update_weight (8, cweight [2], csample [1], left);
			    left = left2;
			    right2 = right + apply_weight (8, cweight [3], left);
			    update_weight (8, cweight [3], left, right);
			    csample [1] = right = right2;
			}

			left2 = apply_weight (8, weight [0] [2], sample [0] [2]) + left;
			right2 = apply_weight (8, weight [1] [2], sample [1] [2]) + right;

			update_weight (8, weight [0] [2], sample [0] [2], left);
			update_weight (8, weight [1] [2], sample [1] [2], right);
		    }
		    else {
			left2 = apply_weight (8, weight [0] [2], sample [0] [2]) + (sample [0] [0] = left2);
			right2 = apply_weight (8, weight [1] [2], sample [1] [2]) + (sample [1] [0] = right2);

			update_weight (8, weight [0] [2], sample [0] [2], sample [0] [0]);
			update_weight (8, weight [1] [2], sample [1] [2], sample [1] [0]);
		    }

		    left2 = apply_weight (8, weight [0] [3], sample [0] [3]) + (sample [0] [2] = left2);
		    right2 = apply_weight (8, weight [1] [3], sample [1] [3]) + (sample [1] [2] = right2);

		    update_weight (8, weight [0] [3], sample [0] [3], sample [0] [2]);
		    update_weight (8, weight [1] [3], sample [1] [3], sample [1] [2]);

		    left2 = apply_weight (8, weight [0] [4], sample [0] [4]) + (sample [0] [3] = left2);
		    right2 = apply_weight (8, weight [1] [4], sample [1] [4]) + (sample [1] [3] = right2);

		    update_weight (8, weight [0] [4], sample [0] [4], sample [0] [3]);
		    update_weight (8, weight [1] [4], sample [1] [4], sample [1] [3]);

		    sample [0] [4] = left = left2;
		    sample [1] [4] = right = right2;
#endif
		}

		if (flags & JOINT_STEREO) {
		    sum = (right << 1) | ((diff = left) & 1);
		    right = (sum - diff) >> 1;
		    left = (sum + diff) >> 1;
		}

		if (flags & OVER_20) {
		    if (wpc->dc.extra_bits_count < 8 || !getbit (inbits)) {
			getbits (&extra_bits, 8, inbits);

			if ((extra_bits &= 0xff) != wpc->dc.last_extra_bits) {
			    wpc->dc.last_extra_bits = extra_bits;
			    wpc->dc.extra_bits_count = 0;
			}
			else
			    ++wpc->dc.extra_bits_count;
		    }

		    crc = crc * 3 + (left = (left << 4) + (wpc->dc.last_extra_bits >> 4));
		    crc = crc * 3 + (right = (right << 4) + (wpc->dc.last_extra_bits & 0xf));
		    PUT_24 (bptr, left);
		    bptr += 3;
		    PUT_24 (bptr, right);
		    bptr += 3;
		}
		else if (flags & BYTES_3) {
		    crc = crc * 3 + left;
		    PUT_24 (bptr, left << shift);
		    bptr += 3;
		    crc = crc * 3 + right;
		    PUT_24 (bptr, right << shift);
		    bptr += 3;
		}
		else {
		    crc = crc * 3 + (*((short*) bptr)++ = left);
		    crc = crc * 3 + (*((short*) bptr)++ = right);
		}
	    }
	}
	else
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long sum, left, right, left2, right2, extra_bits;

		read_word = get_word3 (wpc, 0);

		if (read_word == WORD_EOF)
		    break;

		if (sum_level <= right_level && sum_level <= left_level) {
		    sum = (get_word3 (wpc, 1) << 1) | (read_word & 1);
		    left = (sum + read_word) >> 1;
		    right = (sum - read_word) >> 1;
		}
		else if (left_level <= right_level)
		    sum = left + (right = (left = get_word3 (wpc, 1)) - read_word);
		else
		    sum = right + (left = read_word + (right = get_word3 (wpc, 1)));

		sum_level = sum_level - (sum_level >> 8) + labs (sum >> 1);
		left_level = left_level - (left_level >> 8) + labs (left);
		right_level = right_level - (right_level >> 8) + labs (right);

		left2 = sample [0] [0] + ((sample [0] [1] * weight [0] [0] + 128) >> 8) + left;
		right2 = sample [1] [0] + ((sample [1] [1] * weight [1] [0] + 128) >> 8) + right;

		if ((sample [0] [1] >= 0) == (left > 0)) {
		    if (weight [0] [0]++ == 256)
			weight [0] [0]--;
		}
		else if (weight [0] [0]-- == 0)
		    weight [0] [0]++;

		if ((sample [1] [1] >= 0) == (right > 0)) {
		    if (weight [1] [0]++ == 256)
			weight [1] [0]--;
		}
		else if (weight [1] [0]-- == 0)
		    weight [1] [0]++;

		sample [0] [0] += (sample [0] [1] = left2 - sample [0] [0]);
		sample [1] [0] += (sample [1] [1] = right2 - sample [1] [0]);

		if (flags & OVER_20) {
		    if (wpc->dc.extra_bits_count < 8 || !getbit (inbits)) {
			getbits (&extra_bits, 8, inbits);

			if ((extra_bits &= 0xff) != wpc->dc.last_extra_bits) {
			    wpc->dc.last_extra_bits = extra_bits;
			    wpc->dc.extra_bits_count = 0;
			}
			else
			    ++wpc->dc.extra_bits_count;
		    }

		    crc = crc * 3 + (left2 = (sample [0] [0] << 4) + (wpc->dc.last_extra_bits >> 4));
		    crc = crc * 3 + (right2 = (sample [1] [0] << 4) + (wpc->dc.last_extra_bits & 0xf));
		    PUT_24 (bptr, left2);
		    bptr += 3;
		    PUT_24 (bptr, right2);
		    bptr += 3;
		}
		else if (flags & BYTES_3) {
		    crc = crc * 3 + sample [0] [0];
		    PUT_24 (bptr, sample [0] [0] << shift);
		    bptr += 3;
		    crc = crc * 3 + sample [1] [0];
		    PUT_24 (bptr, sample [1] [0] << shift);
		    bptr += 3;
		}
		else {
		    crc = crc * 3 + (*((short*) bptr)++ = sample [0] [0]);
		    crc = crc * 3 + (*((short*) bptr)++ = sample [1] [0]);
		}
	    }

	if (shift && !(flags & BYTES_3))
	    for (j = 0; j < i; ++j) {
		*((short*) bptr)++ <<= shift;
		*((short*) bptr)++ <<= shift;
	    }

	wpc->dc.left_level = left_level;
	wpc->dc.right_level = right_level;
	wpc->dc.sum_level = sum_level;
	wpc->dc.diff_level = diff_level;
    }

    //////////////// handle version 3 lossy/hybrid mono data //////////////////

    else if (wpc->wphdr->version == 3 && wpc->wphdr->bits && (flags & MONO_FLAG)) {
	if (flags & FAST_FLAG) {
	    if (flags & BYTES_3)
		for (bptr = buffer, i = 0; i < sample_count; bptr += 3, ++i) {

		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    crc = crc * 3 + (sample [0] [0] += sample [0] [1] += read_word);

		    if (sample [0] [0] < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (sample [0] [0] > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, sample [0] [0] << shift);
		}
	    else
		for (bptr = buffer, i = 0; i < sample_count; ++i) {

		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    crc = crc * 3 + (sample [0] [0] += sample [0] [1] += read_word);

		    if (sample [0] [0] < min_value)
			*((short*) bptr)++ = min_value;
		    else if (sample [0] [0] > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = sample [0] [0];
		}
	}
	else if (flags & (HIGH_FLAG | NEW_HIGH_FLAG))
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long temp;

		read_word = (flags & NEW_HIGH_FLAG) ?
		    get_word4 (wpc, 0, correction) : get_word3 (wpc, 0);

		if (read_word == WORD_EOF)
		    break;

		if ((flags & BYTES_3) && shift < 4)
		    for (dpp = wpc->decorr_passes; dpp->term; dpp++) {
			long sam = dpp->samples_A [m];

			temp = apply_weight24 (dpp->weight_A, sam) + read_word;
			update_weight2 (dpp->weight_A, sam, read_word);
			dpp->samples_A [(m + dpp->term) & (MAX_TERM - 1)] = read_word = temp;
		    }
		else
		    for (dpp = wpc->decorr_passes; dpp->term; dpp++) {
			long sam = dpp->samples_A [m];

			temp = apply_weight (8, dpp->weight_A, sam) + read_word;
			update_weight2 (dpp->weight_A, sam, read_word);
			dpp->samples_A [(m + dpp->term) & (MAX_TERM - 1)] = read_word = temp;
		    }

		m = (m + 1) & (MAX_TERM - 1);

		if (flags & WVC_FLAG) {
		    if (flags & LOSSY_SHAPE) {
			crc = crc * 3 + (read_word += correction [0] + wpc->dc.error [0]);
			wpc->dc.error [0] = -correction [0];
		    }
		    else
			crc = crc * 3 + (read_word += correction [0]);

		    if (flags & BYTES_3) {
			PUT_24 (bptr, read_word << shift);
			bptr += 3;
		    }
		    else
			*((short*) bptr)++ = read_word;
		}
		else {
		    crc = crc * 3 + read_word;

		    if (flags & BYTES_3) {
			if (read_word < min_value)
			    PUT_24 (bptr, min_shifted);
			else if (read_word > max_value)
			    PUT_24 (bptr, max_shifted);
			else
			    PUT_24 (bptr, read_word << shift);

			bptr += 3;
		    }
		    else {
			if (read_word < min_value)
			    *((short*) bptr)++ = min_value;
			else if (read_word > max_value)
			    *((short*) bptr)++ = max_value;
			else
			    *((short*) bptr)++ = read_word;
		    }
		}
	    }
	else
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long new_sample;

		if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
		    break;

		new_sample = sample [0] [0] + ((sample [0] [1] * weight [0] [0] + 128) >> 8) + read_word;

		if ((sample [0] [1] >= 0) == (read_word > 0)) {
		    if (weight [0] [0]++ == 256)
			weight [0] [0]--;
		}
		else if (weight [0] [0]-- == 0)
		    weight [0] [0]++;

		sample [0] [1] = new_sample - sample [0] [0];
		crc = crc * 3 + (sample [0] [0] = new_sample);

		if (flags & BYTES_3) {
		    if (sample [0] [0] < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (sample [0] [0] > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, sample [0] [0] << shift);

		    bptr += 3;
		}
		else {
		    if (sample [0] [0] < min_value)
			*((short*) bptr)++ = min_value;
		    else if (sample [0] [0] > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = sample [0] [0];
		}
	    }

	if (shift && !(flags & BYTES_3))
	    for (bptr = buffer, j = 0; j < i; ++j)
		*((short*) bptr)++ <<= shift;
    }

    //////////////// handle version 3 lossy/hybrid stereo data ////////////////

    else if (wpc->wphdr->version == 3 && wpc->wphdr->bits && !(flags & MONO_FLAG)) {
	if (flags & FAST_FLAG) {
	    if (flags & BYTES_3)
		for (bptr = buffer, i = 0; i < sample_count; ++i) {

		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    crc = crc * 3 + (sample [0] [0] += sample [0] [1] += read_word);

		    if (sample [0] [0] < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (sample [0] [0] > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, sample [0] [0] << shift);

		    bptr += 3;
		    crc = crc * 3 + (sample [1] [0] += sample [1] [1] += get_word3 (wpc, 1));

		    if (sample [1] [0] < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (sample [1] [0] > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, sample [1] [0] << shift);

		    bptr += 3;
		}
	    else
		for (bptr = buffer, i = 0; i < sample_count; ++i) {

		    if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
			break;

		    crc = crc * 3 + (sample [0] [0] += sample [0] [1] += read_word);

		    if (sample [0] [0] < min_value)
			*((short*) bptr)++ = min_value;
		    else if (sample [0] [0] > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = sample [0] [0];

		    crc = crc * 3 + (sample [1] [0] += sample [1] [1] += get_word3 (wpc, 1));

		    if (sample [1] [0] < min_value)
			*((short*) bptr)++ = min_value;
		    else if (sample [1] [0] > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = sample [1] [0];
		}
	}
	else if (flags & (HIGH_FLAG | NEW_HIGH_FLAG))
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long left, right, left2, right2, sum, diff;

		if (flags & NEW_HIGH_FLAG) {
		    left = get_word4 (wpc, 0, correction);
		    right = get_word4 (wpc, 1, correction + 1);
		}
		else {
		    left = get_word3 (wpc, 0);
		    right = get_word3 (wpc, 1);
		}

		if (left == WORD_EOF)
		    break;

		if ((flags & BYTES_3) && shift < 4)
		    for (dpp = wpc->decorr_passes; dpp->term; dpp++) {
			long sam_A = dpp->samples_A [m], sam_B = dpp->samples_B [m];
			int k = (m + dpp->term) & (MAX_TERM - 1);

			left2 = apply_weight24 (dpp->weight_A, sam_A) + left;
			update_weight2 (dpp->weight_A, sam_A, left);
			dpp->samples_A [k] = left = left2;

			right2 = apply_weight24 (dpp->weight_B, sam_B) + right;
			update_weight2 (dpp->weight_B, sam_B, right);
			dpp->samples_B [k] = right = right2;
		    }
		else
		    for (dpp = wpc->decorr_passes; dpp->term; dpp++) {
			long sam_A = dpp->samples_A [m], sam_B = dpp->samples_B [m];
			int k = (m + dpp->term) & (MAX_TERM - 1);

			left2 = apply_weight (8, dpp->weight_A, sam_A) + left;
			update_weight2 (dpp->weight_A, sam_A, left);
			dpp->samples_A [k] = left = left2;

			right2 = apply_weight (8, dpp->weight_B, sam_B) + right;
			update_weight2 (dpp->weight_B, sam_B, right);
			dpp->samples_B [k] = right = right2;
		    }

		m = (m + 1) & (MAX_TERM - 1);

		if (flags & WVC_FLAG) {
		    if (flags & LOSSY_SHAPE) {
			left += correction [0] + wpc->dc.error [0];
			right += correction [1] + wpc->dc.error [1];
			wpc->dc.error [0] = -correction [0];
			wpc->dc.error [1] = -correction [1];
		    }
		    else {
			left += correction [0];
			right += correction [1];
		    }
		}

		if (flags & JOINT_STEREO) {
		    right = ((sum = (right << 1) | (left & 1)) - (diff = left)) >> 1;
		    left = (sum + diff) >> 1;
		}

		crc = crc * 3 + left;
		crc = crc * 3 + right;

		if (flags & WVC_FLAG) {
		    if (flags & BYTES_3) {
			PUT_24 (bptr, left << shift);
			bptr += 3;
			PUT_24 (bptr, right << shift);
			bptr += 3;
		    }
		    else {
			*((short*) bptr)++ = left;
			*((short*) bptr)++ = right;
		    }
		}
		else if (flags & BYTES_3) {
		    if (left < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (left > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, left << shift);

		    bptr += 3;

		    if (right < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (right > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, right << shift);

		    bptr += 3;
		}
		else {
		    if (left < min_value)
			*((short*) bptr)++ = min_value;
		    else if (left > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = left;

		    if (right < min_value)
			*((short*) bptr)++ = min_value;
		    else if (right > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = right;
		}
	    }
	else
	    for (bptr = buffer, i = 0; i < sample_count; ++i) {
		long new_sample;

		if ((read_word = get_word3 (wpc, 0)) == WORD_EOF)
		    break;

		new_sample = sample [0] [0] + ((sample [0] [1] * weight [0] [0] + 128) >> 8) + read_word;

		if ((sample [0] [1] >= 0) == (read_word > 0)) {
		    if (weight [0] [0]++ == 256)
			weight [0] [0]--;
		}
		else if (weight [0] [0]-- == 0)
		    weight [0] [0]++;

		sample [0] [1] = new_sample - sample [0] [0];
		crc = crc * 3 + (sample [0] [0] = new_sample);

		read_word = get_word3 (wpc, 1);
		new_sample = sample [1] [0] + ((sample [1] [1] * weight [1] [0] + 128) >> 8) + read_word;

		if ((sample [1] [1] >= 0) == (read_word > 0)) {
		    if (weight [1] [0]++ == 256)
			weight [1] [0]--;
		}
		else if (weight [1] [0]-- == 0)
		    weight [1] [0]++;

		sample [1] [1] = new_sample - sample [1] [0];
		crc = crc * 3 + (sample [1] [0] = new_sample);

		if (flags & BYTES_3) {
		    if (sample [0] [0] < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (sample [0] [0] > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, sample [0] [0] << shift);

		    bptr += 3;

		    if (sample [1] [0] < min_value)
			PUT_24 (bptr, min_shifted);
		    else if (sample [1] [0] > max_value)
			PUT_24 (bptr, max_shifted);
		    else
			PUT_24 (bptr, sample [1] [0] << shift);

		    bptr += 3;
		}
		else {
		    if (sample [0] [0] < min_value)
			*((short*) bptr)++ = min_value;
		    else if (sample [0] [0] > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = sample [0] [0];

		    if (sample [1] [0] < min_value)
			*((short*) bptr)++ = min_value;
		    else if (sample [1] [0] > max_value)
			*((short*) bptr)++ = max_value;
		    else
			*((short*) bptr)++ = sample [1] [0];
		}
	    }

	if (shift && !(flags & BYTES_3))
	    for (j = 0; j < i; ++j) {
		*((short*) bptr)++ <<= shift;
		*((short*) bptr)++ <<= shift;
	    }
    }

    //////////// finally, handle version 1 & 2 data, if selected //////////////

#ifdef VER2
    else if (wpc->wphdr->version < 3 && (flags & MONO_FLAG))
	for (bptr = buffer, i = 0; i < sample_count; ++i) {
	    if ((read_word = get_word2 (wpc, 0)) == WORD_EOF)
		break;

	    sample [0] [0] += sample [0] [1] += read_word;

	    if (wpc->wphdr->bits) {
		if (sample [0] [0] < min_value)
		    sample [0] [0] = min_value;
		else if (sample [0] [0] > max_value)
		    sample [0] [0] = max_value;
	    }

	    *((short*) bptr)++ = sample [0] [0] << shift;
	}
    else if (wpc->wphdr->version < 3 && !(flags & MONO_FLAG))
	for (bptr = buffer, i = 0; i < sample_count; ++i) {
	    long sum, diff;

	    read_word = get_word2 (wpc, 0);

	    if (read_word == WORD_EOF)
		break;

	    sum = (read_word << 1) | ((diff = get_word2 (wpc, 1)) & 1);
	    sample [0] [0] += sample [0] [1] += ((sum + diff) >> 1);
	    sample [1] [0] += sample [1] [1] += ((sum - diff) >> 1);

	    if (wpc->wphdr->bits) {
		if (sample [0] [0] < min_value)
		    sample [0] [0] = min_value;
		else if (sample [0] [0] > max_value)
		    sample [0] [0] = max_value;

		if (sample [1] [0] < min_value)
		    sample [1] [0] = min_value;
		else if (sample [1] [0] > max_value)
		    sample [1] [0] = max_value;
	    }

	    *((short*) bptr)++ = sample [0] [0] << shift;
	    *((short*) bptr)++ = sample [1] [0] << shift;
	}
#endif

#ifdef COMPACT
    memcpy (wpc->dc.sample, sample, sizeof (sample));
    memcpy (wpc->dc.weight, weight, sizeof (weight));
#else
    memcpy (wpc->dc.sample, sample, sizeof (sample));
    memcpy (wpc->dc.dsample, dsample, sizeof (dsample));
    memcpy (wpc->dc.csample, csample, sizeof (csample));
    memcpy (wpc->dc.weight, weight, sizeof (weight));
    memcpy (wpc->dc.cweight, cweight, sizeof (cweight));
#endif

    wpc->dc.crc = crc;
    wpc->dc.m = m;

    return i;
}

//////////////////////////////////////////////////////////////////////////////
// This function returns the accumulated 32-bit CRC value. At the end of    //
// unpacking this should match the value stored in the WavPack header. Note //
// that WavPack's CRC is not a CCITT approved polynomial algorithm, but is  //
// a much simpler method that is virtually as robust for real world data.   //
//////////////////////////////////////////////////////////////////////////////

long unpack_crc (WavpackContext *wpc)
{
    return wpc->dc.crc;
}
