////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2002 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// utils.c

// This module provides general purpose utilities for the WavPack command-line
// utilities and the self-extraction module.

#include "wavpack.h"

#include <stdio.h>
#include <dos.h>
#include <conio.h>
#include <process.h>
#include <io.h>

#ifndef __WIN32__
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#endif

//////////////////////////////////////////////////////////////////////////////
// This function parses a filename (with or without full path) and returns  //
// a pointer to the extension (including the "."). If no extension is found //
// then NULL is returned. Extensions with more than 3 letters don't count.  //
//////////////////////////////////////////////////////////////////////////////

char *filespec_ext (char *filespec)
{
    char *cp = filespec + strlen (filespec);

    while (--cp >= filespec) {

	if (*cp == '\\' || *cp == ':')
	    return NULL;

	if (*cp == '.') {
	    if (strlen (cp) > 1 && strlen (cp) <= 4)
		return cp;
	    else
		return NULL;
	}
    }

    return NULL;
}

//////////////////////////////////////////////////////////////////////////////
// This function determines if the specified filespec is a valid pathname.  //
// If not, NULL is returned. If it is in the format of a pathname, then the //
// original pointer is returned. If the format is ambiguous, then a lookup  //
// is performed to determine if it is in fact a valid path, and if so a "\" //
// is appended so that the pathname can be used and the original pointer is //
// returned.                                                                //
//////////////////////////////////////////////////////////////////////////////

#ifdef __BORLANDC__

#include <dir.h>

char *filespec_path (char *filespec)
{
    char *cp = filespec + strlen (filespec);
    struct ffblk ffblk;

    if (cp == filespec || filespec_wild (filespec))
	return NULL;

    if (*--cp == '\\' || *cp == ':')
	return filespec;

    if (*cp == '.' && cp == filespec)
	return strcat (filespec, "\\");

    if (findfirst (filespec, &ffblk, FA_DIREC) == 0 &&
	findfirst (filespec, &ffblk, 0) == -1)
	    return strcat (filespec, "\\");

    return NULL;
}

#else

char *filespec_path (char *filespec)
{
    char *cp = filespec + strlen (filespec);
    struct _finddata_t _finddata_t;
    long file;

    if (cp == filespec || filespec_wild (filespec))
	return NULL;

    if (*--cp == '\\' || *cp == ':')
	return filespec;

    if (*cp == '.' && cp == filespec)
	return strcat (filespec, "\\");

    if ((file = _findfirst (filespec, &_finddata_t)) != -1L &&
	(_finddata_t.attrib & _A_SUBDIR)) {
	    _findclose (file);
	    return strcat (filespec, "\\");
    }

    return NULL;
}

#endif

//////////////////////////////////////////////////////////////////////////////
// This function returns non-NULL if the specified filename spec has any    //
// wildcard characters.                                                     //
//////////////////////////////////////////////////////////////////////////////

char *filespec_wild (char *filespec)
{
    return strpbrk (filespec, "*?");
}

//////////////////////////////////////////////////////////////////////////////
// This function parses a filename (with or without full path) and returns  //
// a pointer to the actual filename, or NULL if no filename can be found.   //
//////////////////////////////////////////////////////////////////////////////

char *filespec_name (char *filespec)
{
    char *cp = filespec + strlen (filespec);

    while (--cp >= filespec)
	if (*cp == '\\' || *cp == ':')
	    break;

#ifdef __WIN32__
    if (strlen (cp + 1))
#else
    if (strlen (cp + 1) && strlen (cp + 1) <= 12)
#endif
	return cp + 1;
    else
	return NULL;
}

//////////////////////////////////////////////////////////////////////////////
// This function returns 1 if the specified string has any lowercase chars. //
//////////////////////////////////////////////////////////////////////////////

char anylower (char *string)
{
    if (!string)
	return 0;

    while (*string)
	if (islower (*string))
	    return 1;
	else
	    string++;

    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// This function allows the user to type 'y', 'n', or 'a' (with Enter) in   //
// response to a system query. The return value is the key typed as         //
// lowercase (regardless of the typed case).                                //
//////////////////////////////////////////////////////////////////////////////

static int waiting_input;

char yna (void)
{
    char choice = 0, key;

    waiting_input = 1;

    while (1) {
	key = getch ();

	if (key == 3) {
	    fprintf (stderr, "^C\n");
	    exit (1);
	}
	else if (key == '\r') {
	    if (choice) {
		fprintf (stderr, "\n");
		break;
	    }
	    else
		fprintf (stderr, "%c", 7);
	}
	else if (key == 'Y' || key == 'y') {
	    fprintf (stderr, "%c\b", key);
	    choice = 'y';
	}
	else if (key == 'N' || key == 'n') {
	    fprintf (stderr, "%c\b", key);
	    choice = 'n';
	}
	else if (key == 'A' || key == 'a') {
	    fprintf (stderr, "%c\b", key);
	    choice = 'a';
	}
	else
	    fprintf (stderr, "%c", 7);
    }

    waiting_input = 0;

    return choice;
}

//////////////////////////////////////////////////////////////////////////////
// Display the specified message on the console through stderr. Note that   //
// the cursor may start anywhere in the line and all text already on the    //
// line is erased. A terminating newline is not needed and function works   //
// with printf strings and args.                                            //
//////////////////////////////////////////////////////////////////////////////

void error_line (char *error, ...)
{
    char error_msg [512];
    va_list argptr;

    error_msg [0] = '\r';
    va_start (argptr, error);
    vsprintf (error_msg + 1, error, argptr);
    va_end (argptr);
    fputs (error_msg, stderr);
    finish_line ();
}

static int break_flag;

#ifdef __WIN32__

//////////////////////////////////////////////////////////////////////////////
// Function to intercept ^C or ^Break typed at the console.                 //
//////////////////////////////////////////////////////////////////////////////

BOOL WINAPI ctrl_handler (DWORD ctrl)
{
    if (ctrl == CTRL_C_EVENT) {
	break_flag = TRUE;
	return TRUE;
    }

    if (ctrl == CTRL_BREAK_EVENT) {

	if (waiting_input) {
#ifdef __BORLANDC__
	    fprintf (stderr, "^C\n");
#endif
	    return FALSE;
	}
	else {
	    break_flag = TRUE;
	    return TRUE;
	}
    }

    return FALSE;
}

//////////////////////////////////////////////////////////////////////////////
// Function to determine Windows platform (1 = Win95/98/Me)                 //
//////////////////////////////////////////////////////////////////////////////

int get_platform (void)
{
    static OSVERSIONINFO VerInfo;

    if (!VerInfo.dwOSVersionInfoSize) {
	VerInfo.dwOSVersionInfoSize = sizeof (VerInfo);
	VerInfo.dwPlatformId = 0;
	GetVersionEx (&VerInfo);
    }

    return VerInfo.dwPlatformId;
}

//////////////////////////////////////////////////////////////////////////////
// Function to initialize console for intercepting ^C and ^Break.           //
//////////////////////////////////////////////////////////////////////////////

void setup_break (void)
{
    HANDLE hConIn = GetStdHandle (STD_INPUT_HANDLE);

    SetConsoleMode (hConIn, ENABLE_PROCESSED_INPUT);
    FlushConsoleInputBuffer (hConIn);
    SetConsoleCtrlHandler (ctrl_handler, TRUE);
    break_flag = 0;
}

//////////////////////////////////////////////////////////////////////////////
// Function to determine whether ^C or ^Break has been issued by user.      //
//////////////////////////////////////////////////////////////////////////////

int check_break (void)
{
    return break_flag;
}

//////////////////////////////////////////////////////////////////////////////
// Function to clear the stderr console to the end of the current line (and //
// go to the beginning next line).                                          //
//////////////////////////////////////////////////////////////////////////////

void finish_line (void)
{
    HANDLE hConIn = GetStdHandle (STD_ERROR_HANDLE);
    CONSOLE_SCREEN_BUFFER_INFO coninfo;

    if (hConIn && GetConsoleScreenBufferInfo (hConIn, &coninfo)) {
	char spaces = coninfo.dwSize.X - coninfo.dwCursorPosition.X;

	while (spaces--)
	    fputc (' ', stderr);
    }
    else
	fputc ('\n', stderr);
}

#else

//////////////////////////////////////////////////////////////////////////////
// Function to intercept ^C or ^Break typed at the console.                 //
//////////////////////////////////////////////////////////////////////////////

static void interrupt control_break ()
{
   break_flag = 1;
}

//////////////////////////////////////////////////////////////////////////////
// Function to initialize console for intercepting ^C and ^Break.           //
//////////////////////////////////////////////////////////////////////////////

void setup_break (void)
{
    break_flag = 0;
    setvect (0x23, control_break);
}

//////////////////////////////////////////////////////////////////////////////
// Function to determine whether ^C or ^Break has been issued by user. If   //
// we are reading from stdin, then we don't want to call kbhit() even       //
// it makes ^Break checking better.                                         //
//////////////////////////////////////////////////////////////////////////////

int check_break (int call_kbhit)
{
    if (call_kbhit)
	kbhit ();

    return break_flag;
}

//////////////////////////////////////////////////////////////////////////////
// Function to clear the console to the end of the current line (and go to  //
// the beginning next line).                                                //
//////////////////////////////////////////////////////////////////////////////

void finish_line (void)
{
    clreol ();
    putc ('\n', stderr);
}

#endif
