////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2003 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// wavpack.c

// This is the main module for the WavPack command-line compressor.

#include "wavpack.h"

#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <stdlib.h>
#include <dos.h>
#include <math.h>

#ifdef __BORLANDC__
#include <dir.h>
#else
#include <sys/types.h>
#include <sys/timeb.h>
#endif

#ifndef __WIN32__
#include <sys/stat.h>
#include <string.h>
#endif

///////////////////////////// local variable storage //////////////////////////

#ifdef __WIN32__
static const char *sign_on = "\n\
 WAVPACK  Hybrid Lossless Wavefile Compressor  Win32 Version %s %s\n\
 Copyright (c) 1998 - 2003 Conifer Software.  All Rights Reserved.\n\n";
#else
static const char *sign_on = "\n\
 WAVPACK  Hybrid Lossless Wavefile Compressor  DOS Version %s %s\n\
 Copyright (c) 1998 - 2003 Conifer Software.  All Rights Reserved.\n\n";
#endif

#ifdef __WIN32__
static const char *usage = "\
 Usage:   WAVPACK [-options] [@]infile[.wav]|- [[@]outfile[.wv]|outpath]\n\
             (infile may contain wildcards: ?,*)\n\n\
 Options: -bn = enable hybrid compression, 'n' is kbits/second (kbps)\n\
             (n = 24-4800) or bits/sample (n = 4-23); otherwise default\n\
             operation is pure lossless compression\n\
          -c  = create correction file (.wvc) for hybrid mode\n\
          -d  = delete source file if successful (use with caution!)\n\
          -e  = create self-extracting executable\n\
             (WVSELFX.EXE must be in the same directory as WAVPACK.EXE)\n\
          -f (or -ff) = fast (or very fast) lossless compression\n\
          -jn = joint-stereo override (where applicable, 0=LR/1=MS)\n\
          -h  = high quality (best compression in all modes, but slower)\n\
          -n  = calculate average and peak quantization noise (hybrid only)\n\
          -rn = raw mode for non-wav files (1=mono/2=stereo)\n\
          -sn = noise shaping override (hybrid only, 0=off/1=1st order)\n\
          -t  = copy input file's time stamp to output file\n\
          -y  = yes to all warnings (use with caution!)\n\n\
 Web:     Visit www.wavpack.com for latest version and info\n";
#else
static const char *usage = "\
 Usage:   WAVPACK [-options] [@]infile[.wav]|- [[@]outfile[.wv]|outpath]\n\
             (infile may contain wildcards: ?,*)\n\n\
 Options: -bn = enable hybrid compression, 'n' is kbits/second (kbps)\n\
             (n = 24-4800) or bits/sample (n = 4-23); otherwise default\n\
             operation is pure lossless compression\n\
          -c  = create correction file (.wvc) for hybrid mode\n\
          -d  = delete source file if successful (use with caution!)\n\
          -f (or -ff) = fast (or very fast) lossless compression\n\
          -jn = joint-stereo override (where applicable, 0=LR/1=MS)\n\
          -h  = high quality (best compression in all modes, but slower)\n\
          -n  = calculate average and peak quantization noise (hybrid only)\n\
          -rn = raw mode for non-wav files (1=mono/2=stereo)\n\
          -sn = noise shaping override (hybrid only, 0=off/1=1st order)\n\
          -t  = copy input file's time stamp to output file\n\
          -y  = yes to all warnings (use with caution!)\n\n\
 Web:     Visit www.wavpack.com for latest version and info\n";
#endif

static int overwrite_all, num_files, file_index;
static char *wvselfx_image;

/////////////////////////// local function declarations ///////////////////////

static int pack_file (char *infilename, char *outfilename, char *out2filename, int bits, long flags);
static int pack_audio (WavpackHeader *wphdr, HANDLE infile, HANDLE outfile, HANDLE out2file);
static void display_progress (double file_progress);

#define NO_ERROR 0L
#define SOFT_ERROR 1
#define HARD_ERROR 2

//////////////////////////////////////////////////////////////////////////////
// The "main" function for the command-line WavPack compressor.             //
//////////////////////////////////////////////////////////////////////////////

int main (argc, argv) int argc; char **argv;
{
    int bits = 0, delete_source = 0, usage_error = 0, filelist = 0;
    char *infilename = NULL, *outfilename = NULL, *out2filename;
    char *selfname = malloc (strlen (*argv) + 20);
    char **matches = NULL;
    long flags = 0;
    int result, i;

#ifdef __BORLANDC__
	struct ffblk ffblk;
#else
	struct _finddata_t _finddata_t;
#endif

    strcpy (selfname, *argv);
    fprintf (stderr, sign_on, VERSION_STR, DATE_STR);

    // loop through command-line arguments

    while (--argc)
	if ((**++argv == '-' || **argv == '/') && (*argv)[1])
	    while (*++*argv)
		switch (**argv) {

		    case 'Y': case 'y':
			overwrite_all = 1;
			break;

		    case 'D': case 'd':
			delete_source = 1;
			break;

		    case 'C': case 'c':
			flags |= WVC_FLAG;
			break;

		    case 'F': case 'f':
			if (flags & FAST_FLAG)
			    flags |= VERY_FAST_FLAG;
			else
			    flags |= FAST_FLAG;

			break;

		    case 'H': case 'h':
			flags |= HIGH_FLAG;
			break;

		    case 'N': case 'n':
			flags |= CALC_NOISE;
			break;

		    case 'T': case 't':
			flags |= COPY_TIME;
			break;

		    case 'B': case 'b':
			bits = strtol (++*argv, argv, 10);
			--*argv;

			if (bits < 4 || bits > 4800) {
			    error_line ("hybrid spec must be 4 to 4800!");
			    usage_error = 1;
			}

			if (bits < 24)
			    bits -= 3;

			break;

		    case 'J': case 'j':
			switch (strtol (++*argv, argv, 10)) {

			    case 0:
				flags |= JOINT_OVERRIDE;
				flags &= ~JOINT_STEREO;
				break;

			    case 1:
				flags |= (JOINT_OVERRIDE | JOINT_STEREO);
				break;

			    default:
				error_line ("-j0 or -j1 only!");
				usage_error = 1;
			}
			
			--*argv;
			break;

		    case 'S': case 's':
			switch (strtol (++*argv, argv, 10)) {

			    case 0:
				flags |= SHAPE_OVERRIDE;
				flags &= ~LOSSY_SHAPE;
				break;

			    case 1:
				flags |= (LOSSY_SHAPE | SHAPE_OVERRIDE);
				break;

			    default:
				error_line ("-s0 or -s1 only!");
				usage_error = 1;
			}
			
			--*argv;
			break;

		    case 'R': case 'r':
			switch (strtol (++*argv, argv, 10)) {

			    case 1:
				flags |= MONO_FLAG;

			    case 2:
				flags |= RAW_FLAG;
				break;

			    default:
				error_line ("-r1 or -r2 only!");
				usage_error = 1;
				break;
			}

			--*argv;
			break;
#ifdef __WIN32__
		    case 'E': case 'e':
			flags |= CREATE_EXE;
			break;
#endif
		    default:
			error_line ("illegal option: %c !", **argv);
			usage_error = 1;
		}
	else {
	    if (!infilename) {
		infilename = malloc (strlen (*argv) + MAX_PATH);
		strcpy (infilename, *argv);
	    }
	    else if (!outfilename) {
		outfilename = malloc (strlen (*argv) + MAX_PATH);
		strcpy (outfilename, *argv);
	    }
	    else {
		error_line ("extra unknown argument: %s !", *argv);
		usage_error = 1;
	    }
	}

    setup_break ();	// set up console and detect ^C and ^Break

    // check for various command-line argument problems

    if (!(~flags & (HIGH_FLAG | FAST_FLAG))) {
	error_line ("high and fast modes are mutually exclusive!");
	usage_error = 1;
    }

    if (bits) {
	if ((flags & RAW_FLAG) && !overwrite_all) {
	    fprintf (stderr, "are you sure you want to use hybrid mode with raw files? ");
#ifdef __WIN32__
	    SetConsoleTitle ("hybrid+raw?");
#endif
	    usage_error = (yna () == 'n');
	}
    }
    else {
	if (flags & (CALC_NOISE | SHAPE_OVERRIDE | WVC_FLAG)) {
	    error_line ("-s, -n and -c options are for hybrid mode (-b) only!");
	    usage_error = 1;
	}
    }

    if (!infilename) {
	printf ("%s", usage);
	return 0;
    }

    if (usage_error)
	return 0;

    // If we are trying to create self-extracting .exe files, this is where
    // we read the wvselfx.exe file into memory in preparation for pre-pending
    // it to the WavPack files.

#ifdef __WIN32__
    if (flags & CREATE_EXE) {
	HANDLE wvselfx_file;
	ulong bcount;

	strcpy (filespec_name (selfname), "wvselfx.exe");

	wvselfx_file = CreateFile (selfname, GENERIC_READ, FILE_SHARE_READ, NULL,
	    OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

	if (wvselfx_file == INVALID_HANDLE_VALUE) {
	    _searchenv ("wvselfx.exe", "PATH", selfname);

	    wvselfx_file = CreateFile (selfname, GENERIC_READ, FILE_SHARE_READ, NULL,
	       OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

	    if (wvselfx_file == INVALID_HANDLE_VALUE) {
		error_line ("can't find wvselfx.exe file needed to create executable!");
		exit (1);
	    }
	}

	if (GetFileSize (wvselfx_file, NULL) != SELF_SIZE) {
	    error_line ("wvselfx.exe file is corrupt or an incompatible revision!");
	    DoCloseHandle (wvselfx_file);
	    exit (1);
	}

	wvselfx_image = malloc (SELF_SIZE);

	if (!DoReadFile (wvselfx_file, wvselfx_image, SELF_SIZE, &bcount, NULL) || bcount != SELF_SIZE) {
	    error_line ("wvselfx.exe file is not readable!");
	    DoCloseHandle (wvselfx_file);
	    free (wvselfx_image);
	    exit (1);
	}

	DoCloseHandle (wvselfx_file);
    }
#endif

    // If the infile specification begins with a '@', then it actually points
    // to a file that contains the names of the files to be converted. This
    // was included for use by Wim Speekenbrink's frontends, but could be used
    // for other purposes.

    if (infilename [0] == '@') {
	FILE *list = fopen (infilename+1, "rt");
	int c;

	if (list == NULL) {
	    error_line ("file %s not found!", infilename+1);
	    return 1;
	}

	while ((c = getc (list)) != EOF) {

	    while (c == '\n')
		c = getc (list);

	    if (c != EOF) {
		char *fname = malloc (MAX_PATH);
		int ci = 0;

		do
		    fname [ci++] = c;
		while ((c = getc (list)) != '\n' && c != EOF && ci < MAX_PATH);

		fname [ci++] = '\0';
		matches = realloc (matches, ++num_files * sizeof (*matches));
		matches [num_files - 1] = realloc (fname, ci);
	    }
	}

	fclose (list);
	free (infilename);
	infilename = NULL;
	filelist = 1;
    }
    else if (*infilename != '-') {	// skip this if infile is stdin (-)
	if (!(flags & RAW_FLAG) && !filespec_ext (infilename))
	    strcat (infilename, anylower (infilename) ? ".wav" : ".WAV");

	// search for and store any filenames that match the user supplied spec

#ifdef __BORLANDC__
	if (findfirst (infilename, &ffblk, 0) == 0) {
	    do {
		matches = realloc (matches, ++num_files * sizeof (*matches));
		matches [num_files - 1] = strdup (ffblk.ff_name);
	    } while (findnext (&ffblk) == 0);
	}
#else
	if ((i = _findfirst (infilename, &_finddata_t)) != -1L) {
	    do {
		if (!(_finddata_t.attrib & _A_SUBDIR)) {
		    matches = realloc (matches, ++num_files * sizeof (*matches));
		    matches [num_files - 1] = strdup (_finddata_t.name);
		}
	    } while (_findnext (i, &_finddata_t) == 0);

	    _findclose (i);
	}
#endif
    }
    else {	// handle case of stdin (-)
	matches = realloc (matches, ++num_files * sizeof (*matches));
	matches [num_files - 1] = infilename;
    }

    // If the outfile specification begins with a '@', then it actually points
    // to a file that contains the output specification. This was included for
    // use by Wim Speekenbrink's frontends because certain filenames could not
    // be passed on the command-line, but could be used for other purposes.

    if (outfilename && outfilename [0] == '@') {
	FILE *list = fopen (outfilename+1, "rt");
	int c;

	if (list == NULL) {
	    error_line ("file %s not found!", outfilename+1);
	    return 1;
	}

	while ((c = getc (list)) == '\n');

	if (c != EOF) {
	    int ci = 0;

	    do
		outfilename [ci++] = c;
	    while ((c = getc (list)) != '\n' && c != EOF && ci < MAX_PATH);

	    outfilename [ci] = '\0';
	}
	else {
	    error_line ("output spec file is empty!");
	    fclose (list);
	    return 1;
	}

	fclose (list);
    }

    // if we found any files to process, this is where we start

    if (num_files) {
	char outpath, addext;

	if (outfilename) {
	    outpath = (filespec_path (outfilename) != NULL);

	    if (num_files > 1 && !outpath) {
		error_line ("%s is not a valid output path", outfilename);
		return 1;
	    }
	}
	else
	    outpath = 0;

	addext = !outfilename || outpath || !filespec_ext (outfilename);

	// loop through and process files in list

	for (file_index = 0; file_index < num_files; ++file_index) {
#ifdef __WIN32__
	    if (check_break ())
		break;
#else
	    if (check_break (0))
		break;
#endif
	    // get input filename from list

	    if (filelist)
		infilename = matches [file_index];
	    else if (*infilename != '-') {
		*filespec_name (infilename) = '\0';
		strcat (infilename, matches [file_index]);
	    }

	    // generate output filename

	    if (outpath) {
		strcat (outfilename, filespec_name (matches [file_index]));

		if (filespec_ext (outfilename))
		    *filespec_ext (outfilename) = '\0';
	    }
	    else if (!outfilename) {
		outfilename = malloc (strlen (infilename) + 10);
		strcpy (outfilename, infilename);

		if (filespec_ext (outfilename))
		    *filespec_ext (outfilename) = '\0';
	    }

	    if (addext)
#ifdef __WIN32__
		if (flags & CREATE_EXE)
		    strcat (outfilename, anylower (filespec_name (outfilename)) ?
			".exe" : ".EXE");
		else
#endif
		    strcat (outfilename, anylower (filespec_name (outfilename)) ?
			".wv" : ".WV");

	    // if "correction" file is desired, generate name for that

	    if (flags & WVC_FLAG) {
		out2filename = malloc (strlen (outfilename) + 10);
		strcpy (out2filename, outfilename);

		if (filespec_ext (out2filename))
		    *filespec_ext (out2filename) = '\0';

		strcat (out2filename, anylower (filespec_name (out2filename)) ?
		    ".wvc" : ".WVC");
	    }
	    else
		out2filename = NULL;

	    if (num_files > 1)
		fprintf (stderr, "\n%s:\n", infilename);

	    result = pack_file (infilename, outfilename, out2filename, bits, flags);

	    if (result == HARD_ERROR)
		break;

	    // delete source file if that option is enabled

	    if (result == NO_ERROR && delete_source)
		error_line ("%s source file %s", DoDeleteFile (infilename) ?
		    "deleted" : "can't delete", infilename);

	    // clean up in preparation for potentially another file

	    if (outpath)
		*filespec_name (outfilename) = '\0';
	    else {
		free (outfilename);
		outfilename = NULL;
	    }

	    if (out2filename)
		free (out2filename);

	    free (matches [file_index]);
	}

	free (matches);
    }
    else
	error_line (filespec_wild (infilename) ? "nothing to do!" :
	    "file %s not found!", infilename);

    if (wvselfx_image)
	free (wvselfx_image);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// This function packs a single file "infilename" and stores the result at  //
// "outfilename". If "out2filename" is specified, then the "correction"     //
// file would go there. The files are opened and closed in this function    //
// and the "flags" and "bits" parameter specifies the mode of compression.  //
//////////////////////////////////////////////////////////////////////////////

#ifdef NO_BS_WRITE
extern long bytes_not_written;
#endif

static int pack_file (char *infilename, char *outfilename, char *out2filename, int bits, long flags)
{
    ulong infile_length = 0, outfile_length, out2file_length;
    ulong wavpack_header_pos, total_samples;
    HANDLE infile, outfile, out2file;
    RiffChunkHeader RiffChunkHeader;
    WavpackHeader WavpackHeader;
    ChunkHeader ChunkHeader;
    WaveHeader WaveHeader;
    double dtime;
    ulong bcount;
    int result;

#ifdef __BORLANDC__
    struct time time1, time2;
#else
    struct _timeb time1, time2;
#endif

#ifdef __WIN32__
    FILETIME time;
#else
    struct ftime time;
#endif

#ifdef NO_BS_WRITE
    bytes_not_written = 0;
#endif

    // open the source file for reading

#ifdef __WIN32__
    if (*infilename == '-') {
	infile = GetStdHandle (STD_INPUT_HANDLE);
	setmode (fileno (stdin), O_BINARY);
    }
    else if ((infile = CreateFile (infilename, GENERIC_READ, FILE_SHARE_READ, NULL,
	OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
	    error_line ("can't open file %s!", infilename);
	    return SOFT_ERROR;
    }

    GetFileTime (infile, NULL, NULL, &time);
#else
    if (*infilename == '-') {
	infile = fileno (stdin);
	setmode (fileno (stdin), O_BINARY);
    }
    else if ((infile = open (infilename, O_RDONLY | O_BINARY)) == -1) {
	error_line ("can't open file %s!", infilename);
	return SOFT_ERROR;
    }

    getftime (infile, &time);
#endif

    // check both output files for overwrite warning required

#ifdef __WIN32__
    if (!overwrite_all && (outfile = CreateFile (outfilename, GENERIC_READ, 0, NULL,
	OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
#else
    if (!overwrite_all && (outfile = open (outfilename, O_RDONLY | O_BINARY)) != -1) {
#endif
	    DoCloseHandle (outfile);
	    fprintf (stderr, "overwrite %s (yes/no/all)? ", FN_FIT (outfilename));
#ifdef __WIN32__
	    SetConsoleTitle ("overwrite?");
#endif

	    switch (yna ()) {

		case 'n':
		    DoCloseHandle (infile);
		    return SOFT_ERROR;

		case 'a':
		    overwrite_all = 1;
	    }
    }

#ifdef __WIN32__
    if (out2filename && !overwrite_all && (out2file = CreateFile (out2filename, GENERIC_READ, 0, NULL,
	OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
#else
    if (out2filename && !overwrite_all && (out2file = open (out2filename, O_RDONLY | O_BINARY)) != -1) {
#endif
	    DoCloseHandle (out2file);
	    fprintf (stderr, "overwrite %s (yes/no/all)? ", FN_FIT (out2filename));
#ifdef __WIN32__
	    SetConsoleTitle ("overwrite?");
#endif

	    switch (yna ()) {

		case 'n':
		    DoCloseHandle (infile);
		    return SOFT_ERROR;

		case 'a':
		    overwrite_all = 1;
	    }
    }

#ifdef __BORLANDC__
    gettime (&time1);
#else
    _ftime (&time1);
#endif

    // open output file for writing

#ifdef __WIN32__
    if ((outfile = CreateFile (outfilename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
	CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
#else
    if ((outfile = open (outfilename, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, S_IREAD | S_IWRITE)) == -1) {
#endif
	    error_line ("can't create file %s!", outfilename);
	    DoCloseHandle (infile);
	    return SOFT_ERROR;
    }

    if (out2filename)
	fprintf (stderr, "creating %s (+%s),", FN_FIT (outfilename), filespec_ext (out2filename));
    else
	fprintf (stderr, "creating %s,", FN_FIT (outfilename));

    // if we are creating a self-extracting .exe, write that portion first

#ifdef __WIN32__
    if (flags & CREATE_EXE)
	if (!DoWriteFile (outfile, wvselfx_image, SELF_SIZE, &bcount, NULL) || bcount != SELF_SIZE) {
	    error_line ("can't write WavPack data, disk probably full!");
	    DoCloseHandle (infile);
	    DoCloseHandle (outfile);
	    DoDeleteFile (outfilename);
	    return SOFT_ERROR;
	}
#endif

    // if not in "raw" mode, read (and copy to output) initial RIFF form header

    if (!(flags & RAW_FLAG)) {
	if ((!DoReadFile (infile, &RiffChunkHeader, sizeof (RiffChunkHeader), &bcount, NULL) ||
	    bcount != sizeof (RiffChunkHeader) || strncmp (RiffChunkHeader.ckID, "RIFF", 4) ||
	    strncmp (RiffChunkHeader.formType, "WAVE", 4))) {
		error_line ("%s is not a valid .WAV file!", infilename);
		DoCloseHandle (infile);
		DoCloseHandle (outfile);
		DoDeleteFile (outfilename);
		return SOFT_ERROR;
	}
	else if (!DoWriteFile (outfile, &RiffChunkHeader, sizeof (RiffChunkHeader), &bcount, NULL) ||
	    bcount != sizeof (RiffChunkHeader)) {
		error_line ("can't write WavPack data, disk probably full!");
		DoCloseHandle (infile);
		DoCloseHandle (outfile);
		DoDeleteFile (outfilename);
		return SOFT_ERROR;
	}
	else
	    infile_length += bcount;
    }

    // if not in "raw" mode, loop through all elements of the RIFF wav header
    // (until the data chuck) and copy them to the output file

    while (!(flags & RAW_FLAG)) {

	if (!DoReadFile (infile, &ChunkHeader, sizeof (ChunkHeader), &bcount, NULL) ||
	    bcount != sizeof (ChunkHeader)) {
		error_line ("%s is not a valid .WAV file!", infilename);
		DoCloseHandle (infile);
		DoCloseHandle (outfile);
		DoDeleteFile (outfilename);
		return SOFT_ERROR;
	}
	else if (!DoWriteFile (outfile, &ChunkHeader, sizeof (ChunkHeader), &bcount, NULL) ||
	    bcount != sizeof (ChunkHeader)) {
		error_line ("can't write WavPack data, disk probably full!");
		DoCloseHandle (infile);
		DoCloseHandle (outfile);
		DoDeleteFile (outfilename);
		return SOFT_ERROR;
	}
	else
	    infile_length += bcount;

	// if it's the format chunk, we want to get some info out of there and
	// make sure it's a .wav file we can handle

	if (!strncmp (ChunkHeader.ckID, "fmt ", 4)) {

	    if (ChunkHeader.ckSize < sizeof (WaveHeader) ||
		!DoReadFile (infile, &WaveHeader, sizeof (WaveHeader), &bcount, NULL) ||
		bcount != sizeof (WaveHeader)) {
		    error_line ("%s is not a valid .WAV file!", infilename);
		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);
		    return SOFT_ERROR;
	    }
	    else if (!DoWriteFile (outfile, &WaveHeader, sizeof (WaveHeader), &bcount, NULL) ||
		bcount != sizeof (WaveHeader)) {
		    error_line ("can't write WavPack data, disk probably full!");
		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);
		    return SOFT_ERROR;
	    }
	    else
		infile_length += bcount;

	    if (ChunkHeader.ckSize > sizeof (WaveHeader)) {
		int bytes_to_copy = (ChunkHeader.ckSize + 1 - sizeof (WaveHeader)) & ~1L;
		char *buff = malloc (bytes_to_copy);

		if (!DoReadFile (infile, buff, bytes_to_copy, &bcount, NULL) ||
		    bcount != bytes_to_copy ||
		    !DoWriteFile (outfile, buff, bytes_to_copy, &bcount, NULL) ||
		    bcount != bytes_to_copy) {

			error_line ("can't write WavPack data, disk probably full!");
			DoCloseHandle (infile);
			DoCloseHandle (outfile);
			DoDeleteFile (outfilename);
			free (buff);
			return SOFT_ERROR;
		}
		else
		    infile_length += bcount;

		free (buff);
	    }

	    if (WaveHeader.FormatTag != 1 ||
		WaveHeader.NumChannels < 1 || WaveHeader.NumChannels > 2 ||
		WaveHeader.BitsPerSample < 16 || WaveHeader.BitsPerSample > 24 ||
		WaveHeader.BlockAlign / WaveHeader.NumChannels > 3) {
		    error_line ("%s is an unsupported .WAV format!", infilename);
		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);
		    return SOFT_ERROR;
	    }

	    if (WaveHeader.NumChannels == 1)
		flags |= MONO_FLAG;

	    if (WaveHeader.BitsPerSample > 16)
		flags |= BYTES_3;

	    if (bits && !(flags & SHAPE_OVERRIDE) && WaveHeader.SampleRate >= 64000)
		flags |= LOSSY_SHAPE;
	}
	else if (!strncmp (ChunkHeader.ckID, "data", 4)) {

	    // on the data chunk, get size and exit loop

	    total_samples = ChunkHeader.ckSize / (WaveHeader.NumChannels *
		((flags & BYTES_3) ? 3 : 2));

	    infile_length += ChunkHeader.ckSize;
	    break;
	}
	else {		// just copy unknown chunks to output file

	    int bytes_to_copy = (ChunkHeader.ckSize + 1) & ~1L;
	    char *buff = malloc (bytes_to_copy);

	    if (!DoReadFile (infile, buff, bytes_to_copy, &bcount, NULL) ||
		bcount != bytes_to_copy ||
		!DoWriteFile (outfile, buff, bytes_to_copy, &bcount, NULL) ||
		bcount != bytes_to_copy) {

		    error_line ("can't write WavPack data, disk probably full!");
		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);
		    free (buff);
		    return SOFT_ERROR;
	    }
	    else
		infile_length += bcount;

	    free (buff);
	}
    }

    // now create the WavPack header which gets written next in the output

    CLEAR (WavpackHeader);
    memcpy (WavpackHeader.ckID, "wvpk", 4);
    WavpackHeader.ckSize = sizeof (WavpackHeader) - 8;
    WavpackHeader.version = 3;

    if (flags & RAW_FLAG) {
	WaveHeader.BitsPerSample = 16;
	WaveHeader.SampleRate = 44100;
    }

    if (WaveHeader.BitsPerSample == 16)
	WavpackHeader.shift = 0;
    else if (WaveHeader.BitsPerSample > 20 && !bits) {
	WavpackHeader.shift = 4;
	flags |= OVER_20;
    }
    else
	WavpackHeader.shift = 24 - WaveHeader.BitsPerSample;

    // If we are in "raw" mode, we will not know how much .wav data there is if
    // we are reading from a pipe. We can still handle this case, but we will
    // not be able to display progress.

#ifdef __WIN32__
    if (flags & RAW_FLAG) {
	if (GetFileType (infile) == FILE_TYPE_DISK) {
	    infile_length = GetFileSize (infile, NULL);
	    total_samples = infile_length / ((flags & MONO_FLAG) ? 2 : 4);
	}
	else
	    total_samples = 0;
    }
#else
    if (flags & RAW_FLAG) {
	struct stat stat;

	if (!fstat (infile, &stat) && (stat.st_mode & S_IFREG)) {
	    infile_length = filelength (infile);
	    total_samples = infile_length / ((flags & MONO_FLAG) ? 2 : 4);
	}
	else
	    total_samples = 0;
    }
#endif

    WavpackHeader.total_samples = total_samples;

    if (bits) {

	// If the command-line "bits" parameter is < 24 then we translate that
	// to match the old lossy mode so that old command lines will generate
	// the same bitrate (although quality will be much higher with the new
	// hybrid mode). If the parameter is >= 24, then we convert the kbps
	// value into the new "bits" field meaning which is bits/sample * 256.

	if (bits < 21) {
	    WavpackHeader.bits = (bits * 512) + 1374;

	    if (flags & MONO_FLAG)
		WavpackHeader.bits /= 2;
	}
	else {
	    long bps = floor ((bits * 256000.0 / WaveHeader.SampleRate) + 0.5);
	    WavpackHeader.bits = bps <= (64 << 8) ? bps : (64 << 8);
	}

	if (flags & HIGH_FLAG)
	    flags |= (NEW_DECORR_FLAG | NEW_HIGH_FLAG | EXTREME_DECORR);
	else
	    flags |= (NEW_DECORR_FLAG | NEW_HIGH_FLAG);

	flags &= ~(FAST_FLAG | HIGH_FLAG);

//	if (!(flags & JOINT_OVERRIDE))		// joint stereo is no longer
//	    flags |= JOINT_STEREO;		//  the default in hybrid mode
    }
    else {
	WavpackHeader.bits = 0;

	if (!(flags & FAST_FLAG)) {
	    if (flags & HIGH_FLAG)
		flags |= (NEW_HIGH_FLAG | NEW_DECORR_FLAG | EXTREME_DECORR | CROSS_DECORR);
	    else
		flags |= (HIGH_FLAG | NEW_HIGH_FLAG | NEW_DECORR_FLAG | CROSS_DECORR);

	    if (!(flags & JOINT_OVERRIDE))
		flags |= JOINT_STEREO;
	}
	else
	    if (!(flags & VERY_FAST_FLAG))
		flags &= ~FAST_FLAG;
    }

    if (flags & MONO_FLAG)
	flags &= ~(JOINT_STEREO | CROSS_DECORR);

    WavpackHeader.flags = flags;

    if (filespec_ext (infilename))
	strcpy (WavpackHeader.extension, filespec_ext (infilename) + 1);
    else if (*infilename == '-') {
	if (flags & RAW_FLAG)
	    strcpy (WavpackHeader.extension, anylower (filespec_name (outfilename)) ? "raw" : "RAW");
	else
	    strcpy (WavpackHeader.extension, anylower (filespec_name (outfilename)) ? "wav" : "WAV");
    }

    // save the current outfile position because we will rewrite the WavPack
    // header when we have finished packing the file and know the CRC(s)

#ifdef __WIN32__
    wavpack_header_pos = SetFilePointer (outfile, 0, NULL, FILE_CURRENT);
#else
    wavpack_header_pos = tell (outfile);
#endif

    // write WavPack header

    if (!DoWriteFile (outfile, &WavpackHeader, sizeof (WavpackHeader), &bcount, NULL) ||
	bcount != sizeof (WavpackHeader)) {
	    error_line ("can't write WavPack data, disk probably full!");
	    DoCloseHandle (infile);
	    DoCloseHandle (outfile);
	    DoDeleteFile (outfilename);
	    return SOFT_ERROR;
    }

    // if we are creating a "correction" file, open if now for writing

#ifdef __WIN32__
    if (out2filename && (out2file = CreateFile (out2filename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
	CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
#else
    if (out2filename && (out2file = open (out2filename, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, S_IREAD | S_IWRITE)) == -1) {
#endif
	    error_line ("can't create correction file!");
	    DoCloseHandle (infile);
	    DoCloseHandle (outfile);
	    DoDeleteFile (outfilename);
	    return SOFT_ERROR;
    }
    else if (!out2filename)
	out2file = INVALID_HANDLE_VALUE;

    // pack the file now

    result = pack_audio (&WavpackHeader, infile, outfile, out2file);

    // if we created a "correction" file, set the time (if selected) and
    // close it now (or delete it if we didn't store anything there)

    if (out2filename) {
#ifdef __WIN32__
	out2file_length = GetFileSize (out2file, NULL);

	if (flags & COPY_TIME)
	    SetFileTime (out2file, &time, &time, &time);
#else
	out2file_length = filelength (out2file);

	if (flags & COPY_TIME)
	    setftime (out2file, &time);
#endif

	if (!DoCloseHandle (out2file)) {
	    error_line ("can't close .wvc file!");

	    if (result == NO_ERROR)
		result = SOFT_ERROR;
	}

	if (!out2file_length)
	    DoDeleteFile (out2filename);
    }
    else
	out2file_length = 0;

    // If we are in "raw" mode and we didn't know the original filesize (pipe)
    // then figure a length based on the number of samples packed. We cannot
    // handle "raw" files of zero length.

    if ((flags & RAW_FLAG) && !total_samples) {
	infile_length = WavpackHeader.total_samples * ((flags & MONO_FLAG) ? 2 : 4) + WavpackHeader.extra_bc;

	if (!infile_length) {
	    error_line ("raw file contained no data!");
	    DoCloseHandle (infile);
	    DoCloseHandle (outfile);
	    DoDeleteFile (outfilename);

	    if (out2file_length)
		DoDeleteFile (out2filename);

	    return SOFT_ERROR;
	}
    }

    // if there were any errors during packing, delete the outfiles and exit

    if (result != NO_ERROR) {
	DoCloseHandle (infile);
	DoCloseHandle (outfile);
	DoDeleteFile (outfilename);

	if (out2file_length)
	    DoDeleteFile (out2filename);

	return result;
    }

    // If we are not in "raw" mode, then we need to check to see if there is
    // any information stored in the .wav file past the audio data. If there
    // is, we will append 128 '1's to the WavPack file (to recognize this data
    // later and prevent old unpackers from choking) and then copy the extra
    // information. For "raw" files there can be at most 3 extra bytes at the
    // end of the file, and these have a place in the WavPack header.

    if (!(flags & RAW_FLAG)) {
	char buff [16], ones [16], ones_written = 0;
	ulong bcount2;

	while (DoReadFile (infile, buff, sizeof (buff), &bcount, NULL) && bcount) {

	    if (!ones_written) {

		memset (ones, -1, sizeof (ones));

		if (!DoWriteFile (outfile, ones, sizeof (ones), &bcount2, NULL) || bcount2 != sizeof (ones)) {
		    error_line ("can't write WavPack data, disk probably full!");
		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);

		    if (out2file_length)
			DoDeleteFile (out2filename);

		    return SOFT_ERROR;
		}

		ones_written = 1;
	    }

	    if (!DoWriteFile (outfile, buff, bcount, &bcount2, NULL) || bcount2 != bcount) {
		error_line ("can't write WavPack data, disk probably full!");
		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);

		    if (out2file_length)
			DoDeleteFile (out2filename);

		    return SOFT_ERROR;
	    }
	}
    }
    else if ((flags & RAW_FLAG) && !WavpackHeader.extra_bc &&
	DoReadFile (infile, WavpackHeader.extras, 3, &bcount, NULL))
	    WavpackHeader.extra_bc = bcount;

    // close input file and seek the output file back to the WavPack header

    DoCloseHandle (infile);

#ifdef __WIN32__
    SetFilePointer (outfile, wavpack_header_pos, NULL, FILE_BEGIN);
#else
    lseek (outfile, wavpack_header_pos, SEEK_SET);
#endif

    // Because I ran out of flag bits in the WavPack header, an amazingly ugly
    // kludge was forced upon me! This code takes care of preparing the flags
    // field to be written to the file. When the file is read, this is undone.

    if ((flags & CROSS_DECORR) && !(flags & EXTREME_DECORR))
	flags |= EXTREME_DECORR | CANCEL_EXTREME;

    WavpackHeader.flags = flags & STORED_FLAGS;

    // write WavPack header with updated flags field and CRC(s)

    if (!DoWriteFile (outfile, &WavpackHeader, sizeof (WavpackHeader), &bcount, NULL) ||
	bcount != sizeof (WavpackHeader)) {
	    error_line ("can't write WavPack data, disk probably full!");
	    DoCloseHandle (outfile);
	    DoDeleteFile (outfilename);

	    if (out2file_length)
		DoDeleteFile (out2filename);

	    return SOFT_ERROR;
    }

    // get length of output file, set the time (if selected) and close it

#ifdef __WIN32__
    outfile_length = GetFileSize (outfile, NULL) - ((flags & CREATE_EXE) ? SELF_SIZE : 0);
#else
    outfile_length = filelength (outfile);
#endif

#ifdef NO_BS_WRITE
    outfile_length += bytes_not_written;
#endif

    if (flags & COPY_TIME)
#ifdef __WIN32__
	SetFileTime (outfile, &time, &time, &time);
#else
	setftime (outfile, &time);
#endif

    if (!DoCloseHandle (outfile)) {
	error_line ("can't close file!");
	return SOFT_ERROR;
    }

    // compute and display the time consumed along with some other details of
    // the packing operation, and then return NO_ERROR

#ifdef __BORLANDC__
    gettime (&time2);
    dtime = time2.ti_sec * 100.0 + time2.ti_hund + time2.ti_min * 6000.0 + time2.ti_hour * 360000.00;
    dtime -= time1.ti_sec * 100.0 + time1.ti_hund + time1.ti_min * 6000.0 + time1.ti_hour * 360000.00;

    if ((dtime /= 100.0) < 0.0)
	dtime += 86400.0;
#else
    _ftime (&time2);
    dtime = time2.time + time2.millitm / 1000.0;
    dtime -= time1.time + time1.millitm / 1000.0;
#endif

    if (out2file_length)
	error_line ("created %s (+%s) with %.2f%% lossless comp in %.2f secs",
	    FN_FIT (outfilename), filespec_ext (out2filename), 
	    100 - ((out2file_length + outfile_length) * 100.0 / infile_length), dtime);
    else
	error_line ("created %s with %.2f%% %s comp in %.2f secs", FN_FIT (outfilename),
	    100 - (outfile_length * 100.0 / infile_length),
	    (bits && WavpackHeader.crc != WavpackHeader.crc2) ? "lossy" : "lossless", dtime);

    return NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////
// This function performs the actual audio compression on the specified     //
// file handles, which are assumed to be opened and positioned correctly.   //
// This function does not read or write any file headers, nor does it close //
// the files when done. The supplied WavPack header specifies all of the    //
// compression parameters to use. If the "total_samples" field in the       //
// WavPack header is set non-zero, then exactly that many samples are read  //
// from the input file, the file pointer is left at the end of the data and //
// progress status messages are dumped to stderr. If this field is zero     //
// then we will read and convert samples up to the physical EOF, store the  //
// number of converted samples back into the field, store up to 3 leftover  //
// bytes into the WavPack header (24-bit data is not used with "raw" mode)  //
// and NOT generate any progress messages. This function has the code that  //
// handles overlapped reading of the audio data for Win95/98 systems and    //
// displays the noise calculations for lossy mode if requested by the user. //
//////////////////////////////////////////////////////////////////////////////

static int pack_audio (WavpackHeader *wphdr, HANDLE infile, HANDLE outfile, HANDLE out2file)
{
    ulong samples_remaining, samples_read = 0;
    int flags = wphdr->flags;
    double progress = 0.0;
    uchar *buffer, *bptr;
    WavpackContext wpc;

#ifdef __MT__
    uchar *next_buffer = NULL, read_pending = 0;
#endif

    CLEAR (wpc);
    wpc.wphdr = wphdr;
    wpc.outbits.bufsiz = wpc.out2bits.bufsiz = OUTBUF_SIZE;
    buffer = malloc (INBUF_SIZE);
    pack_init (&wpc);

#ifdef __MT__
    if (get_platform () == 1) {
	next_buffer = malloc (INBUF_SIZE);

	if (!buffer || !next_buffer || bs_open_write (&wpc.outbits, outfile, TRUE) ||
	    ((flags & WVC_FLAG) && bs_open_write (&wpc.out2bits, out2file, FALSE))) {
		error_line ("internal error!");

		if (buffer)
		    free (buffer);

		if (next_buffer)
		    free (buffer);

		return HARD_ERROR;
	}
    }
    else {
	if (!buffer || bs_open_write (&wpc.outbits, outfile, FALSE) ||
	    ((flags & WVC_FLAG) && bs_open_write (&wpc.out2bits, out2file, FALSE))) {
		error_line ("internal error!");

		if (buffer)
		    free (buffer);

		return HARD_ERROR;
	}
    }
#else
    if (!buffer || bs_open_write (&wpc.outbits, outfile) || ((flags & WVC_FLAG) && bs_open_write (&wpc.out2bits, out2file))) {
	error_line ("internal error!");

	if (buffer)
	    free (buffer);

	return HARD_ERROR;
    }
#endif

    if (wphdr->total_samples) {
	samples_remaining = wphdr->total_samples;
	fprintf (stderr, "   0%% done...");
#ifdef __WIN32__
	display_progress (progress);
#endif
    }

    while (1) {
	uint sample_count;
	ulong bytes_read;

	bytes_read = 0;

	while (1) {
	    ulong bytes_to_read;

	    if (!wphdr->total_samples || samples_remaining > INBUF_SIZE)
		bytes_to_read = INBUF_SIZE;
	    else {
		bytes_to_read = samples_remaining * BYTES_PER_SAMPLE;

		if (bytes_to_read > INBUF_SIZE)
		    bytes_to_read = INBUF_SIZE;
	    }

	    samples_remaining -= bytes_to_read / BYTES_PER_SAMPLE;

#ifdef __MT__
	    if (next_buffer) {
		if (read_pending) {
		    bytes_read = finish_read ();
		    read_pending = 0;
		    bptr = buffer;
		    buffer = next_buffer;
		    next_buffer = bptr;

		    if (!bytes_read)
			break;
		}

		start_read (infile, next_buffer, bytes_to_read);
		read_pending = 1;

		if (bytes_read)
		    break;
	    }
	    else {
		DoReadFile (infile, buffer, bytes_to_read, &bytes_read, NULL);
		break;
	    }
#else
	    DoReadFile (infile, buffer, bytes_to_read, &bytes_read, NULL);
	    break;
#endif
	}

	if (bytes_read)
	    samples_read += sample_count = bytes_read / BYTES_PER_SAMPLE;
	else
	    break;

	pack_samples (&wpc, buffer, sample_count);

	// this is only used for "raw" files of unknown length

	if (!wphdr->total_samples && (bytes_read %= BYTES_PER_SAMPLE) && bytes_read < 4)
	    for (wphdr->extra_bc = 0; wphdr->extra_bc < bytes_read; ++wphdr->extra_bc)
		wphdr->extras [wphdr->extra_bc] = *bptr++;

	if (bs_error (&wpc.outbits)) {
#ifdef __MT__
	    if (read_pending)
		finish_read ();
#endif
	    error_line ("can't write WavPack data, disk probably full!");
	    bs_close_write (&wpc.outbits);
	    free (buffer);
#ifdef __MT__
	    if (next_buffer)
		free (next_buffer);
#endif
	    return HARD_ERROR;
	}

#ifdef __WIN32__
	if (check_break ()) {
	    fprintf (stderr, "^C\n");
#else
	if (check_break (infile != fileno (stdin))) {
#endif

#ifdef __MT__
	    if (read_pending)
		finish_read ();
#endif
	    bs_close_write (&wpc.outbits);
	    free (buffer);
#ifdef __MT__
	    if (next_buffer)
		free (next_buffer);
#endif
	    return SOFT_ERROR;
	}

	if (wphdr->total_samples && progress != floor (((double) samples_read / wphdr->total_samples) * 100.0 + 0.5)) {
	    progress = (double) samples_read / wphdr->total_samples;
#ifdef __WIN32__
	    display_progress (progress);
#endif
	    progress = floor (progress * 100.0 + 0.5);
	    fprintf (stderr, "\b\b\b\b\b\b\b\b\b\b\b\b%3d%% done...", (int) progress);
	}
    }

    free (buffer);
#ifdef __MT__
    if (next_buffer)
	free (next_buffer);
#endif

    flush_word1 (&wpc);		// be sure to flush any accumulated zeros
    bs_close_write (&wpc.outbits);	//  before writing the last buffer to the file

    if (flags & WVC_FLAG)
	bs_close_write (&wpc.out2bits);

    if (bs_error (&wpc.outbits) || ((flags & WVC_FLAG) && bs_error (&wpc.out2bits))) {
	error_line ("can't write WavPack data, disk probably full!");
	return HARD_ERROR;
    }

    // If the "total_samples" field was zero, then we store the actual number
    // of converted samples there, otherwise we make sure that we really got
    // as many samples as there were supposed to be.

    if (!wphdr->total_samples)
	wphdr->total_samples = samples_read;
    else if (wphdr->total_samples != samples_read) {
	error_line ("couldn't read all samples, file may be corrupt!!");
	return SOFT_ERROR;
    }

    // display the quantization noise if requested by the user

    if ((flags & CALC_NOISE) && pack_noise (&wpc, NULL) > 0.0) {
	double full_scale_rms, sum, peak;

	sum = pack_noise (&wpc, &peak);

	if (flags & BYTES_3)
	    full_scale_rms = 8388608.0 * 8388607.0 * 0.5;
	else
	    full_scale_rms = 32768.0 * 32767.0 * 0.5;

	error_line ("ave noise = %.2f dB, peak noise = %.2f dB",
	    log10 (sum / wphdr->total_samples / full_scale_rms) * 10,
	    log10 (peak / full_scale_rms) * 10);
    }

    return NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////
// This function displays the progress status on the title bar of the DOS   //
// window that WavPack is running in. The "file_progress" argument is for   //
// the current file only and ranges from 0 - 1; this function takes into    //
// account the total number of files to generate a batch progress number.   //
//////////////////////////////////////////////////////////////////////////////

#ifdef __WIN32__
static void display_progress (double file_progress)
{
    char title [40];

    file_progress = (file_index + file_progress) / num_files;
    sprintf (title, "%d%% (WavPack)", (int) ((file_progress * 100.0) + 0.5));
    SetConsoleTitle (title);
}
#endif
