////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2002 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// wavpack.h

// This header file contains all the definitions required by WavPack.

typedef unsigned short	ushort;
typedef unsigned char	uchar;
typedef unsigned long	ulong;
typedef unsigned int	uint;

#ifdef __WIN32__
#include <windows.h>

#define VERSION_STR "3.97 "

#define WIN_INBUF_SIZE (384 * 1024)
#define WIN_OUTBUF_SIZE (384 * 1024)

#define NT_INBUF_SIZE (72 * 1024)
#define NT_OUTBUF_SIZE (3072 * 1024)

#define INBUF_SIZE ((get_platform () == 1) ? WIN_INBUF_SIZE : NT_INBUF_SIZE)
#define OUTBUF_SIZE ((get_platform () == 1) ? WIN_OUTBUF_SIZE : NT_OUTBUF_SIZE)

// This macro must be set to the size in bytes of the wvselfx.exe module. It
// is used in wavpack.c to verify that it is using the correct version (even
// though this is not a very robust test) and it is used by the wvselfx.c
// program itself to locate the start of the WavPack file that has been
// appended to it.

#define SELF_SIZE 26624

#else

#define VERSION_STR "3.4x"

typedef int HANDLE;

#define INVALID_HANDLE_VALUE (HANDLE)(-1)
#define MAX_PATH 80
#define INBUF_SIZE (16 * 1024)
#define OUTBUF_SIZE (16 * 1024)

#define FALSE (0)
#define TRUE (!FALSE)

#endif

#define DATE_STR "2-18-03"

// This macro is handy for benchmarking because it eliminates the writing of
// WavPack data to the output file(s). In this way it is very much like the
// verify (-v) option in WvUnpack except that it works with WavPack instead
// and has no real use outside of benchmarking.

// #define NO_BS_WRITE


// ID3v1 and APEv2 TAG formats (may occur at the end of WavPack files)

typedef struct {
    uchar tag_id [3], title [30], artist [30], album [30];
    uchar year [4], comment [30], genre;
} ID3_Tag;

typedef struct {
    char ID [8];
    long version, length, item_count, flags;
    char res [8];
} APE_Tag_Hdr;

typedef struct {
    ID3_Tag id3_tag;
    APE_Tag_Hdr ape_tag_hdr;
    char *ape_tag_data;
} M_Tag;

int LoadTag (M_Tag *m_tag, HANDLE infile), ValidTag (M_Tag *m_tag);
int GetTagItem (M_Tag *m_tag, const char *item, char *value, int size);
void FreeTag (M_Tag *m_tag);


// RIFF / wav header formats (these occur at the beginning of both wav files
// and WavPack files that are not in the "raw" mode)

typedef struct {
    char ckID [4];
    long ckSize;
    char formType [4];
} RiffChunkHeader;

typedef struct {
    char ckID [4];
    long ckSize;
} ChunkHeader;

typedef struct {
    short FormatTag, NumChannels;
    long SampleRate, BitsPerSecond;
    short BlockAlign, BitsPerSample;
} WaveHeader;


// WavPack header format (this occurs immediately after the RIFF header in a
// standard WavPack file, or as the very first thing in a "raw" WavPack file)

typedef struct {
    char ckID [4];
    long ckSize;
    short version;
    short bits;			// added for version 2.00
    short flags, shift;		// added for version 3.00
    long total_samples, crc, crc2;
    char extension [4], extra_bc, extras [3];
} WavpackHeader;

// these flags added for version 3

#define MONO_FLAG	1	// not stereo
#define FAST_FLAG	2	// non-adaptive predictor and stereo mode
#define RAW_FLAG	4	// raw mode (no .wav header)
#define CALC_NOISE	8	// calc noise in lossy mode (no longer stored)
#define HIGH_FLAG	0x10	// high quality mode (all modes)
#define BYTES_3		0x20	// files have 3-byte samples
#define OVER_20		0x40	// samples are over 20 bits
#define WVC_FLAG	0x80	// create/use .wvc (no longer stored)
#define LOSSY_SHAPE	0x100	// noise shape (lossy mode only)
#define VERY_FAST_FLAG	0x200	// double fast (no longer stored)
#define NEW_HIGH_FLAG	0x400	// new high quality mode (lossless only)
#define CANCEL_EXTREME	0x800	// cancel EXTREME_DECORR
#define CROSS_DECORR	0x1000	// decorrelate chans (with EXTREME_DECORR flag)
#define NEW_DECORR_FLAG	0x2000	// new high-mode decorrelator
#define JOINT_STEREO	0x4000	// joint stereo (lossy and high lossless)
#define EXTREME_DECORR	0x8000	// extra decorrelation (+ enables other flags)

#define UNKNOWN_FLAGS	0x0000	// none of those left  :(
#define STORED_FLAGS	0xfd77	// these are only flags that affect unpacking

#define NOT_STORED_FLAGS (~STORED_FLAGS & 0xffff)

#define SHAPE_OVERRIDE	0x10000	// shaping mode specified
#define JOINT_OVERRIDE	0x20000	// joint-stereo mode specified
#define COPY_TIME	0x40000	// copy file-time from source
#define CREATE_EXE	0x80000	// create executable

// misc macros

#define BYTES_PER_SAMPLE ((flags & BYTES_3) ? ((flags & MONO_FLAG) ? 3 : 6) : ((flags & MONO_FLAG) ? 2 : 4))

#define CLEAR(destin) memset (&destin, 0, sizeof (destin));
#define SAVE(destin, item) { memcpy (destin, &item, sizeof (item)); (char *) destin += sizeof (item); }
#define RESTORE(item, source) { memcpy (&item, source, sizeof (item)); (char *) source += sizeof (item); }


// BitStream stuff (bits.c)

#ifdef __MT__
typedef struct {
    uchar *buf, *altbuf, *end, *ptr;
    ulong bufsiz, sr;
    uint error, bc;
    HANDLE file;
} Bitstream;
#else
typedef struct {
    uchar *buf, *end, *ptr;
    ulong bufsiz, fpos, sr;
    uint error, bc;
    HANDLE file;
} Bitstream;
#endif

#ifdef __MT__
int bs_open_read (Bitstream *bs, HANDLE file, int overlap);
int bs_open_write (Bitstream *bs, HANDLE file, int overlap);
#else
int bs_open_read (Bitstream *bs, HANDLE file);
int bs_open_write (Bitstream *bs, HANDLE file);
#endif

void bs_close_read (Bitstream *bs);
void bs_close_write (Bitstream *bs);
void bs_read (Bitstream *bs);
void bs_restore (Bitstream *bs);
void bs_write (Bitstream *bs);
int bs_error (Bitstream *bs);

int DoReadFile (HANDLE hFile, void *lpBuffer, ulong nNumberOfBytesToRead,
    ulong *lpNumberOfBytesRead, ...);

int DoWriteFile (HANDLE hFile, void *lpBuffer, ulong nNumberOfBytesToWrite,
    ulong *lpNumberOfBytesWritten, ...);

int DoCloseHandle (HANDLE hFile), DoTruncateFile (HANDLE hFile);
int DoDeleteFile (char *filename);

#ifdef __MT__
void start_read (HANDLE file, void *buff, ulong len);
void start_write (HANDLE file, void *buff, ulong len);
ulong finish_write (void), finish_read (void);
#endif

// macros used with BitStreams to read and write bits

#define getbit(bs) ( \
    (((bs)->bc) ? \
	((bs)->bc--, (bs)->sr & 1) : \
	    (((++((bs)->ptr) != (bs)->end) ? (void) 0 : bs_read (bs)), (bs)->bc = 7, ((bs)->sr = *((bs)->ptr)) & 1) \
    ) ? \
	((bs)->sr >>= 1, 1) : \
	((bs)->sr >>= 1, 0) \
)

#define getbits(value, nbits, bs) { \
    while ((nbits) > (bs)->bc) { \
	if (++((bs)->ptr) == (bs)->end) bs_read (bs); \
	(bs)->sr |= (long)*((bs)->ptr) << (bs)->bc; \
	(bs)->bc += 8; \
    } \
    *(value) = (bs)->sr; \
    (bs)->sr >>= (nbits); \
    (bs)->bc -= (nbits); \
}

#define putbit(bit, bs) { if (bit) (bs)->sr |= (1 << (bs)->bc); \
    if (++((bs)->bc) == 8) { \
	*((bs)->ptr) = (bs)->sr; \
	(bs)->sr = (bs)->bc = 0; \
	if (++((bs)->ptr) == (bs)->end) bs_write (bs); \
    }}

#define putbit_0(bs) { \
    if (++((bs)->bc) == 8) { \
	*((bs)->ptr) = (bs)->sr; \
	(bs)->sr = (bs)->bc = 0; \
	if (++((bs)->ptr) == (bs)->end) bs_write (bs); \
    }}

#define putbit_1(bs) { (bs)->sr |= (1 << (bs)->bc); \
    if (++((bs)->bc) == 8) { \
	*((bs)->ptr) = (bs)->sr; \
	(bs)->sr = (bs)->bc = 0; \
	if (++((bs)->ptr) == (bs)->end) bs_write (bs); \
    }}

#define putbits(value, nbits, bs) { \
    (bs)->sr |= (long)(value) << (bs)->bc; \
    if (((bs)->bc += (nbits)) >= 8) \
	do { \
	    *((bs)->ptr) = (bs)->sr; \
	    (bs)->sr >>= 8; \
	    if (++((bs)->ptr) == (bs)->end) bs_write (bs); \
	} while (((bs)->bc -= 8) >= 8); \
}

#define K_DEPTH 3
#define MAX_NTERMS 18
#define MAX_TERM 8

typedef struct {

    WavpackHeader *wphdr;

    Bitstream inbits, in2bits, outbits, out2bits;

    struct {
	double noise_sum, noise_ave, noise_max;
	long sum_level, left_level, right_level, diff_level;
	int last_extra_bits, extra_bits_count, m;
	long error [2], crc;

#ifdef COMPACT
	long sample [2] [2];
	int weight [2] [1];
#else
	long sample [2] [5], dsample [2], csample [2];
	int weight [2] [5], cweight [4];
#endif
    } dc;

    struct decorr_pass {
	int term, weight_A, weight_B;
	long samples_A [MAX_TERM], samples_B [MAX_TERM];
	long aweight_A, aweight_B;
    } decorr_passes [MAX_NTERMS + 1];

    struct {
	uint index [2], k_value [2], ave_k [2];
	ulong zeros_acc, ave_level [K_DEPTH] [2];
    } w1;

    struct { int last_dbits [2], last_delta_sign [2], bit_limit; } w2;

    struct { int ave_dbits [2], bit_limit; } w3;

    struct {
	ulong fast_level [2], slow_level [2];
	int bits_acc [2], bitrate;
    } w4;

} WavpackContext;

// pack.c and unpack.c stuff

void pack_init (WavpackContext *wpc);
void pack_samples (WavpackContext *wpc, void *buffer, uint sample_count);
double pack_noise (WavpackContext *wpc, double *peak);

void unpack_init (WavpackContext *wpc);
long unpack_samples (WavpackContext *wpc, void *buffer, uint sample_count);
int unpack_size (WavpackContext *wpc);
void *unpack_save (WavpackContext *wpc, void *destin);
void *unpack_restore (WavpackContext *wpc, void *source, int keep_resources);
long unpack_crc (WavpackContext *wpc);

// utils.c stuff

char *filespec_ext (char *filespec), *filespec_path (char *filespec);
char *filespec_name (char *filespec), *filespec_wild (char *filespec);
void error_line (char *error, ...), finish_line (void);
char anylower (char *string), yna (void);
void setup_break (void);

#ifdef __WIN32__
int get_platform (void);
int check_break (void);
#else
int check_break (int call_kbhit);
#endif

#define FN_FIT(fn) ((strlen (fn) > 30) ? filespec_name (fn) : fn)

// words?.c stuff

void init_word1 (WavpackContext *wpc);
long __fastcall get_word1 (WavpackContext *wpc, int chan);
long __fastcall get_old_word1 (WavpackContext *wpc, int chan);
void __fastcall send_word1 (WavpackContext *wpc, long value, int chan);
void flush_word1 (WavpackContext *wpc);

void init_word2 (WavpackContext *wpc);
long get_word2 (WavpackContext *wpc, int chan);

void init_word3 (WavpackContext *wpc);
long __fastcall send_word3 (WavpackContext *wpc, long value, int chan);
long __fastcall get_word3 (WavpackContext *wpc, int chan);

void init_word4 (WavpackContext *wpc);
long __fastcall send_word4 (WavpackContext *wpc, long value, int chan);
long __fastcall get_word4 (WavpackContext *wpc, int chan, long *correction);

#define WORD_EOF (1L << 31)
