////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2002 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// words1.c

// This module provides entropy word encoding and decoding functions using
// the Rice method. For versions 3.6 and 3.7, the coding was over a fixed
// block of 128 samples and the 'k' value was transmitted in the bitstream.
// For version 3.8 the fixed blocks were replaced with the 'k' value being
// dynamically calculated for each word and the conversion was done in three
// layers for maximum efficiency when the optimum 'k' value is not near an
// integer (or the data has less than optimum distribution). I believe that
// this method is now very close to the maximum efficiency possible without
// using fractional bits per sample (like arithmetic coding) which would
// complicate the bitstream. Starting with version 3.94, both the default
// and high lossless modes use this method.

#include "wavpack.h"

#include <stdlib.h>

#ifndef __WIN32__
#include <string.h>
#endif

////////////////////////// external table references /////////////////////////

extern const long bitmask [], bitset [];
extern const char nbits_table [];

//////////////////////////////////////////////////////////////////////////////
// This function initializes everything required to send or received words  //
// with this module and must be called BEFORE any other function in this    //
// module.                                                                  //
//////////////////////////////////////////////////////////////////////////////

void init_word1 (WavpackContext *wpc)
{
    CLEAR (wpc->w1);
}

//////////////////////////////////////////////////////////////////////////////
// This function or macro counts the number of bits that are required to    //
// specify the unsigned 32-bit value, counting from the LSB to the most     //
// significant bit that is set. Return range is 0 - 32.                     //
//////////////////////////////////////////////////////////////////////////////

#ifdef COMPACT
int count_bits (ulong avalue)
{
    if (avalue < 256)
	return nbits_table [avalue];
    else if (avalue < (1L << 16))
	return nbits_table [avalue >> 8] + 8;
    else if (avalue < (1L << 24))
	return nbits_table [avalue >> 16] + 16;
    else
	return nbits_table [avalue >> 24] + 24;
}
#else
#define count_bits(av) ( \
 (av) < (1 << 8) ? nbits_table [av] : \
  ( \
   (av) < (1L << 16) ? nbits_table [(av) >> 8] + 8 : \
   ((av) < (1L << 24) ? nbits_table [(av) >> 16] + 16 : nbits_table [(av) >> 24] + 24) \
  ) \
)
#endif

#ifdef PACK

#define outbits (&wpc->outbits)

//////////////////////////////////////////////////////////////////////////////
// Send the specified word to the "outbits" bitstream (which must have been //
// previously opened with bs_open_write()). Note that the "chan" value must //
// either be always 0 or 1, or alternate 0, 1, 0, 1, 0, 1, etc. Because     //
// streams of zeroes are accumulated in some cases, it is important to call //
// flush_word1() after all the data has been sent.                          //
//////////////////////////////////////////////////////////////////////////////

void __fastcall send_word1 (WavpackContext *wpc, long value, int chan)
{
    ulong avalue = labs (value), k;
    uint ones_count;

    // We accumulate zeroes when using the extreme decorrelator (it could be
    // used anytime, but that is when I added it and didn't want to break more
    // than required). However, this cannot be used with over 20-bit data
    // because bits over 20 are transmitted directly (in lossless, anyway) and
    // the bitstreams would become tangled.

    if ((wpc->wphdr->flags & EXTREME_DECORR) && !(wpc->wphdr->flags & OVER_20)) {
	if (wpc->w1.zeros_acc) {
	    if (value) {
		int cbits = count_bits (wpc->w1.zeros_acc);

		while (cbits--) {
		    putbit_1 (outbits);
		}

		putbit_0 (outbits);

		while (wpc->w1.zeros_acc > 1) {
		    putbit (wpc->w1.zeros_acc & 1, outbits);
		    wpc->w1.zeros_acc >>= 1;
		}

		wpc->w1.zeros_acc = 0;
	    }
	    else {
		wpc->w1.zeros_acc++;
		return;
	    }
	}
	else if (wpc->w1.ave_level [0] [0] < 0x20 && wpc->w1.ave_level [0] [1] < 0x20) {
	    if (value) {
		putbit_0 (outbits);
	    }
	    else {
		wpc->w1.zeros_acc = 1;
		return;
	    }
	}
    }

    k = (wpc->w1.ave_level [0] [chan] + (wpc->w1.ave_level [0] [chan] >> 3) + 0x40) >> 7;
    k = count_bits (k);
    wpc->w1.ave_level [0] [chan] -= ((wpc->w1.ave_level [0] [chan] + 0x20) >> 6);
    wpc->w1.ave_level [0] [chan] += avalue;

    if (avalue >= bitset [k]) {
	avalue -= bitset [k];
	k = (wpc->w1.ave_level [1] [chan] + (wpc->w1.ave_level [1] [chan] >> 4) + 0x20) >> 6;
	k = count_bits (k);
	wpc->w1.ave_level [1] [chan] -= ((wpc->w1.ave_level [1] [chan] + 0x10) >> 5);
	wpc->w1.ave_level [1] [chan] += avalue;

	if (avalue >= bitset [k]) {
	    avalue -= bitset [k];
	    k = (wpc->w1.ave_level [2] [chan] + 0x10) >> 5;
	    k = count_bits (k);
	    wpc->w1.ave_level [2] [chan] -= ((wpc->w1.ave_level [2] [chan] + 0x8) >> 4);
	    wpc->w1.ave_level [2] [chan] += avalue;
	    ones_count = 2 + (avalue >> k);

	    // If the ones count is 24 or greater then we transmit exactly 24
	    // ones, a zero, and then we simply transmit the 24-bit unsigned
	    // value literally. This has virtually no statistical significance
	    // but does make it impossible for pathological samples to generate
	    // millions of bits.

	    if (ones_count >= 24) {
		putbits (0xffffff, 25, outbits);
		k = 24;
	    }
	    else
		putbits (bitmask [ones_count], ones_count + 1, outbits);
	}
	else {
	    putbit_1 (outbits);
	    putbit_0 (outbits);
	}
    }
    else
	putbit_0 (outbits);

    putbits (avalue & bitmask [k], k, outbits);

    if (value > 0) {
	putbit_0 (outbits);
    }
    else if (value < 0) {
	putbit_1 (outbits);
    }
}

//////////////////////////////////////////////////////////////////////////////
// Because streams of zeroes are accumulated in some cases, it is important //
// to call this function after all the data has been sent to send_word1().  //
//////////////////////////////////////////////////////////////////////////////

void flush_word1 (WavpackContext *wpc)
{
    if (wpc->w1.zeros_acc) {
	int cbits = count_bits (wpc->w1.zeros_acc);

	while (cbits--) {
	    putbit_1 (outbits);
	}

	putbit_0 (outbits);

	while (wpc->w1.zeros_acc > 1) {
	    putbit (wpc->w1.zeros_acc & 1, outbits);
	    wpc->w1.zeros_acc >>= 1;
	}

	wpc->w1.zeros_acc = 0;
    }
}

#endif

#ifdef UNPACK

#define inbits (&wpc->inbits)

//////////////////////////////////////////////////////////////////////////////
// Get a word from the "inbits" bitstream. Note that the "chan" value must  //
// either be always 0 or 1, or alternate 0, 1, 0, 1, 0, 1, etc.             //
//////////////////////////////////////////////////////////////////////////////

long __fastcall get_word1 (WavpackContext *wpc, int chan)
{
    ulong tmp1, tmp2, avalue, k;
    uint ones_count;

    // We accumulate zeroes when using the extreme decorrelator (it could be
    // used anytime, but that is when I added it and didn't want to break more
    // than required). However, this cannot be used with over 20-bit data
    // because bits over 20 are transmitted directly (in lossless, anyway) and
    // the two bitstreams would become tangled.

    if ((wpc->wphdr->flags & EXTREME_DECORR) && !(wpc->wphdr->flags & OVER_20)) {
	if (wpc->w1.zeros_acc) {
	    if (--wpc->w1.zeros_acc)
		return 0;
	}
	else if (wpc->w1.ave_level [0] [0] < 0x20 && wpc->w1.ave_level [0] [1] < 0x20) {
	    long mask;
	    int cbits;

	    for (cbits = 0; cbits < 33 && getbit (inbits); ++cbits);

	    if (cbits == 33)
		return WORD_EOF;

	    if (cbits < 2)
		wpc->w1.zeros_acc = cbits;
	    else {
		for (mask = 1, wpc->w1.zeros_acc = 0; --cbits; mask <<= 1)
		    if (getbit (inbits))
			wpc->w1.zeros_acc |= mask;

		wpc->w1.zeros_acc |= mask;
	    }

	    if (wpc->w1.zeros_acc)
		return 0;
	}
    }

    // count consecutive ones in bitstream, > 25 indicates error (or EOF)

    for (ones_count = 0; ones_count < 25 && getbit (inbits); ++ones_count);

    if (ones_count == 25)
	return WORD_EOF;

    k = (wpc->w1.ave_level [0] [chan] + (wpc->w1.ave_level [0] [chan] >> 3) + 0x40) >> 7;
    k = count_bits (k);

    if (ones_count == 0) {
	getbits (&avalue, k, inbits);
	avalue &= bitmask [k];
    }
    else {
	tmp1 = bitset [k];
	k = (wpc->w1.ave_level [1] [chan] + (wpc->w1.ave_level [1] [chan] >> 4) + 0x20) >> 6;
	k = count_bits (k);

	if (ones_count == 1) {
	    getbits (&avalue, k, inbits);
	    avalue &= bitmask [k];
	}
	else {
	    tmp2 = bitset [k];

	    // If the ones count is exactly 24, then next 24 bits are literal

	    if (ones_count == 24) {
		getbits (&avalue, 24, inbits);
		avalue &= 0xffffff;
	    }
	    else {
		k = (wpc->w1.ave_level [2] [chan] + 0x10) >> 5;
		k = count_bits (k);
		getbits (&avalue, k, inbits);
		avalue = (avalue & bitmask [k]) + (bitset [k] * (ones_count - 2));
	    }

	    wpc->w1.ave_level [2] [chan] -= ((wpc->w1.ave_level [2] [chan] + 0x8) >> 4);
	    wpc->w1.ave_level [2] [chan] += avalue;
	    avalue += tmp2;
	}

	wpc->w1.ave_level [1] [chan] -= ((wpc->w1.ave_level [1] [chan] + 0x10) >> 5);
	wpc->w1.ave_level [1] [chan] += avalue;
	avalue += tmp1;
    }

    wpc->w1.ave_level [0] [chan] -= ((wpc->w1.ave_level [0] [chan] + 0x20) >> 6);
    wpc->w1.ave_level [0] [chan] += avalue;

    return (avalue && getbit (inbits)) ? -avalue : avalue;
}

//////////////////////////////////////////////////////////////////////////////
// Get a word from the "inbits" bitstream using the old, fixed block, Rice  //
// method. This format is no longer written.                                // 
//////////////////////////////////////////////////////////////////////////////

#define NUM_SAMPLES 128

long __fastcall get_old_word1 (WavpackContext *wpc, int chan)
{
    ulong avalue;
    uint bc, k;

    if (!wpc->w1.index [chan]) {

	int guess_k = (wpc->w1.ave_k [chan] + 128) >> 8, ones;

	for (ones = 0; ones < 72 && getbit (inbits); ++ones);

	if (ones == 72)
	    return WORD_EOF;

	if (ones % 3 == 1)
	    wpc->w1.k_value [chan] = guess_k - (ones / 3) - 1;
	else
	    wpc->w1.k_value [chan] = guess_k + ones - ((ones + 1) / 3);

	wpc->w1.ave_k [chan] -= (wpc->w1.ave_k [chan] + 0x10) >> 5;
	wpc->w1.ave_k [chan] += wpc->w1.k_value [chan] << 3;
    }

    if (++wpc->w1.index [chan] == NUM_SAMPLES)
	wpc->w1.index [chan] = 0;

    k = wpc->w1.k_value [chan];
    getbits (&avalue, k, inbits);

    for (bc = 0; bc < 32 && getbit (inbits); ++bc);

    if (bc == 32)
	return WORD_EOF;

    avalue = (avalue & bitmask [k]) + bitset [k] * bc;
    return (avalue && getbit (inbits)) ? -avalue : avalue;
}

#endif
