////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2002 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// words3.c

// This module provides the entropy word encoding and decoding functions for
// all files generated by WavPack 3.0, both lossless and lossy. Later, the
// higher compression lossless modes started using a variation of Rice codes
// (words1.c) and the hybrid mode (which replaced the old lossy mode)
// introduced another variation of Rice codes (words4.c). Because the old
// lossy mode is no longer written, this module now handles decoding only of
// lossy data (i.e. wphdr->bits != 0).
//
// The technique is the encoder estimates how many bits the next sample will
// take to transmit (not counting the implied MSB but counting the sign bit)
// based on the exponentially decaying average bits used by previous samples.
// Then, this is subtracted from the actual number of bits that the sample will
// require and this becomes the "delta_dbits". This value (which is usually in
// the range +/-1) is transmitted according to a fixed bitcode followed by the
// actual data bits and the sign. It's actually simpler to implement than Rice
// codes (most of the complexity here is optimization), provides almost the
// same compression efficiency, and provides a simple method for lossy
// encoding.

#include "wavpack.h"

#include <stdlib.h>
#include <string.h>


///////////////////////////// local table storage ////////////////////////////

const long bitset [] = {
    1L << 0, 1L << 1, 1L << 2, 1L << 3,
    1L << 4, 1L << 5, 1L << 6, 1L << 7,
    1L << 8, 1L << 9, 1L << 10, 1L << 11,
    1L << 12, 1L << 13, 1L << 14, 1L << 15,
    1L << 16, 1L << 17, 1L << 18, 1L << 19,
    1L << 20, 1L << 21, 1L << 22, 1L << 23,
    1L << 24, 1L << 25, 1L << 26, 1L << 27,
    1L << 28, 1L << 29, 1L << 30, 1L << 31
};

const long bitmask [] = {
    (1L << 0) - 1, (1L << 1) - 1, (1L << 2) - 1, (1L << 3) - 1,
    (1L << 4) - 1, (1L << 5) - 1, (1L << 6) - 1, (1L << 7) - 1,
    (1L << 8) - 1, (1L << 9) - 1, (1L << 10) - 1, (1L << 11) - 1,
    (1L << 12) - 1, (1L << 13) - 1, (1L << 14) - 1, (1L << 15) - 1,
    (1L << 16) - 1, (1L << 17) - 1, (1L << 18) - 1, (1L << 19) - 1,
    (1L << 20) - 1, (1L << 21) - 1, (1L << 22) - 1, (1L << 23) - 1,
    (1L << 24) - 1, (1L << 25) - 1, (1L << 26) - 1, (1L << 27) - 1,
    (1L << 28) - 1, (1L << 29) - 1, (1L << 30) - 1, 0x7fffffff
};

const char nbits_table [] = {
    0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4,	// 0 - 15
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,	// 16 - 31
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,	// 32 - 47
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,	// 48 - 63
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,	// 64 - 79
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,	// 80 - 95
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,	// 96 - 111
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,	// 112 - 127
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 128 - 143
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 144 - 159
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 160 - 175
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 176 - 191
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 192 - 207
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 208 - 223
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,	// 224 - 239
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8	// 240 - 255
};


//////////////////////////////////////////////////////////////////////////////
// This function initializes everything required to send or receive words   //
// with this module and must be called BEFORE any other function in this    //
// module.                                                                  //
//////////////////////////////////////////////////////////////////////////////

void init_word3 (WavpackContext *wpc)
{
    CLEAR (wpc->w3);
}

#ifdef PACK

// This table provides a quick lookup method of determining the bit pattern
// and number of bits to write for each delta_dbits in the range of -16 to +8.
// Values outside this range (which happen VERY infrequently) are calculated.

static const struct {
    ulong cvalue;
    uchar cbits;
} cvalues [] = {
    (1L << 23) - 1, 24,		// delta_dbits = -16
    (1L << 21) - 1, 22,		// delta_dbits = -15
    (1L << 20) - 1, 21,		// delta_dbits = -14
    (1L << 18) - 1, 19,		// delta_dbits = -13
    (1L << 17) - 1, 18,		// delta_dbits = -12
    (1L << 15) - 1, 16,		// delta_dbits = -11
    (1L << 14) - 1, 15,		// delta_dbits = -10
    (1L << 12) - 1, 13,		// delta_dbits = -9
    (1L << 11) - 1, 12,		// delta_dbits = -8
    (1L << 9) - 1, 10,		// delta_dbits = -7
    (1L << 8) - 1, 9,		// delta_dbits = -6
    (1L << 6) - 1, 7,		// delta_dbits = -5
    (1L << 5) - 1, 6,		// delta_dbits = -4
    (1L << 3) - 1, 4,		// delta_dbits = -3
    (1L << 2) - 1, 3,		// delta_dbits = -2
    2L, 2,			// delta_dbits = -1
    0L, 2,			// delta_dbits = 0
    1L, 2,			// delta_dbits = +1
    (1L << 4) - 1, 5,		// delta_dbits = +2
    (1L << 7) - 1, 8,		// delta_dbits = +3
    (1L << 10) - 1, 11,		// delta_dbits = +4
    (1L << 13) - 1, 14,		// delta_dbits = +5
    (1L << 16) - 1, 17,		// delta_dbits = +6
    (1L << 19) - 1, 20,		// delta_dbits = +7
    (1L << 22) - 1, 23		// delta_dbits = +8
};

#define outbits (&wpc->outbits)

//////////////////////////////////////////////////////////////////////////////
// Send the specified word to the "outbits" bitstream (which must have been //
// previously opened with bs_open_write()). Note that the "chan" value must //
// either be always 0 or 1, or alternate 0, 1, 0, 1, 0, 1, etc. Originally  //
// the return value was the value actually transmitted in lossy mode, but   //
// since the lossy mode is no longer supported for writing, the value is    //
// simply returned unchanged.                                               //
//////////////////////////////////////////////////////////////////////////////

long __fastcall send_word3 (WavpackContext *wpc, long value, int chan)
{
    int dbits, cbits, delta_dbits;
    ulong avalue;

    if ((avalue = labs (value)) < 256)
	dbits = nbits_table [avalue];
    else if (avalue < (1L << 16))
	dbits = nbits_table [avalue >> 8] + 8;
    else if (avalue < (1L << 24))
	dbits = nbits_table [avalue >> 16] + 16;
    else
	dbits = nbits_table [avalue >> 24] + 24;

    if (chan) {
	delta_dbits = dbits - (wpc->w3.ave_dbits [1] >> 8) - 1;
	wpc->w3.ave_dbits [1] -= (wpc->w3.ave_dbits [1] + 0x10) >> 5;
	wpc->w3.ave_dbits [1] += dbits << 3;
    }
    else {
	delta_dbits = dbits - (wpc->w3.ave_dbits [0] >> 8) - 1;
	wpc->w3.ave_dbits [0] -= (wpc->w3.ave_dbits [0] + 0x10) >> 5;
	wpc->w3.ave_dbits [0] += dbits << 3;
    }

    if (delta_dbits >= -16 && delta_dbits <= 8) {	// -16 to +8 uses table
	cbits = cvalues [delta_dbits += 16].cbits;
	putbits (cvalues [delta_dbits].cvalue, cbits, outbits);
    }
    else {
	if (delta_dbits > 0)
	    cbits = (delta_dbits * 3) - 1;
	else
	    cbits = -delta_dbits + (-delta_dbits >> 1);

	while (--cbits)
	    putbit_1 (outbits);

	putbit_0 (outbits);
    }

    if (dbits) {
	putbits (avalue & bitmask [(value < 0) ? dbits : dbits - 1], dbits, outbits);
    }

    return value;
}

#endif

#ifdef UNPACK

// This table provides a quick lookup method for determining the next
// delta_dbits value read from the bitstream. The next 8 bits in the bitstream
// is used as an index, and the table provides the delta_dbits value and the
// actual number of bits consumed.

static const struct { char ddbits, ubits; } cbits_table [] = {
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 0 - 7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* 8 - f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 10 - 17 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -4,6,	/* 18 - 1f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 20 - 27 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* 28 - 2f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 30 - 37 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -5,7,	/* 38 - 3f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 40 - 47 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* 48 - 4f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 50 - 57 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -4,6,	/* 58 - 5f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 60 - 67 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* 68 - 6f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 70 - 77 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 3,8,	/* 78 - 7f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 80 - 87 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* 88 - 8f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* 90 - 97 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -4,6,	/* 98 - 9f */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* a0 - a7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* a8 - af */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* b0 - b7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -5,7,	/* b8 - bf */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* c0 - c7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* c8 - cf */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* d0 - d7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -4,6,	/* d8 - df */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* e0 - e7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 2,5,	/* e8 - ef */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, -3,4,	/* f0 - f7 */
    0,2, 1,2, -1,2, -2,3, 0,2, 1,2, -1,2, 0,8,	/* f8 - ff */
};

#define inbits (&wpc->inbits)

//////////////////////////////////////////////////////////////////////////////
// Get a word from the "inbits" bitstream. Note that the "chan" value must  //
// either be always 0 or 1, or alternate 0, 1, 0, 1, 0, 1, etc. Because     //
// this function uses the cbits_table above and this method requires the    //
// next 8 bits in the bitstream to be available, we cannot use the standard //
// macros alone to read the bitstream (getbit() and getbits()) but instead  //
// directly access the members of the structure. This should be remembered  //
// if the Bitstream structure or its macros is updated.                     //
//////////////////////////////////////////////////////////////////////////////

long __fastcall get_word3 (WavpackContext *wpc, int chan)
{
    int cbits, delta_dbits, dbits, next8;
    long value;

    if (inbits->bc < 8) {
	if (++(inbits->ptr) == inbits->end)
	    bs_read (inbits);

	next8 = (inbits->sr |= *(inbits->ptr) << inbits->bc) & 0xff;
	inbits->bc += 8;
    }
    else
	next8 = inbits->sr & 0xff;

    if (next8 != 0xff) {
	delta_dbits = cbits_table [next8].ddbits;
	inbits->sr >>= cbits_table [next8].ubits;
	inbits->bc -= cbits_table [next8].ubits;
    }
    else {
	cbits = 0;

	do {
	    inbits->sr >>= 8;

	    if ((cbits += 8) == 72)
		return WORD_EOF;

	    if (++(inbits->ptr) == inbits->end)
		bs_read (inbits);

	    next8 = (inbits->sr |= *(inbits->ptr) << (inbits->bc - 8)) & 0xff;

	} while (next8 == 0xff);

	do ++cbits;
	while (getbit (inbits));

	if (!((cbits + 1) % 3))
	    delta_dbits = (cbits + 1) / 3;
	else
	    delta_dbits = -(cbits - cbits / 3);
    }

    if (chan) {
	dbits = (wpc->w3.ave_dbits [1] >> 8) + 1 + delta_dbits;
	wpc->w3.ave_dbits [1] -= (wpc->w3.ave_dbits [1] + 0x10) >> 5;
	wpc->w3.ave_dbits [1] += dbits << 3;
    }
    else {
	dbits = (wpc->w3.ave_dbits [0] >> 8) + 1 + delta_dbits;
	wpc->w3.ave_dbits [0] -= (wpc->w3.ave_dbits [0] + 0x10) >> 5;
	wpc->w3.ave_dbits [0] += dbits << 3;
    }

    if (dbits < 0 || dbits > 24)
	return WORD_EOF;

    if (!dbits)
	return 0L;

    if (wpc->wphdr->bits && dbits > wpc->wphdr->bits) {
	getbits (&value, wpc->wphdr->bits, inbits);

	if (value & bitset [wpc->wphdr->bits - 1])
	    return -(value & bitmask [wpc->wphdr->bits]) << (dbits - wpc->wphdr->bits);
	else
	    return ((value & bitmask [wpc->wphdr->bits - 1]) | bitset [wpc->wphdr->bits - 1]) << (dbits - wpc->wphdr->bits);
    }
    else {
	getbits (&value, dbits, inbits);

	if (value & bitset [dbits - 1])
	    return -(value & bitmask [dbits]);
	else
	    return (value & bitmask [dbits - 1]) | bitset [dbits - 1];
    }
}

#endif
