////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2003 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software Licence (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// wvselfx.c

// This module provides the "stub" for the self-extraction WavPack files.

#include "wavpack.h"

#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <stdlib.h>
#include <conio.h>
#include <dir.h>
#include <dos.h>
#include <math.h>

static char *sign_on = "\n\
 WavPack Self-Extracting Hybrid Lossless Archive  Version %s %s\n\
 Copyright (c) 1998 - 2003 Conifer Software.  All Rights Reserved.\n\n";

static char *usage = "\
 Usage:   To create a self-extracting archive from an existing WavPack file,\n\
          simply prepend this .EXE file to any .WV file using the MS-DOS COPY\n\
          command. Be sure to use the /B switch (binary) and give the output\n\
          the .EXE extension.\n\n\
          For new archives you can do this automatically with the -e switch\n\
          in WavPack; just make sure that this file is in the same directory\n\
          as the WAVPACK.EXE file (or in the current path).\n\n\
 Example: COPY/B WVSELFX.EXE+SAMPLE.WV SAMPLE.EXE\n\n\
 Web:     Visit www.wavpack.com for latest version and info\n";

static int unpack_file (char *infilename, char *outfilename, int offset);
static int unpack_audio (WavpackHeader *wphdr, HANDLE infile, HANDLE in2file, HANDLE outfile);
static void display_progress (double file_progress);

#define NO_ERROR 0L
#define SOFT_ERROR 1
#define HARD_ERROR 2

int main (argc, argv) int argc; char **argv;
{
    char *infilename, *outfilename;
    struct ffblk ffblk;
    HANDLE self_file;

    if (!argc)
	return 1;

    fprintf (stderr, sign_on, VERSION_STR, DATE_STR);
    setup_break ();

    if (findfirst (*argv, &ffblk, 0) == 0 &&
	strlen (filespec_name (*argv)) >= strlen (ffblk.ff_name))
	    strcpy (filespec_name (*argv), ffblk.ff_name);

    if ((self_file = CreateFile (*argv, GENERIC_READ, FILE_SHARE_READ, NULL,
	OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) != INVALID_HANDLE_VALUE &&
	GetFileSize (self_file, NULL) > SELF_SIZE + sizeof (WavpackHeader)) {
	    CloseHandle (self_file);
	    infilename = *argv;
	    outfilename = malloc (strlen (infilename) + 10);
	    strcpy (outfilename, infilename);

	    if (filespec_ext (outfilename))
		*filespec_ext (outfilename) = '\0';

	    if (unpack_file (infilename, outfilename, SELF_SIZE)) {
		fprintf (stderr, "\npress any key to continue...\n");
		while (!kbhit ());
		return 1;
	    }
	    else {
		Sleep (2000);
		return 0;
	    }
    }

    if (self_file != INVALID_HANDLE_VALUE)
	CloseHandle (self_file);

    printf ("%s", usage);
    return 0;
}


static int unpack_file (char *infilename, char *outfilename, int offset)
{
    long total_samples, total_header_bytes, outfile_length, in2file_length;
    int result, drive_type = DRIVE_UNKNOWN;
    HANDLE infile, in2file, outfile;
    RiffChunkHeader RiffChunkHeader;
    WavpackHeader WavpackHeader;
    ChunkHeader ChunkHeader;
    WaveHeader WaveHeader;
    struct time time1, time2;
    char *drive = "C:\\";
    char *in2filename;
    FILETIME time;
    double dtime;
    DWORD bcount;

    if (isalpha (*outfilename)) {
	drive [0] = *outfilename;
	drive_type = GetDriveType (drive);
    }

    if ((infile = CreateFile (infilename, GENERIC_READ, FILE_SHARE_READ, NULL,
	OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
	    error_line ("can't open file %s!", filespec_name (infilename));
	    return SOFT_ERROR;
    }

    GetFileTime (infile, NULL, NULL, &time);
    SetFilePointer (infile, offset, NULL, FILE_BEGIN);

    if (ReadFile (infile, &RiffChunkHeader, sizeof (RiffChunkHeader), &bcount, NULL) &&
	bcount == sizeof (RiffChunkHeader) && !strncmp (RiffChunkHeader.ckID, "RIFF", 4) &&
	!strncmp (RiffChunkHeader.formType, "WAVE", 4)) {

	    while (1) {

		if (!ReadFile (infile, &ChunkHeader, sizeof (ChunkHeader), &bcount, NULL) ||
		    bcount != sizeof (ChunkHeader)) {
			error_line ("%s is not a valid Wavpack file!", filespec_name (infilename));
			CloseHandle (infile);
			return SOFT_ERROR;
		}

		if (!strncmp (ChunkHeader.ckID, "fmt ", 4)) {

		    if (ChunkHeader.ckSize < sizeof (WaveHeader) ||
			!ReadFile (infile, &WaveHeader, sizeof (WaveHeader), &bcount, NULL) ||
			bcount != sizeof (WaveHeader)) {
			    error_line ("%s is not a valid Wavpack file!", filespec_name (infilename));
			    CloseHandle (infile);
			    return SOFT_ERROR;
		    }

		    if (ChunkHeader.ckSize > sizeof (WaveHeader))
			SetFilePointer (infile, (ChunkHeader.ckSize + 1 - sizeof (WaveHeader)) & ~1L, NULL, FILE_CURRENT);
		}
		else if (!strncmp (ChunkHeader.ckID, "data", 4)) {
#ifdef VER2
		    total_samples = ChunkHeader.ckSize / WaveHeader.NumChannels /
			((WaveHeader.BitsPerSample > 16) ? 3 : 2);
#endif
		    break;
		}
		else
		    SetFilePointer (infile, (ChunkHeader.ckSize + 1) & ~1L, NULL, FILE_CURRENT);
	    }

	    total_header_bytes = SetFilePointer (infile, 0, NULL, FILE_CURRENT) - offset;
    }
    else {
	SetFilePointer (infile, offset, NULL, FILE_BEGIN);
	total_header_bytes = 0;
    }

    if (!ReadFile (infile, &WavpackHeader, sizeof (WavpackHeader), &bcount, NULL) ||
	bcount != sizeof (WavpackHeader) || strncmp (WavpackHeader.ckID, "wvpk", 4)) {
	    error_line ("%s is not a valid Wavpack file!", filespec_name (infilename));
	    CloseHandle (infile);
	    return SOFT_ERROR;
    }

    if (WavpackHeader.version == 3) {

	if (WavpackHeader.flags & EXTREME_DECORR) {

	    if ((WavpackHeader.flags & NOT_STORED_FLAGS) ||
		((WavpackHeader.bits) &&
		(((WavpackHeader.flags & NEW_HIGH_FLAG) &&
		(WavpackHeader.flags & (FAST_FLAG | HIGH_FLAG))) ||
		(WavpackHeader.flags & CROSS_DECORR)))) {
		    error_line ("not compatible with this version of WavPack file!");
		    CloseHandle (infile);
		    return SOFT_ERROR;
	    }

	    if (WavpackHeader.flags & CANCEL_EXTREME)
		WavpackHeader.flags &= ~(EXTREME_DECORR | CANCEL_EXTREME);
	}
	else
	    WavpackHeader.flags &= ~CROSS_DECORR;
    }
    else {
	error_line ("not compatible with this version of WavPack file!");
	CloseHandle (infile);
	return SOFT_ERROR;
    }

    if (WavpackHeader.version >= 3) {
	if (WavpackHeader.extension [0]) {
	    strcat (outfilename, ".");
	    strcat (outfilename, WavpackHeader.extension);
	}
    }
    else
	strcat (outfilename, anylower (filespec_name (outfilename)) ? ".wav" : ".WAV");

    if (drive_type == DRIVE_REMOVABLE || drive_type == DRIVE_CDROM ||
	(outfile = CreateFile (outfilename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
	CREATE_NEW, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
	    OPENFILENAME ofn;

	    memset (&ofn, 0, sizeof (ofn));
	    ofn.lStructSize = sizeof (ofn);

	    if (!(WavpackHeader.flags & RAW_FLAG))
		ofn.lpstrFilter = "Windows PCM (.wav)\0*.wav\0";

	    ofn.lpstrFile = malloc (ofn.nMaxFile = 512);
	    strcpy (ofn.lpstrFile, filespec_name (outfilename));
	    ofn.lpstrInitialDir = malloc (512);
	    GetCurrentDirectory (512, (char *) ofn.lpstrInitialDir);
	    ofn.lpstrTitle = "Extract To";
	    ofn.Flags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;

	    if (!GetSaveFileName (&ofn)) {
		CloseHandle (infile);
		return NO_ERROR;
	    }

	    outfilename = ofn.lpstrFile;

	    if ((outfile = CreateFile (outfilename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
		CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
		    error_line ("can't create file %s!", filespec_name (outfilename));
		    CloseHandle (infile);
		    return SOFT_ERROR;
	    }
    }

    fprintf (stderr, "extracting %s,", filespec_name (outfilename));
    gettime (&time1);
    SetFilePointer (infile, offset, NULL, FILE_BEGIN);

    if (total_header_bytes) {
	char *buff = malloc (total_header_bytes);

	if (!ReadFile (infile, buff, total_header_bytes, &bcount, NULL) ||
	    bcount != total_header_bytes ||
	    !WriteFile (outfile, buff, total_header_bytes, &bcount, NULL) ||
	    bcount != total_header_bytes) {

		error_line ("can't write .WAV header, disk probably full!");
		CloseHandle (infile);
		CloseHandle (outfile);
		DeleteFile (outfilename);
		free (buff);
		return SOFT_ERROR;
	}

	free (buff);
    }

    if (WavpackHeader.bits && (WavpackHeader.flags & NEW_HIGH_FLAG)) {
	in2filename = malloc (strlen (infilename) + 10);
	strcpy (in2filename, infilename);

	if (filespec_ext (in2filename))
	    *filespec_ext (in2filename) = '\0';

	strcat (in2filename, anylower (filespec_name (in2filename)) ?
	    ".wvc" : ".WVC");

	in2file = CreateFile (in2filename, GENERIC_READ, FILE_SHARE_READ, NULL,
	    OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

	free (in2filename);
    }
    else
	in2file = INVALID_HANDLE_VALUE;

    if (in2file == INVALID_HANDLE_VALUE)
	WavpackHeader.flags &= ~WVC_FLAG;
    else
	WavpackHeader.flags |= WVC_FLAG;

    switch (WavpackHeader.version) {
#ifdef VER2
	case 1:
	    WavpackHeader.bits = 0;
	    SetFilePointer (infile, -2, NULL, FILE_CURRENT);

	case 2:
	    WavpackHeader.total_samples = total_samples;
	    WavpackHeader.flags = WaveHeader.NumChannels == 1 ? MONO_FLAG : 0;
	    WavpackHeader.shift = 16 - WaveHeader.BitsPerSample;
	    SetFilePointer (infile, 12, NULL, FILE_CURRENT);
	    result = unpack_audio (&WavpackHeader, infile, in2file, outfile);
	    break;
#endif
	case 3:
	    SetFilePointer (infile, sizeof (WavpackHeader), NULL, FILE_CURRENT);
	    result = unpack_audio (&WavpackHeader, infile, in2file, outfile);

	    if (in2file != INVALID_HANDLE_VALUE) {
		in2file_length = GetFileSize (in2file, NULL);
		CloseHandle (in2file);		
	    }
	    else
		in2file_length = 0;

	    if (result == NO_ERROR && !(WavpackHeader.flags & RAW_FLAG)) {
		char c, ones [16];

		if (ReadFile (infile, &ones, sizeof (ones), &bcount, NULL) && bcount == sizeof (ones)) {

		    for (c = 0; c < sizeof (ones) && !~ones [c]; ++c);

		    if (c == sizeof (ones))
			while (ReadFile (infile, &c, 1, &bcount, NULL) && bcount == 1)
			    if (!WriteFile (outfile, &c, 1, &bcount, NULL) || bcount != 1) {
				error_line ("can't write .WAV data, disk probably full!");
				SetFilePointer (outfile, 0, NULL, FILE_BEGIN);
				SetEndOfFile (outfile);
				result = SOFT_ERROR;
			    }
		}
	    }

	    if (result == NO_ERROR && (WavpackHeader.flags & RAW_FLAG) &&
		WavpackHeader.extra_bc && (!WriteFile (outfile, WavpackHeader.extras,
		WavpackHeader.extra_bc, &bcount, NULL) || bcount != WavpackHeader.extra_bc)) {
		    error_line ("can't write raw data, disk probably full!");
		    SetFilePointer (outfile, 0, NULL, FILE_BEGIN);
		    SetEndOfFile (outfile);
		    result = SOFT_ERROR;
	    }

	    break;
    }

    outfile_length = GetFileSize (outfile, NULL);
    SetFileTime (outfile, &time, &time, &time);

    if (!CloseHandle (outfile)) {
	error_line ("can't properly close file %s!", filespec_name (outfilename));
	result = SOFT_ERROR;
    }

    gettime (&time2);
    dtime = time2.ti_sec * 100.0 + time2.ti_hund + time2.ti_min * 6000.0 + time2.ti_hour * 360000.00;
    dtime -= time1.ti_sec * 100.0 + time1.ti_hund + time1.ti_min * 6000.0 + time1.ti_hour * 360000.00;

    if ((dtime /= 100.0) < 0.0)
	dtime += 86400.0;

    if (result == NO_ERROR) {
	if (in2file == INVALID_HANDLE_VALUE)
	    error_line ("restored %s in %.2f secs (%s comp: %.2f%%)",
		FN_FIT (outfilename), dtime,
		(WavpackHeader.bits && WavpackHeader.crc != WavpackHeader.crc2) ? "lossy" : "lossless",
		100 - (GetFileSize (infile, NULL) * 100.0 / outfile_length));
	else
	    error_line ("restored %s in %.2f secs (lossless comp: %.2f%%)",
		FN_FIT (outfilename), dtime, 100 - ((GetFileSize (infile, NULL) + in2file_length) * 100.0 / outfile_length));
    }

    if (!outfile_length)
	DeleteFile (outfilename);

    CloseHandle (infile);
    return result;
}


static int unpack_audio (WavpackHeader *wphdr, HANDLE infile, HANDLE in2file, HANDLE outfile)
{
    ulong bytes_written, bytes_to_write, samples_remaining;
    int valid_outfile = (outfile != INVALID_HANDLE_VALUE);
    uint samples_to_unpack, samples_unpacked;
    int flags = wphdr->flags;
    double progress = 0.0;
    WavpackContext wpc;
    uchar *buffer;

    if (!wphdr->total_samples)
	return NO_ERROR;

    CLEAR (wpc);
    wpc.wphdr = wphdr;
    wpc.inbits.bufsiz = wpc.in2bits.bufsiz = INBUF_SIZE;
    buffer = malloc (OUTBUF_SIZE);
    unpack_init (&wpc);

    if (!buffer || bs_open_read (&wpc.inbits, infile) || ((flags & WVC_FLAG) && bs_open_read (&wpc.in2bits, in2file))) {
	error_line ("internal error!");

	if (valid_outfile)
	    DoTruncateFile (outfile);

	if (buffer)
	    free (buffer);

	return HARD_ERROR;
    }

    fprintf (stderr, "   0%% done...");
    display_progress (progress);
    samples_remaining = wphdr->total_samples;

    do {
	samples_to_unpack = OUTBUF_SIZE / BYTES_PER_SAMPLE;

	if (samples_to_unpack > samples_remaining)
	    samples_to_unpack = samples_remaining;

	samples_unpacked = unpack_samples (&wpc, buffer, samples_to_unpack);
	bytes_to_write = samples_unpacked * BYTES_PER_SAMPLE;
	samples_remaining -= samples_unpacked;

	if (valid_outfile &&
	    (!DoWriteFile (outfile, buffer, bytes_to_write, &bytes_written, NULL) ||
	    bytes_written != bytes_to_write)) {
		error_line ("can't write .WAV data, disk probably full!");

		if (flags & WVC_FLAG)
		    bs_close_read (&wpc.in2bits);

		bs_close_read (&wpc.inbits);
		DoTruncateFile (outfile);
		free (buffer);
		return HARD_ERROR;
	}

	if (check_break ()) {
	    fprintf (stderr, "^C\n");

	    if (flags & WVC_FLAG)
		bs_close_read (&wpc.in2bits);

	    bs_close_read (&wpc.inbits);

	    if (valid_outfile)
		DoTruncateFile (outfile);

	    free (buffer);
	    return SOFT_ERROR;
	}

	if (progress != floor (((double)(wphdr->total_samples - samples_remaining) / wphdr->total_samples) * 100.0 + 0.5)) {
	    progress = (double)(wphdr->total_samples - samples_remaining) / wphdr->total_samples;
	    display_progress (progress);
	    progress = floor (progress * 100.0 + 0.5);
	    fprintf (stderr, "\b\b\b\b\b\b\b\b\b\b\b\b%3d%% done...", (int) progress);
	}

    } while (samples_remaining && samples_unpacked == samples_to_unpack);

    if (flags & WVC_FLAG)
	bs_close_read (&wpc.in2bits);

    bs_close_read (&wpc.inbits);
    free (buffer);

    if (samples_remaining || (wphdr->version == 3 && unpack_crc (&wpc) != ((flags & WVC_FLAG) ? wphdr->crc2 : wphdr->crc))) {
	if (valid_outfile)
	    error_line ("CRC error detected, file not correctly uncompressed!!");
	else
	    error_line ("CRC error detected, file does not verify!!");

	return SOFT_ERROR;
    }

    return NO_ERROR;
}

static void display_progress (double file_progress)
{
    char title [40];

    sprintf (title, "%d%% (WavPack)", (int) ((file_progress * 100.0) + 0.5));
    SetConsoleTitle (title);
}
