////////////////////////////////////////////////////////////////////////////
//                           **** WAVPACK ****                            //
//                  Hybrid Lossless Wavefile Compressor                   //
//              Copyright (c) 1998 - 2003 Conifer Software.               //
//                          All Rights Reserved.                          //
//      Distributed under the BSD Software License (see license.txt)      //
////////////////////////////////////////////////////////////////////////////

// wvunpack.c

// This is the main module for the WavPack command-line decompressor.

#include "wavpack.h"

#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <stdlib.h>
#include <dos.h>
#include <math.h>

#ifdef __BORLANDC__
#include <dir.h>
#else
#include <sys/types.h>
#include <sys/timeb.h>
#endif

#ifndef __WIN32__
#include <sys/stat.h>
#include <string.h>
#endif

///////////////////////////// local variable storage //////////////////////////

#ifdef __WIN32__
static const char *sign_on = "\n\
 WVUNPACK  Hybrid Lossless Wavefile Decompressor  Win32 Version %s %s\n\
 Copyright (c) 1998 - 2003 Conifer Software.  All Rights Reserved.\n\n";
#else
static const char *sign_on = "\n\
 WVUNPACK  Hybrid Lossless Wavefile Decompressor  DOS Version %s %s\n\
 Copyright (c) 1998 - 2003 Conifer Software.  All Rights Reserved.\n\n";
#endif

static const char *usage = "\
 Usage:   WVUNPACK [-options] [@]infile[.wv] [[@]outfile[.wav]|outpath|-]\n\
             (infile may contain wildcards: ?,*)\n\n\
 Options: -d  = delete source file if successful (use with caution!)\n\
          -t  = copy input file's time stamp to output file\n\
          -v  = verify source data only (no output file created)\n\
          -y  = yes to overwrite warning (use with caution!)\n\n\
 Web:     Visit www.wavpack.com for latest version and info\n";


static char overwrite_all, copy_time, delete_source, add_extension;
static int num_files, file_index;

/////////////////////////// local function declarations ///////////////////////

static int unpack_file (char *infilename, char *outfilename);
static int unpack_audio (WavpackHeader *wphdr, HANDLE infile, HANDLE in2file, HANDLE outfile);
static void display_progress (double file_progress);

#define NO_ERROR 0L
#define SOFT_ERROR 1
#define HARD_ERROR 2

//////////////////////////////////////////////////////////////////////////////
// The "main" function for the command-line WavPack decompressor.           //
//////////////////////////////////////////////////////////////////////////////

int main (argc, argv) int argc; char **argv;
{
    int verify_only = 0, usage_error = 0, filelist = 0;
    char *infilename = NULL, *outfilename = NULL;
    char outpath, **matches = NULL;
    int result, i;

#ifdef __BORLANDC__
	struct ffblk ffblk;
#else
	struct _finddata_t _finddata_t;
#endif

    fprintf (stderr, sign_on, VERSION_STR, DATE_STR);

    // loop through command-line arguments

    while (--argc) {

	if ((**++argv == '-' || **argv == '/') && (*argv)[1])
	    while (*++*argv)
		switch (**argv) {

		    case 'Y': case 'y':
			overwrite_all = 1;
			break;

		    case 'D': case 'd':
			delete_source = 1;
			break;

		    case 'T': case 't':
			copy_time = 1;
			break;

		    case 'V': case 'v':
			verify_only = 1;
			break;

		    default:
			error_line ("illegal option: %c !", **argv);
			usage_error = 1;
		}
	else {
	    if (!infilename) {
		infilename = malloc (strlen (*argv) + MAX_PATH);
		strcpy (infilename, *argv);
	    }
	    else if (!outfilename) {
		outfilename = malloc (strlen (*argv) + MAX_PATH);
		strcpy (outfilename, *argv);
	    }
	    else {
		error_line ("extra unknown argument: %s !", *argv);
		usage_error = 1;
	    }
	}
    }

    // check for various command-line argument problems

    if (verify_only && delete_source) {
	error_line ("can't delete in verify mode!");
	delete_source = 0;
    }

    if (verify_only && outfilename) {
	error_line ("outfile specification and verify mode are incompatible!");
	usage_error = 1;
    }

    if (!infilename) {
	printf ("%s", usage);
	return 0;
    }

    if (usage_error)
	return 0;

    setup_break ();

    // If the infile specification begins with a '@', then it actually points
    // to a file that contains the names of the files to be converted. This
    // was included for use by Wim Speekenbrink's frontends, but could be used
    // for other purposes.

    if (infilename [0] == '@') {
	FILE *list = fopen (infilename+1, "rt");
	int c;

	if (list == NULL) {
	    error_line ("file %s not found!", infilename+1);
	    return 1;
	}

	while ((c = getc (list)) != EOF) {

	    while (c == '\n')
		c = getc (list);

	    if (c != EOF) {
		char *fname = malloc (MAX_PATH);
		int ci = 0;

		do
		    fname [ci++] = c;
		while ((c = getc (list)) != '\n' && c != EOF && ci < MAX_PATH);

		fname [ci++] = '\0';
		matches = realloc (matches, ++num_files * sizeof (*matches));
		matches [num_files - 1] = realloc (fname, ci);
	    }
	}

	fclose (list);
	free (infilename);
	infilename = NULL;
	filelist = 1;
    }
    else {
	if (!filespec_ext (infilename))
	    strcat (infilename, anylower (infilename) ? ".wv" : ".WV");

	// search for and store any filenames that match the user supplied spec

#ifdef __BORLANDC__
	if (findfirst (infilename, &ffblk, 0) == 0) {
	    do {
		matches = realloc (matches, ++num_files * sizeof (*matches));
		matches [num_files - 1] = strdup (ffblk.ff_name);
	    } while (findnext (&ffblk) == 0);
	}
#else
	if ((i = _findfirst (infilename, &_finddata_t)) != -1L) {
	    do {
		if (!(_finddata_t.attrib & _A_SUBDIR)) {
		    matches = realloc (matches, ++num_files * sizeof (*matches));
		    matches [num_files - 1] = strdup (_finddata_t.name);
		}
	    } while (_findnext (i, &_finddata_t) == 0);

	    _findclose (i);
	}
#endif
    }

    // If the outfile specification begins with a '@', then it actually points
    // to a file that contains the output specification. This was included for
    // use by Wim Speekenbrink's frontends because certain filenames could not
    // be passed on the command-line, but could be used for other purposes.

    if (outfilename && outfilename [0] == '@') {
	FILE *list = fopen (outfilename+1, "rt");
	int c;

	if (list == NULL) {
	    error_line ("file %s not found!", outfilename+1);
	    return 1;
	}

	while ((c = getc (list)) == '\n');

	if (c != EOF) {
	    int ci = 0;

	    do
		outfilename [ci++] = c;
	    while ((c = getc (list)) != '\n' && c != EOF && ci < MAX_PATH);

	    outfilename [ci] = '\0';
	}
	else {
	    error_line ("output spec file is empty!");
	    fclose (list);
	    return 1;
	}

	fclose (list);
    }

    // if we found any files to process, this is where we start

    if (num_files) {
	if (outfilename && *outfilename != '-') {
	    outpath = (filespec_path (outfilename) != NULL);

	    if (num_files > 1 && !outpath) {
		error_line ("%s is not a valid output path", outfilename);
		return 1;
	    }
	}
	else
	    outpath = 0;

	add_extension = !outfilename || outpath || !filespec_ext (outfilename);

	// loop through and process files in list

	for (file_index = 0; file_index < num_files; ++file_index) {
#ifdef __WIN32__
	    if (check_break ())
		break;
#else
	    if (check_break (0))
		break;
#endif
	    // get input filename from list

	    if (!filelist) {
		*filespec_name (infilename) = '\0';
		strcat (infilename, matches [file_index]);
	    }
	    else
		infilename = matches [file_index];

	    // generate output filename

	    if (outpath) {
		strcat (outfilename, filespec_name (matches [file_index]));

		if (filespec_ext (outfilename))
		    *filespec_ext (outfilename) = '\0';
	    }
	    else if (!outfilename) {
		outfilename = malloc (strlen (infilename) + 10);
		strcpy (outfilename, infilename);

		if (filespec_ext (outfilename))
		    *filespec_ext (outfilename) = '\0';
	    }

	    if (num_files > 1)
		fprintf (stderr, "\n%s:\n", infilename);

	    result = unpack_file (infilename, verify_only ? NULL : outfilename);

	    if (result == HARD_ERROR)
		break;

	    // clean up in preparation for potentially another file

	    if (outpath)
		*filespec_name (outfilename) = '\0';
	    else if (*outfilename != '-') {
		free (outfilename);
		outfilename = NULL;
	    }

	    free (matches [file_index]);
	}

	free (matches);
    }
    else
	error_line (filespec_wild (infilename) ? "nothing to do!" :
	    "file %s not found!", infilename);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// This function unpacks a single file "infilename" and stores the result   //
// at "outfilename". Note that "outfilename" can be NULL for verifying only //
// and can point to a single "-" to specify stdout as the destination.      //
//////////////////////////////////////////////////////////////////////////////

static int unpack_file (char *infilename, char *outfilename)
{
    ulong outfile_length, infile_length, in2file_length;
    long total_samples, total_header_bytes;
    HANDLE infile, in2file, outfile;
    RiffChunkHeader RiffChunkHeader;
    WavpackHeader WavpackHeader;
    ChunkHeader ChunkHeader;
    WaveHeader WaveHeader;
    char *in2filename;
    double dtime;
    ulong bcount;
    int result;

#ifdef __BORLANDC__
    struct time time1, time2;
#else
    struct _timeb time1, time2;
#endif

#ifdef __WIN32__
    FILETIME time;
#else
    struct ftime time;
#endif

    // open the source file for reading and store the size and time

#ifdef __WIN32__
    if ((infile = CreateFile (infilename, GENERIC_READ, FILE_SHARE_READ, NULL,
	OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
	    error_line ("can't open file %s!", infilename);
	    return SOFT_ERROR;
    }

    infile_length = GetFileSize (infile, NULL);
    GetFileTime (infile, NULL, NULL, &time);
#else
    if ((infile = open (infilename, O_RDONLY | O_BINARY)) == -1) {
	error_line ("can't open file %s!", infilename);
	return SOFT_ERROR;
    }

    infile_length = filelength (infile);
    getftime (infile, &time);
#endif

    // If the first chunk is a wave RIFF header, then read the various chunks
    // until we get to the "data" chunk (and WavPack header should follow). If
    // the first chunk is not a RIFF, then we assume a "raw" WavPack file and
    // the WavPack header must be first. In either event, "total_header_bytes"
    // will end up set to the number of bytes that we must copy from the source
    // to the destination for headers.

    if (DoReadFile (infile, &RiffChunkHeader, sizeof (RiffChunkHeader), &bcount, NULL) &&
	bcount == sizeof (RiffChunkHeader) && !strncmp (RiffChunkHeader.ckID, "RIFF", 4) &&
	!strncmp (RiffChunkHeader.formType, "WAVE", 4)) {

	    while (1) {

		if (!DoReadFile (infile, &ChunkHeader, sizeof (ChunkHeader), &bcount, NULL) ||
		    bcount != sizeof (ChunkHeader)) {
			error_line ("%s is not a valid WavPack file!", infilename);
			DoCloseHandle (infile);
			return SOFT_ERROR;
		}

		if (!strncmp (ChunkHeader.ckID, "fmt ", 4)) {

		    if (ChunkHeader.ckSize < sizeof (WaveHeader) ||
			!DoReadFile (infile, &WaveHeader, sizeof (WaveHeader), &bcount, NULL) ||
			bcount != sizeof (WaveHeader)) {
			    error_line ("%s is not a valid WavPack file!", infilename);
			    DoCloseHandle (infile);
			    return SOFT_ERROR;
		    }

		    if (ChunkHeader.ckSize > sizeof (WaveHeader))
#ifdef __WIN32__
			SetFilePointer (infile, (ChunkHeader.ckSize + 1 - sizeof (WaveHeader)) & ~1L, NULL, FILE_CURRENT);
#else
			lseek (infile, (ChunkHeader.ckSize + 1 - sizeof (WaveHeader)) & ~1L, SEEK_CUR);
#endif
		}
		else if (!strncmp (ChunkHeader.ckID, "data", 4)) {
#ifdef VER2
		    total_samples = ChunkHeader.ckSize / WaveHeader.NumChannels /
			((WaveHeader.BitsPerSample > 16) ? 3 : 2);
#endif
		    break;
		}
		else
#ifdef __WIN32__
		    SetFilePointer (infile, (ChunkHeader.ckSize + 1) & ~1L, NULL, FILE_CURRENT);
#else
		    lseek (infile, (ChunkHeader.ckSize + 1) & ~1L, SEEK_CUR);
#endif
	    }

#ifdef __WIN32__
	    total_header_bytes = SetFilePointer (infile, 0, NULL, FILE_CURRENT);
#else
	    total_header_bytes = tell (infile);
#endif
    }
    else {
#ifdef __WIN32__
	SetFilePointer (infile, 0, NULL, FILE_BEGIN);
#else
	lseek (infile, 0, SEEK_SET);
#endif
	total_header_bytes = 0;
    }

    // read WavPack header and make sure this is a version we know about

    if (!DoReadFile (infile, &WavpackHeader, sizeof (WavpackHeader), &bcount, NULL) ||
	bcount != sizeof (WavpackHeader) || strncmp (WavpackHeader.ckID, "wvpk", 4)) {
	    error_line ("%s is not a valid WavPack file!", infilename);
	    DoCloseHandle (infile);
	    return SOFT_ERROR;
    }

    if (WavpackHeader.version < 1 || WavpackHeader.version > 3) {
	error_line ("not compatible with this version of WavPack file!");
	DoCloseHandle (infile);
	return SOFT_ERROR;
    }

    // Because I ran out of flag bits in the WavPack header, an amazingly ugly
    // kludge was forced upon me! This code takes care of preparing the flags
    // field for internal use and checking for unknown formats we can't decode

    if (WavpackHeader.version == 3) {

	if (WavpackHeader.flags & EXTREME_DECORR) {

	    if ((WavpackHeader.flags & NOT_STORED_FLAGS) ||
		((WavpackHeader.bits) &&
		(((WavpackHeader.flags & NEW_HIGH_FLAG) &&
		(WavpackHeader.flags & (FAST_FLAG | HIGH_FLAG))) ||
		(WavpackHeader.flags & CROSS_DECORR)))) {
		    error_line ("not compatible with this version of WavPack file!");
		    DoCloseHandle (infile);
		    return SOFT_ERROR;
	    }

	    if (WavpackHeader.flags & CANCEL_EXTREME)
		WavpackHeader.flags &= ~(EXTREME_DECORR | CANCEL_EXTREME);
	}
	else
	    WavpackHeader.flags &= ~CROSS_DECORR;
    }

    // check to see if we should look for a "correction" file, and if so try
    // to open it for reading, then set WVC_FLAG accordingly

    if (WavpackHeader.version == 3 && WavpackHeader.bits && (WavpackHeader.flags & NEW_HIGH_FLAG)) {
	in2filename = malloc (strlen (infilename) + 10);
	strcpy (in2filename, infilename);

	if (filespec_ext (in2filename))
	    *filespec_ext (in2filename) = '\0';

	strcat (in2filename, anylower (filespec_name (in2filename)) ?
	    ".wvc" : ".WVC");

#ifdef __WIN32__
	if ((in2file = CreateFile (in2filename, GENERIC_READ, FILE_SHARE_READ, NULL,
	    OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE)
#else
	if ((in2file = open (in2filename, O_RDONLY | O_BINARY)) == -1)
#endif
		free (in2filename);
    }
    else
	in2file = INVALID_HANDLE_VALUE;

    if (in2file == INVALID_HANDLE_VALUE) {
	WavpackHeader.flags &= ~WVC_FLAG;
	in2filename = NULL;
    }
    else
	WavpackHeader.flags |= WVC_FLAG;

    // if the output file is specified (not verify mode) then we have to get
    // the destination handle ready to go

    if (outfilename) {
	if (*outfilename != '-') {

	    // the destination file's extension is stored in the WavPack header
	    // sometimes, so if we don't already have an extension use that one

	    if (add_extension) {
		if (WavpackHeader.version >= 3) {
		    if (WavpackHeader.extension [0]) {
			strcat (outfilename, ".");
			strcat (outfilename, WavpackHeader.extension);
		    }
		}
		else
		    strcat (outfilename, anylower (filespec_name (outfilename)) ?
			".wav" : ".WAV");
	    }

	    // check the output file for overwrite warning required

#ifdef __WIN32__
	    if (!overwrite_all && (outfile = CreateFile (outfilename, GENERIC_READ, 0, NULL,
		OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {

		    DoCloseHandle (outfile);
		    fprintf (stderr, "overwrite %s (yes/no/all)? ", FN_FIT (outfilename));
		    SetConsoleTitle ("overwrite?");

		    switch (yna ()) {

			case 'n':
			    if (in2filename) {
				DoCloseHandle (in2file);		
				free (in2filename);
			    }

			    DoCloseHandle (infile);
			    return SOFT_ERROR;

			case 'a':
			    overwrite_all = 1;
		    }
	    }

	    // open output file for writing

	    if ((outfile = CreateFile (outfilename, GENERIC_READ | GENERIC_WRITE, 0, NULL,
		CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL)) == INVALID_HANDLE_VALUE) {
		    error_line ("can't create file %s!", outfilename);

		    if (in2filename) {
			DoCloseHandle (in2file);		
			free (in2filename);
		    }

		    DoCloseHandle (infile);
		    return SOFT_ERROR;
	    }

	    fprintf (stderr, "restoring %s,", FN_FIT (outfilename));
	}
	else {	// come here to open stdout as destination

	    outfile = GetStdHandle (STD_OUTPUT_HANDLE);
	    setmode (fileno (stdout), O_BINARY);

	    fprintf (stderr, "unpacking %s%s to stdout,", FN_FIT (infilename), (in2file == INVALID_HANDLE_VALUE) ? "" :
		(anylower (filespec_name (infilename)) ? " (+.wvc)" : " (+.WVC)"));
	}

	SetFilePointer (infile, 0, NULL, FILE_BEGIN);
#else
	    if (!overwrite_all && (outfile = open (outfilename, O_RDONLY | O_BINARY)) != -1) {

		DoCloseHandle (outfile);
		fprintf (stderr, "overwrite %s (yes/no/all)? ", FN_FIT (outfilename));

		switch (yna ()) {

		    case 'n':
			if (in2filename) {
			    DoCloseHandle (in2file);		
			    free (in2filename);
			}

			DoCloseHandle (infile);
			return SOFT_ERROR;

		    case 'a':
			overwrite_all = 1;
		}
	    }

	    if ((outfile = open (outfilename, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, S_IREAD | S_IWRITE)) == -1) {
		error_line ("can't create file %s!", outfilename);

		if (in2filename) {
		    DoCloseHandle (in2file);		
		    free (in2filename);
		}

		DoCloseHandle (infile);
		return SOFT_ERROR;
	    }

	    fprintf (stderr, "restoring %s,", FN_FIT (outfilename));
	}
	else {
	    outfile = fileno (stdout);
	    setmode (fileno (stdout), O_BINARY);

	    fprintf (stderr, "unpacking %s%s to stdout,", FN_FIT (infilename), (in2file == INVALID_HANDLE_VALUE) ? "" :
		(anylower (filespec_name (infilename)) ? " (+.wvc)" : " (+.WVC)"));
	}

	lseek (infile, 0, SEEK_SET);
#endif

	// If "total_header_bytes" is non-zero, then we are restoring a RIFF
	// wave file and we must copy that many header bytes from source to
	// destination.

	if (total_header_bytes) {
	    char *buff = malloc (total_header_bytes);

	    if (!DoReadFile (infile, buff, total_header_bytes, &bcount, NULL) ||
		bcount != total_header_bytes ||
		!DoWriteFile (outfile, buff, total_header_bytes, &bcount, NULL) ||
		bcount != total_header_bytes) {

		    error_line ("can't write .WAV data, disk probably full!");

		    if (in2filename) {
			DoCloseHandle (in2file);
			free (in2filename);
		    }

		    DoCloseHandle (infile);
		    DoCloseHandle (outfile);
		    DoDeleteFile (outfilename);
		    free (buff);
		    return SOFT_ERROR;
	    }

	    free (buff);
	}
    }
    else {	// in verify only mode we don't worry about headers

#ifdef __WIN32__
	SetFilePointer (infile, total_header_bytes, NULL, FILE_BEGIN);
#else
	lseek (infile, total_header_bytes, SEEK_SET);
#endif
	outfile = INVALID_HANDLE_VALUE;

	fprintf (stderr, "verifying %s%s,", FN_FIT (infilename), (in2file == INVALID_HANDLE_VALUE) ? "" :
	    (anylower (filespec_name (infilename)) ? " (+.wvc)" : " (+.WVC)"));
    }

#ifdef __BORLANDC__
    gettime (&time1);
#else
    _ftime (&time1);
#endif

    // switch on WavPack version to handle special requirements of versions
    // before 3.0 that had smaller headers

    switch (WavpackHeader.version) {

#ifdef VER2
	case 1:
	    WavpackHeader.bits = 0;
#ifdef __WIN32__
	    SetFilePointer (infile, -2, NULL, FILE_CURRENT);
#else
	    lseek (infile, -2, SEEK_CUR);
#endif

	case 2:
	    WavpackHeader.total_samples = total_samples;
	    WavpackHeader.flags = WaveHeader.NumChannels == 1 ? MONO_FLAG : 0;
	    WavpackHeader.shift = 16 - WaveHeader.BitsPerSample;
#ifdef __WIN32__
	    SetFilePointer (infile, 12, NULL, FILE_CURRENT);
#else
	    lseek (infile, 12, SEEK_CUR);
#endif
	    result = unpack_audio (&WavpackHeader, infile, in2file, outfile);
	    break;
#endif

	case 3:
#ifdef __WIN32__
	    SetFilePointer (infile, sizeof (WavpackHeader), NULL, FILE_CURRENT);
#else
	    lseek (infile, sizeof (WavpackHeader), SEEK_CUR);
#endif
	    result = unpack_audio (&WavpackHeader, infile, in2file, outfile);

	    if (!outfilename)	// if verifying only, we're done enough
		break;

	    // If this was not a "raw" file, then there may be information
	    // past the audio data that we need to copy onto the end of the
	    // destination file. This is indicated by having 128 '1's past the
	    // end if audio data, so if we find that then copy whatever's past
	    // that. If we find something else, then don't copy anything
	    // because it is probably some info tag that we want to ignore.

	    if (result == NO_ERROR && !(WavpackHeader.flags & RAW_FLAG)) {
		char c, ones [16];

		if (DoReadFile (infile, &ones, sizeof (ones), &bcount, NULL) && bcount == sizeof (ones)) {

		    for (c = 0; c < sizeof (ones) && !~ones [c]; ++c);

		    if (c == sizeof (ones))
			while (DoReadFile (infile, &c, 1, &bcount, NULL) && bcount == 1)
			    if (!DoWriteFile (outfile, &c, 1, &bcount, NULL) || bcount != 1) {
				error_line ("can't write .WAV data, disk probably full!");
				DoTruncateFile (outfile);
				result = SOFT_ERROR;
			    }
		}
	    }

	    // If this was a "raw" file, then there may be up to 3 extra bytes
	    // in the WavPack header that need to be appended to the end of the
	    // file.

	    if (result == NO_ERROR && (WavpackHeader.flags & RAW_FLAG) &&
		WavpackHeader.extra_bc && (!DoWriteFile (outfile, WavpackHeader.extras,
		WavpackHeader.extra_bc, &bcount, NULL) || bcount != WavpackHeader.extra_bc)) {
		    error_line ("can't write .WAV data, disk probably full!");
		    DoTruncateFile (outfile);
		    result = SOFT_ERROR;
	    }

	    break;
    }

    // if we are not just in verify only mode, grab the size of the output
    // file, copy the time to it (if selected) and close the file

    if (outfilename) {
#ifdef __WIN32__
	outfile_length = GetFileSize (outfile, NULL);
#else
	outfile_length = filelength (outfile);
#endif

	if (copy_time)
#ifdef __WIN32__
	    SetFileTime (outfile, &time, &time, &time);
#else
	    setftime (outfile, &time);
#endif

	if (!DoCloseHandle (outfile)) {
	    error_line ("can't close file!");
	    result = SOFT_ERROR;
	}
    }

    // close the input file(s)

    DoCloseHandle (infile);

    if (in2filename) {
#ifdef __WIN32__
	in2file_length = GetFileSize (in2file, NULL);
#else
	in2file_length = filelength (in2file);
#endif
	DoCloseHandle (in2file);
    }
    else
	in2file_length = 0;

    // Compute and display the time consumed along with some other details of
    // the unpacking operation (assuming there was no error). Note that if we
    // didn't actually create a file (i.e. verify only or to stdout) then we
    // can't compute the compression ratio because we don't have an output
    // file size. Call me lazy. :)

#ifdef __BORLANDC__
    gettime (&time2);
    dtime = time2.ti_sec * 100.0 + time2.ti_hund + time2.ti_min * 6000.0 + time2.ti_hour * 360000.00;
    dtime -= time1.ti_sec * 100.0 + time1.ti_hund + time1.ti_min * 6000.0 + time1.ti_hour * 360000.00;

    if ((dtime /= 100.0) < 0.0)
	dtime += 86400.0;
#else
    _ftime (&time2);
    dtime = time2.time + time2.millitm / 1000.0;
    dtime -= time1.time + time1.millitm / 1000.0;
#endif

    if (outfilename && *outfilename != '-') {
	if (result == NO_ERROR) {
	    if (!in2file_length)
		error_line ("restored %s in %.2f secs (%s comp: %.2f%%)",
		    FN_FIT (outfilename), dtime,
		    (WavpackHeader.bits && WavpackHeader.crc != WavpackHeader.crc2) ? "lossy" : "lossless",
		    100 - (infile_length * 100.0 / outfile_length));
	    else
		error_line ("restored %s in %.2f secs (lossless comp: %.2f%%)",
		    FN_FIT (outfilename), dtime, 100 - ((infile_length + in2file_length) * 100.0 / outfile_length));
	}

	if (!outfile_length)
	    DoDeleteFile (outfilename);
    }
    else if (result == NO_ERROR) {
	error_line ("%s %s%s in %.2f seconds",
	    outfilename ? "unpacked" : "verified",
	    FN_FIT (infilename), (!in2file_length) ? "" :
	    (anylower (filespec_name (infilename)) ? " (+.wvc)" : " (+.WVC)"), dtime);
    }

    // delete the source files if requested by the user

    if (result == NO_ERROR && delete_source) {
	error_line ("%s source file %s", DoDeleteFile (infilename) ?
	    "deleted" : "can't delete", infilename);

	if (in2filename)
	    error_line ("%s source file %s", DoDeleteFile (in2filename) ?
		"deleted" : "can't delete", in2filename);
    }

    if (in2filename)
	free (in2filename);

    return result;
}

//////////////////////////////////////////////////////////////////////////////
// This function performs the actual audio decompression on the specified   //
// file handles, which are assumed to be opened and positioned correctly.   //
// This function does not read or write any file headers, nor does it close //
// the files when done. On an error that should cause the deletion of the   //
// output file, this function will simply leave the file truncated to zero  //
// length and the caller should actually delete the file. This function has //
// the code that handles overlapped writing of the audio data for Win95/98  //
// systems and also displays the progress messages on stderr and (on Win32) //
// the window's title bar.                                                  //
//////////////////////////////////////////////////////////////////////////////

static int unpack_audio (WavpackHeader *wphdr, HANDLE infile, HANDLE in2file, HANDLE outfile)
{
    ulong bytes_written, bytes_to_write, samples_remaining;
    int valid_outfile = (outfile != INVALID_HANDLE_VALUE);
    uint samples_to_unpack, samples_unpacked;
    int flags = wphdr->flags;
    double progress = 0.0;
    WavpackContext wpc;
    uchar *buffer;

#ifdef __MT__
    uint pending_write_bytes = 0;
    uchar *next_buffer = NULL;
#endif

    if (!wphdr->total_samples)
	return NO_ERROR;

    CLEAR (wpc);
    wpc.wphdr = wphdr;
    wpc.inbits.bufsiz = wpc.in2bits.bufsiz = INBUF_SIZE;
    buffer = malloc (OUTBUF_SIZE);
    unpack_init (&wpc);

#ifdef __MT__
    if (valid_outfile && get_platform () == 1) {
	next_buffer = malloc (OUTBUF_SIZE);

	if (!buffer || !next_buffer || bs_open_read (&wpc.inbits, infile, TRUE) ||
	    ((flags & WVC_FLAG) && bs_open_read (&wpc.in2bits, in2file, FALSE))) {
		error_line ("internal error!");
		DoTruncateFile (outfile);

		if (buffer)
		    free (buffer);

		if (next_buffer)
		    free (buffer);

		return HARD_ERROR;
	}
    }
    else if (!buffer || bs_open_read (&wpc.inbits, infile, FALSE) ||
	((flags & WVC_FLAG) && bs_open_read (&wpc.in2bits, in2file, FALSE))) {
	    error_line ("internal error!");

	    if (valid_outfile)
		DoTruncateFile (outfile);

	    if (buffer)
		free (buffer);

	    return HARD_ERROR;
    }
#else
    if (!buffer || bs_open_read (&wpc.inbits, infile) || ((flags & WVC_FLAG) && bs_open_read (&wpc.in2bits, in2file))) {
	error_line ("internal error!");

	if (valid_outfile)
	    DoTruncateFile (outfile);

	if (buffer)
	    free (buffer);

	return HARD_ERROR;
    }
#endif

    fprintf (stderr, "   0%% done...");
#ifdef __WIN32__
    display_progress (progress);
#endif
    samples_remaining = wphdr->total_samples;

    do {
	samples_to_unpack = OUTBUF_SIZE / BYTES_PER_SAMPLE;

	if (samples_to_unpack > samples_remaining)
	    samples_to_unpack = samples_remaining;

	samples_unpacked = unpack_samples (&wpc, buffer, samples_to_unpack);
	bytes_to_write = samples_unpacked * BYTES_PER_SAMPLE;
	samples_remaining -= samples_unpacked;

#ifdef __MT__
	if (next_buffer) {
	    if (pending_write_bytes && pending_write_bytes != finish_write ()) {
		error_line ("can't write .WAV data, disk probably full!");

		if (flags & WVC_FLAG)
		    bs_close_read (&wpc.in2bits);

		bs_close_read (&wpc.inbits);
		DoTruncateFile (outfile);
		free (buffer);
		free (next_buffer);
		return HARD_ERROR;
	    }

	    if ((pending_write_bytes = bytes_to_write) != 0) {
		uchar *tmp;

		start_write (outfile, buffer, pending_write_bytes);
		tmp = buffer;
		buffer = next_buffer;
		next_buffer = tmp;
	    }
	}
	else if (valid_outfile &&
	    (!DoWriteFile (outfile, buffer, bytes_to_write, &bytes_written, NULL) ||
	    bytes_written != bytes_to_write)) {
		error_line ("can't write .WAV data, disk probably full!");

		if (flags & WVC_FLAG)
		    bs_close_read (&wpc.in2bits);

		bs_close_read (&wpc.inbits);
		DoTruncateFile (outfile);
		free (buffer);
		return HARD_ERROR;
	}
#else
	if (valid_outfile &&
	    (!DoWriteFile (outfile, buffer, bytes_to_write, &bytes_written, NULL) ||
	    bytes_written != bytes_to_write)) {
		error_line ("can't write .WAV data, disk probably full!");

		if (flags & WVC_FLAG)
		    bs_close_read (&wpc.in2bits);

		bs_close_read (&wpc.inbits);
		DoTruncateFile (outfile);
		free (buffer);
		return HARD_ERROR;
	}
#endif

#ifdef __WIN32__
	if (check_break ()) {
	    fprintf (stderr, "^C\n");
#else
	if (check_break (infile != fileno (stdin))) {
#endif
	    if (flags & WVC_FLAG)
		bs_close_read (&wpc.in2bits);

	    bs_close_read (&wpc.inbits);

	    if (valid_outfile) {
#ifdef __MT__
		if (pending_write_bytes)
		    finish_write ();
#endif
		DoTruncateFile (outfile);
	    }

	    free (buffer);
#ifdef __MT__
	    if (next_buffer)
		free (next_buffer);
#endif
	    return SOFT_ERROR;
	}

	if (progress != floor (((double)(wphdr->total_samples - samples_remaining) / wphdr->total_samples) * 100.0 + 0.5)) {
	    progress = (double)(wphdr->total_samples - samples_remaining) / wphdr->total_samples;
#ifdef __WIN32__
	    display_progress (progress);
#endif
	    progress = floor (progress * 100.0 + 0.5);
	    fprintf (stderr, "\b\b\b\b\b\b\b\b\b\b\b\b%3d%% done...", (int) progress);
	}

    } while (samples_remaining && samples_unpacked == samples_to_unpack);

#ifdef __MT__
    if (pending_write_bytes && pending_write_bytes != finish_write ()) {
	error_line ("can't write .WAV data, disk probably full!");
	DoTruncateFile (outfile);

	if (flags & WVC_FLAG)
	    bs_close_read (&wpc.in2bits);

	bs_close_read (&wpc.inbits);
	free (buffer);
	free (next_buffer);
	return HARD_ERROR;
    }
#endif

    if (flags & WVC_FLAG)
	bs_close_read (&wpc.in2bits);

    bs_close_read (&wpc.inbits);
    free (buffer);
#ifdef __MT__
    if (next_buffer)
	free (next_buffer);
#endif

    if (samples_remaining || (wphdr->version == 3 && unpack_crc (&wpc) != ((flags & WVC_FLAG) ? wphdr->crc2 : wphdr->crc))) {
	if (valid_outfile)
	    error_line ("CRC error detected, file not correctly uncompressed!!");
	else
	    error_line ("CRC error detected, file does not verify!!");

	return SOFT_ERROR;
    }

    return NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////
// This function displays the progress status on the title bar of the DOS   //
// window that WavPack is running in. The "file_progress" argument is for   //
// the current file only and ranges from 0 - 1; this function takes into    //
// account the total number of files to generate a batch progress number.   //
//////////////////////////////////////////////////////////////////////////////

#ifdef __WIN32__
void display_progress (double file_progress)
{
    char title [40];

    file_progress = (file_index + file_progress) / num_files;
    sprintf (title, "%d%% (WvUnpack)", (int) ((file_progress * 100.0) + 0.5));
    SetConsoleTitle (title);
}
#endif
