// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// ParticleBC template definitions for KillBC specialization.
//-----------------------------------------------------------------------------

// include files

#include "Domain/Interval.h"
#include "DynamicArray/DynamicArray.h"
#include "Domain/IndirectionList.h"
#include "Evaluator/PatchFunction.h"
#include "Utilities/PAssert.h"
#include "Tiny/Vector.h"


// KillBC functors for general elements and specialized for Vectors.
// This is the general case in which we assume the Object to be a
// Particles object.

//-----------------------------------------------------------------------------
// KillBCFunc functor
// Apply the kill BC in a patchwise fashion.
// The object of the KillBCFunc functor *must* be a Particles object
// or a DynamicArray (i.e., something that can perform destroys).
//-----------------------------------------------------------------------------

template <class T, class Object>
struct KillBCFunc
{
  // Typedefs
  typedef Object Object_t;

  // Constructor
  KillBCFunc(const T& min, const T& max, Object_t& object)
    : min_m(min), max_m(max), object_m(object)
  {
  }

  // apply method implements BC
  template <class ArrayPatch>
  void apply(const ArrayPatch& sub, int node)
  {
    // store kill list as a DynamicBrick
    Array<1,int,Brick> killlist(sub.domain().size());
    
    // loop over elements in this patch and apply BC
    int from = sub.domain()[0].first();
    int to = sub.domain()[0].last();
    int killed = 0;
    for (int i = from; i <= to; ++i)
      {
        // check both boundaries
        if (sub.read(i) < min_m || sub.read(i) > max_m)
          {
            // add this particle's local index to the kill list
            killlist(killed++) = i;
          }
      }

    // convert the kill list into an IndirectionList and cache it
    IndirectionList<int> klist(killlist(Interval<1>(killed)));
    object_m.deferredDestroy(klist, node);
  }

  // bounds for BC
  T min_m, max_m;

  // store reference to object so we can invoke its destroy method
  Object_t& object_m;
};

//-----------------------------------------------------------------------------
// KillBCFunc functor specialized for Vectors
// Apply the kill BC in a patchwise fashion.
// Each component is treated separately.
//-----------------------------------------------------------------------------

template <int Dim, class T, class E, class Object>
struct KillBCFunc< Vector<Dim,T,E>, Object >
{
  // Typedefs
  typedef Vector<Dim,T,E> Type_t;
  typedef Object Object_t;

  // Constructor
  KillBCFunc(const Type_t& min, const Type_t& max, Object_t& object)
    : min_m(min), max_m(max), object_m(object)
  {
  }

  // apply method implements BC
  template <class ArrayPatch>
  void apply(const ArrayPatch& sub, int node)
  {
    // store kill list as a DynamicBrick
    Array<1,int,Brick> killlist(sub.domain().size());
    
    // loop over elements in this patch and apply BC
    int from = sub.domain()[0].first();
    int to = sub.domain()[0].last();
    int killed = 0;
    for (int i = from; i <= to; ++i)
      {
        bool wasKilled = false;
        int d = 0;
        while (d < Dim && !wasKilled)
          {
            // check both boundaries
            if (sub.read(i)(d) < min_m(d) || sub.read(i)(d) > max_m(d))
              {
                // add this particle's local index to the kill list
                killlist(killed++) = i;
                wasKilled = true;
              }
            ++d;
          }
      }

    // convert the kill list into an IndirectionList and cache it
    IndirectionList<int> klist(killlist(Interval<1>(killed)));
    object_m.deferredDestroy(klist, node);
  }

  // bounds for BC
  Type_t min_m, max_m;

  // store reference to object so we can invoke its destroy method
  Object_t& object_m;
};


// KillBC functors for general elements and specialized for Vectors.
// This is the special case in which the Object is a DynamicArray.

//-----------------------------------------------------------------------------
// KillBCFunc functor
// Apply the kill BC in a patchwise fashion.
// The object of the KillBCFunc functor *must* be a Particles object
// or a DynamicArray (i.e., something that can perform destroys).
//-----------------------------------------------------------------------------

template <class T1, class T2, class E>
struct KillBCFunc< T1, DynamicArray<T2,E> >
{
  // Typedefs
  typedef DynamicArray<T2,E> Object_t;

  // Constructor
  KillBCFunc(const T1& min, const T1& max, Object_t& object)
    : min_m(min), max_m(max), object_m(object)
  {
  }

  // apply method implements BC
  template <class ArrayPatch>
  void apply(const ArrayPatch& sub, int node)
  {
    // store kill list as a DynamicBrick
    Array<1,int,Brick> killlist(sub.domain().size());
    
    // loop over elements in this patch and apply BC
    int from = sub.domain()[0].first();
    int to = sub.domain()[0].last();
    int killed = 0;
    for (int i = from; i <= to; ++i)
      {
        // check both boundaries
        if (sub.read(i) < min_m || sub.read(i) > max_m)
          {
            // add this particle's local index to the kill list
            killlist(killed++) = i;
          }
      }

    // convert the kill list into an IndirectionList and cache it
    IndirectionList<int> klist(killlist(Interval<1>(killed)));
    object_m.destroy(klist, node);
  }

  // bounds for BC
  T1 min_m, max_m;

  // store reference to object so we can invoke its destroy method
  Object_t& object_m;
};

//-----------------------------------------------------------------------------
// KillBCFunc functor specialized for Vectors
// Apply the kill BC in a patchwise fashion.
// Each component is treated separately.
//-----------------------------------------------------------------------------

template <int Dim, class T1, class E1, class T2, class E2>
struct KillBCFunc< Vector<Dim,T1,E1>, DynamicArray<T2,E2> >
{
  // Typedefs
  typedef Vector<Dim,T1,E1> Type_t;
  typedef DynamicArray<T2,E2> Object_t;

  // Constructor
  KillBCFunc(const Type_t& min, const Type_t& max, Object_t& object)
    : min_m(min), max_m(max), object_m(object)
  {
  }

  // apply method implements BC
  template <class ArrayPatch>
  void apply(const ArrayPatch& sub, int node)
  {
    // store kill list as a DynamicBrick
    Array<1,int,Brick> killlist(sub.domain().size());
    
    // loop over elements in this patch and apply BC
    int from = sub.domain()[0].first();
    int to = sub.domain()[0].last();
    int killed = 0;
    for (int i = from; i <= to; ++i)
      {
        bool wasKilled = false;
        int d = 0;
        while (d < Dim && !wasKilled)
          {
            // check both boundaries
            if (sub(i)(d) < min_m(d) || sub(i)(d) > max_m(d))
              {
                // add this particle's local index to the kill list
                killlist(killed++) = i;
                wasKilled = true;
              }
            ++d;
          }
      }
    // convert the kill list into an IndirectionList and cache it
    IndirectionList<int> klist(killlist(Interval<1>(killed)));
    object_m.destroy(klist, node);
  }

  // bounds for BC
  Type_t min_m, max_m;

  // store reference to object so we can invoke its destroy method
  Object_t& object_m;
};


//-----------------------------------------------------------------------------
// void applyBoundaryCondition()
// Apply the "kill" boundary condition to the subject and object.
//-----------------------------------------------------------------------------

template <class Subject, class Object, class T>
void
ParticleBC< Subject, Object, KillBC<T> >::
applyBoundaryCondition(int pid)
{
  // get limits of KillBC range
  T min = bc_m.min(), max = bc_m.max();

  // loop over local patches and apply the BC using functor...
  KillBCFunc<T,Object> bcfun(min,max,object_m);
  if (pid < 0)
    {
      PatchFunction< KillBCFunc<T,Object>,
                     PatchParticle1<false> > patchfun(bcfun);
      patchfun.block(subject_m);
    }
  else
    {
      bcfun.apply(subject_m.patchLocal(pid)(), pid);
    }
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: KillBC.cpp,v $   $Author: sa_smith $
// $Revision: 1.20 $   $Date: 2000/07/28 16:11:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
