// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// FftwCC
// FftwRC
//-----------------------------------------------------------------------------

#ifndef POOMA_TRANSFORM_WRAP_FFTW_H
#define POOMA_TRANSFORM_WRAP_FFTW_H

#if POOMA_FFTW

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// These functors wrap calls to the FFTW functions that perform multiple 1D
// transforms.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include "fftw.h"
#include "rfftw.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
// These functors wrap calls to the FFTW functions that perform multiple 1D
// transforms.
//
// For now, the interface provided by these functors is of the form
// initialize(), apply(), finalize().  To support an interface where you just
// construct the object, use it and later delete it, then these objects
// would require a ref-counted core for efficiency.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FftwCC
//-----------------------------------------------------------------------------

class FftwCC
{
public:
  typedef complex<double> Element_t;
  enum { inPlace = true };
  enum { sliceDimensions = 2 };

  FftwCC(int isign)
    : isign_m(isign), initialized_m(false)
  {
    CTAssert(sizeof(Element_t) == sizeof(fftw_complex));
  }

  FftwCC(const FftwCC &model)
    : isign_m(model.isign_m), initialized_m(false)
  { }

  void initialize(const Loc<2> &lengths, const Loc<2> &strides)
  {
    if (!initialized_m)
    {
      length_m  = lengths[0].first();
      lot_m     = lengths[1].first();
      istride_m = strides[0].first();
      idist_m   = strides[1].first();

      fftw_direction dir = (isign_m == 1) ? FFTW_FORWARD : FFTW_BACKWARD;
      plan_m = fftw_create_plan(length_m, dir,
				FFTW_MEASURE | FFTW_USE_WISDOM |
				FFTW_IN_PLACE);
      initialized_m = true;
    }
  }

  void apply(Element_t *a)
  {
    PAssert(initialized_m);

    fftw_complex *aa = reinterpret_cast<fftw_complex *>(a);
    fftw(plan_m, lot_m, aa, istride_m, idist_m, NULL, 0, 0);
  }

  void finalize()
  {
    if (initialized_m)
    {
      fftw_destroy_plan(plan_m);
      initialized_m = false;
    }
  }

  ~FftwCC()
  {
    finalize();
  }

private:

  FftwCC();
  FftwCC &operator=(const FftwCC &);

  // isign - forward or backward transform?

  int isign_m;

  // length of 1D fft

  int length_m;

  // stride in fft direction

  int istride_m;

  // number of ffts

  int lot_m;

  // stride between 1D strips

  int idist_m;

  bool initialized_m;
  fftw_plan plan_m;
};

//-----------------------------------------------------------------------------
// FftwRC
//-----------------------------------------------------------------------------

class FftwRC
{
public:
  typedef double Element_t;
  enum { inPlace = true };
  enum { sliceDimensions = 2 };

  FftwRC(int isign)
    : isign_m(isign), initialized_m(false)
  {
    CTAssert(sizeof(Element_t) == sizeof(fftw_real));
  }

  FftwRC(const FftwRC &model)
    : isign_m(model.isign_m), initialized_m(false)
  { }

  void initialize(const Loc<2> &lengths, const Loc<2> &strides)
  {
    if (!initialized_m)
    {
      length_m  = lengths[0].first();
      lot_m     = lengths[1].first();
      istride_m = strides[0].first();
      idist_m   = strides[1].first();

      fftw_direction dir = (isign_m == 1) ? FFTW_FORWARD : FFTW_BACKWARD;
      plan_m = rfftw_create_plan(length_m, dir,
				 FFTW_MEASURE | FFTW_USE_WISDOM |
				 FFTW_IN_PLACE);
      initialized_m = true;
    }
  }

  void apply(Element_t *a) const
  {
    PAssert(initialized_m);

    fftw_real *aa = reinterpret_cast<fftw_real *>(a);
    rfftw(plan_m, lot_m, aa, istride_m, idist_m, NULL, 0, 0);
  }

  void finalize()
  {
    if (initialized_m)
    {
      rfftw_destroy_plan(plan_m);
    }
  }

  ~FftwRC()
  {
    finalize();
  }

private:

  FftwRC();
  FftwRC &operator=(const FftwRC &);

  // isign - forward or backward transform?

  int isign_m;

  // length of 1D fft

  int length_m;

  // stride in fft direction

  int istride_m;

  // number of ffts

  int lot_m;

  // stride between 1D strips

  int idist_m;

  // initialized_m - is the plan initialized?

  bool initialized_m;

  // the fft plan:

  fftw_plan plan_m;
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_FFTW

#endif     // POOMA_TRANSFORM_WRAP_FFTW_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: WrapFFTW.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/03/07 13:18:20 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
