// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_WRAPPEDINT_H
#define POOMA_UTILITIES_WRAPPEDINT_H

//-----------------------------------------------------------------------------
// Class: WrappedInt
//-----------------------------------------------------------------------------

#include "Utilities/PurifyConstructors.h"

//-----------------------------------------------------------------------------
//
// Helper class: WrappedInt<int> 
//
// A tag class templated on an integer.  This class is intended to be
// used to let you specialize a funtion on a compile time number.
//
// For example, if you have an object of type T which you want to pass
// to a subroutine foo, but you want to specialize that subroutine based on
// the enum 'bar' defined in T, you could say:
//
// template<class T>  void foo(const T& t) 
// { 
//    foo(t,WrappedInt<T::bar>())
// }
//
// You can then specialize foo(T,WrappedInt<x>) for different values
// of x.
//
// With functor classes you can do this in a slightly slicker manner.
// Define the general functor class like so:
//
// template< class T, class Switch = WrappedInt<T::bar> >
// struct Foo
// {
//   static void foo()(const T& t);
// }
//
// Then you can specialize foo for different values of Switch.  That
// is you write things like Foo<T,WrappedInt<1>>::foo(const T&) to
// specialize it for T::bar==1. 
//
// With this construction you don't have two function calls and the
// WrappedInt is never constructed or passed.  Since it relies on Foo
// being templated though, you can't use it nested in another class.
//
//-----------------------------------------------------------------------------

template<int Integer> class WrappedInt
{
public:
  enum { val = Integer };
  POOMA_PURIFY_CONSTRUCTORS(WrappedInt)
};

#endif   // POOMA_UTILITIES_WRAPPEDINT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: WrappedInt.h,v $   $Author: sa_smith $
// $Revision: 1.10 $   $Date: 2000/05/23 23:18:44 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
