// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"

#ifdef	HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifdef CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

bool Trunk::scrExamine(void)
{
	Trunk *trk;
	Symbol *var;
	const char *pid;
	unsigned len = 0;

	pid = getKeyword("id");
	if(!pid)
		pid = getValue("9999");

	trk = driver->getTrunkId(pid);
	if(!trk)
	{
		error("examine-no-port");
		return true;
	}

	if(trk == this)
	{
		error("examine-self-reference");
		return true;
	}

	pid = getKeyword("var");
	if(!pid)
		pid = getContent(NULL);

	if(!pid)
	{
		error("examine-no-target");
		return true;
	}

	if(*pid == '%')
		++pid;

	var = getEntry(pid, getSymbolSize());
	if(!var)
	{
		error("examine-no-target");
		return true;
	}

	if(var->flags.readonly)
	{
		error("examine-readonly");
		return true;
	}

	var->data[len] = 0;
	trk->enterMutex();
	while(NULL != (pid = getContent(NULL)) && len < var->flags.size)
	{
		if(*pid == '%')
			++pid;

		pid = trk->getSymbol(pid);
		if(!pid)
			continue;

		if(len)
			var->data[len++] = ',';

		strncpy(var->data + len, pid, var->flags.size - len);
		var->data[var->flags.size] = 0;
	}
	if(var->flags.commit)
		commit(var);
	trk->leaveMutex();
	advance();
	return true;
}

bool Trunk::scrService(void)
{
	const char *mem = getMember();
	if(!mem)
		mem = "up";

	if(!stricmp(mem, "up"))
		service[0] = 0;
	else
		snprintf(service, sizeof(service), "down::%s", getValue("service"));

	advance();
	return true;
}
	
bool Trunk::scrBusy(void)
{
	TrunkEvent event;
	Trunk *trk;
	TrunkGroup *grp = NULL;
	const char *mem = getMember();
	int port, tspan;

	if(!mem)
		mem = "self";

	if(!stricmp(mem, "port"))
	{
		event.id = TRUNK_MAKE_BUSY;
		mem = getValue(getKeyword("id"));
		trk = driver->getTrunkId(mem);

		if(!trk)
		{
			error("busy-port-id");
			return true;
		}

		if(trk == this)
		{
			error("busy-self-reference");
			return true;
		}

		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!stricmp(mem, "span"))
	{
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("busy-span-id");
			return true;
		}
		tspan = atoi(mem);
		if(!driver->spanEvent(tspan, &event))
			error("busy-span-invalid");
		else
			advance();
		return true;
	}

	if(!stricmp(mem, "card"))
	{
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("busy-card-id");
			return true;
		}
		tspan = atoi(mem);
		if(!driver->cardEvent(tspan, &event))
			error("busy-card-invalid");
		else
			advance();
		return true;
	}

	if(!stricmp(mem, "group"))
	{
		mem = getValue(getKeyword("id"));
		if(mem)
			grp = getGroup(mem);
		if(!grp)
		{
			error("busy-group-id");
			return true;
		}

	        for(port = 0; port < driver->getTrunkCount(); ++port)
        	{
                	if(driver->getTrunkGroup(port) != grp)
                        	continue;

	                trk = driver->getTrunkPort(port);
        	        if(!trk || trk == this)
                        	continue;

			event.id = TRUNK_MAKE_BUSY;
			trk->postEvent(&event);
		}

		advance();
		return true;
	}
	error("busy-self-reference");
	return true;
}

bool Trunk::scrSlog(void)
{
        unsigned id = getId();
        const char *member = getMember();
	const char *file = getKeyword(file);
        char *val;
        Name *obj = getObject();
	char name[32];
	char buffer[256], encode[256];
	tgicmd_t cmd;

        if(!member)
                member = getKeyword("level");

	if(file)
	{
		buffer[0] = 0;
		while(NULL != (val = getValue(NULL)))
			strcat(buffer, val);
		val = urlEncode(buffer, encode, sizeof(encode));
		file = urlEncode(file, buffer, sizeof(buffer));
		snprintf(cmd.cmd, sizeof(cmd.cmd), "-log %s %s", val, file);
		cmd.port = id;
		cmd.mode = TGI_EXEC_NORMAL;

		data.sleep.rings = 0;
        	data.sleep.loops = 1;
        	data.sleep.save = NULL;
		data.sleep.wakeup = getTimeout("maxTime");
		if(!data.sleep.wakeup)
			data.sleep.wakeup = 30000;
	        ::write(tgipipe[1], &cmd, sizeof(cmd));
        	trunkStep(TRUNK_STEP_SLEEP);
        	return false;
	}

        if(member)
        {
                if(!strnicmp(member, "err", 3))
                        slog(Slog::levelError);
                else if(!strnicmp(member, "warn", 4))
                        slog(Slog::levelWarning);
                else if(!stricmp(member, "debug"))
                        slog(Slog::levelDebug);
                else if(!strnicmp(member, "crit", 4))
                        slog(Slog::levelCritical);
                else
                        slog(Slog::levelInfo);
        }
        else
                slog(Slog::levelInfo);

	getName(name);

        slog() << name << ": " << obj->name;
        if(id)
                slog() << "(" << id << ")";

        slog() << ": ";
        while(NULL != (val = getValue(NULL)))
                slog() << val;
        slog() << endl;
        advance();
        return true;
}

bool Trunk::scrSend(void)
{
	TrunkEvent event;
	Trunk *trk;
	Symbol *sym;
	int dig;
	const char *pid = getKeyword("id");
	const char *msg = getKeyword("message");
	const char *opt = getMember();
	const char *val;

	if(!opt)
		opt = "message";

	if(!pid)
		pid = getKeyword("gid");

	if(!pid)
		pid = getValue(NULL);

	if(!pid)
	{
		error("send-no-id");
		return true;
	}

	trk = driver->getTrunkId(pid);
	if(!trk)
	{
		error("send-no-session");
		return true;
	}

	if(trk == this)
	{
		error("send-self-reference");
		return true;
	}

	if(!stricmp(opt, "copy"))
	{
		trk->enterMutex();
		while(NULL != (opt = getOption(NULL)))
		{
			if(*opt != '%')
				continue;

			sym = getEntry(++opt, 0);
			if(!sym)
				continue;

			trk->setSymbol(opt, sym->flags.size);
			trk->setSymbol(opt, sym->data);
		}
		trk->leaveMutex();
	}
	else if(!strnicmp(opt, "dig", 3))
	{
		opt = getKeyword("digits");
		if(!opt)
			opt = getValue(NULL);
		if(!opt)
		{
			error("no-digits");
			return true;
		}
		while(*opt)
		{
			dig = getDigit(*(opt++));
			if(dig < 0)
				continue;
			event.id = TRUNK_DTMF_KEYUP;
			event.parm.dtmf.digit = dig;
			trk->postEvent(&event);
		}			
	}
	else if(!stricmp(opt, "post"))
	{
		trk->enterMutex();
		while(NULL != (opt = getOption(NULL)))
		{
			val = getValue("");
			if(*opt != '%')
				continue;

			sym = trk->getEntry(++opt, 0);
			if(!sym)
			{
				continue;
			}
			trk->postSymbol(sym, val);
		}
		trk->leaveMutex();
	}
	else
	{
		event.id = TRUNK_SEND_MESSAGE;
		event.parm.send.seq = seq;
		event.parm.send.src = this;
		event.parm.send.msg = getKeyword("message");
		if(!event.parm.send.msg)
			event.parm.send.msg = getValue("");
		trk->postEvent(&event);
	}

	advance();
	return true;
}

bool Trunk::scrAssign(void)
{
	const char *var = getKeyword("var");
	const char *value = getKeyword("value");
	const char *size = getKeyword("size");

	if(!value)
		value = "";

	if(!size)
		setSymbol(var, getSymbolSize());
	else
		setSymbol(var, atoi(size));
	setSymbol(var, value);
	advance();
	return true;
}

bool Trunk::scrPolicy(void)
{
	ScriptCommand *cmd = getCommand();
	Name *scr = getObject();
	Line *line = getScript();
	Symbol *sym;
	char *opt;
	const char *value = NULL, *def;
	const char *member = getMember();
	int argc = 0;
	char buffer[65];
	char local[65];

	while(argc < line->argc)
	{
		opt = line->args[argc++];
                if(*opt != '=')
                        continue;

                if(*(++opt) == '%')
                        ++opt;

		def = line->args[argc++];

                if(member)
                        snprintf(local, sizeof(local), "%s.%s", member, opt);
                else
                        snprintf(local, sizeof(local), "%s", opt);

		if(group)
			value = group->getLast(opt);

		if(!value)
			value = def;

		setConst(local, value);
	}
	advance();
	return true;
}

bool Trunk::scrConfig(void)
{
#ifdef	SCRIPT_LOCAL_DEFINE
	ScriptImage *img = getImage();
#else
	ScriptCommand *img = getCommand();
#endif

	Name *scr = getObject();
	Line *line = getScript();
	Symbol *sym;
	char *opt;
	const char *value, *def;
	const char *member = getMember();
	int argc = 0;
	char buffer[65];
	char appl[65];
	unsigned len;
	char local[65];

	snprintf(appl, sizeof(appl), "%s", scr->name);
	opt = strstr(appl, "::");
	if(opt)
		*opt = 0;

	while(argc < line->argc)
	{
		opt = line->args[argc++];
                if(*opt != '=')
                        continue;

                if(*(++opt) == '%')
                        ++opt;

		def = line->args[argc++];

                if(member)
		{
                        snprintf(local, sizeof(local), "%s.%s", member, opt);
			snprintf(buffer, sizeof(buffer), "%s.%s", member, opt);
		}
                else
		{
                        snprintf(local, sizeof(local), "%s", opt);
			snprintf(buffer, sizeof(buffer), "%s.%s", appl, opt);
		}

		value = img->getLast(buffer);
		if(!value)
			value = img->getLast(opt);

		if(!value)
			value = def;

		setConst(local, value);
	}
	advance();
	return true;
}	

bool Trunk::scrDummy(void)
{
	error("not-supported");
	return true;
}

bool Trunk::scrCleardigits(void)
{
	const char *mem = getMember();
	Line *line = getScript();
	trunksignal_t sig;
	unsigned dig = 0;
	unsigned count;

	if(!mem)
		mem = "all";

	if(!stricmp(mem, "last") && digits)
	{	
		dtmf.bin.data[0] = dtmf.bin.data[digits - 1];
		dtmf.bin.data[1] = 0;
		digits = 1;
	}
	else if(atoi(mem) > 0)
	{
		count = atoi(mem);
		if(count > digits)
			count = digits;

		while(dig < count)
			dtmf.bin.data[dig++] = '-';
	}
	else if(!stricmp(mem, "pop") && digits)
		dtmf.bin.data[0] = '-';
	else if(stricmp(mem, "trap"))
	{
		dtmf.bin.data[0] = 0;
		digits = 0;
	}
	
	if(line->argc > 0)
		scrGoto();
	else
		advance();

retry:
	if(!digits)
		return true;

	switch(dtmf.bin.data[0])
	{
	case '*':
		sig = TRUNK_SIGNAL_STAR;
		break;
	case '#':
		sig = TRUNK_SIGNAL_POUND;
		break;
	case 'a':
	case 'A':
		sig = TRUNK_SIGNAL_A;
		break;
	case 'b':
	case 'B':
		sig = TRUNK_SIGNAL_B;
		break;
	case 'c':
	case 'C':
		sig = TRUNK_SIGNAL_C;
		break;
	case 'd':
	case 'D':
		sig = TRUNK_SIGNAL_D;
		break;
	case '0':
		sig = TRUNK_SIGNAL_0;
		break;
	case '1':
		sig = TRUNK_SIGNAL_1;
		break;
	case '2':
		sig = TRUNK_SIGNAL_2;
		break;
	case '3':
		sig = TRUNK_SIGNAL_3;
		break;
	case '4':
		sig = TRUNK_SIGNAL_4;
		break;
	case '5':
		sig = TRUNK_SIGNAL_5;
		break;
	case '6':
		sig = TRUNK_SIGNAL_6;
		break;
	case '7':
		sig = TRUNK_SIGNAL_7;
		break;
	case '8':
		sig = TRUNK_SIGNAL_8;
		break;
	case '9':
		sig = TRUNK_SIGNAL_9;
		break;
	default:
		sig = TRUNK_SIGNAL_STEP;
	}

	if(sig != TRUNK_SIGNAL_STEP)			
		if(trunkSignal(sig))
			return true;

	dig = 0;
	while(dig < digits)
	{	
		dtmf.bin.data[dig] = dtmf.bin.data[dig + 1];
		++dig;
	}
	dtmf.bin.data[dig] = 0;
	digits = --dig;
	goto retry;	
}

bool Trunk::scrIdle(void)
{
	TrunkEvent event;
	Trunk *trk;
	TrunkGroup *grp = NULL;
	const char *mem = getMember();
	int port, tspan;

	if(!mem)
		mem = "self";

	if(!stricmp(mem, "port"))
	{
		event.id = TRUNK_MAKE_IDLE;
		mem = getValue(getKeyword("id"));
		trk = driver->getTrunkId(mem);

		if(!trk)
		{
			error("idle-port-id");
			return true;
		}

		if(trk == this)
		{
			error("idle-self-reference");
			return true;
		}

		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!stricmp(mem, "span"))
	{
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("idle-span-id");
			return true;
		}
		tspan = atoi(mem);
		if(driver->spanEvent(tspan, &event))
			advance();
		else
			error("idle-span-invalid");
		return true;
	}

        if(!stricmp(mem, "card"))
        {
                mem = getValue(getKeyword("id"));
                if(!mem)
                {
                        error("idle-card-id");
                        return true;
                }
                tspan = atoi(mem);
                if(driver->cardEvent(tspan, &event))
                        advance();
                else
                        error("idle-card-invalid");
                return true;
        }

	if(!stricmp(mem, "group"))
	{
		mem = getValue(getKeyword("id"));
		if(mem)
			grp = getGroup(mem);
		if(!grp)
		{
			error("idle-group-id");
			return true;
		}

	        for(port = 0; port < driver->getTrunkCount(); ++port)
        	{
                	if(driver->getTrunkGroup(port) != grp)
                        	continue;

	                trk = driver->getTrunkPort(port);
        	        if(!trk || trk == this)
                        	continue;

			event.id = TRUNK_MAKE_IDLE;
			trk->postEvent(&event);
		}

		advance();
		return true;
	}

	idle_timer = atoi(getValue("0"));
	advance();
	return true;
}

bool Trunk::scrSchedule(void)
{
	char cmd[65];
	bool ok = false;

	strcpy(cmd, "schedule ");
	strcat(cmd, getValue(""));
	if(!fifo.command(cmd))
	{
		error("schedule-failed");
		return true;
	}
	advance();
	return true;
}

bool Trunk::scrSignal(void)
{
	Trunk *trunk;
	TrunkEvent event;

	trunk = driver->getTrunkPort(atoi(getValue("-1")));
	if(!trunk)
	{
		error("signal-no-such-trunk");
		return true;
	}

	event.id = TRUNK_SIGNAL_NOTIFY;
	event.parm.error = getKeyword("message");
	if(!event.parm.error)
		event.parm.error = getValue(NULL);
	if(!trunk->postEvent(&event))
	{
		error("signal-not-waiting");
		return true;
	}
	advance();
	return true;
}

bool Trunk::scrModule(void)
{
	Line *line = getScript();
	char *cmd = line->cmd;
	char keybuf[33];
	int len = 0;
	char *kw = keybuf;
	char *cp;

	cp = strchr(cmd, '-');
	if(cp)
		cmd = ++cp;
	
	while(len++ < 32 && *cmd && *cmd != '.')
		*(kw++) = *(cmd++);
	*kw = 0;

	Module *module = getModule(MODULE_ANY, keybuf);
	char *err;
	unsigned delay;

	if(!module)
	{
		error("module-not-found");
		return true;
	}

	err = module->dispatch(this);
	if(err)
	{
		error(err);
		return true;
	}

	delay = module->sleep(this);
	if(!delay)
	{
		advance();
		return true;
	}
	if(delay == -1)
		return module->executePrior(this);
	data.sleep.wakeup = delay * 1000;
	data.sleep.rings = 0;
	data.sleep.loops = 1;
	data.sleep.save = NULL;
	trunkStep(TRUNK_STEP_SLEEP);
	module->commit(this);
	return false;
}

bool Trunk::scrExists(void)
{
        const char *mem = getMember();
        char *path = getValue(NULL);
        const char *prefix = getPrefixPath();
        char buf[256];

        if(!mem)
                mem = "file";

        if(!path)
        {
                error("exists-missing-path");
                return true;
        }

        if(prefix)
        {
                snprintf(buf, sizeof(buf), "%s/%s", prefix, path);
                path = buf;
        }

        if(!stricmp(mem, "dir"))
        {
                if(isDir(path))
                        return scrGoto();
        }
        else if(!stricmp(mem, "missing") || !stricmp(mem, "not"))
        {
                if(!canAccess(path))
                        return scrGoto();
        }
        else
        {
                if(canAccess(path))
                        return scrGoto();
        }

        advance();
        return true;
}

bool Trunk::scrMove(void)
{
	const char *prefix = getPrefixPath();
	const char *n1 = getValue(NULL);
	const char *n2 = getValue(NULL);
	char buf1[256], buf2[256];

	if(!n1 || !n2)
	{
		error("move-no-files");
		return true;
	}

	if(prefix)
	{
		snprintf(buf1, sizeof(buf1), "%s/%s", prefix, n1);
		snprintf(buf2, sizeof(buf2), "%s/%s", prefix, n2);
		n1 = (const char *)buf1;
		n2 = (const char *)buf2;
	}
	if(rename(n1, n2))
		error("move-failed");
	else
		advance();
	return true;
}

bool Trunk::scrErase(void)
{
	const char *prefix = getPrefixPath();	
	const char *name = getValue(NULL);

	if(!name)
	{
		error("erase-no-file");
		return true;
	}

	if(prefix)
	{
		snprintf(apppath, sizeof(apppath), "%s/%s",
			prefix, name);
		remove(apppath);
	}
	else
		remove(name);

	advance();
	return true;
}

bool Trunk::scrRecord(void)
{
	const char *member = getMember();
	const char *cp;
	const char *prefix = getPrefixPath();
	char *gain = getKeyword("gain");
	char *vol = getKeyword("volume");

	cp = getKeyword("trim");
	if(!cp)
		cp = getSymbol(SYM_TRIM);
	if(!cp)	
		cp = "0";
	data.record.trim = atoi(cp);

	cp = getKeyword("frames");
	if(!cp)
		cp = "0";

	data.record.frames = atoi(cp);
	cp = getKeyword("minSize");
	if(!cp)
		cp = "0";
	data.record.minsize = atoi(cp);

	if(!vol)
		vol = getSymbol(SYM_VOLUME);

	if(!vol)
		vol = "100";

	apppath[0] = 0;

	if(!member)
		member="all";

	data.record.text = getKeyword("text");
	data.record.name = getValue("");	
	if(!data.record.name)
	{
		error("record-no-file");
		return true;
	}

	if(prefix)
	{
		snprintf(apppath + 1, sizeof(apppath) - 1, "%s/%s",
			prefix, data.record.name);
		data.record.name = apppath + 1;
	}

	data.record.timeout = getTimeout("maxTime");
	data.record.term = getDigitMask("exit");
	data.record.offset = (unsigned long)-1;
	data.record.volume = atoi(vol);
	data.record.silence = 0;
	data.record.encoding = getKeyword("encoding");
	data.record.annotation = getKeyword("annotation");
	data.record.extension = getKeyword("extension");

	if(!data.record.encoding)
		data.record.encoding = getDefaultEncoding();

	if(!data.record.annotation)
		data.record.annotation = "";

	if(!data.record.extension)
		data.record.extension = getSymbol(SYM_EXTENSION);

	if(gain)
		data.record.gain = (float)strtod(gain, NULL);
	else
		data.record.gain = 0.0;
	
	data.record.info = false;

	if(!stricmp(member, "append"))
		data.record.append = true;
	else if(!stricmp(member, "info"))
	{
		data.record.append = true;
		data.record.info = true;
	}
	else
		data.record.append = false;
	if(NULL != (cp = getKeyword("offset")))
		data.record.offset = atoi(cp);
	if(NULL != (cp = getKeyword("volume")))
		data.record.volume = atoi(cp);
	if(NULL != (cp = getKeyword("silence")))
		data.record.silence = atoi(cp);

	trunkStep(TRUNK_STEP_RECORD);
	return false;
}

bool Trunk::scrTransfer(void)
{
	unsigned len;
	char *cp = (char *)group->getLast("transfer");
	if(cp)
		strncpy(data.dialxfer.digits, cp, sizeof(data.dialxfer.digits));
	else
		data.dialxfer.digits[0] = 0;
	len = strlen(data.dialxfer.digits);

	while(NULL != (cp = getValue(NULL)) && len < sizeof(data.dialxfer.digits))
	{
		strncpy(data.dialxfer.digits + len, cp, sizeof(data.dialxfer.digits) - len);
		len = strlen(data.dialxfer.digits);
	}

	data.dialxfer.digits[sizeof(data.dialxfer.digits) - 1] = 0;
	data.dialxfer.interdigit = group->getDialspeed();
	data.dialxfer.digit = data.dialxfer.digits;
	data.dialxfer.exit = true;
	data.dialxfer.timeout = 0;
	cp = getKeyword("onhook");
	if(!cp)
		cp = getKeyword("flash");
	if(!cp)
		cp = getValue(group->getLast("flash"));
	data.dialxfer.onhook = getMSTimeout(cp);
	cp = getKeyword("dialtone");
	if(!cp)
		cp = getKeyword("offhook");
	if(!cp)
		cp = getValue(group->getLast("dialtone"));
        data.dialxfer.offhook = getMSTimeout(cp);
	trunkStep(TRUNK_STEP_FLASH);
	return false;
}

bool Trunk::scrHold(void)
{
	const char *cp = group->getLast("hold");
	if(!cp)
		cp = "";

	strcpy(data.dialxfer.digits, cp);
	data.dialxfer.interdigit = group->getDialspeed();
        data.dialxfer.digit = data.dialxfer.digits;
        data.dialxfer.exit = true;
        data.dialxfer.timeout = 0;
	cp = getKeyword("onhook");
	if(!cp)
		cp = getKeyword("flash");
	if(!cp)
		cp = getValue(group->getLast("flash"));
        data.dialxfer.onhook = getMSTimeout(cp);
	cp = getKeyword("offhook");
	if(!cp)
		cp = getKeyword("dialtone");
	if(!cp)
		cp = getValue(group->getLast("dialtone"));
        data.dialxfer.offhook = getMSTimeout(cp);
	trunkStep(TRUNK_STEP_FLASH);
	return false;
}

bool Trunk::scrSpeak(void)
{
	Translator *tts;
	char *err;
	const char *member = getMember();
	const char *gain = getKeyword("gain");
	const char *lang = getKeyword("language");
	const char *vol = getKeyword("volume");

	if(!vol)
		vol = getSymbol(SYM_VOLUME);

	if(!vol)
		vol = "100";

	if(!lang)
		lang = getSymbol(SYM_LANGUAGE);

	apppath[0] = 0;

	if(!member)
		member="all";

	data.play.text = getKeyword("text");
	if(!stricmp(member, "any"))
		data.play.mode = PLAY_MODE_ANY;
	else
		data.play.mode = PLAY_MODE_NORMAL;

	data.play.term = 0;
	data.play.voice = getKeyword("voice");

	tts = getTranslator(lang);

	if(!tts)
	{
		error("language-unsupported");
		return true;
	}
	err = tts->speak(this);
	if(err)
	{
		error(err);
		return true;
	}
	while(*data.play.name == ',')
		++data.play.name;

	if(gain)
		data.play.gain = (float)strtod(gain, NULL);
	else
		data.play.gain = 0.0;

	data.play.extension = getKeyword("extension");
	if(!data.play.extension)
		data.play.extension = getSymbol(SYM_EXTENSION);

	data.play.offset = data.play.limit = 0;
	data.play.timeout = 0;
	data.play.maxtime = 0;
	data.play.repeat = 0;
	data.play.volume = atoi(vol);
	trunkStep(TRUNK_STEP_PLAY);
	return false;
}
	
bool Trunk::scrDial(void)
{
	unsigned len = 0;
	const char *cp;

	data.dialxfer.digits[0] = 0;
	while(NULL != (cp = getValue(NULL)) && len < 64)
	{
		strncpy(data.dialxfer.digits + len, cp, 64 - len);
		len += strlen(cp);
	}
	data.dialxfer.digits[len] = 0;

	data.dialxfer.callingdigit = getKeyword("origin");
	data.dialxfer.digits[sizeof(data.dialxfer.digits) - 1] = 0;
	data.dialxfer.interdigit = group->getDialspeed();
	data.dialxfer.digit = data.dialxfer.digits;
	data.dialxfer.exit = false;
	cp = getKeyword("maxTime");
	if(cp)
		data.dialxfer.timeout = getSecTimeout(cp);
	else
		data.dialxfer.timeout = group->getAnalysis() * 1000;
	trunkStep(TRUNK_STEP_DIALXFER);
	return false;
}

bool Trunk::scrDTMF(void)
{
	unsigned len = 0;
	char *cp;

	data.dialxfer.digits[0] = 0;

	while(NULL != (cp = getValue(NULL)) && len < sizeof(data.dialxfer.digits))
	{
		strncpy(data.dialxfer.digits + len, cp, sizeof(data.dialxfer.digits));
		len += strlen(cp);
	}
	data.dialxfer.digits[len] = 0;

	data.dialxfer.digits[sizeof(data.dialxfer.digits) - 1] = 0;
	data.dialxfer.interdigit = group->getDialspeed();
	data.dialxfer.digit = data.dialxfer.digits;
	data.dialxfer.exit = false;
	data.dialxfer.timeout = 0;
	trunkStep(TRUNK_STEP_DIALXFER);
	return false;
}

bool Trunk::scrSay(void)
{
	char *cp;
	const char *gain = getKeyword("gain");
	const char *vol = getKeyword("volume");
	unsigned len = 0;

	if(!vol)
		vol = getSymbol(SYM_VOLUME);

	if(!vol)
		vol = "100";

	data.play.list[0] = 0;

	while(NULL != (cp = getValue(NULL)) && len < sizeof(data.play.list))
	{
		if(len)
			data.play.list[len++] = '+';
		strncpy(data.play.list + len, cp, sizeof(data.play.list) - len);
	}
	if(!data.play.list[0])
	{
		error("tts-no-output");
		return true;
	}
	data.play.text = getKeyword("text");
	data.play.voice = getKeyword("voice");
	data.play.list[sizeof(data.play.list) - 1] = 0;
	libtts(data.play.list, TTS_GATEWAY_TEXT);
	sprintf(data.play.list, "temp/.tts.%d.ul", id);
	data.play.name = data.play.list;
	data.play.repeat = 0;
	data.play.maxtime = 0;
	data.play.timeout = getSecTimeout(getSymbol(SYM_PLAYWAIT));
	data.play.volume = atoi(vol);
	data.play.term = 0;
	data.play.mode = PLAY_MODE_TEMP;
	data.play.limit = data.play.offset = 0;
	data.play.extension = NULL;

	if(gain)
		data.play.gain = (float)strtod(gain, NULL);
	else
		data.play.gain = 0.0;

	if(NULL != (cp = getKeyword("offset")))
		data.play.offset = atol(cp);
	if(NULL != (cp = getKeyword("volume")))
		data.play.volume = atoi(cp);
	if(NULL != (cp = getKeyword("timeout")))
		data.play.timeout = getSecTimeout(cp);
	if(NULL != (cp = getKeyword("limit")))
		data.play.limit = atol(cp);
	trunkStep(TRUNK_STEP_PLAYWAIT);
	return false;
}

#ifdef	XML_SCRIPTS
bool Trunk::scrLoad(void)
{
	Module *mod;
	const char *kw = getMember();
	Name *scr = getObject();
	Line *line = getScript();
	int count;
	const char *prefix;
	char *sym;
	char *value;
	char *str;
	char *var;
	char *cp;
	int len;

	if(!kw)
		kw = "put";

	if(!stricmp(kw, "xml"))
	{
		var = getValue("");
		if(altimage)
		{
			altimage->purge();
			delete altimage;
			altimage = NULL;
		}
		mod = getModule(MODULE_XML, var);
		if(mod)
			altimage = mod->getXML();
	}
 
	if(!altimage)
	{
		error("no-xml-parser");
		return true;
	}

	if(!stricmp(kw, "xml"))
	{
		advance();
		return true;
	}

	altimage->purge();
	data.load.attach = false;
	if(!stricmp(kw, "post"))
		data.load.post = true;
	else
		data.load.post = false;
	kw = getKeyword("section");
	if(!kw)
		kw = "#";
	if(*kw != '#')
	{
		var = (char *)altimage->alloc(strlen(kw + 1));
		var[0] = '#';
		strcpy(var + 1, ++kw);
		kw = var;
	}
	data.load.section = kw;
	kw = getKeyword("maxTime");	
	if(!kw)
		kw = "60";
	data.load.image = altimage;
	data.load.timeout = getSecTimeout(kw);
	data.load.parent = NULL;
	count = line->argc + 1;
	data.load.vars = (char **)altimage->alloc(count * sizeof(char *));
	data.load.url = getValue(NULL);
	if(!data.load.url)
	{
		error("no-xml-url");
		return true;
	}

	data.load.userid[0] = 0;
#ifdef	USER_HOSTING
	if(!strnicmp(data.load.url, "~/", 2))
	{
		snprintf(data.load.filepath, 65, "%s", scr->name);
		cp = strchr(data.load.filepath, ':');
		if(cp)
			*cp = 0;
		if(data.load.filepath[0] == '~')
		{
			snprintf(data.load.userid, sizeof(data.load.userid), 
				"%s", data.load.filepath + 1);
			prefix = keyusers.getLast(data.load.filepath + 1);
		}
		else if(data.load.filepath[0] == '#')
		{
			prefix = getSymbol(SYM_HOME);
			if(!prefix)
				prefix = "";
			if(*prefix)
				prefix = keyusers.getLast(prefix);
			else
				prefix = "xml";
		}
		else
			prefix = "xml";
		if(!prefix)
		{
			error("no-such-user");
			return true;
		}
		snprintf(data.load.filepath, sizeof(data.load.filepath),
			"%s/%s", prefix, data.load.url + 2);
		data.load.url = data.load.filepath;
	}
	else if(data.load.url[0] == '~' && NULL != (cp = strchr(data.load.url, '/')))
	{
		strncpy(data.load.filepath, data.load.url, 65);
		str = strchr(data.load.filepath, '/');
		if(str)
			*str = 0;
		prefix = keyusers.getLast(data.load.filepath + 1);
		if(prefix)
			++cp;
		else
		{
			prefix = "xml";
			cp = (char *)(data.load.url + 1);
		}
		snprintf(data.load.filepath, sizeof(data.load.filepath),
			"%s/%s", prefix, cp);
		data.load.url = data.load.filepath;
	}
#endif

	setSymbol(SYM_HOME, data.load.userid);

	if(!strnicmp(data.load.url, "http:", 5))
		altimage->setProxy(keyproxy.getHTTPServer(), keyproxy.getHTTPPort());
	else
		altimage->setProxy(NULL, 0);
		
	count = 0;
	while(NULL != (sym = getOption(NULL)))
	{
		value = getContent(sym);
		if(*sym == '%' || *sym == '@')
			++sym;
		len = (strlen(sym) + strlen(value) * 2);
		var = (char *)altimage->alloc(len);
		data.load.vars[count++] = var;
		urlEncode(sym, var, len);
		strcat(var, "=");
		len -= strlen(var);
		var += strlen(var);
		urlEncode(sym, value, len);
	}
	data.load.vars[count] = NULL;
	if(NULL != (kw = getKeyword("timeout")))
		data.load.timeout = atoi(kw) * 1000;		
	trunkStep(TRUNK_STEP_LOADER);
	return false;
}
#endif

bool Trunk::scrPlay(void)
{
	Name *scr = getObject();
	char *cp;
	unsigned long mask = 0;
	const char *prefix = getPrefixPath();
	char *gain = getKeyword("gain");
	char *vol = getKeyword("volume");
	bool feed = false;
	unsigned len = 0;

	snprintf(apppath, sizeof(apppath), "%s", scr->name);
	cp = strstr(apppath, "::");
	if(cp)
		*cp = 0;

	const char *member = getMember();
	if(!member)
		member = "all";

	if(!stricmp(member, "feed"))
		feed = true;

	if(!stricmp(member, "any"))
		data.play.mode = PLAY_MODE_ANY;
	else if(!stricmp(member, "one"))
		data.play.mode = PLAY_MODE_ONE;
	else if(!stricmp(member, "temp") || !stricmp(member, "tmp"))
		data.play.mode = PLAY_MODE_TEMP;
	/* else if(0 != (mask = getScriptMask(member)))
		data.play.mode = PLAY_MODE_ANY; */
	else
		data.play.mode = PLAY_MODE_NORMAL;

	data.play.text = getKeyword("text");
	data.play.voice = getKeyword("voice");

	if(mask)
	{
		if((mask & scr->mask) != mask)
		{
			advance();
			return true;
		}
	}

	data.play.list[0] = 0;
	while(NULL != (cp = getValue(NULL)))
	{
		if(len)
			data.play.list[len++] = ',';
		if(prefix)
			snprintf(data.play.list + len, sizeof(data.play.list) - len, "%s/%s", prefix, cp);
		else
			strncpy(data.play.list + len, cp, sizeof(data.play.list) - len);
		len = strlen(data.play.list);
		if(feed)
			break;
	}

	if(!vol)
		vol = getSymbol(SYM_VOLUME);
	if(!vol)
		vol = "100";

	data.play.list[sizeof(data.play.list) - 1] = 0;
	data.play.name = data.play.list;
	data.play.volume = atoi(vol);
	data.play.timeout = 0;
	data.play.repeat = 0;
	data.play.maxtime = 0;
	data.play.term = 0;
	data.play.extension = getKeyword("extension");
	if(!data.play.extension)
		data.play.extension = getSymbol(SYM_EXTENSION);

	if(feed)
		data.play.maxtime = getTimeout("maxTime");

	while(*data.play.name == ',')
		++data.play.name;
	if(!*data.play.name)
	{
		error("play-no-files");
		return true;
	}

	if(gain)
		data.play.gain = (float)strtod(gain, NULL);
	else
		data.play.gain = 0.0;

	data.play.limit = data.play.offset = 0;

	if(NULL != (cp = getKeyword("offset")))
		data.play.offset = atol(cp);

	if(NULL != (cp = getKeyword("limit")))
		data.play.limit = atol(cp);

	if(NULL != (cp = getKeyword("volume")))
		data.play.volume = atoi(cp);

	trunkStep(TRUNK_STEP_PLAY);
	return false;
}	

bool Trunk::scrFlash(void)
{
	const char *cp = getKeyword("offhook");
	if(!cp)
		cp = getKeyword("flash");
	if(!cp)
		cp = getValue(group->getLast("flash"));
	data.dialxfer.onhook = getMSTimeout(cp);

	cp = getKeyword("onhook");
	if(!cp)
		cp = getKeyword("dialtone");
	if(!cp)
		cp = getValue(group->getLast("dialtone"));

	data.dialxfer.offhook = getMSTimeout(cp);

	data.dialxfer.digit = NULL;
	trunkStep(TRUNK_STEP_FLASH);
	return false;
}

bool Trunk::scrCollect(void)
{
	unsigned copy = 0;
	Symbol *sym = NULL;
	const char *cp = getKeyword("count");
	const char *mem = getMember();
	const char *var = getKeyword("var");
	const char *ignore = getKeyword("ignore");
	const char *term = getKeyword("term");
	unsigned digpos = 0, digscan;
	bool trim = false;

	if(!ignore)
		ignore = "";

	if(!term)
		term = "";

	if(!mem)
		mem = "all";

	if(!cp)
		cp = getKeyword("digits");

	if(!cp)
		cp = getValue("0");

	data.collect.count = atoi(cp);
	if(data.collect.count > MAX_DIGITS)
		data.collect.count = MAX_DIGITS;

	if(var)
		sym = getLocal(var, data.collect.count);

	if(sym)
	{
		if(sym->flags.readonly)
		{
			error("collect-read-only");
			return true;
		}
	}

	if(sym)
	{
		copy = sym->flags.size;
		sym->data[0] = 0;
		if(sym->flags.commit)
			commit(sym);
	}

	if(copy > data.collect.count)
		copy = data.collect.count;		

	data.collect.var = (void *)sym;
	data.collect.map = NULL;	data.collect.timeout = getInterdigit("timeout");
	data.collect.term = getDigitMask("exit");
	data.collect.ignore = getDigitMask("ignore");
	if(!stricmp(mem, "clear"))
	{
		digits = 0;
		dtmf.bin.data[0] = 0;
	}
	else if(!stricmp(mem, "trim"))
		trim = true;
	else if(!stricmp(mem, "input"))
		trim = true;
	
	while(digpos < digits)
	{
		if(strchr(term, dtmf.bin.data[digpos]))
		{
			if(copy > digpos)
				copy = digpos;
			if(sym)
			{
				if(copy)
					strncpy(sym->data, dtmf.bin.data, copy);
				sym->data[copy] = 0;
				if(sym->flags.commit)
					commit(sym);
				digscan = ++digpos;
				while(digscan < digits)
				{
					dtmf.bin.data[digscan - digpos] = dtmf.bin.data[digscan];
					++digscan;
				}
				digits = digscan - digpos;
				dtmf.bin.data[digits] = 0;
				digits = digscan;
			}	
			else
			{
				digits = digpos;
				dtmf.bin.data[digits] = 0;
			}
			advance();
			return true;
		}

		if(strchr(ignore, dtmf.bin.data[digpos]))
		{
			digscan = digpos;
			while(digscan < digits)
			{
				dtmf.bin.data[digscan] = dtmf.bin.data[digscan + 1];
				++digscan;
			}
			continue;
		}	

		if(++digpos >= data.collect.count)
		{
			if(sym)
			{
				if(copy)
					strncpy(sym->data, dtmf.bin.data, copy);
				sym->data[copy] = 0;
				if(sym->flags.commit)
					commit(sym);
				while(digpos < digits)
				{
					dtmf.bin.data[digpos - data.collect.count] = dtmf.bin.data[digpos];
					++digpos;
				}
				digits = digpos - data.collect.count;
				dtmf.bin.data[digits] = 0;
			}
			else if(trim || *term)
			{
				digits = digpos;
				dtmf.bin.data[digits] = 0;
			}
			advance();
			return true;
		}
	}
	trunkStep(TRUNK_STEP_COLLECT);
	return false;
}

bool Trunk::scrAccept(void)
{
	if (group->getAccept())
	{
		advance();
		return true;
	}

	trunkStep(TRUNK_STEP_ACCEPT);
	return false;
}

bool Trunk::scrReject(void)
{
	trunkStep(TRUNK_STEP_REJECT);
	return false;
}
	
bool Trunk::scrAnswer(void)
{
	const char *kw;

	if(NULL != (kw = getKeyword("maxRing")))
		data.answer.rings = atoi(kw);
	else 	
		data.answer.rings = atoi(getValue("0"));
	if(NULL != (kw = getKeyword("maxTime")))
		data.answer.timeout = getSecTimeout(kw);
	else
		data.answer.timeout =  getSecTimeout(getValue(group->getLast("ringtime")));
	trunkStep(TRUNK_STEP_ANSWER);
	return false;
}	

bool Trunk::scrHangup(void)
{
	TrunkEvent event;
	Trunk *trk;
	TrunkGroup *grp = NULL;
	const char *mem = getMember();
	int port, tspan;

	if(!mem)
		mem = "self";

	if(!stricmp(mem, "port"))
	{
		event.id = TRUNK_STOP_DISCONNECT;
		mem = getValue(getKeyword("id"));
		trk = driver->getTrunkId(mem);

		if(!trk)
		{
			error("hangup-port-id");
			return true;
		}

		if(trk == this)
		{
			error("hangup-self-reference");
			return true;
		}

		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!stricmp(mem, "span"))
	{
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("hangup-span-id");
			return true;
		}
		tspan = atoi(mem);
		if(driver->spanEvent(tspan, &event))
			advance();
		else
			error("hangup-span-invalid");
		return true;			
	}

        if(!stricmp(mem, "card"))
        {
                mem = getValue(getKeyword("id"));
                if(!mem)
                {
                        error("hangup-card-id");
                        return true;
                }
                tspan = atoi(mem);
                if(driver->cardEvent(tspan, &event))
                        advance();
                else
                        error("hangup-card-invalid");
                return true;
        }

	if(!stricmp(mem, "group"))
	{
		mem = getValue(getKeyword("id"));
		if(mem)
			grp = getGroup(mem);
		if(!grp)
		{
			error("hangup-group-id");
			return true;
		}

	        for(port = 0; port < driver->getTrunkCount(); ++port)
        	{
                	if(driver->getTrunkGroup(port) != grp)
                        	continue;

	                trk = driver->getTrunkPort(port);
        	        if(!trk || trk == this)
                        	continue;

			event.id = TRUNK_STOP_DISCONNECT;
			trk->postEvent(&event);
		}

		advance();
		return true;
	}

	if(!ScriptInterp::signal((unsigned int)0))
		scrExit();

	return true;
}

bool Trunk::scrAudit(void)
{
	const char *mem = getMember();
	char buffer[256];
	char overflow[256];
	Line *line = NULL;
	char *tag, *val;
	int argc = 0;
	unsigned len = 0;
	bool post = false;

	if(!mem)
		mem = "set";

	if(!stricmp(mem, "clear"))
		cdrv = NULL;
	else if(!stricmp(mem, "log"))
		line = getScript();
	else if(!stricmp(mem, "post"))
	{
		post = true;
		line = getScript();
	}
	else
		cdrv = getScript();

	if(!line)
	{
		advance();
		return true;
	}

	buffer[0] = 0;
	while(argc < line->argc && len < sizeof(buffer))
	{
		tag = line->args[argc++];
		if(*tag == '%')
			val = getContent(tag);
		else if(*tag == '=')
			val = getContent(line->args[argc++]);
		else
			continue;

		if(!val)
			continue;

		if(!*val)
			continue;

		urlEncode(val, overflow, sizeof(overflow));
		if(len)
			buffer[len++] = ' ';
		snprintf(buffer + len, sizeof(buffer) - len, "%s=%s", ++tag, overflow);
		len = strlen(buffer);
	}
	if(post)
	{
		audit(this, buffer);
		cdrv = NULL;
	}
	else
		alog(this, buffer);
	advance();
	return true;
}

bool Trunk::scrDebug(void)
{
	char buf[256];
	char *value;

	buf[0] = 0;
	while(NULL != (value = getValue(NULL)))
		strcat(buf, value);
	debug->debugScript(this, buf);
	advance();
	return true;
}

bool Trunk::scrSync(void)
{
	const char *cp;
	timeout_t timer = getTimeout("time");
	time_t now;
	const char *mem = getMember();

	if(!mem)
		mem = "none";

	time(&now);


	if(!strnicmp(mem, "max", 3) || !stricmp(mem, "exit"))
	{
		if(timer)
			exittimer = starttime + (timer / 1000);
		else
			exittimer = 0;
		advance();
		return true;
	}

	if(!strnicmp(mem, "time", 4))
	{
		if(timer)
			synctimer = starttime + (timer / 1000);
		else
			synctimer = 0;
	}

	timer = timer - (now - starttime);
	if(timer < 1)
	{
		advance();
		return true;
	}
	data.sleep.wakeup = timer;
	data.sleep.save = NULL;
	cp = getKeyword("maxRing");
	if(!cp)
		cp = getValue("0");
	data.sleep.rings = atoi(cp);
	data.sleep.loops = 1;
	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

bool Trunk::scrStart(void)
{
	Trunk *trunk;
	TrunkGroup *grp;
	TrunkEvent event;
	Symbol *sym;
	int argc = 0;
	char *argv[32];
	char *arg, *tok;
	const char *var = getKeyword("var");
	const char *submit = getKeyword("submit");
	unsigned expires;
	timeout_t exp = 0;
	bool notify = false;
	bool rtn = true;
	const char *mem = getMember();
	char buffer[256];
	char args[512];
	char content[512];
	unsigned alen = 0;
	unsigned offset = 0, last = 0, span = 0;
	bool start = false;
	bool ports = false;
	Name *scr = getObject();

	args[0] = 0;

	if(!mem)
		mem = "none";

	expires = exp / 1000l;

	if(!stricmp(mem, "wait"))
	{
		timeout_t exp = getTimeout("maxTime");
		notify = true;
	}
	else if(!stricmp(mem, "port"))
	{
		arg = getKeyword("first");
		if(arg)
			offset = atoi(arg);
		else
			offset = atoi(getValue("0"));
		arg = getKeyword("last");
		if(arg)
			last = atoi(arg);
		else
			last = offset;
		ports = true;
	}
	else if(!stricmp(mem, "span"))
	{
		span = atoi(getValue("1"));
		offset = 0;
		last = driver->getTrunkCount() - 1;
		ports = true;
	}
	else if(!stricmp(mem, "offset"))
	{
		arg = getKeyword("first");
		if(arg)
			offset = atoi(arg) + id;
		else
			offset = atoi(getValue("1")) + id;
		arg = getKeyword("last");
		if(arg)
			last = id + atoi(arg);
		else
			last = driver->getTrunkCount() - 1;
		ports = true;
	}
	else if(!stricmp(mem, "group"))
		start = true;
	else
		timeout_t exp = getTimeout("maxTime");

	if(!ports && !span)
	{
		arg = getKeyword("group");
		if(!arg)
			arg = getValue("*");

		grp = getGroup(arg);
		if(!grp)
		{
			error("request-unknown-group");
			return true;
		}
	}

	if(start)
	{
		if(grp)
			snprintf(args + alen, sizeof(args) - alen, "start %s ", arg);
		else
			snprintf(args + alen, sizeof(args) - alen, "start %s ", offset);
		alen = strlen(args);
	}

	arg = getKeyword("script");
	if(!arg)
		arg = getValue(NULL);
	if(!arg)
	{
		error("request-no-script");
		return true;
	}

	if(!strnicmp(arg, "::", 2))
	{
		strcpy(content, scr->name);
		tok = strstr(content, "::");
		if(!tok)
			tok = content + strlen(content);
		strcpy(tok, arg);
		arg = content;
	}

	strncpy(args + alen, arg, sizeof(args) - alen);
	alen = strlen(args);

	if(start || ports || span)
		snprintf(args + alen, sizeof(args) - alen, " %s=%s", SYM_PARENT, getSymbol(SYM_GID));
	alen = strlen(args);

	while(NULL != (arg = getOption(NULL)))
	{
		if(*arg != '%')
			continue;
		urlEncode(getContent(arg), content, sizeof(content));
		snprintf(args + alen, sizeof(args) - alen, " %s=%s", ++arg, content);
		alen = strlen(args);
	}

        if(submit)
        {
                snprintf(buffer, 255, "%s", submit);
                submit = strtok_r(buffer, ",", &tok);
	}

	while(submit)
        {
                sym = getEntry(submit, 0);
                submit = strtok_r(NULL, ",", &tok);
                if(!sym)
                        continue;

		urlEncode(sym->data, content, sizeof(content));
		snprintf(args + alen, sizeof(args) - alen, " %s=%s", sym->id, content);
		alen = strlen(args);
	}

	if(var || notify)
	{
		if(!exp)
			exp = 1000;

		data.sleep.save = var;
		data.sleep.wakeup = exp;
	        data.sleep.loops = 1;
	        data.sleep.rings = 0;
        	data.sleep.save = NULL;
        	trunkStep(TRUNK_STEP_SLEEP);
		rtn = false;
	}
	else
		advance();

	if(start)
	{
		fifo.command(args);
		return rtn;
	}

	argv[argc++] = strtok_r(args, " ", &tok);
	while(argc < 31)
		argv[argc++] = strtok_r(NULL, " ", &tok);	

	argv[argc] = NULL;

	if(!ports && !span)
	{
		request(grp, argv, expires, NULL, getSymbol(SYM_GID));
		return rtn;
	}

	if(offset > last)
		last = offset;

	while(offset <= last)
	{
		trunk = driver->getTrunkPort(offset++);
		if(!trunk)
			continue;

		if(span)
			if(trunk->span != span)
				continue;

		event.id = TRUNK_START_SCRIPT;
		event.parm.argv = argv;		
		if(trunk->postEvent(&event))
			break;
	}

	if(offset > last)
	{
		event.id = TRUNK_CHILD_FAIL;
		postEvent(&event);
	}

	return rtn;
}	

bool Trunk::scrOptions(void)
{
	const char *keys;
	const char *err = NULL;
	char voice[256];

	keys = getKeyword("dtmf");
	if(keys)
	{
		if(!stricmp(keys, "on"))
			flags.digits = DTMF_MODE_ON;
		else if(!stricmp(keys, "off"))
			flags.digits = DTMF_MODE_OFF;
		else if(!stricmp(keys, "script"))
			flags.digits = DTMF_MODE_SCRIPT;
		else if(!stricmp(keys, "line"))
			flags.digits = DTMF_MODE_LINE;
		else
			err = "options-dtmf-invalid";
	}
        keys = getKeyword("logging");
        if(keys)
        {
                if(!stricmp(keys, "notice"))
                        slog.level(Slog::levelNotice);
                else if(!stricmp(keys, "debug"))
                        slog.level(Slog::levelDebug);
                else if(!stricmp(keys, "info"))
                        slog.level(Slog::levelInfo);
                else if(!strnicmp(keys, "err", 3))
                        slog.level(Slog::levelError);
                else if(!strnicmp(keys, "warn", 4))
                        slog.level(Slog::levelWarning);
		else
			err = "options-logging-invalid";
        }
	keys = getKeyword("language");
	if(keys)
	{
		if(getTranslator(keys) != NULL)
			setSymbol(SYM_LANGUAGE, keys);
		else
			err = "options-language-invalid";
	}
	keys = getKeyword("voice");
	if(keys)
	{
		snprintf(voice, sizeof(voice), "%s/%s", keypaths.getLast("prompts"), keys);
		if(isDir(voice))
			setSymbol(SYM_VOICE, keys);
		else
			err = "options-voice-invalid";
	}
	if(err)
		error(err);
	else
		advance();
	return true;
}

bool Trunk::scrCopy(void)
{
	char *cc;
	tgicmd_t cmd;
	const char *prefix = getPrefixPath();
	const char *src = getValue(NULL);
	const char *dest = getValue(NULL);
	const char *mem = getMember();
	char buf1[65], buf2[65];

	if(!mem)
		mem = "copy";

	if(!stricmp(mem, "append"))
		cc = "-append";
	else
		cc = "-copy";

	if(!src || !dest)
	{
		error("copy-files-missing");
		return true;
	}

	if(*src == '-' || *dest == '-')
	{
		error("copy-files-invalid");
		return true;
	}

	src = urlEncode(src, buf1, 64);
	dest = urlEncode(dest, buf2, 64);

	if(prefix)
		snprintf(cmd.cmd, sizeof(cmd.cmd), "%s %s/%s %s/%s",
			cc, prefix, src, prefix, dest);
	else
		snprintf(cmd.cmd, sizeof(cmd.cmd), "cc %s %s",
			cc, src, dest);
	
	data.sleep.wakeup = getTimeout("maxTime");
	if(!data.sleep.wakeup)
		data.sleep.wakeup = 30000;

	data.sleep.rings = 0;
	data.sleep.loops = 1;
	data.sleep.save = NULL;

	cmd.port = id;
	cmd.mode = TGI_EXEC_NORMAL;
	::write(tgipipe[1], &cmd, sizeof(cmd));
	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

bool Trunk::scrLibexec(void)
{
	tgicmd_t cmd;
	int argc = 0;
	char *user = NULL;
	char *gain = getKeyword("gain");
	Line *line = getScript();
	Name *scr = getObject();
	char query[sizeof(cmd.cmd) - 160];
#ifdef	HAVE_SSTREAM
	ostringstream str;
	str.str() = "";
#else
	strstream str(cmd.cmd, sizeof(cmd.cmd));
#endif
	unsigned qlen = 0;
	char *qc, *tag, *opt;
	const char *member = getMember();
	char namebuf[64];

	if(!member)
		member="tgi";

	if(!strnicmp(member, "one", 3))
	{
		if(!getOnce())
		{
			advance();
			return true;
		}
	}

	int cnt = 0;
	int args = 0;

	if(!stricmp(member, "play"))
		data.sleep.wakeup =  getSecTimeout(getValue(getSymbol(SYM_PLAYWAIT)));
	else
		data.sleep.wakeup = getTimeout("maxTime");
	data.sleep.rings = 0;
	data.sleep.loops = 1;
	data.sleep.save = NULL;

	cmd.port = id;
	cmd.mode = TGI_EXEC_NORMAL;
	cmd.cmd[0] = 0;
	query[0] = 0;

	opt = getValue("--");
	if(!strnicmp(opt, "~/", 2))
	{
		snprintf(namebuf, sizeof(namebuf), "%s", scr->name);
		qc = strchr(namebuf, ':');
		if(qc)
			*qc = 0;
		if(namebuf[0] == '~')
			user = namebuf + 1;
		else if(namebuf[0] == '#')
		{
			user = getSymbol(SYM_HOME);
			if(!*user)
				user = NULL;
		}
		opt += 2;
	}

	str << opt;
	while(NULL != (qc = getOption(NULL)) && qlen < sizeof(query))
	{
		if(*qc != '%')
			continue;

		if(!strnicmp(qc, "%lib.", 5))
			tag = qc + 5;
		else
			tag = qc + 1;

		if(qlen)
			query[qlen++] = *keyserver.getToken();

		snprintf(query + qlen, sizeof(query) - qlen, "%s=%s", tag, getContent(qc));
		qlen = strlen(query);
	}

	while(argc < line->argc && qlen < sizeof(query))
	{
		opt = line->args[argc++];
		if(*opt != '=')
			continue;

		tag = opt + 1;
		opt = line->args[argc++];

		if(qlen)
			query[qlen++] = *keyserver.getToken();

		snprintf(query + qlen, sizeof(query) - qlen, "%s=%s", tag, getContent(opt));
		qlen = strlen(query);
	}

	if(user)
		str << " user=" << user;

	str << " query=" << query;

	if(dtmf.bin.data)
		str << " digits=" << dtmf.bin.data;

	qc = getSymbol(SYM_CALLER);
	if(qc)
		str << " clid=" << qc;

	qc = getSymbol(SYM_DIALED);
	if(qc)
		str << " dnid=" << qc;

	if(!data.sleep.wakeup)
		cmd.mode = TGI_EXEC_DETACH;

	if(!stricmp(member, "play"))
	{
		data.play.voice = NULL;
		data.play.timeout = data.sleep.wakeup;
		data.play.repeat = 0;
		data.play.name = data.play.list;
		data.play.term = 0;
		data.play.mode = PLAY_MODE_TEMP;
		data.play.limit = data.play.offset = 0;
		data.play.volume = atoi(getSymbol(SYM_VOLUME));
		data.play.extension = NULL;

		if(gain)
			data.play.gain = (float)strtod(gain, NULL);
		else
			data.play.gain = 0.0;

		sprintf(data.play.list, "temp/.tmp.%d.%s",
			id, getSymbol(SYM_EXTENSION));
		str << " audio=" << data.play.name;
	}

#ifdef	HAVE_SSTREAM
	snprintf(cmd.cmd, sizeof(cmd.cmd), "%s", str.str().c_str());
#else
	str << ends;
#endif
	::write(tgipipe[1], &cmd, sizeof(cmd));

	if(!stricmp(member, "play"))
	{
		trunkStep(TRUNK_STEP_PLAYWAIT);
		return false;
	}

	if(!data.sleep.wakeup)
	{
		advance();
		return true;
	}

	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

bool Trunk::scrTone(void)
{
	const char *cp;
	
	cp = getValue(NULL);
	if(cp)
	{
		data.tone.tone = getphTone(cp);
		if(!data.tone.tone)
		{
			error("no-tone");
			return true;
		}
	}
	else
	{
		data.tone.tone = NULL;
		data.tone.freq1 = data.tone.freq2 = 0;
		data.tone.ampl1 = data.tone.ampl2 = 0;
		cp = getKeyword("frequency");
		if(cp)
		{
			data.tone.freq1 = atoi(cp);
			cp = getKeyword("amplitude");
			if(cp)
				data.tone.ampl1 = atoi(cp);
		}
		else
		{
			cp = getKeyword("freq1");
			if(cp)
				data.tone.freq1 = atoi(cp);

			cp = getKeyword("ampl1");
			if(cp)
				data.tone.ampl1 = atoi(cp);

			cp = getKeyword("freq2");
			if(cp)
				data.tone.freq2 = atoi(cp);

			cp = getKeyword("ampl2");
			if(cp)
				data.tone.ampl2 = atoi(cp);
		}
	}

	cp = getKeyword("timeout");
	if(!cp)
		cp = getValue("1");
	data.tone.wakeup = data.tone.duration = getSecTimeout(cp);

	cp = getKeyword("length");
	if(cp)
		data.tone.duration = getMSTimeout(cp);

	if(data.tone.duration > data.tone.wakeup)
		data.tone.duration = data.tone.wakeup;
		
	cp = getKeyword("count");
	if(!cp)
		cp = getValue("1");
	data.tone.loops = atoi(cp);
	trunkStep(TRUNK_STEP_TONE);
	return false;
}

bool Trunk::scrSleep(void)
{
	timeout_t timer = getTimeout("maxTime");

	if(!timer)
	{
		advance();
		return true;
	}

	data.sleep.wakeup = timer;
	data.sleep.loops = 1;
	data.sleep.rings = 0;
	data.sleep.save = NULL;
	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

#ifdef	CCXX_NAMESPACES
};
#endif
