/*
    GNUbik -- A 3 dimensional magic cube game.
    Copyright (C) 2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
static const char RCSID[]="$Id: glarea-gtk.c,v 1.9 2004/01/21 11:25:58 jmd Exp $";


#include <time.h>
#include <getopt.h>
#include <math.h>
#include "drwBlock.h"
#include "cube.h"
#include "gnubik.h"
#include "version.h"
#include "menus.h"
#include "ui.h"
#include "select.h"
#include "gnubik.xpm"
#include "glarea.h"
#include "widget-set.h"
#include "quarternion.h"
#include "cursors.h"
#include "textures.h"
#include "colour-sel.h"

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include <gtk/gtk.h>
#include <gtk/gtkwidget.h>
#include <gdk/gdkkeysyms.h>

#include <gtk/gtkgl.h>


void display(GtkWidget *glarea);
void set_icon(GtkWidget *shell_widget);


static void 
graphics_area_init(GtkWidget *w, gpointer data );

static gboolean handleRedisplay(gpointer glxarea);


extern int cube_dimension; 
int number_of_blocks;

void projection_init(int jitter) ;


static gboolean resize (GtkWidget *w,
			GdkEventConfigure *event, 
			gpointer data); 


static gboolean cube_orientate_keys(GtkWidget *w, 
	       GdkEventKey *event, 
	       gpointer clientData);

static gboolean z_rotate(GtkWidget *w, GdkEventScroll *event, 
		       gpointer clientData);


static gboolean cube_orientate_mouse(GtkWidget *w, 
	       GdkEventMotion *event, 
	       gpointer clientData);


static gboolean buttons(GtkWidget *w, 
	       GdkEventButton *event, 
	       gpointer clientData);



static gboolean cube_controls(GtkWidget *w, 
	      GdkEventButton *event, 
	      gpointer clientData);


static void expose(GtkWidget *w,GdkEventExpose *event);


static GtkWidget *glwidget;


void re_initialize_glarea()
{
  graphics_area_init(glwidget,0);
}


/* Resize callback.  */
static gboolean 
resize (GtkWidget *w,GdkEventConfigure *event, 
	gpointer data)
{
  GLint min_dim ;
  gint height = event->height;
  gint width =  event->width;
  GdkGLContext *glcontext = gtk_widget_get_gl_context (w);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (w);

  if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
     return FALSE;

  glXWaitX();
  min_dim = (width < height) ? width : height ;

  /* Ensure that cube is always the same proportions */
  glViewport((width-min_dim) /2, (height-min_dim) /2, min_dim, min_dim);

  return FALSE;
}




GtkWidget *
create_gl_area(GtkWidget *containerWidget, GtkWidget *adjacentWidget)
{

  GtkWidget *glxarea; 
  GdkGLConfig *glconfig;

  const GtkTargetEntry target[2] ={
      { "text/uri-list", 0, RDRAG_FILELIST },
      { "application/x-color", 0, RDRAG_COLOUR },
  };



  int attribs[] = { GDK_GL_RGBA, 
		    GDK_GL_DEPTH_SIZE, 12, 
		    GDK_GL_DOUBLEBUFFER,
		    GDK_GL_RED_SIZE, 1, 
		    GDK_GL_ACCUM_RED_SIZE, 1,
		    GDK_GL_ACCUM_GREEN_SIZE, 1,
		    GDK_GL_ACCUM_BLUE_SIZE, 1,
		    GDK_GL_ACCUM_ALPHA_SIZE, 1,
		    GDK_GL_ATTRIB_LIST_NONE };

  glconfig = gdk_gl_config_new(attribs);
 
  glxarea = gtk_drawing_area_new();

  gtk_widget_set_gl_capability (glxarea,
                                glconfig,
                                0,
                                TRUE,
                                GDK_GL_RGBA_TYPE);


  gtk_drag_dest_set(glxarea,GTK_DEST_DEFAULT_ALL,
		    target,2,GDK_ACTION_COPY);


  g_signal_connect (GTK_OBJECT (glxarea),"drag_data_received",
		    GTK_SIGNAL_FUNC(drag_data_received),
		    (gpointer)-1);


  

  glwidget = glxarea;

  gtk_box_pack_start (GTK_BOX (containerWidget), glxarea, TRUE, TRUE, 0);

  gtk_widget_show (GTK_WIDGET(glxarea));

  return glxarea ;
}


extern float cursorAngle;

static void 
graphics_area_init(GtkWidget *w, gpointer data )
{

  GdkGLContext *glcontext = gtk_widget_get_gl_context (w);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (w);


  if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext)) {
    g_print("Cannot initialise gl drawable\n");
    return;
  }

  gtk_widget_set_size_request(w,300,300);

  lighting_init();

  texInit();

  projection_init(0);

  modelViewInit();

  set_icon(w);

  set_the_colours(w,"gnubik");

}


void
register_gl_callbacks(GtkWidget *glxarea)
{

  g_signal_connect (GTK_OBJECT(glxarea),"realize",
		      GTK_SIGNAL_FUNC(graphics_area_init),0);

  g_signal_connect (GTK_OBJECT(glxarea),"expose_event", 
		      GTK_SIGNAL_FUNC(expose),0);

  g_signal_connect (GTK_OBJECT(glxarea),"configure_event",
		      GTK_SIGNAL_FUNC(resize),0);



  gtk_widget_add_events(GTK_WIDGET(glxarea),
			GDK_KEY_PRESS_MASK | GDK_BUTTON_PRESS_MASK 
			| GDK_BUTTON_RELEASE_MASK
			/* | GDK_BUTTON_MOTION_MASK */
			| GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
			| GDK_VISIBILITY_NOTIFY_MASK
			| GDK_POINTER_MOTION_MASK );


  GTK_WIDGET_SET_FLAGS(glxarea,GTK_CAN_FOCUS);

  gtk_window_set_focus(GTK_WINDOW(gtk_widget_get_toplevel(glxarea)),glxarea);

  g_signal_connect (GTK_OBJECT(glxarea), "key_press_event", 
		      GTK_SIGNAL_FUNC(cube_orientate_keys),0);



  g_signal_connect (GTK_OBJECT(glxarea),"motion_notify_event",
		      GTK_SIGNAL_FUNC(cube_orientate_mouse),0);


  g_signal_connect (GTK_OBJECT(glxarea),"scroll_event",
		      GTK_SIGNAL_FUNC(z_rotate),0);


  g_signal_connect (GTK_OBJECT(glxarea),"button_press_event",
		      GTK_SIGNAL_FUNC(buttons),0);

  g_signal_connect (GTK_OBJECT(glxarea),"button_release_event",
		      GTK_SIGNAL_FUNC(buttons),0);


  g_signal_connect (GTK_OBJECT(glxarea), "button_press_event", 
		      GTK_SIGNAL_FUNC(cube_controls),0);





}


static gboolean redisplayPending = False ;

static guint idle_id;

void
postRedisplay()
{

  if ( !redisplayPending ) {
    idle_id = gtk_idle_add(handleRedisplay,glwidget);

    redisplayPending = 1;
  }

}




/* Set the icon for the program */
void 
set_icon(GtkWidget *shell_widget)
{

  GdkPixmap *pixmap ;

  GdkWindow *window  ; 

  GdkBitmap *mask;

  window = gtk_widget_get_parent_window(shell_widget);

  pixmap =   gdk_pixmap_create_from_xpm_d(window,&mask,0,(gchar **)gnubik_xpm);
  
  gdk_window_set_icon(window,0,pixmap,mask);

}


/* Error string display.  This is always called by a macro
wrapper, to set the file and line_no opts parameters */
void
error_check(const char *file, int line_no, const char *string)
{
  GLenum err_state;
  if ( GL_NO_ERROR != (err_state = glGetError())  )
    g_print("%s:%d %s:  %s\n",file,line_no ,string, 
	   gluErrorString(err_state) );
}




/* Expose callback.  Just redraw the scene */
static void
expose(GtkWidget *w,GdkEventExpose *event)
{

  postRedisplay();

}




static gboolean button_down = FALSE;

/* Rotate the cube about the z axis (relative to the viewer ) */
gboolean 
z_rotate(GtkWidget *w, 
	GdkEventScroll *event, 
	gpointer clientData)
{

  rotate_cube(2,!event->direction);

  return FALSE;

}

/* Record the state of button 1 */
gboolean 
buttons(GtkWidget *w, 
	GdkEventButton *event, 
	gpointer clientData)
{

  /* In GTK1-2, buttons 4 and 5 mean mouse wheel scrolling */
  if ( event->button == 4 ) 
    rotate_cube(2,1);

  if ( event->button == 5 ) 
    rotate_cube(2,0);



  if ( event->button != 1) 
    return FALSE;


  if ( event->type == GDK_BUTTON_PRESS) { 
    button_down = TRUE;
    disableSelection();
  }
  else if ( event->type == GDK_BUTTON_RELEASE) {
    enableSelection();
    button_down = FALSE;
  }


  return FALSE;
}


gboolean 
cube_orientate_mouse(GtkWidget *w, 
	       GdkEventMotion *event, 
	       gpointer clientData)
{

  static gdouble last_mouse_x=-1; 
  static gdouble last_mouse_y=-1;
  
  gint xmotion = 0;
  gint ymotion = 0;



  if ( ! button_down) 
    return FALSE;

  if ( itemIsSelected() ) 
	return FALSE;


  if ( last_mouse_x >= 0 )
    xmotion = event->x - last_mouse_x ;

  if ( last_mouse_y >= 0 )
    ymotion = event->y - last_mouse_y ;
    

  last_mouse_x = event->x ; 
  last_mouse_y = event->y;

  if ( ymotion > 0 ) 
    rotate_cube(0,1);
  if ( ymotion < 0 ) 
    rotate_cube(0,0);

  if ( xmotion > 0 ) 
    rotate_cube(1,1);
  if ( xmotion < 0 ) 
    rotate_cube(1,0);


  return FALSE;
}

gboolean
cube_orientate_keys(GtkWidget *w, GdkEventKey *event, gpointer clientData)
{

  int shifted=0;

  if ( event->state & GDK_SHIFT_MASK)
    shifted=1;

  arrows(event->keyval, shifted);

  return TRUE;
}





/* Input callback.  Despatch input events to approprite handlers */
gboolean
cube_controls(GtkWidget *w, GdkEventButton *event, gpointer clientData)
{

  if ( event->type != GDK_BUTTON_PRESS)
    return TRUE;


  mouse(event->button);

  return TRUE;
}





/* Pair of mutually co-operating funcs to handle redisplays,
avoiding unnecessary overhead.  For example when the window
is covered by another one */


static gboolean
handleRedisplay(gpointer glxarea)
{

  display((GtkWidget*)glxarea);
  redisplayPending = 0;

  gtk_idle_remove(idle_id);

  return True ; 
}


void
set_mouse_cursor(GtkWidget *glxarea)
{
  unsigned char *mask_bits;
  unsigned char *data_bits;
  int hot_x, hot_y;
  int width, height;


  GdkCursor *cursor;
  GdkPixmap *source, *mask;
  GdkColor fg = { 0, 65535, 65535, 65535 }; /* White. */
  GdkColor bg = { 0, 0, 0, 0 }; /* Black. */

  if ( itemIsSelected() ) { 
    get_cursor(cursorAngle, &data_bits, &mask_bits, &height, &width,
	       &hot_x, &hot_y);


    source = gdk_bitmap_create_from_data (NULL, (const gchar *) data_bits,
					  width,height);
    mask = gdk_bitmap_create_from_data (NULL, (const gchar *) mask_bits,
					width, height);
					
    cursor = gdk_cursor_new_from_pixmap (source, mask, &fg, &bg, hot_x, hot_y);
    g_object_unref (source);
    g_object_unref (mask);
  }
  else { 
    cursor = gdk_cursor_new(GDK_CROSSHAIR);
  }

  gdk_window_set_cursor (glxarea->window, cursor);
  gdk_cursor_unref(cursor);

}


/* Reset the bit planes, and render the scene */
void
display(GtkWidget *glxarea)
{

  int jitter;

  GdkGLContext *glcontext = gtk_widget_get_gl_context (glxarea);
  GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (glxarea);



  if (!gdk_gl_drawable_make_current (gldrawable, glcontext)) {
    g_print("Cannot set gl drawable current\n");
    return;
  }
  ERR_CHECK("Error in display");

  projection_init(0);
  modelViewInit();



  set_mouse_cursor(glxarea);


  glClear(GL_ACCUM_BUFFER_BIT);
  ERR_CHECK("Error in display");

  for( jitter = 0 ; jitter < 8 ; ++jitter ) { 
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    projection_init(jitter);
    modelViewInit();
  
    ERR_CHECK("Error in display");

    drawCube();

    ERR_CHECK("Error in display");

    glAccum(GL_ACCUM, 1.0/8.0);

  }

  glAccum(GL_RETURN,1.0);

  gdk_gl_drawable_swap_buffers (gldrawable);


}


