/*
 *  Copyright (C) 2004-2021 Savoir-faire Linux Inc.
 *
 *  Author: Alexandre Savard <alexandre.savard@savoirfairelinux.com>
 *  Author: Adrien Béraud <adrien.beraud@savoirfairelinux.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

%header %{
#include "dring/dring.h"
#include "dring/configurationmanager_interface.h"

class ConfigurationCallback {
public:
    virtual ~ConfigurationCallback(){}
    virtual void volumeChanged(const std::string& device, int value){}
    virtual void accountsChanged(void){}
    virtual void historyChanged(void){}
    virtual void stunStatusFailure(const std::string& account_id){}
    virtual void registrationStateChanged(const std::string& account_id, const std::string& state, int code, const std::string& detail_str){}
    virtual void volatileAccountDetailsChanged(const std::string& account_id, const std::map<std::string, std::string>& details){}
    virtual void incomingAccountMessage(const std::string& /*account_id*/, const std::string& /*from*/, const std::map<std::string, std::string>& /*payload*/){}
    virtual void accountMessageStatusChanged(const std::string& /*account_id*/, uint64_t /*message_id*/, const std::string& /*to*/, int /*state*/){}
    virtual void knownDevicesChanged(const std::string& /*account_id*/, const std::map<std::string, std::string>& /*devices*/){}
    virtual void exportOnRingEnded(const std::string& /*account_id*/, int /*state*/, const std::string& /*pin*/){}

    virtual void incomingTrustRequest(const std::string& /*account_id*/, const std::string& /*from*/, const std::vector<uint8_t>& /*payload*/, time_t received){}
    virtual void contactAdded(const std::string& /*account_id*/, const std::string& /*uri*/, bool confirmed){}
    virtual void contactRemoved(const std::string& /*account_id*/, const std::string& /*uri*/, bool banned){}

    virtual void certificatePinned(const std::string& /*certId*/){}
    virtual void certificatePathPinned(const std::string& /*path*/, const std::vector<std::string>& /*certId*/){}
    virtual void certificateExpired(const std::string& /*certId*/){}
    virtual void certificateStateChanged(const std::string& /*account_id*/, const std::string& /*certId*/, const std::string& /*state*/){}

    virtual void errorAlert(int alert){}

    virtual void nameRegistrationEnded(const std::string& /*account_id*/, int state, const std::string& /*name*/){}
    virtual void registeredNameFound(const std::string& /*account_id*/, int state, const std::string& /*address*/, const std::string& /*name*/){}
    virtual void userSearchEnded(const std::string& /*account_id*/, int state, const std::string& /*query*/, const std::vector<std::map<std::string, std::string>>& /*results*/){}

    virtual void migrationEnded(const std::string& /*accountId*/, const std::string& /*state*/){}
    virtual void deviceRevocationEnded(const std::string& /*accountId*/, const std::string& /*device*/, int /*status*/){}
    virtual void accountProfileReceived(const std::string& /*accountId*/, const std::string& /*names*/, const std::string& /*photo*/){}

    virtual void hardwareDecodingChanged(bool /*state*/){}
    virtual void hardwareEncodingChanged(bool /*state*/){}

    virtual void audioMeter(const std::string& /*id*/, float /*level*/){}
};
%}

%feature("director") ConfigurationCallback;

namespace DRing {

struct Message
{
    std::string from;
    std::map<std::string, std::string> payloads;
    uint64_t received;
};

std::map<std::string, std::string> getAccountDetails(const std::string& accountID);
std::map<std::string, std::string> getVolatileAccountDetails(const std::string& accountID);
void setAccountDetails(const std::string& accountID, const std::map<std::string, std::string>& details);
void setAccountActive(const std::string& accountID, bool active);
std::map<std::string, std::string> getAccountTemplate(const std::string& accountType);
void monitor(bool continuous);
std::string addAccount(const std::map<std::string, std::string>& details);
void removeAccount(const std::string& accountID);
std::vector<std::string> getAccountList();
void sendRegister(const std::string& accountID, bool enable);
void registerAllAccounts(void);
uint64_t sendAccountTextMessage(const std::string& accountID, const std::string& to, const std::map<std::string, std::string>& message);
std::vector<Message> getLastMessages(const std::string& accountID, const uint64_t& base_timestamp);
int getMessageStatus(uint64_t id);
int getMessageStatus(const std::string& accountID, uint64_t id);
bool cancelMessage(const std::string& accountID, uint64_t id);
bool setMessageDisplayed(const std::string& accountID, const std::string& contactId, const std::string& messageId, int status);
bool lookupName(const std::string& account, const std::string& nameserver, const std::string& name);
bool lookupAddress(const std::string& account, const std::string& nameserver, const std::string& address);
bool registerName(const std::string& account, const std::string& password, const std::string& name);
bool searchUser(const std::string& account, const std::string& query);

std::vector<unsigned> getCodecList();
std::vector<std::string> getSupportedTlsMethod();
std::vector<std::string> getSupportedCiphers(const std::string& accountID);
std::map<std::string, std::string> getCodecDetails(const std::string& accountID, const unsigned& codecId);
bool setCodecDetails(const std::string& accountID, const unsigned& codecId, const std::map<std::string, std::string>& details);
std::vector<unsigned> getActiveCodecList(const std::string& accountID);
bool exportOnRing(const std::string& accountID, const std::string& password);
bool exportToFile(const std::string& accountID, const std::string& destinationPath, const std::string& password);

std::map<std::string, std::string> getKnownRingDevices(const std::string& accountID);
bool revokeDevice(const std::string& accountID, const std::string& password, const std::string& deviceID);

void setActiveCodecList(const std::string& accountID, const std::vector<unsigned>& list);

std::vector<std::string> getAudioPluginList();
void setAudioPlugin(const std::string& audioPlugin);
std::vector<std::string> getAudioOutputDeviceList();
void setAudioOutputDevice(int32_t index);
void setAudioInputDevice(int32_t index);
void setAudioRingtoneDevice(int32_t index);
std::vector<std::string> getAudioInputDeviceList();
std::vector<std::string> getCurrentAudioDevicesIndex();
int32_t getAudioInputDeviceIndex(const std::string& name);
int32_t getAudioOutputDeviceIndex(const std::string& name);
std::string getCurrentAudioOutputPlugin();
bool getNoiseSuppressState();
void setNoiseSuppressState(bool state);

bool isAgcEnabled();
void setAgcState(bool enabled);

void muteDtmf(bool mute);
bool isDtmfMuted();

bool isCaptureMuted();
void muteCapture(bool mute);
bool isPlaybackMuted();
void mutePlayback(bool mute);
bool isRingtoneMuted();
void muteRingtone(bool mute);

std::string getAudioManager();
bool setAudioManager(const std::string& api);

std::string getRecordPath();
void setRecordPath(const std::string& recPath);
bool getIsAlwaysRecording();
void setIsAlwaysRecording(bool rec);
bool getRecordPreview();
void setRecordPreview(bool rec);
int32_t getRecordQuality();
void setRecordQuality(int32_t rec);

void setHistoryLimit(int32_t days);
int32_t getHistoryLimit();

void setRingingTimeout(int32_t timeout);
int32_t getRingingTimeout();

void setAccountsOrder(const std::string& order);

std::vector<std::map<std::string, std::string> > getCredentials(const std::string& accountID);
void setCredentials(const std::string& accountID, const std::vector<std::map<std::string, std::string> >& details);

std::string getAddrFromInterfaceName(const std::string& interface);

std::vector<std::string> getAllIpInterface();
std::vector<std::string> getAllIpInterfaceByName();

std::map<std::string, std::string> getShortcuts();
void setShortcuts(const std::map<std::string, std::string> &shortcutsMap);

void setVolume(const std::string& device, double value);
double getVolume(const std::string& device);

/*
 * Security
 */
std::map<std::string, std::string> validateCertificatePath(const std::string& accountId,
                                                       const std::string& certificate,
                                                       const std::string& privateKey,
                                                       const std::string& privateKeyPassword,
                                                       const std::string& caList);

std::map<std::string, std::string> validateCertificate(const std::string& accountId, const std::string& certificate);

std::map<std::string, std::string> getCertificateDetails(const std::string& certificate);
std::map<std::string, std::string> getCertificateDetailsPath(const std::string& certificate, const std::string& privateKey, const std::string& privateKeyPass);

std::vector<std::string> getPinnedCertificates();

std::vector<std::string> pinCertificate(const std::vector<uint8_t>& certificate, bool local);
bool unpinCertificate(const std::string& certId);

void pinCertificatePath(const std::string& path);
unsigned unpinCertificatePath(const std::string& path);

bool pinRemoteCertificate(const std::string& accountId, const std::string& certId);
bool setCertificateStatus(const std::string& account, const std::string& certId, const std::string& status);
std::vector<std::string> getCertificatesByStatus(const std::string& account, const std::string& status);

/* contact requests */
std::vector<std::map<std::string, std::string>> getTrustRequests(const std::string& accountId);
bool acceptTrustRequest(const std::string& accountId, const std::string& from);
bool discardTrustRequest(const std::string& accountId, const std::string& from);
void sendTrustRequest(const std::string& accountId, const std::string& to, const std::vector<uint8_t>& payload);

/* Contacts */

void addContact(const std::string& accountId, const std::string& uri);
void removeContact(const std::string& accountId, const std::string& uri, const bool& ban);
std::vector<std::map<std::string, std::string>> getContacts(const std::string& accountId);
std::map<std::string, std::string> getContactDetails(const std::string& accountId, const std::string& uri);

void connectivityChanged();

bool isAudioMeterActive(const std::string& id);
void setAudioMeterState(const std::string& id, bool state);

void setDefaultModerator(const std::string& accountID, const std::string& peerURI, bool state);
std::vector<std::string> getDefaultModerators(const std::string& accountID);
void enableLocalModerators(const std::string& accountID, bool isModEnabled);
bool isLocalModeratorsEnabled(const std::string& accountID);
}

class ConfigurationCallback {
public:
    virtual ~ConfigurationCallback(){}
    virtual void volumeChanged(const std::string& device, int value){}
    virtual void accountsChanged(void){}
    virtual void historyChanged(void){}
    virtual void stunStatusFailure(const std::string& account_id){}
    virtual void registrationStateChanged(const std::string& account_id, const std::string& state, int code, const std::string& detail_str){}
    virtual void volatileAccountDetailsChanged(const std::string& account_id, const std::map<std::string, std::string>& details){}
    virtual void incomingAccountMessage(const std::string& /*account_id*/, const std::string& /*from*/, const std::map<std::string, std::string>& /*payload*/){}
    virtual void accountMessageStatusChanged(const std::string& /*account_id*/, uint64_t /*message_id*/, const std::string& /*to*/, int /*state*/){}
    virtual void knownDevicesChanged(const std::string& /*account_id*/, const std::map<std::string, std::string>& /*devices*/){}
    virtual void exportOnRingEnded(const std::string& /*account_id*/, int /*state*/, const std::string& /*pin*/){}

    virtual void incomingTrustRequest(const std::string& /*account_id*/, const std::string& /*from*/, const std::vector<uint8_t>& /*payload*/, time_t received){}
    virtual void contactAdded(const std::string& /*account_id*/, const std::string& /*uri*/, bool confirmed){}
    virtual void contactRemoved(const std::string& /*account_id*/, const std::string& /*uri*/, bool banned){}

    virtual void certificatePinned(const std::string& /*certId*/){}
    virtual void certificatePathPinned(const std::string& /*path*/, const std::vector<std::string>& /*certId*/){}
    virtual void certificateExpired(const std::string& /*certId*/){}
    virtual void certificateStateChanged(const std::string& /*account_id*/, const std::string& /*certId*/, const std::string& /*state*/){}

    virtual void errorAlert(int alert){}

    virtual void nameRegistrationEnded(const std::string& /*account_id*/, int state, const std::string& /*name*/){}
    virtual void registeredNameFound(const std::string& /*account_id*/, int state, const std::string& /*address*/, const std::string& /*name*/){}
    virtual void userSearchEnded(const std::string& /*account_id*/, int state, const std::string& /*query*/, const std::vector<std::map<std::string, std::string>>& /*results*/){}

    virtual void migrationEnded(const std::string& /*accountId*/, const std::string& /*state*/){}
    virtual void deviceRevocationEnded(const std::string& /*accountId*/, const std::string& /*device*/, int /*status*/){}
    virtual void accountProfileReceived(const std::string& /*accountId*/, const std::string& /*names*/, const std::string& /*photo*/){}

    virtual void hardwareDecodingChanged(bool /*state*/){}
    virtual void hardwareEncodingChanged(bool /*state*/){}

    virtual void audioMeter(const std::string& /*id*/, float /*level*/){}
    //virtual void messageSend(const std::string& /*message*/){}
};
