/* Mapping tables for CNS 11643, plane 1 handling.
   Copyright (C) 1998-2017 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Ulrich Drepper <drepper@cygnus.com>, 1998.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

#include <stdint.h>

/* To generate a Unicode 3.2 CNS11643.TXT, take
   http://www.unicode.org/Public/Mappings/EASTASIA/OTHER/CNS11643.TXT
   and add the following lines (see Unicode 3.2 UNIHAN.TXT):
   0x12728	0x4EA0	#	<CJK Ideograph>
   0x1272F	0x51AB	#	<CJK Ideograph>
   0x12734	0x52F9	#	<CJK Ideograph>
 */

/* The following table contains quite some big hole but I'm not sure whether
   it is a good idea to eliminate them.  The algorithm gets slower (has to
   use loops).  [And I'm too lazy in the moment.]

   The table can be generated using

   egrep '^0x1' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x12121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint16_t __cns11643l1_to_ucs4_tab[] =
{
  [0x0000] = 0x3000, [0x0001] = 0xff0c, [0x0002] = 0x3001, [0x0003] = 0x3002,
  [0x0004] = 0xff0e, [0x0005] = 0x30fb, [0x0006] = 0xff1b, [0x0007] = 0xff1a,
  [0x0008] = 0xff1f, [0x0009] = 0xff01, [0x000a] = 0xfe30, [0x000b] = 0x2026,
  [0x000c] = 0x2025, [0x000d] = 0xfe50, [0x000e] = 0xfe51, [0x000f] = 0xfe52,
  [0x0010] = 0x00b7, [0x0011] = 0xfe54, [0x0012] = 0xfe55, [0x0013] = 0xfe56,
  [0x0014] = 0xfe57, [0x0015] = 0xfe31, [0x0016] = 0x2014, [0x0017] = 0xfe32,
  [0x0018] = 0x2013, [0x001d] = 0xff08, [0x001e] = 0xff09, [0x001f] = 0xfe35,
  [0x0020] = 0xfe36, [0x0021] = 0xff5b, [0x0022] = 0xff5d, [0x0023] = 0xfe37,
  [0x0024] = 0xfe38, [0x0025] = 0x3014, [0x0026] = 0x3015, [0x0027] = 0xfe39,
  [0x0028] = 0xfe3a, [0x0029] = 0x3010, [0x002a] = 0x3011, [0x002b] = 0xfe3b,
  [0x002c] = 0xfe3c, [0x002d] = 0x300a, [0x002e] = 0x300b, [0x002f] = 0xfe3d,
  [0x0030] = 0xfe3e, [0x0031] = 0x3008, [0x0032] = 0x3009, [0x0033] = 0xfe3f,
  [0x0034] = 0xfe40, [0x0035] = 0x300c, [0x0036] = 0x300d, [0x0037] = 0xfe41,
  [0x0038] = 0xfe42, [0x0039] = 0x300e, [0x003a] = 0x300f, [0x003b] = 0xfe43,
  [0x003c] = 0xfe44, [0x003d] = 0xfe59, [0x003e] = 0xfe5a, [0x003f] = 0xfe5b,
  [0x0040] = 0xfe5c, [0x0041] = 0xfe5d, [0x0042] = 0xfe5e, [0x0043] = 0x2018,
  [0x0044] = 0x2019, [0x0045] = 0x201c, [0x0046] = 0x201d, [0x0047] = 0x301d,
  [0x0048] = 0x301e, [0x0049] = 0x2032, [0x004a] = 0x2035, [0x004b] = 0xff03,
  [0x004c] = 0xff06, [0x004d] = 0xff0a, [0x004e] = 0x203b, [0x004f] = 0x00a7,
  [0x0050] = 0x3003, [0x0051] = 0x25cb, [0x0052] = 0x25cf, [0x0053] = 0x25b3,
  [0x0054] = 0x25b2, [0x0055] = 0x25ce, [0x0056] = 0x2606, [0x0057] = 0x2605,
  [0x0058] = 0x25c7, [0x0059] = 0x25c6, [0x005a] = 0x25a1, [0x005b] = 0x25a0,
  [0x005c] = 0x25bd, [0x005d] = 0x25bc, [0x005e] = 0x32a3, [0x005f] = 0x2105,
  [0x0060] = 0x203e, [0x0062] = 0xff3f, [0x0064] = 0xfe49, [0x0065] = 0xfe4a,
  [0x0066] = 0xfe4d, [0x0067] = 0xfe4e, [0x0068] = 0xfe4b, [0x0069] = 0xfe4c,
  [0x006a] = 0xfe5f, [0x006b] = 0xfe60, [0x006c] = 0xfe61, [0x006d] = 0xff0b,
  [0x006e] = 0xff0d, [0x006f] = 0x00d7, [0x0070] = 0x00f7, [0x0071] = 0x00b1,
  [0x0072] = 0x221a, [0x0073] = 0xff1c, [0x0074] = 0xff1e, [0x0075] = 0xff1d,
  [0x0076] = 0x2266, [0x0077] = 0x2267, [0x0078] = 0x2260, [0x0079] = 0x221e,
  [0x007a] = 0x2252, [0x007b] = 0x2261, [0x007c] = 0xfe62, [0x007d] = 0xfe63,
  [0x007e] = 0xfe64, [0x007f] = 0xfe66, [0x0080] = 0xfe65, [0x0081] = 0x223c,
  [0x0082] = 0x2229, [0x0083] = 0x222a, [0x0084] = 0x22a5, [0x0085] = 0x2220,
  [0x0086] = 0x221f, [0x0087] = 0x22bf, [0x0088] = 0x33d2, [0x0089] = 0x33d1,
  [0x008a] = 0x222b, [0x008b] = 0x222e, [0x008c] = 0x2235, [0x008d] = 0x2234,
  [0x008e] = 0x2640, [0x008f] = 0x2642, [0x0090] = 0x2641, [0x0091] = 0x2609,
  [0x0092] = 0x2191, [0x0093] = 0x2193, [0x0094] = 0x2192, [0x0095] = 0x2190,
  [0x0096] = 0x2196, [0x0097] = 0x2197, [0x0098] = 0x2199, [0x0099] = 0x2198,
  [0x009a] = 0x2016, [0x009b] = 0xff5c, [0x009c] = 0xff0f, [0x009d] = 0xff3c,
  [0x009e] = 0x2215, [0x009f] = 0xfe68, [0x00a0] = 0xff04, [0x00a1] = 0xffe5,
  [0x00a2] = 0x3012, [0x00a3] = 0xffe0, [0x00a4] = 0xffe1, [0x00a5] = 0xff05,
  [0x00a6] = 0xff20, [0x00a7] = 0x2103, [0x00a8] = 0x2109, [0x00a9] = 0xfe69,
  [0x00aa] = 0xfe6a, [0x00ab] = 0xfe6b, [0x00ac] = 0x33d5, [0x00ad] = 0x339c,
  [0x00ae] = 0x339d, [0x00af] = 0x339e, [0x00b0] = 0x33ce, [0x00b1] = 0x33a1,
  [0x00b2] = 0x338e, [0x00b3] = 0x338f, [0x00b4] = 0x33c4, [0x00b5] = 0x00b0,
  [0x00b6] = 0x5159, [0x00b7] = 0x515b, [0x00b8] = 0x515e, [0x00b9] = 0x515d,
  [0x00ba] = 0x5161, [0x00bb] = 0x5163, [0x00bc] = 0x55e7, [0x00bd] = 0x74e9,
  [0x00be] = 0x7cce, [0x00bf] = 0x2581, [0x00c0] = 0x2582, [0x00c1] = 0x2583,
  [0x00c2] = 0x2584, [0x00c3] = 0x2585, [0x00c4] = 0x2586, [0x00c5] = 0x2587,
  [0x00c6] = 0x2588, [0x00c7] = 0x258f, [0x00c8] = 0x258e, [0x00c9] = 0x258d,
  [0x00ca] = 0x258c, [0x00cb] = 0x258b, [0x00cc] = 0x258a, [0x00cd] = 0x2589,
  [0x00ce] = 0x253c, [0x00cf] = 0x2534, [0x00d0] = 0x252c, [0x00d1] = 0x2524,
  [0x00d2] = 0x251c, [0x00d3] = 0x2594, [0x00d4] = 0x2500, [0x00d5] = 0x2502,
  [0x00d6] = 0x2595, [0x00d7] = 0x250c, [0x00d8] = 0x2510, [0x00d9] = 0x2514,
  [0x00da] = 0x2518, [0x00db] = 0x256d, [0x00dc] = 0x256e, [0x00dd] = 0x2570,
  [0x00de] = 0x256f, [0x00df] = 0x2550, [0x00e0] = 0x255e, [0x00e1] = 0x256a,
  [0x00e2] = 0x2561, [0x00e3] = 0x25e2, [0x00e4] = 0x25e3, [0x00e5] = 0x25e5,
  [0x00e6] = 0x25e4, [0x00e7] = 0x2571, [0x00e8] = 0x2572, [0x00e9] = 0x2573,
  [0x011a] = 0xff10, [0x011b] = 0xff11, [0x011c] = 0xff12, [0x011d] = 0xff13,
  [0x011e] = 0xff14, [0x011f] = 0xff15, [0x0120] = 0xff16, [0x0121] = 0xff17,
  [0x0122] = 0xff18, [0x0123] = 0xff19, [0x0124] = 0x2160, [0x0125] = 0x2161,
  [0x0126] = 0x2162, [0x0127] = 0x2163, [0x0128] = 0x2164, [0x0129] = 0x2165,
  [0x012a] = 0x2166, [0x012b] = 0x2167, [0x012c] = 0x2168, [0x012d] = 0x2169,
  [0x012e] = 0x3021, [0x012f] = 0x3022, [0x0130] = 0x3023, [0x0131] = 0x3024,
  [0x0132] = 0x3025, [0x0133] = 0x3026, [0x0134] = 0x3027, [0x0135] = 0x3028,
  [0x0136] = 0x3029, [0x0138] = 0x5344, [0x013a] = 0xff21, [0x013b] = 0xff22,
  [0x013c] = 0xff23, [0x013d] = 0xff24, [0x013e] = 0xff25, [0x013f] = 0xff26,
  [0x0140] = 0xff27, [0x0141] = 0xff28, [0x0142] = 0xff29, [0x0143] = 0xff2a,
  [0x0144] = 0xff2b, [0x0145] = 0xff2c, [0x0146] = 0xff2d, [0x0147] = 0xff2e,
  [0x0148] = 0xff2f, [0x0149] = 0xff30, [0x014a] = 0xff31, [0x014b] = 0xff32,
  [0x014c] = 0xff33, [0x014d] = 0xff34, [0x014e] = 0xff35, [0x014f] = 0xff36,
  [0x0150] = 0xff37, [0x0151] = 0xff38, [0x0152] = 0xff39, [0x0153] = 0xff3a,
  [0x0154] = 0xff41, [0x0155] = 0xff42, [0x0156] = 0xff43, [0x0157] = 0xff44,
  [0x0158] = 0xff45, [0x0159] = 0xff46, [0x015a] = 0xff47, [0x015b] = 0xff48,
  [0x015c] = 0xff49, [0x015d] = 0xff4a, [0x015e] = 0xff4b, [0x015f] = 0xff4c,
  [0x0160] = 0xff4d, [0x0161] = 0xff4e, [0x0162] = 0xff4f, [0x0163] = 0xff50,
  [0x0164] = 0xff51, [0x0165] = 0xff52, [0x0166] = 0xff53, [0x0167] = 0xff54,
  [0x0168] = 0xff55, [0x0169] = 0xff56, [0x016a] = 0xff57, [0x016b] = 0xff58,
  [0x016c] = 0xff59, [0x016d] = 0xff5a, [0x016e] = 0x0391, [0x016f] = 0x0392,
  [0x0170] = 0x0393, [0x0171] = 0x0394, [0x0172] = 0x0395, [0x0173] = 0x0396,
  [0x0174] = 0x0397, [0x0175] = 0x0398, [0x0176] = 0x0399, [0x0177] = 0x039a,
  [0x0178] = 0x039b, [0x0179] = 0x039c, [0x017a] = 0x039d, [0x017b] = 0x039e,
  [0x017c] = 0x039f, [0x017d] = 0x03a0, [0x017e] = 0x03a1, [0x017f] = 0x03a3,
  [0x0180] = 0x03a4, [0x0181] = 0x03a5, [0x0182] = 0x03a6, [0x0183] = 0x03a7,
  [0x0184] = 0x03a8, [0x0185] = 0x03a9, [0x0186] = 0x03b1, [0x0187] = 0x03b2,
  [0x0188] = 0x03b3, [0x0189] = 0x03b4, [0x018a] = 0x03b5, [0x018b] = 0x03b6,
  [0x018c] = 0x03b7, [0x018d] = 0x03b8, [0x018e] = 0x03b9, [0x018f] = 0x03ba,
  [0x0190] = 0x03bb, [0x0191] = 0x03bc, [0x0192] = 0x03bd, [0x0193] = 0x03be,
  [0x0194] = 0x03bf, [0x0195] = 0x03c0, [0x0196] = 0x03c1, [0x0197] = 0x03c3,
  [0x0198] = 0x03c4, [0x0199] = 0x03c5, [0x019a] = 0x03c6, [0x019b] = 0x03c7,
  [0x019c] = 0x03c8, [0x019d] = 0x03c9, [0x019e] = 0x3105, [0x019f] = 0x3106,
  [0x01a0] = 0x3107, [0x01a1] = 0x3108, [0x01a2] = 0x3109, [0x01a3] = 0x310a,
  [0x01a4] = 0x310b, [0x01a5] = 0x310c, [0x01a6] = 0x310d, [0x01a7] = 0x310e,
  [0x01a8] = 0x310f, [0x01a9] = 0x3110, [0x01aa] = 0x3111, [0x01ab] = 0x3112,
  [0x01ac] = 0x3113, [0x01ad] = 0x3114, [0x01ae] = 0x3115, [0x01af] = 0x3116,
  [0x01b0] = 0x3117, [0x01b1] = 0x3118, [0x01b2] = 0x3119, [0x01b3] = 0x311a,
  [0x01b4] = 0x311b, [0x01b5] = 0x311c, [0x01b6] = 0x311d, [0x01b7] = 0x311e,
  [0x01b8] = 0x311f, [0x01b9] = 0x3120, [0x01ba] = 0x3121, [0x01bb] = 0x3122,
  [0x01bc] = 0x3123, [0x01bd] = 0x3124, [0x01be] = 0x3125, [0x01bf] = 0x3126,
  [0x01c0] = 0x3127, [0x01c1] = 0x3128, [0x01c2] = 0x3129, [0x01c3] = 0x02d9,
  [0x01c4] = 0x02c9, [0x01c5] = 0x02ca, [0x01c6] = 0x02c7, [0x01c7] = 0x02cb,
  [0x01d6] = 0x2460, [0x01d7] = 0x2461, [0x01d8] = 0x2462, [0x01d9] = 0x2463,
  [0x01da] = 0x2464, [0x01db] = 0x2465, [0x01dc] = 0x2466, [0x01dd] = 0x2467,
  [0x01de] = 0x2468, [0x01df] = 0x2469, [0x01e0] = 0x2474, [0x01e1] = 0x2475,
  [0x01e2] = 0x2476, [0x01e3] = 0x2477, [0x01e4] = 0x2478, [0x01e5] = 0x2479,
  [0x01e6] = 0x247a, [0x01e7] = 0x247b, [0x01e8] = 0x247c, [0x01e9] = 0x247d,
  [0x01ea] = 0x2170, [0x01eb] = 0x2171, [0x01ec] = 0x2172, [0x01ed] = 0x2173,
  [0x01ee] = 0x2174, [0x01ef] = 0x2175, [0x01f0] = 0x2176, [0x01f1] = 0x2177,
  [0x01f2] = 0x2178, [0x01f3] = 0x2179, [0x023b] = 0x4ea0, [0x0242] = 0x51ab,
  [0x0247] = 0x52f9, [0x0c1e] = 0x2400, [0x0c1f] = 0x2401, [0x0c20] = 0x2402,
  [0x0c21] = 0x2403, [0x0c22] = 0x2404, [0x0c23] = 0x2405, [0x0c24] = 0x2406,
  [0x0c25] = 0x2407, [0x0c26] = 0x2408, [0x0c27] = 0x2409, [0x0c28] = 0x240a,
  [0x0c29] = 0x240b, [0x0c2a] = 0x240c, [0x0c2b] = 0x240d, [0x0c2c] = 0x240e,
  [0x0c2d] = 0x240f, [0x0c2e] = 0x2410, [0x0c2f] = 0x2411, [0x0c30] = 0x2412,
  [0x0c31] = 0x2413, [0x0c32] = 0x2414, [0x0c33] = 0x2415, [0x0c34] = 0x2416,
  [0x0c35] = 0x2417, [0x0c36] = 0x2418, [0x0c37] = 0x2419, [0x0c38] = 0x241a,
  [0x0c39] = 0x241b, [0x0c3a] = 0x241c, [0x0c3b] = 0x241d, [0x0c3c] = 0x241e,
  [0x0c3d] = 0x241f, [0x0c3e] = 0x2421, [0x0cda] = 0x4e00, [0x0cdb] = 0x4e59,
  [0x0cdc] = 0x4e01, [0x0cdd] = 0x4e03, [0x0cde] = 0x4e43, [0x0cdf] = 0x4e5d,
  [0x0ce0] = 0x4e86, [0x0ce1] = 0x4e8c, [0x0ce2] = 0x4eba, [0x0ce3] = 0x513f,
  [0x0ce4] = 0x5165, [0x0ce5] = 0x516b, [0x0ce6] = 0x51e0, [0x0ce7] = 0x5200,
  [0x0ce8] = 0x5201, [0x0ce9] = 0x529b, [0x0cea] = 0x5315, [0x0ceb] = 0x5341,
  [0x0cec] = 0x535c, [0x0ced] = 0x53c8, [0x0cee] = 0x4e09, [0x0cef] = 0x4e0b,
  [0x0cf0] = 0x4e08, [0x0cf1] = 0x4e0a, [0x0cf2] = 0x4e2b, [0x0cf3] = 0x4e38,
  [0x0cf4] = 0x51e1, [0x0cf5] = 0x4e45, [0x0cf6] = 0x4e48, [0x0cf7] = 0x4e5f,
  [0x0cf8] = 0x4e5e, [0x0cf9] = 0x4e8e, [0x0cfa] = 0x4ea1, [0x0cfb] = 0x5140,
  [0x0cfc] = 0x5203, [0x0cfd] = 0x52fa, [0x0cfe] = 0x5343, [0x0cff] = 0x53c9,
  [0x0d00] = 0x53e3, [0x0d01] = 0x571f, [0x0d02] = 0x58eb, [0x0d03] = 0x5915,
  [0x0d04] = 0x5927, [0x0d05] = 0x5973, [0x0d06] = 0x5b50, [0x0d07] = 0x5b51,
  [0x0d08] = 0x5b53, [0x0d09] = 0x5bf8, [0x0d0a] = 0x5c0f, [0x0d0b] = 0x5c22,
  [0x0d0c] = 0x5c38, [0x0d0d] = 0x5c71, [0x0d0e] = 0x5ddd, [0x0d0f] = 0x5de5,
  [0x0d10] = 0x5df1, [0x0d11] = 0x5df2, [0x0d12] = 0x5df3, [0x0d13] = 0x5dfe,
  [0x0d14] = 0x5e72, [0x0d15] = 0x5efe, [0x0d16] = 0x5f0b, [0x0d17] = 0x5f13,
  [0x0d18] = 0x624d, [0x0d19] = 0x4e11, [0x0d1a] = 0x4e10, [0x0d1b] = 0x4e0d,
  [0x0d1c] = 0x4e2d, [0x0d1d] = 0x4e30, [0x0d1e] = 0x4e39, [0x0d1f] = 0x4e4b,
  [0x0d20] = 0x5c39, [0x0d21] = 0x4e88, [0x0d22] = 0x4e91, [0x0d23] = 0x4e95,
  [0x0d24] = 0x4e92, [0x0d25] = 0x4e94, [0x0d26] = 0x4ea2, [0x0d27] = 0x4ec1,
  [0x0d28] = 0x4ec0, [0x0d29] = 0x4ec3, [0x0d2a] = 0x4ec6, [0x0d2b] = 0x4ec7,
  [0x0d2c] = 0x4ecd, [0x0d2d] = 0x4eca, [0x0d2e] = 0x4ecb, [0x0d2f] = 0x4ec4,
  [0x0d30] = 0x5143, [0x0d31] = 0x5141, [0x0d32] = 0x5167, [0x0d33] = 0x516d,
  [0x0d34] = 0x516e, [0x0d35] = 0x516c, [0x0d36] = 0x5197, [0x0d37] = 0x51f6,
  [0x0d38] = 0x5206, [0x0d39] = 0x5207, [0x0d3a] = 0x5208, [0x0d3b] = 0x52fb,
  [0x0d3c] = 0x52fe, [0x0d3d] = 0x52ff, [0x0d3e] = 0x5316, [0x0d3f] = 0x5339,
  [0x0d40] = 0x5348, [0x0d41] = 0x5347, [0x0d42] = 0x5345, [0x0d43] = 0x535e,
  [0x0d44] = 0x5384, [0x0d45] = 0x53cb, [0x0d46] = 0x53ca, [0x0d47] = 0x53cd,
  [0x0d48] = 0x58ec, [0x0d49] = 0x5929, [0x0d4a] = 0x592b, [0x0d4b] = 0x592a,
  [0x0d4c] = 0x592d, [0x0d4d] = 0x5b54, [0x0d4e] = 0x5c11, [0x0d4f] = 0x5c24,
  [0x0d50] = 0x5c3a, [0x0d51] = 0x5c6f, [0x0d52] = 0x5df4, [0x0d53] = 0x5e7b,
  [0x0d54] = 0x5eff, [0x0d55] = 0x5f14, [0x0d56] = 0x5f15, [0x0d57] = 0x5fc3,
  [0x0d58] = 0x6208, [0x0d59] = 0x6236, [0x0d5a] = 0x624b, [0x0d5b] = 0x624e,
  [0x0d5c] = 0x652f, [0x0d5d] = 0x6587, [0x0d5e] = 0x6597, [0x0d5f] = 0x65a4,
  [0x0d60] = 0x65b9, [0x0d61] = 0x65e5, [0x0d62] = 0x66f0, [0x0d63] = 0x6708,
  [0x0d64] = 0x6728, [0x0d65] = 0x6b20, [0x0d66] = 0x6b62, [0x0d67] = 0x6b79,
  [0x0d68] = 0x6bcb, [0x0d69] = 0x6bd4, [0x0d6a] = 0x6bdb, [0x0d6b] = 0x6c0f,
  [0x0d6c] = 0x6c34, [0x0d6d] = 0x706b, [0x0d6e] = 0x722a, [0x0d6f] = 0x7236,
  [0x0d70] = 0x723b, [0x0d71] = 0x7247, [0x0d72] = 0x7259, [0x0d73] = 0x725b,
  [0x0d74] = 0x72ac, [0x0d75] = 0x738b, [0x0d76] = 0x4e19, [0x0d77] = 0x4e16,
  [0x0d78] = 0x4e15, [0x0d79] = 0x4e14, [0x0d7a] = 0x4e18, [0x0d7b] = 0x4e3b,
  [0x0d7c] = 0x4e4d, [0x0d7d] = 0x4e4f, [0x0d7e] = 0x4e4e, [0x0d7f] = 0x4ee5,
  [0x0d80] = 0x4ed8, [0x0d81] = 0x4ed4, [0x0d82] = 0x4ed5, [0x0d83] = 0x4ed6,
  [0x0d84] = 0x4ed7, [0x0d85] = 0x4ee3, [0x0d86] = 0x4ee4, [0x0d87] = 0x4ed9,
  [0x0d88] = 0x4ede, [0x0d89] = 0x5145, [0x0d8a] = 0x5144, [0x0d8b] = 0x5189,
  [0x0d8c] = 0x518a, [0x0d8d] = 0x51ac, [0x0d8e] = 0x51f9, [0x0d8f] = 0x51fa,
  [0x0d90] = 0x51f8, [0x0d91] = 0x520a, [0x0d92] = 0x52a0, [0x0d93] = 0x529f,
  [0x0d94] = 0x5305, [0x0d95] = 0x5306, [0x0d96] = 0x5317, [0x0d97] = 0x531d,
  [0x0d98] = 0x4edf, [0x0d99] = 0x534a, [0x0d9a] = 0x5349, [0x0d9b] = 0x5361,
  [0x0d9c] = 0x5360, [0x0d9d] = 0x536f, [0x0d9e] = 0x536e, [0x0d9f] = 0x53bb,
  [0x0da0] = 0x53ef, [0x0da1] = 0x53e4, [0x0da2] = 0x53f3, [0x0da3] = 0x53ec,
  [0x0da4] = 0x53ee, [0x0da5] = 0x53e9, [0x0da6] = 0x53e8, [0x0da7] = 0x53fc,
  [0x0da8] = 0x53f8, [0x0da9] = 0x53f5, [0x0daa] = 0x53eb, [0x0dab] = 0x53e6,
  [0x0dac] = 0x53ea, [0x0dad] = 0x53f2, [0x0dae] = 0x53f1, [0x0daf] = 0x53f0,
  [0x0db0] = 0x53e5, [0x0db1] = 0x53ed, [0x0db2] = 0x53fb, [0x0db3] = 0x56db,
  [0x0db4] = 0x56da, [0x0db5] = 0x5916, [0x0db6] = 0x592e, [0x0db7] = 0x5931,
  [0x0db8] = 0x5974, [0x0db9] = 0x5976, [0x0dba] = 0x5b55, [0x0dbb] = 0x5b83,
  [0x0dbc] = 0x5c3c, [0x0dbd] = 0x5de8, [0x0dbe] = 0x5de7, [0x0dbf] = 0x5de6,
  [0x0dc0] = 0x5e02, [0x0dc1] = 0x5e03, [0x0dc2] = 0x5e73, [0x0dc3] = 0x5e7c,
  [0x0dc4] = 0x5f01, [0x0dc5] = 0x5f18, [0x0dc6] = 0x5f17, [0x0dc7] = 0x5fc5,
  [0x0dc8] = 0x620a, [0x0dc9] = 0x6253, [0x0dca] = 0x6254, [0x0dcb] = 0x6252,
  [0x0dcc] = 0x6251, [0x0dcd] = 0x65a5, [0x0dce] = 0x65e6, [0x0dcf] = 0x672e,
  [0x0dd0] = 0x672c, [0x0dd1] = 0x672a, [0x0dd2] = 0x672b, [0x0dd3] = 0x672d,
  [0x0dd4] = 0x6b63, [0x0dd5] = 0x6bcd, [0x0dd6] = 0x6c11, [0x0dd7] = 0x6c10,
  [0x0dd8] = 0x6c38, [0x0dd9] = 0x6c41, [0x0dda] = 0x6c40, [0x0ddb] = 0x6c3e,
  [0x0ddc] = 0x72af, [0x0ddd] = 0x7384, [0x0dde] = 0x7389, [0x0ddf] = 0x74dc,
  [0x0de0] = 0x74e6, [0x0de1] = 0x7518, [0x0de2] = 0x751f, [0x0de3] = 0x7528,
  [0x0de4] = 0x7529, [0x0de5] = 0x7530, [0x0de6] = 0x7531, [0x0de7] = 0x7532,
  [0x0de8] = 0x7533, [0x0de9] = 0x758b, [0x0dea] = 0x767d, [0x0deb] = 0x76ae,
  [0x0dec] = 0x76bf, [0x0ded] = 0x76ee, [0x0dee] = 0x77db, [0x0def] = 0x77e2,
  [0x0df0] = 0x77f3, [0x0df1] = 0x793a, [0x0df2] = 0x79be, [0x0df3] = 0x7a74,
  [0x0df4] = 0x7acb, [0x0df5] = 0x4e1e, [0x0df6] = 0x4e1f, [0x0df7] = 0x4e52,
  [0x0df8] = 0x4e53, [0x0df9] = 0x4e69, [0x0dfa] = 0x4e99, [0x0dfb] = 0x4ea4,
  [0x0dfc] = 0x4ea6, [0x0dfd] = 0x4ea5, [0x0dfe] = 0x4eff, [0x0dff] = 0x4f09,
  [0x0e00] = 0x4f19, [0x0e01] = 0x4f0a, [0x0e02] = 0x4f15, [0x0e03] = 0x4f0d,
  [0x0e04] = 0x4f10, [0x0e05] = 0x4f11, [0x0e06] = 0x4f0f, [0x0e07] = 0x4ef2,
  [0x0e08] = 0x4ef6, [0x0e09] = 0x4efb, [0x0e0a] = 0x4ef0, [0x0e0b] = 0x4ef3,
  [0x0e0c] = 0x4efd, [0x0e0d] = 0x4f01, [0x0e0e] = 0x4f0b, [0x0e0f] = 0x5149,
  [0x0e10] = 0x5147, [0x0e11] = 0x5146, [0x0e12] = 0x5148, [0x0e13] = 0x5168,
  [0x0e14] = 0x5171, [0x0e15] = 0x518d, [0x0e16] = 0x51b0, [0x0e17] = 0x5217,
  [0x0e18] = 0x5211, [0x0e19] = 0x5212, [0x0e1a] = 0x520e, [0x0e1b] = 0x5216,
  [0x0e1c] = 0x52a3, [0x0e1d] = 0x5308, [0x0e1e] = 0x5321, [0x0e1f] = 0x5320,
  [0x0e20] = 0x5370, [0x0e21] = 0x5371, [0x0e22] = 0x5409, [0x0e23] = 0x540f,
  [0x0e24] = 0x540c, [0x0e25] = 0x540a, [0x0e26] = 0x5410, [0x0e27] = 0x5401,
  [0x0e28] = 0x540b, [0x0e29] = 0x5404, [0x0e2a] = 0x5411, [0x0e2b] = 0x540d,
  [0x0e2c] = 0x5408, [0x0e2d] = 0x5403, [0x0e2e] = 0x540e, [0x0e2f] = 0x5406,
  [0x0e30] = 0x5412, [0x0e31] = 0x56e0, [0x0e32] = 0x56de, [0x0e33] = 0x56dd,
  [0x0e34] = 0x5733, [0x0e35] = 0x5730, [0x0e36] = 0x5728, [0x0e37] = 0x572d,
  [0x0e38] = 0x572c, [0x0e39] = 0x572f, [0x0e3a] = 0x5729, [0x0e3b] = 0x5919,
  [0x0e3c] = 0x591a, [0x0e3d] = 0x5937, [0x0e3e] = 0x5938, [0x0e3f] = 0x5984,
  [0x0e40] = 0x5978, [0x0e41] = 0x5983, [0x0e42] = 0x597d, [0x0e43] = 0x5979,
  [0x0e44] = 0x5982, [0x0e45] = 0x5981, [0x0e46] = 0x5b57, [0x0e47] = 0x5b58,
  [0x0e48] = 0x5b87, [0x0e49] = 0x5b88, [0x0e4a] = 0x5b85, [0x0e4b] = 0x5b89,
  [0x0e4c] = 0x5bfa, [0x0e4d] = 0x5c16, [0x0e4e] = 0x5c79, [0x0e4f] = 0x5dde,
  [0x0e50] = 0x5e06, [0x0e51] = 0x5e76, [0x0e52] = 0x5e74, [0x0e53] = 0x5f0f,
  [0x0e54] = 0x5f1b, [0x0e55] = 0x5fd9, [0x0e56] = 0x5fd6, [0x0e57] = 0x620e,
  [0x0e58] = 0x620c, [0x0e59] = 0x620d, [0x0e5a] = 0x6210, [0x0e5b] = 0x6263,
  [0x0e5c] = 0x625b, [0x0e5d] = 0x6258, [0x0e5e] = 0x6536, [0x0e5f] = 0x65e9,
  [0x0e60] = 0x65e8, [0x0e61] = 0x65ec, [0x0e62] = 0x65ed, [0x0e63] = 0x66f2,
  [0x0e64] = 0x66f3, [0x0e65] = 0x6709, [0x0e66] = 0x673d, [0x0e67] = 0x6734,
  [0x0e68] = 0x6731, [0x0e69] = 0x6735, [0x0e6a] = 0x6b21, [0x0e6b] = 0x6b64,
  [0x0e6c] = 0x6b7b, [0x0e6d] = 0x6c16, [0x0e6e] = 0x6c5d, [0x0e6f] = 0x6c57,
  [0x0e70] = 0x6c59, [0x0e71] = 0x6c5f, [0x0e72] = 0x6c60, [0x0e73] = 0x6c50,
  [0x0e74] = 0x6c55, [0x0e75] = 0x6c61, [0x0e76] = 0x6c5b, [0x0e77] = 0x6c4d,
  [0x0e78] = 0x6c4e, [0x0e79] = 0x7070, [0x0e7a] = 0x725f, [0x0e7b] = 0x725d,
  [0x0e7c] = 0x767e, [0x0e7d] = 0x7af9, [0x0e7e] = 0x7c73, [0x0e7f] = 0x7cf8,
  [0x0e80] = 0x7f36, [0x0e81] = 0x7f8a, [0x0e82] = 0x7fbd, [0x0e83] = 0x8001,
  [0x0e84] = 0x8003, [0x0e85] = 0x800c, [0x0e86] = 0x8012, [0x0e87] = 0x8033,
  [0x0e88] = 0x807f, [0x0e89] = 0x8089, [0x0e8a] = 0x808b, [0x0e8b] = 0x808c,
  [0x0e8c] = 0x81e3, [0x0e8d] = 0x81ea, [0x0e8e] = 0x81f3, [0x0e8f] = 0x81fc,
  [0x0e90] = 0x820c, [0x0e91] = 0x821b, [0x0e92] = 0x821f, [0x0e93] = 0x826e,
  [0x0e94] = 0x8272, [0x0e95] = 0x827e, [0x0e96] = 0x866b, [0x0e97] = 0x8840,
  [0x0e98] = 0x884c, [0x0e99] = 0x8863, [0x0e9a] = 0x897f, [0x0e9b] = 0x9621,
  [0x0e9c] = 0x4e32, [0x0e9d] = 0x4ea8, [0x0e9e] = 0x4f4d, [0x0e9f] = 0x4f4f,
  [0x0ea0] = 0x4f47, [0x0ea1] = 0x4f57, [0x0ea2] = 0x4f5e, [0x0ea3] = 0x4f34,
  [0x0ea4] = 0x4f5b, [0x0ea5] = 0x4f55, [0x0ea6] = 0x4f30, [0x0ea7] = 0x4f50,
  [0x0ea8] = 0x4f51, [0x0ea9] = 0x4f3d, [0x0eaa] = 0x4f3a, [0x0eab] = 0x4f38,
  [0x0eac] = 0x4f43, [0x0ead] = 0x4f54, [0x0eae] = 0x4f3c, [0x0eaf] = 0x4f46,
  [0x0eb0] = 0x4f63, [0x0eb1] = 0x4f5c, [0x0eb2] = 0x4f60, [0x0eb3] = 0x4f2f,
  [0x0eb4] = 0x4f4e, [0x0eb5] = 0x4f36, [0x0eb6] = 0x4f59, [0x0eb7] = 0x4f5d,
  [0x0eb8] = 0x4f48, [0x0eb9] = 0x4f5a, [0x0eba] = 0x514c, [0x0ebb] = 0x514b,
  [0x0ebc] = 0x514d, [0x0ebd] = 0x5175, [0x0ebe] = 0x51b6, [0x0ebf] = 0x51b7,
  [0x0ec0] = 0x5225, [0x0ec1] = 0x5224, [0x0ec2] = 0x5229, [0x0ec3] = 0x522a,
  [0x0ec4] = 0x5228, [0x0ec5] = 0x52ab, [0x0ec6] = 0x52a9, [0x0ec7] = 0x52aa,
  [0x0ec8] = 0x52ac, [0x0ec9] = 0x5323, [0x0eca] = 0x5373, [0x0ecb] = 0x5375,
  [0x0ecc] = 0x541d, [0x0ecd] = 0x542d, [0x0ece] = 0x541e, [0x0ecf] = 0x543e,
  [0x0ed0] = 0x5426, [0x0ed1] = 0x544e, [0x0ed2] = 0x5427, [0x0ed3] = 0x5446,
  [0x0ed4] = 0x5443, [0x0ed5] = 0x5433, [0x0ed6] = 0x5448, [0x0ed7] = 0x5442,
  [0x0ed8] = 0x541b, [0x0ed9] = 0x5429, [0x0eda] = 0x544a, [0x0edb] = 0x5439,
  [0x0edc] = 0x543b, [0x0edd] = 0x5438, [0x0ede] = 0x542e, [0x0edf] = 0x5435,
  [0x0ee0] = 0x5436, [0x0ee1] = 0x5420, [0x0ee2] = 0x543c, [0x0ee3] = 0x5440,
  [0x0ee4] = 0x5431, [0x0ee5] = 0x542b, [0x0ee6] = 0x541f, [0x0ee7] = 0x542c,
  [0x0ee8] = 0x56ea, [0x0ee9] = 0x56f0, [0x0eea] = 0x56e4, [0x0eeb] = 0x56eb,
  [0x0eec] = 0x574a, [0x0eed] = 0x5751, [0x0eee] = 0x5740, [0x0eef] = 0x574d,
  [0x0ef0] = 0x5747, [0x0ef1] = 0x574e, [0x0ef2] = 0x573e, [0x0ef3] = 0x5750,
  [0x0ef4] = 0x574f, [0x0ef5] = 0x573b, [0x0ef6] = 0x58ef, [0x0ef7] = 0x593e,
  [0x0ef8] = 0x599d, [0x0ef9] = 0x5992, [0x0efa] = 0x59a8, [0x0efb] = 0x599e,
  [0x0efc] = 0x59a3, [0x0efd] = 0x5999, [0x0efe] = 0x5996, [0x0eff] = 0x598d,
  [0x0f00] = 0x59a4, [0x0f01] = 0x5993, [0x0f02] = 0x598a, [0x0f03] = 0x59a5,
  [0x0f04] = 0x5b5d, [0x0f05] = 0x5b5c, [0x0f06] = 0x5b5a, [0x0f07] = 0x5b5b,
  [0x0f08] = 0x5b8c, [0x0f09] = 0x5b8b, [0x0f0a] = 0x5b8f, [0x0f0b] = 0x5c2c,
  [0x0f0c] = 0x5c40, [0x0f0d] = 0x5c41, [0x0f0e] = 0x5c3f, [0x0f0f] = 0x5c3e,
  [0x0f10] = 0x5c90, [0x0f11] = 0x5c91, [0x0f12] = 0x5c94, [0x0f13] = 0x5c8c,
  [0x0f14] = 0x5deb, [0x0f15] = 0x5e0c, [0x0f16] = 0x5e8f, [0x0f17] = 0x5e87,
  [0x0f18] = 0x5e8a, [0x0f19] = 0x5ef7, [0x0f1a] = 0x5f04, [0x0f1b] = 0x5f1f,
  [0x0f1c] = 0x5f64, [0x0f1d] = 0x5f62, [0x0f1e] = 0x5f77, [0x0f1f] = 0x5f79,
  [0x0f20] = 0x5fd8, [0x0f21] = 0x5fcc, [0x0f22] = 0x5fd7, [0x0f23] = 0x5fcd,
  [0x0f24] = 0x5ff1, [0x0f25] = 0x5feb, [0x0f26] = 0x5ff8, [0x0f27] = 0x5fea,
  [0x0f28] = 0x6212, [0x0f29] = 0x6211, [0x0f2a] = 0x6284, [0x0f2b] = 0x6297,
  [0x0f2c] = 0x6296, [0x0f2d] = 0x6280, [0x0f2e] = 0x6276, [0x0f2f] = 0x6289,
  [0x0f30] = 0x626d, [0x0f31] = 0x628a, [0x0f32] = 0x627c, [0x0f33] = 0x627e,
  [0x0f34] = 0x6279, [0x0f35] = 0x6273, [0x0f36] = 0x6292, [0x0f37] = 0x626f,
  [0x0f38] = 0x6298, [0x0f39] = 0x626e, [0x0f3a] = 0x6295, [0x0f3b] = 0x6293,
  [0x0f3c] = 0x6291, [0x0f3d] = 0x6286, [0x0f3e] = 0x6539, [0x0f3f] = 0x653b,
  [0x0f40] = 0x6538, [0x0f41] = 0x65f1, [0x0f42] = 0x66f4, [0x0f43] = 0x675f,
  [0x0f44] = 0x674e, [0x0f45] = 0x674f, [0x0f46] = 0x6750, [0x0f47] = 0x6751,
  [0x0f48] = 0x675c, [0x0f49] = 0x6756, [0x0f4a] = 0x675e, [0x0f4b] = 0x6749,
  [0x0f4c] = 0x6746, [0x0f4d] = 0x6760, [0x0f4e] = 0x6753, [0x0f4f] = 0x6757,
  [0x0f50] = 0x6b65, [0x0f51] = 0x6bcf, [0x0f52] = 0x6c42, [0x0f53] = 0x6c5e,
  [0x0f54] = 0x6c99, [0x0f55] = 0x6c81, [0x0f56] = 0x6c88, [0x0f57] = 0x6c89,
  [0x0f58] = 0x6c85, [0x0f59] = 0x6c9b, [0x0f5a] = 0x6c6a, [0x0f5b] = 0x6c7a,
  [0x0f5c] = 0x6c90, [0x0f5d] = 0x6c70, [0x0f5e] = 0x6c8c, [0x0f5f] = 0x6c68,
  [0x0f60] = 0x6c96, [0x0f61] = 0x6c92, [0x0f62] = 0x6c7d, [0x0f63] = 0x6c83,
  [0x0f64] = 0x6c72, [0x0f65] = 0x6c7e, [0x0f66] = 0x6c74, [0x0f67] = 0x6c86,
  [0x0f68] = 0x6c76, [0x0f69] = 0x6c8d, [0x0f6a] = 0x6c94, [0x0f6b] = 0x6c98,
  [0x0f6c] = 0x6c82, [0x0f6d] = 0x7076, [0x0f6e] = 0x707c, [0x0f6f] = 0x707d,
  [0x0f70] = 0x7078, [0x0f71] = 0x7262, [0x0f72] = 0x7261, [0x0f73] = 0x7260,
  [0x0f74] = 0x72c4, [0x0f75] = 0x72c2, [0x0f76] = 0x7396, [0x0f77] = 0x752c,
  [0x0f78] = 0x752b, [0x0f79] = 0x7537, [0x0f7a] = 0x7538, [0x0f7b] = 0x7682,
  [0x0f7c] = 0x76ef, [0x0f7d] = 0x77e3, [0x0f7e] = 0x79c1, [0x0f7f] = 0x79c0,
  [0x0f80] = 0x79bf, [0x0f81] = 0x7a76, [0x0f82] = 0x7cfb, [0x0f83] = 0x7f55,
  [0x0f84] = 0x8096, [0x0f85] = 0x8093, [0x0f86] = 0x809d, [0x0f87] = 0x8098,
  [0x0f88] = 0x809b, [0x0f89] = 0x809a, [0x0f8a] = 0x80b2, [0x0f8b] = 0x826f,
  [0x0f8c] = 0x8292, [0x0f8d] = 0x828b, [0x0f8e] = 0x828d, [0x0f8f] = 0x898b,
  [0x0f90] = 0x89d2, [0x0f91] = 0x8a00, [0x0f92] = 0x8c37, [0x0f93] = 0x8c46,
  [0x0f94] = 0x8c55, [0x0f95] = 0x8c9d, [0x0f96] = 0x8d64, [0x0f97] = 0x8d70,
  [0x0f98] = 0x8db3, [0x0f99] = 0x8eab, [0x0f9a] = 0x8eca, [0x0f9b] = 0x8f9b,
  [0x0f9c] = 0x8fb0, [0x0f9d] = 0x8fc2, [0x0f9e] = 0x8fc6, [0x0f9f] = 0x8fc5,
  [0x0fa0] = 0x8fc4, [0x0fa1] = 0x5de1, [0x0fa2] = 0x9091, [0x0fa3] = 0x90a2,
  [0x0fa4] = 0x90aa, [0x0fa5] = 0x90a6, [0x0fa6] = 0x90a3, [0x0fa7] = 0x9149,
  [0x0fa8] = 0x91c6, [0x0fa9] = 0x91cc, [0x0faa] = 0x9632, [0x0fab] = 0x962e,
  [0x0fac] = 0x9631, [0x0fad] = 0x962a, [0x0fae] = 0x962c, [0x0faf] = 0x4e26,
  [0x0fb0] = 0x4e56, [0x0fb1] = 0x4e73, [0x0fb2] = 0x4e8b, [0x0fb3] = 0x4e9b,
  [0x0fb4] = 0x4e9e, [0x0fb5] = 0x4eab, [0x0fb6] = 0x4eac, [0x0fb7] = 0x4f6f,
  [0x0fb8] = 0x4f9d, [0x0fb9] = 0x4f8d, [0x0fba] = 0x4f73, [0x0fbb] = 0x4f7f,
  [0x0fbc] = 0x4f6c, [0x0fbd] = 0x4f9b, [0x0fbe] = 0x4f8b, [0x0fbf] = 0x4f86,
  [0x0fc0] = 0x4f83, [0x0fc1] = 0x4f70, [0x0fc2] = 0x4f75, [0x0fc3] = 0x4f88,
  [0x0fc4] = 0x4f69, [0x0fc5] = 0x4f7b, [0x0fc6] = 0x4f96, [0x0fc7] = 0x4f7e,
  [0x0fc8] = 0x4f8f, [0x0fc9] = 0x4f91, [0x0fca] = 0x4f7a, [0x0fcb] = 0x5154,
  [0x0fcc] = 0x5152, [0x0fcd] = 0x5155, [0x0fce] = 0x5169, [0x0fcf] = 0x5177,
  [0x0fd0] = 0x5176, [0x0fd1] = 0x5178, [0x0fd2] = 0x51bd, [0x0fd3] = 0x51fd,
  [0x0fd4] = 0x523b, [0x0fd5] = 0x5238, [0x0fd6] = 0x5237, [0x0fd7] = 0x523a,
  [0x0fd8] = 0x5230, [0x0fd9] = 0x522e, [0x0fda] = 0x5236, [0x0fdb] = 0x5241,
  [0x0fdc] = 0x52be, [0x0fdd] = 0x52bb, [0x0fde] = 0x5352, [0x0fdf] = 0x5354,
  [0x0fe0] = 0x5353, [0x0fe1] = 0x5351, [0x0fe2] = 0x5366, [0x0fe3] = 0x5377,
  [0x0fe4] = 0x5378, [0x0fe5] = 0x5379, [0x0fe6] = 0x53d6, [0x0fe7] = 0x53d4,
  [0x0fe8] = 0x53d7, [0x0fe9] = 0x5473, [0x0fea] = 0x5475, [0x0feb] = 0x5496,
  [0x0fec] = 0x5478, [0x0fed] = 0x5495, [0x0fee] = 0x5480, [0x0fef] = 0x547b,
  [0x0ff0] = 0x5477, [0x0ff1] = 0x5484, [0x0ff2] = 0x5492, [0x0ff3] = 0x5486,
  [0x0ff4] = 0x547c, [0x0ff5] = 0x5490, [0x0ff6] = 0x5471, [0x0ff7] = 0x5476,
  [0x0ff8] = 0x548c, [0x0ff9] = 0x549a, [0x0ffa] = 0x5462, [0x0ffb] = 0x5468,
  [0x0ffc] = 0x548b, [0x0ffd] = 0x547d, [0x0ffe] = 0x548e, [0x0fff] = 0x56fa,
  [0x1000] = 0x5783, [0x1001] = 0x5777, [0x1002] = 0x576a, [0x1003] = 0x5769,
  [0x1004] = 0x5761, [0x1005] = 0x5766, [0x1006] = 0x5764, [0x1007] = 0x577c,
  [0x1008] = 0x591c, [0x1009] = 0x5949, [0x100a] = 0x5947, [0x100b] = 0x5948,
  [0x100c] = 0x5944, [0x100d] = 0x5954, [0x100e] = 0x59be, [0x100f] = 0x59bb,
  [0x1010] = 0x59d4, [0x1011] = 0x59b9, [0x1012] = 0x59ae, [0x1013] = 0x59d1,
  [0x1014] = 0x59c6, [0x1015] = 0x59d0, [0x1016] = 0x59cd, [0x1017] = 0x59cb,
  [0x1018] = 0x59d3, [0x1019] = 0x59ca, [0x101a] = 0x59af, [0x101b] = 0x59b3,
  [0x101c] = 0x59d2, [0x101d] = 0x59c5, [0x101e] = 0x5b5f, [0x101f] = 0x5b64,
  [0x1020] = 0x5b63, [0x1021] = 0x5b97, [0x1022] = 0x5b9a, [0x1023] = 0x5b98,
  [0x1024] = 0x5b9c, [0x1025] = 0x5b99, [0x1026] = 0x5b9b, [0x1027] = 0x5c1a,
  [0x1028] = 0x5c48, [0x1029] = 0x5c45, [0x102a] = 0x5c46, [0x102b] = 0x5cb7,
  [0x102c] = 0x5ca1, [0x102d] = 0x5cb8, [0x102e] = 0x5ca9, [0x102f] = 0x5cab,
  [0x1030] = 0x5cb1, [0x1031] = 0x5cb3, [0x1032] = 0x5e18, [0x1033] = 0x5e1a,
  [0x1034] = 0x5e16, [0x1035] = 0x5e15, [0x1036] = 0x5e1b, [0x1037] = 0x5e11,
  [0x1038] = 0x5e78, [0x1039] = 0x5e9a, [0x103a] = 0x5e97, [0x103b] = 0x5e9c,
  [0x103c] = 0x5e95, [0x103d] = 0x5e96, [0x103e] = 0x5ef6, [0x103f] = 0x5f26,
  [0x1040] = 0x5f27, [0x1041] = 0x5f29, [0x1042] = 0x5f80, [0x1043] = 0x5f81,
  [0x1044] = 0x5f7f, [0x1045] = 0x5f7c, [0x1046] = 0x5fdd, [0x1047] = 0x5fe0,
  [0x1048] = 0x5ffd, [0x1049] = 0x5ff5, [0x104a] = 0x5fff, [0x104b] = 0x600f,
  [0x104c] = 0x6014, [0x104d] = 0x602f, [0x104e] = 0x6035, [0x104f] = 0x6016,
  [0x1050] = 0x602a, [0x1051] = 0x6015, [0x1052] = 0x6021, [0x1053] = 0x6027,
  [0x1054] = 0x6029, [0x1055] = 0x602b, [0x1056] = 0x601b, [0x1057] = 0x6216,
  [0x1058] = 0x6215, [0x1059] = 0x623f, [0x105a] = 0x623e, [0x105b] = 0x6240,
  [0x105c] = 0x627f, [0x105d] = 0x62c9, [0x105e] = 0x62cc, [0x105f] = 0x62c4,
  [0x1060] = 0x62bf, [0x1061] = 0x62c2, [0x1062] = 0x62b9, [0x1063] = 0x62d2,
  [0x1064] = 0x62db, [0x1065] = 0x62ab, [0x1066] = 0x62d3, [0x1067] = 0x62d4,
  [0x1068] = 0x62cb, [0x1069] = 0x62c8, [0x106a] = 0x62a8, [0x106b] = 0x62bd,
  [0x106c] = 0x62bc, [0x106d] = 0x62d0, [0x106e] = 0x62d9, [0x106f] = 0x62c7,
  [0x1070] = 0x62cd, [0x1071] = 0x62b5, [0x1072] = 0x62da, [0x1073] = 0x62b1,
  [0x1074] = 0x62d8, [0x1075] = 0x62d6, [0x1076] = 0x62d7, [0x1077] = 0x62c6,
  [0x1078] = 0x62ac, [0x1079] = 0x62ce, [0x107a] = 0x653e, [0x107b] = 0x65a7,
  [0x107c] = 0x65bc, [0x107d] = 0x65fa, [0x107e] = 0x6614, [0x107f] = 0x6613,
  [0x1080] = 0x660c, [0x1081] = 0x6606, [0x1082] = 0x6602, [0x1083] = 0x660e,
  [0x1084] = 0x6600, [0x1085] = 0x660f, [0x1086] = 0x6615, [0x1087] = 0x660a,
  [0x1088] = 0x6607, [0x1089] = 0x670d, [0x108a] = 0x670b, [0x108b] = 0x676d,
  [0x108c] = 0x678b, [0x108d] = 0x6795, [0x108e] = 0x6771, [0x108f] = 0x679c,
  [0x1090] = 0x6773, [0x1091] = 0x6777, [0x1092] = 0x6787, [0x1093] = 0x679d,
  [0x1094] = 0x6797, [0x1095] = 0x676f, [0x1096] = 0x6770, [0x1097] = 0x677f,
  [0x1098] = 0x6789, [0x1099] = 0x677e, [0x109a] = 0x6790, [0x109b] = 0x6775,
  [0x109c] = 0x679a, [0x109d] = 0x6793, [0x109e] = 0x677c, [0x109f] = 0x676a,
  [0x10a0] = 0x6772, [0x10a1] = 0x6b23, [0x10a2] = 0x6b66, [0x10a3] = 0x6b67,
  [0x10a4] = 0x6b7f, [0x10a5] = 0x6c13, [0x10a6] = 0x6c1b, [0x10a7] = 0x6ce3,
  [0x10a8] = 0x6ce8, [0x10a9] = 0x6cf3, [0x10aa] = 0x6cb1, [0x10ab] = 0x6ccc,
  [0x10ac] = 0x6ce5, [0x10ad] = 0x6cb3, [0x10ae] = 0x6cbd, [0x10af] = 0x6cbe,
  [0x10b0] = 0x6cbc, [0x10b1] = 0x6ce2, [0x10b2] = 0x6cab, [0x10b3] = 0x6cd5,
  [0x10b4] = 0x6cd3, [0x10b5] = 0x6cb8, [0x10b6] = 0x6cc4, [0x10b7] = 0x6cb9,
  [0x10b8] = 0x6cc1, [0x10b9] = 0x6cae, [0x10ba] = 0x6cd7, [0x10bb] = 0x6cc5,
  [0x10bc] = 0x6cf1, [0x10bd] = 0x6cbf, [0x10be] = 0x6cbb, [0x10bf] = 0x6ce1,
  [0x10c0] = 0x6cdb, [0x10c1] = 0x6cca, [0x10c2] = 0x6cac, [0x10c3] = 0x6cef,
  [0x10c4] = 0x6cdc, [0x10c5] = 0x6cd6, [0x10c6] = 0x6ce0, [0x10c7] = 0x7095,
  [0x10c8] = 0x708e, [0x10c9] = 0x7092, [0x10ca] = 0x708a, [0x10cb] = 0x7099,
  [0x10cc] = 0x722c, [0x10cd] = 0x722d, [0x10ce] = 0x7238, [0x10cf] = 0x7248,
  [0x10d0] = 0x7267, [0x10d1] = 0x7269, [0x10d2] = 0x72c0, [0x10d3] = 0x72ce,
  [0x10d4] = 0x72d9, [0x10d5] = 0x72d7, [0x10d6] = 0x72d0, [0x10d7] = 0x73a9,
  [0x10d8] = 0x73a8, [0x10d9] = 0x739f, [0x10da] = 0x73ab, [0x10db] = 0x73a5,
  [0x10dc] = 0x753d, [0x10dd] = 0x759d, [0x10de] = 0x7599, [0x10df] = 0x759a,
  [0x10e0] = 0x7684, [0x10e1] = 0x76c2, [0x10e2] = 0x76f2, [0x10e3] = 0x76f4,
  [0x10e4] = 0x77e5, [0x10e5] = 0x77fd, [0x10e6] = 0x793e, [0x10e7] = 0x7940,
  [0x10e8] = 0x7941, [0x10e9] = 0x79c9, [0x10ea] = 0x79c8, [0x10eb] = 0x7a7a,
  [0x10ec] = 0x7a79, [0x10ed] = 0x7afa, [0x10ee] = 0x7cfe, [0x10ef] = 0x7f54,
  [0x10f0] = 0x7f8c, [0x10f1] = 0x7f8b, [0x10f2] = 0x8005, [0x10f3] = 0x80ba,
  [0x10f4] = 0x80a5, [0x10f5] = 0x80a2, [0x10f6] = 0x80b1, [0x10f7] = 0x80a1,
  [0x10f8] = 0x80ab, [0x10f9] = 0x80a9, [0x10fa] = 0x80b4, [0x10fb] = 0x80aa,
  [0x10fc] = 0x80af, [0x10fd] = 0x81e5, [0x10fe] = 0x81fe, [0x10ff] = 0x820d,
  [0x1100] = 0x82b3, [0x1101] = 0x829d, [0x1102] = 0x8299, [0x1103] = 0x82ad,
  [0x1104] = 0x82bd, [0x1105] = 0x829f, [0x1106] = 0x82b9, [0x1107] = 0x82b1,
  [0x1108] = 0x82ac, [0x1109] = 0x82a5, [0x110a] = 0x82af, [0x110b] = 0x82b8,
  [0x110c] = 0x82a3, [0x110d] = 0x82b0, [0x110e] = 0x82be, [0x110f] = 0x82b7,
  [0x1110] = 0x864e, [0x1111] = 0x8671, [0x1112] = 0x521d, [0x1113] = 0x8868,
  [0x1114] = 0x8ecb, [0x1115] = 0x8fce, [0x1116] = 0x8fd4, [0x1117] = 0x8fd1,
  [0x1118] = 0x90b5, [0x1119] = 0x90b8, [0x111a] = 0x90b1, [0x111b] = 0x90b6,
  [0x111c] = 0x91c7, [0x111d] = 0x91d1, [0x111e] = 0x9577, [0x111f] = 0x9580,
  [0x1120] = 0x961c, [0x1121] = 0x9640, [0x1122] = 0x963f, [0x1123] = 0x963b,
  [0x1124] = 0x9644, [0x1125] = 0x9642, [0x1126] = 0x96b9, [0x1127] = 0x96e8,
  [0x1128] = 0x9752, [0x1129] = 0x975e, [0x112a] = 0x4e9f, [0x112b] = 0x4ead,
  [0x112c] = 0x4eae, [0x112d] = 0x4fe1, [0x112e] = 0x4fb5, [0x112f] = 0x4faf,
  [0x1130] = 0x4fbf, [0x1131] = 0x4fe0, [0x1132] = 0x4fd1, [0x1133] = 0x4fcf,
  [0x1134] = 0x4fdd, [0x1135] = 0x4fc3, [0x1136] = 0x4fb6, [0x1137] = 0x4fd8,
  [0x1138] = 0x4fdf, [0x1139] = 0x4fca, [0x113a] = 0x4fd7, [0x113b] = 0x4fae,
  [0x113c] = 0x4fd0, [0x113d] = 0x4fc4, [0x113e] = 0x4fc2, [0x113f] = 0x4fda,
  [0x1140] = 0x4fce, [0x1141] = 0x4fde, [0x1142] = 0x4fb7, [0x1143] = 0x5157,
  [0x1144] = 0x5192, [0x1145] = 0x5191, [0x1146] = 0x51a0, [0x1147] = 0x524e,
  [0x1148] = 0x5243, [0x1149] = 0x524a, [0x114a] = 0x524d, [0x114b] = 0x524c,
  [0x114c] = 0x524b, [0x114d] = 0x5247, [0x114e] = 0x52c7, [0x114f] = 0x52c9,
  [0x1150] = 0x52c3, [0x1151] = 0x52c1, [0x1152] = 0x530d, [0x1153] = 0x5357,
  [0x1154] = 0x537b, [0x1155] = 0x539a, [0x1156] = 0x53db, [0x1157] = 0x54ac,
  [0x1158] = 0x54c0, [0x1159] = 0x54a8, [0x115a] = 0x54ce, [0x115b] = 0x54c9,
  [0x115c] = 0x54b8, [0x115d] = 0x54a6, [0x115e] = 0x54b3, [0x115f] = 0x54c7,
  [0x1160] = 0x54c2, [0x1161] = 0x54bd, [0x1162] = 0x54aa, [0x1163] = 0x54c1,
  [0x1164] = 0x54c4, [0x1165] = 0x54c8, [0x1166] = 0x54af, [0x1167] = 0x54ab,
  [0x1168] = 0x54b1, [0x1169] = 0x54bb, [0x116a] = 0x54a9, [0x116b] = 0x54a7,
  [0x116c] = 0x54bf, [0x116d] = 0x56ff, [0x116e] = 0x5782, [0x116f] = 0x578b,
  [0x1170] = 0x57a0, [0x1171] = 0x57a3, [0x1172] = 0x57a2, [0x1173] = 0x57ce,
  [0x1174] = 0x57ae, [0x1175] = 0x5793, [0x1176] = 0x5955, [0x1177] = 0x5951,
  [0x1178] = 0x594f, [0x1179] = 0x594e, [0x117a] = 0x5950, [0x117b] = 0x59dc,
  [0x117c] = 0x59d8, [0x117d] = 0x59ff, [0x117e] = 0x59e3, [0x117f] = 0x59e8,
  [0x1180] = 0x5a03, [0x1181] = 0x59e5, [0x1182] = 0x59ea, [0x1183] = 0x59da,
  [0x1184] = 0x59e6, [0x1185] = 0x5a01, [0x1186] = 0x59fb, [0x1187] = 0x5b69,
  [0x1188] = 0x5ba3, [0x1189] = 0x5ba6, [0x118a] = 0x5ba4, [0x118b] = 0x5ba2,
  [0x118c] = 0x5ba5, [0x118d] = 0x5c01, [0x118e] = 0x5c4e, [0x118f] = 0x5c4f,
  [0x1190] = 0x5c4d, [0x1191] = 0x5c4b, [0x1192] = 0x5cd9, [0x1193] = 0x5cd2,
  [0x1194] = 0x5df7, [0x1195] = 0x5e1d, [0x1196] = 0x5e25, [0x1197] = 0x5e1f,
  [0x1198] = 0x5e7d, [0x1199] = 0x5ea0, [0x119a] = 0x5ea6, [0x119b] = 0x5efa,
  [0x119c] = 0x5f08, [0x119d] = 0x5f2d, [0x119e] = 0x5f65, [0x119f] = 0x5f88,
  [0x11a0] = 0x5f85, [0x11a1] = 0x5f8a, [0x11a2] = 0x5f8b, [0x11a3] = 0x5f87,
  [0x11a4] = 0x5f8c, [0x11a5] = 0x5f89, [0x11a6] = 0x6012, [0x11a7] = 0x601d,
  [0x11a8] = 0x6020, [0x11a9] = 0x6025, [0x11aa] = 0x600e, [0x11ab] = 0x6028,
  [0x11ac] = 0x604d, [0x11ad] = 0x6070, [0x11ae] = 0x6068, [0x11af] = 0x6062,
  [0x11b0] = 0x6046, [0x11b1] = 0x6043, [0x11b2] = 0x606c, [0x11b3] = 0x606b,
  [0x11b4] = 0x606a, [0x11b5] = 0x6064, [0x11b6] = 0x6241, [0x11b7] = 0x62dc,
  [0x11b8] = 0x6316, [0x11b9] = 0x6309, [0x11ba] = 0x62fc, [0x11bb] = 0x62ed,
  [0x11bc] = 0x6301, [0x11bd] = 0x62ee, [0x11be] = 0x62fd, [0x11bf] = 0x6307,
  [0x11c0] = 0x62f1, [0x11c1] = 0x62f7, [0x11c2] = 0x62ef, [0x11c3] = 0x62ec,
  [0x11c4] = 0x62fe, [0x11c5] = 0x62f4, [0x11c6] = 0x6311, [0x11c7] = 0x6302,
  [0x11c8] = 0x653f, [0x11c9] = 0x6545, [0x11ca] = 0x65ab, [0x11cb] = 0x65bd,
  [0x11cc] = 0x65e2, [0x11cd] = 0x6625, [0x11ce] = 0x662d, [0x11cf] = 0x6620,
  [0x11d0] = 0x6627, [0x11d1] = 0x662f, [0x11d2] = 0x661f, [0x11d3] = 0x6628,
  [0x11d4] = 0x6631, [0x11d5] = 0x6624, [0x11d6] = 0x66f7, [0x11d7] = 0x67ff,
  [0x11d8] = 0x67d3, [0x11d9] = 0x67f1, [0x11da] = 0x67d4, [0x11db] = 0x67d0,
  [0x11dc] = 0x67ec, [0x11dd] = 0x67b6, [0x11de] = 0x67af, [0x11df] = 0x67f5,
  [0x11e0] = 0x67e9, [0x11e1] = 0x67ef, [0x11e2] = 0x67c4, [0x11e3] = 0x67d1,
  [0x11e4] = 0x67b4, [0x11e5] = 0x67da, [0x11e6] = 0x67e5, [0x11e7] = 0x67b8,
  [0x11e8] = 0x67cf, [0x11e9] = 0x67de, [0x11ea] = 0x67f3, [0x11eb] = 0x67b0,
  [0x11ec] = 0x67d9, [0x11ed] = 0x67e2, [0x11ee] = 0x67dd, [0x11ef] = 0x67d2,
  [0x11f0] = 0x6b6a, [0x11f1] = 0x6b83, [0x11f2] = 0x6b86, [0x11f3] = 0x6bb5,
  [0x11f4] = 0x6bd2, [0x11f5] = 0x6bd7, [0x11f6] = 0x6c1f, [0x11f7] = 0x6cc9,
  [0x11f8] = 0x6d0b, [0x11f9] = 0x6d32, [0x11fa] = 0x6d2a, [0x11fb] = 0x6d41,
  [0x11fc] = 0x6d25, [0x11fd] = 0x6d0c, [0x11fe] = 0x6d31, [0x11ff] = 0x6d1e,
  [0x1200] = 0x6d17, [0x1201] = 0x6d3b, [0x1202] = 0x6d3d, [0x1203] = 0x6d3e,
  [0x1204] = 0x6d36, [0x1205] = 0x6d1b, [0x1206] = 0x6cf5, [0x1207] = 0x6d39,
  [0x1208] = 0x6d27, [0x1209] = 0x6d38, [0x120a] = 0x6d29, [0x120b] = 0x6d2e,
  [0x120c] = 0x6d35, [0x120d] = 0x6d0e, [0x120e] = 0x6d2b, [0x120f] = 0x70ab,
  [0x1210] = 0x70ba, [0x1211] = 0x70b3, [0x1212] = 0x70ac, [0x1213] = 0x70af,
  [0x1214] = 0x70ad, [0x1215] = 0x70b8, [0x1216] = 0x70ae, [0x1217] = 0x70a4,
  [0x1218] = 0x7230, [0x1219] = 0x7272, [0x121a] = 0x726f, [0x121b] = 0x7274,
  [0x121c] = 0x72e9, [0x121d] = 0x72e0, [0x121e] = 0x72e1, [0x121f] = 0x73b7,
  [0x1220] = 0x73ca, [0x1221] = 0x73bb, [0x1222] = 0x73b2, [0x1223] = 0x73cd,
  [0x1224] = 0x73c0, [0x1225] = 0x73b3, [0x1226] = 0x751a, [0x1227] = 0x752d,
  [0x1228] = 0x754f, [0x1229] = 0x754c, [0x122a] = 0x754e, [0x122b] = 0x754b,
  [0x122c] = 0x75ab, [0x122d] = 0x75a4, [0x122e] = 0x75a5, [0x122f] = 0x75a2,
  [0x1230] = 0x75a3, [0x1231] = 0x7678, [0x1232] = 0x7686, [0x1233] = 0x7687,
  [0x1234] = 0x7688, [0x1235] = 0x76c8, [0x1236] = 0x76c6, [0x1237] = 0x76c3,
  [0x1238] = 0x76c5, [0x1239] = 0x7701, [0x123a] = 0x76f9, [0x123b] = 0x76f8,
  [0x123c] = 0x7709, [0x123d] = 0x770b, [0x123e] = 0x76fe, [0x123f] = 0x76fc,
  [0x1240] = 0x7707, [0x1241] = 0x77dc, [0x1242] = 0x7802, [0x1243] = 0x7814,
  [0x1244] = 0x780c, [0x1245] = 0x780d, [0x1246] = 0x7946, [0x1247] = 0x7949,
  [0x1248] = 0x7948, [0x1249] = 0x7947, [0x124a] = 0x79b9, [0x124b] = 0x79ba,
  [0x124c] = 0x79d1, [0x124d] = 0x79d2, [0x124e] = 0x79cb, [0x124f] = 0x7a7f,
  [0x1250] = 0x7a81, [0x1251] = 0x7aff, [0x1252] = 0x7afd, [0x1253] = 0x7c7d,
  [0x1254] = 0x7d02, [0x1255] = 0x7d05, [0x1256] = 0x7d00, [0x1257] = 0x7d09,
  [0x1258] = 0x7d07, [0x1259] = 0x7d04, [0x125a] = 0x7d06, [0x125b] = 0x7f38,
  [0x125c] = 0x7f8e, [0x125d] = 0x7fbf, [0x125e] = 0x8010, [0x125f] = 0x800d,
  [0x1260] = 0x8011, [0x1261] = 0x8036, [0x1262] = 0x80d6, [0x1263] = 0x80e5,
  [0x1264] = 0x80da, [0x1265] = 0x80c3, [0x1266] = 0x80c4, [0x1267] = 0x80cc,
  [0x1268] = 0x80e1, [0x1269] = 0x80db, [0x126a] = 0x80ce, [0x126b] = 0x80de,
  [0x126c] = 0x80e4, [0x126d] = 0x80dd, [0x126e] = 0x81f4, [0x126f] = 0x8222,
  [0x1270] = 0x82e7, [0x1271] = 0x8303, [0x1272] = 0x8305, [0x1273] = 0x82e3,
  [0x1274] = 0x82db, [0x1275] = 0x82e6, [0x1276] = 0x8304, [0x1277] = 0x82e5,
  [0x1278] = 0x8302, [0x1279] = 0x8309, [0x127a] = 0x82d2, [0x127b] = 0x82d7,
  [0x127c] = 0x82f1, [0x127d] = 0x8301, [0x127e] = 0x82dc, [0x127f] = 0x82d4,
  [0x1280] = 0x82d1, [0x1281] = 0x82de, [0x1282] = 0x82d3, [0x1283] = 0x82df,
  [0x1284] = 0x82ef, [0x1285] = 0x8306, [0x1286] = 0x8650, [0x1287] = 0x8679,
  [0x1288] = 0x867b, [0x1289] = 0x867a, [0x128a] = 0x884d, [0x128b] = 0x886b,
  [0x128c] = 0x8981, [0x128d] = 0x89d4, [0x128e] = 0x8a08, [0x128f] = 0x8a02,
  [0x1290] = 0x8a03, [0x1291] = 0x8c9e, [0x1292] = 0x8ca0, [0x1293] = 0x8d74,
  [0x1294] = 0x8d73, [0x1295] = 0x8db4, [0x1296] = 0x8ecd, [0x1297] = 0x8ecc,
  [0x1298] = 0x8ff0, [0x1299] = 0x8fe6, [0x129a] = 0x8fe2, [0x129b] = 0x8fea,
  [0x129c] = 0x8fe5, [0x129d] = 0x8fed, [0x129e] = 0x8feb, [0x129f] = 0x8fe4,
  [0x12a0] = 0x8fe8, [0x12a1] = 0x90ca, [0x12a2] = 0x90ce, [0x12a3] = 0x90c1,
  [0x12a4] = 0x90c3, [0x12a5] = 0x914b, [0x12a6] = 0x914a, [0x12a7] = 0x91cd,
  [0x12a8] = 0x9582, [0x12a9] = 0x9650, [0x12aa] = 0x964b, [0x12ab] = 0x964c,
  [0x12ac] = 0x964d, [0x12ad] = 0x9762, [0x12ae] = 0x9769, [0x12af] = 0x97cb,
  [0x12b0] = 0x97ed, [0x12b1] = 0x97f3, [0x12b2] = 0x9801, [0x12b3] = 0x98a8,
  [0x12b4] = 0x98db, [0x12b5] = 0x98df, [0x12b6] = 0x9996, [0x12b7] = 0x9999,
  [0x12b8] = 0x4e58, [0x12b9] = 0x4eb3, [0x12ba] = 0x500c, [0x12bb] = 0x500d,
  [0x12bc] = 0x5023, [0x12bd] = 0x4fef, [0x12be] = 0x5026, [0x12bf] = 0x5025,
  [0x12c0] = 0x4ff8, [0x12c1] = 0x5029, [0x12c2] = 0x5016, [0x12c3] = 0x5006,
  [0x12c4] = 0x503c, [0x12c5] = 0x501f, [0x12c6] = 0x501a, [0x12c7] = 0x5012,
  [0x12c8] = 0x5011, [0x12c9] = 0x4ffa, [0x12ca] = 0x5000, [0x12cb] = 0x5014,
  [0x12cc] = 0x5028, [0x12cd] = 0x4ff1, [0x12ce] = 0x5021, [0x12cf] = 0x500b,
  [0x12d0] = 0x5019, [0x12d1] = 0x5018, [0x12d2] = 0x4ff3, [0x12d3] = 0x4fee,
  [0x12d4] = 0x502d, [0x12d5] = 0x502a, [0x12d6] = 0x4ffe, [0x12d7] = 0x502b,
  [0x12d8] = 0x5009, [0x12d9] = 0x517c, [0x12da] = 0x51a4, [0x12db] = 0x51a5,
  [0x12dc] = 0x51a2, [0x12dd] = 0x51cd, [0x12de] = 0x51cc, [0x12df] = 0x51c6,
  [0x12e0] = 0x51cb, [0x12e1] = 0x5256, [0x12e2] = 0x525c, [0x12e3] = 0x5254,
  [0x12e4] = 0x525b, [0x12e5] = 0x525d, [0x12e6] = 0x532a, [0x12e7] = 0x537f,
  [0x12e8] = 0x539f, [0x12e9] = 0x539d, [0x12ea] = 0x53df, [0x12eb] = 0x54e8,
  [0x12ec] = 0x5510, [0x12ed] = 0x5501, [0x12ee] = 0x5537, [0x12ef] = 0x54fc,
  [0x12f0] = 0x54e5, [0x12f1] = 0x54f2, [0x12f2] = 0x5506, [0x12f3] = 0x54fa,
  [0x12f4] = 0x5514, [0x12f5] = 0x54e9, [0x12f6] = 0x54ed, [0x12f7] = 0x54e1,
  [0x12f8] = 0x5509, [0x12f9] = 0x54ee, [0x12fa] = 0x54ea, [0x12fb] = 0x54e6,
  [0x12fc] = 0x5527, [0x12fd] = 0x5507, [0x12fe] = 0x54fd, [0x12ff] = 0x550f,
  [0x1300] = 0x5703, [0x1301] = 0x5704, [0x1302] = 0x57c2, [0x1303] = 0x57d4,
  [0x1304] = 0x57cb, [0x1305] = 0x57c3, [0x1306] = 0x5809, [0x1307] = 0x590f,
  [0x1308] = 0x5957, [0x1309] = 0x5958, [0x130a] = 0x595a, [0x130b] = 0x5a11,
  [0x130c] = 0x5a18, [0x130d] = 0x5a1c, [0x130e] = 0x5a1f, [0x130f] = 0x5a1b,
  [0x1310] = 0x5a13, [0x1311] = 0x59ec, [0x1312] = 0x5a20, [0x1313] = 0x5a23,
  [0x1314] = 0x5a29, [0x1315] = 0x5a25, [0x1316] = 0x5a0c, [0x1317] = 0x5a09,
  [0x1318] = 0x5b6b, [0x1319] = 0x5c58, [0x131a] = 0x5bb0, [0x131b] = 0x5bb3,
  [0x131c] = 0x5bb6, [0x131d] = 0x5bb4, [0x131e] = 0x5bae, [0x131f] = 0x5bb5,
  [0x1320] = 0x5bb9, [0x1321] = 0x5bb8, [0x1322] = 0x5c04, [0x1323] = 0x5c51,
  [0x1324] = 0x5c55, [0x1325] = 0x5c50, [0x1326] = 0x5ced, [0x1327] = 0x5cfd,
  [0x1328] = 0x5cfb, [0x1329] = 0x5cea, [0x132a] = 0x5ce8, [0x132b] = 0x5cf0,
  [0x132c] = 0x5cf6, [0x132d] = 0x5d01, [0x132e] = 0x5cf4, [0x132f] = 0x5dee,
  [0x1330] = 0x5e2d, [0x1331] = 0x5e2b, [0x1332] = 0x5eab, [0x1333] = 0x5ead,
  [0x1334] = 0x5ea7, [0x1335] = 0x5f31, [0x1336] = 0x5f92, [0x1337] = 0x5f91,
  [0x1338] = 0x5f90, [0x1339] = 0x6059, [0x133a] = 0x6063, [0x133b] = 0x6065,
  [0x133c] = 0x6050, [0x133d] = 0x6055, [0x133e] = 0x606d, [0x133f] = 0x6069,
  [0x1340] = 0x606f, [0x1341] = 0x6084, [0x1342] = 0x609f, [0x1343] = 0x609a,
  [0x1344] = 0x608d, [0x1345] = 0x6094, [0x1346] = 0x608c, [0x1347] = 0x6085,
  [0x1348] = 0x6096, [0x1349] = 0x6247, [0x134a] = 0x62f3, [0x134b] = 0x6308,
  [0x134c] = 0x62ff, [0x134d] = 0x634e, [0x134e] = 0x633e, [0x134f] = 0x632f,
  [0x1350] = 0x6355, [0x1351] = 0x6342, [0x1352] = 0x6346, [0x1353] = 0x634f,
  [0x1354] = 0x6349, [0x1355] = 0x633a, [0x1356] = 0x6350, [0x1357] = 0x633d,
  [0x1358] = 0x632a, [0x1359] = 0x632b, [0x135a] = 0x6328, [0x135b] = 0x634d,
  [0x135c] = 0x634c, [0x135d] = 0x6548, [0x135e] = 0x6549, [0x135f] = 0x6599,
  [0x1360] = 0x65c1, [0x1361] = 0x65c5, [0x1362] = 0x6642, [0x1363] = 0x6649,
  [0x1364] = 0x664f, [0x1365] = 0x6643, [0x1366] = 0x6652, [0x1367] = 0x664c,
  [0x1368] = 0x6645, [0x1369] = 0x6641, [0x136a] = 0x66f8, [0x136b] = 0x6714,
  [0x136c] = 0x6715, [0x136d] = 0x6717, [0x136e] = 0x6821, [0x136f] = 0x6838,
  [0x1370] = 0x6848, [0x1371] = 0x6846, [0x1372] = 0x6853, [0x1373] = 0x6839,
  [0x1374] = 0x6842, [0x1375] = 0x6854, [0x1376] = 0x6829, [0x1377] = 0x68b3,
  [0x1378] = 0x6817, [0x1379] = 0x684c, [0x137a] = 0x6851, [0x137b] = 0x683d,
  [0x137c] = 0x67f4, [0x137d] = 0x6850, [0x137e] = 0x6840, [0x137f] = 0x683c,
  [0x1380] = 0x6843, [0x1381] = 0x682a, [0x1382] = 0x6845, [0x1383] = 0x6813,
  [0x1384] = 0x6818, [0x1385] = 0x6841, [0x1386] = 0x6b8a, [0x1387] = 0x6b89,
  [0x1388] = 0x6bb7, [0x1389] = 0x6c23, [0x138a] = 0x6c27, [0x138b] = 0x6c28,
  [0x138c] = 0x6c26, [0x138d] = 0x6c24, [0x138e] = 0x6cf0, [0x138f] = 0x6d6a,
  [0x1390] = 0x6d95, [0x1391] = 0x6d88, [0x1392] = 0x6d87, [0x1393] = 0x6d66,
  [0x1394] = 0x6d78, [0x1395] = 0x6d77, [0x1396] = 0x6d59, [0x1397] = 0x6d93,
  [0x1398] = 0x6d6c, [0x1399] = 0x6d89, [0x139a] = 0x6d6e, [0x139b] = 0x6d5a,
  [0x139c] = 0x6d74, [0x139d] = 0x6d69, [0x139e] = 0x6d8c, [0x139f] = 0x6d8a,
  [0x13a0] = 0x6d79, [0x13a1] = 0x6d85, [0x13a2] = 0x6d65, [0x13a3] = 0x6d94,
  [0x13a4] = 0x70ca, [0x13a5] = 0x70d8, [0x13a6] = 0x70e4, [0x13a7] = 0x70d9,
  [0x13a8] = 0x70c8, [0x13a9] = 0x70cf, [0x13aa] = 0x7239, [0x13ab] = 0x7279,
  [0x13ac] = 0x72fc, [0x13ad] = 0x72f9, [0x13ae] = 0x72fd, [0x13af] = 0x72f8,
  [0x13b0] = 0x72f7, [0x13b1] = 0x7386, [0x13b2] = 0x73ed, [0x13b3] = 0x7409,
  [0x13b4] = 0x73ee, [0x13b5] = 0x73e0, [0x13b6] = 0x73ea, [0x13b7] = 0x73de,
  [0x13b8] = 0x7554, [0x13b9] = 0x755d, [0x13ba] = 0x755c, [0x13bb] = 0x755a,
  [0x13bc] = 0x7559, [0x13bd] = 0x75be, [0x13be] = 0x75c5, [0x13bf] = 0x75c7,
  [0x13c0] = 0x75b2, [0x13c1] = 0x75b3, [0x13c2] = 0x75bd, [0x13c3] = 0x75bc,
  [0x13c4] = 0x75b9, [0x13c5] = 0x75c2, [0x13c6] = 0x75b8, [0x13c7] = 0x768b,
  [0x13c8] = 0x76b0, [0x13c9] = 0x76ca, [0x13ca] = 0x76cd, [0x13cb] = 0x76ce,
  [0x13cc] = 0x7729, [0x13cd] = 0x771f, [0x13ce] = 0x7720, [0x13cf] = 0x7728,
  [0x13d0] = 0x77e9, [0x13d1] = 0x7830, [0x13d2] = 0x7827, [0x13d3] = 0x7838,
  [0x13d4] = 0x781d, [0x13d5] = 0x7834, [0x13d6] = 0x7837, [0x13d7] = 0x7825,
  [0x13d8] = 0x782d, [0x13d9] = 0x7820, [0x13da] = 0x781f, [0x13db] = 0x7832,
  [0x13dc] = 0x7955, [0x13dd] = 0x7950, [0x13de] = 0x7960, [0x13df] = 0x795f,
  [0x13e0] = 0x7956, [0x13e1] = 0x795e, [0x13e2] = 0x795d, [0x13e3] = 0x7957,
  [0x13e4] = 0x795a, [0x13e5] = 0x79e4, [0x13e6] = 0x79e3, [0x13e7] = 0x79e7,
  [0x13e8] = 0x79df, [0x13e9] = 0x79e6, [0x13ea] = 0x79e9, [0x13eb] = 0x79d8,
  [0x13ec] = 0x7a84, [0x13ed] = 0x7a88, [0x13ee] = 0x7ad9, [0x13ef] = 0x7b06,
  [0x13f0] = 0x7b11, [0x13f1] = 0x7c89, [0x13f2] = 0x7d21, [0x13f3] = 0x7d17,
  [0x13f4] = 0x7d0b, [0x13f5] = 0x7d0a, [0x13f6] = 0x7d20, [0x13f7] = 0x7d22,
  [0x13f8] = 0x7d14, [0x13f9] = 0x7d10, [0x13fa] = 0x7d15, [0x13fb] = 0x7d1a,
  [0x13fc] = 0x7d1c, [0x13fd] = 0x7d0d, [0x13fe] = 0x7d19, [0x13ff] = 0x7d1b,
  [0x1400] = 0x7f3a, [0x1401] = 0x7f5f, [0x1402] = 0x7f94, [0x1403] = 0x7fc5,
  [0x1404] = 0x7fc1, [0x1405] = 0x8006, [0x1406] = 0x8004, [0x1407] = 0x8018,
  [0x1408] = 0x8015, [0x1409] = 0x8019, [0x140a] = 0x8017, [0x140b] = 0x803d,
  [0x140c] = 0x803f, [0x140d] = 0x80f1, [0x140e] = 0x8102, [0x140f] = 0x80f0,
  [0x1410] = 0x8105, [0x1411] = 0x80ed, [0x1412] = 0x80f4, [0x1413] = 0x8106,
  [0x1414] = 0x80f8, [0x1415] = 0x80f3, [0x1416] = 0x8108, [0x1417] = 0x80fd,
  [0x1418] = 0x810a, [0x1419] = 0x80fc, [0x141a] = 0x80ef, [0x141b] = 0x81ed,
  [0x141c] = 0x81ec, [0x141d] = 0x8200, [0x141e] = 0x8210, [0x141f] = 0x822a,
  [0x1420] = 0x822b, [0x1421] = 0x8228, [0x1422] = 0x822c, [0x1423] = 0x82bb,
  [0x1424] = 0x832b, [0x1425] = 0x8352, [0x1426] = 0x8354, [0x1427] = 0x834a,
  [0x1428] = 0x8338, [0x1429] = 0x8350, [0x142a] = 0x8349, [0x142b] = 0x8335,
  [0x142c] = 0x8334, [0x142d] = 0x834f, [0x142e] = 0x8332, [0x142f] = 0x8339,
  [0x1430] = 0x8336, [0x1431] = 0x8317, [0x1432] = 0x8340, [0x1433] = 0x8331,
  [0x1434] = 0x8328, [0x1435] = 0x8343, [0x1436] = 0x8654, [0x1437] = 0x868a,
  [0x1438] = 0x86aa, [0x1439] = 0x8693, [0x143a] = 0x86a4, [0x143b] = 0x86a9,
  [0x143c] = 0x868c, [0x143d] = 0x86a3, [0x143e] = 0x869c, [0x143f] = 0x8870,
  [0x1440] = 0x8877, [0x1441] = 0x8881, [0x1442] = 0x8882, [0x1443] = 0x887d,
  [0x1444] = 0x8879, [0x1445] = 0x8a18, [0x1446] = 0x8a10, [0x1447] = 0x8a0e,
  [0x1448] = 0x8a0c, [0x1449] = 0x8a15, [0x144a] = 0x8a0a, [0x144b] = 0x8a17,
  [0x144c] = 0x8a13, [0x144d] = 0x8a16, [0x144e] = 0x8a0f, [0x144f] = 0x8a11,
  [0x1450] = 0x8c48, [0x1451] = 0x8c7a, [0x1452] = 0x8c79, [0x1453] = 0x8ca1,
  [0x1454] = 0x8ca2, [0x1455] = 0x8d77, [0x1456] = 0x8eac, [0x1457] = 0x8ed2,
  [0x1458] = 0x8ed4, [0x1459] = 0x8ecf, [0x145a] = 0x8fb1, [0x145b] = 0x9001,
  [0x145c] = 0x9006, [0x145d] = 0x8ff7, [0x145e] = 0x9000, [0x145f] = 0x8ffa,
  [0x1460] = 0x8ff4, [0x1461] = 0x9003, [0x1462] = 0x8ffd, [0x1463] = 0x9005,
  [0x1464] = 0x8ff8, [0x1465] = 0x9095, [0x1466] = 0x90e1, [0x1467] = 0x90dd,
  [0x1468] = 0x90e2, [0x1469] = 0x9152, [0x146a] = 0x914d, [0x146b] = 0x914c,
  [0x146c] = 0x91d8, [0x146d] = 0x91dd, [0x146e] = 0x91d7, [0x146f] = 0x91dc,
  [0x1470] = 0x91d9, [0x1471] = 0x9583, [0x1472] = 0x9662, [0x1473] = 0x9663,
  [0x1474] = 0x9661, [0x1475] = 0x965b, [0x1476] = 0x965d, [0x1477] = 0x9664,
  [0x1478] = 0x9658, [0x1479] = 0x965e, [0x147a] = 0x96bb, [0x147b] = 0x98e2,
  [0x147c] = 0x99ac, [0x147d] = 0x9aa8, [0x147e] = 0x9ad8, [0x147f] = 0x9b25,
  [0x1480] = 0x9b32, [0x1481] = 0x9b3c, [0x1482] = 0x4e7e, [0x1483] = 0x507a,
  [0x1484] = 0x507d, [0x1485] = 0x505c, [0x1486] = 0x5047, [0x1487] = 0x5043,
  [0x1488] = 0x504c, [0x1489] = 0x505a, [0x148a] = 0x5049, [0x148b] = 0x5065,
  [0x148c] = 0x5076, [0x148d] = 0x504e, [0x148e] = 0x5055, [0x148f] = 0x5075,
  [0x1490] = 0x5074, [0x1491] = 0x5077, [0x1492] = 0x504f, [0x1493] = 0x500f,
  [0x1494] = 0x506f, [0x1495] = 0x506d, [0x1496] = 0x515c, [0x1497] = 0x5195,
  [0x1498] = 0x51f0, [0x1499] = 0x526a, [0x149a] = 0x526f, [0x149b] = 0x52d2,
  [0x149c] = 0x52d9, [0x149d] = 0x52d8, [0x149e] = 0x52d5, [0x149f] = 0x5310,
  [0x14a0] = 0x530f, [0x14a1] = 0x5319, [0x14a2] = 0x533f, [0x14a3] = 0x5340,
  [0x14a4] = 0x533e, [0x14a5] = 0x53c3, [0x14a6] = 0x66fc, [0x14a7] = 0x5546,
  [0x14a8] = 0x556a, [0x14a9] = 0x5566, [0x14aa] = 0x5544, [0x14ab] = 0x555e,
  [0x14ac] = 0x5561, [0x14ad] = 0x5543, [0x14ae] = 0x554a, [0x14af] = 0x5531,
  [0x14b0] = 0x5556, [0x14b1] = 0x554f, [0x14b2] = 0x5555, [0x14b3] = 0x552f,
  [0x14b4] = 0x5564, [0x14b5] = 0x5538, [0x14b6] = 0x552e, [0x14b7] = 0x555c,
  [0x14b8] = 0x552c, [0x14b9] = 0x5563, [0x14ba] = 0x5533, [0x14bb] = 0x5541,
  [0x14bc] = 0x5557, [0x14bd] = 0x5708, [0x14be] = 0x570b, [0x14bf] = 0x5709,
  [0x14c0] = 0x57df, [0x14c1] = 0x5805, [0x14c2] = 0x580a, [0x14c3] = 0x5806,
  [0x14c4] = 0x57e0, [0x14c5] = 0x57e4, [0x14c6] = 0x57fa, [0x14c7] = 0x5802,
  [0x14c8] = 0x5835, [0x14c9] = 0x57f7, [0x14ca] = 0x57f9, [0x14cb] = 0x5920,
  [0x14cc] = 0x5962, [0x14cd] = 0x5a36, [0x14ce] = 0x5a41, [0x14cf] = 0x5a49,
  [0x14d0] = 0x5a66, [0x14d1] = 0x5a6a, [0x14d2] = 0x5a40, [0x14d3] = 0x5a3c,
  [0x14d4] = 0x5a62, [0x14d5] = 0x5a5a, [0x14d6] = 0x5a46, [0x14d7] = 0x5a4a,
  [0x14d8] = 0x5b70, [0x14d9] = 0x5bc7, [0x14da] = 0x5bc5, [0x14db] = 0x5bc4,
  [0x14dc] = 0x5bc2, [0x14dd] = 0x5bbf, [0x14de] = 0x5bc6, [0x14df] = 0x5c09,
  [0x14e0] = 0x5c08, [0x14e1] = 0x5c07, [0x14e2] = 0x5c60, [0x14e3] = 0x5c5c,
  [0x14e4] = 0x5c5d, [0x14e5] = 0x5d07, [0x14e6] = 0x5d06, [0x14e7] = 0x5d0e,
  [0x14e8] = 0x5d1b, [0x14e9] = 0x5d16, [0x14ea] = 0x5d22, [0x14eb] = 0x5d11,
  [0x14ec] = 0x5d29, [0x14ed] = 0x5d14, [0x14ee] = 0x5d19, [0x14ef] = 0x5d24,
  [0x14f0] = 0x5d27, [0x14f1] = 0x5d17, [0x14f2] = 0x5de2, [0x14f3] = 0x5e38,
  [0x14f4] = 0x5e36, [0x14f5] = 0x5e33, [0x14f6] = 0x5e37, [0x14f7] = 0x5eb7,
  [0x14f8] = 0x5eb8, [0x14f9] = 0x5eb6, [0x14fa] = 0x5eb5, [0x14fb] = 0x5ebe,
  [0x14fc] = 0x5f35, [0x14fd] = 0x5f37, [0x14fe] = 0x5f57, [0x14ff] = 0x5f6c,
  [0x1500] = 0x5f69, [0x1501] = 0x5f6b, [0x1502] = 0x5f97, [0x1503] = 0x5f99,
  [0x1504] = 0x5f9e, [0x1505] = 0x5f98, [0x1506] = 0x5fa1, [0x1507] = 0x5fa0,
  [0x1508] = 0x5f9c, [0x1509] = 0x607f, [0x150a] = 0x60a3, [0x150b] = 0x6089,
  [0x150c] = 0x60a0, [0x150d] = 0x60a8, [0x150e] = 0x60cb, [0x150f] = 0x60b4,
  [0x1510] = 0x60e6, [0x1511] = 0x60bd, [0x1512] = 0x60c5, [0x1513] = 0x60bb,
  [0x1514] = 0x60b5, [0x1515] = 0x60dc, [0x1516] = 0x60bc, [0x1517] = 0x60d8,
  [0x1518] = 0x60d5, [0x1519] = 0x60c6, [0x151a] = 0x60df, [0x151b] = 0x60b8,
  [0x151c] = 0x60da, [0x151d] = 0x60c7, [0x151e] = 0x621a, [0x151f] = 0x621b,
  [0x1520] = 0x6248, [0x1521] = 0x63a0, [0x1522] = 0x63a7, [0x1523] = 0x6372,
  [0x1524] = 0x6396, [0x1525] = 0x63a2, [0x1526] = 0x63a5, [0x1527] = 0x6377,
  [0x1528] = 0x6367, [0x1529] = 0x6398, [0x152a] = 0x63aa, [0x152b] = 0x6371,
  [0x152c] = 0x63a9, [0x152d] = 0x6389, [0x152e] = 0x6383, [0x152f] = 0x639b,
  [0x1530] = 0x636b, [0x1531] = 0x63a8, [0x1532] = 0x6384, [0x1533] = 0x6388,
  [0x1534] = 0x6399, [0x1535] = 0x63a1, [0x1536] = 0x63ac, [0x1537] = 0x6392,
  [0x1538] = 0x638f, [0x1539] = 0x6380, [0x153a] = 0x637b, [0x153b] = 0x6369,
  [0x153c] = 0x6368, [0x153d] = 0x637a, [0x153e] = 0x655d, [0x153f] = 0x6556,
  [0x1540] = 0x6551, [0x1541] = 0x6559, [0x1542] = 0x6557, [0x1543] = 0x555f,
  [0x1544] = 0x654f, [0x1545] = 0x6558, [0x1546] = 0x6555, [0x1547] = 0x6554,
  [0x1548] = 0x659c, [0x1549] = 0x659b, [0x154a] = 0x65ac, [0x154b] = 0x65cf,
  [0x154c] = 0x65cb, [0x154d] = 0x65cc, [0x154e] = 0x65ce, [0x154f] = 0x665d,
  [0x1550] = 0x665a, [0x1551] = 0x6664, [0x1552] = 0x6668, [0x1553] = 0x6666,
  [0x1554] = 0x665e, [0x1555] = 0x66f9, [0x1556] = 0x52d7, [0x1557] = 0x671b,
  [0x1558] = 0x6881, [0x1559] = 0x68af, [0x155a] = 0x68a2, [0x155b] = 0x6893,
  [0x155c] = 0x68b5, [0x155d] = 0x687f, [0x155e] = 0x6876, [0x155f] = 0x68b1,
  [0x1560] = 0x68a7, [0x1561] = 0x6897, [0x1562] = 0x68b0, [0x1563] = 0x6883,
  [0x1564] = 0x68c4, [0x1565] = 0x68ad, [0x1566] = 0x6886, [0x1567] = 0x6885,
  [0x1568] = 0x6894, [0x1569] = 0x689d, [0x156a] = 0x68a8, [0x156b] = 0x689f,
  [0x156c] = 0x68a1, [0x156d] = 0x6882, [0x156e] = 0x6b32, [0x156f] = 0x6bba,
  [0x1570] = 0x6beb, [0x1571] = 0x6bec, [0x1572] = 0x6c2b, [0x1573] = 0x6d8e,
  [0x1574] = 0x6dbc, [0x1575] = 0x6df3, [0x1576] = 0x6dd9, [0x1577] = 0x6db2,
  [0x1578] = 0x6de1, [0x1579] = 0x6dcc, [0x157a] = 0x6de4, [0x157b] = 0x6dfb,
  [0x157c] = 0x6dfa, [0x157d] = 0x6e05, [0x157e] = 0x6dc7, [0x157f] = 0x6dcb,
  [0x1580] = 0x6daf, [0x1581] = 0x6dd1, [0x1582] = 0x6dae, [0x1583] = 0x6dde,
  [0x1584] = 0x6df9, [0x1585] = 0x6db8, [0x1586] = 0x6df7, [0x1587] = 0x6df5,
  [0x1588] = 0x6dc5, [0x1589] = 0x6dd2, [0x158a] = 0x6e1a, [0x158b] = 0x6db5,
  [0x158c] = 0x6dda, [0x158d] = 0x6deb, [0x158e] = 0x6dd8, [0x158f] = 0x6dea,
  [0x1590] = 0x6df1, [0x1591] = 0x6dee, [0x1592] = 0x6de8, [0x1593] = 0x6dc6,
  [0x1594] = 0x6dc4, [0x1595] = 0x6daa, [0x1596] = 0x6dec, [0x1597] = 0x6dbf,
  [0x1598] = 0x6de6, [0x1599] = 0x70f9, [0x159a] = 0x7109, [0x159b] = 0x710a,
  [0x159c] = 0x70fd, [0x159d] = 0x70ef, [0x159e] = 0x723d, [0x159f] = 0x727d,
  [0x15a0] = 0x7281, [0x15a1] = 0x731c, [0x15a2] = 0x731b, [0x15a3] = 0x7316,
  [0x15a4] = 0x7313, [0x15a5] = 0x7319, [0x15a6] = 0x7387, [0x15a7] = 0x7405,
  [0x15a8] = 0x740a, [0x15a9] = 0x7403, [0x15aa] = 0x7406, [0x15ab] = 0x73fe,
  [0x15ac] = 0x740d, [0x15ad] = 0x74e0, [0x15ae] = 0x74f6, [0x15af] = 0x74f7,
  [0x15b0] = 0x751c, [0x15b1] = 0x7522, [0x15b2] = 0x7565, [0x15b3] = 0x7566,
  [0x15b4] = 0x7562, [0x15b5] = 0x7570, [0x15b6] = 0x758f, [0x15b7] = 0x75d4,
  [0x15b8] = 0x75d5, [0x15b9] = 0x75b5, [0x15ba] = 0x75ca, [0x15bb] = 0x75cd,
  [0x15bc] = 0x768e, [0x15bd] = 0x76d4, [0x15be] = 0x76d2, [0x15bf] = 0x76db,
  [0x15c0] = 0x7737, [0x15c1] = 0x773e, [0x15c2] = 0x773c, [0x15c3] = 0x7736,
  [0x15c4] = 0x7738, [0x15c5] = 0x773a, [0x15c6] = 0x786b, [0x15c7] = 0x7843,
  [0x15c8] = 0x784e, [0x15c9] = 0x7965, [0x15ca] = 0x7968, [0x15cb] = 0x796d,
  [0x15cc] = 0x79fb, [0x15cd] = 0x7a92, [0x15ce] = 0x7a95, [0x15cf] = 0x7b20,
  [0x15d0] = 0x7b28, [0x15d1] = 0x7b1b, [0x15d2] = 0x7b2c, [0x15d3] = 0x7b26,
  [0x15d4] = 0x7b19, [0x15d5] = 0x7b1e, [0x15d6] = 0x7b2e, [0x15d7] = 0x7c92,
  [0x15d8] = 0x7c97, [0x15d9] = 0x7c95, [0x15da] = 0x7d46, [0x15db] = 0x7d43,
  [0x15dc] = 0x7d71, [0x15dd] = 0x7d2e, [0x15de] = 0x7d39, [0x15df] = 0x7d3c,
  [0x15e0] = 0x7d40, [0x15e1] = 0x7d30, [0x15e2] = 0x7d33, [0x15e3] = 0x7d44,
  [0x15e4] = 0x7d2f, [0x15e5] = 0x7d42, [0x15e6] = 0x7d32, [0x15e7] = 0x7d31,
  [0x15e8] = 0x7f3d, [0x15e9] = 0x7f9e, [0x15ea] = 0x7f9a, [0x15eb] = 0x7fcc,
  [0x15ec] = 0x7fce, [0x15ed] = 0x7fd2, [0x15ee] = 0x801c, [0x15ef] = 0x804a,
  [0x15f0] = 0x8046, [0x15f1] = 0x812f, [0x15f2] = 0x8116, [0x15f3] = 0x8123,
  [0x15f4] = 0x812b, [0x15f5] = 0x8129, [0x15f6] = 0x8130, [0x15f7] = 0x8124,
  [0x15f8] = 0x8202, [0x15f9] = 0x8235, [0x15fa] = 0x8237, [0x15fb] = 0x8236,
  [0x15fc] = 0x8239, [0x15fd] = 0x838e, [0x15fe] = 0x839e, [0x15ff] = 0x8398,
  [0x1600] = 0x8378, [0x1601] = 0x83a2, [0x1602] = 0x8396, [0x1603] = 0x83bd,
  [0x1604] = 0x83ab, [0x1605] = 0x8392, [0x1606] = 0x838a, [0x1607] = 0x8393,
  [0x1608] = 0x8389, [0x1609] = 0x83a0, [0x160a] = 0x8377, [0x160b] = 0x837b,
  [0x160c] = 0x837c, [0x160d] = 0x8386, [0x160e] = 0x83a7, [0x160f] = 0x8655,
  [0x1610] = 0x5f6a, [0x1611] = 0x86c7, [0x1612] = 0x86c0, [0x1613] = 0x86b6,
  [0x1614] = 0x86c4, [0x1615] = 0x86b5, [0x1616] = 0x86c6, [0x1617] = 0x86cb,
  [0x1618] = 0x86b1, [0x1619] = 0x86af, [0x161a] = 0x86c9, [0x161b] = 0x8853,
  [0x161c] = 0x889e, [0x161d] = 0x8888, [0x161e] = 0x88ab, [0x161f] = 0x8892,
  [0x1620] = 0x8896, [0x1621] = 0x888d, [0x1622] = 0x888b, [0x1623] = 0x8993,
  [0x1624] = 0x898f, [0x1625] = 0x8a2a, [0x1626] = 0x8a1d, [0x1627] = 0x8a23,
  [0x1628] = 0x8a25, [0x1629] = 0x8a31, [0x162a] = 0x8a2d, [0x162b] = 0x8a1f,
  [0x162c] = 0x8a1b, [0x162d] = 0x8a22, [0x162e] = 0x8c49, [0x162f] = 0x8c5a,
  [0x1630] = 0x8ca9, [0x1631] = 0x8cac, [0x1632] = 0x8cab, [0x1633] = 0x8ca8,
  [0x1634] = 0x8caa, [0x1635] = 0x8ca7, [0x1636] = 0x8d67, [0x1637] = 0x8d66,
  [0x1638] = 0x8dbe, [0x1639] = 0x8dba, [0x163a] = 0x8edb, [0x163b] = 0x8edf,
  [0x163c] = 0x9019, [0x163d] = 0x900d, [0x163e] = 0x901a, [0x163f] = 0x9017,
  [0x1640] = 0x9023, [0x1641] = 0x901f, [0x1642] = 0x901d, [0x1643] = 0x9010,
  [0x1644] = 0x9015, [0x1645] = 0x901e, [0x1646] = 0x9020, [0x1647] = 0x900f,
  [0x1648] = 0x9022, [0x1649] = 0x9016, [0x164a] = 0x901b, [0x164b] = 0x9014,
  [0x164c] = 0x90e8, [0x164d] = 0x90ed, [0x164e] = 0x90fd, [0x164f] = 0x9157,
  [0x1650] = 0x91ce, [0x1651] = 0x91f5, [0x1652] = 0x91e6, [0x1653] = 0x91e3,
  [0x1654] = 0x91e7, [0x1655] = 0x91ed, [0x1656] = 0x91e9, [0x1657] = 0x9589,
  [0x1658] = 0x966a, [0x1659] = 0x9675, [0x165a] = 0x9673, [0x165b] = 0x9678,
  [0x165c] = 0x9670, [0x165d] = 0x9674, [0x165e] = 0x9676, [0x165f] = 0x9677,
  [0x1660] = 0x966c, [0x1661] = 0x96c0, [0x1662] = 0x96ea, [0x1663] = 0x96e9,
  [0x1664] = 0x7ae0, [0x1665] = 0x7adf, [0x1666] = 0x9802, [0x1667] = 0x9803,
  [0x1668] = 0x9b5a, [0x1669] = 0x9ce5, [0x166a] = 0x9e75, [0x166b] = 0x9e7f,
  [0x166c] = 0x9ea5, [0x166d] = 0x9ebb, [0x166e] = 0x50a2, [0x166f] = 0x508d,
  [0x1670] = 0x5085, [0x1671] = 0x5099, [0x1672] = 0x5091, [0x1673] = 0x5080,
  [0x1674] = 0x5096, [0x1675] = 0x5098, [0x1676] = 0x509a, [0x1677] = 0x6700,
  [0x1678] = 0x51f1, [0x1679] = 0x5272, [0x167a] = 0x5274, [0x167b] = 0x5275,
  [0x167c] = 0x5269, [0x167d] = 0x52de, [0x167e] = 0x52dd, [0x167f] = 0x52db,
  [0x1680] = 0x535a, [0x1681] = 0x53a5, [0x1682] = 0x557b, [0x1683] = 0x5580,
  [0x1684] = 0x55a7, [0x1685] = 0x557c, [0x1686] = 0x558a, [0x1687] = 0x559d,
  [0x1688] = 0x5598, [0x1689] = 0x5582, [0x168a] = 0x559c, [0x168b] = 0x55aa,
  [0x168c] = 0x5594, [0x168d] = 0x5587, [0x168e] = 0x558b, [0x168f] = 0x5583,
  [0x1690] = 0x55b3, [0x1691] = 0x55ae, [0x1692] = 0x559f, [0x1693] = 0x553e,
  [0x1694] = 0x55b2, [0x1695] = 0x559a, [0x1696] = 0x55bb, [0x1697] = 0x55ac,
  [0x1698] = 0x55b1, [0x1699] = 0x557e, [0x169a] = 0x5589, [0x169b] = 0x55ab,
  [0x169c] = 0x5599, [0x169d] = 0x570d, [0x169e] = 0x582f, [0x169f] = 0x582a,
  [0x16a0] = 0x5834, [0x16a1] = 0x5824, [0x16a2] = 0x5830, [0x16a3] = 0x5831,
  [0x16a4] = 0x5821, [0x16a5] = 0x581d, [0x16a6] = 0x5820, [0x16a7] = 0x58f9,
  [0x16a8] = 0x58fa, [0x16a9] = 0x5960, [0x16aa] = 0x5a77, [0x16ab] = 0x5a9a,
  [0x16ac] = 0x5a7f, [0x16ad] = 0x5a92, [0x16ae] = 0x5a9b, [0x16af] = 0x5aa7,
  [0x16b0] = 0x5b73, [0x16b1] = 0x5b71, [0x16b2] = 0x5bd2, [0x16b3] = 0x5bcc,
  [0x16b4] = 0x5bd3, [0x16b5] = 0x5bd0, [0x16b6] = 0x5c0a, [0x16b7] = 0x5c0b,
  [0x16b8] = 0x5c31, [0x16b9] = 0x5d4c, [0x16ba] = 0x5d50, [0x16bb] = 0x5d34,
  [0x16bc] = 0x5d47, [0x16bd] = 0x5dfd, [0x16be] = 0x5e45, [0x16bf] = 0x5e3d,
  [0x16c0] = 0x5e40, [0x16c1] = 0x5e43, [0x16c2] = 0x5e7e, [0x16c3] = 0x5eca,
  [0x16c4] = 0x5ec1, [0x16c5] = 0x5ec2, [0x16c6] = 0x5ec4, [0x16c7] = 0x5f3c,
  [0x16c8] = 0x5f6d, [0x16c9] = 0x5fa9, [0x16ca] = 0x5faa, [0x16cb] = 0x5fa8,
  [0x16cc] = 0x60d1, [0x16cd] = 0x60e1, [0x16ce] = 0x60b2, [0x16cf] = 0x60b6,
  [0x16d0] = 0x60e0, [0x16d1] = 0x611c, [0x16d2] = 0x6123, [0x16d3] = 0x60fa,
  [0x16d4] = 0x6115, [0x16d5] = 0x60f0, [0x16d6] = 0x60fb, [0x16d7] = 0x60f4,
  [0x16d8] = 0x6168, [0x16d9] = 0x60f1, [0x16da] = 0x610e, [0x16db] = 0x60f6,
  [0x16dc] = 0x6109, [0x16dd] = 0x6100, [0x16de] = 0x6112, [0x16df] = 0x621f,
  [0x16e0] = 0x6249, [0x16e1] = 0x63a3, [0x16e2] = 0x638c, [0x16e3] = 0x63cf,
  [0x16e4] = 0x63c0, [0x16e5] = 0x63e9, [0x16e6] = 0x63c9, [0x16e7] = 0x63c6,
  [0x16e8] = 0x63cd, [0x16e9] = 0x63d2, [0x16ea] = 0x63e3, [0x16eb] = 0x63d0,
  [0x16ec] = 0x63e1, [0x16ed] = 0x63d6, [0x16ee] = 0x63ed, [0x16ef] = 0x63ee,
  [0x16f0] = 0x6376, [0x16f1] = 0x63f4, [0x16f2] = 0x63ea, [0x16f3] = 0x63db,
  [0x16f4] = 0x6452, [0x16f5] = 0x63da, [0x16f6] = 0x63f9, [0x16f7] = 0x655e,
  [0x16f8] = 0x6566, [0x16f9] = 0x6562, [0x16fa] = 0x6563, [0x16fb] = 0x6591,
  [0x16fc] = 0x6590, [0x16fd] = 0x65af, [0x16fe] = 0x666e, [0x16ff] = 0x6670,
  [0x1700] = 0x6674, [0x1701] = 0x6676, [0x1702] = 0x666f, [0x1703] = 0x6691,
  [0x1704] = 0x667a, [0x1705] = 0x667e, [0x1706] = 0x6677, [0x1707] = 0x66fe,
  [0x1708] = 0x66ff, [0x1709] = 0x671f, [0x170a] = 0x671d, [0x170b] = 0x68fa,
  [0x170c] = 0x68d5, [0x170d] = 0x68e0, [0x170e] = 0x68d8, [0x170f] = 0x68d7,
  [0x1710] = 0x6905, [0x1711] = 0x68df, [0x1712] = 0x68f5, [0x1713] = 0x68ee,
  [0x1714] = 0x68e7, [0x1715] = 0x68f9, [0x1716] = 0x68d2, [0x1717] = 0x68f2,
  [0x1718] = 0x68e3, [0x1719] = 0x68cb, [0x171a] = 0x68cd, [0x171b] = 0x690d,
  [0x171c] = 0x6912, [0x171d] = 0x690e, [0x171e] = 0x68c9, [0x171f] = 0x68da,
  [0x1720] = 0x696e, [0x1721] = 0x68fb, [0x1722] = 0x6b3e, [0x1723] = 0x6b3a,
  [0x1724] = 0x6b3d, [0x1725] = 0x6b98, [0x1726] = 0x6b96, [0x1727] = 0x6bbc,
  [0x1728] = 0x6bef, [0x1729] = 0x6c2e, [0x172a] = 0x6c2f, [0x172b] = 0x6c2c,
  [0x172c] = 0x6e2f, [0x172d] = 0x6e38, [0x172e] = 0x6e54, [0x172f] = 0x6e21,
  [0x1730] = 0x6e32, [0x1731] = 0x6e67, [0x1732] = 0x6e4a, [0x1733] = 0x6e20,
  [0x1734] = 0x6e25, [0x1735] = 0x6e23, [0x1736] = 0x6e1b, [0x1737] = 0x6e5b,
  [0x1738] = 0x6e58, [0x1739] = 0x6e24, [0x173a] = 0x6e56, [0x173b] = 0x6e6e,
  [0x173c] = 0x6e2d, [0x173d] = 0x6e26, [0x173e] = 0x6e6f, [0x173f] = 0x6e34,
  [0x1740] = 0x6e4d, [0x1741] = 0x6e3a, [0x1742] = 0x6e2c, [0x1743] = 0x6e43,
  [0x1744] = 0x6e1d, [0x1745] = 0x6e3e, [0x1746] = 0x6ecb, [0x1747] = 0x6e89,
  [0x1748] = 0x6e19, [0x1749] = 0x6e4e, [0x174a] = 0x6e63, [0x174b] = 0x6e44,
  [0x174c] = 0x6e72, [0x174d] = 0x6e69, [0x174e] = 0x6e5f, [0x174f] = 0x7119,
  [0x1750] = 0x711a, [0x1751] = 0x7126, [0x1752] = 0x7130, [0x1753] = 0x7121,
  [0x1754] = 0x7136, [0x1755] = 0x716e, [0x1756] = 0x711c, [0x1757] = 0x724c,
  [0x1758] = 0x7284, [0x1759] = 0x7280, [0x175a] = 0x7336, [0x175b] = 0x7325,
  [0x175c] = 0x7334, [0x175d] = 0x7329, [0x175e] = 0x743a, [0x175f] = 0x742a,
  [0x1760] = 0x7433, [0x1761] = 0x7422, [0x1762] = 0x7425, [0x1763] = 0x7435,
  [0x1764] = 0x7436, [0x1765] = 0x7434, [0x1766] = 0x742f, [0x1767] = 0x741b,
  [0x1768] = 0x7426, [0x1769] = 0x7428, [0x176a] = 0x7525, [0x176b] = 0x7526,
  [0x176c] = 0x756b, [0x176d] = 0x756a, [0x176e] = 0x75e2, [0x176f] = 0x75db,
  [0x1770] = 0x75e3, [0x1771] = 0x75d9, [0x1772] = 0x75d8, [0x1773] = 0x75de,
  [0x1774] = 0x75e0, [0x1775] = 0x767b, [0x1776] = 0x767c, [0x1777] = 0x7696,
  [0x1778] = 0x7693, [0x1779] = 0x76b4, [0x177a] = 0x76dc, [0x177b] = 0x774f,
  [0x177c] = 0x77ed, [0x177d] = 0x785d, [0x177e] = 0x786c, [0x177f] = 0x786f,
  [0x1780] = 0x7a0d, [0x1781] = 0x7a08, [0x1782] = 0x7a0b, [0x1783] = 0x7a05,
  [0x1784] = 0x7a00, [0x1785] = 0x7a98, [0x1786] = 0x7a97, [0x1787] = 0x7a96,
  [0x1788] = 0x7ae5, [0x1789] = 0x7ae3, [0x178a] = 0x7b49, [0x178b] = 0x7b56,
  [0x178c] = 0x7b46, [0x178d] = 0x7b50, [0x178e] = 0x7b52, [0x178f] = 0x7b54,
  [0x1790] = 0x7b4d, [0x1791] = 0x7b4b, [0x1792] = 0x7b4f, [0x1793] = 0x7b51,
  [0x1794] = 0x7c9f, [0x1795] = 0x7ca5, [0x1796] = 0x7d5e, [0x1797] = 0x7d50,
  [0x1798] = 0x7d68, [0x1799] = 0x7d55, [0x179a] = 0x7d2b, [0x179b] = 0x7d6e,
  [0x179c] = 0x7d72, [0x179d] = 0x7d61, [0x179e] = 0x7d66, [0x179f] = 0x7d62,
  [0x17a0] = 0x7d70, [0x17a1] = 0x7d73, [0x17a2] = 0x5584, [0x17a3] = 0x7fd4,
  [0x17a4] = 0x7fd5, [0x17a5] = 0x800b, [0x17a6] = 0x8052, [0x17a7] = 0x8085,
  [0x17a8] = 0x8155, [0x17a9] = 0x8154, [0x17aa] = 0x814b, [0x17ab] = 0x8151,
  [0x17ac] = 0x814e, [0x17ad] = 0x8139, [0x17ae] = 0x8146, [0x17af] = 0x813e,
  [0x17b0] = 0x814c, [0x17b1] = 0x8153, [0x17b2] = 0x8174, [0x17b3] = 0x8212,
  [0x17b4] = 0x821c, [0x17b5] = 0x83e9, [0x17b6] = 0x8403, [0x17b7] = 0x83f8,
  [0x17b8] = 0x840d, [0x17b9] = 0x83e0, [0x17ba] = 0x83c5, [0x17bb] = 0x840b,
  [0x17bc] = 0x83c1, [0x17bd] = 0x83ef, [0x17be] = 0x83f1, [0x17bf] = 0x83f4,
  [0x17c0] = 0x8457, [0x17c1] = 0x840a, [0x17c2] = 0x83f0, [0x17c3] = 0x840c,
  [0x17c4] = 0x83cc, [0x17c5] = 0x83fd, [0x17c6] = 0x83f2, [0x17c7] = 0x83ca,
  [0x17c8] = 0x8438, [0x17c9] = 0x840e, [0x17ca] = 0x8404, [0x17cb] = 0x83dc,
  [0x17cc] = 0x8407, [0x17cd] = 0x83d4, [0x17ce] = 0x83df, [0x17cf] = 0x865b,
  [0x17d0] = 0x86df, [0x17d1] = 0x86d9, [0x17d2] = 0x86ed, [0x17d3] = 0x86d4,
  [0x17d4] = 0x86db, [0x17d5] = 0x86e4, [0x17d6] = 0x86d0, [0x17d7] = 0x86de,
  [0x17d8] = 0x8857, [0x17d9] = 0x88c1, [0x17da] = 0x88c2, [0x17db] = 0x88b1,
  [0x17dc] = 0x8983, [0x17dd] = 0x8996, [0x17de] = 0x8a3b, [0x17df] = 0x8a60,
  [0x17e0] = 0x8a55, [0x17e1] = 0x8a5e, [0x17e2] = 0x8a3c, [0x17e3] = 0x8a41,
  [0x17e4] = 0x8a54, [0x17e5] = 0x8a5b, [0x17e6] = 0x8a50, [0x17e7] = 0x8a46,
  [0x17e8] = 0x8a34, [0x17e9] = 0x8a3a, [0x17ea] = 0x8a36, [0x17eb] = 0x8a56,
  [0x17ec] = 0x8c61, [0x17ed] = 0x8c82, [0x17ee] = 0x8caf, [0x17ef] = 0x8cbc,
  [0x17f0] = 0x8cb3, [0x17f1] = 0x8cbd, [0x17f2] = 0x8cc1, [0x17f3] = 0x8cbb,
  [0x17f4] = 0x8cc0, [0x17f5] = 0x8cb4, [0x17f6] = 0x8cb7, [0x17f7] = 0x8cb6,
  [0x17f8] = 0x8cbf, [0x17f9] = 0x8cb8, [0x17fa] = 0x8d8a, [0x17fb] = 0x8d85,
  [0x17fc] = 0x8d81, [0x17fd] = 0x8dce, [0x17fe] = 0x8ddd, [0x17ff] = 0x8dcb,
  [0x1800] = 0x8dda, [0x1801] = 0x8dd1, [0x1802] = 0x8dcc, [0x1803] = 0x8ddb,
  [0x1804] = 0x8dc6, [0x1805] = 0x8efb, [0x1806] = 0x8ef8, [0x1807] = 0x8efc,
  [0x1808] = 0x8f9c, [0x1809] = 0x902e, [0x180a] = 0x9035, [0x180b] = 0x9031,
  [0x180c] = 0x9038, [0x180d] = 0x9032, [0x180e] = 0x9036, [0x180f] = 0x9102,
  [0x1810] = 0x90f5, [0x1811] = 0x9109, [0x1812] = 0x90fe, [0x1813] = 0x9163,
  [0x1814] = 0x9165, [0x1815] = 0x91cf, [0x1816] = 0x9214, [0x1817] = 0x9215,
  [0x1818] = 0x9223, [0x1819] = 0x9209, [0x181a] = 0x921e, [0x181b] = 0x920d,
  [0x181c] = 0x9210, [0x181d] = 0x9207, [0x181e] = 0x9211, [0x181f] = 0x9594,
  [0x1820] = 0x958f, [0x1821] = 0x958b, [0x1822] = 0x9591, [0x1823] = 0x9593,
  [0x1824] = 0x9592, [0x1825] = 0x958e, [0x1826] = 0x968a, [0x1827] = 0x968e,
  [0x1828] = 0x968b, [0x1829] = 0x967d, [0x182a] = 0x9685, [0x182b] = 0x9686,
  [0x182c] = 0x968d, [0x182d] = 0x9672, [0x182e] = 0x9684, [0x182f] = 0x96c1,
  [0x1830] = 0x96c5, [0x1831] = 0x96c4, [0x1832] = 0x96c6, [0x1833] = 0x96c7,
  [0x1834] = 0x96ef, [0x1835] = 0x96f2, [0x1836] = 0x97cc, [0x1837] = 0x9805,
  [0x1838] = 0x9806, [0x1839] = 0x9808, [0x183a] = 0x98e7, [0x183b] = 0x98ea,
  [0x183c] = 0x98ef, [0x183d] = 0x98e9, [0x183e] = 0x98f2, [0x183f] = 0x98ed,
  [0x1840] = 0x99ae, [0x1841] = 0x99ad, [0x1842] = 0x9ec3, [0x1843] = 0x9ecd,
  [0x1844] = 0x9ed1, [0x1845] = 0x4e82, [0x1846] = 0x50ad, [0x1847] = 0x50b5,
  [0x1848] = 0x50b2, [0x1849] = 0x50b3, [0x184a] = 0x50c5, [0x184b] = 0x50be,
  [0x184c] = 0x50ac, [0x184d] = 0x50b7, [0x184e] = 0x50bb, [0x184f] = 0x50af,
  [0x1850] = 0x50c7, [0x1851] = 0x527f, [0x1852] = 0x5277, [0x1853] = 0x527d,
  [0x1854] = 0x52df, [0x1855] = 0x52e6, [0x1856] = 0x52e4, [0x1857] = 0x52e2,
  [0x1858] = 0x52e3, [0x1859] = 0x532f, [0x185a] = 0x55df, [0x185b] = 0x55e8,
  [0x185c] = 0x55d3, [0x185d] = 0x55e6, [0x185e] = 0x55ce, [0x185f] = 0x55dc,
  [0x1860] = 0x55c7, [0x1861] = 0x55d1, [0x1862] = 0x55e3, [0x1863] = 0x55e4,
  [0x1864] = 0x55ef, [0x1865] = 0x55da, [0x1866] = 0x55e1, [0x1867] = 0x55c5,
  [0x1868] = 0x55c6, [0x1869] = 0x55e5, [0x186a] = 0x55c9, [0x186b] = 0x5712,
  [0x186c] = 0x5713, [0x186d] = 0x585e, [0x186e] = 0x5851, [0x186f] = 0x5858,
  [0x1870] = 0x5857, [0x1871] = 0x585a, [0x1872] = 0x5854, [0x1873] = 0x586b,
  [0x1874] = 0x584c, [0x1875] = 0x586d, [0x1876] = 0x584a, [0x1877] = 0x5862,
  [0x1878] = 0x5852, [0x1879] = 0x584b, [0x187a] = 0x5967, [0x187b] = 0x5ac1,
  [0x187c] = 0x5ac9, [0x187d] = 0x5acc, [0x187e] = 0x5abe, [0x187f] = 0x5abd,
  [0x1880] = 0x5abc, [0x1881] = 0x5ab3, [0x1882] = 0x5ac2, [0x1883] = 0x5ab2,
  [0x1884] = 0x5d69, [0x1885] = 0x5d6f, [0x1886] = 0x5e4c, [0x1887] = 0x5e79,
  [0x1888] = 0x5ec9, [0x1889] = 0x5ec8, [0x188a] = 0x5f12, [0x188b] = 0x5f59,
  [0x188c] = 0x5fac, [0x188d] = 0x5fae, [0x188e] = 0x611a, [0x188f] = 0x610f,
  [0x1890] = 0x6148, [0x1891] = 0x611f, [0x1892] = 0x60f3, [0x1893] = 0x611b,
  [0x1894] = 0x60f9, [0x1895] = 0x6101, [0x1896] = 0x6108, [0x1897] = 0x614e,
  [0x1898] = 0x614c, [0x1899] = 0x6144, [0x189a] = 0x614d, [0x189b] = 0x613e,
  [0x189c] = 0x6134, [0x189d] = 0x6127, [0x189e] = 0x610d, [0x189f] = 0x6106,
  [0x18a0] = 0x6137, [0x18a1] = 0x6221, [0x18a2] = 0x6222, [0x18a3] = 0x6413,
  [0x18a4] = 0x643e, [0x18a5] = 0x641e, [0x18a6] = 0x642a, [0x18a7] = 0x642d,
  [0x18a8] = 0x643d, [0x18a9] = 0x642c, [0x18aa] = 0x640f, [0x18ab] = 0x641c,
  [0x18ac] = 0x6414, [0x18ad] = 0x640d, [0x18ae] = 0x6436, [0x18af] = 0x6416,
  [0x18b0] = 0x6417, [0x18b1] = 0x6406, [0x18b2] = 0x656c, [0x18b3] = 0x659f,
  [0x18b4] = 0x65b0, [0x18b5] = 0x6697, [0x18b6] = 0x6689, [0x18b7] = 0x6687,
  [0x18b8] = 0x6688, [0x18b9] = 0x6696, [0x18ba] = 0x6684, [0x18bb] = 0x6698,
  [0x18bc] = 0x668d, [0x18bd] = 0x6703, [0x18be] = 0x6994, [0x18bf] = 0x696d,
  [0x18c0] = 0x695a, [0x18c1] = 0x6977, [0x18c2] = 0x6960, [0x18c3] = 0x6954,
  [0x18c4] = 0x6975, [0x18c5] = 0x6930, [0x18c6] = 0x6982, [0x18c7] = 0x694a,
  [0x18c8] = 0x6968, [0x18c9] = 0x696b, [0x18ca] = 0x695e, [0x18cb] = 0x6953,
  [0x18cc] = 0x6979, [0x18cd] = 0x6986, [0x18ce] = 0x695d, [0x18cf] = 0x6963,
  [0x18d0] = 0x695b, [0x18d1] = 0x6b47, [0x18d2] = 0x6b72, [0x18d3] = 0x6bc0,
  [0x18d4] = 0x6bbf, [0x18d5] = 0x6bd3, [0x18d6] = 0x6bfd, [0x18d7] = 0x6ea2,
  [0x18d8] = 0x6eaf, [0x18d9] = 0x6ed3, [0x18da] = 0x6eb6, [0x18db] = 0x6ec2,
  [0x18dc] = 0x6e90, [0x18dd] = 0x6e9d, [0x18de] = 0x6ec7, [0x18df] = 0x6ec5,
  [0x18e0] = 0x6ea5, [0x18e1] = 0x6e98, [0x18e2] = 0x6ebc, [0x18e3] = 0x6eba,
  [0x18e4] = 0x6eab, [0x18e5] = 0x6ed1, [0x18e6] = 0x6e96, [0x18e7] = 0x6e9c,
  [0x18e8] = 0x6ec4, [0x18e9] = 0x6ed4, [0x18ea] = 0x6eaa, [0x18eb] = 0x6ea7,
  [0x18ec] = 0x6eb4, [0x18ed] = 0x714e, [0x18ee] = 0x7159, [0x18ef] = 0x7169,
  [0x18f0] = 0x7164, [0x18f1] = 0x7149, [0x18f2] = 0x7167, [0x18f3] = 0x715c,
  [0x18f4] = 0x716c, [0x18f5] = 0x7166, [0x18f6] = 0x714c, [0x18f7] = 0x7165,
  [0x18f8] = 0x715e, [0x18f9] = 0x7146, [0x18fa] = 0x7168, [0x18fb] = 0x7156,
  [0x18fc] = 0x723a, [0x18fd] = 0x7252, [0x18fe] = 0x7337, [0x18ff] = 0x7345,
  [0x1900] = 0x733f, [0x1901] = 0x733e, [0x1902] = 0x746f, [0x1903] = 0x745a,
  [0x1904] = 0x7455, [0x1905] = 0x745f, [0x1906] = 0x745e, [0x1907] = 0x7441,
  [0x1908] = 0x743f, [0x1909] = 0x7459, [0x190a] = 0x745b, [0x190b] = 0x745c,
  [0x190c] = 0x7576, [0x190d] = 0x7578, [0x190e] = 0x7600, [0x190f] = 0x75f0,
  [0x1910] = 0x7601, [0x1911] = 0x75f2, [0x1912] = 0x75f1, [0x1913] = 0x75fa,
  [0x1914] = 0x75ff, [0x1915] = 0x75f4, [0x1916] = 0x75f3, [0x1917] = 0x76de,
  [0x1918] = 0x76df, [0x1919] = 0x775b, [0x191a] = 0x776b, [0x191b] = 0x7766,
  [0x191c] = 0x775e, [0x191d] = 0x7763, [0x191e] = 0x7779, [0x191f] = 0x776a,
  [0x1920] = 0x776c, [0x1921] = 0x775c, [0x1922] = 0x7765, [0x1923] = 0x7768,
  [0x1924] = 0x7762, [0x1925] = 0x77ee, [0x1926] = 0x788e, [0x1927] = 0x78b0,
  [0x1928] = 0x7897, [0x1929] = 0x7898, [0x192a] = 0x788c, [0x192b] = 0x7889,
  [0x192c] = 0x787c, [0x192d] = 0x7891, [0x192e] = 0x7893, [0x192f] = 0x787f,
  [0x1930] = 0x797a, [0x1931] = 0x797f, [0x1932] = 0x7981, [0x1933] = 0x842c,
  [0x1934] = 0x79bd, [0x1935] = 0x7a1c, [0x1936] = 0x7a1a, [0x1937] = 0x7a20,
  [0x1938] = 0x7a14, [0x1939] = 0x7a1f, [0x193a] = 0x7a1e, [0x193b] = 0x7a9f,
  [0x193c] = 0x7aa0, [0x193d] = 0x7b77, [0x193e] = 0x7bc0, [0x193f] = 0x7b60,
  [0x1940] = 0x7b6e, [0x1941] = 0x7b67, [0x1942] = 0x7cb1, [0x1943] = 0x7cb3,
  [0x1944] = 0x7cb5, [0x1945] = 0x7d93, [0x1946] = 0x7d79, [0x1947] = 0x7d91,
  [0x1948] = 0x7d81, [0x1949] = 0x7d8f, [0x194a] = 0x7d5b, [0x194b] = 0x7f6e,
  [0x194c] = 0x7f69, [0x194d] = 0x7f6a, [0x194e] = 0x7f72, [0x194f] = 0x7fa9,
  [0x1950] = 0x7fa8, [0x1951] = 0x7fa4, [0x1952] = 0x8056, [0x1953] = 0x8058,
  [0x1954] = 0x8086, [0x1955] = 0x8084, [0x1956] = 0x8171, [0x1957] = 0x8170,
  [0x1958] = 0x8178, [0x1959] = 0x8165, [0x195a] = 0x816e, [0x195b] = 0x8173,
  [0x195c] = 0x816b, [0x195d] = 0x8179, [0x195e] = 0x817a, [0x195f] = 0x8166,
  [0x1960] = 0x8205, [0x1961] = 0x8247, [0x1962] = 0x8482, [0x1963] = 0x8477,
  [0x1964] = 0x843d, [0x1965] = 0x8431, [0x1966] = 0x8475, [0x1967] = 0x8466,
  [0x1968] = 0x846b, [0x1969] = 0x8449, [0x196a] = 0x846c, [0x196b] = 0x845b,
  [0x196c] = 0x843c, [0x196d] = 0x8435, [0x196e] = 0x8461, [0x196f] = 0x8463,
  [0x1970] = 0x8469, [0x1971] = 0x846d, [0x1972] = 0x8446, [0x1973] = 0x865e,
  [0x1974] = 0x865c, [0x1975] = 0x865f, [0x1976] = 0x86f9, [0x1977] = 0x8713,
  [0x1978] = 0x8708, [0x1979] = 0x8707, [0x197a] = 0x8700, [0x197b] = 0x86fe,
  [0x197c] = 0x86fb, [0x197d] = 0x8702, [0x197e] = 0x8703, [0x197f] = 0x8706,
  [0x1980] = 0x870a, [0x1981] = 0x8859, [0x1982] = 0x88df, [0x1983] = 0x88d4,
  [0x1984] = 0x88d9, [0x1985] = 0x88dc, [0x1986] = 0x88d8, [0x1987] = 0x88dd,
  [0x1988] = 0x88e1, [0x1989] = 0x88ca, [0x198a] = 0x88d5, [0x198b] = 0x88d2,
  [0x198c] = 0x899c, [0x198d] = 0x89e3, [0x198e] = 0x8a6b, [0x198f] = 0x8a72,
  [0x1990] = 0x8a73, [0x1991] = 0x8a66, [0x1992] = 0x8a69, [0x1993] = 0x8a70,
  [0x1994] = 0x8a87, [0x1995] = 0x8a7c, [0x1996] = 0x8a63, [0x1997] = 0x8aa0,
  [0x1998] = 0x8a71, [0x1999] = 0x8a85, [0x199a] = 0x8a6d, [0x199b] = 0x8a62,
  [0x199c] = 0x8a6e, [0x199d] = 0x8a6c, [0x199e] = 0x8a79, [0x199f] = 0x8a7b,
  [0x19a0] = 0x8a3e, [0x19a1] = 0x8a68, [0x19a2] = 0x8c62, [0x19a3] = 0x8c8a,
  [0x19a4] = 0x8c89, [0x19a5] = 0x8cca, [0x19a6] = 0x8cc7, [0x19a7] = 0x8cc8,
  [0x19a8] = 0x8cc4, [0x19a9] = 0x8cb2, [0x19aa] = 0x8cc3, [0x19ab] = 0x8cc2,
  [0x19ac] = 0x8cc5, [0x19ad] = 0x8de1, [0x19ae] = 0x8ddf, [0x19af] = 0x8de8,
  [0x19b0] = 0x8def, [0x19b1] = 0x8df3, [0x19b2] = 0x8dfa, [0x19b3] = 0x8dea,
  [0x19b4] = 0x8de4, [0x19b5] = 0x8de6, [0x19b6] = 0x8eb2, [0x19b7] = 0x8f03,
  [0x19b8] = 0x8f09, [0x19b9] = 0x8efe, [0x19ba] = 0x8f0a, [0x19bb] = 0x8f9f,
  [0x19bc] = 0x8fb2, [0x19bd] = 0x904b, [0x19be] = 0x904a, [0x19bf] = 0x9053,
  [0x19c0] = 0x9042, [0x19c1] = 0x9054, [0x19c2] = 0x903c, [0x19c3] = 0x9055,
  [0x19c4] = 0x9050, [0x19c5] = 0x9047, [0x19c6] = 0x904f, [0x19c7] = 0x904e,
  [0x19c8] = 0x904d, [0x19c9] = 0x9051, [0x19ca] = 0x903e, [0x19cb] = 0x9041,
  [0x19cc] = 0x9112, [0x19cd] = 0x9117, [0x19ce] = 0x916c, [0x19cf] = 0x916a,
  [0x19d0] = 0x9169, [0x19d1] = 0x91c9, [0x19d2] = 0x9237, [0x19d3] = 0x9257,
  [0x19d4] = 0x9238, [0x19d5] = 0x923d, [0x19d6] = 0x9240, [0x19d7] = 0x923e,
  [0x19d8] = 0x925b, [0x19d9] = 0x924b, [0x19da] = 0x9264, [0x19db] = 0x9251,
  [0x19dc] = 0x9234, [0x19dd] = 0x9249, [0x19de] = 0x924d, [0x19df] = 0x9245,
  [0x19e0] = 0x9239, [0x19e1] = 0x923f, [0x19e2] = 0x925a, [0x19e3] = 0x9598,
  [0x19e4] = 0x9698, [0x19e5] = 0x9694, [0x19e6] = 0x9695, [0x19e7] = 0x96cd,
  [0x19e8] = 0x96cb, [0x19e9] = 0x96c9, [0x19ea] = 0x96ca, [0x19eb] = 0x96f7,
  [0x19ec] = 0x96fb, [0x19ed] = 0x96f9, [0x19ee] = 0x96f6, [0x19ef] = 0x9756,
  [0x19f0] = 0x9774, [0x19f1] = 0x9776, [0x19f2] = 0x9810, [0x19f3] = 0x9811,
  [0x19f4] = 0x9813, [0x19f5] = 0x980a, [0x19f6] = 0x9812, [0x19f7] = 0x980c,
  [0x19f8] = 0x98fc, [0x19f9] = 0x98f4, [0x19fa] = 0x98fd, [0x19fb] = 0x98fe,
  [0x19fc] = 0x99b3, [0x19fd] = 0x99b1, [0x19fe] = 0x99b4, [0x19ff] = 0x9ae1,
  [0x1a00] = 0x9ce9, [0x1a01] = 0x9e82, [0x1a02] = 0x9f0e, [0x1a03] = 0x9f13,
  [0x1a04] = 0x9f20, [0x1a05] = 0x50e7, [0x1a06] = 0x50ee, [0x1a07] = 0x50e5,
  [0x1a08] = 0x50d6, [0x1a09] = 0x50ed, [0x1a0a] = 0x50da, [0x1a0b] = 0x50d5,
  [0x1a0c] = 0x50cf, [0x1a0d] = 0x50d1, [0x1a0e] = 0x50f1, [0x1a0f] = 0x50ce,
  [0x1a10] = 0x50e9, [0x1a11] = 0x5162, [0x1a12] = 0x51f3, [0x1a13] = 0x5283,
  [0x1a14] = 0x5282, [0x1a15] = 0x5331, [0x1a16] = 0x53ad, [0x1a17] = 0x55fe,
  [0x1a18] = 0x5600, [0x1a19] = 0x561b, [0x1a1a] = 0x5617, [0x1a1b] = 0x55fd,
  [0x1a1c] = 0x5614, [0x1a1d] = 0x5606, [0x1a1e] = 0x5609, [0x1a1f] = 0x560d,
  [0x1a20] = 0x560e, [0x1a21] = 0x55f7, [0x1a22] = 0x5616, [0x1a23] = 0x561f,
  [0x1a24] = 0x5608, [0x1a25] = 0x5610, [0x1a26] = 0x55f6, [0x1a27] = 0x5718,
  [0x1a28] = 0x5716, [0x1a29] = 0x5875, [0x1a2a] = 0x587e, [0x1a2b] = 0x5883,
  [0x1a2c] = 0x5893, [0x1a2d] = 0x588a, [0x1a2e] = 0x5879, [0x1a2f] = 0x5885,
  [0x1a30] = 0x587d, [0x1a31] = 0x58fd, [0x1a32] = 0x5925, [0x1a33] = 0x5922,
  [0x1a34] = 0x5924, [0x1a35] = 0x596a, [0x1a36] = 0x5969, [0x1a37] = 0x5ae1,
  [0x1a38] = 0x5ae6, [0x1a39] = 0x5ae9, [0x1a3a] = 0x5ad7, [0x1a3b] = 0x5ad6,
  [0x1a3c] = 0x5ad8, [0x1a3d] = 0x5ae3, [0x1a3e] = 0x5b75, [0x1a3f] = 0x5bde,
  [0x1a40] = 0x5be7, [0x1a41] = 0x5be1, [0x1a42] = 0x5be5, [0x1a43] = 0x5be6,
  [0x1a44] = 0x5be8, [0x1a45] = 0x5be2, [0x1a46] = 0x5be4, [0x1a47] = 0x5bdf,
  [0x1a48] = 0x5c0d, [0x1a49] = 0x5c62, [0x1a4a] = 0x5d84, [0x1a4b] = 0x5d87,
  [0x1a4c] = 0x5e5b, [0x1a4d] = 0x5e63, [0x1a4e] = 0x5e55, [0x1a4f] = 0x5e57,
  [0x1a50] = 0x5e54, [0x1a51] = 0x5ed3, [0x1a52] = 0x5ed6, [0x1a53] = 0x5f0a,
  [0x1a54] = 0x5f46, [0x1a55] = 0x5f70, [0x1a56] = 0x5fb9, [0x1a57] = 0x6147,
  [0x1a58] = 0x613f, [0x1a59] = 0x614b, [0x1a5a] = 0x6177, [0x1a5b] = 0x6162,
  [0x1a5c] = 0x6163, [0x1a5d] = 0x615f, [0x1a5e] = 0x615a, [0x1a5f] = 0x6158,
  [0x1a60] = 0x6175, [0x1a61] = 0x622a, [0x1a62] = 0x6487, [0x1a63] = 0x6458,
  [0x1a64] = 0x6454, [0x1a65] = 0x64a4, [0x1a66] = 0x6478, [0x1a67] = 0x645f,
  [0x1a68] = 0x647a, [0x1a69] = 0x6451, [0x1a6a] = 0x6467, [0x1a6b] = 0x6434,
  [0x1a6c] = 0x646d, [0x1a6d] = 0x647b, [0x1a6e] = 0x6572, [0x1a6f] = 0x65a1,
  [0x1a70] = 0x65d7, [0x1a71] = 0x65d6, [0x1a72] = 0x66a2, [0x1a73] = 0x66a8,
  [0x1a74] = 0x669d, [0x1a75] = 0x699c, [0x1a76] = 0x69a8, [0x1a77] = 0x6995,
  [0x1a78] = 0x69c1, [0x1a79] = 0x69ae, [0x1a7a] = 0x69d3, [0x1a7b] = 0x69cb,
  [0x1a7c] = 0x699b, [0x1a7d] = 0x69b7, [0x1a7e] = 0x69bb, [0x1a7f] = 0x69ab,
  [0x1a80] = 0x69b4, [0x1a81] = 0x69d0, [0x1a82] = 0x69cd, [0x1a83] = 0x69ad,
  [0x1a84] = 0x69cc, [0x1a85] = 0x69a6, [0x1a86] = 0x69c3, [0x1a87] = 0x69a3,
  [0x1a88] = 0x6b49, [0x1a89] = 0x6b4c, [0x1a8a] = 0x6c33, [0x1a8b] = 0x6f33,
  [0x1a8c] = 0x6f14, [0x1a8d] = 0x6efe, [0x1a8e] = 0x6f13, [0x1a8f] = 0x6ef4,
  [0x1a90] = 0x6f29, [0x1a91] = 0x6f3e, [0x1a92] = 0x6f20, [0x1a93] = 0x6f2c,
  [0x1a94] = 0x6f0f, [0x1a95] = 0x6f02, [0x1a96] = 0x6f22, [0x1a97] = 0x6eff,
  [0x1a98] = 0x6eef, [0x1a99] = 0x6f06, [0x1a9a] = 0x6f31, [0x1a9b] = 0x6f38,
  [0x1a9c] = 0x6f32, [0x1a9d] = 0x6f23, [0x1a9e] = 0x6f15, [0x1a9f] = 0x6f2b,
  [0x1aa0] = 0x6f2f, [0x1aa1] = 0x6f88, [0x1aa2] = 0x6f2a, [0x1aa3] = 0x6eec,
  [0x1aa4] = 0x6f01, [0x1aa5] = 0x6ef2, [0x1aa6] = 0x6ecc, [0x1aa7] = 0x6ef7,
  [0x1aa8] = 0x7194, [0x1aa9] = 0x7199, [0x1aaa] = 0x717d, [0x1aab] = 0x718a,
  [0x1aac] = 0x7184, [0x1aad] = 0x7192, [0x1aae] = 0x723e, [0x1aaf] = 0x7292,
  [0x1ab0] = 0x7296, [0x1ab1] = 0x7344, [0x1ab2] = 0x7350, [0x1ab3] = 0x7464,
  [0x1ab4] = 0x7463, [0x1ab5] = 0x746a, [0x1ab6] = 0x7470, [0x1ab7] = 0x746d,
  [0x1ab8] = 0x7504, [0x1ab9] = 0x7591, [0x1aba] = 0x7627, [0x1abb] = 0x760d,
  [0x1abc] = 0x760b, [0x1abd] = 0x7609, [0x1abe] = 0x7613, [0x1abf] = 0x76e1,
  [0x1ac0] = 0x76e3, [0x1ac1] = 0x7784, [0x1ac2] = 0x777d, [0x1ac3] = 0x777f,
  [0x1ac4] = 0x7761, [0x1ac5] = 0x78c1, [0x1ac6] = 0x789f, [0x1ac7] = 0x78a7,
  [0x1ac8] = 0x78b3, [0x1ac9] = 0x78a9, [0x1aca] = 0x78a3, [0x1acb] = 0x798e,
  [0x1acc] = 0x798f, [0x1acd] = 0x798d, [0x1ace] = 0x7a2e, [0x1acf] = 0x7a31,
  [0x1ad0] = 0x7aaa, [0x1ad1] = 0x7aa9, [0x1ad2] = 0x7aed, [0x1ad3] = 0x7aef,
  [0x1ad4] = 0x7ba1, [0x1ad5] = 0x7b95, [0x1ad6] = 0x7b8b, [0x1ad7] = 0x7b75,
  [0x1ad8] = 0x7b97, [0x1ad9] = 0x7b9d, [0x1ada] = 0x7b94, [0x1adb] = 0x7b8f,
  [0x1adc] = 0x7bb8, [0x1add] = 0x7b87, [0x1ade] = 0x7b84, [0x1adf] = 0x7cb9,
  [0x1ae0] = 0x7cbd, [0x1ae1] = 0x7cbe, [0x1ae2] = 0x7dbb, [0x1ae3] = 0x7db0,
  [0x1ae4] = 0x7d9c, [0x1ae5] = 0x7dbd, [0x1ae6] = 0x7dbe, [0x1ae7] = 0x7da0,
  [0x1ae8] = 0x7dca, [0x1ae9] = 0x7db4, [0x1aea] = 0x7db2, [0x1aeb] = 0x7db1,
  [0x1aec] = 0x7dba, [0x1aed] = 0x7da2, [0x1aee] = 0x7dbf, [0x1aef] = 0x7db5,
  [0x1af0] = 0x7db8, [0x1af1] = 0x7dad, [0x1af2] = 0x7dd2, [0x1af3] = 0x7dc7,
  [0x1af4] = 0x7dac, [0x1af5] = 0x7f70, [0x1af6] = 0x7fe0, [0x1af7] = 0x7fe1,
  [0x1af8] = 0x7fdf, [0x1af9] = 0x805e, [0x1afa] = 0x805a, [0x1afb] = 0x8087,
  [0x1afc] = 0x8150, [0x1afd] = 0x8180, [0x1afe] = 0x818f, [0x1aff] = 0x8188,
  [0x1b00] = 0x818a, [0x1b01] = 0x817f, [0x1b02] = 0x8182, [0x1b03] = 0x81e7,
  [0x1b04] = 0x81fa, [0x1b05] = 0x8207, [0x1b06] = 0x8214, [0x1b07] = 0x821e,
  [0x1b08] = 0x824b, [0x1b09] = 0x84c9, [0x1b0a] = 0x84bf, [0x1b0b] = 0x84c6,
  [0x1b0c] = 0x84c4, [0x1b0d] = 0x8499, [0x1b0e] = 0x849e, [0x1b0f] = 0x84b2,
  [0x1b10] = 0x849c, [0x1b11] = 0x84cb, [0x1b12] = 0x84b8, [0x1b13] = 0x84c0,
  [0x1b14] = 0x84d3, [0x1b15] = 0x8490, [0x1b16] = 0x84bc, [0x1b17] = 0x84d1,
  [0x1b18] = 0x84ca, [0x1b19] = 0x873f, [0x1b1a] = 0x871c, [0x1b1b] = 0x873b,
  [0x1b1c] = 0x8722, [0x1b1d] = 0x8725, [0x1b1e] = 0x8734, [0x1b1f] = 0x8718,
  [0x1b20] = 0x8755, [0x1b21] = 0x8737, [0x1b22] = 0x8729, [0x1b23] = 0x88f3,
  [0x1b24] = 0x8902, [0x1b25] = 0x88f4, [0x1b26] = 0x88f9, [0x1b27] = 0x88f8,
  [0x1b28] = 0x88fd, [0x1b29] = 0x88e8, [0x1b2a] = 0x891a, [0x1b2b] = 0x88ef,
  [0x1b2c] = 0x8aa6, [0x1b2d] = 0x8a8c, [0x1b2e] = 0x8a9e, [0x1b2f] = 0x8aa3,
  [0x1b30] = 0x8a8d, [0x1b31] = 0x8aa1, [0x1b32] = 0x8a93, [0x1b33] = 0x8aa4,
  [0x1b34] = 0x8aaa, [0x1b35] = 0x8aa5, [0x1b36] = 0x8aa8, [0x1b37] = 0x8a98,
  [0x1b38] = 0x8a91, [0x1b39] = 0x8a9a, [0x1b3a] = 0x8aa7, [0x1b3b] = 0x8c6a,
  [0x1b3c] = 0x8c8d, [0x1b3d] = 0x8c8c, [0x1b3e] = 0x8cd3, [0x1b3f] = 0x8cd1,
  [0x1b40] = 0x8cd2, [0x1b41] = 0x8d6b, [0x1b42] = 0x8d99, [0x1b43] = 0x8d95,
  [0x1b44] = 0x8dfc, [0x1b45] = 0x8f14, [0x1b46] = 0x8f12, [0x1b47] = 0x8f15,
  [0x1b48] = 0x8f13, [0x1b49] = 0x8fa3, [0x1b4a] = 0x9060, [0x1b4b] = 0x9058,
  [0x1b4c] = 0x905c, [0x1b4d] = 0x9063, [0x1b4e] = 0x9059, [0x1b4f] = 0x905e,
  [0x1b50] = 0x9062, [0x1b51] = 0x905d, [0x1b52] = 0x905b, [0x1b53] = 0x9119,
  [0x1b54] = 0x9118, [0x1b55] = 0x911e, [0x1b56] = 0x9175, [0x1b57] = 0x9178,
  [0x1b58] = 0x9177, [0x1b59] = 0x9174, [0x1b5a] = 0x9278, [0x1b5b] = 0x92ac,
  [0x1b5c] = 0x9280, [0x1b5d] = 0x9285, [0x1b5e] = 0x9298, [0x1b5f] = 0x9296,
  [0x1b60] = 0x927b, [0x1b61] = 0x9293, [0x1b62] = 0x929c, [0x1b63] = 0x92a8,
  [0x1b64] = 0x927c, [0x1b65] = 0x9291, [0x1b66] = 0x95a1, [0x1b67] = 0x95a8,
  [0x1b68] = 0x95a9, [0x1b69] = 0x95a3, [0x1b6a] = 0x95a5, [0x1b6b] = 0x95a4,
  [0x1b6c] = 0x9699, [0x1b6d] = 0x969c, [0x1b6e] = 0x969b, [0x1b6f] = 0x96cc,
  [0x1b70] = 0x96d2, [0x1b71] = 0x9700, [0x1b72] = 0x977c, [0x1b73] = 0x9785,
  [0x1b74] = 0x97f6, [0x1b75] = 0x9817, [0x1b76] = 0x9818, [0x1b77] = 0x98af,
  [0x1b78] = 0x98b1, [0x1b79] = 0x9903, [0x1b7a] = 0x9905, [0x1b7b] = 0x990c,
  [0x1b7c] = 0x9909, [0x1b7d] = 0x99c1, [0x1b7e] = 0x9aaf, [0x1b7f] = 0x9ab0,
  [0x1b80] = 0x9ae6, [0x1b81] = 0x9b41, [0x1b82] = 0x9b42, [0x1b83] = 0x9cf4,
  [0x1b84] = 0x9cf6, [0x1b85] = 0x9cf3, [0x1b86] = 0x9ebc, [0x1b87] = 0x9f3b,
  [0x1b88] = 0x9f4a, [0x1b89] = 0x5104, [0x1b8a] = 0x5100, [0x1b8b] = 0x50fb,
  [0x1b8c] = 0x50f5, [0x1b8d] = 0x50f9, [0x1b8e] = 0x5102, [0x1b8f] = 0x5108,
  [0x1b90] = 0x5109, [0x1b91] = 0x5105, [0x1b92] = 0x51dc, [0x1b93] = 0x5287,
  [0x1b94] = 0x5288, [0x1b95] = 0x5289, [0x1b96] = 0x528d, [0x1b97] = 0x528a,
  [0x1b98] = 0x52f0, [0x1b99] = 0x53b2, [0x1b9a] = 0x562e, [0x1b9b] = 0x563b,
  [0x1b9c] = 0x5639, [0x1b9d] = 0x5632, [0x1b9e] = 0x563f, [0x1b9f] = 0x5634,
  [0x1ba0] = 0x5629, [0x1ba1] = 0x5653, [0x1ba2] = 0x564e, [0x1ba3] = 0x5657,
  [0x1ba4] = 0x5674, [0x1ba5] = 0x5636, [0x1ba6] = 0x562f, [0x1ba7] = 0x5630,
  [0x1ba8] = 0x5880, [0x1ba9] = 0x589f, [0x1baa] = 0x589e, [0x1bab] = 0x58b3,
  [0x1bac] = 0x589c, [0x1bad] = 0x58ae, [0x1bae] = 0x58a9, [0x1baf] = 0x58a6,
  [0x1bb0] = 0x596d, [0x1bb1] = 0x5b09, [0x1bb2] = 0x5afb, [0x1bb3] = 0x5b0b,
  [0x1bb4] = 0x5af5, [0x1bb5] = 0x5b0c, [0x1bb6] = 0x5b08, [0x1bb7] = 0x5bee,
  [0x1bb8] = 0x5bec, [0x1bb9] = 0x5be9, [0x1bba] = 0x5beb, [0x1bbb] = 0x5c64,
  [0x1bbc] = 0x5c65, [0x1bbd] = 0x5d9d, [0x1bbe] = 0x5d94, [0x1bbf] = 0x5e62,
  [0x1bc0] = 0x5e5f, [0x1bc1] = 0x5e61, [0x1bc2] = 0x5ee2, [0x1bc3] = 0x5eda,
  [0x1bc4] = 0x5edf, [0x1bc5] = 0x5edd, [0x1bc6] = 0x5ee3, [0x1bc7] = 0x5ee0,
  [0x1bc8] = 0x5f48, [0x1bc9] = 0x5f71, [0x1bca] = 0x5fb7, [0x1bcb] = 0x5fb5,
  [0x1bcc] = 0x6176, [0x1bcd] = 0x6167, [0x1bce] = 0x616e, [0x1bcf] = 0x615d,
  [0x1bd0] = 0x6155, [0x1bd1] = 0x6182, [0x1bd2] = 0x617c, [0x1bd3] = 0x6170,
  [0x1bd4] = 0x616b, [0x1bd5] = 0x617e, [0x1bd6] = 0x61a7, [0x1bd7] = 0x6190,
  [0x1bd8] = 0x61ab, [0x1bd9] = 0x618e, [0x1bda] = 0x61ac, [0x1bdb] = 0x619a,
  [0x1bdc] = 0x61a4, [0x1bdd] = 0x6194, [0x1bde] = 0x61ae, [0x1bdf] = 0x622e,
  [0x1be0] = 0x6469, [0x1be1] = 0x646f, [0x1be2] = 0x6479, [0x1be3] = 0x649e,
  [0x1be4] = 0x64b2, [0x1be5] = 0x6488, [0x1be6] = 0x6490, [0x1be7] = 0x64b0,
  [0x1be8] = 0x64a5, [0x1be9] = 0x6493, [0x1bea] = 0x6495, [0x1beb] = 0x64a9,
  [0x1bec] = 0x6492, [0x1bed] = 0x64ae, [0x1bee] = 0x64ad, [0x1bef] = 0x64ab,
  [0x1bf0] = 0x649a, [0x1bf1] = 0x64ac, [0x1bf2] = 0x6499, [0x1bf3] = 0x64a2,
  [0x1bf4] = 0x64b3, [0x1bf5] = 0x6575, [0x1bf6] = 0x6577, [0x1bf7] = 0x6578,
  [0x1bf8] = 0x66ae, [0x1bf9] = 0x66ab, [0x1bfa] = 0x66b4, [0x1bfb] = 0x66b1,
  [0x1bfc] = 0x6a23, [0x1bfd] = 0x6a1f, [0x1bfe] = 0x69e8, [0x1bff] = 0x6a01,
  [0x1c00] = 0x6a1e, [0x1c01] = 0x6a19, [0x1c02] = 0x69fd, [0x1c03] = 0x6a21,
  [0x1c04] = 0x6a13, [0x1c05] = 0x6a0a, [0x1c06] = 0x69f3, [0x1c07] = 0x6a02,
  [0x1c08] = 0x6a05, [0x1c09] = 0x69ed, [0x1c0a] = 0x6a11, [0x1c0b] = 0x6b50,
  [0x1c0c] = 0x6b4e, [0x1c0d] = 0x6ba4, [0x1c0e] = 0x6bc5, [0x1c0f] = 0x6bc6,
  [0x1c10] = 0x6f3f, [0x1c11] = 0x6f7c, [0x1c12] = 0x6f84, [0x1c13] = 0x6f51,
  [0x1c14] = 0x6f66, [0x1c15] = 0x6f54, [0x1c16] = 0x6f86, [0x1c17] = 0x6f6d,
  [0x1c18] = 0x6f5b, [0x1c19] = 0x6f78, [0x1c1a] = 0x6f6e, [0x1c1b] = 0x6f8e,
  [0x1c1c] = 0x6f7a, [0x1c1d] = 0x6f70, [0x1c1e] = 0x6f64, [0x1c1f] = 0x6f97,
  [0x1c20] = 0x6f58, [0x1c21] = 0x6ed5, [0x1c22] = 0x6f6f, [0x1c23] = 0x6f60,
  [0x1c24] = 0x6f5f, [0x1c25] = 0x719f, [0x1c26] = 0x71ac, [0x1c27] = 0x71b1,
  [0x1c28] = 0x71a8, [0x1c29] = 0x7256, [0x1c2a] = 0x729b, [0x1c2b] = 0x734e,
  [0x1c2c] = 0x7357, [0x1c2d] = 0x7469, [0x1c2e] = 0x748b, [0x1c2f] = 0x7483,
  [0x1c30] = 0x747e, [0x1c31] = 0x7480, [0x1c32] = 0x757f, [0x1c33] = 0x7620,
  [0x1c34] = 0x7629, [0x1c35] = 0x761f, [0x1c36] = 0x7624, [0x1c37] = 0x7626,
  [0x1c38] = 0x7621, [0x1c39] = 0x7622, [0x1c3a] = 0x769a, [0x1c3b] = 0x76ba,
  [0x1c3c] = 0x76e4, [0x1c3d] = 0x778e, [0x1c3e] = 0x7787, [0x1c3f] = 0x778c,
  [0x1c40] = 0x7791, [0x1c41] = 0x778b, [0x1c42] = 0x78cb, [0x1c43] = 0x78c5,
  [0x1c44] = 0x78ba, [0x1c45] = 0x78ca, [0x1c46] = 0x78be, [0x1c47] = 0x78d5,
  [0x1c48] = 0x78bc, [0x1c49] = 0x78d0, [0x1c4a] = 0x7a3f, [0x1c4b] = 0x7a3c,
  [0x1c4c] = 0x7a40, [0x1c4d] = 0x7a3d, [0x1c4e] = 0x7a37, [0x1c4f] = 0x7a3b,
  [0x1c50] = 0x7aaf, [0x1c51] = 0x7aae, [0x1c52] = 0x7bad, [0x1c53] = 0x7bb1,
  [0x1c54] = 0x7bc4, [0x1c55] = 0x7bb4, [0x1c56] = 0x7bc6, [0x1c57] = 0x7bc7,
  [0x1c58] = 0x7bc1, [0x1c59] = 0x7ba0, [0x1c5a] = 0x7bcc, [0x1c5b] = 0x7cca,
  [0x1c5c] = 0x7de0, [0x1c5d] = 0x7df4, [0x1c5e] = 0x7def, [0x1c5f] = 0x7dfb,
  [0x1c60] = 0x7dd8, [0x1c61] = 0x7dec, [0x1c62] = 0x7ddd, [0x1c63] = 0x7de8,
  [0x1c64] = 0x7de3, [0x1c65] = 0x7dda, [0x1c66] = 0x7dde, [0x1c67] = 0x7de9,
  [0x1c68] = 0x7d9e, [0x1c69] = 0x7dd9, [0x1c6a] = 0x7df2, [0x1c6b] = 0x7df9,
  [0x1c6c] = 0x7f75, [0x1c6d] = 0x7f77, [0x1c6e] = 0x7faf, [0x1c6f] = 0x7fe9,
  [0x1c70] = 0x8026, [0x1c71] = 0x819b, [0x1c72] = 0x819c, [0x1c73] = 0x819d,
  [0x1c74] = 0x81a0, [0x1c75] = 0x819a, [0x1c76] = 0x8198, [0x1c77] = 0x8517,
  [0x1c78] = 0x853d, [0x1c79] = 0x851a, [0x1c7a] = 0x84ee, [0x1c7b] = 0x852c,
  [0x1c7c] = 0x852d, [0x1c7d] = 0x8513, [0x1c7e] = 0x8511, [0x1c7f] = 0x8523,
  [0x1c80] = 0x8521, [0x1c81] = 0x8514, [0x1c82] = 0x84ec, [0x1c83] = 0x8525,
  [0x1c84] = 0x84ff, [0x1c85] = 0x8506, [0x1c86] = 0x8782, [0x1c87] = 0x8774,
  [0x1c88] = 0x8776, [0x1c89] = 0x8760, [0x1c8a] = 0x8766, [0x1c8b] = 0x8778,
  [0x1c8c] = 0x8768, [0x1c8d] = 0x8759, [0x1c8e] = 0x8757, [0x1c8f] = 0x874c,
  [0x1c90] = 0x8753, [0x1c91] = 0x885b, [0x1c92] = 0x885d, [0x1c93] = 0x8910,
  [0x1c94] = 0x8907, [0x1c95] = 0x8912, [0x1c96] = 0x8913, [0x1c97] = 0x8915,
  [0x1c98] = 0x890a, [0x1c99] = 0x8abc, [0x1c9a] = 0x8ad2, [0x1c9b] = 0x8ac7,
  [0x1c9c] = 0x8ac4, [0x1c9d] = 0x8a95, [0x1c9e] = 0x8acb, [0x1c9f] = 0x8af8,
  [0x1ca0] = 0x8ab2, [0x1ca1] = 0x8ac9, [0x1ca2] = 0x8ac2, [0x1ca3] = 0x8abf,
  [0x1ca4] = 0x8ab0, [0x1ca5] = 0x8ad6, [0x1ca6] = 0x8acd, [0x1ca7] = 0x8ab6,
  [0x1ca8] = 0x8ab9, [0x1ca9] = 0x8adb, [0x1caa] = 0x8c4c, [0x1cab] = 0x8c4e,
  [0x1cac] = 0x8c6c, [0x1cad] = 0x8ce0, [0x1cae] = 0x8cde, [0x1caf] = 0x8ce6,
  [0x1cb0] = 0x8ce4, [0x1cb1] = 0x8cec, [0x1cb2] = 0x8ced, [0x1cb3] = 0x8ce2,
  [0x1cb4] = 0x8ce3, [0x1cb5] = 0x8cdc, [0x1cb6] = 0x8cea, [0x1cb7] = 0x8ce1,
  [0x1cb8] = 0x8d6d, [0x1cb9] = 0x8d9f, [0x1cba] = 0x8da3, [0x1cbb] = 0x8e2b,
  [0x1cbc] = 0x8e10, [0x1cbd] = 0x8e1d, [0x1cbe] = 0x8e22, [0x1cbf] = 0x8e0f,
  [0x1cc0] = 0x8e29, [0x1cc1] = 0x8e1f, [0x1cc2] = 0x8e21, [0x1cc3] = 0x8e1e,
  [0x1cc4] = 0x8eba, [0x1cc5] = 0x8f1d, [0x1cc6] = 0x8f1b, [0x1cc7] = 0x8f1f,
  [0x1cc8] = 0x8f29, [0x1cc9] = 0x8f26, [0x1cca] = 0x8f2a, [0x1ccb] = 0x8f1c,
  [0x1ccc] = 0x8f1e, [0x1ccd] = 0x8f25, [0x1cce] = 0x9069, [0x1ccf] = 0x906e,
  [0x1cd0] = 0x9068, [0x1cd1] = 0x906d, [0x1cd2] = 0x9077, [0x1cd3] = 0x9130,
  [0x1cd4] = 0x912d, [0x1cd5] = 0x9127, [0x1cd6] = 0x9131, [0x1cd7] = 0x9187,
  [0x1cd8] = 0x9189, [0x1cd9] = 0x918b, [0x1cda] = 0x9183, [0x1cdb] = 0x92c5,
  [0x1cdc] = 0x92bb, [0x1cdd] = 0x92b7, [0x1cde] = 0x92ea, [0x1cdf] = 0x92e4,
  [0x1ce0] = 0x92c1, [0x1ce1] = 0x92b3, [0x1ce2] = 0x92bc, [0x1ce3] = 0x92d2,
  [0x1ce4] = 0x92c7, [0x1ce5] = 0x92f0, [0x1ce6] = 0x92b2, [0x1ce7] = 0x95ad,
  [0x1ce8] = 0x95b1, [0x1ce9] = 0x9704, [0x1cea] = 0x9706, [0x1ceb] = 0x9707,
  [0x1cec] = 0x9709, [0x1ced] = 0x9760, [0x1cee] = 0x978d, [0x1cef] = 0x978b,
  [0x1cf0] = 0x978f, [0x1cf1] = 0x9821, [0x1cf2] = 0x982b, [0x1cf3] = 0x981c,
  [0x1cf4] = 0x98b3, [0x1cf5] = 0x990a, [0x1cf6] = 0x9913, [0x1cf7] = 0x9912,
  [0x1cf8] = 0x9918, [0x1cf9] = 0x99dd, [0x1cfa] = 0x99d0, [0x1cfb] = 0x99df,
  [0x1cfc] = 0x99db, [0x1cfd] = 0x99d1, [0x1cfe] = 0x99d5, [0x1cff] = 0x99d2,
  [0x1d00] = 0x99d9, [0x1d01] = 0x9ab7, [0x1d02] = 0x9aee, [0x1d03] = 0x9aef,
  [0x1d04] = 0x9b27, [0x1d05] = 0x9b45, [0x1d06] = 0x9b44, [0x1d07] = 0x9b77,
  [0x1d08] = 0x9b6f, [0x1d09] = 0x9d06, [0x1d0a] = 0x9d09, [0x1d0b] = 0x9d03,
  [0x1d0c] = 0x9ea9, [0x1d0d] = 0x9ebe, [0x1d0e] = 0x9ece, [0x1d0f] = 0x58a8,
  [0x1d10] = 0x9f52, [0x1d11] = 0x5112, [0x1d12] = 0x5118, [0x1d13] = 0x5114,
  [0x1d14] = 0x5110, [0x1d15] = 0x5115, [0x1d16] = 0x5180, [0x1d17] = 0x51aa,
  [0x1d18] = 0x51dd, [0x1d19] = 0x5291, [0x1d1a] = 0x5293, [0x1d1b] = 0x52f3,
  [0x1d1c] = 0x5659, [0x1d1d] = 0x566b, [0x1d1e] = 0x5679, [0x1d1f] = 0x5669,
  [0x1d20] = 0x5664, [0x1d21] = 0x5678, [0x1d22] = 0x566a, [0x1d23] = 0x5668,
  [0x1d24] = 0x5665, [0x1d25] = 0x5671, [0x1d26] = 0x566f, [0x1d27] = 0x566c,
  [0x1d28] = 0x5662, [0x1d29] = 0x5676, [0x1d2a] = 0x58c1, [0x1d2b] = 0x58be,
  [0x1d2c] = 0x58c7, [0x1d2d] = 0x58c5, [0x1d2e] = 0x596e, [0x1d2f] = 0x5b1d,
  [0x1d30] = 0x5b34, [0x1d31] = 0x5b78, [0x1d32] = 0x5bf0, [0x1d33] = 0x5c0e,
  [0x1d34] = 0x5f4a, [0x1d35] = 0x61b2, [0x1d36] = 0x6191, [0x1d37] = 0x61a9,
  [0x1d38] = 0x618a, [0x1d39] = 0x61cd, [0x1d3a] = 0x61b6, [0x1d3b] = 0x61be,
  [0x1d3c] = 0x61ca, [0x1d3d] = 0x61c8, [0x1d3e] = 0x6230, [0x1d3f] = 0x64c5,
  [0x1d40] = 0x64c1, [0x1d41] = 0x64cb, [0x1d42] = 0x64bb, [0x1d43] = 0x64bc,
  [0x1d44] = 0x64da, [0x1d45] = 0x64c4, [0x1d46] = 0x64c7, [0x1d47] = 0x64c2,
  [0x1d48] = 0x64cd, [0x1d49] = 0x64bf, [0x1d4a] = 0x64d2, [0x1d4b] = 0x64d4,
  [0x1d4c] = 0x64be, [0x1d4d] = 0x6574, [0x1d4e] = 0x66c6, [0x1d4f] = 0x66c9,
  [0x1d50] = 0x66b9, [0x1d51] = 0x66c4, [0x1d52] = 0x66c7, [0x1d53] = 0x66b8,
  [0x1d54] = 0x6a3d, [0x1d55] = 0x6a38, [0x1d56] = 0x6a3a, [0x1d57] = 0x6a59,
  [0x1d58] = 0x6a6b, [0x1d59] = 0x6a58, [0x1d5a] = 0x6a39, [0x1d5b] = 0x6a44,
  [0x1d5c] = 0x6a62, [0x1d5d] = 0x6a61, [0x1d5e] = 0x6a4b, [0x1d5f] = 0x6a47,
  [0x1d60] = 0x6a35, [0x1d61] = 0x6a5f, [0x1d62] = 0x6a48, [0x1d63] = 0x6b59,
  [0x1d64] = 0x6b77, [0x1d65] = 0x6c05, [0x1d66] = 0x6fc2, [0x1d67] = 0x6fb1,
  [0x1d68] = 0x6fa1, [0x1d69] = 0x6fc3, [0x1d6a] = 0x6fa4, [0x1d6b] = 0x6fc1,
  [0x1d6c] = 0x6fa7, [0x1d6d] = 0x6fb3, [0x1d6e] = 0x6fc0, [0x1d6f] = 0x6fb9,
  [0x1d70] = 0x6fb6, [0x1d71] = 0x6fa6, [0x1d72] = 0x6fa0, [0x1d73] = 0x6fb4,
  [0x1d74] = 0x71be, [0x1d75] = 0x71c9, [0x1d76] = 0x71d0, [0x1d77] = 0x71d2,
  [0x1d78] = 0x71c8, [0x1d79] = 0x71d5, [0x1d7a] = 0x71b9, [0x1d7b] = 0x71ce,
  [0x1d7c] = 0x71d9, [0x1d7d] = 0x71dc, [0x1d7e] = 0x71c3, [0x1d7f] = 0x71c4,
  [0x1d80] = 0x7368, [0x1d81] = 0x749c, [0x1d82] = 0x74a3, [0x1d83] = 0x7498,
  [0x1d84] = 0x749f, [0x1d85] = 0x749e, [0x1d86] = 0x74e2, [0x1d87] = 0x750c,
  [0x1d88] = 0x750d, [0x1d89] = 0x7634, [0x1d8a] = 0x7638, [0x1d8b] = 0x763a,
  [0x1d8c] = 0x76e7, [0x1d8d] = 0x76e5, [0x1d8e] = 0x77a0, [0x1d8f] = 0x779e,
  [0x1d90] = 0x779f, [0x1d91] = 0x77a5, [0x1d92] = 0x78e8, [0x1d93] = 0x78da,
  [0x1d94] = 0x78ec, [0x1d95] = 0x78e7, [0x1d96] = 0x79a6, [0x1d97] = 0x7a4d,
  [0x1d98] = 0x7a4e, [0x1d99] = 0x7a46, [0x1d9a] = 0x7a4c, [0x1d9b] = 0x7a4b,
  [0x1d9c] = 0x7aba, [0x1d9d] = 0x7bd9, [0x1d9e] = 0x7c11, [0x1d9f] = 0x7bc9,
  [0x1da0] = 0x7be4, [0x1da1] = 0x7bdb, [0x1da2] = 0x7be1, [0x1da3] = 0x7be9,
  [0x1da4] = 0x7be6, [0x1da5] = 0x7cd5, [0x1da6] = 0x7cd6, [0x1da7] = 0x7e0a,
  [0x1da8] = 0x7e11, [0x1da9] = 0x7e08, [0x1daa] = 0x7e1b, [0x1dab] = 0x7e23,
  [0x1dac] = 0x7e1e, [0x1dad] = 0x7e1d, [0x1dae] = 0x7e09, [0x1daf] = 0x7e10,
  [0x1db0] = 0x7f79, [0x1db1] = 0x7fb2, [0x1db2] = 0x7ff0, [0x1db3] = 0x7ff1,
  [0x1db4] = 0x7fee, [0x1db5] = 0x8028, [0x1db6] = 0x81b3, [0x1db7] = 0x81a9,
  [0x1db8] = 0x81a8, [0x1db9] = 0x81fb, [0x1dba] = 0x8208, [0x1dbb] = 0x8258,
  [0x1dbc] = 0x8259, [0x1dbd] = 0x854a, [0x1dbe] = 0x8559, [0x1dbf] = 0x8548,
  [0x1dc0] = 0x8568, [0x1dc1] = 0x8569, [0x1dc2] = 0x8543, [0x1dc3] = 0x8549,
  [0x1dc4] = 0x856d, [0x1dc5] = 0x856a, [0x1dc6] = 0x855e, [0x1dc7] = 0x8783,
  [0x1dc8] = 0x879f, [0x1dc9] = 0x879e, [0x1dca] = 0x87a2, [0x1dcb] = 0x878d,
  [0x1dcc] = 0x8861, [0x1dcd] = 0x892a, [0x1dce] = 0x8932, [0x1dcf] = 0x8925,
  [0x1dd0] = 0x892b, [0x1dd1] = 0x8921, [0x1dd2] = 0x89aa, [0x1dd3] = 0x89a6,
  [0x1dd4] = 0x8ae6, [0x1dd5] = 0x8afa, [0x1dd6] = 0x8aeb, [0x1dd7] = 0x8af1,
  [0x1dd8] = 0x8b00, [0x1dd9] = 0x8adc, [0x1dda] = 0x8ae7, [0x1ddb] = 0x8aee,
  [0x1ddc] = 0x8afe, [0x1ddd] = 0x8b01, [0x1dde] = 0x8b02, [0x1ddf] = 0x8af7,
  [0x1de0] = 0x8aed, [0x1de1] = 0x8af3, [0x1de2] = 0x8af6, [0x1de3] = 0x8afc,
  [0x1de4] = 0x8c6b, [0x1de5] = 0x8c6d, [0x1de6] = 0x8c93, [0x1de7] = 0x8cf4,
  [0x1de8] = 0x8e44, [0x1de9] = 0x8e31, [0x1dea] = 0x8e34, [0x1deb] = 0x8e42,
  [0x1dec] = 0x8e39, [0x1ded] = 0x8e35, [0x1dee] = 0x8f3b, [0x1def] = 0x8f2f,
  [0x1df0] = 0x8f38, [0x1df1] = 0x8f33, [0x1df2] = 0x8fa8, [0x1df3] = 0x8fa6,
  [0x1df4] = 0x9075, [0x1df5] = 0x9074, [0x1df6] = 0x9078, [0x1df7] = 0x9072,
  [0x1df8] = 0x907c, [0x1df9] = 0x907a, [0x1dfa] = 0x9134, [0x1dfb] = 0x9192,
  [0x1dfc] = 0x9320, [0x1dfd] = 0x9336, [0x1dfe] = 0x92f8, [0x1dff] = 0x9333,
  [0x1e00] = 0x932f, [0x1e01] = 0x9322, [0x1e02] = 0x92fc, [0x1e03] = 0x932b,
  [0x1e04] = 0x9304, [0x1e05] = 0x931a, [0x1e06] = 0x9310, [0x1e07] = 0x9326,
  [0x1e08] = 0x9321, [0x1e09] = 0x9315, [0x1e0a] = 0x932e, [0x1e0b] = 0x9319,
  [0x1e0c] = 0x95bb, [0x1e0d] = 0x96a7, [0x1e0e] = 0x96a8, [0x1e0f] = 0x96aa,
  [0x1e10] = 0x96d5, [0x1e11] = 0x970e, [0x1e12] = 0x9711, [0x1e13] = 0x9716,
  [0x1e14] = 0x970d, [0x1e15] = 0x9713, [0x1e16] = 0x970f, [0x1e17] = 0x975b,
  [0x1e18] = 0x975c, [0x1e19] = 0x9766, [0x1e1a] = 0x9798, [0x1e1b] = 0x9830,
  [0x1e1c] = 0x9838, [0x1e1d] = 0x983b, [0x1e1e] = 0x9837, [0x1e1f] = 0x982d,
  [0x1e20] = 0x9839, [0x1e21] = 0x9824, [0x1e22] = 0x9910, [0x1e23] = 0x9928,
  [0x1e24] = 0x991e, [0x1e25] = 0x991b, [0x1e26] = 0x9921, [0x1e27] = 0x991a,
  [0x1e28] = 0x99ed, [0x1e29] = 0x99e2, [0x1e2a] = 0x99f1, [0x1e2b] = 0x9ab8,
  [0x1e2c] = 0x9abc, [0x1e2d] = 0x9afb, [0x1e2e] = 0x9aed, [0x1e2f] = 0x9b28,
  [0x1e30] = 0x9b91, [0x1e31] = 0x9d15, [0x1e32] = 0x9d23, [0x1e33] = 0x9d26,
  [0x1e34] = 0x9d28, [0x1e35] = 0x9d12, [0x1e36] = 0x9d1b, [0x1e37] = 0x9ed8,
  [0x1e38] = 0x9ed4, [0x1e39] = 0x9f8d, [0x1e3a] = 0x9f9c, [0x1e3b] = 0x512a,
  [0x1e3c] = 0x511f, [0x1e3d] = 0x5121, [0x1e3e] = 0x5132, [0x1e3f] = 0x52f5,
  [0x1e40] = 0x568e, [0x1e41] = 0x5680, [0x1e42] = 0x5690, [0x1e43] = 0x5685,
  [0x1e44] = 0x5687, [0x1e45] = 0x568f, [0x1e46] = 0x58d5, [0x1e47] = 0x58d3,
  [0x1e48] = 0x58d1, [0x1e49] = 0x58ce, [0x1e4a] = 0x5b30, [0x1e4b] = 0x5b2a,
  [0x1e4c] = 0x5b24, [0x1e4d] = 0x5b7a, [0x1e4e] = 0x5c37, [0x1e4f] = 0x5c68,
  [0x1e50] = 0x5dbc, [0x1e51] = 0x5dba, [0x1e52] = 0x5dbd, [0x1e53] = 0x5db8,
  [0x1e54] = 0x5e6b, [0x1e55] = 0x5f4c, [0x1e56] = 0x5fbd, [0x1e57] = 0x61c9,
  [0x1e58] = 0x61c2, [0x1e59] = 0x61c7, [0x1e5a] = 0x61e6, [0x1e5b] = 0x61cb,
  [0x1e5c] = 0x6232, [0x1e5d] = 0x6234, [0x1e5e] = 0x64ce, [0x1e5f] = 0x64ca,
  [0x1e60] = 0x64d8, [0x1e61] = 0x64e0, [0x1e62] = 0x64f0, [0x1e63] = 0x64e6,
  [0x1e64] = 0x64ec, [0x1e65] = 0x64f1, [0x1e66] = 0x64e2, [0x1e67] = 0x64ed,
  [0x1e68] = 0x6582, [0x1e69] = 0x6583, [0x1e6a] = 0x66d9, [0x1e6b] = 0x66d6,
  [0x1e6c] = 0x6a80, [0x1e6d] = 0x6a94, [0x1e6e] = 0x6a84, [0x1e6f] = 0x6aa2,
  [0x1e70] = 0x6a9c, [0x1e71] = 0x6adb, [0x1e72] = 0x6aa3, [0x1e73] = 0x6a7e,
  [0x1e74] = 0x6a97, [0x1e75] = 0x6a90, [0x1e76] = 0x6aa0, [0x1e77] = 0x6b5c,
  [0x1e78] = 0x6bae, [0x1e79] = 0x6bda, [0x1e7a] = 0x6c08, [0x1e7b] = 0x6fd8,
  [0x1e7c] = 0x6ff1, [0x1e7d] = 0x6fdf, [0x1e7e] = 0x6fe0, [0x1e7f] = 0x6fdb,
  [0x1e80] = 0x6fe4, [0x1e81] = 0x6feb, [0x1e82] = 0x6fef, [0x1e83] = 0x6f80,
  [0x1e84] = 0x6fec, [0x1e85] = 0x6fe1, [0x1e86] = 0x6fe9, [0x1e87] = 0x6fd5,
  [0x1e88] = 0x6fee, [0x1e89] = 0x6ff0, [0x1e8a] = 0x71e7, [0x1e8b] = 0x71df,
  [0x1e8c] = 0x71ee, [0x1e8d] = 0x71e6, [0x1e8e] = 0x71e5, [0x1e8f] = 0x71ed,
  [0x1e90] = 0x71ec, [0x1e91] = 0x71f4, [0x1e92] = 0x71e0, [0x1e93] = 0x7235,
  [0x1e94] = 0x7246, [0x1e95] = 0x7370, [0x1e96] = 0x7372, [0x1e97] = 0x74a9,
  [0x1e98] = 0x74b0, [0x1e99] = 0x74a6, [0x1e9a] = 0x74a8, [0x1e9b] = 0x7646,
  [0x1e9c] = 0x7642, [0x1e9d] = 0x764c, [0x1e9e] = 0x76ea, [0x1e9f] = 0x77b3,
  [0x1ea0] = 0x77aa, [0x1ea1] = 0x77b0, [0x1ea2] = 0x77ac, [0x1ea3] = 0x77a7,
  [0x1ea4] = 0x77ad, [0x1ea5] = 0x77ef, [0x1ea6] = 0x78f7, [0x1ea7] = 0x78fa,
  [0x1ea8] = 0x78f4, [0x1ea9] = 0x78ef, [0x1eaa] = 0x7901, [0x1eab] = 0x79a7,
  [0x1eac] = 0x79aa, [0x1ead] = 0x7a57, [0x1eae] = 0x7abf, [0x1eaf] = 0x7c07,
  [0x1eb0] = 0x7c0d, [0x1eb1] = 0x7bfe, [0x1eb2] = 0x7bf7, [0x1eb3] = 0x7c0c,
  [0x1eb4] = 0x7be0, [0x1eb5] = 0x7ce0, [0x1eb6] = 0x7cdc, [0x1eb7] = 0x7cde,
  [0x1eb8] = 0x7ce2, [0x1eb9] = 0x7cdf, [0x1eba] = 0x7cd9, [0x1ebb] = 0x7cdd,
  [0x1ebc] = 0x7e2e, [0x1ebd] = 0x7e3e, [0x1ebe] = 0x7e46, [0x1ebf] = 0x7e37,
  [0x1ec0] = 0x7e32, [0x1ec1] = 0x7e43, [0x1ec2] = 0x7e2b, [0x1ec3] = 0x7e3d,
  [0x1ec4] = 0x7e31, [0x1ec5] = 0x7e45, [0x1ec6] = 0x7e41, [0x1ec7] = 0x7e34,
  [0x1ec8] = 0x7e39, [0x1ec9] = 0x7e48, [0x1eca] = 0x7e35, [0x1ecb] = 0x7e3f,
  [0x1ecc] = 0x7e2f, [0x1ecd] = 0x7f44, [0x1ece] = 0x7ff3, [0x1ecf] = 0x7ffc,
  [0x1ed0] = 0x8071, [0x1ed1] = 0x8072, [0x1ed2] = 0x8070, [0x1ed3] = 0x806f,
  [0x1ed4] = 0x8073, [0x1ed5] = 0x81c6, [0x1ed6] = 0x81c3, [0x1ed7] = 0x81ba,
  [0x1ed8] = 0x81c2, [0x1ed9] = 0x81c0, [0x1eda] = 0x81bf, [0x1edb] = 0x81bd,
  [0x1edc] = 0x81c9, [0x1edd] = 0x81be, [0x1ede] = 0x81e8, [0x1edf] = 0x8209,
  [0x1ee0] = 0x8271, [0x1ee1] = 0x85aa, [0x1ee2] = 0x8584, [0x1ee3] = 0x857e,
  [0x1ee4] = 0x859c, [0x1ee5] = 0x8591, [0x1ee6] = 0x8594, [0x1ee7] = 0x85af,
  [0x1ee8] = 0x859b, [0x1ee9] = 0x8587, [0x1eea] = 0x85a8, [0x1eeb] = 0x858a,
  [0x1eec] = 0x85a6, [0x1eed] = 0x8667, [0x1eee] = 0x87c0, [0x1eef] = 0x87d1,
  [0x1ef0] = 0x87b3, [0x1ef1] = 0x87d2, [0x1ef2] = 0x87c6, [0x1ef3] = 0x87ab,
  [0x1ef4] = 0x87bb, [0x1ef5] = 0x87ba, [0x1ef6] = 0x87c8, [0x1ef7] = 0x87cb,
  [0x1ef8] = 0x893b, [0x1ef9] = 0x8936, [0x1efa] = 0x8944, [0x1efb] = 0x8938,
  [0x1efc] = 0x893d, [0x1efd] = 0x89ac, [0x1efe] = 0x8b0e, [0x1eff] = 0x8b17,
  [0x1f00] = 0x8b19, [0x1f01] = 0x8b1b, [0x1f02] = 0x8b0a, [0x1f03] = 0x8b20,
  [0x1f04] = 0x8b1d, [0x1f05] = 0x8b04, [0x1f06] = 0x8b10, [0x1f07] = 0x8c41,
  [0x1f08] = 0x8c3f, [0x1f09] = 0x8c73, [0x1f0a] = 0x8cfa, [0x1f0b] = 0x8cfd,
  [0x1f0c] = 0x8cfc, [0x1f0d] = 0x8cf8, [0x1f0e] = 0x8cfb, [0x1f0f] = 0x8da8,
  [0x1f10] = 0x8e49, [0x1f11] = 0x8e4b, [0x1f12] = 0x8e48, [0x1f13] = 0x8e4a,
  [0x1f14] = 0x8f44, [0x1f15] = 0x8f3e, [0x1f16] = 0x8f42, [0x1f17] = 0x8f45,
  [0x1f18] = 0x8f3f, [0x1f19] = 0x907f, [0x1f1a] = 0x907d, [0x1f1b] = 0x9084,
  [0x1f1c] = 0x9081, [0x1f1d] = 0x9082, [0x1f1e] = 0x9080, [0x1f1f] = 0x9139,
  [0x1f20] = 0x91a3, [0x1f21] = 0x919e, [0x1f22] = 0x919c, [0x1f23] = 0x934d,
  [0x1f24] = 0x9382, [0x1f25] = 0x9328, [0x1f26] = 0x9375, [0x1f27] = 0x934a,
  [0x1f28] = 0x9365, [0x1f29] = 0x934b, [0x1f2a] = 0x9318, [0x1f2b] = 0x937e,
  [0x1f2c] = 0x936c, [0x1f2d] = 0x935b, [0x1f2e] = 0x9370, [0x1f2f] = 0x935a,
  [0x1f30] = 0x9354, [0x1f31] = 0x95ca, [0x1f32] = 0x95cb, [0x1f33] = 0x95cc,
  [0x1f34] = 0x95c8, [0x1f35] = 0x95c6, [0x1f36] = 0x96b1, [0x1f37] = 0x96b8,
  [0x1f38] = 0x96d6, [0x1f39] = 0x971c, [0x1f3a] = 0x971e, [0x1f3b] = 0x97a0,
  [0x1f3c] = 0x97d3, [0x1f3d] = 0x9846, [0x1f3e] = 0x98b6, [0x1f3f] = 0x9935,
  [0x1f40] = 0x9a01, [0x1f41] = 0x99ff, [0x1f42] = 0x9bae, [0x1f43] = 0x9bab,
  [0x1f44] = 0x9baa, [0x1f45] = 0x9bad, [0x1f46] = 0x9d3b, [0x1f47] = 0x9d3f,
  [0x1f48] = 0x9e8b, [0x1f49] = 0x9ecf, [0x1f4a] = 0x9ede, [0x1f4b] = 0x9edc,
  [0x1f4c] = 0x9edd, [0x1f4d] = 0x9edb, [0x1f4e] = 0x9f3e, [0x1f4f] = 0x9f4b,
  [0x1f50] = 0x53e2, [0x1f51] = 0x5695, [0x1f52] = 0x56ae, [0x1f53] = 0x58d9,
  [0x1f54] = 0x58d8, [0x1f55] = 0x5b38, [0x1f56] = 0x5f5e, [0x1f57] = 0x61e3,
  [0x1f58] = 0x6233, [0x1f59] = 0x64f4, [0x1f5a] = 0x64f2, [0x1f5b] = 0x64fe,
  [0x1f5c] = 0x6506, [0x1f5d] = 0x64fa, [0x1f5e] = 0x64fb, [0x1f5f] = 0x64f7,
  [0x1f60] = 0x65b7, [0x1f61] = 0x66dc, [0x1f62] = 0x6726, [0x1f63] = 0x6ab3,
  [0x1f64] = 0x6aac, [0x1f65] = 0x6ac3, [0x1f66] = 0x6abb, [0x1f67] = 0x6ab8,
  [0x1f68] = 0x6ac2, [0x1f69] = 0x6aae, [0x1f6a] = 0x6aaf, [0x1f6b] = 0x6b5f,
  [0x1f6c] = 0x6b78, [0x1f6d] = 0x6baf, [0x1f6e] = 0x7009, [0x1f6f] = 0x700b,
  [0x1f70] = 0x6ffe, [0x1f71] = 0x7006, [0x1f72] = 0x6ffa, [0x1f73] = 0x7011,
  [0x1f74] = 0x700f, [0x1f75] = 0x71fb, [0x1f76] = 0x71fc, [0x1f77] = 0x71fe,
  [0x1f78] = 0x71f8, [0x1f79] = 0x7377, [0x1f7a] = 0x7375, [0x1f7b] = 0x74a7,
  [0x1f7c] = 0x74bf, [0x1f7d] = 0x7515, [0x1f7e] = 0x7656, [0x1f7f] = 0x7658,
  [0x1f80] = 0x7652, [0x1f81] = 0x77bd, [0x1f82] = 0x77bf, [0x1f83] = 0x77bb,
  [0x1f84] = 0x77bc, [0x1f85] = 0x790e, [0x1f86] = 0x79ae, [0x1f87] = 0x7a61,
  [0x1f88] = 0x7a62, [0x1f89] = 0x7a60, [0x1f8a] = 0x7ac4, [0x1f8b] = 0x7ac5,
  [0x1f8c] = 0x7c2b, [0x1f8d] = 0x7c27, [0x1f8e] = 0x7c2a, [0x1f8f] = 0x7c1e,
  [0x1f90] = 0x7c23, [0x1f91] = 0x7c21, [0x1f92] = 0x7ce7, [0x1f93] = 0x7e54,
  [0x1f94] = 0x7e55, [0x1f95] = 0x7e5e, [0x1f96] = 0x7e5a, [0x1f97] = 0x7e61,
  [0x1f98] = 0x7e52, [0x1f99] = 0x7e59, [0x1f9a] = 0x7f48, [0x1f9b] = 0x7ff9,
  [0x1f9c] = 0x7ffb, [0x1f9d] = 0x8077, [0x1f9e] = 0x8076, [0x1f9f] = 0x81cd,
  [0x1fa0] = 0x81cf, [0x1fa1] = 0x820a, [0x1fa2] = 0x85cf, [0x1fa3] = 0x85a9,
  [0x1fa4] = 0x85cd, [0x1fa5] = 0x85d0, [0x1fa6] = 0x85c9, [0x1fa7] = 0x85b0,
  [0x1fa8] = 0x85ba, [0x1fa9] = 0x85b9, [0x1faa] = 0x87ef, [0x1fab] = 0x87ec,
  [0x1fac] = 0x87f2, [0x1fad] = 0x87e0, [0x1fae] = 0x8986, [0x1faf] = 0x89b2,
  [0x1fb0] = 0x89f4, [0x1fb1] = 0x8b28, [0x1fb2] = 0x8b39, [0x1fb3] = 0x8b2c,
  [0x1fb4] = 0x8b2b, [0x1fb5] = 0x8c50, [0x1fb6] = 0x8d05, [0x1fb7] = 0x8e59,
  [0x1fb8] = 0x8e63, [0x1fb9] = 0x8e66, [0x1fba] = 0x8e64, [0x1fbb] = 0x8e5f,
  [0x1fbc] = 0x8e55, [0x1fbd] = 0x8ec0, [0x1fbe] = 0x8f49, [0x1fbf] = 0x8f4d,
  [0x1fc0] = 0x9087, [0x1fc1] = 0x9083, [0x1fc2] = 0x9088, [0x1fc3] = 0x91ab,
  [0x1fc4] = 0x91ac, [0x1fc5] = 0x91d0, [0x1fc6] = 0x9394, [0x1fc7] = 0x938a,
  [0x1fc8] = 0x9396, [0x1fc9] = 0x93a2, [0x1fca] = 0x93b3, [0x1fcb] = 0x93ae,
  [0x1fcc] = 0x93ac, [0x1fcd] = 0x93b0, [0x1fce] = 0x9398, [0x1fcf] = 0x939a,
  [0x1fd0] = 0x9397, [0x1fd1] = 0x95d4, [0x1fd2] = 0x95d6, [0x1fd3] = 0x95d0,
  [0x1fd4] = 0x95d5, [0x1fd5] = 0x96e2, [0x1fd6] = 0x96dc, [0x1fd7] = 0x96d9,
  [0x1fd8] = 0x96db, [0x1fd9] = 0x96de, [0x1fda] = 0x9724, [0x1fdb] = 0x97a3,
  [0x1fdc] = 0x97a6, [0x1fdd] = 0x97ad, [0x1fde] = 0x97f9, [0x1fdf] = 0x984d,
  [0x1fe0] = 0x984f, [0x1fe1] = 0x984c, [0x1fe2] = 0x984e, [0x1fe3] = 0x9853,
  [0x1fe4] = 0x98ba, [0x1fe5] = 0x993e, [0x1fe6] = 0x993f, [0x1fe7] = 0x993d,
  [0x1fe8] = 0x992e, [0x1fe9] = 0x99a5, [0x1fea] = 0x9a0e, [0x1feb] = 0x9ac1,
  [0x1fec] = 0x9b03, [0x1fed] = 0x9b06, [0x1fee] = 0x9b4f, [0x1fef] = 0x9b4e,
  [0x1ff0] = 0x9b4d, [0x1ff1] = 0x9bca, [0x1ff2] = 0x9bc9, [0x1ff3] = 0x9bfd,
  [0x1ff4] = 0x9bc8, [0x1ff5] = 0x9bc0, [0x1ff6] = 0x9d51, [0x1ff7] = 0x9d5d,
  [0x1ff8] = 0x9d60, [0x1ff9] = 0x9ee0, [0x1ffa] = 0x9f15, [0x1ffb] = 0x9f2c,
  [0x1ffc] = 0x5133, [0x1ffd] = 0x56a5, [0x1ffe] = 0x56a8, [0x1fff] = 0x58de,
  [0x2000] = 0x58df, [0x2001] = 0x58e2, [0x2002] = 0x5bf5, [0x2003] = 0x9f90,
  [0x2004] = 0x5eec, [0x2005] = 0x61f2, [0x2006] = 0x61f7, [0x2007] = 0x61f6,
  [0x2008] = 0x61f5, [0x2009] = 0x6500, [0x200a] = 0x650f, [0x200b] = 0x66e0,
  [0x200c] = 0x66dd, [0x200d] = 0x6ae5, [0x200e] = 0x6add, [0x200f] = 0x6ada,
  [0x2010] = 0x6ad3, [0x2011] = 0x701b, [0x2012] = 0x701f, [0x2013] = 0x7028,
  [0x2014] = 0x701a, [0x2015] = 0x701d, [0x2016] = 0x7015, [0x2017] = 0x7018,
  [0x2018] = 0x7206, [0x2019] = 0x720d, [0x201a] = 0x7258, [0x201b] = 0x72a2,
  [0x201c] = 0x7378, [0x201d] = 0x737a, [0x201e] = 0x74bd, [0x201f] = 0x74ca,
  [0x2020] = 0x74e3, [0x2021] = 0x7587, [0x2022] = 0x7586, [0x2023] = 0x765f,
  [0x2024] = 0x7661, [0x2025] = 0x77c7, [0x2026] = 0x7919, [0x2027] = 0x79b1,
  [0x2028] = 0x7a6b, [0x2029] = 0x7a69, [0x202a] = 0x7c3e, [0x202b] = 0x7c3f,
  [0x202c] = 0x7c38, [0x202d] = 0x7c3d, [0x202e] = 0x7c37, [0x202f] = 0x7c40,
  [0x2030] = 0x7e6b, [0x2031] = 0x7e6d, [0x2032] = 0x7e79, [0x2033] = 0x7e69,
  [0x2034] = 0x7e6a, [0x2035] = 0x7e73, [0x2036] = 0x7f85, [0x2037] = 0x7fb6,
  [0x2038] = 0x7fb9, [0x2039] = 0x7fb8, [0x203a] = 0x81d8, [0x203b] = 0x85e9,
  [0x203c] = 0x85dd, [0x203d] = 0x85ea, [0x203e] = 0x85d5, [0x203f] = 0x85e4,
  [0x2040] = 0x85e5, [0x2041] = 0x85f7, [0x2042] = 0x87fb, [0x2043] = 0x8805,
  [0x2044] = 0x880d, [0x2045] = 0x87f9, [0x2046] = 0x87fe, [0x2047] = 0x8960,
  [0x2048] = 0x895f, [0x2049] = 0x8956, [0x204a] = 0x895e, [0x204b] = 0x8b41,
  [0x204c] = 0x8b5c, [0x204d] = 0x8b58, [0x204e] = 0x8b49, [0x204f] = 0x8b5a,
  [0x2050] = 0x8b4e, [0x2051] = 0x8b4f, [0x2052] = 0x8b46, [0x2053] = 0x8b59,
  [0x2054] = 0x8d08, [0x2055] = 0x8d0a, [0x2056] = 0x8e7c, [0x2057] = 0x8e72,
  [0x2058] = 0x8e87, [0x2059] = 0x8e76, [0x205a] = 0x8e6c, [0x205b] = 0x8e7a,
  [0x205c] = 0x8e74, [0x205d] = 0x8f54, [0x205e] = 0x8f4e, [0x205f] = 0x8fad,
  [0x2060] = 0x908a, [0x2061] = 0x908b, [0x2062] = 0x91b1, [0x2063] = 0x91ae,
  [0x2064] = 0x93e1, [0x2065] = 0x93d1, [0x2066] = 0x93df, [0x2067] = 0x93c3,
  [0x2068] = 0x93c8, [0x2069] = 0x93dc, [0x206a] = 0x93dd, [0x206b] = 0x93d6,
  [0x206c] = 0x93e2, [0x206d] = 0x93cd, [0x206e] = 0x93d8, [0x206f] = 0x93e4,
  [0x2070] = 0x93d7, [0x2071] = 0x93e8, [0x2072] = 0x95dc, [0x2073] = 0x96b4,
  [0x2074] = 0x96e3, [0x2075] = 0x972a, [0x2076] = 0x9727, [0x2077] = 0x9761,
  [0x2078] = 0x97dc, [0x2079] = 0x97fb, [0x207a] = 0x985e, [0x207b] = 0x9858,
  [0x207c] = 0x985b, [0x207d] = 0x98bc, [0x207e] = 0x9945, [0x207f] = 0x9949,
  [0x2080] = 0x9a16, [0x2081] = 0x9a19, [0x2082] = 0x9b0d, [0x2083] = 0x9be8,
  [0x2084] = 0x9be7, [0x2085] = 0x9bd6, [0x2086] = 0x9bdb, [0x2087] = 0x9d89,
  [0x2088] = 0x9d61, [0x2089] = 0x9d72, [0x208a] = 0x9d6a, [0x208b] = 0x9d6c,
  [0x208c] = 0x9e92, [0x208d] = 0x9e97, [0x208e] = 0x9e93, [0x208f] = 0x9eb4,
  [0x2090] = 0x52f8, [0x2091] = 0x56b7, [0x2092] = 0x56b6, [0x2093] = 0x56b4,
  [0x2094] = 0x56bc, [0x2095] = 0x58e4, [0x2096] = 0x5b40, [0x2097] = 0x5b43,
  [0x2098] = 0x5b7d, [0x2099] = 0x5bf6, [0x209a] = 0x5dc9, [0x209b] = 0x61f8,
  [0x209c] = 0x61fa, [0x209d] = 0x6518, [0x209e] = 0x6514, [0x209f] = 0x6519,
  [0x20a0] = 0x66e6, [0x20a1] = 0x6727, [0x20a2] = 0x6aec, [0x20a3] = 0x703e,
  [0x20a4] = 0x7030, [0x20a5] = 0x7032, [0x20a6] = 0x7210, [0x20a7] = 0x737b,
  [0x20a8] = 0x74cf, [0x20a9] = 0x7662, [0x20aa] = 0x7665, [0x20ab] = 0x7926,
  [0x20ac] = 0x792a, [0x20ad] = 0x792c, [0x20ae] = 0x792b, [0x20af] = 0x7ac7,
  [0x20b0] = 0x7af6, [0x20b1] = 0x7c4c, [0x20b2] = 0x7c43, [0x20b3] = 0x7c4d,
  [0x20b4] = 0x7cef, [0x20b5] = 0x7cf0, [0x20b6] = 0x8fae, [0x20b7] = 0x7e7d,
  [0x20b8] = 0x7e7c, [0x20b9] = 0x7e82, [0x20ba] = 0x7f4c, [0x20bb] = 0x8000,
  [0x20bc] = 0x81da, [0x20bd] = 0x8266, [0x20be] = 0x85fb, [0x20bf] = 0x85f9,
  [0x20c0] = 0x8611, [0x20c1] = 0x85fa, [0x20c2] = 0x8606, [0x20c3] = 0x860b,
  [0x20c4] = 0x8607, [0x20c5] = 0x860a, [0x20c6] = 0x8814, [0x20c7] = 0x8815,
  [0x20c8] = 0x8964, [0x20c9] = 0x89ba, [0x20ca] = 0x89f8, [0x20cb] = 0x8b70,
  [0x20cc] = 0x8b6c, [0x20cd] = 0x8b66, [0x20ce] = 0x8b6f, [0x20cf] = 0x8b5f,
  [0x20d0] = 0x8b6b, [0x20d1] = 0x8d0f, [0x20d2] = 0x8d0d, [0x20d3] = 0x8e89,
  [0x20d4] = 0x8e81, [0x20d5] = 0x8e85, [0x20d6] = 0x8e82, [0x20d7] = 0x91b4,
  [0x20d8] = 0x91cb, [0x20d9] = 0x9418, [0x20da] = 0x9403, [0x20db] = 0x93fd,
  [0x20dc] = 0x95e1, [0x20dd] = 0x9730, [0x20de] = 0x98c4, [0x20df] = 0x9952,
  [0x20e0] = 0x9951, [0x20e1] = 0x99a8, [0x20e2] = 0x9a2b, [0x20e3] = 0x9a30,
  [0x20e4] = 0x9a37, [0x20e5] = 0x9a35, [0x20e6] = 0x9c13, [0x20e7] = 0x9c0d,
  [0x20e8] = 0x9e79, [0x20e9] = 0x9eb5, [0x20ea] = 0x9ee8, [0x20eb] = 0x9f2f,
  [0x20ec] = 0x9f5f, [0x20ed] = 0x9f63, [0x20ee] = 0x9f61, [0x20ef] = 0x5137,
  [0x20f0] = 0x5138, [0x20f1] = 0x56c1, [0x20f2] = 0x56c0, [0x20f3] = 0x56c2,
  [0x20f4] = 0x5914, [0x20f5] = 0x5c6c, [0x20f6] = 0x5dcd, [0x20f7] = 0x61fc,
  [0x20f8] = 0x61fe, [0x20f9] = 0x651d, [0x20fa] = 0x651c, [0x20fb] = 0x6595,
  [0x20fc] = 0x66e9, [0x20fd] = 0x6afb, [0x20fe] = 0x6b04, [0x20ff] = 0x6afa,
  [0x2100] = 0x6bb2, [0x2101] = 0x704c, [0x2102] = 0x721b, [0x2103] = 0x72a7,
  [0x2104] = 0x74d6, [0x2105] = 0x74d4, [0x2106] = 0x7669, [0x2107] = 0x77d3,
  [0x2108] = 0x7c50, [0x2109] = 0x7e8f, [0x210a] = 0x7e8c, [0x210b] = 0x7fbc,
  [0x210c] = 0x8617, [0x210d] = 0x862d, [0x210e] = 0x861a, [0x210f] = 0x8823,
  [0x2110] = 0x8822, [0x2111] = 0x8821, [0x2112] = 0x881f, [0x2113] = 0x896a,
  [0x2114] = 0x896c, [0x2115] = 0x89bd, [0x2116] = 0x8b74, [0x2117] = 0x8b77,
  [0x2118] = 0x8b7d, [0x2119] = 0x8d13, [0x211a] = 0x8e8a, [0x211b] = 0x8e8d,
  [0x211c] = 0x8e8b, [0x211d] = 0x8f5f, [0x211e] = 0x8faf, [0x211f] = 0x91ba,
  [0x2120] = 0x942e, [0x2121] = 0x9433, [0x2122] = 0x9435, [0x2123] = 0x943a,
  [0x2124] = 0x9438, [0x2125] = 0x9432, [0x2126] = 0x942b, [0x2127] = 0x95e2,
  [0x2128] = 0x9738, [0x2129] = 0x9739, [0x212a] = 0x9732, [0x212b] = 0x97ff,
  [0x212c] = 0x9867, [0x212d] = 0x9865, [0x212e] = 0x9957, [0x212f] = 0x9a45,
  [0x2130] = 0x9a43, [0x2131] = 0x9a40, [0x2132] = 0x9a3e, [0x2133] = 0x9acf,
  [0x2134] = 0x9b54, [0x2135] = 0x9b51, [0x2136] = 0x9c2d, [0x2137] = 0x9c25,
  [0x2138] = 0x9daf, [0x2139] = 0x9db4, [0x213a] = 0x9dc2, [0x213b] = 0x9db8,
  [0x213c] = 0x9e9d, [0x213d] = 0x9eef, [0x213e] = 0x9f19, [0x213f] = 0x9f5c,
  [0x2140] = 0x9f66, [0x2141] = 0x9f67, [0x2142] = 0x513c, [0x2143] = 0x513b,
  [0x2144] = 0x56c8, [0x2145] = 0x56ca, [0x2146] = 0x56c9, [0x2147] = 0x5b7f,
  [0x2148] = 0x5dd4, [0x2149] = 0x5dd2, [0x214a] = 0x5f4e, [0x214b] = 0x61ff,
  [0x214c] = 0x6524, [0x214d] = 0x6b0a, [0x214e] = 0x6b61, [0x214f] = 0x7051,
  [0x2150] = 0x7058, [0x2151] = 0x7380, [0x2152] = 0x74e4, [0x2153] = 0x758a,
  [0x2154] = 0x766e, [0x2155] = 0x766c, [0x2156] = 0x79b3, [0x2157] = 0x7c60,
  [0x2158] = 0x7c5f, [0x2159] = 0x807e, [0x215a] = 0x807d, [0x215b] = 0x81df,
  [0x215c] = 0x8972, [0x215d] = 0x896f, [0x215e] = 0x89fc, [0x215f] = 0x8b80,
  [0x2160] = 0x8d16, [0x2161] = 0x8d17, [0x2162] = 0x8e91, [0x2163] = 0x8e93,
  [0x2164] = 0x8f61, [0x2165] = 0x9148, [0x2166] = 0x9444, [0x2167] = 0x9451,
  [0x2168] = 0x9452, [0x2169] = 0x973d, [0x216a] = 0x973e, [0x216b] = 0x97c3,
  [0x216c] = 0x97c1, [0x216d] = 0x986b, [0x216e] = 0x9955, [0x216f] = 0x9a55,
  [0x2170] = 0x9a4d, [0x2171] = 0x9ad2, [0x2172] = 0x9b1a, [0x2173] = 0x9c49,
  [0x2174] = 0x9c31, [0x2175] = 0x9c3e, [0x2176] = 0x9c3b, [0x2177] = 0x9dd3,
  [0x2178] = 0x9dd7, [0x2179] = 0x9f34, [0x217a] = 0x9f6c, [0x217b] = 0x9f6a,
  [0x217c] = 0x9f94, [0x217d] = 0x56cc, [0x217e] = 0x5dd6, [0x217f] = 0x6200,
  [0x2180] = 0x6523, [0x2181] = 0x652b, [0x2182] = 0x652a, [0x2183] = 0x66ec,
  [0x2184] = 0x6b10, [0x2185] = 0x74da, [0x2186] = 0x7aca, [0x2187] = 0x7c64,
  [0x2188] = 0x7c63, [0x2189] = 0x7c65, [0x218a] = 0x7e93, [0x218b] = 0x7e96,
  [0x218c] = 0x7e94, [0x218d] = 0x81e2, [0x218e] = 0x8638, [0x218f] = 0x863f,
  [0x2190] = 0x8831, [0x2191] = 0x8b8a, [0x2192] = 0x9090, [0x2193] = 0x908f,
  [0x2194] = 0x9463, [0x2195] = 0x9460, [0x2196] = 0x9464, [0x2197] = 0x9768,
  [0x2198] = 0x986f, [0x2199] = 0x995c, [0x219a] = 0x9a5a, [0x219b] = 0x9a5b,
  [0x219c] = 0x9a57, [0x219d] = 0x9ad3, [0x219e] = 0x9ad4, [0x219f] = 0x9ad1,
  [0x21a0] = 0x9c54, [0x21a1] = 0x9c57, [0x21a2] = 0x9c56, [0x21a3] = 0x9de5,
  [0x21a4] = 0x9e9f, [0x21a5] = 0x9ef4, [0x21a6] = 0x56d1, [0x21a7] = 0x58e9,
  [0x21a8] = 0x652c, [0x21a9] = 0x705e, [0x21aa] = 0x7671, [0x21ab] = 0x7672,
  [0x21ac] = 0x77d7, [0x21ad] = 0x7f50, [0x21ae] = 0x7f88, [0x21af] = 0x8836,
  [0x21b0] = 0x8839, [0x21b1] = 0x8862, [0x21b2] = 0x8b93, [0x21b3] = 0x8b92,
  [0x21b4] = 0x8b96, [0x21b5] = 0x8277, [0x21b6] = 0x8d1b, [0x21b7] = 0x91c0,
  [0x21b8] = 0x946a, [0x21b9] = 0x9742, [0x21ba] = 0x9748, [0x21bb] = 0x9744,
  [0x21bc] = 0x97c6, [0x21bd] = 0x9870, [0x21be] = 0x9a5f, [0x21bf] = 0x9b22,
  [0x21c0] = 0x9b58, [0x21c1] = 0x9c5f, [0x21c2] = 0x9df9, [0x21c3] = 0x9dfa,
  [0x21c4] = 0x9e7c, [0x21c5] = 0x9e7d, [0x21c6] = 0x9f07, [0x21c7] = 0x9f77,
  [0x21c8] = 0x9f72, [0x21c9] = 0x5ef3, [0x21ca] = 0x6b16, [0x21cb] = 0x7063,
  [0x21cc] = 0x7c6c, [0x21cd] = 0x7c6e, [0x21ce] = 0x883b, [0x21cf] = 0x89c0,
  [0x21d0] = 0x8ea1, [0x21d1] = 0x91c1, [0x21d2] = 0x9472, [0x21d3] = 0x9470,
  [0x21d4] = 0x9871, [0x21d5] = 0x995e, [0x21d6] = 0x9ad6, [0x21d7] = 0x9b23,
  [0x21d8] = 0x9ecc, [0x21d9] = 0x7064, [0x21da] = 0x77da, [0x21db] = 0x8b9a,
  [0x21dc] = 0x9477, [0x21dd] = 0x97c9, [0x21de] = 0x9a62, [0x21df] = 0x9a65,
  [0x21e0] = 0x7e9c, [0x21e1] = 0x8b9c, [0x21e2] = 0x8eaa, [0x21e3] = 0x91c5,
  [0x21e4] = 0x947d, [0x21e5] = 0x947e, [0x21e6] = 0x947c, [0x21e7] = 0x9c77,
  [0x21e8] = 0x9c78, [0x21e9] = 0x9ef7, [0x21ea] = 0x8c54, [0x21eb] = 0x947f,
  [0x21ec] = 0x9e1a, [0x21ed] = 0x7228, [0x21ee] = 0x9a6a, [0x21ef] = 0x9b31,
  [0x21f0] = 0x9e1b, [0x21f1] = 0x9e1e, [0x21f2] = 0x7c72,
};


/* Some Latin1 characters, starting at U+00a7.  */
const char __cns11643l1_from_ucs4_tab1[][2] =
{
  [0x00] = "\x21\x70", [0x09] = "\x22\x78", [0x0a] = "\x22\x34",
  [0x10] = "\x21\x31", [0x30] = "\x22\x32", [0x50] = "\x22\x33"
};


/* Some phonetic modifiers, starting at U+02c7.  */
const char __cns11643l1_from_ucs4_tab2[][2] =
{
  [0x00] = "\x25\x6f", [0x02] = "\x25\x6d", [0x03] = "\x25\x6e",
  [0x04] = "\x25\x70", [0x12] = "\x25\x6c"
};

/* Greek alphabet.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x03' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x391,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab3[][2] =
{
  [0x0000] = "\x24\x75", [0x0001] = "\x24\x76", [0x0002] = "\x24\x77",
  [0x0003] = "\x24\x78", [0x0004] = "\x24\x79", [0x0005] = "\x24\x7a",
  [0x0006] = "\x24\x7b", [0x0007] = "\x24\x7c", [0x0008] = "\x24\x7d",
  [0x0009] = "\x24\x7e", [0x000a] = "\x25\x21", [0x000b] = "\x25\x22",
  [0x000c] = "\x25\x23", [0x000d] = "\x25\x24", [0x000e] = "\x25\x25",
  [0x000f] = "\x25\x26", [0x0010] = "\x25\x27", [0x0012] = "\x25\x28",
  [0x0013] = "\x25\x29", [0x0014] = "\x25\x2a", [0x0015] = "\x25\x2b",
  [0x0016] = "\x25\x2c", [0x0017] = "\x25\x2d", [0x0018] = "\x25\x2e",
  [0x0020] = "\x25\x2f", [0x0021] = "\x25\x30", [0x0022] = "\x25\x31",
  [0x0023] = "\x25\x32", [0x0024] = "\x25\x33", [0x0025] = "\x25\x34",
  [0x0026] = "\x25\x35", [0x0027] = "\x25\x36", [0x0028] = "\x25\x37",
  [0x0029] = "\x25\x38", [0x002a] = "\x25\x39", [0x002b] = "\x25\x3a",
  [0x002c] = "\x25\x3b", [0x002d] = "\x25\x3c", [0x002e] = "\x25\x3d",
  [0x002f] = "\x25\x3e", [0x0030] = "\x25\x3f", [0x0032] = "\x25\x40",
  [0x0033] = "\x25\x41", [0x0034] = "\x25\x42", [0x0035] = "\x25\x43",
  [0x0036] = "\x25\x44", [0x0037] = "\x25\x45", [0x0038] = "\x25\x46",
};

/* General punctuation.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x20' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2013,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab4[][2] =
{
  [0x0000] = "\x21\x39", [0x0001] = "\x21\x37", [0x0003] = "\x22\x5d",
  [0x0005] = "\x21\x64", [0x0006] = "\x21\x65", [0x0009] = "\x21\x66",
  [0x000a] = "\x21\x67", [0x0012] = "\x21\x2d", [0x0013] = "\x21\x2c",
  [0x001f] = "\x21\x6a", [0x0022] = "\x21\x6b", [0x0028] = "\x21\x6f",
  [0x002b] = "\x22\x23",
};


const char __cns11643l1_from_ucs4_tab5[][2] =
{
  [0x00] = "\x22\x58", [0x01] = "\x22\x55", [0x02] = "\x22\x57",
  [0x03] = "\x22\x56", [0x06] = "\x22\x59", [0x07] = "\x22\x5a",
  [0x08] = "\x22\x5c", [0x09] = "\x22\x5b"
};

/* Mathematical operators.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x22[1-6]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2215,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab6[][2] =
{
  [0x0000] = "\x22\x61", [0x0005] = "\x22\x35", [0x0009] = "\x22\x3c",
  [0x000a] = "\x22\x49", [0x000b] = "\x22\x48", [0x0014] = "\x22\x45",
  [0x0015] = "\x22\x46", [0x0016] = "\x22\x4d", [0x0019] = "\x22\x4e",
  [0x001f] = "\x22\x50", [0x0020] = "\x22\x4f", [0x0027] = "\x22\x44",
  [0x003d] = "\x22\x3d", [0x004b] = "\x22\x3b", [0x004c] = "\x22\x3e",
  [0x0051] = "\x22\x39", [0x0052] = "\x22\x3a",
};

/* Graphic pictures for control codes.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x24[0-2]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2400,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab7[][2] =
{
  [0x0000] = "\x42\x21", [0x0001] = "\x42\x22", [0x0002] = "\x42\x23",
  [0x0003] = "\x42\x24", [0x0004] = "\x42\x25", [0x0005] = "\x42\x26",
  [0x0006] = "\x42\x27", [0x0007] = "\x42\x28", [0x0008] = "\x42\x29",
  [0x0009] = "\x42\x2a", [0x000a] = "\x42\x2b", [0x000b] = "\x42\x2c",
  [0x000c] = "\x42\x2d", [0x000d] = "\x42\x2e", [0x000e] = "\x42\x2f",
  [0x000f] = "\x42\x30", [0x0010] = "\x42\x31", [0x0011] = "\x42\x32",
  [0x0012] = "\x42\x33", [0x0013] = "\x42\x34", [0x0014] = "\x42\x35",
  [0x0015] = "\x42\x36", [0x0016] = "\x42\x37", [0x0017] = "\x42\x38",
  [0x0018] = "\x42\x39", [0x0019] = "\x42\x3a", [0x001a] = "\x42\x3b",
  [0x001b] = "\x42\x3c", [0x001c] = "\x42\x3d", [0x001d] = "\x42\x3e",
  [0x001e] = "\x42\x3f", [0x001f] = "\x42\x40", [0x0021] = "\x42\x41",
};

/* Circled and Parenthesized numbers.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x24[67]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2460,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab8[][2] =
{
  [0x0000] = "\x26\x21", [0x0001] = "\x26\x22", [0x0002] = "\x26\x23",
  [0x0003] = "\x26\x24", [0x0004] = "\x26\x25", [0x0005] = "\x26\x26",
  [0x0006] = "\x26\x27", [0x0007] = "\x26\x28", [0x0008] = "\x26\x29",
  [0x0009] = "\x26\x2a", [0x0014] = "\x26\x2b", [0x0015] = "\x26\x2c",
  [0x0016] = "\x26\x2d", [0x0017] = "\x26\x2e", [0x0018] = "\x26\x2f",
  [0x0019] = "\x26\x30", [0x001a] = "\x26\x31", [0x001b] = "\x26\x32",
  [0x001c] = "\x26\x33", [0x001d] = "\x26\x34",
};

/* Circled and Parenthesized numbers.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x2[56]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2500,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab9[][2] =
{
  [0x0000] = "\x23\x39", [0x0002] = "\x23\x3a", [0x000c] = "\x23\x3c",
  [0x0010] = "\x23\x3d", [0x0014] = "\x23\x3e", [0x0018] = "\x23\x3f",
  [0x001c] = "\x23\x37", [0x0024] = "\x23\x36", [0x002c] = "\x23\x35",
  [0x0034] = "\x23\x34", [0x003c] = "\x23\x33", [0x0050] = "\x23\x44",
  [0x005e] = "\x23\x45", [0x0061] = "\x23\x47", [0x006a] = "\x23\x46",
  [0x006d] = "\x23\x40", [0x006e] = "\x23\x41", [0x006f] = "\x23\x43",
  [0x0070] = "\x23\x42", [0x0071] = "\x23\x4c", [0x0072] = "\x23\x4d",
  [0x0073] = "\x23\x4e", [0x0081] = "\x23\x24", [0x0082] = "\x23\x25",
  [0x0083] = "\x23\x26", [0x0084] = "\x23\x27", [0x0085] = "\x23\x28",
  [0x0086] = "\x23\x29", [0x0087] = "\x23\x2a", [0x0088] = "\x23\x2b",
  [0x0089] = "\x23\x32", [0x008a] = "\x23\x31", [0x008b] = "\x23\x30",
  [0x008c] = "\x23\x2f", [0x008d] = "\x23\x2e", [0x008e] = "\x23\x2d",
  [0x008f] = "\x23\x2c", [0x0094] = "\x23\x38", [0x0095] = "\x23\x3b",
  [0x00a0] = "\x21\x7c", [0x00a1] = "\x21\x7b", [0x00b2] = "\x21\x75",
  [0x00b3] = "\x21\x74", [0x00bc] = "\x21\x7e", [0x00bd] = "\x21\x7d",
  [0x00c6] = "\x21\x7a", [0x00c7] = "\x21\x79", [0x00cb] = "\x21\x72",
  [0x00ce] = "\x21\x76", [0x00cf] = "\x21\x73", [0x00e2] = "\x23\x48",
  [0x00e3] = "\x23\x49", [0x00e4] = "\x23\x4b", [0x00e5] = "\x23\x4a",
  [0x0105] = "\x21\x78", [0x0106] = "\x21\x77", [0x0109] = "\x22\x54",
  [0x0140] = "\x22\x51", [0x0141] = "\x22\x53", [0x0142] = "\x22\x52",
};

/* CJK punctuation and Hangzhou-style numerals.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x30[0-2]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x3000,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab10[][2] =
{
  [0x0000] = "\x21\x21", [0x0001] = "\x21\x23", [0x0002] = "\x21\x24",
  [0x0003] = "\x21\x71", [0x0008] = "\x21\x52", [0x0009] = "\x21\x53",
  [0x000a] = "\x21\x4e", [0x000b] = "\x21\x4f", [0x000c] = "\x21\x56",
  [0x000d] = "\x21\x57", [0x000e] = "\x21\x5a", [0x000f] = "\x21\x5b",
  [0x0010] = "\x21\x4a", [0x0011] = "\x21\x4b", [0x0012] = "\x22\x65",
  [0x0014] = "\x21\x46", [0x0015] = "\x21\x47", [0x001d] = "\x21\x68",
  [0x001e] = "\x21\x69", [0x0021] = "\x24\x35", [0x0022] = "\x24\x36",
  [0x0023] = "\x24\x37", [0x0024] = "\x24\x38", [0x0025] = "\x24\x39",
  [0x0026] = "\x24\x3a", [0x0027] = "\x24\x3b", [0x0028] = "\x24\x3c",
  [0x0029] = "\x24\x3d",
};

/* Squared latin abbreviations.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x33' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x338e,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab11[][2] =
{
  [0x0000] = "\x22\x75", [0x0001] = "\x22\x76", [0x000e] = "\x22\x70",
  [0x000f] = "\x22\x71", [0x0010] = "\x22\x72", [0x0013] = "\x22\x74",
  [0x0036] = "\x22\x77", [0x0040] = "\x22\x73", [0x0043] = "\x22\x4c",
  [0x0044] = "\x22\x4b", [0x0047] = "\x22\x6f",
};

/* CJK Ideographs.  The table can be created using

   egrep '^0x1' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0x[4-9]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x4e00,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab12[][2] =
{
  [0x0000] = "\x44\x21", [0x0001] = "\x44\x23", [0x0003] = "\x44\x24",
  [0x0008] = "\x44\x37", [0x0009] = "\x44\x35", [0x000a] = "\x44\x38",
  [0x000b] = "\x44\x36", [0x000d] = "\x44\x62", [0x0010] = "\x44\x61",
  [0x0011] = "\x44\x60", [0x0014] = "\x45\x62", [0x0015] = "\x45\x61",
  [0x0016] = "\x45\x60", [0x0018] = "\x45\x63", [0x0019] = "\x45\x5f",
  [0x001e] = "\x47\x22", [0x001f] = "\x47\x23", [0x0026] = "\x4b\x64",
  [0x002b] = "\x44\x39", [0x002d] = "\x44\x63", [0x0030] = "\x44\x64",
  [0x0032] = "\x48\x6b", [0x0038] = "\x44\x3a", [0x0039] = "\x44\x65",
  [0x003b] = "\x45\x64", [0x0043] = "\x44\x25", [0x0045] = "\x44\x3c",
  [0x0048] = "\x44\x3d", [0x004b] = "\x44\x66", [0x004d] = "\x45\x65",
  [0x004e] = "\x45\x67", [0x004f] = "\x45\x66", [0x0052] = "\x47\x24",
  [0x0053] = "\x47\x25", [0x0056] = "\x4b\x65", [0x0058] = "\x53\x7d",
  [0x0059] = "\x44\x22", [0x005d] = "\x44\x26", [0x005e] = "\x44\x3f",
  [0x005f] = "\x44\x3e", [0x0069] = "\x47\x26", [0x0073] = "\x4b\x66",
  [0x007e] = "\x58\x71", [0x0082] = "\x63\x2a", [0x0086] = "\x44\x27",
  [0x0088] = "\x44\x68", [0x008b] = "\x4b\x67", [0x008c] = "\x44\x28",
  [0x008e] = "\x44\x40", [0x0091] = "\x44\x69", [0x0092] = "\x44\x6b",
  [0x0094] = "\x44\x6c", [0x0095] = "\x44\x6a", [0x0099] = "\x47\x27",
  [0x009b] = "\x4b\x68", [0x009e] = "\x4b\x69", [0x009f] = "\x4f\x67",
  [0x00a0] = "\x27\x28", [0x00a1] = "\x44\x41", [0x00a2] = "\x44\x6d",
  [0x00a4] = "\x47\x28", [0x00a5] = "\x47\x2a", [0x00a6] = "\x47\x29",
  [0x00a8] = "\x48\x6c", [0x00ab] = "\x4b\x6a", [0x00ac] = "\x4b\x6b",
  [0x00ad] = "\x4f\x68", [0x00ae] = "\x4f\x69", [0x00b3] = "\x53\x7e",
  [0x00ba] = "\x44\x29", [0x00c0] = "\x44\x6f", [0x00c1] = "\x44\x6e",
  [0x00c3] = "\x44\x70", [0x00c4] = "\x44\x76", [0x00c6] = "\x44\x71",
  [0x00c7] = "\x44\x72", [0x00ca] = "\x44\x74", [0x00cb] = "\x44\x75",
  [0x00cd] = "\x44\x73", [0x00d4] = "\x45\x6a", [0x00d5] = "\x45\x6b",
  [0x00d6] = "\x45\x6c", [0x00d7] = "\x45\x6d", [0x00d8] = "\x45\x69",
  [0x00d9] = "\x45\x70", [0x00de] = "\x45\x71", [0x00df] = "\x46\x23",
  [0x00e3] = "\x45\x6e", [0x00e4] = "\x45\x6f", [0x00e5] = "\x45\x68",
  [0x00f0] = "\x47\x37", [0x00f2] = "\x47\x34", [0x00f3] = "\x47\x38",
  [0x00f6] = "\x47\x35", [0x00fb] = "\x47\x36", [0x00fd] = "\x47\x39",
  [0x00ff] = "\x47\x2b", [0x0101] = "\x47\x3a", [0x0109] = "\x47\x2c",
  [0x010a] = "\x47\x2e", [0x010b] = "\x47\x3b", [0x010d] = "\x47\x30",
  [0x010f] = "\x47\x33", [0x0110] = "\x47\x31", [0x0111] = "\x47\x32",
  [0x0115] = "\x47\x2f", [0x0119] = "\x47\x2d", [0x012f] = "\x49\x24",
  [0x0130] = "\x48\x75", [0x0134] = "\x48\x72", [0x0136] = "\x49\x26",
  [0x0138] = "\x48\x7a", [0x013a] = "\x48\x79", [0x013c] = "\x48\x7d",
  [0x013d] = "\x48\x78", [0x0143] = "\x48\x7b", [0x0146] = "\x48\x7e",
  [0x0147] = "\x48\x6f", [0x0148] = "\x49\x29", [0x014d] = "\x48\x6d",
  [0x014e] = "\x49\x25", [0x014f] = "\x48\x6e", [0x0150] = "\x48\x76",
  [0x0151] = "\x48\x77", [0x0154] = "\x48\x7c", [0x0155] = "\x48\x74",
  [0x0157] = "\x48\x70", [0x0159] = "\x49\x27", [0x015a] = "\x49\x2a",
  [0x015b] = "\x48\x73", [0x015c] = "\x49\x22", [0x015d] = "\x49\x28",
  [0x015e] = "\x48\x71", [0x0160] = "\x49\x23", [0x0163] = "\x49\x21",
  [0x0169] = "\x4b\x79", [0x016c] = "\x4b\x71", [0x016f] = "\x4b\x6c",
  [0x0170] = "\x4b\x76", [0x0173] = "\x4b\x6f", [0x0175] = "\x4b\x77",
  [0x017a] = "\x4c\x21", [0x017b] = "\x4b\x7a", [0x017e] = "\x4b\x7c",
  [0x017f] = "\x4b\x70", [0x0183] = "\x4b\x75", [0x0186] = "\x4b\x74",
  [0x0188] = "\x4b\x78", [0x018b] = "\x4b\x73", [0x018d] = "\x4b\x6e",
  [0x018f] = "\x4b\x7d", [0x0191] = "\x4b\x7e", [0x0196] = "\x4b\x7b",
  [0x019b] = "\x4b\x72", [0x019d] = "\x4b\x6d", [0x01ae] = "\x4f\x78",
  [0x01af] = "\x4f\x6c", [0x01b5] = "\x4f\x6b", [0x01b6] = "\x4f\x73",
  [0x01b7] = "\x50\x21", [0x01bf] = "\x4f\x6d", [0x01c2] = "\x4f\x7b",
  [0x01c3] = "\x4f\x72", [0x01c4] = "\x4f\x7a", [0x01ca] = "\x4f\x76",
  [0x01ce] = "\x4f\x7d", [0x01cf] = "\x4f\x70", [0x01d0] = "\x4f\x79",
  [0x01d1] = "\x4f\x6f", [0x01d7] = "\x4f\x77", [0x01d8] = "\x4f\x74",
  [0x01da] = "\x4f\x7c", [0x01dd] = "\x4f\x71", [0x01de] = "\x4f\x7e",
  [0x01df] = "\x4f\x75", [0x01e0] = "\x4f\x6e", [0x01e1] = "\x4f\x6a",
  [0x01ee] = "\x54\x3a", [0x01ef] = "\x54\x24", [0x01f1] = "\x54\x34",
  [0x01f3] = "\x54\x39", [0x01f8] = "\x54\x27", [0x01fa] = "\x54\x30",
  [0x01fe] = "\x54\x3d", [0x0200] = "\x54\x31", [0x0206] = "\x54\x2a",
  [0x0209] = "\x54\x3f", [0x020b] = "\x54\x36", [0x020c] = "\x54\x21",
  [0x020d] = "\x54\x22", [0x020f] = "\x59\x24", [0x0211] = "\x54\x2f",
  [0x0212] = "\x54\x2e", [0x0214] = "\x54\x32", [0x0216] = "\x54\x29",
  [0x0218] = "\x54\x38", [0x0219] = "\x54\x37", [0x021a] = "\x54\x2d",
  [0x021f] = "\x54\x2c", [0x0221] = "\x54\x35", [0x0223] = "\x54\x23",
  [0x0225] = "\x54\x26", [0x0226] = "\x54\x25", [0x0228] = "\x54\x33",
  [0x0229] = "\x54\x28", [0x022a] = "\x54\x3c", [0x022b] = "\x54\x3e",
  [0x022d] = "\x54\x3b", [0x023c] = "\x54\x2b", [0x0243] = "\x58\x76",
  [0x0247] = "\x58\x75", [0x0249] = "\x58\x79", [0x024c] = "\x58\x77",
  [0x024e] = "\x58\x7c", [0x024f] = "\x59\x23", [0x0255] = "\x58\x7d",
  [0x025a] = "\x58\x78", [0x025c] = "\x58\x74", [0x0265] = "\x58\x7a",
  [0x026d] = "\x59\x26", [0x026f] = "\x59\x25", [0x0274] = "\x59\x21",
  [0x0275] = "\x58\x7e", [0x0276] = "\x58\x7b", [0x0277] = "\x59\x22",
  [0x027a] = "\x58\x72", [0x027d] = "\x58\x73", [0x0280] = "\x5e\x2e",
  [0x0285] = "\x5e\x2b", [0x028d] = "\x5e\x2a", [0x0291] = "\x5e\x2d",
  [0x0296] = "\x5e\x2f", [0x0298] = "\x5e\x30", [0x0299] = "\x5e\x2c",
  [0x029a] = "\x5e\x31", [0x02a2] = "\x5e\x29", [0x02ac] = "\x63\x31",
  [0x02ad] = "\x63\x2b", [0x02af] = "\x63\x34", [0x02b2] = "\x63\x2d",
  [0x02b3] = "\x63\x2e", [0x02b5] = "\x63\x2c", [0x02b7] = "\x63\x32",
  [0x02bb] = "\x63\x33", [0x02be] = "\x63\x30", [0x02c5] = "\x63\x2f",
  [0x02c7] = "\x63\x35", [0x02ce] = "\x67\x7c", [0x02cf] = "\x67\x79",
  [0x02d1] = "\x67\x7a", [0x02d5] = "\x67\x78", [0x02d6] = "\x67\x75",
  [0x02da] = "\x67\x77", [0x02e5] = "\x67\x74", [0x02e7] = "\x67\x72",
  [0x02e9] = "\x67\x7d", [0x02ed] = "\x67\x76", [0x02ee] = "\x67\x73",
  [0x02f1] = "\x67\x7b", [0x02f5] = "\x6c\x23", [0x02f9] = "\x6c\x24",
  [0x02fb] = "\x6c\x22", [0x0300] = "\x6c\x21", [0x0302] = "\x6c\x25",
  [0x0304] = "\x6b\x7e", [0x0305] = "\x6c\x28", [0x0308] = "\x6c\x26",
  [0x0309] = "\x6c\x27", [0x0310] = "\x70\x33", [0x0312] = "\x70\x30",
  [0x0314] = "\x70\x32", [0x0315] = "\x70\x34", [0x0318] = "\x70\x31",
  [0x031f] = "\x73\x41", [0x0321] = "\x73\x42", [0x032a] = "\x73\x40",
  [0x0332] = "\x73\x43", [0x0333] = "\x78\x2b", [0x0337] = "\x7a\x62",
  [0x0338] = "\x7a\x63", [0x033b] = "\x7b\x58", [0x033c] = "\x7b\x57",
  [0x033f] = "\x44\x2a", [0x0340] = "\x44\x42", [0x0341] = "\x44\x78",
  [0x0343] = "\x44\x77", [0x0344] = "\x45\x73", [0x0345] = "\x45\x72",
  [0x0346] = "\x47\x3e", [0x0347] = "\x47\x3d", [0x0348] = "\x47\x3f",
  [0x0349] = "\x47\x3c", [0x034b] = "\x49\x2c", [0x034c] = "\x49\x2b",
  [0x034d] = "\x49\x2d", [0x0352] = "\x4c\x23", [0x0354] = "\x4c\x22",
  [0x0355] = "\x4c\x24", [0x0357] = "\x50\x22", [0x0359] = "\x22\x79",
  [0x035b] = "\x22\x7a", [0x035c] = "\x59\x27", [0x035d] = "\x22\x7c",
  [0x035e] = "\x22\x7b", [0x0361] = "\x22\x7d", [0x0362] = "\x67\x7e",
  [0x0363] = "\x22\x7e", [0x0365] = "\x44\x2b", [0x0367] = "\x44\x79",
  [0x0368] = "\x47\x40", [0x0369] = "\x4c\x25", [0x036b] = "\x44\x2c",
  [0x036c] = "\x44\x7c", [0x036d] = "\x44\x7a", [0x036e] = "\x44\x7b",
  [0x0371] = "\x47\x41", [0x0375] = "\x49\x2e", [0x0376] = "\x4c\x27",
  [0x0377] = "\x4c\x26", [0x0378] = "\x4c\x28", [0x037c] = "\x54\x40",
  [0x0380] = "\x70\x35", [0x0389] = "\x45\x74", [0x038a] = "\x45\x75",
  [0x038d] = "\x47\x42", [0x0391] = "\x50\x24", [0x0392] = "\x50\x23",
  [0x0395] = "\x59\x28", [0x0397] = "\x44\x7d", [0x03a0] = "\x50\x25",
  [0x03a2] = "\x54\x43", [0x03a4] = "\x54\x41", [0x03a5] = "\x54\x42",
  [0x03aa] = "\x70\x36", [0x03ab] = "\x27\x2f", [0x03ac] = "\x45\x76",
  [0x03b0] = "\x47\x43", [0x03b6] = "\x49\x2f", [0x03b7] = "\x49\x30",
  [0x03bd] = "\x4c\x29", [0x03c6] = "\x54\x46", [0x03cb] = "\x54\x47",
  [0x03cc] = "\x54\x45", [0x03cd] = "\x54\x44", [0x03dc] = "\x6c\x29",
  [0x03dd] = "\x70\x37", [0x03e0] = "\x44\x2d", [0x03e1] = "\x44\x3b",
  [0x03f0] = "\x59\x29", [0x03f1] = "\x5e\x33", [0x03f3] = "\x68\x21",
  [0x03f6] = "\x44\x7e", [0x03f8] = "\x45\x79", [0x03f9] = "\x45\x77",
  [0x03fa] = "\x45\x78", [0x03fd] = "\x4c\x2a", [0x0400] = "\x44\x2e",
  [0x0401] = "\x44\x2f", [0x0403] = "\x44\x43", [0x0406] = "\x45\x21",
  [0x0407] = "\x45\x22", [0x0408] = "\x45\x23", [0x040a] = "\x45\x7a",
  [0x040e] = "\x47\x47", [0x0411] = "\x47\x45", [0x0412] = "\x47\x46",
  [0x0416] = "\x47\x48", [0x0417] = "\x47\x44", [0x041d] = "\x4f\x4f",
  [0x0424] = "\x49\x32", [0x0425] = "\x49\x31", [0x0428] = "\x49\x35",
  [0x0429] = "\x49\x33", [0x042a] = "\x49\x34", [0x042e] = "\x4c\x30",
  [0x0430] = "\x4c\x2f", [0x0436] = "\x4c\x31", [0x0437] = "\x4c\x2d",
  [0x0438] = "\x4c\x2c", [0x043a] = "\x4c\x2e", [0x043b] = "\x4c\x2b",
  [0x0441] = "\x4c\x32", [0x0443] = "\x50\x27", [0x0447] = "\x50\x2c",
  [0x044a] = "\x50\x28", [0x044b] = "\x50\x2b", [0x044c] = "\x50\x2a",
  [0x044d] = "\x50\x29", [0x044e] = "\x50\x26", [0x0454] = "\x54\x4a",
  [0x0456] = "\x54\x48", [0x045b] = "\x54\x4b", [0x045c] = "\x54\x49",
  [0x045d] = "\x54\x4c", [0x0469] = "\x5e\x37", [0x046a] = "\x59\x2a",
  [0x046f] = "\x59\x2b", [0x0472] = "\x5e\x34", [0x0474] = "\x5e\x35",
  [0x0475] = "\x5e\x36", [0x0477] = "\x63\x37", [0x047d] = "\x63\x38",
  [0x047f] = "\x63\x36", [0x0482] = "\x68\x23", [0x0483] = "\x68\x22",
  [0x0487] = "\x6c\x2a", [0x0488] = "\x6c\x2b", [0x0489] = "\x6c\x2c",
  [0x048a] = "\x6c\x2e", [0x048d] = "\x6c\x2d", [0x0491] = "\x70\x38",
  [0x0493] = "\x70\x39", [0x049b] = "\x44\x30", [0x049f] = "\x45\x7c",
  [0x04a0] = "\x45\x7b", [0x04a3] = "\x47\x49", [0x04a9] = "\x49\x37",
  [0x04aa] = "\x49\x38", [0x04ab] = "\x49\x36", [0x04ac] = "\x49\x39",
  [0x04bb] = "\x4c\x34", [0x04be] = "\x4c\x33", [0x04c1] = "\x50\x30",
  [0x04c3] = "\x50\x2f", [0x04c7] = "\x50\x2d", [0x04c9] = "\x50\x2e",
  [0x04d2] = "\x59\x2c", [0x04d5] = "\x59\x2f", [0x04d7] = "\x5b\x2b",
  [0x04d8] = "\x59\x2e", [0x04d9] = "\x59\x2d", [0x04db] = "\x5e\x3a",
  [0x04dd] = "\x5e\x39", [0x04de] = "\x5e\x38", [0x04df] = "\x63\x39",
  [0x04e2] = "\x63\x3c", [0x04e3] = "\x63\x3d", [0x04e4] = "\x63\x3b",
  [0x04e6] = "\x63\x3a", [0x04f0] = "\x6c\x2f", [0x04f3] = "\x70\x3a",
  [0x04f5] = "\x73\x44", [0x04f8] = "\x79\x61", [0x04f9] = "\x27\x34",
  [0x04fa] = "\x44\x44", [0x04fb] = "\x45\x24", [0x04fe] = "\x45\x25",
  [0x04ff] = "\x45\x26", [0x0505] = "\x45\x7d", [0x0506] = "\x45\x7e",
  [0x0508] = "\x47\x4a", [0x050d] = "\x50\x31", [0x050f] = "\x59\x31",
  [0x0510] = "\x59\x30", [0x0515] = "\x44\x31", [0x0516] = "\x45\x27",
  [0x0517] = "\x46\x21", [0x0519] = "\x59\x32", [0x051d] = "\x46\x22",
  [0x0520] = "\x47\x4c", [0x0521] = "\x47\x4b", [0x0523] = "\x49\x3a",
  [0x052a] = "\x54\x4d", [0x052f] = "\x63\x3e", [0x0531] = "\x68\x24",
  [0x0539] = "\x45\x28", [0x053e] = "\x59\x35", [0x053f] = "\x59\x33",
  [0x0540] = "\x59\x34", [0x0541] = "\x44\x32", [0x0543] = "\x44\x45",
  [0x0544] = "\x24\x3f", [0x0545] = "\x45\x2b", [0x0547] = "\x45\x2a",
  [0x0548] = "\x45\x29", [0x0549] = "\x46\x25", [0x054a] = "\x46\x24",
  [0x0551] = "\x4c\x38", [0x0552] = "\x4c\x35", [0x0553] = "\x4c\x37",
  [0x0554] = "\x4c\x36", [0x0557] = "\x50\x32", [0x055a] = "\x5e\x3b",
  [0x055c] = "\x44\x33", [0x055e] = "\x45\x2c", [0x0560] = "\x46\x27",
  [0x0561] = "\x46\x26", [0x0566] = "\x4c\x39", [0x056e] = "\x46\x29",
  [0x056f] = "\x46\x28", [0x0570] = "\x47\x4d", [0x0571] = "\x47\x4e",
  [0x0573] = "\x49\x3b", [0x0575] = "\x49\x3c", [0x0577] = "\x4c\x3a",
  [0x0578] = "\x4c\x3b", [0x0579] = "\x4c\x3c", [0x057b] = "\x50\x33",
  [0x057f] = "\x54\x4e", [0x0584] = "\x45\x2d", [0x059a] = "\x50\x34",
  [0x059d] = "\x54\x50", [0x059f] = "\x54\x4f", [0x05a5] = "\x5e\x3c",
  [0x05ad] = "\x68\x25", [0x05b2] = "\x6c\x30", [0x05bb] = "\x46\x2a",
  [0x05c3] = "\x59\x36", [0x05c8] = "\x44\x34", [0x05c9] = "\x44\x46",
  [0x05ca] = "\x45\x2f", [0x05cb] = "\x45\x2e", [0x05cd] = "\x45\x30",
  [0x05d4] = "\x4c\x3e", [0x05d6] = "\x4c\x3d", [0x05d7] = "\x4c\x3f",
  [0x05db] = "\x50\x35", [0x05df] = "\x54\x51", [0x05e2] = "\x76\x3b",
  [0x05e3] = "\x44\x47", [0x05e4] = "\x46\x2c", [0x05e5] = "\x46\x3b",
  [0x05e6] = "\x46\x36", [0x05e8] = "\x46\x31", [0x05e9] = "\x46\x30",
  [0x05ea] = "\x46\x37", [0x05eb] = "\x46\x35", [0x05ec] = "\x46\x2e",
  [0x05ed] = "\x46\x3c", [0x05ee] = "\x46\x2f", [0x05ef] = "\x46\x2b",
  [0x05f0] = "\x46\x3a", [0x05f1] = "\x46\x39", [0x05f2] = "\x46\x38",
  [0x05f3] = "\x46\x2d", [0x05f5] = "\x46\x34", [0x05f8] = "\x46\x33",
  [0x05fb] = "\x46\x3d", [0x05fc] = "\x46\x32", [0x0601] = "\x47\x54",
  [0x0603] = "\x47\x5a", [0x0604] = "\x47\x56", [0x0606] = "\x47\x5c",
  [0x0608] = "\x47\x59", [0x0609] = "\x47\x4f", [0x060a] = "\x47\x52",
  [0x060b] = "\x47\x55", [0x060c] = "\x47\x51", [0x060d] = "\x47\x58",
  [0x060e] = "\x47\x5b", [0x060f] = "\x47\x50", [0x0610] = "\x47\x53",
  [0x0611] = "\x47\x57", [0x0612] = "\x47\x5d", [0x061b] = "\x49\x49",
  [0x061d] = "\x49\x3d", [0x061e] = "\x49\x3f", [0x061f] = "\x49\x57",
  [0x0620] = "\x49\x52", [0x0626] = "\x49\x41", [0x0627] = "\x49\x43",
  [0x0629] = "\x49\x4a", [0x062b] = "\x49\x56", [0x062c] = "\x49\x58",
  [0x062d] = "\x49\x3e", [0x062e] = "\x49\x4f", [0x0631] = "\x49\x55",
  [0x0633] = "\x49\x46", [0x0635] = "\x49\x50", [0x0636] = "\x49\x51",
  [0x0638] = "\x49\x4e", [0x0639] = "\x49\x4c", [0x063b] = "\x49\x4d",
  [0x063c] = "\x49\x53", [0x063e] = "\x49\x40", [0x0640] = "\x49\x54",
  [0x0642] = "\x49\x48", [0x0643] = "\x49\x45", [0x0646] = "\x49\x44",
  [0x0648] = "\x49\x47", [0x064a] = "\x49\x4b", [0x064e] = "\x49\x42",
  [0x0662] = "\x4c\x51", [0x0668] = "\x4c\x52", [0x0671] = "\x4c\x4d",
  [0x0673] = "\x4c\x40", [0x0675] = "\x4c\x41", [0x0676] = "\x4c\x4e",
  [0x0677] = "\x4c\x47", [0x0678] = "\x4c\x43", [0x067b] = "\x4c\x46",
  [0x067c] = "\x4c\x4b", [0x067d] = "\x4c\x54", [0x0680] = "\x4c\x45",
  [0x0684] = "\x4c\x48", [0x0686] = "\x4c\x4a", [0x068b] = "\x4c\x53",
  [0x068c] = "\x4c\x4f", [0x068e] = "\x4c\x55", [0x0690] = "\x4c\x4c",
  [0x0692] = "\x4c\x49", [0x0695] = "\x4c\x44", [0x0696] = "\x4c\x42",
  [0x069a] = "\x4c\x50", [0x06a6] = "\x50\x3c", [0x06a7] = "\x50\x4a",
  [0x06a8] = "\x50\x38", [0x06a9] = "\x50\x49", [0x06aa] = "\x50\x41",
  [0x06ab] = "\x50\x46", [0x06ac] = "\x50\x36", [0x06af] = "\x50\x45",
  [0x06b1] = "\x50\x47", [0x06b3] = "\x50\x3d", [0x06b8] = "\x50\x3b",
  [0x06bb] = "\x50\x48", [0x06bd] = "\x50\x40", [0x06bf] = "\x50\x4b",
  [0x06c0] = "\x50\x37", [0x06c1] = "\x50\x42", [0x06c2] = "\x50\x3f",
  [0x06c4] = "\x50\x43", [0x06c7] = "\x50\x3e", [0x06c8] = "\x50\x44",
  [0x06c9] = "\x50\x3a", [0x06ce] = "\x50\x39", [0x06e1] = "\x54\x5e",
  [0x06e5] = "\x54\x57", [0x06e6] = "\x54\x62", [0x06e8] = "\x54\x52",
  [0x06e9] = "\x54\x5c", [0x06ea] = "\x54\x61", [0x06ed] = "\x54\x5d",
  [0x06ee] = "\x54\x60", [0x06f2] = "\x54\x58", [0x06fa] = "\x54\x5a",
  [0x06fc] = "\x54\x56", [0x06fd] = "\x54\x65", [0x0701] = "\x54\x54",
  [0x0706] = "\x54\x59", [0x0707] = "\x54\x64", [0x0709] = "\x54\x5f",
  [0x070f] = "\x54\x66", [0x0710] = "\x54\x53", [0x0714] = "\x54\x5b",
  [0x0727] = "\x54\x63", [0x072c] = "\x59\x49", [0x072e] = "\x59\x47",
  [0x072f] = "\x59\x44", [0x0731] = "\x59\x40", [0x0733] = "\x59\x4b",
  [0x0737] = "\x54\x55", [0x0738] = "\x59\x46", [0x073e] = "\x5e\x4e",
  [0x0741] = "\x59\x4c", [0x0743] = "\x59\x3e", [0x0744] = "\x59\x3b",
  [0x0746] = "\x59\x38", [0x074a] = "\x59\x3f", [0x074f] = "\x59\x42",
  [0x0755] = "\x59\x43", [0x0756] = "\x59\x41", [0x0757] = "\x59\x4d",
  [0x075c] = "\x59\x48", [0x075e] = "\x59\x3c", [0x075f] = "\x5a\x76",
  [0x0761] = "\x59\x3d", [0x0763] = "\x59\x4a", [0x0764] = "\x59\x45",
  [0x0766] = "\x59\x3a", [0x076a] = "\x59\x39", [0x077b] = "\x5e\x3d",
  [0x077c] = "\x5e\x40", [0x077e] = "\x5e\x54", [0x0780] = "\x5e\x3e",
  [0x0782] = "\x5e\x44", [0x0783] = "\x5e\x4a", [0x0784] = "\x61\x43",
  [0x0787] = "\x5e\x48", [0x0789] = "\x5e\x55", [0x078a] = "\x5e\x41",
  [0x078b] = "\x5e\x49", [0x0794] = "\x5e\x47", [0x0798] = "\x5e\x43",
  [0x0799] = "\x5e\x57", [0x079a] = "\x5e\x50", [0x079c] = "\x5e\x45",
  [0x079d] = "\x5e\x42", [0x079f] = "\x5e\x4d", [0x07a7] = "\x5e\x3f",
  [0x07aa] = "\x5e\x46", [0x07ab] = "\x5e\x56", [0x07ac] = "\x5e\x52",
  [0x07ae] = "\x5e\x4c", [0x07b1] = "\x5e\x53", [0x07b2] = "\x5e\x4f",
  [0x07b3] = "\x5e\x4b", [0x07bb] = "\x5e\x51", [0x07c5] = "\x63\x4c",
  [0x07c6] = "\x63\x4d", [0x07c7] = "\x63\x45", [0x07c9] = "\x63\x4f",
  [0x07ce] = "\x63\x43", [0x07d1] = "\x63\x46", [0x07d3] = "\x63\x41",
  [0x07da] = "\x63\x4a", [0x07dc] = "\x63\x44", [0x07df] = "\x63\x3f",
  [0x07e1] = "\x63\x4b", [0x07e3] = "\x63\x47", [0x07e4] = "\x63\x48",
  [0x07e5] = "\x63\x4e", [0x07e6] = "\x63\x42", [0x07e7] = "\x23\x21",
  [0x07e8] = "\x63\x40", [0x07ef] = "\x63\x49", [0x07f6] = "\x68\x35",
  [0x07f7] = "\x68\x30", [0x07fd] = "\x68\x2a", [0x07fe] = "\x68\x26",
  [0x0800] = "\x68\x27", [0x0806] = "\x68\x2c", [0x0808] = "\x68\x33",
  [0x0809] = "\x68\x2d", [0x080d] = "\x68\x2e", [0x080e] = "\x68\x2f",
  [0x0810] = "\x68\x34", [0x0814] = "\x68\x2b", [0x0816] = "\x68\x31",
  [0x0817] = "\x68\x29", [0x081b] = "\x68\x28", [0x081f] = "\x68\x32",
  [0x0829] = "\x6c\x37", [0x082e] = "\x6c\x31", [0x082f] = "\x6c\x3d",
  [0x0830] = "\x6c\x3e", [0x0832] = "\x6c\x34", [0x0834] = "\x6c\x36",
  [0x0836] = "\x6c\x3c", [0x0839] = "\x6c\x33", [0x083b] = "\x6c\x32",
  [0x083f] = "\x6c\x35", [0x084e] = "\x6c\x39", [0x0853] = "\x6c\x38",
  [0x0857] = "\x6c\x3a", [0x0859] = "\x70\x3b", [0x0862] = "\x70\x47",
  [0x0864] = "\x70\x3f", [0x0865] = "\x70\x43", [0x0868] = "\x70\x42",
  [0x0869] = "\x70\x3e", [0x086a] = "\x70\x41", [0x086b] = "\x70\x3c",
  [0x086c] = "\x70\x46", [0x086f] = "\x70\x45", [0x0871] = "\x70\x44",
  [0x0874] = "\x6c\x3b", [0x0876] = "\x70\x48", [0x0878] = "\x70\x40",
  [0x0879] = "\x70\x3d", [0x0880] = "\x73\x46", [0x0885] = "\x73\x48",
  [0x0887] = "\x73\x49", [0x088e] = "\x73\x45", [0x088f] = "\x73\x4a",
  [0x0890] = "\x73\x47", [0x0895] = "\x76\x3c", [0x08a5] = "\x78\x2c",
  [0x08a8] = "\x78\x2d", [0x08ae] = "\x76\x3d", [0x08b4] = "\x79\x64",
  [0x08b6] = "\x79\x63", [0x08b7] = "\x79\x62", [0x08bc] = "\x79\x65",
  [0x08c0] = "\x7a\x65", [0x08c1] = "\x7a\x64", [0x08c2] = "\x7a\x66",
  [0x08c8] = "\x7b\x59", [0x08c9] = "\x7b\x5b", [0x08ca] = "\x7b\x5a",
  [0x08cc] = "\x7c\x34", [0x08d1] = "\x7c\x5d", [0x08da] = "\x46\x3f",
  [0x08db] = "\x46\x3e", [0x08dd] = "\x47\x60", [0x08de] = "\x47\x5f",
  [0x08e0] = "\x47\x5e", [0x08e4] = "\x49\x5b", [0x08ea] = "\x49\x59",
  [0x08eb] = "\x49\x5c", [0x08f0] = "\x49\x5a", [0x08fa] = "\x4c\x56",
  [0x08ff] = "\x50\x4c", [0x0903] = "\x54\x67", [0x0904] = "\x54\x68",
  [0x0908] = "\x59\x4e", [0x0909] = "\x59\x50", [0x090b] = "\x59\x4f",
  [0x090d] = "\x5e\x58", [0x0912] = "\x63\x50", [0x0913] = "\x63\x51",
  [0x0916] = "\x68\x37", [0x0918] = "\x68\x36", [0x091f] = "\x44\x48",
  [0x0928] = "\x47\x63", [0x0929] = "\x47\x67", [0x092c] = "\x47\x65",
  [0x092d] = "\x47\x64", [0x092f] = "\x47\x66", [0x0930] = "\x47\x62",
  [0x0933] = "\x47\x61", [0x093b] = "\x49\x66", [0x093e] = "\x49\x63",
  [0x0940] = "\x49\x5f", [0x0947] = "\x49\x61", [0x094a] = "\x49\x5d",
  [0x094d] = "\x49\x60", [0x094e] = "\x49\x62", [0x094f] = "\x49\x65",
  [0x0950] = "\x49\x64", [0x0951] = "\x49\x5e", [0x0961] = "\x4c\x5b",
  [0x0964] = "\x4c\x5d", [0x0966] = "\x4c\x5c", [0x0969] = "\x4c\x5a",
  [0x096a] = "\x4c\x59", [0x0977] = "\x4c\x58", [0x097c] = "\x4c\x5e",
  [0x0982] = "\x50\x4d", [0x0983] = "\x4c\x57", [0x098b] = "\x50\x4e",
  [0x0993] = "\x50\x54", [0x09a0] = "\x50\x4f", [0x09a2] = "\x50\x51",
  [0x09a3] = "\x50\x50", [0x09ae] = "\x50\x53", [0x09c2] = "\x54\x69",
  [0x09c3] = "\x54\x6c", [0x09cb] = "\x54\x6b", [0x09ce] = "\x50\x52",
  [0x09d4] = "\x54\x6a", [0x09df] = "\x59\x51", [0x09e0] = "\x59\x55",
  [0x09e4] = "\x59\x56", [0x09f7] = "\x59\x5a", [0x09f9] = "\x59\x5b",
  [0x09fa] = "\x59\x57", [0x0a02] = "\x59\x58", [0x0a05] = "\x59\x52",
  [0x0a06] = "\x59\x54", [0x0a09] = "\x54\x6d", [0x0a0a] = "\x59\x53",
  [0x0a1d] = "\x5e\x60", [0x0a20] = "\x5e\x61", [0x0a21] = "\x5e\x5f",
  [0x0a24] = "\x5e\x5c", [0x0a2a] = "\x5e\x5a", [0x0a2f] = "\x5e\x59",
  [0x0a30] = "\x5e\x5d", [0x0a31] = "\x5e\x5e", [0x0a34] = "\x5e\x5b",
  [0x0a35] = "\x59\x59", [0x0a4a] = "\x63\x5b", [0x0a4b] = "\x63\x5e",
  [0x0a4c] = "\x63\x59", [0x0a51] = "\x63\x53", [0x0a52] = "\x63\x5d",
  [0x0a54] = "\x63\x57", [0x0a57] = "\x63\x55", [0x0a58] = "\x63\x54",
  [0x0a5a] = "\x63\x56", [0x0a5e] = "\x63\x52", [0x0a62] = "\x63\x5c",
  [0x0a6b] = "\x63\x58", [0x0a6d] = "\x63\x5a", [0x0a75] = "\x68\x38",
  [0x0a79] = "\x68\x3d", [0x0a7d] = "\x68\x3f", [0x0a7e] = "\x68\x39",
  [0x0a80] = "\x6c\x3f", [0x0a83] = "\x68\x3a", [0x0a85] = "\x68\x3e",
  [0x0a8a] = "\x68\x3c", [0x0a93] = "\x68\x3b", [0x0a9c] = "\x6c\x43",
  [0x0a9e] = "\x6c\x41", [0x0a9f] = "\x6c\x40", [0x0aa6] = "\x6c\x46",
  [0x0aa8] = "\x70\x2e", [0x0aa9] = "\x6c\x45", [0x0aae] = "\x6c\x44",
  [0x0ab3] = "\x6c\x42", [0x0abe] = "\x70\x4a", [0x0ac1] = "\x70\x49",
  [0x0ac5] = "\x70\x4c", [0x0ac7] = "\x70\x4b", [0x0ace] = "\x73\x4e",
  [0x0ad1] = "\x73\x4d", [0x0ad3] = "\x73\x4c", [0x0ad5] = "\x73\x4b",
  [0x0ad8] = "\x76\x3f", [0x0ad9] = "\x76\x3e", [0x0ade] = "\x78\x2e",
  [0x0adf] = "\x78\x2f", [0x0ae2] = "\x78\x30", [0x0ae4] = "\x79\x66",
  [0x0ae9] = "\x7c\x5e", [0x0aeb] = "\x44\x49", [0x0aec] = "\x45\x31",
  [0x0aef] = "\x49\x67", [0x0af9] = "\x5e\x62", [0x0afa] = "\x5e\x63",
  [0x0afd] = "\x68\x40", [0x0b0f] = "\x54\x6e", [0x0b14] = "\x7a\x67",
  [0x0b15] = "\x44\x4a", [0x0b16] = "\x46\x40", [0x0b19] = "\x47\x68",
  [0x0b1a] = "\x47\x69", [0x0b1c] = "\x4c\x5f", [0x0b20] = "\x59\x5c",
  [0x0b22] = "\x68\x42", [0x0b24] = "\x68\x43", [0x0b25] = "\x68\x41",
  [0x0b27] = "\x44\x4b", [0x0b29] = "\x45\x32", [0x0b2a] = "\x45\x34",
  [0x0b2b] = "\x45\x33", [0x0b2d] = "\x45\x35", [0x0b2e] = "\x46\x41",
  [0x0b31] = "\x46\x42", [0x0b37] = "\x47\x6a", [0x0b38] = "\x47\x6b",
  [0x0b3e] = "\x49\x68", [0x0b44] = "\x4c\x63", [0x0b47] = "\x4c\x61",
  [0x0b48] = "\x4c\x62", [0x0b49] = "\x4c\x60", [0x0b4e] = "\x50\x58",
  [0x0b4f] = "\x50\x57", [0x0b50] = "\x50\x59", [0x0b51] = "\x50\x56",
  [0x0b54] = "\x4c\x64", [0x0b55] = "\x50\x55", [0x0b57] = "\x54\x6f",
  [0x0b58] = "\x54\x70", [0x0b5a] = "\x54\x71", [0x0b60] = "\x5e\x64",
  [0x0b62] = "\x59\x5d", [0x0b67] = "\x63\x5f", [0x0b69] = "\x68\x45",
  [0x0b6a] = "\x68\x44", [0x0b6d] = "\x6c\x47", [0x0b6e] = "\x70\x4d",
  [0x0b73] = "\x44\x4c", [0x0b74] = "\x46\x43", [0x0b76] = "\x46\x44",
  [0x0b78] = "\x47\x6d", [0x0b79] = "\x47\x70", [0x0b7d] = "\x47\x6f",
  [0x0b81] = "\x47\x72", [0x0b82] = "\x47\x71", [0x0b83] = "\x47\x6e",
  [0x0b84] = "\x47\x6c", [0x0b8a] = "\x49\x73", [0x0b8d] = "\x49\x70",
  [0x0b92] = "\x49\x6a", [0x0b93] = "\x49\x72", [0x0b96] = "\x49\x6f",
  [0x0b99] = "\x49\x6e", [0x0b9d] = "\x49\x69", [0x0b9e] = "\x49\x6c",
  [0x0ba3] = "\x49\x6d", [0x0ba4] = "\x49\x71", [0x0ba5] = "\x49\x74",
  [0x0ba8] = "\x49\x6b", [0x0bae] = "\x4c\x69", [0x0baf] = "\x4c\x71",
  [0x0bb3] = "\x4c\x72", [0x0bb9] = "\x4c\x68", [0x0bbb] = "\x4c\x66",
  [0x0bbe] = "\x4c\x65", [0x0bc5] = "\x4c\x74", [0x0bc6] = "\x4c\x6b",
  [0x0bca] = "\x4c\x70", [0x0bcb] = "\x4c\x6e", [0x0bcd] = "\x4c\x6d",
  [0x0bd0] = "\x4c\x6c", [0x0bd1] = "\x4c\x6a", [0x0bd2] = "\x4c\x73",
  [0x0bd3] = "\x4c\x6f", [0x0bd4] = "\x4c\x67", [0x0bd8] = "\x50\x5b",
  [0x0bda] = "\x50\x62", [0x0bdc] = "\x50\x5a", [0x0be3] = "\x50\x5d",
  [0x0be5] = "\x50\x60", [0x0be6] = "\x50\x63", [0x0be8] = "\x50\x5e",
  [0x0bea] = "\x50\x61", [0x0bec] = "\x54\x78", [0x0bfb] = "\x50\x65",
  [0x0bff] = "\x50\x5c", [0x0c01] = "\x50\x64", [0x0c03] = "\x50\x5f",
  [0x0c09] = "\x54\x7e", [0x0c0c] = "\x54\x7d", [0x0c11] = "\x54\x72",
  [0x0c13] = "\x54\x77", [0x0c18] = "\x54\x73", [0x0c1b] = "\x54\x76",
  [0x0c1c] = "\x54\x74", [0x0c1f] = "\x54\x75", [0x0c20] = "\x54\x79",
  [0x0c23] = "\x54\x7a", [0x0c25] = "\x54\x7c", [0x0c29] = "\x54\x7b",
  [0x0c36] = "\x59\x5e", [0x0c3c] = "\x59\x64", [0x0c40] = "\x59\x63",
  [0x0c41] = "\x59\x5f", [0x0c46] = "\x59\x67", [0x0c49] = "\x59\x60",
  [0x0c4a] = "\x59\x68", [0x0c5a] = "\x59\x66", [0x0c62] = "\x59\x65",
  [0x0c66] = "\x59\x61", [0x0c6a] = "\x59\x62", [0x0c77] = "\x5e\x65",
  [0x0c7f] = "\x5e\x67", [0x0c92] = "\x5e\x68", [0x0c9a] = "\x5e\x66",
  [0x0c9b] = "\x5e\x69", [0x0ca7] = "\x5e\x6a", [0x0cb2] = "\x63\x68",
  [0x0cb3] = "\x63\x66", [0x0cbc] = "\x63\x65", [0x0cbd] = "\x63\x64",
  [0x0cbe] = "\x63\x63", [0x0cc1] = "\x63\x60", [0x0cc2] = "\x63\x67",
  [0x0cc9] = "\x63\x61", [0x0ccc] = "\x63\x62", [0x0cd6] = "\x68\x4a",
  [0x0cd7] = "\x68\x49", [0x0cd8] = "\x68\x4b", [0x0ce1] = "\x68\x46",
  [0x0ce3] = "\x68\x4c", [0x0ce6] = "\x68\x47", [0x0ce9] = "\x68\x48",
  [0x0cf5] = "\x6c\x4b", [0x0cfb] = "\x6c\x49", [0x0d08] = "\x6c\x4d",
  [0x0d09] = "\x6c\x48", [0x0d0b] = "\x6c\x4a", [0x0d0c] = "\x6c\x4c",
  [0x0d1d] = "\x70\x4e", [0x0d24] = "\x73\x51", [0x0d2a] = "\x73\x50",
  [0x0d30] = "\x73\x4f", [0x0d34] = "\x70\x4f", [0x0d38] = "\x76\x40",
  [0x0d40] = "\x79\x67", [0x0d43] = "\x79\x68", [0x0d50] = "\x44\x4d",
  [0x0d51] = "\x44\x4e", [0x0d53] = "\x44\x4f", [0x0d54] = "\x45\x36",
  [0x0d55] = "\x46\x45", [0x0d57] = "\x47\x73", [0x0d58] = "\x47\x74",
  [0x0d5a] = "\x49\x77", [0x0d5b] = "\x49\x78", [0x0d5c] = "\x49\x76",
  [0x0d5d] = "\x49\x75", [0x0d5f] = "\x4c\x75", [0x0d63] = "\x4c\x77",
  [0x0d64] = "\x4c\x76", [0x0d69] = "\x50\x66", [0x0d6b] = "\x55\x21",
  [0x0d70] = "\x59\x69", [0x0d71] = "\x5e\x6c", [0x0d73] = "\x5e\x6b",
  [0x0d75] = "\x68\x4d", [0x0d78] = "\x70\x50", [0x0d7a] = "\x73\x52",
  [0x0d7d] = "\x79\x69", [0x0d7f] = "\x7b\x5c", [0x0d83] = "\x46\x46",
  [0x0d85] = "\x47\x77", [0x0d87] = "\x47\x75", [0x0d88] = "\x47\x76",
  [0x0d89] = "\x47\x78", [0x0d8b] = "\x49\x7a", [0x0d8c] = "\x49\x79",
  [0x0d8f] = "\x49\x7b", [0x0d97] = "\x4c\x78", [0x0d98] = "\x4c\x7a",
  [0x0d99] = "\x4c\x7c", [0x0d9a] = "\x4c\x79", [0x0d9b] = "\x4c\x7d",
  [0x0d9c] = "\x4c\x7b", [0x0da2] = "\x50\x6a", [0x0da3] = "\x50\x67",
  [0x0da4] = "\x50\x69", [0x0da5] = "\x50\x6b", [0x0da6] = "\x50\x68",
  [0x0dae] = "\x55\x27", [0x0db0] = "\x55\x23", [0x0db3] = "\x55\x24",
  [0x0db4] = "\x55\x26", [0x0db5] = "\x55\x28", [0x0db6] = "\x55\x25",
  [0x0db8] = "\x55\x2a", [0x0db9] = "\x55\x29", [0x0dbf] = "\x59\x6e",
  [0x0dc2] = "\x59\x6d", [0x0dc4] = "\x59\x6c", [0x0dc5] = "\x59\x6b",
  [0x0dc6] = "\x59\x6f", [0x0dc7] = "\x59\x6a", [0x0dcc] = "\x5e\x6e",
  [0x0dd0] = "\x5e\x70", [0x0dd2] = "\x5e\x6d", [0x0dd3] = "\x5e\x6f",
  [0x0dde] = "\x68\x4e", [0x0ddf] = "\x68\x56", [0x0de1] = "\x68\x50",
  [0x0de2] = "\x68\x54", [0x0de4] = "\x68\x55", [0x0de5] = "\x68\x51",
  [0x0de6] = "\x68\x52", [0x0de7] = "\x68\x4f", [0x0de8] = "\x68\x53",
  [0x0de9] = "\x6c\x50", [0x0deb] = "\x6c\x51", [0x0dec] = "\x6c\x4f",
  [0x0dee] = "\x6c\x4e", [0x0df0] = "\x70\x51", [0x0df5] = "\x78\x31",
  [0x0df6] = "\x79\x6a", [0x0df8] = "\x44\x50", [0x0dfa] = "\x47\x79",
  [0x0e01] = "\x50\x6c", [0x0e04] = "\x55\x2b", [0x0e07] = "\x59\x72",
  [0x0e08] = "\x59\x71", [0x0e09] = "\x59\x70", [0x0e0a] = "\x5e\x71",
  [0x0e0b] = "\x5e\x72", [0x0e0d] = "\x68\x57", [0x0e0e] = "\x70\x52",
  [0x0e0f] = "\x44\x51", [0x0e11] = "\x45\x37", [0x0e16] = "\x47\x7a",
  [0x0e1a] = "\x4c\x7e", [0x0e22] = "\x44\x52", [0x0e24] = "\x45\x38",
  [0x0e2c] = "\x49\x7c", [0x0e31] = "\x5e\x73", [0x0e37] = "\x73\x53",
  [0x0e38] = "\x44\x53", [0x0e39] = "\x44\x67", [0x0e3a] = "\x45\x39",
  [0x0e3c] = "\x46\x47", [0x0e3e] = "\x4a\x22", [0x0e3f] = "\x4a\x21",
  [0x0e40] = "\x49\x7d", [0x0e41] = "\x49\x7e", [0x0e45] = "\x4d\x22",
  [0x0e46] = "\x4d\x23", [0x0e48] = "\x4d\x21", [0x0e4b] = "\x50\x70",
  [0x0e4d] = "\x50\x6f", [0x0e4e] = "\x50\x6d", [0x0e4f] = "\x50\x6e",
  [0x0e50] = "\x55\x2e", [0x0e51] = "\x55\x2c", [0x0e55] = "\x55\x2d",
  [0x0e58] = "\x55\x22", [0x0e5c] = "\x59\x74", [0x0e5d] = "\x59\x75",
  [0x0e60] = "\x59\x73", [0x0e62] = "\x68\x58", [0x0e64] = "\x6c\x52",
  [0x0e65] = "\x6c\x53", [0x0e68] = "\x73\x54", [0x0e6c] = "\x7a\x68",
  [0x0e6f] = "\x45\x3a", [0x0e71] = "\x44\x54", [0x0e79] = "\x47\x7b",
  [0x0e8c] = "\x4a\x26", [0x0e90] = "\x4a\x23", [0x0e91] = "\x4a\x24",
  [0x0e94] = "\x4a\x25", [0x0ea1] = "\x4d\x25", [0x0ea9] = "\x4d\x27",
  [0x0eab] = "\x4d\x28", [0x0eb1] = "\x4d\x29", [0x0eb3] = "\x4d\x2a",
  [0x0eb7] = "\x4d\x24", [0x0eb8] = "\x4d\x26", [0x0ed2] = "\x50\x72",
  [0x0ed9] = "\x50\x71", [0x0ee8] = "\x55\x33", [0x0eea] = "\x55\x32",
  [0x0eed] = "\x55\x2f", [0x0ef0] = "\x55\x34", [0x0ef4] = "\x55\x37",
  [0x0ef6] = "\x55\x35", [0x0efb] = "\x55\x31", [0x0efd] = "\x55\x30",
  [0x0f01] = "\x55\x36", [0x0f06] = "\x59\x77", [0x0f07] = "\x59\x76",
  [0x0f0e] = "\x59\x78", [0x0f11] = "\x59\x7c", [0x0f14] = "\x59\x7e",
  [0x0f16] = "\x59\x7a", [0x0f17] = "\x5a\x24", [0x0f19] = "\x5a\x21",
  [0x0f1b] = "\x59\x79", [0x0f22] = "\x59\x7b", [0x0f24] = "\x5a\x22",
  [0x0f27] = "\x5a\x23", [0x0f29] = "\x59\x7d", [0x0f34] = "\x5e\x76",
  [0x0f47] = "\x5e\x77", [0x0f4c] = "\x5e\x74", [0x0f50] = "\x5e\x75",
  [0x0f69] = "\x63\x69", [0x0f6f] = "\x63\x6a", [0x0f84] = "\x68\x59",
  [0x0f87] = "\x68\x5a", [0x0f94] = "\x6c\x55", [0x0f9d] = "\x6c\x54",
  [0x0fb8] = "\x73\x58", [0x0fba] = "\x73\x56", [0x0fbc] = "\x73\x55",
  [0x0fbd] = "\x73\x57", [0x0fc9] = "\x79\x6b", [0x0fcd] = "\x7a\x69",
  [0x0fd2] = "\x7b\x5e", [0x0fd4] = "\x7b\x5d", [0x0fd6] = "\x7c\x35",
  [0x0fdd] = "\x44\x55", [0x0fde] = "\x47\x7c", [0x0fe1] = "\x4b\x56",
  [0x0fe2] = "\x5a\x25", [0x0fe5] = "\x44\x56", [0x0fe6] = "\x46\x4a",
  [0x0fe7] = "\x46\x49", [0x0fe8] = "\x46\x48", [0x0feb] = "\x4a\x27",
  [0x0fee] = "\x55\x38", [0x0ff1] = "\x44\x57", [0x0ff2] = "\x44\x58",
  [0x0ff3] = "\x44\x59", [0x0ff4] = "\x45\x3b", [0x0ff7] = "\x50\x73",
  [0x0ffd] = "\x5e\x78", [0x0ffe] = "\x44\x5a", [0x1002] = "\x46\x4b",
  [0x1003] = "\x46\x4c", [0x1006] = "\x47\x7d", [0x100c] = "\x4a\x28",
  [0x1011] = "\x4d\x30", [0x1015] = "\x4d\x2e", [0x1016] = "\x4d\x2d",
  [0x1018] = "\x4d\x2b", [0x101a] = "\x4d\x2c", [0x101b] = "\x4d\x2f",
  [0x101d] = "\x50\x74", [0x101f] = "\x50\x76", [0x1025] = "\x50\x75",
  [0x102b] = "\x55\x3a", [0x102d] = "\x55\x39", [0x1033] = "\x5a\x28",
  [0x1036] = "\x5a\x27", [0x1037] = "\x5a\x29", [0x1038] = "\x5a\x26",
  [0x103d] = "\x5e\x7a", [0x1040] = "\x5e\x7b", [0x1043] = "\x5e\x7c",
  [0x1045] = "\x5e\x79", [0x104c] = "\x63\x6b", [0x1054] = "\x68\x5f",
  [0x1055] = "\x68\x5d", [0x1057] = "\x68\x5e", [0x105b] = "\x68\x5b",
  [0x105f] = "\x6c\x57", [0x1061] = "\x6c\x58", [0x1062] = "\x6c\x56",
  [0x1063] = "\x68\x5c", [0x106b] = "\x73\x59", [0x1072] = "\x44\x5b",
  [0x1073] = "\x46\x4d", [0x1074] = "\x48\x21", [0x1076] = "\x47\x7e",
  [0x1078] = "\x4d\x31", [0x1079] = "\x63\x6c", [0x107b] = "\x45\x3c",
  [0x107c] = "\x46\x4e", [0x107d] = "\x50\x77", [0x107e] = "\x5e\x7d",
  [0x1087] = "\x4a\x2a", [0x108a] = "\x4a\x2b", [0x108f] = "\x4a\x29",
  [0x1095] = "\x4d\x35", [0x1096] = "\x4d\x36", [0x1097] = "\x4d\x33",
  [0x109a] = "\x4d\x32", [0x109c] = "\x4d\x34", [0x10a0] = "\x50\x78",
  [0x10a6] = "\x50\x79", [0x10a7] = "\x55\x3d", [0x10ab] = "\x55\x3b",
  [0x10ad] = "\x55\x3c", [0x10b5] = "\x5a\x2d", [0x10b6] = "\x5a\x2c",
  [0x10b7] = "\x5a\x2a", [0x10b8] = "\x5a\x2b", [0x10be] = "\x5a\x2e",
  [0x10c1] = "\x5f\x21", [0x10c2] = "\x5f\x22", [0x10c4] = "\x5f\x23",
  [0x10c8] = "\x63\x6e", [0x10c9] = "\x63\x6d", [0x10ca] = "\x5e\x7e",
  [0x10d3] = "\x68\x60", [0x10d6] = "\x68\x61", [0x10da] = "\x6c\x5a",
  [0x10dd] = "\x6c\x5c", [0x10df] = "\x6c\x5b", [0x10e0] = "\x6c\x5e",
  [0x10e2] = "\x6c\x59", [0x10e3] = "\x6c\x5d", [0x10ec] = "\x78\x33",
  [0x10f3] = "\x7d\x22", [0x10f6] = "\x4d\x37", [0x10f7] = "\x4a\x2c",
  [0x10fa] = "\x50\x7a", [0x10fe] = "\x44\x5c", [0x10ff] = "\x45\x3d",
  [0x1101] = "\x46\x4f", [0x1104] = "\x4a\x2d", [0x1108] = "\x50\x7b",
  [0x110a] = "\x68\x62", [0x110b] = "\x44\x5d", [0x110f] = "\x48\x22",
  [0x1112] = "\x63\x6f", [0x1113] = "\x44\x5e", [0x1114] = "\x45\x3e",
  [0x1115] = "\x45\x3f", [0x1117] = "\x46\x51", [0x1118] = "\x46\x50",
  [0x111b] = "\x48\x23", [0x111f] = "\x4a\x2e", [0x1126] = "\x4d\x38",
  [0x1127] = "\x4d\x39", [0x1129] = "\x4d\x3a", [0x112d] = "\x50\x7c",
  [0x1131] = "\x55\x3e", [0x1135] = "\x5a\x2f", [0x1137] = "\x5a\x30",
  [0x113c] = "\x5f\x24", [0x1146] = "\x68\x63", [0x1148] = "\x6c\x5f",
  [0x114a] = "\x70\x53", [0x114c] = "\x73\x5a", [0x114e] = "\x7b\x5f",
  [0x1157] = "\x5a\x31", [0x1159] = "\x63\x70", [0x115e] = "\x76\x41",
  [0x1162] = "\x4a\x30", [0x1164] = "\x4a\x2f", [0x1165] = "\x50\x7d",
  [0x1169] = "\x5a\x33", [0x116a] = "\x5d\x29", [0x116b] = "\x5a\x34",
  [0x116c] = "\x5a\x32", [0x116d] = "\x5f\x25", [0x1170] = "\x68\x64",
  [0x1171] = "\x6c\x60", [0x1177] = "\x4a\x31", [0x1179] = "\x4a\x32",
  [0x117c] = "\x4d\x3e", [0x117f] = "\x4d\x3d", [0x1180] = "\x4d\x3b",
  [0x1181] = "\x4d\x3c", [0x1185] = "\x51\x21", [0x1187] = "\x51\x24",
  [0x1188] = "\x50\x7e", [0x1189] = "\x51\x26", [0x118a] = "\x51\x22",
  [0x118b] = "\x51\x23", [0x118c] = "\x51\x25", [0x1190] = "\x55\x41",
  [0x1191] = "\x55\x40", [0x1192] = "\x55\x3f", [0x1197] = "\x5a\x35",
  [0x1198] = "\x5a\x38", [0x1199] = "\x5a\x36", [0x119c] = "\x5a\x3b",
  [0x119e] = "\x5a\x37", [0x11a0] = "\x5a\x3a", [0x11a1] = "\x5a\x39",
  [0x11a8] = "\x5f\x28", [0x11a9] = "\x5f\x26", [0x11aa] = "\x5f\x27",
  [0x11ac] = "\x63\x71", [0x11ae] = "\x63\x72", [0x11b5] = "\x6c\x62",
  [0x11b7] = "\x6c\x61", [0x11b9] = "\x68\x65", [0x11bd] = "\x73\x5b",
  [0x11c3] = "\x45\x40", [0x11c5] = "\x46\x52", [0x11cc] = "\x4a\x34",
  [0x11cd] = "\x4a\x36", [0x11d6] = "\x48\x25", [0x11d7] = "\x4a\x35",
  [0x11d8] = "\x4a\x33", [0x11d9] = "\x48\x24", [0x11dd] = "\x4d\x3f",
  [0x11e0] = "\x4d\x40", [0x11ea] = "\x4a\x3a", [0x11eb] = "\x4a\x38",
  [0x11f1] = "\x4a\x37", [0x11f5] = "\x4d\x42", [0x11f8] = "\x4a\x39",
  [0x11fd] = "\x4d\x41", [0x11ff] = "\x4d\x43", [0x120e] = "\x51\x2b",
  [0x120f] = "\x4d\x44", [0x1212] = "\x51\x27", [0x1214] = "\x4d\x45",
  [0x1215] = "\x4d\x4a", [0x1216] = "\x4d\x48", [0x121b] = "\x4d\x4f",
  [0x121d] = "\x51\x28", [0x1220] = "\x51\x29", [0x1221] = "\x4d\x4b",
  [0x1225] = "\x51\x2a", [0x1227] = "\x4d\x4c", [0x1228] = "\x51\x2c",
  [0x1229] = "\x4d\x4d", [0x122a] = "\x4d\x49", [0x122b] = "\x4d\x4e",
  [0x122f] = "\x4d\x46", [0x1235] = "\x4d\x47", [0x1243] = "\x51\x32",
  [0x1246] = "\x51\x31", [0x124d] = "\x51\x2d", [0x1250] = "\x55\x45",
  [0x1255] = "\x55\x46", [0x1259] = "\x55\x42", [0x1262] = "\x51\x30",
  [0x1263] = "\x55\x43", [0x1264] = "\x51\x36", [0x1265] = "\x55\x44",
  [0x1268] = "\x51\x2f", [0x1269] = "\x55\x48", [0x126a] = "\x51\x35",
  [0x126b] = "\x51\x34", [0x126c] = "\x51\x33", [0x126d] = "\x55\x47",
  [0x126f] = "\x55\x49", [0x1270] = "\x51\x2e", [0x127f] = "\x5a\x3c",
  [0x1284] = "\x55\x4a", [0x1285] = "\x55\x50", [0x1289] = "\x5a\x3e",
  [0x128c] = "\x55\x4f", [0x128d] = "\x55\x4d", [0x1294] = "\x55\x4e",
  [0x1296] = "\x55\x51", [0x129a] = "\x55\x4c", [0x129f] = "\x55\x4b",
  [0x12a0] = "\x5a\x3f", [0x12a3] = "\x5a\x3d", [0x12a8] = "\x5a\x40",
  [0x12b2] = "\x5f\x2b", [0x12b4] = "\x5a\x42", [0x12b5] = "\x5a\x47",
  [0x12b6] = "\x5f\x2c", [0x12b8] = "\x5a\x4e", [0x12bb] = "\x5a\x46",
  [0x12bc] = "\x5a\x49", [0x12bd] = "\x5a\x44", [0x12c5] = "\x5a\x45",
  [0x12c6] = "\x5a\x4c", [0x12c7] = "\x5a\x50", [0x12cb] = "\x5a\x41",
  [0x12d1] = "\x5f\x29", [0x12d5] = "\x5a\x4b", [0x12d8] = "\x5a\x4a",
  [0x12da] = "\x5a\x4f", [0x12dc] = "\x5a\x48", [0x12df] = "\x5a\x4d",
  [0x12e0] = "\x5f\x2d", [0x12e1] = "\x5f\x2a", [0x12e6] = "\x5a\x43",
  [0x12f0] = "\x5f\x32", [0x12f1] = "\x5f\x36", [0x12f3] = "\x63\x77",
  [0x12f4] = "\x5f\x34", [0x12f6] = "\x5f\x38", [0x12f9] = "\x63\x79",
  [0x12fa] = "\x5f\x30", [0x12fb] = "\x5f\x33", [0x1300] = "\x5f\x3a",
  [0x1301] = "\x63\x7a", [0x1306] = "\x64\x26", [0x1308] = "\x63\x7b",
  [0x1309] = "\x5f\x39", [0x130d] = "\x64\x25", [0x130e] = "\x5f\x37",
  [0x130f] = "\x63\x74", [0x1312] = "\x5f\x3b", [0x1315] = "\x5f\x31",
  [0x131a] = "\x63\x73", [0x131b] = "\x63\x78", [0x131c] = "\x5f\x2e",
  [0x131f] = "\x63\x76", [0x1323] = "\x5f\x2f", [0x1327] = "\x64\x24",
  [0x1334] = "\x64\x23", [0x1337] = "\x64\x27", [0x133e] = "\x64\x22",
  [0x133f] = "\x68\x67", [0x1344] = "\x63\x7e", [0x1347] = "\x68\x66",
  [0x1348] = "\x63\x75", [0x134b] = "\x68\x68", [0x134c] = "\x63\x7d",
  [0x134d] = "\x64\x21", [0x134e] = "\x63\x7c", [0x1355] = "\x6c\x67",
  [0x1358] = "\x68\x6e", [0x135a] = "\x68\x6d", [0x135d] = "\x6c\x66",
  [0x135f] = "\x68\x6c", [0x1362] = "\x68\x6a", [0x1363] = "\x68\x6b",
  [0x1367] = "\x6c\x64", [0x1368] = "\x5f\x35", [0x136b] = "\x6c\x6b",
  [0x136e] = "\x6c\x65", [0x1370] = "\x6c\x6a", [0x1375] = "\x68\x6f",
  [0x1376] = "\x6c\x63", [0x1377] = "\x68\x69", [0x137c] = "\x6c\x69",
  [0x137e] = "\x6c\x6c", [0x1382] = "\x6c\x68", [0x138a] = "\x70\x57",
  [0x138e] = "\x6c\x70", [0x1390] = "\x6c\x6e", [0x1391] = "\x70\x55",
  [0x1394] = "\x6c\x74", [0x139a] = "\x6c\x72", [0x13a4] = "\x6c\x73",
  [0x13a7] = "\x6c\x6d", [0x13a9] = "\x70\x56", [0x13ab] = "\x6c\x6f",
  [0x13ac] = "\x6c\x71", [0x13ae] = "\x6c\x75", [0x13b2] = "\x70\x54",
  [0x13b6] = "\x70\x59", [0x13be] = "\x70\x5a", [0x13c2] = "\x73\x5d",
  [0x13c7] = "\x73\x5e", [0x13c8] = "\x70\x5c", [0x13c9] = "\x73\x5c",
  [0x13ca] = "\x70\x5b", [0x13cb] = "\x73\x60", [0x13cd] = "\x70\x58",
  [0x13e3] = "\x76\x42", [0x13e6] = "\x73\x5f", [0x13f2] = "\x78\x34",
  [0x13f5] = "\x78\x37", [0x13f6] = "\x78\x36", [0x13f7] = "\x78\x35",
  [0x13f8] = "\x79\x6c", [0x13fa] = "\x79\x6d", [0x13fc] = "\x7a\x6a",
  [0x13fe] = "\x7a\x6b", [0x13ff] = "\x7b\x60", [0x1400] = "\x7c\x36",
  [0x1408] = "\x45\x41", [0x140a] = "\x46\x53", [0x140c] = "\x48\x27",
  [0x140d] = "\x48\x28", [0x140e] = "\x48\x26", [0x1410] = "\x48\x29",
  [0x1411] = "\x4a\x3c", [0x1412] = "\x4a\x3b", [0x1415] = "\x4d\x51",
  [0x1416] = "\x4d\x50", [0x141a] = "\x5a\x51", [0x141b] = "\x5a\x52",
  [0x141f] = "\x5f\x3c", [0x1421] = "\x64\x28", [0x1422] = "\x64\x29",
  [0x142a] = "\x68\x70", [0x142e] = "\x6c\x76", [0x1430] = "\x70\x5d",
  [0x1432] = "\x73\x61", [0x1433] = "\x76\x43", [0x1434] = "\x73\x62",
  [0x1436] = "\x45\x42", [0x143e] = "\x4d\x53", [0x143f] = "\x4d\x52",
  [0x1440] = "\x4d\x54", [0x1441] = "\x51\x37", [0x1447] = "\x55\x52",
  [0x1448] = "\x5a\x53", [0x1449] = "\x5f\x3d", [0x144b] = "\x45\x43",
  [0x144d] = "\x44\x5f", [0x144e] = "\x45\x44", [0x1451] = "\x46\x57",
  [0x1452] = "\x46\x56", [0x1453] = "\x46\x54", [0x1454] = "\x46\x55",
  [0x1458] = "\x48\x2c", [0x145b] = "\x48\x2b", [0x1463] = "\x48\x2a",
  [0x146d] = "\x4a\x43", [0x146e] = "\x4a\x4c", [0x146f] = "\x4a\x4a",
  [0x1473] = "\x4a\x48", [0x1476] = "\x4a\x41", [0x1479] = "\x4a\x47",
  [0x147c] = "\x4a\x45", [0x147e] = "\x4a\x46", [0x147f] = "\x4d\x55",
  [0x1480] = "\x4a\x40", [0x1484] = "\x4a\x3d", [0x1486] = "\x4a\x50",
  [0x1489] = "\x4a\x42", [0x148a] = "\x4a\x44", [0x1491] = "\x4a\x4f",
  [0x1492] = "\x4a\x49", [0x1493] = "\x4a\x4e", [0x1495] = "\x4a\x4d",
  [0x1496] = "\x4a\x3f", [0x1497] = "\x4a\x3e", [0x1498] = "\x4a\x4b",
  [0x14a8] = "\x4d\x63", [0x14ab] = "\x4d\x5e", [0x14ac] = "\x4d\x71",
  [0x14b1] = "\x4d\x6c", [0x14b5] = "\x4d\x6a", [0x14b9] = "\x4d\x5b",
  [0x14bc] = "\x4d\x65", [0x14bd] = "\x4d\x64", [0x14bf] = "\x4d\x59",
  [0x14c2] = "\x4d\x5a", [0x14c4] = "\x4d\x58", [0x14c6] = "\x4d\x70",
  [0x14c7] = "\x4d\x68", [0x14c8] = "\x4d\x62", [0x14c9] = "\x4d\x56",
  [0x14cb] = "\x4d\x61", [0x14cc] = "\x4d\x57", [0x14cd] = "\x4d\x69",
  [0x14ce] = "\x4d\x72", [0x14d0] = "\x4d\x66", [0x14d2] = "\x4d\x5c",
  [0x14d3] = "\x4d\x5f", [0x14d4] = "\x4d\x60", [0x14d6] = "\x4d\x6e",
  [0x14d7] = "\x4d\x6f", [0x14d8] = "\x4d\x6d", [0x14d9] = "\x4d\x67",
  [0x14da] = "\x4d\x6b", [0x14db] = "\x4d\x5d", [0x14dc] = "\x51\x38",
  [0x14ec] = "\x51\x44", [0x14ed] = "\x51\x3c", [0x14ee] = "\x51\x3e",
  [0x14ef] = "\x51\x43", [0x14f1] = "\x51\x41", [0x14f3] = "\x55\x53",
  [0x14f4] = "\x51\x46", [0x14f7] = "\x51\x42", [0x14fc] = "\x51\x3b",
  [0x14fd] = "\x51\x3f", [0x14fe] = "\x51\x45", [0x14ff] = "\x55\x55",
  [0x1501] = "\x51\x3d", [0x1502] = "\x51\x48", [0x1507] = "\x51\x40",
  [0x1508] = "\x55\x54", [0x1509] = "\x51\x3a", [0x1511] = "\x51\x47",
  [0x1516] = "\x51\x39", [0x1528] = "\x55\x63", [0x152a] = "\x55\x61",
  [0x152b] = "\x55\x62", [0x152f] = "\x55\x58", [0x153a] = "\x55\x5e",
  [0x153d] = "\x55\x60", [0x153e] = "\x55\x57", [0x1542] = "\x55\x5a",
  [0x1546] = "\x55\x5b", [0x1549] = "\x55\x5d", [0x154c] = "\x55\x65",
  [0x154d] = "\x55\x64", [0x154e] = "\x55\x56", [0x154f] = "\x55\x5c",
  [0x1550] = "\x55\x5f", [0x1555] = "\x55\x59", [0x1567] = "\x5a\x5b",
  [0x1568] = "\x5a\x6f", [0x1569] = "\x5a\x6e", [0x156b] = "\x5a\x63",
  [0x1571] = "\x5a\x5e", [0x1572] = "\x5a\x56", [0x1576] = "\x5f\x4d",
  [0x1577] = "\x5a\x5a", [0x157a] = "\x5a\x70", [0x157b] = "\x5a\x6d",
  [0x1580] = "\x5a\x6c", [0x1583] = "\x5a\x61", [0x1584] = "\x5a\x65",
  [0x1588] = "\x5a\x66", [0x1589] = "\x5a\x60", [0x158c] = "\x5f\x3f",
  [0x158f] = "\x5a\x6b", [0x1592] = "\x5a\x6a", [0x1596] = "\x5a\x57",
  [0x1598] = "\x5a\x5c", [0x1599] = "\x5a\x67", [0x159b] = "\x5a\x62",
  [0x15a0] = "\x5a\x54", [0x15a1] = "\x5a\x68", [0x15a2] = "\x5a\x58",
  [0x15a3] = "\x5f\x3e", [0x15a5] = "\x5a\x59", [0x15a7] = "\x5a\x55",
  [0x15a8] = "\x5a\x64", [0x15a9] = "\x5a\x5f", [0x15aa] = "\x5a\x5d",
  [0x15ac] = "\x5a\x69", [0x15c0] = "\x5f\x41", [0x15c6] = "\x5f\x44",
  [0x15c9] = "\x5f\x43", [0x15cd] = "\x5f\x45", [0x15cf] = "\x5f\x40",
  [0x15d0] = "\x5f\x48", [0x15d2] = "\x5f\x46", [0x15d6] = "\x5f\x4a",
  [0x15da] = "\x5f\x52", [0x15db] = "\x5f\x50", [0x15e1] = "\x5f\x49",
  [0x15e3] = "\x5f\x47", [0x15e9] = "\x5f\x42", [0x15ea] = "\x5f\x4f",
  [0x15ed] = "\x5f\x4b", [0x15ee] = "\x5f\x4c", [0x15f4] = "\x5f\x4e",
  [0x15f9] = "\x5f\x53", [0x1606] = "\x64\x38", [0x160d] = "\x64\x34",
  [0x160f] = "\x64\x31", [0x1613] = "\x64\x2a", [0x1614] = "\x64\x33",
  [0x1616] = "\x64\x36", [0x1617] = "\x64\x37", [0x161c] = "\x64\x32",
  [0x161e] = "\x64\x2c", [0x162a] = "\x64\x2d", [0x162c] = "\x64\x30",
  [0x162d] = "\x64\x2e", [0x1634] = "\x68\x7a", [0x1636] = "\x64\x35",
  [0x163d] = "\x64\x2f", [0x163e] = "\x64\x2b", [0x1651] = "\x68\x78",
  [0x1652] = "\x5f\x51", [0x1654] = "\x68\x73", [0x1658] = "\x68\x72",
  [0x165f] = "\x68\x76", [0x1667] = "\x68\x79", [0x1669] = "\x6c\x77",
  [0x166d] = "\x68\x7b", [0x166f] = "\x6c\x78", [0x1678] = "\x68\x75",
  [0x1679] = "\x6c\x79", [0x167a] = "\x68\x77", [0x167b] = "\x68\x7c",
  [0x1687] = "\x68\x71", [0x1688] = "\x6c\x7c", [0x1690] = "\x6c\x7d",
  [0x1692] = "\x6d\x25", [0x1693] = "\x6d\x22", [0x1695] = "\x6d\x23",
  [0x1699] = "\x6d\x2b", [0x169a] = "\x6d\x29", [0x169e] = "\x6c\x7a",
  [0x16a2] = "\x6d\x2c", [0x16a4] = "\x68\x74", [0x16a5] = "\x6d\x21",
  [0x16a9] = "\x6d\x24", [0x16ab] = "\x6d\x28", [0x16ac] = "\x6d\x2a",
  [0x16ad] = "\x6d\x27", [0x16ae] = "\x6d\x26", [0x16b0] = "\x6c\x7e",
  [0x16b2] = "\x6c\x7b", [0x16b3] = "\x6d\x2d", [0x16bb] = "\x70\x61",
  [0x16bc] = "\x70\x62", [0x16be] = "\x70\x6b", [0x16bf] = "\x70\x68",
  [0x16c1] = "\x70\x5f", [0x16c2] = "\x70\x66", [0x16c4] = "\x70\x64",
  [0x16c5] = "\x70\x5e", [0x16c7] = "\x70\x65", [0x16ca] = "\x73\x64",
  [0x16cb] = "\x70\x60", [0x16cd] = "\x70\x67", [0x16ce] = "\x73\x63",
  [0x16d2] = "\x70\x69", [0x16d4] = "\x70\x6a", [0x16d8] = "\x73\x65",
  [0x16da] = "\x70\x63", [0x16e0] = "\x73\x66", [0x16e2] = "\x73\x6b",
  [0x16e6] = "\x73\x68", [0x16ec] = "\x73\x69", [0x16ed] = "\x73\x6c",
  [0x16f0] = "\x73\x67", [0x16f1] = "\x73\x6a", [0x16f2] = "\x76\x45",
  [0x16f4] = "\x76\x44", [0x16f7] = "\x76\x4a", [0x16fa] = "\x76\x48",
  [0x16fb] = "\x76\x49", [0x16fe] = "\x76\x46", [0x1700] = "\x78\x38",
  [0x1706] = "\x76\x47", [0x170f] = "\x78\x39", [0x1714] = "\x79\x6f",
  [0x1718] = "\x79\x6e", [0x1719] = "\x79\x70", [0x171c] = "\x7a\x6d",
  [0x171d] = "\x7a\x6c", [0x1723] = "\x7c\x37", [0x1724] = "\x7b\x61",
  [0x172a] = "\x7c\x39", [0x172b] = "\x7c\x38", [0x172c] = "\x7c\x5f",
  [0x172f] = "\x45\x45", [0x1736] = "\x48\x2d", [0x1738] = "\x4a\x53",
  [0x1739] = "\x4a\x51", [0x173b] = "\x4a\x52", [0x173e] = "\x4d\x73",
  [0x173f] = "\x51\x49", [0x1745] = "\x51\x4a", [0x1748] = "\x55\x66",
  [0x1749] = "\x55\x67", [0x174f] = "\x5a\x77", [0x1751] = "\x5a\x73",
  [0x1754] = "\x5a\x7a", [0x1755] = "\x5a\x79", [0x1756] = "\x5a\x72",
  [0x1757] = "\x5a\x75", [0x1758] = "\x5a\x78", [0x1759] = "\x5a\x74",
  [0x175d] = "\x5a\x71", [0x175e] = "\x5f\x54", [0x1762] = "\x5f\x56",
  [0x1763] = "\x5f\x57", [0x1766] = "\x5f\x55", [0x176c] = "\x64\x39",
  [0x1772] = "\x68\x7d", [0x1774] = "\x70\x6c", [0x1775] = "\x6d\x2e",
  [0x1777] = "\x6d\x2f", [0x1778] = "\x6d\x30", [0x1782] = "\x73\x6d",
  [0x1783] = "\x73\x6e", [0x1787] = "\x45\x46", [0x1790] = "\x5f\x59",
  [0x1791] = "\x5f\x58", [0x1795] = "\x7a\x6e", [0x1797] = "\x45\x47",
  [0x1799] = "\x55\x68", [0x179b] = "\x5a\x7c", [0x179c] = "\x5a\x7b",
  [0x179f] = "\x64\x3a", [0x17a1] = "\x68\x7e", [0x17a4] = "\x45\x48",
  [0x17a5] = "\x46\x58", [0x17a7] = "\x4d\x74", [0x17ab] = "\x51\x4b",
  [0x17ac] = "\x5a\x7d", [0x17af] = "\x5f\x5a", [0x17b0] = "\x64\x3b",
  [0x17b7] = "\x76\x4b", [0x17b9] = "\x45\x49", [0x17bc] = "\x4d\x75",
  [0x17bd] = "\x51\x4c", [0x17c1] = "\x55\x69", [0x17c5] = "\x55\x6a",
  [0x17cb] = "\x5b\x21", [0x17cc] = "\x5b\x22", [0x17ce] = "\x5b\x23",
  [0x17cf] = "\x5a\x7e", [0x17d6] = "\x69\x22", [0x17d7] = "\x69\x21",
  [0x17e2] = "\x51\x4d", [0x17e5] = "\x45\x4a", [0x17e6] = "\x46\x59",
  [0x17e8] = "\x48\x2f", [0x17e9] = "\x48\x2e", [0x17ec] = "\x48\x30",
  [0x17ed] = "\x48\x31", [0x17f1] = "\x4a\x54", [0x17fa] = "\x4d\x76",
  [0x1800] = "\x4d\x7d", [0x1802] = "\x4d\x7b", [0x1806] = "\x4d\x7a",
  [0x1807] = "\x4e\x23", [0x180a] = "\x4e\x22", [0x180c] = "\x4d\x79",
  [0x180e] = "\x4d\x7c", [0x180f] = "\x4d\x7e", [0x1813] = "\x4d\x78",
  [0x1814] = "\x4d\x77", [0x1815] = "\x4e\x21", [0x181f] = "\x51\x53",
  [0x1820] = "\x51\x50", [0x1824] = "\x51\x56", [0x1825] = "\x51\x4e",
  [0x1827] = "\x51\x51", [0x1828] = "\x51\x54", [0x182d] = "\x51\x4f",
  [0x182f] = "\x51\x52", [0x1831] = "\x51\x55", [0x1841] = "\x55\x72",
  [0x1842] = "\x55\x6b", [0x1843] = "\x55\x6e", [0x1845] = "\x55\x71",
  [0x1849] = "\x55\x6c", [0x184c] = "\x55\x70", [0x184f] = "\x55\x6d",
  [0x1852] = "\x55\x6f", [0x185a] = "\x5b\x25", [0x185d] = "\x5b\x24",
  [0x185e] = "\x5b\x29", [0x1864] = "\x5b\x26", [0x1866] = "\x5b\x28",
  [0x1868] = "\x5b\x27", [0x186e] = "\x5f\x5b", [0x186f] = "\x5f\x5f",
  [0x1870] = "\x5f\x5c", [0x1874] = "\x5f\x5d", [0x1876] = "\x5f\x5e",
  [0x1877] = "\x5f\x63", [0x187a] = "\x5f\x61", [0x187e] = "\x5f\x62",
  [0x1884] = "\x64\x41", [0x1887] = "\x64\x3e", [0x1888] = "\x64\x3f",
  [0x1889] = "\x64\x3d", [0x188d] = "\x64\x43", [0x1891] = "\x5f\x60",
  [0x1896] = "\x64\x40", [0x1897] = "\x64\x3c", [0x1898] = "\x64\x42",
  [0x189d] = "\x69\x25", [0x18a2] = "\x69\x23", [0x18a8] = "\x69\x24",
  [0x18ab] = "\x6d\x32", [0x18ae] = "\x6d\x31", [0x18b1] = "\x6d\x34",
  [0x18b4] = "\x6d\x33", [0x18b8] = "\x70\x72", [0x18b9] = "\x70\x6f",
  [0x18c4] = "\x70\x70", [0x18c6] = "\x70\x6d", [0x18c7] = "\x70\x71",
  [0x18c9] = "\x70\x6e", [0x18d6] = "\x73\x70", [0x18d9] = "\x73\x6f",
  [0x18dc] = "\x76\x4c", [0x18dd] = "\x78\x3b", [0x18e0] = "\x78\x3a",
  [0x18e6] = "\x79\x71", [0x18e9] = "\x7a\x6f", [0x18ec] = "\x7c\x3a",
  [0x18f0] = "\x45\x4b", [0x18f2] = "\x48\x32", [0x18f3] = "\x48\x33",
  [0x18f4] = "\x4a\x55", [0x18f7] = "\x51\x57", [0x18f8] = "\x55\x73",
  [0x18f9] = "\x5b\x2a", [0x18fc] = "\x59\x37", [0x18fe] = "\x5f\x64",
  [0x18ff] = "\x5f\x65", [0x1900] = "\x5e\x32", [0x1903] = "\x64\x44",
  [0x1908] = "\x45\x4c", [0x1909] = "\x48\x34", [0x190b] = "\x4e\x25",
  [0x190d] = "\x4e\x24", [0x1914] = "\x55\x74", [0x1915] = "\x55\x75",
  [0x1917] = "\x55\x76", [0x191b] = "\x5b\x2c", [0x191d] = "\x5f\x67",
  [0x191f] = "\x5f\x66", [0x1926] = "\x76\x4d", [0x1927] = "\x79\x72",
  [0x1928] = "\x45\x4d", [0x192a] = "\x46\x5c", [0x192b] = "\x46\x5d",
  [0x192c] = "\x46\x5b", [0x192d] = "\x46\x5e", [0x192e] = "\x46\x5a",
  [0x1931] = "\x48\x37", [0x1934] = "\x48\x36", [0x1935] = "\x48\x38",
  [0x193d] = "\x48\x35", [0x1946] = "\x4a\x5f", [0x1949] = "\x4a\x5e",
  [0x194e] = "\x4a\x57", [0x194f] = "\x4a\x58", [0x1950] = "\x4a\x59",
  [0x1951] = "\x4a\x5a", [0x1953] = "\x4a\x61", [0x1956] = "\x4a\x5c",
  [0x1957] = "\x4a\x62", [0x195c] = "\x4a\x5b", [0x195e] = "\x4a\x5d",
  [0x195f] = "\x4a\x56", [0x1960] = "\x4a\x60", [0x196a] = "\x4e\x3a",
  [0x196d] = "\x4e\x26", [0x196f] = "\x4e\x30", [0x1970] = "\x4e\x31",
  [0x1971] = "\x4e\x29", [0x1972] = "\x4e\x3b", [0x1973] = "\x4e\x2b",
  [0x1975] = "\x4e\x36", [0x1977] = "\x4e\x2c", [0x197c] = "\x4e\x39",
  [0x197e] = "\x4e\x34", [0x197f] = "\x4e\x32", [0x1987] = "\x4e\x2d",
  [0x1989] = "\x4e\x33", [0x198b] = "\x4e\x27", [0x1990] = "\x4e\x35",
  [0x1993] = "\x4e\x38", [0x1995] = "\x4e\x28", [0x1997] = "\x4e\x2f",
  [0x199a] = "\x4e\x37", [0x199c] = "\x4e\x2a", [0x199d] = "\x4e\x2e",
  [0x19af] = "\x51\x5f", [0x19b0] = "\x51\x6c", [0x19b4] = "\x51\x65",
  [0x19b6] = "\x51\x5e", [0x19b8] = "\x51\x68", [0x19c4] = "\x51\x63",
  [0x19cf] = "\x51\x69", [0x19d0] = "\x51\x5c", [0x19d1] = "\x51\x64",
  [0x19d2] = "\x51\x70", [0x19d3] = "\x51\x59", [0x19d4] = "\x51\x5b",
  [0x19d9] = "\x51\x6d", [0x19da] = "\x51\x66", [0x19dd] = "\x51\x6f",
  [0x19de] = "\x51\x6a", [0x19e2] = "\x51\x6e", [0x19e5] = "\x51\x67",
  [0x19e9] = "\x51\x61", [0x19ec] = "\x51\x5d", [0x19ef] = "\x51\x62",
  [0x19f1] = "\x51\x5a", [0x19f3] = "\x51\x6b", [0x19f4] = "\x56\x27",
  [0x19f5] = "\x51\x60", [0x19ff] = "\x51\x58", [0x1a13] = "\x56\x2e",
  [0x1a17] = "\x56\x23", [0x1a18] = "\x56\x2f", [0x1a21] = "\x55\x77",
  [0x1a29] = "\x56\x21", [0x1a2a] = "\x56\x2c", [0x1a38] = "\x55\x78",
  [0x1a39] = "\x55\x7c", [0x1a3c] = "\x56\x2a", [0x1a3d] = "\x56\x26",
  [0x1a40] = "\x56\x29", [0x1a41] = "\x56\x30", [0x1a42] = "\x55\x7d",
  [0x1a43] = "\x56\x2b", [0x1a45] = "\x56\x2d", [0x1a46] = "\x55\x7a",
  [0x1a48] = "\x55\x79", [0x1a4c] = "\x56\x24", [0x1a50] = "\x56\x28",
  [0x1a51] = "\x56\x25", [0x1a53] = "\x55\x7b", [0x1a54] = "\x55\x7e",
  [0x1a76] = "\x5b\x33", [0x1a7f] = "\x5b\x32", [0x1a81] = "\x5b\x2d",
  [0x1a82] = "\x5b\x42", [0x1a83] = "\x5b\x38", [0x1a85] = "\x5b\x3c",
  [0x1a86] = "\x5b\x3b", [0x1a93] = "\x5b\x30", [0x1a94] = "\x5b\x3d",
  [0x1a97] = "\x5b\x36", [0x1a9d] = "\x5b\x3e", [0x1a9f] = "\x5b\x40",
  [0x1aa1] = "\x5b\x41", [0x1aa2] = "\x5b\x2f", [0x1aa7] = "\x5b\x35",
  [0x1aa8] = "\x5b\x3f", [0x1aad] = "\x5b\x3a", [0x1aaf] = "\x5b\x2e",
  [0x1ab0] = "\x5b\x37", [0x1ab1] = "\x5b\x34", [0x1ab3] = "\x56\x22",
  [0x1ab5] = "\x5b\x31", [0x1ac4] = "\x5b\x39", [0x1ac9] = "\x5f\x7b",
  [0x1acb] = "\x5f\x76", [0x1acd] = "\x5f\x77", [0x1ad2] = "\x5f\x73",
  [0x1ad5] = "\x5f\x69", [0x1ad7] = "\x5f\x6c", [0x1ad8] = "\x5f\x6b",
  [0x1ada] = "\x5f\x7c", [0x1adf] = "\x5f\x6e", [0x1ae0] = "\x5f\x6a",
  [0x1ae3] = "\x5f\x75", [0x1ae7] = "\x5f\x71", [0x1aee] = "\x5f\x70",
  [0x1af2] = "\x5f\x74", [0x1af5] = "\x5f\x6f", [0x1af9] = "\x5f\x72",
  [0x1afa] = "\x5f\x68", [0x1afb] = "\x5f\x7e", [0x1b05] = "\x5f\x6d",
  [0x1b0d] = "\x5f\x78", [0x1b0e] = "\x5f\x7a", [0x1b12] = "\x5f\x79",
  [0x1b30] = "\x64\x4c", [0x1b4a] = "\x64\x4e", [0x1b53] = "\x64\x52",
  [0x1b54] = "\x64\x4a", [0x1b5a] = "\x64\x47", [0x1b5b] = "\x64\x57",
  [0x1b5d] = "\x64\x55", [0x1b5e] = "\x64\x51", [0x1b60] = "\x64\x49",
  [0x1b63] = "\x64\x56", [0x1b68] = "\x64\x4f", [0x1b6b] = "\x64\x50",
  [0x1b6d] = "\x64\x46", [0x1b6e] = "\x5f\x7d", [0x1b75] = "\x64\x4b",
  [0x1b77] = "\x64\x48", [0x1b79] = "\x64\x53", [0x1b82] = "\x64\x4d",
  [0x1b86] = "\x64\x54", [0x1b94] = "\x64\x45", [0x1b95] = "\x69\x28",
  [0x1b9b] = "\x69\x2d", [0x1b9c] = "\x69\x26", [0x1ba3] = "\x69\x38",
  [0x1ba6] = "\x69\x36", [0x1ba8] = "\x69\x27", [0x1bab] = "\x69\x30",
  [0x1bad] = "\x69\x34", [0x1bae] = "\x69\x2a", [0x1bb4] = "\x69\x31",
  [0x1bb7] = "\x69\x2e", [0x1bbb] = "\x69\x2f", [0x1bc1] = "\x69\x29",
  [0x1bc3] = "\x69\x37", [0x1bcb] = "\x69\x2c", [0x1bcc] = "\x69\x35",
  [0x1bcd] = "\x69\x33", [0x1bd0] = "\x69\x32", [0x1bd3] = "\x69\x2b",
  [0x1be8] = "\x6d\x37", [0x1bed] = "\x6d\x42", [0x1bf3] = "\x6d\x3f",
  [0x1bfd] = "\x6d\x3b", [0x1c01] = "\x6d\x38", [0x1c02] = "\x6d\x40",
  [0x1c05] = "\x6d\x41", [0x1c0a] = "\x6d\x3e", [0x1c11] = "\x6d\x43",
  [0x1c13] = "\x6d\x3d", [0x1c19] = "\x6d\x3a", [0x1c1e] = "\x6d\x39",
  [0x1c1f] = "\x6d\x36", [0x1c21] = "\x6d\x3c", [0x1c23] = "\x6d\x35",
  [0x1c35] = "\x71\x21", [0x1c38] = "\x70\x74", [0x1c39] = "\x70\x79",
  [0x1c3a] = "\x70\x75", [0x1c3d] = "\x70\x73", [0x1c44] = "\x70\x7a",
  [0x1c47] = "\x70\x7e", [0x1c48] = "\x71\x23", [0x1c4b] = "\x70\x7d",
  [0x1c58] = "\x70\x78", [0x1c59] = "\x70\x76", [0x1c5f] = "\x71\x22",
  [0x1c61] = "\x70\x7c", [0x1c62] = "\x70\x7b", [0x1c6b] = "\x70\x77",
  [0x1c7e] = "\x73\x78", [0x1c80] = "\x73\x71", [0x1c84] = "\x73\x73",
  [0x1c90] = "\x73\x7a", [0x1c94] = "\x73\x72", [0x1c97] = "\x73\x79",
  [0x1c9c] = "\x73\x75", [0x1ca0] = "\x73\x7b", [0x1ca2] = "\x73\x74",
  [0x1ca3] = "\x73\x77", [0x1cac] = "\x76\x4f", [0x1cae] = "\x76\x54",
  [0x1caf] = "\x76\x55", [0x1cb3] = "\x76\x4e", [0x1cb8] = "\x76\x52",
  [0x1cbb] = "\x76\x51", [0x1cc2] = "\x76\x53", [0x1cc3] = "\x76\x50",
  [0x1cd3] = "\x78\x3f", [0x1cda] = "\x78\x3e", [0x1cdb] = "\x73\x76",
  [0x1cdd] = "\x78\x3d", [0x1ce5] = "\x78\x3c", [0x1cec] = "\x79\x73",
  [0x1cfa] = "\x7a\x72", [0x1cfb] = "\x7a\x70", [0x1d04] = "\x7a\x71",
  [0x1d0a] = "\x7b\x62", [0x1d10] = "\x7c\x3b", [0x1d16] = "\x7d\x23",
  [0x1d20] = "\x45\x4e", [0x1d21] = "\x48\x39", [0x1d23] = "\x4e\x3c",
  [0x1d32] = "\x5b\x43", [0x1d3a] = "\x60\x22", [0x1d3d] = "\x60\x23",
  [0x1d3e] = "\x60\x21", [0x1d47] = "\x64\x58", [0x1d49] = "\x69\x39",
  [0x1d4c] = "\x69\x3a", [0x1d4e] = "\x6d\x45", [0x1d50] = "\x6d\x44",
  [0x1d59] = "\x71\x24", [0x1d5c] = "\x73\x7c", [0x1d5f] = "\x76\x56",
  [0x1d61] = "\x7b\x63", [0x1d62] = "\x45\x4f", [0x1d63] = "\x46\x5f",
  [0x1d64] = "\x48\x3a", [0x1d65] = "\x4a\x63", [0x1d66] = "\x4e\x3d",
  [0x1d67] = "\x4e\x3e", [0x1d6a] = "\x51\x71", [0x1d72] = "\x64\x59",
  [0x1d77] = "\x71\x25", [0x1d78] = "\x76\x57", [0x1d79] = "\x45\x50",
  [0x1d7b] = "\x48\x3b", [0x1d7f] = "\x4e\x3f", [0x1d83] = "\x51\x72",
  [0x1d86] = "\x51\x73", [0x1d89] = "\x56\x32", [0x1d8a] = "\x56\x31",
  [0x1d96] = "\x60\x25", [0x1d98] = "\x60\x24", [0x1da4] = "\x6d\x46",
  [0x1dae] = "\x73\x7d", [0x1daf] = "\x76\x58", [0x1db2] = "\x7a\x73",
  [0x1db5] = "\x51\x74", [0x1db7] = "\x56\x33", [0x1dba] = "\x5b\x44",
  [0x1dbc] = "\x60\x26", [0x1dbf] = "\x64\x5b", [0x1dc0] = "\x64\x5a",
  [0x1dc5] = "\x6d\x47", [0x1dc6] = "\x6d\x48", [0x1dcb] = "\x45\x51",
  [0x1dcd] = "\x46\x60", [0x1dcf] = "\x4a\x64", [0x1dd2] = "\x51\x75",
  [0x1dd3] = "\x64\x5c", [0x1dd4] = "\x45\x52", [0x1dd7] = "\x51\x76",
  [0x1dda] = "\x73\x7e", [0x1ddb] = "\x45\x53", [0x1deb] = "\x5b\x45",
  [0x1dec] = "\x5b\x46", [0x1def] = "\x60\x27", [0x1dfd] = "\x64\x5d",
  [0x1e05] = "\x71\x26", [0x1e08] = "\x74\x21", [0x1e0f] = "\x45\x54",
  [0x1e10] = "\x46\x62", [0x1e11] = "\x46\x61", [0x1e13] = "\x4e\x40",
  [0x1e16] = "\x48\x3c", [0x1e1b] = "\x4e\x41", [0x1e1f] = "\x51\x77",
  [0x1e23] = "\x56\x34", [0x1e24] = "\x56\x38", [0x1e26] = "\x56\x37",
  [0x1e27] = "\x56\x35", [0x1e28] = "\x56\x36", [0x1e2b] = "\x5b\x47",
  [0x1e2c] = "\x60\x2a", [0x1e2e] = "\x60\x28", [0x1e2f] = "\x60\x29",
  [0x1e33] = "\x69\x3b", [0x1e34] = "\x45\x55", [0x1e38] = "\x46\x63",
  [0x1e3e] = "\x46\x66", [0x1e40] = "\x46\x65", [0x1e41] = "\x46\x64",
  [0x1e42] = "\x4a\x65", [0x1e4d] = "\x48\x46", [0x1e4e] = "\x48\x47",
  [0x1e50] = "\x48\x42", [0x1e55] = "\x48\x43", [0x1e57] = "\x48\x3e",
  [0x1e59] = "\x48\x3f", [0x1e5b] = "\x48\x45", [0x1e5d] = "\x48\x3d",
  [0x1e5e] = "\x4a\x66", [0x1e5f] = "\x48\x40", [0x1e60] = "\x48\x41",
  [0x1e61] = "\x48\x44", [0x1e68] = "\x4a\x72", [0x1e6a] = "\x4a\x6d",
  [0x1e70] = "\x4a\x70", [0x1e72] = "\x4a\x77", [0x1e74] = "\x4a\x79",
  [0x1e76] = "\x4a\x7b", [0x1e7a] = "\x4a\x6e", [0x1e7d] = "\x4a\x75",
  [0x1e7e] = "\x4a\x78", [0x1e81] = "\x4a\x68", [0x1e82] = "\x4b\x21",
  [0x1e83] = "\x4a\x76", [0x1e85] = "\x4a\x6b", [0x1e86] = "\x4a\x7a",
  [0x1e88] = "\x4a\x69", [0x1e89] = "\x4a\x6a", [0x1e8c] = "\x4a\x71",
  [0x1e8d] = "\x4a\x7c", [0x1e90] = "\x4a\x6f", [0x1e92] = "\x4a\x74",
  [0x1e94] = "\x4a\x7d", [0x1e96] = "\x4a\x73", [0x1e98] = "\x4a\x7e",
  [0x1e99] = "\x4a\x67", [0x1e9b] = "\x4a\x6c", [0x1eab] = "\x4e\x4d",
  [0x1eac] = "\x4e\x5d", [0x1eae] = "\x4e\x54", [0x1eb1] = "\x4e\x45",
  [0x1eb3] = "\x4e\x48", [0x1eb8] = "\x4e\x50", [0x1eb9] = "\x4e\x52",
  [0x1ebb] = "\x4e\x59", [0x1ebc] = "\x4e\x4b", [0x1ebd] = "\x4e\x49",
  [0x1ebe] = "\x4e\x4a", [0x1ebf] = "\x4e\x58", [0x1ec1] = "\x4e\x53",
  [0x1ec4] = "\x4e\x51", [0x1ec5] = "\x4e\x56", [0x1ec9] = "\x51\x78",
  [0x1eca] = "\x4e\x5c", [0x1ecc] = "\x4e\x46", [0x1ed3] = "\x4e\x4f",
  [0x1ed5] = "\x4e\x4e", [0x1ed6] = "\x4e\x60", [0x1ed7] = "\x4e\x55",
  [0x1edb] = "\x4e\x5b", [0x1edc] = "\x4e\x5f", [0x1ee0] = "\x4e\x61",
  [0x1ee1] = "\x4e\x5a", [0x1ee2] = "\x4e\x4c", [0x1ee3] = "\x4e\x42",
  [0x1ee5] = "\x4e\x47", [0x1ee8] = "\x4e\x43", [0x1eef] = "\x4e\x5e",
  [0x1ef0] = "\x56\x39", [0x1ef1] = "\x4e\x57", [0x1ef3] = "\x4e\x44",
  [0x1ef5] = "\x52\x29", [0x1f0b] = "\x51\x79", [0x1f0c] = "\x51\x7e",
  [0x1f0e] = "\x52\x30", [0x1f17] = "\x52\x23", [0x1f1b] = "\x52\x28",
  [0x1f1e] = "\x52\x22", [0x1f25] = "\x51\x7d", [0x1f27] = "\x52\x2b",
  [0x1f29] = "\x52\x2d", [0x1f2a] = "\x51\x7b", [0x1f2b] = "\x52\x31",
  [0x1f2e] = "\x52\x2e", [0x1f31] = "\x52\x21", [0x1f32] = "\x51\x7a",
  [0x1f35] = "\x52\x2f", [0x1f36] = "\x52\x27", [0x1f38] = "\x52\x2c",
  [0x1f39] = "\x52\x2a", [0x1f3b] = "\x52\x24", [0x1f3d] = "\x52\x25",
  [0x1f3e] = "\x52\x26", [0x1f41] = "\x51\x7c", [0x1f59] = "\x56\x41",
  [0x1f5a] = "\x56\x46", [0x1f65] = "\x56\x4d", [0x1f66] = "\x56\x3e",
  [0x1f69] = "\x56\x48", [0x1f6a] = "\x56\x3a", [0x1f6c] = "\x56\x43",
  [0x1f6e] = "\x56\x45", [0x1f74] = "\x56\x47", [0x1f77] = "\x56\x40",
  [0x1f78] = "\x56\x3f", [0x1f79] = "\x56\x4b", [0x1f85] = "\x56\x4c",
  [0x1f87] = "\x56\x3d", [0x1f88] = "\x56\x3c", [0x1f89] = "\x56\x44",
  [0x1f8a] = "\x56\x4a", [0x1f8c] = "\x56\x49", [0x1f8e] = "\x5b\x48",
  [0x1f93] = "\x56\x42", [0x1f94] = "\x56\x4e", [0x1f95] = "\x56\x3b",
  [0x1faa] = "\x5b\x6a", [0x1fae] = "\x5b\x57", [0x1faf] = "\x5b\x55",
  [0x1fb2] = "\x5b\x4c", [0x1fb5] = "\x5b\x60", [0x1fb8] = "\x5b\x5a",
  [0x1fbc] = "\x5b\x49", [0x1fbf] = "\x5b\x6c", [0x1fc4] = "\x5b\x69",
  [0x1fc5] = "\x5b\x5d", [0x1fc6] = "\x5b\x68", [0x1fc7] = "\x5b\x53",
  [0x1fcb] = "\x5b\x54", [0x1fcc] = "\x5b\x4e", [0x1fd1] = "\x5b\x56",
  [0x1fd2] = "\x5b\x5e", [0x1fd8] = "\x5b\x63", [0x1fd9] = "\x5b\x4b",
  [0x1fda] = "\x5b\x61", [0x1fde] = "\x5b\x58", [0x1fe1] = "\x5b\x4d",
  [0x1fe4] = "\x5b\x4f", [0x1fe6] = "\x5b\x6d", [0x1fe8] = "\x5b\x67",
  [0x1fea] = "\x5b\x64", [0x1feb] = "\x5b\x62", [0x1fec] = "\x5b\x6b",
  [0x1fee] = "\x5b\x66", [0x1ff1] = "\x5b\x65", [0x1ff3] = "\x5b\x4a",
  [0x1ff5] = "\x5b\x5c", [0x1ff7] = "\x5b\x5b", [0x1ff9] = "\x5b\x59",
  [0x1ffa] = "\x5b\x51", [0x1ffb] = "\x5b\x50", [0x2005] = "\x5b\x52",
  [0x2019] = "\x60\x47", [0x201a] = "\x5b\x5f", [0x201b] = "\x60\x35",
  [0x201d] = "\x60\x43", [0x2020] = "\x60\x32", [0x2021] = "\x60\x2e",
  [0x2023] = "\x60\x34", [0x2024] = "\x60\x38", [0x2025] = "\x60\x33",
  [0x2026] = "\x60\x3c", [0x202c] = "\x60\x41", [0x202d] = "\x60\x3b",
  [0x202f] = "\x60\x2b", [0x2032] = "\x60\x2f", [0x2034] = "\x60\x3e",
  [0x2038] = "\x60\x2c", [0x203a] = "\x60\x40", [0x203e] = "\x60\x44",
  [0x2043] = "\x60\x42", [0x2044] = "\x60\x4a", [0x204a] = "\x60\x31",
  [0x204d] = "\x60\x3f", [0x204e] = "\x60\x48", [0x2054] = "\x60\x2d",
  [0x2056] = "\x60\x39", [0x2058] = "\x60\x37", [0x205b] = "\x60\x36",
  [0x205f] = "\x60\x4d", [0x2063] = "\x60\x49", [0x2067] = "\x60\x30",
  [0x2069] = "\x60\x4c", [0x206e] = "\x60\x3a", [0x206f] = "\x60\x3d",
  [0x2072] = "\x60\x4b", [0x2089] = "\x60\x46", [0x2090] = "\x64\x63",
  [0x2096] = "\x64\x6d", [0x2098] = "\x64\x68", [0x209c] = "\x64\x6e",
  [0x209d] = "\x64\x64", [0x20a2] = "\x64\x5e", [0x20a5] = "\x64\x67",
  [0x20a7] = "\x64\x72", [0x20aa] = "\x64\x71", [0x20ab] = "\x64\x6b",
  [0x20af] = "\x64\x5f", [0x20b4] = "\x64\x73", [0x20b6] = "\x64\x61",
  [0x20ba] = "\x64\x6a", [0x20bc] = "\x64\x69", [0x20c2] = "\x64\x62",
  [0x20c4] = "\x64\x6f", [0x20c5] = "\x64\x66", [0x20c7] = "\x64\x65",
  [0x20cb] = "\x60\x45", [0x20cc] = "\x69\x57", [0x20d1] = "\x64\x6c",
  [0x20d3] = "\x64\x60", [0x20d4] = "\x64\x70", [0x20d5] = "\x6d\x5a",
  [0x20ec] = "\x69\x54", [0x20ef] = "\x69\x49", [0x20f2] = "\x69\x56",
  [0x20f4] = "\x69\x40", [0x20f7] = "\x69\x58", [0x20fe] = "\x69\x3e",
  [0x20ff] = "\x69\x48", [0x2101] = "\x69\x55", [0x2102] = "\x69\x46",
  [0x2106] = "\x69\x4a", [0x210f] = "\x69\x45", [0x2113] = "\x69\x3f",
  [0x2114] = "\x69\x3d", [0x2115] = "\x69\x4f", [0x2120] = "\x69\x43",
  [0x2122] = "\x69\x47", [0x2123] = "\x69\x4e", [0x2129] = "\x69\x41",
  [0x212a] = "\x69\x53", [0x212b] = "\x69\x50", [0x212c] = "\x69\x44",
  [0x212f] = "\x69\x51", [0x2131] = "\x69\x4b", [0x2132] = "\x69\x4d",
  [0x2133] = "\x69\x3c", [0x2138] = "\x69\x4c", [0x213e] = "\x69\x42",
  [0x213f] = "\x6d\x49", [0x2151] = "\x6d\x4c", [0x2154] = "\x6d\x4e",
  [0x2158] = "\x6d\x59", [0x215b] = "\x6d\x51", [0x215f] = "\x6d\x5d",
  [0x2160] = "\x6d\x5c", [0x2164] = "\x6d\x57", [0x2166] = "\x6d\x4d",
  [0x216d] = "\x6d\x50", [0x216e] = "\x6d\x53", [0x216f] = "\x6d\x5b",
  [0x2170] = "\x6d\x56", [0x2178] = "\x6d\x52", [0x217a] = "\x6d\x55",
  [0x217c] = "\x6d\x4a", [0x2180] = "\x74\x2a", [0x2184] = "\x6d\x4b",
  [0x2186] = "\x6d\x4f", [0x2188] = "\x69\x52", [0x218e] = "\x6d\x54",
  [0x2197] = "\x6d\x58", [0x21a0] = "\x71\x33", [0x21a1] = "\x71\x29",
  [0x21a4] = "\x71\x2b", [0x21a6] = "\x71\x32", [0x21a7] = "\x71\x2d",
  [0x21b1] = "\x71\x28", [0x21b3] = "\x71\x2e", [0x21b4] = "\x71\x34",
  [0x21b6] = "\x71\x31", [0x21b9] = "\x71\x30", [0x21c0] = "\x71\x2f",
  [0x21c1] = "\x71\x2c", [0x21c2] = "\x71\x27", [0x21c3] = "\x71\x2a",
  [0x21d5] = "\x74\x2e", [0x21d8] = "\x74\x22", [0x21db] = "\x74\x26",
  [0x21df] = "\x74\x24", [0x21e0] = "\x74\x25", [0x21e1] = "\x74\x2c",
  [0x21e4] = "\x74\x27", [0x21e9] = "\x74\x2d", [0x21eb] = "\x74\x28",
  [0x21ec] = "\x74\x2b", [0x21ee] = "\x74\x2f", [0x21ef] = "\x74\x29",
  [0x21f0] = "\x74\x30", [0x21f1] = "\x74\x23", [0x21fa] = "\x76\x5d",
  [0x21fe] = "\x76\x5b", [0x2206] = "\x76\x5c", [0x2209] = "\x76\x59",
  [0x220b] = "\x76\x5a", [0x220f] = "\x76\x5f", [0x2211] = "\x76\x5e",
  [0x2215] = "\x78\x45", [0x2218] = "\x78\x46", [0x221a] = "\x78\x43",
  [0x221b] = "\x78\x40", [0x221d] = "\x78\x44", [0x221f] = "\x78\x41",
  [0x2228] = "\x78\x42", [0x2230] = "\x79\x75", [0x2232] = "\x79\x76",
  [0x223e] = "\x79\x74", [0x224c] = "\x7a\x74", [0x2251] = "\x7b\x64",
  [0x2258] = "\x7b\x65", [0x225e] = "\x7c\x60", [0x2263] = "\x7d\x24",
  [0x2264] = "\x7d\x32", [0x226b] = "\x45\x56", [0x2270] = "\x48\x48",
  [0x2276] = "\x4b\x22", [0x2278] = "\x4b\x25", [0x227c] = "\x4b\x23",
  [0x227d] = "\x4b\x24", [0x228a] = "\x4e\x65", [0x228e] = "\x4e\x63",
  [0x2292] = "\x4e\x64", [0x2295] = "\x4e\x62", [0x2299] = "\x4e\x66",
  [0x22a4] = "\x52\x3a", [0x22ab] = "\x52\x32", [0x22ac] = "\x52\x35",
  [0x22ad] = "\x52\x37", [0x22ae] = "\x52\x39", [0x22af] = "\x52\x36",
  [0x22b3] = "\x52\x34", [0x22b8] = "\x52\x38", [0x22ba] = "\x52\x33",
  [0x22c8] = "\x56\x53", [0x22ca] = "\x56\x4f", [0x22cf] = "\x56\x54",
  [0x22d8] = "\x56\x50", [0x22d9] = "\x56\x52", [0x22e4] = "\x56\x51",
  [0x22ef] = "\x5b\x72", [0x22f9] = "\x5b\x6e", [0x22fd] = "\x5b\x71",
  [0x2309] = "\x5b\x6f", [0x230a] = "\x5b\x70", [0x2319] = "\x60\x4e",
  [0x231a] = "\x60\x4f", [0x231c] = "\x60\x55", [0x2321] = "\x60\x52",
  [0x2326] = "\x60\x50", [0x2330] = "\x60\x51", [0x2336] = "\x60\x53",
  [0x2346] = "\x65\x22", [0x2349] = "\x64\x78", [0x234c] = "\x64\x7d",
  [0x234e] = "\x64\x74", [0x2356] = "\x65\x24", [0x2359] = "\x64\x75",
  [0x235c] = "\x64\x7a", [0x235e] = "\x65\x21", [0x2364] = "\x64\x77",
  [0x2365] = "\x64\x7e", [0x2366] = "\x64\x7c", [0x2367] = "\x64\x79",
  [0x2368] = "\x65\x23", [0x2369] = "\x64\x76", [0x236c] = "\x64\x7b",
  [0x236e] = "\x60\x54", [0x237d] = "\x69\x5b", [0x2384] = "\x69\x5d",
  [0x238a] = "\x69\x5c", [0x2392] = "\x69\x5e", [0x2394] = "\x69\x59",
  [0x2399] = "\x69\x5a", [0x239f] = "\x6d\x5e", [0x23a8] = "\x6d\x61",
  [0x23ac] = "\x6d\x5f", [0x23b1] = "\x6d\x60", [0x23b9] = "\x71\x3b",
  [0x23be] = "\x71\x35", [0x23c3] = "\x71\x3f", [0x23c4] = "\x71\x40",
  [0x23c8] = "\x71\x39", [0x23c9] = "\x71\x36", [0x23ce] = "\x71\x3c",
  [0x23d0] = "\x71\x37", [0x23d2] = "\x71\x38", [0x23d5] = "\x71\x3a",
  [0x23d9] = "\x71\x3d", [0x23dc] = "\x71\x3e", [0x23df] = "\x74\x32",
  [0x23e0] = "\x74\x39", [0x23e5] = "\x74\x35", [0x23e6] = "\x74\x34",
  [0x23e7] = "\x74\x31", [0x23ec] = "\x74\x37", [0x23ed] = "\x74\x36",
  [0x23ee] = "\x74\x33", [0x23f4] = "\x74\x38", [0x23f8] = "\x76\x63",
  [0x23fb] = "\x76\x60", [0x23fc] = "\x76\x61", [0x23fe] = "\x76\x62",
  [0x2406] = "\x78\x47", [0x240d] = "\x78\x48", [0x2410] = "\x79\x77",
  [0x241b] = "\x7a\x75", [0x2428] = "\x7d\x46", [0x242a] = "\x45\x57",
  [0x242c] = "\x4e\x67", [0x242d] = "\x4e\x68", [0x2430] = "\x52\x3b",
  [0x2435] = "\x74\x3a", [0x2436] = "\x45\x58", [0x2438] = "\x4e\x69",
  [0x2439] = "\x56\x55", [0x243a] = "\x65\x25", [0x243b] = "\x45\x59",
  [0x243d] = "\x5b\x73", [0x243e] = "\x69\x5f", [0x2446] = "\x74\x3b",
  [0x2447] = "\x45\x5a", [0x2448] = "\x4e\x6a", [0x244c] = "\x60\x56",
  [0x2452] = "\x65\x26", [0x2456] = "\x6d\x62", [0x2458] = "\x78\x49",
  [0x2459] = "\x45\x5b", [0x245b] = "\x45\x5c", [0x245d] = "\x48\x4a",
  [0x245f] = "\x48\x49", [0x2460] = "\x4b\x28", [0x2461] = "\x4b\x27",
  [0x2462] = "\x4b\x26", [0x2467] = "\x4e\x6b", [0x2469] = "\x4e\x6c",
  [0x246f] = "\x52\x3d", [0x2472] = "\x52\x3c", [0x2474] = "\x52\x3e",
  [0x2479] = "\x56\x56", [0x247d] = "\x5b\x74", [0x2480] = "\x60\x58",
  [0x2481] = "\x5b\x75", [0x2484] = "\x60\x57", [0x2492] = "\x69\x60",
  [0x2496] = "\x69\x61", [0x249b] = "\x6d\x63", [0x24a2] = "\x78\x4a",
  [0x24a7] = "\x7a\x76", [0x24ac] = "\x45\x5d", [0x24af] = "\x46\x67",
  [0x24c0] = "\x4e\x6d", [0x24c2] = "\x4b\x2a", [0x24c4] = "\x4b\x29",
  [0x24ce] = "\x4e\x6e", [0x24d0] = "\x4e\x71", [0x24d7] = "\x4e\x70",
  [0x24d9] = "\x4e\x6f", [0x24e0] = "\x52\x40", [0x24e1] = "\x52\x41",
  [0x24e9] = "\x52\x3f", [0x24f7] = "\x56\x5b", [0x24f8] = "\x56\x5a",
  [0x24f9] = "\x56\x58", [0x24fc] = "\x56\x57", [0x24fd] = "\x56\x59",
  [0x2513] = "\x5b\x79", [0x2516] = "\x5b\x78", [0x2519] = "\x5b\x7a",
  [0x251b] = "\x5b\x77", [0x251c] = "\x5b\x76", [0x2525] = "\x60\x5a",
  [0x2529] = "\x60\x5c", [0x2534] = "\x60\x5b", [0x2536] = "\x60\x59",
  [0x2537] = "\x65\x27", [0x253e] = "\x65\x2a", [0x253f] = "\x65\x29",
  [0x2544] = "\x69\x62", [0x2545] = "\x65\x28", [0x254e] = "\x6d\x64",
  [0x2550] = "\x69\x63", [0x2557] = "\x6d\x65", [0x2568] = "\x71\x41",
  [0x2570] = "\x74\x3c", [0x2572] = "\x74\x3d", [0x2575] = "\x76\x65",
  [0x2577] = "\x76\x64", [0x2578] = "\x78\x4b", [0x257a] = "\x78\x4c",
  [0x257b] = "\x79\x78", [0x2580] = "\x7b\x66", [0x2584] = "\x46\x68",
  [0x2586] = "\x56\x5c", [0x2587] = "\x5b\x7b", [0x2589] = "\x46\x69",
  [0x258b] = "\x45\x5e", [0x2596] = "\x4b\x2b", [0x259f] = "\x4e\x74",
  [0x25a5] = "\x4e\x76", [0x25a8] = "\x4e\x73", [0x25a9] = "\x4e\x72",
  [0x25ab] = "\x4e\x75", [0x25b2] = "\x52\x45", [0x25b3] = "\x52\x48",
  [0x25b7] = "\x52\x42", [0x25bb] = "\x52\x44", [0x25c0] = "\x52\x47",
  [0x25ca] = "\x52\x43", [0x25cd] = "\x52\x46", [0x25de] = "\x56\x62",
  [0x25e0] = "\x56\x60", [0x25ea] = "\x56\x61", [0x25ed] = "\x56\x5d",
  [0x25ee] = "\x56\x5f", [0x25fe] = "\x5c\x22", [0x2603] = "\x5b\x7e",
  [0x2605] = "\x5b\x7c", [0x2606] = "\x5c\x21", [0x2609] = "\x56\x5e",
  [0x260a] = "\x5b\x7d", [0x260d] = "\x5c\x23", [0x261b] = "\x60\x66",
  [0x2622] = "\x60\x60", [0x2625] = "\x60\x61", [0x2626] = "\x60\x67",
  [0x2628] = "\x60\x68", [0x262a] = "\x60\x5e", [0x262f] = "\x60\x65",
  [0x2633] = "\x60\x5f", [0x2634] = "\x60\x64", [0x2635] = "\x60\x62",
  [0x2636] = "\x60\x63", [0x263a] = "\x60\x5d", [0x263f] = "\x65\x31",
  [0x2641] = "\x65\x30", [0x2655] = "\x65\x2d", [0x2659] = "\x65\x32",
  [0x265a] = "\x65\x2c", [0x265b] = "\x65\x33", [0x265c] = "\x65\x34",
  [0x265e] = "\x65\x2f", [0x265f] = "\x65\x2e", [0x2663] = "\x69\x65",
  [0x2664] = "\x69\x64", [0x2669] = "\x6d\x66", [0x266a] = "\x69\x66",
  [0x266d] = "\x69\x68", [0x266f] = "\x65\x2b", [0x2670] = "\x69\x67",
  [0x267e] = "\x6d\x69", [0x2680] = "\x6d\x6a", [0x2683] = "\x6d\x68",
  [0x268b] = "\x6d\x67", [0x2698] = "\x71\x44", [0x269c] = "\x71\x42",
  [0x269e] = "\x71\x46", [0x269f] = "\x71\x45", [0x26a3] = "\x71\x43",
  [0x26a6] = "\x74\x40", [0x26a7] = "\x76\x66", [0x26a8] = "\x74\x41",
  [0x26a9] = "\x74\x3e", [0x26b0] = "\x74\x3f", [0x26bd] = "\x78\x4d",
  [0x26bf] = "\x76\x67", [0x26ca] = "\x78\x4e", [0x26cf] = "\x79\x79",
  [0x26d4] = "\x7a\x78", [0x26d6] = "\x7a\x77", [0x26da] = "\x7c\x3c",
  [0x26dc] = "\x46\x6a", [0x26e0] = "\x5c\x24", [0x26e2] = "\x71\x47",
  [0x26e3] = "\x78\x4f", [0x26e4] = "\x7b\x67", [0x26e6] = "\x46\x6b",
  [0x26e9] = "\x23\x22", [0x26f6] = "\x5c\x25", [0x26f7] = "\x5c\x26",
  [0x2704] = "\x69\x69", [0x270c] = "\x71\x48", [0x270d] = "\x71\x49",
  [0x2715] = "\x76\x68", [0x2718] = "\x46\x6c", [0x271a] = "\x52\x49",
  [0x271c] = "\x5c\x27", [0x271f] = "\x46\x6d", [0x2722] = "\x5c\x28",
  [0x2725] = "\x60\x69", [0x2726] = "\x60\x6a", [0x2728] = "\x46\x6e",
  [0x2729] = "\x46\x6f", [0x272b] = "\x4b\x2d", [0x272c] = "\x4b\x2c",
  [0x272d] = "\x52\x4a", [0x2730] = "\x46\x70", [0x2731] = "\x46\x71",
  [0x2732] = "\x46\x72", [0x2733] = "\x46\x73", [0x2737] = "\x4b\x2e",
  [0x2738] = "\x4b\x2f", [0x273d] = "\x4e\x77", [0x274b] = "\x52\x4e",
  [0x274c] = "\x52\x4c", [0x274e] = "\x52\x4d", [0x274f] = "\x52\x4b",
  [0x2754] = "\x56\x63", [0x2759] = "\x56\x67", [0x275a] = "\x56\x66",
  [0x275c] = "\x56\x65", [0x275d] = "\x56\x64", [0x2762] = "\x5c\x2b",
  [0x2765] = "\x5c\x29", [0x2766] = "\x5c\x2a", [0x276a] = "\x60\x6c",
  [0x276b] = "\x60\x6b", [0x2770] = "\x5c\x2c", [0x2776] = "\x65\x35",
  [0x2778] = "\x65\x36", [0x277f] = "\x6d\x6b", [0x2786] = "\x78\x51",
  [0x2787] = "\x78\x50", [0x278a] = "\x7b\x68", [0x278b] = "\x46\x74",
  [0x278f] = "\x5c\x2d", [0x2791] = "\x69\x6a", [0x2799] = "\x4e\x79",
  [0x279a] = "\x4e\x7a", [0x279d] = "\x4e\x78", [0x27a2] = "\x52\x52",
  [0x27a3] = "\x52\x53", [0x27a4] = "\x52\x50", [0x27a5] = "\x52\x51",
  [0x27ab] = "\x52\x4f", [0x27b2] = "\x56\x6b", [0x27b3] = "\x56\x6c",
  [0x27b5] = "\x5c\x30", [0x27b8] = "\x56\x71", [0x27b9] = "\x56\x6f",
  [0x27bc] = "\x56\x6e", [0x27bd] = "\x56\x6d", [0x27be] = "\x56\x68",
  [0x27c2] = "\x56\x70", [0x27c5] = "\x56\x69", [0x27c7] = "\x56\x6a",
  [0x27ca] = "\x5c\x31", [0x27cd] = "\x5c\x32", [0x27d4] = "\x5c\x2e",
  [0x27d5] = "\x5c\x2f", [0x27d8] = "\x60\x71", [0x27d9] = "\x60\x70",
  [0x27db] = "\x60\x6e", [0x27de] = "\x60\x72", [0x27e0] = "\x60\x73",
  [0x27e2] = "\x60\x6d", [0x27e3] = "\x60\x6f", [0x27f0] = "\x65\x38",
  [0x27f1] = "\x65\x3b", [0x27f2] = "\x65\x3a", [0x27f3] = "\x65\x3f",
  [0x27f4] = "\x65\x3e", [0x27fa] = "\x65\x3c", [0x27ff] = "\x65\x3d",
  [0x2800] = "\x65\x37", [0x2801] = "\x65\x39", [0x2809] = "\x69\x6e",
  [0x280b] = "\x69\x6d", [0x280d] = "\x69\x6c", [0x2813] = "\x69\x6f",
  [0x281f] = "\x6d\x6e", [0x2820] = "\x6d\x6c", [0x2821] = "\x6d\x71",
  [0x2822] = "\x6d\x72", [0x2824] = "\x6d\x6f", [0x2826] = "\x6d\x70",
  [0x2827] = "\x69\x6b", [0x2829] = "\x6d\x6d", [0x2834] = "\x71\x4a",
  [0x2838] = "\x71\x4b", [0x283a] = "\x71\x4c", [0x2842] = "\x74\x43",
  [0x2846] = "\x74\x42", [0x284c] = "\x74\x44", [0x2852] = "\x76\x6b",
  [0x2856] = "\x76\x69", [0x2858] = "\x76\x6a", [0x285f] = "\x78\x52",
  [0x2861] = "\x78\x53", [0x2862] = "\x79\x7a", [0x2865] = "\x79\x7b",
  [0x2869] = "\x7a\x79", [0x286c] = "\x7b\x6a", [0x286e] = "\x7b\x69",
  [0x2871] = "\x7c\x61", [0x2872] = "\x7c\x62", [0x2878] = "\x52\x54",
  [0x287b] = "\x60\x74", [0x287c] = "\x60\x75", [0x287d] = "\x46\x75",
  [0x287e] = "\x48\x4b", [0x2882] = "\x4b\x30", [0x2884] = "\x4e\x7b",
  [0x2886] = "\x52\x55", [0x2887] = "\x52\x56", [0x2888] = "\x52\x57",
  [0x288b] = "\x56\x72", [0x288e] = "\x5c\x33", [0x2893] = "\x60\x77",
  [0x2896] = "\x60\x76", [0x289a] = "\x6d\x73", [0x28ae] = "\x46\x76",
  [0x28b0] = "\x56\x73", [0x28b4] = "\x60\x78", [0x28ba] = "\x6d\x74",
  [0x28bf] = "\x46\x77", [0x28c2] = "\x4e\x7c", [0x28c3] = "\x52\x5a",
  [0x28c5] = "\x52\x5b", [0x28c6] = "\x52\x59", [0x28c8] = "\x52\x58",
  [0x28ca] = "\x56\x74", [0x28cd] = "\x56\x75", [0x28ce] = "\x56\x76",
  [0x28d2] = "\x5c\x35", [0x28d4] = "\x5c\x34", [0x28db] = "\x5c\x36",
  [0x28dc] = "\x60\x79", [0x28de] = "\x65\x40", [0x28df] = "\x65\x41",
  [0x28e1] = "\x69\x70", [0x28e3] = "\x69\x71", [0x28e4] = "\x6d\x75",
  [0x28e5] = "\x71\x4e", [0x28e7] = "\x71\x4d", [0x28ea] = "\x74\x45",
  [0x28ee] = "\x46\x78", [0x28ef] = "\x4b\x31", [0x28f2] = "\x4e\x7d",
  [0x28f4] = "\x4e\x7e", [0x28f8] = "\x52\x5e", [0x28f9] = "\x52\x5d",
  [0x28fc] = "\x52\x62", [0x28fe] = "\x52\x61", [0x2901] = "\x52\x5c",
  [0x2907] = "\x52\x63", [0x2909] = "\x52\x5f", [0x290b] = "\x52\x60",
  [0x291f] = "\x56\x78", [0x2920] = "\x56\x79", [0x2928] = "\x56\x7a",
  [0x2929] = "\x56\x77", [0x2936] = "\x5c\x3a", [0x2937] = "\x5c\x37",
  [0x2938] = "\x5c\x3b", [0x293a] = "\x5c\x3c", [0x293c] = "\x5c\x39",
  [0x293e] = "\x5c\x38", [0x294f] = "\x60\x7a", [0x295b] = "\x65\x42",
  [0x295c] = "\x65\x4a", [0x295e] = "\x65\x45", [0x2961] = "\x69\x75",
  [0x2962] = "\x65\x4d", [0x2963] = "\x65\x46", [0x2965] = "\x65\x4b",
  [0x2966] = "\x65\x44", [0x2968] = "\x65\x4c", [0x296a] = "\x65\x48",
  [0x296b] = "\x65\x43", [0x296c] = "\x65\x49", [0x2979] = "\x65\x47",
  [0x297d] = "\x69\x73", [0x297f] = "\x69\x74", [0x2984] = "\x69\x72",
  [0x2987] = "\x6d\x77", [0x298b] = "\x6d\x7a", [0x298c] = "\x6d\x78",
  [0x298e] = "\x6d\x76", [0x2991] = "\x6d\x79", [0x299e] = "\x71\x50",
  [0x299f] = "\x71\x51", [0x29a0] = "\x71\x4f", [0x29a5] = "\x71\x52",
  [0x29a7] = "\x74\x4a", [0x29aa] = "\x74\x47", [0x29ac] = "\x74\x49",
  [0x29ad] = "\x74\x4b", [0x29b0] = "\x74\x48", [0x29b3] = "\x74\x46",
  [0x29bb] = "\x76\x6e", [0x29bc] = "\x76\x6f", [0x29bd] = "\x76\x6c",
  [0x29bf] = "\x76\x6d", [0x29c7] = "\x78\x54", [0x29d3] = "\x7a\x7a",
  [0x29d7] = "\x7c\x63", [0x29da] = "\x7d\x33", [0x29db] = "\x46\x79",
  [0x29dc] = "\x52\x64", [0x29e2] = "\x46\x7a", [0x29e3] = "\x4b\x32",
  [0x29e5] = "\x4f\x21", [0x29e9] = "\x56\x7b", [0x29ed] = "\x60\x7b",
  [0x29ee] = "\x65\x4e", [0x29ef] = "\x74\x4c", [0x29f3] = "\x46\x7b",
  [0x29fd] = "\x4f\x22", [0x2a02] = "\x52\x65", [0x2a0c] = "\x52\x67",
  [0x2a0d] = "\x52\x68", [0x2a14] = "\x52\x66", [0x2a1d] = "\x57\x21",
  [0x2a1f] = "\x57\x27", [0x2a20] = "\x57\x26", [0x2a25] = "\x57\x24",
  [0x2a27] = "\x56\x7d", [0x2a2d] = "\x57\x25", [0x2a30] = "\x56\x7c",
  [0x2a32] = "\x57\x28", [0x2a34] = "\x57\x22", [0x2a37] = "\x57\x23",
  [0x2a38] = "\x56\x7e", [0x2a43] = "\x5c\x3e", [0x2a4e] = "\x5c\x3f",
  [0x2a5d] = "\x60\x7c", [0x2a6b] = "\x5c\x3d", [0x2a6c] = "\x60\x7d",
  [0x2a6f] = "\x60\x7e", [0x2a7c] = "\x65\x55", [0x2a7f] = "\x65\x58",
  [0x2a89] = "\x65\x54", [0x2a8c] = "\x65\x53", [0x2a8e] = "\x65\x4f",
  [0x2a91] = "\x65\x56", [0x2a93] = "\x65\x57", [0x2a97] = "\x65\x51",
  [0x2a98] = "\x65\x52", [0x2a9f] = "\x69\x77", [0x2aa3] = "\x69\x7b",
  [0x2aa7] = "\x69\x78", [0x2aa9] = "\x69\x7a", [0x2ab0] = "\x65\x50",
  [0x2ab3] = "\x69\x79", [0x2aba] = "\x6d\x7d", [0x2abc] = "\x6e\x23",
  [0x2abe] = "\x6e\x21", [0x2ac1] = "\x69\x76", [0x2ac5] = "\x6d\x7c",
  [0x2aca] = "\x6d\x7e", [0x2acb] = "\x6d\x7b", [0x2ad0] = "\x6e\x24",
  [0x2ad5] = "\x6e\x22", [0x2ada] = "\x71\x54", [0x2ae7] = "\x71\x56",
  [0x2ae8] = "\x71\x53", [0x2aec] = "\x71\x55", [0x2aef] = "\x74\x50",
  [0x2af4] = "\x74\x4f", [0x2af7] = "\x74\x4d", [0x2afa] = "\x74\x4e",
  [0x2b01] = "\x74\x51", [0x2b0e] = "\x76\x70", [0x2b19] = "\x78\x55",
  [0x2b26] = "\x79\x7c", [0x2b2a] = "\x79\x7d", [0x2b2b] = "\x7a\x21",
  [0x2b2c] = "\x79\x7e", [0x2b3a] = "\x46\x7c", [0x2b3e] = "\x4f\x23",
  [0x2b40] = "\x4f\x24", [0x2b41] = "\x4f\x25", [0x2b46] = "\x52\x69",
  [0x2b47] = "\x52\x6c", [0x2b48] = "\x52\x6b", [0x2b49] = "\x52\x6a",
  [0x2b50] = "\x57\x2a", [0x2b55] = "\x57\x29", [0x2b56] = "\x57\x2d",
  [0x2b57] = "\x57\x30", [0x2b5a] = "\x57\x31", [0x2b5d] = "\x57\x2f",
  [0x2b5e] = "\x57\x2e", [0x2b5f] = "\x57\x2c", [0x2b60] = "\x57\x2b",
  [0x2b65] = "\x5c\x40", [0x2b68] = "\x5c\x41", [0x2b6d] = "\x5c\x42",
  [0x2b7a] = "\x65\x59", [0x2b7f] = "\x65\x5a", [0x2b81] = "\x65\x5b",
  [0x2b8d] = "\x69\x7e", [0x2b8e] = "\x69\x7c", [0x2b8f] = "\x69\x7d",
  [0x2ba6] = "\x71\x57", [0x2ba7] = "\x74\x52", [0x2baa] = "\x74\x53",
  [0x2bae] = "\x76\x71", [0x2bb1] = "\x78\x56", [0x2bb3] = "\x7b\x6b",
  [0x2bb9] = "\x52\x6d", [0x2bba] = "\x52\x6e", [0x2bbd] = "\x65\x5d",
  [0x2bbe] = "\x46\x7d", [0x2bbf] = "\x4b\x35", [0x2bc0] = "\x4b\x34",
  [0x2bc1] = "\x4b\x33", [0x2bc8] = "\x4f\x27", [0x2bc9] = "\x4f\x26",
  [0x2bcb] = "\x52\x71", [0x2bd1] = "\x52\x6f", [0x2bd2] = "\x52\x70",
  [0x2bd8] = "\x57\x38", [0x2bdf] = "\x57\x35", [0x2be3] = "\x57\x33",
  [0x2be4] = "\x57\x32", [0x2be6] = "\x57\x36", [0x2be7] = "\x57\x34",
  [0x2be9] = "\x57\x37", [0x2bfb] = "\x5c\x43", [0x2c00] = "\x61\x25",
  [0x2c05] = "\x61\x24", [0x2c08] = "\x61\x22", [0x2c0b] = "\x61\x23",
  [0x2c0d] = "\x61\x21", [0x2c14] = "\x65\x61", [0x2c1a] = "\x65\x5f",
  [0x2c1c] = "\x65\x5e", [0x2c1e] = "\x65\x63", [0x2c1f] = "\x65\x62",
  [0x2c20] = "\x65\x60", [0x2c2e] = "\x6a\x21", [0x2c31] = "\x6a\x22",
  [0x2c37] = "\x6e\x29", [0x2c3b] = "\x6e\x2a", [0x2c3c] = "\x6e\x26",
  [0x2c3d] = "\x6e\x28", [0x2c3f] = "\x6e\x25", [0x2c40] = "\x6e\x27",
  [0x2c46] = "\x71\x5a", [0x2c4b] = "\x71\x5c", [0x2c4c] = "\x71\x5b",
  [0x2c4d] = "\x71\x58", [0x2c4e] = "\x71\x59", [0x2c57] = "\x74\x54",
  [0x2c60] = "\x76\x74", [0x2c61] = "\x76\x72", [0x2c62] = "\x76\x73",
  [0x2c69] = "\x78\x58", [0x2c6b] = "\x78\x57", [0x2c74] = "\x46\x7e",
  [0x2c76] = "\x4b\x36", [0x2c79] = "\x4f\x29", [0x2c7a] = "\x4f\x28",
  [0x2c7f] = "\x52\x72", [0x2c81] = "\x52\x73", [0x2c84] = "\x57\x39",
  [0x2c88] = "\x57\x3a", [0x2c92] = "\x5c\x44", [0x2c95] = "\x5c\x45",
  [0x2c96] = "\x61\x28", [0x2c97] = "\x61\x27", [0x2c98] = "\x61\x26",
  [0x2c9f] = "\x65\x64", [0x2ca0] = "\x65\x65", [0x2ca9] = "\x6a\x24",
  [0x2caa] = "\x6a\x23", [0x2cae] = "\x6e\x2c", [0x2caf] = "\x6e\x2b",
  [0x2cba] = "\x71\x5d", [0x2cbf] = "\x74\x55", [0x2cc4] = "\x76\x75",
  [0x2cc5] = "\x76\x76", [0x2cc7] = "\x7a\x22", [0x2cca] = "\x7c\x3d",
  [0x2ccb] = "\x47\x21", [0x2cd9] = "\x57\x3b", [0x2cdf] = "\x5d\x7e",
  [0x2ce0] = "\x5d\x7d", [0x2ce3] = "\x61\x2a", [0x2ce5] = "\x61\x29",
  [0x2ced] = "\x6a\x25", [0x2cef] = "\x6a\x26", [0x2cf6] = "\x7a\x23",
  [0x2cf9] = "\x48\x4c", [0x2cfa] = "\x4f\x2a", [0x2cfd] = "\x52\x75",
  [0x2cff] = "\x52\x74", [0x2d06] = "\x57\x3c", [0x2d11] = "\x57\x3d",
  [0x2d19] = "\x5c\x4b", [0x2d1b] = "\x5c\x48", [0x2d1e] = "\x5c\x4c",
  [0x2d20] = "\x5c\x46", [0x2d26] = "\x5c\x4a", [0x2d28] = "\x5c\x47",
  [0x2d2c] = "\x5c\x49", [0x2d2e] = "\x5c\x4d", [0x2d46] = "\x61\x2d",
  [0x2d49] = "\x61\x2b", [0x2d4b] = "\x61\x32", [0x2d4d] = "\x61\x31",
  [0x2d4f] = "\x61\x33", [0x2d50] = "\x61\x2e", [0x2d51] = "\x61\x34",
  [0x2d52] = "\x61\x2f", [0x2d54] = "\x61\x30", [0x2d56] = "\x61\x2c",
  [0x2d60] = "\x65\x68", [0x2d67] = "\x65\x6a", [0x2d6e] = "\x65\x69",
  [0x2d75] = "\x6a\x2a", [0x2d77] = "\x65\x66", [0x2d84] = "\x6a\x31",
  [0x2d87] = "\x6a\x30", [0x2d8b] = "\x6a\x29", [0x2d8f] = "\x6a\x2e",
  [0x2d94] = "\x6a\x2d", [0x2d95] = "\x6a\x28", [0x2d97] = "\x6a\x2b",
  [0x2d9d] = "\x6a\x2c", [0x2da0] = "\x6e\x34", [0x2da1] = "\x6a\x27",
  [0x2dad] = "\x6e\x2d", [0x2db1] = "\x6e\x2e", [0x2db4] = "\x6e\x30",
  [0x2db8] = "\x6a\x2f", [0x2dc0] = "\x65\x67", [0x2dc1] = "\x6e\x33",
  [0x2dc4] = "\x6e\x2f", [0x2dc6] = "\x6e\x31", [0x2dc7] = "\x6e\x32",
  [0x2dc9] = "\x71\x60", [0x2dcc] = "\x6e\x35", [0x2dd9] = "\x71\x5e",
  [0x2ddb] = "\x71\x62", [0x2de0] = "\x74\x5b", [0x2de1] = "\x71\x63",
  [0x2de4] = "\x71\x61", [0x2de6] = "\x71\x65", [0x2de9] = "\x71\x64",
  [0x2df7] = "\x74\x59", [0x2dfe] = "\x74\x58", [0x2e07] = "\x74\x56",
  [0x2e0c] = "\x74\x5a", [0x2e0d] = "\x74\x57", [0x2e11] = "\x71\x5f",
  [0x2e1e] = "\x76\x7a", [0x2e21] = "\x76\x7c", [0x2e23] = "\x76\x7b",
  [0x2e27] = "\x76\x78", [0x2e2a] = "\x76\x79", [0x2e2b] = "\x76\x77",
  [0x2e37] = "\x78\x5d", [0x2e38] = "\x78\x5b", [0x2e3d] = "\x78\x5c",
  [0x2e3e] = "\x78\x59", [0x2e3f] = "\x78\x5a", [0x2e40] = "\x78\x5e",
  [0x2e43] = "\x7a\x25", [0x2e4c] = "\x7a\x24", [0x2e4d] = "\x7a\x26",
  [0x2e50] = "\x7a\x7b", [0x2e5f] = "\x7b\x6d", [0x2e60] = "\x7b\x6c",
  [0x2e63] = "\x7c\x3f", [0x2e64] = "\x7c\x3e", [0x2e65] = "\x7c\x40",
  [0x2e6c] = "\x7d\x25", [0x2e6e] = "\x7d\x26", [0x2e72] = "\x7d\x4b",
  [0x2e73] = "\x48\x4d", [0x2e7d] = "\x52\x76", [0x2e89] = "\x57\x3e",
  [0x2e92] = "\x5c\x4e", [0x2e95] = "\x5c\x50", [0x2e97] = "\x5c\x4f",
  [0x2e9f] = "\x61\x35", [0x2ea5] = "\x61\x36", [0x2eb1] = "\x65\x6b",
  [0x2eb3] = "\x65\x6c", [0x2eb5] = "\x65\x6d", [0x2eb9] = "\x6a\x32",
  [0x2ebd] = "\x6a\x33", [0x2ebe] = "\x6a\x34", [0x2eca] = "\x6e\x36",
  [0x2ece] = "\x23\x23", [0x2ed5] = "\x71\x66", [0x2ed6] = "\x71\x67",
  [0x2ed9] = "\x74\x61", [0x2edc] = "\x74\x5d", [0x2edd] = "\x74\x62",
  [0x2ede] = "\x74\x5e", [0x2edf] = "\x74\x60", [0x2ee0] = "\x74\x5c",
  [0x2ee2] = "\x74\x5f", [0x2ee7] = "\x76\x7d", [0x2eef] = "\x7a\x27",
  [0x2ef0] = "\x7a\x28", [0x2ef8] = "\x48\x4e", [0x2efb] = "\x4b\x37",
  [0x2efe] = "\x4f\x2b", [0x2f00] = "\x52\x79", [0x2f02] = "\x52\x77",
  [0x2f04] = "\x52\x7c", [0x2f05] = "\x52\x78", [0x2f06] = "\x52\x7d",
  [0x2f07] = "\x52\x7b", [0x2f09] = "\x52\x7a", [0x2f0a] = "\x57\x42",
  [0x2f0b] = "\x57\x41", [0x2f0d] = "\x57\x4a", [0x2f10] = "\x57\x46",
  [0x2f14] = "\x57\x45", [0x2f15] = "\x57\x47", [0x2f17] = "\x57\x40",
  [0x2f19] = "\x57\x4b", [0x2f1a] = "\x57\x48", [0x2f1b] = "\x57\x4c",
  [0x2f1c] = "\x57\x49", [0x2f20] = "\x57\x43", [0x2f21] = "\x57\x3f",
  [0x2f22] = "\x57\x44", [0x2f2b] = "\x61\x3b", [0x2f2e] = "\x5c\x54",
  [0x2f2f] = "\x5c\x5b", [0x2f30] = "\x5c\x58", [0x2f31] = "\x5c\x5e",
  [0x2f32] = "\x5c\x5d", [0x2f33] = "\x5c\x59", [0x2f39] = "\x5c\x55",
  [0x2f3c] = "\x5c\x56", [0x2f40] = "\x5c\x57", [0x2f42] = "\x5c\x5c",
  [0x2f43] = "\x5c\x52", [0x2f44] = "\x5c\x5a", [0x2f46] = "\x5c\x51",
  [0x2f50] = "\x61\x38", [0x2f55] = "\x61\x3a", [0x2f5b] = "\x65\x73",
  [0x2f5e] = "\x61\x37", [0x2f61] = "\x61\x3e", [0x2f62] = "\x61\x40",
  [0x2f66] = "\x61\x3f", [0x2f68] = "\x61\x39", [0x2f6e] = "\x61\x3c",
  [0x2f70] = "\x61\x41", [0x2f71] = "\x5c\x53", [0x2f72] = "\x61\x3d",
  [0x2f73] = "\x61\x42", [0x2f79] = "\x65\x6f", [0x2f81] = "\x65\x71",
  [0x2f8f] = "\x65\x72", [0x2f91] = "\x65\x70", [0x2f93] = "\x65\x6e",
  [0x2f9c] = "\x6a\x37", [0x2f9e] = "\x6e\x43", [0x2fa0] = "\x6a\x3a",
  [0x2fa2] = "\x6a\x40", [0x2fac] = "\x6a\x47", [0x2fad] = "\x6a\x44",
  [0x2fb0] = "\x6a\x36", [0x2fb1] = "\x6a\x3e", [0x2fb2] = "\x6a\x3d",
  [0x2fb4] = "\x6a\x3c", [0x2fb5] = "\x6a\x42", [0x2fb8] = "\x6a\x43",
  [0x2fba] = "\x6a\x3f", [0x2fbb] = "\x6a\x35", [0x2fbd] = "\x6a\x38",
  [0x2fbe] = "\x6a\x39", [0x2fbf] = "\x6a\x41", [0x2fc7] = "\x6a\x46",
  [0x2fca] = "\x6a\x3b", [0x2fd2] = "\x6a\x45", [0x2fd8] = "\x6e\x3b",
  [0x2fd9] = "\x6e\x44", [0x2fda] = "\x6e\x40", [0x2fdd] = "\x6e\x3d",
  [0x2fde] = "\x6e\x41", [0x2fe0] = "\x6e\x37", [0x2fe3] = "\x6e\x3f",
  [0x2fe8] = "\x6e\x3e", [0x2fe9] = "\x6e\x42", [0x2fec] = "\x6e\x3c",
  [0x2fef] = "\x6e\x39", [0x2ff2] = "\x6e\x45", [0x2ff4] = "\x6e\x38",
  [0x2ff9] = "\x6e\x46", [0x2ffb] = "\x6e\x3a", [0x3008] = "\x71\x6a",
  [0x3009] = "\x71\x6f", [0x300a] = "\x71\x68", [0x3010] = "\x71\x70",
  [0x3011] = "\x71\x69", [0x301b] = "\x71\x6b", [0x301d] = "\x71\x6e",
  [0x301e] = "\x71\x6d", [0x3023] = "\x71\x6c", [0x302b] = "\x74\x69",
  [0x302e] = "\x74\x63", [0x302f] = "\x74\x73", [0x3031] = "\x74\x6b",
  [0x3032] = "\x74\x67", [0x3034] = "\x74\x6e", [0x3035] = "\x74\x71",
  [0x3037] = "\x74\x66", [0x3039] = "\x74\x6f", [0x303d] = "\x74\x6a",
  [0x303e] = "\x74\x64", [0x303f] = "\x74\x72", [0x3041] = "\x74\x6d",
  [0x3043] = "\x74\x68", [0x3045] = "\x74\x6c", [0x3046] = "\x74\x65",
  [0x3048] = "\x74\x70", [0x3052] = "\x77\x25", [0x3054] = "\x76\x7e",
  [0x3055] = "\x77\x21", [0x3059] = "\x77\x26", [0x305a] = "\x77\x23",
  [0x305e] = "\x77\x22", [0x3061] = "\x77\x24", [0x3069] = "\x78\x62",
  [0x306a] = "\x78\x63", [0x306b] = "\x78\x5f", [0x306d] = "\x78\x60",
  [0x3073] = "\x78\x64", [0x3079] = "\x78\x61", [0x307c] = "\x7a\x2b",
  [0x307d] = "\x7a\x2a", [0x3082] = "\x7a\x2c", [0x308c] = "\x7a\x7d",
  [0x308f] = "\x7a\x7c", [0x3093] = "\x7c\x41", [0x3094] = "\x7c\x43",
  [0x3096] = "\x7c\x42", [0x309c] = "\x7d\x39", [0x3136] = "\x48\x4f",
  [0x3138] = "\x52\x7e", [0x313a] = "\x57\x4d", [0x313d] = "\x5c\x5f",
  [0x3144] = "\x74\x74", [0x3148] = "\x77\x27", [0x314c] = "\x7a\x2d",
  [0x3150] = "\x7c\x64", [0x3154] = "\x4f\x2c", [0x3155] = "\x4b\x38",
  [0x315f] = "\x57\x4e", [0x3169] = "\x65\x75", [0x316a] = "\x65\x76",
  [0x316e] = "\x65\x74", [0x3170] = "\x6a\x48", [0x3172] = "\x65\x77",
  [0x3175] = "\x6e\x47", [0x3177] = "\x6e\x48", [0x3179] = "\x71\x71",
  [0x3185] = "\x78\x65", [0x3188] = "\x7c\x65", [0x318a] = "\x48\x50",
  [0x318b] = "\x4f\x2e", [0x318c] = "\x4f\x2d", [0x318e] = "\x53\x21",
  [0x3194] = "\x57\x4f", [0x319a] = "\x5c\x61", [0x319e] = "\x5c\x60",
  [0x31a4] = "\x65\x7a", [0x31a8] = "\x65\x79", [0x31a9] = "\x65\x78",
  [0x31af] = "\x6e\x49", [0x31b2] = "\x71\x72", [0x31b6] = "\x78\x66",
  [0x31b8] = "\x78\x68", [0x31b9] = "\x78\x67", [0x31bc] = "\x7a\x7e",
  [0x31bd] = "\x48\x51", [0x31bf] = "\x53\x22", [0x31c1] = "\x57\x51",
  [0x31c5] = "\x57\x50", [0x31cc] = "\x5c\x62", [0x31ce] = "\x5c\x63",
  [0x31d2] = "\x5c\x64", [0x31d4] = "\x61\x44", [0x31d5] = "\x61\x45",
  [0x31df] = "\x6a\x4b", [0x31e0] = "\x6a\x49", [0x31e1] = "\x6a\x4a",
  [0x31e9] = "\x6e\x4a", [0x31ee] = "\x71\x75", [0x31f0] = "\x71\x73",
  [0x31f1] = "\x71\x74", [0x31f3] = "\x74\x75", [0x31f9] = "\x77\x28",
  [0x31fb] = "\x77\x29", [0x31fc] = "\x74\x76", [0x3200] = "\x7a\x2e",
  [0x3201] = "\x48\x52", [0x3203] = "\x48\x53", [0x3204] = "\x57\x53",
  [0x3205] = "\x4f\x2f", [0x3206] = "\x57\x52", [0x320b] = "\x61\x46",
  [0x320c] = "\x48\x54", [0x320d] = "\x53\x24", [0x3210] = "\x53\x23",
  [0x3211] = "\x53\x25", [0x3212] = "\x48\x55", [0x3215] = "\x57\x55",
  [0x3217] = "\x57\x57", [0x3218] = "\x57\x54", [0x3219] = "\x57\x56",
  [0x321c] = "\x5c\x65", [0x3226] = "\x6e\x4b", [0x3228] = "\x71\x76",
  [0x3233] = "\x48\x56", [0x3236] = "\x53\x26", [0x323d] = "\x57\x58",
  [0x323f] = "\x57\x59", [0x3246] = "\x5c\x67", [0x324a] = "\x5c\x66",
  [0x3252] = "\x61\x47", [0x3256] = "\x65\x7b", [0x3258] = "\x65\x7c",
  [0x325a] = "\x6a\x4d", [0x325e] = "\x6a\x4c", [0x326f] = "\x74\x7a",
  [0x3270] = "\x74\x79", [0x3271] = "\x74\x77", [0x3272] = "\x74\x78",
  [0x3273] = "\x74\x7b", [0x3276] = "\x77\x2b", [0x3277] = "\x77\x2a",
  [0x327d] = "\x7b\x6f", [0x327e] = "\x7b\x6e", [0x327f] = "\x48\x57",
  [0x3284] = "\x65\x7e", [0x3285] = "\x61\x48", [0x3286] = "\x65\x7d",
  [0x3287] = "\x6a\x4e", [0x3289] = "\x48\x58", [0x328b] = "\x48\x59",
  [0x328c] = "\x48\x5a", [0x3293] = "\x4b\x3a", [0x3296] = "\x4b\x39",
  [0x3298] = "\x4b\x3c", [0x329a] = "\x4b\x3e", [0x329b] = "\x4b\x3d",
  [0x329d] = "\x4b\x3b", [0x32a1] = "\x4f\x34", [0x32a2] = "\x4f\x32",
  [0x32a5] = "\x4f\x31", [0x32a9] = "\x4f\x36", [0x32aa] = "\x4f\x38",
  [0x32ab] = "\x4f\x35", [0x32af] = "\x4f\x39", [0x32b1] = "\x4f\x33",
  [0x32b2] = "\x4b\x3f", [0x32b4] = "\x4f\x37", [0x32ba] = "\x4f\x30",
  [0x32c3] = "\x53\x2a", [0x32c4] = "\x53\x2b", [0x32cc] = "\x53\x2c",
  [0x32ce] = "\x53\x2f", [0x32d6] = "\x53\x27", [0x32da] = "\x53\x29",
  [0x32db] = "\x53\x2e", [0x32dd] = "\x53\x32", [0x32de] = "\x53\x30",
  [0x32e1] = "\x53\x2d", [0x32e4] = "\x53\x31", [0x32e5] = "\x53\x28",
  [0x32ed] = "\x57\x5e", [0x32ef] = "\x57\x67", [0x32f0] = "\x57\x5c",
  [0x32f1] = "\x57\x5a", [0x32f3] = "\x57\x62", [0x32f4] = "\x57\x5f",
  [0x32f8] = "\x57\x61", [0x32fc] = "\x57\x66", [0x32fd] = "\x57\x64",
  [0x3302] = "\x57\x5b", [0x3305] = "\x57\x5d", [0x3306] = "\x57\x60",
  [0x3308] = "\x57\x63", [0x330a] = "\x57\x65", [0x3316] = "\x5c\x69",
  [0x3323] = "\x5c\x6a", [0x3324] = "\x5c\x6e", [0x3329] = "\x5c\x6c",
  [0x332b] = "\x5c\x6b", [0x332f] = "\x5c\x68", [0x3330] = "\x5c\x6d",
  [0x3339] = "\x61\x4e", [0x333e] = "\x61\x50", [0x3346] = "\x61\x4f",
  [0x334b] = "\x61\x4b", [0x334c] = "\x61\x51", [0x334e] = "\x61\x4d",
  [0x3350] = "\x6a\x4f", [0x3351] = "\x61\x4c", [0x3353] = "\x61\x52",
  [0x3354] = "\x61\x4a", [0x3355] = "\x61\x49", [0x3365] = "\x66\x24",
  [0x3366] = "\x66\x2a", [0x336b] = "\x66\x27", [0x336e] = "\x66\x25",
  [0x3370] = "\x66\x22", [0x3371] = "\x66\x21", [0x3373] = "\x66\x26",
  [0x3374] = "\x61\x53", [0x3378] = "\x66\x23", [0x3379] = "\x66\x28",
  [0x337a] = "\x66\x29", [0x337f] = "\x6a\x54", [0x3380] = "\x6a\x50",
  [0x3382] = "\x6a\x55", [0x3388] = "\x6a\x52", [0x338a] = "\x6a\x53",
  [0x338f] = "\x6a\x51", [0x3398] = "\x6e\x51", [0x339a] = "\x6e\x50",
  [0x339b] = "\x6e\x4c", [0x339c] = "\x6e\x4d", [0x339d] = "\x6e\x4e",
  [0x33a0] = "\x6e\x4f", [0x33a8] = "\x71\x79", [0x33a9] = "\x71\x78",
  [0x33b3] = "\x71\x77", [0x33ba] = "\x74\x7e", [0x33bd] = "\x75\x24",
  [0x33be] = "\x75\x26", [0x33bf] = "\x75\x23", [0x33c0] = "\x75\x22",
  [0x33c2] = "\x75\x21", [0x33c3] = "\x74\x7d", [0x33c6] = "\x74\x7c",
  [0x33c9] = "\x75\x25", [0x33cd] = "\x77\x2c", [0x33cf] = "\x77\x2d",
  [0x33d8] = "\x78\x69", [0x33da] = "\x7a\x2f", [0x33df] = "\x7b\x70",
  [0x33e2] = "\x7c\x44", [0x33e3] = "\x48\x5b", [0x33e5] = "\x4f\x3a",
  [0x33e7] = "\x6a\x56", [0x33e8] = "\x75\x27", [0x33ea] = "\x48\x5c",
  [0x33ec] = "\x57\x69", [0x33ed] = "\x57\x68", [0x33f3] = "\x48\x5d",
  [0x33f4] = "\x53\x33", [0x33fa] = "\x6a\x57", [0x33fb] = "\x71\x7a",
  [0x33fc] = "\x48\x5e", [0x33fe] = "\x4f\x3b", [0x3400] = "\x57\x6a",
  [0x3402] = "\x5c\x6f", [0x3405] = "\x66\x2b", [0x3407] = "\x6a\x58",
  [0x3408] = "\x71\x7b", [0x3409] = "\x75\x28", [0x340a] = "\x77\x2e",
  [0x340c] = "\x48\x5f", [0x340d] = "\x4f\x3c", [0x3410] = "\x57\x6b",
  [0x3412] = "\x61\x54", [0x3414] = "\x6a\x59", [0x341b] = "\x48\x60",
  [0x341c] = "\x61\x55", [0x341e] = "\x6a\x5a", [0x341f] = "\x48\x61",
  [0x3422] = "\x53\x34", [0x3428] = "\x57\x6e", [0x342a] = "\x57\x6c",
  [0x342b] = "\x57\x6d", [0x342c] = "\x57\x6f", [0x3435] = "\x5c\x70",
  [0x3436] = "\x5c\x72", [0x3437] = "\x5c\x71", [0x3439] = "\x5c\x73",
  [0x3447] = "\x66\x2c", [0x344b] = "\x6a\x5b", [0x3458] = "\x71\x7c",
  [0x3459] = "\x71\x7d", [0x3466] = "\x7a\x30", [0x346e] = "\x48\x62",
  [0x346f] = "\x4b\x40", [0x3471] = "\x75\x29", [0x3472] = "\x48\x63",
  [0x3477] = "\x7c\x6c", [0x347e] = "\x48\x64", [0x348b] = "\x4b\x42",
  [0x348d] = "\x4b\x43", [0x3492] = "\x4b\x41", [0x3499] = "\x4f\x3f",
  [0x349d] = "\x4f\x3e", [0x349f] = "\x4f\x42", [0x34a3] = "\x4f\x49",
  [0x34a5] = "\x4f\x46", [0x34ac] = "\x4f\x45", [0x34ad] = "\x4f\x40",
  [0x34af] = "\x4f\x47", [0x34b0] = "\x4f\x4a", [0x34b1] = "\x4f\x44",
  [0x34b3] = "\x4f\x3d", [0x34b7] = "\x4f\x4c", [0x34b8] = "\x4f\x48",
  [0x34b9] = "\x4f\x43", [0x34bb] = "\x57\x70", [0x34bd] = "\x4f\x41",
  [0x34be] = "\x4f\x4b", [0x34d1] = "\x53\x45", [0x34d2] = "\x53\x3f",
  [0x34d3] = "\x53\x47", [0x34d4] = "\x53\x44", [0x34d7] = "\x53\x40",
  [0x34db] = "\x53\x39", [0x34dc] = "\x53\x43", [0x34de] = "\x53\x46",
  [0x34df] = "\x53\x48", [0x34e3] = "\x53\x38", [0x34e5] = "\x53\x3c",
  [0x34e6] = "\x53\x3a", [0x34e7] = "\x53\x35", [0x34ef] = "\x53\x49",
  [0x34f1] = "\x53\x41", [0x3501] = "\x53\x42", [0x3502] = "\x53\x3d",
  [0x3503] = "\x53\x36", [0x3504] = "\x53\x3b", [0x3505] = "\x53\x37",
  [0x3506] = "\x53\x4a", [0x3509] = "\x53\x3e", [0x3517] = "\x57\x7e",
  [0x3528] = "\x58\x23", [0x352b] = "\x57\x71", [0x3531] = "\x58\x22",
  [0x3532] = "\x57\x7b", [0x3534] = "\x57\x79", [0x3535] = "\x57\x78",
  [0x3536] = "\x57\x7d", [0x3538] = "\x57\x75", [0x3539] = "\x57\x7c",
  [0x3540] = "\x58\x21", [0x3543] = "\x58\x24", [0x3549] = "\x57\x77",
  [0x354a] = "\x57\x74", [0x354f] = "\x57\x7a", [0x3550] = "\x57\x76",
  [0x3552] = "\x57\x72", [0x3554] = "\x57\x73", [0x3577] = "\x5d\x23",
  [0x3578] = "\x5c\x77", [0x357b] = "\x5d\x24", [0x357c] = "\x5d\x25",
  [0x3586] = "\x5d\x26", [0x3589] = "\x5d\x21", [0x358a] = "\x5c\x7d",
  [0x358e] = "\x5c\x74", [0x3592] = "\x5c\x7c", [0x3593] = "\x5c\x7e",
  [0x3596] = "\x5c\x79", [0x3598] = "\x5c\x76", [0x359e] = "\x5c\x75",
  [0x35a0] = "\x5d\x22", [0x35a2] = "\x5c\x78", [0x35a7] = "\x5d\x27",
  [0x35ab] = "\x5c\x7b", [0x35bd] = "\x5c\x7a", [0x35c1] = "\x61\x5d",
  [0x35c5] = "\x61\x5b", [0x35ca] = "\x61\x68", [0x35cc] = "\x61\x65",
  [0x35d4] = "\x61\x6e", [0x35dc] = "\x61\x6c", [0x35df] = "\x61\x6f",
  [0x35e0] = "\x61\x5a", [0x35e9] = "\x61\x56", [0x35ef] = "\x61\x5e",
  [0x35f0] = "\x61\x63", [0x35f1] = "\x61\x5f", [0x35f2] = "\x61\x67",
  [0x35f4] = "\x61\x60", [0x35f8] = "\x61\x58", [0x35fd] = "\x61\x66",
  [0x3603] = "\x61\x57", [0x3604] = "\x61\x6b", [0x3607] = "\x61\x6d",
  [0x360a] = "\x61\x62", [0x360b] = "\x61\x5c", [0x360c] = "\x61\x64",
  [0x360d] = "\x61\x59", [0x360e] = "\x61\x6a", [0x362c] = "\x65\x5c",
  [0x3631] = "\x66\x30", [0x3635] = "\x66\x38", [0x3638] = "\x61\x69",
  [0x363c] = "\x66\x37", [0x363d] = "\x66\x2f", [0x3646] = "\x66\x3d",
  [0x3649] = "\x66\x34", [0x3657] = "\x61\x61", [0x365b] = "\x66\x36",
  [0x3661] = "\x66\x39", [0x3663] = "\x66\x3a", [0x3666] = "\x66\x32",
  [0x3669] = "\x66\x3b", [0x366b] = "\x66\x33", [0x366c] = "\x66\x35",
  [0x366d] = "\x66\x3c", [0x3675] = "\x66\x31", [0x3677] = "\x66\x2e",
  [0x3682] = "\x66\x2d", [0x3690] = "\x6a\x68", [0x3699] = "\x6a\x60",
  [0x369c] = "\x6a\x63", [0x369e] = "\x6a\x61", [0x36b2] = "\x6a\x62",
  [0x36b8] = "\x6a\x65", [0x36bc] = "\x6a\x69", [0x36bf] = "\x6a\x5d",
  [0x36c0] = "\x6a\x66", [0x36c4] = "\x6a\x5f", [0x36c6] = "\x6a\x5e",
  [0x36c9] = "\x6a\x5c", [0x36ca] = "\x6a\x6b", [0x36cb] = "\x6a\x64",
  [0x36d1] = "\x6a\x6a", [0x36d3] = "\x6a\x67", [0x36ec] = "\x6e\x5d",
  [0x36ee] = "\x6e\x55", [0x36ff] = "\x6e\x5f", [0x3706] = "\x6e\x60",
  [0x3711] = "\x6e\x59", [0x3713] = "\x6e\x58", [0x3714] = "\x6e\x5c",
  [0x3717] = "\x6e\x52", [0x371a] = "\x6e\x54", [0x3721] = "\x6e\x5b",
  [0x3723] = "\x6e\x5a", [0x3725] = "\x6e\x5e", [0x372c] = "\x6e\x56",
  [0x372d] = "\x6e\x57", [0x373d] = "\x6e\x53", [0x3743] = "\x72\x25",
  [0x3748] = "\x72\x22", [0x3749] = "\x72\x26", [0x374a] = "\x71\x7e",
  [0x3759] = "\x72\x21", [0x375e] = "\x72\x29", [0x3768] = "\x72\x23",
  [0x3769] = "\x72\x24", [0x376a] = "\x72\x28", [0x376d] = "\x72\x27",
  [0x377e] = "\x75\x2c", [0x3784] = "\x75\x2b", [0x3787] = "\x75\x32",
  [0x378a] = "\x75\x34", [0x3791] = "\x75\x2e", [0x3794] = "\x75\x2f",
  [0x379b] = "\x75\x31", [0x379c] = "\x75\x2d", [0x37a6] = "\x75\x35",
  [0x37a8] = "\x75\x33", [0x37a9] = "\x77\x30", [0x37aa] = "\x75\x2a",
  [0x37af] = "\x75\x30", [0x37b0] = "\x77\x34", [0x37b9] = "\x77\x36",
  [0x37ba] = "\x77\x35", [0x37c9] = "\x77\x33", [0x37cd] = "\x77\x31",
  [0x37cf] = "\x77\x2f", [0x37d0] = "\x77\x32", [0x37d5] = "\x78\x6d",
  [0x37dd] = "\x78\x6b", [0x37e4] = "\x78\x6e", [0x37e5] = "\x78\x6f",
  [0x37e9] = "\x78\x6a", [0x37ea] = "\x78\x6c", [0x37f7] = "\x78\x70",
  [0x37f9] = "\x7a\x32", [0x37fa] = "\x7a\x34", [0x37fb] = "\x7a\x31",
  [0x3806] = "\x7a\x35", [0x3807] = "\x7a\x37", [0x380a] = "\x7a\x38",
  [0x380b] = "\x7a\x36", [0x3811] = "\x7a\x33", [0x3817] = "\x7b\x21",
  [0x381a] = "\x7b\x23", [0x382d] = "\x7b\x22", [0x3838] = "\x7c\x45",
  [0x383f] = "\x7c\x46", [0x384e] = "\x4f\x4d", [0x3850] = "\x53\x4b",
  [0x3854] = "\x58\x25", [0x3855] = "\x5d\x28", [0x385b] = "\x61\x70",
  [0x385c] = "\x66\x3f", [0x385e] = "\x66\x3e", [0x385f] = "\x66\x40",
  [0x3867] = "\x75\x36", [0x386b] = "\x48\x65", [0x3871] = "\x4f\x4e",
  [0x3879] = "\x53\x4c", [0x387a] = "\x53\x4e", [0x387b] = "\x53\x4d",
  [0x388a] = "\x58\x26", [0x388c] = "\x58\x2b", [0x3893] = "\x58\x28",
  [0x389c] = "\x58\x2d", [0x38a3] = "\x58\x2c", [0x38a4] = "\x58\x29",
  [0x38a9] = "\x58\x2a", [0x38aa] = "\x58\x27", [0x38af] = "\x5d\x32",
  [0x38b1] = "\x5d\x31", [0x38b5] = "\x5d\x2e", [0x38b6] = "\x5d\x2c",
  [0x38c0] = "\x5d\x2b", [0x38c4] = "\x5d\x2d", [0x38c6] = "\x5d\x2f",
  [0x38c7] = "\x5d\x2a", [0x38c9] = "\x5d\x33", [0x38cb] = "\x5d\x30",
  [0x38d0] = "\x61\x77", [0x38d4] = "\x61\x74", [0x38d9] = "\x61\x72",
  [0x38db] = "\x61\x75", [0x38de] = "\x61\x78", [0x38df] = "\x61\x71",
  [0x38e4] = "\x61\x76", [0x38ed] = "\x61\x73", [0x38f9] = "\x66\x41",
  [0x38fb] = "\x66\x47", [0x38fe] = "\x66\x46", [0x3900] = "\x66\x45",
  [0x3902] = "\x66\x48", [0x3903] = "\x66\x49", [0x3906] = "\x66\x4a",
  [0x3907] = "\x66\x44", [0x3908] = "\x66\x43", [0x390a] = "\x66\x4b",
  [0x3913] = "\x66\x42", [0x3918] = "\x6a\x72", [0x391c] = "\x6a\x6d",
  [0x3922] = "\x6a\x6f", [0x3925] = "\x6a\x70", [0x3929] = "\x6a\x75",
  [0x3934] = "\x6a\x71", [0x3937] = "\x6a\x74", [0x393b] = "\x6a\x6e",
  [0x393f] = "\x6a\x6c", [0x394c] = "\x6e\x6a", [0x3953] = "\x6e\x6b",
  [0x3955] = "\x6a\x73", [0x3957] = "\x6e\x69", [0x3959] = "\x6e\x68",
  [0x3960] = "\x6e\x64", [0x3966] = "\x6e\x65", [0x3968] = "\x6e\x67",
  [0x3974] = "\x6e\x62", [0x3976] = "\x6e\x63", [0x3978] = "\x6e\x66",
  [0x3982] = "\x6e\x61", [0x3983] = "\x72\x2a", [0x398d] = "\x72\x2e",
  [0x399e] = "\x72\x2c", [0x399f] = "\x72\x2b", [0x39a2] = "\x72\x2d",
  [0x39ab] = "\x75\x3c", [0x39b3] = "\x75\x39", [0x39ba] = "\x75\x3e",
  [0x39bb] = "\x75\x3d", [0x39c0] = "\x75\x37", [0x39c6] = "\x75\x3b",
  [0x39c8] = "\x75\x3f", [0x39cb] = "\x75\x40", [0x39d1] = "\x75\x38",
  [0x39d2] = "\x75\x3a", [0x39e0] = "\x77\x3a", [0x39ec] = "\x77\x38",
  [0x39ef] = "\x77\x37", [0x39f2] = "\x77\x39", [0x39f9] = "\x78\x74",
  [0x39fb] = "\x78\x71", [0x39fe] = "\x78\x75", [0x3a05] = "\x78\x72",
  [0x3a0d] = "\x78\x73", [0x3a14] = "\x7a\x39", [0x3a15] = "\x7a\x3a",
  [0x3a1f] = "\x7b\x27", [0x3a21] = "\x7b\x26", [0x3a22] = "\x7b\x25",
  [0x3a23] = "\x7b\x24", [0x3a31] = "\x7c\x47", [0x3a36] = "\x7c\x66",
  [0x3a39] = "\x7c\x67", [0x3a3b] = "\x7d\x27", [0x3a40] = "\x48\x66",
  [0x3a4c] = "\x48\x67", [0x3a4d] = "\x53\x4f", [0x3a53] = "\x5d\x34",
  [0x3a57] = "\x61\x79", [0x3a59] = "\x66\x4c", [0x3a5b] = "\x6e\x6c",
  [0x3a5d] = "\x6e\x6d", [0x3a61] = "\x72\x2f", [0x3a62] = "\x7c\x68",
  [0x3a63] = "\x48\x68", [0x3a68] = "\x4f\x50", [0x3a6b] = "\x53\x50",
  [0x3a70] = "\x58\x2e", [0x3a77] = "\x58\x2f", [0x3a79] = "\x58\x33",
  [0x3a7d] = "\x58\x32", [0x3a81] = "\x58\x30", [0x3a82] = "\x58\x31",
  [0x3a88] = "\x5d\x36", [0x3a8b] = "\x5d\x3b", [0x3a8d] = "\x5d\x3a",
  [0x3a92] = "\x5d\x38", [0x3a96] = "\x5d\x39", [0x3a9e] = "\x5d\x35",
  [0x3aab] = "\x5d\x37", [0x3ab1] = "\x61\x7c", [0x3ac1] = "\x61\x7a",
  [0x3ac2] = "\x61\x7b", [0x3aca] = "\x66\x54", [0x3ad2] = "\x66\x56",
  [0x3ad4] = "\x66\x4e", [0x3ad5] = "\x66\x55", [0x3ad8] = "\x66\x51",
  [0x3ad9] = "\x66\x4f", [0x3adc] = "\x66\x50", [0x3add] = "\x66\x52",
  [0x3adf] = "\x66\x4d", [0x3ae1] = "\x66\x53", [0x3ae8] = "\x6a\x7c",
  [0x3aef] = "\x6a\x7e", [0x3af3] = "\x6a\x76", [0x3af4] = "\x6a\x78",
  [0x3af8] = "\x6a\x7a", [0x3af9] = "\x6a\x79", [0x3afd] = "\x6a\x7b",
  [0x3b02] = "\x6a\x77", [0x3b07] = "\x6e\x6f", [0x3b0a] = "\x6e\x73",
  [0x3b10] = "\x6e\x6e", [0x3b12] = "\x6e\x70", [0x3b13] = "\x6e\x71",
  [0x3b15] = "\x6e\x72", [0x3b1a] = "\x6a\x7d", [0x3b21] = "\x72\x34",
  [0x3b25] = "\x72\x32", [0x3b2a] = "\x72\x30", [0x3b2b] = "\x72\x33",
  [0x3b32] = "\x72\x31", [0x3b36] = "\x75\x42", [0x3b38] = "\x75\x44",
  [0x3b3b] = "\x75\x41", [0x3b3d] = "\x75\x45", [0x3b44] = "\x75\x43",
  [0x3b56] = "\x78\x78", [0x3b5e] = "\x78\x79", [0x3b5f] = "\x78\x77",
  [0x3b60] = "\x78\x76", [0x3b64] = "\x7a\x3b", [0x3b6a] = "\x7b\x28",
  [0x3b6c] = "\x7b\x29", [0x3b6f] = "\x7b\x72", [0x3b72] = "\x7b\x71",
  [0x3b7f] = "\x48\x69", [0x3b81] = "\x53\x51", [0x3b83] = "\x61\x7d",
  [0x3b86] = "\x77\x3b", [0x3b8b] = "\x4b\x44", [0x3b8f] = "\x5d\x3d",
  [0x3b93] = "\x5d\x3c", [0x3b96] = "\x61\x7e", [0x3b9c] = "\x66\x57",
  [0x3ba6] = "\x72\x36", [0x3baa] = "\x72\x35", [0x3bac] = "\x75\x46",
  [0x3bb2] = "\x77\x3c", [0x3bba] = "\x7a\x3c", [0x3bbd] = "\x7b\x2a",
  [0x3bc0] = "\x7d\x28", [0x3bd2] = "\x4b\x45", [0x3bd4] = "\x53\x52",
  [0x3be3] = "\x66\x58", [0x3bf4] = "\x77\x3d", [0x3bf8] = "\x7a\x3d",
  [0x3bfc] = "\x7b\x73", [0x3c00] = "\x4b\x46", [0x3c02] = "\x53\x54",
  [0x3c03] = "\x53\x55", [0x3c08] = "\x53\x53", [0x3c0a] = "\x58\x39",
  [0x3c0c] = "\x58\x37", [0x3c0e] = "\x58\x36", [0x3c0f] = "\x58\x3d",
  [0x3c10] = "\x58\x35", [0x3c11] = "\x58\x3e", [0x3c13] = "\x58\x3b",
  [0x3c15] = "\x58\x38", [0x3c16] = "\x58\x3c", [0x3c17] = "\x58\x3a",
  [0x3c18] = "\x58\x34", [0x3c1b] = "\x5d\x45", [0x3c1d] = "\x5d\x3f",
  [0x3c1f] = "\x5d\x44", [0x3c22] = "\x5d\x46", [0x3c23] = "\x5d\x40",
  [0x3c25] = "\x5d\x41", [0x3c2a] = "\x5d\x3e", [0x3c2d] = "\x5d\x43",
  [0x3c31] = "\x5d\x42", [0x3c34] = "\x62\x2b", [0x3c36] = "\x62\x2d",
  [0x3c3a] = "\x62\x2c", [0x3c3b] = "\x62\x21", [0x3c3c] = "\x62\x25",
  [0x3c3e] = "\x66\x6b", [0x3c41] = "\x62\x26", [0x3c46] = "\x62\x2a",
  [0x3c50] = "\x62\x29", [0x3c54] = "\x62\x27", [0x3c55] = "\x62\x23",
  [0x3c56] = "\x62\x2e", [0x3c5b] = "\x62\x28", [0x3c5e] = "\x62\x24",
  [0x3c60] = "\x62\x22", [0x3c62] = "\x66\x66", [0x3c63] = "\x66\x61",
  [0x3c66] = "\x66\x5c", [0x3c68] = "\x66\x6c", [0x3c69] = "\x66\x5d",
  [0x3c6b] = "\x66\x59", [0x3c6c] = "\x66\x68", [0x3c6d] = "\x66\x65",
  [0x3c6e] = "\x66\x67", [0x3c70] = "\x66\x5e", [0x3c71] = "\x66\x63",
  [0x3c72] = "\x66\x5a", [0x3c73] = "\x66\x5b", [0x3c79] = "\x66\x69",
  [0x3c7b] = "\x66\x6a", [0x3c7c] = "\x66\x60", [0x3c85] = "\x66\x64",
  [0x3c87] = "\x66\x5f", [0x3c8c] = "\x6b\x22", [0x3c8d] = "\x6b\x25",
  [0x3c91] = "\x6b\x2d", [0x3c93] = "\x6b\x27", [0x3c95] = "\x6e\x78",
  [0x3c98] = "\x6b\x2c", [0x3c9a] = "\x6b\x2e", [0x3c9e] = "\x6b\x23",
  [0x3ca0] = "\x66\x62", [0x3ca1] = "\x6b\x26", [0x3ca3] = "\x6b\x24",
  [0x3ca4] = "\x6b\x28", [0x3ca5] = "\x6b\x2a", [0x3ca6] = "\x6b\x21",
  [0x3ca7] = "\x6b\x2f", [0x3ca8] = "\x6b\x2b", [0x3caa] = "\x6b\x29",
  [0x3cb0] = "\x6f\x21", [0x3cb2] = "\x6e\x7b", [0x3cb6] = "\x6f\x24",
  [0x3cb9] = "\x6f\x25", [0x3cbc] = "\x6e\x74", [0x3cbf] = "\x6e\x7e",
  [0x3cc2] = "\x6e\x7d", [0x3cc4] = "\x6e\x77", [0x3cc7] = "\x6e\x76",
  [0x3cc9] = "\x6e\x7c", [0x3ccb] = "\x6e\x79", [0x3ccd] = "\x6f\x23",
  [0x3cd2] = "\x6e\x75", [0x3cd6] = "\x6f\x22", [0x3cdb] = "\x6f\x26",
  [0x3cdc] = "\x72\x3c", [0x3ce6] = "\x72\x37", [0x3ce7] = "\x72\x3d",
  [0x3ceb] = "\x72\x39", [0x3ced] = "\x72\x43", [0x3cee] = "\x72\x3e",
  [0x3cf1] = "\x72\x3a", [0x3cf3] = "\x72\x44", [0x3cf6] = "\x72\x45",
  [0x3cf7] = "\x72\x42", [0x3cf8] = "\x6e\x7a", [0x3cfa] = "\x72\x38",
  [0x3cfc] = "\x72\x46", [0x3cfe] = "\x72\x3f", [0x3d00] = "\x72\x3b",
  [0x3d01] = "\x72\x40", [0x3d02] = "\x72\x41", [0x3d04] = "\x75\x4e",
  [0x3d0a] = "\x75\x4b", [0x3d0e] = "\x75\x47", [0x3d10] = "\x75\x4f",
  [0x3d17] = "\x75\x48", [0x3d19] = "\x75\x49", [0x3d1b] = "\x75\x4a",
  [0x3d1d] = "\x75\x4d", [0x3d20] = "\x75\x4c", [0x3d28] = "\x77\x3e",
  [0x3d2b] = "\x77\x41", [0x3d2c] = "\x77\x40", [0x3d39] = "\x77\x3f",
  [0x3d41] = "\x78\x7a", [0x3d46] = "\x79\x23", [0x3d49] = "\x78\x7d",
  [0x3d4e] = "\x79\x21", [0x3d4f] = "\x79\x22", [0x3d58] = "\x78\x7c",
  [0x3d59] = "\x79\x24", [0x3d5a] = "\x78\x7e", [0x3d5c] = "\x78\x7b",
  [0x3d5f] = "\x7a\x42", [0x3d66] = "\x7a\x40", [0x3d6b] = "\x7a\x43",
  [0x3d6c] = "\x7a\x3f", [0x3d6f] = "\x7a\x41", [0x3d70] = "\x7a\x3e",
  [0x3d74] = "\x7b\x2b", [0x3d77] = "\x7b\x2c", [0x3d7d] = "\x7b\x2d",
  [0x3d80] = "\x7b\x74", [0x3d8a] = "\x7c\x48", [0x3d92] = "\x7c\x6a",
  [0x3d93] = "\x7c\x69", [0x3d96] = "\x7c\x6b", [0x3d9a] = "\x7d\x34",
  [0x3d9c] = "\x7d\x3a", [0x3e37] = "\x4b\x47", [0x3e3f] = "\x75\x51",
  [0x3e41] = "\x75\x50", [0x3e46] = "\x4b\x48", [0x3e48] = "\x58\x3f",
  [0x3e49] = "\x5d\x47", [0x3e4c] = "\x6f\x27", [0x3e4e] = "\x6f\x28",
  [0x3e50] = "\x77\x42", [0x3e54] = "\x7d\x43", [0x3e55] = "\x4b\x49",
  [0x3e5a] = "\x5d\x48", [0x3e61] = "\x62\x2f", [0x3e62] = "\x66\x6d",
  [0x3e6a] = "\x6b\x30", [0x3e6b] = "\x72\x47", [0x3e6c] = "\x6f\x29",
  [0x3e6d] = "\x72\x48", [0x3e73] = "\x75\x52", [0x3e79] = "\x58\x41",
  [0x3e7a] = "\x58\x40", [0x3e82] = "\x62\x30", [0x3e89] = "\x66\x6f",
  [0x3e8a] = "\x66\x6e", [0x3e8c] = "\x6b\x32", [0x3e8d] = "\x6b\x31",
  [0x3e93] = "\x72\x49", [0x3e9d] = "\x4b\x4a", [0x3e9e] = "\x53\x56",
  [0x3ea0] = "\x53\x57", [0x3ea1] = "\x58\x42", [0x3ea2] = "\x58\x43",
  [0x3ea7] = "\x5d\x4e", [0x3ea8] = "\x5d\x4c", [0x3ea9] = "\x5d\x49",
  [0x3eaa] = "\x5d\x4d", [0x3eab] = "\x5d\x4b", [0x3eac] = "\x5d\x4a",
  [0x3eaf] = "\x62\x31", [0x3eb2] = "\x66\x74", [0x3eb3] = "\x62\x33",
  [0x3eb4] = "\x62\x38", [0x3eb6] = "\x62\x3a", [0x3eb7] = "\x62\x39",
  [0x3eb8] = "\x62\x3c", [0x3ebb] = "\x62\x36", [0x3ebc] = "\x62\x32",
  [0x3ebd] = "\x62\x34", [0x3ebf] = "\x62\x3b", [0x3ec0] = "\x62\x37",
  [0x3ec1] = "\x62\x35", [0x3ec2] = "\x66\x76", [0x3ec3] = "\x66\x75",
  [0x3ec4] = "\x66\x73", [0x3ec5] = "\x66\x77", [0x3ec7] = "\x66\x71",
  [0x3ec8] = "\x66\x72", [0x3eca] = "\x66\x70", [0x3ed1] = "\x6b\x34",
  [0x3ed2] = "\x6b\x35", [0x3ed3] = "\x6b\x33", [0x3edc] = "\x6f\x32",
  [0x3ede] = "\x6f\x2b", [0x3ee0] = "\x6f\x2a", [0x3ee1] = "\x6f\x34",
  [0x3ee2] = "\x6f\x30", [0x3ee3] = "\x6f\x31", [0x3ee4] = "\x6f\x2d",
  [0x3ee6] = "\x6f\x2c", [0x3eea] = "\x6f\x33", [0x3eec] = "\x6f\x2e",
  [0x3eed] = "\x6f\x2f", [0x3ef4] = "\x72\x4a", [0x3ef8] = "\x75\x56",
  [0x3efa] = "\x75\x53", [0x3efb] = "\x75\x57", [0x3efc] = "\x75\x55",
  [0x3efd] = "\x75\x54", [0x3f05] = "\x77\x43", [0x3f08] = "\x79\x25",
  [0x3f0a] = "\x79\x26", [0x3f0d] = "\x7a\x45", [0x3f0f] = "\x7a\x44",
  [0x3f13] = "\x7b\x2e", [0x3f16] = "\x7b\x75", [0x3f17] = "\x7b\x76",
  [0x3f1b] = "\x7c\x6d", [0x3f64] = "\x4b\x4b", [0x3f66] = "\x5d\x50",
  [0x3f67] = "\x5d\x4f", [0x3f6b] = "\x6b\x36", [0x3f6d] = "\x6f\x35",
  [0x3f70] = "\x4b\x4c", [0x3f73] = "\x53\x59", [0x3f74] = "\x53\x58",
  [0x3f77] = "\x58\x44", [0x3f81] = "\x62\x3f", [0x3f85] = "\x62\x3e",
  [0x3f8a] = "\x62\x3d", [0x3f95] = "\x6b\x38", [0x3f99] = "\x6b\x37",
  [0x3f9f] = "\x6f\x36", [0x3fa3] = "\x6f\x37", [0x3fa8] = "\x75\x58",
  [0x3fb3] = "\x4b\x4d", [0x3fb4] = "\x53\x5a", [0x3fba] = "\x5d\x52",
  [0x3fbe] = "\x5d\x51", [0x3fc6] = "\x62\x47", [0x3fcb] = "\x62\x42",
  [0x3fcc] = "\x62\x45", [0x3fce] = "\x62\x40", [0x3fd1] = "\x62\x44",
  [0x3fda] = "\x62\x43", [0x3fdb] = "\x62\x46", [0x3fdd] = "\x62\x41",
  [0x3fdf] = "\x66\x79", [0x3fe1] = "\x66\x78", [0x3fe4] = "\x67\x21",
  [0x3fe6] = "\x67\x22", [0x3fe8] = "\x66\x7a", [0x3fea] = "\x66\x7e",
  [0x3fef] = "\x66\x7b", [0x3ff3] = "\x66\x7c", [0x3ffa] = "\x66\x7d",
  [0x3ffc] = "\x6b\x39", [0x400f] = "\x6f\x3c", [0x4010] = "\x6f\x39",
  [0x401d] = "\x6f\x3a", [0x401e] = "\x6f\x40", [0x401f] = "\x6f\x3e",
  [0x4021] = "\x6f\x3f", [0x4022] = "\x6f\x3b", [0x4029] = "\x6f\x3d",
  [0x402b] = "\x6f\x38", [0x4031] = "\x72\x4c", [0x4034] = "\x72\x4d",
  [0x4035] = "\x72\x50", [0x4039] = "\x72\x4f", [0x4042] = "\x72\x4e",
  [0x4044] = "\x72\x4b", [0x4048] = "\x75\x5b", [0x4049] = "\x75\x59",
  [0x404a] = "\x75\x5c", [0x404b] = "\x75\x5a", [0x4055] = "\x77\x49",
  [0x4059] = "\x77\x44", [0x405f] = "\x77\x48", [0x4063] = "\x77\x45",
  [0x4064] = "\x77\x47", [0x4066] = "\x77\x46", [0x406c] = "\x79\x2b",
  [0x4072] = "\x79\x28", [0x4074] = "\x79\x2d", [0x4076] = "\x79\x2a",
  [0x407a] = "\x79\x2c", [0x407c] = "\x79\x27", [0x4081] = "\x7a\x47",
  [0x4082] = "\x7a\x49", [0x4085] = "\x7a\x48", [0x4087] = "\x79\x29",
  [0x4089] = "\x7a\x46", [0x408a] = "\x7b\x2f", [0x408b] = "\x7b\x31",
  [0x408d] = "\x7b\x30", [0x4091] = "\x7b\x77", [0x4093] = "\x7b\x78",
  [0x40a1] = "\x7d\x29", [0x40aa] = "\x7d\x3b", [0x40ab] = "\x4b\x4e",
  [0x40ac] = "\x58\x45", [0x40b2] = "\x67\x23", [0x40ba] = "\x6f\x41",
  [0x40c0] = "\x77\x4a", [0x40ca] = "\x4b\x4f", [0x40cb] = "\x4f\x51",
  [0x40cc] = "\x53\x5c", [0x40cd] = "\x53\x5b", [0x40cf] = "\x58\x48",
  [0x40d2] = "\x58\x46", [0x40d4] = "\x58\x47", [0x40db] = "\x5d\x53",
  [0x40df] = "\x5d\x54", [0x40f8] = "\x62\x49", [0x40fb] = "\x62\x48",
  [0x40fc] = "\x62\x4a", [0x40fe] = "\x67\x26", [0x4103] = "\x67\x24",
  [0x4109] = "\x67\x25", [0x410a] = "\x67\x27", [0x4112] = "\x6b\x3b",
  [0x4113] = "\x6b\x3d", [0x4114] = "\x6b\x3a", [0x4115] = "\x6b\x3c",
  [0x411b] = "\x6f\x43", [0x411c] = "\x6f\x48", [0x411d] = "\x6f\x42",
  [0x411e] = "\x6f\x49", [0x411f] = "\x6f\x44", [0x4125] = "\x6f\x4a",
  [0x4126] = "\x6f\x46", [0x4129] = "\x6f\x45", [0x412a] = "\x6f\x47",
  [0x412f] = "\x72\x52", [0x4133] = "\x72\x54", [0x4138] = "\x72\x53",
  [0x413b] = "\x72\x51", [0x413e] = "\x75\x5e", [0x413f] = "\x75\x61",
  [0x4142] = "\x75\x5f", [0x4144] = "\x75\x5d", [0x4145] = "\x75\x60",
  [0x4149] = "\x77\x4b", [0x414d] = "\x77\x4c", [0x414e] = "\x79\x2f",
  [0x4154] = "\x79\x2e", [0x415f] = "\x7b\x32", [0x4161] = "\x7b\x79",
  [0x419b] = "\x4b\x50", [0x419c] = "\x62\x4b", [0x419f] = "\x67\x28",
  [0x41a3] = "\x6b\x3e", [0x41a6] = "\x72\x56", [0x41a8] = "\x72\x55",
  [0x41ad] = "\x79\x30", [0x41ae] = "\x7a\x29", [0x41af] = "\x7b\x33",
  [0x41b0] = "\x4b\x51", [0x41b1] = "\x58\x49", [0x41b2] = "\x67\x29",
  [0x41c2] = "\x4b\x52", [0x41c4] = "\x4b\x55", [0x41c5] = "\x4b\x54",
  [0x41c6] = "\x4b\x53", [0x41ce] = "\x4f\x52", [0x41d1] = "\x4f\x54",
  [0x41d4] = "\x4f\x53", [0x41e2] = "\x53\x5f", [0x41e4] = "\x53\x64",
  [0x41e5] = "\x53\x61", [0x41e6] = "\x53\x5e", [0x41e8] = "\x53\x65",
  [0x41ea] = "\x53\x60", [0x41eb] = "\x53\x63", [0x41ed] = "\x53\x62",
  [0x41f0] = "\x53\x5d", [0x41f4] = "\x58\x4f", [0x41f7] = "\x58\x4c",
  [0x41f8] = "\x58\x53", [0x41fa] = "\x58\x4e", [0x41fd] = "\x58\x51",
  [0x4200] = "\x58\x4d", [0x4201] = "\x58\x4a", [0x4203] = "\x58\x50",
  [0x4205] = "\x58\x52", [0x4206] = "\x58\x4b", [0x420d] = "\x5d\x56",
  [0x420f] = "\x5d\x60", [0x4210] = "\x5d\x5c", [0x4214] = "\x5d\x64",
  [0x4215] = "\x5d\x5d", [0x4216] = "\x5d\x62", [0x4217] = "\x5d\x58",
  [0x4219] = "\x5d\x55", [0x421a] = "\x5d\x57", [0x421b] = "\x5d\x63",
  [0x421d] = "\x5d\x5b", [0x421e] = "\x5d\x5e", [0x421f] = "\x5d\x5a",
  [0x4220] = "\x5d\x5f", [0x4222] = "\x5d\x61", [0x4223] = "\x5d\x59",
  [0x422e] = "\x62\x4c", [0x4231] = "\x62\x4e", [0x4232] = "\x62\x50",
  [0x4235] = "\x62\x4d", [0x4236] = "\x62\x51", [0x4238] = "\x62\x4f",
  [0x423c] = "\x67\x2f", [0x423e] = "\x67\x37", [0x4241] = "\x67\x38",
  [0x4242] = "\x67\x2d", [0x4247] = "\x67\x32", [0x424a] = "\x67\x2b",
  [0x424b] = "\x67\x2a", [0x424d] = "\x67\x35", [0x424e] = "\x67\x34",
  [0x424f] = "\x67\x33", [0x4250] = "\x67\x31", [0x4251] = "\x67\x36",
  [0x4253] = "\x67\x2c", [0x4254] = "\x67\x2e", [0x4255] = "\x67\x30",
  [0x4258] = "\x6b\x40", [0x4259] = "\x6b\x43", [0x425b] = "\x6b\x47",
  [0x425c] = "\x6b\x41", [0x425d] = "\x6b\x46", [0x425e] = "\x6b\x44",
  [0x4260] = "\x6b\x3f", [0x4262] = "\x6b\x45", [0x4263] = "\x6b\x42",
  [0x4268] = "\x6f\x4d", [0x4269] = "\x6f\x4b", [0x426d] = "\x6f\x4e",
  [0x426e] = "\x6f\x4c", [0x4272] = "\x72\x5a", [0x4274] = "\x72\x58",
  [0x4275] = "\x72\x57", [0x4277] = "\x6f\x4f", [0x4278] = "\x72\x59",
  [0x427a] = "\x72\x5c", [0x427c] = "\x72\x5b", [0x427d] = "\x75\x63",
  [0x427f] = "\x75\x62", [0x4280] = "\x75\x67", [0x4281] = "\x75\x65",
  [0x4282] = "\x75\x66", [0x4283] = "\x77\x4e", [0x4284] = "\x75\x64",
  [0x4287] = "\x77\x4d", [0x4288] = "\x77\x4f", [0x428a] = "\x79\x31",
  [0x428b] = "\x79\x32", [0x428f] = "\x7c\x4a", [0x4290] = "\x7c\x49",
  [0x4291] = "\x4b\x57", [0x4295] = "\x58\x54", [0x42a2] = "\x4b\x58",
  [0x42a3] = "\x4b\x5b", [0x42a6] = "\x4b\x5a", [0x42aa] = "\x4b\x59",
  [0x42b1] = "\x4f\x57", [0x42b5] = "\x4f\x55", [0x42b6] = "\x4f\x58",
  [0x42b8] = "\x4f\x56", [0x42c1] = "\x53\x68", [0x42c3] = "\x53\x69",
  [0x42ca] = "\x53\x66", [0x42ce] = "\x53\x67", [0x42dd] = "\x58\x56",
  [0x42e1] = "\x58\x55", [0x42e2] = "\x58\x57", [0x42e8] = "\x5d\x65",
  [0x42ed] = "\x5d\x66", [0x42f5] = "\x62\x53", [0x42fd] = "\x5d\x67",
  [0x42fe] = "\x62\x55", [0x4302] = "\x62\x52", [0x4309] = "\x62\x54",
  [0x4312] = "\x67\x39", [0x4317] = "\x67\x3a", [0x4318] = "\x6b\x49",
  [0x4319] = "\x6b\x48", [0x431e] = "\x6b\x4a", [0x4327] = "\x6f\x52",
  [0x432d] = "\x6f\x51", [0x4330] = "\x6f\x50", [0x4331] = "\x6f\x53",
  [0x4334] = "\x72\x5d", [0x4339] = "\x75\x68", [0x4348] = "\x7b\x7a",
  [0x4349] = "\x4b\x5c", [0x434a] = "\x53\x6b", [0x434b] = "\x53\x6a",
  [0x434c] = "\x58\x5a", [0x434d] = "\x58\x59", [0x4352] = "\x58\x58",
  [0x4357] = "\x5d\x68", [0x4363] = "\x62\x56", [0x4365] = "\x62\x57",
  [0x4369] = "\x67\x3d", [0x436a] = "\x67\x3c", [0x436c] = "\x67\x3b",
  [0x4374] = "\x6b\x4e", [0x4375] = "\x6b\x4b", [0x4377] = "\x6b\x4d",
  [0x4378] = "\x6b\x4c", [0x4383] = "\x6f\x57", [0x4387] = "\x6f\x54",
  [0x4389] = "\x6f\x55", [0x438b] = "\x6f\x56", [0x4392] = "\x72\x5e",
  [0x439c] = "\x75\x6b", [0x439e] = "\x75\x6a", [0x43a3] = "\x75\x69",
  [0x43ab] = "\x77\x50", [0x43ac] = "\x77\x51", [0x43ae] = "\x79\x34",
  [0x43b1] = "\x79\x33", [0x43b4] = "\x7a\x4a", [0x43ba] = "\x7b\x34",
  [0x43c0] = "\x7c\x6e", [0x43c1] = "\x7d\x2a", [0x43c5] = "\x7d\x3c",
  [0x43c6] = "\x4b\x5d", [0x43c7] = "\x4f\x59", [0x43c9] = "\x67\x3e",
  [0x43cb] = "\x7a\x4b", [0x43cc] = "\x4b\x5e", [0x43cd] = "\x53\x6c",
  [0x43ce] = "\x5d\x69", [0x43cf] = "\x62\x58", [0x43d0] = "\x77\x52",
  [0x43d1] = "\x4f\x5a", [0x43d7] = "\x58\x5d", [0x43d8] = "\x58\x5b",
  [0x43d9] = "\x58\x5f", [0x43dc] = "\x58\x5e", [0x43dd] = "\x58\x5c",
  [0x43e3] = "\x5d\x6c", [0x43e6] = "\x5d\x6b", [0x43e7] = "\x5d\x6d",
  [0x43e9] = "\x5d\x6f", [0x43ed] = "\x5d\x6e", [0x43f5] = "\x5d\x6a",
  [0x4407] = "\x62\x60", [0x4409] = "\x62\x5c", [0x440d] = "\x62\x5e",
  [0x4410] = "\x62\x5f", [0x4411] = "\x62\x61", [0x4414] = "\x62\x59",
  [0x4415] = "\x62\x5a", [0x441e] = "\x62\x5d", [0x4423] = "\x62\x5b",
  [0x4434] = "\x67\x49", [0x4437] = "\x67\x3f", [0x4438] = "\x67\x41",
  [0x4439] = "\x67\x4d", [0x443d] = "\x67\x42", [0x443e] = "\x67\x44",
  [0x443f] = "\x67\x4e", [0x4440] = "\x67\x43", [0x4445] = "\x67\x4c",
  [0x4449] = "\x67\x4a", [0x444b] = "\x67\x46", [0x444d] = "\x67\x4b",
  [0x4451] = "\x67\x48", [0x4457] = "\x67\x40", [0x445a] = "\x67\x4f",
  [0x445b] = "\x67\x45", [0x4464] = "\x67\x47", [0x4478] = "\x6b\x4f",
  [0x447b] = "\x6b\x55", [0x447c] = "\x6b\x59", [0x4480] = "\x6b\x51",
  [0x4485] = "\x6b\x52", [0x4491] = "\x6b\x5a", [0x4493] = "\x6b\x56",
  [0x4496] = "\x6b\x54", [0x4498] = "\x6b\x53", [0x449c] = "\x6b\x57",
  [0x44a8] = "\x6b\x58", [0x44ac] = "\x6b\x50", [0x44b2] = "\x6f\x63",
  [0x44b3] = "\x6f\x5e", [0x44b7] = "\x6f\x5a", [0x44bb] = "\x6f\x59",
  [0x44bc] = "\x6f\x5f", [0x44c1] = "\x6f\x5d", [0x44c5] = "\x6f\x58",
  [0x44c7] = "\x6f\x61", [0x44d2] = "\x6f\x60", [0x44e4] = "\x6f\x5c",
  [0x44ea] = "\x6f\x5b", [0x44f0] = "\x6f\x62", [0x44f8] = "\x72\x61",
  [0x44fc] = "\x72\x65", [0x4504] = "\x72\x67", [0x4510] = "\x72\x69",
  [0x4515] = "\x72\x6c", [0x4518] = "\x75\x73", [0x4519] = "\x72\x6e",
  [0x451a] = "\x72\x68", [0x4520] = "\x72\x5f", [0x4521] = "\x72\x6b",
  [0x4522] = "\x72\x64", [0x4526] = "\x72\x6a", [0x4528] = "\x75\x6e",
  [0x452b] = "\x72\x66", [0x452e] = "\x72\x6d", [0x452f] = "\x72\x63",
  [0x4533] = "\x72\x62", [0x4536] = "\x72\x60", [0x454a] = "\x75\x70",
  [0x454b] = "\x75\x72", [0x454d] = "\x75\x6c", [0x4554] = "\x75\x79",
  [0x455a] = "\x75\x78", [0x455b] = "\x75\x76", [0x4565] = "\x75\x71",
  [0x456c] = "\x75\x75", [0x4570] = "\x75\x77", [0x4575] = "\x75\x6f",
  [0x457e] = "\x75\x74", [0x4582] = "\x75\x6d", [0x458a] = "\x77\x54",
  [0x4594] = "\x77\x53", [0x4596] = "\x77\x55", [0x4597] = "\x77\x5d",
  [0x4598] = "\x77\x5b", [0x459a] = "\x77\x5c", [0x45a2] = "\x77\x56",
  [0x45ac] = "\x77\x59", [0x45ae] = "\x77\x58", [0x45b0] = "\x77\x5a",
  [0x45b3] = "\x77\x57", [0x45c3] = "\x79\x38", [0x45c8] = "\x79\x39",
  [0x45cd] = "\x79\x3e", [0x45d1] = "\x79\x36", [0x45d6] = "\x79\x3c",
  [0x45d7] = "\x79\x41", [0x45d8] = "\x79\x3f", [0x45dc] = "\x79\x3a",
  [0x45dd] = "\x79\x3b", [0x45df] = "\x79\x37", [0x45e1] = "\x79\x35",
  [0x45e2] = "\x79\x3d", [0x45e4] = "\x79\x40", [0x45e8] = "\x79\x42",
  [0x45fd] = "\x7a\x4e", [0x4603] = "\x7a\x4d", [0x4618] = "\x7a\x4c",
  [0x462b] = "\x7b\x3b", [0x462e] = "\x7b\x35", [0x4632] = "\x7b\x3a",
  [0x4633] = "\x7b\x36", [0x4635] = "\x7b\x37", [0x4638] = "\x7b\x39",
  [0x463a] = "\x7b\x38", [0x4644] = "\x7b\x7b", [0x4651] = "\x7b\x7c",
  [0x4652] = "\x7b\x7d", [0x4660] = "\x7c\x4c", [0x4663] = "\x7c\x4b",
  [0x4664] = "\x7c\x4d", [0x466a] = "\x7c\x6f", [0x4670] = "\x7d\x2c",
  [0x4672] = "\x7d\x2b", [0x4677] = "\x7d\x35", [0x467c] = "\x7d\x3f",
  [0x467d] = "\x7d\x3d", [0x467e] = "\x7d\x3e", [0x467f] = "\x7d\x44",
  [0x4777] = "\x4f\x5b", [0x4780] = "\x4f\x5c", [0x4782] = "\x53\x6d",
  [0x4783] = "\x58\x60", [0x4789] = "\x5d\x70", [0x478b] = "\x62\x64",
  [0x478e] = "\x62\x68", [0x478f] = "\x62\x63", [0x4791] = "\x62\x65",
  [0x4792] = "\x62\x67", [0x4793] = "\x62\x66", [0x4794] = "\x62\x62",
  [0x4798] = "\x67\x50", [0x47a1] = "\x6b\x5b", [0x47a3] = "\x6b\x5e",
  [0x47a4] = "\x6b\x60", [0x47a5] = "\x6b\x5f", [0x47a8] = "\x6b\x5c",
  [0x47a9] = "\x6b\x5d", [0x47ad] = "\x6f\x64", [0x47b1] = "\x6f\x65",
  [0x47bb] = "\x72\x6f", [0x47c6] = "\x75\x7e", [0x47c8] = "\x75\x7d",
  [0x47ca] = "\x75\x7a", [0x47cb] = "\x75\x7b", [0x47cc] = "\x75\x7c",
  [0x47d0] = "\x77\x60", [0x47d4] = "\x77\x5e", [0x47d5] = "\x77\x61",
  [0x47d6] = "\x77\x5f", [0x47dc] = "\x79\x43", [0x47e1] = "\x7a\x4f",
  [0x47e2] = "\x7b\x3c", [0x481c] = "\x4f\x5d", [0x4821] = "\x48\x6a",
  [0x482a] = "\x4b\x62", [0x482c] = "\x4b\x63", [0x482e] = "\x4b\x60",
  [0x4831] = "\x4b\x61", [0x4832] = "\x4b\x5f", [0x483b] = "\x4f\x60",
  [0x483f] = "\x4f\x5f", [0x4840] = "\x4f\x5e", [0x4842] = "\x4f\x62",
  [0x4844] = "\x4f\x61", [0x484b] = "\x53\x6f", [0x484c] = "\x53\x70",
  [0x484d] = "\x53\x71", [0x4850] = "\x53\x6e", [0x4858] = "\x58\x67",
  [0x485b] = "\x58\x64", [0x485d] = "\x58\x65", [0x485e] = "\x58\x68",
  [0x4861] = "\x58\x63", [0x4862] = "\x58\x61", [0x4863] = "\x58\x62",
  [0x4864] = "\x58\x66", [0x486a] = "\x5d\x71", [0x486c] = "\x5d\x79",
  [0x4870] = "\x5d\x75", [0x4872] = "\x62\x70", [0x4873] = "\x5d\x73",
  [0x4874] = "\x5d\x76", [0x4875] = "\x5d\x72", [0x4876] = "\x5d\x77",
  [0x4877] = "\x5d\x78", [0x4878] = "\x5d\x74", [0x487d] = "\x62\x6c",
  [0x4884] = "\x62\x71", [0x4885] = "\x62\x6d", [0x4886] = "\x62\x6e",
  [0x488a] = "\x62\x69", [0x488b] = "\x62\x6b", [0x488d] = "\x62\x6f",
  [0x488e] = "\x62\x6a", [0x4894] = "\x67\x52", [0x4895] = "\x67\x53",
  [0x4898] = "\x67\x51", [0x4899] = "\x6b\x61", [0x489b] = "\x6b\x63",
  [0x489c] = "\x6b\x62", [0x48a7] = "\x72\x70", [0x48a8] = "\x72\x71",
  [0x48aa] = "\x72\x72", [0x48b1] = "\x76\x21", [0x48b4] = "\x79\x44",
  [0x48b8] = "\x76\x22", [0x48b9] = "\x4f\x63", [0x48bb] = "\x58\x69",
  [0x48c0] = "\x5d\x7a", [0x48c1] = "\x62\x72", [0x48c4] = "\x62\x74",
  [0x48c5] = "\x62\x73", [0x48c6] = "\x62\x75", [0x48c7] = "\x62\x76",
  [0x48c9] = "\x67\x56", [0x48ca] = "\x67\x57", [0x48cb] = "\x67\x55",
  [0x48cc] = "\x6b\x64", [0x48cd] = "\x67\x54", [0x48d2] = "\x6b\x65",
  [0x48d5] = "\x72\x73", [0x48d6] = "\x76\x23", [0x48d9] = "\x77\x64",
  [0x48db] = "\x77\x65", [0x48dc] = "\x77\x63", [0x48de] = "\x77\x66",
  [0x48e2] = "\x77\x62", [0x48e3] = "\x79\x45", [0x48e8] = "\x4f\x64",
  [0x48e9] = "\x5d\x7c", [0x48ea] = "\x5d\x7b", [0x48ef] = "\x62\x77",
  [0x48f2] = "\x62\x78", [0x48f6] = "\x67\x5b", [0x48f7] = "\x67\x58",
  [0x48f9] = "\x67\x5a", [0x48fb] = "\x67\x59", [0x4900] = "\x6b\x66",
  [0x4904] = "\x6f\x66", [0x4906] = "\x6f\x67", [0x4907] = "\x6f\x68",
  [0x4909] = "\x6f\x69", [0x490d] = "\x72\x77", [0x490e] = "\x72\x74",
  [0x490f] = "\x72\x79", [0x4911] = "\x72\x75", [0x4913] = "\x72\x78",
  [0x4916] = "\x72\x76", [0x491c] = "\x76\x24", [0x491e] = "\x76\x25",
  [0x4924] = "\x77\x67", [0x4927] = "\x79\x47", [0x492a] = "\x79\x46",
  [0x4930] = "\x7a\x50", [0x4932] = "\x7b\x3f", [0x4938] = "\x7b\x3d",
  [0x4939] = "\x7b\x3e", [0x493d] = "\x7b\x7e", [0x493e] = "\x7c\x21",
  [0x4942] = "\x7c\x70", [0x4944] = "\x7c\x72", [0x4948] = "\x7c\x71",
  [0x4952] = "\x4f\x65", [0x4956] = "\x67\x5c", [0x495b] = "\x72\x7a",
  [0x495c] = "\x72\x7b", [0x495e] = "\x4f\x66", [0x4960] = "\x6f\x6a",
  [0x4961] = "\x79\x48", [0x4962] = "\x53\x72", [0x4966] = "\x72\x7c",
  [0x4968] = "\x7c\x4e", [0x4969] = "\x53\x73", [0x4974] = "\x67\x5d",
  [0x4976] = "\x67\x5e", [0x497c] = "\x6b\x67", [0x4985] = "\x6b\x68",
  [0x498b] = "\x6f\x6c", [0x498d] = "\x6f\x6b", [0x498f] = "\x6f\x6d",
  [0x4998] = "\x72\x7d", [0x49a0] = "\x76\x26", [0x49a3] = "\x77\x68",
  [0x49a6] = "\x77\x69", [0x49ad] = "\x77\x6a", [0x49c1] = "\x7c\x23",
  [0x49c3] = "\x7c\x22", [0x49c6] = "\x7c\x73", [0x49c9] = "\x7d\x36",
  [0x49cb] = "\x53\x74", [0x49cc] = "\x62\x79", [0x49d3] = "\x76\x27",
  [0x49dc] = "\x79\x49", [0x49ed] = "\x53\x75", [0x49f3] = "\x53\x76",
  [0x49f6] = "\x6b\x69", [0x49f9] = "\x77\x6b", [0x49fb] = "\x79\x4a",
  [0x49ff] = "\x7b\x40", [0x4a01] = "\x53\x77", [0x4a02] = "\x5e\x21",
  [0x4a03] = "\x5e\x22", [0x4a05] = "\x62\x7a", [0x4a06] = "\x62\x7b",
  [0x4a08] = "\x62\x7c", [0x4a0a] = "\x67\x62", [0x4a0c] = "\x67\x64",
  [0x4a10] = "\x67\x5f", [0x4a11] = "\x67\x60", [0x4a12] = "\x67\x63",
  [0x4a13] = "\x67\x61", [0x4a17] = "\x6b\x6a", [0x4a18] = "\x6b\x6b",
  [0x4a1c] = "\x6f\x70", [0x4a21] = "\x6f\x6e", [0x4a24] = "\x73\x26",
  [0x4a2b] = "\x6f\x6f", [0x4a2d] = "\x73\x24", [0x4a30] = "\x72\x7e",
  [0x4a37] = "\x73\x23", [0x4a38] = "\x73\x21", [0x4a39] = "\x73\x25",
  [0x4a3b] = "\x73\x22", [0x4a46] = "\x76\x28", [0x4a4c] = "\x77\x6e",
  [0x4a4d] = "\x77\x6c", [0x4a4e] = "\x77\x6f", [0x4a4f] = "\x77\x6d",
  [0x4a53] = "\x77\x70", [0x4a58] = "\x79\x4c", [0x4a5b] = "\x79\x4d",
  [0x4a5e] = "\x79\x4b", [0x4a65] = "\x7b\x42", [0x4a67] = "\x7b\x41",
  [0x4a6b] = "\x7c\x24", [0x4a6f] = "\x7c\x4f", [0x4a70] = "\x7c\x74",
  [0x4a71] = "\x7d\x2d", [0x4aa8] = "\x53\x78", [0x4aaf] = "\x6b\x6c",
  [0x4ab1] = "\x6b\x6d", [0x4ab3] = "\x6f\x71", [0x4ab6] = "\x76\x29",
  [0x4aba] = "\x77\x71", [0x4abc] = "\x79\x4e", [0x4ac4] = "\x7a\x51",
  [0x4adb] = "\x53\x79", [0x4adf] = "\x53\x7a", [0x4ae2] = "\x58\x6a",
  [0x4ae7] = "\x62\x7d", [0x4ae9] = "\x63\x22", [0x4aea] = "\x62\x7e",
  [0x4aed] = "\x63\x24", [0x4aef] = "\x63\x21", [0x4af2] = "\x63\x23",
  [0x4af4] = "\x67\x66", [0x4afc] = "\x67\x65", [0x4afd] = "\x67\x67",
  [0x4afe] = "\x67\x68", [0x4b03] = "\x6b\x6e", [0x4b05] = "\x6b\x6f",
  [0x4b09] = "\x6b\x71", [0x4b0a] = "\x6f\x72", [0x4b0c] = "\x6b\x70",
  [0x4b10] = "\x73\x27", [0x4b12] = "\x6f\x74", [0x4b13] = "\x6f\x73",
  [0x4b18] = "\x6f\x75", [0x4b1a] = "\x73\x2c", [0x4b1b] = "\x73\x2a",
  [0x4b1e] = "\x73\x29", [0x4b21] = "\x73\x2b", [0x4b28] = "\x73\x28",
  [0x4b2e] = "\x77\x75", [0x4b35] = "\x76\x2a", [0x4b3d] = "\x77\x74",
  [0x4b3e] = "\x77\x72", [0x4b3f] = "\x77\x73", [0x4b45] = "\x79\x4f",
  [0x4b49] = "\x79\x50", [0x4b51] = "\x7a\x53", [0x4b52] = "\x7a\x52",
  [0x4b55] = "\x7c\x25", [0x4b57] = "\x7b\x43", [0x4b5c] = "\x7c\x50",
  [0x4b5e] = "\x7d\x2e", [0x4b96] = "\x53\x7b", [0x4b99] = "\x53\x7c",
  [0x4ba5] = "\x77\x76", [0x4ba8] = "\x7a\x54", [0x4bac] = "\x58\x6b",
  [0x4bad] = "\x63\x26", [0x4bae] = "\x63\x25", [0x4bb1] = "\x67\x6a",
  [0x4bb3] = "\x67\x69", [0x4bb4] = "\x67\x6b", [0x4bc1] = "\x6b\x72",
  [0x4bd0] = "\x6f\x77", [0x4bd1] = "\x6f\x7a", [0x4bd2] = "\x6f\x7c",
  [0x4bd5] = "\x6f\x7b", [0x4bd9] = "\x6f\x7d", [0x4bdb] = "\x6f\x79",
  [0x4bdd] = "\x6f\x76", [0x4bdf] = "\x6f\x78", [0x4be2] = "\x73\x2e",
  [0x4bed] = "\x73\x2d", [0x4bf1] = "\x73\x2f", [0x4bff] = "\x76\x2c",
  [0x4c01] = "\x76\x2b", [0x4c0e] = "\x77\x77", [0x4c16] = "\x79\x51",
  [0x4c19] = "\x79\x52", [0x4c2b] = "\x7a\x55", [0x4c30] = "\x7a\x56",
  [0x4c35] = "\x7a\x58", [0x4c37] = "\x7a\x57", [0x4c3e] = "\x7b\x47",
  [0x4c40] = "\x7b\x46", [0x4c43] = "\x7b\x45", [0x4c45] = "\x7b\x44",
  [0x4c4d] = "\x7c\x27", [0x4c55] = "\x7c\x26", [0x4c57] = "\x7c\x53",
  [0x4c5a] = "\x7c\x51", [0x4c5b] = "\x7c\x52", [0x4c5f] = "\x7c\x75",
  [0x4c62] = "\x7d\x37", [0x4c65] = "\x7d\x38", [0x4c6a] = "\x7d\x47",
  [0x4ca8] = "\x58\x6c", [0x4caf] = "\x6b\x73", [0x4cb0] = "\x6b\x74",
  [0x4cb7] = "\x6f\x7e", [0x4cb8] = "\x73\x30", [0x4cbc] = "\x73\x31",
  [0x4cc1] = "\x77\x78", [0x4ccf] = "\x7b\x48", [0x4cd1] = "\x7c\x56",
  [0x4cd2] = "\x7c\x28", [0x4cd3] = "\x7c\x54", [0x4cd4] = "\x7c\x55",
  [0x4cd6] = "\x7d\x2f", [0x4cd8] = "\x58\x6d", [0x4ce1] = "\x67\x6c",
  [0x4ce6] = "\x6b\x75", [0x4ced] = "\x73\x33", [0x4cee] = "\x70\x21",
  [0x4cef] = "\x70\x22", [0x4cfb] = "\x73\x32", [0x4d03] = "\x77\x79",
  [0x4d06] = "\x77\x7a", [0x4d0d] = "\x79\x53", [0x4d1a] = "\x7c\x29",
  [0x4d22] = "\x7c\x76", [0x4d23] = "\x7d\x30", [0x4d25] = "\x58\x6e",
  [0x4d27] = "\x70\x23", [0x4d28] = "\x73\x34", [0x4d31] = "\x7d\x48",
  [0x4d32] = "\x58\x6f", [0x4d3c] = "\x58\x70", [0x4d41] = "\x6b\x76",
  [0x4d42] = "\x6b\x77", [0x4d44] = "\x70\x25", [0x4d45] = "\x70\x24",
  [0x4d4d] = "\x77\x7d", [0x4d4e] = "\x77\x7c", [0x4d4f] = "\x77\x7b",
  [0x4d51] = "\x7b\x4a", [0x4d54] = "\x7b\x49", [0x4d58] = "\x7c\x77",
  [0x4d5a] = "\x5e\x23", [0x4d6f] = "\x70\x27", [0x4d77] = "\x70\x26",
  [0x4d91] = "\x73\x35", [0x4daa] = "\x76\x2f", [0x4dab] = "\x76\x2e",
  [0x4dad] = "\x76\x30", [0x4dae] = "\x76\x2d", [0x4dc0] = "\x78\x24",
  [0x4dc8] = "\x78\x23", [0x4dc9] = "\x78\x21", [0x4dca] = "\x77\x7e",
  [0x4dd6] = "\x79\x56", [0x4ddb] = "\x79\x57", [0x4de7] = "\x79\x55",
  [0x4de8] = "\x79\x54", [0x4dfd] = "\x78\x22", [0x4e0d] = "\x7a\x5a",
  [0x4e13] = "\x7a\x59", [0x4e25] = "\x7b\x4c", [0x4e2d] = "\x7b\x4b",
  [0x4e31] = "\x7c\x2b", [0x4e3b] = "\x7c\x2d", [0x4e3e] = "\x7c\x2c",
  [0x4e49] = "\x7c\x2a", [0x4e54] = "\x7c\x57", [0x4e56] = "\x7c\x59",
  [0x4e57] = "\x7c\x58", [0x4e5f] = "\x7c\x78", [0x4e77] = "\x7d\x40",
  [0x4e78] = "\x7d\x41", [0x4ee5] = "\x5e\x24", [0x4ee9] = "\x67\x6d",
  [0x4ef3] = "\x6b\x7a", [0x4ef4] = "\x6b\x78", [0x4ef6] = "\x6b\x79",
  [0x4f03] = "\x70\x2a", [0x4f06] = "\x70\x28", [0x4f09] = "\x70\x29",
  [0x4f12] = "\x73\x3a", [0x4f15] = "\x73\x36", [0x4f1b] = "\x73\x3b",
  [0x4f23] = "\x73\x37", [0x4f26] = "\x73\x38", [0x4f28] = "\x73\x39",
  [0x4f3b] = "\x76\x31", [0x4f3f] = "\x76\x32", [0x4f51] = "\x78\x25",
  [0x4f5d] = "\x78\x26", [0x4f60] = "\x78\x27", [0x4f61] = "\x79\x59",
  [0x4f6a] = "\x79\x5b", [0x4f6c] = "\x79\x5c", [0x4f72] = "\x79\x5a",
  [0x4f89] = "\x79\x58", [0x4faf] = "\x7b\x4d", [0x4fb4] = "\x7b\x4e",
  [0x4fb8] = "\x7b\x50", [0x4fc2] = "\x7b\x4f", [0x4fd3] = "\x7c\x2e",
  [0x4fd7] = "\x7c\x2f", [0x4fe5] = "\x7c\x5a", [0x4ff9] = "\x7c\x79",
  [0x4ffa] = "\x7c\x7a", [0x501a] = "\x7d\x45", [0x501b] = "\x7d\x49",
  [0x501e] = "\x7d\x4a", [0x5075] = "\x5e\x25", [0x5079] = "\x7a\x5b",
  [0x507c] = "\x7c\x7b", [0x507d] = "\x7c\x7c", [0x507f] = "\x5e\x26",
  [0x5082] = "\x67\x6e", [0x508b] = "\x76\x33", [0x5092] = "\x79\x5d",
  [0x5093] = "\x79\x5f", [0x5097] = "\x79\x5e", [0x509d] = "\x7b\x51",
  [0x509f] = "\x7c\x5b", [0x50a5] = "\x5e\x27", [0x50a9] = "\x70\x2b",
  [0x50b4] = "\x79\x60", [0x50b5] = "\x7a\x5c", [0x50bb] = "\x5e\x28",
  [0x50bc] = "\x6b\x7b", [0x50be] = "\x70\x2c", [0x50c3] = "\x63\x27",
  [0x50cc] = "\x7d\x31", [0x50cd] = "\x63\x28", [0x50ce] = "\x70\x2d",
  [0x50cf] = "\x76\x34", [0x50d1] = "\x63\x29", [0x50d4] = "\x73\x3d",
  [0x50d8] = "\x73\x3c", [0x50db] = "\x76\x38", [0x50dc] = "\x76\x36",
  [0x50dd] = "\x76\x37", [0x50de] = "\x76\x35", [0x50e0] = "\x78\x28",
  [0x50e8] = "\x7a\x5d", [0x50ef] = "\x7b\x52", [0x50f4] = "\x7c\x5c",
  [0x50f7] = "\x7d\x42", [0x5107] = "\x7c\x7d", [0x510e] = "\x67\x6f",
  [0x5113] = "\x67\x70", [0x5115] = "\x78\x29", [0x5119] = "\x7b\x53",
  [0x5120] = "\x67\x71", [0x512c] = "\x78\x2a", [0x512f] = "\x7a\x5e",
  [0x5134] = "\x7c\x30", [0x513b] = "\x6b\x7c", [0x513e] = "\x76\x39",
  [0x514a] = "\x6b\x7d", [0x514b] = "\x76\x3a", [0x5152] = "\x70\x2f",
  [0x515c] = "\x7b\x54", [0x515f] = "\x7a\x5f", [0x5161] = "\x7a\x61",
  [0x5163] = "\x7a\x60", [0x5166] = "\x7b\x55", [0x5167] = "\x7b\x56",
  [0x516a] = "\x7c\x32", [0x516c] = "\x7c\x31", [0x5172] = "\x7d\x21",
  [0x5177] = "\x7c\x7e", [0x518d] = "\x73\x3e", [0x5190] = "\x78\x32",
  [0x5194] = "\x7c\x33", [0x519c] = "\x73\x3f",
};

/* Glyphs for vertical variants.  The table can be created using

   egrep '^0x' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0xFE' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xfe30,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab13[][2] =
{
  [0x0000] = "\x21\x2b", [0x0001] = "\x21\x36", [0x0002] = "\x21\x38",
  [0x0005] = "\x21\x40", [0x0006] = "\x21\x41", [0x0007] = "\x21\x44",
  [0x0008] = "\x21\x45", [0x0009] = "\x21\x48", [0x000a] = "\x21\x49",
  [0x000b] = "\x21\x4c", [0x000c] = "\x21\x4d", [0x000d] = "\x21\x50",
  [0x000e] = "\x21\x51", [0x000f] = "\x21\x54", [0x0010] = "\x21\x55",
  [0x0011] = "\x21\x58", [0x0012] = "\x21\x59", [0x0013] = "\x21\x5c",
  [0x0014] = "\x21\x5d", [0x0019] = "\x22\x27", [0x001a] = "\x22\x28",
  [0x001b] = "\x22\x2b", [0x001c] = "\x22\x2c", [0x001d] = "\x22\x29",
  [0x001e] = "\x22\x2a", [0x0020] = "\x21\x2e", [0x0021] = "\x21\x2f",
  [0x0022] = "\x21\x30", [0x0024] = "\x21\x32", [0x0025] = "\x21\x33",
  [0x0026] = "\x21\x34", [0x0027] = "\x21\x35", [0x0029] = "\x21\x5e",
  [0x002a] = "\x21\x5f", [0x002b] = "\x21\x60", [0x002c] = "\x21\x61",
  [0x002d] = "\x21\x62", [0x002e] = "\x21\x63", [0x002f] = "\x22\x2d",
  [0x0030] = "\x22\x2e", [0x0031] = "\x22\x2f", [0x0032] = "\x22\x3f",
  [0x0033] = "\x22\x40", [0x0034] = "\x22\x41", [0x0035] = "\x22\x43",
  [0x0036] = "\x22\x42", [0x0038] = "\x22\x62", [0x0039] = "\x22\x6c",
  [0x003a] = "\x22\x6d", [0x003b] = "\x22\x6e",
};

/* Fullwidth ASCII variants.  The table can be created using

   egrep '^0x' CNS11643.TXT |
   awk '{ print $2, $1 }' | sort | egrep '^0xFF[0-5]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns) - 0x10000;
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xff01,
             $c < 0x100 ? $c : int($c/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643l1_from_ucs4_tab14[][2] =
{
  [0x0000] = "\x21\x2a", [0x0002] = "\x21\x6c", [0x0003] = "\x22\x63",
  [0x0004] = "\x22\x68", [0x0005] = "\x21\x6d", [0x0007] = "\x21\x3e",
  [0x0008] = "\x21\x3f", [0x0009] = "\x21\x6e", [0x000a] = "\x22\x30",
  [0x000b] = "\x21\x22", [0x000c] = "\x22\x31", [0x000d] = "\x21\x25",
  [0x000e] = "\x22\x5f", [0x000f] = "\x24\x21", [0x0010] = "\x24\x22",
  [0x0011] = "\x24\x23", [0x0012] = "\x24\x24", [0x0013] = "\x24\x25",
  [0x0014] = "\x24\x26", [0x0015] = "\x24\x27", [0x0016] = "\x24\x28",
  [0x0017] = "\x24\x29", [0x0018] = "\x24\x2a", [0x0019] = "\x21\x28",
  [0x001a] = "\x21\x27", [0x001b] = "\x22\x36", [0x001c] = "\x22\x38",
  [0x001d] = "\x22\x37", [0x001e] = "\x21\x29", [0x001f] = "\x22\x69",
  [0x0020] = "\x24\x41", [0x0021] = "\x24\x42", [0x0022] = "\x24\x43",
  [0x0023] = "\x24\x44", [0x0024] = "\x24\x45", [0x0025] = "\x24\x46",
  [0x0026] = "\x24\x47", [0x0027] = "\x24\x48", [0x0028] = "\x24\x49",
  [0x0029] = "\x24\x4a", [0x002a] = "\x24\x4b", [0x002b] = "\x24\x4c",
  [0x002c] = "\x24\x4d", [0x002d] = "\x24\x4e", [0x002e] = "\x24\x4f",
  [0x002f] = "\x24\x50", [0x0030] = "\x24\x51", [0x0031] = "\x24\x52",
  [0x0032] = "\x24\x53", [0x0033] = "\x24\x54", [0x0034] = "\x24\x55",
  [0x0035] = "\x24\x56", [0x0036] = "\x24\x57", [0x0037] = "\x24\x58",
  [0x0038] = "\x24\x59", [0x0039] = "\x24\x5a", [0x003b] = "\x22\x60",
  [0x003e] = "\x22\x25", [0x0040] = "\x24\x5b", [0x0041] = "\x24\x5c",
  [0x0042] = "\x24\x5d", [0x0043] = "\x24\x5e", [0x0044] = "\x24\x5f",
  [0x0045] = "\x24\x60", [0x0046] = "\x24\x61", [0x0047] = "\x24\x62",
  [0x0048] = "\x24\x63", [0x0049] = "\x24\x64", [0x004a] = "\x24\x65",
  [0x004b] = "\x24\x66", [0x004c] = "\x24\x67", [0x004d] = "\x24\x68",
  [0x004e] = "\x24\x69", [0x004f] = "\x24\x6a", [0x0050] = "\x24\x6b",
  [0x0051] = "\x24\x6c", [0x0052] = "\x24\x6d", [0x0053] = "\x24\x6e",
  [0x0054] = "\x24\x6f", [0x0055] = "\x24\x70", [0x0056] = "\x24\x71",
  [0x0057] = "\x24\x72", [0x0058] = "\x24\x73", [0x0059] = "\x24\x74",
  [0x005a] = "\x21\x42", [0x005b] = "\x22\x5e", [0x005c] = "\x21\x43",
};
