/* Mapping tables for Big5 handling.
   Copyright (C) 1997-1999, 2000-2002 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Ulrich Drepper <drepper@cygnus.com>, 1997.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.  */

#include <dlfcn.h>
#include <gconv.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <wchar.h>


/* Table for Big5 to UCS conversion. This is generated from the BIG5
   charmap, using the following command:

	egrep "/x../x.." BIG5 | egrep -v "^% " | sed 's/%IRREVERSIBLE%//' | \
	perl segA.pl > segA

   where segA.pl script is:

  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
print "static const uint16_t big5_to_ucs[13973] =\n{";
while (<>) {
    local($ucs, $big5, %rest) = split;
    $ucs =~ s/<U(....)>/0x$1/;
    $big5 =~ s/\/x//g;
    $big5 = hex($big5) - 0xa140;
    local($ridx) = int($big5 / 256) * 157;
    if ($big5 % 256 > 0x3e) {
      $ridx += ($big5 % 256) - 0x22;
    } else {
      $ridx += ($big5 % 256);
    }
    printf "\n " if (($n % 4) == 0);
    ++$n;
    printf " [%5d] = %s,", $ridx, $ucs;
}
print "\n};\n";

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const uint16_t big5_to_ucs[13973] =
{
  [    0] = 0x3000, [    1] = 0xFF0C, [    2] = 0x3001, [    3] = 0x3002,
  [    4] = 0xFF0E, [    5] = 0x2027, [    6] = 0xFF1B, [    7] = 0xFF1A,
  [    8] = 0xFF1F, [    9] = 0xFF01, [   10] = 0xFE30, [   11] = 0x2026,
  [   12] = 0x2025, [   13] = 0xFE50, [   14] = 0xFE51, [   15] = 0xFE52,
  [   16] = 0x00B7, [   17] = 0xFE54, [   18] = 0xFE55, [   19] = 0xFE56,
  [   20] = 0xFE57, [   21] = 0xFF5C, [   22] = 0x2013, [   23] = 0xFE31,
  [   24] = 0x2014, [   25] = 0xFE33, [   26] = 0x2574, [   27] = 0xFE34,
  [   28] = 0xFE4F, [   29] = 0xFF08, [   30] = 0xFF09, [   31] = 0xFE35,
  [   32] = 0xFE36, [   33] = 0xFF5B, [   34] = 0xFF5D, [   35] = 0xFE37,
  [   36] = 0xFE38, [   37] = 0x3014, [   38] = 0x3015, [   39] = 0xFE39,
  [   40] = 0xFE3A, [   41] = 0x3010, [   42] = 0x3011, [   43] = 0xFE3B,
  [   44] = 0xFE3C, [   45] = 0x300A, [   46] = 0x300B, [   47] = 0xFE3D,
  [   48] = 0xFE3E, [   49] = 0x3008, [   50] = 0x3009, [   51] = 0xFE3F,
  [   52] = 0xFE40, [   53] = 0x300C, [   54] = 0x300D, [   55] = 0xFE41,
  [   56] = 0xFE42, [   57] = 0x300E, [   58] = 0x300F, [   59] = 0xFE43,
  [   60] = 0xFE44, [   61] = 0xFE59, [   62] = 0xFE5A, [   63] = 0xFE5B,
  [   64] = 0xFE5C, [   65] = 0xFE5D, [   66] = 0xFE5E, [   67] = 0x2018,
  [   68] = 0x2019, [   69] = 0x201C, [   70] = 0x201D, [   71] = 0x301D,
  [   72] = 0x301E, [   73] = 0x2035, [   74] = 0x2032, [   75] = 0xFF03,
  [   76] = 0xFF06, [   77] = 0xFF0A, [   78] = 0x203B, [   79] = 0x00A7,
  [   80] = 0x3003, [   81] = 0x25CB, [   82] = 0x25CF, [   83] = 0x25B3,
  [   84] = 0x25B2, [   85] = 0x25CE, [   86] = 0x2606, [   87] = 0x2605,
  [   88] = 0x25C7, [   89] = 0x25C6, [   90] = 0x25A1, [   91] = 0x25A0,
  [   92] = 0x25BD, [   93] = 0x25BC, [   94] = 0x32A3, [   95] = 0x2105,
  [   96] = 0x00AF, [   97] = 0xFFE3, [   98] = 0xFF3F, [   99] = 0x02CD,
  [  100] = 0xFE49, [  101] = 0xFE4A, [  102] = 0xFE4D, [  103] = 0xFE4E,
  [  104] = 0xFE4B, [  105] = 0xFE4C, [  106] = 0xFE5F, [  107] = 0xFE60,
  [  108] = 0xFE61, [  109] = 0xFF0B, [  110] = 0xFF0D, [  111] = 0x00D7,
  [  112] = 0x00F7, [  113] = 0x00B1, [  114] = 0x221A, [  115] = 0xFF1C,
  [  116] = 0xFF1E, [  117] = 0xFF1D, [  118] = 0x2266, [  119] = 0x2267,
  [  120] = 0x2260, [  121] = 0x221E, [  122] = 0x2252, [  123] = 0x2261,
  [  124] = 0xFE62, [  125] = 0xFE63, [  126] = 0xFE64, [  127] = 0xFE65,
  [  128] = 0xFE66, [  129] = 0xFF5E, [  130] = 0x2229, [  131] = 0x222A,
  [  132] = 0x22A5, [  133] = 0x2220, [  134] = 0x221F, [  135] = 0x22BF,
  [  136] = 0x33D2, [  137] = 0x33D1, [  138] = 0x222B, [  139] = 0x222E,
  [  140] = 0x2235, [  141] = 0x2234, [  142] = 0x2640, [  143] = 0x2642,
  [  144] = 0x2295, [  145] = 0x2299, [  146] = 0x2191, [  147] = 0x2193,
  [  148] = 0x2190, [  149] = 0x2192, [  150] = 0x2196, [  151] = 0x2197,
  [  152] = 0x2199, [  153] = 0x2198, [  154] = 0x2225, [  155] = 0x2223,
  [  156] = 0xFF0F, [  157] = 0xFF3C, [  158] = 0x2215, [  159] = 0xFE68,
  [  160] = 0xFF04, [  161] = 0xFFE5, [  162] = 0x3012, [  163] = 0xFFE0,
  [  164] = 0xFFE1, [  165] = 0xFF05, [  166] = 0xFF20, [  167] = 0x2103,
  [  168] = 0x2109, [  169] = 0xFE69, [  170] = 0xFE6A, [  171] = 0xFE6B,
  [  172] = 0x33D5, [  173] = 0x339C, [  174] = 0x339D, [  175] = 0x339E,
  [  176] = 0x33CE, [  177] = 0x33A1, [  178] = 0x338E, [  179] = 0x338F,
  [  180] = 0x33C4, [  181] = 0x00B0, [  182] = 0x5159, [  183] = 0x515B,
  [  184] = 0x515E, [  185] = 0x515D, [  186] = 0x5161, [  187] = 0x5163,
  [  188] = 0x55E7, [  189] = 0x74E9, [  190] = 0x7CCE, [  191] = 0x2581,
  [  192] = 0x2582, [  193] = 0x2583, [  194] = 0x2584, [  195] = 0x2585,
  [  196] = 0x2586, [  197] = 0x2587, [  198] = 0x2588, [  199] = 0x258F,
  [  200] = 0x258E, [  201] = 0x258D, [  202] = 0x258C, [  203] = 0x258B,
  [  204] = 0x258A, [  205] = 0x2589, [  206] = 0x253C, [  207] = 0x2534,
  [  208] = 0x252C, [  209] = 0x2524, [  210] = 0x251C, [  211] = 0x2594,
  [  212] = 0x2500, [  213] = 0x2502, [  214] = 0x2595, [  215] = 0x250C,
  [  216] = 0x2510, [  217] = 0x2514, [  218] = 0x2518, [  219] = 0x256D,
  [  220] = 0x256E, [  221] = 0x2570, [  222] = 0x256F, [  223] = 0x2550,
  [  224] = 0x255E, [  225] = 0x256A, [  226] = 0x2561, [  227] = 0x25E2,
  [  228] = 0x25E3, [  229] = 0x25E5, [  230] = 0x25E4, [  231] = 0x2571,
  [  232] = 0x2572, [  233] = 0x2573, [  234] = 0xFF10, [  235] = 0xFF11,
  [  236] = 0xFF12, [  237] = 0xFF13, [  238] = 0xFF14, [  239] = 0xFF15,
  [  240] = 0xFF16, [  241] = 0xFF17, [  242] = 0xFF18, [  243] = 0xFF19,
  [  244] = 0x2160, [  245] = 0x2161, [  246] = 0x2162, [  247] = 0x2163,
  [  248] = 0x2164, [  249] = 0x2165, [  250] = 0x2166, [  251] = 0x2167,
  [  252] = 0x2168, [  253] = 0x2169, [  254] = 0x3021, [  255] = 0x3022,
  [  256] = 0x3023, [  257] = 0x3024, [  258] = 0x3025, [  259] = 0x3026,
  [  260] = 0x3027, [  261] = 0x3028, [  262] = 0x3029, [  263] = 0x5341,
  [  264] = 0x5344, [  265] = 0x5345, [  266] = 0xFF21, [  267] = 0xFF22,
  [  268] = 0xFF23, [  269] = 0xFF24, [  270] = 0xFF25, [  271] = 0xFF26,
  [  272] = 0xFF27, [  273] = 0xFF28, [  274] = 0xFF29, [  275] = 0xFF2A,
  [  276] = 0xFF2B, [  277] = 0xFF2C, [  278] = 0xFF2D, [  279] = 0xFF2E,
  [  280] = 0xFF2F, [  281] = 0xFF30, [  282] = 0xFF31, [  283] = 0xFF32,
  [  284] = 0xFF33, [  285] = 0xFF34, [  286] = 0xFF35, [  287] = 0xFF36,
  [  288] = 0xFF37, [  289] = 0xFF38, [  290] = 0xFF39, [  291] = 0xFF3A,
  [  292] = 0xFF41, [  293] = 0xFF42, [  294] = 0xFF43, [  295] = 0xFF44,
  [  296] = 0xFF45, [  297] = 0xFF46, [  298] = 0xFF47, [  299] = 0xFF48,
  [  300] = 0xFF49, [  301] = 0xFF4A, [  302] = 0xFF4B, [  303] = 0xFF4C,
  [  304] = 0xFF4D, [  305] = 0xFF4E, [  306] = 0xFF4F, [  307] = 0xFF50,
  [  308] = 0xFF51, [  309] = 0xFF52, [  310] = 0xFF53, [  311] = 0xFF54,
  [  312] = 0xFF55, [  313] = 0xFF56, [  314] = 0xFF57, [  315] = 0xFF58,
  [  316] = 0xFF59, [  317] = 0xFF5A, [  318] = 0x0391, [  319] = 0x0392,
  [  320] = 0x0393, [  321] = 0x0394, [  322] = 0x0395, [  323] = 0x0396,
  [  324] = 0x0397, [  325] = 0x0398, [  326] = 0x0399, [  327] = 0x039A,
  [  328] = 0x039B, [  329] = 0x039C, [  330] = 0x039D, [  331] = 0x039E,
  [  332] = 0x039F, [  333] = 0x03A0, [  334] = 0x03A1, [  335] = 0x03A3,
  [  336] = 0x03A4, [  337] = 0x03A5, [  338] = 0x03A6, [  339] = 0x03A7,
  [  340] = 0x03A8, [  341] = 0x03A9, [  342] = 0x03B1, [  343] = 0x03B2,
  [  344] = 0x03B3, [  345] = 0x03B4, [  346] = 0x03B5, [  347] = 0x03B6,
  [  348] = 0x03B7, [  349] = 0x03B8, [  350] = 0x03B9, [  351] = 0x03BA,
  [  352] = 0x03BB, [  353] = 0x03BC, [  354] = 0x03BD, [  355] = 0x03BE,
  [  356] = 0x03BF, [  357] = 0x03C0, [  358] = 0x03C1, [  359] = 0x03C3,
  [  360] = 0x03C4, [  361] = 0x03C5, [  362] = 0x03C6, [  363] = 0x03C7,
  [  364] = 0x03C8, [  365] = 0x03C9, [  366] = 0x3105, [  367] = 0x3106,
  [  368] = 0x3107, [  369] = 0x3108, [  370] = 0x3109, [  371] = 0x310A,
  [  372] = 0x310B, [  373] = 0x310C, [  374] = 0x310D, [  375] = 0x310E,
  [  376] = 0x310F, [  377] = 0x3110, [  378] = 0x3111, [  379] = 0x3112,
  [  380] = 0x3113, [  381] = 0x3114, [  382] = 0x3115, [  383] = 0x3116,
  [  384] = 0x3117, [  385] = 0x3118, [  386] = 0x3119, [  387] = 0x311A,
  [  388] = 0x311B, [  389] = 0x311C, [  390] = 0x311D, [  391] = 0x311E,
  [  392] = 0x311F, [  393] = 0x3120, [  394] = 0x3121, [  395] = 0x3122,
  [  396] = 0x3123, [  397] = 0x3124, [  398] = 0x3125, [  399] = 0x3126,
  [  400] = 0x3127, [  401] = 0x3128, [  402] = 0x3129, [  403] = 0x02D9,
  [  404] = 0x02C9, [  405] = 0x02CA, [  406] = 0x02C7, [  407] = 0x02CB,
  [  441] = 0x20AC, [  471] = 0x4E00, [  472] = 0x4E59, [  473] = 0x4E01,
  [  474] = 0x4E03, [  475] = 0x4E43, [  476] = 0x4E5D, [  477] = 0x4E86,
  [  478] = 0x4E8C, [  479] = 0x4EBA, [  480] = 0x513F, [  481] = 0x5165,
  [  482] = 0x516B, [  483] = 0x51E0, [  484] = 0x5200, [  485] = 0x5201,
  [  486] = 0x529B, [  487] = 0x5315, [  488] = 0x5341, [  489] = 0x535C,
  [  490] = 0x53C8, [  491] = 0x4E09, [  492] = 0x4E0B, [  493] = 0x4E08,
  [  494] = 0x4E0A, [  495] = 0x4E2B, [  496] = 0x4E38, [  497] = 0x51E1,
  [  498] = 0x4E45, [  499] = 0x4E48, [  500] = 0x4E5F, [  501] = 0x4E5E,
  [  502] = 0x4E8E, [  503] = 0x4EA1, [  504] = 0x5140, [  505] = 0x5203,
  [  506] = 0x52FA, [  507] = 0x5343, [  508] = 0x53C9, [  509] = 0x53E3,
  [  510] = 0x571F, [  511] = 0x58EB, [  512] = 0x5915, [  513] = 0x5927,
  [  514] = 0x5973, [  515] = 0x5B50, [  516] = 0x5B51, [  517] = 0x5B53,
  [  518] = 0x5BF8, [  519] = 0x5C0F, [  520] = 0x5C22, [  521] = 0x5C38,
  [  522] = 0x5C71, [  523] = 0x5DDD, [  524] = 0x5DE5, [  525] = 0x5DF1,
  [  526] = 0x5DF2, [  527] = 0x5DF3, [  528] = 0x5DFE, [  529] = 0x5E72,
  [  530] = 0x5EFE, [  531] = 0x5F0B, [  532] = 0x5F13, [  533] = 0x624D,
  [  534] = 0x4E11, [  535] = 0x4E10, [  536] = 0x4E0D, [  537] = 0x4E2D,
  [  538] = 0x4E30, [  539] = 0x4E39, [  540] = 0x4E4B, [  541] = 0x5C39,
  [  542] = 0x4E88, [  543] = 0x4E91, [  544] = 0x4E95, [  545] = 0x4E92,
  [  546] = 0x4E94, [  547] = 0x4EA2, [  548] = 0x4EC1, [  549] = 0x4EC0,
  [  550] = 0x4EC3, [  551] = 0x4EC6, [  552] = 0x4EC7, [  553] = 0x4ECD,
  [  554] = 0x4ECA, [  555] = 0x4ECB, [  556] = 0x4EC4, [  557] = 0x5143,
  [  558] = 0x5141, [  559] = 0x5167, [  560] = 0x516D, [  561] = 0x516E,
  [  562] = 0x516C, [  563] = 0x5197, [  564] = 0x51F6, [  565] = 0x5206,
  [  566] = 0x5207, [  567] = 0x5208, [  568] = 0x52FB, [  569] = 0x52FE,
  [  570] = 0x52FF, [  571] = 0x5316, [  572] = 0x5339, [  573] = 0x5348,
  [  574] = 0x5347, [  575] = 0x5345, [  576] = 0x535E, [  577] = 0x5384,
  [  578] = 0x53CB, [  579] = 0x53CA, [  580] = 0x53CD, [  581] = 0x58EC,
  [  582] = 0x5929, [  583] = 0x592B, [  584] = 0x592A, [  585] = 0x592D,
  [  586] = 0x5B54, [  587] = 0x5C11, [  588] = 0x5C24, [  589] = 0x5C3A,
  [  590] = 0x5C6F, [  591] = 0x5DF4, [  592] = 0x5E7B, [  593] = 0x5EFF,
  [  594] = 0x5F14, [  595] = 0x5F15, [  596] = 0x5FC3, [  597] = 0x6208,
  [  598] = 0x6236, [  599] = 0x624B, [  600] = 0x624E, [  601] = 0x652F,
  [  602] = 0x6587, [  603] = 0x6597, [  604] = 0x65A4, [  605] = 0x65B9,
  [  606] = 0x65E5, [  607] = 0x66F0, [  608] = 0x6708, [  609] = 0x6728,
  [  610] = 0x6B20, [  611] = 0x6B62, [  612] = 0x6B79, [  613] = 0x6BCB,
  [  614] = 0x6BD4, [  615] = 0x6BDB, [  616] = 0x6C0F, [  617] = 0x6C34,
  [  618] = 0x706B, [  619] = 0x722A, [  620] = 0x7236, [  621] = 0x723B,
  [  622] = 0x7247, [  623] = 0x7259, [  624] = 0x725B, [  625] = 0x72AC,
  [  626] = 0x738B, [  627] = 0x4E19, [  628] = 0x4E16, [  629] = 0x4E15,
  [  630] = 0x4E14, [  631] = 0x4E18, [  632] = 0x4E3B, [  633] = 0x4E4D,
  [  634] = 0x4E4F, [  635] = 0x4E4E, [  636] = 0x4EE5, [  637] = 0x4ED8,
  [  638] = 0x4ED4, [  639] = 0x4ED5, [  640] = 0x4ED6, [  641] = 0x4ED7,
  [  642] = 0x4EE3, [  643] = 0x4EE4, [  644] = 0x4ED9, [  645] = 0x4EDE,
  [  646] = 0x5145, [  647] = 0x5144, [  648] = 0x5189, [  649] = 0x518A,
  [  650] = 0x51AC, [  651] = 0x51F9, [  652] = 0x51FA, [  653] = 0x51F8,
  [  654] = 0x520A, [  655] = 0x52A0, [  656] = 0x529F, [  657] = 0x5305,
  [  658] = 0x5306, [  659] = 0x5317, [  660] = 0x531D, [  661] = 0x4EDF,
  [  662] = 0x534A, [  663] = 0x5349, [  664] = 0x5361, [  665] = 0x5360,
  [  666] = 0x536F, [  667] = 0x536E, [  668] = 0x53BB, [  669] = 0x53EF,
  [  670] = 0x53E4, [  671] = 0x53F3, [  672] = 0x53EC, [  673] = 0x53EE,
  [  674] = 0x53E9, [  675] = 0x53E8, [  676] = 0x53FC, [  677] = 0x53F8,
  [  678] = 0x53F5, [  679] = 0x53EB, [  680] = 0x53E6, [  681] = 0x53EA,
  [  682] = 0x53F2, [  683] = 0x53F1, [  684] = 0x53F0, [  685] = 0x53E5,
  [  686] = 0x53ED, [  687] = 0x53FB, [  688] = 0x56DB, [  689] = 0x56DA,
  [  690] = 0x5916, [  691] = 0x592E, [  692] = 0x5931, [  693] = 0x5974,
  [  694] = 0x5976, [  695] = 0x5B55, [  696] = 0x5B83, [  697] = 0x5C3C,
  [  698] = 0x5DE8, [  699] = 0x5DE7, [  700] = 0x5DE6, [  701] = 0x5E02,
  [  702] = 0x5E03, [  703] = 0x5E73, [  704] = 0x5E7C, [  705] = 0x5F01,
  [  706] = 0x5F18, [  707] = 0x5F17, [  708] = 0x5FC5, [  709] = 0x620A,
  [  710] = 0x6253, [  711] = 0x6254, [  712] = 0x6252, [  713] = 0x6251,
  [  714] = 0x65A5, [  715] = 0x65E6, [  716] = 0x672E, [  717] = 0x672C,
  [  718] = 0x672A, [  719] = 0x672B, [  720] = 0x672D, [  721] = 0x6B63,
  [  722] = 0x6BCD, [  723] = 0x6C11, [  724] = 0x6C10, [  725] = 0x6C38,
  [  726] = 0x6C41, [  727] = 0x6C40, [  728] = 0x6C3E, [  729] = 0x72AF,
  [  730] = 0x7384, [  731] = 0x7389, [  732] = 0x74DC, [  733] = 0x74E6,
  [  734] = 0x7518, [  735] = 0x751F, [  736] = 0x7528, [  737] = 0x7529,
  [  738] = 0x7530, [  739] = 0x7531, [  740] = 0x7532, [  741] = 0x7533,
  [  742] = 0x758B, [  743] = 0x767D, [  744] = 0x76AE, [  745] = 0x76BF,
  [  746] = 0x76EE, [  747] = 0x77DB, [  748] = 0x77E2, [  749] = 0x77F3,
  [  750] = 0x793A, [  751] = 0x79BE, [  752] = 0x7A74, [  753] = 0x7ACB,
  [  754] = 0x4E1E, [  755] = 0x4E1F, [  756] = 0x4E52, [  757] = 0x4E53,
  [  758] = 0x4E69, [  759] = 0x4E99, [  760] = 0x4EA4, [  761] = 0x4EA6,
  [  762] = 0x4EA5, [  763] = 0x4EFF, [  764] = 0x4F09, [  765] = 0x4F19,
  [  766] = 0x4F0A, [  767] = 0x4F15, [  768] = 0x4F0D, [  769] = 0x4F10,
  [  770] = 0x4F11, [  771] = 0x4F0F, [  772] = 0x4EF2, [  773] = 0x4EF6,
  [  774] = 0x4EFB, [  775] = 0x4EF0, [  776] = 0x4EF3, [  777] = 0x4EFD,
  [  778] = 0x4F01, [  779] = 0x4F0B, [  780] = 0x5149, [  781] = 0x5147,
  [  782] = 0x5146, [  783] = 0x5148, [  784] = 0x5168, [  785] = 0x5171,
  [  786] = 0x518D, [  787] = 0x51B0, [  788] = 0x5217, [  789] = 0x5211,
  [  790] = 0x5212, [  791] = 0x520E, [  792] = 0x5216, [  793] = 0x52A3,
  [  794] = 0x5308, [  795] = 0x5321, [  796] = 0x5320, [  797] = 0x5370,
  [  798] = 0x5371, [  799] = 0x5409, [  800] = 0x540F, [  801] = 0x540C,
  [  802] = 0x540A, [  803] = 0x5410, [  804] = 0x5401, [  805] = 0x540B,
  [  806] = 0x5404, [  807] = 0x5411, [  808] = 0x540D, [  809] = 0x5408,
  [  810] = 0x5403, [  811] = 0x540E, [  812] = 0x5406, [  813] = 0x5412,
  [  814] = 0x56E0, [  815] = 0x56DE, [  816] = 0x56DD, [  817] = 0x5733,
  [  818] = 0x5730, [  819] = 0x5728, [  820] = 0x572D, [  821] = 0x572C,
  [  822] = 0x572F, [  823] = 0x5729, [  824] = 0x5919, [  825] = 0x591A,
  [  826] = 0x5937, [  827] = 0x5938, [  828] = 0x5984, [  829] = 0x5978,
  [  830] = 0x5983, [  831] = 0x597D, [  832] = 0x5979, [  833] = 0x5982,
  [  834] = 0x5981, [  835] = 0x5B57, [  836] = 0x5B58, [  837] = 0x5B87,
  [  838] = 0x5B88, [  839] = 0x5B85, [  840] = 0x5B89, [  841] = 0x5BFA,
  [  842] = 0x5C16, [  843] = 0x5C79, [  844] = 0x5DDE, [  845] = 0x5E06,
  [  846] = 0x5E76, [  847] = 0x5E74, [  848] = 0x5F0F, [  849] = 0x5F1B,
  [  850] = 0x5FD9, [  851] = 0x5FD6, [  852] = 0x620E, [  853] = 0x620C,
  [  854] = 0x620D, [  855] = 0x6210, [  856] = 0x6263, [  857] = 0x625B,
  [  858] = 0x6258, [  859] = 0x6536, [  860] = 0x65E9, [  861] = 0x65E8,
  [  862] = 0x65EC, [  863] = 0x65ED, [  864] = 0x66F2, [  865] = 0x66F3,
  [  866] = 0x6709, [  867] = 0x673D, [  868] = 0x6734, [  869] = 0x6731,
  [  870] = 0x6735, [  871] = 0x6B21, [  872] = 0x6B64, [  873] = 0x6B7B,
  [  874] = 0x6C16, [  875] = 0x6C5D, [  876] = 0x6C57, [  877] = 0x6C59,
  [  878] = 0x6C5F, [  879] = 0x6C60, [  880] = 0x6C50, [  881] = 0x6C55,
  [  882] = 0x6C61, [  883] = 0x6C5B, [  884] = 0x6C4D, [  885] = 0x6C4E,
  [  886] = 0x7070, [  887] = 0x725F, [  888] = 0x725D, [  889] = 0x767E,
  [  890] = 0x7AF9, [  891] = 0x7C73, [  892] = 0x7CF8, [  893] = 0x7F36,
  [  894] = 0x7F8A, [  895] = 0x7FBD, [  896] = 0x8001, [  897] = 0x8003,
  [  898] = 0x800C, [  899] = 0x8012, [  900] = 0x8033, [  901] = 0x807F,
  [  902] = 0x8089, [  903] = 0x808B, [  904] = 0x808C, [  905] = 0x81E3,
  [  906] = 0x81EA, [  907] = 0x81F3, [  908] = 0x81FC, [  909] = 0x820C,
  [  910] = 0x821B, [  911] = 0x821F, [  912] = 0x826E, [  913] = 0x8272,
  [  914] = 0x827E, [  915] = 0x866B, [  916] = 0x8840, [  917] = 0x884C,
  [  918] = 0x8863, [  919] = 0x897F, [  920] = 0x9621, [  921] = 0x4E32,
  [  922] = 0x4EA8, [  923] = 0x4F4D, [  924] = 0x4F4F, [  925] = 0x4F47,
  [  926] = 0x4F57, [  927] = 0x4F5E, [  928] = 0x4F34, [  929] = 0x4F5B,
  [  930] = 0x4F55, [  931] = 0x4F30, [  932] = 0x4F50, [  933] = 0x4F51,
  [  934] = 0x4F3D, [  935] = 0x4F3A, [  936] = 0x4F38, [  937] = 0x4F43,
  [  938] = 0x4F54, [  939] = 0x4F3C, [  940] = 0x4F46, [  941] = 0x4F63,
  [  942] = 0x4F5C, [  943] = 0x4F60, [  944] = 0x4F2F, [  945] = 0x4F4E,
  [  946] = 0x4F36, [  947] = 0x4F59, [  948] = 0x4F5D, [  949] = 0x4F48,
  [  950] = 0x4F5A, [  951] = 0x514C, [  952] = 0x514B, [  953] = 0x514D,
  [  954] = 0x5175, [  955] = 0x51B6, [  956] = 0x51B7, [  957] = 0x5225,
  [  958] = 0x5224, [  959] = 0x5229, [  960] = 0x522A, [  961] = 0x5228,
  [  962] = 0x52AB, [  963] = 0x52A9, [  964] = 0x52AA, [  965] = 0x52AC,
  [  966] = 0x5323, [  967] = 0x5373, [  968] = 0x5375, [  969] = 0x541D,
  [  970] = 0x542D, [  971] = 0x541E, [  972] = 0x543E, [  973] = 0x5426,
  [  974] = 0x544E, [  975] = 0x5427, [  976] = 0x5446, [  977] = 0x5443,
  [  978] = 0x5433, [  979] = 0x5448, [  980] = 0x5442, [  981] = 0x541B,
  [  982] = 0x5429, [  983] = 0x544A, [  984] = 0x5439, [  985] = 0x543B,
  [  986] = 0x5438, [  987] = 0x542E, [  988] = 0x5435, [  989] = 0x5436,
  [  990] = 0x5420, [  991] = 0x543C, [  992] = 0x5440, [  993] = 0x5431,
  [  994] = 0x542B, [  995] = 0x541F, [  996] = 0x542C, [  997] = 0x56EA,
  [  998] = 0x56F0, [  999] = 0x56E4, [ 1000] = 0x56EB, [ 1001] = 0x574A,
  [ 1002] = 0x5751, [ 1003] = 0x5740, [ 1004] = 0x574D, [ 1005] = 0x5747,
  [ 1006] = 0x574E, [ 1007] = 0x573E, [ 1008] = 0x5750, [ 1009] = 0x574F,
  [ 1010] = 0x573B, [ 1011] = 0x58EF, [ 1012] = 0x593E, [ 1013] = 0x599D,
  [ 1014] = 0x5992, [ 1015] = 0x59A8, [ 1016] = 0x599E, [ 1017] = 0x59A3,
  [ 1018] = 0x5999, [ 1019] = 0x5996, [ 1020] = 0x598D, [ 1021] = 0x59A4,
  [ 1022] = 0x5993, [ 1023] = 0x598A, [ 1024] = 0x59A5, [ 1025] = 0x5B5D,
  [ 1026] = 0x5B5C, [ 1027] = 0x5B5A, [ 1028] = 0x5B5B, [ 1029] = 0x5B8C,
  [ 1030] = 0x5B8B, [ 1031] = 0x5B8F, [ 1032] = 0x5C2C, [ 1033] = 0x5C40,
  [ 1034] = 0x5C41, [ 1035] = 0x5C3F, [ 1036] = 0x5C3E, [ 1037] = 0x5C90,
  [ 1038] = 0x5C91, [ 1039] = 0x5C94, [ 1040] = 0x5C8C, [ 1041] = 0x5DEB,
  [ 1042] = 0x5E0C, [ 1043] = 0x5E8F, [ 1044] = 0x5E87, [ 1045] = 0x5E8A,
  [ 1046] = 0x5EF7, [ 1047] = 0x5F04, [ 1048] = 0x5F1F, [ 1049] = 0x5F64,
  [ 1050] = 0x5F62, [ 1051] = 0x5F77, [ 1052] = 0x5F79, [ 1053] = 0x5FD8,
  [ 1054] = 0x5FCC, [ 1055] = 0x5FD7, [ 1056] = 0x5FCD, [ 1057] = 0x5FF1,
  [ 1058] = 0x5FEB, [ 1059] = 0x5FF8, [ 1060] = 0x5FEA, [ 1061] = 0x6212,
  [ 1062] = 0x6211, [ 1063] = 0x6284, [ 1064] = 0x6297, [ 1065] = 0x6296,
  [ 1066] = 0x6280, [ 1067] = 0x6276, [ 1068] = 0x6289, [ 1069] = 0x626D,
  [ 1070] = 0x628A, [ 1071] = 0x627C, [ 1072] = 0x627E, [ 1073] = 0x6279,
  [ 1074] = 0x6273, [ 1075] = 0x6292, [ 1076] = 0x626F, [ 1077] = 0x6298,
  [ 1078] = 0x626E, [ 1079] = 0x6295, [ 1080] = 0x6293, [ 1081] = 0x6291,
  [ 1082] = 0x6286, [ 1083] = 0x6539, [ 1084] = 0x653B, [ 1085] = 0x6538,
  [ 1086] = 0x65F1, [ 1087] = 0x66F4, [ 1088] = 0x675F, [ 1089] = 0x674E,
  [ 1090] = 0x674F, [ 1091] = 0x6750, [ 1092] = 0x6751, [ 1093] = 0x675C,
  [ 1094] = 0x6756, [ 1095] = 0x675E, [ 1096] = 0x6749, [ 1097] = 0x6746,
  [ 1098] = 0x6760, [ 1099] = 0x6753, [ 1100] = 0x6757, [ 1101] = 0x6B65,
  [ 1102] = 0x6BCF, [ 1103] = 0x6C42, [ 1104] = 0x6C5E, [ 1105] = 0x6C99,
  [ 1106] = 0x6C81, [ 1107] = 0x6C88, [ 1108] = 0x6C89, [ 1109] = 0x6C85,
  [ 1110] = 0x6C9B, [ 1111] = 0x6C6A, [ 1112] = 0x6C7A, [ 1113] = 0x6C90,
  [ 1114] = 0x6C70, [ 1115] = 0x6C8C, [ 1116] = 0x6C68, [ 1117] = 0x6C96,
  [ 1118] = 0x6C92, [ 1119] = 0x6C7D, [ 1120] = 0x6C83, [ 1121] = 0x6C72,
  [ 1122] = 0x6C7E, [ 1123] = 0x6C74, [ 1124] = 0x6C86, [ 1125] = 0x6C76,
  [ 1126] = 0x6C8D, [ 1127] = 0x6C94, [ 1128] = 0x6C98, [ 1129] = 0x6C82,
  [ 1130] = 0x7076, [ 1131] = 0x707C, [ 1132] = 0x707D, [ 1133] = 0x7078,
  [ 1134] = 0x7262, [ 1135] = 0x7261, [ 1136] = 0x7260, [ 1137] = 0x72C4,
  [ 1138] = 0x72C2, [ 1139] = 0x7396, [ 1140] = 0x752C, [ 1141] = 0x752B,
  [ 1142] = 0x7537, [ 1143] = 0x7538, [ 1144] = 0x7682, [ 1145] = 0x76EF,
  [ 1146] = 0x77E3, [ 1147] = 0x79C1, [ 1148] = 0x79C0, [ 1149] = 0x79BF,
  [ 1150] = 0x7A76, [ 1151] = 0x7CFB, [ 1152] = 0x7F55, [ 1153] = 0x8096,
  [ 1154] = 0x8093, [ 1155] = 0x809D, [ 1156] = 0x8098, [ 1157] = 0x809B,
  [ 1158] = 0x809A, [ 1159] = 0x80B2, [ 1160] = 0x826F, [ 1161] = 0x8292,
  [ 1162] = 0x828B, [ 1163] = 0x828D, [ 1164] = 0x898B, [ 1165] = 0x89D2,
  [ 1166] = 0x8A00, [ 1167] = 0x8C37, [ 1168] = 0x8C46, [ 1169] = 0x8C55,
  [ 1170] = 0x8C9D, [ 1171] = 0x8D64, [ 1172] = 0x8D70, [ 1173] = 0x8DB3,
  [ 1174] = 0x8EAB, [ 1175] = 0x8ECA, [ 1176] = 0x8F9B, [ 1177] = 0x8FB0,
  [ 1178] = 0x8FC2, [ 1179] = 0x8FC6, [ 1180] = 0x8FC5, [ 1181] = 0x8FC4,
  [ 1182] = 0x5DE1, [ 1183] = 0x9091, [ 1184] = 0x90A2, [ 1185] = 0x90AA,
  [ 1186] = 0x90A6, [ 1187] = 0x90A3, [ 1188] = 0x9149, [ 1189] = 0x91C6,
  [ 1190] = 0x91CC, [ 1191] = 0x9632, [ 1192] = 0x962E, [ 1193] = 0x9631,
  [ 1194] = 0x962A, [ 1195] = 0x962C, [ 1196] = 0x4E26, [ 1197] = 0x4E56,
  [ 1198] = 0x4E73, [ 1199] = 0x4E8B, [ 1200] = 0x4E9B, [ 1201] = 0x4E9E,
  [ 1202] = 0x4EAB, [ 1203] = 0x4EAC, [ 1204] = 0x4F6F, [ 1205] = 0x4F9D,
  [ 1206] = 0x4F8D, [ 1207] = 0x4F73, [ 1208] = 0x4F7F, [ 1209] = 0x4F6C,
  [ 1210] = 0x4F9B, [ 1211] = 0x4F8B, [ 1212] = 0x4F86, [ 1213] = 0x4F83,
  [ 1214] = 0x4F70, [ 1215] = 0x4F75, [ 1216] = 0x4F88, [ 1217] = 0x4F69,
  [ 1218] = 0x4F7B, [ 1219] = 0x4F96, [ 1220] = 0x4F7E, [ 1221] = 0x4F8F,
  [ 1222] = 0x4F91, [ 1223] = 0x4F7A, [ 1224] = 0x5154, [ 1225] = 0x5152,
  [ 1226] = 0x5155, [ 1227] = 0x5169, [ 1228] = 0x5177, [ 1229] = 0x5176,
  [ 1230] = 0x5178, [ 1231] = 0x51BD, [ 1232] = 0x51FD, [ 1233] = 0x523B,
  [ 1234] = 0x5238, [ 1235] = 0x5237, [ 1236] = 0x523A, [ 1237] = 0x5230,
  [ 1238] = 0x522E, [ 1239] = 0x5236, [ 1240] = 0x5241, [ 1241] = 0x52BE,
  [ 1242] = 0x52BB, [ 1243] = 0x5352, [ 1244] = 0x5354, [ 1245] = 0x5353,
  [ 1246] = 0x5351, [ 1247] = 0x5366, [ 1248] = 0x5377, [ 1249] = 0x5378,
  [ 1250] = 0x5379, [ 1251] = 0x53D6, [ 1252] = 0x53D4, [ 1253] = 0x53D7,
  [ 1254] = 0x5473, [ 1255] = 0x5475, [ 1256] = 0x5496, [ 1257] = 0x5478,
  [ 1258] = 0x5495, [ 1259] = 0x5480, [ 1260] = 0x547B, [ 1261] = 0x5477,
  [ 1262] = 0x5484, [ 1263] = 0x5492, [ 1264] = 0x5486, [ 1265] = 0x547C,
  [ 1266] = 0x5490, [ 1267] = 0x5471, [ 1268] = 0x5476, [ 1269] = 0x548C,
  [ 1270] = 0x549A, [ 1271] = 0x5462, [ 1272] = 0x5468, [ 1273] = 0x548B,
  [ 1274] = 0x547D, [ 1275] = 0x548E, [ 1276] = 0x56FA, [ 1277] = 0x5783,
  [ 1278] = 0x5777, [ 1279] = 0x576A, [ 1280] = 0x5769, [ 1281] = 0x5761,
  [ 1282] = 0x5766, [ 1283] = 0x5764, [ 1284] = 0x577C, [ 1285] = 0x591C,
  [ 1286] = 0x5949, [ 1287] = 0x5947, [ 1288] = 0x5948, [ 1289] = 0x5944,
  [ 1290] = 0x5954, [ 1291] = 0x59BE, [ 1292] = 0x59BB, [ 1293] = 0x59D4,
  [ 1294] = 0x59B9, [ 1295] = 0x59AE, [ 1296] = 0x59D1, [ 1297] = 0x59C6,
  [ 1298] = 0x59D0, [ 1299] = 0x59CD, [ 1300] = 0x59CB, [ 1301] = 0x59D3,
  [ 1302] = 0x59CA, [ 1303] = 0x59AF, [ 1304] = 0x59B3, [ 1305] = 0x59D2,
  [ 1306] = 0x59C5, [ 1307] = 0x5B5F, [ 1308] = 0x5B64, [ 1309] = 0x5B63,
  [ 1310] = 0x5B97, [ 1311] = 0x5B9A, [ 1312] = 0x5B98, [ 1313] = 0x5B9C,
  [ 1314] = 0x5B99, [ 1315] = 0x5B9B, [ 1316] = 0x5C1A, [ 1317] = 0x5C48,
  [ 1318] = 0x5C45, [ 1319] = 0x5C46, [ 1320] = 0x5CB7, [ 1321] = 0x5CA1,
  [ 1322] = 0x5CB8, [ 1323] = 0x5CA9, [ 1324] = 0x5CAB, [ 1325] = 0x5CB1,
  [ 1326] = 0x5CB3, [ 1327] = 0x5E18, [ 1328] = 0x5E1A, [ 1329] = 0x5E16,
  [ 1330] = 0x5E15, [ 1331] = 0x5E1B, [ 1332] = 0x5E11, [ 1333] = 0x5E78,
  [ 1334] = 0x5E9A, [ 1335] = 0x5E97, [ 1336] = 0x5E9C, [ 1337] = 0x5E95,
  [ 1338] = 0x5E96, [ 1339] = 0x5EF6, [ 1340] = 0x5F26, [ 1341] = 0x5F27,
  [ 1342] = 0x5F29, [ 1343] = 0x5F80, [ 1344] = 0x5F81, [ 1345] = 0x5F7F,
  [ 1346] = 0x5F7C, [ 1347] = 0x5FDD, [ 1348] = 0x5FE0, [ 1349] = 0x5FFD,
  [ 1350] = 0x5FF5, [ 1351] = 0x5FFF, [ 1352] = 0x600F, [ 1353] = 0x6014,
  [ 1354] = 0x602F, [ 1355] = 0x6035, [ 1356] = 0x6016, [ 1357] = 0x602A,
  [ 1358] = 0x6015, [ 1359] = 0x6021, [ 1360] = 0x6027, [ 1361] = 0x6029,
  [ 1362] = 0x602B, [ 1363] = 0x601B, [ 1364] = 0x6216, [ 1365] = 0x6215,
  [ 1366] = 0x623F, [ 1367] = 0x623E, [ 1368] = 0x6240, [ 1369] = 0x627F,
  [ 1370] = 0x62C9, [ 1371] = 0x62CC, [ 1372] = 0x62C4, [ 1373] = 0x62BF,
  [ 1374] = 0x62C2, [ 1375] = 0x62B9, [ 1376] = 0x62D2, [ 1377] = 0x62DB,
  [ 1378] = 0x62AB, [ 1379] = 0x62D3, [ 1380] = 0x62D4, [ 1381] = 0x62CB,
  [ 1382] = 0x62C8, [ 1383] = 0x62A8, [ 1384] = 0x62BD, [ 1385] = 0x62BC,
  [ 1386] = 0x62D0, [ 1387] = 0x62D9, [ 1388] = 0x62C7, [ 1389] = 0x62CD,
  [ 1390] = 0x62B5, [ 1391] = 0x62DA, [ 1392] = 0x62B1, [ 1393] = 0x62D8,
  [ 1394] = 0x62D6, [ 1395] = 0x62D7, [ 1396] = 0x62C6, [ 1397] = 0x62AC,
  [ 1398] = 0x62CE, [ 1399] = 0x653E, [ 1400] = 0x65A7, [ 1401] = 0x65BC,
  [ 1402] = 0x65FA, [ 1403] = 0x6614, [ 1404] = 0x6613, [ 1405] = 0x660C,
  [ 1406] = 0x6606, [ 1407] = 0x6602, [ 1408] = 0x660E, [ 1409] = 0x6600,
  [ 1410] = 0x660F, [ 1411] = 0x6615, [ 1412] = 0x660A, [ 1413] = 0x6607,
  [ 1414] = 0x670D, [ 1415] = 0x670B, [ 1416] = 0x676D, [ 1417] = 0x678B,
  [ 1418] = 0x6795, [ 1419] = 0x6771, [ 1420] = 0x679C, [ 1421] = 0x6773,
  [ 1422] = 0x6777, [ 1423] = 0x6787, [ 1424] = 0x679D, [ 1425] = 0x6797,
  [ 1426] = 0x676F, [ 1427] = 0x6770, [ 1428] = 0x677F, [ 1429] = 0x6789,
  [ 1430] = 0x677E, [ 1431] = 0x6790, [ 1432] = 0x6775, [ 1433] = 0x679A,
  [ 1434] = 0x6793, [ 1435] = 0x677C, [ 1436] = 0x676A, [ 1437] = 0x6772,
  [ 1438] = 0x6B23, [ 1439] = 0x6B66, [ 1440] = 0x6B67, [ 1441] = 0x6B7F,
  [ 1442] = 0x6C13, [ 1443] = 0x6C1B, [ 1444] = 0x6CE3, [ 1445] = 0x6CE8,
  [ 1446] = 0x6CF3, [ 1447] = 0x6CB1, [ 1448] = 0x6CCC, [ 1449] = 0x6CE5,
  [ 1450] = 0x6CB3, [ 1451] = 0x6CBD, [ 1452] = 0x6CBE, [ 1453] = 0x6CBC,
  [ 1454] = 0x6CE2, [ 1455] = 0x6CAB, [ 1456] = 0x6CD5, [ 1457] = 0x6CD3,
  [ 1458] = 0x6CB8, [ 1459] = 0x6CC4, [ 1460] = 0x6CB9, [ 1461] = 0x6CC1,
  [ 1462] = 0x6CAE, [ 1463] = 0x6CD7, [ 1464] = 0x6CC5, [ 1465] = 0x6CF1,
  [ 1466] = 0x6CBF, [ 1467] = 0x6CBB, [ 1468] = 0x6CE1, [ 1469] = 0x6CDB,
  [ 1470] = 0x6CCA, [ 1471] = 0x6CAC, [ 1472] = 0x6CEF, [ 1473] = 0x6CDC,
  [ 1474] = 0x6CD6, [ 1475] = 0x6CE0, [ 1476] = 0x7095, [ 1477] = 0x708E,
  [ 1478] = 0x7092, [ 1479] = 0x708A, [ 1480] = 0x7099, [ 1481] = 0x722C,
  [ 1482] = 0x722D, [ 1483] = 0x7238, [ 1484] = 0x7248, [ 1485] = 0x7267,
  [ 1486] = 0x7269, [ 1487] = 0x72C0, [ 1488] = 0x72CE, [ 1489] = 0x72D9,
  [ 1490] = 0x72D7, [ 1491] = 0x72D0, [ 1492] = 0x73A9, [ 1493] = 0x73A8,
  [ 1494] = 0x739F, [ 1495] = 0x73AB, [ 1496] = 0x73A5, [ 1497] = 0x753D,
  [ 1498] = 0x759D, [ 1499] = 0x7599, [ 1500] = 0x759A, [ 1501] = 0x7684,
  [ 1502] = 0x76C2, [ 1503] = 0x76F2, [ 1504] = 0x76F4, [ 1505] = 0x77E5,
  [ 1506] = 0x77FD, [ 1507] = 0x793E, [ 1508] = 0x7940, [ 1509] = 0x7941,
  [ 1510] = 0x79C9, [ 1511] = 0x79C8, [ 1512] = 0x7A7A, [ 1513] = 0x7A79,
  [ 1514] = 0x7AFA, [ 1515] = 0x7CFE, [ 1516] = 0x7F54, [ 1517] = 0x7F8C,
  [ 1518] = 0x7F8B, [ 1519] = 0x8005, [ 1520] = 0x80BA, [ 1521] = 0x80A5,
  [ 1522] = 0x80A2, [ 1523] = 0x80B1, [ 1524] = 0x80A1, [ 1525] = 0x80AB,
  [ 1526] = 0x80A9, [ 1527] = 0x80B4, [ 1528] = 0x80AA, [ 1529] = 0x80AF,
  [ 1530] = 0x81E5, [ 1531] = 0x81FE, [ 1532] = 0x820D, [ 1533] = 0x82B3,
  [ 1534] = 0x829D, [ 1535] = 0x8299, [ 1536] = 0x82AD, [ 1537] = 0x82BD,
  [ 1538] = 0x829F, [ 1539] = 0x82B9, [ 1540] = 0x82B1, [ 1541] = 0x82AC,
  [ 1542] = 0x82A5, [ 1543] = 0x82AF, [ 1544] = 0x82B8, [ 1545] = 0x82A3,
  [ 1546] = 0x82B0, [ 1547] = 0x82BE, [ 1548] = 0x82B7, [ 1549] = 0x864E,
  [ 1550] = 0x8671, [ 1551] = 0x521D, [ 1552] = 0x8868, [ 1553] = 0x8ECB,
  [ 1554] = 0x8FCE, [ 1555] = 0x8FD4, [ 1556] = 0x8FD1, [ 1557] = 0x90B5,
  [ 1558] = 0x90B8, [ 1559] = 0x90B1, [ 1560] = 0x90B6, [ 1561] = 0x91C7,
  [ 1562] = 0x91D1, [ 1563] = 0x9577, [ 1564] = 0x9580, [ 1565] = 0x961C,
  [ 1566] = 0x9640, [ 1567] = 0x963F, [ 1568] = 0x963B, [ 1569] = 0x9644,
  [ 1570] = 0x9642, [ 1571] = 0x96B9, [ 1572] = 0x96E8, [ 1573] = 0x9752,
  [ 1574] = 0x975E, [ 1575] = 0x4E9F, [ 1576] = 0x4EAD, [ 1577] = 0x4EAE,
  [ 1578] = 0x4FE1, [ 1579] = 0x4FB5, [ 1580] = 0x4FAF, [ 1581] = 0x4FBF,
  [ 1582] = 0x4FE0, [ 1583] = 0x4FD1, [ 1584] = 0x4FCF, [ 1585] = 0x4FDD,
  [ 1586] = 0x4FC3, [ 1587] = 0x4FB6, [ 1588] = 0x4FD8, [ 1589] = 0x4FDF,
  [ 1590] = 0x4FCA, [ 1591] = 0x4FD7, [ 1592] = 0x4FAE, [ 1593] = 0x4FD0,
  [ 1594] = 0x4FC4, [ 1595] = 0x4FC2, [ 1596] = 0x4FDA, [ 1597] = 0x4FCE,
  [ 1598] = 0x4FDE, [ 1599] = 0x4FB7, [ 1600] = 0x5157, [ 1601] = 0x5192,
  [ 1602] = 0x5191, [ 1603] = 0x51A0, [ 1604] = 0x524E, [ 1605] = 0x5243,
  [ 1606] = 0x524A, [ 1607] = 0x524D, [ 1608] = 0x524C, [ 1609] = 0x524B,
  [ 1610] = 0x5247, [ 1611] = 0x52C7, [ 1612] = 0x52C9, [ 1613] = 0x52C3,
  [ 1614] = 0x52C1, [ 1615] = 0x530D, [ 1616] = 0x5357, [ 1617] = 0x537B,
  [ 1618] = 0x539A, [ 1619] = 0x53DB, [ 1620] = 0x54AC, [ 1621] = 0x54C0,
  [ 1622] = 0x54A8, [ 1623] = 0x54CE, [ 1624] = 0x54C9, [ 1625] = 0x54B8,
  [ 1626] = 0x54A6, [ 1627] = 0x54B3, [ 1628] = 0x54C7, [ 1629] = 0x54C2,
  [ 1630] = 0x54BD, [ 1631] = 0x54AA, [ 1632] = 0x54C1, [ 1633] = 0x54C4,
  [ 1634] = 0x54C8, [ 1635] = 0x54AF, [ 1636] = 0x54AB, [ 1637] = 0x54B1,
  [ 1638] = 0x54BB, [ 1639] = 0x54A9, [ 1640] = 0x54A7, [ 1641] = 0x54BF,
  [ 1642] = 0x56FF, [ 1643] = 0x5782, [ 1644] = 0x578B, [ 1645] = 0x57A0,
  [ 1646] = 0x57A3, [ 1647] = 0x57A2, [ 1648] = 0x57CE, [ 1649] = 0x57AE,
  [ 1650] = 0x5793, [ 1651] = 0x5955, [ 1652] = 0x5951, [ 1653] = 0x594F,
  [ 1654] = 0x594E, [ 1655] = 0x5950, [ 1656] = 0x59DC, [ 1657] = 0x59D8,
  [ 1658] = 0x59FF, [ 1659] = 0x59E3, [ 1660] = 0x59E8, [ 1661] = 0x5A03,
  [ 1662] = 0x59E5, [ 1663] = 0x59EA, [ 1664] = 0x59DA, [ 1665] = 0x59E6,
  [ 1666] = 0x5A01, [ 1667] = 0x59FB, [ 1668] = 0x5B69, [ 1669] = 0x5BA3,
  [ 1670] = 0x5BA6, [ 1671] = 0x5BA4, [ 1672] = 0x5BA2, [ 1673] = 0x5BA5,
  [ 1674] = 0x5C01, [ 1675] = 0x5C4E, [ 1676] = 0x5C4F, [ 1677] = 0x5C4D,
  [ 1678] = 0x5C4B, [ 1679] = 0x5CD9, [ 1680] = 0x5CD2, [ 1681] = 0x5DF7,
  [ 1682] = 0x5E1D, [ 1683] = 0x5E25, [ 1684] = 0x5E1F, [ 1685] = 0x5E7D,
  [ 1686] = 0x5EA0, [ 1687] = 0x5EA6, [ 1688] = 0x5EFA, [ 1689] = 0x5F08,
  [ 1690] = 0x5F2D, [ 1691] = 0x5F65, [ 1692] = 0x5F88, [ 1693] = 0x5F85,
  [ 1694] = 0x5F8A, [ 1695] = 0x5F8B, [ 1696] = 0x5F87, [ 1697] = 0x5F8C,
  [ 1698] = 0x5F89, [ 1699] = 0x6012, [ 1700] = 0x601D, [ 1701] = 0x6020,
  [ 1702] = 0x6025, [ 1703] = 0x600E, [ 1704] = 0x6028, [ 1705] = 0x604D,
  [ 1706] = 0x6070, [ 1707] = 0x6068, [ 1708] = 0x6062, [ 1709] = 0x6046,
  [ 1710] = 0x6043, [ 1711] = 0x606C, [ 1712] = 0x606B, [ 1713] = 0x606A,
  [ 1714] = 0x6064, [ 1715] = 0x6241, [ 1716] = 0x62DC, [ 1717] = 0x6316,
  [ 1718] = 0x6309, [ 1719] = 0x62FC, [ 1720] = 0x62ED, [ 1721] = 0x6301,
  [ 1722] = 0x62EE, [ 1723] = 0x62FD, [ 1724] = 0x6307, [ 1725] = 0x62F1,
  [ 1726] = 0x62F7, [ 1727] = 0x62EF, [ 1728] = 0x62EC, [ 1729] = 0x62FE,
  [ 1730] = 0x62F4, [ 1731] = 0x6311, [ 1732] = 0x6302, [ 1733] = 0x653F,
  [ 1734] = 0x6545, [ 1735] = 0x65AB, [ 1736] = 0x65BD, [ 1737] = 0x65E2,
  [ 1738] = 0x6625, [ 1739] = 0x662D, [ 1740] = 0x6620, [ 1741] = 0x6627,
  [ 1742] = 0x662F, [ 1743] = 0x661F, [ 1744] = 0x6628, [ 1745] = 0x6631,
  [ 1746] = 0x6624, [ 1747] = 0x66F7, [ 1748] = 0x67FF, [ 1749] = 0x67D3,
  [ 1750] = 0x67F1, [ 1751] = 0x67D4, [ 1752] = 0x67D0, [ 1753] = 0x67EC,
  [ 1754] = 0x67B6, [ 1755] = 0x67AF, [ 1756] = 0x67F5, [ 1757] = 0x67E9,
  [ 1758] = 0x67EF, [ 1759] = 0x67C4, [ 1760] = 0x67D1, [ 1761] = 0x67B4,
  [ 1762] = 0x67DA, [ 1763] = 0x67E5, [ 1764] = 0x67B8, [ 1765] = 0x67CF,
  [ 1766] = 0x67DE, [ 1767] = 0x67F3, [ 1768] = 0x67B0, [ 1769] = 0x67D9,
  [ 1770] = 0x67E2, [ 1771] = 0x67DD, [ 1772] = 0x67D2, [ 1773] = 0x6B6A,
  [ 1774] = 0x6B83, [ 1775] = 0x6B86, [ 1776] = 0x6BB5, [ 1777] = 0x6BD2,
  [ 1778] = 0x6BD7, [ 1779] = 0x6C1F, [ 1780] = 0x6CC9, [ 1781] = 0x6D0B,
  [ 1782] = 0x6D32, [ 1783] = 0x6D2A, [ 1784] = 0x6D41, [ 1785] = 0x6D25,
  [ 1786] = 0x6D0C, [ 1787] = 0x6D31, [ 1788] = 0x6D1E, [ 1789] = 0x6D17,
  [ 1790] = 0x6D3B, [ 1791] = 0x6D3D, [ 1792] = 0x6D3E, [ 1793] = 0x6D36,
  [ 1794] = 0x6D1B, [ 1795] = 0x6CF5, [ 1796] = 0x6D39, [ 1797] = 0x6D27,
  [ 1798] = 0x6D38, [ 1799] = 0x6D29, [ 1800] = 0x6D2E, [ 1801] = 0x6D35,
  [ 1802] = 0x6D0E, [ 1803] = 0x6D2B, [ 1804] = 0x70AB, [ 1805] = 0x70BA,
  [ 1806] = 0x70B3, [ 1807] = 0x70AC, [ 1808] = 0x70AF, [ 1809] = 0x70AD,
  [ 1810] = 0x70B8, [ 1811] = 0x70AE, [ 1812] = 0x70A4, [ 1813] = 0x7230,
  [ 1814] = 0x7272, [ 1815] = 0x726F, [ 1816] = 0x7274, [ 1817] = 0x72E9,
  [ 1818] = 0x72E0, [ 1819] = 0x72E1, [ 1820] = 0x73B7, [ 1821] = 0x73CA,
  [ 1822] = 0x73BB, [ 1823] = 0x73B2, [ 1824] = 0x73CD, [ 1825] = 0x73C0,
  [ 1826] = 0x73B3, [ 1827] = 0x751A, [ 1828] = 0x752D, [ 1829] = 0x754F,
  [ 1830] = 0x754C, [ 1831] = 0x754E, [ 1832] = 0x754B, [ 1833] = 0x75AB,
  [ 1834] = 0x75A4, [ 1835] = 0x75A5, [ 1836] = 0x75A2, [ 1837] = 0x75A3,
  [ 1838] = 0x7678, [ 1839] = 0x7686, [ 1840] = 0x7687, [ 1841] = 0x7688,
  [ 1842] = 0x76C8, [ 1843] = 0x76C6, [ 1844] = 0x76C3, [ 1845] = 0x76C5,
  [ 1846] = 0x7701, [ 1847] = 0x76F9, [ 1848] = 0x76F8, [ 1849] = 0x7709,
  [ 1850] = 0x770B, [ 1851] = 0x76FE, [ 1852] = 0x76FC, [ 1853] = 0x7707,
  [ 1854] = 0x77DC, [ 1855] = 0x7802, [ 1856] = 0x7814, [ 1857] = 0x780C,
  [ 1858] = 0x780D, [ 1859] = 0x7946, [ 1860] = 0x7949, [ 1861] = 0x7948,
  [ 1862] = 0x7947, [ 1863] = 0x79B9, [ 1864] = 0x79BA, [ 1865] = 0x79D1,
  [ 1866] = 0x79D2, [ 1867] = 0x79CB, [ 1868] = 0x7A7F, [ 1869] = 0x7A81,
  [ 1870] = 0x7AFF, [ 1871] = 0x7AFD, [ 1872] = 0x7C7D, [ 1873] = 0x7D02,
  [ 1874] = 0x7D05, [ 1875] = 0x7D00, [ 1876] = 0x7D09, [ 1877] = 0x7D07,
  [ 1878] = 0x7D04, [ 1879] = 0x7D06, [ 1880] = 0x7F38, [ 1881] = 0x7F8E,
  [ 1882] = 0x7FBF, [ 1883] = 0x8004, [ 1884] = 0x8010, [ 1885] = 0x800D,
  [ 1886] = 0x8011, [ 1887] = 0x8036, [ 1888] = 0x80D6, [ 1889] = 0x80E5,
  [ 1890] = 0x80DA, [ 1891] = 0x80C3, [ 1892] = 0x80C4, [ 1893] = 0x80CC,
  [ 1894] = 0x80E1, [ 1895] = 0x80DB, [ 1896] = 0x80CE, [ 1897] = 0x80DE,
  [ 1898] = 0x80E4, [ 1899] = 0x80DD, [ 1900] = 0x81F4, [ 1901] = 0x8222,
  [ 1902] = 0x82E7, [ 1903] = 0x8303, [ 1904] = 0x8305, [ 1905] = 0x82E3,
  [ 1906] = 0x82DB, [ 1907] = 0x82E6, [ 1908] = 0x8304, [ 1909] = 0x82E5,
  [ 1910] = 0x8302, [ 1911] = 0x8309, [ 1912] = 0x82D2, [ 1913] = 0x82D7,
  [ 1914] = 0x82F1, [ 1915] = 0x8301, [ 1916] = 0x82DC, [ 1917] = 0x82D4,
  [ 1918] = 0x82D1, [ 1919] = 0x82DE, [ 1920] = 0x82D3, [ 1921] = 0x82DF,
  [ 1922] = 0x82EF, [ 1923] = 0x8306, [ 1924] = 0x8650, [ 1925] = 0x8679,
  [ 1926] = 0x867B, [ 1927] = 0x867A, [ 1928] = 0x884D, [ 1929] = 0x886B,
  [ 1930] = 0x8981, [ 1931] = 0x89D4, [ 1932] = 0x8A08, [ 1933] = 0x8A02,
  [ 1934] = 0x8A03, [ 1935] = 0x8C9E, [ 1936] = 0x8CA0, [ 1937] = 0x8D74,
  [ 1938] = 0x8D73, [ 1939] = 0x8DB4, [ 1940] = 0x8ECD, [ 1941] = 0x8ECC,
  [ 1942] = 0x8FF0, [ 1943] = 0x8FE6, [ 1944] = 0x8FE2, [ 1945] = 0x8FEA,
  [ 1946] = 0x8FE5, [ 1947] = 0x8FED, [ 1948] = 0x8FEB, [ 1949] = 0x8FE4,
  [ 1950] = 0x8FE8, [ 1951] = 0x90CA, [ 1952] = 0x90CE, [ 1953] = 0x90C1,
  [ 1954] = 0x90C3, [ 1955] = 0x914B, [ 1956] = 0x914A, [ 1957] = 0x91CD,
  [ 1958] = 0x9582, [ 1959] = 0x9650, [ 1960] = 0x964B, [ 1961] = 0x964C,
  [ 1962] = 0x964D, [ 1963] = 0x9762, [ 1964] = 0x9769, [ 1965] = 0x97CB,
  [ 1966] = 0x97ED, [ 1967] = 0x97F3, [ 1968] = 0x9801, [ 1969] = 0x98A8,
  [ 1970] = 0x98DB, [ 1971] = 0x98DF, [ 1972] = 0x9996, [ 1973] = 0x9999,
  [ 1974] = 0x4E58, [ 1975] = 0x4EB3, [ 1976] = 0x500C, [ 1977] = 0x500D,
  [ 1978] = 0x5023, [ 1979] = 0x4FEF, [ 1980] = 0x5026, [ 1981] = 0x5025,
  [ 1982] = 0x4FF8, [ 1983] = 0x5029, [ 1984] = 0x5016, [ 1985] = 0x5006,
  [ 1986] = 0x503C, [ 1987] = 0x501F, [ 1988] = 0x501A, [ 1989] = 0x5012,
  [ 1990] = 0x5011, [ 1991] = 0x4FFA, [ 1992] = 0x5000, [ 1993] = 0x5014,
  [ 1994] = 0x5028, [ 1995] = 0x4FF1, [ 1996] = 0x5021, [ 1997] = 0x500B,
  [ 1998] = 0x5019, [ 1999] = 0x5018, [ 2000] = 0x4FF3, [ 2001] = 0x4FEE,
  [ 2002] = 0x502D, [ 2003] = 0x502A, [ 2004] = 0x4FFE, [ 2005] = 0x502B,
  [ 2006] = 0x5009, [ 2007] = 0x517C, [ 2008] = 0x51A4, [ 2009] = 0x51A5,
  [ 2010] = 0x51A2, [ 2011] = 0x51CD, [ 2012] = 0x51CC, [ 2013] = 0x51C6,
  [ 2014] = 0x51CB, [ 2015] = 0x5256, [ 2016] = 0x525C, [ 2017] = 0x5254,
  [ 2018] = 0x525B, [ 2019] = 0x525D, [ 2020] = 0x532A, [ 2021] = 0x537F,
  [ 2022] = 0x539F, [ 2023] = 0x539D, [ 2024] = 0x53DF, [ 2025] = 0x54E8,
  [ 2026] = 0x5510, [ 2027] = 0x5501, [ 2028] = 0x5537, [ 2029] = 0x54FC,
  [ 2030] = 0x54E5, [ 2031] = 0x54F2, [ 2032] = 0x5506, [ 2033] = 0x54FA,
  [ 2034] = 0x5514, [ 2035] = 0x54E9, [ 2036] = 0x54ED, [ 2037] = 0x54E1,
  [ 2038] = 0x5509, [ 2039] = 0x54EE, [ 2040] = 0x54EA, [ 2041] = 0x54E6,
  [ 2042] = 0x5527, [ 2043] = 0x5507, [ 2044] = 0x54FD, [ 2045] = 0x550F,
  [ 2046] = 0x5703, [ 2047] = 0x5704, [ 2048] = 0x57C2, [ 2049] = 0x57D4,
  [ 2050] = 0x57CB, [ 2051] = 0x57C3, [ 2052] = 0x5809, [ 2053] = 0x590F,
  [ 2054] = 0x5957, [ 2055] = 0x5958, [ 2056] = 0x595A, [ 2057] = 0x5A11,
  [ 2058] = 0x5A18, [ 2059] = 0x5A1C, [ 2060] = 0x5A1F, [ 2061] = 0x5A1B,
  [ 2062] = 0x5A13, [ 2063] = 0x59EC, [ 2064] = 0x5A20, [ 2065] = 0x5A23,
  [ 2066] = 0x5A29, [ 2067] = 0x5A25, [ 2068] = 0x5A0C, [ 2069] = 0x5A09,
  [ 2070] = 0x5B6B, [ 2071] = 0x5C58, [ 2072] = 0x5BB0, [ 2073] = 0x5BB3,
  [ 2074] = 0x5BB6, [ 2075] = 0x5BB4, [ 2076] = 0x5BAE, [ 2077] = 0x5BB5,
  [ 2078] = 0x5BB9, [ 2079] = 0x5BB8, [ 2080] = 0x5C04, [ 2081] = 0x5C51,
  [ 2082] = 0x5C55, [ 2083] = 0x5C50, [ 2084] = 0x5CED, [ 2085] = 0x5CFD,
  [ 2086] = 0x5CFB, [ 2087] = 0x5CEA, [ 2088] = 0x5CE8, [ 2089] = 0x5CF0,
  [ 2090] = 0x5CF6, [ 2091] = 0x5D01, [ 2092] = 0x5CF4, [ 2093] = 0x5DEE,
  [ 2094] = 0x5E2D, [ 2095] = 0x5E2B, [ 2096] = 0x5EAB, [ 2097] = 0x5EAD,
  [ 2098] = 0x5EA7, [ 2099] = 0x5F31, [ 2100] = 0x5F92, [ 2101] = 0x5F91,
  [ 2102] = 0x5F90, [ 2103] = 0x6059, [ 2104] = 0x6063, [ 2105] = 0x6065,
  [ 2106] = 0x6050, [ 2107] = 0x6055, [ 2108] = 0x606D, [ 2109] = 0x6069,
  [ 2110] = 0x606F, [ 2111] = 0x6084, [ 2112] = 0x609F, [ 2113] = 0x609A,
  [ 2114] = 0x608D, [ 2115] = 0x6094, [ 2116] = 0x608C, [ 2117] = 0x6085,
  [ 2118] = 0x6096, [ 2119] = 0x6247, [ 2120] = 0x62F3, [ 2121] = 0x6308,
  [ 2122] = 0x62FF, [ 2123] = 0x634E, [ 2124] = 0x633E, [ 2125] = 0x632F,
  [ 2126] = 0x6355, [ 2127] = 0x6342, [ 2128] = 0x6346, [ 2129] = 0x634F,
  [ 2130] = 0x6349, [ 2131] = 0x633A, [ 2132] = 0x6350, [ 2133] = 0x633D,
  [ 2134] = 0x632A, [ 2135] = 0x632B, [ 2136] = 0x6328, [ 2137] = 0x634D,
  [ 2138] = 0x634C, [ 2139] = 0x6548, [ 2140] = 0x6549, [ 2141] = 0x6599,
  [ 2142] = 0x65C1, [ 2143] = 0x65C5, [ 2144] = 0x6642, [ 2145] = 0x6649,
  [ 2146] = 0x664F, [ 2147] = 0x6643, [ 2148] = 0x6652, [ 2149] = 0x664C,
  [ 2150] = 0x6645, [ 2151] = 0x6641, [ 2152] = 0x66F8, [ 2153] = 0x6714,
  [ 2154] = 0x6715, [ 2155] = 0x6717, [ 2156] = 0x6821, [ 2157] = 0x6838,
  [ 2158] = 0x6848, [ 2159] = 0x6846, [ 2160] = 0x6853, [ 2161] = 0x6839,
  [ 2162] = 0x6842, [ 2163] = 0x6854, [ 2164] = 0x6829, [ 2165] = 0x68B3,
  [ 2166] = 0x6817, [ 2167] = 0x684C, [ 2168] = 0x6851, [ 2169] = 0x683D,
  [ 2170] = 0x67F4, [ 2171] = 0x6850, [ 2172] = 0x6840, [ 2173] = 0x683C,
  [ 2174] = 0x6843, [ 2175] = 0x682A, [ 2176] = 0x6845, [ 2177] = 0x6813,
  [ 2178] = 0x6818, [ 2179] = 0x6841, [ 2180] = 0x6B8A, [ 2181] = 0x6B89,
  [ 2182] = 0x6BB7, [ 2183] = 0x6C23, [ 2184] = 0x6C27, [ 2185] = 0x6C28,
  [ 2186] = 0x6C26, [ 2187] = 0x6C24, [ 2188] = 0x6CF0, [ 2189] = 0x6D6A,
  [ 2190] = 0x6D95, [ 2191] = 0x6D88, [ 2192] = 0x6D87, [ 2193] = 0x6D66,
  [ 2194] = 0x6D78, [ 2195] = 0x6D77, [ 2196] = 0x6D59, [ 2197] = 0x6D93,
  [ 2198] = 0x6D6C, [ 2199] = 0x6D89, [ 2200] = 0x6D6E, [ 2201] = 0x6D5A,
  [ 2202] = 0x6D74, [ 2203] = 0x6D69, [ 2204] = 0x6D8C, [ 2205] = 0x6D8A,
  [ 2206] = 0x6D79, [ 2207] = 0x6D85, [ 2208] = 0x6D65, [ 2209] = 0x6D94,
  [ 2210] = 0x70CA, [ 2211] = 0x70D8, [ 2212] = 0x70E4, [ 2213] = 0x70D9,
  [ 2214] = 0x70C8, [ 2215] = 0x70CF, [ 2216] = 0x7239, [ 2217] = 0x7279,
  [ 2218] = 0x72FC, [ 2219] = 0x72F9, [ 2220] = 0x72FD, [ 2221] = 0x72F8,
  [ 2222] = 0x72F7, [ 2223] = 0x7386, [ 2224] = 0x73ED, [ 2225] = 0x7409,
  [ 2226] = 0x73EE, [ 2227] = 0x73E0, [ 2228] = 0x73EA, [ 2229] = 0x73DE,
  [ 2230] = 0x7554, [ 2231] = 0x755D, [ 2232] = 0x755C, [ 2233] = 0x755A,
  [ 2234] = 0x7559, [ 2235] = 0x75BE, [ 2236] = 0x75C5, [ 2237] = 0x75C7,
  [ 2238] = 0x75B2, [ 2239] = 0x75B3, [ 2240] = 0x75BD, [ 2241] = 0x75BC,
  [ 2242] = 0x75B9, [ 2243] = 0x75C2, [ 2244] = 0x75B8, [ 2245] = 0x768B,
  [ 2246] = 0x76B0, [ 2247] = 0x76CA, [ 2248] = 0x76CD, [ 2249] = 0x76CE,
  [ 2250] = 0x7729, [ 2251] = 0x771F, [ 2252] = 0x7720, [ 2253] = 0x7728,
  [ 2254] = 0x77E9, [ 2255] = 0x7830, [ 2256] = 0x7827, [ 2257] = 0x7838,
  [ 2258] = 0x781D, [ 2259] = 0x7834, [ 2260] = 0x7837, [ 2261] = 0x7825,
  [ 2262] = 0x782D, [ 2263] = 0x7820, [ 2264] = 0x781F, [ 2265] = 0x7832,
  [ 2266] = 0x7955, [ 2267] = 0x7950, [ 2268] = 0x7960, [ 2269] = 0x795F,
  [ 2270] = 0x7956, [ 2271] = 0x795E, [ 2272] = 0x795D, [ 2273] = 0x7957,
  [ 2274] = 0x795A, [ 2275] = 0x79E4, [ 2276] = 0x79E3, [ 2277] = 0x79E7,
  [ 2278] = 0x79DF, [ 2279] = 0x79E6, [ 2280] = 0x79E9, [ 2281] = 0x79D8,
  [ 2282] = 0x7A84, [ 2283] = 0x7A88, [ 2284] = 0x7AD9, [ 2285] = 0x7B06,
  [ 2286] = 0x7B11, [ 2287] = 0x7C89, [ 2288] = 0x7D21, [ 2289] = 0x7D17,
  [ 2290] = 0x7D0B, [ 2291] = 0x7D0A, [ 2292] = 0x7D20, [ 2293] = 0x7D22,
  [ 2294] = 0x7D14, [ 2295] = 0x7D10, [ 2296] = 0x7D15, [ 2297] = 0x7D1A,
  [ 2298] = 0x7D1C, [ 2299] = 0x7D0D, [ 2300] = 0x7D19, [ 2301] = 0x7D1B,
  [ 2302] = 0x7F3A, [ 2303] = 0x7F5F, [ 2304] = 0x7F94, [ 2305] = 0x7FC5,
  [ 2306] = 0x7FC1, [ 2307] = 0x8006, [ 2308] = 0x8018, [ 2309] = 0x8015,
  [ 2310] = 0x8019, [ 2311] = 0x8017, [ 2312] = 0x803D, [ 2313] = 0x803F,
  [ 2314] = 0x80F1, [ 2315] = 0x8102, [ 2316] = 0x80F0, [ 2317] = 0x8105,
  [ 2318] = 0x80ED, [ 2319] = 0x80F4, [ 2320] = 0x8106, [ 2321] = 0x80F8,
  [ 2322] = 0x80F3, [ 2323] = 0x8108, [ 2324] = 0x80FD, [ 2325] = 0x810A,
  [ 2326] = 0x80FC, [ 2327] = 0x80EF, [ 2328] = 0x81ED, [ 2329] = 0x81EC,
  [ 2330] = 0x8200, [ 2331] = 0x8210, [ 2332] = 0x822A, [ 2333] = 0x822B,
  [ 2334] = 0x8228, [ 2335] = 0x822C, [ 2336] = 0x82BB, [ 2337] = 0x832B,
  [ 2338] = 0x8352, [ 2339] = 0x8354, [ 2340] = 0x834A, [ 2341] = 0x8338,
  [ 2342] = 0x8350, [ 2343] = 0x8349, [ 2344] = 0x8335, [ 2345] = 0x8334,
  [ 2346] = 0x834F, [ 2347] = 0x8332, [ 2348] = 0x8339, [ 2349] = 0x8336,
  [ 2350] = 0x8317, [ 2351] = 0x8340, [ 2352] = 0x8331, [ 2353] = 0x8328,
  [ 2354] = 0x8343, [ 2355] = 0x8654, [ 2356] = 0x868A, [ 2357] = 0x86AA,
  [ 2358] = 0x8693, [ 2359] = 0x86A4, [ 2360] = 0x86A9, [ 2361] = 0x868C,
  [ 2362] = 0x86A3, [ 2363] = 0x869C, [ 2364] = 0x8870, [ 2365] = 0x8877,
  [ 2366] = 0x8881, [ 2367] = 0x8882, [ 2368] = 0x887D, [ 2369] = 0x8879,
  [ 2370] = 0x8A18, [ 2371] = 0x8A10, [ 2372] = 0x8A0E, [ 2373] = 0x8A0C,
  [ 2374] = 0x8A15, [ 2375] = 0x8A0A, [ 2376] = 0x8A17, [ 2377] = 0x8A13,
  [ 2378] = 0x8A16, [ 2379] = 0x8A0F, [ 2380] = 0x8A11, [ 2381] = 0x8C48,
  [ 2382] = 0x8C7A, [ 2383] = 0x8C79, [ 2384] = 0x8CA1, [ 2385] = 0x8CA2,
  [ 2386] = 0x8D77, [ 2387] = 0x8EAC, [ 2388] = 0x8ED2, [ 2389] = 0x8ED4,
  [ 2390] = 0x8ECF, [ 2391] = 0x8FB1, [ 2392] = 0x9001, [ 2393] = 0x9006,
  [ 2394] = 0x8FF7, [ 2395] = 0x9000, [ 2396] = 0x8FFA, [ 2397] = 0x8FF4,
  [ 2398] = 0x9003, [ 2399] = 0x8FFD, [ 2400] = 0x9005, [ 2401] = 0x8FF8,
  [ 2402] = 0x9095, [ 2403] = 0x90E1, [ 2404] = 0x90DD, [ 2405] = 0x90E2,
  [ 2406] = 0x9152, [ 2407] = 0x914D, [ 2408] = 0x914C, [ 2409] = 0x91D8,
  [ 2410] = 0x91DD, [ 2411] = 0x91D7, [ 2412] = 0x91DC, [ 2413] = 0x91D9,
  [ 2414] = 0x9583, [ 2415] = 0x9662, [ 2416] = 0x9663, [ 2417] = 0x9661,
  [ 2418] = 0x965B, [ 2419] = 0x965D, [ 2420] = 0x9664, [ 2421] = 0x9658,
  [ 2422] = 0x965E, [ 2423] = 0x96BB, [ 2424] = 0x98E2, [ 2425] = 0x99AC,
  [ 2426] = 0x9AA8, [ 2427] = 0x9AD8, [ 2428] = 0x9B25, [ 2429] = 0x9B32,
  [ 2430] = 0x9B3C, [ 2431] = 0x4E7E, [ 2432] = 0x507A, [ 2433] = 0x507D,
  [ 2434] = 0x505C, [ 2435] = 0x5047, [ 2436] = 0x5043, [ 2437] = 0x504C,
  [ 2438] = 0x505A, [ 2439] = 0x5049, [ 2440] = 0x5065, [ 2441] = 0x5076,
  [ 2442] = 0x504E, [ 2443] = 0x5055, [ 2444] = 0x5075, [ 2445] = 0x5074,
  [ 2446] = 0x5077, [ 2447] = 0x504F, [ 2448] = 0x500F, [ 2449] = 0x506F,
  [ 2450] = 0x506D, [ 2451] = 0x515C, [ 2452] = 0x5195, [ 2453] = 0x51F0,
  [ 2454] = 0x526A, [ 2455] = 0x526F, [ 2456] = 0x52D2, [ 2457] = 0x52D9,
  [ 2458] = 0x52D8, [ 2459] = 0x52D5, [ 2460] = 0x5310, [ 2461] = 0x530F,
  [ 2462] = 0x5319, [ 2463] = 0x533F, [ 2464] = 0x5340, [ 2465] = 0x533E,
  [ 2466] = 0x53C3, [ 2467] = 0x66FC, [ 2468] = 0x5546, [ 2469] = 0x556A,
  [ 2470] = 0x5566, [ 2471] = 0x5544, [ 2472] = 0x555E, [ 2473] = 0x5561,
  [ 2474] = 0x5543, [ 2475] = 0x554A, [ 2476] = 0x5531, [ 2477] = 0x5556,
  [ 2478] = 0x554F, [ 2479] = 0x5555, [ 2480] = 0x552F, [ 2481] = 0x5564,
  [ 2482] = 0x5538, [ 2483] = 0x552E, [ 2484] = 0x555C, [ 2485] = 0x552C,
  [ 2486] = 0x5563, [ 2487] = 0x5533, [ 2488] = 0x5541, [ 2489] = 0x5557,
  [ 2490] = 0x5708, [ 2491] = 0x570B, [ 2492] = 0x5709, [ 2493] = 0x57DF,
  [ 2494] = 0x5805, [ 2495] = 0x580A, [ 2496] = 0x5806, [ 2497] = 0x57E0,
  [ 2498] = 0x57E4, [ 2499] = 0x57FA, [ 2500] = 0x5802, [ 2501] = 0x5835,
  [ 2502] = 0x57F7, [ 2503] = 0x57F9, [ 2504] = 0x5920, [ 2505] = 0x5962,
  [ 2506] = 0x5A36, [ 2507] = 0x5A41, [ 2508] = 0x5A49, [ 2509] = 0x5A66,
  [ 2510] = 0x5A6A, [ 2511] = 0x5A40, [ 2512] = 0x5A3C, [ 2513] = 0x5A62,
  [ 2514] = 0x5A5A, [ 2515] = 0x5A46, [ 2516] = 0x5A4A, [ 2517] = 0x5B70,
  [ 2518] = 0x5BC7, [ 2519] = 0x5BC5, [ 2520] = 0x5BC4, [ 2521] = 0x5BC2,
  [ 2522] = 0x5BBF, [ 2523] = 0x5BC6, [ 2524] = 0x5C09, [ 2525] = 0x5C08,
  [ 2526] = 0x5C07, [ 2527] = 0x5C60, [ 2528] = 0x5C5C, [ 2529] = 0x5C5D,
  [ 2530] = 0x5D07, [ 2531] = 0x5D06, [ 2532] = 0x5D0E, [ 2533] = 0x5D1B,
  [ 2534] = 0x5D16, [ 2535] = 0x5D22, [ 2536] = 0x5D11, [ 2537] = 0x5D29,
  [ 2538] = 0x5D14, [ 2539] = 0x5D19, [ 2540] = 0x5D24, [ 2541] = 0x5D27,
  [ 2542] = 0x5D17, [ 2543] = 0x5DE2, [ 2544] = 0x5E38, [ 2545] = 0x5E36,
  [ 2546] = 0x5E33, [ 2547] = 0x5E37, [ 2548] = 0x5EB7, [ 2549] = 0x5EB8,
  [ 2550] = 0x5EB6, [ 2551] = 0x5EB5, [ 2552] = 0x5EBE, [ 2553] = 0x5F35,
  [ 2554] = 0x5F37, [ 2555] = 0x5F57, [ 2556] = 0x5F6C, [ 2557] = 0x5F69,
  [ 2558] = 0x5F6B, [ 2559] = 0x5F97, [ 2560] = 0x5F99, [ 2561] = 0x5F9E,
  [ 2562] = 0x5F98, [ 2563] = 0x5FA1, [ 2564] = 0x5FA0, [ 2565] = 0x5F9C,
  [ 2566] = 0x607F, [ 2567] = 0x60A3, [ 2568] = 0x6089, [ 2569] = 0x60A0,
  [ 2570] = 0x60A8, [ 2571] = 0x60CB, [ 2572] = 0x60B4, [ 2573] = 0x60E6,
  [ 2574] = 0x60BD, [ 2575] = 0x60C5, [ 2576] = 0x60BB, [ 2577] = 0x60B5,
  [ 2578] = 0x60DC, [ 2579] = 0x60BC, [ 2580] = 0x60D8, [ 2581] = 0x60D5,
  [ 2582] = 0x60C6, [ 2583] = 0x60DF, [ 2584] = 0x60B8, [ 2585] = 0x60DA,
  [ 2586] = 0x60C7, [ 2587] = 0x621A, [ 2588] = 0x621B, [ 2589] = 0x6248,
  [ 2590] = 0x63A0, [ 2591] = 0x63A7, [ 2592] = 0x6372, [ 2593] = 0x6396,
  [ 2594] = 0x63A2, [ 2595] = 0x63A5, [ 2596] = 0x6377, [ 2597] = 0x6367,
  [ 2598] = 0x6398, [ 2599] = 0x63AA, [ 2600] = 0x6371, [ 2601] = 0x63A9,
  [ 2602] = 0x6389, [ 2603] = 0x6383, [ 2604] = 0x639B, [ 2605] = 0x636B,
  [ 2606] = 0x63A8, [ 2607] = 0x6384, [ 2608] = 0x6388, [ 2609] = 0x6399,
  [ 2610] = 0x63A1, [ 2611] = 0x63AC, [ 2612] = 0x6392, [ 2613] = 0x638F,
  [ 2614] = 0x6380, [ 2615] = 0x637B, [ 2616] = 0x6369, [ 2617] = 0x6368,
  [ 2618] = 0x637A, [ 2619] = 0x655D, [ 2620] = 0x6556, [ 2621] = 0x6551,
  [ 2622] = 0x6559, [ 2623] = 0x6557, [ 2624] = 0x555F, [ 2625] = 0x654F,
  [ 2626] = 0x6558, [ 2627] = 0x6555, [ 2628] = 0x6554, [ 2629] = 0x659C,
  [ 2630] = 0x659B, [ 2631] = 0x65AC, [ 2632] = 0x65CF, [ 2633] = 0x65CB,
  [ 2634] = 0x65CC, [ 2635] = 0x65CE, [ 2636] = 0x665D, [ 2637] = 0x665A,
  [ 2638] = 0x6664, [ 2639] = 0x6668, [ 2640] = 0x6666, [ 2641] = 0x665E,
  [ 2642] = 0x66F9, [ 2643] = 0x52D7, [ 2644] = 0x671B, [ 2645] = 0x6881,
  [ 2646] = 0x68AF, [ 2647] = 0x68A2, [ 2648] = 0x6893, [ 2649] = 0x68B5,
  [ 2650] = 0x687F, [ 2651] = 0x6876, [ 2652] = 0x68B1, [ 2653] = 0x68A7,
  [ 2654] = 0x6897, [ 2655] = 0x68B0, [ 2656] = 0x6883, [ 2657] = 0x68C4,
  [ 2658] = 0x68AD, [ 2659] = 0x6886, [ 2660] = 0x6885, [ 2661] = 0x6894,
  [ 2662] = 0x689D, [ 2663] = 0x68A8, [ 2664] = 0x689F, [ 2665] = 0x68A1,
  [ 2666] = 0x6882, [ 2667] = 0x6B32, [ 2668] = 0x6BBA, [ 2669] = 0x6BEB,
  [ 2670] = 0x6BEC, [ 2671] = 0x6C2B, [ 2672] = 0x6D8E, [ 2673] = 0x6DBC,
  [ 2674] = 0x6DF3, [ 2675] = 0x6DD9, [ 2676] = 0x6DB2, [ 2677] = 0x6DE1,
  [ 2678] = 0x6DCC, [ 2679] = 0x6DE4, [ 2680] = 0x6DFB, [ 2681] = 0x6DFA,
  [ 2682] = 0x6E05, [ 2683] = 0x6DC7, [ 2684] = 0x6DCB, [ 2685] = 0x6DAF,
  [ 2686] = 0x6DD1, [ 2687] = 0x6DAE, [ 2688] = 0x6DDE, [ 2689] = 0x6DF9,
  [ 2690] = 0x6DB8, [ 2691] = 0x6DF7, [ 2692] = 0x6DF5, [ 2693] = 0x6DC5,
  [ 2694] = 0x6DD2, [ 2695] = 0x6E1A, [ 2696] = 0x6DB5, [ 2697] = 0x6DDA,
  [ 2698] = 0x6DEB, [ 2699] = 0x6DD8, [ 2700] = 0x6DEA, [ 2701] = 0x6DF1,
  [ 2702] = 0x6DEE, [ 2703] = 0x6DE8, [ 2704] = 0x6DC6, [ 2705] = 0x6DC4,
  [ 2706] = 0x6DAA, [ 2707] = 0x6DEC, [ 2708] = 0x6DBF, [ 2709] = 0x6DE6,
  [ 2710] = 0x70F9, [ 2711] = 0x7109, [ 2712] = 0x710A, [ 2713] = 0x70FD,
  [ 2714] = 0x70EF, [ 2715] = 0x723D, [ 2716] = 0x727D, [ 2717] = 0x7281,
  [ 2718] = 0x731C, [ 2719] = 0x731B, [ 2720] = 0x7316, [ 2721] = 0x7313,
  [ 2722] = 0x7319, [ 2723] = 0x7387, [ 2724] = 0x7405, [ 2725] = 0x740A,
  [ 2726] = 0x7403, [ 2727] = 0x7406, [ 2728] = 0x73FE, [ 2729] = 0x740D,
  [ 2730] = 0x74E0, [ 2731] = 0x74F6, [ 2732] = 0x74F7, [ 2733] = 0x751C,
  [ 2734] = 0x7522, [ 2735] = 0x7565, [ 2736] = 0x7566, [ 2737] = 0x7562,
  [ 2738] = 0x7570, [ 2739] = 0x758F, [ 2740] = 0x75D4, [ 2741] = 0x75D5,
  [ 2742] = 0x75B5, [ 2743] = 0x75CA, [ 2744] = 0x75CD, [ 2745] = 0x768E,
  [ 2746] = 0x76D4, [ 2747] = 0x76D2, [ 2748] = 0x76DB, [ 2749] = 0x7737,
  [ 2750] = 0x773E, [ 2751] = 0x773C, [ 2752] = 0x7736, [ 2753] = 0x7738,
  [ 2754] = 0x773A, [ 2755] = 0x786B, [ 2756] = 0x7843, [ 2757] = 0x784E,
  [ 2758] = 0x7965, [ 2759] = 0x7968, [ 2760] = 0x796D, [ 2761] = 0x79FB,
  [ 2762] = 0x7A92, [ 2763] = 0x7A95, [ 2764] = 0x7B20, [ 2765] = 0x7B28,
  [ 2766] = 0x7B1B, [ 2767] = 0x7B2C, [ 2768] = 0x7B26, [ 2769] = 0x7B19,
  [ 2770] = 0x7B1E, [ 2771] = 0x7B2E, [ 2772] = 0x7C92, [ 2773] = 0x7C97,
  [ 2774] = 0x7C95, [ 2775] = 0x7D46, [ 2776] = 0x7D43, [ 2777] = 0x7D71,
  [ 2778] = 0x7D2E, [ 2779] = 0x7D39, [ 2780] = 0x7D3C, [ 2781] = 0x7D40,
  [ 2782] = 0x7D30, [ 2783] = 0x7D33, [ 2784] = 0x7D44, [ 2785] = 0x7D2F,
  [ 2786] = 0x7D42, [ 2787] = 0x7D32, [ 2788] = 0x7D31, [ 2789] = 0x7F3D,
  [ 2790] = 0x7F9E, [ 2791] = 0x7F9A, [ 2792] = 0x7FCC, [ 2793] = 0x7FCE,
  [ 2794] = 0x7FD2, [ 2795] = 0x801C, [ 2796] = 0x804A, [ 2797] = 0x8046,
  [ 2798] = 0x812F, [ 2799] = 0x8116, [ 2800] = 0x8123, [ 2801] = 0x812B,
  [ 2802] = 0x8129, [ 2803] = 0x8130, [ 2804] = 0x8124, [ 2805] = 0x8202,
  [ 2806] = 0x8235, [ 2807] = 0x8237, [ 2808] = 0x8236, [ 2809] = 0x8239,
  [ 2810] = 0x838E, [ 2811] = 0x839E, [ 2812] = 0x8398, [ 2813] = 0x8378,
  [ 2814] = 0x83A2, [ 2815] = 0x8396, [ 2816] = 0x83BD, [ 2817] = 0x83AB,
  [ 2818] = 0x8392, [ 2819] = 0x838A, [ 2820] = 0x8393, [ 2821] = 0x8389,
  [ 2822] = 0x83A0, [ 2823] = 0x8377, [ 2824] = 0x837B, [ 2825] = 0x837C,
  [ 2826] = 0x8386, [ 2827] = 0x83A7, [ 2828] = 0x8655, [ 2829] = 0x5F6A,
  [ 2830] = 0x86C7, [ 2831] = 0x86C0, [ 2832] = 0x86B6, [ 2833] = 0x86C4,
  [ 2834] = 0x86B5, [ 2835] = 0x86C6, [ 2836] = 0x86CB, [ 2837] = 0x86B1,
  [ 2838] = 0x86AF, [ 2839] = 0x86C9, [ 2840] = 0x8853, [ 2841] = 0x889E,
  [ 2842] = 0x8888, [ 2843] = 0x88AB, [ 2844] = 0x8892, [ 2845] = 0x8896,
  [ 2846] = 0x888D, [ 2847] = 0x888B, [ 2848] = 0x8993, [ 2849] = 0x898F,
  [ 2850] = 0x8A2A, [ 2851] = 0x8A1D, [ 2852] = 0x8A23, [ 2853] = 0x8A25,
  [ 2854] = 0x8A31, [ 2855] = 0x8A2D, [ 2856] = 0x8A1F, [ 2857] = 0x8A1B,
  [ 2858] = 0x8A22, [ 2859] = 0x8C49, [ 2860] = 0x8C5A, [ 2861] = 0x8CA9,
  [ 2862] = 0x8CAC, [ 2863] = 0x8CAB, [ 2864] = 0x8CA8, [ 2865] = 0x8CAA,
  [ 2866] = 0x8CA7, [ 2867] = 0x8D67, [ 2868] = 0x8D66, [ 2869] = 0x8DBE,
  [ 2870] = 0x8DBA, [ 2871] = 0x8EDB, [ 2872] = 0x8EDF, [ 2873] = 0x9019,
  [ 2874] = 0x900D, [ 2875] = 0x901A, [ 2876] = 0x9017, [ 2877] = 0x9023,
  [ 2878] = 0x901F, [ 2879] = 0x901D, [ 2880] = 0x9010, [ 2881] = 0x9015,
  [ 2882] = 0x901E, [ 2883] = 0x9020, [ 2884] = 0x900F, [ 2885] = 0x9022,
  [ 2886] = 0x9016, [ 2887] = 0x901B, [ 2888] = 0x9014, [ 2889] = 0x90E8,
  [ 2890] = 0x90ED, [ 2891] = 0x90FD, [ 2892] = 0x9157, [ 2893] = 0x91CE,
  [ 2894] = 0x91F5, [ 2895] = 0x91E6, [ 2896] = 0x91E3, [ 2897] = 0x91E7,
  [ 2898] = 0x91ED, [ 2899] = 0x91E9, [ 2900] = 0x9589, [ 2901] = 0x966A,
  [ 2902] = 0x9675, [ 2903] = 0x9673, [ 2904] = 0x9678, [ 2905] = 0x9670,
  [ 2906] = 0x9674, [ 2907] = 0x9676, [ 2908] = 0x9677, [ 2909] = 0x966C,
  [ 2910] = 0x96C0, [ 2911] = 0x96EA, [ 2912] = 0x96E9, [ 2913] = 0x7AE0,
  [ 2914] = 0x7ADF, [ 2915] = 0x9802, [ 2916] = 0x9803, [ 2917] = 0x9B5A,
  [ 2918] = 0x9CE5, [ 2919] = 0x9E75, [ 2920] = 0x9E7F, [ 2921] = 0x9EA5,
  [ 2922] = 0x9EBB, [ 2923] = 0x50A2, [ 2924] = 0x508D, [ 2925] = 0x5085,
  [ 2926] = 0x5099, [ 2927] = 0x5091, [ 2928] = 0x5080, [ 2929] = 0x5096,
  [ 2930] = 0x5098, [ 2931] = 0x509A, [ 2932] = 0x6700, [ 2933] = 0x51F1,
  [ 2934] = 0x5272, [ 2935] = 0x5274, [ 2936] = 0x5275, [ 2937] = 0x5269,
  [ 2938] = 0x52DE, [ 2939] = 0x52DD, [ 2940] = 0x52DB, [ 2941] = 0x535A,
  [ 2942] = 0x53A5, [ 2943] = 0x557B, [ 2944] = 0x5580, [ 2945] = 0x55A7,
  [ 2946] = 0x557C, [ 2947] = 0x558A, [ 2948] = 0x559D, [ 2949] = 0x5598,
  [ 2950] = 0x5582, [ 2951] = 0x559C, [ 2952] = 0x55AA, [ 2953] = 0x5594,
  [ 2954] = 0x5587, [ 2955] = 0x558B, [ 2956] = 0x5583, [ 2957] = 0x55B3,
  [ 2958] = 0x55AE, [ 2959] = 0x559F, [ 2960] = 0x553E, [ 2961] = 0x55B2,
  [ 2962] = 0x559A, [ 2963] = 0x55BB, [ 2964] = 0x55AC, [ 2965] = 0x55B1,
  [ 2966] = 0x557E, [ 2967] = 0x5589, [ 2968] = 0x55AB, [ 2969] = 0x5599,
  [ 2970] = 0x570D, [ 2971] = 0x582F, [ 2972] = 0x582A, [ 2973] = 0x5834,
  [ 2974] = 0x5824, [ 2975] = 0x5830, [ 2976] = 0x5831, [ 2977] = 0x5821,
  [ 2978] = 0x581D, [ 2979] = 0x5820, [ 2980] = 0x58F9, [ 2981] = 0x58FA,
  [ 2982] = 0x5960, [ 2983] = 0x5A77, [ 2984] = 0x5A9A, [ 2985] = 0x5A7F,
  [ 2986] = 0x5A92, [ 2987] = 0x5A9B, [ 2988] = 0x5AA7, [ 2989] = 0x5B73,
  [ 2990] = 0x5B71, [ 2991] = 0x5BD2, [ 2992] = 0x5BCC, [ 2993] = 0x5BD3,
  [ 2994] = 0x5BD0, [ 2995] = 0x5C0A, [ 2996] = 0x5C0B, [ 2997] = 0x5C31,
  [ 2998] = 0x5D4C, [ 2999] = 0x5D50, [ 3000] = 0x5D34, [ 3001] = 0x5D47,
  [ 3002] = 0x5DFD, [ 3003] = 0x5E45, [ 3004] = 0x5E3D, [ 3005] = 0x5E40,
  [ 3006] = 0x5E43, [ 3007] = 0x5E7E, [ 3008] = 0x5ECA, [ 3009] = 0x5EC1,
  [ 3010] = 0x5EC2, [ 3011] = 0x5EC4, [ 3012] = 0x5F3C, [ 3013] = 0x5F6D,
  [ 3014] = 0x5FA9, [ 3015] = 0x5FAA, [ 3016] = 0x5FA8, [ 3017] = 0x60D1,
  [ 3018] = 0x60E1, [ 3019] = 0x60B2, [ 3020] = 0x60B6, [ 3021] = 0x60E0,
  [ 3022] = 0x611C, [ 3023] = 0x6123, [ 3024] = 0x60FA, [ 3025] = 0x6115,
  [ 3026] = 0x60F0, [ 3027] = 0x60FB, [ 3028] = 0x60F4, [ 3029] = 0x6168,
  [ 3030] = 0x60F1, [ 3031] = 0x610E, [ 3032] = 0x60F6, [ 3033] = 0x6109,
  [ 3034] = 0x6100, [ 3035] = 0x6112, [ 3036] = 0x621F, [ 3037] = 0x6249,
  [ 3038] = 0x63A3, [ 3039] = 0x638C, [ 3040] = 0x63CF, [ 3041] = 0x63C0,
  [ 3042] = 0x63E9, [ 3043] = 0x63C9, [ 3044] = 0x63C6, [ 3045] = 0x63CD,
  [ 3046] = 0x63D2, [ 3047] = 0x63E3, [ 3048] = 0x63D0, [ 3049] = 0x63E1,
  [ 3050] = 0x63D6, [ 3051] = 0x63ED, [ 3052] = 0x63EE, [ 3053] = 0x6376,
  [ 3054] = 0x63F4, [ 3055] = 0x63EA, [ 3056] = 0x63DB, [ 3057] = 0x6452,
  [ 3058] = 0x63DA, [ 3059] = 0x63F9, [ 3060] = 0x655E, [ 3061] = 0x6566,
  [ 3062] = 0x6562, [ 3063] = 0x6563, [ 3064] = 0x6591, [ 3065] = 0x6590,
  [ 3066] = 0x65AF, [ 3067] = 0x666E, [ 3068] = 0x6670, [ 3069] = 0x6674,
  [ 3070] = 0x6676, [ 3071] = 0x666F, [ 3072] = 0x6691, [ 3073] = 0x667A,
  [ 3074] = 0x667E, [ 3075] = 0x6677, [ 3076] = 0x66FE, [ 3077] = 0x66FF,
  [ 3078] = 0x671F, [ 3079] = 0x671D, [ 3080] = 0x68FA, [ 3081] = 0x68D5,
  [ 3082] = 0x68E0, [ 3083] = 0x68D8, [ 3084] = 0x68D7, [ 3085] = 0x6905,
  [ 3086] = 0x68DF, [ 3087] = 0x68F5, [ 3088] = 0x68EE, [ 3089] = 0x68E7,
  [ 3090] = 0x68F9, [ 3091] = 0x68D2, [ 3092] = 0x68F2, [ 3093] = 0x68E3,
  [ 3094] = 0x68CB, [ 3095] = 0x68CD, [ 3096] = 0x690D, [ 3097] = 0x6912,
  [ 3098] = 0x690E, [ 3099] = 0x68C9, [ 3100] = 0x68DA, [ 3101] = 0x696E,
  [ 3102] = 0x68FB, [ 3103] = 0x6B3E, [ 3104] = 0x6B3A, [ 3105] = 0x6B3D,
  [ 3106] = 0x6B98, [ 3107] = 0x6B96, [ 3108] = 0x6BBC, [ 3109] = 0x6BEF,
  [ 3110] = 0x6C2E, [ 3111] = 0x6C2F, [ 3112] = 0x6C2C, [ 3113] = 0x6E2F,
  [ 3114] = 0x6E38, [ 3115] = 0x6E54, [ 3116] = 0x6E21, [ 3117] = 0x6E32,
  [ 3118] = 0x6E67, [ 3119] = 0x6E4A, [ 3120] = 0x6E20, [ 3121] = 0x6E25,
  [ 3122] = 0x6E23, [ 3123] = 0x6E1B, [ 3124] = 0x6E5B, [ 3125] = 0x6E58,
  [ 3126] = 0x6E24, [ 3127] = 0x6E56, [ 3128] = 0x6E6E, [ 3129] = 0x6E2D,
  [ 3130] = 0x6E26, [ 3131] = 0x6E6F, [ 3132] = 0x6E34, [ 3133] = 0x6E4D,
  [ 3134] = 0x6E3A, [ 3135] = 0x6E2C, [ 3136] = 0x6E43, [ 3137] = 0x6E1D,
  [ 3138] = 0x6E3E, [ 3139] = 0x6ECB, [ 3140] = 0x6E89, [ 3141] = 0x6E19,
  [ 3142] = 0x6E4E, [ 3143] = 0x6E63, [ 3144] = 0x6E44, [ 3145] = 0x6E72,
  [ 3146] = 0x6E69, [ 3147] = 0x6E5F, [ 3148] = 0x7119, [ 3149] = 0x711A,
  [ 3150] = 0x7126, [ 3151] = 0x7130, [ 3152] = 0x7121, [ 3153] = 0x7136,
  [ 3154] = 0x716E, [ 3155] = 0x711C, [ 3156] = 0x724C, [ 3157] = 0x7284,
  [ 3158] = 0x7280, [ 3159] = 0x7336, [ 3160] = 0x7325, [ 3161] = 0x7334,
  [ 3162] = 0x7329, [ 3163] = 0x743A, [ 3164] = 0x742A, [ 3165] = 0x7433,
  [ 3166] = 0x7422, [ 3167] = 0x7425, [ 3168] = 0x7435, [ 3169] = 0x7436,
  [ 3170] = 0x7434, [ 3171] = 0x742F, [ 3172] = 0x741B, [ 3173] = 0x7426,
  [ 3174] = 0x7428, [ 3175] = 0x7525, [ 3176] = 0x7526, [ 3177] = 0x756B,
  [ 3178] = 0x756A, [ 3179] = 0x75E2, [ 3180] = 0x75DB, [ 3181] = 0x75E3,
  [ 3182] = 0x75D9, [ 3183] = 0x75D8, [ 3184] = 0x75DE, [ 3185] = 0x75E0,
  [ 3186] = 0x767B, [ 3187] = 0x767C, [ 3188] = 0x7696, [ 3189] = 0x7693,
  [ 3190] = 0x76B4, [ 3191] = 0x76DC, [ 3192] = 0x774F, [ 3193] = 0x77ED,
  [ 3194] = 0x785D, [ 3195] = 0x786C, [ 3196] = 0x786F, [ 3197] = 0x7A0D,
  [ 3198] = 0x7A08, [ 3199] = 0x7A0B, [ 3200] = 0x7A05, [ 3201] = 0x7A00,
  [ 3202] = 0x7A98, [ 3203] = 0x7A97, [ 3204] = 0x7A96, [ 3205] = 0x7AE5,
  [ 3206] = 0x7AE3, [ 3207] = 0x7B49, [ 3208] = 0x7B56, [ 3209] = 0x7B46,
  [ 3210] = 0x7B50, [ 3211] = 0x7B52, [ 3212] = 0x7B54, [ 3213] = 0x7B4D,
  [ 3214] = 0x7B4B, [ 3215] = 0x7B4F, [ 3216] = 0x7B51, [ 3217] = 0x7C9F,
  [ 3218] = 0x7CA5, [ 3219] = 0x7D5E, [ 3220] = 0x7D50, [ 3221] = 0x7D68,
  [ 3222] = 0x7D55, [ 3223] = 0x7D2B, [ 3224] = 0x7D6E, [ 3225] = 0x7D72,
  [ 3226] = 0x7D61, [ 3227] = 0x7D66, [ 3228] = 0x7D62, [ 3229] = 0x7D70,
  [ 3230] = 0x7D73, [ 3231] = 0x5584, [ 3232] = 0x7FD4, [ 3233] = 0x7FD5,
  [ 3234] = 0x800B, [ 3235] = 0x8052, [ 3236] = 0x8085, [ 3237] = 0x8155,
  [ 3238] = 0x8154, [ 3239] = 0x814B, [ 3240] = 0x8151, [ 3241] = 0x814E,
  [ 3242] = 0x8139, [ 3243] = 0x8146, [ 3244] = 0x813E, [ 3245] = 0x814C,
  [ 3246] = 0x8153, [ 3247] = 0x8174, [ 3248] = 0x8212, [ 3249] = 0x821C,
  [ 3250] = 0x83E9, [ 3251] = 0x8403, [ 3252] = 0x83F8, [ 3253] = 0x840D,
  [ 3254] = 0x83E0, [ 3255] = 0x83C5, [ 3256] = 0x840B, [ 3257] = 0x83C1,
  [ 3258] = 0x83EF, [ 3259] = 0x83F1, [ 3260] = 0x83F4, [ 3261] = 0x8457,
  [ 3262] = 0x840A, [ 3263] = 0x83F0, [ 3264] = 0x840C, [ 3265] = 0x83CC,
  [ 3266] = 0x83FD, [ 3267] = 0x83F2, [ 3268] = 0x83CA, [ 3269] = 0x8438,
  [ 3270] = 0x840E, [ 3271] = 0x8404, [ 3272] = 0x83DC, [ 3273] = 0x8407,
  [ 3274] = 0x83D4, [ 3275] = 0x83DF, [ 3276] = 0x865B, [ 3277] = 0x86DF,
  [ 3278] = 0x86D9, [ 3279] = 0x86ED, [ 3280] = 0x86D4, [ 3281] = 0x86DB,
  [ 3282] = 0x86E4, [ 3283] = 0x86D0, [ 3284] = 0x86DE, [ 3285] = 0x8857,
  [ 3286] = 0x88C1, [ 3287] = 0x88C2, [ 3288] = 0x88B1, [ 3289] = 0x8983,
  [ 3290] = 0x8996, [ 3291] = 0x8A3B, [ 3292] = 0x8A60, [ 3293] = 0x8A55,
  [ 3294] = 0x8A5E, [ 3295] = 0x8A3C, [ 3296] = 0x8A41, [ 3297] = 0x8A54,
  [ 3298] = 0x8A5B, [ 3299] = 0x8A50, [ 3300] = 0x8A46, [ 3301] = 0x8A34,
  [ 3302] = 0x8A3A, [ 3303] = 0x8A36, [ 3304] = 0x8A56, [ 3305] = 0x8C61,
  [ 3306] = 0x8C82, [ 3307] = 0x8CAF, [ 3308] = 0x8CBC, [ 3309] = 0x8CB3,
  [ 3310] = 0x8CBD, [ 3311] = 0x8CC1, [ 3312] = 0x8CBB, [ 3313] = 0x8CC0,
  [ 3314] = 0x8CB4, [ 3315] = 0x8CB7, [ 3316] = 0x8CB6, [ 3317] = 0x8CBF,
  [ 3318] = 0x8CB8, [ 3319] = 0x8D8A, [ 3320] = 0x8D85, [ 3321] = 0x8D81,
  [ 3322] = 0x8DCE, [ 3323] = 0x8DDD, [ 3324] = 0x8DCB, [ 3325] = 0x8DDA,
  [ 3326] = 0x8DD1, [ 3327] = 0x8DCC, [ 3328] = 0x8DDB, [ 3329] = 0x8DC6,
  [ 3330] = 0x8EFB, [ 3331] = 0x8EF8, [ 3332] = 0x8EFC, [ 3333] = 0x8F9C,
  [ 3334] = 0x902E, [ 3335] = 0x9035, [ 3336] = 0x9031, [ 3337] = 0x9038,
  [ 3338] = 0x9032, [ 3339] = 0x9036, [ 3340] = 0x9102, [ 3341] = 0x90F5,
  [ 3342] = 0x9109, [ 3343] = 0x90FE, [ 3344] = 0x9163, [ 3345] = 0x9165,
  [ 3346] = 0x91CF, [ 3347] = 0x9214, [ 3348] = 0x9215, [ 3349] = 0x9223,
  [ 3350] = 0x9209, [ 3351] = 0x921E, [ 3352] = 0x920D, [ 3353] = 0x9210,
  [ 3354] = 0x9207, [ 3355] = 0x9211, [ 3356] = 0x9594, [ 3357] = 0x958F,
  [ 3358] = 0x958B, [ 3359] = 0x9591, [ 3360] = 0x9593, [ 3361] = 0x9592,
  [ 3362] = 0x958E, [ 3363] = 0x968A, [ 3364] = 0x968E, [ 3365] = 0x968B,
  [ 3366] = 0x967D, [ 3367] = 0x9685, [ 3368] = 0x9686, [ 3369] = 0x968D,
  [ 3370] = 0x9672, [ 3371] = 0x9684, [ 3372] = 0x96C1, [ 3373] = 0x96C5,
  [ 3374] = 0x96C4, [ 3375] = 0x96C6, [ 3376] = 0x96C7, [ 3377] = 0x96EF,
  [ 3378] = 0x96F2, [ 3379] = 0x97CC, [ 3380] = 0x9805, [ 3381] = 0x9806,
  [ 3382] = 0x9808, [ 3383] = 0x98E7, [ 3384] = 0x98EA, [ 3385] = 0x98EF,
  [ 3386] = 0x98E9, [ 3387] = 0x98F2, [ 3388] = 0x98ED, [ 3389] = 0x99AE,
  [ 3390] = 0x99AD, [ 3391] = 0x9EC3, [ 3392] = 0x9ECD, [ 3393] = 0x9ED1,
  [ 3394] = 0x4E82, [ 3395] = 0x50AD, [ 3396] = 0x50B5, [ 3397] = 0x50B2,
  [ 3398] = 0x50B3, [ 3399] = 0x50C5, [ 3400] = 0x50BE, [ 3401] = 0x50AC,
  [ 3402] = 0x50B7, [ 3403] = 0x50BB, [ 3404] = 0x50AF, [ 3405] = 0x50C7,
  [ 3406] = 0x527F, [ 3407] = 0x5277, [ 3408] = 0x527D, [ 3409] = 0x52DF,
  [ 3410] = 0x52E6, [ 3411] = 0x52E4, [ 3412] = 0x52E2, [ 3413] = 0x52E3,
  [ 3414] = 0x532F, [ 3415] = 0x55DF, [ 3416] = 0x55E8, [ 3417] = 0x55D3,
  [ 3418] = 0x55E6, [ 3419] = 0x55CE, [ 3420] = 0x55DC, [ 3421] = 0x55C7,
  [ 3422] = 0x55D1, [ 3423] = 0x55E3, [ 3424] = 0x55E4, [ 3425] = 0x55EF,
  [ 3426] = 0x55DA, [ 3427] = 0x55E1, [ 3428] = 0x55C5, [ 3429] = 0x55C6,
  [ 3430] = 0x55E5, [ 3431] = 0x55C9, [ 3432] = 0x5712, [ 3433] = 0x5713,
  [ 3434] = 0x585E, [ 3435] = 0x5851, [ 3436] = 0x5858, [ 3437] = 0x5857,
  [ 3438] = 0x585A, [ 3439] = 0x5854, [ 3440] = 0x586B, [ 3441] = 0x584C,
  [ 3442] = 0x586D, [ 3443] = 0x584A, [ 3444] = 0x5862, [ 3445] = 0x5852,
  [ 3446] = 0x584B, [ 3447] = 0x5967, [ 3448] = 0x5AC1, [ 3449] = 0x5AC9,
  [ 3450] = 0x5ACC, [ 3451] = 0x5ABE, [ 3452] = 0x5ABD, [ 3453] = 0x5ABC,
  [ 3454] = 0x5AB3, [ 3455] = 0x5AC2, [ 3456] = 0x5AB2, [ 3457] = 0x5D69,
  [ 3458] = 0x5D6F, [ 3459] = 0x5E4C, [ 3460] = 0x5E79, [ 3461] = 0x5EC9,
  [ 3462] = 0x5EC8, [ 3463] = 0x5F12, [ 3464] = 0x5F59, [ 3465] = 0x5FAC,
  [ 3466] = 0x5FAE, [ 3467] = 0x611A, [ 3468] = 0x610F, [ 3469] = 0x6148,
  [ 3470] = 0x611F, [ 3471] = 0x60F3, [ 3472] = 0x611B, [ 3473] = 0x60F9,
  [ 3474] = 0x6101, [ 3475] = 0x6108, [ 3476] = 0x614E, [ 3477] = 0x614C,
  [ 3478] = 0x6144, [ 3479] = 0x614D, [ 3480] = 0x613E, [ 3481] = 0x6134,
  [ 3482] = 0x6127, [ 3483] = 0x610D, [ 3484] = 0x6106, [ 3485] = 0x6137,
  [ 3486] = 0x6221, [ 3487] = 0x6222, [ 3488] = 0x6413, [ 3489] = 0x643E,
  [ 3490] = 0x641E, [ 3491] = 0x642A, [ 3492] = 0x642D, [ 3493] = 0x643D,
  [ 3494] = 0x642C, [ 3495] = 0x640F, [ 3496] = 0x641C, [ 3497] = 0x6414,
  [ 3498] = 0x640D, [ 3499] = 0x6436, [ 3500] = 0x6416, [ 3501] = 0x6417,
  [ 3502] = 0x6406, [ 3503] = 0x656C, [ 3504] = 0x659F, [ 3505] = 0x65B0,
  [ 3506] = 0x6697, [ 3507] = 0x6689, [ 3508] = 0x6687, [ 3509] = 0x6688,
  [ 3510] = 0x6696, [ 3511] = 0x6684, [ 3512] = 0x6698, [ 3513] = 0x668D,
  [ 3514] = 0x6703, [ 3515] = 0x6994, [ 3516] = 0x696D, [ 3517] = 0x695A,
  [ 3518] = 0x6977, [ 3519] = 0x6960, [ 3520] = 0x6954, [ 3521] = 0x6975,
  [ 3522] = 0x6930, [ 3523] = 0x6982, [ 3524] = 0x694A, [ 3525] = 0x6968,
  [ 3526] = 0x696B, [ 3527] = 0x695E, [ 3528] = 0x6953, [ 3529] = 0x6979,
  [ 3530] = 0x6986, [ 3531] = 0x695D, [ 3532] = 0x6963, [ 3533] = 0x695B,
  [ 3534] = 0x6B47, [ 3535] = 0x6B72, [ 3536] = 0x6BC0, [ 3537] = 0x6BBF,
  [ 3538] = 0x6BD3, [ 3539] = 0x6BFD, [ 3540] = 0x6EA2, [ 3541] = 0x6EAF,
  [ 3542] = 0x6ED3, [ 3543] = 0x6EB6, [ 3544] = 0x6EC2, [ 3545] = 0x6E90,
  [ 3546] = 0x6E9D, [ 3547] = 0x6EC7, [ 3548] = 0x6EC5, [ 3549] = 0x6EA5,
  [ 3550] = 0x6E98, [ 3551] = 0x6EBC, [ 3552] = 0x6EBA, [ 3553] = 0x6EAB,
  [ 3554] = 0x6ED1, [ 3555] = 0x6E96, [ 3556] = 0x6E9C, [ 3557] = 0x6EC4,
  [ 3558] = 0x6ED4, [ 3559] = 0x6EAA, [ 3560] = 0x6EA7, [ 3561] = 0x6EB4,
  [ 3562] = 0x714E, [ 3563] = 0x7159, [ 3564] = 0x7169, [ 3565] = 0x7164,
  [ 3566] = 0x7149, [ 3567] = 0x7167, [ 3568] = 0x715C, [ 3569] = 0x716C,
  [ 3570] = 0x7166, [ 3571] = 0x714C, [ 3572] = 0x7165, [ 3573] = 0x715E,
  [ 3574] = 0x7146, [ 3575] = 0x7168, [ 3576] = 0x7156, [ 3577] = 0x723A,
  [ 3578] = 0x7252, [ 3579] = 0x7337, [ 3580] = 0x7345, [ 3581] = 0x733F,
  [ 3582] = 0x733E, [ 3583] = 0x746F, [ 3584] = 0x745A, [ 3585] = 0x7455,
  [ 3586] = 0x745F, [ 3587] = 0x745E, [ 3588] = 0x7441, [ 3589] = 0x743F,
  [ 3590] = 0x7459, [ 3591] = 0x745B, [ 3592] = 0x745C, [ 3593] = 0x7576,
  [ 3594] = 0x7578, [ 3595] = 0x7600, [ 3596] = 0x75F0, [ 3597] = 0x7601,
  [ 3598] = 0x75F2, [ 3599] = 0x75F1, [ 3600] = 0x75FA, [ 3601] = 0x75FF,
  [ 3602] = 0x75F4, [ 3603] = 0x75F3, [ 3604] = 0x76DE, [ 3605] = 0x76DF,
  [ 3606] = 0x775B, [ 3607] = 0x776B, [ 3608] = 0x7766, [ 3609] = 0x775E,
  [ 3610] = 0x7763, [ 3611] = 0x7779, [ 3612] = 0x776A, [ 3613] = 0x776C,
  [ 3614] = 0x775C, [ 3615] = 0x7765, [ 3616] = 0x7768, [ 3617] = 0x7762,
  [ 3618] = 0x77EE, [ 3619] = 0x788E, [ 3620] = 0x78B0, [ 3621] = 0x7897,
  [ 3622] = 0x7898, [ 3623] = 0x788C, [ 3624] = 0x7889, [ 3625] = 0x787C,
  [ 3626] = 0x7891, [ 3627] = 0x7893, [ 3628] = 0x787F, [ 3629] = 0x797A,
  [ 3630] = 0x797F, [ 3631] = 0x7981, [ 3632] = 0x842C, [ 3633] = 0x79BD,
  [ 3634] = 0x7A1C, [ 3635] = 0x7A1A, [ 3636] = 0x7A20, [ 3637] = 0x7A14,
  [ 3638] = 0x7A1F, [ 3639] = 0x7A1E, [ 3640] = 0x7A9F, [ 3641] = 0x7AA0,
  [ 3642] = 0x7B77, [ 3643] = 0x7BC0, [ 3644] = 0x7B60, [ 3645] = 0x7B6E,
  [ 3646] = 0x7B67, [ 3647] = 0x7CB1, [ 3648] = 0x7CB3, [ 3649] = 0x7CB5,
  [ 3650] = 0x7D93, [ 3651] = 0x7D79, [ 3652] = 0x7D91, [ 3653] = 0x7D81,
  [ 3654] = 0x7D8F, [ 3655] = 0x7D5B, [ 3656] = 0x7F6E, [ 3657] = 0x7F69,
  [ 3658] = 0x7F6A, [ 3659] = 0x7F72, [ 3660] = 0x7FA9, [ 3661] = 0x7FA8,
  [ 3662] = 0x7FA4, [ 3663] = 0x8056, [ 3664] = 0x8058, [ 3665] = 0x8086,
  [ 3666] = 0x8084, [ 3667] = 0x8171, [ 3668] = 0x8170, [ 3669] = 0x8178,
  [ 3670] = 0x8165, [ 3671] = 0x816E, [ 3672] = 0x8173, [ 3673] = 0x816B,
  [ 3674] = 0x8179, [ 3675] = 0x817A, [ 3676] = 0x8166, [ 3677] = 0x8205,
  [ 3678] = 0x8247, [ 3679] = 0x8482, [ 3680] = 0x8477, [ 3681] = 0x843D,
  [ 3682] = 0x8431, [ 3683] = 0x8475, [ 3684] = 0x8466, [ 3685] = 0x846B,
  [ 3686] = 0x8449, [ 3687] = 0x846C, [ 3688] = 0x845B, [ 3689] = 0x843C,
  [ 3690] = 0x8435, [ 3691] = 0x8461, [ 3692] = 0x8463, [ 3693] = 0x8469,
  [ 3694] = 0x846D, [ 3695] = 0x8446, [ 3696] = 0x865E, [ 3697] = 0x865C,
  [ 3698] = 0x865F, [ 3699] = 0x86F9, [ 3700] = 0x8713, [ 3701] = 0x8708,
  [ 3702] = 0x8707, [ 3703] = 0x8700, [ 3704] = 0x86FE, [ 3705] = 0x86FB,
  [ 3706] = 0x8702, [ 3707] = 0x8703, [ 3708] = 0x8706, [ 3709] = 0x870A,
  [ 3710] = 0x8859, [ 3711] = 0x88DF, [ 3712] = 0x88D4, [ 3713] = 0x88D9,
  [ 3714] = 0x88DC, [ 3715] = 0x88D8, [ 3716] = 0x88DD, [ 3717] = 0x88E1,
  [ 3718] = 0x88CA, [ 3719] = 0x88D5, [ 3720] = 0x88D2, [ 3721] = 0x899C,
  [ 3722] = 0x89E3, [ 3723] = 0x8A6B, [ 3724] = 0x8A72, [ 3725] = 0x8A73,
  [ 3726] = 0x8A66, [ 3727] = 0x8A69, [ 3728] = 0x8A70, [ 3729] = 0x8A87,
  [ 3730] = 0x8A7C, [ 3731] = 0x8A63, [ 3732] = 0x8AA0, [ 3733] = 0x8A71,
  [ 3734] = 0x8A85, [ 3735] = 0x8A6D, [ 3736] = 0x8A62, [ 3737] = 0x8A6E,
  [ 3738] = 0x8A6C, [ 3739] = 0x8A79, [ 3740] = 0x8A7B, [ 3741] = 0x8A3E,
  [ 3742] = 0x8A68, [ 3743] = 0x8C62, [ 3744] = 0x8C8A, [ 3745] = 0x8C89,
  [ 3746] = 0x8CCA, [ 3747] = 0x8CC7, [ 3748] = 0x8CC8, [ 3749] = 0x8CC4,
  [ 3750] = 0x8CB2, [ 3751] = 0x8CC3, [ 3752] = 0x8CC2, [ 3753] = 0x8CC5,
  [ 3754] = 0x8DE1, [ 3755] = 0x8DDF, [ 3756] = 0x8DE8, [ 3757] = 0x8DEF,
  [ 3758] = 0x8DF3, [ 3759] = 0x8DFA, [ 3760] = 0x8DEA, [ 3761] = 0x8DE4,
  [ 3762] = 0x8DE6, [ 3763] = 0x8EB2, [ 3764] = 0x8F03, [ 3765] = 0x8F09,
  [ 3766] = 0x8EFE, [ 3767] = 0x8F0A, [ 3768] = 0x8F9F, [ 3769] = 0x8FB2,
  [ 3770] = 0x904B, [ 3771] = 0x904A, [ 3772] = 0x9053, [ 3773] = 0x9042,
  [ 3774] = 0x9054, [ 3775] = 0x903C, [ 3776] = 0x9055, [ 3777] = 0x9050,
  [ 3778] = 0x9047, [ 3779] = 0x904F, [ 3780] = 0x904E, [ 3781] = 0x904D,
  [ 3782] = 0x9051, [ 3783] = 0x903E, [ 3784] = 0x9041, [ 3785] = 0x9112,
  [ 3786] = 0x9117, [ 3787] = 0x916C, [ 3788] = 0x916A, [ 3789] = 0x9169,
  [ 3790] = 0x91C9, [ 3791] = 0x9237, [ 3792] = 0x9257, [ 3793] = 0x9238,
  [ 3794] = 0x923D, [ 3795] = 0x9240, [ 3796] = 0x923E, [ 3797] = 0x925B,
  [ 3798] = 0x924B, [ 3799] = 0x9264, [ 3800] = 0x9251, [ 3801] = 0x9234,
  [ 3802] = 0x9249, [ 3803] = 0x924D, [ 3804] = 0x9245, [ 3805] = 0x9239,
  [ 3806] = 0x923F, [ 3807] = 0x925A, [ 3808] = 0x9598, [ 3809] = 0x9698,
  [ 3810] = 0x9694, [ 3811] = 0x9695, [ 3812] = 0x96CD, [ 3813] = 0x96CB,
  [ 3814] = 0x96C9, [ 3815] = 0x96CA, [ 3816] = 0x96F7, [ 3817] = 0x96FB,
  [ 3818] = 0x96F9, [ 3819] = 0x96F6, [ 3820] = 0x9756, [ 3821] = 0x9774,
  [ 3822] = 0x9776, [ 3823] = 0x9810, [ 3824] = 0x9811, [ 3825] = 0x9813,
  [ 3826] = 0x980A, [ 3827] = 0x9812, [ 3828] = 0x980C, [ 3829] = 0x98FC,
  [ 3830] = 0x98F4, [ 3831] = 0x98FD, [ 3832] = 0x98FE, [ 3833] = 0x99B3,
  [ 3834] = 0x99B1, [ 3835] = 0x99B4, [ 3836] = 0x9AE1, [ 3837] = 0x9CE9,
  [ 3838] = 0x9E82, [ 3839] = 0x9F0E, [ 3840] = 0x9F13, [ 3841] = 0x9F20,
  [ 3842] = 0x50E7, [ 3843] = 0x50EE, [ 3844] = 0x50E5, [ 3845] = 0x50D6,
  [ 3846] = 0x50ED, [ 3847] = 0x50DA, [ 3848] = 0x50D5, [ 3849] = 0x50CF,
  [ 3850] = 0x50D1, [ 3851] = 0x50F1, [ 3852] = 0x50CE, [ 3853] = 0x50E9,
  [ 3854] = 0x5162, [ 3855] = 0x51F3, [ 3856] = 0x5283, [ 3857] = 0x5282,
  [ 3858] = 0x5331, [ 3859] = 0x53AD, [ 3860] = 0x55FE, [ 3861] = 0x5600,
  [ 3862] = 0x561B, [ 3863] = 0x5617, [ 3864] = 0x55FD, [ 3865] = 0x5614,
  [ 3866] = 0x5606, [ 3867] = 0x5609, [ 3868] = 0x560D, [ 3869] = 0x560E,
  [ 3870] = 0x55F7, [ 3871] = 0x5616, [ 3872] = 0x561F, [ 3873] = 0x5608,
  [ 3874] = 0x5610, [ 3875] = 0x55F6, [ 3876] = 0x5718, [ 3877] = 0x5716,
  [ 3878] = 0x5875, [ 3879] = 0x587E, [ 3880] = 0x5883, [ 3881] = 0x5893,
  [ 3882] = 0x588A, [ 3883] = 0x5879, [ 3884] = 0x5885, [ 3885] = 0x587D,
  [ 3886] = 0x58FD, [ 3887] = 0x5925, [ 3888] = 0x5922, [ 3889] = 0x5924,
  [ 3890] = 0x596A, [ 3891] = 0x5969, [ 3892] = 0x5AE1, [ 3893] = 0x5AE6,
  [ 3894] = 0x5AE9, [ 3895] = 0x5AD7, [ 3896] = 0x5AD6, [ 3897] = 0x5AD8,
  [ 3898] = 0x5AE3, [ 3899] = 0x5B75, [ 3900] = 0x5BDE, [ 3901] = 0x5BE7,
  [ 3902] = 0x5BE1, [ 3903] = 0x5BE5, [ 3904] = 0x5BE6, [ 3905] = 0x5BE8,
  [ 3906] = 0x5BE2, [ 3907] = 0x5BE4, [ 3908] = 0x5BDF, [ 3909] = 0x5C0D,
  [ 3910] = 0x5C62, [ 3911] = 0x5D84, [ 3912] = 0x5D87, [ 3913] = 0x5E5B,
  [ 3914] = 0x5E63, [ 3915] = 0x5E55, [ 3916] = 0x5E57, [ 3917] = 0x5E54,
  [ 3918] = 0x5ED3, [ 3919] = 0x5ED6, [ 3920] = 0x5F0A, [ 3921] = 0x5F46,
  [ 3922] = 0x5F70, [ 3923] = 0x5FB9, [ 3924] = 0x6147, [ 3925] = 0x613F,
  [ 3926] = 0x614B, [ 3927] = 0x6177, [ 3928] = 0x6162, [ 3929] = 0x6163,
  [ 3930] = 0x615F, [ 3931] = 0x615A, [ 3932] = 0x6158, [ 3933] = 0x6175,
  [ 3934] = 0x622A, [ 3935] = 0x6487, [ 3936] = 0x6458, [ 3937] = 0x6454,
  [ 3938] = 0x64A4, [ 3939] = 0x6478, [ 3940] = 0x645F, [ 3941] = 0x647A,
  [ 3942] = 0x6451, [ 3943] = 0x6467, [ 3944] = 0x6434, [ 3945] = 0x646D,
  [ 3946] = 0x647B, [ 3947] = 0x6572, [ 3948] = 0x65A1, [ 3949] = 0x65D7,
  [ 3950] = 0x65D6, [ 3951] = 0x66A2, [ 3952] = 0x66A8, [ 3953] = 0x669D,
  [ 3954] = 0x699C, [ 3955] = 0x69A8, [ 3956] = 0x6995, [ 3957] = 0x69C1,
  [ 3958] = 0x69AE, [ 3959] = 0x69D3, [ 3960] = 0x69CB, [ 3961] = 0x699B,
  [ 3962] = 0x69B7, [ 3963] = 0x69BB, [ 3964] = 0x69AB, [ 3965] = 0x69B4,
  [ 3966] = 0x69D0, [ 3967] = 0x69CD, [ 3968] = 0x69AD, [ 3969] = 0x69CC,
  [ 3970] = 0x69A6, [ 3971] = 0x69C3, [ 3972] = 0x69A3, [ 3973] = 0x6B49,
  [ 3974] = 0x6B4C, [ 3975] = 0x6C33, [ 3976] = 0x6F33, [ 3977] = 0x6F14,
  [ 3978] = 0x6EFE, [ 3979] = 0x6F13, [ 3980] = 0x6EF4, [ 3981] = 0x6F29,
  [ 3982] = 0x6F3E, [ 3983] = 0x6F20, [ 3984] = 0x6F2C, [ 3985] = 0x6F0F,
  [ 3986] = 0x6F02, [ 3987] = 0x6F22, [ 3988] = 0x6EFF, [ 3989] = 0x6EEF,
  [ 3990] = 0x6F06, [ 3991] = 0x6F31, [ 3992] = 0x6F38, [ 3993] = 0x6F32,
  [ 3994] = 0x6F23, [ 3995] = 0x6F15, [ 3996] = 0x6F2B, [ 3997] = 0x6F2F,
  [ 3998] = 0x6F88, [ 3999] = 0x6F2A, [ 4000] = 0x6EEC, [ 4001] = 0x6F01,
  [ 4002] = 0x6EF2, [ 4003] = 0x6ECC, [ 4004] = 0x6EF7, [ 4005] = 0x7194,
  [ 4006] = 0x7199, [ 4007] = 0x717D, [ 4008] = 0x718A, [ 4009] = 0x7184,
  [ 4010] = 0x7192, [ 4011] = 0x723E, [ 4012] = 0x7292, [ 4013] = 0x7296,
  [ 4014] = 0x7344, [ 4015] = 0x7350, [ 4016] = 0x7464, [ 4017] = 0x7463,
  [ 4018] = 0x746A, [ 4019] = 0x7470, [ 4020] = 0x746D, [ 4021] = 0x7504,
  [ 4022] = 0x7591, [ 4023] = 0x7627, [ 4024] = 0x760D, [ 4025] = 0x760B,
  [ 4026] = 0x7609, [ 4027] = 0x7613, [ 4028] = 0x76E1, [ 4029] = 0x76E3,
  [ 4030] = 0x7784, [ 4031] = 0x777D, [ 4032] = 0x777F, [ 4033] = 0x7761,
  [ 4034] = 0x78C1, [ 4035] = 0x789F, [ 4036] = 0x78A7, [ 4037] = 0x78B3,
  [ 4038] = 0x78A9, [ 4039] = 0x78A3, [ 4040] = 0x798E, [ 4041] = 0x798F,
  [ 4042] = 0x798D, [ 4043] = 0x7A2E, [ 4044] = 0x7A31, [ 4045] = 0x7AAA,
  [ 4046] = 0x7AA9, [ 4047] = 0x7AED, [ 4048] = 0x7AEF, [ 4049] = 0x7BA1,
  [ 4050] = 0x7B95, [ 4051] = 0x7B8B, [ 4052] = 0x7B75, [ 4053] = 0x7B97,
  [ 4054] = 0x7B9D, [ 4055] = 0x7B94, [ 4056] = 0x7B8F, [ 4057] = 0x7BB8,
  [ 4058] = 0x7B87, [ 4059] = 0x7B84, [ 4060] = 0x7CB9, [ 4061] = 0x7CBD,
  [ 4062] = 0x7CBE, [ 4063] = 0x7DBB, [ 4064] = 0x7DB0, [ 4065] = 0x7D9C,
  [ 4066] = 0x7DBD, [ 4067] = 0x7DBE, [ 4068] = 0x7DA0, [ 4069] = 0x7DCA,
  [ 4070] = 0x7DB4, [ 4071] = 0x7DB2, [ 4072] = 0x7DB1, [ 4073] = 0x7DBA,
  [ 4074] = 0x7DA2, [ 4075] = 0x7DBF, [ 4076] = 0x7DB5, [ 4077] = 0x7DB8,
  [ 4078] = 0x7DAD, [ 4079] = 0x7DD2, [ 4080] = 0x7DC7, [ 4081] = 0x7DAC,
  [ 4082] = 0x7F70, [ 4083] = 0x7FE0, [ 4084] = 0x7FE1, [ 4085] = 0x7FDF,
  [ 4086] = 0x805E, [ 4087] = 0x805A, [ 4088] = 0x8087, [ 4089] = 0x8150,
  [ 4090] = 0x8180, [ 4091] = 0x818F, [ 4092] = 0x8188, [ 4093] = 0x818A,
  [ 4094] = 0x817F, [ 4095] = 0x8182, [ 4096] = 0x81E7, [ 4097] = 0x81FA,
  [ 4098] = 0x8207, [ 4099] = 0x8214, [ 4100] = 0x821E, [ 4101] = 0x824B,
  [ 4102] = 0x84C9, [ 4103] = 0x84BF, [ 4104] = 0x84C6, [ 4105] = 0x84C4,
  [ 4106] = 0x8499, [ 4107] = 0x849E, [ 4108] = 0x84B2, [ 4109] = 0x849C,
  [ 4110] = 0x84CB, [ 4111] = 0x84B8, [ 4112] = 0x84C0, [ 4113] = 0x84D3,
  [ 4114] = 0x8490, [ 4115] = 0x84BC, [ 4116] = 0x84D1, [ 4117] = 0x84CA,
  [ 4118] = 0x873F, [ 4119] = 0x871C, [ 4120] = 0x873B, [ 4121] = 0x8722,
  [ 4122] = 0x8725, [ 4123] = 0x8734, [ 4124] = 0x8718, [ 4125] = 0x8755,
  [ 4126] = 0x8737, [ 4127] = 0x8729, [ 4128] = 0x88F3, [ 4129] = 0x8902,
  [ 4130] = 0x88F4, [ 4131] = 0x88F9, [ 4132] = 0x88F8, [ 4133] = 0x88FD,
  [ 4134] = 0x88E8, [ 4135] = 0x891A, [ 4136] = 0x88EF, [ 4137] = 0x8AA6,
  [ 4138] = 0x8A8C, [ 4139] = 0x8A9E, [ 4140] = 0x8AA3, [ 4141] = 0x8A8D,
  [ 4142] = 0x8AA1, [ 4143] = 0x8A93, [ 4144] = 0x8AA4, [ 4145] = 0x8AAA,
  [ 4146] = 0x8AA5, [ 4147] = 0x8AA8, [ 4148] = 0x8A98, [ 4149] = 0x8A91,
  [ 4150] = 0x8A9A, [ 4151] = 0x8AA7, [ 4152] = 0x8C6A, [ 4153] = 0x8C8D,
  [ 4154] = 0x8C8C, [ 4155] = 0x8CD3, [ 4156] = 0x8CD1, [ 4157] = 0x8CD2,
  [ 4158] = 0x8D6B, [ 4159] = 0x8D99, [ 4160] = 0x8D95, [ 4161] = 0x8DFC,
  [ 4162] = 0x8F14, [ 4163] = 0x8F12, [ 4164] = 0x8F15, [ 4165] = 0x8F13,
  [ 4166] = 0x8FA3, [ 4167] = 0x9060, [ 4168] = 0x9058, [ 4169] = 0x905C,
  [ 4170] = 0x9063, [ 4171] = 0x9059, [ 4172] = 0x905E, [ 4173] = 0x9062,
  [ 4174] = 0x905D, [ 4175] = 0x905B, [ 4176] = 0x9119, [ 4177] = 0x9118,
  [ 4178] = 0x911E, [ 4179] = 0x9175, [ 4180] = 0x9178, [ 4181] = 0x9177,
  [ 4182] = 0x9174, [ 4183] = 0x9278, [ 4184] = 0x9280, [ 4185] = 0x9285,
  [ 4186] = 0x9298, [ 4187] = 0x9296, [ 4188] = 0x927B, [ 4189] = 0x9293,
  [ 4190] = 0x929C, [ 4191] = 0x92A8, [ 4192] = 0x927C, [ 4193] = 0x9291,
  [ 4194] = 0x95A1, [ 4195] = 0x95A8, [ 4196] = 0x95A9, [ 4197] = 0x95A3,
  [ 4198] = 0x95A5, [ 4199] = 0x95A4, [ 4200] = 0x9699, [ 4201] = 0x969C,
  [ 4202] = 0x969B, [ 4203] = 0x96CC, [ 4204] = 0x96D2, [ 4205] = 0x9700,
  [ 4206] = 0x977C, [ 4207] = 0x9785, [ 4208] = 0x97F6, [ 4209] = 0x9817,
  [ 4210] = 0x9818, [ 4211] = 0x98AF, [ 4212] = 0x98B1, [ 4213] = 0x9903,
  [ 4214] = 0x9905, [ 4215] = 0x990C, [ 4216] = 0x9909, [ 4217] = 0x99C1,
  [ 4218] = 0x9AAF, [ 4219] = 0x9AB0, [ 4220] = 0x9AE6, [ 4221] = 0x9B41,
  [ 4222] = 0x9B42, [ 4223] = 0x9CF4, [ 4224] = 0x9CF6, [ 4225] = 0x9CF3,
  [ 4226] = 0x9EBC, [ 4227] = 0x9F3B, [ 4228] = 0x9F4A, [ 4229] = 0x5104,
  [ 4230] = 0x5100, [ 4231] = 0x50FB, [ 4232] = 0x50F5, [ 4233] = 0x50F9,
  [ 4234] = 0x5102, [ 4235] = 0x5108, [ 4236] = 0x5109, [ 4237] = 0x5105,
  [ 4238] = 0x51DC, [ 4239] = 0x5287, [ 4240] = 0x5288, [ 4241] = 0x5289,
  [ 4242] = 0x528D, [ 4243] = 0x528A, [ 4244] = 0x52F0, [ 4245] = 0x53B2,
  [ 4246] = 0x562E, [ 4247] = 0x563B, [ 4248] = 0x5639, [ 4249] = 0x5632,
  [ 4250] = 0x563F, [ 4251] = 0x5634, [ 4252] = 0x5629, [ 4253] = 0x5653,
  [ 4254] = 0x564E, [ 4255] = 0x5657, [ 4256] = 0x5674, [ 4257] = 0x5636,
  [ 4258] = 0x562F, [ 4259] = 0x5630, [ 4260] = 0x5880, [ 4261] = 0x589F,
  [ 4262] = 0x589E, [ 4263] = 0x58B3, [ 4264] = 0x589C, [ 4265] = 0x58AE,
  [ 4266] = 0x58A9, [ 4267] = 0x58A6, [ 4268] = 0x596D, [ 4269] = 0x5B09,
  [ 4270] = 0x5AFB, [ 4271] = 0x5B0B, [ 4272] = 0x5AF5, [ 4273] = 0x5B0C,
  [ 4274] = 0x5B08, [ 4275] = 0x5BEE, [ 4276] = 0x5BEC, [ 4277] = 0x5BE9,
  [ 4278] = 0x5BEB, [ 4279] = 0x5C64, [ 4280] = 0x5C65, [ 4281] = 0x5D9D,
  [ 4282] = 0x5D94, [ 4283] = 0x5E62, [ 4284] = 0x5E5F, [ 4285] = 0x5E61,
  [ 4286] = 0x5EE2, [ 4287] = 0x5EDA, [ 4288] = 0x5EDF, [ 4289] = 0x5EDD,
  [ 4290] = 0x5EE3, [ 4291] = 0x5EE0, [ 4292] = 0x5F48, [ 4293] = 0x5F71,
  [ 4294] = 0x5FB7, [ 4295] = 0x5FB5, [ 4296] = 0x6176, [ 4297] = 0x6167,
  [ 4298] = 0x616E, [ 4299] = 0x615D, [ 4300] = 0x6155, [ 4301] = 0x6182,
  [ 4302] = 0x617C, [ 4303] = 0x6170, [ 4304] = 0x616B, [ 4305] = 0x617E,
  [ 4306] = 0x61A7, [ 4307] = 0x6190, [ 4308] = 0x61AB, [ 4309] = 0x618E,
  [ 4310] = 0x61AC, [ 4311] = 0x619A, [ 4312] = 0x61A4, [ 4313] = 0x6194,
  [ 4314] = 0x61AE, [ 4315] = 0x622E, [ 4316] = 0x6469, [ 4317] = 0x646F,
  [ 4318] = 0x6479, [ 4319] = 0x649E, [ 4320] = 0x64B2, [ 4321] = 0x6488,
  [ 4322] = 0x6490, [ 4323] = 0x64B0, [ 4324] = 0x64A5, [ 4325] = 0x6493,
  [ 4326] = 0x6495, [ 4327] = 0x64A9, [ 4328] = 0x6492, [ 4329] = 0x64AE,
  [ 4330] = 0x64AD, [ 4331] = 0x64AB, [ 4332] = 0x649A, [ 4333] = 0x64AC,
  [ 4334] = 0x6499, [ 4335] = 0x64A2, [ 4336] = 0x64B3, [ 4337] = 0x6575,
  [ 4338] = 0x6577, [ 4339] = 0x6578, [ 4340] = 0x66AE, [ 4341] = 0x66AB,
  [ 4342] = 0x66B4, [ 4343] = 0x66B1, [ 4344] = 0x6A23, [ 4345] = 0x6A1F,
  [ 4346] = 0x69E8, [ 4347] = 0x6A01, [ 4348] = 0x6A1E, [ 4349] = 0x6A19,
  [ 4350] = 0x69FD, [ 4351] = 0x6A21, [ 4352] = 0x6A13, [ 4353] = 0x6A0A,
  [ 4354] = 0x69F3, [ 4355] = 0x6A02, [ 4356] = 0x6A05, [ 4357] = 0x69ED,
  [ 4358] = 0x6A11, [ 4359] = 0x6B50, [ 4360] = 0x6B4E, [ 4361] = 0x6BA4,
  [ 4362] = 0x6BC5, [ 4363] = 0x6BC6, [ 4364] = 0x6F3F, [ 4365] = 0x6F7C,
  [ 4366] = 0x6F84, [ 4367] = 0x6F51, [ 4368] = 0x6F66, [ 4369] = 0x6F54,
  [ 4370] = 0x6F86, [ 4371] = 0x6F6D, [ 4372] = 0x6F5B, [ 4373] = 0x6F78,
  [ 4374] = 0x6F6E, [ 4375] = 0x6F8E, [ 4376] = 0x6F7A, [ 4377] = 0x6F70,
  [ 4378] = 0x6F64, [ 4379] = 0x6F97, [ 4380] = 0x6F58, [ 4381] = 0x6ED5,
  [ 4382] = 0x6F6F, [ 4383] = 0x6F60, [ 4384] = 0x6F5F, [ 4385] = 0x719F,
  [ 4386] = 0x71AC, [ 4387] = 0x71B1, [ 4388] = 0x71A8, [ 4389] = 0x7256,
  [ 4390] = 0x729B, [ 4391] = 0x734E, [ 4392] = 0x7357, [ 4393] = 0x7469,
  [ 4394] = 0x748B, [ 4395] = 0x7483, [ 4396] = 0x747E, [ 4397] = 0x7480,
  [ 4398] = 0x757F, [ 4399] = 0x7620, [ 4400] = 0x7629, [ 4401] = 0x761F,
  [ 4402] = 0x7624, [ 4403] = 0x7626, [ 4404] = 0x7621, [ 4405] = 0x7622,
  [ 4406] = 0x769A, [ 4407] = 0x76BA, [ 4408] = 0x76E4, [ 4409] = 0x778E,
  [ 4410] = 0x7787, [ 4411] = 0x778C, [ 4412] = 0x7791, [ 4413] = 0x778B,
  [ 4414] = 0x78CB, [ 4415] = 0x78C5, [ 4416] = 0x78BA, [ 4417] = 0x78CA,
  [ 4418] = 0x78BE, [ 4419] = 0x78D5, [ 4420] = 0x78BC, [ 4421] = 0x78D0,
  [ 4422] = 0x7A3F, [ 4423] = 0x7A3C, [ 4424] = 0x7A40, [ 4425] = 0x7A3D,
  [ 4426] = 0x7A37, [ 4427] = 0x7A3B, [ 4428] = 0x7AAF, [ 4429] = 0x7AAE,
  [ 4430] = 0x7BAD, [ 4431] = 0x7BB1, [ 4432] = 0x7BC4, [ 4433] = 0x7BB4,
  [ 4434] = 0x7BC6, [ 4435] = 0x7BC7, [ 4436] = 0x7BC1, [ 4437] = 0x7BA0,
  [ 4438] = 0x7BCC, [ 4439] = 0x7CCA, [ 4440] = 0x7DE0, [ 4441] = 0x7DF4,
  [ 4442] = 0x7DEF, [ 4443] = 0x7DFB, [ 4444] = 0x7DD8, [ 4445] = 0x7DEC,
  [ 4446] = 0x7DDD, [ 4447] = 0x7DE8, [ 4448] = 0x7DE3, [ 4449] = 0x7DDA,
  [ 4450] = 0x7DDE, [ 4451] = 0x7DE9, [ 4452] = 0x7D9E, [ 4453] = 0x7DD9,
  [ 4454] = 0x7DF2, [ 4455] = 0x7DF9, [ 4456] = 0x7F75, [ 4457] = 0x7F77,
  [ 4458] = 0x7FAF, [ 4459] = 0x7FE9, [ 4460] = 0x8026, [ 4461] = 0x819B,
  [ 4462] = 0x819C, [ 4463] = 0x819D, [ 4464] = 0x81A0, [ 4465] = 0x819A,
  [ 4466] = 0x8198, [ 4467] = 0x8517, [ 4468] = 0x853D, [ 4469] = 0x851A,
  [ 4470] = 0x84EE, [ 4471] = 0x852C, [ 4472] = 0x852D, [ 4473] = 0x8513,
  [ 4474] = 0x8511, [ 4475] = 0x8523, [ 4476] = 0x8521, [ 4477] = 0x8514,
  [ 4478] = 0x84EC, [ 4479] = 0x8525, [ 4480] = 0x84FF, [ 4481] = 0x8506,
  [ 4482] = 0x8782, [ 4483] = 0x8774, [ 4484] = 0x8776, [ 4485] = 0x8760,
  [ 4486] = 0x8766, [ 4487] = 0x8778, [ 4488] = 0x8768, [ 4489] = 0x8759,
  [ 4490] = 0x8757, [ 4491] = 0x874C, [ 4492] = 0x8753, [ 4493] = 0x885B,
  [ 4494] = 0x885D, [ 4495] = 0x8910, [ 4496] = 0x8907, [ 4497] = 0x8912,
  [ 4498] = 0x8913, [ 4499] = 0x8915, [ 4500] = 0x890A, [ 4501] = 0x8ABC,
  [ 4502] = 0x8AD2, [ 4503] = 0x8AC7, [ 4504] = 0x8AC4, [ 4505] = 0x8A95,
  [ 4506] = 0x8ACB, [ 4507] = 0x8AF8, [ 4508] = 0x8AB2, [ 4509] = 0x8AC9,
  [ 4510] = 0x8AC2, [ 4511] = 0x8ABF, [ 4512] = 0x8AB0, [ 4513] = 0x8AD6,
  [ 4514] = 0x8ACD, [ 4515] = 0x8AB6, [ 4516] = 0x8AB9, [ 4517] = 0x8ADB,
  [ 4518] = 0x8C4C, [ 4519] = 0x8C4E, [ 4520] = 0x8C6C, [ 4521] = 0x8CE0,
  [ 4522] = 0x8CDE, [ 4523] = 0x8CE6, [ 4524] = 0x8CE4, [ 4525] = 0x8CEC,
  [ 4526] = 0x8CED, [ 4527] = 0x8CE2, [ 4528] = 0x8CE3, [ 4529] = 0x8CDC,
  [ 4530] = 0x8CEA, [ 4531] = 0x8CE1, [ 4532] = 0x8D6D, [ 4533] = 0x8D9F,
  [ 4534] = 0x8DA3, [ 4535] = 0x8E2B, [ 4536] = 0x8E10, [ 4537] = 0x8E1D,
  [ 4538] = 0x8E22, [ 4539] = 0x8E0F, [ 4540] = 0x8E29, [ 4541] = 0x8E1F,
  [ 4542] = 0x8E21, [ 4543] = 0x8E1E, [ 4544] = 0x8EBA, [ 4545] = 0x8F1D,
  [ 4546] = 0x8F1B, [ 4547] = 0x8F1F, [ 4548] = 0x8F29, [ 4549] = 0x8F26,
  [ 4550] = 0x8F2A, [ 4551] = 0x8F1C, [ 4552] = 0x8F1E, [ 4553] = 0x8F25,
  [ 4554] = 0x9069, [ 4555] = 0x906E, [ 4556] = 0x9068, [ 4557] = 0x906D,
  [ 4558] = 0x9077, [ 4559] = 0x9130, [ 4560] = 0x912D, [ 4561] = 0x9127,
  [ 4562] = 0x9131, [ 4563] = 0x9187, [ 4564] = 0x9189, [ 4565] = 0x918B,
  [ 4566] = 0x9183, [ 4567] = 0x92C5, [ 4568] = 0x92BB, [ 4569] = 0x92B7,
  [ 4570] = 0x92EA, [ 4571] = 0x92AC, [ 4572] = 0x92E4, [ 4573] = 0x92C1,
  [ 4574] = 0x92B3, [ 4575] = 0x92BC, [ 4576] = 0x92D2, [ 4577] = 0x92C7,
  [ 4578] = 0x92F0, [ 4579] = 0x92B2, [ 4580] = 0x95AD, [ 4581] = 0x95B1,
  [ 4582] = 0x9704, [ 4583] = 0x9706, [ 4584] = 0x9707, [ 4585] = 0x9709,
  [ 4586] = 0x9760, [ 4587] = 0x978D, [ 4588] = 0x978B, [ 4589] = 0x978F,
  [ 4590] = 0x9821, [ 4591] = 0x982B, [ 4592] = 0x981C, [ 4593] = 0x98B3,
  [ 4594] = 0x990A, [ 4595] = 0x9913, [ 4596] = 0x9912, [ 4597] = 0x9918,
  [ 4598] = 0x99DD, [ 4599] = 0x99D0, [ 4600] = 0x99DF, [ 4601] = 0x99DB,
  [ 4602] = 0x99D1, [ 4603] = 0x99D5, [ 4604] = 0x99D2, [ 4605] = 0x99D9,
  [ 4606] = 0x9AB7, [ 4607] = 0x9AEE, [ 4608] = 0x9AEF, [ 4609] = 0x9B27,
  [ 4610] = 0x9B45, [ 4611] = 0x9B44, [ 4612] = 0x9B77, [ 4613] = 0x9B6F,
  [ 4614] = 0x9D06, [ 4615] = 0x9D09, [ 4616] = 0x9D03, [ 4617] = 0x9EA9,
  [ 4618] = 0x9EBE, [ 4619] = 0x9ECE, [ 4620] = 0x58A8, [ 4621] = 0x9F52,
  [ 4622] = 0x5112, [ 4623] = 0x5118, [ 4624] = 0x5114, [ 4625] = 0x5110,
  [ 4626] = 0x5115, [ 4627] = 0x5180, [ 4628] = 0x51AA, [ 4629] = 0x51DD,
  [ 4630] = 0x5291, [ 4631] = 0x5293, [ 4632] = 0x52F3, [ 4633] = 0x5659,
  [ 4634] = 0x566B, [ 4635] = 0x5679, [ 4636] = 0x5669, [ 4637] = 0x5664,
  [ 4638] = 0x5678, [ 4639] = 0x566A, [ 4640] = 0x5668, [ 4641] = 0x5665,
  [ 4642] = 0x5671, [ 4643] = 0x566F, [ 4644] = 0x566C, [ 4645] = 0x5662,
  [ 4646] = 0x5676, [ 4647] = 0x58C1, [ 4648] = 0x58BE, [ 4649] = 0x58C7,
  [ 4650] = 0x58C5, [ 4651] = 0x596E, [ 4652] = 0x5B1D, [ 4653] = 0x5B34,
  [ 4654] = 0x5B78, [ 4655] = 0x5BF0, [ 4656] = 0x5C0E, [ 4657] = 0x5F4A,
  [ 4658] = 0x61B2, [ 4659] = 0x6191, [ 4660] = 0x61A9, [ 4661] = 0x618A,
  [ 4662] = 0x61CD, [ 4663] = 0x61B6, [ 4664] = 0x61BE, [ 4665] = 0x61CA,
  [ 4666] = 0x61C8, [ 4667] = 0x6230, [ 4668] = 0x64C5, [ 4669] = 0x64C1,
  [ 4670] = 0x64CB, [ 4671] = 0x64BB, [ 4672] = 0x64BC, [ 4673] = 0x64DA,
  [ 4674] = 0x64C4, [ 4675] = 0x64C7, [ 4676] = 0x64C2, [ 4677] = 0x64CD,
  [ 4678] = 0x64BF, [ 4679] = 0x64D2, [ 4680] = 0x64D4, [ 4681] = 0x64BE,
  [ 4682] = 0x6574, [ 4683] = 0x66C6, [ 4684] = 0x66C9, [ 4685] = 0x66B9,
  [ 4686] = 0x66C4, [ 4687] = 0x66C7, [ 4688] = 0x66B8, [ 4689] = 0x6A3D,
  [ 4690] = 0x6A38, [ 4691] = 0x6A3A, [ 4692] = 0x6A59, [ 4693] = 0x6A6B,
  [ 4694] = 0x6A58, [ 4695] = 0x6A39, [ 4696] = 0x6A44, [ 4697] = 0x6A62,
  [ 4698] = 0x6A61, [ 4699] = 0x6A4B, [ 4700] = 0x6A47, [ 4701] = 0x6A35,
  [ 4702] = 0x6A5F, [ 4703] = 0x6A48, [ 4704] = 0x6B59, [ 4705] = 0x6B77,
  [ 4706] = 0x6C05, [ 4707] = 0x6FC2, [ 4708] = 0x6FB1, [ 4709] = 0x6FA1,
  [ 4710] = 0x6FC3, [ 4711] = 0x6FA4, [ 4712] = 0x6FC1, [ 4713] = 0x6FA7,
  [ 4714] = 0x6FB3, [ 4715] = 0x6FC0, [ 4716] = 0x6FB9, [ 4717] = 0x6FB6,
  [ 4718] = 0x6FA6, [ 4719] = 0x6FA0, [ 4720] = 0x6FB4, [ 4721] = 0x71BE,
  [ 4722] = 0x71C9, [ 4723] = 0x71D0, [ 4724] = 0x71D2, [ 4725] = 0x71C8,
  [ 4726] = 0x71D5, [ 4727] = 0x71B9, [ 4728] = 0x71CE, [ 4729] = 0x71D9,
  [ 4730] = 0x71DC, [ 4731] = 0x71C3, [ 4732] = 0x71C4, [ 4733] = 0x7368,
  [ 4734] = 0x749C, [ 4735] = 0x74A3, [ 4736] = 0x7498, [ 4737] = 0x749F,
  [ 4738] = 0x749E, [ 4739] = 0x74E2, [ 4740] = 0x750C, [ 4741] = 0x750D,
  [ 4742] = 0x7634, [ 4743] = 0x7638, [ 4744] = 0x763A, [ 4745] = 0x76E7,
  [ 4746] = 0x76E5, [ 4747] = 0x77A0, [ 4748] = 0x779E, [ 4749] = 0x779F,
  [ 4750] = 0x77A5, [ 4751] = 0x78E8, [ 4752] = 0x78DA, [ 4753] = 0x78EC,
  [ 4754] = 0x78E7, [ 4755] = 0x79A6, [ 4756] = 0x7A4D, [ 4757] = 0x7A4E,
  [ 4758] = 0x7A46, [ 4759] = 0x7A4C, [ 4760] = 0x7A4B, [ 4761] = 0x7ABA,
  [ 4762] = 0x7BD9, [ 4763] = 0x7C11, [ 4764] = 0x7BC9, [ 4765] = 0x7BE4,
  [ 4766] = 0x7BDB, [ 4767] = 0x7BE1, [ 4768] = 0x7BE9, [ 4769] = 0x7BE6,
  [ 4770] = 0x7CD5, [ 4771] = 0x7CD6, [ 4772] = 0x7E0A, [ 4773] = 0x7E11,
  [ 4774] = 0x7E08, [ 4775] = 0x7E1B, [ 4776] = 0x7E23, [ 4777] = 0x7E1E,
  [ 4778] = 0x7E1D, [ 4779] = 0x7E09, [ 4780] = 0x7E10, [ 4781] = 0x7F79,
  [ 4782] = 0x7FB2, [ 4783] = 0x7FF0, [ 4784] = 0x7FF1, [ 4785] = 0x7FEE,
  [ 4786] = 0x8028, [ 4787] = 0x81B3, [ 4788] = 0x81A9, [ 4789] = 0x81A8,
  [ 4790] = 0x81FB, [ 4791] = 0x8208, [ 4792] = 0x8258, [ 4793] = 0x8259,
  [ 4794] = 0x854A, [ 4795] = 0x8559, [ 4796] = 0x8548, [ 4797] = 0x8568,
  [ 4798] = 0x8569, [ 4799] = 0x8543, [ 4800] = 0x8549, [ 4801] = 0x856D,
  [ 4802] = 0x856A, [ 4803] = 0x855E, [ 4804] = 0x8783, [ 4805] = 0x879F,
  [ 4806] = 0x879E, [ 4807] = 0x87A2, [ 4808] = 0x878D, [ 4809] = 0x8861,
  [ 4810] = 0x892A, [ 4811] = 0x8932, [ 4812] = 0x8925, [ 4813] = 0x892B,
  [ 4814] = 0x8921, [ 4815] = 0x89AA, [ 4816] = 0x89A6, [ 4817] = 0x8AE6,
  [ 4818] = 0x8AFA, [ 4819] = 0x8AEB, [ 4820] = 0x8AF1, [ 4821] = 0x8B00,
  [ 4822] = 0x8ADC, [ 4823] = 0x8AE7, [ 4824] = 0x8AEE, [ 4825] = 0x8AFE,
  [ 4826] = 0x8B01, [ 4827] = 0x8B02, [ 4828] = 0x8AF7, [ 4829] = 0x8AED,
  [ 4830] = 0x8AF3, [ 4831] = 0x8AF6, [ 4832] = 0x8AFC, [ 4833] = 0x8C6B,
  [ 4834] = 0x8C6D, [ 4835] = 0x8C93, [ 4836] = 0x8CF4, [ 4837] = 0x8E44,
  [ 4838] = 0x8E31, [ 4839] = 0x8E34, [ 4840] = 0x8E42, [ 4841] = 0x8E39,
  [ 4842] = 0x8E35, [ 4843] = 0x8F3B, [ 4844] = 0x8F2F, [ 4845] = 0x8F38,
  [ 4846] = 0x8F33, [ 4847] = 0x8FA8, [ 4848] = 0x8FA6, [ 4849] = 0x9075,
  [ 4850] = 0x9074, [ 4851] = 0x9078, [ 4852] = 0x9072, [ 4853] = 0x907C,
  [ 4854] = 0x907A, [ 4855] = 0x9134, [ 4856] = 0x9192, [ 4857] = 0x9320,
  [ 4858] = 0x9336, [ 4859] = 0x92F8, [ 4860] = 0x9333, [ 4861] = 0x932F,
  [ 4862] = 0x9322, [ 4863] = 0x92FC, [ 4864] = 0x932B, [ 4865] = 0x9304,
  [ 4866] = 0x931A, [ 4867] = 0x9310, [ 4868] = 0x9326, [ 4869] = 0x9321,
  [ 4870] = 0x9315, [ 4871] = 0x932E, [ 4872] = 0x9319, [ 4873] = 0x95BB,
  [ 4874] = 0x96A7, [ 4875] = 0x96A8, [ 4876] = 0x96AA, [ 4877] = 0x96D5,
  [ 4878] = 0x970E, [ 4879] = 0x9711, [ 4880] = 0x9716, [ 4881] = 0x970D,
  [ 4882] = 0x9713, [ 4883] = 0x970F, [ 4884] = 0x975B, [ 4885] = 0x975C,
  [ 4886] = 0x9766, [ 4887] = 0x9798, [ 4888] = 0x9830, [ 4889] = 0x9838,
  [ 4890] = 0x983B, [ 4891] = 0x9837, [ 4892] = 0x982D, [ 4893] = 0x9839,
  [ 4894] = 0x9824, [ 4895] = 0x9910, [ 4896] = 0x9928, [ 4897] = 0x991E,
  [ 4898] = 0x991B, [ 4899] = 0x9921, [ 4900] = 0x991A, [ 4901] = 0x99ED,
  [ 4902] = 0x99E2, [ 4903] = 0x99F1, [ 4904] = 0x9AB8, [ 4905] = 0x9ABC,
  [ 4906] = 0x9AFB, [ 4907] = 0x9AED, [ 4908] = 0x9B28, [ 4909] = 0x9B91,
  [ 4910] = 0x9D15, [ 4911] = 0x9D23, [ 4912] = 0x9D26, [ 4913] = 0x9D28,
  [ 4914] = 0x9D12, [ 4915] = 0x9D1B, [ 4916] = 0x9ED8, [ 4917] = 0x9ED4,
  [ 4918] = 0x9F8D, [ 4919] = 0x9F9C, [ 4920] = 0x512A, [ 4921] = 0x511F,
  [ 4922] = 0x5121, [ 4923] = 0x5132, [ 4924] = 0x52F5, [ 4925] = 0x568E,
  [ 4926] = 0x5680, [ 4927] = 0x5690, [ 4928] = 0x5685, [ 4929] = 0x5687,
  [ 4930] = 0x568F, [ 4931] = 0x58D5, [ 4932] = 0x58D3, [ 4933] = 0x58D1,
  [ 4934] = 0x58CE, [ 4935] = 0x5B30, [ 4936] = 0x5B2A, [ 4937] = 0x5B24,
  [ 4938] = 0x5B7A, [ 4939] = 0x5C37, [ 4940] = 0x5C68, [ 4941] = 0x5DBC,
  [ 4942] = 0x5DBA, [ 4943] = 0x5DBD, [ 4944] = 0x5DB8, [ 4945] = 0x5E6B,
  [ 4946] = 0x5F4C, [ 4947] = 0x5FBD, [ 4948] = 0x61C9, [ 4949] = 0x61C2,
  [ 4950] = 0x61C7, [ 4951] = 0x61E6, [ 4952] = 0x61CB, [ 4953] = 0x6232,
  [ 4954] = 0x6234, [ 4955] = 0x64CE, [ 4956] = 0x64CA, [ 4957] = 0x64D8,
  [ 4958] = 0x64E0, [ 4959] = 0x64F0, [ 4960] = 0x64E6, [ 4961] = 0x64EC,
  [ 4962] = 0x64F1, [ 4963] = 0x64E2, [ 4964] = 0x64ED, [ 4965] = 0x6582,
  [ 4966] = 0x6583, [ 4967] = 0x66D9, [ 4968] = 0x66D6, [ 4969] = 0x6A80,
  [ 4970] = 0x6A94, [ 4971] = 0x6A84, [ 4972] = 0x6AA2, [ 4973] = 0x6A9C,
  [ 4974] = 0x6ADB, [ 4975] = 0x6AA3, [ 4976] = 0x6A7E, [ 4977] = 0x6A97,
  [ 4978] = 0x6A90, [ 4979] = 0x6AA0, [ 4980] = 0x6B5C, [ 4981] = 0x6BAE,
  [ 4982] = 0x6BDA, [ 4983] = 0x6C08, [ 4984] = 0x6FD8, [ 4985] = 0x6FF1,
  [ 4986] = 0x6FDF, [ 4987] = 0x6FE0, [ 4988] = 0x6FDB, [ 4989] = 0x6FE4,
  [ 4990] = 0x6FEB, [ 4991] = 0x6FEF, [ 4992] = 0x6F80, [ 4993] = 0x6FEC,
  [ 4994] = 0x6FE1, [ 4995] = 0x6FE9, [ 4996] = 0x6FD5, [ 4997] = 0x6FEE,
  [ 4998] = 0x6FF0, [ 4999] = 0x71E7, [ 5000] = 0x71DF, [ 5001] = 0x71EE,
  [ 5002] = 0x71E6, [ 5003] = 0x71E5, [ 5004] = 0x71ED, [ 5005] = 0x71EC,
  [ 5006] = 0x71F4, [ 5007] = 0x71E0, [ 5008] = 0x7235, [ 5009] = 0x7246,
  [ 5010] = 0x7370, [ 5011] = 0x7372, [ 5012] = 0x74A9, [ 5013] = 0x74B0,
  [ 5014] = 0x74A6, [ 5015] = 0x74A8, [ 5016] = 0x7646, [ 5017] = 0x7642,
  [ 5018] = 0x764C, [ 5019] = 0x76EA, [ 5020] = 0x77B3, [ 5021] = 0x77AA,
  [ 5022] = 0x77B0, [ 5023] = 0x77AC, [ 5024] = 0x77A7, [ 5025] = 0x77AD,
  [ 5026] = 0x77EF, [ 5027] = 0x78F7, [ 5028] = 0x78FA, [ 5029] = 0x78F4,
  [ 5030] = 0x78EF, [ 5031] = 0x7901, [ 5032] = 0x79A7, [ 5033] = 0x79AA,
  [ 5034] = 0x7A57, [ 5035] = 0x7ABF, [ 5036] = 0x7C07, [ 5037] = 0x7C0D,
  [ 5038] = 0x7BFE, [ 5039] = 0x7BF7, [ 5040] = 0x7C0C, [ 5041] = 0x7BE0,
  [ 5042] = 0x7CE0, [ 5043] = 0x7CDC, [ 5044] = 0x7CDE, [ 5045] = 0x7CE2,
  [ 5046] = 0x7CDF, [ 5047] = 0x7CD9, [ 5048] = 0x7CDD, [ 5049] = 0x7E2E,
  [ 5050] = 0x7E3E, [ 5051] = 0x7E46, [ 5052] = 0x7E37, [ 5053] = 0x7E32,
  [ 5054] = 0x7E43, [ 5055] = 0x7E2B, [ 5056] = 0x7E3D, [ 5057] = 0x7E31,
  [ 5058] = 0x7E45, [ 5059] = 0x7E41, [ 5060] = 0x7E34, [ 5061] = 0x7E39,
  [ 5062] = 0x7E48, [ 5063] = 0x7E35, [ 5064] = 0x7E3F, [ 5065] = 0x7E2F,
  [ 5066] = 0x7F44, [ 5067] = 0x7FF3, [ 5068] = 0x7FFC, [ 5069] = 0x8071,
  [ 5070] = 0x8072, [ 5071] = 0x8070, [ 5072] = 0x806F, [ 5073] = 0x8073,
  [ 5074] = 0x81C6, [ 5075] = 0x81C3, [ 5076] = 0x81BA, [ 5077] = 0x81C2,
  [ 5078] = 0x81C0, [ 5079] = 0x81BF, [ 5080] = 0x81BD, [ 5081] = 0x81C9,
  [ 5082] = 0x81BE, [ 5083] = 0x81E8, [ 5084] = 0x8209, [ 5085] = 0x8271,
  [ 5086] = 0x85AA, [ 5087] = 0x8584, [ 5088] = 0x857E, [ 5089] = 0x859C,
  [ 5090] = 0x8591, [ 5091] = 0x8594, [ 5092] = 0x85AF, [ 5093] = 0x859B,
  [ 5094] = 0x8587, [ 5095] = 0x85A8, [ 5096] = 0x858A, [ 5097] = 0x8667,
  [ 5098] = 0x87C0, [ 5099] = 0x87D1, [ 5100] = 0x87B3, [ 5101] = 0x87D2,
  [ 5102] = 0x87C6, [ 5103] = 0x87AB, [ 5104] = 0x87BB, [ 5105] = 0x87BA,
  [ 5106] = 0x87C8, [ 5107] = 0x87CB, [ 5108] = 0x893B, [ 5109] = 0x8936,
  [ 5110] = 0x8944, [ 5111] = 0x8938, [ 5112] = 0x893D, [ 5113] = 0x89AC,
  [ 5114] = 0x8B0E, [ 5115] = 0x8B17, [ 5116] = 0x8B19, [ 5117] = 0x8B1B,
  [ 5118] = 0x8B0A, [ 5119] = 0x8B20, [ 5120] = 0x8B1D, [ 5121] = 0x8B04,
  [ 5122] = 0x8B10, [ 5123] = 0x8C41, [ 5124] = 0x8C3F, [ 5125] = 0x8C73,
  [ 5126] = 0x8CFA, [ 5127] = 0x8CFD, [ 5128] = 0x8CFC, [ 5129] = 0x8CF8,
  [ 5130] = 0x8CFB, [ 5131] = 0x8DA8, [ 5132] = 0x8E49, [ 5133] = 0x8E4B,
  [ 5134] = 0x8E48, [ 5135] = 0x8E4A, [ 5136] = 0x8F44, [ 5137] = 0x8F3E,
  [ 5138] = 0x8F42, [ 5139] = 0x8F45, [ 5140] = 0x8F3F, [ 5141] = 0x907F,
  [ 5142] = 0x907D, [ 5143] = 0x9084, [ 5144] = 0x9081, [ 5145] = 0x9082,
  [ 5146] = 0x9080, [ 5147] = 0x9139, [ 5148] = 0x91A3, [ 5149] = 0x919E,
  [ 5150] = 0x919C, [ 5151] = 0x934D, [ 5152] = 0x9382, [ 5153] = 0x9328,
  [ 5154] = 0x9375, [ 5155] = 0x934A, [ 5156] = 0x9365, [ 5157] = 0x934B,
  [ 5158] = 0x9318, [ 5159] = 0x937E, [ 5160] = 0x936C, [ 5161] = 0x935B,
  [ 5162] = 0x9370, [ 5163] = 0x935A, [ 5164] = 0x9354, [ 5165] = 0x95CA,
  [ 5166] = 0x95CB, [ 5167] = 0x95CC, [ 5168] = 0x95C8, [ 5169] = 0x95C6,
  [ 5170] = 0x96B1, [ 5171] = 0x96B8, [ 5172] = 0x96D6, [ 5173] = 0x971C,
  [ 5174] = 0x971E, [ 5175] = 0x97A0, [ 5176] = 0x97D3, [ 5177] = 0x9846,
  [ 5178] = 0x98B6, [ 5179] = 0x9935, [ 5180] = 0x9A01, [ 5181] = 0x99FF,
  [ 5182] = 0x9BAE, [ 5183] = 0x9BAB, [ 5184] = 0x9BAA, [ 5185] = 0x9BAD,
  [ 5186] = 0x9D3B, [ 5187] = 0x9D3F, [ 5188] = 0x9E8B, [ 5189] = 0x9ECF,
  [ 5190] = 0x9EDE, [ 5191] = 0x9EDC, [ 5192] = 0x9EDD, [ 5193] = 0x9EDB,
  [ 5194] = 0x9F3E, [ 5195] = 0x9F4B, [ 5196] = 0x53E2, [ 5197] = 0x5695,
  [ 5198] = 0x56AE, [ 5199] = 0x58D9, [ 5200] = 0x58D8, [ 5201] = 0x5B38,
  [ 5202] = 0x5F5D, [ 5203] = 0x61E3, [ 5204] = 0x6233, [ 5205] = 0x64F4,
  [ 5206] = 0x64F2, [ 5207] = 0x64FE, [ 5208] = 0x6506, [ 5209] = 0x64FA,
  [ 5210] = 0x64FB, [ 5211] = 0x64F7, [ 5212] = 0x65B7, [ 5213] = 0x66DC,
  [ 5214] = 0x6726, [ 5215] = 0x6AB3, [ 5216] = 0x6AAC, [ 5217] = 0x6AC3,
  [ 5218] = 0x6ABB, [ 5219] = 0x6AB8, [ 5220] = 0x6AC2, [ 5221] = 0x6AAE,
  [ 5222] = 0x6AAF, [ 5223] = 0x6B5F, [ 5224] = 0x6B78, [ 5225] = 0x6BAF,
  [ 5226] = 0x7009, [ 5227] = 0x700B, [ 5228] = 0x6FFE, [ 5229] = 0x7006,
  [ 5230] = 0x6FFA, [ 5231] = 0x7011, [ 5232] = 0x700F, [ 5233] = 0x71FB,
  [ 5234] = 0x71FC, [ 5235] = 0x71FE, [ 5236] = 0x71F8, [ 5237] = 0x7377,
  [ 5238] = 0x7375, [ 5239] = 0x74A7, [ 5240] = 0x74BF, [ 5241] = 0x7515,
  [ 5242] = 0x7656, [ 5243] = 0x7658, [ 5244] = 0x7652, [ 5245] = 0x77BD,
  [ 5246] = 0x77BF, [ 5247] = 0x77BB, [ 5248] = 0x77BC, [ 5249] = 0x790E,
  [ 5250] = 0x79AE, [ 5251] = 0x7A61, [ 5252] = 0x7A62, [ 5253] = 0x7A60,
  [ 5254] = 0x7AC4, [ 5255] = 0x7AC5, [ 5256] = 0x7C2B, [ 5257] = 0x7C27,
  [ 5258] = 0x7C2A, [ 5259] = 0x7C1E, [ 5260] = 0x7C23, [ 5261] = 0x7C21,
  [ 5262] = 0x7CE7, [ 5263] = 0x7E54, [ 5264] = 0x7E55, [ 5265] = 0x7E5E,
  [ 5266] = 0x7E5A, [ 5267] = 0x7E61, [ 5268] = 0x7E52, [ 5269] = 0x7E59,
  [ 5270] = 0x7F48, [ 5271] = 0x7FF9, [ 5272] = 0x7FFB, [ 5273] = 0x8077,
  [ 5274] = 0x8076, [ 5275] = 0x81CD, [ 5276] = 0x81CF, [ 5277] = 0x820A,
  [ 5278] = 0x85CF, [ 5279] = 0x85A9, [ 5280] = 0x85CD, [ 5281] = 0x85D0,
  [ 5282] = 0x85C9, [ 5283] = 0x85B0, [ 5284] = 0x85BA, [ 5285] = 0x85B9,
  [ 5286] = 0x85A6, [ 5287] = 0x87EF, [ 5288] = 0x87EC, [ 5289] = 0x87F2,
  [ 5290] = 0x87E0, [ 5291] = 0x8986, [ 5292] = 0x89B2, [ 5293] = 0x89F4,
  [ 5294] = 0x8B28, [ 5295] = 0x8B39, [ 5296] = 0x8B2C, [ 5297] = 0x8B2B,
  [ 5298] = 0x8C50, [ 5299] = 0x8D05, [ 5300] = 0x8E59, [ 5301] = 0x8E63,
  [ 5302] = 0x8E66, [ 5303] = 0x8E64, [ 5304] = 0x8E5F, [ 5305] = 0x8E55,
  [ 5306] = 0x8EC0, [ 5307] = 0x8F49, [ 5308] = 0x8F4D, [ 5309] = 0x9087,
  [ 5310] = 0x9083, [ 5311] = 0x9088, [ 5312] = 0x91AB, [ 5313] = 0x91AC,
  [ 5314] = 0x91D0, [ 5315] = 0x9394, [ 5316] = 0x938A, [ 5317] = 0x9396,
  [ 5318] = 0x93A2, [ 5319] = 0x93B3, [ 5320] = 0x93AE, [ 5321] = 0x93AC,
  [ 5322] = 0x93B0, [ 5323] = 0x9398, [ 5324] = 0x939A, [ 5325] = 0x9397,
  [ 5326] = 0x95D4, [ 5327] = 0x95D6, [ 5328] = 0x95D0, [ 5329] = 0x95D5,
  [ 5330] = 0x96E2, [ 5331] = 0x96DC, [ 5332] = 0x96D9, [ 5333] = 0x96DB,
  [ 5334] = 0x96DE, [ 5335] = 0x9724, [ 5336] = 0x97A3, [ 5337] = 0x97A6,
  [ 5338] = 0x97AD, [ 5339] = 0x97F9, [ 5340] = 0x984D, [ 5341] = 0x984F,
  [ 5342] = 0x984C, [ 5343] = 0x984E, [ 5344] = 0x9853, [ 5345] = 0x98BA,
  [ 5346] = 0x993E, [ 5347] = 0x993F, [ 5348] = 0x993D, [ 5349] = 0x992E,
  [ 5350] = 0x99A5, [ 5351] = 0x9A0E, [ 5352] = 0x9AC1, [ 5353] = 0x9B03,
  [ 5354] = 0x9B06, [ 5355] = 0x9B4F, [ 5356] = 0x9B4E, [ 5357] = 0x9B4D,
  [ 5358] = 0x9BCA, [ 5359] = 0x9BC9, [ 5360] = 0x9BFD, [ 5361] = 0x9BC8,
  [ 5362] = 0x9BC0, [ 5363] = 0x9D51, [ 5364] = 0x9D5D, [ 5365] = 0x9D60,
  [ 5366] = 0x9EE0, [ 5367] = 0x9F15, [ 5368] = 0x9F2C, [ 5369] = 0x5133,
  [ 5370] = 0x56A5, [ 5371] = 0x58DE, [ 5372] = 0x58DF, [ 5373] = 0x58E2,
  [ 5374] = 0x5BF5, [ 5375] = 0x9F90, [ 5376] = 0x5EEC, [ 5377] = 0x61F2,
  [ 5378] = 0x61F7, [ 5379] = 0x61F6, [ 5380] = 0x61F5, [ 5381] = 0x6500,
  [ 5382] = 0x650F, [ 5383] = 0x66E0, [ 5384] = 0x66DD, [ 5385] = 0x6AE5,
  [ 5386] = 0x6ADD, [ 5387] = 0x6ADA, [ 5388] = 0x6AD3, [ 5389] = 0x701B,
  [ 5390] = 0x701F, [ 5391] = 0x7028, [ 5392] = 0x701A, [ 5393] = 0x701D,
  [ 5394] = 0x7015, [ 5395] = 0x7018, [ 5396] = 0x7206, [ 5397] = 0x720D,
  [ 5398] = 0x7258, [ 5399] = 0x72A2, [ 5400] = 0x7378, [ 5401] = 0x737A,
  [ 5402] = 0x74BD, [ 5403] = 0x74CA, [ 5404] = 0x74E3, [ 5405] = 0x7587,
  [ 5406] = 0x7586, [ 5407] = 0x765F, [ 5408] = 0x7661, [ 5409] = 0x77C7,
  [ 5410] = 0x7919, [ 5411] = 0x79B1, [ 5412] = 0x7A6B, [ 5413] = 0x7A69,
  [ 5414] = 0x7C3E, [ 5415] = 0x7C3F, [ 5416] = 0x7C38, [ 5417] = 0x7C3D,
  [ 5418] = 0x7C37, [ 5419] = 0x7C40, [ 5420] = 0x7E6B, [ 5421] = 0x7E6D,
  [ 5422] = 0x7E79, [ 5423] = 0x7E69, [ 5424] = 0x7E6A, [ 5425] = 0x7F85,
  [ 5426] = 0x7E73, [ 5427] = 0x7FB6, [ 5428] = 0x7FB9, [ 5429] = 0x7FB8,
  [ 5430] = 0x81D8, [ 5431] = 0x85E9, [ 5432] = 0x85DD, [ 5433] = 0x85EA,
  [ 5434] = 0x85D5, [ 5435] = 0x85E4, [ 5436] = 0x85E5, [ 5437] = 0x85F7,
  [ 5438] = 0x87FB, [ 5439] = 0x8805, [ 5440] = 0x880D, [ 5441] = 0x87F9,
  [ 5442] = 0x87FE, [ 5443] = 0x8960, [ 5444] = 0x895F, [ 5445] = 0x8956,
  [ 5446] = 0x895E, [ 5447] = 0x8B41, [ 5448] = 0x8B5C, [ 5449] = 0x8B58,
  [ 5450] = 0x8B49, [ 5451] = 0x8B5A, [ 5452] = 0x8B4E, [ 5453] = 0x8B4F,
  [ 5454] = 0x8B46, [ 5455] = 0x8B59, [ 5456] = 0x8D08, [ 5457] = 0x8D0A,
  [ 5458] = 0x8E7C, [ 5459] = 0x8E72, [ 5460] = 0x8E87, [ 5461] = 0x8E76,
  [ 5462] = 0x8E6C, [ 5463] = 0x8E7A, [ 5464] = 0x8E74, [ 5465] = 0x8F54,
  [ 5466] = 0x8F4E, [ 5467] = 0x8FAD, [ 5468] = 0x908A, [ 5469] = 0x908B,
  [ 5470] = 0x91B1, [ 5471] = 0x91AE, [ 5472] = 0x93E1, [ 5473] = 0x93D1,
  [ 5474] = 0x93DF, [ 5475] = 0x93C3, [ 5476] = 0x93C8, [ 5477] = 0x93DC,
  [ 5478] = 0x93DD, [ 5479] = 0x93D6, [ 5480] = 0x93E2, [ 5481] = 0x93CD,
  [ 5482] = 0x93D8, [ 5483] = 0x93E4, [ 5484] = 0x93D7, [ 5485] = 0x93E8,
  [ 5486] = 0x95DC, [ 5487] = 0x96B4, [ 5488] = 0x96E3, [ 5489] = 0x972A,
  [ 5490] = 0x9727, [ 5491] = 0x9761, [ 5492] = 0x97DC, [ 5493] = 0x97FB,
  [ 5494] = 0x985E, [ 5495] = 0x9858, [ 5496] = 0x985B, [ 5497] = 0x98BC,
  [ 5498] = 0x9945, [ 5499] = 0x9949, [ 5500] = 0x9A16, [ 5501] = 0x9A19,
  [ 5502] = 0x9B0D, [ 5503] = 0x9BE8, [ 5504] = 0x9BE7, [ 5505] = 0x9BD6,
  [ 5506] = 0x9BDB, [ 5507] = 0x9D89, [ 5508] = 0x9D61, [ 5509] = 0x9D72,
  [ 5510] = 0x9D6A, [ 5511] = 0x9D6C, [ 5512] = 0x9E92, [ 5513] = 0x9E97,
  [ 5514] = 0x9E93, [ 5515] = 0x9EB4, [ 5516] = 0x52F8, [ 5517] = 0x56A8,
  [ 5518] = 0x56B7, [ 5519] = 0x56B6, [ 5520] = 0x56B4, [ 5521] = 0x56BC,
  [ 5522] = 0x58E4, [ 5523] = 0x5B40, [ 5524] = 0x5B43, [ 5525] = 0x5B7D,
  [ 5526] = 0x5BF6, [ 5527] = 0x5DC9, [ 5528] = 0x61F8, [ 5529] = 0x61FA,
  [ 5530] = 0x6518, [ 5531] = 0x6514, [ 5532] = 0x6519, [ 5533] = 0x66E6,
  [ 5534] = 0x6727, [ 5535] = 0x6AEC, [ 5536] = 0x703E, [ 5537] = 0x7030,
  [ 5538] = 0x7032, [ 5539] = 0x7210, [ 5540] = 0x737B, [ 5541] = 0x74CF,
  [ 5542] = 0x7662, [ 5543] = 0x7665, [ 5544] = 0x7926, [ 5545] = 0x792A,
  [ 5546] = 0x792C, [ 5547] = 0x792B, [ 5548] = 0x7AC7, [ 5549] = 0x7AF6,
  [ 5550] = 0x7C4C, [ 5551] = 0x7C43, [ 5552] = 0x7C4D, [ 5553] = 0x7CEF,
  [ 5554] = 0x7CF0, [ 5555] = 0x8FAE, [ 5556] = 0x7E7D, [ 5557] = 0x7E7C,
  [ 5558] = 0x7E82, [ 5559] = 0x7F4C, [ 5560] = 0x8000, [ 5561] = 0x81DA,
  [ 5562] = 0x8266, [ 5563] = 0x85FB, [ 5564] = 0x85F9, [ 5565] = 0x8611,
  [ 5566] = 0x85FA, [ 5567] = 0x8606, [ 5568] = 0x860B, [ 5569] = 0x8607,
  [ 5570] = 0x860A, [ 5571] = 0x8814, [ 5572] = 0x8815, [ 5573] = 0x8964,
  [ 5574] = 0x89BA, [ 5575] = 0x89F8, [ 5576] = 0x8B70, [ 5577] = 0x8B6C,
  [ 5578] = 0x8B66, [ 5579] = 0x8B6F, [ 5580] = 0x8B5F, [ 5581] = 0x8B6B,
  [ 5582] = 0x8D0F, [ 5583] = 0x8D0D, [ 5584] = 0x8E89, [ 5585] = 0x8E81,
  [ 5586] = 0x8E85, [ 5587] = 0x8E82, [ 5588] = 0x91B4, [ 5589] = 0x91CB,
  [ 5590] = 0x9418, [ 5591] = 0x9403, [ 5592] = 0x93FD, [ 5593] = 0x95E1,
  [ 5594] = 0x9730, [ 5595] = 0x98C4, [ 5596] = 0x9952, [ 5597] = 0x9951,
  [ 5598] = 0x99A8, [ 5599] = 0x9A2B, [ 5600] = 0x9A30, [ 5601] = 0x9A37,
  [ 5602] = 0x9A35, [ 5603] = 0x9C13, [ 5604] = 0x9C0D, [ 5605] = 0x9E79,
  [ 5606] = 0x9EB5, [ 5607] = 0x9EE8, [ 5608] = 0x9F2F, [ 5609] = 0x9F5F,
  [ 5610] = 0x9F63, [ 5611] = 0x9F61, [ 5612] = 0x5137, [ 5613] = 0x5138,
  [ 5614] = 0x56C1, [ 5615] = 0x56C0, [ 5616] = 0x56C2, [ 5617] = 0x5914,
  [ 5618] = 0x5C6C, [ 5619] = 0x5DCD, [ 5620] = 0x61FC, [ 5621] = 0x61FE,
  [ 5622] = 0x651D, [ 5623] = 0x651C, [ 5624] = 0x6595, [ 5625] = 0x66E9,
  [ 5626] = 0x6AFB, [ 5627] = 0x6B04, [ 5628] = 0x6AFA, [ 5629] = 0x6BB2,
  [ 5630] = 0x704C, [ 5631] = 0x721B, [ 5632] = 0x72A7, [ 5633] = 0x74D6,
  [ 5634] = 0x74D4, [ 5635] = 0x7669, [ 5636] = 0x77D3, [ 5637] = 0x7C50,
  [ 5638] = 0x7E8F, [ 5639] = 0x7E8C, [ 5640] = 0x7FBC, [ 5641] = 0x8617,
  [ 5642] = 0x862D, [ 5643] = 0x861A, [ 5644] = 0x8823, [ 5645] = 0x8822,
  [ 5646] = 0x8821, [ 5647] = 0x881F, [ 5648] = 0x896A, [ 5649] = 0x896C,
  [ 5650] = 0x89BD, [ 5651] = 0x8B74, [ 5652] = 0x8B77, [ 5653] = 0x8B7D,
  [ 5654] = 0x8D13, [ 5655] = 0x8E8A, [ 5656] = 0x8E8D, [ 5657] = 0x8E8B,
  [ 5658] = 0x8F5F, [ 5659] = 0x8FAF, [ 5660] = 0x91BA, [ 5661] = 0x942E,
  [ 5662] = 0x9433, [ 5663] = 0x9435, [ 5664] = 0x943A, [ 5665] = 0x9438,
  [ 5666] = 0x9432, [ 5667] = 0x942B, [ 5668] = 0x95E2, [ 5669] = 0x9738,
  [ 5670] = 0x9739, [ 5671] = 0x9732, [ 5672] = 0x97FF, [ 5673] = 0x9867,
  [ 5674] = 0x9865, [ 5675] = 0x9957, [ 5676] = 0x9A45, [ 5677] = 0x9A43,
  [ 5678] = 0x9A40, [ 5679] = 0x9A3E, [ 5680] = 0x9ACF, [ 5681] = 0x9B54,
  [ 5682] = 0x9B51, [ 5683] = 0x9C2D, [ 5684] = 0x9C25, [ 5685] = 0x9DAF,
  [ 5686] = 0x9DB4, [ 5687] = 0x9DC2, [ 5688] = 0x9DB8, [ 5689] = 0x9E9D,
  [ 5690] = 0x9EEF, [ 5691] = 0x9F19, [ 5692] = 0x9F5C, [ 5693] = 0x9F66,
  [ 5694] = 0x9F67, [ 5695] = 0x513C, [ 5696] = 0x513B, [ 5697] = 0x56C8,
  [ 5698] = 0x56CA, [ 5699] = 0x56C9, [ 5700] = 0x5B7F, [ 5701] = 0x5DD4,
  [ 5702] = 0x5DD2, [ 5703] = 0x5F4E, [ 5704] = 0x61FF, [ 5705] = 0x6524,
  [ 5706] = 0x6B0A, [ 5707] = 0x6B61, [ 5708] = 0x7051, [ 5709] = 0x7058,
  [ 5710] = 0x7380, [ 5711] = 0x74E4, [ 5712] = 0x758A, [ 5713] = 0x766E,
  [ 5714] = 0x766C, [ 5715] = 0x79B3, [ 5716] = 0x7C60, [ 5717] = 0x7C5F,
  [ 5718] = 0x807E, [ 5719] = 0x807D, [ 5720] = 0x81DF, [ 5721] = 0x8972,
  [ 5722] = 0x896F, [ 5723] = 0x89FC, [ 5724] = 0x8B80, [ 5725] = 0x8D16,
  [ 5726] = 0x8D17, [ 5727] = 0x8E91, [ 5728] = 0x8E93, [ 5729] = 0x8F61,
  [ 5730] = 0x9148, [ 5731] = 0x9444, [ 5732] = 0x9451, [ 5733] = 0x9452,
  [ 5734] = 0x973D, [ 5735] = 0x973E, [ 5736] = 0x97C3, [ 5737] = 0x97C1,
  [ 5738] = 0x986B, [ 5739] = 0x9955, [ 5740] = 0x9A55, [ 5741] = 0x9A4D,
  [ 5742] = 0x9AD2, [ 5743] = 0x9B1A, [ 5744] = 0x9C49, [ 5745] = 0x9C31,
  [ 5746] = 0x9C3E, [ 5747] = 0x9C3B, [ 5748] = 0x9DD3, [ 5749] = 0x9DD7,
  [ 5750] = 0x9F34, [ 5751] = 0x9F6C, [ 5752] = 0x9F6A, [ 5753] = 0x9F94,
  [ 5754] = 0x56CC, [ 5755] = 0x5DD6, [ 5756] = 0x6200, [ 5757] = 0x6523,
  [ 5758] = 0x652B, [ 5759] = 0x652A, [ 5760] = 0x66EC, [ 5761] = 0x6B10,
  [ 5762] = 0x74DA, [ 5763] = 0x7ACA, [ 5764] = 0x7C64, [ 5765] = 0x7C63,
  [ 5766] = 0x7C65, [ 5767] = 0x7E93, [ 5768] = 0x7E96, [ 5769] = 0x7E94,
  [ 5770] = 0x81E2, [ 5771] = 0x8638, [ 5772] = 0x863F, [ 5773] = 0x8831,
  [ 5774] = 0x8B8A, [ 5775] = 0x9090, [ 5776] = 0x908F, [ 5777] = 0x9463,
  [ 5778] = 0x9460, [ 5779] = 0x9464, [ 5780] = 0x9768, [ 5781] = 0x986F,
  [ 5782] = 0x995C, [ 5783] = 0x9A5A, [ 5784] = 0x9A5B, [ 5785] = 0x9A57,
  [ 5786] = 0x9AD3, [ 5787] = 0x9AD4, [ 5788] = 0x9AD1, [ 5789] = 0x9C54,
  [ 5790] = 0x9C57, [ 5791] = 0x9C56, [ 5792] = 0x9DE5, [ 5793] = 0x9E9F,
  [ 5794] = 0x9EF4, [ 5795] = 0x56D1, [ 5796] = 0x58E9, [ 5797] = 0x652C,
  [ 5798] = 0x705E, [ 5799] = 0x7671, [ 5800] = 0x7672, [ 5801] = 0x77D7,
  [ 5802] = 0x7F50, [ 5803] = 0x7F88, [ 5804] = 0x8836, [ 5805] = 0x8839,
  [ 5806] = 0x8862, [ 5807] = 0x8B93, [ 5808] = 0x8B92, [ 5809] = 0x8B96,
  [ 5810] = 0x8277, [ 5811] = 0x8D1B, [ 5812] = 0x91C0, [ 5813] = 0x946A,
  [ 5814] = 0x9742, [ 5815] = 0x9748, [ 5816] = 0x9744, [ 5817] = 0x97C6,
  [ 5818] = 0x9870, [ 5819] = 0x9A5F, [ 5820] = 0x9B22, [ 5821] = 0x9B58,
  [ 5822] = 0x9C5F, [ 5823] = 0x9DF9, [ 5824] = 0x9DFA, [ 5825] = 0x9E7C,
  [ 5826] = 0x9E7D, [ 5827] = 0x9F07, [ 5828] = 0x9F77, [ 5829] = 0x9F72,
  [ 5830] = 0x5EF3, [ 5831] = 0x6B16, [ 5832] = 0x7063, [ 5833] = 0x7C6C,
  [ 5834] = 0x7C6E, [ 5835] = 0x883B, [ 5836] = 0x89C0, [ 5837] = 0x8EA1,
  [ 5838] = 0x91C1, [ 5839] = 0x9472, [ 5840] = 0x9470, [ 5841] = 0x9871,
  [ 5842] = 0x995E, [ 5843] = 0x9AD6, [ 5844] = 0x9B23, [ 5845] = 0x9ECC,
  [ 5846] = 0x7064, [ 5847] = 0x77DA, [ 5848] = 0x8B9A, [ 5849] = 0x9477,
  [ 5850] = 0x97C9, [ 5851] = 0x9A62, [ 5852] = 0x9A65, [ 5853] = 0x7E9C,
  [ 5854] = 0x8B9C, [ 5855] = 0x8EAA, [ 5856] = 0x91C5, [ 5857] = 0x947D,
  [ 5858] = 0x947E, [ 5859] = 0x947C, [ 5860] = 0x9C77, [ 5861] = 0x9C78,
  [ 5862] = 0x9EF7, [ 5863] = 0x8C54, [ 5864] = 0x947F, [ 5865] = 0x9E1A,
  [ 5866] = 0x7228, [ 5867] = 0x9A6A, [ 5868] = 0x9B31, [ 5869] = 0x9E1B,
  [ 5870] = 0x9E1E, [ 5871] = 0x7C72, [ 5872] = 0xF6B1, [ 5873] = 0xF6B2,
  [ 5874] = 0xF6B3, [ 5875] = 0xF6B4, [ 5876] = 0xF6B5, [ 5877] = 0xF6B6,
  [ 5878] = 0xF6B7, [ 5879] = 0xF6B8, [ 5880] = 0xF6B9, [ 5881] = 0xF6BA,
  [ 5882] = 0xF6BB, [ 5883] = 0xF6BC, [ 5884] = 0xF6BD, [ 5885] = 0xF6BE,
  [ 5886] = 0xF6BF, [ 5887] = 0xF6C0, [ 5888] = 0xF6C1, [ 5889] = 0xF6C2,
  [ 5890] = 0xF6C3, [ 5891] = 0xF6C4, [ 5892] = 0xF6C5, [ 5893] = 0xF6C6,
  [ 5894] = 0xF6C7, [ 5895] = 0xF6C8, [ 5896] = 0xF6C9, [ 5897] = 0xF6CA,
  [ 5898] = 0xF6CB, [ 5899] = 0xF6CC, [ 5900] = 0xF6CD, [ 5901] = 0xF6CE,
  [ 5902] = 0xF6CF, [ 5903] = 0xF6D0, [ 5904] = 0xF6D1, [ 5905] = 0xF6D2,
  [ 5906] = 0xF6D3, [ 5907] = 0xF6D4, [ 5908] = 0xF6D5, [ 5909] = 0xF6D6,
  [ 5910] = 0xF6D7, [ 5911] = 0xF6D8, [ 5912] = 0xF6D9, [ 5913] = 0xF6DA,
  [ 5914] = 0xF6DB, [ 5915] = 0xF6DC, [ 5916] = 0xF6DD, [ 5917] = 0xF6DE,
  [ 5918] = 0xF6DF, [ 5919] = 0xF6E0, [ 5920] = 0xF6E1, [ 5921] = 0xF6E2,
  [ 5922] = 0xF6E3, [ 5923] = 0xF6E4, [ 5924] = 0xF6E5, [ 5925] = 0xF6E6,
  [ 5926] = 0xF6E7, [ 5927] = 0xF6E8, [ 5928] = 0xF6E9, [ 5929] = 0xF6EA,
  [ 5930] = 0xF6EB, [ 5931] = 0xF6EC, [ 5932] = 0xF6ED, [ 5933] = 0xF6EE,
  [ 5934] = 0xF6EF, [ 5935] = 0xF6F0, [ 5936] = 0xF6F1, [ 5937] = 0xF6F2,
  [ 5938] = 0xF6F3, [ 5939] = 0xF6F4, [ 5940] = 0xF6F5, [ 5941] = 0xF6F6,
  [ 5942] = 0xF6F7, [ 5943] = 0xF6F8, [ 5944] = 0xF6F9, [ 5945] = 0xF6FA,
  [ 5946] = 0xF6FB, [ 5947] = 0xF6FC, [ 5948] = 0xF6FD, [ 5949] = 0xF6FE,
  [ 5950] = 0xF6FF, [ 5951] = 0xF700, [ 5952] = 0xF701, [ 5953] = 0xF702,
  [ 5954] = 0xF703, [ 5955] = 0xF704, [ 5956] = 0xF705, [ 5957] = 0xF706,
  [ 5958] = 0xF707, [ 5959] = 0xF708, [ 5960] = 0xF709, [ 5961] = 0xF70A,
  [ 5962] = 0xF70B, [ 5963] = 0xF70C, [ 5964] = 0xF70D, [ 5965] = 0xF70E,
  [ 5966] = 0xF70F, [ 5967] = 0xF710, [ 5968] = 0xF711, [ 5969] = 0xF712,
  [ 5970] = 0xF713, [ 5971] = 0xF714, [ 5972] = 0xF715, [ 5973] = 0xF716,
  [ 5974] = 0xF717, [ 5975] = 0xF718, [ 5976] = 0xF719, [ 5977] = 0xF71A,
  [ 5978] = 0xF71B, [ 5979] = 0xF71C, [ 5980] = 0xF71D, [ 5981] = 0xF71E,
  [ 5982] = 0xF71F, [ 5983] = 0xF720, [ 5984] = 0xF721, [ 5985] = 0xF722,
  [ 5986] = 0xF723, [ 5987] = 0xF724, [ 5988] = 0xF725, [ 5989] = 0xF726,
  [ 5990] = 0xF727, [ 5991] = 0xF728, [ 5992] = 0xF729, [ 5993] = 0xF72A,
  [ 5994] = 0xF72B, [ 5995] = 0xF72C, [ 5996] = 0xF72D, [ 5997] = 0xF72E,
  [ 5998] = 0xF72F, [ 5999] = 0xF730, [ 6000] = 0xF731, [ 6001] = 0xF732,
  [ 6002] = 0xF733, [ 6003] = 0xF734, [ 6004] = 0xF735, [ 6005] = 0xF736,
  [ 6006] = 0xF737, [ 6007] = 0xF738, [ 6008] = 0xF739, [ 6009] = 0xF73A,
  [ 6010] = 0xF73B, [ 6011] = 0xF73C, [ 6012] = 0xF73D, [ 6013] = 0xF73E,
  [ 6014] = 0xF73F, [ 6015] = 0xF740, [ 6016] = 0xF741, [ 6017] = 0xF742,
  [ 6018] = 0xF743, [ 6019] = 0xF744, [ 6020] = 0xF745, [ 6021] = 0xF746,
  [ 6022] = 0xF747, [ 6023] = 0xF748, [ 6024] = 0xF749, [ 6025] = 0xF74A,
  [ 6026] = 0xF74B, [ 6027] = 0xF74C, [ 6028] = 0xF74D, [ 6029] = 0xF74E,
  [ 6030] = 0xF74F, [ 6031] = 0xF750, [ 6032] = 0xF751, [ 6033] = 0xF752,
  [ 6034] = 0xF753, [ 6035] = 0xF754, [ 6036] = 0xF755, [ 6037] = 0xF756,
  [ 6038] = 0xF757, [ 6039] = 0xF758, [ 6040] = 0xF759, [ 6041] = 0xF75A,
  [ 6042] = 0xF75B, [ 6043] = 0xF75C, [ 6044] = 0xF75D, [ 6045] = 0xF75E,
  [ 6046] = 0xF75F, [ 6047] = 0xF760, [ 6048] = 0xF761, [ 6049] = 0xF762,
  [ 6050] = 0xF763, [ 6051] = 0xF764, [ 6052] = 0xF765, [ 6053] = 0xF766,
  [ 6054] = 0xF767, [ 6055] = 0xF768, [ 6056] = 0xF769, [ 6057] = 0xF76A,
  [ 6058] = 0xF76B, [ 6059] = 0xF76C, [ 6060] = 0xF76D, [ 6061] = 0xF76E,
  [ 6062] = 0xF76F, [ 6063] = 0xF770, [ 6064] = 0xF771, [ 6065] = 0xF772,
  [ 6066] = 0xF773, [ 6067] = 0xF774, [ 6068] = 0xF775, [ 6069] = 0xF776,
  [ 6070] = 0xF777, [ 6071] = 0xF778, [ 6072] = 0xF779, [ 6073] = 0xF77A,
  [ 6074] = 0xF77B, [ 6075] = 0xF77C, [ 6076] = 0xF77D, [ 6077] = 0xF77E,
  [ 6078] = 0xF77F, [ 6079] = 0xF780, [ 6080] = 0xF781, [ 6081] = 0xF782,
  [ 6082] = 0xF783, [ 6083] = 0xF784, [ 6084] = 0xF785, [ 6085] = 0xF786,
  [ 6086] = 0xF787, [ 6087] = 0xF788, [ 6088] = 0xF789, [ 6089] = 0xF78A,
  [ 6090] = 0xF78B, [ 6091] = 0xF78C, [ 6092] = 0xF78D, [ 6093] = 0xF78E,
  [ 6094] = 0xF78F, [ 6095] = 0xF790, [ 6096] = 0xF791, [ 6097] = 0xF792,
  [ 6098] = 0xF793, [ 6099] = 0xF794, [ 6100] = 0xF795, [ 6101] = 0xF796,
  [ 6102] = 0xF797, [ 6103] = 0xF798, [ 6104] = 0xF799, [ 6105] = 0xF79A,
  [ 6106] = 0xF79B, [ 6107] = 0xF79C, [ 6108] = 0xF79D, [ 6109] = 0xF79E,
  [ 6110] = 0xF79F, [ 6111] = 0xF7A0, [ 6112] = 0xF7A1, [ 6113] = 0xF7A2,
  [ 6114] = 0xF7A3, [ 6115] = 0xF7A4, [ 6116] = 0xF7A5, [ 6117] = 0xF7A6,
  [ 6118] = 0xF7A7, [ 6119] = 0xF7A8, [ 6120] = 0xF7A9, [ 6121] = 0xF7AA,
  [ 6122] = 0xF7AB, [ 6123] = 0xF7AC, [ 6124] = 0xF7AD, [ 6125] = 0xF7AE,
  [ 6126] = 0xF7AF, [ 6127] = 0xF7B0, [ 6128] = 0xF7B1, [ 6129] = 0xF7B2,
  [ 6130] = 0xF7B3, [ 6131] = 0xF7B4, [ 6132] = 0xF7B5, [ 6133] = 0xF7B6,
  [ 6134] = 0xF7B7, [ 6135] = 0xF7B8, [ 6136] = 0xF7B9, [ 6137] = 0xF7BA,
  [ 6138] = 0xF7BB, [ 6139] = 0xF7BC, [ 6140] = 0xF7BD, [ 6141] = 0xF7BE,
  [ 6142] = 0xF7BF, [ 6143] = 0xF7C0, [ 6144] = 0xF7C1, [ 6145] = 0xF7C2,
  [ 6146] = 0xF7C3, [ 6147] = 0xF7C4, [ 6148] = 0xF7C5, [ 6149] = 0xF7C6,
  [ 6150] = 0xF7C7, [ 6151] = 0xF7C8, [ 6152] = 0xF7C9, [ 6153] = 0xF7CA,
  [ 6154] = 0xF7CB, [ 6155] = 0xF7CC, [ 6156] = 0xF7CD, [ 6157] = 0xF7CE,
  [ 6158] = 0xF7CF, [ 6159] = 0xF7D0, [ 6160] = 0xF7D1, [ 6161] = 0xF7D2,
  [ 6162] = 0xF7D3, [ 6163] = 0xF7D4, [ 6164] = 0xF7D5, [ 6165] = 0xF7D6,
  [ 6166] = 0xF7D7, [ 6167] = 0xF7D8, [ 6168] = 0xF7D9, [ 6169] = 0xF7DA,
  [ 6170] = 0xF7DB, [ 6171] = 0xF7DC, [ 6172] = 0xF7DD, [ 6173] = 0xF7DE,
  [ 6174] = 0xF7DF, [ 6175] = 0xF7E0, [ 6176] = 0xF7E1, [ 6177] = 0xF7E2,
  [ 6178] = 0xF7E3, [ 6179] = 0xF7E4, [ 6180] = 0xF7E5, [ 6181] = 0xF7E6,
  [ 6182] = 0xF7E7, [ 6183] = 0xF7E8, [ 6184] = 0xF7E9, [ 6185] = 0xF7EA,
  [ 6186] = 0xF7EB, [ 6187] = 0xF7EC, [ 6188] = 0xF7ED, [ 6189] = 0xF7EE,
  [ 6190] = 0xF7EF, [ 6191] = 0xF7F0, [ 6192] = 0xF7F1, [ 6193] = 0xF7F2,
  [ 6194] = 0xF7F3, [ 6195] = 0xF7F4, [ 6196] = 0xF7F5, [ 6197] = 0xF7F6,
  [ 6198] = 0xF7F7, [ 6199] = 0xF7F8, [ 6200] = 0xF7F9, [ 6201] = 0xF7FA,
  [ 6202] = 0xF7FB, [ 6203] = 0xF7FC, [ 6204] = 0xF7FD, [ 6205] = 0xF7FE,
  [ 6206] = 0xF7FF, [ 6207] = 0xF800, [ 6208] = 0xF801, [ 6209] = 0xF802,
  [ 6210] = 0xF803, [ 6211] = 0xF804, [ 6212] = 0xF805, [ 6213] = 0xF806,
  [ 6214] = 0xF807, [ 6215] = 0xF808, [ 6216] = 0xF809, [ 6217] = 0xF80A,
  [ 6218] = 0xF80B, [ 6219] = 0xF80C, [ 6220] = 0xF80D, [ 6221] = 0xF80E,
  [ 6222] = 0xF80F, [ 6223] = 0xF810, [ 6224] = 0xF811, [ 6225] = 0xF812,
  [ 6226] = 0xF813, [ 6227] = 0xF814, [ 6228] = 0xF815, [ 6229] = 0xF816,
  [ 6230] = 0xF817, [ 6231] = 0xF818, [ 6232] = 0xF819, [ 6233] = 0xF81A,
  [ 6234] = 0xF81B, [ 6235] = 0xF81C, [ 6236] = 0xF81D, [ 6237] = 0xF81E,
  [ 6238] = 0xF81F, [ 6239] = 0xF820, [ 6240] = 0xF821, [ 6241] = 0xF822,
  [ 6242] = 0xF823, [ 6243] = 0xF824, [ 6244] = 0xF825, [ 6245] = 0xF826,
  [ 6246] = 0xF827, [ 6247] = 0xF828, [ 6248] = 0xF829, [ 6249] = 0xF82A,
  [ 6250] = 0xF82B, [ 6251] = 0xF82C, [ 6252] = 0xF82D, [ 6253] = 0xF82E,
  [ 6254] = 0xF82F, [ 6255] = 0xF830, [ 6256] = 0xF831, [ 6257] = 0xF832,
  [ 6258] = 0xF833, [ 6259] = 0xF834, [ 6260] = 0xF835, [ 6261] = 0xF836,
  [ 6262] = 0xF837, [ 6263] = 0xF838, [ 6264] = 0xF839, [ 6265] = 0xF83A,
  [ 6266] = 0xF83B, [ 6267] = 0xF83C, [ 6268] = 0xF83D, [ 6269] = 0xF83E,
  [ 6270] = 0xF83F, [ 6271] = 0xF840, [ 6272] = 0xF841, [ 6273] = 0xF842,
  [ 6274] = 0xF843, [ 6275] = 0xF844, [ 6276] = 0xF845, [ 6277] = 0xF846,
  [ 6278] = 0xF847, [ 6279] = 0xF848, [ 6280] = 0x4E42, [ 6281] = 0x4E5C,
  [ 6282] = 0x51F5, [ 6283] = 0x531A, [ 6284] = 0x5382, [ 6285] = 0x4E07,
  [ 6286] = 0x4E0C, [ 6287] = 0x4E47, [ 6288] = 0x4E8D, [ 6289] = 0x56D7,
  [ 6290] = 0xFA0C, [ 6291] = 0x5C6E, [ 6292] = 0x5F73, [ 6293] = 0x4E0F,
  [ 6294] = 0x5187, [ 6295] = 0x4E0E, [ 6296] = 0x4E2E, [ 6297] = 0x4E93,
  [ 6298] = 0x4EC2, [ 6299] = 0x4EC9, [ 6300] = 0x4EC8, [ 6301] = 0x5198,
  [ 6302] = 0x52FC, [ 6303] = 0x536C, [ 6304] = 0x53B9, [ 6305] = 0x5720,
  [ 6306] = 0x5903, [ 6307] = 0x592C, [ 6308] = 0x5C10, [ 6309] = 0x5DFF,
  [ 6310] = 0x65E1, [ 6311] = 0x6BB3, [ 6312] = 0x6BCC, [ 6313] = 0x6C14,
  [ 6314] = 0x723F, [ 6315] = 0x4E31, [ 6316] = 0x4E3C, [ 6317] = 0x4EE8,
  [ 6318] = 0x4EDC, [ 6319] = 0x4EE9, [ 6320] = 0x4EE1, [ 6321] = 0x4EDD,
  [ 6322] = 0x4EDA, [ 6323] = 0x520C, [ 6324] = 0x531C, [ 6325] = 0x534C,
  [ 6326] = 0x5722, [ 6327] = 0x5723, [ 6328] = 0x5917, [ 6329] = 0x592F,
  [ 6330] = 0x5B81, [ 6331] = 0x5B84, [ 6332] = 0x5C12, [ 6333] = 0x5C3B,
  [ 6334] = 0x5C74, [ 6335] = 0x5C73, [ 6336] = 0x5E04, [ 6337] = 0x5E80,
  [ 6338] = 0x5E82, [ 6339] = 0x5FC9, [ 6340] = 0x6209, [ 6341] = 0x6250,
  [ 6342] = 0x6C15, [ 6343] = 0x6C36, [ 6344] = 0x6C43, [ 6345] = 0x6C3F,
  [ 6346] = 0x6C3B, [ 6347] = 0x72AE, [ 6348] = 0x72B0, [ 6349] = 0x738A,
  [ 6350] = 0x79B8, [ 6351] = 0x808A, [ 6352] = 0x961E, [ 6353] = 0x4F0E,
  [ 6354] = 0x4F18, [ 6355] = 0x4F2C, [ 6356] = 0x4EF5, [ 6357] = 0x4F14,
  [ 6358] = 0x4EF1, [ 6359] = 0x4F00, [ 6360] = 0x4EF7, [ 6361] = 0x4F08,
  [ 6362] = 0x4F1D, [ 6363] = 0x4F02, [ 6364] = 0x4F05, [ 6365] = 0x4F22,
  [ 6366] = 0x4F13, [ 6367] = 0x4F04, [ 6368] = 0x4EF4, [ 6369] = 0x4F12,
  [ 6370] = 0x51B1, [ 6371] = 0x5213, [ 6372] = 0x5209, [ 6373] = 0x5210,
  [ 6374] = 0x52A6, [ 6375] = 0x5322, [ 6376] = 0x531F, [ 6377] = 0x534D,
  [ 6378] = 0x538A, [ 6379] = 0x5407, [ 6380] = 0x56E1, [ 6381] = 0x56DF,
  [ 6382] = 0x572E, [ 6383] = 0x572A, [ 6384] = 0x5734, [ 6385] = 0x593C,
  [ 6386] = 0x5980, [ 6387] = 0x597C, [ 6388] = 0x5985, [ 6389] = 0x597B,
  [ 6390] = 0x597E, [ 6391] = 0x5977, [ 6392] = 0x597F, [ 6393] = 0x5B56,
  [ 6394] = 0x5C15, [ 6395] = 0x5C25, [ 6396] = 0x5C7C, [ 6397] = 0x5C7A,
  [ 6398] = 0x5C7B, [ 6399] = 0x5C7E, [ 6400] = 0x5DDF, [ 6401] = 0x5E75,
  [ 6402] = 0x5E84, [ 6403] = 0x5F02, [ 6404] = 0x5F1A, [ 6405] = 0x5F74,
  [ 6406] = 0x5FD5, [ 6407] = 0x5FD4, [ 6408] = 0x5FCF, [ 6409] = 0x625C,
  [ 6410] = 0x625E, [ 6411] = 0x6264, [ 6412] = 0x6261, [ 6413] = 0x6266,
  [ 6414] = 0x6262, [ 6415] = 0x6259, [ 6416] = 0x6260, [ 6417] = 0x625A,
  [ 6418] = 0x6265, [ 6419] = 0x65EF, [ 6420] = 0x65EE, [ 6421] = 0x673E,
  [ 6422] = 0x6739, [ 6423] = 0x6738, [ 6424] = 0x673B, [ 6425] = 0x673A,
  [ 6426] = 0x673F, [ 6427] = 0x673C, [ 6428] = 0x6733, [ 6429] = 0x6C18,
  [ 6430] = 0x6C46, [ 6431] = 0x6C52, [ 6432] = 0x6C5C, [ 6433] = 0x6C4F,
  [ 6434] = 0x6C4A, [ 6435] = 0x6C54, [ 6436] = 0x6C4B, [ 6437] = 0x6C4C,
  [ 6438] = 0x7071, [ 6439] = 0x725E, [ 6440] = 0x72B4, [ 6441] = 0x72B5,
  [ 6442] = 0x738E, [ 6443] = 0x752A, [ 6444] = 0x767F, [ 6445] = 0x7A75,
  [ 6446] = 0x7F51, [ 6447] = 0x8278, [ 6448] = 0x827C, [ 6449] = 0x8280,
  [ 6450] = 0x827D, [ 6451] = 0x827F, [ 6452] = 0x864D, [ 6453] = 0x897E,
  [ 6454] = 0x9099, [ 6455] = 0x9097, [ 6456] = 0x9098, [ 6457] = 0x909B,
  [ 6458] = 0x9094, [ 6459] = 0x9622, [ 6460] = 0x9624, [ 6461] = 0x9620,
  [ 6462] = 0x9623, [ 6463] = 0x4F56, [ 6464] = 0x4F3B, [ 6465] = 0x4F62,
  [ 6466] = 0x4F49, [ 6467] = 0x4F53, [ 6468] = 0x4F64, [ 6469] = 0x4F3E,
  [ 6470] = 0x4F67, [ 6471] = 0x4F52, [ 6472] = 0x4F5F, [ 6473] = 0x4F41,
  [ 6474] = 0x4F58, [ 6475] = 0x4F2D, [ 6476] = 0x4F33, [ 6477] = 0x4F3F,
  [ 6478] = 0x4F61, [ 6479] = 0x518F, [ 6480] = 0x51B9, [ 6481] = 0x521C,
  [ 6482] = 0x521E, [ 6483] = 0x5221, [ 6484] = 0x52AD, [ 6485] = 0x52AE,
  [ 6486] = 0x5309, [ 6487] = 0x5363, [ 6488] = 0x5372, [ 6489] = 0x538E,
  [ 6490] = 0x538F, [ 6491] = 0x5430, [ 6492] = 0x5437, [ 6493] = 0x542A,
  [ 6494] = 0x5454, [ 6495] = 0x5445, [ 6496] = 0x5419, [ 6497] = 0x541C,
  [ 6498] = 0x5425, [ 6499] = 0x5418, [ 6500] = 0x543D, [ 6501] = 0x544F,
  [ 6502] = 0x5441, [ 6503] = 0x5428, [ 6504] = 0x5424, [ 6505] = 0x5447,
  [ 6506] = 0x56EE, [ 6507] = 0x56E7, [ 6508] = 0x56E5, [ 6509] = 0x5741,
  [ 6510] = 0x5745, [ 6511] = 0x574C, [ 6512] = 0x5749, [ 6513] = 0x574B,
  [ 6514] = 0x5752, [ 6515] = 0x5906, [ 6516] = 0x5940, [ 6517] = 0x59A6,
  [ 6518] = 0x5998, [ 6519] = 0x59A0, [ 6520] = 0x5997, [ 6521] = 0x598E,
  [ 6522] = 0x59A2, [ 6523] = 0x5990, [ 6524] = 0x598F, [ 6525] = 0x59A7,
  [ 6526] = 0x59A1, [ 6527] = 0x5B8E, [ 6528] = 0x5B92, [ 6529] = 0x5C28,
  [ 6530] = 0x5C2A, [ 6531] = 0x5C8D, [ 6532] = 0x5C8F, [ 6533] = 0x5C88,
  [ 6534] = 0x5C8B, [ 6535] = 0x5C89, [ 6536] = 0x5C92, [ 6537] = 0x5C8A,
  [ 6538] = 0x5C86, [ 6539] = 0x5C93, [ 6540] = 0x5C95, [ 6541] = 0x5DE0,
  [ 6542] = 0x5E0A, [ 6543] = 0x5E0E, [ 6544] = 0x5E8B, [ 6545] = 0x5E89,
  [ 6546] = 0x5E8C, [ 6547] = 0x5E88, [ 6548] = 0x5E8D, [ 6549] = 0x5F05,
  [ 6550] = 0x5F1D, [ 6551] = 0x5F78, [ 6552] = 0x5F76, [ 6553] = 0x5FD2,
  [ 6554] = 0x5FD1, [ 6555] = 0x5FD0, [ 6556] = 0x5FED, [ 6557] = 0x5FE8,
  [ 6558] = 0x5FEE, [ 6559] = 0x5FF3, [ 6560] = 0x5FE1, [ 6561] = 0x5FE4,
  [ 6562] = 0x5FE3, [ 6563] = 0x5FFA, [ 6564] = 0x5FEF, [ 6565] = 0x5FF7,
  [ 6566] = 0x5FFB, [ 6567] = 0x6000, [ 6568] = 0x5FF4, [ 6569] = 0x623A,
  [ 6570] = 0x6283, [ 6571] = 0x628C, [ 6572] = 0x628E, [ 6573] = 0x628F,
  [ 6574] = 0x6294, [ 6575] = 0x6287, [ 6576] = 0x6271, [ 6577] = 0x627B,
  [ 6578] = 0x627A, [ 6579] = 0x6270, [ 6580] = 0x6281, [ 6581] = 0x6288,
  [ 6582] = 0x6277, [ 6583] = 0x627D, [ 6584] = 0x6272, [ 6585] = 0x6274,
  [ 6586] = 0x6537, [ 6587] = 0x65F0, [ 6588] = 0x65F4, [ 6589] = 0x65F3,
  [ 6590] = 0x65F2, [ 6591] = 0x65F5, [ 6592] = 0x6745, [ 6593] = 0x6747,
  [ 6594] = 0x6759, [ 6595] = 0x6755, [ 6596] = 0x674C, [ 6597] = 0x6748,
  [ 6598] = 0x675D, [ 6599] = 0x674D, [ 6600] = 0x675A, [ 6601] = 0x674B,
  [ 6602] = 0x6BD0, [ 6603] = 0x6C19, [ 6604] = 0x6C1A, [ 6605] = 0x6C78,
  [ 6606] = 0x6C67, [ 6607] = 0x6C6B, [ 6608] = 0x6C84, [ 6609] = 0x6C8B,
  [ 6610] = 0x6C8F, [ 6611] = 0x6C71, [ 6612] = 0x6C6F, [ 6613] = 0x6C69,
  [ 6614] = 0x6C9A, [ 6615] = 0x6C6D, [ 6616] = 0x6C87, [ 6617] = 0x6C95,
  [ 6618] = 0x6C9C, [ 6619] = 0x6C66, [ 6620] = 0x6C73, [ 6621] = 0x6C65,
  [ 6622] = 0x6C7B, [ 6623] = 0x6C8E, [ 6624] = 0x7074, [ 6625] = 0x707A,
  [ 6626] = 0x7263, [ 6627] = 0x72BF, [ 6628] = 0x72BD, [ 6629] = 0x72C3,
  [ 6630] = 0x72C6, [ 6631] = 0x72C1, [ 6632] = 0x72BA, [ 6633] = 0x72C5,
  [ 6634] = 0x7395, [ 6635] = 0x7397, [ 6636] = 0x7393, [ 6637] = 0x7394,
  [ 6638] = 0x7392, [ 6639] = 0x753A, [ 6640] = 0x7539, [ 6641] = 0x7594,
  [ 6642] = 0x7595, [ 6643] = 0x7681, [ 6644] = 0x793D, [ 6645] = 0x8034,
  [ 6646] = 0x8095, [ 6647] = 0x8099, [ 6648] = 0x8090, [ 6649] = 0x8092,
  [ 6650] = 0x809C, [ 6651] = 0x8290, [ 6652] = 0x828F, [ 6653] = 0x8285,
  [ 6654] = 0x828E, [ 6655] = 0x8291, [ 6656] = 0x8293, [ 6657] = 0x828A,
  [ 6658] = 0x8283, [ 6659] = 0x8284, [ 6660] = 0x8C78, [ 6661] = 0x8FC9,
  [ 6662] = 0x8FBF, [ 6663] = 0x909F, [ 6664] = 0x90A1, [ 6665] = 0x90A5,
  [ 6666] = 0x909E, [ 6667] = 0x90A7, [ 6668] = 0x90A0, [ 6669] = 0x9630,
  [ 6670] = 0x9628, [ 6671] = 0x962F, [ 6672] = 0x962D, [ 6673] = 0x4E33,
  [ 6674] = 0x4F98, [ 6675] = 0x4F7C, [ 6676] = 0x4F85, [ 6677] = 0x4F7D,
  [ 6678] = 0x4F80, [ 6679] = 0x4F87, [ 6680] = 0x4F76, [ 6681] = 0x4F74,
  [ 6682] = 0x4F89, [ 6683] = 0x4F84, [ 6684] = 0x4F77, [ 6685] = 0x4F4C,
  [ 6686] = 0x4F97, [ 6687] = 0x4F6A, [ 6688] = 0x4F9A, [ 6689] = 0x4F79,
  [ 6690] = 0x4F81, [ 6691] = 0x4F78, [ 6692] = 0x4F90, [ 6693] = 0x4F9C,
  [ 6694] = 0x4F94, [ 6695] = 0x4F9E, [ 6696] = 0x4F92, [ 6697] = 0x4F82,
  [ 6698] = 0x4F95, [ 6699] = 0x4F6B, [ 6700] = 0x4F6E, [ 6701] = 0x519E,
  [ 6702] = 0x51BC, [ 6703] = 0x51BE, [ 6704] = 0x5235, [ 6705] = 0x5232,
  [ 6706] = 0x5233, [ 6707] = 0x5246, [ 6708] = 0x5231, [ 6709] = 0x52BC,
  [ 6710] = 0x530A, [ 6711] = 0x530B, [ 6712] = 0x533C, [ 6713] = 0x5392,
  [ 6714] = 0x5394, [ 6715] = 0x5487, [ 6716] = 0x547F, [ 6717] = 0x5481,
  [ 6718] = 0x5491, [ 6719] = 0x5482, [ 6720] = 0x5488, [ 6721] = 0x546B,
  [ 6722] = 0x547A, [ 6723] = 0x547E, [ 6724] = 0x5465, [ 6725] = 0x546C,
  [ 6726] = 0x5474, [ 6727] = 0x5466, [ 6728] = 0x548D, [ 6729] = 0x546F,
  [ 6730] = 0x5461, [ 6731] = 0x5460, [ 6732] = 0x5498, [ 6733] = 0x5463,
  [ 6734] = 0x5467, [ 6735] = 0x5464, [ 6736] = 0x56F7, [ 6737] = 0x56F9,
  [ 6738] = 0x576F, [ 6739] = 0x5772, [ 6740] = 0x576D, [ 6741] = 0x576B,
  [ 6742] = 0x5771, [ 6743] = 0x5770, [ 6744] = 0x5776, [ 6745] = 0x5780,
  [ 6746] = 0x5775, [ 6747] = 0x577B, [ 6748] = 0x5773, [ 6749] = 0x5774,
  [ 6750] = 0x5762, [ 6751] = 0x5768, [ 6752] = 0x577D, [ 6753] = 0x590C,
  [ 6754] = 0x5945, [ 6755] = 0x59B5, [ 6756] = 0x59BA, [ 6757] = 0x59CF,
  [ 6758] = 0x59CE, [ 6759] = 0x59B2, [ 6760] = 0x59CC, [ 6761] = 0x59C1,
  [ 6762] = 0x59B6, [ 6763] = 0x59BC, [ 6764] = 0x59C3, [ 6765] = 0x59D6,
  [ 6766] = 0x59B1, [ 6767] = 0x59BD, [ 6768] = 0x59C0, [ 6769] = 0x59C8,
  [ 6770] = 0x59B4, [ 6771] = 0x59C7, [ 6772] = 0x5B62, [ 6773] = 0x5B65,
  [ 6774] = 0x5B93, [ 6775] = 0x5B95, [ 6776] = 0x5C44, [ 6777] = 0x5C47,
  [ 6778] = 0x5CAE, [ 6779] = 0x5CA4, [ 6780] = 0x5CA0, [ 6781] = 0x5CB5,
  [ 6782] = 0x5CAF, [ 6783] = 0x5CA8, [ 6784] = 0x5CAC, [ 6785] = 0x5C9F,
  [ 6786] = 0x5CA3, [ 6787] = 0x5CAD, [ 6788] = 0x5CA2, [ 6789] = 0x5CAA,
  [ 6790] = 0x5CA7, [ 6791] = 0x5C9D, [ 6792] = 0x5CA5, [ 6793] = 0x5CB6,
  [ 6794] = 0x5CB0, [ 6795] = 0x5CA6, [ 6796] = 0x5E17, [ 6797] = 0x5E14,
  [ 6798] = 0x5E19, [ 6799] = 0x5F28, [ 6800] = 0x5F22, [ 6801] = 0x5F23,
  [ 6802] = 0x5F24, [ 6803] = 0x5F54, [ 6804] = 0x5F82, [ 6805] = 0x5F7E,
  [ 6806] = 0x5F7D, [ 6807] = 0x5FDE, [ 6808] = 0x5FE5, [ 6809] = 0x602D,
  [ 6810] = 0x6026, [ 6811] = 0x6019, [ 6812] = 0x6032, [ 6813] = 0x600B,
  [ 6814] = 0x6034, [ 6815] = 0x600A, [ 6816] = 0x6017, [ 6817] = 0x6033,
  [ 6818] = 0x601A, [ 6819] = 0x601E, [ 6820] = 0x602C, [ 6821] = 0x6022,
  [ 6822] = 0x600D, [ 6823] = 0x6010, [ 6824] = 0x602E, [ 6825] = 0x6013,
  [ 6826] = 0x6011, [ 6827] = 0x600C, [ 6828] = 0x6009, [ 6829] = 0x601C,
  [ 6830] = 0x6214, [ 6831] = 0x623D, [ 6832] = 0x62AD, [ 6833] = 0x62B4,
  [ 6834] = 0x62D1, [ 6835] = 0x62BE, [ 6836] = 0x62AA, [ 6837] = 0x62B6,
  [ 6838] = 0x62CA, [ 6839] = 0x62AE, [ 6840] = 0x62B3, [ 6841] = 0x62AF,
  [ 6842] = 0x62BB, [ 6843] = 0x62A9, [ 6844] = 0x62B0, [ 6845] = 0x62B8,
  [ 6846] = 0x653D, [ 6847] = 0x65A8, [ 6848] = 0x65BB, [ 6849] = 0x6609,
  [ 6850] = 0x65FC, [ 6851] = 0x6604, [ 6852] = 0x6612, [ 6853] = 0x6608,
  [ 6854] = 0x65FB, [ 6855] = 0x6603, [ 6856] = 0x660B, [ 6857] = 0x660D,
  [ 6858] = 0x6605, [ 6859] = 0x65FD, [ 6860] = 0x6611, [ 6861] = 0x6610,
  [ 6862] = 0x66F6, [ 6863] = 0x670A, [ 6864] = 0x6785, [ 6865] = 0x676C,
  [ 6866] = 0x678E, [ 6867] = 0x6792, [ 6868] = 0x6776, [ 6869] = 0x677B,
  [ 6870] = 0x6798, [ 6871] = 0x6786, [ 6872] = 0x6784, [ 6873] = 0x6774,
  [ 6874] = 0x678D, [ 6875] = 0x678C, [ 6876] = 0x677A, [ 6877] = 0x679F,
  [ 6878] = 0x6791, [ 6879] = 0x6799, [ 6880] = 0x6783, [ 6881] = 0x677D,
  [ 6882] = 0x6781, [ 6883] = 0x6778, [ 6884] = 0x6779, [ 6885] = 0x6794,
  [ 6886] = 0x6B25, [ 6887] = 0x6B80, [ 6888] = 0x6B7E, [ 6889] = 0x6BDE,
  [ 6890] = 0x6C1D, [ 6891] = 0x6C93, [ 6892] = 0x6CEC, [ 6893] = 0x6CEB,
  [ 6894] = 0x6CEE, [ 6895] = 0x6CD9, [ 6896] = 0x6CB6, [ 6897] = 0x6CD4,
  [ 6898] = 0x6CAD, [ 6899] = 0x6CE7, [ 6900] = 0x6CB7, [ 6901] = 0x6CD0,
  [ 6902] = 0x6CC2, [ 6903] = 0x6CBA, [ 6904] = 0x6CC3, [ 6905] = 0x6CC6,
  [ 6906] = 0x6CED, [ 6907] = 0x6CF2, [ 6908] = 0x6CD2, [ 6909] = 0x6CDD,
  [ 6910] = 0x6CB4, [ 6911] = 0x6C8A, [ 6912] = 0x6C9D, [ 6913] = 0x6C80,
  [ 6914] = 0x6CDE, [ 6915] = 0x6CC0, [ 6916] = 0x6D30, [ 6917] = 0x6CCD,
  [ 6918] = 0x6CC7, [ 6919] = 0x6CB0, [ 6920] = 0x6CF9, [ 6921] = 0x6CCF,
  [ 6922] = 0x6CE9, [ 6923] = 0x6CD1, [ 6924] = 0x7094, [ 6925] = 0x7098,
  [ 6926] = 0x7085, [ 6927] = 0x7093, [ 6928] = 0x7086, [ 6929] = 0x7084,
  [ 6930] = 0x7091, [ 6931] = 0x7096, [ 6932] = 0x7082, [ 6933] = 0x709A,
  [ 6934] = 0x7083, [ 6935] = 0x726A, [ 6936] = 0x72D6, [ 6937] = 0x72CB,
  [ 6938] = 0x72D8, [ 6939] = 0x72C9, [ 6940] = 0x72DC, [ 6941] = 0x72D2,
  [ 6942] = 0x72D4, [ 6943] = 0x72DA, [ 6944] = 0x72CC, [ 6945] = 0x72D1,
  [ 6946] = 0x73A4, [ 6947] = 0x73A1, [ 6948] = 0x73AD, [ 6949] = 0x73A6,
  [ 6950] = 0x73A2, [ 6951] = 0x73A0, [ 6952] = 0x73AC, [ 6953] = 0x739D,
  [ 6954] = 0x74DD, [ 6955] = 0x74E8, [ 6956] = 0x753F, [ 6957] = 0x7540,
  [ 6958] = 0x753E, [ 6959] = 0x758C, [ 6960] = 0x7598, [ 6961] = 0x76AF,
  [ 6962] = 0x76F3, [ 6963] = 0x76F1, [ 6964] = 0x76F0, [ 6965] = 0x76F5,
  [ 6966] = 0x77F8, [ 6967] = 0x77FC, [ 6968] = 0x77F9, [ 6969] = 0x77FB,
  [ 6970] = 0x77FA, [ 6971] = 0x77F7, [ 6972] = 0x7942, [ 6973] = 0x793F,
  [ 6974] = 0x79C5, [ 6975] = 0x7A78, [ 6976] = 0x7A7B, [ 6977] = 0x7AFB,
  [ 6978] = 0x7C75, [ 6979] = 0x7CFD, [ 6980] = 0x8035, [ 6981] = 0x808F,
  [ 6982] = 0x80AE, [ 6983] = 0x80A3, [ 6984] = 0x80B8, [ 6985] = 0x80B5,
  [ 6986] = 0x80AD, [ 6987] = 0x8220, [ 6988] = 0x82A0, [ 6989] = 0x82C0,
  [ 6990] = 0x82AB, [ 6991] = 0x829A, [ 6992] = 0x8298, [ 6993] = 0x829B,
  [ 6994] = 0x82B5, [ 6995] = 0x82A7, [ 6996] = 0x82AE, [ 6997] = 0x82BC,
  [ 6998] = 0x829E, [ 6999] = 0x82BA, [ 7000] = 0x82B4, [ 7001] = 0x82A8,
  [ 7002] = 0x82A1, [ 7003] = 0x82A9, [ 7004] = 0x82C2, [ 7005] = 0x82A4,
  [ 7006] = 0x82C3, [ 7007] = 0x82B6, [ 7008] = 0x82A2, [ 7009] = 0x8670,
  [ 7010] = 0x866F, [ 7011] = 0x866D, [ 7012] = 0x866E, [ 7013] = 0x8C56,
  [ 7014] = 0x8FD2, [ 7015] = 0x8FCB, [ 7016] = 0x8FD3, [ 7017] = 0x8FCD,
  [ 7018] = 0x8FD6, [ 7019] = 0x8FD5, [ 7020] = 0x8FD7, [ 7021] = 0x90B2,
  [ 7022] = 0x90B4, [ 7023] = 0x90AF, [ 7024] = 0x90B3, [ 7025] = 0x90B0,
  [ 7026] = 0x9639, [ 7027] = 0x963D, [ 7028] = 0x963C, [ 7029] = 0x963A,
  [ 7030] = 0x9643, [ 7031] = 0x4FCD, [ 7032] = 0x4FC5, [ 7033] = 0x4FD3,
  [ 7034] = 0x4FB2, [ 7035] = 0x4FC9, [ 7036] = 0x4FCB, [ 7037] = 0x4FC1,
  [ 7038] = 0x4FD4, [ 7039] = 0x4FDC, [ 7040] = 0x4FD9, [ 7041] = 0x4FBB,
  [ 7042] = 0x4FB3, [ 7043] = 0x4FDB, [ 7044] = 0x4FC7, [ 7045] = 0x4FD6,
  [ 7046] = 0x4FBA, [ 7047] = 0x4FC0, [ 7048] = 0x4FB9, [ 7049] = 0x4FEC,
  [ 7050] = 0x5244, [ 7051] = 0x5249, [ 7052] = 0x52C0, [ 7053] = 0x52C2,
  [ 7054] = 0x533D, [ 7055] = 0x537C, [ 7056] = 0x5397, [ 7057] = 0x5396,
  [ 7058] = 0x5399, [ 7059] = 0x5398, [ 7060] = 0x54BA, [ 7061] = 0x54A1,
  [ 7062] = 0x54AD, [ 7063] = 0x54A5, [ 7064] = 0x54CF, [ 7065] = 0x54C3,
  [ 7066] = 0x830D, [ 7067] = 0x54B7, [ 7068] = 0x54AE, [ 7069] = 0x54D6,
  [ 7070] = 0x54B6, [ 7071] = 0x54C5, [ 7072] = 0x54C6, [ 7073] = 0x54A0,
  [ 7074] = 0x5470, [ 7075] = 0x54BC, [ 7076] = 0x54A2, [ 7077] = 0x54BE,
  [ 7078] = 0x5472, [ 7079] = 0x54DE, [ 7080] = 0x54B0, [ 7081] = 0x57B5,
  [ 7082] = 0x579E, [ 7083] = 0x579F, [ 7084] = 0x57A4, [ 7085] = 0x578C,
  [ 7086] = 0x5797, [ 7087] = 0x579D, [ 7088] = 0x579B, [ 7089] = 0x5794,
  [ 7090] = 0x5798, [ 7091] = 0x578F, [ 7092] = 0x5799, [ 7093] = 0x57A5,
  [ 7094] = 0x579A, [ 7095] = 0x5795, [ 7096] = 0x58F4, [ 7097] = 0x590D,
  [ 7098] = 0x5953, [ 7099] = 0x59E1, [ 7100] = 0x59DE, [ 7101] = 0x59EE,
  [ 7102] = 0x5A00, [ 7103] = 0x59F1, [ 7104] = 0x59DD, [ 7105] = 0x59FA,
  [ 7106] = 0x59FD, [ 7107] = 0x59FC, [ 7108] = 0x59F6, [ 7109] = 0x59E4,
  [ 7110] = 0x59F2, [ 7111] = 0x59F7, [ 7112] = 0x59DB, [ 7113] = 0x59E9,
  [ 7114] = 0x59F3, [ 7115] = 0x59F5, [ 7116] = 0x59E0, [ 7117] = 0x59FE,
  [ 7118] = 0x59F4, [ 7119] = 0x59ED, [ 7120] = 0x5BA8, [ 7121] = 0x5C4C,
  [ 7122] = 0x5CD0, [ 7123] = 0x5CD8, [ 7124] = 0x5CCC, [ 7125] = 0x5CD7,
  [ 7126] = 0x5CCB, [ 7127] = 0x5CDB, [ 7128] = 0x5CDE, [ 7129] = 0x5CDA,
  [ 7130] = 0x5CC9, [ 7131] = 0x5CC7, [ 7132] = 0x5CCA, [ 7133] = 0x5CD6,
  [ 7134] = 0x5CD3, [ 7135] = 0x5CD4, [ 7136] = 0x5CCF, [ 7137] = 0x5CC8,
  [ 7138] = 0x5CC6, [ 7139] = 0x5CCE, [ 7140] = 0x5CDF, [ 7141] = 0x5CF8,
  [ 7142] = 0x5DF9, [ 7143] = 0x5E21, [ 7144] = 0x5E22, [ 7145] = 0x5E23,
  [ 7146] = 0x5E20, [ 7147] = 0x5E24, [ 7148] = 0x5EB0, [ 7149] = 0x5EA4,
  [ 7150] = 0x5EA2, [ 7151] = 0x5E9B, [ 7152] = 0x5EA3, [ 7153] = 0x5EA5,
  [ 7154] = 0x5F07, [ 7155] = 0x5F2E, [ 7156] = 0x5F56, [ 7157] = 0x5F86,
  [ 7158] = 0x6037, [ 7159] = 0x6039, [ 7160] = 0x6054, [ 7161] = 0x6072,
  [ 7162] = 0x605E, [ 7163] = 0x6045, [ 7164] = 0x6053, [ 7165] = 0x6047,
  [ 7166] = 0x6049, [ 7167] = 0x605B, [ 7168] = 0x604C, [ 7169] = 0x6040,
  [ 7170] = 0x6042, [ 7171] = 0x605F, [ 7172] = 0x6024, [ 7173] = 0x6044,
  [ 7174] = 0x6058, [ 7175] = 0x6066, [ 7176] = 0x606E, [ 7177] = 0x6242,
  [ 7178] = 0x6243, [ 7179] = 0x62CF, [ 7180] = 0x630D, [ 7181] = 0x630B,
  [ 7182] = 0x62F5, [ 7183] = 0x630E, [ 7184] = 0x6303, [ 7185] = 0x62EB,
  [ 7186] = 0x62F9, [ 7187] = 0x630F, [ 7188] = 0x630C, [ 7189] = 0x62F8,
  [ 7190] = 0x62F6, [ 7191] = 0x6300, [ 7192] = 0x6313, [ 7193] = 0x6314,
  [ 7194] = 0x62FA, [ 7195] = 0x6315, [ 7196] = 0x62FB, [ 7197] = 0x62F0,
  [ 7198] = 0x6541, [ 7199] = 0x6543, [ 7200] = 0x65AA, [ 7201] = 0x65BF,
  [ 7202] = 0x6636, [ 7203] = 0x6621, [ 7204] = 0x6632, [ 7205] = 0x6635,
  [ 7206] = 0x661C, [ 7207] = 0x6626, [ 7208] = 0x6622, [ 7209] = 0x6633,
  [ 7210] = 0x662B, [ 7211] = 0x663A, [ 7212] = 0x661D, [ 7213] = 0x6634,
  [ 7214] = 0x6639, [ 7215] = 0x662E, [ 7216] = 0x670F, [ 7217] = 0x6710,
  [ 7218] = 0x67C1, [ 7219] = 0x67F2, [ 7220] = 0x67C8, [ 7221] = 0x67BA,
  [ 7222] = 0x67DC, [ 7223] = 0x67BB, [ 7224] = 0x67F8, [ 7225] = 0x67D8,
  [ 7226] = 0x67C0, [ 7227] = 0x67B7, [ 7228] = 0x67C5, [ 7229] = 0x67EB,
  [ 7230] = 0x67E4, [ 7231] = 0x67DF, [ 7232] = 0x67B5, [ 7233] = 0x67CD,
  [ 7234] = 0x67B3, [ 7235] = 0x67F7, [ 7236] = 0x67F6, [ 7237] = 0x67EE,
  [ 7238] = 0x67E3, [ 7239] = 0x67C2, [ 7240] = 0x67B9, [ 7241] = 0x67CE,
  [ 7242] = 0x67E7, [ 7243] = 0x67F0, [ 7244] = 0x67B2, [ 7245] = 0x67FC,
  [ 7246] = 0x67C6, [ 7247] = 0x67ED, [ 7248] = 0x67CC, [ 7249] = 0x67AE,
  [ 7250] = 0x67E6, [ 7251] = 0x67DB, [ 7252] = 0x67FA, [ 7253] = 0x67C9,
  [ 7254] = 0x67CA, [ 7255] = 0x67C3, [ 7256] = 0x67EA, [ 7257] = 0x67CB,
  [ 7258] = 0x6B28, [ 7259] = 0x6B82, [ 7260] = 0x6B84, [ 7261] = 0x6BB6,
  [ 7262] = 0x6BD6, [ 7263] = 0x6BD8, [ 7264] = 0x6BE0, [ 7265] = 0x6C20,
  [ 7266] = 0x6C21, [ 7267] = 0x6D28, [ 7268] = 0x6D34, [ 7269] = 0x6D2D,
  [ 7270] = 0x6D1F, [ 7271] = 0x6D3C, [ 7272] = 0x6D3F, [ 7273] = 0x6D12,
  [ 7274] = 0x6D0A, [ 7275] = 0x6CDA, [ 7276] = 0x6D33, [ 7277] = 0x6D04,
  [ 7278] = 0x6D19, [ 7279] = 0x6D3A, [ 7280] = 0x6D1A, [ 7281] = 0x6D11,
  [ 7282] = 0x6D00, [ 7283] = 0x6D1D, [ 7284] = 0x6D42, [ 7285] = 0x6D01,
  [ 7286] = 0x6D18, [ 7287] = 0x6D37, [ 7288] = 0x6D03, [ 7289] = 0x6D0F,
  [ 7290] = 0x6D40, [ 7291] = 0x6D07, [ 7292] = 0x6D20, [ 7293] = 0x6D2C,
  [ 7294] = 0x6D08, [ 7295] = 0x6D22, [ 7296] = 0x6D09, [ 7297] = 0x6D10,
  [ 7298] = 0x70B7, [ 7299] = 0x709F, [ 7300] = 0x70BE, [ 7301] = 0x70B1,
  [ 7302] = 0x70B0, [ 7303] = 0x70A1, [ 7304] = 0x70B4, [ 7305] = 0x70B5,
  [ 7306] = 0x70A9, [ 7307] = 0x7241, [ 7308] = 0x7249, [ 7309] = 0x724A,
  [ 7310] = 0x726C, [ 7311] = 0x7270, [ 7312] = 0x7273, [ 7313] = 0x726E,
  [ 7314] = 0x72CA, [ 7315] = 0x72E4, [ 7316] = 0x72E8, [ 7317] = 0x72EB,
  [ 7318] = 0x72DF, [ 7319] = 0x72EA, [ 7320] = 0x72E6, [ 7321] = 0x72E3,
  [ 7322] = 0x7385, [ 7323] = 0x73CC, [ 7324] = 0x73C2, [ 7325] = 0x73C8,
  [ 7326] = 0x73C5, [ 7327] = 0x73B9, [ 7328] = 0x73B6, [ 7329] = 0x73B5,
  [ 7330] = 0x73B4, [ 7331] = 0x73EB, [ 7332] = 0x73BF, [ 7333] = 0x73C7,
  [ 7334] = 0x73BE, [ 7335] = 0x73C3, [ 7336] = 0x73C6, [ 7337] = 0x73B8,
  [ 7338] = 0x73CB, [ 7339] = 0x74EC, [ 7340] = 0x74EE, [ 7341] = 0x752E,
  [ 7342] = 0x7547, [ 7343] = 0x7548, [ 7344] = 0x75A7, [ 7345] = 0x75AA,
  [ 7346] = 0x7679, [ 7347] = 0x76C4, [ 7348] = 0x7708, [ 7349] = 0x7703,
  [ 7350] = 0x7704, [ 7351] = 0x7705, [ 7352] = 0x770A, [ 7353] = 0x76F7,
  [ 7354] = 0x76FB, [ 7355] = 0x76FA, [ 7356] = 0x77E7, [ 7357] = 0x77E8,
  [ 7358] = 0x7806, [ 7359] = 0x7811, [ 7360] = 0x7812, [ 7361] = 0x7805,
  [ 7362] = 0x7810, [ 7363] = 0x780F, [ 7364] = 0x780E, [ 7365] = 0x7809,
  [ 7366] = 0x7803, [ 7367] = 0x7813, [ 7368] = 0x794A, [ 7369] = 0x794C,
  [ 7370] = 0x794B, [ 7371] = 0x7945, [ 7372] = 0x7944, [ 7373] = 0x79D5,
  [ 7374] = 0x79CD, [ 7375] = 0x79CF, [ 7376] = 0x79D6, [ 7377] = 0x79CE,
  [ 7378] = 0x7A80, [ 7379] = 0x7A7E, [ 7380] = 0x7AD1, [ 7381] = 0x7B00,
  [ 7382] = 0x7B01, [ 7383] = 0x7C7A, [ 7384] = 0x7C78, [ 7385] = 0x7C79,
  [ 7386] = 0x7C7F, [ 7387] = 0x7C80, [ 7388] = 0x7C81, [ 7389] = 0x7D03,
  [ 7390] = 0x7D08, [ 7391] = 0x7D01, [ 7392] = 0x7F58, [ 7393] = 0x7F91,
  [ 7394] = 0x7F8D, [ 7395] = 0x7FBE, [ 7396] = 0x8007, [ 7397] = 0x800E,
  [ 7398] = 0x800F, [ 7399] = 0x8014, [ 7400] = 0x8037, [ 7401] = 0x80D8,
  [ 7402] = 0x80C7, [ 7403] = 0x80E0, [ 7404] = 0x80D1, [ 7405] = 0x80C8,
  [ 7406] = 0x80C2, [ 7407] = 0x80D0, [ 7408] = 0x80C5, [ 7409] = 0x80E3,
  [ 7410] = 0x80D9, [ 7411] = 0x80DC, [ 7412] = 0x80CA, [ 7413] = 0x80D5,
  [ 7414] = 0x80C9, [ 7415] = 0x80CF, [ 7416] = 0x80D7, [ 7417] = 0x80E6,
  [ 7418] = 0x80CD, [ 7419] = 0x81FF, [ 7420] = 0x8221, [ 7421] = 0x8294,
  [ 7422] = 0x82D9, [ 7423] = 0x82FE, [ 7424] = 0x82F9, [ 7425] = 0x8307,
  [ 7426] = 0x82E8, [ 7427] = 0x8300, [ 7428] = 0x82D5, [ 7429] = 0x833A,
  [ 7430] = 0x82EB, [ 7431] = 0x82D6, [ 7432] = 0x82F4, [ 7433] = 0x82EC,
  [ 7434] = 0x82E1, [ 7435] = 0x82F2, [ 7436] = 0x82F5, [ 7437] = 0x830C,
  [ 7438] = 0x82FB, [ 7439] = 0x82F6, [ 7440] = 0x82F0, [ 7441] = 0x82EA,
  [ 7442] = 0x82E4, [ 7443] = 0x82E0, [ 7444] = 0x82FA, [ 7445] = 0x82F3,
  [ 7446] = 0x82ED, [ 7447] = 0x8677, [ 7448] = 0x8674, [ 7449] = 0x867C,
  [ 7450] = 0x8673, [ 7451] = 0x8841, [ 7452] = 0x884E, [ 7453] = 0x8867,
  [ 7454] = 0x886A, [ 7455] = 0x8869, [ 7456] = 0x89D3, [ 7457] = 0x8A04,
  [ 7458] = 0x8A07, [ 7459] = 0x8D72, [ 7460] = 0x8FE3, [ 7461] = 0x8FE1,
  [ 7462] = 0x8FEE, [ 7463] = 0x8FE0, [ 7464] = 0x90F1, [ 7465] = 0x90BD,
  [ 7466] = 0x90BF, [ 7467] = 0x90D5, [ 7468] = 0x90C5, [ 7469] = 0x90BE,
  [ 7470] = 0x90C7, [ 7471] = 0x90CB, [ 7472] = 0x90C8, [ 7473] = 0x91D4,
  [ 7474] = 0x91D3, [ 7475] = 0x9654, [ 7476] = 0x964F, [ 7477] = 0x9651,
  [ 7478] = 0x9653, [ 7479] = 0x964A, [ 7480] = 0x964E, [ 7481] = 0x501E,
  [ 7482] = 0x5005, [ 7483] = 0x5007, [ 7484] = 0x5013, [ 7485] = 0x5022,
  [ 7486] = 0x5030, [ 7487] = 0x501B, [ 7488] = 0x4FF5, [ 7489] = 0x4FF4,
  [ 7490] = 0x5033, [ 7491] = 0x5037, [ 7492] = 0x502C, [ 7493] = 0x4FF6,
  [ 7494] = 0x4FF7, [ 7495] = 0x5017, [ 7496] = 0x501C, [ 7497] = 0x5020,
  [ 7498] = 0x5027, [ 7499] = 0x5035, [ 7500] = 0x502F, [ 7501] = 0x5031,
  [ 7502] = 0x500E, [ 7503] = 0x515A, [ 7504] = 0x5194, [ 7505] = 0x5193,
  [ 7506] = 0x51CA, [ 7507] = 0x51C4, [ 7508] = 0x51C5, [ 7509] = 0x51C8,
  [ 7510] = 0x51CE, [ 7511] = 0x5261, [ 7512] = 0x525A, [ 7513] = 0x5252,
  [ 7514] = 0x525E, [ 7515] = 0x525F, [ 7516] = 0x5255, [ 7517] = 0x5262,
  [ 7518] = 0x52CD, [ 7519] = 0x530E, [ 7520] = 0x539E, [ 7521] = 0x5526,
  [ 7522] = 0x54E2, [ 7523] = 0x5517, [ 7524] = 0x5512, [ 7525] = 0x54E7,
  [ 7526] = 0x54F3, [ 7527] = 0x54E4, [ 7528] = 0x551A, [ 7529] = 0x54FF,
  [ 7530] = 0x5504, [ 7531] = 0x5508, [ 7532] = 0x54EB, [ 7533] = 0x5511,
  [ 7534] = 0x5505, [ 7535] = 0x54F1, [ 7536] = 0x550A, [ 7537] = 0x54FB,
  [ 7538] = 0x54F7, [ 7539] = 0x54F8, [ 7540] = 0x54E0, [ 7541] = 0x550E,
  [ 7542] = 0x5503, [ 7543] = 0x550B, [ 7544] = 0x5701, [ 7545] = 0x5702,
  [ 7546] = 0x57CC, [ 7547] = 0x5832, [ 7548] = 0x57D5, [ 7549] = 0x57D2,
  [ 7550] = 0x57BA, [ 7551] = 0x57C6, [ 7552] = 0x57BD, [ 7553] = 0x57BC,
  [ 7554] = 0x57B8, [ 7555] = 0x57B6, [ 7556] = 0x57BF, [ 7557] = 0x57C7,
  [ 7558] = 0x57D0, [ 7559] = 0x57B9, [ 7560] = 0x57C1, [ 7561] = 0x590E,
  [ 7562] = 0x594A, [ 7563] = 0x5A19, [ 7564] = 0x5A16, [ 7565] = 0x5A2D,
  [ 7566] = 0x5A2E, [ 7567] = 0x5A15, [ 7568] = 0x5A0F, [ 7569] = 0x5A17,
  [ 7570] = 0x5A0A, [ 7571] = 0x5A1E, [ 7572] = 0x5A33, [ 7573] = 0x5B6C,
  [ 7574] = 0x5BA7, [ 7575] = 0x5BAD, [ 7576] = 0x5BAC, [ 7577] = 0x5C03,
  [ 7578] = 0x5C56, [ 7579] = 0x5C54, [ 7580] = 0x5CEC, [ 7581] = 0x5CFF,
  [ 7582] = 0x5CEE, [ 7583] = 0x5CF1, [ 7584] = 0x5CF7, [ 7585] = 0x5D00,
  [ 7586] = 0x5CF9, [ 7587] = 0x5E29, [ 7588] = 0x5E28, [ 7589] = 0x5EA8,
  [ 7590] = 0x5EAE, [ 7591] = 0x5EAA, [ 7592] = 0x5EAC, [ 7593] = 0x5F33,
  [ 7594] = 0x5F30, [ 7595] = 0x5F67, [ 7596] = 0x605D, [ 7597] = 0x605A,
  [ 7598] = 0x6067, [ 7599] = 0x6041, [ 7600] = 0x60A2, [ 7601] = 0x6088,
  [ 7602] = 0x6080, [ 7603] = 0x6092, [ 7604] = 0x6081, [ 7605] = 0x609D,
  [ 7606] = 0x6083, [ 7607] = 0x6095, [ 7608] = 0x609B, [ 7609] = 0x6097,
  [ 7610] = 0x6087, [ 7611] = 0x609C, [ 7612] = 0x608E, [ 7613] = 0x6219,
  [ 7614] = 0x6246, [ 7615] = 0x62F2, [ 7616] = 0x6310, [ 7617] = 0x6356,
  [ 7618] = 0x632C, [ 7619] = 0x6344, [ 7620] = 0x6345, [ 7621] = 0x6336,
  [ 7622] = 0x6343, [ 7623] = 0x63E4, [ 7624] = 0x6339, [ 7625] = 0x634B,
  [ 7626] = 0x634A, [ 7627] = 0x633C, [ 7628] = 0x6329, [ 7629] = 0x6341,
  [ 7630] = 0x6334, [ 7631] = 0x6358, [ 7632] = 0x6354, [ 7633] = 0x6359,
  [ 7634] = 0x632D, [ 7635] = 0x6347, [ 7636] = 0x6333, [ 7637] = 0x635A,
  [ 7638] = 0x6351, [ 7639] = 0x6338, [ 7640] = 0x6357, [ 7641] = 0x6340,
  [ 7642] = 0x6348, [ 7643] = 0x654A, [ 7644] = 0x6546, [ 7645] = 0x65C6,
  [ 7646] = 0x65C3, [ 7647] = 0x65C4, [ 7648] = 0x65C2, [ 7649] = 0x664A,
  [ 7650] = 0x665F, [ 7651] = 0x6647, [ 7652] = 0x6651, [ 7653] = 0x6712,
  [ 7654] = 0x6713, [ 7655] = 0x681F, [ 7656] = 0x681A, [ 7657] = 0x6849,
  [ 7658] = 0x6832, [ 7659] = 0x6833, [ 7660] = 0x683B, [ 7661] = 0x684B,
  [ 7662] = 0x684F, [ 7663] = 0x6816, [ 7664] = 0x6831, [ 7665] = 0x681C,
  [ 7666] = 0x6835, [ 7667] = 0x682B, [ 7668] = 0x682D, [ 7669] = 0x682F,
  [ 7670] = 0x684E, [ 7671] = 0x6844, [ 7672] = 0x6834, [ 7673] = 0x681D,
  [ 7674] = 0x6812, [ 7675] = 0x6814, [ 7676] = 0x6826, [ 7677] = 0x6828,
  [ 7678] = 0x682E, [ 7679] = 0x684D, [ 7680] = 0x683A, [ 7681] = 0x6825,
  [ 7682] = 0x6820, [ 7683] = 0x6B2C, [ 7684] = 0x6B2F, [ 7685] = 0x6B2D,
  [ 7686] = 0x6B31, [ 7687] = 0x6B34, [ 7688] = 0x6B6D, [ 7689] = 0x8082,
  [ 7690] = 0x6B88, [ 7691] = 0x6BE6, [ 7692] = 0x6BE4, [ 7693] = 0x6BE8,
  [ 7694] = 0x6BE3, [ 7695] = 0x6BE2, [ 7696] = 0x6BE7, [ 7697] = 0x6C25,
  [ 7698] = 0x6D7A, [ 7699] = 0x6D63, [ 7700] = 0x6D64, [ 7701] = 0x6D76,
  [ 7702] = 0x6D0D, [ 7703] = 0x6D61, [ 7704] = 0x6D92, [ 7705] = 0x6D58,
  [ 7706] = 0x6D62, [ 7707] = 0x6D6D, [ 7708] = 0x6D6F, [ 7709] = 0x6D91,
  [ 7710] = 0x6D8D, [ 7711] = 0x6DEF, [ 7712] = 0x6D7F, [ 7713] = 0x6D86,
  [ 7714] = 0x6D5E, [ 7715] = 0x6D67, [ 7716] = 0x6D60, [ 7717] = 0x6D97,
  [ 7718] = 0x6D70, [ 7719] = 0x6D7C, [ 7720] = 0x6D5F, [ 7721] = 0x6D82,
  [ 7722] = 0x6D98, [ 7723] = 0x6D2F, [ 7724] = 0x6D68, [ 7725] = 0x6D8B,
  [ 7726] = 0x6D7E, [ 7727] = 0x6D80, [ 7728] = 0x6D84, [ 7729] = 0x6D16,
  [ 7730] = 0x6D83, [ 7731] = 0x6D7B, [ 7732] = 0x6D7D, [ 7733] = 0x6D75,
  [ 7734] = 0x6D90, [ 7735] = 0x70DC, [ 7736] = 0x70D3, [ 7737] = 0x70D1,
  [ 7738] = 0x70DD, [ 7739] = 0x70CB, [ 7740] = 0x7F39, [ 7741] = 0x70E2,
  [ 7742] = 0x70D7, [ 7743] = 0x70D2, [ 7744] = 0x70DE, [ 7745] = 0x70E0,
  [ 7746] = 0x70D4, [ 7747] = 0x70CD, [ 7748] = 0x70C5, [ 7749] = 0x70C6,
  [ 7750] = 0x70C7, [ 7751] = 0x70DA, [ 7752] = 0x70CE, [ 7753] = 0x70E1,
  [ 7754] = 0x7242, [ 7755] = 0x7278, [ 7756] = 0x7277, [ 7757] = 0x7276,
  [ 7758] = 0x7300, [ 7759] = 0x72FA, [ 7760] = 0x72F4, [ 7761] = 0x72FE,
  [ 7762] = 0x72F6, [ 7763] = 0x72F3, [ 7764] = 0x72FB, [ 7765] = 0x7301,
  [ 7766] = 0x73D3, [ 7767] = 0x73D9, [ 7768] = 0x73E5, [ 7769] = 0x73D6,
  [ 7770] = 0x73BC, [ 7771] = 0x73E7, [ 7772] = 0x73E3, [ 7773] = 0x73E9,
  [ 7774] = 0x73DC, [ 7775] = 0x73D2, [ 7776] = 0x73DB, [ 7777] = 0x73D4,
  [ 7778] = 0x73DD, [ 7779] = 0x73DA, [ 7780] = 0x73D7, [ 7781] = 0x73D8,
  [ 7782] = 0x73E8, [ 7783] = 0x74DE, [ 7784] = 0x74DF, [ 7785] = 0x74F4,
  [ 7786] = 0x74F5, [ 7787] = 0x7521, [ 7788] = 0x755B, [ 7789] = 0x755F,
  [ 7790] = 0x75B0, [ 7791] = 0x75C1, [ 7792] = 0x75BB, [ 7793] = 0x75C4,
  [ 7794] = 0x75C0, [ 7795] = 0x75BF, [ 7796] = 0x75B6, [ 7797] = 0x75BA,
  [ 7798] = 0x768A, [ 7799] = 0x76C9, [ 7800] = 0x771D, [ 7801] = 0x771B,
  [ 7802] = 0x7710, [ 7803] = 0x7713, [ 7804] = 0x7712, [ 7805] = 0x7723,
  [ 7806] = 0x7711, [ 7807] = 0x7715, [ 7808] = 0x7719, [ 7809] = 0x771A,
  [ 7810] = 0x7722, [ 7811] = 0x7727, [ 7812] = 0x7823, [ 7813] = 0x782C,
  [ 7814] = 0x7822, [ 7815] = 0x7835, [ 7816] = 0x782F, [ 7817] = 0x7828,
  [ 7818] = 0x782E, [ 7819] = 0x782B, [ 7820] = 0x7821, [ 7821] = 0x7829,
  [ 7822] = 0x7833, [ 7823] = 0x782A, [ 7824] = 0x7831, [ 7825] = 0x7954,
  [ 7826] = 0x795B, [ 7827] = 0x794F, [ 7828] = 0x795C, [ 7829] = 0x7953,
  [ 7830] = 0x7952, [ 7831] = 0x7951, [ 7832] = 0x79EB, [ 7833] = 0x79EC,
  [ 7834] = 0x79E0, [ 7835] = 0x79EE, [ 7836] = 0x79ED, [ 7837] = 0x79EA,
  [ 7838] = 0x79DC, [ 7839] = 0x79DE, [ 7840] = 0x79DD, [ 7841] = 0x7A86,
  [ 7842] = 0x7A89, [ 7843] = 0x7A85, [ 7844] = 0x7A8B, [ 7845] = 0x7A8C,
  [ 7846] = 0x7A8A, [ 7847] = 0x7A87, [ 7848] = 0x7AD8, [ 7849] = 0x7B10,
  [ 7850] = 0x7B04, [ 7851] = 0x7B13, [ 7852] = 0x7B05, [ 7853] = 0x7B0F,
  [ 7854] = 0x7B08, [ 7855] = 0x7B0A, [ 7856] = 0x7B0E, [ 7857] = 0x7B09,
  [ 7858] = 0x7B12, [ 7859] = 0x7C84, [ 7860] = 0x7C91, [ 7861] = 0x7C8A,
  [ 7862] = 0x7C8C, [ 7863] = 0x7C88, [ 7864] = 0x7C8D, [ 7865] = 0x7C85,
  [ 7866] = 0x7D1E, [ 7867] = 0x7D1D, [ 7868] = 0x7D11, [ 7869] = 0x7D0E,
  [ 7870] = 0x7D18, [ 7871] = 0x7D16, [ 7872] = 0x7D13, [ 7873] = 0x7D1F,
  [ 7874] = 0x7D12, [ 7875] = 0x7D0F, [ 7876] = 0x7D0C, [ 7877] = 0x7F5C,
  [ 7878] = 0x7F61, [ 7879] = 0x7F5E, [ 7880] = 0x7F60, [ 7881] = 0x7F5D,
  [ 7882] = 0x7F5B, [ 7883] = 0x7F96, [ 7884] = 0x7F92, [ 7885] = 0x7FC3,
  [ 7886] = 0x7FC2, [ 7887] = 0x7FC0, [ 7888] = 0x8016, [ 7889] = 0x803E,
  [ 7890] = 0x8039, [ 7891] = 0x80FA, [ 7892] = 0x80F2, [ 7893] = 0x80F9,
  [ 7894] = 0x80F5, [ 7895] = 0x8101, [ 7896] = 0x80FB, [ 7897] = 0x8100,
  [ 7898] = 0x8201, [ 7899] = 0x822F, [ 7900] = 0x8225, [ 7901] = 0x8333,
  [ 7902] = 0x832D, [ 7903] = 0x8344, [ 7904] = 0x8319, [ 7905] = 0x8351,
  [ 7906] = 0x8325, [ 7907] = 0x8356, [ 7908] = 0x833F, [ 7909] = 0x8341,
  [ 7910] = 0x8326, [ 7911] = 0x831C, [ 7912] = 0x8322, [ 7913] = 0x8342,
  [ 7914] = 0x834E, [ 7915] = 0x831B, [ 7916] = 0x832A, [ 7917] = 0x8308,
  [ 7918] = 0x833C, [ 7919] = 0x834D, [ 7920] = 0x8316, [ 7921] = 0x8324,
  [ 7922] = 0x8320, [ 7923] = 0x8337, [ 7924] = 0x832F, [ 7925] = 0x8329,
  [ 7926] = 0x8347, [ 7927] = 0x8345, [ 7928] = 0x834C, [ 7929] = 0x8353,
  [ 7930] = 0x831E, [ 7931] = 0x832C, [ 7932] = 0x834B, [ 7933] = 0x8327,
  [ 7934] = 0x8348, [ 7935] = 0x8653, [ 7936] = 0x8652, [ 7937] = 0x86A2,
  [ 7938] = 0x86A8, [ 7939] = 0x8696, [ 7940] = 0x868D, [ 7941] = 0x8691,
  [ 7942] = 0x869E, [ 7943] = 0x8687, [ 7944] = 0x8697, [ 7945] = 0x8686,
  [ 7946] = 0x868B, [ 7947] = 0x869A, [ 7948] = 0x8685, [ 7949] = 0x86A5,
  [ 7950] = 0x8699, [ 7951] = 0x86A1, [ 7952] = 0x86A7, [ 7953] = 0x8695,
  [ 7954] = 0x8698, [ 7955] = 0x868E, [ 7956] = 0x869D, [ 7957] = 0x8690,
  [ 7958] = 0x8694, [ 7959] = 0x8843, [ 7960] = 0x8844, [ 7961] = 0x886D,
  [ 7962] = 0x8875, [ 7963] = 0x8876, [ 7964] = 0x8872, [ 7965] = 0x8880,
  [ 7966] = 0x8871, [ 7967] = 0x887F, [ 7968] = 0x886F, [ 7969] = 0x8883,
  [ 7970] = 0x887E, [ 7971] = 0x8874, [ 7972] = 0x887C, [ 7973] = 0x8A12,
  [ 7974] = 0x8C47, [ 7975] = 0x8C57, [ 7976] = 0x8C7B, [ 7977] = 0x8CA4,
  [ 7978] = 0x8CA3, [ 7979] = 0x8D76, [ 7980] = 0x8D78, [ 7981] = 0x8DB5,
  [ 7982] = 0x8DB7, [ 7983] = 0x8DB6, [ 7984] = 0x8ED1, [ 7985] = 0x8ED3,
  [ 7986] = 0x8FFE, [ 7987] = 0x8FF5, [ 7988] = 0x9002, [ 7989] = 0x8FFF,
  [ 7990] = 0x8FFB, [ 7991] = 0x9004, [ 7992] = 0x8FFC, [ 7993] = 0x8FF6,
  [ 7994] = 0x90D6, [ 7995] = 0x90E0, [ 7996] = 0x90D9, [ 7997] = 0x90DA,
  [ 7998] = 0x90E3, [ 7999] = 0x90DF, [ 8000] = 0x90E5, [ 8001] = 0x90D8,
  [ 8002] = 0x90DB, [ 8003] = 0x90D7, [ 8004] = 0x90DC, [ 8005] = 0x90E4,
  [ 8006] = 0x9150, [ 8007] = 0x914E, [ 8008] = 0x914F, [ 8009] = 0x91D5,
  [ 8010] = 0x91E2, [ 8011] = 0x91DA, [ 8012] = 0x965C, [ 8013] = 0x965F,
  [ 8014] = 0x96BC, [ 8015] = 0x98E3, [ 8016] = 0x9ADF, [ 8017] = 0x9B2F,
  [ 8018] = 0x4E7F, [ 8019] = 0x5070, [ 8020] = 0x506A, [ 8021] = 0x5061,
  [ 8022] = 0x505E, [ 8023] = 0x5060, [ 8024] = 0x5053, [ 8025] = 0x504B,
  [ 8026] = 0x505D, [ 8027] = 0x5072, [ 8028] = 0x5048, [ 8029] = 0x504D,
  [ 8030] = 0x5041, [ 8031] = 0x505B, [ 8032] = 0x504A, [ 8033] = 0x5062,
  [ 8034] = 0x5015, [ 8035] = 0x5045, [ 8036] = 0x505F, [ 8037] = 0x5069,
  [ 8038] = 0x506B, [ 8039] = 0x5063, [ 8040] = 0x5064, [ 8041] = 0x5046,
  [ 8042] = 0x5040, [ 8043] = 0x506E, [ 8044] = 0x5073, [ 8045] = 0x5057,
  [ 8046] = 0x5051, [ 8047] = 0x51D0, [ 8048] = 0x526B, [ 8049] = 0x526D,
  [ 8050] = 0x526C, [ 8051] = 0x526E, [ 8052] = 0x52D6, [ 8053] = 0x52D3,
  [ 8054] = 0x532D, [ 8055] = 0x539C, [ 8056] = 0x5575, [ 8057] = 0x5576,
  [ 8058] = 0x553C, [ 8059] = 0x554D, [ 8060] = 0x5550, [ 8061] = 0x5534,
  [ 8062] = 0x552A, [ 8063] = 0x5551, [ 8064] = 0x5562, [ 8065] = 0x5536,
  [ 8066] = 0x5535, [ 8067] = 0x5530, [ 8068] = 0x5552, [ 8069] = 0x5545,
  [ 8070] = 0x550C, [ 8071] = 0x5532, [ 8072] = 0x5565, [ 8073] = 0x554E,
  [ 8074] = 0x5539, [ 8075] = 0x5548, [ 8076] = 0x552D, [ 8077] = 0x553B,
  [ 8078] = 0x5540, [ 8079] = 0x554B, [ 8080] = 0x570A, [ 8081] = 0x5707,
  [ 8082] = 0x57FB, [ 8083] = 0x5814, [ 8084] = 0x57E2, [ 8085] = 0x57F6,
  [ 8086] = 0x57DC, [ 8087] = 0x57F4, [ 8088] = 0x5800, [ 8089] = 0x57ED,
  [ 8090] = 0x57FD, [ 8091] = 0x5808, [ 8092] = 0x57F8, [ 8093] = 0x580B,
  [ 8094] = 0x57F3, [ 8095] = 0x57CF, [ 8096] = 0x5807, [ 8097] = 0x57EE,
  [ 8098] = 0x57E3, [ 8099] = 0x57F2, [ 8100] = 0x57E5, [ 8101] = 0x57EC,
  [ 8102] = 0x57E1, [ 8103] = 0x580E, [ 8104] = 0x57FC, [ 8105] = 0x5810,
  [ 8106] = 0x57E7, [ 8107] = 0x5801, [ 8108] = 0x580C, [ 8109] = 0x57F1,
  [ 8110] = 0x57E9, [ 8111] = 0x57F0, [ 8112] = 0x580D, [ 8113] = 0x5804,
  [ 8114] = 0x595C, [ 8115] = 0x5A60, [ 8116] = 0x5A58, [ 8117] = 0x5A55,
  [ 8118] = 0x5A67, [ 8119] = 0x5A5E, [ 8120] = 0x5A38, [ 8121] = 0x5A35,
  [ 8122] = 0x5A6D, [ 8123] = 0x5A50, [ 8124] = 0x5A5F, [ 8125] = 0x5A65,
  [ 8126] = 0x5A6C, [ 8127] = 0x5A53, [ 8128] = 0x5A64, [ 8129] = 0x5A57,
  [ 8130] = 0x5A43, [ 8131] = 0x5A5D, [ 8132] = 0x5A52, [ 8133] = 0x5A44,
  [ 8134] = 0x5A5B, [ 8135] = 0x5A48, [ 8136] = 0x5A8E, [ 8137] = 0x5A3E,
  [ 8138] = 0x5A4D, [ 8139] = 0x5A39, [ 8140] = 0x5A4C, [ 8141] = 0x5A70,
  [ 8142] = 0x5A69, [ 8143] = 0x5A47, [ 8144] = 0x5A51, [ 8145] = 0x5A56,
  [ 8146] = 0x5A42, [ 8147] = 0x5A5C, [ 8148] = 0x5B72, [ 8149] = 0x5B6E,
  [ 8150] = 0x5BC1, [ 8151] = 0x5BC0, [ 8152] = 0x5C59, [ 8153] = 0x5D1E,
  [ 8154] = 0x5D0B, [ 8155] = 0x5D1D, [ 8156] = 0x5D1A, [ 8157] = 0x5D20,
  [ 8158] = 0x5D0C, [ 8159] = 0x5D28, [ 8160] = 0x5D0D, [ 8161] = 0x5D26,
  [ 8162] = 0x5D25, [ 8163] = 0x5D0F, [ 8164] = 0x5D30, [ 8165] = 0x5D12,
  [ 8166] = 0x5D23, [ 8167] = 0x5D1F, [ 8168] = 0x5D2E, [ 8169] = 0x5E3E,
  [ 8170] = 0x5E34, [ 8171] = 0x5EB1, [ 8172] = 0x5EB4, [ 8173] = 0x5EB9,
  [ 8174] = 0x5EB2, [ 8175] = 0x5EB3, [ 8176] = 0x5F36, [ 8177] = 0x5F38,
  [ 8178] = 0x5F9B, [ 8179] = 0x5F96, [ 8180] = 0x5F9F, [ 8181] = 0x608A,
  [ 8182] = 0x6090, [ 8183] = 0x6086, [ 8184] = 0x60BE, [ 8185] = 0x60B0,
  [ 8186] = 0x60BA, [ 8187] = 0x60D3, [ 8188] = 0x60D4, [ 8189] = 0x60CF,
  [ 8190] = 0x60E4, [ 8191] = 0x60D9, [ 8192] = 0x60DD, [ 8193] = 0x60C8,
  [ 8194] = 0x60B1, [ 8195] = 0x60DB, [ 8196] = 0x60B7, [ 8197] = 0x60CA,
  [ 8198] = 0x60BF, [ 8199] = 0x60C3, [ 8200] = 0x60CD, [ 8201] = 0x60C0,
  [ 8202] = 0x6332, [ 8203] = 0x6365, [ 8204] = 0x638A, [ 8205] = 0x6382,
  [ 8206] = 0x637D, [ 8207] = 0x63BD, [ 8208] = 0x639E, [ 8209] = 0x63AD,
  [ 8210] = 0x639D, [ 8211] = 0x6397, [ 8212] = 0x63AB, [ 8213] = 0x638E,
  [ 8214] = 0x636F, [ 8215] = 0x6387, [ 8216] = 0x6390, [ 8217] = 0x636E,
  [ 8218] = 0x63AF, [ 8219] = 0x6375, [ 8220] = 0x639C, [ 8221] = 0x636D,
  [ 8222] = 0x63AE, [ 8223] = 0x637C, [ 8224] = 0x63A4, [ 8225] = 0x633B,
  [ 8226] = 0x639F, [ 8227] = 0x6378, [ 8228] = 0x6385, [ 8229] = 0x6381,
  [ 8230] = 0x6391, [ 8231] = 0x638D, [ 8232] = 0x6370, [ 8233] = 0x6553,
  [ 8234] = 0x65CD, [ 8235] = 0x6665, [ 8236] = 0x6661, [ 8237] = 0x665B,
  [ 8238] = 0x6659, [ 8239] = 0x665C, [ 8240] = 0x6662, [ 8241] = 0x6718,
  [ 8242] = 0x6879, [ 8243] = 0x6887, [ 8244] = 0x6890, [ 8245] = 0x689C,
  [ 8246] = 0x686D, [ 8247] = 0x686E, [ 8248] = 0x68AE, [ 8249] = 0x68AB,
  [ 8250] = 0x6956, [ 8251] = 0x686F, [ 8252] = 0x68A3, [ 8253] = 0x68AC,
  [ 8254] = 0x68A9, [ 8255] = 0x6875, [ 8256] = 0x6874, [ 8257] = 0x68B2,
  [ 8258] = 0x688F, [ 8259] = 0x6877, [ 8260] = 0x6892, [ 8261] = 0x687C,
  [ 8262] = 0x686B, [ 8263] = 0x6872, [ 8264] = 0x68AA, [ 8265] = 0x6880,
  [ 8266] = 0x6871, [ 8267] = 0x687E, [ 8268] = 0x689B, [ 8269] = 0x6896,
  [ 8270] = 0x688B, [ 8271] = 0x68A0, [ 8272] = 0x6889, [ 8273] = 0x68A4,
  [ 8274] = 0x6878, [ 8275] = 0x687B, [ 8276] = 0x6891, [ 8277] = 0x688C,
  [ 8278] = 0x688A, [ 8279] = 0x687D, [ 8280] = 0x6B36, [ 8281] = 0x6B33,
  [ 8282] = 0x6B37, [ 8283] = 0x6B38, [ 8284] = 0x6B91, [ 8285] = 0x6B8F,
  [ 8286] = 0x6B8D, [ 8287] = 0x6B8E, [ 8288] = 0x6B8C, [ 8289] = 0x6C2A,
  [ 8290] = 0x6DC0, [ 8291] = 0x6DAB, [ 8292] = 0x6DB4, [ 8293] = 0x6DB3,
  [ 8294] = 0x6E74, [ 8295] = 0x6DAC, [ 8296] = 0x6DE9, [ 8297] = 0x6DE2,
  [ 8298] = 0x6DB7, [ 8299] = 0x6DF6, [ 8300] = 0x6DD4, [ 8301] = 0x6E00,
  [ 8302] = 0x6DC8, [ 8303] = 0x6DE0, [ 8304] = 0x6DDF, [ 8305] = 0x6DD6,
  [ 8306] = 0x6DBE, [ 8307] = 0x6DE5, [ 8308] = 0x6DDC, [ 8309] = 0x6DDD,
  [ 8310] = 0x6DDB, [ 8311] = 0x6DF4, [ 8312] = 0x6DCA, [ 8313] = 0x6DBD,
  [ 8314] = 0x6DED, [ 8315] = 0x6DF0, [ 8316] = 0x6DBA, [ 8317] = 0x6DD5,
  [ 8318] = 0x6DC2, [ 8319] = 0x6DCF, [ 8320] = 0x6DC9, [ 8321] = 0x6DD0,
  [ 8322] = 0x6DF2, [ 8323] = 0x6DD3, [ 8324] = 0x6DFD, [ 8325] = 0x6DD7,
  [ 8326] = 0x6DCD, [ 8327] = 0x6DE3, [ 8328] = 0x6DBB, [ 8329] = 0x70FA,
  [ 8330] = 0x710D, [ 8331] = 0x70F7, [ 8332] = 0x7117, [ 8333] = 0x70F4,
  [ 8334] = 0x710C, [ 8335] = 0x70F0, [ 8336] = 0x7104, [ 8337] = 0x70F3,
  [ 8338] = 0x7110, [ 8339] = 0x70FC, [ 8340] = 0x70FF, [ 8341] = 0x7106,
  [ 8342] = 0x7113, [ 8343] = 0x7100, [ 8344] = 0x70F8, [ 8345] = 0x70F6,
  [ 8346] = 0x710B, [ 8347] = 0x7102, [ 8348] = 0x710E, [ 8349] = 0x727E,
  [ 8350] = 0x727B, [ 8351] = 0x727C, [ 8352] = 0x727F, [ 8353] = 0x731D,
  [ 8354] = 0x7317, [ 8355] = 0x7307, [ 8356] = 0x7311, [ 8357] = 0x7318,
  [ 8358] = 0x730A, [ 8359] = 0x7308, [ 8360] = 0x72FF, [ 8361] = 0x730F,
  [ 8362] = 0x731E, [ 8363] = 0x7388, [ 8364] = 0x73F6, [ 8365] = 0x73F8,
  [ 8366] = 0x73F5, [ 8367] = 0x7404, [ 8368] = 0x7401, [ 8369] = 0x73FD,
  [ 8370] = 0x7407, [ 8371] = 0x7400, [ 8372] = 0x73FA, [ 8373] = 0x73FC,
  [ 8374] = 0x73FF, [ 8375] = 0x740C, [ 8376] = 0x740B, [ 8377] = 0x73F4,
  [ 8378] = 0x7408, [ 8379] = 0x7564, [ 8380] = 0x7563, [ 8381] = 0x75CE,
  [ 8382] = 0x75D2, [ 8383] = 0x75CF, [ 8384] = 0x75CB, [ 8385] = 0x75CC,
  [ 8386] = 0x75D1, [ 8387] = 0x75D0, [ 8388] = 0x768F, [ 8389] = 0x7689,
  [ 8390] = 0x76D3, [ 8391] = 0x7739, [ 8392] = 0x772F, [ 8393] = 0x772D,
  [ 8394] = 0x7731, [ 8395] = 0x7732, [ 8396] = 0x7734, [ 8397] = 0x7733,
  [ 8398] = 0x773D, [ 8399] = 0x7725, [ 8400] = 0x773B, [ 8401] = 0x7735,
  [ 8402] = 0x7848, [ 8403] = 0x7852, [ 8404] = 0x7849, [ 8405] = 0x784D,
  [ 8406] = 0x784A, [ 8407] = 0x784C, [ 8408] = 0x7826, [ 8409] = 0x7845,
  [ 8410] = 0x7850, [ 8411] = 0x7964, [ 8412] = 0x7967, [ 8413] = 0x7969,
  [ 8414] = 0x796A, [ 8415] = 0x7963, [ 8416] = 0x796B, [ 8417] = 0x7961,
  [ 8418] = 0x79BB, [ 8419] = 0x79FA, [ 8420] = 0x79F8, [ 8421] = 0x79F6,
  [ 8422] = 0x79F7, [ 8423] = 0x7A8F, [ 8424] = 0x7A94, [ 8425] = 0x7A90,
  [ 8426] = 0x7B35, [ 8427] = 0x7B47, [ 8428] = 0x7B34, [ 8429] = 0x7B25,
  [ 8430] = 0x7B30, [ 8431] = 0x7B22, [ 8432] = 0x7B24, [ 8433] = 0x7B33,
  [ 8434] = 0x7B18, [ 8435] = 0x7B2A, [ 8436] = 0x7B1D, [ 8437] = 0x7B31,
  [ 8438] = 0x7B2B, [ 8439] = 0x7B2D, [ 8440] = 0x7B2F, [ 8441] = 0x7B32,
  [ 8442] = 0x7B38, [ 8443] = 0x7B1A, [ 8444] = 0x7B23, [ 8445] = 0x7C94,
  [ 8446] = 0x7C98, [ 8447] = 0x7C96, [ 8448] = 0x7CA3, [ 8449] = 0x7D35,
  [ 8450] = 0x7D3D, [ 8451] = 0x7D38, [ 8452] = 0x7D36, [ 8453] = 0x7D3A,
  [ 8454] = 0x7D45, [ 8455] = 0x7D2C, [ 8456] = 0x7D29, [ 8457] = 0x7D41,
  [ 8458] = 0x7D47, [ 8459] = 0x7D3E, [ 8460] = 0x7D3F, [ 8461] = 0x7D4A,
  [ 8462] = 0x7D3B, [ 8463] = 0x7D28, [ 8464] = 0x7F63, [ 8465] = 0x7F95,
  [ 8466] = 0x7F9C, [ 8467] = 0x7F9D, [ 8468] = 0x7F9B, [ 8469] = 0x7FCA,
  [ 8470] = 0x7FCB, [ 8471] = 0x7FCD, [ 8472] = 0x7FD0, [ 8473] = 0x7FD1,
  [ 8474] = 0x7FC7, [ 8475] = 0x7FCF, [ 8476] = 0x7FC9, [ 8477] = 0x801F,
  [ 8478] = 0x801E, [ 8479] = 0x801B, [ 8480] = 0x8047, [ 8481] = 0x8043,
  [ 8482] = 0x8048, [ 8483] = 0x8118, [ 8484] = 0x8125, [ 8485] = 0x8119,
  [ 8486] = 0x811B, [ 8487] = 0x812D, [ 8488] = 0x811F, [ 8489] = 0x812C,
  [ 8490] = 0x811E, [ 8491] = 0x8121, [ 8492] = 0x8115, [ 8493] = 0x8127,
  [ 8494] = 0x811D, [ 8495] = 0x8122, [ 8496] = 0x8211, [ 8497] = 0x8238,
  [ 8498] = 0x8233, [ 8499] = 0x823A, [ 8500] = 0x8234, [ 8501] = 0x8232,
  [ 8502] = 0x8274, [ 8503] = 0x8390, [ 8504] = 0x83A3, [ 8505] = 0x83A8,
  [ 8506] = 0x838D, [ 8507] = 0x837A, [ 8508] = 0x8373, [ 8509] = 0x83A4,
  [ 8510] = 0x8374, [ 8511] = 0x838F, [ 8512] = 0x8381, [ 8513] = 0x8395,
  [ 8514] = 0x8399, [ 8515] = 0x8375, [ 8516] = 0x8394, [ 8517] = 0x83A9,
  [ 8518] = 0x837D, [ 8519] = 0x8383, [ 8520] = 0x838C, [ 8521] = 0x839D,
  [ 8522] = 0x839B, [ 8523] = 0x83AA, [ 8524] = 0x838B, [ 8525] = 0x837E,
  [ 8526] = 0x83A5, [ 8527] = 0x83AF, [ 8528] = 0x8388, [ 8529] = 0x8397,
  [ 8530] = 0x83B0, [ 8531] = 0x837F, [ 8532] = 0x83A6, [ 8533] = 0x8387,
  [ 8534] = 0x83AE, [ 8535] = 0x8376, [ 8536] = 0x839A, [ 8537] = 0x8659,
  [ 8538] = 0x8656, [ 8539] = 0x86BF, [ 8540] = 0x86B7, [ 8541] = 0x86C2,
  [ 8542] = 0x86C1, [ 8543] = 0x86C5, [ 8544] = 0x86BA, [ 8545] = 0x86B0,
  [ 8546] = 0x86C8, [ 8547] = 0x86B9, [ 8548] = 0x86B3, [ 8549] = 0x86B8,
  [ 8550] = 0x86CC, [ 8551] = 0x86B4, [ 8552] = 0x86BB, [ 8553] = 0x86BC,
  [ 8554] = 0x86C3, [ 8555] = 0x86BD, [ 8556] = 0x86BE, [ 8557] = 0x8852,
  [ 8558] = 0x8889, [ 8559] = 0x8895, [ 8560] = 0x88A8, [ 8561] = 0x88A2,
  [ 8562] = 0x88AA, [ 8563] = 0x889A, [ 8564] = 0x8891, [ 8565] = 0x88A1,
  [ 8566] = 0x889F, [ 8567] = 0x8898, [ 8568] = 0x88A7, [ 8569] = 0x8899,
  [ 8570] = 0x889B, [ 8571] = 0x8897, [ 8572] = 0x88A4, [ 8573] = 0x88AC,
  [ 8574] = 0x888C, [ 8575] = 0x8893, [ 8576] = 0x888E, [ 8577] = 0x8982,
  [ 8578] = 0x89D6, [ 8579] = 0x89D9, [ 8580] = 0x89D5, [ 8581] = 0x8A30,
  [ 8582] = 0x8A27, [ 8583] = 0x8A2C, [ 8584] = 0x8A1E, [ 8585] = 0x8C39,
  [ 8586] = 0x8C3B, [ 8587] = 0x8C5C, [ 8588] = 0x8C5D, [ 8589] = 0x8C7D,
  [ 8590] = 0x8CA5, [ 8591] = 0x8D7D, [ 8592] = 0x8D7B, [ 8593] = 0x8D79,
  [ 8594] = 0x8DBC, [ 8595] = 0x8DC2, [ 8596] = 0x8DB9, [ 8597] = 0x8DBF,
  [ 8598] = 0x8DC1, [ 8599] = 0x8ED8, [ 8600] = 0x8EDE, [ 8601] = 0x8EDD,
  [ 8602] = 0x8EDC, [ 8603] = 0x8ED7, [ 8604] = 0x8EE0, [ 8605] = 0x8EE1,
  [ 8606] = 0x9024, [ 8607] = 0x900B, [ 8608] = 0x9011, [ 8609] = 0x901C,
  [ 8610] = 0x900C, [ 8611] = 0x9021, [ 8612] = 0x90EF, [ 8613] = 0x90EA,
  [ 8614] = 0x90F0, [ 8615] = 0x90F4, [ 8616] = 0x90F2, [ 8617] = 0x90F3,
  [ 8618] = 0x90D4, [ 8619] = 0x90EB, [ 8620] = 0x90EC, [ 8621] = 0x90E9,
  [ 8622] = 0x9156, [ 8623] = 0x9158, [ 8624] = 0x915A, [ 8625] = 0x9153,
  [ 8626] = 0x9155, [ 8627] = 0x91EC, [ 8628] = 0x91F4, [ 8629] = 0x91F1,
  [ 8630] = 0x91F3, [ 8631] = 0x91F8, [ 8632] = 0x91E4, [ 8633] = 0x91F9,
  [ 8634] = 0x91EA, [ 8635] = 0x91EB, [ 8636] = 0x91F7, [ 8637] = 0x91E8,
  [ 8638] = 0x91EE, [ 8639] = 0x957A, [ 8640] = 0x9586, [ 8641] = 0x9588,
  [ 8642] = 0x967C, [ 8643] = 0x966D, [ 8644] = 0x966B, [ 8645] = 0x9671,
  [ 8646] = 0x966F, [ 8647] = 0x96BF, [ 8648] = 0x976A, [ 8649] = 0x9804,
  [ 8650] = 0x98E5, [ 8651] = 0x9997, [ 8652] = 0x509B, [ 8653] = 0x5095,
  [ 8654] = 0x5094, [ 8655] = 0x509E, [ 8656] = 0x508B, [ 8657] = 0x50A3,
  [ 8658] = 0x5083, [ 8659] = 0x508C, [ 8660] = 0x508E, [ 8661] = 0x509D,
  [ 8662] = 0x5068, [ 8663] = 0x509C, [ 8664] = 0x5092, [ 8665] = 0x5082,
  [ 8666] = 0x5087, [ 8667] = 0x515F, [ 8668] = 0x51D4, [ 8669] = 0x5312,
  [ 8670] = 0x5311, [ 8671] = 0x53A4, [ 8672] = 0x53A7, [ 8673] = 0x5591,
  [ 8674] = 0x55A8, [ 8675] = 0x55A5, [ 8676] = 0x55AD, [ 8677] = 0x5577,
  [ 8678] = 0x5645, [ 8679] = 0x55A2, [ 8680] = 0x5593, [ 8681] = 0x5588,
  [ 8682] = 0x558F, [ 8683] = 0x55B5, [ 8684] = 0x5581, [ 8685] = 0x55A3,
  [ 8686] = 0x5592, [ 8687] = 0x55A4, [ 8688] = 0x557D, [ 8689] = 0x558C,
  [ 8690] = 0x55A6, [ 8691] = 0x557F, [ 8692] = 0x5595, [ 8693] = 0x55A1,
  [ 8694] = 0x558E, [ 8695] = 0x570C, [ 8696] = 0x5829, [ 8697] = 0x5837,
  [ 8698] = 0x5819, [ 8699] = 0x581E, [ 8700] = 0x5827, [ 8701] = 0x5823,
  [ 8702] = 0x5828, [ 8703] = 0x57F5, [ 8704] = 0x5848, [ 8705] = 0x5825,
  [ 8706] = 0x581C, [ 8707] = 0x581B, [ 8708] = 0x5833, [ 8709] = 0x583F,
  [ 8710] = 0x5836, [ 8711] = 0x582E, [ 8712] = 0x5839, [ 8713] = 0x5838,
  [ 8714] = 0x582D, [ 8715] = 0x582C, [ 8716] = 0x583B, [ 8717] = 0x5961,
  [ 8718] = 0x5AAF, [ 8719] = 0x5A94, [ 8720] = 0x5A9F, [ 8721] = 0x5A7A,
  [ 8722] = 0x5AA2, [ 8723] = 0x5A9E, [ 8724] = 0x5A78, [ 8725] = 0x5AA6,
  [ 8726] = 0x5A7C, [ 8727] = 0x5AA5, [ 8728] = 0x5AAC, [ 8729] = 0x5A95,
  [ 8730] = 0x5AAE, [ 8731] = 0x5A37, [ 8732] = 0x5A84, [ 8733] = 0x5A8A,
  [ 8734] = 0x5A97, [ 8735] = 0x5A83, [ 8736] = 0x5A8B, [ 8737] = 0x5AA9,
  [ 8738] = 0x5A7B, [ 8739] = 0x5A7D, [ 8740] = 0x5A8C, [ 8741] = 0x5A9C,
  [ 8742] = 0x5A8F, [ 8743] = 0x5A93, [ 8744] = 0x5A9D, [ 8745] = 0x5BEA,
  [ 8746] = 0x5BCD, [ 8747] = 0x5BCB, [ 8748] = 0x5BD4, [ 8749] = 0x5BD1,
  [ 8750] = 0x5BCA, [ 8751] = 0x5BCE, [ 8752] = 0x5C0C, [ 8753] = 0x5C30,
  [ 8754] = 0x5D37, [ 8755] = 0x5D43, [ 8756] = 0x5D6B, [ 8757] = 0x5D41,
  [ 8758] = 0x5D4B, [ 8759] = 0x5D3F, [ 8760] = 0x5D35, [ 8761] = 0x5D51,
  [ 8762] = 0x5D4E, [ 8763] = 0x5D55, [ 8764] = 0x5D33, [ 8765] = 0x5D3A,
  [ 8766] = 0x5D52, [ 8767] = 0x5D3D, [ 8768] = 0x5D31, [ 8769] = 0x5D59,
  [ 8770] = 0x5D42, [ 8771] = 0x5D39, [ 8772] = 0x5D49, [ 8773] = 0x5D38,
  [ 8774] = 0x5D3C, [ 8775] = 0x5D32, [ 8776] = 0x5D36, [ 8777] = 0x5D40,
  [ 8778] = 0x5D45, [ 8779] = 0x5E44, [ 8780] = 0x5E41, [ 8781] = 0x5F58,
  [ 8782] = 0x5FA6, [ 8783] = 0x5FA5, [ 8784] = 0x5FAB, [ 8785] = 0x60C9,
  [ 8786] = 0x60B9, [ 8787] = 0x60CC, [ 8788] = 0x60E2, [ 8789] = 0x60CE,
  [ 8790] = 0x60C4, [ 8791] = 0x6114, [ 8792] = 0x60F2, [ 8793] = 0x610A,
  [ 8794] = 0x6116, [ 8795] = 0x6105, [ 8796] = 0x60F5, [ 8797] = 0x6113,
  [ 8798] = 0x60F8, [ 8799] = 0x60FC, [ 8800] = 0x60FE, [ 8801] = 0x60C1,
  [ 8802] = 0x6103, [ 8803] = 0x6118, [ 8804] = 0x611D, [ 8805] = 0x6110,
  [ 8806] = 0x60FF, [ 8807] = 0x6104, [ 8808] = 0x610B, [ 8809] = 0x624A,
  [ 8810] = 0x6394, [ 8811] = 0x63B1, [ 8812] = 0x63B0, [ 8813] = 0x63CE,
  [ 8814] = 0x63E5, [ 8815] = 0x63E8, [ 8816] = 0x63EF, [ 8817] = 0x63C3,
  [ 8818] = 0x649D, [ 8819] = 0x63F3, [ 8820] = 0x63CA, [ 8821] = 0x63E0,
  [ 8822] = 0x63F6, [ 8823] = 0x63D5, [ 8824] = 0x63F2, [ 8825] = 0x63F5,
  [ 8826] = 0x6461, [ 8827] = 0x63DF, [ 8828] = 0x63BE, [ 8829] = 0x63DD,
  [ 8830] = 0x63DC, [ 8831] = 0x63C4, [ 8832] = 0x63D8, [ 8833] = 0x63D3,
  [ 8834] = 0x63C2, [ 8835] = 0x63C7, [ 8836] = 0x63CC, [ 8837] = 0x63CB,
  [ 8838] = 0x63C8, [ 8839] = 0x63F0, [ 8840] = 0x63D7, [ 8841] = 0x63D9,
  [ 8842] = 0x6532, [ 8843] = 0x6567, [ 8844] = 0x656A, [ 8845] = 0x6564,
  [ 8846] = 0x655C, [ 8847] = 0x6568, [ 8848] = 0x6565, [ 8849] = 0x658C,
  [ 8850] = 0x659D, [ 8851] = 0x659E, [ 8852] = 0x65AE, [ 8853] = 0x65D0,
  [ 8854] = 0x65D2, [ 8855] = 0x667C, [ 8856] = 0x666C, [ 8857] = 0x667B,
  [ 8858] = 0x6680, [ 8859] = 0x6671, [ 8860] = 0x6679, [ 8861] = 0x666A,
  [ 8862] = 0x6672, [ 8863] = 0x6701, [ 8864] = 0x690C, [ 8865] = 0x68D3,
  [ 8866] = 0x6904, [ 8867] = 0x68DC, [ 8868] = 0x692A, [ 8869] = 0x68EC,
  [ 8870] = 0x68EA, [ 8871] = 0x68F1, [ 8872] = 0x690F, [ 8873] = 0x68D6,
  [ 8874] = 0x68F7, [ 8875] = 0x68EB, [ 8876] = 0x68E4, [ 8877] = 0x68F6,
  [ 8878] = 0x6913, [ 8879] = 0x6910, [ 8880] = 0x68F3, [ 8881] = 0x68E1,
  [ 8882] = 0x6907, [ 8883] = 0x68CC, [ 8884] = 0x6908, [ 8885] = 0x6970,
  [ 8886] = 0x68B4, [ 8887] = 0x6911, [ 8888] = 0x68EF, [ 8889] = 0x68C6,
  [ 8890] = 0x6914, [ 8891] = 0x68F8, [ 8892] = 0x68D0, [ 8893] = 0x68FD,
  [ 8894] = 0x68FC, [ 8895] = 0x68E8, [ 8896] = 0x690B, [ 8897] = 0x690A,
  [ 8898] = 0x6917, [ 8899] = 0x68CE, [ 8900] = 0x68C8, [ 8901] = 0x68DD,
  [ 8902] = 0x68DE, [ 8903] = 0x68E6, [ 8904] = 0x68F4, [ 8905] = 0x68D1,
  [ 8906] = 0x6906, [ 8907] = 0x68D4, [ 8908] = 0x68E9, [ 8909] = 0x6915,
  [ 8910] = 0x6925, [ 8911] = 0x68C7, [ 8912] = 0x6B39, [ 8913] = 0x6B3B,
  [ 8914] = 0x6B3F, [ 8915] = 0x6B3C, [ 8916] = 0x6B94, [ 8917] = 0x6B97,
  [ 8918] = 0x6B99, [ 8919] = 0x6B95, [ 8920] = 0x6BBD, [ 8921] = 0x6BF0,
  [ 8922] = 0x6BF2, [ 8923] = 0x6BF3, [ 8924] = 0x6C30, [ 8925] = 0x6DFC,
  [ 8926] = 0x6E46, [ 8927] = 0x6E47, [ 8928] = 0x6E1F, [ 8929] = 0x6E49,
  [ 8930] = 0x6E88, [ 8931] = 0x6E3C, [ 8932] = 0x6E3D, [ 8933] = 0x6E45,
  [ 8934] = 0x6E62, [ 8935] = 0x6E2B, [ 8936] = 0x6E3F, [ 8937] = 0x6E41,
  [ 8938] = 0x6E5D, [ 8939] = 0x6E73, [ 8940] = 0x6E1C, [ 8941] = 0x6E33,
  [ 8942] = 0x6E4B, [ 8943] = 0x6E40, [ 8944] = 0x6E51, [ 8945] = 0x6E3B,
  [ 8946] = 0x6E03, [ 8947] = 0x6E2E, [ 8948] = 0x6E5E, [ 8949] = 0x6E68,
  [ 8950] = 0x6E5C, [ 8951] = 0x6E61, [ 8952] = 0x6E31, [ 8953] = 0x6E28,
  [ 8954] = 0x6E60, [ 8955] = 0x6E71, [ 8956] = 0x6E6B, [ 8957] = 0x6E39,
  [ 8958] = 0x6E22, [ 8959] = 0x6E30, [ 8960] = 0x6E53, [ 8961] = 0x6E65,
  [ 8962] = 0x6E27, [ 8963] = 0x6E78, [ 8964] = 0x6E64, [ 8965] = 0x6E77,
  [ 8966] = 0x6E55, [ 8967] = 0x6E79, [ 8968] = 0x6E52, [ 8969] = 0x6E66,
  [ 8970] = 0x6E35, [ 8971] = 0x6E36, [ 8972] = 0x6E5A, [ 8973] = 0x7120,
  [ 8974] = 0x711E, [ 8975] = 0x712F, [ 8976] = 0x70FB, [ 8977] = 0x712E,
  [ 8978] = 0x7131, [ 8979] = 0x7123, [ 8980] = 0x7125, [ 8981] = 0x7122,
  [ 8982] = 0x7132, [ 8983] = 0x711F, [ 8984] = 0x7128, [ 8985] = 0x713A,
  [ 8986] = 0x711B, [ 8987] = 0x724B, [ 8988] = 0x725A, [ 8989] = 0x7288,
  [ 8990] = 0x7289, [ 8991] = 0x7286, [ 8992] = 0x7285, [ 8993] = 0x728B,
  [ 8994] = 0x7312, [ 8995] = 0x730B, [ 8996] = 0x7330, [ 8997] = 0x7322,
  [ 8998] = 0x7331, [ 8999] = 0x7333, [ 9000] = 0x7327, [ 9001] = 0x7332,
  [ 9002] = 0x732D, [ 9003] = 0x7326, [ 9004] = 0x7323, [ 9005] = 0x7335,
  [ 9006] = 0x730C, [ 9007] = 0x742E, [ 9008] = 0x742C, [ 9009] = 0x7430,
  [ 9010] = 0x742B, [ 9011] = 0x7416, [ 9012] = 0x741A, [ 9013] = 0x7421,
  [ 9014] = 0x742D, [ 9015] = 0x7431, [ 9016] = 0x7424, [ 9017] = 0x7423,
  [ 9018] = 0x741D, [ 9019] = 0x7429, [ 9020] = 0x7420, [ 9021] = 0x7432,
  [ 9022] = 0x74FB, [ 9023] = 0x752F, [ 9024] = 0x756F, [ 9025] = 0x756C,
  [ 9026] = 0x75E7, [ 9027] = 0x75DA, [ 9028] = 0x75E1, [ 9029] = 0x75E6,
  [ 9030] = 0x75DD, [ 9031] = 0x75DF, [ 9032] = 0x75E4, [ 9033] = 0x75D7,
  [ 9034] = 0x7695, [ 9035] = 0x7692, [ 9036] = 0x76DA, [ 9037] = 0x7746,
  [ 9038] = 0x7747, [ 9039] = 0x7744, [ 9040] = 0x774D, [ 9041] = 0x7745,
  [ 9042] = 0x774A, [ 9043] = 0x774E, [ 9044] = 0x774B, [ 9045] = 0x774C,
  [ 9046] = 0x77DE, [ 9047] = 0x77EC, [ 9048] = 0x7860, [ 9049] = 0x7864,
  [ 9050] = 0x7865, [ 9051] = 0x785C, [ 9052] = 0x786D, [ 9053] = 0x7871,
  [ 9054] = 0x786A, [ 9055] = 0x786E, [ 9056] = 0x7870, [ 9057] = 0x7869,
  [ 9058] = 0x7868, [ 9059] = 0x785E, [ 9060] = 0x7862, [ 9061] = 0x7974,
  [ 9062] = 0x7973, [ 9063] = 0x7972, [ 9064] = 0x7970, [ 9065] = 0x7A02,
  [ 9066] = 0x7A0A, [ 9067] = 0x7A03, [ 9068] = 0x7A0C, [ 9069] = 0x7A04,
  [ 9070] = 0x7A99, [ 9071] = 0x7AE6, [ 9072] = 0x7AE4, [ 9073] = 0x7B4A,
  [ 9074] = 0x7B3B, [ 9075] = 0x7B44, [ 9076] = 0x7B48, [ 9077] = 0x7B4C,
  [ 9078] = 0x7B4E, [ 9079] = 0x7B40, [ 9080] = 0x7B58, [ 9081] = 0x7B45,
  [ 9082] = 0x7CA2, [ 9083] = 0x7C9E, [ 9084] = 0x7CA8, [ 9085] = 0x7CA1,
  [ 9086] = 0x7D58, [ 9087] = 0x7D6F, [ 9088] = 0x7D63, [ 9089] = 0x7D53,
  [ 9090] = 0x7D56, [ 9091] = 0x7D67, [ 9092] = 0x7D6A, [ 9093] = 0x7D4F,
  [ 9094] = 0x7D6D, [ 9095] = 0x7D5C, [ 9096] = 0x7D6B, [ 9097] = 0x7D52,
  [ 9098] = 0x7D54, [ 9099] = 0x7D69, [ 9100] = 0x7D51, [ 9101] = 0x7D5F,
  [ 9102] = 0x7D4E, [ 9103] = 0x7F3E, [ 9104] = 0x7F3F, [ 9105] = 0x7F65,
  [ 9106] = 0x7F66, [ 9107] = 0x7FA2, [ 9108] = 0x7FA0, [ 9109] = 0x7FA1,
  [ 9110] = 0x7FD7, [ 9111] = 0x8051, [ 9112] = 0x804F, [ 9113] = 0x8050,
  [ 9114] = 0x80FE, [ 9115] = 0x80D4, [ 9116] = 0x8143, [ 9117] = 0x814A,
  [ 9118] = 0x8152, [ 9119] = 0x814F, [ 9120] = 0x8147, [ 9121] = 0x813D,
  [ 9122] = 0x814D, [ 9123] = 0x813A, [ 9124] = 0x81E6, [ 9125] = 0x81EE,
  [ 9126] = 0x81F7, [ 9127] = 0x81F8, [ 9128] = 0x81F9, [ 9129] = 0x8204,
  [ 9130] = 0x823C, [ 9131] = 0x823D, [ 9132] = 0x823F, [ 9133] = 0x8275,
  [ 9134] = 0x833B, [ 9135] = 0x83CF, [ 9136] = 0x83F9, [ 9137] = 0x8423,
  [ 9138] = 0x83C0, [ 9139] = 0x83E8, [ 9140] = 0x8412, [ 9141] = 0x83E7,
  [ 9142] = 0x83E4, [ 9143] = 0x83FC, [ 9144] = 0x83F6, [ 9145] = 0x8410,
  [ 9146] = 0x83C6, [ 9147] = 0x83C8, [ 9148] = 0x83EB, [ 9149] = 0x83E3,
  [ 9150] = 0x83BF, [ 9151] = 0x8401, [ 9152] = 0x83DD, [ 9153] = 0x83E5,
  [ 9154] = 0x83D8, [ 9155] = 0x83FF, [ 9156] = 0x83E1, [ 9157] = 0x83CB,
  [ 9158] = 0x83CE, [ 9159] = 0x83D6, [ 9160] = 0x83F5, [ 9161] = 0x83C9,
  [ 9162] = 0x8409, [ 9163] = 0x840F, [ 9164] = 0x83DE, [ 9165] = 0x8411,
  [ 9166] = 0x8406, [ 9167] = 0x83C2, [ 9168] = 0x83F3, [ 9169] = 0x83D5,
  [ 9170] = 0x83FA, [ 9171] = 0x83C7, [ 9172] = 0x83D1, [ 9173] = 0x83EA,
  [ 9174] = 0x8413, [ 9175] = 0x83C3, [ 9176] = 0x83EC, [ 9177] = 0x83EE,
  [ 9178] = 0x83C4, [ 9179] = 0x83FB, [ 9180] = 0x83D7, [ 9181] = 0x83E2,
  [ 9182] = 0x841B, [ 9183] = 0x83DB, [ 9184] = 0x83FE, [ 9185] = 0x86D8,
  [ 9186] = 0x86E2, [ 9187] = 0x86E6, [ 9188] = 0x86D3, [ 9189] = 0x86E3,
  [ 9190] = 0x86DA, [ 9191] = 0x86EA, [ 9192] = 0x86DD, [ 9193] = 0x86EB,
  [ 9194] = 0x86DC, [ 9195] = 0x86EC, [ 9196] = 0x86E9, [ 9197] = 0x86D7,
  [ 9198] = 0x86E8, [ 9199] = 0x86D1, [ 9200] = 0x8848, [ 9201] = 0x8856,
  [ 9202] = 0x8855, [ 9203] = 0x88BA, [ 9204] = 0x88D7, [ 9205] = 0x88B9,
  [ 9206] = 0x88B8, [ 9207] = 0x88C0, [ 9208] = 0x88BE, [ 9209] = 0x88B6,
  [ 9210] = 0x88BC, [ 9211] = 0x88B7, [ 9212] = 0x88BD, [ 9213] = 0x88B2,
  [ 9214] = 0x8901, [ 9215] = 0x88C9, [ 9216] = 0x8995, [ 9217] = 0x8998,
  [ 9218] = 0x8997, [ 9219] = 0x89DD, [ 9220] = 0x89DA, [ 9221] = 0x89DB,
  [ 9222] = 0x8A4E, [ 9223] = 0x8A4D, [ 9224] = 0x8A39, [ 9225] = 0x8A59,
  [ 9226] = 0x8A40, [ 9227] = 0x8A57, [ 9228] = 0x8A58, [ 9229] = 0x8A44,
  [ 9230] = 0x8A45, [ 9231] = 0x8A52, [ 9232] = 0x8A48, [ 9233] = 0x8A51,
  [ 9234] = 0x8A4A, [ 9235] = 0x8A4C, [ 9236] = 0x8A4F, [ 9237] = 0x8C5F,
  [ 9238] = 0x8C81, [ 9239] = 0x8C80, [ 9240] = 0x8CBA, [ 9241] = 0x8CBE,
  [ 9242] = 0x8CB0, [ 9243] = 0x8CB9, [ 9244] = 0x8CB5, [ 9245] = 0x8D84,
  [ 9246] = 0x8D80, [ 9247] = 0x8D89, [ 9248] = 0x8DD8, [ 9249] = 0x8DD3,
  [ 9250] = 0x8DCD, [ 9251] = 0x8DC7, [ 9252] = 0x8DD6, [ 9253] = 0x8DDC,
  [ 9254] = 0x8DCF, [ 9255] = 0x8DD5, [ 9256] = 0x8DD9, [ 9257] = 0x8DC8,
  [ 9258] = 0x8DD7, [ 9259] = 0x8DC5, [ 9260] = 0x8EEF, [ 9261] = 0x8EF7,
  [ 9262] = 0x8EFA, [ 9263] = 0x8EF9, [ 9264] = 0x8EE6, [ 9265] = 0x8EEE,
  [ 9266] = 0x8EE5, [ 9267] = 0x8EF5, [ 9268] = 0x8EE7, [ 9269] = 0x8EE8,
  [ 9270] = 0x8EF6, [ 9271] = 0x8EEB, [ 9272] = 0x8EF1, [ 9273] = 0x8EEC,
  [ 9274] = 0x8EF4, [ 9275] = 0x8EE9, [ 9276] = 0x902D, [ 9277] = 0x9034,
  [ 9278] = 0x902F, [ 9279] = 0x9106, [ 9280] = 0x912C, [ 9281] = 0x9104,
  [ 9282] = 0x90FF, [ 9283] = 0x90FC, [ 9284] = 0x9108, [ 9285] = 0x90F9,
  [ 9286] = 0x90FB, [ 9287] = 0x9101, [ 9288] = 0x9100, [ 9289] = 0x9107,
  [ 9290] = 0x9105, [ 9291] = 0x9103, [ 9292] = 0x9161, [ 9293] = 0x9164,
  [ 9294] = 0x915F, [ 9295] = 0x9162, [ 9296] = 0x9160, [ 9297] = 0x9201,
  [ 9298] = 0x920A, [ 9299] = 0x9225, [ 9300] = 0x9203, [ 9301] = 0x921A,
  [ 9302] = 0x9226, [ 9303] = 0x920F, [ 9304] = 0x920C, [ 9305] = 0x9200,
  [ 9306] = 0x9212, [ 9307] = 0x91FF, [ 9308] = 0x91FD, [ 9309] = 0x9206,
  [ 9310] = 0x9204, [ 9311] = 0x9227, [ 9312] = 0x9202, [ 9313] = 0x921C,
  [ 9314] = 0x9224, [ 9315] = 0x9219, [ 9316] = 0x9217, [ 9317] = 0x9205,
  [ 9318] = 0x9216, [ 9319] = 0x957B, [ 9320] = 0x958D, [ 9321] = 0x958C,
  [ 9322] = 0x9590, [ 9323] = 0x9687, [ 9324] = 0x967E, [ 9325] = 0x9688,
  [ 9326] = 0x9689, [ 9327] = 0x9683, [ 9328] = 0x9680, [ 9329] = 0x96C2,
  [ 9330] = 0x96C8, [ 9331] = 0x96C3, [ 9332] = 0x96F1, [ 9333] = 0x96F0,
  [ 9334] = 0x976C, [ 9335] = 0x9770, [ 9336] = 0x976E, [ 9337] = 0x9807,
  [ 9338] = 0x98A9, [ 9339] = 0x98EB, [ 9340] = 0x9CE6, [ 9341] = 0x9EF9,
  [ 9342] = 0x4E83, [ 9343] = 0x4E84, [ 9344] = 0x4EB6, [ 9345] = 0x50BD,
  [ 9346] = 0x50BF, [ 9347] = 0x50C6, [ 9348] = 0x50AE, [ 9349] = 0x50C4,
  [ 9350] = 0x50CA, [ 9351] = 0x50B4, [ 9352] = 0x50C8, [ 9353] = 0x50C2,
  [ 9354] = 0x50B0, [ 9355] = 0x50C1, [ 9356] = 0x50BA, [ 9357] = 0x50B1,
  [ 9358] = 0x50CB, [ 9359] = 0x50C9, [ 9360] = 0x50B6, [ 9361] = 0x50B8,
  [ 9362] = 0x51D7, [ 9363] = 0x527A, [ 9364] = 0x5278, [ 9365] = 0x527B,
  [ 9366] = 0x527C, [ 9367] = 0x55C3, [ 9368] = 0x55DB, [ 9369] = 0x55CC,
  [ 9370] = 0x55D0, [ 9371] = 0x55CB, [ 9372] = 0x55CA, [ 9373] = 0x55DD,
  [ 9374] = 0x55C0, [ 9375] = 0x55D4, [ 9376] = 0x55C4, [ 9377] = 0x55E9,
  [ 9378] = 0x55BF, [ 9379] = 0x55D2, [ 9380] = 0x558D, [ 9381] = 0x55CF,
  [ 9382] = 0x55D5, [ 9383] = 0x55E2, [ 9384] = 0x55D6, [ 9385] = 0x55C8,
  [ 9386] = 0x55F2, [ 9387] = 0x55CD, [ 9388] = 0x55D9, [ 9389] = 0x55C2,
  [ 9390] = 0x5714, [ 9391] = 0x5853, [ 9392] = 0x5868, [ 9393] = 0x5864,
  [ 9394] = 0x584F, [ 9395] = 0x584D, [ 9396] = 0x5849, [ 9397] = 0x586F,
  [ 9398] = 0x5855, [ 9399] = 0x584E, [ 9400] = 0x585D, [ 9401] = 0x5859,
  [ 9402] = 0x5865, [ 9403] = 0x585B, [ 9404] = 0x583D, [ 9405] = 0x5863,
  [ 9406] = 0x5871, [ 9407] = 0x58FC, [ 9408] = 0x5AC7, [ 9409] = 0x5AC4,
  [ 9410] = 0x5ACB, [ 9411] = 0x5ABA, [ 9412] = 0x5AB8, [ 9413] = 0x5AB1,
  [ 9414] = 0x5AB5, [ 9415] = 0x5AB0, [ 9416] = 0x5ABF, [ 9417] = 0x5AC8,
  [ 9418] = 0x5ABB, [ 9419] = 0x5AC6, [ 9420] = 0x5AB7, [ 9421] = 0x5AC0,
  [ 9422] = 0x5ACA, [ 9423] = 0x5AB4, [ 9424] = 0x5AB6, [ 9425] = 0x5ACD,
  [ 9426] = 0x5AB9, [ 9427] = 0x5A90, [ 9428] = 0x5BD6, [ 9429] = 0x5BD8,
  [ 9430] = 0x5BD9, [ 9431] = 0x5C1F, [ 9432] = 0x5C33, [ 9433] = 0x5D71,
  [ 9434] = 0x5D63, [ 9435] = 0x5D4A, [ 9436] = 0x5D65, [ 9437] = 0x5D72,
  [ 9438] = 0x5D6C, [ 9439] = 0x5D5E, [ 9440] = 0x5D68, [ 9441] = 0x5D67,
  [ 9442] = 0x5D62, [ 9443] = 0x5DF0, [ 9444] = 0x5E4F, [ 9445] = 0x5E4E,
  [ 9446] = 0x5E4A, [ 9447] = 0x5E4D, [ 9448] = 0x5E4B, [ 9449] = 0x5EC5,
  [ 9450] = 0x5ECC, [ 9451] = 0x5EC6, [ 9452] = 0x5ECB, [ 9453] = 0x5EC7,
  [ 9454] = 0x5F40, [ 9455] = 0x5FAF, [ 9456] = 0x5FAD, [ 9457] = 0x60F7,
  [ 9458] = 0x6149, [ 9459] = 0x614A, [ 9460] = 0x612B, [ 9461] = 0x6145,
  [ 9462] = 0x6136, [ 9463] = 0x6132, [ 9464] = 0x612E, [ 9465] = 0x6146,
  [ 9466] = 0x612F, [ 9467] = 0x614F, [ 9468] = 0x6129, [ 9469] = 0x6140,
  [ 9470] = 0x6220, [ 9471] = 0x9168, [ 9472] = 0x6223, [ 9473] = 0x6225,
  [ 9474] = 0x6224, [ 9475] = 0x63C5, [ 9476] = 0x63F1, [ 9477] = 0x63EB,
  [ 9478] = 0x6410, [ 9479] = 0x6412, [ 9480] = 0x6409, [ 9481] = 0x6420,
  [ 9482] = 0x6424, [ 9483] = 0x6433, [ 9484] = 0x6443, [ 9485] = 0x641F,
  [ 9486] = 0x6415, [ 9487] = 0x6418, [ 9488] = 0x6439, [ 9489] = 0x6437,
  [ 9490] = 0x6422, [ 9491] = 0x6423, [ 9492] = 0x640C, [ 9493] = 0x6426,
  [ 9494] = 0x6430, [ 9495] = 0x6428, [ 9496] = 0x6441, [ 9497] = 0x6435,
  [ 9498] = 0x642F, [ 9499] = 0x640A, [ 9500] = 0x641A, [ 9501] = 0x6440,
  [ 9502] = 0x6425, [ 9503] = 0x6427, [ 9504] = 0x640B, [ 9505] = 0x63E7,
  [ 9506] = 0x641B, [ 9507] = 0x642E, [ 9508] = 0x6421, [ 9509] = 0x640E,
  [ 9510] = 0x656F, [ 9511] = 0x6592, [ 9512] = 0x65D3, [ 9513] = 0x6686,
  [ 9514] = 0x668C, [ 9515] = 0x6695, [ 9516] = 0x6690, [ 9517] = 0x668B,
  [ 9518] = 0x668A, [ 9519] = 0x6699, [ 9520] = 0x6694, [ 9521] = 0x6678,
  [ 9522] = 0x6720, [ 9523] = 0x6966, [ 9524] = 0x695F, [ 9525] = 0x6938,
  [ 9526] = 0x694E, [ 9527] = 0x6962, [ 9528] = 0x6971, [ 9529] = 0x693F,
  [ 9530] = 0x6945, [ 9531] = 0x696A, [ 9532] = 0x6939, [ 9533] = 0x6942,
  [ 9534] = 0x6957, [ 9535] = 0x6959, [ 9536] = 0x697A, [ 9537] = 0x6948,
  [ 9538] = 0x6949, [ 9539] = 0x6935, [ 9540] = 0x696C, [ 9541] = 0x6933,
  [ 9542] = 0x693D, [ 9543] = 0x6965, [ 9544] = 0x68F0, [ 9545] = 0x6978,
  [ 9546] = 0x6934, [ 9547] = 0x6969, [ 9548] = 0x6940, [ 9549] = 0x696F,
  [ 9550] = 0x6944, [ 9551] = 0x6976, [ 9552] = 0x6958, [ 9553] = 0x6941,
  [ 9554] = 0x6974, [ 9555] = 0x694C, [ 9556] = 0x693B, [ 9557] = 0x694B,
  [ 9558] = 0x6937, [ 9559] = 0x695C, [ 9560] = 0x694F, [ 9561] = 0x6951,
  [ 9562] = 0x6932, [ 9563] = 0x6952, [ 9564] = 0x692F, [ 9565] = 0x697B,
  [ 9566] = 0x693C, [ 9567] = 0x6B46, [ 9568] = 0x6B45, [ 9569] = 0x6B43,
  [ 9570] = 0x6B42, [ 9571] = 0x6B48, [ 9572] = 0x6B41, [ 9573] = 0x6B9B,
  [ 9574] = 0xFA0D, [ 9575] = 0x6BFB, [ 9576] = 0x6BFC, [ 9577] = 0x6BF9,
  [ 9578] = 0x6BF7, [ 9579] = 0x6BF8, [ 9580] = 0x6E9B, [ 9581] = 0x6ED6,
  [ 9582] = 0x6EC8, [ 9583] = 0x6E8F, [ 9584] = 0x6EC0, [ 9585] = 0x6E9F,
  [ 9586] = 0x6E93, [ 9587] = 0x6E94, [ 9588] = 0x6EA0, [ 9589] = 0x6EB1,
  [ 9590] = 0x6EB9, [ 9591] = 0x6EC6, [ 9592] = 0x6ED2, [ 9593] = 0x6EBD,
  [ 9594] = 0x6EC1, [ 9595] = 0x6E9E, [ 9596] = 0x6EC9, [ 9597] = 0x6EB7,
  [ 9598] = 0x6EB0, [ 9599] = 0x6ECD, [ 9600] = 0x6EA6, [ 9601] = 0x6ECF,
  [ 9602] = 0x6EB2, [ 9603] = 0x6EBE, [ 9604] = 0x6EC3, [ 9605] = 0x6EDC,
  [ 9606] = 0x6ED8, [ 9607] = 0x6E99, [ 9608] = 0x6E92, [ 9609] = 0x6E8E,
  [ 9610] = 0x6E8D, [ 9611] = 0x6EA4, [ 9612] = 0x6EA1, [ 9613] = 0x6EBF,
  [ 9614] = 0x6EB3, [ 9615] = 0x6ED0, [ 9616] = 0x6ECA, [ 9617] = 0x6E97,
  [ 9618] = 0x6EAE, [ 9619] = 0x6EA3, [ 9620] = 0x7147, [ 9621] = 0x7154,
  [ 9622] = 0x7152, [ 9623] = 0x7163, [ 9624] = 0x7160, [ 9625] = 0x7141,
  [ 9626] = 0x715D, [ 9627] = 0x7162, [ 9628] = 0x7172, [ 9629] = 0x7178,
  [ 9630] = 0x716A, [ 9631] = 0x7161, [ 9632] = 0x7142, [ 9633] = 0x7158,
  [ 9634] = 0x7143, [ 9635] = 0x714B, [ 9636] = 0x7170, [ 9637] = 0x715F,
  [ 9638] = 0x7150, [ 9639] = 0x7153, [ 9640] = 0x7144, [ 9641] = 0x714D,
  [ 9642] = 0x715A, [ 9643] = 0x724F, [ 9644] = 0x728D, [ 9645] = 0x728C,
  [ 9646] = 0x7291, [ 9647] = 0x7290, [ 9648] = 0x728E, [ 9649] = 0x733C,
  [ 9650] = 0x7342, [ 9651] = 0x733B, [ 9652] = 0x733A, [ 9653] = 0x7340,
  [ 9654] = 0x734A, [ 9655] = 0x7349, [ 9656] = 0x7444, [ 9657] = 0x744A,
  [ 9658] = 0x744B, [ 9659] = 0x7452, [ 9660] = 0x7451, [ 9661] = 0x7457,
  [ 9662] = 0x7440, [ 9663] = 0x744F, [ 9664] = 0x7450, [ 9665] = 0x744E,
  [ 9666] = 0x7442, [ 9667] = 0x7446, [ 9668] = 0x744D, [ 9669] = 0x7454,
  [ 9670] = 0x74E1, [ 9671] = 0x74FF, [ 9672] = 0x74FE, [ 9673] = 0x74FD,
  [ 9674] = 0x751D, [ 9675] = 0x7579, [ 9676] = 0x7577, [ 9677] = 0x6983,
  [ 9678] = 0x75EF, [ 9679] = 0x760F, [ 9680] = 0x7603, [ 9681] = 0x75F7,
  [ 9682] = 0x75FE, [ 9683] = 0x75FC, [ 9684] = 0x75F9, [ 9685] = 0x75F8,
  [ 9686] = 0x7610, [ 9687] = 0x75FB, [ 9688] = 0x75F6, [ 9689] = 0x75ED,
  [ 9690] = 0x75F5, [ 9691] = 0x75FD, [ 9692] = 0x7699, [ 9693] = 0x76B5,
  [ 9694] = 0x76DD, [ 9695] = 0x7755, [ 9696] = 0x775F, [ 9697] = 0x7760,
  [ 9698] = 0x7752, [ 9699] = 0x7756, [ 9700] = 0x775A, [ 9701] = 0x7769,
  [ 9702] = 0x7767, [ 9703] = 0x7754, [ 9704] = 0x7759, [ 9705] = 0x776D,
  [ 9706] = 0x77E0, [ 9707] = 0x7887, [ 9708] = 0x789A, [ 9709] = 0x7894,
  [ 9710] = 0x788F, [ 9711] = 0x7884, [ 9712] = 0x7895, [ 9713] = 0x7885,
  [ 9714] = 0x7886, [ 9715] = 0x78A1, [ 9716] = 0x7883, [ 9717] = 0x7879,
  [ 9718] = 0x7899, [ 9719] = 0x7880, [ 9720] = 0x7896, [ 9721] = 0x787B,
  [ 9722] = 0x797C, [ 9723] = 0x7982, [ 9724] = 0x797D, [ 9725] = 0x7979,
  [ 9726] = 0x7A11, [ 9727] = 0x7A18, [ 9728] = 0x7A19, [ 9729] = 0x7A12,
  [ 9730] = 0x7A17, [ 9731] = 0x7A15, [ 9732] = 0x7A22, [ 9733] = 0x7A13,
  [ 9734] = 0x7A1B, [ 9735] = 0x7A10, [ 9736] = 0x7AA3, [ 9737] = 0x7AA2,
  [ 9738] = 0x7A9E, [ 9739] = 0x7AEB, [ 9740] = 0x7B66, [ 9741] = 0x7B64,
  [ 9742] = 0x7B6D, [ 9743] = 0x7B74, [ 9744] = 0x7B69, [ 9745] = 0x7B72,
  [ 9746] = 0x7B65, [ 9747] = 0x7B73, [ 9748] = 0x7B71, [ 9749] = 0x7B70,
  [ 9750] = 0x7B61, [ 9751] = 0x7B78, [ 9752] = 0x7B76, [ 9753] = 0x7B63,
  [ 9754] = 0x7CB2, [ 9755] = 0x7CB4, [ 9756] = 0x7CAF, [ 9757] = 0x7D88,
  [ 9758] = 0x7D86, [ 9759] = 0x7D80, [ 9760] = 0x7D8D, [ 9761] = 0x7D7F,
  [ 9762] = 0x7D85, [ 9763] = 0x7D7A, [ 9764] = 0x7D8E, [ 9765] = 0x7D7B,
  [ 9766] = 0x7D83, [ 9767] = 0x7D7C, [ 9768] = 0x7D8C, [ 9769] = 0x7D94,
  [ 9770] = 0x7D84, [ 9771] = 0x7D7D, [ 9772] = 0x7D92, [ 9773] = 0x7F6D,
  [ 9774] = 0x7F6B, [ 9775] = 0x7F67, [ 9776] = 0x7F68, [ 9777] = 0x7F6C,
  [ 9778] = 0x7FA6, [ 9779] = 0x7FA5, [ 9780] = 0x7FA7, [ 9781] = 0x7FDB,
  [ 9782] = 0x7FDC, [ 9783] = 0x8021, [ 9784] = 0x8164, [ 9785] = 0x8160,
  [ 9786] = 0x8177, [ 9787] = 0x815C, [ 9788] = 0x8169, [ 9789] = 0x815B,
  [ 9790] = 0x8162, [ 9791] = 0x8172, [ 9792] = 0x6721, [ 9793] = 0x815E,
  [ 9794] = 0x8176, [ 9795] = 0x8167, [ 9796] = 0x816F, [ 9797] = 0x8144,
  [ 9798] = 0x8161, [ 9799] = 0x821D, [ 9800] = 0x8249, [ 9801] = 0x8244,
  [ 9802] = 0x8240, [ 9803] = 0x8242, [ 9804] = 0x8245, [ 9805] = 0x84F1,
  [ 9806] = 0x843F, [ 9807] = 0x8456, [ 9808] = 0x8476, [ 9809] = 0x8479,
  [ 9810] = 0x848F, [ 9811] = 0x848D, [ 9812] = 0x8465, [ 9813] = 0x8451,
  [ 9814] = 0x8440, [ 9815] = 0x8486, [ 9816] = 0x8467, [ 9817] = 0x8430,
  [ 9818] = 0x844D, [ 9819] = 0x847D, [ 9820] = 0x845A, [ 9821] = 0x8459,
  [ 9822] = 0x8474, [ 9823] = 0x8473, [ 9824] = 0x845D, [ 9825] = 0x8507,
  [ 9826] = 0x845E, [ 9827] = 0x8437, [ 9828] = 0x843A, [ 9829] = 0x8434,
  [ 9830] = 0x847A, [ 9831] = 0x8443, [ 9832] = 0x8478, [ 9833] = 0x8432,
  [ 9834] = 0x8445, [ 9835] = 0x8429, [ 9836] = 0x83D9, [ 9837] = 0x844B,
  [ 9838] = 0x842F, [ 9839] = 0x8442, [ 9840] = 0x842D, [ 9841] = 0x845F,
  [ 9842] = 0x8470, [ 9843] = 0x8439, [ 9844] = 0x844E, [ 9845] = 0x844C,
  [ 9846] = 0x8452, [ 9847] = 0x846F, [ 9848] = 0x84C5, [ 9849] = 0x848E,
  [ 9850] = 0x843B, [ 9851] = 0x8447, [ 9852] = 0x8436, [ 9853] = 0x8433,
  [ 9854] = 0x8468, [ 9855] = 0x847E, [ 9856] = 0x8444, [ 9857] = 0x842B,
  [ 9858] = 0x8460, [ 9859] = 0x8454, [ 9860] = 0x846E, [ 9861] = 0x8450,
  [ 9862] = 0x870B, [ 9863] = 0x8704, [ 9864] = 0x86F7, [ 9865] = 0x870C,
  [ 9866] = 0x86FA, [ 9867] = 0x86D6, [ 9868] = 0x86F5, [ 9869] = 0x874D,
  [ 9870] = 0x86F8, [ 9871] = 0x870E, [ 9872] = 0x8709, [ 9873] = 0x8701,
  [ 9874] = 0x86F6, [ 9875] = 0x870D, [ 9876] = 0x8705, [ 9877] = 0x88D6,
  [ 9878] = 0x88CB, [ 9879] = 0x88CD, [ 9880] = 0x88CE, [ 9881] = 0x88DE,
  [ 9882] = 0x88DB, [ 9883] = 0x88DA, [ 9884] = 0x88CC, [ 9885] = 0x88D0,
  [ 9886] = 0x8985, [ 9887] = 0x899B, [ 9888] = 0x89DF, [ 9889] = 0x89E5,
  [ 9890] = 0x89E4, [ 9891] = 0x89E1, [ 9892] = 0x89E0, [ 9893] = 0x89E2,
  [ 9894] = 0x89DC, [ 9895] = 0x89E6, [ 9896] = 0x8A76, [ 9897] = 0x8A86,
  [ 9898] = 0x8A7F, [ 9899] = 0x8A61, [ 9900] = 0x8A3F, [ 9901] = 0x8A77,
  [ 9902] = 0x8A82, [ 9903] = 0x8A84, [ 9904] = 0x8A75, [ 9905] = 0x8A83,
  [ 9906] = 0x8A81, [ 9907] = 0x8A74, [ 9908] = 0x8A7A, [ 9909] = 0x8C3C,
  [ 9910] = 0x8C4B, [ 9911] = 0x8C4A, [ 9912] = 0x8C65, [ 9913] = 0x8C64,
  [ 9914] = 0x8C66, [ 9915] = 0x8C86, [ 9916] = 0x8C84, [ 9917] = 0x8C85,
  [ 9918] = 0x8CCC, [ 9919] = 0x8D68, [ 9920] = 0x8D69, [ 9921] = 0x8D91,
  [ 9922] = 0x8D8C, [ 9923] = 0x8D8E, [ 9924] = 0x8D8F, [ 9925] = 0x8D8D,
  [ 9926] = 0x8D93, [ 9927] = 0x8D94, [ 9928] = 0x8D90, [ 9929] = 0x8D92,
  [ 9930] = 0x8DF0, [ 9931] = 0x8DE0, [ 9932] = 0x8DEC, [ 9933] = 0x8DF1,
  [ 9934] = 0x8DEE, [ 9935] = 0x8DD0, [ 9936] = 0x8DE9, [ 9937] = 0x8DE3,
  [ 9938] = 0x8DE2, [ 9939] = 0x8DE7, [ 9940] = 0x8DF2, [ 9941] = 0x8DEB,
  [ 9942] = 0x8DF4, [ 9943] = 0x8F06, [ 9944] = 0x8EFF, [ 9945] = 0x8F01,
  [ 9946] = 0x8F00, [ 9947] = 0x8F05, [ 9948] = 0x8F07, [ 9949] = 0x8F08,
  [ 9950] = 0x8F02, [ 9951] = 0x8F0B, [ 9952] = 0x9052, [ 9953] = 0x903F,
  [ 9954] = 0x9044, [ 9955] = 0x9049, [ 9956] = 0x903D, [ 9957] = 0x9110,
  [ 9958] = 0x910D, [ 9959] = 0x910F, [ 9960] = 0x9111, [ 9961] = 0x9116,
  [ 9962] = 0x9114, [ 9963] = 0x910B, [ 9964] = 0x910E, [ 9965] = 0x916E,
  [ 9966] = 0x916F, [ 9967] = 0x9248, [ 9968] = 0x9252, [ 9969] = 0x9230,
  [ 9970] = 0x923A, [ 9971] = 0x9266, [ 9972] = 0x9233, [ 9973] = 0x9265,
  [ 9974] = 0x925E, [ 9975] = 0x9283, [ 9976] = 0x922E, [ 9977] = 0x924A,
  [ 9978] = 0x9246, [ 9979] = 0x926D, [ 9980] = 0x926C, [ 9981] = 0x924F,
  [ 9982] = 0x9260, [ 9983] = 0x9267, [ 9984] = 0x926F, [ 9985] = 0x9236,
  [ 9986] = 0x9261, [ 9987] = 0x9270, [ 9988] = 0x9231, [ 9989] = 0x9254,
  [ 9990] = 0x9263, [ 9991] = 0x9250, [ 9992] = 0x9272, [ 9993] = 0x924E,
  [ 9994] = 0x9253, [ 9995] = 0x924C, [ 9996] = 0x9256, [ 9997] = 0x9232,
  [ 9998] = 0x959F, [ 9999] = 0x959C, [10000] = 0x959E, [10001] = 0x959B,
  [10002] = 0x9692, [10003] = 0x9693, [10004] = 0x9691, [10005] = 0x9697,
  [10006] = 0x96CE, [10007] = 0x96FA, [10008] = 0x96FD, [10009] = 0x96F8,
  [10010] = 0x96F5, [10011] = 0x9773, [10012] = 0x9777, [10013] = 0x9778,
  [10014] = 0x9772, [10015] = 0x980F, [10016] = 0x980D, [10017] = 0x980E,
  [10018] = 0x98AC, [10019] = 0x98F6, [10020] = 0x98F9, [10021] = 0x99AF,
  [10022] = 0x99B2, [10023] = 0x99B0, [10024] = 0x99B5, [10025] = 0x9AAD,
  [10026] = 0x9AAB, [10027] = 0x9B5B, [10028] = 0x9CEA, [10029] = 0x9CED,
  [10030] = 0x9CE7, [10031] = 0x9E80, [10032] = 0x9EFD, [10033] = 0x50E6,
  [10034] = 0x50D4, [10035] = 0x50D7, [10036] = 0x50E8, [10037] = 0x50F3,
  [10038] = 0x50DB, [10039] = 0x50EA, [10040] = 0x50DD, [10041] = 0x50E4,
  [10042] = 0x50D3, [10043] = 0x50EC, [10044] = 0x50F0, [10045] = 0x50EF,
  [10046] = 0x50E3, [10047] = 0x50E0, [10048] = 0x51D8, [10049] = 0x5280,
  [10050] = 0x5281, [10051] = 0x52E9, [10052] = 0x52EB, [10053] = 0x5330,
  [10054] = 0x53AC, [10055] = 0x5627, [10056] = 0x5615, [10057] = 0x560C,
  [10058] = 0x5612, [10059] = 0x55FC, [10060] = 0x560F, [10061] = 0x561C,
  [10062] = 0x5601, [10063] = 0x5613, [10064] = 0x5602, [10065] = 0x55FA,
  [10066] = 0x561D, [10067] = 0x5604, [10068] = 0x55FF, [10069] = 0x55F9,
  [10070] = 0x5889, [10071] = 0x587C, [10072] = 0x5890, [10073] = 0x5898,
  [10074] = 0x5886, [10075] = 0x5881, [10076] = 0x587F, [10077] = 0x5874,
  [10078] = 0x588B, [10079] = 0x587A, [10080] = 0x5887, [10081] = 0x5891,
  [10082] = 0x588E, [10083] = 0x5876, [10084] = 0x5882, [10085] = 0x5888,
  [10086] = 0x587B, [10087] = 0x5894, [10088] = 0x588F, [10089] = 0x58FE,
  [10090] = 0x596B, [10091] = 0x5ADC, [10092] = 0x5AEE, [10093] = 0x5AE5,
  [10094] = 0x5AD5, [10095] = 0x5AEA, [10096] = 0x5ADA, [10097] = 0x5AED,
  [10098] = 0x5AEB, [10099] = 0x5AF3, [10100] = 0x5AE2, [10101] = 0x5AE0,
  [10102] = 0x5ADB, [10103] = 0x5AEC, [10104] = 0x5ADE, [10105] = 0x5ADD,
  [10106] = 0x5AD9, [10107] = 0x5AE8, [10108] = 0x5ADF, [10109] = 0x5B77,
  [10110] = 0x5BE0, [10111] = 0x5BE3, [10112] = 0x5C63, [10113] = 0x5D82,
  [10114] = 0x5D80, [10115] = 0x5D7D, [10116] = 0x5D86, [10117] = 0x5D7A,
  [10118] = 0x5D81, [10119] = 0x5D77, [10120] = 0x5D8A, [10121] = 0x5D89,
  [10122] = 0x5D88, [10123] = 0x5D7E, [10124] = 0x5D7C, [10125] = 0x5D8D,
  [10126] = 0x5D79, [10127] = 0x5D7F, [10128] = 0x5E58, [10129] = 0x5E59,
  [10130] = 0x5E53, [10131] = 0x5ED8, [10132] = 0x5ED1, [10133] = 0x5ED7,
  [10134] = 0x5ECE, [10135] = 0x5EDC, [10136] = 0x5ED5, [10137] = 0x5ED9,
  [10138] = 0x5ED2, [10139] = 0x5ED4, [10140] = 0x5F44, [10141] = 0x5F43,
  [10142] = 0x5F6F, [10143] = 0x5FB6, [10144] = 0x612C, [10145] = 0x6128,
  [10146] = 0x6141, [10147] = 0x615E, [10148] = 0x6171, [10149] = 0x6173,
  [10150] = 0x6152, [10151] = 0x6153, [10152] = 0x6172, [10153] = 0x616C,
  [10154] = 0x6180, [10155] = 0x6174, [10156] = 0x6154, [10157] = 0x617A,
  [10158] = 0x615B, [10159] = 0x6165, [10160] = 0x613B, [10161] = 0x616A,
  [10162] = 0x6161, [10163] = 0x6156, [10164] = 0x6229, [10165] = 0x6227,
  [10166] = 0x622B, [10167] = 0x642B, [10168] = 0x644D, [10169] = 0x645B,
  [10170] = 0x645D, [10171] = 0x6474, [10172] = 0x6476, [10173] = 0x6472,
  [10174] = 0x6473, [10175] = 0x647D, [10176] = 0x6475, [10177] = 0x6466,
  [10178] = 0x64A6, [10179] = 0x644E, [10180] = 0x6482, [10181] = 0x645E,
  [10182] = 0x645C, [10183] = 0x644B, [10184] = 0x6453, [10185] = 0x6460,
  [10186] = 0x6450, [10187] = 0x647F, [10188] = 0x643F, [10189] = 0x646C,
  [10190] = 0x646B, [10191] = 0x6459, [10192] = 0x6465, [10193] = 0x6477,
  [10194] = 0x6573, [10195] = 0x65A0, [10196] = 0x66A1, [10197] = 0x66A0,
  [10198] = 0x669F, [10199] = 0x6705, [10200] = 0x6704, [10201] = 0x6722,
  [10202] = 0x69B1, [10203] = 0x69B6, [10204] = 0x69C9, [10205] = 0x69A0,
  [10206] = 0x69CE, [10207] = 0x6996, [10208] = 0x69B0, [10209] = 0x69AC,
  [10210] = 0x69BC, [10211] = 0x6991, [10212] = 0x6999, [10213] = 0x698E,
  [10214] = 0x69A7, [10215] = 0x698D, [10216] = 0x69A9, [10217] = 0x69BE,
  [10218] = 0x69AF, [10219] = 0x69BF, [10220] = 0x69C4, [10221] = 0x69BD,
  [10222] = 0x69A4, [10223] = 0x69D4, [10224] = 0x69B9, [10225] = 0x69CA,
  [10226] = 0x699A, [10227] = 0x69CF, [10228] = 0x69B3, [10229] = 0x6993,
  [10230] = 0x69AA, [10231] = 0x69A1, [10232] = 0x699E, [10233] = 0x69D9,
  [10234] = 0x6997, [10235] = 0x6990, [10236] = 0x69C2, [10237] = 0x69B5,
  [10238] = 0x69A5, [10239] = 0x69C6, [10240] = 0x6B4A, [10241] = 0x6B4D,
  [10242] = 0x6B4B, [10243] = 0x6B9E, [10244] = 0x6B9F, [10245] = 0x6BA0,
  [10246] = 0x6BC3, [10247] = 0x6BC4, [10248] = 0x6BFE, [10249] = 0x6ECE,
  [10250] = 0x6EF5, [10251] = 0x6EF1, [10252] = 0x6F03, [10253] = 0x6F25,
  [10254] = 0x6EF8, [10255] = 0x6F37, [10256] = 0x6EFB, [10257] = 0x6F2E,
  [10258] = 0x6F09, [10259] = 0x6F4E, [10260] = 0x6F19, [10261] = 0x6F1A,
  [10262] = 0x6F27, [10263] = 0x6F18, [10264] = 0x6F3B, [10265] = 0x6F12,
  [10266] = 0x6EED, [10267] = 0x6F0A, [10268] = 0x6F36, [10269] = 0x6F73,
  [10270] = 0x6EF9, [10271] = 0x6EEE, [10272] = 0x6F2D, [10273] = 0x6F40,
  [10274] = 0x6F30, [10275] = 0x6F3C, [10276] = 0x6F35, [10277] = 0x6EEB,
  [10278] = 0x6F07, [10279] = 0x6F0E, [10280] = 0x6F43, [10281] = 0x6F05,
  [10282] = 0x6EFD, [10283] = 0x6EF6, [10284] = 0x6F39, [10285] = 0x6F1C,
  [10286] = 0x6EFC, [10287] = 0x6F3A, [10288] = 0x6F1F, [10289] = 0x6F0D,
  [10290] = 0x6F1E, [10291] = 0x6F08, [10292] = 0x6F21, [10293] = 0x7187,
  [10294] = 0x7190, [10295] = 0x7189, [10296] = 0x7180, [10297] = 0x7185,
  [10298] = 0x7182, [10299] = 0x718F, [10300] = 0x717B, [10301] = 0x7186,
  [10302] = 0x7181, [10303] = 0x7197, [10304] = 0x7244, [10305] = 0x7253,
  [10306] = 0x7297, [10307] = 0x7295, [10308] = 0x7293, [10309] = 0x7343,
  [10310] = 0x734D, [10311] = 0x7351, [10312] = 0x734C, [10313] = 0x7462,
  [10314] = 0x7473, [10315] = 0x7471, [10316] = 0x7475, [10317] = 0x7472,
  [10318] = 0x7467, [10319] = 0x746E, [10320] = 0x7500, [10321] = 0x7502,
  [10322] = 0x7503, [10323] = 0x757D, [10324] = 0x7590, [10325] = 0x7616,
  [10326] = 0x7608, [10327] = 0x760C, [10328] = 0x7615, [10329] = 0x7611,
  [10330] = 0x760A, [10331] = 0x7614, [10332] = 0x76B8, [10333] = 0x7781,
  [10334] = 0x777C, [10335] = 0x7785, [10336] = 0x7782, [10337] = 0x776E,
  [10338] = 0x7780, [10339] = 0x776F, [10340] = 0x777E, [10341] = 0x7783,
  [10342] = 0x78B2, [10343] = 0x78AA, [10344] = 0x78B4, [10345] = 0x78AD,
  [10346] = 0x78A8, [10347] = 0x787E, [10348] = 0x78AB, [10349] = 0x789E,
  [10350] = 0x78A5, [10351] = 0x78A0, [10352] = 0x78AC, [10353] = 0x78A2,
  [10354] = 0x78A4, [10355] = 0x7998, [10356] = 0x798A, [10357] = 0x798B,
  [10358] = 0x7996, [10359] = 0x7995, [10360] = 0x7994, [10361] = 0x7993,
  [10362] = 0x7997, [10363] = 0x7988, [10364] = 0x7992, [10365] = 0x7990,
  [10366] = 0x7A2B, [10367] = 0x7A4A, [10368] = 0x7A30, [10369] = 0x7A2F,
  [10370] = 0x7A28, [10371] = 0x7A26, [10372] = 0x7AA8, [10373] = 0x7AAB,
  [10374] = 0x7AAC, [10375] = 0x7AEE, [10376] = 0x7B88, [10377] = 0x7B9C,
  [10378] = 0x7B8A, [10379] = 0x7B91, [10380] = 0x7B90, [10381] = 0x7B96,
  [10382] = 0x7B8D, [10383] = 0x7B8C, [10384] = 0x7B9B, [10385] = 0x7B8E,
  [10386] = 0x7B85, [10387] = 0x7B98, [10388] = 0x5284, [10389] = 0x7B99,
  [10390] = 0x7BA4, [10391] = 0x7B82, [10392] = 0x7CBB, [10393] = 0x7CBF,
  [10394] = 0x7CBC, [10395] = 0x7CBA, [10396] = 0x7DA7, [10397] = 0x7DB7,
  [10398] = 0x7DC2, [10399] = 0x7DA3, [10400] = 0x7DAA, [10401] = 0x7DC1,
  [10402] = 0x7DC0, [10403] = 0x7DC5, [10404] = 0x7D9D, [10405] = 0x7DCE,
  [10406] = 0x7DC4, [10407] = 0x7DC6, [10408] = 0x7DCB, [10409] = 0x7DCC,
  [10410] = 0x7DAF, [10411] = 0x7DB9, [10412] = 0x7D96, [10413] = 0x7DBC,
  [10414] = 0x7D9F, [10415] = 0x7DA6, [10416] = 0x7DAE, [10417] = 0x7DA9,
  [10418] = 0x7DA1, [10419] = 0x7DC9, [10420] = 0x7F73, [10421] = 0x7FE2,
  [10422] = 0x7FE3, [10423] = 0x7FE5, [10424] = 0x7FDE, [10425] = 0x8024,
  [10426] = 0x805D, [10427] = 0x805C, [10428] = 0x8189, [10429] = 0x8186,
  [10430] = 0x8183, [10431] = 0x8187, [10432] = 0x818D, [10433] = 0x818C,
  [10434] = 0x818B, [10435] = 0x8215, [10436] = 0x8497, [10437] = 0x84A4,
  [10438] = 0x84A1, [10439] = 0x849F, [10440] = 0x84BA, [10441] = 0x84CE,
  [10442] = 0x84C2, [10443] = 0x84AC, [10444] = 0x84AE, [10445] = 0x84AB,
  [10446] = 0x84B9, [10447] = 0x84B4, [10448] = 0x84C1, [10449] = 0x84CD,
  [10450] = 0x84AA, [10451] = 0x849A, [10452] = 0x84B1, [10453] = 0x84D0,
  [10454] = 0x849D, [10455] = 0x84A7, [10456] = 0x84BB, [10457] = 0x84A2,
  [10458] = 0x8494, [10459] = 0x84C7, [10460] = 0x84CC, [10461] = 0x849B,
  [10462] = 0x84A9, [10463] = 0x84AF, [10464] = 0x84A8, [10465] = 0x84D6,
  [10466] = 0x8498, [10467] = 0x84B6, [10468] = 0x84CF, [10469] = 0x84A0,
  [10470] = 0x84D7, [10471] = 0x84D4, [10472] = 0x84D2, [10473] = 0x84DB,
  [10474] = 0x84B0, [10475] = 0x8491, [10476] = 0x8661, [10477] = 0x8733,
  [10478] = 0x8723, [10479] = 0x8728, [10480] = 0x876B, [10481] = 0x8740,
  [10482] = 0x872E, [10483] = 0x871E, [10484] = 0x8721, [10485] = 0x8719,
  [10486] = 0x871B, [10487] = 0x8743, [10488] = 0x872C, [10489] = 0x8741,
  [10490] = 0x873E, [10491] = 0x8746, [10492] = 0x8720, [10493] = 0x8732,
  [10494] = 0x872A, [10495] = 0x872D, [10496] = 0x873C, [10497] = 0x8712,
  [10498] = 0x873A, [10499] = 0x8731, [10500] = 0x8735, [10501] = 0x8742,
  [10502] = 0x8726, [10503] = 0x8727, [10504] = 0x8738, [10505] = 0x8724,
  [10506] = 0x871A, [10507] = 0x8730, [10508] = 0x8711, [10509] = 0x88F7,
  [10510] = 0x88E7, [10511] = 0x88F1, [10512] = 0x88F2, [10513] = 0x88FA,
  [10514] = 0x88FE, [10515] = 0x88EE, [10516] = 0x88FC, [10517] = 0x88F6,
  [10518] = 0x88FB, [10519] = 0x88F0, [10520] = 0x88EC, [10521] = 0x88EB,
  [10522] = 0x899D, [10523] = 0x89A1, [10524] = 0x899F, [10525] = 0x899E,
  [10526] = 0x89E9, [10527] = 0x89EB, [10528] = 0x89E8, [10529] = 0x8AAB,
  [10530] = 0x8A99, [10531] = 0x8A8B, [10532] = 0x8A92, [10533] = 0x8A8F,
  [10534] = 0x8A96, [10535] = 0x8C3D, [10536] = 0x8C68, [10537] = 0x8C69,
  [10538] = 0x8CD5, [10539] = 0x8CCF, [10540] = 0x8CD7, [10541] = 0x8D96,
  [10542] = 0x8E09, [10543] = 0x8E02, [10544] = 0x8DFF, [10545] = 0x8E0D,
  [10546] = 0x8DFD, [10547] = 0x8E0A, [10548] = 0x8E03, [10549] = 0x8E07,
  [10550] = 0x8E06, [10551] = 0x8E05, [10552] = 0x8DFE, [10553] = 0x8E00,
  [10554] = 0x8E04, [10555] = 0x8F10, [10556] = 0x8F11, [10557] = 0x8F0E,
  [10558] = 0x8F0D, [10559] = 0x9123, [10560] = 0x911C, [10561] = 0x9120,
  [10562] = 0x9122, [10563] = 0x911F, [10564] = 0x911D, [10565] = 0x911A,
  [10566] = 0x9124, [10567] = 0x9121, [10568] = 0x911B, [10569] = 0x917A,
  [10570] = 0x9172, [10571] = 0x9179, [10572] = 0x9173, [10573] = 0x92A5,
  [10574] = 0x92A4, [10575] = 0x9276, [10576] = 0x929B, [10577] = 0x927A,
  [10578] = 0x92A0, [10579] = 0x9294, [10580] = 0x92AA, [10581] = 0x928D,
  [10582] = 0x92A6, [10583] = 0x929A, [10584] = 0x92AB, [10585] = 0x9279,
  [10586] = 0x9297, [10587] = 0x927F, [10588] = 0x92A3, [10589] = 0x92EE,
  [10590] = 0x928E, [10591] = 0x9282, [10592] = 0x9295, [10593] = 0x92A2,
  [10594] = 0x927D, [10595] = 0x9288, [10596] = 0x92A1, [10597] = 0x928A,
  [10598] = 0x9286, [10599] = 0x928C, [10600] = 0x9299, [10601] = 0x92A7,
  [10602] = 0x927E, [10603] = 0x9287, [10604] = 0x92A9, [10605] = 0x929D,
  [10606] = 0x928B, [10607] = 0x922D, [10608] = 0x969E, [10609] = 0x96A1,
  [10610] = 0x96FF, [10611] = 0x9758, [10612] = 0x977D, [10613] = 0x977A,
  [10614] = 0x977E, [10615] = 0x9783, [10616] = 0x9780, [10617] = 0x9782,
  [10618] = 0x977B, [10619] = 0x9784, [10620] = 0x9781, [10621] = 0x977F,
  [10622] = 0x97CE, [10623] = 0x97CD, [10624] = 0x9816, [10625] = 0x98AD,
  [10626] = 0x98AE, [10627] = 0x9902, [10628] = 0x9900, [10629] = 0x9907,
  [10630] = 0x999D, [10631] = 0x999C, [10632] = 0x99C3, [10633] = 0x99B9,
  [10634] = 0x99BB, [10635] = 0x99BA, [10636] = 0x99C2, [10637] = 0x99BD,
  [10638] = 0x99C7, [10639] = 0x9AB1, [10640] = 0x9AE3, [10641] = 0x9AE7,
  [10642] = 0x9B3E, [10643] = 0x9B3F, [10644] = 0x9B60, [10645] = 0x9B61,
  [10646] = 0x9B5F, [10647] = 0x9CF1, [10648] = 0x9CF2, [10649] = 0x9CF5,
  [10650] = 0x9EA7, [10651] = 0x50FF, [10652] = 0x5103, [10653] = 0x5130,
  [10654] = 0x50F8, [10655] = 0x5106, [10656] = 0x5107, [10657] = 0x50F6,
  [10658] = 0x50FE, [10659] = 0x510B, [10660] = 0x510C, [10661] = 0x50FD,
  [10662] = 0x510A, [10663] = 0x528B, [10664] = 0x528C, [10665] = 0x52F1,
  [10666] = 0x52EF, [10667] = 0x5648, [10668] = 0x5642, [10669] = 0x564C,
  [10670] = 0x5635, [10671] = 0x5641, [10672] = 0x564A, [10673] = 0x5649,
  [10674] = 0x5646, [10675] = 0x5658, [10676] = 0x565A, [10677] = 0x5640,
  [10678] = 0x5633, [10679] = 0x563D, [10680] = 0x562C, [10681] = 0x563E,
  [10682] = 0x5638, [10683] = 0x562A, [10684] = 0x563A, [10685] = 0x571A,
  [10686] = 0x58AB, [10687] = 0x589D, [10688] = 0x58B1, [10689] = 0x58A0,
  [10690] = 0x58A3, [10691] = 0x58AF, [10692] = 0x58AC, [10693] = 0x58A5,
  [10694] = 0x58A1, [10695] = 0x58FF, [10696] = 0x5AFF, [10697] = 0x5AF4,
  [10698] = 0x5AFD, [10699] = 0x5AF7, [10700] = 0x5AF6, [10701] = 0x5B03,
  [10702] = 0x5AF8, [10703] = 0x5B02, [10704] = 0x5AF9, [10705] = 0x5B01,
  [10706] = 0x5B07, [10707] = 0x5B05, [10708] = 0x5B0F, [10709] = 0x5C67,
  [10710] = 0x5D99, [10711] = 0x5D97, [10712] = 0x5D9F, [10713] = 0x5D92,
  [10714] = 0x5DA2, [10715] = 0x5D93, [10716] = 0x5D95, [10717] = 0x5DA0,
  [10718] = 0x5D9C, [10719] = 0x5DA1, [10720] = 0x5D9A, [10721] = 0x5D9E,
  [10722] = 0x5E69, [10723] = 0x5E5D, [10724] = 0x5E60, [10725] = 0x5E5C,
  [10726] = 0x7DF3, [10727] = 0x5EDB, [10728] = 0x5EDE, [10729] = 0x5EE1,
  [10730] = 0x5F49, [10731] = 0x5FB2, [10732] = 0x618B, [10733] = 0x6183,
  [10734] = 0x6179, [10735] = 0x61B1, [10736] = 0x61B0, [10737] = 0x61A2,
  [10738] = 0x6189, [10739] = 0x619B, [10740] = 0x6193, [10741] = 0x61AF,
  [10742] = 0x61AD, [10743] = 0x619F, [10744] = 0x6192, [10745] = 0x61AA,
  [10746] = 0x61A1, [10747] = 0x618D, [10748] = 0x6166, [10749] = 0x61B3,
  [10750] = 0x622D, [10751] = 0x646E, [10752] = 0x6470, [10753] = 0x6496,
  [10754] = 0x64A0, [10755] = 0x6485, [10756] = 0x6497, [10757] = 0x649C,
  [10758] = 0x648F, [10759] = 0x648B, [10760] = 0x648A, [10761] = 0x648C,
  [10762] = 0x64A3, [10763] = 0x649F, [10764] = 0x6468, [10765] = 0x64B1,
  [10766] = 0x6498, [10767] = 0x6576, [10768] = 0x657A, [10769] = 0x6579,
  [10770] = 0x657B, [10771] = 0x65B2, [10772] = 0x65B3, [10773] = 0x66B5,
  [10774] = 0x66B0, [10775] = 0x66A9, [10776] = 0x66B2, [10777] = 0x66B7,
  [10778] = 0x66AA, [10779] = 0x66AF, [10780] = 0x6A00, [10781] = 0x6A06,
  [10782] = 0x6A17, [10783] = 0x69E5, [10784] = 0x69F8, [10785] = 0x6A15,
  [10786] = 0x69F1, [10787] = 0x69E4, [10788] = 0x6A20, [10789] = 0x69FF,
  [10790] = 0x69EC, [10791] = 0x69E2, [10792] = 0x6A1B, [10793] = 0x6A1D,
  [10794] = 0x69FE, [10795] = 0x6A27, [10796] = 0x69F2, [10797] = 0x69EE,
  [10798] = 0x6A14, [10799] = 0x69F7, [10800] = 0x69E7, [10801] = 0x6A40,
  [10802] = 0x6A08, [10803] = 0x69E6, [10804] = 0x69FB, [10805] = 0x6A0D,
  [10806] = 0x69FC, [10807] = 0x69EB, [10808] = 0x6A09, [10809] = 0x6A04,
  [10810] = 0x6A18, [10811] = 0x6A25, [10812] = 0x6A0F, [10813] = 0x69F6,
  [10814] = 0x6A26, [10815] = 0x6A07, [10816] = 0x69F4, [10817] = 0x6A16,
  [10818] = 0x6B51, [10819] = 0x6BA5, [10820] = 0x6BA3, [10821] = 0x6BA2,
  [10822] = 0x6BA6, [10823] = 0x6C01, [10824] = 0x6C00, [10825] = 0x6BFF,
  [10826] = 0x6C02, [10827] = 0x6F41, [10828] = 0x6F26, [10829] = 0x6F7E,
  [10830] = 0x6F87, [10831] = 0x6FC6, [10832] = 0x6F92, [10833] = 0x6F8D,
  [10834] = 0x6F89, [10835] = 0x6F8C, [10836] = 0x6F62, [10837] = 0x6F4F,
  [10838] = 0x6F85, [10839] = 0x6F5A, [10840] = 0x6F96, [10841] = 0x6F76,
  [10842] = 0x6F6C, [10843] = 0x6F82, [10844] = 0x6F55, [10845] = 0x6F72,
  [10846] = 0x6F52, [10847] = 0x6F50, [10848] = 0x6F57, [10849] = 0x6F94,
  [10850] = 0x6F93, [10851] = 0x6F5D, [10852] = 0x6F00, [10853] = 0x6F61,
  [10854] = 0x6F6B, [10855] = 0x6F7D, [10856] = 0x6F67, [10857] = 0x6F90,
  [10858] = 0x6F53, [10859] = 0x6F8B, [10860] = 0x6F69, [10861] = 0x6F7F,
  [10862] = 0x6F95, [10863] = 0x6F63, [10864] = 0x6F77, [10865] = 0x6F6A,
  [10866] = 0x6F7B, [10867] = 0x71B2, [10868] = 0x71AF, [10869] = 0x719B,
  [10870] = 0x71B0, [10871] = 0x71A0, [10872] = 0x719A, [10873] = 0x71A9,
  [10874] = 0x71B5, [10875] = 0x719D, [10876] = 0x71A5, [10877] = 0x719E,
  [10878] = 0x71A4, [10879] = 0x71A1, [10880] = 0x71AA, [10881] = 0x719C,
  [10882] = 0x71A7, [10883] = 0x71B3, [10884] = 0x7298, [10885] = 0x729A,
  [10886] = 0x7358, [10887] = 0x7352, [10888] = 0x735E, [10889] = 0x735F,
  [10890] = 0x7360, [10891] = 0x735D, [10892] = 0x735B, [10893] = 0x7361,
  [10894] = 0x735A, [10895] = 0x7359, [10896] = 0x7362, [10897] = 0x7487,
  [10898] = 0x7489, [10899] = 0x748A, [10900] = 0x7486, [10901] = 0x7481,
  [10902] = 0x747D, [10903] = 0x7485, [10904] = 0x7488, [10905] = 0x747C,
  [10906] = 0x7479, [10907] = 0x7508, [10908] = 0x7507, [10909] = 0x757E,
  [10910] = 0x7625, [10911] = 0x761E, [10912] = 0x7619, [10913] = 0x761D,
  [10914] = 0x761C, [10915] = 0x7623, [10916] = 0x761A, [10917] = 0x7628,
  [10918] = 0x761B, [10919] = 0x769C, [10920] = 0x769D, [10921] = 0x769E,
  [10922] = 0x769B, [10923] = 0x778D, [10924] = 0x778F, [10925] = 0x7789,
  [10926] = 0x7788, [10927] = 0x78CD, [10928] = 0x78BB, [10929] = 0x78CF,
  [10930] = 0x78CC, [10931] = 0x78D1, [10932] = 0x78CE, [10933] = 0x78D4,
  [10934] = 0x78C8, [10935] = 0x78C3, [10936] = 0x78C4, [10937] = 0x78C9,
  [10938] = 0x799A, [10939] = 0x79A1, [10940] = 0x79A0, [10941] = 0x799C,
  [10942] = 0x79A2, [10943] = 0x799B, [10944] = 0x6B76, [10945] = 0x7A39,
  [10946] = 0x7AB2, [10947] = 0x7AB4, [10948] = 0x7AB3, [10949] = 0x7BB7,
  [10950] = 0x7BCB, [10951] = 0x7BBE, [10952] = 0x7BAC, [10953] = 0x7BCE,
  [10954] = 0x7BAF, [10955] = 0x7BB9, [10956] = 0x7BCA, [10957] = 0x7BB5,
  [10958] = 0x7CC5, [10959] = 0x7CC8, [10960] = 0x7CCC, [10961] = 0x7CCB,
  [10962] = 0x7DF7, [10963] = 0x7DDB, [10964] = 0x7DEA, [10965] = 0x7DE7,
  [10966] = 0x7DD7, [10967] = 0x7DE1, [10968] = 0x7E03, [10969] = 0x7DFA,
  [10970] = 0x7DE6, [10971] = 0x7DF6, [10972] = 0x7DF1, [10973] = 0x7DF0,
  [10974] = 0x7DEE, [10975] = 0x7DDF, [10976] = 0x7F76, [10977] = 0x7FAC,
  [10978] = 0x7FB0, [10979] = 0x7FAD, [10980] = 0x7FED, [10981] = 0x7FEB,
  [10982] = 0x7FEA, [10983] = 0x7FEC, [10984] = 0x7FE6, [10985] = 0x7FE8,
  [10986] = 0x8064, [10987] = 0x8067, [10988] = 0x81A3, [10989] = 0x819F,
  [10990] = 0x819E, [10991] = 0x8195, [10992] = 0x81A2, [10993] = 0x8199,
  [10994] = 0x8197, [10995] = 0x8216, [10996] = 0x824F, [10997] = 0x8253,
  [10998] = 0x8252, [10999] = 0x8250, [11000] = 0x824E, [11001] = 0x8251,
  [11002] = 0x8524, [11003] = 0x853B, [11004] = 0x850F, [11005] = 0x8500,
  [11006] = 0x8529, [11007] = 0x850E, [11008] = 0x8509, [11009] = 0x850D,
  [11010] = 0x851F, [11011] = 0x850A, [11012] = 0x8527, [11013] = 0x851C,
  [11014] = 0x84FB, [11015] = 0x852B, [11016] = 0x84FA, [11017] = 0x8508,
  [11018] = 0x850C, [11019] = 0x84F4, [11020] = 0x852A, [11021] = 0x84F2,
  [11022] = 0x8515, [11023] = 0x84F7, [11024] = 0x84EB, [11025] = 0x84F3,
  [11026] = 0x84FC, [11027] = 0x8512, [11028] = 0x84EA, [11029] = 0x84E9,
  [11030] = 0x8516, [11031] = 0x84FE, [11032] = 0x8528, [11033] = 0x851D,
  [11034] = 0x852E, [11035] = 0x8502, [11036] = 0x84FD, [11037] = 0x851E,
  [11038] = 0x84F6, [11039] = 0x8531, [11040] = 0x8526, [11041] = 0x84E7,
  [11042] = 0x84E8, [11043] = 0x84F0, [11044] = 0x84EF, [11045] = 0x84F9,
  [11046] = 0x8518, [11047] = 0x8520, [11048] = 0x8530, [11049] = 0x850B,
  [11050] = 0x8519, [11051] = 0x852F, [11052] = 0x8662, [11053] = 0x8756,
  [11054] = 0x8763, [11055] = 0x8764, [11056] = 0x8777, [11057] = 0x87E1,
  [11058] = 0x8773, [11059] = 0x8758, [11060] = 0x8754, [11061] = 0x875B,
  [11062] = 0x8752, [11063] = 0x8761, [11064] = 0x875A, [11065] = 0x8751,
  [11066] = 0x875E, [11067] = 0x876D, [11068] = 0x876A, [11069] = 0x8750,
  [11070] = 0x874E, [11071] = 0x875F, [11072] = 0x875D, [11073] = 0x876F,
  [11074] = 0x876C, [11075] = 0x877A, [11076] = 0x876E, [11077] = 0x875C,
  [11078] = 0x8765, [11079] = 0x874F, [11080] = 0x877B, [11081] = 0x8775,
  [11082] = 0x8762, [11083] = 0x8767, [11084] = 0x8769, [11085] = 0x885A,
  [11086] = 0x8905, [11087] = 0x890C, [11088] = 0x8914, [11089] = 0x890B,
  [11090] = 0x8917, [11091] = 0x8918, [11092] = 0x8919, [11093] = 0x8906,
  [11094] = 0x8916, [11095] = 0x8911, [11096] = 0x890E, [11097] = 0x8909,
  [11098] = 0x89A2, [11099] = 0x89A4, [11100] = 0x89A3, [11101] = 0x89ED,
  [11102] = 0x89F0, [11103] = 0x89EC, [11104] = 0x8ACF, [11105] = 0x8AC6,
  [11106] = 0x8AB8, [11107] = 0x8AD3, [11108] = 0x8AD1, [11109] = 0x8AD4,
  [11110] = 0x8AD5, [11111] = 0x8ABB, [11112] = 0x8AD7, [11113] = 0x8ABE,
  [11114] = 0x8AC0, [11115] = 0x8AC5, [11116] = 0x8AD8, [11117] = 0x8AC3,
  [11118] = 0x8ABA, [11119] = 0x8ABD, [11120] = 0x8AD9, [11121] = 0x8C3E,
  [11122] = 0x8C4D, [11123] = 0x8C8F, [11124] = 0x8CE5, [11125] = 0x8CDF,
  [11126] = 0x8CD9, [11127] = 0x8CE8, [11128] = 0x8CDA, [11129] = 0x8CDD,
  [11130] = 0x8CE7, [11131] = 0x8DA0, [11132] = 0x8D9C, [11133] = 0x8DA1,
  [11134] = 0x8D9B, [11135] = 0x8E20, [11136] = 0x8E23, [11137] = 0x8E25,
  [11138] = 0x8E24, [11139] = 0x8E2E, [11140] = 0x8E15, [11141] = 0x8E1B,
  [11142] = 0x8E16, [11143] = 0x8E11, [11144] = 0x8E19, [11145] = 0x8E26,
  [11146] = 0x8E27, [11147] = 0x8E14, [11148] = 0x8E12, [11149] = 0x8E18,
  [11150] = 0x8E13, [11151] = 0x8E1C, [11152] = 0x8E17, [11153] = 0x8E1A,
  [11154] = 0x8F2C, [11155] = 0x8F24, [11156] = 0x8F18, [11157] = 0x8F1A,
  [11158] = 0x8F20, [11159] = 0x8F23, [11160] = 0x8F16, [11161] = 0x8F17,
  [11162] = 0x9073, [11163] = 0x9070, [11164] = 0x906F, [11165] = 0x9067,
  [11166] = 0x906B, [11167] = 0x912F, [11168] = 0x912B, [11169] = 0x9129,
  [11170] = 0x912A, [11171] = 0x9132, [11172] = 0x9126, [11173] = 0x912E,
  [11174] = 0x9185, [11175] = 0x9186, [11176] = 0x918A, [11177] = 0x9181,
  [11178] = 0x9182, [11179] = 0x9184, [11180] = 0x9180, [11181] = 0x92D0,
  [11182] = 0x92C3, [11183] = 0x92C4, [11184] = 0x92C0, [11185] = 0x92D9,
  [11186] = 0x92B6, [11187] = 0x92CF, [11188] = 0x92F1, [11189] = 0x92DF,
  [11190] = 0x92D8, [11191] = 0x92E9, [11192] = 0x92D7, [11193] = 0x92DD,
  [11194] = 0x92CC, [11195] = 0x92EF, [11196] = 0x92C2, [11197] = 0x92E8,
  [11198] = 0x92CA, [11199] = 0x92C8, [11200] = 0x92CE, [11201] = 0x92E6,
  [11202] = 0x92CD, [11203] = 0x92D5, [11204] = 0x92C9, [11205] = 0x92E0,
  [11206] = 0x92DE, [11207] = 0x92E7, [11208] = 0x92D1, [11209] = 0x92D3,
  [11210] = 0x92B5, [11211] = 0x92E1, [11212] = 0x92C6, [11213] = 0x92B4,
  [11214] = 0x957C, [11215] = 0x95AC, [11216] = 0x95AB, [11217] = 0x95AE,
  [11218] = 0x95B0, [11219] = 0x96A4, [11220] = 0x96A2, [11221] = 0x96D3,
  [11222] = 0x9705, [11223] = 0x9708, [11224] = 0x9702, [11225] = 0x975A,
  [11226] = 0x978A, [11227] = 0x978E, [11228] = 0x9788, [11229] = 0x97D0,
  [11230] = 0x97CF, [11231] = 0x981E, [11232] = 0x981D, [11233] = 0x9826,
  [11234] = 0x9829, [11235] = 0x9828, [11236] = 0x9820, [11237] = 0x981B,
  [11238] = 0x9827, [11239] = 0x98B2, [11240] = 0x9908, [11241] = 0x98FA,
  [11242] = 0x9911, [11243] = 0x9914, [11244] = 0x9916, [11245] = 0x9917,
  [11246] = 0x9915, [11247] = 0x99DC, [11248] = 0x99CD, [11249] = 0x99CF,
  [11250] = 0x99D3, [11251] = 0x99D4, [11252] = 0x99CE, [11253] = 0x99C9,
  [11254] = 0x99D6, [11255] = 0x99D8, [11256] = 0x99CB, [11257] = 0x99D7,
  [11258] = 0x99CC, [11259] = 0x9AB3, [11260] = 0x9AEC, [11261] = 0x9AEB,
  [11262] = 0x9AF3, [11263] = 0x9AF2, [11264] = 0x9AF1, [11265] = 0x9B46,
  [11266] = 0x9B43, [11267] = 0x9B67, [11268] = 0x9B74, [11269] = 0x9B71,
  [11270] = 0x9B66, [11271] = 0x9B76, [11272] = 0x9B75, [11273] = 0x9B70,
  [11274] = 0x9B68, [11275] = 0x9B64, [11276] = 0x9B6C, [11277] = 0x9CFC,
  [11278] = 0x9CFA, [11279] = 0x9CFD, [11280] = 0x9CFF, [11281] = 0x9CF7,
  [11282] = 0x9D07, [11283] = 0x9D00, [11284] = 0x9CF9, [11285] = 0x9CFB,
  [11286] = 0x9D08, [11287] = 0x9D05, [11288] = 0x9D04, [11289] = 0x9E83,
  [11290] = 0x9ED3, [11291] = 0x9F0F, [11292] = 0x9F10, [11293] = 0x511C,
  [11294] = 0x5113, [11295] = 0x5117, [11296] = 0x511A, [11297] = 0x5111,
  [11298] = 0x51DE, [11299] = 0x5334, [11300] = 0x53E1, [11301] = 0x5670,
  [11302] = 0x5660, [11303] = 0x566E, [11304] = 0x5673, [11305] = 0x5666,
  [11306] = 0x5663, [11307] = 0x566D, [11308] = 0x5672, [11309] = 0x565E,
  [11310] = 0x5677, [11311] = 0x571C, [11312] = 0x571B, [11313] = 0x58C8,
  [11314] = 0x58BD, [11315] = 0x58C9, [11316] = 0x58BF, [11317] = 0x58BA,
  [11318] = 0x58C2, [11319] = 0x58BC, [11320] = 0x58C6, [11321] = 0x5B17,
  [11322] = 0x5B19, [11323] = 0x5B1B, [11324] = 0x5B21, [11325] = 0x5B14,
  [11326] = 0x5B13, [11327] = 0x5B10, [11328] = 0x5B16, [11329] = 0x5B28,
  [11330] = 0x5B1A, [11331] = 0x5B20, [11332] = 0x5B1E, [11333] = 0x5BEF,
  [11334] = 0x5DAC, [11335] = 0x5DB1, [11336] = 0x5DA9, [11337] = 0x5DA7,
  [11338] = 0x5DB5, [11339] = 0x5DB0, [11340] = 0x5DAE, [11341] = 0x5DAA,
  [11342] = 0x5DA8, [11343] = 0x5DB2, [11344] = 0x5DAD, [11345] = 0x5DAF,
  [11346] = 0x5DB4, [11347] = 0x5E67, [11348] = 0x5E68, [11349] = 0x5E66,
  [11350] = 0x5E6F, [11351] = 0x5EE9, [11352] = 0x5EE7, [11353] = 0x5EE6,
  [11354] = 0x5EE8, [11355] = 0x5EE5, [11356] = 0x5F4B, [11357] = 0x5FBC,
  [11358] = 0x619D, [11359] = 0x61A8, [11360] = 0x6196, [11361] = 0x61C5,
  [11362] = 0x61B4, [11363] = 0x61C6, [11364] = 0x61C1, [11365] = 0x61CC,
  [11366] = 0x61BA, [11367] = 0x61BF, [11368] = 0x61B8, [11369] = 0x618C,
  [11370] = 0x64D7, [11371] = 0x64D6, [11372] = 0x64D0, [11373] = 0x64CF,
  [11374] = 0x64C9, [11375] = 0x64BD, [11376] = 0x6489, [11377] = 0x64C3,
  [11378] = 0x64DB, [11379] = 0x64F3, [11380] = 0x64D9, [11381] = 0x6533,
  [11382] = 0x657F, [11383] = 0x657C, [11384] = 0x65A2, [11385] = 0x66C8,
  [11386] = 0x66BE, [11387] = 0x66C0, [11388] = 0x66CA, [11389] = 0x66CB,
  [11390] = 0x66CF, [11391] = 0x66BD, [11392] = 0x66BB, [11393] = 0x66BA,
  [11394] = 0x66CC, [11395] = 0x6723, [11396] = 0x6A34, [11397] = 0x6A66,
  [11398] = 0x6A49, [11399] = 0x6A67, [11400] = 0x6A32, [11401] = 0x6A68,
  [11402] = 0x6A3E, [11403] = 0x6A5D, [11404] = 0x6A6D, [11405] = 0x6A76,
  [11406] = 0x6A5B, [11407] = 0x6A51, [11408] = 0x6A28, [11409] = 0x6A5A,
  [11410] = 0x6A3B, [11411] = 0x6A3F, [11412] = 0x6A41, [11413] = 0x6A6A,
  [11414] = 0x6A64, [11415] = 0x6A50, [11416] = 0x6A4F, [11417] = 0x6A54,
  [11418] = 0x6A6F, [11419] = 0x6A69, [11420] = 0x6A60, [11421] = 0x6A3C,
  [11422] = 0x6A5E, [11423] = 0x6A56, [11424] = 0x6A55, [11425] = 0x6A4D,
  [11426] = 0x6A4E, [11427] = 0x6A46, [11428] = 0x6B55, [11429] = 0x6B54,
  [11430] = 0x6B56, [11431] = 0x6BA7, [11432] = 0x6BAA, [11433] = 0x6BAB,
  [11434] = 0x6BC8, [11435] = 0x6BC7, [11436] = 0x6C04, [11437] = 0x6C03,
  [11438] = 0x6C06, [11439] = 0x6FAD, [11440] = 0x6FCB, [11441] = 0x6FA3,
  [11442] = 0x6FC7, [11443] = 0x6FBC, [11444] = 0x6FCE, [11445] = 0x6FC8,
  [11446] = 0x6F5E, [11447] = 0x6FC4, [11448] = 0x6FBD, [11449] = 0x6F9E,
  [11450] = 0x6FCA, [11451] = 0x6FA8, [11452] = 0x7004, [11453] = 0x6FA5,
  [11454] = 0x6FAE, [11455] = 0x6FBA, [11456] = 0x6FAC, [11457] = 0x6FAA,
  [11458] = 0x6FCF, [11459] = 0x6FBF, [11460] = 0x6FB8, [11461] = 0x6FA2,
  [11462] = 0x6FC9, [11463] = 0x6FAB, [11464] = 0x6FCD, [11465] = 0x6FAF,
  [11466] = 0x6FB2, [11467] = 0x6FB0, [11468] = 0x71C5, [11469] = 0x71C2,
  [11470] = 0x71BF, [11471] = 0x71B8, [11472] = 0x71D6, [11473] = 0x71C0,
  [11474] = 0x71C1, [11475] = 0x71CB, [11476] = 0x71D4, [11477] = 0x71CA,
  [11478] = 0x71C7, [11479] = 0x71CF, [11480] = 0x71BD, [11481] = 0x71D8,
  [11482] = 0x71BC, [11483] = 0x71C6, [11484] = 0x71DA, [11485] = 0x71DB,
  [11486] = 0x729D, [11487] = 0x729E, [11488] = 0x7369, [11489] = 0x7366,
  [11490] = 0x7367, [11491] = 0x736C, [11492] = 0x7365, [11493] = 0x736B,
  [11494] = 0x736A, [11495] = 0x747F, [11496] = 0x749A, [11497] = 0x74A0,
  [11498] = 0x7494, [11499] = 0x7492, [11500] = 0x7495, [11501] = 0x74A1,
  [11502] = 0x750B, [11503] = 0x7580, [11504] = 0x762F, [11505] = 0x762D,
  [11506] = 0x7631, [11507] = 0x763D, [11508] = 0x7633, [11509] = 0x763C,
  [11510] = 0x7635, [11511] = 0x7632, [11512] = 0x7630, [11513] = 0x76BB,
  [11514] = 0x76E6, [11515] = 0x779A, [11516] = 0x779D, [11517] = 0x77A1,
  [11518] = 0x779C, [11519] = 0x779B, [11520] = 0x77A2, [11521] = 0x77A3,
  [11522] = 0x7795, [11523] = 0x7799, [11524] = 0x7797, [11525] = 0x78DD,
  [11526] = 0x78E9, [11527] = 0x78E5, [11528] = 0x78EA, [11529] = 0x78DE,
  [11530] = 0x78E3, [11531] = 0x78DB, [11532] = 0x78E1, [11533] = 0x78E2,
  [11534] = 0x78ED, [11535] = 0x78DF, [11536] = 0x78E0, [11537] = 0x79A4,
  [11538] = 0x7A44, [11539] = 0x7A48, [11540] = 0x7A47, [11541] = 0x7AB6,
  [11542] = 0x7AB8, [11543] = 0x7AB5, [11544] = 0x7AB1, [11545] = 0x7AB7,
  [11546] = 0x7BDE, [11547] = 0x7BE3, [11548] = 0x7BE7, [11549] = 0x7BDD,
  [11550] = 0x7BD5, [11551] = 0x7BE5, [11552] = 0x7BDA, [11553] = 0x7BE8,
  [11554] = 0x7BF9, [11555] = 0x7BD4, [11556] = 0x7BEA, [11557] = 0x7BE2,
  [11558] = 0x7BDC, [11559] = 0x7BEB, [11560] = 0x7BD8, [11561] = 0x7BDF,
  [11562] = 0x7CD2, [11563] = 0x7CD4, [11564] = 0x7CD7, [11565] = 0x7CD0,
  [11566] = 0x7CD1, [11567] = 0x7E12, [11568] = 0x7E21, [11569] = 0x7E17,
  [11570] = 0x7E0C, [11571] = 0x7E1F, [11572] = 0x7E20, [11573] = 0x7E13,
  [11574] = 0x7E0E, [11575] = 0x7E1C, [11576] = 0x7E15, [11577] = 0x7E1A,
  [11578] = 0x7E22, [11579] = 0x7E0B, [11580] = 0x7E0F, [11581] = 0x7E16,
  [11582] = 0x7E0D, [11583] = 0x7E14, [11584] = 0x7E25, [11585] = 0x7E24,
  [11586] = 0x7F43, [11587] = 0x7F7B, [11588] = 0x7F7C, [11589] = 0x7F7A,
  [11590] = 0x7FB1, [11591] = 0x7FEF, [11592] = 0x802A, [11593] = 0x8029,
  [11594] = 0x806C, [11595] = 0x81B1, [11596] = 0x81A6, [11597] = 0x81AE,
  [11598] = 0x81B9, [11599] = 0x81B5, [11600] = 0x81AB, [11601] = 0x81B0,
  [11602] = 0x81AC, [11603] = 0x81B4, [11604] = 0x81B2, [11605] = 0x81B7,
  [11606] = 0x81A7, [11607] = 0x81F2, [11608] = 0x8255, [11609] = 0x8256,
  [11610] = 0x8257, [11611] = 0x8556, [11612] = 0x8545, [11613] = 0x856B,
  [11614] = 0x854D, [11615] = 0x8553, [11616] = 0x8561, [11617] = 0x8558,
  [11618] = 0x8540, [11619] = 0x8546, [11620] = 0x8564, [11621] = 0x8541,
  [11622] = 0x8562, [11623] = 0x8544, [11624] = 0x8551, [11625] = 0x8547,
  [11626] = 0x8563, [11627] = 0x853E, [11628] = 0x855B, [11629] = 0x8571,
  [11630] = 0x854E, [11631] = 0x856E, [11632] = 0x8575, [11633] = 0x8555,
  [11634] = 0x8567, [11635] = 0x8560, [11636] = 0x858C, [11637] = 0x8566,
  [11638] = 0x855D, [11639] = 0x8554, [11640] = 0x8565, [11641] = 0x856C,
  [11642] = 0x8663, [11643] = 0x8665, [11644] = 0x8664, [11645] = 0x879B,
  [11646] = 0x878F, [11647] = 0x8797, [11648] = 0x8793, [11649] = 0x8792,
  [11650] = 0x8788, [11651] = 0x8781, [11652] = 0x8796, [11653] = 0x8798,
  [11654] = 0x8779, [11655] = 0x8787, [11656] = 0x87A3, [11657] = 0x8785,
  [11658] = 0x8790, [11659] = 0x8791, [11660] = 0x879D, [11661] = 0x8784,
  [11662] = 0x8794, [11663] = 0x879C, [11664] = 0x879A, [11665] = 0x8789,
  [11666] = 0x891E, [11667] = 0x8926, [11668] = 0x8930, [11669] = 0x892D,
  [11670] = 0x892E, [11671] = 0x8927, [11672] = 0x8931, [11673] = 0x8922,
  [11674] = 0x8929, [11675] = 0x8923, [11676] = 0x892F, [11677] = 0x892C,
  [11678] = 0x891F, [11679] = 0x89F1, [11680] = 0x8AE0, [11681] = 0x8AE2,
  [11682] = 0x8AF2, [11683] = 0x8AF4, [11684] = 0x8AF5, [11685] = 0x8ADD,
  [11686] = 0x8B14, [11687] = 0x8AE4, [11688] = 0x8ADF, [11689] = 0x8AF0,
  [11690] = 0x8AC8, [11691] = 0x8ADE, [11692] = 0x8AE1, [11693] = 0x8AE8,
  [11694] = 0x8AFF, [11695] = 0x8AEF, [11696] = 0x8AFB, [11697] = 0x8C91,
  [11698] = 0x8C92, [11699] = 0x8C90, [11700] = 0x8CF5, [11701] = 0x8CEE,
  [11702] = 0x8CF1, [11703] = 0x8CF0, [11704] = 0x8CF3, [11705] = 0x8D6C,
  [11706] = 0x8D6E, [11707] = 0x8DA5, [11708] = 0x8DA7, [11709] = 0x8E33,
  [11710] = 0x8E3E, [11711] = 0x8E38, [11712] = 0x8E40, [11713] = 0x8E45,
  [11714] = 0x8E36, [11715] = 0x8E3C, [11716] = 0x8E3D, [11717] = 0x8E41,
  [11718] = 0x8E30, [11719] = 0x8E3F, [11720] = 0x8EBD, [11721] = 0x8F36,
  [11722] = 0x8F2E, [11723] = 0x8F35, [11724] = 0x8F32, [11725] = 0x8F39,
  [11726] = 0x8F37, [11727] = 0x8F34, [11728] = 0x9076, [11729] = 0x9079,
  [11730] = 0x907B, [11731] = 0x9086, [11732] = 0x90FA, [11733] = 0x9133,
  [11734] = 0x9135, [11735] = 0x9136, [11736] = 0x9193, [11737] = 0x9190,
  [11738] = 0x9191, [11739] = 0x918D, [11740] = 0x918F, [11741] = 0x9327,
  [11742] = 0x931E, [11743] = 0x9308, [11744] = 0x931F, [11745] = 0x9306,
  [11746] = 0x930F, [11747] = 0x937A, [11748] = 0x9338, [11749] = 0x933C,
  [11750] = 0x931B, [11751] = 0x9323, [11752] = 0x9312, [11753] = 0x9301,
  [11754] = 0x9346, [11755] = 0x932D, [11756] = 0x930E, [11757] = 0x930D,
  [11758] = 0x92CB, [11759] = 0x931D, [11760] = 0x92FA, [11761] = 0x9325,
  [11762] = 0x9313, [11763] = 0x92F9, [11764] = 0x92F7, [11765] = 0x9334,
  [11766] = 0x9302, [11767] = 0x9324, [11768] = 0x92FF, [11769] = 0x9329,
  [11770] = 0x9339, [11771] = 0x9335, [11772] = 0x932A, [11773] = 0x9314,
  [11774] = 0x930C, [11775] = 0x930B, [11776] = 0x92FE, [11777] = 0x9309,
  [11778] = 0x9300, [11779] = 0x92FB, [11780] = 0x9316, [11781] = 0x95BC,
  [11782] = 0x95CD, [11783] = 0x95BE, [11784] = 0x95B9, [11785] = 0x95BA,
  [11786] = 0x95B6, [11787] = 0x95BF, [11788] = 0x95B5, [11789] = 0x95BD,
  [11790] = 0x96A9, [11791] = 0x96D4, [11792] = 0x970B, [11793] = 0x9712,
  [11794] = 0x9710, [11795] = 0x9799, [11796] = 0x9797, [11797] = 0x9794,
  [11798] = 0x97F0, [11799] = 0x97F8, [11800] = 0x9835, [11801] = 0x982F,
  [11802] = 0x9832, [11803] = 0x9924, [11804] = 0x991F, [11805] = 0x9927,
  [11806] = 0x9929, [11807] = 0x999E, [11808] = 0x99EE, [11809] = 0x99EC,
  [11810] = 0x99E5, [11811] = 0x99E4, [11812] = 0x99F0, [11813] = 0x99E3,
  [11814] = 0x99EA, [11815] = 0x99E9, [11816] = 0x99E7, [11817] = 0x9AB9,
  [11818] = 0x9ABF, [11819] = 0x9AB4, [11820] = 0x9ABB, [11821] = 0x9AF6,
  [11822] = 0x9AFA, [11823] = 0x9AF9, [11824] = 0x9AF7, [11825] = 0x9B33,
  [11826] = 0x9B80, [11827] = 0x9B85, [11828] = 0x9B87, [11829] = 0x9B7C,
  [11830] = 0x9B7E, [11831] = 0x9B7B, [11832] = 0x9B82, [11833] = 0x9B93,
  [11834] = 0x9B92, [11835] = 0x9B90, [11836] = 0x9B7A, [11837] = 0x9B95,
  [11838] = 0x9B7D, [11839] = 0x9B88, [11840] = 0x9D25, [11841] = 0x9D17,
  [11842] = 0x9D20, [11843] = 0x9D1E, [11844] = 0x9D14, [11845] = 0x9D29,
  [11846] = 0x9D1D, [11847] = 0x9D18, [11848] = 0x9D22, [11849] = 0x9D10,
  [11850] = 0x9D19, [11851] = 0x9D1F, [11852] = 0x9E88, [11853] = 0x9E86,
  [11854] = 0x9E87, [11855] = 0x9EAE, [11856] = 0x9EAD, [11857] = 0x9ED5,
  [11858] = 0x9ED6, [11859] = 0x9EFA, [11860] = 0x9F12, [11861] = 0x9F3D,
  [11862] = 0x5126, [11863] = 0x5125, [11864] = 0x5122, [11865] = 0x5124,
  [11866] = 0x5120, [11867] = 0x5129, [11868] = 0x52F4, [11869] = 0x5693,
  [11870] = 0x568C, [11871] = 0x568D, [11872] = 0x5686, [11873] = 0x5684,
  [11874] = 0x5683, [11875] = 0x567E, [11876] = 0x5682, [11877] = 0x567F,
  [11878] = 0x5681, [11879] = 0x58D6, [11880] = 0x58D4, [11881] = 0x58CF,
  [11882] = 0x58D2, [11883] = 0x5B2D, [11884] = 0x5B25, [11885] = 0x5B32,
  [11886] = 0x5B23, [11887] = 0x5B2C, [11888] = 0x5B27, [11889] = 0x5B26,
  [11890] = 0x5B2F, [11891] = 0x5B2E, [11892] = 0x5B7B, [11893] = 0x5BF1,
  [11894] = 0x5BF2, [11895] = 0x5DB7, [11896] = 0x5E6C, [11897] = 0x5E6A,
  [11898] = 0x5FBE, [11899] = 0x5FBB, [11900] = 0x61C3, [11901] = 0x61B5,
  [11902] = 0x61BC, [11903] = 0x61E7, [11904] = 0x61E0, [11905] = 0x61E5,
  [11906] = 0x61E4, [11907] = 0x61E8, [11908] = 0x61DE, [11909] = 0x64EF,
  [11910] = 0x64E9, [11911] = 0x64E3, [11912] = 0x64EB, [11913] = 0x64E4,
  [11914] = 0x64E8, [11915] = 0x6581, [11916] = 0x6580, [11917] = 0x65B6,
  [11918] = 0x65DA, [11919] = 0x66D2, [11920] = 0x6A8D, [11921] = 0x6A96,
  [11922] = 0x6A81, [11923] = 0x6AA5, [11924] = 0x6A89, [11925] = 0x6A9F,
  [11926] = 0x6A9B, [11927] = 0x6AA1, [11928] = 0x6A9E, [11929] = 0x6A87,
  [11930] = 0x6A93, [11931] = 0x6A8E, [11932] = 0x6A95, [11933] = 0x6A83,
  [11934] = 0x6AA8, [11935] = 0x6AA4, [11936] = 0x6A91, [11937] = 0x6A7F,
  [11938] = 0x6AA6, [11939] = 0x6A9A, [11940] = 0x6A85, [11941] = 0x6A8C,
  [11942] = 0x6A92, [11943] = 0x6B5B, [11944] = 0x6BAD, [11945] = 0x6C09,
  [11946] = 0x6FCC, [11947] = 0x6FA9, [11948] = 0x6FF4, [11949] = 0x6FD4,
  [11950] = 0x6FE3, [11951] = 0x6FDC, [11952] = 0x6FED, [11953] = 0x6FE7,
  [11954] = 0x6FE6, [11955] = 0x6FDE, [11956] = 0x6FF2, [11957] = 0x6FDD,
  [11958] = 0x6FE2, [11959] = 0x6FE8, [11960] = 0x71E1, [11961] = 0x71F1,
  [11962] = 0x71E8, [11963] = 0x71F2, [11964] = 0x71E4, [11965] = 0x71F0,
  [11966] = 0x71E2, [11967] = 0x7373, [11968] = 0x736E, [11969] = 0x736F,
  [11970] = 0x7497, [11971] = 0x74B2, [11972] = 0x74AB, [11973] = 0x7490,
  [11974] = 0x74AA, [11975] = 0x74AD, [11976] = 0x74B1, [11977] = 0x74A5,
  [11978] = 0x74AF, [11979] = 0x7510, [11980] = 0x7511, [11981] = 0x7512,
  [11982] = 0x750F, [11983] = 0x7584, [11984] = 0x7643, [11985] = 0x7648,
  [11986] = 0x7649, [11987] = 0x7647, [11988] = 0x76A4, [11989] = 0x76E9,
  [11990] = 0x77B5, [11991] = 0x77AB, [11992] = 0x77B2, [11993] = 0x77B7,
  [11994] = 0x77B6, [11995] = 0x77B4, [11996] = 0x77B1, [11997] = 0x77A8,
  [11998] = 0x77F0, [11999] = 0x78F3, [12000] = 0x78FD, [12001] = 0x7902,
  [12002] = 0x78FB, [12003] = 0x78FC, [12004] = 0x78F2, [12005] = 0x7905,
  [12006] = 0x78F9, [12007] = 0x78FE, [12008] = 0x7904, [12009] = 0x79AB,
  [12010] = 0x79A8, [12011] = 0x7A5C, [12012] = 0x7A5B, [12013] = 0x7A56,
  [12014] = 0x7A58, [12015] = 0x7A54, [12016] = 0x7A5A, [12017] = 0x7ABE,
  [12018] = 0x7AC0, [12019] = 0x7AC1, [12020] = 0x7C05, [12021] = 0x7C0F,
  [12022] = 0x7BF2, [12023] = 0x7C00, [12024] = 0x7BFF, [12025] = 0x7BFB,
  [12026] = 0x7C0E, [12027] = 0x7BF4, [12028] = 0x7C0B, [12029] = 0x7BF3,
  [12030] = 0x7C02, [12031] = 0x7C09, [12032] = 0x7C03, [12033] = 0x7C01,
  [12034] = 0x7BF8, [12035] = 0x7BFD, [12036] = 0x7C06, [12037] = 0x7BF0,
  [12038] = 0x7BF1, [12039] = 0x7C10, [12040] = 0x7C0A, [12041] = 0x7CE8,
  [12042] = 0x7E2D, [12043] = 0x7E3C, [12044] = 0x7E42, [12045] = 0x7E33,
  [12046] = 0x9848, [12047] = 0x7E38, [12048] = 0x7E2A, [12049] = 0x7E49,
  [12050] = 0x7E40, [12051] = 0x7E47, [12052] = 0x7E29, [12053] = 0x7E4C,
  [12054] = 0x7E30, [12055] = 0x7E3B, [12056] = 0x7E36, [12057] = 0x7E44,
  [12058] = 0x7E3A, [12059] = 0x7F45, [12060] = 0x7F7F, [12061] = 0x7F7E,
  [12062] = 0x7F7D, [12063] = 0x7FF4, [12064] = 0x7FF2, [12065] = 0x802C,
  [12066] = 0x81BB, [12067] = 0x81C4, [12068] = 0x81CC, [12069] = 0x81CA,
  [12070] = 0x81C5, [12071] = 0x81C7, [12072] = 0x81BC, [12073] = 0x81E9,
  [12074] = 0x825B, [12075] = 0x825A, [12076] = 0x825C, [12077] = 0x8583,
  [12078] = 0x8580, [12079] = 0x858F, [12080] = 0x85A7, [12081] = 0x8595,
  [12082] = 0x85A0, [12083] = 0x858B, [12084] = 0x85A3, [12085] = 0x857B,
  [12086] = 0x85A4, [12087] = 0x859A, [12088] = 0x859E, [12089] = 0x8577,
  [12090] = 0x857C, [12091] = 0x8589, [12092] = 0x85A1, [12093] = 0x857A,
  [12094] = 0x8578, [12095] = 0x8557, [12096] = 0x858E, [12097] = 0x8596,
  [12098] = 0x8586, [12099] = 0x858D, [12100] = 0x8599, [12101] = 0x859D,
  [12102] = 0x8581, [12103] = 0x85A2, [12104] = 0x8582, [12105] = 0x8588,
  [12106] = 0x8585, [12107] = 0x8579, [12108] = 0x8576, [12109] = 0x8598,
  [12110] = 0x8590, [12111] = 0x859F, [12112] = 0x8668, [12113] = 0x87BE,
  [12114] = 0x87AA, [12115] = 0x87AD, [12116] = 0x87C5, [12117] = 0x87B0,
  [12118] = 0x87AC, [12119] = 0x87B9, [12120] = 0x87B5, [12121] = 0x87BC,
  [12122] = 0x87AE, [12123] = 0x87C9, [12124] = 0x87C3, [12125] = 0x87C2,
  [12126] = 0x87CC, [12127] = 0x87B7, [12128] = 0x87AF, [12129] = 0x87C4,
  [12130] = 0x87CA, [12131] = 0x87B4, [12132] = 0x87B6, [12133] = 0x87BF,
  [12134] = 0x87B8, [12135] = 0x87BD, [12136] = 0x87DE, [12137] = 0x87B2,
  [12138] = 0x8935, [12139] = 0x8933, [12140] = 0x893C, [12141] = 0x893E,
  [12142] = 0x8941, [12143] = 0x8952, [12144] = 0x8937, [12145] = 0x8942,
  [12146] = 0x89AD, [12147] = 0x89AF, [12148] = 0x89AE, [12149] = 0x89F2,
  [12150] = 0x89F3, [12151] = 0x8B1E, [12152] = 0x8B18, [12153] = 0x8B16,
  [12154] = 0x8B11, [12155] = 0x8B05, [12156] = 0x8B0B, [12157] = 0x8B22,
  [12158] = 0x8B0F, [12159] = 0x8B12, [12160] = 0x8B15, [12161] = 0x8B07,
  [12162] = 0x8B0D, [12163] = 0x8B08, [12164] = 0x8B06, [12165] = 0x8B1C,
  [12166] = 0x8B13, [12167] = 0x8B1A, [12168] = 0x8C4F, [12169] = 0x8C70,
  [12170] = 0x8C72, [12171] = 0x8C71, [12172] = 0x8C6F, [12173] = 0x8C95,
  [12174] = 0x8C94, [12175] = 0x8CF9, [12176] = 0x8D6F, [12177] = 0x8E4E,
  [12178] = 0x8E4D, [12179] = 0x8E53, [12180] = 0x8E50, [12181] = 0x8E4C,
  [12182] = 0x8E47, [12183] = 0x8F43, [12184] = 0x8F40, [12185] = 0x9085,
  [12186] = 0x907E, [12187] = 0x9138, [12188] = 0x919A, [12189] = 0x91A2,
  [12190] = 0x919B, [12191] = 0x9199, [12192] = 0x919F, [12193] = 0x91A1,
  [12194] = 0x919D, [12195] = 0x91A0, [12196] = 0x93A1, [12197] = 0x9383,
  [12198] = 0x93AF, [12199] = 0x9364, [12200] = 0x9356, [12201] = 0x9347,
  [12202] = 0x937C, [12203] = 0x9358, [12204] = 0x935C, [12205] = 0x9376,
  [12206] = 0x9349, [12207] = 0x9350, [12208] = 0x9351, [12209] = 0x9360,
  [12210] = 0x936D, [12211] = 0x938F, [12212] = 0x934C, [12213] = 0x936A,
  [12214] = 0x9379, [12215] = 0x9357, [12216] = 0x9355, [12217] = 0x9352,
  [12218] = 0x934F, [12219] = 0x9371, [12220] = 0x9377, [12221] = 0x937B,
  [12222] = 0x9361, [12223] = 0x935E, [12224] = 0x9363, [12225] = 0x9367,
  [12226] = 0x9380, [12227] = 0x934E, [12228] = 0x9359, [12229] = 0x95C7,
  [12230] = 0x95C0, [12231] = 0x95C9, [12232] = 0x95C3, [12233] = 0x95C5,
  [12234] = 0x95B7, [12235] = 0x96AE, [12236] = 0x96B0, [12237] = 0x96AC,
  [12238] = 0x9720, [12239] = 0x971F, [12240] = 0x9718, [12241] = 0x971D,
  [12242] = 0x9719, [12243] = 0x979A, [12244] = 0x97A1, [12245] = 0x979C,
  [12246] = 0x979E, [12247] = 0x979D, [12248] = 0x97D5, [12249] = 0x97D4,
  [12250] = 0x97F1, [12251] = 0x9841, [12252] = 0x9844, [12253] = 0x984A,
  [12254] = 0x9849, [12255] = 0x9845, [12256] = 0x9843, [12257] = 0x9925,
  [12258] = 0x992B, [12259] = 0x992C, [12260] = 0x992A, [12261] = 0x9933,
  [12262] = 0x9932, [12263] = 0x992F, [12264] = 0x992D, [12265] = 0x9931,
  [12266] = 0x9930, [12267] = 0x9998, [12268] = 0x99A3, [12269] = 0x99A1,
  [12270] = 0x9A02, [12271] = 0x99FA, [12272] = 0x99F4, [12273] = 0x99F7,
  [12274] = 0x99F9, [12275] = 0x99F8, [12276] = 0x99F6, [12277] = 0x99FB,
  [12278] = 0x99FD, [12279] = 0x99FE, [12280] = 0x99FC, [12281] = 0x9A03,
  [12282] = 0x9ABE, [12283] = 0x9AFE, [12284] = 0x9AFD, [12285] = 0x9B01,
  [12286] = 0x9AFC, [12287] = 0x9B48, [12288] = 0x9B9A, [12289] = 0x9BA8,
  [12290] = 0x9B9E, [12291] = 0x9B9B, [12292] = 0x9BA6, [12293] = 0x9BA1,
  [12294] = 0x9BA5, [12295] = 0x9BA4, [12296] = 0x9B86, [12297] = 0x9BA2,
  [12298] = 0x9BA0, [12299] = 0x9BAF, [12300] = 0x9D33, [12301] = 0x9D41,
  [12302] = 0x9D67, [12303] = 0x9D36, [12304] = 0x9D2E, [12305] = 0x9D2F,
  [12306] = 0x9D31, [12307] = 0x9D38, [12308] = 0x9D30, [12309] = 0x9D45,
  [12310] = 0x9D42, [12311] = 0x9D43, [12312] = 0x9D3E, [12313] = 0x9D37,
  [12314] = 0x9D40, [12315] = 0x9D3D, [12316] = 0x7FF5, [12317] = 0x9D2D,
  [12318] = 0x9E8A, [12319] = 0x9E89, [12320] = 0x9E8D, [12321] = 0x9EB0,
  [12322] = 0x9EC8, [12323] = 0x9EDA, [12324] = 0x9EFB, [12325] = 0x9EFF,
  [12326] = 0x9F24, [12327] = 0x9F23, [12328] = 0x9F22, [12329] = 0x9F54,
  [12330] = 0x9FA0, [12331] = 0x5131, [12332] = 0x512D, [12333] = 0x512E,
  [12334] = 0x5698, [12335] = 0x569C, [12336] = 0x5697, [12337] = 0x569A,
  [12338] = 0x569D, [12339] = 0x5699, [12340] = 0x5970, [12341] = 0x5B3C,
  [12342] = 0x5C69, [12343] = 0x5C6A, [12344] = 0x5DC0, [12345] = 0x5E6D,
  [12346] = 0x5E6E, [12347] = 0x61D8, [12348] = 0x61DF, [12349] = 0x61ED,
  [12350] = 0x61EE, [12351] = 0x61F1, [12352] = 0x61EA, [12353] = 0x61F0,
  [12354] = 0x61EB, [12355] = 0x61D6, [12356] = 0x61E9, [12357] = 0x64FF,
  [12358] = 0x6504, [12359] = 0x64FD, [12360] = 0x64F8, [12361] = 0x6501,
  [12362] = 0x6503, [12363] = 0x64FC, [12364] = 0x6594, [12365] = 0x65DB,
  [12366] = 0x66DA, [12367] = 0x66DB, [12368] = 0x66D8, [12369] = 0x6AC5,
  [12370] = 0x6AB9, [12371] = 0x6ABD, [12372] = 0x6AE1, [12373] = 0x6AC6,
  [12374] = 0x6ABA, [12375] = 0x6AB6, [12376] = 0x6AB7, [12377] = 0x6AC7,
  [12378] = 0x6AB4, [12379] = 0x6AAD, [12380] = 0x6B5E, [12381] = 0x6BC9,
  [12382] = 0x6C0B, [12383] = 0x7007, [12384] = 0x700C, [12385] = 0x700D,
  [12386] = 0x7001, [12387] = 0x7005, [12388] = 0x7014, [12389] = 0x700E,
  [12390] = 0x6FFF, [12391] = 0x7000, [12392] = 0x6FFB, [12393] = 0x7026,
  [12394] = 0x6FFC, [12395] = 0x6FF7, [12396] = 0x700A, [12397] = 0x7201,
  [12398] = 0x71FF, [12399] = 0x71F9, [12400] = 0x7203, [12401] = 0x71FD,
  [12402] = 0x7376, [12403] = 0x74B8, [12404] = 0x74C0, [12405] = 0x74B5,
  [12406] = 0x74C1, [12407] = 0x74BE, [12408] = 0x74B6, [12409] = 0x74BB,
  [12410] = 0x74C2, [12411] = 0x7514, [12412] = 0x7513, [12413] = 0x765C,
  [12414] = 0x7664, [12415] = 0x7659, [12416] = 0x7650, [12417] = 0x7653,
  [12418] = 0x7657, [12419] = 0x765A, [12420] = 0x76A6, [12421] = 0x76BD,
  [12422] = 0x76EC, [12423] = 0x77C2, [12424] = 0x77BA, [12425] = 0x78FF,
  [12426] = 0x790C, [12427] = 0x7913, [12428] = 0x7914, [12429] = 0x7909,
  [12430] = 0x7910, [12431] = 0x7912, [12432] = 0x7911, [12433] = 0x79AD,
  [12434] = 0x79AC, [12435] = 0x7A5F, [12436] = 0x7C1C, [12437] = 0x7C29,
  [12438] = 0x7C19, [12439] = 0x7C20, [12440] = 0x7C1F, [12441] = 0x7C2D,
  [12442] = 0x7C1D, [12443] = 0x7C26, [12444] = 0x7C28, [12445] = 0x7C22,
  [12446] = 0x7C25, [12447] = 0x7C30, [12448] = 0x7E5C, [12449] = 0x7E50,
  [12450] = 0x7E56, [12451] = 0x7E63, [12452] = 0x7E58, [12453] = 0x7E62,
  [12454] = 0x7E5F, [12455] = 0x7E51, [12456] = 0x7E60, [12457] = 0x7E57,
  [12458] = 0x7E53, [12459] = 0x7FB5, [12460] = 0x7FB3, [12461] = 0x7FF7,
  [12462] = 0x7FF8, [12463] = 0x8075, [12464] = 0x81D1, [12465] = 0x81D2,
  [12466] = 0x81D0, [12467] = 0x825F, [12468] = 0x825E, [12469] = 0x85B4,
  [12470] = 0x85C6, [12471] = 0x85C0, [12472] = 0x85C3, [12473] = 0x85C2,
  [12474] = 0x85B3, [12475] = 0x85B5, [12476] = 0x85BD, [12477] = 0x85C7,
  [12478] = 0x85C4, [12479] = 0x85BF, [12480] = 0x85CB, [12481] = 0x85CE,
  [12482] = 0x85C8, [12483] = 0x85C5, [12484] = 0x85B1, [12485] = 0x85B6,
  [12486] = 0x85D2, [12487] = 0x8624, [12488] = 0x85B8, [12489] = 0x85B7,
  [12490] = 0x85BE, [12491] = 0x8669, [12492] = 0x87E7, [12493] = 0x87E6,
  [12494] = 0x87E2, [12495] = 0x87DB, [12496] = 0x87EB, [12497] = 0x87EA,
  [12498] = 0x87E5, [12499] = 0x87DF, [12500] = 0x87F3, [12501] = 0x87E4,
  [12502] = 0x87D4, [12503] = 0x87DC, [12504] = 0x87D3, [12505] = 0x87ED,
  [12506] = 0x87D8, [12507] = 0x87E3, [12508] = 0x87A4, [12509] = 0x87D7,
  [12510] = 0x87D9, [12511] = 0x8801, [12512] = 0x87F4, [12513] = 0x87E8,
  [12514] = 0x87DD, [12515] = 0x8953, [12516] = 0x894B, [12517] = 0x894F,
  [12518] = 0x894C, [12519] = 0x8946, [12520] = 0x8950, [12521] = 0x8951,
  [12522] = 0x8949, [12523] = 0x8B2A, [12524] = 0x8B27, [12525] = 0x8B23,
  [12526] = 0x8B33, [12527] = 0x8B30, [12528] = 0x8B35, [12529] = 0x8B47,
  [12530] = 0x8B2F, [12531] = 0x8B3C, [12532] = 0x8B3E, [12533] = 0x8B31,
  [12534] = 0x8B25, [12535] = 0x8B37, [12536] = 0x8B26, [12537] = 0x8B36,
  [12538] = 0x8B2E, [12539] = 0x8B24, [12540] = 0x8B3B, [12541] = 0x8B3D,
  [12542] = 0x8B3A, [12543] = 0x8C42, [12544] = 0x8C75, [12545] = 0x8C99,
  [12546] = 0x8C98, [12547] = 0x8C97, [12548] = 0x8CFE, [12549] = 0x8D04,
  [12550] = 0x8D02, [12551] = 0x8D00, [12552] = 0x8E5C, [12553] = 0x8E62,
  [12554] = 0x8E60, [12555] = 0x8E57, [12556] = 0x8E56, [12557] = 0x8E5E,
  [12558] = 0x8E65, [12559] = 0x8E67, [12560] = 0x8E5B, [12561] = 0x8E5A,
  [12562] = 0x8E61, [12563] = 0x8E5D, [12564] = 0x8E69, [12565] = 0x8E54,
  [12566] = 0x8F46, [12567] = 0x8F47, [12568] = 0x8F48, [12569] = 0x8F4B,
  [12570] = 0x9128, [12571] = 0x913A, [12572] = 0x913B, [12573] = 0x913E,
  [12574] = 0x91A8, [12575] = 0x91A5, [12576] = 0x91A7, [12577] = 0x91AF,
  [12578] = 0x91AA, [12579] = 0x93B5, [12580] = 0x938C, [12581] = 0x9392,
  [12582] = 0x93B7, [12583] = 0x939B, [12584] = 0x939D, [12585] = 0x9389,
  [12586] = 0x93A7, [12587] = 0x938E, [12588] = 0x93AA, [12589] = 0x939E,
  [12590] = 0x93A6, [12591] = 0x9395, [12592] = 0x9388, [12593] = 0x9399,
  [12594] = 0x939F, [12595] = 0x938D, [12596] = 0x93B1, [12597] = 0x9391,
  [12598] = 0x93B2, [12599] = 0x93A4, [12600] = 0x93A8, [12601] = 0x93B4,
  [12602] = 0x93A3, [12603] = 0x93A5, [12604] = 0x95D2, [12605] = 0x95D3,
  [12606] = 0x95D1, [12607] = 0x96B3, [12608] = 0x96D7, [12609] = 0x96DA,
  [12610] = 0x5DC2, [12611] = 0x96DF, [12612] = 0x96D8, [12613] = 0x96DD,
  [12614] = 0x9723, [12615] = 0x9722, [12616] = 0x9725, [12617] = 0x97AC,
  [12618] = 0x97AE, [12619] = 0x97A8, [12620] = 0x97AB, [12621] = 0x97A4,
  [12622] = 0x97AA, [12623] = 0x97A2, [12624] = 0x97A5, [12625] = 0x97D7,
  [12626] = 0x97D9, [12627] = 0x97D6, [12628] = 0x97D8, [12629] = 0x97FA,
  [12630] = 0x9850, [12631] = 0x9851, [12632] = 0x9852, [12633] = 0x98B8,
  [12634] = 0x9941, [12635] = 0x993C, [12636] = 0x993A, [12637] = 0x9A0F,
  [12638] = 0x9A0B, [12639] = 0x9A09, [12640] = 0x9A0D, [12641] = 0x9A04,
  [12642] = 0x9A11, [12643] = 0x9A0A, [12644] = 0x9A05, [12645] = 0x9A07,
  [12646] = 0x9A06, [12647] = 0x9AC0, [12648] = 0x9ADC, [12649] = 0x9B08,
  [12650] = 0x9B04, [12651] = 0x9B05, [12652] = 0x9B29, [12653] = 0x9B35,
  [12654] = 0x9B4A, [12655] = 0x9B4C, [12656] = 0x9B4B, [12657] = 0x9BC7,
  [12658] = 0x9BC6, [12659] = 0x9BC3, [12660] = 0x9BBF, [12661] = 0x9BC1,
  [12662] = 0x9BB5, [12663] = 0x9BB8, [12664] = 0x9BD3, [12665] = 0x9BB6,
  [12666] = 0x9BC4, [12667] = 0x9BB9, [12668] = 0x9BBD, [12669] = 0x9D5C,
  [12670] = 0x9D53, [12671] = 0x9D4F, [12672] = 0x9D4A, [12673] = 0x9D5B,
  [12674] = 0x9D4B, [12675] = 0x9D59, [12676] = 0x9D56, [12677] = 0x9D4C,
  [12678] = 0x9D57, [12679] = 0x9D52, [12680] = 0x9D54, [12681] = 0x9D5F,
  [12682] = 0x9D58, [12683] = 0x9D5A, [12684] = 0x9E8E, [12685] = 0x9E8C,
  [12686] = 0x9EDF, [12687] = 0x9F01, [12688] = 0x9F00, [12689] = 0x9F16,
  [12690] = 0x9F25, [12691] = 0x9F2B, [12692] = 0x9F2A, [12693] = 0x9F29,
  [12694] = 0x9F28, [12695] = 0x9F4C, [12696] = 0x9F55, [12697] = 0x5134,
  [12698] = 0x5135, [12699] = 0x5296, [12700] = 0x52F7, [12701] = 0x53B4,
  [12702] = 0x56AB, [12703] = 0x56AD, [12704] = 0x56A6, [12705] = 0x56A7,
  [12706] = 0x56AA, [12707] = 0x56AC, [12708] = 0x58DA, [12709] = 0x58DD,
  [12710] = 0x58DB, [12711] = 0x5912, [12712] = 0x5B3D, [12713] = 0x5B3E,
  [12714] = 0x5B3F, [12715] = 0x5DC3, [12716] = 0x5E70, [12717] = 0x5FBF,
  [12718] = 0x61FB, [12719] = 0x6507, [12720] = 0x6510, [12721] = 0x650D,
  [12722] = 0x6509, [12723] = 0x650C, [12724] = 0x650E, [12725] = 0x6584,
  [12726] = 0x65DE, [12727] = 0x65DD, [12728] = 0x66DE, [12729] = 0x6AE7,
  [12730] = 0x6AE0, [12731] = 0x6ACC, [12732] = 0x6AD1, [12733] = 0x6AD9,
  [12734] = 0x6ACB, [12735] = 0x6ADF, [12736] = 0x6ADC, [12737] = 0x6AD0,
  [12738] = 0x6AEB, [12739] = 0x6ACF, [12740] = 0x6ACD, [12741] = 0x6ADE,
  [12742] = 0x6B60, [12743] = 0x6BB0, [12744] = 0x6C0C, [12745] = 0x7019,
  [12746] = 0x7027, [12747] = 0x7020, [12748] = 0x7016, [12749] = 0x702B,
  [12750] = 0x7021, [12751] = 0x7022, [12752] = 0x7023, [12753] = 0x7029,
  [12754] = 0x7017, [12755] = 0x7024, [12756] = 0x701C, [12757] = 0x702A,
  [12758] = 0x720C, [12759] = 0x720A, [12760] = 0x7207, [12761] = 0x7202,
  [12762] = 0x7205, [12763] = 0x72A5, [12764] = 0x72A6, [12765] = 0x72A4,
  [12766] = 0x72A3, [12767] = 0x72A1, [12768] = 0x74CB, [12769] = 0x74C5,
  [12770] = 0x74B7, [12771] = 0x74C3, [12772] = 0x7516, [12773] = 0x7660,
  [12774] = 0x77C9, [12775] = 0x77CA, [12776] = 0x77C4, [12777] = 0x77F1,
  [12778] = 0x791D, [12779] = 0x791B, [12780] = 0x7921, [12781] = 0x791C,
  [12782] = 0x7917, [12783] = 0x791E, [12784] = 0x79B0, [12785] = 0x7A67,
  [12786] = 0x7A68, [12787] = 0x7C33, [12788] = 0x7C3C, [12789] = 0x7C39,
  [12790] = 0x7C2C, [12791] = 0x7C3B, [12792] = 0x7CEC, [12793] = 0x7CEA,
  [12794] = 0x7E76, [12795] = 0x7E75, [12796] = 0x7E78, [12797] = 0x7E70,
  [12798] = 0x7E77, [12799] = 0x7E6F, [12800] = 0x7E7A, [12801] = 0x7E72,
  [12802] = 0x7E74, [12803] = 0x7E68, [12804] = 0x7F4B, [12805] = 0x7F4A,
  [12806] = 0x7F83, [12807] = 0x7F86, [12808] = 0x7FB7, [12809] = 0x7FFD,
  [12810] = 0x7FFE, [12811] = 0x8078, [12812] = 0x81D7, [12813] = 0x81D5,
  [12814] = 0x8264, [12815] = 0x8261, [12816] = 0x8263, [12817] = 0x85EB,
  [12818] = 0x85F1, [12819] = 0x85ED, [12820] = 0x85D9, [12821] = 0x85E1,
  [12822] = 0x85E8, [12823] = 0x85DA, [12824] = 0x85D7, [12825] = 0x85EC,
  [12826] = 0x85F2, [12827] = 0x85F8, [12828] = 0x85D8, [12829] = 0x85DF,
  [12830] = 0x85E3, [12831] = 0x85DC, [12832] = 0x85D1, [12833] = 0x85F0,
  [12834] = 0x85E6, [12835] = 0x85EF, [12836] = 0x85DE, [12837] = 0x85E2,
  [12838] = 0x8800, [12839] = 0x87FA, [12840] = 0x8803, [12841] = 0x87F6,
  [12842] = 0x87F7, [12843] = 0x8809, [12844] = 0x880C, [12845] = 0x880B,
  [12846] = 0x8806, [12847] = 0x87FC, [12848] = 0x8808, [12849] = 0x87FF,
  [12850] = 0x880A, [12851] = 0x8802, [12852] = 0x8962, [12853] = 0x895A,
  [12854] = 0x895B, [12855] = 0x8957, [12856] = 0x8961, [12857] = 0x895C,
  [12858] = 0x8958, [12859] = 0x895D, [12860] = 0x8959, [12861] = 0x8988,
  [12862] = 0x89B7, [12863] = 0x89B6, [12864] = 0x89F6, [12865] = 0x8B50,
  [12866] = 0x8B48, [12867] = 0x8B4A, [12868] = 0x8B40, [12869] = 0x8B53,
  [12870] = 0x8B56, [12871] = 0x8B54, [12872] = 0x8B4B, [12873] = 0x8B55,
  [12874] = 0x8B51, [12875] = 0x8B42, [12876] = 0x8B52, [12877] = 0x8B57,
  [12878] = 0x8C43, [12879] = 0x8C77, [12880] = 0x8C76, [12881] = 0x8C9A,
  [12882] = 0x8D06, [12883] = 0x8D07, [12884] = 0x8D09, [12885] = 0x8DAC,
  [12886] = 0x8DAA, [12887] = 0x8DAD, [12888] = 0x8DAB, [12889] = 0x8E6D,
  [12890] = 0x8E78, [12891] = 0x8E73, [12892] = 0x8E6A, [12893] = 0x8E6F,
  [12894] = 0x8E7B, [12895] = 0x8EC2, [12896] = 0x8F52, [12897] = 0x8F51,
  [12898] = 0x8F4F, [12899] = 0x8F50, [12900] = 0x8F53, [12901] = 0x8FB4,
  [12902] = 0x9140, [12903] = 0x913F, [12904] = 0x91B0, [12905] = 0x91AD,
  [12906] = 0x93DE, [12907] = 0x93C7, [12908] = 0x93CF, [12909] = 0x93C2,
  [12910] = 0x93DA, [12911] = 0x93D0, [12912] = 0x93F9, [12913] = 0x93EC,
  [12914] = 0x93CC, [12915] = 0x93D9, [12916] = 0x93A9, [12917] = 0x93E6,
  [12918] = 0x93CA, [12919] = 0x93D4, [12920] = 0x93EE, [12921] = 0x93E3,
  [12922] = 0x93D5, [12923] = 0x93C4, [12924] = 0x93CE, [12925] = 0x93C0,
  [12926] = 0x93D2, [12927] = 0x93E7, [12928] = 0x957D, [12929] = 0x95DA,
  [12930] = 0x95DB, [12931] = 0x96E1, [12932] = 0x9729, [12933] = 0x972B,
  [12934] = 0x972C, [12935] = 0x9728, [12936] = 0x9726, [12937] = 0x97B3,
  [12938] = 0x97B7, [12939] = 0x97B6, [12940] = 0x97DD, [12941] = 0x97DE,
  [12942] = 0x97DF, [12943] = 0x985C, [12944] = 0x9859, [12945] = 0x985D,
  [12946] = 0x9857, [12947] = 0x98BF, [12948] = 0x98BD, [12949] = 0x98BB,
  [12950] = 0x98BE, [12951] = 0x9948, [12952] = 0x9947, [12953] = 0x9943,
  [12954] = 0x99A6, [12955] = 0x99A7, [12956] = 0x9A1A, [12957] = 0x9A15,
  [12958] = 0x9A25, [12959] = 0x9A1D, [12960] = 0x9A24, [12961] = 0x9A1B,
  [12962] = 0x9A22, [12963] = 0x9A20, [12964] = 0x9A27, [12965] = 0x9A23,
  [12966] = 0x9A1E, [12967] = 0x9A1C, [12968] = 0x9A14, [12969] = 0x9AC2,
  [12970] = 0x9B0B, [12971] = 0x9B0A, [12972] = 0x9B0E, [12973] = 0x9B0C,
  [12974] = 0x9B37, [12975] = 0x9BEA, [12976] = 0x9BEB, [12977] = 0x9BE0,
  [12978] = 0x9BDE, [12979] = 0x9BE4, [12980] = 0x9BE6, [12981] = 0x9BE2,
  [12982] = 0x9BF0, [12983] = 0x9BD4, [12984] = 0x9BD7, [12985] = 0x9BEC,
  [12986] = 0x9BDC, [12987] = 0x9BD9, [12988] = 0x9BE5, [12989] = 0x9BD5,
  [12990] = 0x9BE1, [12991] = 0x9BDA, [12992] = 0x9D77, [12993] = 0x9D81,
  [12994] = 0x9D8A, [12995] = 0x9D84, [12996] = 0x9D88, [12997] = 0x9D71,
  [12998] = 0x9D80, [12999] = 0x9D78, [13000] = 0x9D86, [13001] = 0x9D8B,
  [13002] = 0x9D8C, [13003] = 0x9D7D, [13004] = 0x9D6B, [13005] = 0x9D74,
  [13006] = 0x9D75, [13007] = 0x9D70, [13008] = 0x9D69, [13009] = 0x9D85,
  [13010] = 0x9D73, [13011] = 0x9D7B, [13012] = 0x9D82, [13013] = 0x9D6F,
  [13014] = 0x9D79, [13015] = 0x9D7F, [13016] = 0x9D87, [13017] = 0x9D68,
  [13018] = 0x9E94, [13019] = 0x9E91, [13020] = 0x9EC0, [13021] = 0x9EFC,
  [13022] = 0x9F2D, [13023] = 0x9F40, [13024] = 0x9F41, [13025] = 0x9F4D,
  [13026] = 0x9F56, [13027] = 0x9F57, [13028] = 0x9F58, [13029] = 0x5337,
  [13030] = 0x56B2, [13031] = 0x56B5, [13032] = 0x56B3, [13033] = 0x58E3,
  [13034] = 0x5B45, [13035] = 0x5DC6, [13036] = 0x5DC7, [13037] = 0x5EEE,
  [13038] = 0x5EEF, [13039] = 0x5FC0, [13040] = 0x5FC1, [13041] = 0x61F9,
  [13042] = 0x6517, [13043] = 0x6516, [13044] = 0x6515, [13045] = 0x6513,
  [13046] = 0x65DF, [13047] = 0x66E8, [13048] = 0x66E3, [13049] = 0x66E4,
  [13050] = 0x6AF3, [13051] = 0x6AF0, [13052] = 0x6AEA, [13053] = 0x6AE8,
  [13054] = 0x6AF9, [13055] = 0x6AF1, [13056] = 0x6AEE, [13057] = 0x6AEF,
  [13058] = 0x703C, [13059] = 0x7035, [13060] = 0x702F, [13061] = 0x7037,
  [13062] = 0x7034, [13063] = 0x7031, [13064] = 0x7042, [13065] = 0x7038,
  [13066] = 0x703F, [13067] = 0x703A, [13068] = 0x7039, [13069] = 0x7040,
  [13070] = 0x703B, [13071] = 0x7033, [13072] = 0x7041, [13073] = 0x7213,
  [13074] = 0x7214, [13075] = 0x72A8, [13076] = 0x737D, [13077] = 0x737C,
  [13078] = 0x74BA, [13079] = 0x76AB, [13080] = 0x76AA, [13081] = 0x76BE,
  [13082] = 0x76ED, [13083] = 0x77CC, [13084] = 0x77CE, [13085] = 0x77CF,
  [13086] = 0x77CD, [13087] = 0x77F2, [13088] = 0x7925, [13089] = 0x7923,
  [13090] = 0x7927, [13091] = 0x7928, [13092] = 0x7924, [13093] = 0x7929,
  [13094] = 0x79B2, [13095] = 0x7A6E, [13096] = 0x7A6C, [13097] = 0x7A6D,
  [13098] = 0x7AF7, [13099] = 0x7C49, [13100] = 0x7C48, [13101] = 0x7C4A,
  [13102] = 0x7C47, [13103] = 0x7C45, [13104] = 0x7CEE, [13105] = 0x7E7B,
  [13106] = 0x7E7E, [13107] = 0x7E81, [13108] = 0x7E80, [13109] = 0x7FBA,
  [13110] = 0x7FFF, [13111] = 0x8079, [13112] = 0x81DB, [13113] = 0x81D9,
  [13114] = 0x820B, [13115] = 0x8268, [13116] = 0x8269, [13117] = 0x8622,
  [13118] = 0x85FF, [13119] = 0x8601, [13120] = 0x85FE, [13121] = 0x861B,
  [13122] = 0x8600, [13123] = 0x85F6, [13124] = 0x8604, [13125] = 0x8609,
  [13126] = 0x8605, [13127] = 0x860C, [13128] = 0x85FD, [13129] = 0x8819,
  [13130] = 0x8810, [13131] = 0x8811, [13132] = 0x8817, [13133] = 0x8813,
  [13134] = 0x8816, [13135] = 0x8963, [13136] = 0x8966, [13137] = 0x89B9,
  [13138] = 0x89F7, [13139] = 0x8B60, [13140] = 0x8B6A, [13141] = 0x8B5D,
  [13142] = 0x8B68, [13143] = 0x8B63, [13144] = 0x8B65, [13145] = 0x8B67,
  [13146] = 0x8B6D, [13147] = 0x8DAE, [13148] = 0x8E86, [13149] = 0x8E88,
  [13150] = 0x8E84, [13151] = 0x8F59, [13152] = 0x8F56, [13153] = 0x8F57,
  [13154] = 0x8F55, [13155] = 0x8F58, [13156] = 0x8F5A, [13157] = 0x908D,
  [13158] = 0x9143, [13159] = 0x9141, [13160] = 0x91B7, [13161] = 0x91B5,
  [13162] = 0x91B2, [13163] = 0x91B3, [13164] = 0x940B, [13165] = 0x9413,
  [13166] = 0x93FB, [13167] = 0x9420, [13168] = 0x940F, [13169] = 0x9414,
  [13170] = 0x93FE, [13171] = 0x9415, [13172] = 0x9410, [13173] = 0x9428,
  [13174] = 0x9419, [13175] = 0x940D, [13176] = 0x93F5, [13177] = 0x9400,
  [13178] = 0x93F7, [13179] = 0x9407, [13180] = 0x940E, [13181] = 0x9416,
  [13182] = 0x9412, [13183] = 0x93FA, [13184] = 0x9409, [13185] = 0x93F8,
  [13186] = 0x940A, [13187] = 0x93FF, [13188] = 0x93FC, [13189] = 0x940C,
  [13190] = 0x93F6, [13191] = 0x9411, [13192] = 0x9406, [13193] = 0x95DE,
  [13194] = 0x95E0, [13195] = 0x95DF, [13196] = 0x972E, [13197] = 0x972F,
  [13198] = 0x97B9, [13199] = 0x97BB, [13200] = 0x97FD, [13201] = 0x97FE,
  [13202] = 0x9860, [13203] = 0x9862, [13204] = 0x9863, [13205] = 0x985F,
  [13206] = 0x98C1, [13207] = 0x98C2, [13208] = 0x9950, [13209] = 0x994E,
  [13210] = 0x9959, [13211] = 0x994C, [13212] = 0x994B, [13213] = 0x9953,
  [13214] = 0x9A32, [13215] = 0x9A34, [13216] = 0x9A31, [13217] = 0x9A2C,
  [13218] = 0x9A2A, [13219] = 0x9A36, [13220] = 0x9A29, [13221] = 0x9A2E,
  [13222] = 0x9A38, [13223] = 0x9A2D, [13224] = 0x9AC7, [13225] = 0x9ACA,
  [13226] = 0x9AC6, [13227] = 0x9B10, [13228] = 0x9B12, [13229] = 0x9B11,
  [13230] = 0x9C0B, [13231] = 0x9C08, [13232] = 0x9BF7, [13233] = 0x9C05,
  [13234] = 0x9C12, [13235] = 0x9BF8, [13236] = 0x9C40, [13237] = 0x9C07,
  [13238] = 0x9C0E, [13239] = 0x9C06, [13240] = 0x9C17, [13241] = 0x9C14,
  [13242] = 0x9C09, [13243] = 0x9D9F, [13244] = 0x9D99, [13245] = 0x9DA4,
  [13246] = 0x9D9D, [13247] = 0x9D92, [13248] = 0x9D98, [13249] = 0x9D90,
  [13250] = 0x9D9B, [13251] = 0x9DA0, [13252] = 0x9D94, [13253] = 0x9D9C,
  [13254] = 0x9DAA, [13255] = 0x9D97, [13256] = 0x9DA1, [13257] = 0x9D9A,
  [13258] = 0x9DA2, [13259] = 0x9DA8, [13260] = 0x9D9E, [13261] = 0x9DA3,
  [13262] = 0x9DBF, [13263] = 0x9DA9, [13264] = 0x9D96, [13265] = 0x9DA6,
  [13266] = 0x9DA7, [13267] = 0x9E99, [13268] = 0x9E9B, [13269] = 0x9E9A,
  [13270] = 0x9EE5, [13271] = 0x9EE4, [13272] = 0x9EE7, [13273] = 0x9EE6,
  [13274] = 0x9F30, [13275] = 0x9F2E, [13276] = 0x9F5B, [13277] = 0x9F60,
  [13278] = 0x9F5E, [13279] = 0x9F5D, [13280] = 0x9F59, [13281] = 0x9F91,
  [13282] = 0x513A, [13283] = 0x5139, [13284] = 0x5298, [13285] = 0x5297,
  [13286] = 0x56C3, [13287] = 0x56BD, [13288] = 0x56BE, [13289] = 0x5B48,
  [13290] = 0x5B47, [13291] = 0x5DCB, [13292] = 0x5DCF, [13293] = 0x5EF1,
  [13294] = 0x61FD, [13295] = 0x651B, [13296] = 0x6B02, [13297] = 0x6AFC,
  [13298] = 0x6B03, [13299] = 0x6AF8, [13300] = 0x6B00, [13301] = 0x7043,
  [13302] = 0x7044, [13303] = 0x704A, [13304] = 0x7048, [13305] = 0x7049,
  [13306] = 0x7045, [13307] = 0x7046, [13308] = 0x721D, [13309] = 0x721A,
  [13310] = 0x7219, [13311] = 0x737E, [13312] = 0x7517, [13313] = 0x766A,
  [13314] = 0x77D0, [13315] = 0x792D, [13316] = 0x7931, [13317] = 0x792F,
  [13318] = 0x7C54, [13319] = 0x7C53, [13320] = 0x7CF2, [13321] = 0x7E8A,
  [13322] = 0x7E87, [13323] = 0x7E88, [13324] = 0x7E8B, [13325] = 0x7E86,
  [13326] = 0x7E8D, [13327] = 0x7F4D, [13328] = 0x7FBB, [13329] = 0x8030,
  [13330] = 0x81DD, [13331] = 0x8618, [13332] = 0x862A, [13333] = 0x8626,
  [13334] = 0x861F, [13335] = 0x8623, [13336] = 0x861C, [13337] = 0x8619,
  [13338] = 0x8627, [13339] = 0x862E, [13340] = 0x8621, [13341] = 0x8620,
  [13342] = 0x8629, [13343] = 0x861E, [13344] = 0x8625, [13345] = 0x8829,
  [13346] = 0x881D, [13347] = 0x881B, [13348] = 0x8820, [13349] = 0x8824,
  [13350] = 0x881C, [13351] = 0x882B, [13352] = 0x884A, [13353] = 0x896D,
  [13354] = 0x8969, [13355] = 0x896E, [13356] = 0x896B, [13357] = 0x89FA,
  [13358] = 0x8B79, [13359] = 0x8B78, [13360] = 0x8B45, [13361] = 0x8B7A,
  [13362] = 0x8B7B, [13363] = 0x8D10, [13364] = 0x8D14, [13365] = 0x8DAF,
  [13366] = 0x8E8E, [13367] = 0x8E8C, [13368] = 0x8F5E, [13369] = 0x8F5B,
  [13370] = 0x8F5D, [13371] = 0x9146, [13372] = 0x9144, [13373] = 0x9145,
  [13374] = 0x91B9, [13375] = 0x943F, [13376] = 0x943B, [13377] = 0x9436,
  [13378] = 0x9429, [13379] = 0x943D, [13380] = 0x943C, [13381] = 0x9430,
  [13382] = 0x9439, [13383] = 0x942A, [13384] = 0x9437, [13385] = 0x942C,
  [13386] = 0x9440, [13387] = 0x9431, [13388] = 0x95E5, [13389] = 0x95E4,
  [13390] = 0x95E3, [13391] = 0x9735, [13392] = 0x973A, [13393] = 0x97BF,
  [13394] = 0x97E1, [13395] = 0x9864, [13396] = 0x98C9, [13397] = 0x98C6,
  [13398] = 0x98C0, [13399] = 0x9958, [13400] = 0x9956, [13401] = 0x9A39,
  [13402] = 0x9A3D, [13403] = 0x9A46, [13404] = 0x9A44, [13405] = 0x9A42,
  [13406] = 0x9A41, [13407] = 0x9A3A, [13408] = 0x9A3F, [13409] = 0x9ACD,
  [13410] = 0x9B15, [13411] = 0x9B17, [13412] = 0x9B18, [13413] = 0x9B16,
  [13414] = 0x9B3A, [13415] = 0x9B52, [13416] = 0x9C2B, [13417] = 0x9C1D,
  [13418] = 0x9C1C, [13419] = 0x9C2C, [13420] = 0x9C23, [13421] = 0x9C28,
  [13422] = 0x9C29, [13423] = 0x9C24, [13424] = 0x9C21, [13425] = 0x9DB7,
  [13426] = 0x9DB6, [13427] = 0x9DBC, [13428] = 0x9DC1, [13429] = 0x9DC7,
  [13430] = 0x9DCA, [13431] = 0x9DCF, [13432] = 0x9DBE, [13433] = 0x9DC5,
  [13434] = 0x9DC3, [13435] = 0x9DBB, [13436] = 0x9DB5, [13437] = 0x9DCE,
  [13438] = 0x9DB9, [13439] = 0x9DBA, [13440] = 0x9DAC, [13441] = 0x9DC8,
  [13442] = 0x9DB1, [13443] = 0x9DAD, [13444] = 0x9DCC, [13445] = 0x9DB3,
  [13446] = 0x9DCD, [13447] = 0x9DB2, [13448] = 0x9E7A, [13449] = 0x9E9C,
  [13450] = 0x9EEB, [13451] = 0x9EEE, [13452] = 0x9EED, [13453] = 0x9F1B,
  [13454] = 0x9F18, [13455] = 0x9F1A, [13456] = 0x9F31, [13457] = 0x9F4E,
  [13458] = 0x9F65, [13459] = 0x9F64, [13460] = 0x9F92, [13461] = 0x4EB9,
  [13462] = 0x56C6, [13463] = 0x56C5, [13464] = 0x56CB, [13465] = 0x5971,
  [13466] = 0x5B4B, [13467] = 0x5B4C, [13468] = 0x5DD5, [13469] = 0x5DD1,
  [13470] = 0x5EF2, [13471] = 0x6521, [13472] = 0x6520, [13473] = 0x6526,
  [13474] = 0x6522, [13475] = 0x6B0B, [13476] = 0x6B08, [13477] = 0x6B09,
  [13478] = 0x6C0D, [13479] = 0x7055, [13480] = 0x7056, [13481] = 0x7057,
  [13482] = 0x7052, [13483] = 0x721E, [13484] = 0x721F, [13485] = 0x72A9,
  [13486] = 0x737F, [13487] = 0x74D8, [13488] = 0x74D5, [13489] = 0x74D9,
  [13490] = 0x74D7, [13491] = 0x766D, [13492] = 0x76AD, [13493] = 0x7935,
  [13494] = 0x79B4, [13495] = 0x7A70, [13496] = 0x7A71, [13497] = 0x7C57,
  [13498] = 0x7C5C, [13499] = 0x7C59, [13500] = 0x7C5B, [13501] = 0x7C5A,
  [13502] = 0x7CF4, [13503] = 0x7CF1, [13504] = 0x7E91, [13505] = 0x7F4F,
  [13506] = 0x7F87, [13507] = 0x81DE, [13508] = 0x826B, [13509] = 0x8634,
  [13510] = 0x8635, [13511] = 0x8633, [13512] = 0x862C, [13513] = 0x8632,
  [13514] = 0x8636, [13515] = 0x882C, [13516] = 0x8828, [13517] = 0x8826,
  [13518] = 0x882A, [13519] = 0x8825, [13520] = 0x8971, [13521] = 0x89BF,
  [13522] = 0x89BE, [13523] = 0x89FB, [13524] = 0x8B7E, [13525] = 0x8B84,
  [13526] = 0x8B82, [13527] = 0x8B86, [13528] = 0x8B85, [13529] = 0x8B7F,
  [13530] = 0x8D15, [13531] = 0x8E95, [13532] = 0x8E94, [13533] = 0x8E9A,
  [13534] = 0x8E92, [13535] = 0x8E90, [13536] = 0x8E96, [13537] = 0x8E97,
  [13538] = 0x8F60, [13539] = 0x8F62, [13540] = 0x9147, [13541] = 0x944C,
  [13542] = 0x9450, [13543] = 0x944A, [13544] = 0x944B, [13545] = 0x944F,
  [13546] = 0x9447, [13547] = 0x9445, [13548] = 0x9448, [13549] = 0x9449,
  [13550] = 0x9446, [13551] = 0x973F, [13552] = 0x97E3, [13553] = 0x986A,
  [13554] = 0x9869, [13555] = 0x98CB, [13556] = 0x9954, [13557] = 0x995B,
  [13558] = 0x9A4E, [13559] = 0x9A53, [13560] = 0x9A54, [13561] = 0x9A4C,
  [13562] = 0x9A4F, [13563] = 0x9A48, [13564] = 0x9A4A, [13565] = 0x9A49,
  [13566] = 0x9A52, [13567] = 0x9A50, [13568] = 0x9AD0, [13569] = 0x9B19,
  [13570] = 0x9B2B, [13571] = 0x9B3B, [13572] = 0x9B56, [13573] = 0x9B55,
  [13574] = 0x9C46, [13575] = 0x9C48, [13576] = 0x9C3F, [13577] = 0x9C44,
  [13578] = 0x9C39, [13579] = 0x9C33, [13580] = 0x9C41, [13581] = 0x9C3C,
  [13582] = 0x9C37, [13583] = 0x9C34, [13584] = 0x9C32, [13585] = 0x9C3D,
  [13586] = 0x9C36, [13587] = 0x9DDB, [13588] = 0x9DD2, [13589] = 0x9DDE,
  [13590] = 0x9DDA, [13591] = 0x9DCB, [13592] = 0x9DD0, [13593] = 0x9DDC,
  [13594] = 0x9DD1, [13595] = 0x9DDF, [13596] = 0x9DE9, [13597] = 0x9DD9,
  [13598] = 0x9DD8, [13599] = 0x9DD6, [13600] = 0x9DF5, [13601] = 0x9DD5,
  [13602] = 0x9DDD, [13603] = 0x9EB6, [13604] = 0x9EF0, [13605] = 0x9F35,
  [13606] = 0x9F33, [13607] = 0x9F32, [13608] = 0x9F42, [13609] = 0x9F6B,
  [13610] = 0x9F95, [13611] = 0x9FA2, [13612] = 0x513D, [13613] = 0x5299,
  [13614] = 0x58E8, [13615] = 0x58E7, [13616] = 0x5972, [13617] = 0x5B4D,
  [13618] = 0x5DD8, [13619] = 0x882F, [13620] = 0x5F4F, [13621] = 0x6201,
  [13622] = 0x6203, [13623] = 0x6204, [13624] = 0x6529, [13625] = 0x6525,
  [13626] = 0x6596, [13627] = 0x66EB, [13628] = 0x6B11, [13629] = 0x6B12,
  [13630] = 0x6B0F, [13631] = 0x6BCA, [13632] = 0x705B, [13633] = 0x705A,
  [13634] = 0x7222, [13635] = 0x7382, [13636] = 0x7381, [13637] = 0x7383,
  [13638] = 0x7670, [13639] = 0x77D4, [13640] = 0x7C67, [13641] = 0x7C66,
  [13642] = 0x7E95, [13643] = 0x826C, [13644] = 0x863A, [13645] = 0x8640,
  [13646] = 0x8639, [13647] = 0x863C, [13648] = 0x8631, [13649] = 0x863B,
  [13650] = 0x863E, [13651] = 0x8830, [13652] = 0x8832, [13653] = 0x882E,
  [13654] = 0x8833, [13655] = 0x8976, [13656] = 0x8974, [13657] = 0x8973,
  [13658] = 0x89FE, [13659] = 0x8B8C, [13660] = 0x8B8E, [13661] = 0x8B8B,
  [13662] = 0x8B88, [13663] = 0x8C45, [13664] = 0x8D19, [13665] = 0x8E98,
  [13666] = 0x8F64, [13667] = 0x8F63, [13668] = 0x91BC, [13669] = 0x9462,
  [13670] = 0x9455, [13671] = 0x945D, [13672] = 0x9457, [13673] = 0x945E,
  [13674] = 0x97C4, [13675] = 0x97C5, [13676] = 0x9800, [13677] = 0x9A56,
  [13678] = 0x9A59, [13679] = 0x9B1E, [13680] = 0x9B1F, [13681] = 0x9B20,
  [13682] = 0x9C52, [13683] = 0x9C58, [13684] = 0x9C50, [13685] = 0x9C4A,
  [13686] = 0x9C4D, [13687] = 0x9C4B, [13688] = 0x9C55, [13689] = 0x9C59,
  [13690] = 0x9C4C, [13691] = 0x9C4E, [13692] = 0x9DFB, [13693] = 0x9DF7,
  [13694] = 0x9DEF, [13695] = 0x9DE3, [13696] = 0x9DEB, [13697] = 0x9DF8,
  [13698] = 0x9DE4, [13699] = 0x9DF6, [13700] = 0x9DE1, [13701] = 0x9DEE,
  [13702] = 0x9DE6, [13703] = 0x9DF2, [13704] = 0x9DF0, [13705] = 0x9DE2,
  [13706] = 0x9DEC, [13707] = 0x9DF4, [13708] = 0x9DF3, [13709] = 0x9DE8,
  [13710] = 0x9DED, [13711] = 0x9EC2, [13712] = 0x9ED0, [13713] = 0x9EF2,
  [13714] = 0x9EF3, [13715] = 0x9F06, [13716] = 0x9F1C, [13717] = 0x9F38,
  [13718] = 0x9F37, [13719] = 0x9F36, [13720] = 0x9F43, [13721] = 0x9F4F,
  [13722] = 0x9F71, [13723] = 0x9F70, [13724] = 0x9F6E, [13725] = 0x9F6F,
  [13726] = 0x56D3, [13727] = 0x56CD, [13728] = 0x5B4E, [13729] = 0x5C6D,
  [13730] = 0x652D, [13731] = 0x66ED, [13732] = 0x66EE, [13733] = 0x6B13,
  [13734] = 0x705F, [13735] = 0x7061, [13736] = 0x705D, [13737] = 0x7060,
  [13738] = 0x7223, [13739] = 0x74DB, [13740] = 0x74E5, [13741] = 0x77D5,
  [13742] = 0x7938, [13743] = 0x79B7, [13744] = 0x79B6, [13745] = 0x7C6A,
  [13746] = 0x7E97, [13747] = 0x7F89, [13748] = 0x826D, [13749] = 0x8643,
  [13750] = 0x8838, [13751] = 0x8837, [13752] = 0x8835, [13753] = 0x884B,
  [13754] = 0x8B94, [13755] = 0x8B95, [13756] = 0x8E9E, [13757] = 0x8E9F,
  [13758] = 0x8EA0, [13759] = 0x8E9D, [13760] = 0x91BE, [13761] = 0x91BD,
  [13762] = 0x91C2, [13763] = 0x946B, [13764] = 0x9468, [13765] = 0x9469,
  [13766] = 0x96E5, [13767] = 0x9746, [13768] = 0x9743, [13769] = 0x9747,
  [13770] = 0x97C7, [13771] = 0x97E5, [13772] = 0x9A5E, [13773] = 0x9AD5,
  [13774] = 0x9B59, [13775] = 0x9C63, [13776] = 0x9C67, [13777] = 0x9C66,
  [13778] = 0x9C62, [13779] = 0x9C5E, [13780] = 0x9C60, [13781] = 0x9E02,
  [13782] = 0x9DFE, [13783] = 0x9E07, [13784] = 0x9E03, [13785] = 0x9E06,
  [13786] = 0x9E05, [13787] = 0x9E00, [13788] = 0x9E01, [13789] = 0x9E09,
  [13790] = 0x9DFF, [13791] = 0x9DFD, [13792] = 0x9E04, [13793] = 0x9EA0,
  [13794] = 0x9F1E, [13795] = 0x9F46, [13796] = 0x9F74, [13797] = 0x9F75,
  [13798] = 0x9F76, [13799] = 0x56D4, [13800] = 0x652E, [13801] = 0x65B8,
  [13802] = 0x6B18, [13803] = 0x6B19, [13804] = 0x6B17, [13805] = 0x6B1A,
  [13806] = 0x7062, [13807] = 0x7226, [13808] = 0x72AA, [13809] = 0x77D8,
  [13810] = 0x77D9, [13811] = 0x7939, [13812] = 0x7C69, [13813] = 0x7C6B,
  [13814] = 0x7CF6, [13815] = 0x7E9A, [13816] = 0x7E98, [13817] = 0x7E9B,
  [13818] = 0x7E99, [13819] = 0x81E0, [13820] = 0x81E1, [13821] = 0x8646,
  [13822] = 0x8647, [13823] = 0x8648, [13824] = 0x8979, [13825] = 0x897A,
  [13826] = 0x897C, [13827] = 0x897B, [13828] = 0x89FF, [13829] = 0x8B98,
  [13830] = 0x8B99, [13831] = 0x8EA5, [13832] = 0x8EA4, [13833] = 0x8EA3,
  [13834] = 0x946E, [13835] = 0x946D, [13836] = 0x946F, [13837] = 0x9471,
  [13838] = 0x9473, [13839] = 0x9749, [13840] = 0x9872, [13841] = 0x995F,
  [13842] = 0x9C68, [13843] = 0x9C6E, [13844] = 0x9C6D, [13845] = 0x9E0B,
  [13846] = 0x9E0D, [13847] = 0x9E10, [13848] = 0x9E0F, [13849] = 0x9E12,
  [13850] = 0x9E11, [13851] = 0x9EA1, [13852] = 0x9EF5, [13853] = 0x9F09,
  [13854] = 0x9F47, [13855] = 0x9F78, [13856] = 0x9F7B, [13857] = 0x9F7A,
  [13858] = 0x9F79, [13859] = 0x571E, [13860] = 0x7066, [13861] = 0x7C6F,
  [13862] = 0x883C, [13863] = 0x8DB2, [13864] = 0x8EA6, [13865] = 0x91C3,
  [13866] = 0x9474, [13867] = 0x9478, [13868] = 0x9476, [13869] = 0x9475,
  [13870] = 0x9A60, [13871] = 0x9C74, [13872] = 0x9C73, [13873] = 0x9C71,
  [13874] = 0x9C75, [13875] = 0x9E14, [13876] = 0x9E13, [13877] = 0x9EF6,
  [13878] = 0x9F0A, [13879] = 0x9FA4, [13880] = 0x7068, [13881] = 0x7065,
  [13882] = 0x7CF7, [13883] = 0x866A, [13884] = 0x883E, [13885] = 0x883D,
  [13886] = 0x883F, [13887] = 0x8B9E, [13888] = 0x8C9C, [13889] = 0x8EA9,
  [13890] = 0x8EC9, [13891] = 0x974B, [13892] = 0x9873, [13893] = 0x9874,
  [13894] = 0x98CC, [13895] = 0x9961, [13896] = 0x99AB, [13897] = 0x9A64,
  [13898] = 0x9A66, [13899] = 0x9A67, [13900] = 0x9B24, [13901] = 0x9E15,
  [13902] = 0x9E17, [13903] = 0x9F48, [13904] = 0x6207, [13905] = 0x6B1E,
  [13906] = 0x7227, [13907] = 0x864C, [13908] = 0x8EA8, [13909] = 0x9482,
  [13910] = 0x9480, [13911] = 0x9481, [13912] = 0x9A69, [13913] = 0x9A68,
  [13914] = 0x9B2E, [13915] = 0x9E19, [13916] = 0x7229, [13917] = 0x864B,
  [13918] = 0x8B9F, [13919] = 0x9483, [13920] = 0x9C79, [13921] = 0x9EB7,
  [13922] = 0x7675, [13923] = 0x9A6B, [13924] = 0x9C7A, [13925] = 0x9E1D,
  [13926] = 0x7069, [13927] = 0x706A, [13928] = 0x9EA4, [13929] = 0x9F7E,
  [13930] = 0x9F49, [13931] = 0x9F98, [13932] = 0x7881, [13933] = 0x92B9,
  [13934] = 0x88CF, [13935] = 0x58BB, [13936] = 0x6052, [13937] = 0x7CA7,
  [13938] = 0x5AFA, [13939] = 0x2554, [13940] = 0x2566, [13941] = 0x2557,
  [13942] = 0x2560, [13943] = 0x256C, [13944] = 0x2563, [13945] = 0x255A,
  [13946] = 0x2569, [13947] = 0x255D, [13948] = 0x2552, [13949] = 0x2564,
  [13950] = 0x2555, [13951] = 0x255E, [13952] = 0x256A, [13953] = 0x2561,
  [13954] = 0x2558, [13955] = 0x2567, [13956] = 0x255B, [13957] = 0x2553,
  [13958] = 0x2565, [13959] = 0x2556, [13960] = 0x255F, [13961] = 0x256B,
  [13962] = 0x2562, [13963] = 0x2559, [13964] = 0x2568, [13965] = 0x255C,
  [13966] = 0x2551, [13967] = 0x2550, [13968] = 0x256D, [13969] = 0x256E,
  [13970] = 0x2570, [13971] = 0x256F, [13972] = 0x2593,
};


/* The mapping from Unicode to Big5 is not easy since we have several
   separate blocks.  First a block of characters with the second byte
   being zero. */

/* Table starting at U+0000, containing all U+00XX */
static const char from_ucs4_tab1[256][2] =
{
  /* 0x00 */ "\x00\x00", "\x01\x00", "\x02\x00", "\x03\x00", "\x04\x00",
  /* 0x05 */ "\x05\x00", "\x06\x00", "\x07\x00", "\x08\x00", "\x09\x00",
  /* 0x0a */ "\x0a\x00", "\x0b\x00", "\x0c\x00", "\x0d\x00", "\x0e\x00",
  /* 0x0f */ "\x0f\x00", "\x10\x00", "\x11\x00", "\x12\x00", "\x13\x00",
  /* 0x14 */ "\x14\x00", "\x15\x00", "\x16\x00", "\x17\x00", "\x18\x00",
  /* 0x19 */ "\x19\x00", "\x1a\x00", "\x1b\x00", "\x1c\x00", "\x1d\x00",
  /* 0x1e */ "\x1e\x00", "\x1f\x00", "\x20\x00", "\x21\x00", "\x22\x00",
  /* 0x23 */ "\x23\x00", "\x24\x00", "\x25\x00", "\x26\x00", "\x27\x00",
  /* 0x28 */ "\x28\x00", "\x29\x00", "\x2a\x00", "\x2b\x00", "\x2c\x00",
  /* 0x2d */ "\x2d\x00", "\x2e\x00", "\x2f\x00", "\x30\x00", "\x31\x00",
  /* 0x32 */ "\x32\x00", "\x33\x00", "\x34\x00", "\x35\x00", "\x36\x00",
  /* 0x37 */ "\x37\x00", "\x38\x00", "\x39\x00", "\x3a\x00", "\x3b\x00",
  /* 0x3c */ "\x3c\x00", "\x3d\x00", "\x3e\x00", "\x3f\x00", "\x40\x00",
  /* 0x41 */ "\x41\x00", "\x42\x00", "\x43\x00", "\x44\x00", "\x45\x00",
  /* 0x46 */ "\x46\x00", "\x47\x00", "\x48\x00", "\x49\x00", "\x4a\x00",
  /* 0x4b */ "\x4b\x00", "\x4c\x00", "\x4d\x00", "\x4e\x00", "\x4f\x00",
  /* 0x50 */ "\x50\x00", "\x51\x00", "\x52\x00", "\x53\x00", "\x54\x00",
  /* 0x55 */ "\x55\x00", "\x56\x00", "\x57\x00", "\x58\x00", "\x59\x00",
  /* 0x5a */ "\x5a\x00", "\x5b\x00", "\x5c\x00", "\x5d\x00", "\x5e\x00",
  /* 0x5f */ "\x5f\x00", "\x60\x00", "\x61\x00", "\x62\x00", "\x63\x00",
  /* 0x64 */ "\x64\x00", "\x65\x00", "\x66\x00", "\x67\x00", "\x68\x00",
  /* 0x69 */ "\x69\x00", "\x6a\x00", "\x6b\x00", "\x6c\x00", "\x6d\x00",
  /* 0x6e */ "\x6e\x00", "\x6f\x00", "\x70\x00", "\x71\x00", "\x72\x00",
  /* 0x73 */ "\x73\x00", "\x74\x00", "\x75\x00", "\x76\x00", "\x77\x00",
  /* 0x78 */ "\x78\x00", "\x79\x00", "\x7a\x00", "\x7b\x00", "\x7c\x00",
  /* 0x7d */ "\x7d\x00", "\x7e\x00", "\x7f\x00", "\x80\x00", "\x00\x00",
  /* 0x82 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0x87 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0x8c */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0x91 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0x96 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0x9b */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xa0 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xa5 */ "\x00\x00", "\x00\x00", "\xa1\xb1", "\x00\x00", "\x00\x00",
  /* 0xaa */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xaf */ "\xa1\xc2", "\xa2\x58", "\xa1\xd3", "\x00\x00", "\x00\x00",
  /* 0xb4 */ "\x00\x00", "\x00\x00", "\x00\x00", "\xa1\x50", "\x00\x00",
  /* 0xb9 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xbe */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xc3 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xc8 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xcd */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xd2 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xd7 */ "\xa1\xd1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xdc */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xe1 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xe6 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xeb */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xf0 */ "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  /* 0xf5 */ "\x00\x00", "\x00\x00", "\xa1\xd2"
};

/* Tables for UCS to Big5 conversion. These are generated from the BIG5
   charmap, using the following commands:

---------------------------------------------------------------------
rm -f segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U02" | sort | \
	perl segB2.pl 2 0x02C7 0x02D9 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U03" | sort | \
	perl segB2.pl 3 0x0391 0x03C9 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U20[1-3]" | sort | \
	perl segB2.pl 4 0x2013 0x203B >> segB2

( Separate U20AC, U2103, U2105, U2109 )

egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U216" | sort | \
	perl segB2.pl 5 0x2160 0x2169 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U219" | sort | \
	perl segB2.pl 6 0x2190 0x2199 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U22[1-6]" | sort | \
	perl segB2.pl 7 0x2215 0x2267 >> segB2

( Separate U2295, U2299, U22A5, U22BF )

egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U2[56]" | sort | \
	perl segB2.pl 8 0x2500 0x2642 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U30" | sort | \
	perl segB2.pl 9 0x3000 0x3029 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U31" | sort | \
	perl segB2.pl 10 0x3105 0x3129 >> segB2

( Separate U32A3 )

egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U33" | sort | \
	perl segB2.pl 11 0x338E 0x33D5 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<U[4-9]" | sort | \
	perl segB2.pl 12 0x4E00 0x9FA4 >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<UF[678]" | sort | \
	perl segB2.pl 13 0xF6B1 0xF848 >> segB2

( Separate UFA0C, UFA0D )

egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<UFE" | sort | \
	perl segB2.pl 14 0xFE30 0xFE6B >> segB2
egrep "/x../x.." BIG5 | egrep -v "^%" | egrep "^<UFF" | sort | \
	perl segB2.pl 15 0xFF01 0xFFE5 >> segB2
---------------------------------------------------------------------

   where the segB2.pl script is:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
$tnum   = $ARGV[0];
$ubegin = $ARGV[1];
$uend   = $ARGV[2];

printf("|* Table of range U+%04X - U+%04X. *|\n", hex($ubegin), hex($uend));
printf("static const char from_ucs4_tab%d[][2] =\n{", $tnum);

$n=0;
while (<STDIN>) {
     local($ucs4, $big5, %rest) = split;
     $ucs4 =~ s/<U(....)>/$1/;
     $big5 =~ s/\/x//g;
     local($u)=hex($ucs4);
     local($b)=hex($big5);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - hex($ubegin),
             $b < 0x100 ? $b : int($b/256), $b < 0x100 ? 0 : $b&255);
}
printf ("\n};\n\n");
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/

/* Table of range U+02C7 - U+02D9. */
static const char from_ucs4_tab2[][2] =
{
  [0x0000] = "\xa3\xbe", [0x0002] = "\xa3\xbc", [0x0003] = "\xa3\xbd",
  [0x0004] = "\xa3\xbf", [0x0006] = "\xa1\xc5", [0x0012] = "\xa3\xbb",
};

/* Table of range U+0391 - U+03C9. */
static const char from_ucs4_tab3[][2] =
{
  [0x0000] = "\xa3\x44", [0x0001] = "\xa3\x45", [0x0002] = "\xa3\x46",
  [0x0003] = "\xa3\x47", [0x0004] = "\xa3\x48", [0x0005] = "\xa3\x49",
  [0x0006] = "\xa3\x4a", [0x0007] = "\xa3\x4b", [0x0008] = "\xa3\x4c",
  [0x0009] = "\xa3\x4d", [0x000a] = "\xa3\x4e", [0x000b] = "\xa3\x4f",
  [0x000c] = "\xa3\x50", [0x000d] = "\xa3\x51", [0x000e] = "\xa3\x52",
  [0x000f] = "\xa3\x53", [0x0010] = "\xa3\x54", [0x0012] = "\xa3\x55",
  [0x0013] = "\xa3\x56", [0x0014] = "\xa3\x57", [0x0015] = "\xa3\x58",
  [0x0016] = "\xa3\x59", [0x0017] = "\xa3\x5a", [0x0018] = "\xa3\x5b",
  [0x0020] = "\xa3\x5c", [0x0021] = "\xa3\x5d", [0x0022] = "\xa3\x5e",
  [0x0023] = "\xa3\x5f", [0x0024] = "\xa3\x60", [0x0025] = "\xa3\x61",
  [0x0026] = "\xa3\x62", [0x0027] = "\xa3\x63", [0x0028] = "\xa3\x64",
  [0x0029] = "\xa3\x65", [0x002a] = "\xa3\x66", [0x002b] = "\xa3\x67",
  [0x002c] = "\xa3\x68", [0x002d] = "\xa3\x69", [0x002e] = "\xa3\x6a",
  [0x002f] = "\xa3\x6b", [0x0030] = "\xa3\x6c", [0x0032] = "\xa3\x6d",
  [0x0033] = "\xa3\x6e", [0x0034] = "\xa3\x6f", [0x0035] = "\xa3\x70",
  [0x0036] = "\xa3\x71", [0x0037] = "\xa3\x72", [0x0038] = "\xa3\x73",
};

/* Table of range U+2013 - U+203B. */
static const char from_ucs4_tab4[][2] =
{
  [0x0000] = "\xa1\x56", [0x0001] = "\xa1\x58", [0x0005] = "\xa1\xa5",
  [0x0006] = "\xa1\xa6", [0x0009] = "\xa1\xa7", [0x000a] = "\xa1\xa8",
  [0x0012] = "\xa1\x4c", [0x0013] = "\xa1\x4b", [0x0014] = "\xa1\x45",
  [0x001f] = "\xa1\xac", [0x0022] = "\xa1\xab", [0x0028] = "\xa1\xb0",
};

/* Table of range U+2160 - U+2169. */
static const char from_ucs4_tab5[][2] =
{
  [0x0000] = "\xa2\xb9", [0x0001] = "\xa2\xba", [0x0002] = "\xa2\xbb",
  [0x0003] = "\xa2\xbc", [0x0004] = "\xa2\xbd", [0x0005] = "\xa2\xbe",
  [0x0006] = "\xa2\xbf", [0x0007] = "\xa2\xc0", [0x0008] = "\xa2\xc1",
  [0x0009] = "\xa2\xc2",
};

/* Table of range U+2190 - U+2199. */
static const char from_ucs4_tab6[][2] =
{
  [0x0000] = "\xa1\xf6", [0x0001] = "\xa1\xf4", [0x0002] = "\xa1\xf7",
  [0x0003] = "\xa1\xf5", [0x0006] = "\xa1\xf8", [0x0007] = "\xa1\xf9",
  [0x0008] = "\xa1\xfb", [0x0009] = "\xa1\xfa",
};

/* Table of range U+2215 - U+2267. */
static const char from_ucs4_tab7[][2] =
{
  [0x0000] = "\xa2\x41", [0x0005] = "\xa1\xd4", [0x0009] = "\xa1\xdb",
  [0x000a] = "\xa1\xe8", [0x000b] = "\xa1\xe7", [0x000e] = "\xa1\xfd",
  [0x0010] = "\xa1\xfc", [0x0014] = "\xa1\xe4", [0x0015] = "\xa1\xe5",
  [0x0016] = "\xa1\xec", [0x0019] = "\xa1\xed", [0x001f] = "\xa1\xef",
  [0x0020] = "\xa1\xee", [0x003d] = "\xa1\xdc", [0x004b] = "\xa1\xda",
  [0x004c] = "\xa1\xdd", [0x0051] = "\xa1\xd8", [0x0052] = "\xa1\xd9",
};

/* Table of range U+2500 - U+2642. */
static const char from_ucs4_tab8[][2] =
{
  [0x0000] = "\xa2\x77", [0x0002] = "\xa2\x78", [0x000c] = "\xa2\x7a",
  [0x0010] = "\xa2\x7b", [0x0014] = "\xa2\x7c", [0x0018] = "\xa2\x7d",
  [0x001c] = "\xa2\x75", [0x0024] = "\xa2\x74", [0x002c] = "\xa2\x73",
  [0x0034] = "\xa2\x72", [0x003c] = "\xa2\x71", [0x0050] = "\xa2\xa4",
  [0x0051] = "\xf9\xf8", [0x0052] = "\xf9\xe6", [0x0053] = "\xf9\xef",
  [0x0054] = "\xf9\xdd", [0x0055] = "\xf9\xe8", [0x0056] = "\xf9\xf1",
  [0x0057] = "\xf9\xdf", [0x0058] = "\xf9\xec", [0x0059] = "\xf9\xf5",
  [0x005a] = "\xf9\xe3", [0x005b] = "\xf9\xee", [0x005c] = "\xf9\xf7",
  [0x005d] = "\xf9\xe5", [0x005e] = "\xa2\xa5", [0x005f] = "\xf9\xf2",
  [0x0060] = "\xf9\xe0", [0x0061] = "\xa2\xa7", [0x0062] = "\xf9\xf4",
  [0x0063] = "\xf9\xe2", [0x0064] = "\xf9\xe7", [0x0065] = "\xf9\xf0",
  [0x0066] = "\xf9\xde", [0x0067] = "\xf9\xed", [0x0068] = "\xf9\xf6",
  [0x0069] = "\xf9\xe4", [0x006a] = "\xa2\xa6", [0x006b] = "\xf9\xf3",
  [0x006c] = "\xf9\xe1", [0x006d] = "\xa2\x7e", [0x006e] = "\xa2\xa1",
  [0x006f] = "\xa2\xa3", [0x0070] = "\xa2\xa2", [0x0071] = "\xa2\xac",
  [0x0072] = "\xa2\xad", [0x0073] = "\xa2\xae", [0x0074] = "\xa1\x5a",
  [0x0081] = "\xa2\x62", [0x0082] = "\xa2\x63", [0x0083] = "\xa2\x64",
  [0x0084] = "\xa2\x65", [0x0085] = "\xa2\x66", [0x0086] = "\xa2\x67",
  [0x0087] = "\xa2\x68", [0x0088] = "\xa2\x69", [0x0089] = "\xa2\x70",
  [0x008a] = "\xa2\x6f", [0x008b] = "\xa2\x6e", [0x008c] = "\xa2\x6d",
  [0x008d] = "\xa2\x6c", [0x008e] = "\xa2\x6b", [0x008f] = "\xa2\x6a",
  [0x0093] = "\xf9\xfe", [0x0094] = "\xa2\x76", [0x0095] = "\xa2\x79",
  [0x00a0] = "\xa1\xbd", [0x00a1] = "\xa1\xbc", [0x00b2] = "\xa1\xb6",
  [0x00b3] = "\xa1\xb5", [0x00bc] = "\xa1\xbf", [0x00bd] = "\xa1\xbe",
  [0x00c6] = "\xa1\xbb", [0x00c7] = "\xa1\xba", [0x00cb] = "\xa1\xb3",
  [0x00ce] = "\xa1\xb7", [0x00cf] = "\xa1\xb4", [0x00e2] = "\xa2\xa8",
  [0x00e3] = "\xa2\xa9", [0x00e4] = "\xa2\xab", [0x00e5] = "\xa2\xaa",
  [0x0105] = "\xa1\xb9", [0x0106] = "\xa1\xb8", [0x0140] = "\xa1\xf0",
  [0x0142] = "\xa1\xf1",
};

/* Table of range U+3000 - U+3029. */
static const char from_ucs4_tab9[][2] =
{
  [0x0000] = "\xa1\x40", [0x0001] = "\xa1\x42", [0x0002] = "\xa1\x43",
  [0x0003] = "\xa1\xb2", [0x0008] = "\xa1\x71", [0x0009] = "\xa1\x72",
  [0x000a] = "\xa1\x6d", [0x000b] = "\xa1\x6e", [0x000c] = "\xa1\x75",
  [0x000d] = "\xa1\x76", [0x000e] = "\xa1\x79", [0x000f] = "\xa1\x7a",
  [0x0010] = "\xa1\x69", [0x0011] = "\xa1\x6a", [0x0012] = "\xa2\x45",
  [0x0014] = "\xa1\x65", [0x0015] = "\xa1\x66", [0x001d] = "\xa1\xa9",
  [0x001e] = "\xa1\xaa", [0x0021] = "\xa2\xc3", [0x0022] = "\xa2\xc4",
  [0x0023] = "\xa2\xc5", [0x0024] = "\xa2\xc6", [0x0025] = "\xa2\xc7",
  [0x0026] = "\xa2\xc8", [0x0027] = "\xa2\xc9", [0x0028] = "\xa2\xca",
  [0x0029] = "\xa2\xcb",
};

/* Table of range U+3105 - U+3129. */
static const char from_ucs4_tab10[][2] =
{
  [0x0000] = "\xa3\x74", [0x0001] = "\xa3\x75", [0x0002] = "\xa3\x76",
  [0x0003] = "\xa3\x77", [0x0004] = "\xa3\x78", [0x0005] = "\xa3\x79",
  [0x0006] = "\xa3\x7a", [0x0007] = "\xa3\x7b", [0x0008] = "\xa3\x7c",
  [0x0009] = "\xa3\x7d", [0x000a] = "\xa3\x7e", [0x000b] = "\xa3\xa1",
  [0x000c] = "\xa3\xa2", [0x000d] = "\xa3\xa3", [0x000e] = "\xa3\xa4",
  [0x000f] = "\xa3\xa5", [0x0010] = "\xa3\xa6", [0x0011] = "\xa3\xa7",
  [0x0012] = "\xa3\xa8", [0x0013] = "\xa3\xa9", [0x0014] = "\xa3\xaa",
  [0x0015] = "\xa3\xab", [0x0016] = "\xa3\xac", [0x0017] = "\xa3\xad",
  [0x0018] = "\xa3\xae", [0x0019] = "\xa3\xaf", [0x001a] = "\xa3\xb0",
  [0x001b] = "\xa3\xb1", [0x001c] = "\xa3\xb2", [0x001d] = "\xa3\xb3",
  [0x001e] = "\xa3\xb4", [0x001f] = "\xa3\xb5", [0x0020] = "\xa3\xb6",
  [0x0021] = "\xa3\xb7", [0x0022] = "\xa3\xb8", [0x0023] = "\xa3\xb9",
  [0x0024] = "\xa3\xba",
};

/* Table of range U+338E - U+33D5. */
static const char from_ucs4_tab11[][2] =
{
  [0x0000] = "\xa2\x55", [0x0001] = "\xa2\x56", [0x000e] = "\xa2\x50",
  [0x000f] = "\xa2\x51", [0x0010] = "\xa2\x52", [0x0013] = "\xa2\x54",
  [0x0036] = "\xa2\x57", [0x0040] = "\xa2\x53", [0x0043] = "\xa1\xeb",
  [0x0044] = "\xa1\xea", [0x0047] = "\xa2\x4f",
};

/* Table of range U+4E00 - U+9FA4. */
static const char from_ucs4_tab12[][2] =
{
  [0x0000] = "\xa4\x40", [0x0001] = "\xa4\x42", [0x0003] = "\xa4\x43",
  [0x0007] = "\xc9\x45", [0x0008] = "\xa4\x56", [0x0009] = "\xa4\x54",
  [0x000a] = "\xa4\x57", [0x000b] = "\xa4\x55", [0x000c] = "\xc9\x46",
  [0x000d] = "\xa4\xa3", [0x000e] = "\xc9\x4f", [0x000f] = "\xc9\x4d",
  [0x0010] = "\xa4\xa2", [0x0011] = "\xa4\xa1", [0x0014] = "\xa5\x42",
  [0x0015] = "\xa5\x41", [0x0016] = "\xa5\x40", [0x0018] = "\xa5\x43",
  [0x0019] = "\xa4\xfe", [0x001e] = "\xa5\xe0", [0x001f] = "\xa5\xe1",
  [0x0026] = "\xa8\xc3", [0x002b] = "\xa4\x58", [0x002d] = "\xa4\xa4",
  [0x002e] = "\xc9\x50", [0x0030] = "\xa4\xa5", [0x0031] = "\xc9\x63",
  [0x0032] = "\xa6\xea", [0x0033] = "\xcb\xb1", [0x0038] = "\xa4\x59",
  [0x0039] = "\xa4\xa6", [0x003b] = "\xa5\x44", [0x003c] = "\xc9\x64",
  [0x0042] = "\xc9\x40", [0x0043] = "\xa4\x44", [0x0045] = "\xa4\x5b",
  [0x0047] = "\xc9\x47", [0x0048] = "\xa4\x5c", [0x004b] = "\xa4\xa7",
  [0x004d] = "\xa5\x45", [0x004e] = "\xa5\x47", [0x004f] = "\xa5\x46",
  [0x0052] = "\xa5\xe2", [0x0053] = "\xa5\xe3", [0x0056] = "\xa8\xc4",
  [0x0058] = "\xad\xbc", [0x0059] = "\xa4\x41", [0x005c] = "\xc9\x41",
  [0x005d] = "\xa4\x45", [0x005e] = "\xa4\x5e", [0x005f] = "\xa4\x5d",
  [0x0069] = "\xa5\xe4", [0x0073] = "\xa8\xc5", [0x007e] = "\xb0\xae",
  [0x007f] = "\xd4\x4b", [0x0082] = "\xb6\xc3", [0x0083] = "\xdc\xb1",
  [0x0084] = "\xdc\xb2", [0x0086] = "\xa4\x46", [0x0088] = "\xa4\xa9",
  [0x008b] = "\xa8\xc6", [0x008c] = "\xa4\x47", [0x008d] = "\xc9\x48",
  [0x008e] = "\xa4\x5f", [0x0091] = "\xa4\xaa", [0x0092] = "\xa4\xac",
  [0x0093] = "\xc9\x51", [0x0094] = "\xa4\xad", [0x0095] = "\xa4\xab",
  [0x0099] = "\xa5\xe5", [0x009b] = "\xa8\xc7", [0x009e] = "\xa8\xc8",
  [0x009f] = "\xab\x45", [0x00a1] = "\xa4\x60", [0x00a2] = "\xa4\xae",
  [0x00a4] = "\xa5\xe6", [0x00a5] = "\xa5\xe8", [0x00a6] = "\xa5\xe7",
  [0x00a8] = "\xa6\xeb", [0x00ab] = "\xa8\xc9", [0x00ac] = "\xa8\xca",
  [0x00ad] = "\xab\x46", [0x00ae] = "\xab\x47", [0x00b3] = "\xad\xbd",
  [0x00b6] = "\xdc\xb3", [0x00b9] = "\xf6\xd6", [0x00ba] = "\xa4\x48",
  [0x00c0] = "\xa4\xb0", [0x00c1] = "\xa4\xaf", [0x00c2] = "\xc9\x52",
  [0x00c3] = "\xa4\xb1", [0x00c4] = "\xa4\xb7", [0x00c6] = "\xa4\xb2",
  [0x00c7] = "\xa4\xb3", [0x00c8] = "\xc9\x54", [0x00c9] = "\xc9\x53",
  [0x00ca] = "\xa4\xb5", [0x00cb] = "\xa4\xb6", [0x00cd] = "\xa4\xb4",
  [0x00d4] = "\xa5\x4a", [0x00d5] = "\xa5\x4b", [0x00d6] = "\xa5\x4c",
  [0x00d7] = "\xa5\x4d", [0x00d8] = "\xa5\x49", [0x00d9] = "\xa5\x50",
  [0x00da] = "\xc9\x6a", [0x00dc] = "\xc9\x66", [0x00dd] = "\xc9\x69",
  [0x00de] = "\xa5\x51", [0x00df] = "\xa5\x61", [0x00e1] = "\xc9\x68",
  [0x00e3] = "\xa5\x4e", [0x00e4] = "\xa5\x4f", [0x00e5] = "\xa5\x48",
  [0x00e8] = "\xc9\x65", [0x00e9] = "\xc9\x67", [0x00f0] = "\xa5\xf5",
  [0x00f1] = "\xc9\xb0", [0x00f2] = "\xa5\xf2", [0x00f3] = "\xa5\xf6",
  [0x00f4] = "\xc9\xba", [0x00f5] = "\xc9\xae", [0x00f6] = "\xa5\xf3",
  [0x00f7] = "\xc9\xb2", [0x00fb] = "\xa5\xf4", [0x00fd] = "\xa5\xf7",
  [0x00ff] = "\xa5\xe9", [0x0100] = "\xc9\xb1", [0x0101] = "\xa5\xf8",
  [0x0102] = "\xc9\xb5", [0x0104] = "\xc9\xb9", [0x0105] = "\xc9\xb6",
  [0x0108] = "\xc9\xb3", [0x0109] = "\xa5\xea", [0x010a] = "\xa5\xec",
  [0x010b] = "\xa5\xf9", [0x010d] = "\xa5\xee", [0x010e] = "\xc9\xab",
  [0x010f] = "\xa5\xf1", [0x0110] = "\xa5\xef", [0x0111] = "\xa5\xf0",
  [0x0112] = "\xc9\xbb", [0x0113] = "\xc9\xb8", [0x0114] = "\xc9\xaf",
  [0x0115] = "\xa5\xed", [0x0118] = "\xc9\xac", [0x0119] = "\xa5\xeb",
  [0x011d] = "\xc9\xb4", [0x0122] = "\xc9\xb7", [0x012c] = "\xc9\xad",
  [0x012d] = "\xca\x66", [0x012f] = "\xa7\x42", [0x0130] = "\xa6\xf4",
  [0x0133] = "\xca\x67", [0x0134] = "\xa6\xf1", [0x0136] = "\xa7\x44",
  [0x0138] = "\xa6\xf9", [0x013a] = "\xa6\xf8", [0x013b] = "\xca\x5b",
  [0x013c] = "\xa6\xfc", [0x013d] = "\xa6\xf7", [0x013e] = "\xca\x60",
  [0x013f] = "\xca\x68", [0x0141] = "\xca\x64", [0x0143] = "\xa6\xfa",
  [0x0146] = "\xa6\xfd", [0x0147] = "\xa6\xee", [0x0148] = "\xa7\x47",
  [0x0149] = "\xca\x5d", [0x014c] = "\xcb\xbd", [0x014d] = "\xa6\xec",
  [0x014e] = "\xa7\x43", [0x014f] = "\xa6\xed", [0x0150] = "\xa6\xf5",
  [0x0151] = "\xa6\xf6", [0x0152] = "\xca\x62", [0x0153] = "\xca\x5e",
  [0x0154] = "\xa6\xfb", [0x0155] = "\xa6\xf3", [0x0156] = "\xca\x5a",
  [0x0157] = "\xa6\xef", [0x0158] = "\xca\x65", [0x0159] = "\xa7\x45",
  [0x015a] = "\xa7\x48", [0x015b] = "\xa6\xf2", [0x015c] = "\xa7\x40",
  [0x015d] = "\xa7\x46", [0x015e] = "\xa6\xf0", [0x015f] = "\xca\x63",
  [0x0160] = "\xa7\x41", [0x0161] = "\xca\x69", [0x0162] = "\xca\x5c",
  [0x0163] = "\xa6\xfe", [0x0164] = "\xca\x5f", [0x0167] = "\xca\x61",
  [0x0169] = "\xa8\xd8", [0x016a] = "\xcb\xbf", [0x016b] = "\xcb\xcb",
  [0x016c] = "\xa8\xd0", [0x016e] = "\xcb\xcc", [0x016f] = "\xa8\xcb",
  [0x0170] = "\xa8\xd5", [0x0173] = "\xa8\xce", [0x0174] = "\xcb\xb9",
  [0x0175] = "\xa8\xd6", [0x0176] = "\xcb\xb8", [0x0177] = "\xcb\xbc",
  [0x0178] = "\xcb\xc3", [0x0179] = "\xcb\xc1", [0x017a] = "\xa8\xde",
  [0x017b] = "\xa8\xd9", [0x017c] = "\xcb\xb3", [0x017d] = "\xcb\xb5",
  [0x017e] = "\xa8\xdb", [0x017f] = "\xa8\xcf", [0x0180] = "\xcb\xb6",
  [0x0181] = "\xcb\xc2", [0x0182] = "\xcb\xc9", [0x0183] = "\xa8\xd4",
  [0x0184] = "\xcb\xbb", [0x0185] = "\xcb\xb4", [0x0186] = "\xa8\xd3",
  [0x0187] = "\xcb\xb7", [0x0188] = "\xa8\xd7", [0x0189] = "\xcb\xba",
  [0x018b] = "\xa8\xd2", [0x018d] = "\xa8\xcd", [0x018f] = "\xa8\xdc",
  [0x0190] = "\xcb\xc4", [0x0191] = "\xa8\xdd", [0x0192] = "\xcb\xc8",
  [0x0194] = "\xcb\xc6", [0x0195] = "\xcb\xca", [0x0196] = "\xa8\xda",
  [0x0197] = "\xcb\xbe", [0x0198] = "\xcb\xb2", [0x019a] = "\xcb\xc0",
  [0x019b] = "\xa8\xd1", [0x019c] = "\xcb\xc5", [0x019d] = "\xa8\xcc",
  [0x019e] = "\xcb\xc7", [0x01ae] = "\xab\x56", [0x01af] = "\xab\x4a",
  [0x01b2] = "\xcd\xe0", [0x01b3] = "\xcd\xe8", [0x01b5] = "\xab\x49",
  [0x01b6] = "\xab\x51", [0x01b7] = "\xab\x5d", [0x01b9] = "\xcd\xee",
  [0x01ba] = "\xcd\xec", [0x01bb] = "\xcd\xe7", [0x01bf] = "\xab\x4b",
  [0x01c0] = "\xcd\xed", [0x01c1] = "\xcd\xe3", [0x01c2] = "\xab\x59",
  [0x01c3] = "\xab\x50", [0x01c4] = "\xab\x58", [0x01c5] = "\xcd\xde",
  [0x01c7] = "\xcd\xea", [0x01c9] = "\xcd\xe1", [0x01ca] = "\xab\x54",
  [0x01cb] = "\xcd\xe2", [0x01cd] = "\xcd\xdd", [0x01ce] = "\xab\x5b",
  [0x01cf] = "\xab\x4e", [0x01d0] = "\xab\x57", [0x01d1] = "\xab\x4d",
  [0x01d3] = "\xcd\xdf", [0x01d4] = "\xcd\xe4", [0x01d6] = "\xcd\xeb",
  [0x01d7] = "\xab\x55", [0x01d8] = "\xab\x52", [0x01d9] = "\xcd\xe6",
  [0x01da] = "\xab\x5a", [0x01db] = "\xcd\xe9", [0x01dc] = "\xcd\xe5",
  [0x01dd] = "\xab\x4f", [0x01de] = "\xab\x5c", [0x01df] = "\xab\x53",
  [0x01e0] = "\xab\x4c", [0x01e1] = "\xab\x48", [0x01ec] = "\xcd\xef",
  [0x01ee] = "\xad\xd7", [0x01ef] = "\xad\xc1", [0x01f1] = "\xad\xd1",
  [0x01f3] = "\xad\xd6", [0x01f4] = "\xd0\xd0", [0x01f5] = "\xd0\xcf",
  [0x01f6] = "\xd0\xd4", [0x01f7] = "\xd0\xd5", [0x01f8] = "\xad\xc4",
  [0x01fa] = "\xad\xcd", [0x01fe] = "\xad\xda", [0x0200] = "\xad\xce",
  [0x0205] = "\xd0\xc9", [0x0206] = "\xad\xc7", [0x0207] = "\xd0\xca",
  [0x0209] = "\xad\xdc", [0x020b] = "\xad\xd3", [0x020c] = "\xad\xbe",
  [0x020d] = "\xad\xbf", [0x020e] = "\xd0\xdd", [0x020f] = "\xb0\xbf",
  [0x0211] = "\xad\xcc", [0x0212] = "\xad\xcb", [0x0213] = "\xd0\xcb",
  [0x0214] = "\xad\xcf", [0x0215] = "\xd4\x5b", [0x0216] = "\xad\xc6",
  [0x0217] = "\xd0\xd6", [0x0218] = "\xad\xd5", [0x0219] = "\xad\xd4",
  [0x021a] = "\xad\xca", [0x021b] = "\xd0\xce", [0x021c] = "\xd0\xd7",
  [0x021e] = "\xd0\xc8", [0x021f] = "\xad\xc9", [0x0220] = "\xd0\xd8",
  [0x0221] = "\xad\xd2", [0x0222] = "\xd0\xcc", [0x0223] = "\xad\xc0",
  [0x0225] = "\xad\xc3", [0x0226] = "\xad\xc2", [0x0227] = "\xd0\xd9",
  [0x0228] = "\xad\xd0", [0x0229] = "\xad\xc5", [0x022a] = "\xad\xd9",
  [0x022b] = "\xad\xdb", [0x022c] = "\xd0\xd3", [0x022d] = "\xad\xd8",
  [0x022f] = "\xd0\xdb", [0x0230] = "\xd0\xcd", [0x0231] = "\xd0\xdc",
  [0x0233] = "\xd0\xd1", [0x0235] = "\xd0\xda", [0x0237] = "\xd0\xd2",
  [0x023c] = "\xad\xc8", [0x0240] = "\xd4\x63", [0x0241] = "\xd4\x57",
  [0x0243] = "\xb0\xb3", [0x0245] = "\xd4\x5c", [0x0246] = "\xd4\x62",
  [0x0247] = "\xb0\xb2", [0x0248] = "\xd4\x55", [0x0249] = "\xb0\xb6",
  [0x024a] = "\xd4\x59", [0x024b] = "\xd4\x52", [0x024c] = "\xb0\xb4",
  [0x024d] = "\xd4\x56", [0x024e] = "\xb0\xb9", [0x024f] = "\xb0\xbe",
  [0x0251] = "\xd4\x67", [0x0253] = "\xd4\x51", [0x0255] = "\xb0\xba",
  [0x0257] = "\xd4\x66", [0x025a] = "\xb0\xb5", [0x025b] = "\xd4\x58",
  [0x025c] = "\xb0\xb1", [0x025d] = "\xd4\x53", [0x025e] = "\xd4\x4f",
  [0x025f] = "\xd4\x5d", [0x0260] = "\xd4\x50", [0x0261] = "\xd4\x4e",
  [0x0262] = "\xd4\x5a", [0x0263] = "\xd4\x60", [0x0264] = "\xd4\x61",
  [0x0265] = "\xb0\xb7", [0x0268] = "\xd8\x5b", [0x0269] = "\xd4\x5e",
  [0x026a] = "\xd4\x4d", [0x026b] = "\xd4\x5f", [0x026d] = "\xb0\xc1",
  [0x026e] = "\xd4\x64", [0x026f] = "\xb0\xc0", [0x0270] = "\xd4\x4c",
  [0x0272] = "\xd4\x54", [0x0273] = "\xd4\x65", [0x0274] = "\xb0\xbc",
  [0x0275] = "\xb0\xbb", [0x0276] = "\xb0\xb8", [0x0277] = "\xb0\xbd",
  [0x027a] = "\xb0\xaf", [0x027d] = "\xb0\xb0", [0x0280] = "\xb3\xc8",
  [0x0282] = "\xd8\x5e", [0x0283] = "\xd8\x57", [0x0285] = "\xb3\xc5",
  [0x0287] = "\xd8\x5f", [0x028b] = "\xd8\x55", [0x028c] = "\xd8\x58",
  [0x028d] = "\xb3\xc4", [0x028e] = "\xd8\x59", [0x0291] = "\xb3\xc7",
  [0x0292] = "\xd8\x5d", [0x0294] = "\xd8\x53", [0x0295] = "\xd8\x52",
  [0x0296] = "\xb3\xc9", [0x0298] = "\xb3\xca", [0x0299] = "\xb3\xc6",
  [0x029a] = "\xb3\xcb", [0x029b] = "\xd8\x51", [0x029c] = "\xd8\x5c",
  [0x029d] = "\xd8\x5a", [0x029e] = "\xd8\x54", [0x02a2] = "\xb3\xc3",
  [0x02a3] = "\xd8\x56", [0x02ac] = "\xb6\xca", [0x02ad] = "\xb6\xc4",
  [0x02ae] = "\xdc\xb7", [0x02af] = "\xb6\xcd", [0x02b0] = "\xdc\xbd",
  [0x02b1] = "\xdc\xc0", [0x02b2] = "\xb6\xc6", [0x02b3] = "\xb6\xc7",
  [0x02b4] = "\xdc\xba", [0x02b5] = "\xb6\xc5", [0x02b6] = "\xdc\xc3",
  [0x02b7] = "\xb6\xcb", [0x02b8] = "\xdc\xc4", [0x02ba] = "\xdc\xbf",
  [0x02bb] = "\xb6\xcc", [0x02bd] = "\xdc\xb4", [0x02be] = "\xb6\xc9",
  [0x02bf] = "\xdc\xb5", [0x02c1] = "\xdc\xbe", [0x02c2] = "\xdc\xbc",
  [0x02c4] = "\xdc\xb8", [0x02c5] = "\xb6\xc8", [0x02c6] = "\xdc\xb6",
  [0x02c7] = "\xb6\xce", [0x02c8] = "\xdc\xbb", [0x02c9] = "\xdc\xc2",
  [0x02ca] = "\xdc\xb9", [0x02cb] = "\xdc\xc1", [0x02ce] = "\xb9\xb6",
  [0x02cf] = "\xb9\xb3", [0x02d1] = "\xb9\xb4", [0x02d3] = "\xe0\xf9",
  [0x02d4] = "\xe0\xf1", [0x02d5] = "\xb9\xb2", [0x02d6] = "\xb9\xaf",
  [0x02d7] = "\xe0\xf2", [0x02da] = "\xb9\xb1", [0x02db] = "\xe0\xf5",
  [0x02dd] = "\xe0\xf7", [0x02e0] = "\xe0\xfe", [0x02e3] = "\xe0\xfd",
  [0x02e4] = "\xe0\xf8", [0x02e5] = "\xb9\xae", [0x02e6] = "\xe0\xf0",
  [0x02e7] = "\xb9\xac", [0x02e8] = "\xe0\xf3", [0x02e9] = "\xb9\xb7",
  [0x02ea] = "\xe0\xf6", [0x02ec] = "\xe0\xfa", [0x02ed] = "\xb9\xb0",
  [0x02ee] = "\xb9\xad", [0x02ef] = "\xe0\xfc", [0x02f0] = "\xe0\xfb",
  [0x02f1] = "\xb9\xb5", [0x02f3] = "\xe0\xf4", [0x02f5] = "\xbb\xf8",
  [0x02f6] = "\xe4\xec", [0x02f8] = "\xe4\xe9", [0x02f9] = "\xbb\xf9",
  [0x02fb] = "\xbb\xf7", [0x02fd] = "\xe4\xf0", [0x02fe] = "\xe4\xed",
  [0x02ff] = "\xe4\xe6", [0x0300] = "\xbb\xf6", [0x0302] = "\xbb\xfa",
  [0x0303] = "\xe4\xe7", [0x0304] = "\xbb\xf5", [0x0305] = "\xbb\xfd",
  [0x0306] = "\xe4\xea", [0x0307] = "\xe4\xeb", [0x0308] = "\xbb\xfb",
  [0x0309] = "\xbb\xfc", [0x030a] = "\xe4\xf1", [0x030b] = "\xe4\xee",
  [0x030c] = "\xe4\xef", [0x0310] = "\xbe\xaa", [0x0311] = "\xe8\xf8",
  [0x0312] = "\xbe\xa7", [0x0313] = "\xe8\xf5", [0x0314] = "\xbe\xa9",
  [0x0315] = "\xbe\xab", [0x0317] = "\xe8\xf6", [0x0318] = "\xbe\xa8",
  [0x031a] = "\xe8\xf7", [0x031c] = "\xe8\xf4", [0x031f] = "\xc0\x76",
  [0x0320] = "\xec\xbd", [0x0321] = "\xc0\x77", [0x0322] = "\xec\xbb",
  [0x0324] = "\xec\xbc", [0x0325] = "\xec\xba", [0x0326] = "\xec\xb9",
  [0x0329] = "\xec\xbe", [0x032a] = "\xc0\x75", [0x032d] = "\xef\xb8",
  [0x032e] = "\xef\xb9", [0x0330] = "\xe4\xe8", [0x0331] = "\xef\xb7",
  [0x0332] = "\xc0\x78", [0x0333] = "\xc3\x5f", [0x0334] = "\xf1\xeb",
  [0x0335] = "\xf1\xec", [0x0337] = "\xc4\xd7", [0x0338] = "\xc4\xd8",
  [0x0339] = "\xf5\xc1", [0x033a] = "\xf5\xc0", [0x033b] = "\xc5\x6c",
  [0x033c] = "\xc5\x6b", [0x033d] = "\xf7\xd0", [0x033f] = "\xa4\x49",
  [0x0340] = "\xa4\x61", [0x0341] = "\xa4\xb9", [0x0343] = "\xa4\xb8",
  [0x0344] = "\xa5\x53", [0x0345] = "\xa5\x52", [0x0346] = "\xa5\xfc",
  [0x0347] = "\xa5\xfb", [0x0348] = "\xa5\xfd", [0x0349] = "\xa5\xfa",
  [0x034b] = "\xa7\x4a", [0x034c] = "\xa7\x49", [0x034d] = "\xa7\x4b",
  [0x0352] = "\xa8\xe0", [0x0354] = "\xa8\xdf", [0x0355] = "\xa8\xe1",
  [0x0357] = "\xab\x5e", [0x0359] = "\xa2\x59", [0x035a] = "\xd0\xde",
  [0x035b] = "\xa2\x5a", [0x035c] = "\xb0\xc2", [0x035d] = "\xa2\x5c",
  [0x035e] = "\xa2\x5b", [0x035f] = "\xd8\x60", [0x0361] = "\xa2\x5d",
  [0x0362] = "\xb9\xb8", [0x0363] = "\xa2\x5e", [0x0365] = "\xa4\x4a",
  [0x0367] = "\xa4\xba", [0x0368] = "\xa5\xfe", [0x0369] = "\xa8\xe2",
  [0x036b] = "\xa4\x4b", [0x036c] = "\xa4\xbd", [0x036d] = "\xa4\xbb",
  [0x036e] = "\xa4\xbc", [0x0371] = "\xa6\x40", [0x0375] = "\xa7\x4c",
  [0x0376] = "\xa8\xe4", [0x0377] = "\xa8\xe3", [0x0378] = "\xa8\xe5",
  [0x037c] = "\xad\xdd", [0x0380] = "\xbe\xac", [0x0387] = "\xc9\x4e",
  [0x0389] = "\xa5\x54", [0x038a] = "\xa5\x55", [0x038d] = "\xa6\x41",
  [0x038f] = "\xca\x6a", [0x0391] = "\xab\x60", [0x0392] = "\xab\x5f",
  [0x0393] = "\xd0\xe0", [0x0394] = "\xd0\xdf", [0x0395] = "\xb0\xc3",
  [0x0397] = "\xa4\xbe", [0x0398] = "\xc9\x55", [0x039e] = "\xcb\xcd",
  [0x03a0] = "\xab\x61", [0x03a2] = "\xad\xe0", [0x03a4] = "\xad\xde",
  [0x03a5] = "\xad\xdf", [0x03aa] = "\xbe\xad", [0x03ac] = "\xa5\x56",
  [0x03b0] = "\xa6\x42", [0x03b1] = "\xc9\xbc", [0x03b6] = "\xa7\x4d",
  [0x03b7] = "\xa7\x4e", [0x03b9] = "\xca\x6b", [0x03bc] = "\xcb\xce",
  [0x03bd] = "\xa8\xe6", [0x03be] = "\xcb\xcf", [0x03c4] = "\xd0\xe2",
  [0x03c5] = "\xd0\xe3", [0x03c6] = "\xad\xe3", [0x03c8] = "\xd0\xe4",
  [0x03ca] = "\xd0\xe1", [0x03cb] = "\xad\xe4", [0x03cc] = "\xad\xe2",
  [0x03cd] = "\xad\xe1", [0x03ce] = "\xd0\xe5", [0x03d0] = "\xd4\x68",
  [0x03d4] = "\xd8\x61", [0x03d7] = "\xdc\xc5", [0x03d8] = "\xe1\x40",
  [0x03dc] = "\xbb\xfe", [0x03dd] = "\xbe\xae", [0x03de] = "\xe8\xf9",
  [0x03e0] = "\xa4\x4c", [0x03e1] = "\xa4\x5a", [0x03f0] = "\xb0\xc4",
  [0x03f1] = "\xb3\xcd", [0x03f3] = "\xb9\xb9", [0x03f5] = "\xc9\x42",
  [0x03f6] = "\xa4\xbf", [0x03f8] = "\xa5\x59", [0x03f9] = "\xa5\x57",
  [0x03fa] = "\xa5\x58", [0x03fd] = "\xa8\xe7", [0x0400] = "\xa4\x4d",
  [0x0401] = "\xa4\x4e", [0x0403] = "\xa4\x62", [0x0406] = "\xa4\xc0",
  [0x0407] = "\xa4\xc1", [0x0408] = "\xa4\xc2", [0x0409] = "\xc9\xbe",
  [0x040a] = "\xa5\x5a", [0x040c] = "\xc9\x6b", [0x040e] = "\xa6\x46",
  [0x0410] = "\xc9\xbf", [0x0411] = "\xa6\x44", [0x0412] = "\xa6\x45",
  [0x0413] = "\xc9\xbd", [0x0416] = "\xa6\x47", [0x0417] = "\xa6\x43",
  [0x041c] = "\xca\x6c", [0x041d] = "\xaa\xec", [0x041e] = "\xca\x6d",
  [0x0421] = "\xca\x6e", [0x0424] = "\xa7\x50", [0x0425] = "\xa7\x4f",
  [0x0428] = "\xa7\x53", [0x0429] = "\xa7\x51", [0x042a] = "\xa7\x52",
  [0x042e] = "\xa8\xed", [0x0430] = "\xa8\xec", [0x0431] = "\xcb\xd4",
  [0x0432] = "\xcb\xd1", [0x0433] = "\xcb\xd2", [0x0435] = "\xcb\xd0",
  [0x0436] = "\xa8\xee", [0x0437] = "\xa8\xea", [0x0438] = "\xa8\xe9",
  [0x043a] = "\xa8\xeb", [0x043b] = "\xa8\xe8", [0x0441] = "\xa8\xef",
  [0x0443] = "\xab\x63", [0x0444] = "\xcd\xf0", [0x0446] = "\xcb\xd3",
  [0x0447] = "\xab\x68", [0x0449] = "\xcd\xf1", [0x044a] = "\xab\x64",
  [0x044b] = "\xab\x67", [0x044c] = "\xab\x66", [0x044d] = "\xab\x65",
  [0x044e] = "\xab\x62", [0x0452] = "\xd0\xe8", [0x0454] = "\xad\xe7",
  [0x0455] = "\xd0\xeb", [0x0456] = "\xad\xe5", [0x045a] = "\xd0\xe7",
  [0x045b] = "\xad\xe8", [0x045c] = "\xad\xe6", [0x045d] = "\xad\xe9",
  [0x045e] = "\xd0\xe9", [0x045f] = "\xd0\xea", [0x0461] = "\xd0\xe6",
  [0x0462] = "\xd0\xec", [0x0469] = "\xb3\xd1", [0x046a] = "\xb0\xc5",
  [0x046b] = "\xd4\x69", [0x046c] = "\xd4\x6b", [0x046d] = "\xd4\x6a",
  [0x046e] = "\xd4\x6c", [0x046f] = "\xb0\xc6", [0x0472] = "\xb3\xce",
  [0x0474] = "\xb3\xcf", [0x0475] = "\xb3\xd0", [0x0477] = "\xb6\xd0",
  [0x0478] = "\xdc\xc7", [0x047a] = "\xdc\xc6", [0x047b] = "\xdc\xc8",
  [0x047c] = "\xdc\xc9", [0x047d] = "\xb6\xd1", [0x047f] = "\xb6\xcf",
  [0x0480] = "\xe1\x41", [0x0481] = "\xe1\x42", [0x0482] = "\xb9\xbb",
  [0x0483] = "\xb9\xba", [0x0484] = "\xe3\x5a", [0x0487] = "\xbc\x40",
  [0x0488] = "\xbc\x41", [0x0489] = "\xbc\x42", [0x048a] = "\xbc\x44",
  [0x048b] = "\xe4\xf2", [0x048c] = "\xe4\xf3", [0x048d] = "\xbc\x43",
  [0x0491] = "\xbe\xaf", [0x0493] = "\xbe\xb0", [0x0496] = "\xf1\xed",
  [0x0497] = "\xf5\xc3", [0x0498] = "\xf5\xc2", [0x0499] = "\xf7\xd1",
  [0x049b] = "\xa4\x4f", [0x049f] = "\xa5\x5c", [0x04a0] = "\xa5\x5b",
  [0x04a3] = "\xa6\x48", [0x04a6] = "\xc9\xc0", [0x04a9] = "\xa7\x55",
  [0x04aa] = "\xa7\x56", [0x04ab] = "\xa7\x54", [0x04ac] = "\xa7\x57",
  [0x04ad] = "\xca\x6f", [0x04ae] = "\xca\x70", [0x04bb] = "\xa8\xf1",
  [0x04bc] = "\xcb\xd5", [0x04be] = "\xa8\xf0", [0x04c0] = "\xcd\xf2",
  [0x04c1] = "\xab\x6c", [0x04c2] = "\xcd\xf3", [0x04c3] = "\xab\x6b",
  [0x04c7] = "\xab\x69", [0x04c9] = "\xab\x6a", [0x04cd] = "\xd0\xed",
  [0x04d2] = "\xb0\xc7", [0x04d3] = "\xd4\x6e", [0x04d5] = "\xb0\xca",
  [0x04d6] = "\xd4\x6d", [0x04d7] = "\xb1\xe5", [0x04d8] = "\xb0\xc9",
  [0x04d9] = "\xb0\xc8", [0x04db] = "\xb3\xd4", [0x04dd] = "\xb3\xd3",
  [0x04de] = "\xb3\xd2", [0x04df] = "\xb6\xd2", [0x04e2] = "\xb6\xd5",
  [0x04e3] = "\xb6\xd6", [0x04e4] = "\xb6\xd4", [0x04e6] = "\xb6\xd3",
  [0x04e9] = "\xe1\x43", [0x04eb] = "\xe1\x44", [0x04ef] = "\xe4\xf5",
  [0x04f0] = "\xbc\x45", [0x04f1] = "\xe4\xf4", [0x04f3] = "\xbe\xb1",
  [0x04f4] = "\xec\xbf", [0x04f5] = "\xc0\x79", [0x04f7] = "\xf1\xee",
  [0x04f8] = "\xc4\x55", [0x04fa] = "\xa4\x63", [0x04fb] = "\xa4\xc3",
  [0x04fc] = "\xc9\x56", [0x04fe] = "\xa4\xc4", [0x04ff] = "\xa4\xc5",
  [0x0505] = "\xa5\x5d", [0x0506] = "\xa5\x5e", [0x0508] = "\xa6\x49",
  [0x0509] = "\xca\x71", [0x050a] = "\xcb\xd6", [0x050b] = "\xcb\xd7",
  [0x050d] = "\xab\x6d", [0x050e] = "\xd0\xee", [0x050f] = "\xb0\xcc",
  [0x0510] = "\xb0\xcb", [0x0511] = "\xd8\x63", [0x0512] = "\xd8\x62",
  [0x0515] = "\xa4\x50", [0x0516] = "\xa4\xc6", [0x0517] = "\xa5\x5f",
  [0x0519] = "\xb0\xcd", [0x051a] = "\xc9\x43", [0x051c] = "\xc9\x6c",
  [0x051d] = "\xa5\x60", [0x051f] = "\xc9\xc2", [0x0520] = "\xa6\x4b",
  [0x0521] = "\xa6\x4a", [0x0522] = "\xc9\xc1", [0x0523] = "\xa7\x58",
  [0x052a] = "\xad\xea", [0x052d] = "\xd4\x6f", [0x052f] = "\xb6\xd7",
  [0x0530] = "\xe1\x45", [0x0531] = "\xb9\xbc", [0x0534] = "\xe8\xfa",
  [0x0537] = "\xf3\xfd", [0x0539] = "\xa4\xc7", [0x053c] = "\xcb\xd8",
  [0x053d] = "\xcd\xf4", [0x053e] = "\xb0\xd0", [0x053f] = "\xb0\xce",
  [0x0540] = "\xb0\xcf", [0x0541] = "\xa4\x51", [0x0543] = "\xa4\x64",
  [0x0544] = "\xa2\xcd", [0x0545] = "\xa4\xca", [0x0547] = "\xa4\xc9",
  [0x0548] = "\xa4\xc8", [0x0549] = "\xa5\x63", [0x054a] = "\xa5\x62",
  [0x054c] = "\xc9\x6d", [0x054d] = "\xc9\xc3", [0x0551] = "\xa8\xf5",
  [0x0552] = "\xa8\xf2", [0x0553] = "\xa8\xf4", [0x0554] = "\xa8\xf3",
  [0x0557] = "\xab\x6e", [0x055a] = "\xb3\xd5", [0x055c] = "\xa4\x52",
  [0x055e] = "\xa4\xcb", [0x0560] = "\xa5\x65", [0x0561] = "\xa5\x64",
  [0x0563] = "\xca\x72", [0x0566] = "\xa8\xf6", [0x056c] = "\xc9\x57",
  [0x056e] = "\xa5\x67", [0x056f] = "\xa5\x66", [0x0570] = "\xa6\x4c",
  [0x0571] = "\xa6\x4d", [0x0572] = "\xca\x73", [0x0573] = "\xa7\x59",
  [0x0575] = "\xa7\x5a", [0x0577] = "\xa8\xf7", [0x0578] = "\xa8\xf8",
  [0x0579] = "\xa8\xf9", [0x057b] = "\xab\x6f", [0x057c] = "\xcd\xf5",
  [0x057f] = "\xad\xeb", [0x0582] = "\xc9\x44", [0x0584] = "\xa4\xcc",
  [0x058a] = "\xc9\xc4", [0x058e] = "\xca\x74", [0x058f] = "\xca\x75",
  [0x0592] = "\xcb\xd9", [0x0594] = "\xcb\xda", [0x0596] = "\xcd\xf7",
  [0x0597] = "\xcd\xf6", [0x0598] = "\xcd\xf9", [0x0599] = "\xcd\xf8",
  [0x059a] = "\xab\x70", [0x059c] = "\xd4\x70", [0x059d] = "\xad\xed",
  [0x059e] = "\xd0\xef", [0x059f] = "\xad\xec", [0x05a4] = "\xd8\x64",
  [0x05a5] = "\xb3\xd6", [0x05a7] = "\xd8\x65", [0x05ac] = "\xe1\x46",
  [0x05ad] = "\xb9\xbd", [0x05b2] = "\xbc\x46", [0x05b4] = "\xf1\xef",
  [0x05b9] = "\xc9\x58", [0x05bb] = "\xa5\x68", [0x05c3] = "\xb0\xd1",
  [0x05c8] = "\xa4\x53", [0x05c9] = "\xa4\x65", [0x05ca] = "\xa4\xce",
  [0x05cb] = "\xa4\xcd", [0x05cd] = "\xa4\xcf", [0x05d4] = "\xa8\xfb",
  [0x05d6] = "\xa8\xfa", [0x05d7] = "\xa8\xfc", [0x05db] = "\xab\x71",
  [0x05df] = "\xad\xee", [0x05e1] = "\xe8\xfb", [0x05e2] = "\xc2\x4f",
  [0x05e3] = "\xa4\x66", [0x05e4] = "\xa5\x6a", [0x05e5] = "\xa5\x79",
  [0x05e6] = "\xa5\x74", [0x05e8] = "\xa5\x6f", [0x05e9] = "\xa5\x6e",
  [0x05ea] = "\xa5\x75", [0x05eb] = "\xa5\x73", [0x05ec] = "\xa5\x6c",
  [0x05ed] = "\xa5\x7a", [0x05ee] = "\xa5\x6d", [0x05ef] = "\xa5\x69",
  [0x05f0] = "\xa5\x78", [0x05f1] = "\xa5\x77", [0x05f2] = "\xa5\x76",
  [0x05f3] = "\xa5\x6b", [0x05f5] = "\xa5\x72", [0x05f8] = "\xa5\x71",
  [0x05fb] = "\xa5\x7b", [0x05fc] = "\xa5\x70", [0x0601] = "\xa6\x53",
  [0x0603] = "\xa6\x59", [0x0604] = "\xa6\x55", [0x0606] = "\xa6\x5b",
  [0x0607] = "\xc9\xc5", [0x0608] = "\xa6\x58", [0x0609] = "\xa6\x4e",
  [0x060a] = "\xa6\x51", [0x060b] = "\xa6\x54", [0x060c] = "\xa6\x50",
  [0x060d] = "\xa6\x57", [0x060e] = "\xa6\x5a", [0x060f] = "\xa6\x4f",
  [0x0610] = "\xa6\x52", [0x0611] = "\xa6\x56", [0x0612] = "\xa6\x5c",
  [0x0618] = "\xca\x7e", [0x0619] = "\xca\x7b", [0x061b] = "\xa7\x67",
  [0x061c] = "\xca\x7c", [0x061d] = "\xa7\x5b", [0x061e] = "\xa7\x5d",
  [0x061f] = "\xa7\x75", [0x0620] = "\xa7\x70", [0x0624] = "\xca\xa5",
  [0x0625] = "\xca\x7d", [0x0626] = "\xa7\x5f", [0x0627] = "\xa7\x61",
  [0x0628] = "\xca\xa4", [0x0629] = "\xa7\x68", [0x062a] = "\xca\x78",
  [0x062b] = "\xa7\x74", [0x062c] = "\xa7\x76", [0x062d] = "\xa7\x5c",
  [0x062e] = "\xa7\x6d", [0x0630] = "\xca\x76", [0x0631] = "\xa7\x73",
  [0x0633] = "\xa7\x64", [0x0635] = "\xa7\x6e", [0x0636] = "\xa7\x6f",
  [0x0637] = "\xca\x77", [0x0638] = "\xa7\x6c", [0x0639] = "\xa7\x6a",
  [0x063b] = "\xa7\x6b", [0x063c] = "\xa7\x71", [0x063d] = "\xca\xa1",
  [0x063e] = "\xa7\x5e", [0x0640] = "\xa7\x72", [0x0641] = "\xca\xa3",
  [0x0642] = "\xa7\x66", [0x0643] = "\xa7\x63", [0x0645] = "\xca\x7a",
  [0x0646] = "\xa7\x62", [0x0647] = "\xca\xa6", [0x0648] = "\xa7\x65",
  [0x064a] = "\xa7\x69", [0x064e] = "\xa7\x60", [0x064f] = "\xca\xa2",
  [0x0654] = "\xca\x79", [0x0660] = "\xcb\xeb", [0x0661] = "\xcb\xea",
  [0x0662] = "\xa9\x4f", [0x0663] = "\xcb\xed", [0x0664] = "\xcb\xef",
  [0x0665] = "\xcb\xe4", [0x0666] = "\xcb\xe7", [0x0667] = "\xcb\xee",
  [0x0668] = "\xa9\x50", [0x066b] = "\xcb\xe1", [0x066c] = "\xcb\xe5",
  [0x066f] = "\xcb\xe9", [0x0670] = "\xce\x49", [0x0671] = "\xa9\x4b",
  [0x0672] = "\xce\x4d", [0x0673] = "\xa8\xfd", [0x0674] = "\xcb\xe6",
  [0x0675] = "\xa8\xfe", [0x0676] = "\xa9\x4c", [0x0677] = "\xa9\x45",
  [0x0678] = "\xa9\x41", [0x067a] = "\xcb\xe2", [0x067b] = "\xa9\x44",
  [0x067c] = "\xa9\x49", [0x067d] = "\xa9\x52", [0x067e] = "\xcb\xe3",
  [0x067f] = "\xcb\xdc", [0x0680] = "\xa9\x43", [0x0681] = "\xcb\xdd",
  [0x0682] = "\xcb\xdf", [0x0684] = "\xa9\x46", [0x0686] = "\xa9\x48",
  [0x0687] = "\xcb\xdb", [0x0688] = "\xcb\xe0", [0x068b] = "\xa9\x51",
  [0x068c] = "\xa9\x4d", [0x068d] = "\xcb\xe8", [0x068e] = "\xa9\x53",
  [0x0690] = "\xa9\x4a", [0x0691] = "\xcb\xde", [0x0692] = "\xa9\x47",
  [0x0695] = "\xa9\x42", [0x0696] = "\xa9\x40", [0x0698] = "\xcb\xec",
  [0x069a] = "\xa9\x4e", [0x06a0] = "\xce\x48", [0x06a1] = "\xcd\xfb",
  [0x06a2] = "\xce\x4b", [0x06a5] = "\xcd\xfd", [0x06a6] = "\xab\x78",
  [0x06a7] = "\xab\xa8", [0x06a8] = "\xab\x74", [0x06a9] = "\xab\xa7",
  [0x06aa] = "\xab\x7d", [0x06ab] = "\xab\xa4", [0x06ac] = "\xab\x72",
  [0x06ad] = "\xcd\xfc", [0x06ae] = "\xce\x43", [0x06af] = "\xab\xa3",
  [0x06b0] = "\xce\x4f", [0x06b1] = "\xab\xa5", [0x06b3] = "\xab\x79",
  [0x06b6] = "\xce\x45", [0x06b7] = "\xce\x42", [0x06b8] = "\xab\x77",
  [0x06ba] = "\xcd\xfa", [0x06bb] = "\xab\xa6", [0x06bc] = "\xce\x4a",
  [0x06bd] = "\xab\x7c", [0x06be] = "\xce\x4c", [0x06bf] = "\xab\xa9",
  [0x06c0] = "\xab\x73", [0x06c1] = "\xab\x7e", [0x06c2] = "\xab\x7b",
  [0x06c3] = "\xce\x40", [0x06c4] = "\xab\xa1", [0x06c5] = "\xce\x46",
  [0x06c6] = "\xce\x47", [0x06c7] = "\xab\x7a", [0x06c8] = "\xab\xa2",
  [0x06c9] = "\xab\x76", [0x06ce] = "\xab\x75", [0x06cf] = "\xcd\xfe",
  [0x06d6] = "\xce\x44", [0x06de] = "\xce\x4e", [0x06e0] = "\xd1\x44",
  [0x06e1] = "\xad\xfb", [0x06e2] = "\xd0\xf1", [0x06e4] = "\xd0\xf6",
  [0x06e5] = "\xad\xf4", [0x06e6] = "\xae\x40", [0x06e7] = "\xd0\xf4",
  [0x06e8] = "\xad\xef", [0x06e9] = "\xad\xf9", [0x06ea] = "\xad\xfe",
  [0x06eb] = "\xd0\xfb", [0x06ed] = "\xad\xfa", [0x06ee] = "\xad\xfd",
  [0x06f1] = "\xd0\xfe", [0x06f2] = "\xad\xf5", [0x06f3] = "\xd0\xf5",
  [0x06f7] = "\xd1\x42", [0x06f8] = "\xd1\x43", [0x06fa] = "\xad\xf7",
  [0x06fb] = "\xd1\x41", [0x06fc] = "\xad\xf3", [0x06fd] = "\xae\x43",
  [0x06ff] = "\xd0\xf8", [0x0701] = "\xad\xf1", [0x0703] = "\xd1\x46",
  [0x0704] = "\xd0\xf9", [0x0705] = "\xd0\xfd", [0x0706] = "\xad\xf6",
  [0x0707] = "\xae\x42", [0x0708] = "\xd0\xfa", [0x0709] = "\xad\xfc",
  [0x070a] = "\xd1\x40", [0x070b] = "\xd1\x47", [0x070c] = "\xd4\xa1",
  [0x070e] = "\xd1\x45", [0x070f] = "\xae\x44", [0x0710] = "\xad\xf0",
  [0x0711] = "\xd0\xfc", [0x0712] = "\xd0\xf3", [0x0714] = "\xad\xf8",
  [0x0717] = "\xd0\xf2", [0x071a] = "\xd0\xf7", [0x0726] = "\xd0\xf0",
  [0x0727] = "\xae\x41", [0x072a] = "\xd4\x77", [0x072c] = "\xb0\xe4",
  [0x072d] = "\xd4\xa7", [0x072e] = "\xb0\xe2", [0x072f] = "\xb0\xdf",
  [0x0730] = "\xd4\x7c", [0x0731] = "\xb0\xdb", [0x0732] = "\xd4\xa2",
  [0x0733] = "\xb0\xe6", [0x0734] = "\xd4\x76", [0x0735] = "\xd4\x7b",
  [0x0736] = "\xd4\x7a", [0x0737] = "\xad\xf2", [0x0738] = "\xb0\xe1",
  [0x0739] = "\xd4\xa5", [0x073b] = "\xd4\xa8", [0x073c] = "\xd4\x73",
  [0x073e] = "\xb3\xe8", [0x0740] = "\xd4\xa9", [0x0741] = "\xb0\xe7",
  [0x0743] = "\xb0\xd9", [0x0744] = "\xb0\xd6", [0x0745] = "\xd4\x7e",
  [0x0746] = "\xb0\xd3", [0x0748] = "\xd4\xa6", [0x074a] = "\xb0\xda",
  [0x074b] = "\xd4\xaa", [0x074d] = "\xd4\x74", [0x074e] = "\xd4\xa4",
  [0x074f] = "\xb0\xdd", [0x0750] = "\xd4\x75", [0x0751] = "\xd4\x78",
  [0x0752] = "\xd4\x7d", [0x0755] = "\xb0\xde", [0x0756] = "\xb0\xdc",
  [0x0757] = "\xb0\xe8", [0x075c] = "\xb0\xe3", [0x075e] = "\xb0\xd7",
  [0x075f] = "\xb1\xd2", [0x0761] = "\xb0\xd8", [0x0762] = "\xd4\x79",
  [0x0763] = "\xb0\xe5", [0x0764] = "\xb0\xe0", [0x0765] = "\xd4\xa3",
  [0x0766] = "\xb0\xd5", [0x076a] = "\xb0\xd4", [0x0775] = "\xd4\x71",
  [0x0776] = "\xd4\x72", [0x0777] = "\xd8\x6a", [0x077b] = "\xb3\xd7",
  [0x077c] = "\xb3\xda", [0x077d] = "\xd8\x75", [0x077e] = "\xb3\xee",
  [0x077f] = "\xd8\x78", [0x0780] = "\xb3\xd8", [0x0781] = "\xd8\x71",
  [0x0782] = "\xb3\xde", [0x0783] = "\xb3\xe4", [0x0784] = "\xb5\xbd",
  [0x0787] = "\xb3\xe2", [0x0788] = "\xd8\x6e", [0x0789] = "\xb3\xef",
  [0x078a] = "\xb3\xdb", [0x078b] = "\xb3\xe3", [0x078c] = "\xd8\x76",
  [0x078d] = "\xdc\xd7", [0x078e] = "\xd8\x7b", [0x078f] = "\xd8\x6f",
  [0x0791] = "\xd8\x66", [0x0792] = "\xd8\x73", [0x0793] = "\xd8\x6d",
  [0x0794] = "\xb3\xe1", [0x0795] = "\xd8\x79", [0x0798] = "\xb3\xdd",
  [0x0799] = "\xb3\xf1", [0x079a] = "\xb3\xea", [0x079c] = "\xb3\xdf",
  [0x079d] = "\xb3\xdc", [0x079f] = "\xb3\xe7", [0x07a1] = "\xd8\x7a",
  [0x07a2] = "\xd8\x6c", [0x07a3] = "\xd8\x72", [0x07a4] = "\xd8\x74",
  [0x07a5] = "\xd8\x68", [0x07a6] = "\xd8\x77", [0x07a7] = "\xb3\xd9",
  [0x07a8] = "\xd8\x67", [0x07aa] = "\xb3\xe0", [0x07ab] = "\xb3\xf0",
  [0x07ac] = "\xb3\xec", [0x07ad] = "\xd8\x69", [0x07ae] = "\xb3\xe6",
  [0x07b1] = "\xb3\xed", [0x07b2] = "\xb3\xe9", [0x07b3] = "\xb3\xe5",
  [0x07b5] = "\xd8\x70", [0x07bb] = "\xb3\xeb", [0x07bf] = "\xdc\xd5",
  [0x07c0] = "\xdc\xd1", [0x07c2] = "\xdc\xe0", [0x07c3] = "\xdc\xca",
  [0x07c4] = "\xdc\xd3", [0x07c5] = "\xb6\xe5", [0x07c6] = "\xb6\xe6",
  [0x07c7] = "\xb6\xde", [0x07c8] = "\xdc\xdc", [0x07c9] = "\xb6\xe8",
  [0x07ca] = "\xdc\xcf", [0x07cb] = "\xdc\xce", [0x07cc] = "\xdc\xcc",
  [0x07cd] = "\xdc\xde", [0x07ce] = "\xb6\xdc", [0x07cf] = "\xdc\xd8",
  [0x07d0] = "\xdc\xcd", [0x07d1] = "\xb6\xdf", [0x07d2] = "\xdc\xd6",
  [0x07d3] = "\xb6\xda", [0x07d4] = "\xdc\xd2", [0x07d5] = "\xdc\xd9",
  [0x07d6] = "\xdc\xdb", [0x07d9] = "\xdc\xdf", [0x07da] = "\xb6\xe3",
  [0x07db] = "\xdc\xcb", [0x07dc] = "\xb6\xdd", [0x07dd] = "\xdc\xd0",
  [0x07df] = "\xb6\xd8", [0x07e1] = "\xb6\xe4", [0x07e2] = "\xdc\xda",
  [0x07e3] = "\xb6\xe0", [0x07e4] = "\xb6\xe1", [0x07e5] = "\xb6\xe7",
  [0x07e6] = "\xb6\xdb", [0x07e7] = "\xa2\x5f", [0x07e8] = "\xb6\xd9",
  [0x07e9] = "\xdc\xd4", [0x07ef] = "\xb6\xe2", [0x07f2] = "\xdc\xdd",
  [0x07f6] = "\xb9\xcd", [0x07f7] = "\xb9\xc8", [0x07f9] = "\xe1\x55",
  [0x07fa] = "\xe1\x51", [0x07fc] = "\xe1\x4b", [0x07fd] = "\xb9\xc2",
  [0x07fe] = "\xb9\xbe", [0x07ff] = "\xe1\x54", [0x0800] = "\xb9\xbf",
  [0x0801] = "\xe1\x4e", [0x0802] = "\xe1\x50", [0x0804] = "\xe1\x53",
  [0x0806] = "\xb9\xc4", [0x0808] = "\xb9\xcb", [0x0809] = "\xb9\xc5",
  [0x080c] = "\xe1\x49", [0x080d] = "\xb9\xc6", [0x080e] = "\xb9\xc7",
  [0x080f] = "\xe1\x4c", [0x0810] = "\xb9\xcc", [0x0812] = "\xe1\x4a",
  [0x0813] = "\xe1\x4f", [0x0814] = "\xb9\xc3", [0x0815] = "\xe1\x48",
  [0x0816] = "\xb9\xc9", [0x0817] = "\xb9\xc1", [0x081b] = "\xb9\xc0",
  [0x081c] = "\xe1\x4d", [0x081d] = "\xe1\x52", [0x081f] = "\xb9\xca",
  [0x0827] = "\xe1\x47", [0x0829] = "\xbc\x4d", [0x082a] = "\xe5\x47",
  [0x082c] = "\xe5\x44", [0x082e] = "\xbc\x47", [0x082f] = "\xbc\x53",
  [0x0830] = "\xbc\x54", [0x0832] = "\xbc\x4a", [0x0833] = "\xe5\x42",
  [0x0834] = "\xbc\x4c", [0x0835] = "\xe4\xf9", [0x0836] = "\xbc\x52",
  [0x0838] = "\xe5\x46", [0x0839] = "\xbc\x49", [0x083a] = "\xe5\x48",
  [0x083b] = "\xbc\x48", [0x083d] = "\xe5\x43", [0x083e] = "\xe5\x45",
  [0x083f] = "\xbc\x4b", [0x0840] = "\xe5\x41", [0x0841] = "\xe4\xfa",
  [0x0842] = "\xe4\xf7", [0x0845] = "\xd8\x6b", [0x0846] = "\xe4\xfd",
  [0x0848] = "\xe4\xf6", [0x0849] = "\xe4\xfc", [0x084a] = "\xe4\xfb",
  [0x084c] = "\xe4\xf8", [0x084e] = "\xbc\x4f", [0x0853] = "\xbc\x4e",
  [0x0857] = "\xbc\x50", [0x0858] = "\xe4\xfe", [0x0859] = "\xbe\xb2",
  [0x085a] = "\xe5\x40", [0x085e] = "\xe9\x45", [0x0860] = "\xe8\xfd",
  [0x0862] = "\xbe\xbe", [0x0863] = "\xe9\x42", [0x0864] = "\xbe\xb6",
  [0x0865] = "\xbe\xba", [0x0866] = "\xe9\x41", [0x0868] = "\xbe\xb9",
  [0x0869] = "\xbe\xb5", [0x086a] = "\xbe\xb8", [0x086b] = "\xbe\xb3",
  [0x086c] = "\xbe\xbd", [0x086d] = "\xe9\x43", [0x086e] = "\xe8\xfe",
  [0x086f] = "\xbe\xbc", [0x0870] = "\xe8\xfc", [0x0871] = "\xbe\xbb",
  [0x0872] = "\xe9\x44", [0x0873] = "\xe9\x40", [0x0874] = "\xbc\x51",
  [0x0876] = "\xbe\xbf", [0x0877] = "\xe9\x46", [0x0878] = "\xbe\xb7",
  [0x0879] = "\xbe\xb4", [0x087e] = "\xec\xc6", [0x087f] = "\xec\xc8",
  [0x0880] = "\xc0\x7b", [0x0881] = "\xec\xc9", [0x0882] = "\xec\xc7",
  [0x0883] = "\xec\xc5", [0x0884] = "\xec\xc4", [0x0885] = "\xc0\x7d",
  [0x0886] = "\xec\xc3", [0x0887] = "\xc0\x7e", [0x088c] = "\xec\xc1",
  [0x088d] = "\xec\xc2", [0x088e] = "\xc0\x7a", [0x088f] = "\xc0\xa1",
  [0x0890] = "\xc0\x7c", [0x0893] = "\xec\xc0", [0x0895] = "\xc2\x50",
  [0x0897] = "\xef\xbc", [0x0898] = "\xef\xba", [0x0899] = "\xef\xbf",
  [0x089a] = "\xef\xbd", [0x089c] = "\xef\xbb", [0x089d] = "\xef\xbe",
  [0x08a5] = "\xc3\x60", [0x08a6] = "\xf1\xf2", [0x08a7] = "\xf1\xf3",
  [0x08a8] = "\xc4\x56", [0x08aa] = "\xf1\xf4", [0x08ab] = "\xf1\xf0",
  [0x08ac] = "\xf1\xf5", [0x08ad] = "\xf1\xf1", [0x08ae] = "\xc2\x51",
  [0x08b2] = "\xf3\xfe", [0x08b3] = "\xf4\x41", [0x08b4] = "\xc4\x59",
  [0x08b5] = "\xf4\x40", [0x08b6] = "\xc4\x58", [0x08b7] = "\xc4\x57",
  [0x08bc] = "\xc4\x5a", [0x08bd] = "\xf5\xc5", [0x08be] = "\xf5\xc6",
  [0x08c0] = "\xc4\xda", [0x08c1] = "\xc4\xd9", [0x08c2] = "\xc4\xdb",
  [0x08c3] = "\xf5\xc4", [0x08c5] = "\xf6\xd8", [0x08c6] = "\xf6\xd7",
  [0x08c8] = "\xc5\x6d", [0x08c9] = "\xc5\x6f", [0x08ca] = "\xc5\x6e",
  [0x08cb] = "\xf6\xd9", [0x08cc] = "\xc5\xc8", [0x08cd] = "\xf8\xa6",
  [0x08d1] = "\xc5\xf1", [0x08d3] = "\xf8\xa5", [0x08d4] = "\xf8\xee",
  [0x08d7] = "\xc9\x49", [0x08da] = "\xa5\x7d", [0x08db] = "\xa5\x7c",
  [0x08dd] = "\xa6\x5f", [0x08de] = "\xa6\x5e", [0x08df] = "\xc9\xc7",
  [0x08e0] = "\xa6\x5d", [0x08e1] = "\xc9\xc6", [0x08e4] = "\xa7\x79",
  [0x08e5] = "\xca\xa9", [0x08e7] = "\xca\xa8", [0x08ea] = "\xa7\x77",
  [0x08eb] = "\xa7\x7a", [0x08ee] = "\xca\xa7", [0x08f0] = "\xa7\x78",
  [0x08f7] = "\xcb\xf0", [0x08f9] = "\xcb\xf1", [0x08fa] = "\xa9\x54",
  [0x08ff] = "\xab\xaa", [0x0901] = "\xd1\x48", [0x0902] = "\xd1\x49",
  [0x0903] = "\xae\x45", [0x0904] = "\xae\x46", [0x0907] = "\xd4\xac",
  [0x0908] = "\xb0\xe9", [0x0909] = "\xb0\xeb", [0x090a] = "\xd4\xab",
  [0x090b] = "\xb0\xea", [0x090c] = "\xd8\x7c", [0x090d] = "\xb3\xf2",
  [0x0912] = "\xb6\xe9", [0x0913] = "\xb6\xea", [0x0914] = "\xdc\xe1",
  [0x0916] = "\xb9\xcf", [0x0918] = "\xb9\xce", [0x091a] = "\xe5\x49",
  [0x091b] = "\xe9\x48", [0x091c] = "\xe9\x47", [0x091e] = "\xf9\x6b",
  [0x091f] = "\xa4\x67", [0x0920] = "\xc9\x59", [0x0922] = "\xc9\x6e",
  [0x0923] = "\xc9\x6f", [0x0928] = "\xa6\x62", [0x0929] = "\xa6\x66",
  [0x092a] = "\xc9\xc9", [0x092c] = "\xa6\x64", [0x092d] = "\xa6\x63",
  [0x092e] = "\xc9\xc8", [0x092f] = "\xa6\x65", [0x0930] = "\xa6\x61",
  [0x0933] = "\xa6\x60", [0x0934] = "\xc9\xca", [0x093b] = "\xa7\xa6",
  [0x093e] = "\xa7\xa3", [0x0940] = "\xa7\x7d", [0x0941] = "\xca\xaa",
  [0x0945] = "\xca\xab", [0x0947] = "\xa7\xa1", [0x0949] = "\xca\xad",
  [0x094a] = "\xa7\x7b", [0x094b] = "\xca\xae", [0x094c] = "\xca\xac",
  [0x094d] = "\xa7\x7e", [0x094e] = "\xa7\xa2", [0x094f] = "\xa7\xa5",
  [0x0950] = "\xa7\xa4", [0x0951] = "\xa7\x7c", [0x0952] = "\xca\xaf",
  [0x0961] = "\xa9\x59", [0x0962] = "\xcb\xfe", [0x0964] = "\xa9\x5b",
  [0x0966] = "\xa9\x5a", [0x0968] = "\xcc\x40", [0x0969] = "\xa9\x58",
  [0x096a] = "\xa9\x57", [0x096b] = "\xcb\xf5", [0x096d] = "\xcb\xf4",
  [0x096f] = "\xcb\xf2", [0x0970] = "\xcb\xf7", [0x0971] = "\xcb\xf6",
  [0x0972] = "\xcb\xf3", [0x0973] = "\xcb\xfc", [0x0974] = "\xcb\xfd",
  [0x0975] = "\xcb\xfa", [0x0976] = "\xcb\xf8", [0x0977] = "\xa9\x56",
  [0x097b] = "\xcb\xfb", [0x097c] = "\xa9\x5c", [0x097d] = "\xcc\x41",
  [0x0980] = "\xcb\xf9", [0x0982] = "\xab\xab", [0x0983] = "\xa9\x55",
  [0x098b] = "\xab\xac", [0x098c] = "\xce\x54", [0x098f] = "\xce\x5a",
  [0x0993] = "\xab\xb2", [0x0994] = "\xce\x58", [0x0995] = "\xce\x5e",
  [0x0997] = "\xce\x55", [0x0998] = "\xce\x59", [0x0999] = "\xce\x5b",
  [0x099a] = "\xce\x5d", [0x099b] = "\xce\x57", [0x099d] = "\xce\x56",
  [0x099e] = "\xce\x51", [0x099f] = "\xce\x52", [0x09a0] = "\xab\xad",
  [0x09a2] = "\xab\xaf", [0x09a3] = "\xab\xae", [0x09a4] = "\xce\x53",
  [0x09a5] = "\xce\x5c", [0x09ae] = "\xab\xb1", [0x09b5] = "\xce\x50",
  [0x09b6] = "\xd1\x53", [0x09b8] = "\xd1\x52", [0x09b9] = "\xd1\x57",
  [0x09ba] = "\xd1\x4e", [0x09bc] = "\xd1\x51", [0x09bd] = "\xd1\x50",
  [0x09bf] = "\xd1\x54", [0x09c1] = "\xd1\x58", [0x09c2] = "\xae\x47",
  [0x09c3] = "\xae\x4a", [0x09c6] = "\xd1\x4f", [0x09c7] = "\xd1\x55",
  [0x09cb] = "\xae\x49", [0x09cc] = "\xd1\x4a", [0x09ce] = "\xab\xb0",
  [0x09cf] = "\xd4\xba", [0x09d0] = "\xd1\x56", [0x09d2] = "\xd1\x4d",
  [0x09d4] = "\xae\x48", [0x09d5] = "\xd1\x4c", [0x09dc] = "\xd4\xb1",
  [0x09df] = "\xb0\xec", [0x09e0] = "\xb0\xf0", [0x09e1] = "\xd4\xc1",
  [0x09e2] = "\xd4\xaf", [0x09e3] = "\xd4\xbd", [0x09e4] = "\xb0\xf1",
  [0x09e5] = "\xd4\xbf", [0x09e7] = "\xd4\xc5", [0x09e9] = "\xd4\xc9",
  [0x09ec] = "\xd4\xc0", [0x09ed] = "\xd4\xb4", [0x09ee] = "\xd4\xbc",
  [0x09f0] = "\xd4\xca", [0x09f1] = "\xd4\xc8", [0x09f2] = "\xd4\xbe",
  [0x09f3] = "\xd4\xb9", [0x09f4] = "\xd4\xb2", [0x09f5] = "\xd8\xa6",
  [0x09f6] = "\xd4\xb0", [0x09f7] = "\xb0\xf5", [0x09f8] = "\xd4\xb7",
  [0x09f9] = "\xb0\xf6", [0x09fa] = "\xb0\xf2", [0x09fb] = "\xd4\xad",
  [0x09fc] = "\xd4\xc3", [0x09fd] = "\xd4\xb5", [0x0a00] = "\xd4\xb3",
  [0x0a01] = "\xd4\xc6", [0x0a02] = "\xb0\xf3", [0x0a04] = "\xd4\xcc",
  [0x0a05] = "\xb0\xed", [0x0a06] = "\xb0\xef", [0x0a07] = "\xd4\xbb",
  [0x0a08] = "\xd4\xb6", [0x0a09] = "\xae\x4b", [0x0a0a] = "\xb0\xee",
  [0x0a0b] = "\xd4\xb8", [0x0a0c] = "\xd4\xc7", [0x0a0d] = "\xd4\xcb",
  [0x0a0e] = "\xd4\xc2", [0x0a10] = "\xd4\xc4", [0x0a14] = "\xd4\xae",
  [0x0a19] = "\xd8\xa1", [0x0a1b] = "\xd8\xaa", [0x0a1c] = "\xd8\xa9",
  [0x0a1d] = "\xb3\xfa", [0x0a1e] = "\xd8\xa2", [0x0a20] = "\xb3\xfb",
  [0x0a21] = "\xb3\xf9", [0x0a23] = "\xd8\xa4", [0x0a24] = "\xb3\xf6",
  [0x0a25] = "\xd8\xa8", [0x0a27] = "\xd8\xa3", [0x0a28] = "\xd8\xa5",
  [0x0a29] = "\xd8\x7d", [0x0a2a] = "\xb3\xf4", [0x0a2c] = "\xd8\xb2",
  [0x0a2d] = "\xd8\xb1", [0x0a2e] = "\xd8\xae", [0x0a2f] = "\xb3\xf3",
  [0x0a30] = "\xb3\xf7", [0x0a31] = "\xb3\xf8", [0x0a32] = "\xd1\x4b",
  [0x0a33] = "\xd8\xab", [0x0a34] = "\xb3\xf5", [0x0a35] = "\xb0\xf4",
  [0x0a36] = "\xd8\xad", [0x0a37] = "\xd8\x7e", [0x0a38] = "\xd8\xb0",
  [0x0a39] = "\xd8\xaf", [0x0a3b] = "\xd8\xb3", [0x0a3d] = "\xdc\xef",
  [0x0a3f] = "\xd8\xac", [0x0a48] = "\xd8\xa7", [0x0a49] = "\xdc\xe7",
  [0x0a4a] = "\xb6\xf4", [0x0a4b] = "\xb6\xf7", [0x0a4c] = "\xb6\xf2",
  [0x0a4d] = "\xdc\xe6", [0x0a4e] = "\xdc\xea", [0x0a4f] = "\xdc\xe5",
  [0x0a51] = "\xb6\xec", [0x0a52] = "\xb6\xf6", [0x0a53] = "\xdc\xe2",
  [0x0a54] = "\xb6\xf0", [0x0a55] = "\xdc\xe9", [0x0a57] = "\xb6\xee",
  [0x0a58] = "\xb6\xed", [0x0a59] = "\xdc\xec", [0x0a5a] = "\xb6\xef",
  [0x0a5b] = "\xdc\xee", [0x0a5d] = "\xdc\xeb", [0x0a5e] = "\xb6\xeb",
  [0x0a62] = "\xb6\xf5", [0x0a63] = "\xdc\xf0", [0x0a64] = "\xdc\xe4",
  [0x0a65] = "\xdc\xed", [0x0a68] = "\xdc\xe3", [0x0a6b] = "\xb6\xf1",
  [0x0a6d] = "\xb6\xf3", [0x0a6f] = "\xdc\xe8", [0x0a71] = "\xdc\xf1",
  [0x0a74] = "\xe1\x5d", [0x0a75] = "\xb9\xd0", [0x0a76] = "\xe1\x63",
  [0x0a79] = "\xb9\xd5", [0x0a7a] = "\xe1\x5f", [0x0a7b] = "\xe1\x66",
  [0x0a7c] = "\xe1\x57", [0x0a7d] = "\xb9\xd7", [0x0a7e] = "\xb9\xd1",
  [0x0a7f] = "\xe1\x5c", [0x0a80] = "\xbc\x55", [0x0a81] = "\xe1\x5b",
  [0x0a82] = "\xe1\x64", [0x0a83] = "\xb9\xd2", [0x0a85] = "\xb9\xd6",
  [0x0a86] = "\xe1\x5a", [0x0a87] = "\xe1\x60", [0x0a88] = "\xe1\x65",
  [0x0a89] = "\xe1\x56", [0x0a8a] = "\xb9\xd4", [0x0a8b] = "\xe1\x5e",
  [0x0a8e] = "\xe1\x62", [0x0a8f] = "\xe1\x68", [0x0a90] = "\xe1\x58",
  [0x0a91] = "\xe1\x61", [0x0a93] = "\xb9\xd3", [0x0a94] = "\xe1\x67",
  [0x0a98] = "\xe1\x59", [0x0a9c] = "\xbc\x59", [0x0a9d] = "\xe5\x4b",
  [0x0a9e] = "\xbc\x57", [0x0a9f] = "\xbc\x56", [0x0aa0] = "\xe5\x4d",
  [0x0aa1] = "\xe5\x52", [0x0aa3] = "\xe5\x4e", [0x0aa5] = "\xe5\x51",
  [0x0aa6] = "\xbc\x5c", [0x0aa8] = "\xbe\xa5", [0x0aa9] = "\xbc\x5b",
  [0x0aab] = "\xe5\x4a", [0x0aac] = "\xe5\x50", [0x0aae] = "\xbc\x5a",
  [0x0aaf] = "\xe5\x4f", [0x0ab1] = "\xe5\x4c", [0x0ab3] = "\xbc\x58",
  [0x0aba] = "\xe9\x4d", [0x0abb] = "\xf9\xd9", [0x0abc] = "\xe9\x4f",
  [0x0abd] = "\xe9\x4a", [0x0abe] = "\xbe\xc1", [0x0abf] = "\xe9\x4c",
  [0x0ac1] = "\xbe\xc0", [0x0ac2] = "\xe9\x4e", [0x0ac5] = "\xbe\xc3",
  [0x0ac6] = "\xe9\x50", [0x0ac7] = "\xbe\xc2", [0x0ac8] = "\xe9\x49",
  [0x0ac9] = "\xe9\x4b", [0x0ace] = "\xc0\xa5", [0x0acf] = "\xec\xcc",
  [0x0ad1] = "\xc0\xa4", [0x0ad2] = "\xec\xcd", [0x0ad3] = "\xc0\xa3",
  [0x0ad4] = "\xec\xcb", [0x0ad5] = "\xc0\xa2", [0x0ad6] = "\xec\xca",
  [0x0ad8] = "\xc2\x53", [0x0ad9] = "\xc2\x52", [0x0ada] = "\xf1\xf6",
  [0x0adb] = "\xf1\xf8", [0x0add] = "\xf1\xf7", [0x0ade] = "\xc3\x61",
  [0x0adf] = "\xc3\x62", [0x0ae2] = "\xc3\x63", [0x0ae3] = "\xf4\x42",
  [0x0ae4] = "\xc4\x5b", [0x0ae7] = "\xf7\xd3", [0x0ae8] = "\xf7\xd2",
  [0x0ae9] = "\xc5\xf2", [0x0aeb] = "\xa4\x68", [0x0aec] = "\xa4\xd0",
  [0x0aef] = "\xa7\xa7", [0x0af4] = "\xce\x5f", [0x0af9] = "\xb3\xfc",
  [0x0afa] = "\xb3\xfd", [0x0afc] = "\xdc\xf2", [0x0afd] = "\xb9\xd8",
  [0x0afe] = "\xe1\x69", [0x0aff] = "\xe5\x53", [0x0b03] = "\xc9\x5a",
  [0x0b06] = "\xca\xb0", [0x0b0c] = "\xcc\x42", [0x0b0d] = "\xce\x60",
  [0x0b0e] = "\xd1\x59", [0x0b0f] = "\xae\x4c", [0x0b12] = "\xf1\xf9",
  [0x0b14] = "\xc4\xdc", [0x0b15] = "\xa4\x69", [0x0b16] = "\xa5\x7e",
  [0x0b17] = "\xc9\x70", [0x0b19] = "\xa6\x67", [0x0b1a] = "\xa6\x68",
  [0x0b1c] = "\xa9\x5d", [0x0b20] = "\xb0\xf7", [0x0b22] = "\xb9\xda",
  [0x0b24] = "\xb9\xdb", [0x0b25] = "\xb9\xd9", [0x0b27] = "\xa4\x6a",
  [0x0b29] = "\xa4\xd1", [0x0b2a] = "\xa4\xd3", [0x0b2b] = "\xa4\xd2",
  [0x0b2c] = "\xc9\x5b", [0x0b2d] = "\xa4\xd4", [0x0b2e] = "\xa5\xa1",
  [0x0b2f] = "\xc9\x71", [0x0b31] = "\xa5\xa2", [0x0b37] = "\xa6\x69",
  [0x0b38] = "\xa6\x6a", [0x0b3c] = "\xc9\xcb", [0x0b3e] = "\xa7\xa8",
  [0x0b40] = "\xca\xb1", [0x0b44] = "\xa9\x61", [0x0b45] = "\xcc\x43",
  [0x0b47] = "\xa9\x5f", [0x0b48] = "\xa9\x60", [0x0b49] = "\xa9\x5e",
  [0x0b4a] = "\xd1\x5a", [0x0b4e] = "\xab\xb6", [0x0b4f] = "\xab\xb5",
  [0x0b50] = "\xab\xb7", [0x0b51] = "\xab\xb4", [0x0b53] = "\xce\x61",
  [0x0b54] = "\xa9\x62", [0x0b55] = "\xab\xb3", [0x0b57] = "\xae\x4d",
  [0x0b58] = "\xae\x4e", [0x0b5a] = "\xae\x4f", [0x0b5c] = "\xd4\xcd",
  [0x0b60] = "\xb3\xfe", [0x0b61] = "\xd8\xb4", [0x0b62] = "\xb0\xf8",
  [0x0b67] = "\xb6\xf8", [0x0b69] = "\xb9\xdd", [0x0b6a] = "\xb9\xdc",
  [0x0b6b] = "\xe1\x6a", [0x0b6d] = "\xbc\x5d", [0x0b6e] = "\xbe\xc4",
  [0x0b70] = "\xef\xc0", [0x0b71] = "\xf6\xda", [0x0b72] = "\xf7\xd4",
  [0x0b73] = "\xa4\x6b", [0x0b74] = "\xa5\xa3", [0x0b76] = "\xa5\xa4",
  [0x0b77] = "\xc9\xd1", [0x0b78] = "\xa6\x6c", [0x0b79] = "\xa6\x6f",
  [0x0b7b] = "\xc9\xcf", [0x0b7c] = "\xc9\xcd", [0x0b7d] = "\xa6\x6e",
  [0x0b7e] = "\xc9\xd0", [0x0b7f] = "\xc9\xd2", [0x0b80] = "\xc9\xcc",
  [0x0b81] = "\xa6\x71", [0x0b82] = "\xa6\x70", [0x0b83] = "\xa6\x6d",
  [0x0b84] = "\xa6\x6b", [0x0b85] = "\xc9\xce", [0x0b8a] = "\xa7\xb3",
  [0x0b8d] = "\xa7\xb0", [0x0b8e] = "\xca\xb6", [0x0b8f] = "\xca\xb9",
  [0x0b90] = "\xca\xb8", [0x0b92] = "\xa7\xaa", [0x0b93] = "\xa7\xb2",
  [0x0b96] = "\xa7\xaf", [0x0b97] = "\xca\xb5", [0x0b98] = "\xca\xb3",
  [0x0b99] = "\xa7\xae", [0x0b9d] = "\xa7\xa9", [0x0b9e] = "\xa7\xac",
  [0x0ba0] = "\xca\xb4", [0x0ba1] = "\xca\xbb", [0x0ba2] = "\xca\xb7",
  [0x0ba3] = "\xa7\xad", [0x0ba4] = "\xa7\xb1", [0x0ba5] = "\xa7\xb4",
  [0x0ba6] = "\xca\xb2", [0x0ba7] = "\xca\xba", [0x0ba8] = "\xa7\xab",
  [0x0bae] = "\xa9\x67", [0x0baf] = "\xa9\x6f", [0x0bb1] = "\xcc\x4f",
  [0x0bb2] = "\xcc\x48", [0x0bb3] = "\xa9\x70", [0x0bb4] = "\xcc\x53",
  [0x0bb5] = "\xcc\x44", [0x0bb6] = "\xcc\x4b", [0x0bb9] = "\xa9\x66",
  [0x0bba] = "\xcc\x45", [0x0bbb] = "\xa9\x64", [0x0bbc] = "\xcc\x4c",
  [0x0bbd] = "\xcc\x50", [0x0bbe] = "\xa9\x63", [0x0bc0] = "\xcc\x51",
  [0x0bc1] = "\xcc\x4a", [0x0bc3] = "\xcc\x4d", [0x0bc5] = "\xa9\x72",
  [0x0bc6] = "\xa9\x69", [0x0bc7] = "\xcc\x54", [0x0bc8] = "\xcc\x52",
  [0x0bca] = "\xa9\x6e", [0x0bcb] = "\xa9\x6c", [0x0bcc] = "\xcc\x49",
  [0x0bcd] = "\xa9\x6b", [0x0bce] = "\xcc\x47", [0x0bcf] = "\xcc\x46",
  [0x0bd0] = "\xa9\x6a", [0x0bd1] = "\xa9\x68", [0x0bd2] = "\xa9\x71",
  [0x0bd3] = "\xa9\x6d", [0x0bd4] = "\xa9\x65", [0x0bd6] = "\xcc\x4e",
  [0x0bd8] = "\xab\xb9", [0x0bda] = "\xab\xc0", [0x0bdb] = "\xce\x6f",
  [0x0bdc] = "\xab\xb8", [0x0bdd] = "\xce\x67", [0x0bde] = "\xce\x63",
  [0x0be0] = "\xce\x73", [0x0be1] = "\xce\x62", [0x0be3] = "\xab\xbb",
  [0x0be4] = "\xce\x6c", [0x0be5] = "\xab\xbe", [0x0be6] = "\xab\xc1",
  [0x0be8] = "\xab\xbc", [0x0be9] = "\xce\x70", [0x0bea] = "\xab\xbf",
  [0x0bec] = "\xae\x56", [0x0bed] = "\xce\x76", [0x0bee] = "\xce\x64",
  [0x0bf1] = "\xce\x66", [0x0bf2] = "\xce\x6d", [0x0bf3] = "\xce\x71",
  [0x0bf4] = "\xce\x75", [0x0bf5] = "\xce\x72", [0x0bf6] = "\xce\x6b",
  [0x0bf7] = "\xce\x6e", [0x0bfa] = "\xce\x68", [0x0bfb] = "\xab\xc3",
  [0x0bfc] = "\xce\x6a", [0x0bfd] = "\xce\x69", [0x0bfe] = "\xce\x74",
  [0x0bff] = "\xab\xba", [0x0c00] = "\xce\x65", [0x0c01] = "\xab\xc2",
  [0x0c03] = "\xab\xbd", [0x0c09] = "\xae\x5c", [0x0c0a] = "\xd1\x62",
  [0x0c0c] = "\xae\x5b", [0x0c0f] = "\xd1\x60", [0x0c11] = "\xae\x50",
  [0x0c13] = "\xae\x55", [0x0c15] = "\xd1\x5f", [0x0c16] = "\xd1\x5c",
  [0x0c17] = "\xd1\x61", [0x0c18] = "\xae\x51", [0x0c19] = "\xd1\x5b",
  [0x0c1b] = "\xae\x54", [0x0c1c] = "\xae\x52", [0x0c1e] = "\xd1\x63",
  [0x0c1f] = "\xae\x53", [0x0c20] = "\xae\x57", [0x0c23] = "\xae\x58",
  [0x0c25] = "\xae\x5a", [0x0c29] = "\xae\x59", [0x0c2d] = "\xd1\x5d",
  [0x0c2e] = "\xd1\x5e", [0x0c33] = "\xd1\x64", [0x0c35] = "\xd4\xd4",
  [0x0c36] = "\xb0\xf9", [0x0c37] = "\xd8\xc2", [0x0c38] = "\xd4\xd3",
  [0x0c39] = "\xd4\xe6", [0x0c3c] = "\xb1\x40", [0x0c3e] = "\xd4\xe4",
  [0x0c40] = "\xb0\xfe", [0x0c41] = "\xb0\xfa", [0x0c42] = "\xd4\xed",
  [0x0c43] = "\xd4\xdd", [0x0c44] = "\xd4\xe0", [0x0c46] = "\xb1\x43",
  [0x0c47] = "\xd4\xea", [0x0c48] = "\xd4\xe2", [0x0c49] = "\xb0\xfb",
  [0x0c4a] = "\xb1\x44", [0x0c4c] = "\xd4\xe7", [0x0c4d] = "\xd4\xe5",
  [0x0c50] = "\xd4\xd6", [0x0c51] = "\xd4\xeb", [0x0c52] = "\xd4\xdf",
  [0x0c53] = "\xd4\xda", [0x0c55] = "\xd4\xd0", [0x0c56] = "\xd4\xec",
  [0x0c57] = "\xd4\xdc", [0x0c58] = "\xd4\xcf", [0x0c5a] = "\xb1\x42",
  [0x0c5b] = "\xd4\xe1", [0x0c5c] = "\xd4\xee", [0x0c5d] = "\xd4\xde",
  [0x0c5e] = "\xd4\xd2", [0x0c5f] = "\xd4\xd7", [0x0c60] = "\xd4\xce",
  [0x0c62] = "\xb1\x41", [0x0c64] = "\xd4\xdb", [0x0c65] = "\xd4\xd8",
  [0x0c66] = "\xb0\xfc", [0x0c67] = "\xd4\xd1", [0x0c69] = "\xd4\xe9",
  [0x0c6a] = "\xb0\xfd", [0x0c6c] = "\xd4\xd9", [0x0c6d] = "\xd4\xd5",
  [0x0c70] = "\xd4\xe8", [0x0c77] = "\xb4\x40", [0x0c78] = "\xd8\xbb",
  [0x0c7a] = "\xd8\xb8", [0x0c7b] = "\xd8\xc9", [0x0c7c] = "\xd8\xbd",
  [0x0c7d] = "\xd8\xca", [0x0c7f] = "\xb4\x42", [0x0c83] = "\xd8\xc6",
  [0x0c84] = "\xd8\xc3", [0x0c8a] = "\xd8\xc4", [0x0c8b] = "\xd8\xc7",
  [0x0c8c] = "\xd8\xcb", [0x0c8e] = "\xd4\xe3", [0x0c8f] = "\xd8\xcd",
  [0x0c90] = "\xdd\x47", [0x0c92] = "\xb4\x43", [0x0c93] = "\xd8\xce",
  [0x0c94] = "\xd8\xb6", [0x0c95] = "\xd8\xc0", [0x0c97] = "\xd8\xc5",
  [0x0c9a] = "\xb4\x41", [0x0c9b] = "\xb4\x44", [0x0c9c] = "\xd8\xcc",
  [0x0c9d] = "\xd8\xcf", [0x0c9e] = "\xd8\xba", [0x0c9f] = "\xd8\xb7",
  [0x0ca2] = "\xd8\xb9", [0x0ca5] = "\xd8\xbe", [0x0ca6] = "\xd8\xbc",
  [0x0ca7] = "\xb4\x45", [0x0ca9] = "\xd8\xc8", [0x0cac] = "\xd8\xbf",
  [0x0cae] = "\xd8\xc1", [0x0caf] = "\xd8\xb5", [0x0cb0] = "\xdc\xfa",
  [0x0cb1] = "\xdc\xf8", [0x0cb2] = "\xb7\x42", [0x0cb3] = "\xb7\x40",
  [0x0cb4] = "\xdd\x43", [0x0cb5] = "\xdc\xf9", [0x0cb6] = "\xdd\x44",
  [0x0cb7] = "\xdd\x40", [0x0cb8] = "\xdc\xf7", [0x0cb9] = "\xdd\x46",
  [0x0cba] = "\xdc\xf6", [0x0cbb] = "\xdc\xfd", [0x0cbc] = "\xb6\xfe",
  [0x0cbd] = "\xb6\xfd", [0x0cbe] = "\xb6\xfc", [0x0cbf] = "\xdc\xfb",
  [0x0cc0] = "\xdd\x41", [0x0cc1] = "\xb6\xf9", [0x0cc2] = "\xb7\x41",
  [0x0cc4] = "\xdc\xf4", [0x0cc6] = "\xdc\xfe", [0x0cc7] = "\xdc\xf3",
  [0x0cc8] = "\xdc\xfc", [0x0cc9] = "\xb6\xfa", [0x0cca] = "\xdd\x42",
  [0x0ccb] = "\xdc\xf5", [0x0ccc] = "\xb6\xfb", [0x0ccd] = "\xdd\x45",
  [0x0cd5] = "\xe1\x6e", [0x0cd6] = "\xb9\xe2", [0x0cd7] = "\xb9\xe1",
  [0x0cd8] = "\xb9\xe3", [0x0cd9] = "\xe1\x7a", [0x0cda] = "\xe1\x70",
  [0x0cdb] = "\xe1\x76", [0x0cdc] = "\xe1\x6b", [0x0cdd] = "\xe1\x79",
  [0x0cde] = "\xe1\x78", [0x0cdf] = "\xe1\x7c", [0x0ce0] = "\xe1\x75",
  [0x0ce1] = "\xb9\xde", [0x0ce2] = "\xe1\x74", [0x0ce3] = "\xb9\xe4",
  [0x0ce5] = "\xe1\x6d", [0x0ce6] = "\xb9\xdf", [0x0ce8] = "\xe1\x7b",
  [0x0ce9] = "\xb9\xe0", [0x0cea] = "\xe1\x6f", [0x0ceb] = "\xe1\x72",
  [0x0cec] = "\xe1\x77", [0x0ced] = "\xe1\x71", [0x0cee] = "\xe1\x6c",
  [0x0cf3] = "\xe1\x73", [0x0cf4] = "\xe5\x55", [0x0cf5] = "\xbc\x61",
  [0x0cf6] = "\xe5\x58", [0x0cf7] = "\xe5\x57", [0x0cf8] = "\xe5\x5a",
  [0x0cf9] = "\xe5\x5c", [0x0cfa] = "\xf9\xdc", [0x0cfb] = "\xbc\x5f",
  [0x0cfd] = "\xe5\x56", [0x0cff] = "\xe5\x54", [0x0d01] = "\xe5\x5d",
  [0x0d02] = "\xe5\x5b", [0x0d03] = "\xe5\x59", [0x0d05] = "\xe5\x5f",
  [0x0d07] = "\xe5\x5e", [0x0d08] = "\xbc\x63", [0x0d09] = "\xbc\x5e",
  [0x0d0b] = "\xbc\x60", [0x0d0c] = "\xbc\x62", [0x0d0f] = "\xe5\x60",
  [0x0d10] = "\xe9\x57", [0x0d13] = "\xe9\x56", [0x0d14] = "\xe9\x55",
  [0x0d16] = "\xe9\x58", [0x0d17] = "\xe9\x51", [0x0d19] = "\xe9\x52",
  [0x0d1a] = "\xe9\x5a", [0x0d1b] = "\xe9\x53", [0x0d1d] = "\xbe\xc5",
  [0x0d1e] = "\xe9\x5c", [0x0d20] = "\xe9\x5b", [0x0d21] = "\xe9\x54",
  [0x0d23] = "\xec\xd1", [0x0d24] = "\xc0\xa8", [0x0d25] = "\xec\xcf",
  [0x0d26] = "\xec\xd4", [0x0d27] = "\xec\xd3", [0x0d28] = "\xe9\x59",
  [0x0d2a] = "\xc0\xa7", [0x0d2c] = "\xec\xd2", [0x0d2d] = "\xec\xce",
  [0x0d2e] = "\xec\xd6", [0x0d2f] = "\xec\xd5", [0x0d30] = "\xc0\xa6",
  [0x0d32] = "\xec\xd0", [0x0d34] = "\xbe\xc6", [0x0d38] = "\xc2\x54",
  [0x0d3c] = "\xef\xc1", [0x0d3d] = "\xf1\xfa", [0x0d3e] = "\xf1\xfb",
  [0x0d3f] = "\xf1\xfc", [0x0d40] = "\xc4\x5c", [0x0d43] = "\xc4\x5d",
  [0x0d45] = "\xf4\x43", [0x0d47] = "\xf5\xc8", [0x0d48] = "\xf5\xc7",
  [0x0d4b] = "\xf6\xdb", [0x0d4c] = "\xf6\xdc", [0x0d4d] = "\xf7\xd5",
  [0x0d4e] = "\xf8\xa7", [0x0d50] = "\xa4\x6c", [0x0d51] = "\xa4\x6d",
  [0x0d53] = "\xa4\x6e", [0x0d54] = "\xa4\xd5", [0x0d55] = "\xa5\xa5",
  [0x0d56] = "\xc9\xd3", [0x0d57] = "\xa6\x72", [0x0d58] = "\xa6\x73",
  [0x0d5a] = "\xa7\xb7", [0x0d5b] = "\xa7\xb8", [0x0d5c] = "\xa7\xb6",
  [0x0d5d] = "\xa7\xb5", [0x0d5f] = "\xa9\x73", [0x0d62] = "\xcc\x55",
  [0x0d63] = "\xa9\x75", [0x0d64] = "\xa9\x74", [0x0d65] = "\xcc\x56",
  [0x0d69] = "\xab\xc4", [0x0d6b] = "\xae\x5d", [0x0d6c] = "\xd1\x65",
  [0x0d6e] = "\xd4\xf0", [0x0d70] = "\xb1\x45", [0x0d71] = "\xb4\x47",
  [0x0d72] = "\xd4\xef", [0x0d73] = "\xb4\x46", [0x0d75] = "\xb9\xe5",
  [0x0d77] = "\xe1\x7d", [0x0d78] = "\xbe\xc7", [0x0d7a] = "\xc0\xa9",
  [0x0d7b] = "\xec\xd7", [0x0d7d] = "\xc4\x5e", [0x0d7f] = "\xc5\x70",
  [0x0d81] = "\xc9\x72", [0x0d83] = "\xa5\xa6", [0x0d84] = "\xc9\x73",
  [0x0d85] = "\xa6\x76", [0x0d87] = "\xa6\x74", [0x0d88] = "\xa6\x75",
  [0x0d89] = "\xa6\x77", [0x0d8b] = "\xa7\xba", [0x0d8c] = "\xa7\xb9",
  [0x0d8e] = "\xca\xbc", [0x0d8f] = "\xa7\xbb", [0x0d92] = "\xca\xbd",
  [0x0d93] = "\xcc\x57", [0x0d95] = "\xcc\x58", [0x0d97] = "\xa9\x76",
  [0x0d98] = "\xa9\x78", [0x0d99] = "\xa9\x7a", [0x0d9a] = "\xa9\x77",
  [0x0d9b] = "\xa9\x7b", [0x0d9c] = "\xa9\x79", [0x0da2] = "\xab\xc8",
  [0x0da3] = "\xab\xc5", [0x0da4] = "\xab\xc7", [0x0da5] = "\xab\xc9",
  [0x0da6] = "\xab\xc6", [0x0da7] = "\xd1\x66", [0x0da8] = "\xce\x77",
  [0x0dac] = "\xd1\x68", [0x0dad] = "\xd1\x67", [0x0dae] = "\xae\x63",
  [0x0db0] = "\xae\x5f", [0x0db3] = "\xae\x60", [0x0db4] = "\xae\x62",
  [0x0db5] = "\xae\x64", [0x0db6] = "\xae\x61", [0x0db8] = "\xae\x66",
  [0x0db9] = "\xae\x65", [0x0dbf] = "\xb1\x4a", [0x0dc0] = "\xd4\xf2",
  [0x0dc1] = "\xd4\xf1", [0x0dc2] = "\xb1\x49", [0x0dc4] = "\xb1\x48",
  [0x0dc5] = "\xb1\x47", [0x0dc6] = "\xb1\x4b", [0x0dc7] = "\xb1\x46",
  [0x0dca] = "\xd8\xd5", [0x0dcb] = "\xd8\xd2", [0x0dcc] = "\xb4\x49",
  [0x0dcd] = "\xd8\xd1", [0x0dce] = "\xd8\xd6", [0x0dd0] = "\xb4\x4b",
  [0x0dd1] = "\xd8\xd4", [0x0dd2] = "\xb4\x48", [0x0dd3] = "\xb4\x4a",
  [0x0dd4] = "\xd8\xd3", [0x0dd6] = "\xdd\x48", [0x0dd8] = "\xdd\x49",
  [0x0dd9] = "\xdd\x4a", [0x0dde] = "\xb9\xe6", [0x0ddf] = "\xb9\xee",
  [0x0de0] = "\xe1\x7e", [0x0de1] = "\xb9\xe8", [0x0de2] = "\xb9\xec",
  [0x0de3] = "\xe1\xa1", [0x0de4] = "\xb9\xed", [0x0de5] = "\xb9\xe9",
  [0x0de6] = "\xb9\xea", [0x0de7] = "\xb9\xe7", [0x0de8] = "\xb9\xeb",
  [0x0de9] = "\xbc\x66", [0x0dea] = "\xd8\xd0", [0x0deb] = "\xbc\x67",
  [0x0dec] = "\xbc\x65", [0x0dee] = "\xbc\x64", [0x0def] = "\xe9\x5d",
  [0x0df0] = "\xbe\xc8", [0x0df1] = "\xec\xd8", [0x0df2] = "\xec\xd9",
  [0x0df5] = "\xc3\x64", [0x0df6] = "\xc4\x5f", [0x0df8] = "\xa4\x6f",
  [0x0dfa] = "\xa6\x78", [0x0e01] = "\xab\xca", [0x0e03] = "\xd1\x69",
  [0x0e04] = "\xae\x67", [0x0e07] = "\xb1\x4e", [0x0e08] = "\xb1\x4d",
  [0x0e09] = "\xb1\x4c", [0x0e0a] = "\xb4\x4c", [0x0e0b] = "\xb4\x4d",
  [0x0e0c] = "\xd8\xd7", [0x0e0d] = "\xb9\xef", [0x0e0e] = "\xbe\xc9",
  [0x0e0f] = "\xa4\x70", [0x0e10] = "\xc9\x5c", [0x0e11] = "\xa4\xd6",
  [0x0e12] = "\xc9\x74", [0x0e15] = "\xc9\xd4", [0x0e16] = "\xa6\x79",
  [0x0e1a] = "\xa9\x7c", [0x0e1f] = "\xdd\x4b", [0x0e22] = "\xa4\x71",
  [0x0e24] = "\xa4\xd7", [0x0e25] = "\xc9\xd5", [0x0e28] = "\xca\xbe",
  [0x0e2a] = "\xca\xbf", [0x0e2c] = "\xa7\xbc", [0x0e30] = "\xd8\xd8",
  [0x0e31] = "\xb4\x4e", [0x0e33] = "\xdd\x4c", [0x0e37] = "\xc0\xaa",
  [0x0e38] = "\xa4\x72", [0x0e39] = "\xa4\xa8", [0x0e3a] = "\xa4\xd8",
  [0x0e3b] = "\xc9\x75", [0x0e3c] = "\xa5\xa7", [0x0e3e] = "\xa7\xc0",
  [0x0e3f] = "\xa7\xbf", [0x0e40] = "\xa7\xbd", [0x0e41] = "\xa7\xbe",
  [0x0e44] = "\xcc\x59", [0x0e45] = "\xa9\x7e", [0x0e46] = "\xa9\xa1",
  [0x0e47] = "\xcc\x5a", [0x0e48] = "\xa9\x7d", [0x0e4b] = "\xab\xce",
  [0x0e4c] = "\xce\x78", [0x0e4d] = "\xab\xcd", [0x0e4e] = "\xab\xcb",
  [0x0e4f] = "\xab\xcc", [0x0e50] = "\xae\x6a", [0x0e51] = "\xae\x68",
  [0x0e54] = "\xd1\x6b", [0x0e55] = "\xae\x69", [0x0e56] = "\xd1\x6a",
  [0x0e58] = "\xae\x5e", [0x0e59] = "\xd4\xf3", [0x0e5c] = "\xb1\x50",
  [0x0e5d] = "\xb1\x51", [0x0e60] = "\xb1\x4f", [0x0e62] = "\xb9\xf0",
  [0x0e63] = "\xe1\xa2", [0x0e64] = "\xbc\x68", [0x0e65] = "\xbc\x69",
  [0x0e67] = "\xe5\x61", [0x0e68] = "\xc0\xab", [0x0e69] = "\xef\xc2",
  [0x0e6a] = "\xef\xc3", [0x0e6c] = "\xc4\xdd", [0x0e6d] = "\xf8\xa8",
  [0x0e6e] = "\xc9\x4b", [0x0e6f] = "\xa4\xd9", [0x0e71] = "\xa4\x73",
  [0x0e73] = "\xc9\x77", [0x0e74] = "\xc9\x76", [0x0e79] = "\xa6\x7a",
  [0x0e7a] = "\xc9\xd7", [0x0e7b] = "\xc9\xd8", [0x0e7c] = "\xc9\xd6",
  [0x0e7e] = "\xc9\xd9", [0x0e86] = "\xca\xc7", [0x0e88] = "\xca\xc2",
  [0x0e89] = "\xca\xc4", [0x0e8a] = "\xca\xc6", [0x0e8b] = "\xca\xc3",
  [0x0e8c] = "\xa7\xc4", [0x0e8d] = "\xca\xc0", [0x0e8f] = "\xca\xc1",
  [0x0e90] = "\xa7\xc1", [0x0e91] = "\xa7\xc2", [0x0e92] = "\xca\xc5",
  [0x0e93] = "\xca\xc8", [0x0e94] = "\xa7\xc3", [0x0e95] = "\xca\xc9",
  [0x0e9d] = "\xcc\x68", [0x0e9f] = "\xcc\x62", [0x0ea0] = "\xcc\x5d",
  [0x0ea1] = "\xa9\xa3", [0x0ea2] = "\xcc\x65", [0x0ea3] = "\xcc\x63",
  [0x0ea4] = "\xcc\x5c", [0x0ea5] = "\xcc\x69", [0x0ea6] = "\xcc\x6c",
  [0x0ea7] = "\xcc\x67", [0x0ea8] = "\xcc\x60", [0x0ea9] = "\xa9\xa5",
  [0x0eaa] = "\xcc\x66", [0x0eab] = "\xa9\xa6", [0x0eac] = "\xcc\x61",
  [0x0ead] = "\xcc\x64", [0x0eae] = "\xcc\x5b", [0x0eaf] = "\xcc\x5f",
  [0x0eb0] = "\xcc\x6b", [0x0eb1] = "\xa9\xa7", [0x0eb3] = "\xa9\xa8",
  [0x0eb5] = "\xcc\x5e", [0x0eb6] = "\xcc\x6a", [0x0eb7] = "\xa9\xa2",
  [0x0eb8] = "\xa9\xa4", [0x0ec6] = "\xce\xab", [0x0ec7] = "\xce\xa4",
  [0x0ec8] = "\xce\xaa", [0x0ec9] = "\xce\xa3", [0x0eca] = "\xce\xa5",
  [0x0ecb] = "\xce\x7d", [0x0ecc] = "\xce\x7b", [0x0ece] = "\xce\xac",
  [0x0ecf] = "\xce\xa9", [0x0ed0] = "\xce\x79", [0x0ed2] = "\xab\xd0",
  [0x0ed3] = "\xce\xa7", [0x0ed4] = "\xce\xa8", [0x0ed6] = "\xce\xa6",
  [0x0ed7] = "\xce\x7c", [0x0ed8] = "\xce\x7a", [0x0ed9] = "\xab\xcf",
  [0x0eda] = "\xce\xa2", [0x0edb] = "\xce\x7e", [0x0ede] = "\xce\xa1",
  [0x0edf] = "\xce\xad", [0x0ee8] = "\xae\x6f", [0x0eea] = "\xae\x6e",
  [0x0eec] = "\xd1\x6c", [0x0eed] = "\xae\x6b", [0x0eee] = "\xd1\x6e",
  [0x0ef0] = "\xae\x70", [0x0ef1] = "\xd1\x6f", [0x0ef4] = "\xae\x73",
  [0x0ef6] = "\xae\x71", [0x0ef7] = "\xd1\x70", [0x0ef8] = "\xce\xae",
  [0x0ef9] = "\xd1\x72", [0x0efb] = "\xae\x6d", [0x0efd] = "\xae\x6c",
  [0x0eff] = "\xd1\x6d", [0x0f00] = "\xd1\x71", [0x0f01] = "\xae\x72",
  [0x0f06] = "\xb1\x53", [0x0f07] = "\xb1\x52", [0x0f0b] = "\xd4\xf5",
  [0x0f0c] = "\xd4\xf9", [0x0f0d] = "\xd4\xfb", [0x0f0e] = "\xb1\x54",
  [0x0f0f] = "\xd4\xfe", [0x0f11] = "\xb1\x58", [0x0f12] = "\xd5\x41",
  [0x0f14] = "\xb1\x5a", [0x0f16] = "\xb1\x56", [0x0f17] = "\xb1\x5e",
  [0x0f19] = "\xb1\x5b", [0x0f1a] = "\xd4\xf7", [0x0f1b] = "\xb1\x55",
  [0x0f1d] = "\xd4\xf6", [0x0f1e] = "\xd4\xf4", [0x0f1f] = "\xd5\x43",
  [0x0f20] = "\xd4\xf8", [0x0f22] = "\xb1\x57", [0x0f23] = "\xd5\x42",
  [0x0f24] = "\xb1\x5c", [0x0f25] = "\xd4\xfd", [0x0f26] = "\xd4\xfc",
  [0x0f27] = "\xb1\x5d", [0x0f28] = "\xd4\xfa", [0x0f29] = "\xb1\x59",
  [0x0f2e] = "\xd5\x44", [0x0f30] = "\xd5\x40", [0x0f31] = "\xd8\xe7",
  [0x0f32] = "\xd8\xee", [0x0f33] = "\xd8\xe3", [0x0f34] = "\xb4\x51",
  [0x0f35] = "\xd8\xdf", [0x0f36] = "\xd8\xef", [0x0f37] = "\xd8\xd9",
  [0x0f38] = "\xd8\xec", [0x0f39] = "\xd8\xea", [0x0f3a] = "\xd8\xe4",
  [0x0f3c] = "\xd8\xed", [0x0f3d] = "\xd8\xe6", [0x0f3f] = "\xd8\xde",
  [0x0f40] = "\xd8\xf0", [0x0f41] = "\xd8\xdc", [0x0f42] = "\xd8\xe9",
  [0x0f43] = "\xd8\xda", [0x0f45] = "\xd8\xf1", [0x0f47] = "\xb4\x52",
  [0x0f49] = "\xd8\xeb", [0x0f4a] = "\xdd\x4f", [0x0f4b] = "\xd8\xdd",
  [0x0f4c] = "\xb4\x4f", [0x0f4e] = "\xd8\xe1", [0x0f50] = "\xb4\x50",
  [0x0f51] = "\xd8\xe0", [0x0f52] = "\xd8\xe5", [0x0f55] = "\xd8\xe2",
  [0x0f59] = "\xd8\xe8", [0x0f5e] = "\xdd\x53", [0x0f62] = "\xdd\x56",
  [0x0f63] = "\xdd\x4e", [0x0f65] = "\xdd\x50", [0x0f67] = "\xdd\x55",
  [0x0f68] = "\xdd\x54", [0x0f69] = "\xb7\x43", [0x0f6b] = "\xd8\xdb",
  [0x0f6c] = "\xdd\x52", [0x0f6f] = "\xb7\x44", [0x0f71] = "\xdd\x4d",
  [0x0f72] = "\xdd\x51", [0x0f77] = "\xe1\xa9", [0x0f79] = "\xe1\xb0",
  [0x0f7a] = "\xe1\xa7", [0x0f7c] = "\xe1\xae", [0x0f7d] = "\xe1\xa5",
  [0x0f7e] = "\xe1\xad", [0x0f7f] = "\xe1\xb1", [0x0f80] = "\xe1\xa4",
  [0x0f81] = "\xe1\xa8", [0x0f82] = "\xe1\xa3", [0x0f84] = "\xb9\xf1",
  [0x0f86] = "\xe1\xa6", [0x0f87] = "\xb9\xf2", [0x0f88] = "\xe1\xac",
  [0x0f89] = "\xe1\xab", [0x0f8a] = "\xe1\xaa", [0x0f8d] = "\xe1\xaf",
  [0x0f92] = "\xe5\x65", [0x0f93] = "\xe5\x67", [0x0f94] = "\xbc\x6b",
  [0x0f95] = "\xe5\x68", [0x0f97] = "\xe5\x63", [0x0f99] = "\xe5\x62",
  [0x0f9a] = "\xe5\x6c", [0x0f9c] = "\xe5\x6a", [0x0f9d] = "\xbc\x6a",
  [0x0f9e] = "\xe5\x6d", [0x0f9f] = "\xe5\x64", [0x0fa0] = "\xe5\x69",
  [0x0fa1] = "\xe5\x6b", [0x0fa2] = "\xe5\x66", [0x0fa7] = "\xe9\x61",
  [0x0fa8] = "\xe9\x66", [0x0fa9] = "\xe9\x60", [0x0faa] = "\xe9\x65",
  [0x0fac] = "\xe9\x5e", [0x0fad] = "\xe9\x68", [0x0fae] = "\xe9\x64",
  [0x0faf] = "\xe9\x69", [0x0fb0] = "\xe9\x63", [0x0fb1] = "\xe9\x5f",
  [0x0fb2] = "\xe9\x67", [0x0fb4] = "\xe9\x6a", [0x0fb5] = "\xe9\x62",
  [0x0fb7] = "\xec\xda", [0x0fb8] = "\xc0\xaf", [0x0fba] = "\xc0\xad",
  [0x0fbc] = "\xc0\xac", [0x0fbd] = "\xc0\xae", [0x0fc0] = "\xef\xc4",
  [0x0fc2] = "\xf1\x72", [0x0fc3] = "\xf1\xfd", [0x0fc6] = "\xf4\x44",
  [0x0fc7] = "\xf4\x45", [0x0fc9] = "\xc4\x60", [0x0fcb] = "\xf5\xc9",
  [0x0fcd] = "\xc4\xde", [0x0fcf] = "\xf5\xca", [0x0fd1] = "\xf6\xde",
  [0x0fd2] = "\xc5\x72", [0x0fd4] = "\xc5\x71", [0x0fd5] = "\xf6\xdd",
  [0x0fd6] = "\xc5\xc9", [0x0fd8] = "\xf7\xd6", [0x0fdd] = "\xa4\x74",
  [0x0fde] = "\xa6\x7b", [0x0fdf] = "\xc9\xda", [0x0fe0] = "\xca\xca",
  [0x0fe1] = "\xa8\xb5", [0x0fe2] = "\xb1\x5f", [0x0fe5] = "\xa4\x75",
  [0x0fe6] = "\xa5\xaa", [0x0fe7] = "\xa5\xa9", [0x0fe8] = "\xa5\xa8",
  [0x0feb] = "\xa7\xc5", [0x0fee] = "\xae\x74", [0x0ff0] = "\xdd\x57",
  [0x0ff1] = "\xa4\x76", [0x0ff2] = "\xa4\x77", [0x0ff3] = "\xa4\x78",
  [0x0ff4] = "\xa4\xda", [0x0ff7] = "\xab\xd1", [0x0ff9] = "\xce\xaf",
  [0x0ffd] = "\xb4\x53", [0x0ffe] = "\xa4\x79", [0x0fff] = "\xc9\x5d",
  [0x1002] = "\xa5\xab", [0x1003] = "\xa5\xac", [0x1004] = "\xc9\x78",
  [0x1006] = "\xa6\x7c", [0x100a] = "\xca\xcb", [0x100c] = "\xa7\xc6",
  [0x100e] = "\xca\xcc", [0x1011] = "\xa9\xae", [0x1014] = "\xcc\x6e",
  [0x1015] = "\xa9\xac", [0x1016] = "\xa9\xab", [0x1017] = "\xcc\x6d",
  [0x1018] = "\xa9\xa9", [0x1019] = "\xcc\x6f", [0x101a] = "\xa9\xaa",
  [0x101b] = "\xa9\xad", [0x101d] = "\xab\xd2", [0x101f] = "\xab\xd4",
  [0x1020] = "\xce\xb3", [0x1021] = "\xce\xb0", [0x1022] = "\xce\xb1",
  [0x1023] = "\xce\xb2", [0x1024] = "\xce\xb4", [0x1025] = "\xab\xd3",
  [0x1028] = "\xd1\x74", [0x1029] = "\xd1\x73", [0x102b] = "\xae\x76",
  [0x102d] = "\xae\x75", [0x1033] = "\xb1\x62", [0x1034] = "\xd5\x46",
  [0x1036] = "\xb1\x61", [0x1037] = "\xb1\x63", [0x1038] = "\xb1\x60",
  [0x103d] = "\xb4\x55", [0x103e] = "\xd5\x45", [0x1040] = "\xb4\x56",
  [0x1041] = "\xd8\xf3", [0x1043] = "\xb4\x57", [0x1044] = "\xd8\xf2",
  [0x1045] = "\xb4\x54", [0x104a] = "\xdd\x5a", [0x104b] = "\xdd\x5c",
  [0x104c] = "\xb7\x45", [0x104d] = "\xdd\x5b", [0x104e] = "\xdd\x59",
  [0x104f] = "\xdd\x58", [0x1053] = "\xe1\xb4", [0x1054] = "\xb9\xf7",
  [0x1055] = "\xb9\xf5", [0x1057] = "\xb9\xf6", [0x1058] = "\xe1\xb2",
  [0x1059] = "\xe1\xb3", [0x105b] = "\xb9\xf3", [0x105c] = "\xe5\x71",
  [0x105d] = "\xe5\x6f", [0x105f] = "\xbc\x6d", [0x1060] = "\xe5\x70",
  [0x1061] = "\xbc\x6e", [0x1062] = "\xbc\x6c", [0x1063] = "\xb9\xf4",
  [0x1066] = "\xe9\x6d", [0x1067] = "\xe9\x6b", [0x1068] = "\xe9\x6c",
  [0x1069] = "\xe5\x6e", [0x106a] = "\xec\xdc", [0x106b] = "\xc0\xb0",
  [0x106c] = "\xec\xdb", [0x106d] = "\xef\xc5", [0x106e] = "\xef\xc6",
  [0x106f] = "\xe9\x6e", [0x1070] = "\xf1\xfe", [0x1072] = "\xa4\x7a",
  [0x1073] = "\xa5\xad", [0x1074] = "\xa6\x7e", [0x1075] = "\xc9\xdb",
  [0x1076] = "\xa6\x7d", [0x1078] = "\xa9\xaf", [0x1079] = "\xb7\x46",
  [0x107b] = "\xa4\xdb", [0x107c] = "\xa5\xae", [0x107d] = "\xab\xd5",
  [0x107e] = "\xb4\x58", [0x1080] = "\xc9\x79", [0x1082] = "\xc9\x7a",
  [0x1084] = "\xc9\xdc", [0x1087] = "\xa7\xc8", [0x1088] = "\xca\xd0",
  [0x1089] = "\xca\xce", [0x108a] = "\xa7\xc9", [0x108b] = "\xca\xcd",
  [0x108c] = "\xca\xcf", [0x108d] = "\xca\xd1", [0x108f] = "\xa7\xc7",
  [0x1095] = "\xa9\xb3", [0x1096] = "\xa9\xb4", [0x1097] = "\xa9\xb1",
  [0x109a] = "\xa9\xb0", [0x109b] = "\xce\xb8", [0x109c] = "\xa9\xb2",
  [0x10a0] = "\xab\xd6", [0x10a2] = "\xce\xb7", [0x10a3] = "\xce\xb9",
  [0x10a4] = "\xce\xb6", [0x10a5] = "\xce\xba", [0x10a6] = "\xab\xd7",
  [0x10a7] = "\xae\x79", [0x10a8] = "\xd1\x75", [0x10aa] = "\xd1\x77",
  [0x10ab] = "\xae\x77", [0x10ac] = "\xd1\x78", [0x10ad] = "\xae\x78",
  [0x10ae] = "\xd1\x76", [0x10b0] = "\xce\xb5", [0x10b1] = "\xd5\x47",
  [0x10b2] = "\xd5\x4a", [0x10b3] = "\xd5\x4b", [0x10b4] = "\xd5\x48",
  [0x10b5] = "\xb1\x67", [0x10b6] = "\xb1\x66", [0x10b7] = "\xb1\x64",
  [0x10b8] = "\xb1\x65", [0x10b9] = "\xd5\x49", [0x10be] = "\xb1\x68",
  [0x10c1] = "\xb4\x5a", [0x10c2] = "\xb4\x5b", [0x10c4] = "\xb4\x5c",
  [0x10c5] = "\xdd\x5d", [0x10c6] = "\xdd\x5f", [0x10c7] = "\xdd\x61",
  [0x10c8] = "\xb7\x48", [0x10c9] = "\xb7\x47", [0x10ca] = "\xb4\x59",
  [0x10cb] = "\xdd\x60", [0x10cc] = "\xdd\x5e", [0x10ce] = "\xe1\xb8",
  [0x10d1] = "\xe1\xb6", [0x10d2] = "\xe1\xbc", [0x10d3] = "\xb9\xf8",
  [0x10d4] = "\xe1\xbd", [0x10d5] = "\xe1\xba", [0x10d6] = "\xb9\xf9",
  [0x10d7] = "\xe1\xb7", [0x10d8] = "\xe1\xb5", [0x10d9] = "\xe1\xbb",
  [0x10da] = "\xbc\x70", [0x10db] = "\xe5\x73", [0x10dc] = "\xe1\xb9",
  [0x10dd] = "\xbc\x72", [0x10de] = "\xe5\x74", [0x10df] = "\xbc\x71",
  [0x10e0] = "\xbc\x74", [0x10e1] = "\xe5\x75", [0x10e2] = "\xbc\x6f",
  [0x10e3] = "\xbc\x73", [0x10e5] = "\xe9\x73", [0x10e6] = "\xe9\x71",
  [0x10e7] = "\xe9\x70", [0x10e8] = "\xe9\x72", [0x10e9] = "\xe9\x6f",
  [0x10ec] = "\xc3\x66", [0x10ee] = "\xf4\x46", [0x10ef] = "\xf4\x47",
  [0x10f1] = "\xf5\xcb", [0x10f2] = "\xf6\xdf", [0x10f3] = "\xc6\x55",
  [0x10f6] = "\xa9\xb5", [0x10f7] = "\xa7\xca", [0x10fa] = "\xab\xd8",
  [0x10fe] = "\xa4\x7b", [0x10ff] = "\xa4\xdc", [0x1101] = "\xa5\xaf",
  [0x1102] = "\xc9\xdd", [0x1104] = "\xa7\xcb", [0x1105] = "\xca\xd2",
  [0x1107] = "\xce\xbb", [0x1108] = "\xab\xd9", [0x110a] = "\xb9\xfa",
  [0x110b] = "\xa4\x7c", [0x110f] = "\xa6\xa1", [0x1112] = "\xb7\x49",
  [0x1113] = "\xa4\x7d", [0x1114] = "\xa4\xdd", [0x1115] = "\xa4\xde",
  [0x1117] = "\xa5\xb1", [0x1118] = "\xa5\xb0", [0x111a] = "\xc9\xde",
  [0x111b] = "\xa6\xa2", [0x111d] = "\xca\xd3", [0x111f] = "\xa7\xcc",
  [0x1122] = "\xcc\x71", [0x1123] = "\xcc\x72", [0x1124] = "\xcc\x73",
  [0x1126] = "\xa9\xb6", [0x1127] = "\xa9\xb7", [0x1128] = "\xcc\x70",
  [0x1129] = "\xa9\xb8", [0x112d] = "\xab\xda", [0x112e] = "\xce\xbc",
  [0x1130] = "\xd1\x7a", [0x1131] = "\xae\x7a", [0x1133] = "\xd1\x79",
  [0x1135] = "\xb1\x69", [0x1136] = "\xd5\x4c", [0x1137] = "\xb1\x6a",
  [0x1138] = "\xd5\x4d", [0x113c] = "\xb4\x5d", [0x1140] = "\xdd\x62",
  [0x1143] = "\xe1\xbf", [0x1144] = "\xe1\xbe", [0x1146] = "\xb9\xfb",
  [0x1148] = "\xbc\x75", [0x1149] = "\xe5\x76", [0x114a] = "\xbe\xca",
  [0x114b] = "\xe9\x74", [0x114c] = "\xc0\xb1", [0x114e] = "\xc5\x73",
  [0x114f] = "\xf7\xd8", [0x1154] = "\xcc\x74", [0x1156] = "\xce\xbd",
  [0x1157] = "\xb1\x6b", [0x1158] = "\xd8\xf4", [0x1159] = "\xb7\x4a",
  [0x115d] = "\xc2\x55", [0x1162] = "\xa7\xce", [0x1164] = "\xa7\xcd",
  [0x1165] = "\xab\xdb", [0x1167] = "\xd1\x7b", [0x1169] = "\xb1\x6d",
  [0x116a] = "\xb3\x43", [0x116b] = "\xb1\x6e", [0x116c] = "\xb1\x6c",
  [0x116d] = "\xb4\x5e", [0x116f] = "\xe1\xc0", [0x1170] = "\xb9\xfc",
  [0x1171] = "\xbc\x76", [0x1173] = "\xc9\x4c", [0x1174] = "\xc9\xdf",
  [0x1176] = "\xca\xd5", [0x1177] = "\xa7\xcf", [0x1178] = "\xca\xd4",
  [0x1179] = "\xa7\xd0", [0x117c] = "\xa9\xbc", [0x117d] = "\xcc\x77",
  [0x117e] = "\xcc\x76", [0x117f] = "\xa9\xbb", [0x1180] = "\xa9\xb9",
  [0x1181] = "\xa9\xba", [0x1182] = "\xcc\x75", [0x1185] = "\xab\xdd",
  [0x1186] = "\xce\xbe", [0x1187] = "\xab\xe0", [0x1188] = "\xab\xdc",
  [0x1189] = "\xab\xe2", [0x118a] = "\xab\xde", [0x118b] = "\xab\xdf",
  [0x118c] = "\xab\xe1", [0x1190] = "\xae\x7d", [0x1191] = "\xae\x7c",
  [0x1192] = "\xae\x7b", [0x1196] = "\xd5\x4f", [0x1197] = "\xb1\x6f",
  [0x1198] = "\xb1\x72", [0x1199] = "\xb1\x70", [0x119b] = "\xd5\x4e",
  [0x119c] = "\xb1\x75", [0x119e] = "\xb1\x71", [0x119f] = "\xd5\x50",
  [0x11a0] = "\xb1\x74", [0x11a1] = "\xb1\x73", [0x11a5] = "\xd8\xf6",
  [0x11a6] = "\xd8\xf5", [0x11a8] = "\xb4\x61", [0x11a9] = "\xb4\x5f",
  [0x11aa] = "\xb4\x60", [0x11ab] = "\xd8\xf7", [0x11ac] = "\xb7\x4b",
  [0x11ad] = "\xdd\x64", [0x11ae] = "\xb7\x4c", [0x11af] = "\xdd\x63",
  [0x11b2] = "\xe5\x77", [0x11b5] = "\xbc\x78", [0x11b6] = "\xe1\xc1",
  [0x11b7] = "\xbc\x77", [0x11b9] = "\xb9\xfd", [0x11bb] = "\xec\xde",
  [0x11bc] = "\xe9\x75", [0x11bd] = "\xc0\xb2", [0x11be] = "\xec\xdd",
  [0x11bf] = "\xf2\x40", [0x11c0] = "\xf4\x48", [0x11c1] = "\xf4\x49",
  [0x11c3] = "\xa4\xdf", [0x11c5] = "\xa5\xb2", [0x11c9] = "\xc9\x7b",
  [0x11cc] = "\xa7\xd2", [0x11cd] = "\xa7\xd4", [0x11cf] = "\xc9\xe2",
  [0x11d0] = "\xca\xd8", [0x11d1] = "\xca\xd7", [0x11d2] = "\xca\xd6",
  [0x11d4] = "\xc9\xe1", [0x11d5] = "\xc9\xe0", [0x11d6] = "\xa6\xa4",
  [0x11d7] = "\xa7\xd3", [0x11d8] = "\xa7\xd1", [0x11d9] = "\xa6\xa3",
  [0x11dd] = "\xa9\xbd", [0x11de] = "\xcc\x78", [0x11e0] = "\xa9\xbe",
  [0x11e1] = "\xca\xdd", [0x11e3] = "\xca\xdf", [0x11e4] = "\xca\xde",
  [0x11e5] = "\xcc\x79", [0x11e8] = "\xca\xda", [0x11ea] = "\xa7\xd8",
  [0x11eb] = "\xa7\xd6", [0x11ed] = "\xca\xd9", [0x11ee] = "\xca\xdb",
  [0x11ef] = "\xca\xe1", [0x11f1] = "\xa7\xd5", [0x11f3] = "\xca\xdc",
  [0x11f4] = "\xca\xe5", [0x11f5] = "\xa9\xc0", [0x11f7] = "\xca\xe2",
  [0x11f8] = "\xa7\xd7", [0x11fa] = "\xca\xe0", [0x11fb] = "\xca\xe3",
  [0x11fd] = "\xa9\xbf", [0x11ff] = "\xa9\xc1", [0x1200] = "\xca\xe4",
  [0x1209] = "\xcc\xaf", [0x120a] = "\xcc\xa2", [0x120b] = "\xcc\x7e",
  [0x120c] = "\xcc\xae", [0x120d] = "\xcc\xa9", [0x120e] = "\xab\xe7",
  [0x120f] = "\xa9\xc2", [0x1210] = "\xcc\xaa", [0x1211] = "\xcc\xad",
  [0x1212] = "\xab\xe3", [0x1213] = "\xcc\xac", [0x1214] = "\xa9\xc3",
  [0x1215] = "\xa9\xc8", [0x1216] = "\xa9\xc6", [0x1217] = "\xcc\xa3",
  [0x1219] = "\xcc\x7c", [0x121a] = "\xcc\xa5", [0x121b] = "\xa9\xcd",
  [0x121c] = "\xcc\xb0", [0x121d] = "\xab\xe4", [0x121e] = "\xcc\xa6",
  [0x1220] = "\xab\xe5", [0x1221] = "\xa9\xc9", [0x1222] = "\xcc\xa8",
  [0x1224] = "\xce\xcd", [0x1225] = "\xab\xe6", [0x1226] = "\xcc\x7b",
  [0x1227] = "\xa9\xca", [0x1228] = "\xab\xe8", [0x1229] = "\xa9\xcb",
  [0x122a] = "\xa9\xc7", [0x122b] = "\xa9\xcc", [0x122c] = "\xcc\xa7",
  [0x122d] = "\xcc\x7a", [0x122e] = "\xcc\xab", [0x122f] = "\xa9\xc4",
  [0x1232] = "\xcc\x7d", [0x1233] = "\xcc\xa4", [0x1234] = "\xcc\xa1",
  [0x1235] = "\xa9\xc5", [0x1237] = "\xce\xbf", [0x1239] = "\xce\xc0",
  [0x1240] = "\xce\xca", [0x1241] = "\xd1\xa1", [0x1242] = "\xce\xcb",
  [0x1243] = "\xab\xee", [0x1244] = "\xce\xce", [0x1245] = "\xce\xc4",
  [0x1246] = "\xab\xed", [0x1247] = "\xce\xc6", [0x1249] = "\xce\xc7",
  [0x124c] = "\xce\xc9", [0x124d] = "\xab\xe9", [0x1250] = "\xae\xa3",
  [0x1252] = "\xf9\xda", [0x1253] = "\xce\xc5", [0x1254] = "\xce\xc1",
  [0x1255] = "\xae\xa4", [0x1258] = "\xce\xcf", [0x1259] = "\xae\x7e",
  [0x125a] = "\xd1\x7d", [0x125b] = "\xce\xc8", [0x125d] = "\xd1\x7c",
  [0x125e] = "\xce\xc3", [0x125f] = "\xce\xcc", [0x1262] = "\xab\xec",
  [0x1263] = "\xae\xa1", [0x1264] = "\xab\xf2", [0x1265] = "\xae\xa2",
  [0x1266] = "\xce\xd0", [0x1267] = "\xd1\x7e", [0x1268] = "\xab\xeb",
  [0x1269] = "\xae\xa6", [0x126a] = "\xab\xf1", [0x126b] = "\xab\xf0",
  [0x126c] = "\xab\xef", [0x126d] = "\xae\xa5", [0x126e] = "\xce\xd1",
  [0x126f] = "\xae\xa7", [0x1270] = "\xab\xea", [0x1272] = "\xce\xc2",
  [0x127f] = "\xb1\x76", [0x1280] = "\xd1\xa4", [0x1281] = "\xd1\xa6",
  [0x1283] = "\xd1\xa8", [0x1284] = "\xae\xa8", [0x1285] = "\xae\xae",
  [0x1286] = "\xd5\x53", [0x1287] = "\xd1\xac", [0x1288] = "\xd1\xa3",
  [0x1289] = "\xb1\x78", [0x128a] = "\xd5\x51", [0x128c] = "\xae\xad",
  [0x128d] = "\xae\xab", [0x128e] = "\xd1\xae", [0x1290] = "\xd5\x52",
  [0x1292] = "\xd1\xa5", [0x1294] = "\xae\xac", [0x1295] = "\xd1\xa9",
  [0x1296] = "\xae\xaf", [0x1297] = "\xd1\xab", [0x129a] = "\xae\xaa",
  [0x129b] = "\xd1\xaa", [0x129c] = "\xd1\xad", [0x129d] = "\xd1\xa7",
  [0x129f] = "\xae\xa9", [0x12a0] = "\xb1\x79", [0x12a2] = "\xd1\xa2",
  [0x12a3] = "\xb1\x77", [0x12a8] = "\xb1\x7a", [0x12b0] = "\xd5\x55",
  [0x12b1] = "\xd5\x5e", [0x12b2] = "\xb4\x64", [0x12b4] = "\xb1\x7c",
  [0x12b5] = "\xb1\xa3", [0x12b6] = "\xb4\x65", [0x12b7] = "\xd5\x60",
  [0x12b8] = "\xb1\xaa", [0x12b9] = "\xd8\xf9", [0x12ba] = "\xd5\x56",
  [0x12bb] = "\xb1\xa2", [0x12bc] = "\xb1\xa5", [0x12bd] = "\xb1\x7e",
  [0x12be] = "\xd5\x54", [0x12bf] = "\xd5\x62", [0x12c0] = "\xd5\x65",
  [0x12c1] = "\xd9\x49", [0x12c3] = "\xd5\x63", [0x12c4] = "\xd8\xfd",
  [0x12c5] = "\xb1\xa1", [0x12c6] = "\xb1\xa8", [0x12c7] = "\xb1\xac",
  [0x12c8] = "\xd5\x5d", [0x12c9] = "\xd8\xf8", [0x12ca] = "\xd5\x61",
  [0x12cb] = "\xb1\x7b", [0x12cc] = "\xd8\xfa", [0x12cd] = "\xd5\x64",
  [0x12ce] = "\xd8\xfc", [0x12cf] = "\xd5\x59", [0x12d1] = "\xb4\x62",
  [0x12d3] = "\xd5\x57", [0x12d4] = "\xd5\x58", [0x12d5] = "\xb1\xa7",
  [0x12d8] = "\xb1\xa6", [0x12d9] = "\xd5\x5b", [0x12da] = "\xb1\xab",
  [0x12db] = "\xd5\x5f", [0x12dc] = "\xb1\xa4", [0x12dd] = "\xd5\x5c",
  [0x12df] = "\xb1\xa9", [0x12e0] = "\xb4\x66", [0x12e1] = "\xb4\x63",
  [0x12e2] = "\xd8\xfb", [0x12e4] = "\xd5\x5a", [0x12e6] = "\xb1\x7d",
  [0x12f0] = "\xb4\x6b", [0x12f1] = "\xb4\x6f", [0x12f2] = "\xd9\x40",
  [0x12f3] = "\xb7\x51", [0x12f4] = "\xb4\x6d", [0x12f5] = "\xd9\x44",
  [0x12f6] = "\xb4\x71", [0x12f7] = "\xdd\x65", [0x12f8] = "\xd9\x46",
  [0x12f9] = "\xb7\x53", [0x12fa] = "\xb4\x69", [0x12fb] = "\xb4\x6c",
  [0x12fc] = "\xd9\x47", [0x12fe] = "\xd9\x48", [0x12ff] = "\xd9\x4e",
  [0x1300] = "\xb4\x73", [0x1301] = "\xb7\x54", [0x1303] = "\xd9\x4a",
  [0x1304] = "\xd9\x4f", [0x1305] = "\xd9\x43", [0x1306] = "\xb7\x5e",
  [0x1308] = "\xb7\x55", [0x1309] = "\xb4\x72", [0x130a] = "\xd9\x41",
  [0x130b] = "\xd9\x50", [0x130d] = "\xb7\x5d", [0x130e] = "\xb4\x70",
  [0x130f] = "\xb7\x4e", [0x1310] = "\xd9\x4d", [0x1312] = "\xb4\x74",
  [0x1313] = "\xd9\x45", [0x1314] = "\xd8\xfe", [0x1315] = "\xb4\x6a",
  [0x1316] = "\xd9\x42", [0x1318] = "\xd9\x4b", [0x131a] = "\xb7\x4d",
  [0x131b] = "\xb7\x52", [0x131c] = "\xb4\x67", [0x131d] = "\xd9\x4c",
  [0x131f] = "\xb7\x50", [0x1323] = "\xb4\x68", [0x1327] = "\xb7\x5c",
  [0x1328] = "\xe1\xc3", [0x1329] = "\xdd\x70", [0x132b] = "\xdd\x68",
  [0x132c] = "\xe1\xc2", [0x132e] = "\xdd\x6c", [0x132f] = "\xdd\x6e",
  [0x1332] = "\xdd\x6b", [0x1334] = "\xb7\x5b", [0x1336] = "\xdd\x6a",
  [0x1337] = "\xb7\x5f", [0x133b] = "\xe1\xd2", [0x133e] = "\xb7\x5a",
  [0x133f] = "\xba\x40", [0x1340] = "\xdd\x71", [0x1341] = "\xe1\xc4",
  [0x1344] = "\xb7\x58", [0x1345] = "\xdd\x69", [0x1346] = "\xdd\x6d",
  [0x1347] = "\xb9\xfe", [0x1348] = "\xb7\x4f", [0x1349] = "\xdd\x66",
  [0x134a] = "\xdd\x67", [0x134b] = "\xba\x41", [0x134c] = "\xb7\x57",
  [0x134d] = "\xb7\x59", [0x134e] = "\xb7\x56", [0x134f] = "\xdd\x6f",
  [0x1352] = "\xe1\xc8", [0x1353] = "\xe1\xc9", [0x1354] = "\xe1\xce",
  [0x1355] = "\xbc\x7d", [0x1356] = "\xe1\xd5", [0x1358] = "\xba\x47",
  [0x135a] = "\xba\x46", [0x135b] = "\xe1\xd0", [0x135d] = "\xbc\x7c",
  [0x135e] = "\xe1\xc5", [0x135f] = "\xba\x45", [0x1361] = "\xe1\xd4",
  [0x1362] = "\xba\x43", [0x1363] = "\xba\x44", [0x1365] = "\xe1\xd1",
  [0x1366] = "\xe5\xaa", [0x1367] = "\xbc\x7a", [0x1368] = "\xb4\x6e",
  [0x136a] = "\xe1\xd3", [0x136b] = "\xbc\xa3", [0x136c] = "\xe1\xcb",
  [0x136e] = "\xbc\x7b", [0x1370] = "\xbc\xa2", [0x1371] = "\xe1\xc6",
  [0x1372] = "\xe1\xca", [0x1373] = "\xe1\xc7", [0x1374] = "\xe1\xcd",
  [0x1375] = "\xba\x48", [0x1376] = "\xbc\x79", [0x1377] = "\xba\x42",
  [0x1379] = "\xe5\x7a", [0x137a] = "\xe1\xcf", [0x137c] = "\xbc\xa1",
  [0x137e] = "\xbc\xa4", [0x1380] = "\xe1\xcc", [0x1382] = "\xbc\x7e",
  [0x1383] = "\xe5\x79", [0x1389] = "\xe5\x7e", [0x138a] = "\xbe\xce",
  [0x138b] = "\xe5\x78", [0x138c] = "\xe9\xa3", [0x138d] = "\xe5\xa9",
  [0x138e] = "\xbc\xa8", [0x1390] = "\xbc\xa6", [0x1391] = "\xbe\xcc",
  [0x1392] = "\xe5\xa6", [0x1393] = "\xe5\xa2", [0x1394] = "\xbc\xac",
  [0x1396] = "\xe9\x78", [0x139a] = "\xbc\xaa", [0x139b] = "\xe5\xa1",
  [0x139d] = "\xe9\x76", [0x139f] = "\xe5\xa5", [0x13a1] = "\xe5\xa8",
  [0x13a2] = "\xe5\x7d", [0x13a4] = "\xbc\xab", [0x13a7] = "\xbc\xa5",
  [0x13a8] = "\xe9\x77", [0x13a9] = "\xbe\xcd", [0x13aa] = "\xe5\xa7",
  [0x13ab] = "\xbc\xa7", [0x13ac] = "\xbc\xa9", [0x13ad] = "\xe5\xa4",
  [0x13ae] = "\xbc\xad", [0x13af] = "\xe5\xa3", [0x13b0] = "\xe5\x7c",
  [0x13b1] = "\xe5\x7b", [0x13b2] = "\xbe\xcb", [0x13b3] = "\xe5\xab",
  [0x13b4] = "\xe9\x7a", [0x13b5] = "\xec\xe0", [0x13b6] = "\xbe\xd0",
  [0x13b8] = "\xe9\xa2", [0x13ba] = "\xe9\x7e", [0x13bc] = "\xec\xe1",
  [0x13be] = "\xbe\xd1", [0x13bf] = "\xe9\xa1", [0x13c1] = "\xe9\x7c",
  [0x13c2] = "\xc0\xb4", [0x13c3] = "\xec\xdf", [0x13c5] = "\xe9\x79",
  [0x13c6] = "\xe9\x7b", [0x13c7] = "\xc0\xb5", [0x13c8] = "\xbe\xd3",
  [0x13c9] = "\xc0\xb3", [0x13ca] = "\xbe\xd2", [0x13cb] = "\xc0\xb7",
  [0x13cc] = "\xe9\x7d", [0x13cd] = "\xbe\xcf", [0x13d6] = "\xef\xcf",
  [0x13d8] = "\xef\xc7", [0x13de] = "\xec\xe7", [0x13df] = "\xef\xc8",
  [0x13e0] = "\xec\xe3", [0x13e3] = "\xc2\x56", [0x13e4] = "\xec\xe5",
  [0x13e5] = "\xec\xe4", [0x13e6] = "\xc0\xb6", [0x13e7] = "\xec\xe2",
  [0x13e8] = "\xec\xe6", [0x13e9] = "\xef\xd0", [0x13ea] = "\xef\xcc",
  [0x13eb] = "\xef\xce", [0x13ed] = "\xef\xc9", [0x13ee] = "\xef\xca",
  [0x13f0] = "\xef\xcd", [0x13f1] = "\xef\xcb", [0x13f2] = "\xc3\x67",
  [0x13f5] = "\xc3\x6a", [0x13f6] = "\xc3\x69", [0x13f7] = "\xc3\x68",
  [0x13f8] = "\xc4\x61", [0x13f9] = "\xf4\x4a", [0x13fa] = "\xc4\x62",
  [0x13fb] = "\xf2\x41", [0x13fc] = "\xc4\xdf", [0x13fd] = "\xf5\xcc",
  [0x13fe] = "\xc4\xe0", [0x13ff] = "\xc5\x74", [0x1400] = "\xc5\xca",
  [0x1401] = "\xf7\xd9", [0x1403] = "\xf7\xda", [0x1404] = "\xf7\xdb",
  [0x1407] = "\xf9\xba", [0x1408] = "\xa4\xe0", [0x1409] = "\xc9\x7c",
  [0x140a] = "\xa5\xb3", [0x140c] = "\xa6\xa6", [0x140d] = "\xa6\xa7",
  [0x140e] = "\xa6\xa5", [0x1410] = "\xa6\xa8", [0x1411] = "\xa7\xda",
  [0x1412] = "\xa7\xd9", [0x1414] = "\xcc\xb1", [0x1415] = "\xa9\xcf",
  [0x1416] = "\xa9\xce", [0x1419] = "\xd1\xaf", [0x141a] = "\xb1\xad",
  [0x141b] = "\xb1\xae", [0x141f] = "\xb4\x75", [0x1420] = "\xdd\x72",
  [0x1421] = "\xb7\x60", [0x1422] = "\xb7\x61", [0x1423] = "\xdd\x74",
  [0x1424] = "\xdd\x76", [0x1425] = "\xdd\x75", [0x1427] = "\xe1\xd7",
  [0x1429] = "\xe1\xd6", [0x142a] = "\xba\x49", [0x142b] = "\xe1\xd8",
  [0x142d] = "\xe5\xac", [0x142e] = "\xbc\xae", [0x1430] = "\xbe\xd4",
  [0x1432] = "\xc0\xb8", [0x1433] = "\xc2\x57", [0x1434] = "\xc0\xb9",
  [0x1436] = "\xa4\xe1", [0x143a] = "\xca\xe6", [0x143d] = "\xcc\xb2",
  [0x143e] = "\xa9\xd1", [0x143f] = "\xa9\xd0", [0x1440] = "\xa9\xd2",
  [0x1441] = "\xab\xf3", [0x1442] = "\xce\xd2", [0x1443] = "\xce\xd3",
  [0x1446] = "\xd1\xb0", [0x1447] = "\xae\xb0", [0x1448] = "\xb1\xaf",
  [0x1449] = "\xb4\x76", [0x144a] = "\xd9\x51", [0x144b] = "\xa4\xe2",
  [0x144d] = "\xa4\x7e", [0x144e] = "\xa4\xe3", [0x1450] = "\xc9\x7d",
  [0x1451] = "\xa5\xb7", [0x1452] = "\xa5\xb6", [0x1453] = "\xa5\xb4",
  [0x1454] = "\xa5\xb5", [0x1458] = "\xa6\xab", [0x1459] = "\xc9\xe9",
  [0x145a] = "\xc9\xeb", [0x145b] = "\xa6\xaa", [0x145c] = "\xc9\xe3",
  [0x145e] = "\xc9\xe4", [0x1460] = "\xc9\xea", [0x1461] = "\xc9\xe6",
  [0x1462] = "\xc9\xe8", [0x1463] = "\xa6\xa9", [0x1464] = "\xc9\xe5",
  [0x1465] = "\xc9\xec", [0x1466] = "\xc9\xe7", [0x146d] = "\xa7\xe1",
  [0x146e] = "\xa7\xea", [0x146f] = "\xa7\xe8", [0x1470] = "\xca\xf0",
  [0x1471] = "\xca\xed", [0x1472] = "\xca\xf5", [0x1473] = "\xa7\xe6",
  [0x1474] = "\xca\xf6", [0x1476] = "\xa7\xdf", [0x1477] = "\xca\xf3",
  [0x1479] = "\xa7\xe5", [0x147a] = "\xca\xef", [0x147b] = "\xca\xee",
  [0x147c] = "\xa7\xe3", [0x147d] = "\xca\xf4", [0x147e] = "\xa7\xe4",
  [0x147f] = "\xa9\xd3", [0x1480] = "\xa7\xde", [0x1481] = "\xca\xf1",
  [0x1483] = "\xca\xe7", [0x1484] = "\xa7\xdb", [0x1486] = "\xa7\xee",
  [0x1487] = "\xca\xec", [0x1488] = "\xca\xf2", [0x1489] = "\xa7\xe0",
  [0x148a] = "\xa7\xe2", [0x148c] = "\xca\xe8", [0x148e] = "\xca\xe9",
  [0x148f] = "\xca\xea", [0x1491] = "\xa7\xed", [0x1492] = "\xa7\xe7",
  [0x1493] = "\xa7\xec", [0x1494] = "\xca\xeb", [0x1495] = "\xa7\xeb",
  [0x1496] = "\xa7\xdd", [0x1497] = "\xa7\xdc", [0x1498] = "\xa7\xe9",
  [0x14a8] = "\xa9\xe1", [0x14a9] = "\xcc\xbe", [0x14aa] = "\xcc\xb7",
  [0x14ab] = "\xa9\xdc", [0x14ac] = "\xa9\xef", [0x14ad] = "\xcc\xb3",
  [0x14ae] = "\xcc\xba", [0x14af] = "\xcc\xbc", [0x14b0] = "\xcc\xbf",
  [0x14b1] = "\xa9\xea", [0x14b3] = "\xcc\xbb", [0x14b4] = "\xcc\xb4",
  [0x14b5] = "\xa9\xe8", [0x14b6] = "\xcc\xb8", [0x14b8] = "\xcc\xc0",
  [0x14b9] = "\xa9\xd9", [0x14bb] = "\xcc\xbd", [0x14bc] = "\xa9\xe3",
  [0x14bd] = "\xa9\xe2", [0x14be] = "\xcc\xb6", [0x14bf] = "\xa9\xd7",
  [0x14c2] = "\xa9\xd8", [0x14c4] = "\xa9\xd6", [0x14c6] = "\xa9\xee",
  [0x14c7] = "\xa9\xe6", [0x14c8] = "\xa9\xe0", [0x14c9] = "\xa9\xd4",
  [0x14ca] = "\xcc\xb9", [0x14cb] = "\xa9\xdf", [0x14cc] = "\xa9\xd5",
  [0x14cd] = "\xa9\xe7", [0x14ce] = "\xa9\xf0", [0x14cf] = "\xce\xd4",
  [0x14d0] = "\xa9\xe4", [0x14d1] = "\xcc\xb5", [0x14d2] = "\xa9\xda",
  [0x14d3] = "\xa9\xdd", [0x14d4] = "\xa9\xde", [0x14d6] = "\xa9\xec",
  [0x14d7] = "\xa9\xed", [0x14d8] = "\xa9\xeb", [0x14d9] = "\xa9\xe5",
  [0x14da] = "\xa9\xe9", [0x14db] = "\xa9\xdb", [0x14dc] = "\xab\xf4",
  [0x14eb] = "\xce\xda", [0x14ec] = "\xac\x41", [0x14ed] = "\xab\xf8",
  [0x14ee] = "\xab\xfa", [0x14ef] = "\xac\x40", [0x14f0] = "\xce\xe6",
  [0x14f1] = "\xab\xfd", [0x14f2] = "\xd1\xb1", [0x14f3] = "\xae\xb1",
  [0x14f4] = "\xac\x43", [0x14f5] = "\xce\xd7", [0x14f6] = "\xce\xdf",
  [0x14f7] = "\xab\xfe", [0x14f8] = "\xce\xde", [0x14f9] = "\xce\xdb",
  [0x14fa] = "\xce\xe3", [0x14fb] = "\xce\xe5", [0x14fc] = "\xab\xf7",
  [0x14fd] = "\xab\xfb", [0x14fe] = "\xac\x42", [0x14ff] = "\xae\xb3",
  [0x1500] = "\xce\xe0", [0x1501] = "\xab\xf9", [0x1502] = "\xac\x45",
  [0x1503] = "\xce\xd9", [0x1507] = "\xab\xfc", [0x1508] = "\xae\xb2",
  [0x1509] = "\xab\xf6", [0x150b] = "\xce\xd6", [0x150c] = "\xce\xdd",
  [0x150d] = "\xce\xd5", [0x150e] = "\xce\xd8", [0x150f] = "\xce\xdc",
  [0x1510] = "\xd1\xb2", [0x1511] = "\xac\x44", [0x1513] = "\xce\xe1",
  [0x1514] = "\xce\xe2", [0x1515] = "\xce\xe4", [0x1516] = "\xab\xf5",
  [0x1528] = "\xae\xc1", [0x1529] = "\xd1\xbe", [0x152a] = "\xae\xbf",
  [0x152b] = "\xae\xc0", [0x152c] = "\xd1\xb4", [0x152d] = "\xd1\xc4",
  [0x152f] = "\xae\xb6", [0x1532] = "\xd5\x66", [0x1533] = "\xd1\xc6",
  [0x1534] = "\xd1\xc0", [0x1536] = "\xd1\xb7", [0x1538] = "\xd1\xc9",
  [0x1539] = "\xd1\xba", [0x153a] = "\xae\xbc", [0x153b] = "\xd5\x7d",
  [0x153c] = "\xd1\xbd", [0x153d] = "\xae\xbe", [0x153e] = "\xae\xb5",
  [0x1540] = "\xd1\xcb", [0x1541] = "\xd1\xbf", [0x1542] = "\xae\xb8",
  [0x1543] = "\xd1\xb8", [0x1544] = "\xd1\xb5", [0x1545] = "\xd1\xb6",
  [0x1546] = "\xae\xb9", [0x1547] = "\xd1\xc5", [0x1548] = "\xd1\xcc",
  [0x1549] = "\xae\xbb", [0x154a] = "\xd1\xbc", [0x154b] = "\xd1\xbb",
  [0x154c] = "\xae\xc3", [0x154d] = "\xae\xc2", [0x154e] = "\xae\xb4",
  [0x154f] = "\xae\xba", [0x1550] = "\xae\xbd", [0x1551] = "\xd1\xc8",
  [0x1554] = "\xd1\xc2", [0x1555] = "\xae\xb7", [0x1556] = "\xd1\xb3",
  [0x1557] = "\xd1\xca", [0x1558] = "\xd1\xc1", [0x1559] = "\xd1\xc3",
  [0x155a] = "\xd1\xc7", [0x1565] = "\xd5\x67", [0x1567] = "\xb1\xb7",
  [0x1568] = "\xb1\xcb", [0x1569] = "\xb1\xca", [0x156b] = "\xb1\xbf",
  [0x156d] = "\xd5\x79", [0x156e] = "\xd5\x75", [0x156f] = "\xd5\x72",
  [0x1570] = "\xd5\xa6", [0x1571] = "\xb1\xba", [0x1572] = "\xb1\xb2",
  [0x1575] = "\xd5\x77", [0x1576] = "\xb4\xa8", [0x1577] = "\xb1\xb6",
  [0x1578] = "\xd5\xa1", [0x157a] = "\xb1\xcc", [0x157b] = "\xb1\xc9",
  [0x157c] = "\xd5\x7b", [0x157d] = "\xd5\x6a", [0x1580] = "\xb1\xc8",
  [0x1581] = "\xd5\xa3", [0x1582] = "\xd5\x69", [0x1583] = "\xb1\xbd",
  [0x1584] = "\xb1\xc1", [0x1585] = "\xd5\xa2", [0x1587] = "\xd5\x73",
  [0x1588] = "\xb1\xc2", [0x1589] = "\xb1\xbc", [0x158a] = "\xd5\x68",
  [0x158c] = "\xb4\x78", [0x158d] = "\xd5\xa5", [0x158e] = "\xd5\x71",
  [0x158f] = "\xb1\xc7", [0x1590] = "\xd5\x74", [0x1591] = "\xd5\xa4",
  [0x1592] = "\xb1\xc6", [0x1594] = "\xd9\x52", [0x1596] = "\xb1\xb3",
  [0x1597] = "\xd5\x6f", [0x1598] = "\xb1\xb8", [0x1599] = "\xb1\xc3",
  [0x159b] = "\xb1\xbe", [0x159c] = "\xd5\x78", [0x159d] = "\xd5\x6e",
  [0x159e] = "\xd5\x6c", [0x159f] = "\xd5\x7e", [0x15a0] = "\xb1\xb0",
  [0x15a1] = "\xb1\xc4", [0x15a2] = "\xb1\xb4", [0x15a3] = "\xb4\x77",
  [0x15a4] = "\xd5\x7c", [0x15a5] = "\xb1\xb5", [0x15a7] = "\xb1\xb1",
  [0x15a8] = "\xb1\xc0", [0x15a9] = "\xb1\xbb", [0x15aa] = "\xb1\xb9",
  [0x15ab] = "\xd5\x70", [0x15ac] = "\xb1\xc5", [0x15ad] = "\xd5\x6d",
  [0x15ae] = "\xd5\x7a", [0x15af] = "\xd5\x76", [0x15b0] = "\xd9\x54",
  [0x15b1] = "\xd9\x53", [0x15bd] = "\xd5\x6b", [0x15be] = "\xd9\x64",
  [0x15c0] = "\xb4\x7a", [0x15c2] = "\xd9\x6a", [0x15c3] = "\xd9\x59",
  [0x15c4] = "\xd9\x67", [0x15c5] = "\xdd\x77", [0x15c6] = "\xb4\x7d",
  [0x15c7] = "\xd9\x6b", [0x15c8] = "\xd9\x6e", [0x15c9] = "\xb4\x7c",
  [0x15ca] = "\xd9\x5c", [0x15cb] = "\xd9\x6d", [0x15cc] = "\xd9\x6c",
  [0x15cd] = "\xb4\x7e", [0x15ce] = "\xd9\x55", [0x15cf] = "\xb4\x79",
  [0x15d0] = "\xb4\xa3", [0x15d2] = "\xb4\xa1", [0x15d3] = "\xd9\x69",
  [0x15d5] = "\xd9\x5f", [0x15d6] = "\xb4\xa5", [0x15d7] = "\xd9\x70",
  [0x15d8] = "\xd9\x68", [0x15d9] = "\xd9\x71", [0x15da] = "\xb4\xad",
  [0x15db] = "\xb4\xab", [0x15dc] = "\xd9\x66", [0x15dd] = "\xd9\x65",
  [0x15df] = "\xd9\x63", [0x15e0] = "\xd9\x5d", [0x15e1] = "\xb4\xa4",
  [0x15e3] = "\xb4\xa2", [0x15e4] = "\xd1\xb9", [0x15e5] = "\xd9\x56",
  [0x15e7] = "\xdd\xb7", [0x15e8] = "\xd9\x57", [0x15e9] = "\xb4\x7b",
  [0x15ea] = "\xb4\xaa", [0x15eb] = "\xdd\x79", [0x15ed] = "\xb4\xa6",
  [0x15ee] = "\xb4\xa7", [0x15ef] = "\xd9\x58", [0x15f0] = "\xd9\x6f",
  [0x15f1] = "\xdd\x78", [0x15f2] = "\xd9\x60", [0x15f3] = "\xd9\x5b",
  [0x15f4] = "\xb4\xa9", [0x15f5] = "\xd9\x61", [0x15f6] = "\xd9\x5e",
  [0x15f9] = "\xb4\xae", [0x1606] = "\xb7\x70", [0x1609] = "\xdd\x7c",
  [0x160a] = "\xdd\xb1", [0x160b] = "\xdd\xb6", [0x160c] = "\xdd\xaa",
  [0x160d] = "\xb7\x6c", [0x160e] = "\xdd\xbb", [0x160f] = "\xb7\x69",
  [0x1610] = "\xdd\x7a", [0x1612] = "\xdd\x7b", [0x1613] = "\xb7\x62",
  [0x1614] = "\xb7\x6b", [0x1615] = "\xdd\xa4", [0x1616] = "\xb7\x6e",
  [0x1617] = "\xb7\x6f", [0x1618] = "\xdd\xa5", [0x161a] = "\xdd\xb2",
  [0x161b] = "\xdd\xb8", [0x161c] = "\xb7\x6a", [0x161e] = "\xb7\x64",
  [0x161f] = "\xdd\xa3", [0x1620] = "\xdd\x7d", [0x1621] = "\xdd\xba",
  [0x1622] = "\xdd\xa8", [0x1623] = "\xdd\xa9", [0x1624] = "\xdd\x7e",
  [0x1625] = "\xdd\xb4", [0x1626] = "\xdd\xab", [0x1627] = "\xdd\xb5",
  [0x1628] = "\xdd\xad", [0x162a] = "\xb7\x65", [0x162b] = "\xe1\xd9",
  [0x162c] = "\xb7\x68", [0x162d] = "\xb7\x66", [0x162e] = "\xdd\xb9",
  [0x162f] = "\xdd\xb0", [0x1630] = "\xdd\xac", [0x1633] = "\xdd\xa1",
  [0x1634] = "\xba\x53", [0x1635] = "\xdd\xaf", [0x1636] = "\xb7\x6d",
  [0x1637] = "\xdd\xa7", [0x1639] = "\xdd\xa6", [0x163d] = "\xb7\x67",
  [0x163e] = "\xb7\x63", [0x163f] = "\xe1\xee", [0x1640] = "\xdd\xb3",
  [0x1641] = "\xdd\xae", [0x1643] = "\xdd\xa2", [0x164b] = "\xe1\xe9",
  [0x164d] = "\xe1\xda", [0x164e] = "\xe1\xe5", [0x1650] = "\xe1\xec",
  [0x1651] = "\xba\x51", [0x1652] = "\xb4\xac", [0x1653] = "\xe1\xea",
  [0x1654] = "\xba\x4c", [0x1658] = "\xba\x4b", [0x1659] = "\xe1\xf1",
  [0x165b] = "\xe1\xdb", [0x165c] = "\xe1\xe8", [0x165d] = "\xe1\xdc",
  [0x165e] = "\xe1\xe7", [0x165f] = "\xba\x4f", [0x1660] = "\xe1\xeb",
  [0x1661] = "\xd9\x62", [0x1665] = "\xe1\xf2", [0x1666] = "\xe1\xe3",
  [0x1667] = "\xba\x52", [0x1668] = "\xe5\xba", [0x1669] = "\xbc\xaf",
  [0x166b] = "\xe1\xf0", [0x166c] = "\xe1\xef", [0x166d] = "\xba\x54",
  [0x166e] = "\xe5\xad", [0x166f] = "\xbc\xb0", [0x1670] = "\xe5\xae",
  [0x1672] = "\xe1\xdf", [0x1673] = "\xe1\xe0", [0x1674] = "\xe1\xdd",
  [0x1675] = "\xe1\xe2", [0x1676] = "\xe1\xde", [0x1677] = "\xe1\xf3",
  [0x1678] = "\xba\x4e", [0x1679] = "\xbc\xb1", [0x167a] = "\xba\x50",
  [0x167b] = "\xba\x55", [0x167d] = "\xe1\xe1", [0x167f] = "\xe1\xed",
  [0x1682] = "\xe1\xe6", [0x1685] = "\xe5\xb1", [0x1687] = "\xba\x4a",
  [0x1688] = "\xbc\xb4", [0x1689] = "\xe9\xaa", [0x168a] = "\xe5\xb6",
  [0x168b] = "\xe5\xb5", [0x168c] = "\xe5\xb7", [0x168f] = "\xe5\xb4",
  [0x1690] = "\xbc\xb5", [0x1692] = "\xbc\xbb", [0x1693] = "\xbc\xb8",
  [0x1695] = "\xbc\xb9", [0x1696] = "\xe5\xaf", [0x1697] = "\xe5\xb2",
  [0x1698] = "\xe5\xbc", [0x1699] = "\xbc\xc1", [0x169a] = "\xbc\xbf",
  [0x169c] = "\xe5\xb3", [0x169d] = "\xd9\x5a", [0x169e] = "\xbc\xb2",
  [0x169f] = "\xe5\xb9", [0x16a0] = "\xe5\xb0", [0x16a2] = "\xbc\xc2",
  [0x16a3] = "\xe5\xb8", [0x16a4] = "\xba\x4d", [0x16a5] = "\xbc\xb7",
  [0x16a6] = "\xe1\xe4", [0x16a9] = "\xbc\xba", [0x16ab] = "\xbc\xbe",
  [0x16ac] = "\xbc\xc0", [0x16ad] = "\xbc\xbd", [0x16ae] = "\xbc\xbc",
  [0x16b0] = "\xbc\xb6", [0x16b1] = "\xe5\xbb", [0x16b2] = "\xbc\xb3",
  [0x16b3] = "\xbc\xc3", [0x16bb] = "\xbe\xd8", [0x16bc] = "\xbe\xd9",
  [0x16bd] = "\xe9\xa9", [0x16be] = "\xbe\xe2", [0x16bf] = "\xbe\xdf",
  [0x16c1] = "\xbe\xd6", [0x16c2] = "\xbe\xdd", [0x16c3] = "\xe9\xab",
  [0x16c4] = "\xbe\xdb", [0x16c5] = "\xbe\xd5", [0x16c7] = "\xbe\xdc",
  [0x16c9] = "\xe9\xa8", [0x16ca] = "\xc0\xbb", [0x16cb] = "\xbe\xd7",
  [0x16cd] = "\xbe\xde", [0x16ce] = "\xc0\xba", [0x16cf] = "\xe9\xa7",
  [0x16d0] = "\xe9\xa6", [0x16d2] = "\xbe\xe0", [0x16d4] = "\xbe\xe1",
  [0x16d6] = "\xe9\xa5", [0x16d7] = "\xe9\xa4", [0x16d8] = "\xc0\xbc",
  [0x16d9] = "\xe9\xae", [0x16da] = "\xbe\xda", [0x16db] = "\xe9\xac",
  [0x16e0] = "\xc0\xbd", [0x16e2] = "\xc0\xc2", [0x16e3] = "\xec\xea",
  [0x16e4] = "\xec\xec", [0x16e6] = "\xc0\xbf", [0x16e8] = "\xec\xed",
  [0x16e9] = "\xec\xe9", [0x16eb] = "\xec\xeb", [0x16ec] = "\xc0\xc0",
  [0x16ed] = "\xc0\xc3", [0x16ef] = "\xec\xe8", [0x16f0] = "\xc0\xbe",
  [0x16f1] = "\xc0\xc1", [0x16f2] = "\xc2\x59", [0x16f3] = "\xe9\xad",
  [0x16f4] = "\xc2\x58", [0x16f7] = "\xc2\x5e", [0x16f8] = "\xef\xd4",
  [0x16fa] = "\xc2\x5c", [0x16fb] = "\xc2\x5d", [0x16fc] = "\xef\xd7",
  [0x16fd] = "\xef\xd3", [0x16fe] = "\xc2\x5a", [0x16ff] = "\xef\xd1",
  [0x1700] = "\xc3\x6b", [0x1701] = "\xef\xd5", [0x1703] = "\xef\xd6",
  [0x1704] = "\xef\xd2", [0x1706] = "\xc2\x5b", [0x1707] = "\xf2\x42",
  [0x1709] = "\xf2\x45", [0x170c] = "\xf2\x46", [0x170d] = "\xf2\x44",
  [0x170e] = "\xf2\x47", [0x170f] = "\xc3\x6c", [0x1710] = "\xf2\x43",
  [0x1713] = "\xf4\x4e", [0x1714] = "\xc4\x64", [0x1715] = "\xf4\x4d",
  [0x1716] = "\xf4\x4c", [0x1717] = "\xf4\x4b", [0x1718] = "\xc4\x63",
  [0x1719] = "\xc4\x65", [0x171b] = "\xf5\xcd", [0x171c] = "\xc4\xe2",
  [0x171d] = "\xc4\xe1", [0x1720] = "\xf6\xe1", [0x1721] = "\xf6\xe0",
  [0x1722] = "\xf6\xe3", [0x1723] = "\xc5\xcb", [0x1724] = "\xc5\x75",
  [0x1725] = "\xf7\xdd", [0x1726] = "\xf6\xe2", [0x1729] = "\xf7\xdc",
  [0x172a] = "\xc5\xcd", [0x172b] = "\xc5\xcc", [0x172c] = "\xc5\xf3",
  [0x172d] = "\xf8\xa9", [0x172e] = "\xf8\xef", [0x172f] = "\xa4\xe4",
  [0x1732] = "\xd9\x72", [0x1733] = "\xe9\xaf", [0x1736] = "\xa6\xac",
  [0x1737] = "\xca\xf7", [0x1738] = "\xa7\xf1", [0x1739] = "\xa7\xef",
  [0x173b] = "\xa7\xf0", [0x173d] = "\xcc\xc1", [0x173e] = "\xa9\xf1",
  [0x173f] = "\xac\x46", [0x1741] = "\xce\xe7", [0x1743] = "\xce\xe8",
  [0x1745] = "\xac\x47", [0x1746] = "\xd1\xce", [0x1748] = "\xae\xc4",
  [0x1749] = "\xae\xc5", [0x174a] = "\xd1\xcd", [0x174f] = "\xb1\xd3",
  [0x1751] = "\xb1\xcf", [0x1753] = "\xd5\xa7", [0x1754] = "\xb1\xd6",
  [0x1755] = "\xb1\xd5", [0x1756] = "\xb1\xce", [0x1757] = "\xb1\xd1",
  [0x1758] = "\xb1\xd4", [0x1759] = "\xb1\xd0", [0x175c] = "\xd9\x76",
  [0x175d] = "\xb1\xcd", [0x175e] = "\xb4\xaf", [0x1762] = "\xb4\xb1",
  [0x1763] = "\xb4\xb2", [0x1764] = "\xd9\x75", [0x1765] = "\xd9\x78",
  [0x1766] = "\xb4\xb0", [0x1767] = "\xd9\x73", [0x1768] = "\xd9\x77",
  [0x176a] = "\xd9\x74", [0x176c] = "\xb7\x71", [0x176f] = "\xdd\xbc",
  [0x1772] = "\xba\x56", [0x1773] = "\xe1\xf4", [0x1774] = "\xbe\xe3",
  [0x1775] = "\xbc\xc4", [0x1776] = "\xe5\xbd", [0x1777] = "\xbc\xc5",
  [0x1778] = "\xbc\xc6", [0x1779] = "\xe5\xbf", [0x177a] = "\xe5\xbe",
  [0x177b] = "\xe5\xc0", [0x177c] = "\xe9\xb1", [0x177f] = "\xe9\xb0",
  [0x1780] = "\xec\xef", [0x1781] = "\xec\xee", [0x1782] = "\xc0\xc4",
  [0x1783] = "\xc0\xc5", [0x1784] = "\xf2\x48", [0x1787] = "\xa4\xe5",
  [0x178c] = "\xd9\x79", [0x1790] = "\xb4\xb4", [0x1791] = "\xb4\xb3",
  [0x1792] = "\xdd\xbd", [0x1794] = "\xef\xd8", [0x1795] = "\xc4\xe3",
  [0x1796] = "\xf7\xde", [0x1797] = "\xa4\xe6", [0x1799] = "\xae\xc6",
  [0x179b] = "\xb1\xd8", [0x179c] = "\xb1\xd7", [0x179d] = "\xd9\x7a",
  [0x179e] = "\xd9\x7b", [0x179f] = "\xb7\x72", [0x17a0] = "\xe1\xf5",
  [0x17a1] = "\xba\x57", [0x17a2] = "\xe9\xb2", [0x17a4] = "\xa4\xe7",
  [0x17a5] = "\xa5\xb8", [0x17a7] = "\xa9\xf2", [0x17a8] = "\xcc\xc2",
  [0x17aa] = "\xce\xe9", [0x17ab] = "\xac\x48", [0x17ac] = "\xb1\xd9",
  [0x17ae] = "\xd9\x7c", [0x17af] = "\xb4\xb5", [0x17b0] = "\xb7\x73",
  [0x17b2] = "\xe5\xc1", [0x17b3] = "\xe5\xc2", [0x17b6] = "\xec\xf0",
  [0x17b7] = "\xc2\x5f", [0x17b8] = "\xf8\xf0", [0x17b9] = "\xa4\xe8",
  [0x17bb] = "\xcc\xc3", [0x17bc] = "\xa9\xf3", [0x17bd] = "\xac\x49",
  [0x17bf] = "\xce\xea", [0x17c1] = "\xae\xc7", [0x17c2] = "\xd1\xd2",
  [0x17c3] = "\xd1\xd0", [0x17c4] = "\xd1\xd1", [0x17c5] = "\xae\xc8",
  [0x17c6] = "\xd1\xcf", [0x17cb] = "\xb1\xdb", [0x17cc] = "\xb1\xdc",
  [0x17cd] = "\xd5\xa8", [0x17ce] = "\xb1\xdd", [0x17cf] = "\xb1\xda",
  [0x17d0] = "\xd9\x7d", [0x17d2] = "\xd9\x7e", [0x17d3] = "\xdd\xbe",
  [0x17d6] = "\xba\x59", [0x17d7] = "\xba\x58", [0x17da] = "\xec\xf1",
  [0x17db] = "\xef\xd9", [0x17dd] = "\xf2\x4a", [0x17de] = "\xf2\x49",
  [0x17df] = "\xf4\x4f", [0x17e1] = "\xc9\x5e", [0x17e2] = "\xac\x4a",
  [0x17e5] = "\xa4\xe9", [0x17e6] = "\xa5\xb9", [0x17e8] = "\xa6\xae",
  [0x17e9] = "\xa6\xad", [0x17ec] = "\xa6\xaf", [0x17ed] = "\xa6\xb0",
  [0x17ee] = "\xc9\xee", [0x17ef] = "\xc9\xed", [0x17f0] = "\xca\xf8",
  [0x17f1] = "\xa7\xf2", [0x17f2] = "\xca\xfb", [0x17f3] = "\xca\xfa",
  [0x17f4] = "\xca\xf9", [0x17f5] = "\xca\xfc", [0x17fa] = "\xa9\xf4",
  [0x17fb] = "\xcc\xc9", [0x17fc] = "\xcc\xc5", [0x17fd] = "\xcc\xce",
  [0x1800] = "\xa9\xfb", [0x1802] = "\xa9\xf9", [0x1803] = "\xcc\xca",
  [0x1804] = "\xcc\xc6", [0x1805] = "\xcc\xcd", [0x1806] = "\xa9\xf8",
  [0x1807] = "\xaa\x40", [0x1808] = "\xcc\xc8", [0x1809] = "\xcc\xc4",
  [0x180a] = "\xa9\xfe", [0x180b] = "\xcc\xcb", [0x180c] = "\xa9\xf7",
  [0x180d] = "\xcc\xcc", [0x180e] = "\xa9\xfa", [0x180f] = "\xa9\xfc",
  [0x1810] = "\xcc\xd0", [0x1811] = "\xcc\xcf", [0x1812] = "\xcc\xc7",
  [0x1813] = "\xa9\xf6", [0x1814] = "\xa9\xf5", [0x1815] = "\xa9\xfd",
  [0x181c] = "\xce\xef", [0x181d] = "\xce\xf5", [0x181f] = "\xac\x50",
  [0x1820] = "\xac\x4d", [0x1821] = "\xce\xec", [0x1822] = "\xce\xf1",
  [0x1824] = "\xac\x53", [0x1825] = "\xac\x4b", [0x1826] = "\xce\xf0",
  [0x1827] = "\xac\x4e", [0x1828] = "\xac\x51", [0x182b] = "\xce\xf3",
  [0x182d] = "\xac\x4c", [0x182e] = "\xce\xf8", [0x182f] = "\xac\x4f",
  [0x1831] = "\xac\x52", [0x1832] = "\xce\xed", [0x1833] = "\xce\xf2",
  [0x1834] = "\xce\xf6", [0x1835] = "\xce\xee", [0x1836] = "\xce\xeb",
  [0x1839] = "\xce\xf7", [0x183a] = "\xce\xf4", [0x1841] = "\xae\xd0",
  [0x1842] = "\xae\xc9", [0x1843] = "\xae\xcc", [0x1845] = "\xae\xcf",
  [0x1847] = "\xd1\xd5", [0x1849] = "\xae\xca", [0x184a] = "\xd1\xd3",
  [0x184c] = "\xae\xce", [0x184f] = "\xae\xcb", [0x1851] = "\xd1\xd6",
  [0x1852] = "\xae\xcd", [0x1859] = "\xd5\xac", [0x185a] = "\xb1\xdf",
  [0x185b] = "\xd5\xab", [0x185c] = "\xd5\xad", [0x185d] = "\xb1\xde",
  [0x185e] = "\xb1\xe3", [0x185f] = "\xd1\xd4", [0x1861] = "\xd5\xaa",
  [0x1862] = "\xd5\xae", [0x1864] = "\xb1\xe0", [0x1865] = "\xd5\xa9",
  [0x1866] = "\xb1\xe2", [0x1868] = "\xb1\xe1", [0x186a] = "\xd9\xa7",
  [0x186c] = "\xd9\xa2", [0x186e] = "\xb4\xb6", [0x186f] = "\xb4\xba",
  [0x1870] = "\xb4\xb7", [0x1871] = "\xd9\xa5", [0x1872] = "\xd9\xa8",
  [0x1874] = "\xb4\xb8", [0x1876] = "\xb4\xb9", [0x1877] = "\xb4\xbe",
  [0x1878] = "\xdd\xc7", [0x1879] = "\xd9\xa6", [0x187a] = "\xb4\xbc",
  [0x187b] = "\xd9\xa3", [0x187c] = "\xd9\xa1", [0x187e] = "\xb4\xbd",
  [0x1880] = "\xd9\xa4", [0x1884] = "\xb7\x79", [0x1886] = "\xdd\xbf",
  [0x1887] = "\xb7\x76", [0x1888] = "\xb7\x77", [0x1889] = "\xb7\x75",
  [0x188a] = "\xdd\xc4", [0x188b] = "\xdd\xc3", [0x188c] = "\xdd\xc0",
  [0x188d] = "\xb7\x7b", [0x1890] = "\xdd\xc2", [0x1891] = "\xb4\xbb",
  [0x1894] = "\xdd\xc6", [0x1895] = "\xdd\xc1", [0x1896] = "\xb7\x78",
  [0x1897] = "\xb7\x74", [0x1898] = "\xb7\x7a", [0x1899] = "\xdd\xc5",
  [0x189d] = "\xba\x5c", [0x189f] = "\xe1\xf8", [0x18a0] = "\xe1\xf7",
  [0x18a1] = "\xe1\xf6", [0x18a2] = "\xba\x5a", [0x18a8] = "\xba\x5b",
  [0x18a9] = "\xe5\xc5", [0x18aa] = "\xe5\xc8", [0x18ab] = "\xbc\xc8",
  [0x18ae] = "\xbc\xc7", [0x18af] = "\xe5\xc9", [0x18b0] = "\xe5\xc4",
  [0x18b1] = "\xbc\xca", [0x18b2] = "\xe5\xc6", [0x18b4] = "\xbc\xc9",
  [0x18b5] = "\xe5\xc3", [0x18b7] = "\xe5\xc7", [0x18b8] = "\xbe\xe9",
  [0x18b9] = "\xbe\xe6", [0x18ba] = "\xe9\xbb", [0x18bb] = "\xe9\xba",
  [0x18bd] = "\xe9\xb9", [0x18be] = "\xe9\xb4", [0x18c0] = "\xe9\xb5",
  [0x18c4] = "\xbe\xe7", [0x18c6] = "\xbe\xe4", [0x18c7] = "\xbe\xe8",
  [0x18c8] = "\xe9\xb3", [0x18c9] = "\xbe\xe5", [0x18ca] = "\xe9\xb6",
  [0x18cb] = "\xe9\xb7", [0x18cc] = "\xe9\xbc", [0x18cf] = "\xe9\xb8",
  [0x18d2] = "\xec\xf2", [0x18d6] = "\xc0\xc7", [0x18d8] = "\xef\xdc",
  [0x18d9] = "\xc0\xc6", [0x18da] = "\xef\xda", [0x18db] = "\xef\xdb",
  [0x18dc] = "\xc2\x60", [0x18dd] = "\xc3\x6e", [0x18de] = "\xf2\x4b",
  [0x18e0] = "\xc3\x6d", [0x18e3] = "\xf4\x51", [0x18e4] = "\xf4\x52",
  [0x18e6] = "\xc4\x66", [0x18e8] = "\xf4\x50", [0x18e9] = "\xc4\xe4",
  [0x18eb] = "\xf7\xdf", [0x18ec] = "\xc5\xce", [0x18ed] = "\xf8\xaa",
  [0x18ee] = "\xf8\xab", [0x18f0] = "\xa4\xea", [0x18f2] = "\xa6\xb1",
  [0x18f3] = "\xa6\xb2", [0x18f4] = "\xa7\xf3", [0x18f6] = "\xcc\xd1",
  [0x18f7] = "\xac\x54", [0x18f8] = "\xae\xd1", [0x18f9] = "\xb1\xe4",
  [0x18fc] = "\xb0\xd2", [0x18fe] = "\xb4\xbf", [0x18ff] = "\xb4\xc0",
  [0x1900] = "\xb3\xcc", [0x1901] = "\xd9\xa9", [0x1903] = "\xb7\x7c",
  [0x1904] = "\xe1\xfa", [0x1905] = "\xe1\xf9", [0x1908] = "\xa4\xeb",
  [0x1909] = "\xa6\xb3", [0x190a] = "\xcc\xd2", [0x190b] = "\xaa\x42",
  [0x190d] = "\xaa\x41", [0x190f] = "\xce\xf9", [0x1910] = "\xce\xfa",
  [0x1912] = "\xd1\xd7", [0x1913] = "\xd1\xd8", [0x1914] = "\xae\xd2",
  [0x1915] = "\xae\xd3", [0x1917] = "\xae\xd4", [0x1918] = "\xd5\xaf",
  [0x191b] = "\xb1\xe6", [0x191d] = "\xb4\xc2", [0x191f] = "\xb4\xc1",
  [0x1920] = "\xdd\xc8", [0x1921] = "\xdf\x7a", [0x1922] = "\xe1\xfb",
  [0x1923] = "\xe9\xbd", [0x1926] = "\xc2\x61", [0x1927] = "\xc4\x67",
  [0x1928] = "\xa4\xec", [0x192a] = "\xa5\xbc", [0x192b] = "\xa5\xbd",
  [0x192c] = "\xa5\xbb", [0x192d] = "\xa5\xbe", [0x192e] = "\xa5\xba",
  [0x1931] = "\xa6\xb6", [0x1933] = "\xc9\xf6", [0x1934] = "\xa6\xb5",
  [0x1935] = "\xa6\xb7", [0x1938] = "\xc9\xf1", [0x1939] = "\xc9\xf0",
  [0x193a] = "\xc9\xf3", [0x193b] = "\xc9\xf2", [0x193c] = "\xc9\xf5",
  [0x193d] = "\xa6\xb4", [0x193e] = "\xc9\xef", [0x193f] = "\xc9\xf4",
  [0x1945] = "\xca\xfd", [0x1946] = "\xa7\xfd", [0x1947] = "\xca\xfe",
  [0x1948] = "\xcb\x43", [0x1949] = "\xa7\xfc", [0x194b] = "\xcb\x47",
  [0x194c] = "\xcb\x42", [0x194d] = "\xcb\x45", [0x194e] = "\xa7\xf5",
  [0x194f] = "\xa7\xf6", [0x1950] = "\xa7\xf7", [0x1951] = "\xa7\xf8",
  [0x1953] = "\xa8\x40", [0x1955] = "\xcb\x41", [0x1956] = "\xa7\xfa",
  [0x1957] = "\xa8\x41", [0x1959] = "\xcb\x40", [0x195a] = "\xcb\x46",
  [0x195c] = "\xa7\xf9", [0x195d] = "\xcb\x44", [0x195e] = "\xa7\xfb",
  [0x195f] = "\xa7\xf4", [0x1960] = "\xa7\xfe", [0x196a] = "\xaa\x57",
  [0x196c] = "\xcc\xd4", [0x196d] = "\xaa\x43", [0x196f] = "\xaa\x4d",
  [0x1970] = "\xaa\x4e", [0x1971] = "\xaa\x46", [0x1972] = "\xaa\x58",
  [0x1973] = "\xaa\x48", [0x1974] = "\xcc\xdc", [0x1975] = "\xaa\x53",
  [0x1976] = "\xcc\xd7", [0x1977] = "\xaa\x49", [0x1978] = "\xcc\xe6",
  [0x1979] = "\xcc\xe7", [0x197a] = "\xcc\xdf", [0x197b] = "\xcc\xd8",
  [0x197c] = "\xaa\x56", [0x197d] = "\xcc\xe4", [0x197e] = "\xaa\x51",
  [0x197f] = "\xaa\x4f", [0x1981] = "\xcc\xe5", [0x1983] = "\xcc\xe3",
  [0x1984] = "\xcc\xdb", [0x1985] = "\xcc\xd3", [0x1986] = "\xcc\xda",
  [0x1987] = "\xaa\x4a", [0x1989] = "\xaa\x50", [0x198b] = "\xaa\x44",
  [0x198c] = "\xcc\xde", [0x198d] = "\xcc\xdd", [0x198e] = "\xcc\xd5",
  [0x1990] = "\xaa\x52", [0x1991] = "\xcc\xe1", [0x1992] = "\xcc\xd6",
  [0x1993] = "\xaa\x55", [0x1994] = "\xcc\xe8", [0x1995] = "\xaa\x45",
  [0x1997] = "\xaa\x4c", [0x1998] = "\xcc\xd9", [0x1999] = "\xcc\xe2",
  [0x199a] = "\xaa\x54", [0x199c] = "\xaa\x47", [0x199d] = "\xaa\x4b",
  [0x199f] = "\xcc\xe0", [0x19ae] = "\xcf\x5b", [0x19af] = "\xac\x5c",
  [0x19b0] = "\xac\x69", [0x19b2] = "\xcf\x56", [0x19b3] = "\xcf\x4c",
  [0x19b4] = "\xac\x62", [0x19b5] = "\xcf\x4a", [0x19b6] = "\xac\x5b",
  [0x19b7] = "\xcf\x45", [0x19b8] = "\xac\x65", [0x19b9] = "\xcf\x52",
  [0x19ba] = "\xce\xfe", [0x19bb] = "\xcf\x41", [0x19c0] = "\xcf\x44",
  [0x19c1] = "\xce\xfb", [0x19c2] = "\xcf\x51", [0x19c3] = "\xcf\x61",
  [0x19c4] = "\xac\x60", [0x19c5] = "\xcf\x46", [0x19c6] = "\xcf\x58",
  [0x19c8] = "\xce\xfd", [0x19c9] = "\xcf\x5f", [0x19ca] = "\xcf\x60",
  [0x19cb] = "\xcf\x63", [0x19cc] = "\xcf\x5a", [0x19cd] = "\xcf\x4b",
  [0x19ce] = "\xcf\x53", [0x19cf] = "\xac\x66", [0x19d0] = "\xac\x59",
  [0x19d1] = "\xac\x61", [0x19d2] = "\xac\x6d", [0x19d3] = "\xac\x56",
  [0x19d4] = "\xac\x58", [0x19d8] = "\xcf\x43", [0x19d9] = "\xac\x6a",
  [0x19da] = "\xac\x63", [0x19db] = "\xcf\x5d", [0x19dc] = "\xcf\x40",
  [0x19dd] = "\xac\x6c", [0x19de] = "\xac\x67", [0x19df] = "\xcf\x49",
  [0x19e2] = "\xac\x6b", [0x19e3] = "\xcf\x50", [0x19e4] = "\xcf\x48",
  [0x19e5] = "\xac\x64", [0x19e6] = "\xcf\x5c", [0x19e7] = "\xcf\x54",
  [0x19e9] = "\xac\x5e", [0x19ea] = "\xcf\x62", [0x19eb] = "\xcf\x47",
  [0x19ec] = "\xac\x5a", [0x19ed] = "\xcf\x59", [0x19ee] = "\xcf\x4f",
  [0x19ef] = "\xac\x5f", [0x19f0] = "\xcf\x55", [0x19f1] = "\xac\x57",
  [0x19f2] = "\xce\xfc", [0x19f3] = "\xac\x68", [0x19f4] = "\xae\xe3",
  [0x19f5] = "\xac\x5d", [0x19f6] = "\xcf\x4e", [0x19f7] = "\xcf\x4d",
  [0x19f8] = "\xcf\x42", [0x19fa] = "\xcf\x5e", [0x19fc] = "\xcf\x57",
  [0x19ff] = "\xac\x55", [0x1a12] = "\xd1\xec", [0x1a13] = "\xae\xea",
  [0x1a14] = "\xd1\xed", [0x1a16] = "\xd1\xe1", [0x1a17] = "\xae\xdf",
  [0x1a18] = "\xae\xeb", [0x1a1a] = "\xd1\xda", [0x1a1c] = "\xd1\xe3",
  [0x1a1d] = "\xd1\xeb", [0x1a1f] = "\xd1\xd9", [0x1a20] = "\xd1\xf4",
  [0x1a21] = "\xae\xd5", [0x1a25] = "\xd1\xf3", [0x1a26] = "\xd1\xee",
  [0x1a28] = "\xd1\xef", [0x1a29] = "\xae\xdd", [0x1a2a] = "\xae\xe8",
  [0x1a2b] = "\xd1\xe5", [0x1a2d] = "\xd1\xe6", [0x1a2e] = "\xd1\xf0",
  [0x1a2f] = "\xd1\xe7", [0x1a31] = "\xd1\xe2", [0x1a32] = "\xd1\xdc",
  [0x1a33] = "\xd1\xdd", [0x1a34] = "\xd1\xea", [0x1a35] = "\xd1\xe4",
  [0x1a38] = "\xae\xd6", [0x1a39] = "\xae\xda", [0x1a3a] = "\xd1\xf2",
  [0x1a3b] = "\xd1\xde", [0x1a3c] = "\xae\xe6", [0x1a3d] = "\xae\xe2",
  [0x1a40] = "\xae\xe5", [0x1a41] = "\xae\xec", [0x1a42] = "\xae\xdb",
  [0x1a43] = "\xae\xe7", [0x1a44] = "\xd1\xe9", [0x1a45] = "\xae\xe9",
  [0x1a46] = "\xae\xd8", [0x1a48] = "\xae\xd7", [0x1a49] = "\xd1\xdb",
  [0x1a4b] = "\xd1\xdf", [0x1a4c] = "\xae\xe0", [0x1a4d] = "\xd1\xf1",
  [0x1a4e] = "\xd1\xe8", [0x1a4f] = "\xd1\xe0", [0x1a50] = "\xae\xe4",
  [0x1a51] = "\xae\xe1", [0x1a53] = "\xae\xd9", [0x1a54] = "\xae\xdc",
  [0x1a6b] = "\xd5\xc4", [0x1a6d] = "\xd5\xb4", [0x1a6e] = "\xd5\xb5",
  [0x1a6f] = "\xd5\xb9", [0x1a71] = "\xd5\xc8", [0x1a72] = "\xd5\xc5",
  [0x1a74] = "\xd5\xbe", [0x1a75] = "\xd5\xbd", [0x1a76] = "\xb1\xed",
  [0x1a77] = "\xd5\xc1", [0x1a78] = "\xd5\xd0", [0x1a79] = "\xd5\xb0",
  [0x1a7b] = "\xd5\xd1", [0x1a7c] = "\xd5\xc3", [0x1a7d] = "\xd5\xd5",
  [0x1a7e] = "\xd5\xc9", [0x1a7f] = "\xb1\xec", [0x1a80] = "\xd5\xc7",
  [0x1a81] = "\xb1\xe7", [0x1a82] = "\xb1\xfc", [0x1a83] = "\xb1\xf2",
  [0x1a85] = "\xb1\xf6", [0x1a86] = "\xb1\xf5", [0x1a87] = "\xd5\xb1",
  [0x1a89] = "\xd5\xce", [0x1a8a] = "\xd5\xd4", [0x1a8b] = "\xd5\xcc",
  [0x1a8c] = "\xd5\xd3", [0x1a8f] = "\xd5\xc0", [0x1a90] = "\xd5\xb2",
  [0x1a91] = "\xd5\xd2", [0x1a92] = "\xd5\xc2", [0x1a93] = "\xb1\xea",
  [0x1a94] = "\xb1\xf7", [0x1a96] = "\xd5\xcb", [0x1a97] = "\xb1\xf0",
  [0x1a9b] = "\xd5\xca", [0x1a9c] = "\xd5\xb3", [0x1a9d] = "\xb1\xf8",
  [0x1a9f] = "\xb1\xfa", [0x1aa0] = "\xd5\xcd", [0x1aa1] = "\xb1\xfb",
  [0x1aa2] = "\xb1\xe9", [0x1aa3] = "\xd5\xba", [0x1aa4] = "\xd5\xcf",
  [0x1aa7] = "\xb1\xef", [0x1aa8] = "\xb1\xf9", [0x1aa9] = "\xd5\xbc",
  [0x1aaa] = "\xd5\xc6", [0x1aab] = "\xd5\xb7", [0x1aac] = "\xd5\xbb",
  [0x1aad] = "\xb1\xf4", [0x1aae] = "\xd5\xb6", [0x1aaf] = "\xb1\xe8",
  [0x1ab0] = "\xb1\xf1", [0x1ab1] = "\xb1\xee", [0x1ab2] = "\xd5\xbf",
  [0x1ab3] = "\xae\xde", [0x1ab4] = "\xd9\xc0", [0x1ab5] = "\xb1\xeb",
  [0x1ac4] = "\xb1\xf3", [0x1ac6] = "\xd9\xc3", [0x1ac7] = "\xd9\xd9",
  [0x1ac8] = "\xd9\xce", [0x1ac9] = "\xb4\xd6", [0x1acb] = "\xb4\xd1",
  [0x1acc] = "\xd9\xbd", [0x1acd] = "\xb4\xd2", [0x1ace] = "\xd9\xcd",
  [0x1ad0] = "\xd9\xc6", [0x1ad1] = "\xd9\xd3", [0x1ad2] = "\xb4\xce",
  [0x1ad3] = "\xd9\xab", [0x1ad4] = "\xd9\xd5", [0x1ad5] = "\xb4\xc4",
  [0x1ad6] = "\xd9\xb3", [0x1ad7] = "\xb4\xc7", [0x1ad8] = "\xb4\xc6",
  [0x1ada] = "\xb4\xd7", [0x1adc] = "\xd9\xad", [0x1add] = "\xd9\xcf",
  [0x1ade] = "\xd9\xd0", [0x1adf] = "\xb4\xc9", [0x1ae0] = "\xb4\xc5",
  [0x1ae1] = "\xd9\xbb", [0x1ae3] = "\xb4\xd0", [0x1ae4] = "\xd9\xb6",
  [0x1ae6] = "\xd9\xd1", [0x1ae7] = "\xb4\xcc", [0x1ae8] = "\xd9\xc9",
  [0x1ae9] = "\xd9\xd6", [0x1aea] = "\xd9\xb0", [0x1aeb] = "\xd9\xb5",
  [0x1aec] = "\xd9\xaf", [0x1aee] = "\xb4\xcb", [0x1aef] = "\xd9\xc2",
  [0x1af0] = "\xdd\xde", [0x1af1] = "\xd9\xb1", [0x1af2] = "\xb4\xcf",
  [0x1af3] = "\xd9\xba", [0x1af4] = "\xd9\xd2", [0x1af5] = "\xb4\xca",
  [0x1af6] = "\xd9\xb7", [0x1af7] = "\xd9\xb4", [0x1af8] = "\xd9\xc5",
  [0x1af9] = "\xb4\xcd", [0x1afa] = "\xb4\xc3", [0x1afb] = "\xb4\xd9",
  [0x1afc] = "\xd9\xc8", [0x1afd] = "\xd9\xc7", [0x1b04] = "\xd9\xac",
  [0x1b05] = "\xb4\xc8", [0x1b06] = "\xd9\xd4", [0x1b07] = "\xd9\xbc",
  [0x1b08] = "\xd9\xbe", [0x1b0a] = "\xd9\xcb", [0x1b0b] = "\xd9\xca",
  [0x1b0c] = "\xd9\xaa", [0x1b0d] = "\xb4\xd3", [0x1b0e] = "\xb4\xd5",
  [0x1b0f] = "\xd9\xb2", [0x1b10] = "\xd9\xb9", [0x1b11] = "\xd9\xc1",
  [0x1b12] = "\xb4\xd4", [0x1b13] = "\xd9\xb8", [0x1b14] = "\xd9\xc4",
  [0x1b15] = "\xd9\xd7", [0x1b17] = "\xd9\xcc", [0x1b25] = "\xd9\xd8",
  [0x1b2a] = "\xd9\xae", [0x1b2f] = "\xdd\xf2", [0x1b30] = "\xb7\xa6",
  [0x1b32] = "\xdd\xf0", [0x1b33] = "\xdd\xdb", [0x1b34] = "\xdd\xe0",
  [0x1b35] = "\xdd\xd9", [0x1b37] = "\xdd\xec", [0x1b38] = "\xdd\xcb",
  [0x1b39] = "\xdd\xd2", [0x1b3b] = "\xdd\xea", [0x1b3c] = "\xdd\xf4",
  [0x1b3d] = "\xdd\xdc", [0x1b3f] = "\xdd\xcf", [0x1b40] = "\xdd\xe2",
  [0x1b41] = "\xdd\xe7", [0x1b42] = "\xdd\xd3", [0x1b44] = "\xdd\xe4",
  [0x1b45] = "\xdd\xd0", [0x1b48] = "\xdd\xd7", [0x1b49] = "\xdd\xd8",
  [0x1b4a] = "\xb7\xa8", [0x1b4b] = "\xdd\xeb", [0x1b4c] = "\xdd\xe9",
  [0x1b4e] = "\xdd\xcc", [0x1b4f] = "\xdd\xee", [0x1b51] = "\xdd\xef",
  [0x1b52] = "\xdd\xf1", [0x1b53] = "\xb7\xac", [0x1b54] = "\xb7\xa4",
  [0x1b56] = "\xd5\xb8", [0x1b57] = "\xdd\xd4", [0x1b58] = "\xdd\xe6",
  [0x1b59] = "\xdd\xd5", [0x1b5a] = "\xb7\xa1", [0x1b5b] = "\xb7\xb1",
  [0x1b5c] = "\xdd\xed", [0x1b5d] = "\xb7\xaf", [0x1b5e] = "\xb7\xab",
  [0x1b5f] = "\xdd\xca", [0x1b60] = "\xb7\xa3", [0x1b62] = "\xdd\xcd",
  [0x1b63] = "\xb7\xb0", [0x1b65] = "\xdd\xdd", [0x1b66] = "\xdd\xc9",
  [0x1b68] = "\xb7\xa9", [0x1b69] = "\xdd\xe1", [0x1b6a] = "\xdd\xd1",
  [0x1b6b] = "\xb7\xaa", [0x1b6c] = "\xdd\xda", [0x1b6d] = "\xb7\x7e",
  [0x1b6e] = "\xb4\xd8", [0x1b6f] = "\xdd\xe3", [0x1b70] = "\xd9\xbf",
  [0x1b71] = "\xdd\xce", [0x1b74] = "\xdd\xe8", [0x1b75] = "\xb7\xa5",
  [0x1b76] = "\xdd\xe5", [0x1b77] = "\xb7\xa2", [0x1b78] = "\xdd\xdf",
  [0x1b79] = "\xb7\xad", [0x1b7a] = "\xdd\xd6", [0x1b7b] = "\xdd\xf3",
  [0x1b82] = "\xb7\xa7", [0x1b83] = "\xde\xc6", [0x1b86] = "\xb7\xae",
  [0x1b8d] = "\xe2\x4a", [0x1b8e] = "\xe2\x48", [0x1b90] = "\xe2\x5e",
  [0x1b91] = "\xe2\x46", [0x1b93] = "\xe2\x58", [0x1b94] = "\xb7\x7d",
  [0x1b95] = "\xba\x5f", [0x1b96] = "\xe2\x42", [0x1b97] = "\xe2\x5d",
  [0x1b99] = "\xe2\x47", [0x1b9a] = "\xe2\x55", [0x1b9b] = "\xba\x64",
  [0x1b9c] = "\xba\x5d", [0x1b9e] = "\xe2\x5b", [0x1ba0] = "\xe2\x40",
  [0x1ba1] = "\xe2\x5a", [0x1ba3] = "\xba\x6f", [0x1ba4] = "\xe2\x51",
  [0x1ba5] = "\xe2\x61", [0x1ba6] = "\xba\x6d", [0x1ba7] = "\xe2\x49",
  [0x1ba8] = "\xba\x5e", [0x1ba9] = "\xe2\x4b", [0x1baa] = "\xe2\x59",
  [0x1bab] = "\xba\x67", [0x1bac] = "\xe2\x44", [0x1bad] = "\xba\x6b",
  [0x1bae] = "\xba\x61", [0x1baf] = "\xe2\x4d", [0x1bb0] = "\xe2\x43",
  [0x1bb1] = "\xe1\xfc", [0x1bb3] = "\xe2\x57", [0x1bb4] = "\xba\x68",
  [0x1bb5] = "\xe2\x60", [0x1bb6] = "\xe1\xfd", [0x1bb7] = "\xba\x65",
  [0x1bb9] = "\xe2\x53", [0x1bbb] = "\xba\x66", [0x1bbc] = "\xe2\x45",
  [0x1bbd] = "\xe2\x50", [0x1bbe] = "\xe2\x4c", [0x1bbf] = "\xe2\x4e",
  [0x1bc1] = "\xba\x60", [0x1bc2] = "\xe2\x5f", [0x1bc3] = "\xba\x6e",
  [0x1bc4] = "\xe2\x4f", [0x1bc6] = "\xe2\x62", [0x1bc9] = "\xe1\xfe",
  [0x1bca] = "\xe2\x54", [0x1bcb] = "\xba\x63", [0x1bcc] = "\xba\x6c",
  [0x1bcd] = "\xba\x6a", [0x1bce] = "\xe2\x41", [0x1bcf] = "\xe2\x56",
  [0x1bd0] = "\xba\x69", [0x1bd3] = "\xba\x62", [0x1bd4] = "\xe2\x52",
  [0x1bd9] = "\xe2\x5c", [0x1be2] = "\xe5\xd5", [0x1be4] = "\xe5\xd1",
  [0x1be5] = "\xe5\xcd", [0x1be6] = "\xe5\xe1", [0x1be7] = "\xe5\xde",
  [0x1be8] = "\xbc\xcd", [0x1beb] = "\xe5\xe5", [0x1bec] = "\xe5\xd4",
  [0x1bed] = "\xbc\xd8", [0x1bee] = "\xe5\xdb", [0x1bf1] = "\xe5\xd0",
  [0x1bf2] = "\xe5\xda", [0x1bf3] = "\xbc\xd5", [0x1bf4] = "\xe5\xee",
  [0x1bf6] = "\xe5\xeb", [0x1bf7] = "\xe5\xdd", [0x1bf8] = "\xe5\xce",
  [0x1bfb] = "\xe5\xe2", [0x1bfc] = "\xe5\xe4", [0x1bfd] = "\xbc\xd1",
  [0x1bfe] = "\xe5\xd8", [0x1bff] = "\xe5\xd3", [0x1c00] = "\xe5\xca",
  [0x1c01] = "\xbc\xce", [0x1c02] = "\xbc\xd6", [0x1c04] = "\xe5\xe7",
  [0x1c05] = "\xbc\xd7", [0x1c06] = "\xe5\xcb", [0x1c07] = "\xe5\xed",
  [0x1c08] = "\xe5\xe0", [0x1c09] = "\xe5\xe6", [0x1c0a] = "\xbc\xd4",
  [0x1c0d] = "\xe5\xe3", [0x1c0f] = "\xe5\xea", [0x1c11] = "\xbc\xd9",
  [0x1c13] = "\xbc\xd3", [0x1c14] = "\xe5\xdc", [0x1c15] = "\xe5\xcf",
  [0x1c16] = "\xe5\xef", [0x1c17] = "\xe5\xcc", [0x1c18] = "\xe5\xe8",
  [0x1c19] = "\xbc\xd0", [0x1c1b] = "\xe5\xd6", [0x1c1d] = "\xe5\xd7",
  [0x1c1e] = "\xbc\xcf", [0x1c1f] = "\xbc\xcc", [0x1c20] = "\xe5\xd2",
  [0x1c21] = "\xbc\xd2", [0x1c23] = "\xbc\xcb", [0x1c25] = "\xe5\xe9",
  [0x1c26] = "\xe5\xec", [0x1c27] = "\xe5\xd9", [0x1c28] = "\xe9\xca",
  [0x1c32] = "\xe9\xc2", [0x1c34] = "\xe9\xbe", [0x1c35] = "\xbe\xf6",
  [0x1c38] = "\xbe\xeb", [0x1c39] = "\xbe\xf0", [0x1c3a] = "\xbe\xec",
  [0x1c3b] = "\xe9\xcc", [0x1c3c] = "\xe9\xd7", [0x1c3d] = "\xbe\xea",
  [0x1c3e] = "\xe9\xc4", [0x1c3f] = "\xe9\xcd", [0x1c40] = "\xe5\xdf",
  [0x1c41] = "\xe9\xce", [0x1c44] = "\xbe\xf1", [0x1c46] = "\xe9\xdd",
  [0x1c47] = "\xbe\xf5", [0x1c48] = "\xbe\xf8", [0x1c49] = "\xe9\xc0",
  [0x1c4b] = "\xbe\xf4", [0x1c4d] = "\xe9\xdb", [0x1c4e] = "\xe9\xdc",
  [0x1c4f] = "\xe9\xd2", [0x1c50] = "\xe9\xd1", [0x1c51] = "\xe9\xc9",
  [0x1c54] = "\xe9\xd3", [0x1c55] = "\xe9\xda", [0x1c56] = "\xe9\xd9",
  [0x1c58] = "\xbe\xef", [0x1c59] = "\xbe\xed", [0x1c5a] = "\xe9\xcb",
  [0x1c5b] = "\xe9\xc8", [0x1c5d] = "\xe9\xc5", [0x1c5e] = "\xe9\xd8",
  [0x1c5f] = "\xbe\xf7", [0x1c60] = "\xe9\xd6", [0x1c61] = "\xbe\xf3",
  [0x1c62] = "\xbe\xf2", [0x1c64] = "\xe9\xd0", [0x1c66] = "\xe9\xbf",
  [0x1c67] = "\xe9\xc1", [0x1c68] = "\xe9\xc3", [0x1c69] = "\xe9\xd5",
  [0x1c6a] = "\xe9\xcf", [0x1c6b] = "\xbe\xee", [0x1c6d] = "\xe9\xc6",
  [0x1c6f] = "\xe9\xd4", [0x1c76] = "\xe9\xc7", [0x1c7e] = "\xc0\xcf",
  [0x1c7f] = "\xed\x45", [0x1c80] = "\xc0\xc8", [0x1c81] = "\xec\xf5",
  [0x1c83] = "\xed\x41", [0x1c84] = "\xc0\xca", [0x1c85] = "\xed\x48",
  [0x1c87] = "\xec\xfc", [0x1c89] = "\xec\xf7", [0x1c8c] = "\xed\x49",
  [0x1c8d] = "\xec\xf3", [0x1c8e] = "\xec\xfe", [0x1c90] = "\xc0\xd1",
  [0x1c91] = "\xed\x44", [0x1c92] = "\xed\x4a", [0x1c93] = "\xec\xfd",
  [0x1c94] = "\xc0\xc9", [0x1c95] = "\xed\x40", [0x1c96] = "\xec\xf4",
  [0x1c97] = "\xc0\xd0", [0x1c9a] = "\xed\x47", [0x1c9b] = "\xec\xf9",
  [0x1c9c] = "\xc0\xcc", [0x1c9e] = "\xec\xfb", [0x1c9f] = "\xec\xf8",
  [0x1ca0] = "\xc0\xd2", [0x1ca1] = "\xec\xfa", [0x1ca2] = "\xc0\xcb",
  [0x1ca3] = "\xc0\xce", [0x1ca4] = "\xed\x43", [0x1ca5] = "\xec\xf6",
  [0x1ca6] = "\xed\x46", [0x1ca8] = "\xed\x42", [0x1cac] = "\xc2\x63",
  [0x1cad] = "\xef\xe7", [0x1cae] = "\xc2\x68", [0x1caf] = "\xc2\x69",
  [0x1cb3] = "\xc2\x62", [0x1cb4] = "\xef\xe6", [0x1cb6] = "\xef\xe3",
  [0x1cb7] = "\xef\xe4", [0x1cb8] = "\xc2\x66", [0x1cb9] = "\xef\xde",
  [0x1cba] = "\xef\xe2", [0x1cbb] = "\xc2\x65", [0x1cbd] = "\xef\xdf",
  [0x1cc2] = "\xc2\x67", [0x1cc3] = "\xc2\x64", [0x1cc5] = "\xef\xdd",
  [0x1cc6] = "\xef\xe1", [0x1cc7] = "\xef\xe5", [0x1ccb] = "\xf2\x51",
  [0x1ccc] = "\xf2\x4e", [0x1ccd] = "\xf2\x57", [0x1ccf] = "\xf2\x56",
  [0x1cd0] = "\xf2\x54", [0x1cd1] = "\xf2\x4f", [0x1cd3] = "\xc3\x72",
  [0x1cd9] = "\xf2\x50", [0x1cda] = "\xc3\x71", [0x1cdb] = "\xc0\xcd",
  [0x1cdc] = "\xf2\x53", [0x1cdd] = "\xc3\x70", [0x1cde] = "\xf2\x58",
  [0x1cdf] = "\xf2\x52", [0x1ce0] = "\xf2\x4d", [0x1ce1] = "\xef\xe0",
  [0x1ce5] = "\xc3\x6f", [0x1ce7] = "\xf2\x4c", [0x1ce8] = "\xf4\x56",
  [0x1cea] = "\xf4\x55", [0x1ceb] = "\xf2\x55", [0x1cec] = "\xc4\x68",
  [0x1cee] = "\xf4\x59", [0x1cef] = "\xf4\x5a", [0x1cf0] = "\xf4\x54",
  [0x1cf1] = "\xf4\x58", [0x1cf3] = "\xf4\x53", [0x1cf8] = "\xf5\xd1",
  [0x1cf9] = "\xf4\x57", [0x1cfa] = "\xc4\xe7", [0x1cfb] = "\xc4\xe5",
  [0x1cfc] = "\xf5\xcf", [0x1d00] = "\xf5\xd2", [0x1d02] = "\xf5\xce",
  [0x1d03] = "\xf5\xd0", [0x1d04] = "\xc4\xe6", [0x1d08] = "\xf6\xe5",
  [0x1d09] = "\xf6\xe6", [0x1d0a] = "\xc5\x76", [0x1d0b] = "\xf6\xe4",
  [0x1d0f] = "\xf7\xe2", [0x1d10] = "\xc5\xcf", [0x1d11] = "\xf7\xe0",
  [0x1d12] = "\xf7\xe1", [0x1d13] = "\xf8\xac", [0x1d16] = "\xc6\x56",
  [0x1d17] = "\xf8\xf3", [0x1d18] = "\xf8\xf1", [0x1d19] = "\xf8\xf2",
  [0x1d1a] = "\xf8\xf4", [0x1d1e] = "\xf9\xbb", [0x1d20] = "\xa4\xed",
  [0x1d21] = "\xa6\xb8", [0x1d23] = "\xaa\x59", [0x1d25] = "\xcc\xe9",
  [0x1d28] = "\xcf\x64", [0x1d2c] = "\xd1\xf5", [0x1d2d] = "\xd1\xf7",
  [0x1d2f] = "\xd1\xf6", [0x1d31] = "\xd1\xf8", [0x1d32] = "\xb1\xfd",
  [0x1d33] = "\xd5\xd7", [0x1d34] = "\xd1\xf9", [0x1d36] = "\xd5\xd6",
  [0x1d37] = "\xd5\xd8", [0x1d38] = "\xd5\xd9", [0x1d39] = "\xd9\xda",
  [0x1d3a] = "\xb4\xdb", [0x1d3b] = "\xd9\xdb", [0x1d3c] = "\xd9\xdd",
  [0x1d3d] = "\xb4\xdc", [0x1d3e] = "\xb4\xda", [0x1d3f] = "\xd9\xdc",
  [0x1d41] = "\xdd\xfa", [0x1d42] = "\xdd\xf8", [0x1d43] = "\xdd\xf7",
  [0x1d45] = "\xdd\xf6", [0x1d46] = "\xdd\xf5", [0x1d47] = "\xb7\xb2",
  [0x1d48] = "\xdd\xf9", [0x1d49] = "\xba\x70", [0x1d4a] = "\xe2\x63",
  [0x1d4b] = "\xe2\x65", [0x1d4c] = "\xba\x71", [0x1d4d] = "\xe2\x64",
  [0x1d4e] = "\xbc\xdb", [0x1d50] = "\xbc\xda", [0x1d51] = "\xe5\xf0",
  [0x1d54] = "\xe9\xdf", [0x1d55] = "\xe9\xde", [0x1d56] = "\xe9\xe0",
  [0x1d59] = "\xbe\xf9", [0x1d5b] = "\xed\x4b", [0x1d5c] = "\xc0\xd3",
  [0x1d5e] = "\xef\xe8", [0x1d5f] = "\xc2\x6a", [0x1d60] = "\xf2\x59",
  [0x1d61] = "\xc5\x77", [0x1d62] = "\xa4\xee", [0x1d63] = "\xa5\xbf",
  [0x1d64] = "\xa6\xb9", [0x1d65] = "\xa8\x42", [0x1d66] = "\xaa\x5a",
  [0x1d67] = "\xaa\x5b", [0x1d6a] = "\xac\x6e", [0x1d6d] = "\xd1\xfa",
  [0x1d72] = "\xb7\xb3", [0x1d76] = "\xe6\xd1", [0x1d77] = "\xbe\xfa",
  [0x1d78] = "\xc2\x6b", [0x1d79] = "\xa4\xef", [0x1d7b] = "\xa6\xba",
  [0x1d7e] = "\xcc\xeb", [0x1d7f] = "\xaa\x5c", [0x1d80] = "\xcc\xea",
  [0x1d82] = "\xcf\x65", [0x1d83] = "\xac\x6f", [0x1d84] = "\xcf\x66",
  [0x1d86] = "\xac\x70", [0x1d88] = "\xd1\xfc", [0x1d89] = "\xae\xee",
  [0x1d8a] = "\xae\xed", [0x1d8c] = "\xd5\xde", [0x1d8d] = "\xd5\xdc",
  [0x1d8e] = "\xd5\xdd", [0x1d8f] = "\xd5\xdb", [0x1d91] = "\xd5\xda",
  [0x1d94] = "\xd9\xde", [0x1d95] = "\xd9\xe1", [0x1d96] = "\xb4\xde",
  [0x1d97] = "\xd9\xdf", [0x1d98] = "\xb4\xdd", [0x1d99] = "\xd9\xe0",
  [0x1d9b] = "\xdd\xfb", [0x1d9e] = "\xe2\x66", [0x1d9f] = "\xe2\x67",
  [0x1da0] = "\xe2\x68", [0x1da2] = "\xe5\xf3", [0x1da3] = "\xe5\xf2",
  [0x1da4] = "\xbc\xdc", [0x1da5] = "\xe5\xf1", [0x1da6] = "\xe5\xf4",
  [0x1da7] = "\xe9\xe1", [0x1daa] = "\xe9\xe2", [0x1dab] = "\xe9\xe3",
  [0x1dad] = "\xed\x4c", [0x1dae] = "\xc0\xd4", [0x1daf] = "\xc2\x6c",
  [0x1db0] = "\xf2\x5a", [0x1db2] = "\xc4\xe8", [0x1db3] = "\xc9\x5f",
  [0x1db5] = "\xac\x71", [0x1db6] = "\xcf\x67", [0x1db7] = "\xae\xef",
  [0x1dba] = "\xb1\xfe", [0x1dbc] = "\xb4\xdf", [0x1dbd] = "\xd9\xe2",
  [0x1dbf] = "\xb7\xb5", [0x1dc0] = "\xb7\xb4", [0x1dc3] = "\xe2\x69",
  [0x1dc4] = "\xe2\x6a", [0x1dc5] = "\xbc\xdd", [0x1dc6] = "\xbc\xde",
  [0x1dc7] = "\xe9\xe5", [0x1dc8] = "\xe9\xe4", [0x1dc9] = "\xef\xe9",
  [0x1dca] = "\xf7\xe3", [0x1dcb] = "\xa4\xf0", [0x1dcc] = "\xc9\x60",
  [0x1dcd] = "\xa5\xc0", [0x1dcf] = "\xa8\x43", [0x1dd0] = "\xcb\x48",
  [0x1dd2] = "\xac\x72", [0x1dd3] = "\xb7\xb6", [0x1dd4] = "\xa4\xf1",
  [0x1dd6] = "\xcf\x68", [0x1dd7] = "\xac\x73", [0x1dd8] = "\xcf\x69",
  [0x1dda] = "\xc0\xd5", [0x1ddb] = "\xa4\xf2", [0x1dde] = "\xcc\xec",
  [0x1de0] = "\xcf\x6a", [0x1de2] = "\xd2\x42", [0x1de3] = "\xd2\x41",
  [0x1de4] = "\xd1\xfe", [0x1de6] = "\xd1\xfd", [0x1de7] = "\xd2\x43",
  [0x1de8] = "\xd2\x40", [0x1deb] = "\xb2\x40", [0x1dec] = "\xb2\x41",
  [0x1def] = "\xb4\xe0", [0x1df0] = "\xd9\xe3", [0x1df2] = "\xd9\xe4",
  [0x1df3] = "\xd9\xe5", [0x1df7] = "\xde\x41", [0x1df8] = "\xde\x42",
  [0x1df9] = "\xde\x40", [0x1dfb] = "\xdd\xfd", [0x1dfc] = "\xdd\xfe",
  [0x1dfd] = "\xb7\xb7", [0x1dfe] = "\xe2\x6b", [0x1dff] = "\xe5\xf7",
  [0x1e00] = "\xe5\xf6", [0x1e01] = "\xe5\xf5", [0x1e02] = "\xe5\xf8",
  [0x1e03] = "\xe9\xe7", [0x1e04] = "\xe9\xe6", [0x1e05] = "\xbe\xfb",
  [0x1e06] = "\xe9\xe8", [0x1e08] = "\xc0\xd6", [0x1e09] = "\xed\x4d",
  [0x1e0b] = "\xef\xea", [0x1e0c] = "\xf2\x5b", [0x1e0d] = "\xf6\xe7",
  [0x1e0f] = "\xa4\xf3", [0x1e10] = "\xa5\xc2", [0x1e11] = "\xa5\xc1",
  [0x1e13] = "\xaa\x5d", [0x1e14] = "\xc9\x61", [0x1e15] = "\xc9\x7e",
  [0x1e16] = "\xa6\xbb", [0x1e18] = "\xc9\xf7", [0x1e19] = "\xcb\x49",
  [0x1e1a] = "\xcb\x4a", [0x1e1b] = "\xaa\x5e", [0x1e1d] = "\xcc\xed",
  [0x1e1f] = "\xac\x74", [0x1e20] = "\xcf\x6b", [0x1e21] = "\xcf\x6c",
  [0x1e23] = "\xae\xf0", [0x1e24] = "\xae\xf4", [0x1e25] = "\xd2\x44",
  [0x1e26] = "\xae\xf3", [0x1e27] = "\xae\xf1", [0x1e28] = "\xae\xf2",
  [0x1e2a] = "\xd5\xdf", [0x1e2b] = "\xb2\x42", [0x1e2c] = "\xb4\xe3",
  [0x1e2e] = "\xb4\xe1", [0x1e2f] = "\xb4\xe2", [0x1e30] = "\xd9\xe6",
  [0x1e33] = "\xba\x72", [0x1e34] = "\xa4\xf4", [0x1e36] = "\xc9\xa1",
  [0x1e38] = "\xa5\xc3", [0x1e3b] = "\xc9\xa4", [0x1e3e] = "\xa5\xc6",
  [0x1e3f] = "\xc9\xa3", [0x1e40] = "\xa5\xc5", [0x1e41] = "\xa5\xc4",
  [0x1e42] = "\xa8\x44", [0x1e43] = "\xc9\xa2", [0x1e46] = "\xc9\xf8",
  [0x1e4a] = "\xc9\xfc", [0x1e4b] = "\xc9\xfe", [0x1e4c] = "\xca\x40",
  [0x1e4d] = "\xa6\xc5", [0x1e4e] = "\xa6\xc6", [0x1e4f] = "\xc9\xfb",
  [0x1e50] = "\xa6\xc1", [0x1e52] = "\xc9\xf9", [0x1e54] = "\xc9\xfd",
  [0x1e55] = "\xa6\xc2", [0x1e57] = "\xa6\xbd", [0x1e59] = "\xa6\xbe",
  [0x1e5b] = "\xa6\xc4", [0x1e5c] = "\xc9\xfa", [0x1e5d] = "\xa6\xbc",
  [0x1e5e] = "\xa8\x45", [0x1e5f] = "\xa6\xbf", [0x1e60] = "\xa6\xc0",
  [0x1e61] = "\xa6\xc3", [0x1e65] = "\xcb\x5b", [0x1e66] = "\xcb\x59",
  [0x1e67] = "\xcb\x4c", [0x1e68] = "\xa8\x51", [0x1e69] = "\xcb\x53",
  [0x1e6a] = "\xa8\x4c", [0x1e6b] = "\xcb\x4d", [0x1e6d] = "\xcb\x55",
  [0x1e6f] = "\xcb\x52", [0x1e70] = "\xa8\x4f", [0x1e71] = "\xcb\x51",
  [0x1e72] = "\xa8\x56", [0x1e73] = "\xcb\x5a", [0x1e74] = "\xa8\x58",
  [0x1e76] = "\xa8\x5a", [0x1e78] = "\xcb\x4b", [0x1e7a] = "\xa8\x4d",
  [0x1e7b] = "\xcb\x5c", [0x1e7d] = "\xa8\x54", [0x1e7e] = "\xa8\x57",
  [0x1e80] = "\xcd\x45", [0x1e81] = "\xa8\x47", [0x1e82] = "\xa8\x5e",
  [0x1e83] = "\xa8\x55", [0x1e84] = "\xcb\x4e", [0x1e85] = "\xa8\x4a",
  [0x1e86] = "\xa8\x59", [0x1e87] = "\xcb\x56", [0x1e88] = "\xa8\x48",
  [0x1e89] = "\xa8\x49", [0x1e8a] = "\xcd\x43", [0x1e8b] = "\xcb\x4f",
  [0x1e8c] = "\xa8\x50", [0x1e8d] = "\xa8\x5b", [0x1e8e] = "\xcb\x5d",
  [0x1e8f] = "\xcb\x50", [0x1e90] = "\xa8\x4e", [0x1e92] = "\xa8\x53",
  [0x1e93] = "\xcc\xee", [0x1e94] = "\xa8\x5c", [0x1e95] = "\xcb\x57",
  [0x1e96] = "\xa8\x52", [0x1e98] = "\xa8\x5d", [0x1e99] = "\xa8\x46",
  [0x1e9a] = "\xcb\x54", [0x1e9b] = "\xa8\x4b", [0x1e9c] = "\xcb\x58",
  [0x1e9d] = "\xcd\x44", [0x1eab] = "\xaa\x6a", [0x1eac] = "\xaa\x7a",
  [0x1ead] = "\xcc\xf5", [0x1eae] = "\xaa\x71", [0x1eb0] = "\xcd\x4b",
  [0x1eb1] = "\xaa\x62", [0x1eb3] = "\xaa\x65", [0x1eb4] = "\xcd\x42",
  [0x1eb6] = "\xcc\xf3", [0x1eb7] = "\xcc\xf7", [0x1eb8] = "\xaa\x6d",
  [0x1eb9] = "\xaa\x6f", [0x1eba] = "\xcc\xfa", [0x1ebb] = "\xaa\x76",
  [0x1ebc] = "\xaa\x68", [0x1ebd] = "\xaa\x66", [0x1ebe] = "\xaa\x67",
  [0x1ebf] = "\xaa\x75", [0x1ec0] = "\xcd\x47", [0x1ec1] = "\xaa\x70",
  [0x1ec2] = "\xcc\xf9", [0x1ec3] = "\xcc\xfb", [0x1ec4] = "\xaa\x6e",
  [0x1ec5] = "\xaa\x73", [0x1ec6] = "\xcc\xfc", [0x1ec7] = "\xcd\x4a",
  [0x1ec9] = "\xac\x75", [0x1eca] = "\xaa\x79", [0x1ecc] = "\xaa\x63",
  [0x1ecd] = "\xcd\x49", [0x1ecf] = "\xcd\x4d", [0x1ed0] = "\xcc\xf8",
  [0x1ed1] = "\xcd\x4f", [0x1ed2] = "\xcd\x40", [0x1ed3] = "\xaa\x6c",
  [0x1ed4] = "\xcc\xf4", [0x1ed5] = "\xaa\x6b", [0x1ed6] = "\xaa\x7d",
  [0x1ed7] = "\xaa\x72", [0x1ed9] = "\xcc\xf2", [0x1eda] = "\xcf\x75",
  [0x1edb] = "\xaa\x78", [0x1edc] = "\xaa\x7c", [0x1edd] = "\xcd\x41",
  [0x1ede] = "\xcd\x46", [0x1ee0] = "\xaa\x7e", [0x1ee1] = "\xaa\x77",
  [0x1ee2] = "\xaa\x69", [0x1ee3] = "\xaa\x5f", [0x1ee5] = "\xaa\x64",
  [0x1ee7] = "\xcc\xf6", [0x1ee8] = "\xaa\x60", [0x1ee9] = "\xcd\x4e",
  [0x1eeb] = "\xcc\xf0", [0x1eec] = "\xcc\xef", [0x1eed] = "\xcc\xfd",
  [0x1eee] = "\xcc\xf1", [0x1eef] = "\xaa\x7b", [0x1ef0] = "\xae\xf5",
  [0x1ef1] = "\xaa\x74", [0x1ef2] = "\xcc\xfe", [0x1ef3] = "\xaa\x61",
  [0x1ef5] = "\xac\xa6", [0x1ef9] = "\xcd\x4c", [0x1f00] = "\xcf\x7c",
  [0x1f01] = "\xcf\xa1", [0x1f03] = "\xcf\xa4", [0x1f04] = "\xcf\x77",
  [0x1f07] = "\xcf\xa7", [0x1f08] = "\xcf\xaa", [0x1f09] = "\xcf\xac",
  [0x1f0a] = "\xcf\x74", [0x1f0b] = "\xac\x76", [0x1f0c] = "\xac\x7b",
  [0x1f0d] = "\xd2\x49", [0x1f0e] = "\xac\xad", [0x1f0f] = "\xcf\xa5",
  [0x1f10] = "\xcf\xad", [0x1f11] = "\xcf\x7b", [0x1f12] = "\xcf\x73",
  [0x1f16] = "\xd2\x64", [0x1f17] = "\xac\x7e", [0x1f18] = "\xcf\xa2",
  [0x1f19] = "\xcf\x78", [0x1f1a] = "\xcf\x7a", [0x1f1b] = "\xac\xa5",
  [0x1f1d] = "\xcf\x7d", [0x1f1e] = "\xac\x7d", [0x1f1f] = "\xcf\x70",
  [0x1f20] = "\xcf\xa8", [0x1f22] = "\xcf\xab", [0x1f25] = "\xac\x7a",
  [0x1f27] = "\xac\xa8", [0x1f28] = "\xcf\x6d", [0x1f29] = "\xac\xaa",
  [0x1f2a] = "\xac\x78", [0x1f2b] = "\xac\xae", [0x1f2c] = "\xcf\xa9",
  [0x1f2d] = "\xcf\x6f", [0x1f2e] = "\xac\xab", [0x1f2f] = "\xd2\x5e",
  [0x1f30] = "\xcd\x48", [0x1f31] = "\xac\x7c", [0x1f32] = "\xac\x77",
  [0x1f33] = "\xcf\x76", [0x1f34] = "\xcf\x6e", [0x1f35] = "\xac\xac",
  [0x1f36] = "\xac\xa4", [0x1f37] = "\xcf\xa3", [0x1f38] = "\xac\xa9",
  [0x1f39] = "\xac\xa7", [0x1f3a] = "\xcf\x79", [0x1f3b] = "\xac\xa1",
  [0x1f3c] = "\xcf\x71", [0x1f3d] = "\xac\xa2", [0x1f3e] = "\xac\xa3",
  [0x1f3f] = "\xcf\x72", [0x1f40] = "\xcf\xa6", [0x1f41] = "\xac\x79",
  [0x1f42] = "\xcf\x7e", [0x1f58] = "\xd2\x4c", [0x1f59] = "\xae\xfd",
  [0x1f5a] = "\xaf\x43", [0x1f5e] = "\xd2\x55", [0x1f5f] = "\xd2\x5b",
  [0x1f60] = "\xd2\x57", [0x1f61] = "\xd2\x4a", [0x1f62] = "\xd2\x4d",
  [0x1f63] = "\xd2\x46", [0x1f64] = "\xd2\x47", [0x1f65] = "\xaf\x4a",
  [0x1f66] = "\xae\xfa", [0x1f67] = "\xd2\x56", [0x1f68] = "\xd2\x5f",
  [0x1f69] = "\xaf\x45", [0x1f6a] = "\xae\xf6", [0x1f6c] = "\xaf\x40",
  [0x1f6d] = "\xd2\x4e", [0x1f6e] = "\xaf\x42", [0x1f6f] = "\xd2\x4f",
  [0x1f70] = "\xd2\x59", [0x1f74] = "\xaf\x44", [0x1f75] = "\xd2\x68",
  [0x1f76] = "\xd2\x48", [0x1f77] = "\xae\xfc", [0x1f78] = "\xae\xfb",
  [0x1f79] = "\xaf\x48", [0x1f7a] = "\xd2\x45", [0x1f7b] = "\xd2\x66",
  [0x1f7c] = "\xd2\x5a", [0x1f7d] = "\xd2\x67", [0x1f7e] = "\xd2\x61",
  [0x1f7f] = "\xd2\x53", [0x1f80] = "\xd2\x62", [0x1f82] = "\xd2\x5c",
  [0x1f83] = "\xd2\x65", [0x1f84] = "\xd2\x63", [0x1f85] = "\xaf\x49",
  [0x1f86] = "\xd2\x54", [0x1f87] = "\xae\xf9", [0x1f88] = "\xae\xf8",
  [0x1f89] = "\xaf\x41", [0x1f8a] = "\xaf\x47", [0x1f8b] = "\xd2\x60",
  [0x1f8c] = "\xaf\x46", [0x1f8d] = "\xd2\x51", [0x1f8e] = "\xb2\x43",
  [0x1f90] = "\xd2\x69", [0x1f91] = "\xd2\x50", [0x1f92] = "\xd2\x4b",
  [0x1f93] = "\xae\xfe", [0x1f94] = "\xaf\x4b", [0x1f95] = "\xae\xf7",
  [0x1f97] = "\xd2\x58", [0x1f98] = "\xd2\x5d", [0x1faa] = "\xb2\x65",
  [0x1fab] = "\xd5\xe1", [0x1fac] = "\xd5\xe5", [0x1fae] = "\xb2\x52",
  [0x1faf] = "\xb2\x50", [0x1fb2] = "\xb2\x47", [0x1fb3] = "\xd5\xe3",
  [0x1fb4] = "\xd5\xe2", [0x1fb5] = "\xb2\x5b", [0x1fb7] = "\xd5\xe8",
  [0x1fb8] = "\xb2\x55", [0x1fba] = "\xd5\xfa", [0x1fbb] = "\xd6\x47",
  [0x1fbc] = "\xb2\x44", [0x1fbd] = "\xd5\xf7", [0x1fbe] = "\xd5\xf0",
  [0x1fbf] = "\xb2\x67", [0x1fc0] = "\xd5\xe0", [0x1fc2] = "\xd5\xfc",
  [0x1fc4] = "\xb2\x64", [0x1fc5] = "\xb2\x58", [0x1fc6] = "\xb2\x63",
  [0x1fc7] = "\xb2\x4e", [0x1fc8] = "\xd5\xec", [0x1fc9] = "\xd5\xfe",
  [0x1fca] = "\xd5\xf6", [0x1fcb] = "\xb2\x4f", [0x1fcc] = "\xb2\x49",
  [0x1fcd] = "\xd6\x45", [0x1fcf] = "\xd5\xfd", [0x1fd0] = "\xd6\x40",
  [0x1fd1] = "\xb2\x51", [0x1fd2] = "\xb2\x59", [0x1fd3] = "\xd6\x42",
  [0x1fd4] = "\xd5\xea", [0x1fd5] = "\xd5\xfb", [0x1fd6] = "\xd5\xef",
  [0x1fd7] = "\xd6\x44", [0x1fd8] = "\xb2\x5e", [0x1fd9] = "\xb2\x46",
  [0x1fda] = "\xb2\x5c", [0x1fdb] = "\xd5\xf4", [0x1fdc] = "\xd5\xf2",
  [0x1fdd] = "\xd5\xf3", [0x1fde] = "\xb2\x53", [0x1fdf] = "\xd5\xee",
  [0x1fe0] = "\xd5\xed", [0x1fe1] = "\xb2\x48", [0x1fe2] = "\xd5\xe7",
  [0x1fe3] = "\xd6\x46", [0x1fe4] = "\xb2\x4a", [0x1fe5] = "\xd5\xf1",
  [0x1fe6] = "\xb2\x68", [0x1fe8] = "\xb2\x62", [0x1fe9] = "\xd5\xe6",
  [0x1fea] = "\xb2\x5f", [0x1feb] = "\xb2\x5d", [0x1fec] = "\xb2\x66",
  [0x1fed] = "\xd5\xf8", [0x1fee] = "\xb2\x61", [0x1fef] = "\xd2\x52",
  [0x1ff0] = "\xd5\xf9", [0x1ff1] = "\xb2\x60", [0x1ff2] = "\xd6\x41",
  [0x1ff3] = "\xb2\x45", [0x1ff4] = "\xd5\xf5", [0x1ff5] = "\xb2\x57",
  [0x1ff6] = "\xd5\xe9", [0x1ff7] = "\xb2\x56", [0x1ff9] = "\xb2\x54",
  [0x1ffa] = "\xb2\x4c", [0x1ffb] = "\xb2\x4b", [0x1ffc] = "\xd9\xe7",
  [0x1ffd] = "\xd6\x43", [0x2000] = "\xd5\xeb", [0x2003] = "\xd9\xfc",
  [0x2005] = "\xb2\x4d", [0x2019] = "\xb5\x41", [0x201a] = "\xb2\x5a",
  [0x201b] = "\xb4\xee", [0x201c] = "\xd9\xf6", [0x201d] = "\xb4\xfc",
  [0x201f] = "\xd9\xea", [0x2020] = "\xb4\xeb", [0x2021] = "\xb4\xe7",
  [0x2022] = "\xda\x49", [0x2023] = "\xb4\xed", [0x2024] = "\xb4\xf1",
  [0x2025] = "\xb4\xec", [0x2026] = "\xb4\xf5", [0x2027] = "\xda\x4d",
  [0x2028] = "\xda\x44", [0x202b] = "\xd9\xf1", [0x202c] = "\xb4\xfa",
  [0x202d] = "\xb4\xf4", [0x202e] = "\xd9\xfd", [0x202f] = "\xb4\xe4",
  [0x2030] = "\xda\x4a", [0x2031] = "\xda\x43", [0x2032] = "\xb4\xe8",
  [0x2033] = "\xd9\xf7", [0x2034] = "\xb4\xf7", [0x2035] = "\xda\x55",
  [0x2036] = "\xda\x56", [0x2038] = "\xb4\xe5", [0x2039] = "\xda\x48",
  [0x203a] = "\xb4\xf9", [0x203b] = "\xd9\xfb", [0x203c] = "\xd9\xed",
  [0x203d] = "\xd9\xee", [0x203e] = "\xb4\xfd", [0x203f] = "\xd9\xf2",
  [0x2040] = "\xd9\xf9", [0x2041] = "\xd9\xf3", [0x2043] = "\xb4\xfb",
  [0x2044] = "\xb5\x44", [0x2045] = "\xd9\xef", [0x2046] = "\xd9\xe8",
  [0x2047] = "\xd9\xe9", [0x2049] = "\xd9\xeb", [0x204a] = "\xb4\xea",
  [0x204b] = "\xd9\xf8", [0x204d] = "\xb4\xf8", [0x204e] = "\xb5\x42",
  [0x2051] = "\xd9\xfa", [0x2052] = "\xda\x53", [0x2053] = "\xda\x4b",
  [0x2054] = "\xb4\xe6", [0x2055] = "\xda\x51", [0x2056] = "\xb4\xf2",
  [0x2058] = "\xb4\xf0", [0x205a] = "\xda\x57", [0x205b] = "\xb4\xef",
  [0x205c] = "\xda\x41", [0x205d] = "\xd9\xf4", [0x205e] = "\xd9\xfe",
  [0x205f] = "\xb5\x47", [0x2060] = "\xda\x45", [0x2061] = "\xda\x42",
  [0x2062] = "\xd9\xf0", [0x2063] = "\xb5\x43", [0x2064] = "\xda\x4f",
  [0x2065] = "\xda\x4c", [0x2066] = "\xda\x54", [0x2067] = "\xb4\xe9",
  [0x2068] = "\xda\x40", [0x2069] = "\xb5\x46", [0x206b] = "\xda\x47",
  [0x206e] = "\xb4\xf3", [0x206f] = "\xb4\xf6", [0x2071] = "\xda\x46",
  [0x2072] = "\xb5\x45", [0x2073] = "\xd9\xf5", [0x2074] = "\xd5\xe4",
  [0x2077] = "\xda\x50", [0x2078] = "\xda\x4e", [0x2079] = "\xda\x52",
  [0x2088] = "\xd9\xec", [0x2089] = "\xb5\x40", [0x208d] = "\xde\x61",
  [0x208e] = "\xde\x60", [0x208f] = "\xde\x46", [0x2090] = "\xb7\xbd",
  [0x2092] = "\xde\x5f", [0x2093] = "\xde\x49", [0x2094] = "\xde\x4a",
  [0x2096] = "\xb7\xc7", [0x2097] = "\xde\x68", [0x2098] = "\xb7\xc2",
  [0x2099] = "\xde\x5e", [0x209b] = "\xde\x43", [0x209c] = "\xb7\xc8",
  [0x209d] = "\xb7\xbe", [0x209e] = "\xde\x52", [0x209f] = "\xde\x48",
  [0x20a0] = "\xde\x4b", [0x20a1] = "\xde\x63", [0x20a2] = "\xb7\xb8",
  [0x20a3] = "\xde\x6a", [0x20a4] = "\xde\x62", [0x20a5] = "\xb7\xc1",
  [0x20a6] = "\xde\x57", [0x20a7] = "\xb7\xcc", [0x20aa] = "\xb7\xcb",
  [0x20ab] = "\xb7\xc5", [0x20ae] = "\xde\x69", [0x20af] = "\xb7\xb9",
  [0x20b0] = "\xde\x55", [0x20b1] = "\xde\x4c", [0x20b2] = "\xde\x59",
  [0x20b3] = "\xde\x65", [0x20b4] = "\xb7\xcd", [0x20b6] = "\xb7\xbb",
  [0x20b7] = "\xde\x54", [0x20b9] = "\xde\x4d", [0x20ba] = "\xb7\xc4",
  [0x20bc] = "\xb7\xc3", [0x20bd] = "\xde\x50", [0x20be] = "\xde\x5a",
  [0x20bf] = "\xde\x64", [0x20c0] = "\xde\x47", [0x20c1] = "\xde\x51",
  [0x20c2] = "\xb7\xbc", [0x20c3] = "\xde\x5b", [0x20c4] = "\xb7\xc9",
  [0x20c5] = "\xb7\xc0", [0x20c6] = "\xde\x4e", [0x20c7] = "\xb7\xbf",
  [0x20c8] = "\xde\x45", [0x20c9] = "\xde\x53", [0x20ca] = "\xde\x67",
  [0x20cb] = "\xb4\xfe", [0x20cc] = "\xba\xb0", [0x20cd] = "\xde\x56",
  [0x20ce] = "\xe2\x6c", [0x20cf] = "\xde\x58", [0x20d0] = "\xde\x66",
  [0x20d1] = "\xb7\xc6", [0x20d2] = "\xde\x4f", [0x20d3] = "\xb7\xba",
  [0x20d4] = "\xb7\xca", [0x20d5] = "\xbc\xf0", [0x20d6] = "\xde\x44",
  [0x20d8] = "\xde\x5d", [0x20dc] = "\xde\x5c", [0x20eb] = "\xe2\xaa",
  [0x20ec] = "\xba\xad", [0x20ed] = "\xe2\x7d", [0x20ee] = "\xe2\xa4",
  [0x20ef] = "\xba\xa2", [0x20f1] = "\xe2\x6e", [0x20f2] = "\xba\xaf",
  [0x20f4] = "\xba\x77", [0x20f5] = "\xe2\x6d", [0x20f6] = "\xe2\xb0",
  [0x20f7] = "\xba\xb1", [0x20f8] = "\xe2\x71", [0x20f9] = "\xe2\xa3",
  [0x20fb] = "\xe2\x73", [0x20fc] = "\xe2\xb3", [0x20fd] = "\xe2\xaf",
  [0x20fe] = "\xba\x75", [0x20ff] = "\xba\xa1", [0x2100] = "\xe6\x53",
  [0x2101] = "\xba\xae", [0x2102] = "\xba\x7d", [0x2103] = "\xe2\x6f",
  [0x2105] = "\xe2\xae", [0x2106] = "\xba\xa3", [0x2107] = "\xe2\xab",
  [0x2108] = "\xe2\xb8", [0x2109] = "\xe2\x75", [0x210a] = "\xe2\x7e",
  [0x210d] = "\xe2\xb6", [0x210e] = "\xe2\xac", [0x210f] = "\xba\x7c",
  [0x2112] = "\xe2\x7c", [0x2113] = "\xba\x76", [0x2114] = "\xba\x74",
  [0x2115] = "\xba\xa8", [0x2118] = "\xe2\x7a", [0x2119] = "\xe2\x77",
  [0x211a] = "\xe2\x78", [0x211c] = "\xe2\xb2", [0x211e] = "\xe2\xb7",
  [0x211f] = "\xe2\xb5", [0x2120] = "\xba\x7a", [0x2121] = "\xe2\xb9",
  [0x2122] = "\xba\x7e", [0x2123] = "\xba\xa7", [0x2125] = "\xe2\x70",
  [0x2126] = "\xe5\xfa", [0x2127] = "\xe2\x79", [0x2129] = "\xba\x78",
  [0x212a] = "\xba\xac", [0x212b] = "\xba\xa9", [0x212c] = "\xba\x7b",
  [0x212d] = "\xe2\xa5", [0x212e] = "\xe2\x74", [0x212f] = "\xba\xaa",
  [0x2130] = "\xe2\xa7", [0x2131] = "\xba\xa4", [0x2132] = "\xba\xa6",
  [0x2133] = "\xba\x73", [0x2135] = "\xe2\xa9", [0x2136] = "\xe2\xa1",
  [0x2137] = "\xe2\x72", [0x2138] = "\xba\xa5", [0x2139] = "\xe2\xb1",
  [0x213a] = "\xe2\xb4", [0x213b] = "\xe2\x7b", [0x213c] = "\xe2\xa8",
  [0x213e] = "\xba\x79", [0x213f] = "\xbc\xdf", [0x2140] = "\xe2\xa6",
  [0x2141] = "\xe5\xf9", [0x2143] = "\xe2\xad", [0x214e] = "\xe2\x76",
  [0x214f] = "\xe6\x44", [0x2150] = "\xe6\x4e", [0x2151] = "\xbc\xe2",
  [0x2152] = "\xe6\x4d", [0x2153] = "\xe6\x59", [0x2154] = "\xbc\xe4",
  [0x2155] = "\xe6\x4b", [0x2157] = "\xe6\x4f", [0x2158] = "\xbc\xef",
  [0x215a] = "\xe6\x46", [0x215b] = "\xbc\xe7", [0x215d] = "\xe6\x52",
  [0x215e] = "\xe9\xf0", [0x215f] = "\xbc\xf3", [0x2160] = "\xbc\xf2",
  [0x2161] = "\xe6\x54", [0x2162] = "\xe6\x43", [0x2163] = "\xe6\x5e",
  [0x2164] = "\xbc\xed", [0x2166] = "\xbc\xe3", [0x2167] = "\xe6\x57",
  [0x2169] = "\xe6\x5b", [0x216a] = "\xe6\x60", [0x216b] = "\xe6\x55",
  [0x216c] = "\xe6\x49", [0x216d] = "\xbc\xe6", [0x216e] = "\xbc\xe9",
  [0x216f] = "\xbc\xf1", [0x2170] = "\xbc\xec", [0x2172] = "\xe6\x4c",
  [0x2173] = "\xe2\xa2", [0x2176] = "\xe6\x48", [0x2177] = "\xe6\x5f",
  [0x2178] = "\xbc\xe8", [0x217a] = "\xbc\xeb", [0x217b] = "\xe6\x61",
  [0x217c] = "\xbc\xe0", [0x217d] = "\xe6\x56", [0x217e] = "\xe5\xfb",
  [0x217f] = "\xe6\x5c", [0x2180] = "\xc0\xdf", [0x2182] = "\xe6\x4a",
  [0x2184] = "\xbc\xe1", [0x2185] = "\xe6\x45", [0x2186] = "\xbc\xe5",
  [0x2187] = "\xe5\xfc", [0x2188] = "\xba\xab", [0x2189] = "\xe6\x41",
  [0x218b] = "\xe6\x5a", [0x218c] = "\xe6\x42", [0x218d] = "\xe6\x40",
  [0x218e] = "\xbc\xea", [0x2190] = "\xe6\x58", [0x2192] = "\xe5\xfe",
  [0x2193] = "\xe6\x51", [0x2194] = "\xe6\x50", [0x2195] = "\xe6\x5d",
  [0x2196] = "\xe6\x47", [0x2197] = "\xbc\xee", [0x219e] = "\xe9\xf3",
  [0x21a0] = "\xbf\x49", [0x21a1] = "\xbe\xfe", [0x21a2] = "\xea\x40",
  [0x21a3] = "\xe9\xeb", [0x21a4] = "\xbf\x41", [0x21a5] = "\xe9\xf7",
  [0x21a6] = "\xbf\x48", [0x21a7] = "\xbf\x43", [0x21a8] = "\xe9\xf5",
  [0x21a9] = "\xed\x4f", [0x21aa] = "\xe9\xfb", [0x21ab] = "\xea\x42",
  [0x21ac] = "\xe9\xfa", [0x21ad] = "\xe9\xe9", [0x21ae] = "\xe9\xf8",
  [0x21af] = "\xea\x44", [0x21b0] = "\xea\x46", [0x21b1] = "\xbe\xfd",
  [0x21b2] = "\xea\x45", [0x21b3] = "\xbf\x44", [0x21b4] = "\xbf\x4a",
  [0x21b6] = "\xbf\x47", [0x21b8] = "\xe9\xfe", [0x21b9] = "\xbf\x46",
  [0x21ba] = "\xe9\xf9", [0x21bc] = "\xe9\xed", [0x21bd] = "\xe9\xf2",
  [0x21bf] = "\xe9\xfd", [0x21c0] = "\xbf\x45", [0x21c1] = "\xbf\x42",
  [0x21c2] = "\xbe\xfc", [0x21c3] = "\xbf\x40", [0x21c4] = "\xe9\xf1",
  [0x21c6] = "\xe5\xfd", [0x21c7] = "\xe9\xec", [0x21c8] = "\xe9\xef",
  [0x21c9] = "\xea\x41", [0x21ca] = "\xe9\xf4", [0x21cb] = "\xe9\xea",
  [0x21cc] = "\xed\x4e", [0x21cd] = "\xea\x43", [0x21ce] = "\xe9\xee",
  [0x21cf] = "\xe9\xfc", [0x21d4] = "\xed\x51", [0x21d5] = "\xc0\xe3",
  [0x21d8] = "\xc0\xd7", [0x21db] = "\xc0\xdb", [0x21dc] = "\xed\x53",
  [0x21dd] = "\xed\x59", [0x21de] = "\xed\x57", [0x21df] = "\xc0\xd9",
  [0x21e0] = "\xc0\xda", [0x21e1] = "\xc0\xe1", [0x21e2] = "\xed\x5a",
  [0x21e3] = "\xed\x52", [0x21e4] = "\xc0\xdc", [0x21e6] = "\xed\x56",
  [0x21e7] = "\xed\x55", [0x21e8] = "\xed\x5b", [0x21e9] = "\xc0\xe2",
  [0x21eb] = "\xc0\xdd", [0x21ec] = "\xc0\xe0", [0x21ed] = "\xed\x54",
  [0x21ee] = "\xc0\xe4", [0x21ef] = "\xc0\xde", [0x21f0] = "\xc0\xe5",
  [0x21f1] = "\xc0\xd8", [0x21f2] = "\xed\x58", [0x21f4] = "\xed\x50",
  [0x21f7] = "\xef\xf7", [0x21fa] = "\xc2\x71", [0x21fb] = "\xef\xf4",
  [0x21fc] = "\xef\xf6", [0x21fe] = "\xc2\x6f", [0x21ff] = "\xef\xf2",
  [0x2200] = "\xef\xf3", [0x2201] = "\xef\xee", [0x2204] = "\xe9\xf6",
  [0x2205] = "\xef\xef", [0x2206] = "\xc2\x70", [0x2207] = "\xef\xeb",
  [0x2209] = "\xc2\x6d", [0x220a] = "\xef\xf8", [0x220b] = "\xc2\x6e",
  [0x220c] = "\xef\xec", [0x220d] = "\xef\xed", [0x220e] = "\xef\xf1",
  [0x220f] = "\xc2\x73", [0x2211] = "\xc2\x72", [0x2214] = "\xef\xf0",
  [0x2215] = "\xc3\x78", [0x2216] = "\xf2\x5f", [0x2217] = "\xf2\x65",
  [0x2218] = "\xc3\x79", [0x2219] = "\xf2\x5c", [0x221a] = "\xc3\x76",
  [0x221b] = "\xc3\x73", [0x221c] = "\xf2\x67", [0x221d] = "\xc3\x77",
  [0x221f] = "\xc3\x74", [0x2220] = "\xf2\x5e", [0x2221] = "\xf2\x61",
  [0x2222] = "\xf2\x62", [0x2223] = "\xf2\x63", [0x2224] = "\xf2\x66",
  [0x2226] = "\xef\xf5", [0x2227] = "\xf2\x5d", [0x2228] = "\xc3\x75",
  [0x2229] = "\xf2\x64", [0x222a] = "\xf2\x68", [0x222b] = "\xf2\x60",
  [0x222f] = "\xf4\x5d", [0x2230] = "\xc4\x6a", [0x2231] = "\xf4\x60",
  [0x2232] = "\xc4\x6b", [0x2233] = "\xf4\x68", [0x2234] = "\xf4\x5f",
  [0x2235] = "\xf4\x5c", [0x2237] = "\xf4\x5e", [0x2238] = "\xf4\x62",
  [0x2239] = "\xf4\x65", [0x223a] = "\xf4\x64", [0x223b] = "\xf4\x67",
  [0x223c] = "\xf4\x5b", [0x223e] = "\xc4\x69", [0x223f] = "\xf4\x63",
  [0x2240] = "\xf4\x66", [0x2241] = "\xf4\x69", [0x2242] = "\xf4\x61",
  [0x2243] = "\xf5\xd3", [0x2244] = "\xf5\xd4", [0x2245] = "\xf5\xd8",
  [0x2246] = "\xf5\xd9", [0x2248] = "\xf5\xd6", [0x2249] = "\xf5\xd7",
  [0x224a] = "\xf5\xd5", [0x224c] = "\xc4\xe9", [0x2251] = "\xc5\x78",
  [0x2252] = "\xf6\xeb", [0x2255] = "\xf6\xe8", [0x2256] = "\xf6\xe9",
  [0x2257] = "\xf6\xea", [0x2258] = "\xc5\x79", [0x225a] = "\xf7\xe5",
  [0x225b] = "\xf7\xe4", [0x225d] = "\xf8\xaf", [0x225e] = "\xc5\xf4",
  [0x225f] = "\xf8\xad", [0x2260] = "\xf8\xb0", [0x2261] = "\xf8\xae",
  [0x2262] = "\xf8\xf5", [0x2263] = "\xc6\x57", [0x2264] = "\xc6\x65",
  [0x2265] = "\xf9\xa3", [0x2266] = "\xf9\x6c", [0x2268] = "\xf9\xa2",
  [0x2269] = "\xf9\xd0", [0x226a] = "\xf9\xd1", [0x226b] = "\xa4\xf5",
  [0x2270] = "\xa6\xc7", [0x2271] = "\xca\x41", [0x2274] = "\xcb\x5e",
  [0x2276] = "\xa8\x5f", [0x2278] = "\xa8\x62", [0x227a] = "\xcb\x5f",
  [0x227c] = "\xa8\x60", [0x227d] = "\xa8\x61", [0x2282] = "\xcd\x58",
  [0x2283] = "\xcd\x5a", [0x2284] = "\xcd\x55", [0x2285] = "\xcd\x52",
  [0x2286] = "\xcd\x54", [0x228a] = "\xaa\xa4", [0x228e] = "\xaa\xa2",
  [0x2291] = "\xcd\x56", [0x2292] = "\xaa\xa3", [0x2293] = "\xcd\x53",
  [0x2294] = "\xcd\x50", [0x2295] = "\xaa\xa1", [0x2296] = "\xcd\x57",
  [0x2298] = "\xcd\x51", [0x2299] = "\xaa\xa5", [0x229a] = "\xcd\x59",
  [0x229f] = "\xcf\xaf", [0x22a1] = "\xcf\xb3", [0x22a4] = "\xac\xb7",
  [0x22a9] = "\xcf\xb6", [0x22ab] = "\xac\xaf", [0x22ac] = "\xac\xb2",
  [0x22ad] = "\xac\xb4", [0x22ae] = "\xac\xb6", [0x22af] = "\xac\xb3",
  [0x22b0] = "\xcf\xb2", [0x22b1] = "\xcf\xb1", [0x22b3] = "\xac\xb1",
  [0x22b4] = "\xcf\xb4", [0x22b5] = "\xcf\xb5", [0x22b7] = "\xcf\xae",
  [0x22b8] = "\xac\xb5", [0x22ba] = "\xac\xb0", [0x22be] = "\xcf\xb0",
  [0x22c5] = "\xd2\x77", [0x22c6] = "\xd2\x78", [0x22c7] = "\xd2\x79",
  [0x22c8] = "\xaf\x50", [0x22ca] = "\xaf\x4c", [0x22cb] = "\xd2\x6e",
  [0x22cd] = "\xd2\x76", [0x22ce] = "\xd2\x7b", [0x22cf] = "\xaf\x51",
  [0x22d1] = "\xd2\x6c", [0x22d2] = "\xd2\x72", [0x22d3] = "\xd2\x6b",
  [0x22d4] = "\xd2\x75", [0x22d7] = "\xd2\x71", [0x22d8] = "\xaf\x4d",
  [0x22d9] = "\xaf\x4f", [0x22da] = "\xd2\x7a", [0x22dc] = "\xd2\x6a",
  [0x22dd] = "\xd2\x6d", [0x22de] = "\xd2\x73", [0x22e0] = "\xd2\x74",
  [0x22e1] = "\xd2\x7c", [0x22e2] = "\xd2\x70", [0x22e4] = "\xaf\x4e",
  [0x22ef] = "\xb2\x6d", [0x22f0] = "\xd6\x4e", [0x22f3] = "\xd6\x50",
  [0x22f4] = "\xd6\x4c", [0x22f6] = "\xd6\x58", [0x22f7] = "\xd6\x4a",
  [0x22f8] = "\xd6\x57", [0x22f9] = "\xb2\x69", [0x22fa] = "\xd6\x48",
  [0x22fb] = "\xda\x5b", [0x22fc] = "\xd6\x52", [0x22fd] = "\xb2\x6c",
  [0x22ff] = "\xd6\x53", [0x2300] = "\xd6\x56", [0x2302] = "\xd6\x5a",
  [0x2304] = "\xd6\x4f", [0x2306] = "\xd6\x54", [0x2309] = "\xb2\x6a",
  [0x230a] = "\xb2\x6b", [0x230b] = "\xd6\x59", [0x230c] = "\xd6\x4d",
  [0x230d] = "\xd6\x49", [0x230e] = "\xd6\x5b", [0x2310] = "\xd6\x51",
  [0x2313] = "\xd6\x55", [0x2317] = "\xd6\x4b", [0x2319] = "\xb5\x48",
  [0x231a] = "\xb5\x49", [0x231b] = "\xda\x65", [0x231c] = "\xb5\x4f",
  [0x231e] = "\xda\x59", [0x231f] = "\xda\x62", [0x2320] = "\xda\x58",
  [0x2321] = "\xb5\x4c", [0x2322] = "\xda\x60", [0x2323] = "\xda\x5e",
  [0x2325] = "\xda\x5f", [0x2326] = "\xb5\x4a", [0x2328] = "\xda\x63",
  [0x232e] = "\xda\x5c", [0x232f] = "\xda\x5a", [0x2330] = "\xb5\x4b",
  [0x2331] = "\xda\x5d", [0x2332] = "\xda\x61", [0x2336] = "\xb5\x4d",
  [0x233a] = "\xda\x64", [0x2341] = "\xde\x70", [0x2342] = "\xde\x77",
  [0x2343] = "\xde\x79", [0x2344] = "\xde\xa1", [0x2346] = "\xb7\xda",
  [0x2347] = "\xde\x6b", [0x2349] = "\xb7\xd2", [0x234b] = "\xde\x7a",
  [0x234c] = "\xb7\xd7", [0x234d] = "\xde\xa2", [0x234e] = "\xb7\xce",
  [0x2350] = "\xde\x7d", [0x2352] = "\xde\x6d", [0x2353] = "\xde\x7e",
  [0x2354] = "\xde\x6c", [0x2356] = "\xb7\xdc", [0x2358] = "\xde\x78",
  [0x2359] = "\xb7\xcf", [0x235a] = "\xde\xa3", [0x235c] = "\xb7\xd4",
  [0x235d] = "\xde\x71", [0x235e] = "\xb7\xd9", [0x235f] = "\xde\x7c",
  [0x2360] = "\xde\x6f", [0x2361] = "\xde\x76", [0x2362] = "\xde\x72",
  [0x2363] = "\xde\x6e", [0x2364] = "\xb7\xd1", [0x2365] = "\xb7\xd8",
  [0x2366] = "\xb7\xd6", [0x2367] = "\xb7\xd3", [0x2368] = "\xb7\xdb",
  [0x2369] = "\xb7\xd0", [0x236a] = "\xde\x75", [0x236c] = "\xb7\xd5",
  [0x236e] = "\xb5\x4e", [0x2370] = "\xde\x7b", [0x2372] = "\xde\x73",
  [0x2378] = "\xde\x74", [0x237b] = "\xe2\xc1", [0x237d] = "\xba\xb4",
  [0x2380] = "\xe2\xbd", [0x2381] = "\xe2\xc3", [0x2382] = "\xe2\xbf",
  [0x2384] = "\xba\xb6", [0x2385] = "\xe2\xbe", [0x2386] = "\xe2\xc2",
  [0x2387] = "\xe2\xba", [0x2389] = "\xe2\xbc", [0x238a] = "\xba\xb5",
  [0x238f] = "\xe2\xc0", [0x2390] = "\xe2\xbb", [0x2392] = "\xba\xb7",
  [0x2394] = "\xba\xb2", [0x2397] = "\xe2\xc4", [0x2399] = "\xba\xb3",
  [0x239a] = "\xe6\x67", [0x239b] = "\xe6\x64", [0x239c] = "\xe6\x70",
  [0x239d] = "\xe6\x6a", [0x239e] = "\xe6\x6c", [0x239f] = "\xbc\xf4",
  [0x23a0] = "\xe6\x66", [0x23a1] = "\xe6\x6e", [0x23a4] = "\xe6\x6d",
  [0x23a5] = "\xe6\x6b", [0x23a7] = "\xe6\x71", [0x23a8] = "\xbc\xf7",
  [0x23a9] = "\xe6\x68", [0x23aa] = "\xe6\x6f", [0x23ac] = "\xbc\xf5",
  [0x23af] = "\xe6\x63", [0x23b0] = "\xe6\x65", [0x23b1] = "\xbc\xf6",
  [0x23b2] = "\xe6\x62", [0x23b3] = "\xe6\x72", [0x23b5] = "\xe6\x69",
  [0x23b8] = "\xea\x4a", [0x23b9] = "\xbf\x51", [0x23bc] = "\xea\x55",
  [0x23bd] = "\xea\x53", [0x23be] = "\xbf\x4b", [0x23bf] = "\xea\x49",
  [0x23c0] = "\xea\x4c", [0x23c1] = "\xea\x4d", [0x23c2] = "\xea\x48",
  [0x23c3] = "\xbf\x55", [0x23c4] = "\xbf\x56", [0x23c5] = "\xea\x47",
  [0x23c6] = "\xea\x56", [0x23c7] = "\xea\x51", [0x23c8] = "\xbf\x4f",
  [0x23c9] = "\xbf\x4c", [0x23ca] = "\xea\x50", [0x23cb] = "\xea\x4e",
  [0x23ce] = "\xbf\x52", [0x23cf] = "\xea\x52", [0x23d0] = "\xbf\x4d",
  [0x23d2] = "\xbf\x4e", [0x23d4] = "\xea\x4f", [0x23d5] = "\xbf\x50",
  [0x23d6] = "\xea\x4b", [0x23d8] = "\xea\x54", [0x23d9] = "\xbf\x53",
  [0x23da] = "\xea\x57", [0x23db] = "\xea\x58", [0x23dc] = "\xbf\x54",
  [0x23df] = "\xc0\xe7", [0x23e0] = "\xc0\xee", [0x23e1] = "\xed\x5c",
  [0x23e2] = "\xed\x62", [0x23e4] = "\xed\x60", [0x23e5] = "\xc0\xea",
  [0x23e6] = "\xc0\xe9", [0x23e7] = "\xc0\xe6", [0x23e8] = "\xed\x5e",
  [0x23ec] = "\xc0\xec", [0x23ed] = "\xc0\xeb", [0x23ee] = "\xc0\xe8",
  [0x23f0] = "\xed\x61", [0x23f1] = "\xed\x5d", [0x23f2] = "\xed\x5f",
  [0x23f4] = "\xc0\xed", [0x23f8] = "\xc2\x77", [0x23f9] = "\xef\xfb",
  [0x23fb] = "\xc2\x74", [0x23fc] = "\xc2\x75", [0x23fd] = "\xef\xfd",
  [0x23fe] = "\xc2\x76", [0x23ff] = "\xef\xfa", [0x2401] = "\xef\xf9",
  [0x2402] = "\xf2\x6c", [0x2403] = "\xef\xfc", [0x2405] = "\xf2\x6d",
  [0x2406] = "\xc3\x7a", [0x2407] = "\xf2\x6b", [0x240a] = "\xf2\x6a",
  [0x240c] = "\xf2\x69", [0x240d] = "\xc3\x7b", [0x2410] = "\xc4\x6c",
  [0x2413] = "\xf4\x6a", [0x2414] = "\xf4\x6b", [0x2419] = "\xf5\xdc",
  [0x241a] = "\xf5\xdb", [0x241b] = "\xc4\xea", [0x241d] = "\xf5\xda",
  [0x241e] = "\xf6\xec", [0x241f] = "\xf6\xed", [0x2422] = "\xf7\xe6",
  [0x2423] = "\xf8\xb1", [0x2426] = "\xf8\xf6", [0x2427] = "\xf9\xbc",
  [0x2428] = "\xc6\x79", [0x2429] = "\xf9\xc6", [0x242a] = "\xa4\xf6",
  [0x242c] = "\xaa\xa6", [0x242d] = "\xaa\xa7", [0x2430] = "\xac\xb8",
  [0x2435] = "\xc0\xef", [0x2436] = "\xa4\xf7", [0x2438] = "\xaa\xa8",
  [0x2439] = "\xaf\x52", [0x243a] = "\xb7\xdd", [0x243b] = "\xa4\xf8",
  [0x243d] = "\xb2\x6e", [0x243e] = "\xba\xb8", [0x243f] = "\xc9\x62",
  [0x2441] = "\xcf\xb7", [0x2442] = "\xd2\x7d", [0x2444] = "\xe2\xc5",
  [0x2446] = "\xc0\xf0", [0x2447] = "\xa4\xf9", [0x2448] = "\xaa\xa9",
  [0x2449] = "\xcf\xb8", [0x244a] = "\xcf\xb9", [0x244b] = "\xda\x66",
  [0x244c] = "\xb5\x50", [0x244f] = "\xde\xa4", [0x2452] = "\xb7\xde",
  [0x2453] = "\xe2\xc6", [0x2456] = "\xbc\xf8", [0x2458] = "\xc3\x7c",
  [0x2459] = "\xa4\xfa", [0x245a] = "\xda\x67", [0x245b] = "\xa4\xfb",
  [0x245d] = "\xa6\xc9", [0x245e] = "\xca\x42", [0x245f] = "\xa6\xc8",
  [0x2460] = "\xa8\x65", [0x2461] = "\xa8\x64", [0x2462] = "\xa8\x63",
  [0x2463] = "\xcb\x60", [0x2467] = "\xaa\xaa", [0x2469] = "\xaa\xab",
  [0x246a] = "\xcd\x5b", [0x246c] = "\xcf\xba", [0x246e] = "\xcf\xbd",
  [0x246f] = "\xac\xba", [0x2470] = "\xcf\xbb", [0x2472] = "\xac\xb9",
  [0x2473] = "\xcf\xbc", [0x2474] = "\xac\xbb", [0x2476] = "\xd2\xa2",
  [0x2477] = "\xd2\xa1", [0x2478] = "\xd2\x7e", [0x2479] = "\xaf\x53",
  [0x247b] = "\xd6\x5d", [0x247c] = "\xd6\x5e", [0x247d] = "\xb2\x6f",
  [0x247e] = "\xd6\x5c", [0x247f] = "\xd6\x5f", [0x2480] = "\xb5\x52",
  [0x2481] = "\xb2\x70", [0x2484] = "\xb5\x51", [0x2485] = "\xda\x6b",
  [0x2486] = "\xda\x6a", [0x2488] = "\xda\x68", [0x2489] = "\xda\x69",
  [0x248b] = "\xda\x6c", [0x248c] = "\xde\xa6", [0x248d] = "\xde\xa5",
  [0x248e] = "\xde\xa9", [0x2490] = "\xde\xa8", [0x2491] = "\xde\xa7",
  [0x2492] = "\xba\xb9", [0x2493] = "\xe2\xc9", [0x2495] = "\xe2\xc8",
  [0x2496] = "\xba\xba", [0x2497] = "\xe2\xc7", [0x2498] = "\xe6\x73",
  [0x249a] = "\xe6\x74", [0x249b] = "\xbc\xf9", [0x249d] = "\xea\x59",
  [0x249e] = "\xea\x5a", [0x24a1] = "\xf2\x72", [0x24a2] = "\xc3\x7d",
  [0x24a3] = "\xf2\x71", [0x24a4] = "\xf2\x70", [0x24a5] = "\xf2\x6e",
  [0x24a6] = "\xf2\x6f", [0x24a7] = "\xc4\xeb", [0x24a8] = "\xf4\x6c",
  [0x24a9] = "\xf6\xee", [0x24aa] = "\xf8\xf7", [0x24ac] = "\xa4\xfc",
  [0x24ae] = "\xc9\xa5", [0x24af] = "\xa5\xc7", [0x24b0] = "\xc9\xa6",
  [0x24b4] = "\xca\x43", [0x24b5] = "\xca\x44", [0x24ba] = "\xcb\x66",
  [0x24bd] = "\xcb\x62", [0x24bf] = "\xcb\x61", [0x24c0] = "\xaa\xac",
  [0x24c1] = "\xcb\x65", [0x24c2] = "\xa8\x67", [0x24c3] = "\xcb\x63",
  [0x24c4] = "\xa8\x66", [0x24c5] = "\xcb\x67", [0x24c6] = "\xcb\x64",
  [0x24c9] = "\xcd\x5f", [0x24ca] = "\xcf\xbe", [0x24cb] = "\xcd\x5d",
  [0x24cc] = "\xcd\x64", [0x24ce] = "\xaa\xad", [0x24d0] = "\xaa\xb0",
  [0x24d1] = "\xcd\x65", [0x24d2] = "\xcd\x61", [0x24d4] = "\xcd\x62",
  [0x24d6] = "\xcd\x5c", [0x24d7] = "\xaa\xaf", [0x24d8] = "\xcd\x5e",
  [0x24d9] = "\xaa\xae", [0x24da] = "\xcd\x63", [0x24dc] = "\xcd\x60",
  [0x24df] = "\xcf\xc2", [0x24e0] = "\xac\xbd", [0x24e1] = "\xac\xbe",
  [0x24e3] = "\xcf\xc5", [0x24e4] = "\xcf\xbf", [0x24e6] = "\xcf\xc4",
  [0x24e8] = "\xcf\xc0", [0x24e9] = "\xac\xbc", [0x24ea] = "\xcf\xc3",
  [0x24eb] = "\xcf\xc1", [0x24f3] = "\xd2\xa8", [0x24f4] = "\xd2\xa5",
  [0x24f6] = "\xd2\xa7", [0x24f7] = "\xaf\x58", [0x24f8] = "\xaf\x57",
  [0x24f9] = "\xaf\x55", [0x24fa] = "\xd2\xa4", [0x24fb] = "\xd2\xa9",
  [0x24fc] = "\xaf\x54", [0x24fd] = "\xaf\x56", [0x24fe] = "\xd2\xa6",
  [0x24ff] = "\xd6\x67", [0x2500] = "\xd2\xa3", [0x2501] = "\xd2\xaa",
  [0x2507] = "\xd6\x62", [0x2508] = "\xd6\x66", [0x250a] = "\xd6\x65",
  [0x250b] = "\xda\x6e", [0x250c] = "\xda\x79", [0x250f] = "\xd6\x68",
  [0x2511] = "\xd6\x63", [0x2512] = "\xda\x6d", [0x2513] = "\xb2\x74",
  [0x2516] = "\xb2\x73", [0x2517] = "\xd6\x61", [0x2518] = "\xd6\x64",
  [0x2519] = "\xb2\x75", [0x251b] = "\xb2\x72", [0x251c] = "\xb2\x71",
  [0x251d] = "\xd6\x60", [0x251e] = "\xd6\x69", [0x2522] = "\xda\x70",
  [0x2523] = "\xda\x77", [0x2525] = "\xb5\x54", [0x2526] = "\xda\x76",
  [0x2527] = "\xda\x73", [0x2529] = "\xb5\x56", [0x252d] = "\xda\x75",
  [0x2530] = "\xda\x6f", [0x2531] = "\xda\x71", [0x2532] = "\xda\x74",
  [0x2533] = "\xda\x72", [0x2534] = "\xb5\x55", [0x2535] = "\xda\x78",
  [0x2536] = "\xb5\x53", [0x2537] = "\xb7\xdf", [0x253a] = "\xde\xad",
  [0x253b] = "\xde\xac", [0x253c] = "\xde\xaa", [0x253e] = "\xb7\xe2",
  [0x253f] = "\xb7\xe1", [0x2540] = "\xde\xae", [0x2542] = "\xde\xab",
  [0x2543] = "\xe2\xca", [0x2544] = "\xba\xbb", [0x2545] = "\xb7\xe0",
  [0x2549] = "\xde\xb0", [0x254a] = "\xde\xaf", [0x254c] = "\xe2\xcd",
  [0x254d] = "\xe2\xcb", [0x254e] = "\xbc\xfa", [0x2550] = "\xba\xbc",
  [0x2551] = "\xe2\xcc", [0x2552] = "\xe6\x76", [0x2557] = "\xbc\xfb",
  [0x2558] = "\xe6\x75", [0x2559] = "\xe6\x7e", [0x255a] = "\xe6\x7d",
  [0x255b] = "\xe6\x7b", [0x255d] = "\xe6\x7a", [0x255e] = "\xe6\x77",
  [0x255f] = "\xe6\x78", [0x2560] = "\xe6\x79", [0x2561] = "\xe6\x7c",
  [0x2562] = "\xe6\xa1", [0x2565] = "\xea\x5f", [0x2566] = "\xea\x5c",
  [0x2567] = "\xea\x5d", [0x2568] = "\xbf\x57", [0x2569] = "\xea\x5b",
  [0x256a] = "\xea\x61", [0x256b] = "\xea\x60", [0x256c] = "\xea\x5e",
  [0x256e] = "\xed\x64", [0x256f] = "\xed\x65", [0x2570] = "\xc0\xf1",
  [0x2572] = "\xc0\xf2", [0x2573] = "\xed\x63", [0x2575] = "\xc2\x79",
  [0x2576] = "\xef\xfe", [0x2577] = "\xc2\x78", [0x2578] = "\xc3\x7e",
  [0x257a] = "\xc3\xa1", [0x257b] = "\xc4\x6d", [0x257c] = "\xf4\x6e",
  [0x257d] = "\xf4\x6d", [0x257e] = "\xf5\xdd", [0x257f] = "\xf6\xef",
  [0x2580] = "\xc5\x7a", [0x2581] = "\xf7\xe8", [0x2582] = "\xf7\xe7",
  [0x2583] = "\xf7\xe9", [0x2584] = "\xa5\xc8", [0x2585] = "\xcf\xc6",
  [0x2586] = "\xaf\x59", [0x2587] = "\xb2\x76", [0x2588] = "\xd6\x6a",
  [0x2589] = "\xa5\xc9", [0x258a] = "\xc9\xa7", [0x258b] = "\xa4\xfd",
  [0x258e] = "\xca\x45", [0x2592] = "\xcb\x6c", [0x2593] = "\xcb\x6a",
  [0x2594] = "\xcb\x6b", [0x2595] = "\xcb\x68", [0x2596] = "\xa8\x68",
  [0x2597] = "\xcb\x69", [0x259d] = "\xcd\x6d", [0x259f] = "\xaa\xb3",
  [0x25a0] = "\xcd\x6b", [0x25a1] = "\xcd\x67", [0x25a2] = "\xcd\x6a",
  [0x25a4] = "\xcd\x66", [0x25a5] = "\xaa\xb5", [0x25a6] = "\xcd\x69",
  [0x25a8] = "\xaa\xb2", [0x25a9] = "\xaa\xb1", [0x25ab] = "\xaa\xb4",
  [0x25ac] = "\xcd\x6c", [0x25ad] = "\xcd\x68", [0x25b2] = "\xac\xc2",
  [0x25b3] = "\xac\xc5", [0x25b4] = "\xcf\xce", [0x25b5] = "\xcf\xcd",
  [0x25b6] = "\xcf\xcc", [0x25b7] = "\xac\xbf", [0x25b8] = "\xcf\xd5",
  [0x25b9] = "\xcf\xcb", [0x25bb] = "\xac\xc1", [0x25bc] = "\xd2\xaf",
  [0x25be] = "\xcf\xd2", [0x25bf] = "\xcf\xd0", [0x25c0] = "\xac\xc4",
  [0x25c2] = "\xcf\xc8", [0x25c3] = "\xcf\xd3", [0x25c5] = "\xcf\xca",
  [0x25c6] = "\xcf\xd4", [0x25c7] = "\xcf\xd1", [0x25c8] = "\xcf\xc9",
  [0x25ca] = "\xac\xc0", [0x25cb] = "\xcf\xd6", [0x25cc] = "\xcf\xc7",
  [0x25cd] = "\xac\xc3", [0x25d2] = "\xd2\xb4", [0x25d3] = "\xd2\xab",
  [0x25d4] = "\xd2\xb6", [0x25d6] = "\xd2\xae", [0x25d7] = "\xd2\xb9",
  [0x25d8] = "\xd2\xba", [0x25d9] = "\xd2\xac", [0x25da] = "\xd2\xb8",
  [0x25db] = "\xd2\xb5", [0x25dc] = "\xd2\xb3", [0x25dd] = "\xd2\xb7",
  [0x25de] = "\xaf\x5f", [0x25e0] = "\xaf\x5d", [0x25e3] = "\xd2\xb1",
  [0x25e5] = "\xd2\xad", [0x25e7] = "\xd2\xb0", [0x25e8] = "\xd2\xbb",
  [0x25e9] = "\xd2\xb2", [0x25ea] = "\xaf\x5e", [0x25eb] = "\xcf\xcf",
  [0x25ed] = "\xaf\x5a", [0x25ee] = "\xaf\x5c", [0x25f4] = "\xd6\x78",
  [0x25f5] = "\xd6\x6d", [0x25f6] = "\xd6\x6b", [0x25f8] = "\xd6\x6c",
  [0x25fa] = "\xd6\x73", [0x25fc] = "\xd6\x74", [0x25fd] = "\xd6\x70",
  [0x25fe] = "\xb2\x7b", [0x25ff] = "\xd6\x75", [0x2600] = "\xd6\x72",
  [0x2601] = "\xd6\x6f", [0x2603] = "\xb2\x79", [0x2604] = "\xd6\x6e",
  [0x2605] = "\xb2\x77", [0x2606] = "\xb2\x7a", [0x2607] = "\xd6\x71",
  [0x2608] = "\xd6\x79", [0x2609] = "\xaf\x5b", [0x260a] = "\xb2\x78",
  [0x260b] = "\xd6\x77", [0x260c] = "\xd6\x76", [0x260d] = "\xb2\x7c",
  [0x2616] = "\xda\x7e", [0x261a] = "\xda\xa1", [0x261b] = "\xb5\x60",
  [0x261d] = "\xda\xa7", [0x2620] = "\xda\xa9", [0x2621] = "\xda\xa2",
  [0x2622] = "\xb5\x5a", [0x2623] = "\xda\xa6", [0x2624] = "\xda\xa5",
  [0x2625] = "\xb5\x5b", [0x2626] = "\xb5\x61", [0x2628] = "\xb5\x62",
  [0x2629] = "\xda\xa8", [0x262a] = "\xb5\x58", [0x262b] = "\xda\x7d",
  [0x262c] = "\xda\x7b", [0x262d] = "\xda\xa3", [0x262e] = "\xda\x7a",
  [0x262f] = "\xb5\x5f", [0x2630] = "\xda\x7c", [0x2631] = "\xda\xa4",
  [0x2632] = "\xda\xaa", [0x2633] = "\xb5\x59", [0x2634] = "\xb5\x5e",
  [0x2635] = "\xb5\x5c", [0x2636] = "\xb5\x5d", [0x263a] = "\xb5\x57",
  [0x263f] = "\xb7\xe9", [0x2640] = "\xde\xb7", [0x2641] = "\xb7\xe8",
  [0x2642] = "\xde\xbb", [0x2644] = "\xde\xb1", [0x2646] = "\xde\xbc",
  [0x264a] = "\xde\xb2", [0x264b] = "\xde\xb3", [0x264d] = "\xde\xbd",
  [0x264e] = "\xde\xba", [0x264f] = "\xde\xb8", [0x2650] = "\xde\xb9",
  [0x2651] = "\xde\xb5", [0x2652] = "\xde\xb4", [0x2654] = "\xde\xbe",
  [0x2655] = "\xb7\xe5", [0x2657] = "\xde\xb6", [0x2659] = "\xb7\xea",
  [0x265a] = "\xb7\xe4", [0x265b] = "\xb7\xeb", [0x265c] = "\xb7\xec",
  [0x265e] = "\xb7\xe7", [0x265f] = "\xb7\xe6", [0x2662] = "\xe2\xce",
  [0x2663] = "\xba\xbe", [0x2664] = "\xba\xbd", [0x2667] = "\xe2\xd3",
  [0x2669] = "\xbc\xfc", [0x266a] = "\xba\xbf", [0x266d] = "\xba\xc1",
  [0x266e] = "\xe2\xd4", [0x266f] = "\xb7\xe3", [0x2670] = "\xba\xc0",
  [0x2671] = "\xe2\xd0", [0x2672] = "\xe2\xd2", [0x2673] = "\xe2\xcf",
  [0x2675] = "\xe2\xd1", [0x2679] = "\xe6\xab", [0x267c] = "\xe6\xaa",
  [0x267d] = "\xe6\xa7", [0x267e] = "\xbd\x40", [0x267f] = "\xea\x62",
  [0x2680] = "\xbd\x41", [0x2681] = "\xe6\xa6", [0x2683] = "\xbc\xfe",
  [0x2685] = "\xe6\xa8", [0x2686] = "\xe6\xa5", [0x2687] = "\xe6\xa2",
  [0x2688] = "\xe6\xa9", [0x2689] = "\xe6\xa3", [0x268a] = "\xe6\xa4",
  [0x268b] = "\xbc\xfd", [0x2690] = "\xed\x69", [0x2692] = "\xea\x66",
  [0x2694] = "\xea\x65", [0x2695] = "\xea\x67", [0x2697] = "\xed\x66",
  [0x2698] = "\xbf\x5a", [0x269a] = "\xea\x63", [0x269c] = "\xbf\x58",
  [0x269e] = "\xbf\x5c", [0x269f] = "\xbf\x5b", [0x26a0] = "\xea\x64",
  [0x26a1] = "\xea\x68", [0x26a3] = "\xbf\x59", [0x26a5] = "\xed\x6d",
  [0x26a6] = "\xc0\xf5", [0x26a7] = "\xc2\x7a", [0x26a8] = "\xc0\xf6",
  [0x26a9] = "\xc0\xf3", [0x26aa] = "\xed\x6a", [0x26ab] = "\xed\x68",
  [0x26ad] = "\xed\x6b", [0x26af] = "\xed\x6e", [0x26b0] = "\xc0\xf4",
  [0x26b1] = "\xed\x6c", [0x26b2] = "\xed\x67", [0x26b5] = "\xf0\x42",
  [0x26b6] = "\xf0\x45", [0x26b7] = "\xf2\x75", [0x26b8] = "\xf0\x40",
  [0x26ba] = "\xf4\x6f", [0x26bb] = "\xf0\x46", [0x26bd] = "\xc3\xa2",
  [0x26be] = "\xf0\x44", [0x26bf] = "\xc2\x7b", [0x26c0] = "\xf0\x41",
  [0x26c1] = "\xf0\x43", [0x26c2] = "\xf0\x47", [0x26c3] = "\xf2\x76",
  [0x26c5] = "\xf2\x74", [0x26ca] = "\xc3\xa3", [0x26cb] = "\xf2\x73",
  [0x26cf] = "\xc4\x6e", [0x26d4] = "\xc4\xed", [0x26d5] = "\xf6\xf1",
  [0x26d6] = "\xc4\xec", [0x26d7] = "\xf6\xf3", [0x26d8] = "\xf6\xf0",
  [0x26d9] = "\xf6\xf2", [0x26da] = "\xc5\xd0", [0x26db] = "\xf8\xb2",
  [0x26dc] = "\xa5\xca", [0x26dd] = "\xcd\x6e", [0x26de] = "\xd2\xbc",
  [0x26df] = "\xd2\xbd", [0x26e0] = "\xb2\x7d", [0x26e1] = "\xde\xbf",
  [0x26e2] = "\xbf\x5d", [0x26e3] = "\xc3\xa4", [0x26e4] = "\xc5\x7b",
  [0x26e5] = "\xf8\xb3", [0x26e6] = "\xa5\xcb", [0x26e8] = "\xcd\x6f",
  [0x26e9] = "\xa2\x60", [0x26ec] = "\xcf\xd7", [0x26ee] = "\xcf\xd8",
  [0x26f4] = "\xd2\xbe", [0x26f5] = "\xd2\xbf", [0x26f6] = "\xb2\x7e",
  [0x26f7] = "\xb2\xa1", [0x26fb] = "\xda\xab", [0x26fd] = "\xde\xc2",
  [0x26fe] = "\xde\xc1", [0x26ff] = "\xde\xc0", [0x2700] = "\xe2\xd5",
  [0x2702] = "\xe2\xd6", [0x2703] = "\xe2\xd7", [0x2704] = "\xba\xc2",
  [0x2707] = "\xe6\xad", [0x2708] = "\xe6\xac", [0x270b] = "\xea\x69",
  [0x270c] = "\xbf\x5e", [0x270d] = "\xbf\x5f", [0x270f] = "\xed\x72",
  [0x2710] = "\xed\x6f", [0x2711] = "\xed\x70", [0x2712] = "\xed\x71",
  [0x2713] = "\xf0\x49", [0x2714] = "\xf0\x48", [0x2715] = "\xc2\x7c",
  [0x2716] = "\xf2\x77", [0x2717] = "\xf5\xde", [0x2718] = "\xa5\xcc",
  [0x271a] = "\xac\xc6", [0x271c] = "\xb2\xa2", [0x271d] = "\xde\xc3",
  [0x271f] = "\xa5\xcd", [0x2721] = "\xd2\xc0", [0x2722] = "\xb2\xa3",
  [0x2725] = "\xb5\x63", [0x2726] = "\xb5\x64", [0x2728] = "\xa5\xce",
  [0x2729] = "\xa5\xcf", [0x272a] = "\xca\x46", [0x272b] = "\xa8\x6a",
  [0x272c] = "\xa8\x69", [0x272d] = "\xac\xc7", [0x272e] = "\xcf\xd9",
  [0x272f] = "\xda\xac", [0x2730] = "\xa5\xd0", [0x2731] = "\xa5\xd1",
  [0x2732] = "\xa5\xd2", [0x2733] = "\xa5\xd3", [0x2737] = "\xa8\x6b",
  [0x2738] = "\xa8\x6c", [0x2739] = "\xcb\x6e", [0x273a] = "\xcb\x6d",
  [0x273d] = "\xaa\xb6", [0x273e] = "\xcd\x72", [0x273f] = "\xcd\x70",
  [0x2740] = "\xcd\x71", [0x2747] = "\xcf\xda", [0x2748] = "\xcf\xdb",
  [0x274b] = "\xac\xcb", [0x274c] = "\xac\xc9", [0x274e] = "\xac\xca",
  [0x274f] = "\xac\xc8", [0x2754] = "\xaf\x60", [0x2759] = "\xaf\x64",
  [0x275a] = "\xaf\x63", [0x275b] = "\xd2\xc1", [0x275c] = "\xaf\x62",
  [0x275d] = "\xaf\x61", [0x275f] = "\xd2\xc2", [0x2762] = "\xb2\xa6",
  [0x2763] = "\xd6\x7b", [0x2764] = "\xd6\x7a", [0x2765] = "\xb2\xa4",
  [0x2766] = "\xb2\xa5", [0x276a] = "\xb5\x66", [0x276b] = "\xb5\x65",
  [0x276c] = "\xda\xae", [0x276f] = "\xda\xad", [0x2770] = "\xb2\xa7",
  [0x2776] = "\xb7\xed", [0x2777] = "\xde\xc5", [0x2778] = "\xb7\xee",
  [0x2779] = "\xde\xc4", [0x277d] = "\xe2\xd8", [0x277e] = "\xe6\xae",
  [0x277f] = "\xbd\x42", [0x2780] = "\xea\x6a", [0x2784] = "\xed\x73",
  [0x2786] = "\xc3\xa6", [0x2787] = "\xc3\xa5", [0x278a] = "\xc5\x7c",
  [0x278b] = "\xa5\xd4", [0x278c] = "\xcd\x73", [0x278f] = "\xb2\xa8",
  [0x2790] = "\xe2\xd9", [0x2791] = "\xba\xc3", [0x2794] = "\xcb\x6f",
  [0x2795] = "\xcb\x70", [0x2798] = "\xcd\x74", [0x2799] = "\xaa\xb8",
  [0x279a] = "\xaa\xb9", [0x279d] = "\xaa\xb7", [0x27a2] = "\xac\xcf",
  [0x27a3] = "\xac\xd0", [0x27a4] = "\xac\xcd", [0x27a5] = "\xac\xce",
  [0x27a7] = "\xcf\xdc", [0x27aa] = "\xcf\xdd", [0x27ab] = "\xac\xcc",
  [0x27b0] = "\xd2\xc3", [0x27b2] = "\xaf\x68", [0x27b3] = "\xaf\x69",
  [0x27b5] = "\xb2\xab", [0x27b6] = "\xd2\xc9", [0x27b8] = "\xaf\x6e",
  [0x27b9] = "\xaf\x6c", [0x27ba] = "\xd2\xca", [0x27bb] = "\xd2\xc5",
  [0x27bc] = "\xaf\x6b", [0x27bd] = "\xaf\x6a", [0x27be] = "\xaf\x65",
  [0x27bf] = "\xd2\xc8", [0x27c0] = "\xd2\xc7", [0x27c1] = "\xd2\xc4",
  [0x27c2] = "\xaf\x6d", [0x27c4] = "\xd2\xc6", [0x27c5] = "\xaf\x66",
  [0x27c7] = "\xaf\x67", [0x27ca] = "\xb2\xac", [0x27cb] = "\xd6\xa1",
  [0x27cc] = "\xd6\xa2", [0x27cd] = "\xb2\xad", [0x27ce] = "\xd6\x7c",
  [0x27cf] = "\xd6\x7e", [0x27d0] = "\xd6\xa4", [0x27d1] = "\xd6\xa3",
  [0x27d2] = "\xd6\x7d", [0x27d4] = "\xb2\xa9", [0x27d5] = "\xb2\xaa",
  [0x27d7] = "\xda\xb6", [0x27d8] = "\xb5\x6b", [0x27d9] = "\xb5\x6a",
  [0x27da] = "\xda\xb0", [0x27db] = "\xb5\x68", [0x27dd] = "\xda\xb3",
  [0x27de] = "\xb5\x6c", [0x27df] = "\xda\xb4", [0x27e0] = "\xb5\x6d",
  [0x27e1] = "\xda\xb1", [0x27e2] = "\xb5\x67", [0x27e3] = "\xb5\x69",
  [0x27e4] = "\xda\xb5", [0x27e6] = "\xda\xb2", [0x27e7] = "\xda\xaf",
  [0x27ed] = "\xde\xd2", [0x27ef] = "\xde\xc7", [0x27f0] = "\xb7\xf0",
  [0x27f1] = "\xb7\xf3", [0x27f2] = "\xb7\xf2", [0x27f3] = "\xb7\xf7",
  [0x27f4] = "\xb7\xf6", [0x27f5] = "\xde\xd3", [0x27f6] = "\xde\xd1",
  [0x27f7] = "\xde\xca", [0x27f8] = "\xde\xce", [0x27f9] = "\xde\xcd",
  [0x27fa] = "\xb7\xf4", [0x27fb] = "\xde\xd0", [0x27fc] = "\xde\xcc",
  [0x27fd] = "\xde\xd4", [0x27fe] = "\xde\xcb", [0x27ff] = "\xb7\xf5",
  [0x2800] = "\xb7\xef", [0x2801] = "\xb7\xf1", [0x2803] = "\xde\xc9",
  [0x2808] = "\xe2\xdb", [0x2809] = "\xba\xc7", [0x280a] = "\xe2\xdf",
  [0x280b] = "\xba\xc6", [0x280c] = "\xe2\xdc", [0x280d] = "\xba\xc5",
  [0x280f] = "\xde\xc8", [0x2810] = "\xde\xcf", [0x2811] = "\xe2\xde",
  [0x2813] = "\xba\xc8", [0x2814] = "\xe2\xe0", [0x2815] = "\xe2\xdd",
  [0x2816] = "\xe2\xda", [0x2819] = "\xe6\xb1", [0x281a] = "\xe6\xb5",
  [0x281b] = "\xe6\xb7", [0x281c] = "\xe6\xb3", [0x281d] = "\xe6\xb2",
  [0x281e] = "\xe6\xb0", [0x281f] = "\xbd\x45", [0x2820] = "\xbd\x43",
  [0x2821] = "\xbd\x48", [0x2822] = "\xbd\x49", [0x2823] = "\xe6\xb4",
  [0x2824] = "\xbd\x46", [0x2825] = "\xe6\xaf", [0x2826] = "\xbd\x47",
  [0x2827] = "\xba\xc4", [0x2828] = "\xe6\xb6", [0x2829] = "\xbd\x44",
  [0x282d] = "\xea\x6c", [0x282f] = "\xea\x6b", [0x2830] = "\xea\x73",
  [0x2831] = "\xea\x6d", [0x2832] = "\xea\x72", [0x2833] = "\xea\x6f",
  [0x2834] = "\xbf\x60", [0x2835] = "\xea\x71", [0x2838] = "\xbf\x61",
  [0x283a] = "\xbf\x62", [0x283c] = "\xea\x70", [0x283d] = "\xea\x6e",
  [0x2842] = "\xc0\xf8", [0x2843] = "\xed\x74", [0x2846] = "\xc0\xf7",
  [0x2847] = "\xed\x77", [0x2848] = "\xed\x75", [0x2849] = "\xed\x76",
  [0x284c] = "\xc0\xf9", [0x2850] = "\xf0\x4d", [0x2852] = "\xc2\xa1",
  [0x2853] = "\xf0\x4e", [0x2856] = "\xc2\x7d", [0x2857] = "\xf0\x4f",
  [0x2858] = "\xc2\x7e", [0x2859] = "\xf0\x4c", [0x285a] = "\xf0\x50",
  [0x285c] = "\xf0\x4a", [0x285f] = "\xc3\xa7", [0x2860] = "\xf2\x78",
  [0x2861] = "\xc3\xa8", [0x2862] = "\xc4\x6f", [0x2864] = "\xf0\x4b",
  [0x2865] = "\xc4\x70", [0x2869] = "\xc4\xee", [0x286a] = "\xf5\xdf",
  [0x286c] = "\xc5\x7e", [0x286d] = "\xf6\xf4", [0x286e] = "\xc5\x7d",
  [0x2870] = "\xf7\xea", [0x2871] = "\xc5\xf5", [0x2872] = "\xc5\xf6",
  [0x2875] = "\xf9\xcc", [0x2878] = "\xac\xd1", [0x2879] = "\xcf\xde",
  [0x287b] = "\xb5\x6e", [0x287c] = "\xb5\x6f", [0x287d] = "\xa5\xd5",
  [0x287e] = "\xa6\xca", [0x287f] = "\xca\x47", [0x2881] = "\xcb\x71",
  [0x2882] = "\xa8\x6d", [0x2884] = "\xaa\xba", [0x2886] = "\xac\xd2",
  [0x2887] = "\xac\xd3", [0x2888] = "\xac\xd4", [0x2889] = "\xd6\xa6",
  [0x288a] = "\xd2\xcb", [0x288b] = "\xaf\x6f", [0x288e] = "\xb2\xae",
  [0x288f] = "\xd6\xa5", [0x2892] = "\xda\xb8", [0x2893] = "\xb5\x71",
  [0x2895] = "\xda\xb7", [0x2896] = "\xb5\x70", [0x2899] = "\xde\xd5",
  [0x289a] = "\xbd\x4a", [0x289b] = "\xe6\xbb", [0x289c] = "\xe6\xb8",
  [0x289d] = "\xe6\xb9", [0x289e] = "\xe6\xba", [0x28a4] = "\xed\x78",
  [0x28a6] = "\xf0\x51", [0x28aa] = "\xf4\x71", [0x28ab] = "\xf4\x70",
  [0x28ad] = "\xf6\xf5", [0x28ae] = "\xa5\xd6", [0x28af] = "\xcd\x75",
  [0x28b0] = "\xaf\x70", [0x28b4] = "\xb5\x72", [0x28b5] = "\xde\xd6",
  [0x28b8] = "\xe2\xe1", [0x28ba] = "\xbd\x4b", [0x28bb] = "\xea\x74",
  [0x28bd] = "\xf0\x52", [0x28be] = "\xf4\x72", [0x28bf] = "\xa5\xd7",
  [0x28c2] = "\xaa\xbb", [0x28c3] = "\xac\xd7", [0x28c4] = "\xcf\xdf",
  [0x28c5] = "\xac\xd8", [0x28c6] = "\xac\xd6", [0x28c8] = "\xac\xd5",
  [0x28c9] = "\xd2\xcc", [0x28ca] = "\xaf\x71", [0x28cd] = "\xaf\x72",
  [0x28ce] = "\xaf\x73", [0x28d2] = "\xb2\xb0", [0x28d3] = "\xd6\xa7",
  [0x28d4] = "\xb2\xaf", [0x28da] = "\xda\xb9", [0x28db] = "\xb2\xb1",
  [0x28dc] = "\xb5\x73", [0x28dd] = "\xde\xd7", [0x28de] = "\xb7\xf8",
  [0x28df] = "\xb7\xf9", [0x28e1] = "\xba\xc9", [0x28e3] = "\xba\xca",
  [0x28e4] = "\xbd\x4c", [0x28e5] = "\xbf\x64", [0x28e6] = "\xea\x75",
  [0x28e7] = "\xbf\x63", [0x28e9] = "\xed\x79", [0x28ea] = "\xc0\xfa",
  [0x28ec] = "\xf0\x53", [0x28ed] = "\xf4\x73", [0x28ee] = "\xa5\xd8",
  [0x28ef] = "\xa8\x6e", [0x28f0] = "\xcd\x78", [0x28f1] = "\xcd\x77",
  [0x28f2] = "\xaa\xbc", [0x28f3] = "\xcd\x76", [0x28f4] = "\xaa\xbd",
  [0x28f5] = "\xcd\x79", [0x28f7] = "\xcf\xe5", [0x28f8] = "\xac\xdb",
  [0x28f9] = "\xac\xda", [0x28fa] = "\xcf\xe7", [0x28fb] = "\xcf\xe6",
  [0x28fc] = "\xac\xdf", [0x28fe] = "\xac\xde", [0x2901] = "\xac\xd9",
  [0x2903] = "\xcf\xe1", [0x2904] = "\xcf\xe2", [0x2905] = "\xcf\xe3",
  [0x2907] = "\xac\xe0", [0x2908] = "\xcf\xe0", [0x2909] = "\xac\xdc",
  [0x290a] = "\xcf\xe4", [0x290b] = "\xac\xdd", [0x2910] = "\xd2\xcf",
  [0x2911] = "\xd2\xd3", [0x2912] = "\xd2\xd1", [0x2913] = "\xd2\xd0",
  [0x2915] = "\xd2\xd4", [0x2919] = "\xd2\xd5", [0x291a] = "\xd2\xd6",
  [0x291b] = "\xd2\xce", [0x291d] = "\xd2\xcd", [0x291f] = "\xaf\x75",
  [0x2920] = "\xaf\x76", [0x2922] = "\xd2\xd7", [0x2923] = "\xd2\xd2",
  [0x2925] = "\xd6\xb0", [0x2927] = "\xd2\xd8", [0x2928] = "\xaf\x77",
  [0x2929] = "\xaf\x74", [0x292d] = "\xd6\xaa", [0x292f] = "\xd6\xa9",
  [0x2931] = "\xd6\xab", [0x2932] = "\xd6\xac", [0x2933] = "\xd6\xae",
  [0x2934] = "\xd6\xad", [0x2935] = "\xd6\xb2", [0x2936] = "\xb2\xb5",
  [0x2937] = "\xb2\xb2", [0x2938] = "\xb2\xb6", [0x2939] = "\xd6\xa8",
  [0x293a] = "\xb2\xb7", [0x293b] = "\xd6\xb1", [0x293c] = "\xb2\xb4",
  [0x293d] = "\xd6\xaf", [0x293e] = "\xb2\xb3", [0x2944] = "\xda\xbc",
  [0x2945] = "\xda\xbe", [0x2946] = "\xda\xba", [0x2947] = "\xda\xbb",
  [0x294a] = "\xda\xbf", [0x294b] = "\xda\xc1", [0x294c] = "\xda\xc2",
  [0x294d] = "\xda\xbd", [0x294e] = "\xda\xc0", [0x294f] = "\xb5\x74",
  [0x2952] = "\xde\xdb", [0x2954] = "\xde\xe0", [0x2955] = "\xde\xd8",
  [0x2956] = "\xde\xdc", [0x2959] = "\xde\xe1", [0x295a] = "\xde\xdd",
  [0x295b] = "\xb7\xfa", [0x295c] = "\xb8\x43", [0x295e] = "\xb7\xfd",
  [0x295f] = "\xde\xd9", [0x2960] = "\xde\xda", [0x2961] = "\xba\xce",
  [0x2962] = "\xb8\x46", [0x2963] = "\xb7\xfe", [0x2965] = "\xb8\x44",
  [0x2966] = "\xb7\xfc", [0x2967] = "\xde\xdf", [0x2968] = "\xb8\x45",
  [0x2969] = "\xde\xde", [0x296a] = "\xb8\x41", [0x296b] = "\xb7\xfb",
  [0x296c] = "\xb8\x42", [0x296d] = "\xde\xe2", [0x296e] = "\xe2\xe6",
  [0x296f] = "\xe2\xe8", [0x2979] = "\xb8\x40", [0x297c] = "\xe2\xe3",
  [0x297d] = "\xba\xcc", [0x297e] = "\xe2\xe9", [0x297f] = "\xba\xcd",
  [0x2980] = "\xe2\xe7", [0x2981] = "\xe2\xe2", [0x2982] = "\xe2\xe5",
  [0x2983] = "\xe2\xea", [0x2984] = "\xba\xcb", [0x2985] = "\xe2\xe4",
  [0x2987] = "\xbd\x4e", [0x2988] = "\xe6\xbf", [0x2989] = "\xe6\xbe",
  [0x298b] = "\xbd\x51", [0x298c] = "\xbd\x4f", [0x298d] = "\xe6\xbc",
  [0x298e] = "\xbd\x4d", [0x298f] = "\xe6\xbd", [0x2991] = "\xbd\x50",
  [0x2995] = "\xea\x7d", [0x2997] = "\xea\xa1", [0x2999] = "\xea\x7e",
  [0x299a] = "\xea\x76", [0x299b] = "\xea\x7a", [0x299c] = "\xea\x79",
  [0x299d] = "\xea\x77", [0x299e] = "\xbf\x66", [0x299f] = "\xbf\x67",
  [0x29a0] = "\xbf\x65", [0x29a1] = "\xea\x78", [0x29a2] = "\xea\x7b",
  [0x29a3] = "\xea\x7c", [0x29a5] = "\xbf\x68", [0x29a7] = "\xc1\x40",
  [0x29a8] = "\xed\xa3", [0x29aa] = "\xc0\xfc", [0x29ab] = "\xed\x7b",
  [0x29ac] = "\xc0\xfe", [0x29ad] = "\xc1\x41", [0x29b0] = "\xc0\xfd",
  [0x29b1] = "\xed\xa2", [0x29b2] = "\xed\x7c", [0x29b3] = "\xc0\xfb",
  [0x29b4] = "\xed\xa1", [0x29b5] = "\xed\x7a", [0x29b6] = "\xed\x7e",
  [0x29b7] = "\xed\x7d", [0x29ba] = "\xf0\x55", [0x29bb] = "\xc2\xa4",
  [0x29bc] = "\xc2\xa5", [0x29bd] = "\xc2\xa2", [0x29bf] = "\xc2\xa3",
  [0x29c2] = "\xf0\x54", [0x29c4] = "\xf2\x7b", [0x29c7] = "\xc3\xa9",
  [0x29c9] = "\xf2\x79", [0x29ca] = "\xf2\x7a", [0x29cc] = "\xf4\x74",
  [0x29cd] = "\xf4\x77", [0x29ce] = "\xf4\x75", [0x29cf] = "\xf4\x76",
  [0x29d0] = "\xf5\xe0", [0x29d3] = "\xc4\xef", [0x29d4] = "\xf7\xeb",
  [0x29d5] = "\xf8\xb4", [0x29d7] = "\xc5\xf7", [0x29d8] = "\xf8\xf8",
  [0x29d9] = "\xf8\xf9", [0x29da] = "\xc6\x66", [0x29db] = "\xa5\xd9",
  [0x29dc] = "\xac\xe1", [0x29de] = "\xda\xc3", [0x29e0] = "\xde\xe3",
  [0x29e2] = "\xa5\xda", [0x29e3] = "\xa8\x6f", [0x29e5] = "\xaa\xbe",
  [0x29e7] = "\xcf\xe8", [0x29e8] = "\xcf\xe9", [0x29e9] = "\xaf\x78",
  [0x29ec] = "\xda\xc4", [0x29ed] = "\xb5\x75", [0x29ee] = "\xb8\x47",
  [0x29ef] = "\xc1\x42", [0x29f0] = "\xed\xa4", [0x29f1] = "\xf2\x7c",
  [0x29f2] = "\xf4\x78", [0x29f3] = "\xa5\xdb", [0x29f7] = "\xcd\xa1",
  [0x29f8] = "\xcd\x7a", [0x29f9] = "\xcd\x7c", [0x29fa] = "\xcd\x7e",
  [0x29fb] = "\xcd\x7d", [0x29fc] = "\xcd\x7b", [0x29fd] = "\xaa\xbf",
  [0x2a02] = "\xac\xe2", [0x2a03] = "\xcf\xf2", [0x2a05] = "\xcf\xed",
  [0x2a06] = "\xcf\xea", [0x2a09] = "\xcf\xf1", [0x2a0c] = "\xac\xe4",
  [0x2a0d] = "\xac\xe5", [0x2a0e] = "\xcf\xf0", [0x2a0f] = "\xcf\xef",
  [0x2a10] = "\xcf\xee", [0x2a11] = "\xcf\xeb", [0x2a12] = "\xcf\xec",
  [0x2a13] = "\xcf\xf3", [0x2a14] = "\xac\xe3", [0x2a1d] = "\xaf\x7c",
  [0x2a1f] = "\xaf\xa4", [0x2a20] = "\xaf\xa3", [0x2a21] = "\xd2\xe1",
  [0x2a22] = "\xd2\xdb", [0x2a23] = "\xd2\xd9", [0x2a25] = "\xaf\xa1",
  [0x2a26] = "\xd6\xb9", [0x2a27] = "\xaf\x7a", [0x2a28] = "\xd2\xde",
  [0x2a29] = "\xd2\xe2", [0x2a2a] = "\xd2\xe4", [0x2a2b] = "\xd2\xe0",
  [0x2a2c] = "\xd2\xda", [0x2a2d] = "\xaf\xa2", [0x2a2e] = "\xd2\xdf",
  [0x2a2f] = "\xd2\xdd", [0x2a30] = "\xaf\x79", [0x2a31] = "\xd2\xe5",
  [0x2a32] = "\xaf\xa5", [0x2a33] = "\xd2\xe3", [0x2a34] = "\xaf\x7d",
  [0x2a35] = "\xd2\xdc", [0x2a37] = "\xaf\x7e", [0x2a38] = "\xaf\x7b",
  [0x2a43] = "\xb2\xb9", [0x2a45] = "\xd6\xba", [0x2a48] = "\xd6\xb3",
  [0x2a49] = "\xd6\xb5", [0x2a4a] = "\xd6\xb7", [0x2a4c] = "\xd6\xb8",
  [0x2a4d] = "\xd6\xb6", [0x2a4e] = "\xb2\xba", [0x2a50] = "\xd6\xbb",
  [0x2a52] = "\xd6\xb4", [0x2a5c] = "\xda\xc8", [0x2a5d] = "\xb5\x76",
  [0x2a5e] = "\xda\xd0", [0x2a60] = "\xda\xc5", [0x2a62] = "\xda\xd1",
  [0x2a64] = "\xda\xc6", [0x2a65] = "\xda\xc7", [0x2a68] = "\xda\xcf",
  [0x2a69] = "\xda\xce", [0x2a6a] = "\xda\xcb", [0x2a6b] = "\xb2\xb8",
  [0x2a6c] = "\xb5\x77", [0x2a6d] = "\xda\xc9", [0x2a6e] = "\xda\xcc",
  [0x2a6f] = "\xb5\x78", [0x2a70] = "\xda\xcd", [0x2a71] = "\xda\xca",
  [0x2a79] = "\xde\xee", [0x2a7b] = "\xde\xf2", [0x2a7c] = "\xb8\x4e",
  [0x2a7e] = "\xe2\xf0", [0x2a7f] = "\xb8\x51", [0x2a80] = "\xde\xf0",
  [0x2a81] = "\xf9\xd6", [0x2a83] = "\xde\xed", [0x2a84] = "\xde\xe8",
  [0x2a85] = "\xde\xea", [0x2a86] = "\xde\xeb", [0x2a87] = "\xde\xe4",
  [0x2a89] = "\xb8\x4d", [0x2a8c] = "\xb8\x4c", [0x2a8e] = "\xb8\x48",
  [0x2a8f] = "\xde\xe7", [0x2a91] = "\xb8\x4f", [0x2a93] = "\xb8\x50",
  [0x2a94] = "\xde\xe6", [0x2a95] = "\xde\xe9", [0x2a96] = "\xde\xf1",
  [0x2a97] = "\xb8\x4a", [0x2a98] = "\xb8\x4b", [0x2a99] = "\xde\xef",
  [0x2a9a] = "\xde\xe5", [0x2a9e] = "\xe2\xf2", [0x2a9f] = "\xba\xd0",
  [0x2aa0] = "\xe2\xf4", [0x2aa1] = "\xde\xec", [0x2aa2] = "\xe2\xf6",
  [0x2aa3] = "\xba\xd4", [0x2aa4] = "\xe2\xf7", [0x2aa5] = "\xe2\xf3",
  [0x2aa7] = "\xba\xd1", [0x2aa8] = "\xe2\xef", [0x2aa9] = "\xba\xd3",
  [0x2aaa] = "\xe2\xec", [0x2aab] = "\xe2\xf1", [0x2aac] = "\xe2\xf5",
  [0x2aad] = "\xe2\xee", [0x2ab0] = "\xb8\x49", [0x2ab2] = "\xe2\xeb",
  [0x2ab3] = "\xba\xd2", [0x2ab4] = "\xe2\xed", [0x2aba] = "\xbd\x54",
  [0x2abb] = "\xe6\xc1", [0x2abc] = "\xbd\x58", [0x2abe] = "\xbd\x56",
  [0x2ac1] = "\xba\xcf", [0x2ac3] = "\xe6\xc8", [0x2ac4] = "\xe6\xc9",
  [0x2ac5] = "\xbd\x53", [0x2ac8] = "\xe6\xc7", [0x2ac9] = "\xe6\xca",
  [0x2aca] = "\xbd\x55", [0x2acb] = "\xbd\x52", [0x2acc] = "\xe6\xc3",
  [0x2acd] = "\xe6\xc0", [0x2ace] = "\xe6\xc5", [0x2acf] = "\xe6\xc2",
  [0x2ad0] = "\xbd\x59", [0x2ad1] = "\xe6\xc4", [0x2ad4] = "\xe6\xc6",
  [0x2ad5] = "\xbd\x57", [0x2ada] = "\xbf\x6a", [0x2adb] = "\xea\xa8",
  [0x2add] = "\xea\xa2", [0x2ade] = "\xea\xa6", [0x2adf] = "\xea\xac",
  [0x2ae0] = "\xea\xad", [0x2ae1] = "\xea\xa9", [0x2ae2] = "\xea\xaa",
  [0x2ae3] = "\xea\xa7", [0x2ae5] = "\xea\xa4", [0x2ae7] = "\xbf\x6c",
  [0x2ae8] = "\xbf\x69", [0x2ae9] = "\xea\xa3", [0x2aea] = "\xea\xa5",
  [0x2aec] = "\xbf\x6b", [0x2aed] = "\xea\xab", [0x2aef] = "\xc1\x46",
  [0x2af2] = "\xed\xaa", [0x2af3] = "\xed\xa5", [0x2af4] = "\xc1\x45",
  [0x2af7] = "\xc1\x43", [0x2af9] = "\xed\xac", [0x2afa] = "\xc1\x44",
  [0x2afb] = "\xed\xa8", [0x2afc] = "\xed\xa9", [0x2afd] = "\xed\xa6",
  [0x2afe] = "\xed\xad", [0x2aff] = "\xf0\x56", [0x2b01] = "\xc1\x47",
  [0x2b02] = "\xed\xa7", [0x2b04] = "\xed\xae", [0x2b05] = "\xed\xab",
  [0x2b09] = "\xf0\x5a", [0x2b0c] = "\xf0\x57", [0x2b0e] = "\xc2\xa6",
  [0x2b10] = "\xf0\x5b", [0x2b11] = "\xf0\x5d", [0x2b12] = "\xf0\x5c",
  [0x2b13] = "\xf0\x58", [0x2b14] = "\xf0\x59", [0x2b17] = "\xf2\xa3",
  [0x2b19] = "\xc3\xaa", [0x2b1b] = "\xf2\x7e", [0x2b1c] = "\xf2\xa2",
  [0x2b1d] = "\xf2\x7d", [0x2b1e] = "\xf2\xa4", [0x2b21] = "\xf2\xa1",
  [0x2b23] = "\xf4\x7a", [0x2b24] = "\xf4\x7d", [0x2b25] = "\xf4\x79",
  [0x2b26] = "\xc4\x71", [0x2b27] = "\xf4\x7b", [0x2b28] = "\xf4\x7c",
  [0x2b29] = "\xf4\x7e", [0x2b2a] = "\xc4\x72", [0x2b2b] = "\xc4\x74",
  [0x2b2c] = "\xc4\x73", [0x2b2d] = "\xf5\xe1", [0x2b2f] = "\xf5\xe3",
  [0x2b31] = "\xf5\xe2", [0x2b35] = "\xf6\xf6", [0x2b38] = "\xf8\xb5",
  [0x2b39] = "\xf8\xfa", [0x2b3a] = "\xa5\xdc", [0x2b3d] = "\xcb\x72",
  [0x2b3e] = "\xaa\xc0", [0x2b3f] = "\xcd\xa3", [0x2b40] = "\xaa\xc1",
  [0x2b41] = "\xaa\xc2", [0x2b42] = "\xcd\xa2", [0x2b44] = "\xcf\xf8",
  [0x2b45] = "\xcf\xf7", [0x2b46] = "\xac\xe6", [0x2b47] = "\xac\xe9",
  [0x2b48] = "\xac\xe8", [0x2b49] = "\xac\xe7", [0x2b4a] = "\xcf\xf4",
  [0x2b4b] = "\xcf\xf6", [0x2b4c] = "\xcf\xf5", [0x2b4f] = "\xd2\xe8",
  [0x2b50] = "\xaf\xa7", [0x2b51] = "\xd2\xec", [0x2b52] = "\xd2\xeb",
  [0x2b53] = "\xd2\xea", [0x2b54] = "\xd2\xe6", [0x2b55] = "\xaf\xa6",
  [0x2b56] = "\xaf\xaa", [0x2b57] = "\xaf\xad", [0x2b5a] = "\xaf\xae",
  [0x2b5b] = "\xd2\xe7", [0x2b5c] = "\xd2\xe9", [0x2b5d] = "\xaf\xac",
  [0x2b5e] = "\xaf\xab", [0x2b5f] = "\xaf\xa9", [0x2b60] = "\xaf\xa8",
  [0x2b61] = "\xd6\xc2", [0x2b63] = "\xd6\xc0", [0x2b64] = "\xd6\xbc",
  [0x2b65] = "\xb2\xbb", [0x2b67] = "\xd6\xbd", [0x2b68] = "\xb2\xbc",
  [0x2b69] = "\xd6\xbe", [0x2b6a] = "\xd6\xbf", [0x2b6b] = "\xd6\xc1",
  [0x2b6d] = "\xb2\xbd", [0x2b70] = "\xda\xd5", [0x2b72] = "\xda\xd4",
  [0x2b73] = "\xda\xd3", [0x2b74] = "\xda\xd2", [0x2b79] = "\xde\xf6",
  [0x2b7a] = "\xb8\x52", [0x2b7c] = "\xde\xf3", [0x2b7d] = "\xde\xf5",
  [0x2b7f] = "\xb8\x53", [0x2b81] = "\xb8\x54", [0x2b82] = "\xde\xf4",
  [0x2b88] = "\xe3\x41", [0x2b8a] = "\xe2\xf9", [0x2b8b] = "\xe2\xfa",
  [0x2b8d] = "\xba\xd7", [0x2b8e] = "\xba\xd5", [0x2b8f] = "\xba\xd6",
  [0x2b90] = "\xe3\x43", [0x2b92] = "\xe3\x42", [0x2b93] = "\xe2\xfe",
  [0x2b94] = "\xe2\xfd", [0x2b95] = "\xe2\xfc", [0x2b96] = "\xe2\xfb",
  [0x2b97] = "\xe3\x40", [0x2b98] = "\xe2\xf8", [0x2b9a] = "\xe6\xcb",
  [0x2b9b] = "\xe6\xd0", [0x2b9c] = "\xe6\xce", [0x2ba0] = "\xe6\xcd",
  [0x2ba1] = "\xe6\xcc", [0x2ba2] = "\xe6\xcf", [0x2ba4] = "\xea\xae",
  [0x2ba6] = "\xbf\x6d", [0x2ba7] = "\xc1\x48", [0x2ba8] = "\xed\xb0",
  [0x2baa] = "\xc1\x49", [0x2bab] = "\xed\xaf", [0x2bac] = "\xf0\x5f",
  [0x2bad] = "\xf0\x5e", [0x2bae] = "\xc2\xa7", [0x2bb0] = "\xf2\xa5",
  [0x2bb1] = "\xc3\xab", [0x2bb2] = "\xf4\xa1", [0x2bb3] = "\xc5\xa1",
  [0x2bb4] = "\xf6\xf7", [0x2bb6] = "\xf8\xb7", [0x2bb7] = "\xf8\xb6",
  [0x2bb8] = "\xc9\xa8", [0x2bb9] = "\xac\xea", [0x2bba] = "\xac\xeb",
  [0x2bbb] = "\xd6\xc3", [0x2bbd] = "\xb8\x56", [0x2bbe] = "\xa5\xdd",
  [0x2bbf] = "\xa8\x72", [0x2bc0] = "\xa8\x71", [0x2bc1] = "\xa8\x70",
  [0x2bc5] = "\xcd\xa4", [0x2bc8] = "\xaa\xc4", [0x2bc9] = "\xaa\xc3",
  [0x2bcb] = "\xac\xee", [0x2bcd] = "\xcf\xfa", [0x2bce] = "\xcf\xfd",
  [0x2bcf] = "\xcf\xfb", [0x2bd1] = "\xac\xec", [0x2bd2] = "\xac\xed",
  [0x2bd5] = "\xcf\xf9", [0x2bd6] = "\xcf\xfc", [0x2bd8] = "\xaf\xb5",
  [0x2bdc] = "\xd2\xf3", [0x2bdd] = "\xd2\xf5", [0x2bde] = "\xd2\xf4",
  [0x2bdf] = "\xaf\xb2", [0x2be0] = "\xd2\xef", [0x2be3] = "\xaf\xb0",
  [0x2be4] = "\xaf\xaf", [0x2be6] = "\xaf\xb3", [0x2be7] = "\xaf\xb1",
  [0x2be9] = "\xaf\xb4", [0x2bea] = "\xd2\xf2", [0x2beb] = "\xd2\xed",
  [0x2bec] = "\xd2\xee", [0x2bed] = "\xd2\xf1", [0x2bee] = "\xd2\xf0",
  [0x2bf6] = "\xd6\xc6", [0x2bf7] = "\xd6\xc7", [0x2bf8] = "\xd6\xc5",
  [0x2bfa] = "\xd6\xc4", [0x2bfb] = "\xb2\xbe", [0x2c00] = "\xb5\x7d",
  [0x2c02] = "\xda\xd6", [0x2c03] = "\xda\xd8", [0x2c04] = "\xda\xda",
  [0x2c05] = "\xb5\x7c", [0x2c08] = "\xb5\x7a", [0x2c0a] = "\xda\xd7",
  [0x2c0b] = "\xb5\x7b", [0x2c0c] = "\xda\xd9", [0x2c0d] = "\xb5\x79",
  [0x2c10] = "\xdf\x41", [0x2c11] = "\xde\xf7", [0x2c12] = "\xde\xfa",
  [0x2c13] = "\xde\xfe", [0x2c14] = "\xb8\x5a", [0x2c15] = "\xde\xfc",
  [0x2c17] = "\xde\xfb", [0x2c18] = "\xde\xf8", [0x2c19] = "\xde\xf9",
  [0x2c1a] = "\xb8\x58", [0x2c1b] = "\xdf\x40", [0x2c1c] = "\xb8\x57",
  [0x2c1e] = "\xb8\x5c", [0x2c1f] = "\xb8\x5b", [0x2c20] = "\xb8\x59",
  [0x2c22] = "\xde\xfd", [0x2c26] = "\xe3\x49", [0x2c28] = "\xe3\x48",
  [0x2c2b] = "\xe3\x44", [0x2c2e] = "\xba\xd8", [0x2c2f] = "\xe3\x47",
  [0x2c30] = "\xe3\x46", [0x2c31] = "\xba\xd9", [0x2c37] = "\xbd\x5e",
  [0x2c39] = "\xe6\xd2", [0x2c3b] = "\xbd\x5f", [0x2c3c] = "\xbd\x5b",
  [0x2c3d] = "\xbd\x5d", [0x2c3f] = "\xbd\x5a", [0x2c40] = "\xbd\x5c",
  [0x2c44] = "\xea\xaf", [0x2c46] = "\xbf\x70", [0x2c47] = "\xea\xb1",
  [0x2c48] = "\xea\xb0", [0x2c4a] = "\xe3\x45", [0x2c4b] = "\xbf\x72",
  [0x2c4c] = "\xbf\x71", [0x2c4d] = "\xbf\x6e", [0x2c4e] = "\xbf\x6f",
  [0x2c54] = "\xed\xb5", [0x2c56] = "\xed\xb3", [0x2c57] = "\xc1\x4a",
  [0x2c58] = "\xed\xb4", [0x2c5a] = "\xed\xb6", [0x2c5b] = "\xed\xb2",
  [0x2c5c] = "\xed\xb1", [0x2c5f] = "\xf0\x60", [0x2c60] = "\xc2\xaa",
  [0x2c61] = "\xc2\xa8", [0x2c62] = "\xc2\xa9", [0x2c67] = "\xf2\xa6",
  [0x2c68] = "\xf2\xa7", [0x2c69] = "\xc3\xad", [0x2c6b] = "\xc3\xac",
  [0x2c6c] = "\xf4\xa3", [0x2c6d] = "\xf4\xa4", [0x2c6e] = "\xf4\xa2",
  [0x2c70] = "\xf6\xf8", [0x2c71] = "\xf6\xf9", [0x2c74] = "\xa5\xde",
  [0x2c75] = "\xca\x48", [0x2c76] = "\xa8\x73", [0x2c78] = "\xcd\xa5",
  [0x2c79] = "\xaa\xc6", [0x2c7a] = "\xaa\xc5", [0x2c7b] = "\xcd\xa6",
  [0x2c7e] = "\xd0\x40", [0x2c7f] = "\xac\xef", [0x2c80] = "\xcf\xfe",
  [0x2c81] = "\xac\xf0", [0x2c84] = "\xaf\xb6", [0x2c85] = "\xd2\xf8",
  [0x2c86] = "\xd2\xf6", [0x2c87] = "\xd2\xfc", [0x2c88] = "\xaf\xb7",
  [0x2c89] = "\xd2\xf7", [0x2c8a] = "\xd2\xfb", [0x2c8b] = "\xd2\xf9",
  [0x2c8c] = "\xd2\xfa", [0x2c8f] = "\xd6\xc8", [0x2c90] = "\xd6\xca",
  [0x2c92] = "\xb2\xbf", [0x2c94] = "\xd6\xc9", [0x2c95] = "\xb2\xc0",
  [0x2c96] = "\xb5\xa2", [0x2c97] = "\xb5\xa1", [0x2c98] = "\xb5\x7e",
  [0x2c99] = "\xda\xdb", [0x2c9e] = "\xdf\x44", [0x2c9f] = "\xb8\x5d",
  [0x2ca0] = "\xb8\x5e", [0x2ca2] = "\xdf\x43", [0x2ca3] = "\xdf\x42",
  [0x2ca8] = "\xe3\x4a", [0x2ca9] = "\xba\xdb", [0x2caa] = "\xba\xda",
  [0x2cab] = "\xe3\x4b", [0x2cac] = "\xe3\x4c", [0x2cae] = "\xbd\x61",
  [0x2caf] = "\xbd\x60", [0x2cb1] = "\xea\xb5", [0x2cb2] = "\xe6\xd3",
  [0x2cb3] = "\xe6\xd5", [0x2cb4] = "\xe6\xd4", [0x2cb5] = "\xea\xb4",
  [0x2cb6] = "\xea\xb2", [0x2cb7] = "\xea\xb6", [0x2cb8] = "\xea\xb3",
  [0x2cba] = "\xbf\x73", [0x2cbe] = "\xed\xb7", [0x2cbf] = "\xc1\x4b",
  [0x2cc0] = "\xed\xb8", [0x2cc1] = "\xed\xb9", [0x2cc4] = "\xc2\xab",
  [0x2cc5] = "\xc2\xac", [0x2cc7] = "\xc4\x75", [0x2cca] = "\xc5\xd1",
  [0x2ccb] = "\xa5\xdf", [0x2cd1] = "\xd0\x41", [0x2cd8] = "\xd2\xfd",
  [0x2cd9] = "\xaf\xb8", [0x2cdf] = "\xb3\xba", [0x2ce0] = "\xb3\xb9",
  [0x2ce3] = "\xb5\xa4", [0x2ce4] = "\xda\xdd", [0x2ce5] = "\xb5\xa3",
  [0x2ce6] = "\xda\xdc", [0x2ceb] = "\xdf\x45", [0x2ced] = "\xba\xdc",
  [0x2cee] = "\xe3\x4d", [0x2cef] = "\xba\xdd", [0x2cf6] = "\xc4\x76",
  [0x2cf7] = "\xf4\xa5", [0x2cf9] = "\xa6\xcb", [0x2cfa] = "\xaa\xc7",
  [0x2cfb] = "\xcd\xa7", [0x2cfd] = "\xac\xf2", [0x2cff] = "\xac\xf1",
  [0x2d00] = "\xd0\x42", [0x2d01] = "\xd0\x43", [0x2d04] = "\xd3\x40",
  [0x2d05] = "\xd3\x42", [0x2d06] = "\xaf\xb9", [0x2d08] = "\xd3\x44",
  [0x2d09] = "\xd3\x47", [0x2d0a] = "\xd3\x45", [0x2d0e] = "\xd3\x46",
  [0x2d0f] = "\xd3\x43", [0x2d10] = "\xd2\xfe", [0x2d11] = "\xaf\xba",
  [0x2d12] = "\xd3\x48", [0x2d13] = "\xd3\x41", [0x2d18] = "\xd6\xd3",
  [0x2d19] = "\xb2\xc6", [0x2d1a] = "\xd6\xdc", [0x2d1b] = "\xb2\xc3",
  [0x2d1d] = "\xd6\xd5", [0x2d1e] = "\xb2\xc7", [0x2d20] = "\xb2\xc1",
  [0x2d22] = "\xd6\xd0", [0x2d23] = "\xd6\xdd", [0x2d24] = "\xd6\xd1",
  [0x2d25] = "\xd6\xce", [0x2d26] = "\xb2\xc5", [0x2d28] = "\xb2\xc2",
  [0x2d2a] = "\xd6\xd4", [0x2d2b] = "\xd6\xd7", [0x2d2c] = "\xb2\xc4",
  [0x2d2d] = "\xd6\xd8", [0x2d2e] = "\xb2\xc8", [0x2d2f] = "\xd6\xd9",
  [0x2d30] = "\xd6\xcf", [0x2d31] = "\xd6\xd6", [0x2d32] = "\xd6\xda",
  [0x2d33] = "\xd6\xd2", [0x2d34] = "\xd6\xcd", [0x2d35] = "\xd6\xcb",
  [0x2d38] = "\xd6\xdb", [0x2d3b] = "\xda\xdf", [0x2d40] = "\xda\xe4",
  [0x2d44] = "\xda\xe0", [0x2d45] = "\xda\xe6", [0x2d46] = "\xb5\xa7",
  [0x2d47] = "\xd6\xcc", [0x2d48] = "\xda\xe1", [0x2d49] = "\xb5\xa5",
  [0x2d4a] = "\xda\xde", [0x2d4b] = "\xb5\xac", [0x2d4c] = "\xda\xe2",
  [0x2d4d] = "\xb5\xab", [0x2d4e] = "\xda\xe3", [0x2d4f] = "\xb5\xad",
  [0x2d50] = "\xb5\xa8", [0x2d51] = "\xb5\xae", [0x2d52] = "\xb5\xa9",
  [0x2d54] = "\xb5\xaa", [0x2d56] = "\xb5\xa6", [0x2d58] = "\xda\xe5",
  [0x2d60] = "\xb8\x61", [0x2d61] = "\xdf\x50", [0x2d63] = "\xdf\x53",
  [0x2d64] = "\xdf\x47", [0x2d65] = "\xdf\x4c", [0x2d66] = "\xdf\x46",
  [0x2d67] = "\xb8\x63", [0x2d69] = "\xdf\x4a", [0x2d6d] = "\xdf\x48",
  [0x2d6e] = "\xb8\x62", [0x2d70] = "\xdf\x4f", [0x2d71] = "\xdf\x4e",
  [0x2d72] = "\xdf\x4b", [0x2d73] = "\xdf\x4d", [0x2d74] = "\xdf\x49",
  [0x2d75] = "\xba\xe1", [0x2d76] = "\xdf\x52", [0x2d77] = "\xb8\x5f",
  [0x2d78] = "\xdf\x51", [0x2d82] = "\xe3\x5d", [0x2d84] = "\xba\xe8",
  [0x2d85] = "\xe3\x58", [0x2d87] = "\xba\xe7", [0x2d88] = "\xe3\x4e",
  [0x2d8a] = "\xe3\x50", [0x2d8b] = "\xba\xe0", [0x2d8c] = "\xe3\x55",
  [0x2d8d] = "\xe3\x54", [0x2d8e] = "\xe3\x57", [0x2d8f] = "\xba\xe5",
  [0x2d90] = "\xe3\x52", [0x2d91] = "\xe3\x51", [0x2d94] = "\xba\xe4",
  [0x2d95] = "\xba\xdf", [0x2d96] = "\xe3\x53", [0x2d97] = "\xba\xe2",
  [0x2d98] = "\xe3\x59", [0x2d99] = "\xe3\x5b", [0x2d9b] = "\xe3\x56",
  [0x2d9c] = "\xe3\x4f", [0x2d9d] = "\xba\xe3", [0x2da0] = "\xbd\x69",
  [0x2da1] = "\xba\xde", [0x2da4] = "\xe3\x5c", [0x2dac] = "\xe6\xd9",
  [0x2dad] = "\xbd\x62", [0x2daf] = "\xe6\xdb", [0x2db1] = "\xbd\x63",
  [0x2db4] = "\xbd\x65", [0x2db5] = "\xe6\xde", [0x2db7] = "\xe6\xd6",
  [0x2db8] = "\xba\xe6", [0x2db9] = "\xe6\xdc", [0x2dbe] = "\xe6\xd8",
  [0x2dc0] = "\xb8\x60", [0x2dc1] = "\xbd\x68", [0x2dc4] = "\xbd\x64",
  [0x2dc6] = "\xbd\x66", [0x2dc7] = "\xbd\x67", [0x2dc9] = "\xbf\x76",
  [0x2dca] = "\xe6\xdd", [0x2dcb] = "\xe6\xd7", [0x2dcc] = "\xbd\x6a",
  [0x2dce] = "\xe6\xda", [0x2dd4] = "\xea\xc0", [0x2dd5] = "\xea\xbb",
  [0x2dd8] = "\xea\xc5", [0x2dd9] = "\xbf\x74", [0x2dda] = "\xea\xbd",
  [0x2ddb] = "\xbf\x78", [0x2ddc] = "\xea\xc3", [0x2ddd] = "\xea\xba",
  [0x2dde] = "\xea\xb7", [0x2ddf] = "\xea\xc6", [0x2de0] = "\xc1\x51",
  [0x2de1] = "\xbf\x79", [0x2de2] = "\xea\xc2", [0x2de3] = "\xea\xb8",
  [0x2de4] = "\xbf\x77", [0x2de5] = "\xea\xbc", [0x2de6] = "\xbf\x7b",
  [0x2de7] = "\xea\xb9", [0x2de8] = "\xea\xbe", [0x2de9] = "\xbf\x7a",
  [0x2dea] = "\xea\xc1", [0x2deb] = "\xea\xc4", [0x2df0] = "\xed\xcb",
  [0x2df1] = "\xed\xcc", [0x2df2] = "\xed\xbc", [0x2df3] = "\xed\xc3",
  [0x2df4] = "\xed\xc1", [0x2df7] = "\xc1\x4f", [0x2df8] = "\xed\xc8",
  [0x2df9] = "\xea\xbf", [0x2dfb] = "\xed\xbf", [0x2dfd] = "\xed\xc9",
  [0x2dfe] = "\xc1\x4e", [0x2dff] = "\xed\xbe", [0x2e00] = "\xed\xbd",
  [0x2e01] = "\xed\xc7", [0x2e02] = "\xed\xc4", [0x2e03] = "\xed\xc6",
  [0x2e05] = "\xed\xba", [0x2e06] = "\xed\xca", [0x2e07] = "\xc1\x4c",
  [0x2e09] = "\xed\xc5", [0x2e0a] = "\xed\xce", [0x2e0b] = "\xed\xc2",
  [0x2e0c] = "\xc1\x50", [0x2e0d] = "\xc1\x4d", [0x2e0e] = "\xed\xc0",
  [0x2e0f] = "\xed\xbb", [0x2e10] = "\xed\xcd", [0x2e11] = "\xbf\x75",
  [0x2e19] = "\xf0\x63", [0x2e1c] = "\xf0\x61", [0x2e1d] = "\xf0\x67",
  [0x2e1e] = "\xc2\xb0", [0x2e1f] = "\xf0\x65", [0x2e20] = "\xf0\x64",
  [0x2e21] = "\xc2\xb2", [0x2e22] = "\xf0\x6a", [0x2e23] = "\xc2\xb1",
  [0x2e25] = "\xf0\x6b", [0x2e26] = "\xf0\x68", [0x2e27] = "\xc2\xae",
  [0x2e28] = "\xf0\x69", [0x2e29] = "\xf0\x62", [0x2e2a] = "\xc2\xaf",
  [0x2e2b] = "\xc2\xad", [0x2e2c] = "\xf2\xab", [0x2e2d] = "\xf0\x66",
  [0x2e30] = "\xf0\x6c", [0x2e33] = "\xf2\xa8", [0x2e37] = "\xc3\xb2",
  [0x2e38] = "\xc3\xb0", [0x2e39] = "\xf2\xaa", [0x2e3b] = "\xf2\xac",
  [0x2e3c] = "\xf2\xa9", [0x2e3d] = "\xc3\xb1", [0x2e3e] = "\xc3\xae",
  [0x2e3f] = "\xc3\xaf", [0x2e40] = "\xc3\xb3", [0x2e43] = "\xc4\x78",
  [0x2e45] = "\xf4\xaa", [0x2e47] = "\xf4\xa9", [0x2e48] = "\xf4\xa7",
  [0x2e49] = "\xf4\xa6", [0x2e4a] = "\xf4\xa8", [0x2e4c] = "\xc4\x77",
  [0x2e4d] = "\xc4\x79", [0x2e50] = "\xc4\xf0", [0x2e53] = "\xf5\xe5",
  [0x2e54] = "\xf5\xe4", [0x2e57] = "\xf6\xfa", [0x2e59] = "\xf6\xfc",
  [0x2e5a] = "\xf6\xfe", [0x2e5b] = "\xf6\xfd", [0x2e5c] = "\xf6\xfb",
  [0x2e5f] = "\xc5\xa3", [0x2e60] = "\xc5\xa2", [0x2e63] = "\xc5\xd3",
  [0x2e64] = "\xc5\xd2", [0x2e65] = "\xc5\xd4", [0x2e66] = "\xf7\xed",
  [0x2e67] = "\xf7\xec", [0x2e69] = "\xf8\xfb", [0x2e6a] = "\xf8\xb8",
  [0x2e6b] = "\xf8\xfc", [0x2e6c] = "\xc6\x58", [0x2e6e] = "\xc6\x59",
  [0x2e6f] = "\xf9\x6d", [0x2e72] = "\xc6\x7e", [0x2e73] = "\xa6\xcc",
  [0x2e75] = "\xcd\xa8", [0x2e78] = "\xd0\x45", [0x2e79] = "\xd0\x46",
  [0x2e7a] = "\xd0\x44", [0x2e7d] = "\xac\xf3", [0x2e7f] = "\xd0\x47",
  [0x2e80] = "\xd0\x48", [0x2e81] = "\xd0\x49", [0x2e84] = "\xd3\x49",
  [0x2e85] = "\xd3\x4f", [0x2e88] = "\xd3\x4d", [0x2e89] = "\xaf\xbb",
  [0x2e8a] = "\xd3\x4b", [0x2e8c] = "\xd3\x4c", [0x2e8d] = "\xd3\x4e",
  [0x2e91] = "\xd3\x4a", [0x2e92] = "\xb2\xc9", [0x2e94] = "\xd6\xde",
  [0x2e95] = "\xb2\xcb", [0x2e96] = "\xd6\xe0", [0x2e97] = "\xb2\xca",
  [0x2e98] = "\xd6\xdf", [0x2e9e] = "\xda\xe8", [0x2e9f] = "\xb5\xaf",
  [0x2ea1] = "\xda\xea", [0x2ea2] = "\xda\xe7", [0x2ea3] = "\xd6\xe1",
  [0x2ea5] = "\xb5\xb0", [0x2ea7] = "\xf9\xdb", [0x2ea8] = "\xda\xe9",
  [0x2eaf] = "\xdf\x56", [0x2eb1] = "\xb8\x64", [0x2eb2] = "\xdf\x54",
  [0x2eb3] = "\xb8\x65", [0x2eb4] = "\xdf\x55", [0x2eb5] = "\xb8\x66",
  [0x2eb9] = "\xba\xe9", [0x2eba] = "\xe3\x61", [0x2ebb] = "\xe3\x5e",
  [0x2ebc] = "\xe3\x60", [0x2ebd] = "\xba\xea", [0x2ebe] = "\xba\xeb",
  [0x2ebf] = "\xe3\x5f", [0x2ec5] = "\xe6\xdf", [0x2ec8] = "\xe6\xe0",
  [0x2eca] = "\xbd\x6b", [0x2ecb] = "\xe6\xe2", [0x2ecc] = "\xe6\xe1",
  [0x2ece] = "\xa2\x61", [0x2ed0] = "\xea\xca", [0x2ed1] = "\xea\xcb",
  [0x2ed2] = "\xea\xc7", [0x2ed4] = "\xea\xc8", [0x2ed5] = "\xbf\x7c",
  [0x2ed6] = "\xbf\x7d", [0x2ed7] = "\xea\xc9", [0x2ed9] = "\xc1\x57",
  [0x2edc] = "\xc1\x53", [0x2edd] = "\xc1\x58", [0x2ede] = "\xc1\x54",
  [0x2edf] = "\xc1\x56", [0x2ee0] = "\xc1\x52", [0x2ee2] = "\xc1\x55",
  [0x2ee7] = "\xc2\xb3", [0x2ee8] = "\xed\xcf", [0x2eea] = "\xf2\xae",
  [0x2eec] = "\xf2\xad", [0x2eee] = "\xf4\xab", [0x2eef] = "\xc4\x7a",
  [0x2ef0] = "\xc4\x7b", [0x2ef1] = "\xf7\x41", [0x2ef2] = "\xf5\xe6",
  [0x2ef4] = "\xf7\x40", [0x2ef6] = "\xf8\xfd", [0x2ef7] = "\xf9\xa4",
  [0x2ef8] = "\xa6\xcd", [0x2efb] = "\xa8\x74", [0x2efd] = "\xcd\xa9",
  [0x2efe] = "\xaa\xc8", [0x2f00] = "\xac\xf6", [0x2f01] = "\xd0\x4c",
  [0x2f02] = "\xac\xf4", [0x2f03] = "\xd0\x4a", [0x2f04] = "\xac\xf9",
  [0x2f05] = "\xac\xf5", [0x2f06] = "\xac\xfa", [0x2f07] = "\xac\xf8",
  [0x2f08] = "\xd0\x4b", [0x2f09] = "\xac\xf7", [0x2f0a] = "\xaf\xbf",
  [0x2f0b] = "\xaf\xbe", [0x2f0c] = "\xd3\x5a", [0x2f0d] = "\xaf\xc7",
  [0x2f0e] = "\xd3\x53", [0x2f0f] = "\xd3\x59", [0x2f10] = "\xaf\xc3",
  [0x2f11] = "\xd3\x52", [0x2f12] = "\xd3\x58", [0x2f13] = "\xd3\x56",
  [0x2f14] = "\xaf\xc2", [0x2f15] = "\xaf\xc4", [0x2f16] = "\xd3\x55",
  [0x2f17] = "\xaf\xbd", [0x2f18] = "\xd3\x54", [0x2f19] = "\xaf\xc8",
  [0x2f1a] = "\xaf\xc5", [0x2f1b] = "\xaf\xc9", [0x2f1c] = "\xaf\xc6",
  [0x2f1d] = "\xd3\x51", [0x2f1e] = "\xd3\x50", [0x2f1f] = "\xd3\x57",
  [0x2f20] = "\xaf\xc0", [0x2f21] = "\xaf\xbc", [0x2f22] = "\xaf\xc1",
  [0x2f28] = "\xd6\xf0", [0x2f29] = "\xd6\xe9", [0x2f2b] = "\xb5\xb5",
  [0x2f2c] = "\xd6\xe8", [0x2f2e] = "\xb2\xcf", [0x2f2f] = "\xb2\xd6",
  [0x2f30] = "\xb2\xd3", [0x2f31] = "\xb2\xd9", [0x2f32] = "\xb2\xd8",
  [0x2f33] = "\xb2\xd4", [0x2f35] = "\xd6\xe2", [0x2f36] = "\xd6\xe5",
  [0x2f38] = "\xd6\xe4", [0x2f39] = "\xb2\xd0", [0x2f3a] = "\xd6\xe6",
  [0x2f3b] = "\xd6\xef", [0x2f3c] = "\xb2\xd1", [0x2f3d] = "\xd6\xe3",
  [0x2f3e] = "\xd6\xec", [0x2f3f] = "\xd6\xed", [0x2f40] = "\xb2\xd2",
  [0x2f41] = "\xd6\xea", [0x2f42] = "\xb2\xd7", [0x2f43] = "\xb2\xcd",
  [0x2f44] = "\xb2\xd5", [0x2f45] = "\xd6\xe7", [0x2f46] = "\xb2\xcc",
  [0x2f47] = "\xd6\xeb", [0x2f4a] = "\xd6\xee", [0x2f4e] = "\xda\xfb",
  [0x2f4f] = "\xda\xf2", [0x2f50] = "\xb5\xb2", [0x2f51] = "\xda\xf9",
  [0x2f52] = "\xda\xf6", [0x2f53] = "\xda\xee", [0x2f54] = "\xda\xf7",
  [0x2f55] = "\xb5\xb4", [0x2f56] = "\xda\xef", [0x2f58] = "\xda\xeb",
  [0x2f5b] = "\xb8\x6c", [0x2f5c] = "\xda\xf4", [0x2f5e] = "\xb5\xb1",
  [0x2f5f] = "\xda\xfa", [0x2f61] = "\xb5\xb8", [0x2f62] = "\xb5\xba",
  [0x2f63] = "\xda\xed", [0x2f66] = "\xb5\xb9", [0x2f67] = "\xda\xf0",
  [0x2f68] = "\xb5\xb3", [0x2f69] = "\xda\xf8", [0x2f6a] = "\xda\xf1",
  [0x2f6b] = "\xda\xf5", [0x2f6d] = "\xda\xf3", [0x2f6e] = "\xb5\xb6",
  [0x2f6f] = "\xda\xec", [0x2f70] = "\xb5\xbb", [0x2f71] = "\xb2\xce",
  [0x2f72] = "\xb5\xb7", [0x2f73] = "\xb5\xbc", [0x2f79] = "\xb8\x68",
  [0x2f7a] = "\xdf\x5d", [0x2f7b] = "\xdf\x5f", [0x2f7c] = "\xdf\x61",
  [0x2f7d] = "\xdf\x65", [0x2f7f] = "\xdf\x5b", [0x2f80] = "\xdf\x59",
  [0x2f81] = "\xb8\x6a", [0x2f83] = "\xdf\x60", [0x2f84] = "\xdf\x64",
  [0x2f85] = "\xdf\x5c", [0x2f86] = "\xdf\x58", [0x2f88] = "\xdf\x57",
  [0x2f8c] = "\xdf\x62", [0x2f8d] = "\xdf\x5a", [0x2f8e] = "\xdf\x5e",
  [0x2f8f] = "\xb8\x6b", [0x2f91] = "\xb8\x69", [0x2f92] = "\xdf\x66",
  [0x2f93] = "\xb8\x67", [0x2f94] = "\xdf\x63", [0x2f96] = "\xe3\x72",
  [0x2f9c] = "\xba\xee", [0x2f9d] = "\xe3\x6a", [0x2f9e] = "\xbd\x78",
  [0x2f9f] = "\xe3\x74", [0x2fa0] = "\xba\xf1", [0x2fa1] = "\xe3\x78",
  [0x2fa2] = "\xba\xf7", [0x2fa3] = "\xe3\x65", [0x2fa6] = "\xe3\x75",
  [0x2fa7] = "\xe3\x62", [0x2fa9] = "\xe3\x77", [0x2faa] = "\xe3\x66",
  [0x2fac] = "\xba\xfe", [0x2fad] = "\xba\xfb", [0x2fae] = "\xe3\x76",
  [0x2faf] = "\xe3\x70", [0x2fb0] = "\xba\xed", [0x2fb1] = "\xba\xf5",
  [0x2fb2] = "\xba\xf4", [0x2fb4] = "\xba\xf3", [0x2fb5] = "\xba\xf9",
  [0x2fb7] = "\xe3\x63", [0x2fb8] = "\xba\xfa", [0x2fb9] = "\xe3\x71",
  [0x2fba] = "\xba\xf6", [0x2fbb] = "\xba\xec", [0x2fbc] = "\xe3\x73",
  [0x2fbd] = "\xba\xef", [0x2fbe] = "\xba\xf0", [0x2fbf] = "\xba\xf8",
  [0x2fc0] = "\xe3\x68", [0x2fc1] = "\xe3\x67", [0x2fc2] = "\xe3\x64",
  [0x2fc4] = "\xe3\x6c", [0x2fc5] = "\xe3\x69", [0x2fc6] = "\xe3\x6d",
  [0x2fc7] = "\xba\xfd", [0x2fc9] = "\xe3\x79", [0x2fca] = "\xba\xf2",
  [0x2fcb] = "\xe3\x6e", [0x2fcc] = "\xe3\x6f", [0x2fce] = "\xe3\x6b",
  [0x2fd2] = "\xba\xfc", [0x2fd7] = "\xe6\xe7", [0x2fd8] = "\xbd\x70",
  [0x2fd9] = "\xbd\x79", [0x2fda] = "\xbd\x75", [0x2fdb] = "\xe6\xe4",
  [0x2fdd] = "\xbd\x72", [0x2fde] = "\xbd\x76", [0x2fdf] = "\xe6\xf0",
  [0x2fe0] = "\xbd\x6c", [0x2fe1] = "\xe6\xe8", [0x2fe3] = "\xbd\x74",
  [0x2fe6] = "\xe6\xeb", [0x2fe7] = "\xe6\xe6", [0x2fe8] = "\xbd\x73",
  [0x2fe9] = "\xbd\x77", [0x2fea] = "\xe6\xe5", [0x2fec] = "\xbd\x71",
  [0x2fee] = "\xe6\xef", [0x2fef] = "\xbd\x6e", [0x2ff0] = "\xe6\xee",
  [0x2ff1] = "\xe6\xed", [0x2ff2] = "\xbd\x7a", [0x2ff3] = "\xe5\x72",
  [0x2ff4] = "\xbd\x6d", [0x2ff6] = "\xe6\xec", [0x2ff7] = "\xe6\xe3",
  [0x2ff9] = "\xbd\x7b", [0x2ffa] = "\xe6\xea", [0x2ffb] = "\xbd\x6f",
  [0x3003] = "\xe6\xe9", [0x3008] = "\xbf\xa2", [0x3009] = "\xbf\xa7",
  [0x300a] = "\xbf\x7e", [0x300b] = "\xea\xd8", [0x300c] = "\xea\xcf",
  [0x300d] = "\xea\xdb", [0x300e] = "\xea\xd3", [0x300f] = "\xea\xd9",
  [0x3010] = "\xbf\xa8", [0x3011] = "\xbf\xa1", [0x3012] = "\xea\xcc",
  [0x3013] = "\xea\xd2", [0x3014] = "\xea\xdc", [0x3015] = "\xea\xd5",
  [0x3016] = "\xea\xda", [0x3017] = "\xea\xce", [0x301a] = "\xea\xd6",
  [0x301b] = "\xbf\xa3", [0x301c] = "\xea\xd4", [0x301d] = "\xbf\xa6",
  [0x301e] = "\xbf\xa5", [0x301f] = "\xea\xd0", [0x3020] = "\xea\xd1",
  [0x3021] = "\xea\xcd", [0x3022] = "\xea\xd7", [0x3023] = "\xbf\xa4",
  [0x3024] = "\xea\xde", [0x3025] = "\xea\xdd", [0x3029] = "\xed\xda",
  [0x302a] = "\xed\xd6", [0x302b] = "\xc1\x5f", [0x302d] = "\xed\xd0",
  [0x302e] = "\xc1\x59", [0x302f] = "\xc1\x69", [0x3030] = "\xed\xdc",
  [0x3031] = "\xc1\x61", [0x3032] = "\xc1\x5d", [0x3033] = "\xed\xd3",
  [0x3034] = "\xc1\x64", [0x3035] = "\xc1\x67", [0x3036] = "\xed\xde",
  [0x3037] = "\xc1\x5c", [0x3038] = "\xed\xd5", [0x3039] = "\xc1\x65",
  [0x303a] = "\xed\xe0", [0x303b] = "\xed\xdd", [0x303c] = "\xed\xd1",
  [0x303d] = "\xc1\x60", [0x303e] = "\xc1\x5a", [0x303f] = "\xc1\x68",
  [0x3040] = "\xed\xd8", [0x3041] = "\xc1\x63", [0x3042] = "\xed\xd2",
  [0x3043] = "\xc1\x5e", [0x3044] = "\xed\xdf", [0x3045] = "\xc1\x62",
  [0x3046] = "\xc1\x5b", [0x3047] = "\xed\xd9", [0x3048] = "\xc1\x66",
  [0x3049] = "\xed\xd7", [0x304c] = "\xed\xdb", [0x3050] = "\xf0\x6e",
  [0x3051] = "\xf0\x74", [0x3052] = "\xc2\xb9", [0x3053] = "\xf0\x77",
  [0x3054] = "\xc2\xb4", [0x3055] = "\xc2\xb5", [0x3056] = "\xf0\x6f",
  [0x3057] = "\xf0\x76", [0x3058] = "\xf0\x71", [0x3059] = "\xc2\xba",
  [0x305a] = "\xc2\xb7", [0x305c] = "\xf0\x6d", [0x305e] = "\xc2\xb6",
  [0x305f] = "\xf0\x73", [0x3060] = "\xf0\x75", [0x3061] = "\xc2\xb8",
  [0x3062] = "\xf0\x72", [0x3063] = "\xf0\x70", [0x3068] = "\xf2\xb8",
  [0x3069] = "\xc3\xb7", [0x306a] = "\xc3\xb8", [0x306b] = "\xc3\xb4",
  [0x306d] = "\xc3\xb5", [0x306f] = "\xf2\xb4", [0x3070] = "\xf2\xb2",
  [0x3072] = "\xf2\xb6", [0x3073] = "\xc3\xba", [0x3074] = "\xf2\xb7",
  [0x3075] = "\xf2\xb0", [0x3076] = "\xf2\xaf", [0x3077] = "\xf2\xb3",
  [0x3078] = "\xf2\xb1", [0x3079] = "\xc3\xb6", [0x307a] = "\xf2\xb5",
  [0x307b] = "\xf4\xac", [0x307c] = "\xc4\x7e", [0x307d] = "\xc4\x7d",
  [0x307e] = "\xf4\xad", [0x3080] = "\xf4\xaf", [0x3081] = "\xf4\xae",
  [0x3082] = "\xc4\xa1", [0x3086] = "\xf5\xeb", [0x3087] = "\xf5\xe8",
  [0x3088] = "\xf5\xe9", [0x308a] = "\xf5\xe7", [0x308b] = "\xf5\xea",
  [0x308c] = "\xc4\xf2", [0x308d] = "\xf5\xec", [0x308f] = "\xc4\xf1",
  [0x3091] = "\xf7\x42", [0x3093] = "\xc5\xd5", [0x3094] = "\xc5\xd7",
  [0x3095] = "\xf7\xee", [0x3096] = "\xc5\xd6", [0x3097] = "\xf8\xb9",
  [0x3098] = "\xf9\x40", [0x3099] = "\xf9\x42", [0x309a] = "\xf8\xfe",
  [0x309b] = "\xf9\x41", [0x309c] = "\xc6\x6c", [0x3136] = "\xa6\xce",
  [0x3138] = "\xac\xfb", [0x3139] = "\xd2\x6f", [0x313a] = "\xaf\xca",
  [0x313d] = "\xb2\xda", [0x313e] = "\xda\xfc", [0x313f] = "\xda\xfd",
  [0x3143] = "\xea\xdf", [0x3144] = "\xc1\x6a", [0x3145] = "\xed\xe1",
  [0x3148] = "\xc2\xbb", [0x314a] = "\xf2\xba", [0x314b] = "\xf2\xb9",
  [0x314c] = "\xc4\xa2", [0x314d] = "\xf5\xed", [0x314f] = "\xf7\x43",
  [0x3150] = "\xc5\xf8", [0x3151] = "\xca\x49", [0x3154] = "\xaa\xc9",
  [0x3155] = "\xa8\x75", [0x3158] = "\xd0\x4d", [0x315b] = "\xd3\x60",
  [0x315c] = "\xd3\x5b", [0x315d] = "\xd3\x5f", [0x315e] = "\xd3\x5d",
  [0x315f] = "\xaf\xcb", [0x3160] = "\xd3\x5e", [0x3161] = "\xd3\x5c",
  [0x3163] = "\xd6\xf1", [0x3165] = "\xda\xfe", [0x3166] = "\xdb\x40",
  [0x3167] = "\xdf\x69", [0x3168] = "\xdf\x6a", [0x3169] = "\xb8\x6e",
  [0x316a] = "\xb8\x6f", [0x316b] = "\xdf\x68", [0x316c] = "\xdf\x6b",
  [0x316d] = "\xdf\x67", [0x316e] = "\xb8\x6d", [0x3170] = "\xbb\x40",
  [0x3172] = "\xb8\x70", [0x3173] = "\xe3\x7a", [0x3175] = "\xbd\x7c",
  [0x3176] = "\xe6\xf1", [0x3177] = "\xbd\x7d", [0x3179] = "\xbf\xa9",
  [0x317a] = "\xea\xe2", [0x317b] = "\xea\xe0", [0x317c] = "\xea\xe1",
  [0x317d] = "\xed\xe4", [0x317e] = "\xed\xe3", [0x317f] = "\xed\xe2",
  [0x3183] = "\xf2\xbb", [0x3185] = "\xc3\xb9", [0x3186] = "\xf2\xbc",
  [0x3187] = "\xf7\x44", [0x3188] = "\xc5\xf9", [0x3189] = "\xf8\xba",
  [0x318a] = "\xa6\xcf", [0x318b] = "\xaa\xcb", [0x318c] = "\xaa\xca",
  [0x318d] = "\xd0\x4f", [0x318e] = "\xac\xfc", [0x3191] = "\xd0\x4e",
  [0x3192] = "\xd3\x62", [0x3194] = "\xaf\xcc", [0x3195] = "\xd6\xf2",
  [0x3196] = "\xd3\x61", [0x319a] = "\xb2\xdc", [0x319b] = "\xd6\xf5",
  [0x319c] = "\xd6\xf3", [0x319d] = "\xd6\xf4", [0x319e] = "\xb2\xdb",
  [0x31a0] = "\xdb\x42", [0x31a1] = "\xdb\x43", [0x31a2] = "\xdb\x41",
  [0x31a4] = "\xb8\x73", [0x31a5] = "\xdf\x6d", [0x31a6] = "\xdf\x6c",
  [0x31a7] = "\xdf\x6e", [0x31a8] = "\xb8\x72", [0x31a9] = "\xb8\x71",
  [0x31ac] = "\xe6\xf2", [0x31ad] = "\xe6\xf4", [0x31af] = "\xbd\x7e",
  [0x31b0] = "\xe6\xf3", [0x31b1] = "\xea\xe3", [0x31b2] = "\xbf\xaa",
  [0x31b3] = "\xf0\x79", [0x31b5] = "\xf0\x78", [0x31b6] = "\xc3\xbb",
  [0x31b7] = "\xf2\xbd", [0x31b8] = "\xc3\xbd", [0x31b9] = "\xc3\xbc",
  [0x31ba] = "\xf4\xb0", [0x31bb] = "\xf5\xee", [0x31bc] = "\xc4\xf3",
  [0x31bd] = "\xa6\xd0", [0x31be] = "\xd0\x50", [0x31bf] = "\xac\xfd",
  [0x31c0] = "\xd3\x65", [0x31c1] = "\xaf\xce", [0x31c2] = "\xd3\x64",
  [0x31c3] = "\xd3\x63", [0x31c5] = "\xaf\xcd", [0x31c7] = "\xd6\xfb",
  [0x31c9] = "\xd6\xfd", [0x31ca] = "\xd6\xf6", [0x31cb] = "\xd6\xf7",
  [0x31cc] = "\xb2\xdd", [0x31cd] = "\xd6\xf8", [0x31ce] = "\xb2\xde",
  [0x31cf] = "\xd6\xfc", [0x31d0] = "\xd6\xf9", [0x31d1] = "\xd6\xfa",
  [0x31d2] = "\xb2\xdf", [0x31d4] = "\xb5\xbe", [0x31d5] = "\xb5\xbf",
  [0x31d7] = "\xdb\x44", [0x31db] = "\xdf\x6f", [0x31dc] = "\xdf\x70",
  [0x31de] = "\xe3\x7e", [0x31df] = "\xbb\x43", [0x31e0] = "\xbb\x41",
  [0x31e1] = "\xbb\x42", [0x31e2] = "\xe3\x7b", [0x31e3] = "\xe3\x7c",
  [0x31e5] = "\xe3\x7d", [0x31e6] = "\xe6\xf9", [0x31e8] = "\xe6\xfa",
  [0x31e9] = "\xbd\xa1", [0x31ea] = "\xe6\xf7", [0x31eb] = "\xe6\xf6",
  [0x31ec] = "\xe6\xf8", [0x31ed] = "\xe6\xf5", [0x31ee] = "\xbf\xad",
  [0x31ef] = "\xea\xe4", [0x31f0] = "\xbf\xab", [0x31f1] = "\xbf\xac",
  [0x31f2] = "\xed\xe6", [0x31f3] = "\xc1\x6b", [0x31f4] = "\xed\xe5",
  [0x31f5] = "\xef\xa8", [0x31f7] = "\xf0\x7a", [0x31f8] = "\xf0\x7b",
  [0x31f9] = "\xc2\xbc", [0x31fb] = "\xc2\xbd", [0x31fc] = "\xc1\x6c",
  [0x31fd] = "\xf2\xbe", [0x31fe] = "\xf2\xbf", [0x31ff] = "\xf4\xb1",
  [0x3200] = "\xc4\xa3", [0x3201] = "\xa6\xd1", [0x3203] = "\xa6\xd2",
  [0x3204] = "\xac\xfe", [0x3205] = "\xaa\xcc", [0x3206] = "\xaf\xcf",
  [0x3207] = "\xd0\x51", [0x320b] = "\xb5\xc0", [0x320c] = "\xa6\xd3",
  [0x320d] = "\xad\x41", [0x320e] = "\xd0\x52", [0x320f] = "\xd0\x53",
  [0x3210] = "\xad\x40", [0x3211] = "\xad\x42", [0x3212] = "\xa6\xd4",
  [0x3214] = "\xd0\x54", [0x3215] = "\xaf\xd1", [0x3216] = "\xd3\x66",
  [0x3217] = "\xaf\xd3", [0x3218] = "\xaf\xd0", [0x3219] = "\xaf\xd2",
  [0x321b] = "\xd7\x41", [0x321c] = "\xb2\xe0", [0x321e] = "\xd7\x40",
  [0x321f] = "\xd6\xfe", [0x3221] = "\xdf\x71", [0x3224] = "\xe3\xa1",
  [0x3226] = "\xbd\xa2", [0x3228] = "\xbf\xae", [0x3229] = "\xea\xe6",
  [0x322a] = "\xea\xe5", [0x322c] = "\xed\xe7", [0x3230] = "\xf5\xef",
  [0x3233] = "\xa6\xd5", [0x3234] = "\xcb\x73", [0x3235] = "\xcd\xaa",
  [0x3236] = "\xad\x43", [0x3237] = "\xd0\x55", [0x3239] = "\xd3\x68",
  [0x323d] = "\xaf\xd4", [0x323e] = "\xd3\x67", [0x323f] = "\xaf\xd5",
  [0x3243] = "\xd7\x43", [0x3246] = "\xb2\xe2", [0x3247] = "\xd7\x42",
  [0x3248] = "\xd7\x44", [0x324a] = "\xb2\xe1", [0x324f] = "\xdb\x46",
  [0x3250] = "\xdb\x47", [0x3251] = "\xdb\x45", [0x3252] = "\xb5\xc1",
  [0x3256] = "\xb8\x74", [0x3258] = "\xb8\x75", [0x325a] = "\xbb\x45",
  [0x325c] = "\xe3\xa3", [0x325d] = "\xe3\xa2", [0x325e] = "\xbb\x44",
  [0x3264] = "\xe6\xfb", [0x3267] = "\xe6\xfc", [0x326c] = "\xea\xe7",
  [0x326f] = "\xc1\x70", [0x3270] = "\xc1\x6f", [0x3271] = "\xc1\x6d",
  [0x3272] = "\xc1\x6e", [0x3273] = "\xc1\x71", [0x3275] = "\xf0\x7c",
  [0x3276] = "\xc2\xbf", [0x3277] = "\xc2\xbe", [0x3278] = "\xf2\xc0",
  [0x3279] = "\xf4\xb2", [0x327d] = "\xc5\xa5", [0x327e] = "\xc5\xa4",
  [0x327f] = "\xa6\xd6", [0x3282] = "\xd1\xfb", [0x3284] = "\xb8\x77",
  [0x3285] = "\xb5\xc2", [0x3286] = "\xb8\x76", [0x3287] = "\xbb\x46",
  [0x3289] = "\xa6\xd7", [0x328a] = "\xc9\xa9", [0x328b] = "\xa6\xd8",
  [0x328c] = "\xa6\xd9", [0x328f] = "\xcd\xab", [0x3290] = "\xcb\x76",
  [0x3292] = "\xcb\x77", [0x3293] = "\xa8\x77", [0x3295] = "\xcb\x74",
  [0x3296] = "\xa8\x76", [0x3298] = "\xa8\x79", [0x3299] = "\xcb\x75",
  [0x329a] = "\xa8\x7b", [0x329b] = "\xa8\x7a", [0x329c] = "\xcb\x78",
  [0x329d] = "\xa8\x78", [0x32a1] = "\xaa\xd1", [0x32a2] = "\xaa\xcf",
  [0x32a3] = "\xcd\xad", [0x32a5] = "\xaa\xce", [0x32a9] = "\xaa\xd3",
  [0x32aa] = "\xaa\xd5", [0x32ab] = "\xaa\xd2", [0x32ad] = "\xcd\xb0",
  [0x32ae] = "\xcd\xac", [0x32af] = "\xaa\xd6", [0x32b1] = "\xaa\xd0",
  [0x32b2] = "\xa8\x7c", [0x32b4] = "\xaa\xd4", [0x32b5] = "\xcd\xaf",
  [0x32b8] = "\xcd\xae", [0x32ba] = "\xaa\xcd", [0x32c2] = "\xd0\x5b",
  [0x32c3] = "\xad\x47", [0x32c4] = "\xad\x48", [0x32c5] = "\xd0\x5d",
  [0x32c7] = "\xd0\x57", [0x32c8] = "\xd0\x5a", [0x32c9] = "\xd0\x63",
  [0x32ca] = "\xd0\x61", [0x32cc] = "\xad\x49", [0x32cd] = "\xd0\x67",
  [0x32ce] = "\xad\x4c", [0x32cf] = "\xd0\x64", [0x32d0] = "\xd0\x5c",
  [0x32d1] = "\xd0\x59", [0x32d4] = "\xdb\x49", [0x32d5] = "\xd0\x62",
  [0x32d6] = "\xad\x44", [0x32d7] = "\xd0\x65", [0x32d8] = "\xd0\x56",
  [0x32d9] = "\xd0\x5f", [0x32da] = "\xad\x46", [0x32db] = "\xad\x4b",
  [0x32dc] = "\xd0\x60", [0x32dd] = "\xad\x4f", [0x32de] = "\xad\x4d",
  [0x32e0] = "\xd0\x58", [0x32e1] = "\xad\x4a", [0x32e3] = "\xd0\x5e",
  [0x32e4] = "\xad\x4e", [0x32e5] = "\xad\x45", [0x32e6] = "\xd0\x66",
  [0x32ed] = "\xaf\xda", [0x32ef] = "\xaf\xe3", [0x32f0] = "\xaf\xd8",
  [0x32f1] = "\xaf\xd6", [0x32f2] = "\xd3\x6a", [0x32f3] = "\xaf\xde",
  [0x32f4] = "\xaf\xdb", [0x32f5] = "\xd3\x6c", [0x32f8] = "\xaf\xdd",
  [0x32f9] = "\xd3\x6b", [0x32fa] = "\xd3\x69", [0x32fb] = "\xd3\x6e",
  [0x32fc] = "\xaf\xe2", [0x32fd] = "\xaf\xe0", [0x32fe] = "\xdb\x48",
  [0x3300] = "\xd3\x6f", [0x3301] = "\xd3\x6d", [0x3302] = "\xaf\xd7",
  [0x3305] = "\xaf\xd9", [0x3306] = "\xaf\xdc", [0x3308] = "\xaf\xdf",
  [0x330a] = "\xaf\xe1", [0x3315] = "\xd7\x4e", [0x3316] = "\xb2\xe4",
  [0x3318] = "\xd7\x45", [0x3319] = "\xd7\x47", [0x331b] = "\xd7\x48",
  [0x331d] = "\xd7\x50", [0x331e] = "\xd7\x4c", [0x331f] = "\xd7\x4a",
  [0x3321] = "\xd7\x4d", [0x3322] = "\xd7\x51", [0x3323] = "\xb2\xe5",
  [0x3324] = "\xb2\xe9", [0x3325] = "\xd7\x46", [0x3327] = "\xd7\x4f",
  [0x3329] = "\xb2\xe7", [0x332b] = "\xb2\xe6", [0x332c] = "\xd7\x4b",
  [0x332d] = "\xd7\x49", [0x332f] = "\xb2\xe3", [0x3330] = "\xb2\xe8",
  [0x3339] = "\xb5\xc8", [0x333a] = "\xdb\x51", [0x333d] = "\xdb\x4f",
  [0x333e] = "\xb5\xca", [0x3343] = "\xdb\x4a", [0x3344] = "\xdf\xa1",
  [0x3346] = "\xb5\xc9", [0x3347] = "\xdb\x4e", [0x334a] = "\xdb\x4b",
  [0x334b] = "\xb5\xc5", [0x334c] = "\xb5\xcb", [0x334d] = "\xdb\x50",
  [0x334e] = "\xb5\xc7", [0x334f] = "\xdb\x4d", [0x3350] = "\xbb\x47",
  [0x3351] = "\xb5\xc6", [0x3352] = "\xdb\x4c", [0x3353] = "\xb5\xcc",
  [0x3354] = "\xb5\xc4", [0x3355] = "\xb5\xc3", [0x335b] = "\xdf\x77",
  [0x335c] = "\xdf\x75", [0x335e] = "\xdf\x7b", [0x3360] = "\xdf\x73",
  [0x3361] = "\xdf\xa2", [0x3362] = "\xdf\x78", [0x3364] = "\xdf\x72",
  [0x3365] = "\xb8\x7b", [0x3366] = "\xb8\xa3", [0x3367] = "\xdf\x7d",
  [0x3369] = "\xdf\x76", [0x336b] = "\xb8\x7e", [0x336e] = "\xb8\x7c",
  [0x336f] = "\xdf\x7e", [0x3370] = "\xb8\x79", [0x3371] = "\xb8\x78",
  [0x3372] = "\xdf\x79", [0x3373] = "\xb8\x7d", [0x3374] = "\xb5\xcd",
  [0x3376] = "\xdf\x7c", [0x3377] = "\xdf\x74", [0x3378] = "\xb8\x7a",
  [0x3379] = "\xb8\xa1", [0x337a] = "\xb8\xa2", [0x337f] = "\xbb\x4c",
  [0x3380] = "\xbb\x48", [0x3382] = "\xbb\x4d", [0x3383] = "\xe3\xa6",
  [0x3386] = "\xe3\xa5", [0x3387] = "\xe3\xa7", [0x3388] = "\xbb\x4a",
  [0x3389] = "\xe3\xa4", [0x338a] = "\xbb\x4b", [0x338b] = "\xe3\xaa",
  [0x338c] = "\xe3\xa9", [0x338d] = "\xe3\xa8", [0x338f] = "\xbb\x49",
  [0x3395] = "\xe7\x41", [0x3397] = "\xe7\x44", [0x3398] = "\xbd\xa8",
  [0x3399] = "\xe7\x43", [0x339a] = "\xbd\xa7", [0x339b] = "\xbd\xa3",
  [0x339c] = "\xbd\xa4", [0x339d] = "\xbd\xa5", [0x339e] = "\xe7\x40",
  [0x339f] = "\xe6\xfe", [0x33a0] = "\xbd\xa6", [0x33a2] = "\xe7\x42",
  [0x33a3] = "\xe6\xfd", [0x33a6] = "\xea\xe9", [0x33a7] = "\xea\xf3",
  [0x33a8] = "\xbf\xb1", [0x33a9] = "\xbf\xb0", [0x33ab] = "\xea\xed",
  [0x33ac] = "\xea\xef", [0x33ae] = "\xea\xea", [0x33b0] = "\xea\xee",
  [0x33b1] = "\xea\xe8", [0x33b2] = "\xea\xf1", [0x33b3] = "\xbf\xaf",
  [0x33b4] = "\xea\xf0", [0x33b5] = "\xea\xec", [0x33b7] = "\xea\xf2",
  [0x33b9] = "\xea\xeb", [0x33ba] = "\xc1\x74", [0x33bb] = "\xed\xe8",
  [0x33bc] = "\xed\xee", [0x33bd] = "\xc1\x78", [0x33be] = "\xc1\x7a",
  [0x33bf] = "\xc1\x77", [0x33c0] = "\xc1\x76", [0x33c2] = "\xc1\x75",
  [0x33c3] = "\xc1\x73", [0x33c4] = "\xed\xe9", [0x33c5] = "\xed\xec",
  [0x33c6] = "\xc1\x72", [0x33c7] = "\xed\xed", [0x33c9] = "\xc1\x79",
  [0x33ca] = "\xed\xeb", [0x33cc] = "\xed\xea", [0x33cd] = "\xc2\xc0",
  [0x33cf] = "\xc2\xc1", [0x33d0] = "\xf0\xa1", [0x33d1] = "\xf0\x7d",
  [0x33d2] = "\xf0\x7e", [0x33d5] = "\xf2\xc2", [0x33d7] = "\xf2\xc1",
  [0x33d8] = "\xc3\xbe", [0x33d9] = "\xf4\xb4", [0x33da] = "\xc4\xa4",
  [0x33db] = "\xf4\xb3", [0x33dd] = "\xf5\xf0", [0x33de] = "\xf7\x45",
  [0x33df] = "\xc5\xa6", [0x33e0] = "\xf9\x43", [0x33e1] = "\xf9\x44",
  [0x33e2] = "\xc5\xd8", [0x33e3] = "\xa6\xda", [0x33e5] = "\xaa\xd7",
  [0x33e6] = "\xdb\x52", [0x33e7] = "\xbb\x4e", [0x33e8] = "\xc1\x7b",
  [0x33e9] = "\xed\xef", [0x33ea] = "\xa6\xdb", [0x33ec] = "\xaf\xe5",
  [0x33ed] = "\xaf\xe4", [0x33ee] = "\xdb\x53", [0x33f2] = "\xea\xf4",
  [0x33f3] = "\xa6\xdc", [0x33f4] = "\xad\x50", [0x33f7] = "\xdb\x54",
  [0x33f8] = "\xdb\x55", [0x33f9] = "\xdb\x56", [0x33fa] = "\xbb\x4f",
  [0x33fb] = "\xbf\xb2", [0x33fc] = "\xa6\xdd", [0x33fe] = "\xaa\xd8",
  [0x33ff] = "\xd0\x68", [0x3400] = "\xaf\xe6", [0x3401] = "\xd3\x70",
  [0x3402] = "\xb2\xea", [0x3404] = "\xdb\x57", [0x3405] = "\xb8\xa4",
  [0x3407] = "\xbb\x50", [0x3408] = "\xbf\xb3", [0x3409] = "\xc1\x7c",
  [0x340a] = "\xc2\xc2", [0x340b] = "\xf4\xb5", [0x340c] = "\xa6\xde",
  [0x340d] = "\xaa\xd9", [0x3410] = "\xaf\xe7", [0x3411] = "\xd7\x52",
  [0x3412] = "\xb5\xce", [0x3414] = "\xbb\x51", [0x3415] = "\xe3\xab",
  [0x3416] = "\xe7\x45", [0x341b] = "\xa6\xdf", [0x341c] = "\xb5\xcf",
  [0x341d] = "\xdf\xa3", [0x341e] = "\xbb\x52", [0x341f] = "\xa6\xe0",
  [0x3420] = "\xcd\xb1", [0x3421] = "\xd0\x69", [0x3422] = "\xad\x51",
  [0x3425] = "\xd3\x72", [0x3428] = "\xaf\xea", [0x342a] = "\xaf\xe8",
  [0x342b] = "\xaf\xe9", [0x342c] = "\xaf\xeb", [0x342f] = "\xd3\x71",
  [0x3432] = "\xd7\x57", [0x3433] = "\xd7\x54", [0x3434] = "\xd7\x56",
  [0x3435] = "\xb2\xeb", [0x3436] = "\xb2\xed", [0x3437] = "\xb2\xec",
  [0x3438] = "\xd7\x53", [0x3439] = "\xb2\xee", [0x343a] = "\xd7\x55",
  [0x343c] = "\xdb\x58", [0x343d] = "\xdb\x59", [0x343f] = "\xdb\x5a",
  [0x3440] = "\xdf\xa6", [0x3442] = "\xdf\xa7", [0x3444] = "\xdf\xa5",
  [0x3445] = "\xdf\xa8", [0x3447] = "\xb8\xa5", [0x3449] = "\xdf\xa4",
  [0x344b] = "\xbb\x53", [0x344e] = "\xe7\x4a", [0x344f] = "\xe7\x46",
  [0x3450] = "\xe7\x49", [0x3451] = "\xe7\x4b", [0x3452] = "\xe7\x48",
  [0x3453] = "\xe7\x47", [0x3455] = "\xea\xf5", [0x3456] = "\xea\xf6",
  [0x3457] = "\xea\xf7", [0x3458] = "\xbf\xb4", [0x3459] = "\xbf\xb5",
  [0x345a] = "\xed\xf1", [0x345b] = "\xed\xf0", [0x345c] = "\xed\xf2",
  [0x345e] = "\xf0\xa3", [0x345f] = "\xf0\xa2", [0x3461] = "\xf2\xc4",
  [0x3463] = "\xf2\xc5", [0x3464] = "\xf2\xc3", [0x3466] = "\xc4\xa5",
  [0x3468] = "\xf4\xb6", [0x3469] = "\xf4\xb7", [0x346b] = "\xf7\x46",
  [0x346c] = "\xf7\xef", [0x346d] = "\xf8\xbb", [0x346e] = "\xa6\xe1",
  [0x346f] = "\xa8\x7d", [0x3471] = "\xc1\x7d", [0x3472] = "\xa6\xe2",
  [0x3474] = "\xd7\x58", [0x3475] = "\xdb\x5b", [0x3477] = "\xc6\x41",
  [0x3478] = "\xca\x4a", [0x347c] = "\xca\x4b", [0x347d] = "\xca\x4d",
  [0x347e] = "\xa6\xe3", [0x347f] = "\xca\x4e", [0x3480] = "\xca\x4c",
  [0x3483] = "\xcb\xa2", [0x3484] = "\xcb\xa3", [0x3485] = "\xcb\x7b",
  [0x348a] = "\xcb\xa1", [0x348b] = "\xa8\xa1", [0x348d] = "\xa8\xa2",
  [0x348e] = "\xcb\x7c", [0x348f] = "\xcb\x7a", [0x3490] = "\xcb\x79",
  [0x3491] = "\xcb\x7d", [0x3492] = "\xa8\x7e", [0x3493] = "\xcb\x7e",
  [0x3494] = "\xd0\x6a", [0x3498] = "\xcd\xb6", [0x3499] = "\xaa\xdc",
  [0x349a] = "\xcd\xb5", [0x349b] = "\xcd\xb7", [0x349d] = "\xaa\xdb",
  [0x349e] = "\xcd\xbc", [0x349f] = "\xaa\xdf", [0x34a0] = "\xcd\xb2",
  [0x34a1] = "\xcd\xc0", [0x34a2] = "\xcd\xc6", [0x34a3] = "\xaa\xe6",
  [0x34a4] = "\xcd\xc3", [0x34a5] = "\xaa\xe3", [0x34a7] = "\xcd\xb9",
  [0x34a8] = "\xcd\xbf", [0x34a9] = "\xcd\xc1", [0x34ab] = "\xcd\xb4",
  [0x34ac] = "\xaa\xe2", [0x34ad] = "\xaa\xdd", [0x34ae] = "\xcd\xba",
  [0x34af] = "\xaa\xe4", [0x34b0] = "\xaa\xe7", [0x34b1] = "\xaa\xe1",
  [0x34b3] = "\xaa\xda", [0x34b4] = "\xcd\xbe", [0x34b5] = "\xcd\xb8",
  [0x34b6] = "\xcd\xc5", [0x34b7] = "\xaa\xe9", [0x34b8] = "\xaa\xe5",
  [0x34b9] = "\xaa\xe0", [0x34ba] = "\xcd\xbd", [0x34bb] = "\xaf\xec",
  [0x34bc] = "\xcd\xbb", [0x34bd] = "\xaa\xde", [0x34be] = "\xaa\xe8",
  [0x34c0] = "\xcd\xb3", [0x34c2] = "\xcd\xc2", [0x34c3] = "\xcd\xc4",
  [0x34d1] = "\xad\x62", [0x34d2] = "\xad\x5c", [0x34d3] = "\xad\x64",
  [0x34d4] = "\xad\x61", [0x34d5] = "\xd0\x71", [0x34d6] = "\xd0\x74",
  [0x34d7] = "\xad\x5d", [0x34d9] = "\xd0\x6b", [0x34db] = "\xad\x56",
  [0x34dc] = "\xad\x60", [0x34de] = "\xad\x63", [0x34df] = "\xad\x65",
  [0x34e0] = "\xd0\xa2", [0x34e1] = "\xd0\x77", [0x34e3] = "\xad\x55",
  [0x34e4] = "\xd0\xa1", [0x34e5] = "\xad\x59", [0x34e6] = "\xad\x57",
  [0x34e7] = "\xad\x52", [0x34e8] = "\xd0\x6f", [0x34ea] = "\xd0\x7e",
  [0x34eb] = "\xd0\x73", [0x34ec] = "\xd0\x76", [0x34ed] = "\xd0\xa5",
  [0x34ef] = "\xad\x66", [0x34f0] = "\xd0\x7d", [0x34f1] = "\xad\x5e",
  [0x34f2] = "\xd0\x78", [0x34f3] = "\xd0\xa4", [0x34f4] = "\xd0\x75",
  [0x34f5] = "\xd0\x79", [0x34f6] = "\xd0\x7c", [0x34f9] = "\xd0\x6d",
  [0x34fa] = "\xd0\xa3", [0x34fb] = "\xd0\x7b", [0x34fe] = "\xd0\x6c",
  [0x3500] = "\xd0\x70", [0x3501] = "\xad\x5f", [0x3502] = "\xad\x5a",
  [0x3503] = "\xad\x53", [0x3504] = "\xad\x58", [0x3505] = "\xad\x54",
  [0x3506] = "\xad\x67", [0x3507] = "\xd0\x6e", [0x3508] = "\xd3\xa5",
  [0x3509] = "\xad\x5b", [0x350c] = "\xd0\x7a", [0x350d] = "\xce\x41",
  [0x3516] = "\xd3\xa8", [0x3517] = "\xaf\xfa", [0x3519] = "\xd3\x76",
  [0x351b] = "\xd3\xa3", [0x351c] = "\xd3\x7d", [0x351e] = "\xd3\xb2",
  [0x3520] = "\xd3\xaa", [0x3522] = "\xd3\x7e", [0x3524] = "\xd3\xa9",
  [0x3525] = "\xd3\x78", [0x3526] = "\xd3\x7c", [0x3527] = "\xd3\xb5",
  [0x3528] = "\xaf\xfd", [0x3529] = "\xd3\xad", [0x352a] = "\xd3\xa4",
  [0x352b] = "\xaf\xed", [0x352c] = "\xd3\xb3", [0x352d] = "\xd3\x74",
  [0x352f] = "\xd3\xac", [0x3531] = "\xaf\xfc", [0x3532] = "\xaf\xf7",
  [0x3533] = "\xd3\x73", [0x3534] = "\xaf\xf5", [0x3535] = "\xaf\xf4",
  [0x3536] = "\xaf\xf9", [0x3537] = "\xd3\xab", [0x3538] = "\xaf\xf1",
  [0x3539] = "\xaf\xf8", [0x353a] = "\xd0\x72", [0x353b] = "\xdb\x5c",
  [0x353c] = "\xd3\xa6", [0x353f] = "\xd3\x7a", [0x3540] = "\xaf\xfb",
  [0x3541] = "\xd3\x7b", [0x3542] = "\xd3\xa1", [0x3543] = "\xaf\xfe",
  [0x3544] = "\xd3\x75", [0x3545] = "\xd3\xaf", [0x3547] = "\xd3\xae",
  [0x3548] = "\xd3\xb6", [0x3549] = "\xaf\xf3", [0x354a] = "\xaf\xf0",
  [0x354b] = "\xd3\xb4", [0x354c] = "\xd3\xb0", [0x354d] = "\xd3\xa7",
  [0x354e] = "\xd3\xa2", [0x354f] = "\xaf\xf6", [0x3550] = "\xaf\xf2",
  [0x3551] = "\xd3\x77", [0x3552] = "\xaf\xee", [0x3553] = "\xd3\xb1",
  [0x3554] = "\xaf\xef", [0x3556] = "\xd3\x79", [0x3573] = "\xd7\x5e",
  [0x3574] = "\xd7\x60", [0x3575] = "\xd7\x65", [0x3576] = "\xd7\x79",
  [0x3577] = "\xb2\xfc", [0x3578] = "\xb2\xf2", [0x357a] = "\xd7\x5d",
  [0x357b] = "\xb2\xfd", [0x357c] = "\xb2\xfe", [0x357d] = "\xd7\x68",
  [0x357e] = "\xd7\x6f", [0x357f] = "\xd7\x75", [0x3581] = "\xd7\x62",
  [0x3583] = "\xd7\x69", [0x3586] = "\xb3\x40", [0x3587] = "\xd7\x77",
  [0x3588] = "\xd7\x72", [0x3589] = "\xb2\xfa", [0x358a] = "\xb2\xf8",
  [0x358b] = "\xd7\x6e", [0x358c] = "\xd7\x6a", [0x358d] = "\xd7\x5c",
  [0x358e] = "\xb2\xef", [0x358f] = "\xd7\x61", [0x3590] = "\xd7\x59",
  [0x3592] = "\xb2\xf7", [0x3593] = "\xb2\xf9", [0x3594] = "\xd7\x66",
  [0x3595] = "\xd7\x63", [0x3596] = "\xb2\xf4", [0x3597] = "\xd7\x73",
  [0x3598] = "\xb2\xf1", [0x3599] = "\xd7\x64", [0x359a] = "\xd7\x7a",
  [0x359b] = "\xd7\x6c", [0x359d] = "\xd7\x6b", [0x359e] = "\xb2\xf0",
  [0x35a0] = "\xb2\xfb", [0x35a2] = "\xb2\xf3", [0x35a3] = "\xd7\x5a",
  [0x35a4] = "\xd7\x5f", [0x35a5] = "\xd7\x70", [0x35a6] = "\xd7\x76",
  [0x35a7] = "\xb3\x41", [0x35a8] = "\xd7\x5b", [0x35a9] = "\xd7\x67",
  [0x35aa] = "\xd7\x6d", [0x35ab] = "\xb2\xf6", [0x35ae] = "\xd7\x78",
  [0x35af] = "\xd7\x71", [0x35b0] = "\xd7\x74", [0x35bd] = "\xb2\xf5",
  [0x35bf] = "\xdb\x6c", [0x35c0] = "\xdb\x60", [0x35c1] = "\xb5\xd7",
  [0x35c2] = "\xdb\x7d", [0x35c3] = "\xdb\xa7", [0x35c4] = "\xdb\xaa",
  [0x35c5] = "\xb5\xd5", [0x35c6] = "\xdb\x68", [0x35c7] = "\xdb\xa3",
  [0x35c8] = "\xdb\x69", [0x35c9] = "\xdb\x77", [0x35ca] = "\xb5\xe2",
  [0x35cb] = "\xdb\x73", [0x35cc] = "\xb5\xdf", [0x35ce] = "\xdb\x74",
  [0x35cf] = "\xdb\x5d", [0x35d1] = "\xdb\xa4", [0x35d4] = "\xb5\xe8",
  [0x35d5] = "\xdb\xa1", [0x35d6] = "\xdb\x75", [0x35d7] = "\xdb\xac",
  [0x35d8] = "\xdb\x70", [0x35d9] = "\xdf\xc8", [0x35db] = "\xdb\xaf",
  [0x35dc] = "\xb5\xe6", [0x35dd] = "\xdb\x6e", [0x35de] = "\xdb\x7a",
  [0x35df] = "\xb5\xe9", [0x35e0] = "\xb5\xd4", [0x35e1] = "\xdb\x72",
  [0x35e2] = "\xdb\xad", [0x35e3] = "\xdb\x6b", [0x35e4] = "\xdb\x64",
  [0x35e5] = "\xdb\x6f", [0x35e7] = "\xdb\x63", [0x35e8] = "\xdb\x61",
  [0x35e9] = "\xb5\xd0", [0x35ea] = "\xdb\xa5", [0x35eb] = "\xdb\x6a",
  [0x35ec] = "\xdb\xa8", [0x35ee] = "\xdb\xa9", [0x35ef] = "\xb5\xd8",
  [0x35f0] = "\xb5\xdd", [0x35f1] = "\xb5\xd9", [0x35f2] = "\xb5\xe1",
  [0x35f3] = "\xdb\x7e", [0x35f4] = "\xb5\xda", [0x35f5] = "\xdb\x76",
  [0x35f6] = "\xdb\x66", [0x35f8] = "\xb5\xd2", [0x35f9] = "\xdb\x5e",
  [0x35fa] = "\xdb\xa2", [0x35fb] = "\xdb\xab", [0x35fc] = "\xdb\x65",
  [0x35fd] = "\xb5\xe0", [0x35fe] = "\xdb\xb0", [0x35ff] = "\xdb\x71",
  [0x3601] = "\xdb\x6d", [0x3603] = "\xb5\xd1", [0x3604] = "\xb5\xe5",
  [0x3606] = "\xdb\x7c", [0x3607] = "\xb5\xe7", [0x3609] = "\xdb\x78",
  [0x360a] = "\xb5\xdc", [0x360b] = "\xb5\xd6", [0x360c] = "\xb5\xde",
  [0x360d] = "\xb5\xd3", [0x360e] = "\xb5\xe4", [0x360f] = "\xdb\x79",
  [0x3610] = "\xdb\x67", [0x3611] = "\xdb\x7b", [0x3612] = "\xdb\x62",
  [0x3613] = "\xdb\xa6", [0x361b] = "\xdb\xae", [0x3623] = "\xdb\x5f",
  [0x3629] = "\xdf\xc7", [0x362b] = "\xdf\xdd", [0x362c] = "\xb8\x55",
  [0x362d] = "\xdf\xcc", [0x362f] = "\xdf\xca", [0x3630] = "\xdf\xb5",
  [0x3631] = "\xb8\xa9", [0x3632] = "\xdf\xc5", [0x3633] = "\xdf\xd9",
  [0x3634] = "\xdf\xc1", [0x3635] = "\xb8\xb1", [0x3636] = "\xdf\xd8",
  [0x3637] = "\xdf\xbf", [0x3638] = "\xb5\xe3", [0x3639] = "\xdf\xcf",
  [0x363a] = "\xdf\xc0", [0x363b] = "\xdf\xd6", [0x363c] = "\xb8\xb0",
  [0x363d] = "\xb8\xa8", [0x363f] = "\xdf\xaa", [0x3640] = "\xdf\xb2",
  [0x3642] = "\xdf\xcb", [0x3643] = "\xdf\xc3", [0x3644] = "\xdf\xdc",
  [0x3645] = "\xdf\xc6", [0x3646] = "\xb8\xb6", [0x3647] = "\xdf\xd7",
  [0x3649] = "\xb8\xad", [0x364b] = "\xdf\xc9", [0x364c] = "\xdf\xd1",
  [0x364d] = "\xdf\xb6", [0x364e] = "\xdf\xd0", [0x3650] = "\xdf\xe1",
  [0x3651] = "\xdf\xb1", [0x3652] = "\xdf\xd2", [0x3654] = "\xdf\xdf",
  [0x3656] = "\xdf\xab", [0x3657] = "\xb5\xdb", [0x3659] = "\xdf\xb9",
  [0x365a] = "\xdf\xb8", [0x365b] = "\xb8\xaf", [0x365d] = "\xdf\xbc",
  [0x365e] = "\xdf\xbe", [0x365f] = "\xdf\xcd", [0x3660] = "\xdf\xde",
  [0x3661] = "\xb8\xb2", [0x3663] = "\xb8\xb3", [0x3665] = "\xdf\xb0",
  [0x3666] = "\xb8\xab", [0x3667] = "\xdf\xb4", [0x3668] = "\xdf\xda",
  [0x3669] = "\xb8\xb4", [0x366b] = "\xb8\xac", [0x366c] = "\xb8\xae",
  [0x366d] = "\xb8\xb5", [0x366e] = "\xdf\xe0", [0x366f] = "\xdf\xd3",
  [0x3670] = "\xdf\xce", [0x3673] = "\xdf\xbb", [0x3674] = "\xdf\xba",
  [0x3675] = "\xb8\xaa", [0x3676] = "\xdf\xac", [0x3677] = "\xb8\xa7",
  [0x3678] = "\xdf\xc4", [0x3679] = "\xdf\xad", [0x367a] = "\xdf\xc2",
  [0x367d] = "\xdf\xb7", [0x367e] = "\xdf\xdb", [0x3682] = "\xb8\xa6",
  [0x3686] = "\xdf\xb3", [0x368d] = "\xdf\xaf", [0x368e] = "\xdf\xd5",
  [0x368f] = "\xdf\xae", [0x3690] = "\xbb\x60", [0x3691] = "\xe3\xd3",
  [0x3694] = "\xe3\xc2", [0x3697] = "\xe3\xac", [0x3698] = "\xe3\xca",
  [0x3699] = "\xbb\x58", [0x369a] = "\xe3\xbb", [0x369b] = "\xe3\xc5",
  [0x369c] = "\xbb\x5b", [0x369d] = "\xe3\xbe", [0x369e] = "\xbb\x59",
  [0x369f] = "\xe3\xaf", [0x36a0] = "\xe3\xcd", [0x36a1] = "\xe3\xae",
  [0x36a2] = "\xe3\xc1", [0x36a4] = "\xe3\xad", [0x36a7] = "\xe3\xbf",
  [0x36a8] = "\xe3\xc8", [0x36a9] = "\xe3\xc6", [0x36aa] = "\xe3\xba",
  [0x36ab] = "\xe3\xb5", [0x36ac] = "\xe3\xb3", [0x36ae] = "\xe3\xb4",
  [0x36af] = "\xe3\xc7", [0x36b0] = "\xe3\xd2", [0x36b1] = "\xe3\xbc",
  [0x36b2] = "\xbb\x5a", [0x36b4] = "\xe3\xb7", [0x36b6] = "\xe3\xcb",
  [0x36b8] = "\xbb\x5d", [0x36b9] = "\xe3\xb6", [0x36ba] = "\xe3\xb0",
  [0x36bb] = "\xe3\xc0", [0x36bc] = "\xbb\x61", [0x36bf] = "\xbb\x55",
  [0x36c0] = "\xbb\x5e", [0x36c1] = "\xe3\xb8", [0x36c2] = "\xe3\xb2",
  [0x36c4] = "\xbb\x57", [0x36c5] = "\xdf\xd4", [0x36c6] = "\xbb\x56",
  [0x36c7] = "\xe3\xc3", [0x36c9] = "\xbb\x54", [0x36ca] = "\xbb\x63",
  [0x36cb] = "\xbb\x5c", [0x36cc] = "\xe3\xc4", [0x36cd] = "\xe3\xb9",
  [0x36ce] = "\xe3\xb1", [0x36cf] = "\xe3\xcc", [0x36d0] = "\xe3\xbd",
  [0x36d1] = "\xbb\x62", [0x36d2] = "\xe3\xd0", [0x36d3] = "\xbb\x5f",
  [0x36d4] = "\xe3\xcf", [0x36d6] = "\xe3\xc9", [0x36d7] = "\xe3\xce",
  [0x36db] = "\xe3\xd1", [0x36e7] = "\xe7\x73", [0x36e8] = "\xe7\x74",
  [0x36e9] = "\xe7\x67", [0x36ea] = "\xe7\x66", [0x36eb] = "\xe7\x62",
  [0x36ec] = "\xbd\xb4", [0x36ee] = "\xbd\xac", [0x36ef] = "\xe7\x76",
  [0x36f0] = "\xe7\x75", [0x36f1] = "\xdf\xa9", [0x36f2] = "\xe7\x5f",
  [0x36f3] = "\xe7\x63", [0x36f4] = "\xe7\x5d", [0x36f6] = "\xe7\x70",
  [0x36f7] = "\xe7\x61", [0x36f9] = "\xe7\x77", [0x36fa] = "\xe7\x5a",
  [0x36fb] = "\xe7\x58", [0x36fc] = "\xe7\x64", [0x36fd] = "\xe7\x6e",
  [0x36fe] = "\xe7\x69", [0x36ff] = "\xbd\xb6", [0x3700] = "\xe7\x4f",
  [0x3702] = "\xe7\x6d", [0x3706] = "\xbd\xb7", [0x3707] = "\xdf\xbd",
  [0x3708] = "\xe7\x5b", [0x3709] = "\xe7\x52", [0x370a] = "\xe7\x55",
  [0x370b] = "\xe7\x7b", [0x370c] = "\xe7\x5c", [0x370d] = "\xe7\x53",
  [0x370e] = "\xe7\x51", [0x370f] = "\xe7\x4e", [0x3711] = "\xbd\xb0",
  [0x3712] = "\xe7\x65", [0x3713] = "\xbd\xaf", [0x3714] = "\xbd\xb3",
  [0x3715] = "\xe7\x60", [0x3716] = "\xe7\x68", [0x3717] = "\xbd\xa9",
  [0x3718] = "\xe7\x78", [0x3719] = "\xe7\x7c", [0x371a] = "\xbd\xab",
  [0x371c] = "\xe7\x57", [0x371d] = "\xe7\x6b", [0x371e] = "\xe7\x6f",
  [0x371f] = "\xe7\x54", [0x3720] = "\xe7\x79", [0x3721] = "\xbd\xb2",
  [0x3723] = "\xbd\xb1", [0x3724] = "\xe7\x4c", [0x3725] = "\xbd\xb5",
  [0x3726] = "\xe7\x72", [0x3727] = "\xe7\x56", [0x3728] = "\xe7\x6a",
  [0x3729] = "\xe7\x50", [0x372a] = "\xe7\x5e", [0x372b] = "\xe7\x59",
  [0x372c] = "\xbd\xad", [0x372d] = "\xbd\xae", [0x372e] = "\xe7\x6c",
  [0x372f] = "\xe7\x7d", [0x3730] = "\xe7\x7a", [0x3731] = "\xe7\x71",
  [0x373b] = "\xe7\x4d", [0x373d] = "\xbd\xaa", [0x373e] = "\xeb\x49",
  [0x3740] = "\xeb\x40", [0x3741] = "\xeb\x43", [0x3743] = "\xbf\xbb",
  [0x3744] = "\xeb\x45", [0x3745] = "\xea\xf9", [0x3746] = "\xeb\x41",
  [0x3747] = "\xeb\x47", [0x3748] = "\xbf\xb8", [0x3749] = "\xbf\xbc",
  [0x374a] = "\xbf\xb6", [0x374d] = "\xea\xfb", [0x374e] = "\xeb\x4c",
  [0x3751] = "\xeb\x46", [0x3753] = "\xea\xfc", [0x3754] = "\xeb\x55",
  [0x3755] = "\xeb\x4f", [0x3756] = "\xea\xf8", [0x3757] = "\xee\x46",
  [0x3758] = "\xea\xfe", [0x3759] = "\xbf\xb7", [0x375b] = "\xeb\x4a",
  [0x375d] = "\xeb\x54", [0x375e] = "\xbf\xbf", [0x3760] = "\xeb\x51",
  [0x3761] = "\xea\xfd", [0x3762] = "\xeb\x44", [0x3763] = "\xeb\x48",
  [0x3764] = "\xeb\x42", [0x3765] = "\xeb\x56", [0x3766] = "\xeb\x53",
  [0x3767] = "\xeb\x50", [0x3768] = "\xbf\xb9", [0x3769] = "\xbf\xba",
  [0x376a] = "\xbf\xbe", [0x376b] = "\xea\xfa", [0x376c] = "\xeb\x57",
  [0x376d] = "\xbf\xbd", [0x376e] = "\xeb\x4d", [0x3771] = "\xeb\x4b",
  [0x3775] = "\xeb\x4e", [0x3776] = "\xee\x53", [0x3777] = "\xee\x40",
  [0x3778] = "\xee\x45", [0x3779] = "\xee\x52", [0x377a] = "\xee\x44",
  [0x377b] = "\xed\xfb", [0x377c] = "\xee\x41", [0x377e] = "\xc1\xa2",
  [0x3780] = "\xed\xf4", [0x3781] = "\xee\x4d", [0x3782] = "\xee\x4f",
  [0x3783] = "\xed\xf3", [0x3784] = "\xc1\xa1", [0x3785] = "\xee\x51",
  [0x3786] = "\xee\x49", [0x3787] = "\xc1\xa8", [0x3788] = "\xee\x50",
  [0x3789] = "\xee\x42", [0x378a] = "\xc1\xaa", [0x378b] = "\xed\xf9",
  [0x378c] = "\xeb\x52", [0x378d] = "\xee\x4a", [0x378e] = "\xee\x47",
  [0x378f] = "\xed\xf5", [0x3790] = "\xee\x55", [0x3791] = "\xc1\xa4",
  [0x3794] = "\xc1\xa5", [0x3795] = "\xed\xf7", [0x3796] = "\xee\x48",
  [0x3798] = "\xee\x54", [0x3799] = "\xee\x4b", [0x379a] = "\xed\xfd",
  [0x379b] = "\xc1\xa7", [0x379c] = "\xc1\xa3", [0x379d] = "\xee\x4c",
  [0x379e] = "\xed\xfe", [0x379f] = "\xee\x56", [0x37a0] = "\xed\xf8",
  [0x37a1] = "\xee\x43", [0x37a2] = "\xee\x4e", [0x37a3] = "\xed\xfa",
  [0x37a4] = "\xed\xfc", [0x37a6] = "\xc2\xcb", [0x37a7] = "\xed\xf6",
  [0x37a8] = "\xc1\xa9", [0x37a9] = "\xc2\xc4", [0x37aa] = "\xc1\x7e",
  [0x37af] = "\xc1\xa6", [0x37b0] = "\xc2\xc8", [0x37b1] = "\xf0\xb3",
  [0x37b3] = "\xf0\xa9", [0x37b4] = "\xf0\xa4", [0x37b5] = "\xf0\xaa",
  [0x37b6] = "\xf0\xb4", [0x37b7] = "\xf0\xb8", [0x37b8] = "\xf0\xb7",
  [0x37b9] = "\xc2\xca", [0x37ba] = "\xc2\xc9", [0x37bd] = "\xf0\xab",
  [0x37be] = "\xf0\xb9", [0x37bf] = "\xf0\xae", [0x37c0] = "\xf0\xa6",
  [0x37c2] = "\xf0\xa8", [0x37c3] = "\xf0\xa7", [0x37c4] = "\xf0\xad",
  [0x37c5] = "\xf0\xb2", [0x37c6] = "\xf0\xa5", [0x37c7] = "\xf0\xac",
  [0x37c8] = "\xf0\xb1", [0x37c9] = "\xc2\xc7", [0x37cb] = "\xf0\xaf",
  [0x37cd] = "\xc2\xc5", [0x37ce] = "\xf0\xb0", [0x37cf] = "\xc2\xc3",
  [0x37d0] = "\xc2\xc6", [0x37d1] = "\xf2\xd5", [0x37d2] = "\xf0\xb5",
  [0x37d5] = "\xc3\xc2", [0x37d7] = "\xf2\xcd", [0x37d8] = "\xf2\xd1",
  [0x37d9] = "\xf2\xc9", [0x37da] = "\xf2\xcc", [0x37dc] = "\xf2\xd4",
  [0x37dd] = "\xc3\xc0", [0x37de] = "\xf2\xd9", [0x37df] = "\xf2\xd2",
  [0x37e1] = "\xf2\xca", [0x37e2] = "\xf2\xda", [0x37e3] = "\xf2\xd3",
  [0x37e4] = "\xc3\xc3", [0x37e5] = "\xc3\xc4", [0x37e6] = "\xf2\xd7",
  [0x37e8] = "\xf2\xcb", [0x37e9] = "\xc3\xbf", [0x37ea] = "\xc3\xc1",
  [0x37eb] = "\xf2\xc6", [0x37ec] = "\xf2\xce", [0x37ed] = "\xf2\xc8",
  [0x37ef] = "\xf2\xd8", [0x37f0] = "\xf2\xd6", [0x37f1] = "\xf2\xc7",
  [0x37f2] = "\xf2\xcf", [0x37f6] = "\xf4\xbe", [0x37f7] = "\xc3\xc5",
  [0x37f8] = "\xf2\xd0", [0x37f9] = "\xc4\xa7", [0x37fa] = "\xc4\xa9",
  [0x37fb] = "\xc4\xa6", [0x37fd] = "\xf4\xc3", [0x37fe] = "\xf4\xbb",
  [0x37ff] = "\xf4\xb9", [0x3800] = "\xf4\xbd", [0x3801] = "\xf4\xba",
  [0x3804] = "\xf4\xbf", [0x3805] = "\xf4\xc1", [0x3806] = "\xc4\xaa",
  [0x3807] = "\xc4\xac", [0x3809] = "\xf4\xc0", [0x380a] = "\xc4\xad",
  [0x380b] = "\xc4\xab", [0x380c] = "\xf4\xc2", [0x3811] = "\xc4\xa8",
  [0x3817] = "\xc4\xf4", [0x3818] = "\xf5\xf1", [0x3819] = "\xf5\xf7",
  [0x381a] = "\xc4\xf6", [0x381b] = "\xf4\xbc", [0x381c] = "\xf5\xf6",
  [0x381e] = "\xf5\xfd", [0x381f] = "\xf5\xf4", [0x3820] = "\xf5\xfb",
  [0x3821] = "\xf5\xfa", [0x3822] = "\xf4\xb8", [0x3823] = "\xf5\xf5",
  [0x3824] = "\xf0\xb6", [0x3825] = "\xf5\xfe", [0x3826] = "\xf5\xf3",
  [0x3827] = "\xf5\xf8", [0x3829] = "\xf5\xfc", [0x382a] = "\xf5\xf2",
  [0x382c] = "\xf7\x4a", [0x382d] = "\xc4\xf5", [0x382e] = "\xf5\xf9",
  [0x3831] = "\xf7\xf4", [0x3832] = "\xf7\x4b", [0x3833] = "\xf7\x49",
  [0x3834] = "\xf7\x47", [0x3835] = "\xf7\x48", [0x3836] = "\xf7\x4c",
  [0x3838] = "\xc5\xd9", [0x3839] = "\xf7\xf2", [0x383a] = "\xf7\xf0",
  [0x383b] = "\xf7\xf5", [0x383c] = "\xf7\xf3", [0x383e] = "\xf7\xf6",
  [0x383f] = "\xc5\xda", [0x3840] = "\xf7\xf1", [0x3843] = "\xf8\xbc",
  [0x3846] = "\xf9\x45", [0x3847] = "\xf9\x46", [0x3848] = "\xf9\x47",
  [0x384b] = "\xf9\xc7", [0x384c] = "\xf9\xbd", [0x384d] = "\xca\x4f",
  [0x384e] = "\xaa\xea", [0x3850] = "\xad\x68", [0x3852] = "\xd3\xb8",
  [0x3853] = "\xd3\xb7", [0x3854] = "\xb0\x40", [0x3855] = "\xb3\x42",
  [0x3856] = "\xd7\x7c", [0x3859] = "\xd7\x7b", [0x385b] = "\xb5\xea",
  [0x385c] = "\xb8\xb8", [0x385e] = "\xb8\xb7", [0x385f] = "\xb8\xb9",
  [0x3861] = "\xe3\xd4", [0x3862] = "\xe7\x7e", [0x3863] = "\xeb\x58",
  [0x3864] = "\xeb\x5a", [0x3865] = "\xeb\x59", [0x3867] = "\xc1\xab",
  [0x3868] = "\xee\x57", [0x3869] = "\xf0\xba", [0x386a] = "\xf9\xa5",
  [0x386b] = "\xa6\xe4", [0x386d] = "\xcd\xc9", [0x386e] = "\xcd\xca",
  [0x386f] = "\xcd\xc8", [0x3870] = "\xcd\xc7", [0x3871] = "\xaa\xeb",
  [0x3873] = "\xd0\xa9", [0x3874] = "\xd0\xa7", [0x3877] = "\xd0\xa6",
  [0x3879] = "\xad\x69", [0x387a] = "\xad\x6b", [0x387b] = "\xad\x6a",
  [0x387c] = "\xd0\xa8", [0x3885] = "\xd3\xc4", [0x3886] = "\xd3\xc1",
  [0x3887] = "\xd3\xbf", [0x388a] = "\xb0\x41", [0x388b] = "\xd3\xc2",
  [0x388c] = "\xb0\x46", [0x388d] = "\xd3\xbc", [0x388e] = "\xd3\xcb",
  [0x3890] = "\xd3\xcd", [0x3891] = "\xd3\xbd", [0x3893] = "\xb0\x43",
  [0x3894] = "\xd3\xce", [0x3895] = "\xd3\xc9", [0x3896] = "\xd3\xbb",
  [0x3897] = "\xd3\xc0", [0x3898] = "\xd3\xca", [0x3899] = "\xd3\xc6",
  [0x389a] = "\xd3\xc3", [0x389c] = "\xb0\x48", [0x389d] = "\xd3\xcc",
  [0x389e] = "\xd3\xbe", [0x38a1] = "\xd3\xc7", [0x38a2] = "\xd3\xb9",
  [0x38a3] = "\xb0\x47", [0x38a4] = "\xb0\x44", [0x38a5] = "\xd3\xc5",
  [0x38a7] = "\xd3\xc8", [0x38a8] = "\xd3\xba", [0x38a9] = "\xb0\x45",
  [0x38aa] = "\xb0\x42", [0x38af] = "\xb3\x4c", [0x38b0] = "\xd7\xa5",
  [0x38b1] = "\xb3\x4b", [0x38b3] = "\xd7\xa8", [0x38b4] = "\xd7\xab",
  [0x38b5] = "\xb3\x48", [0x38b6] = "\xb3\x46", [0x38b7] = "\xd7\x7e",
  [0x38b8] = "\xd7\xa9", [0x38b9] = "\xd7\xa7", [0x38ba] = "\xd7\xa4",
  [0x38bb] = "\xd7\xac", [0x38bc] = "\xd7\xad", [0x38bd] = "\xd7\xaf",
  [0x38be] = "\xd7\xb0", [0x38bf] = "\xd7\x7d", [0x38c0] = "\xb3\x45",
  [0x38c1] = "\xd7\xa2", [0x38c2] = "\xd7\xa1", [0x38c3] = "\xd7\xae",
  [0x38c4] = "\xb3\x47", [0x38c5] = "\xd7\xa3", [0x38c6] = "\xb3\x49",
  [0x38c7] = "\xb3\x44", [0x38c8] = "\xd7\xa6", [0x38c9] = "\xb3\x4d",
  [0x38cb] = "\xb3\x4a", [0x38cc] = "\xd7\xaa", [0x38d0] = "\xb5\xf1",
  [0x38d1] = "\xdb\xbf", [0x38d3] = "\xdb\xb4", [0x38d4] = "\xb5\xee",
  [0x38d6] = "\xdf\xe7", [0x38d7] = "\xdb\xbd", [0x38d8] = "\xdb\xb1",
  [0x38d9] = "\xb5\xec", [0x38da] = "\xdb\xb6", [0x38db] = "\xb5\xef",
  [0x38dc] = "\xdb\xba", [0x38dd] = "\xdb\xb8", [0x38de] = "\xb5\xf2",
  [0x38df] = "\xb5\xeb", [0x38e2] = "\xdb\xb2", [0x38e3] = "\xdb\xb5",
  [0x38e4] = "\xb5\xf0", [0x38e6] = "\xdb\xb3", [0x38e8] = "\xdb\xbe",
  [0x38e9] = "\xdb\xbc", [0x38ea] = "\xdb\xb7", [0x38eb] = "\xdb\xb9",
  [0x38ec] = "\xdb\xbb", [0x38ed] = "\xb5\xed", [0x38f5] = "\xdf\xe8",
  [0x38f6] = "\xdf\xee", [0x38f7] = "\xdf\xe4", [0x38f8] = "\xdf\xea",
  [0x38f9] = "\xb8\xba", [0x38fa] = "\xdf\xe6", [0x38fb] = "\xb8\xc0",
  [0x38fe] = "\xb8\xbf", [0x3900] = "\xb8\xbe", [0x3901] = "\xdf\xed",
  [0x3902] = "\xb8\xc1", [0x3903] = "\xb8\xc2", [0x3904] = "\xdf\xe3",
  [0x3905] = "\xdf\xf0", [0x3906] = "\xb8\xc3", [0x3907] = "\xb8\xbd",
  [0x3908] = "\xb8\xbc", [0x3909] = "\xdf\xec", [0x390a] = "\xb8\xc4",
  [0x390b] = "\xdf\xe2", [0x390c] = "\xdf\xe5", [0x390d] = "\xdf\xef",
  [0x390e] = "\xdf\xeb", [0x3911] = "\xe3\xf4", [0x3912] = "\xe3\xe9",
  [0x3913] = "\xb8\xbb", [0x3918] = "\xbb\x6a", [0x3919] = "\xe3\xdd",
  [0x391a] = "\xe3\xf2", [0x391b] = "\xe3\xde", [0x391c] = "\xbb\x65",
  [0x391e] = "\xe3\xdb", [0x3920] = "\xe3\xe4", [0x3921] = "\xe3\xdc",
  [0x3922] = "\xbb\x67", [0x3923] = "\xe3\xd6", [0x3924] = "\xe3\xf1",
  [0x3925] = "\xbb\x68", [0x3926] = "\xe3\xee", [0x3927] = "\xe3\xef",
  [0x3928] = "\xe3\xd7", [0x3929] = "\xbb\x6d", [0x392a] = "\xe3\xe6",
  [0x392c] = "\xe3\xe0", [0x392d] = "\xe3\xe7", [0x392e] = "\xe3\xda",
  [0x3930] = "\xe3\xf3", [0x3931] = "\xe3\xeb", [0x3932] = "\xe3\xe5",
  [0x3933] = "\xe3\xd5", [0x3934] = "\xbb\x69", [0x3935] = "\xe3\xec",
  [0x3937] = "\xbb\x6c", [0x3938] = "\xe3\xf0", [0x393a] = "\xe3\xea",
  [0x393b] = "\xbb\x66", [0x393c] = "\xe3\xe8", [0x393e] = "\xe3\xe2",
  [0x393f] = "\xbb\x64", [0x3940] = "\xe3\xd9", [0x3941] = "\xe3\xe1",
  [0x3942] = "\xe3\xed", [0x3943] = "\xe3\xdf", [0x3946] = "\xe3\xe3",
  [0x394c] = "\xbd\xc1", [0x394d] = "\xdf\xe9", [0x394e] = "\xe7\xb2",
  [0x394f] = "\xe7\xbb", [0x3950] = "\xe7\xb1", [0x3951] = "\xe7\xad",
  [0x3952] = "\xe7\xaa", [0x3953] = "\xbd\xc2", [0x3954] = "\xe7\xa8",
  [0x3955] = "\xbb\x6b", [0x3956] = "\xe7\xa1", [0x3957] = "\xbd\xc0",
  [0x3958] = "\xe7\xa7", [0x3959] = "\xbd\xbf", [0x395a] = "\xe7\xac",
  [0x395b] = "\xe7\xa9", [0x395c] = "\xe7\xb9", [0x395d] = "\xe7\xb4",
  [0x395e] = "\xe7\xae", [0x395f] = "\xe7\xb3", [0x3960] = "\xbd\xbb",
  [0x3961] = "\xe7\xab", [0x3962] = "\xe7\xbe", [0x3963] = "\xe7\xa2",
  [0x3964] = "\xe7\xa3", [0x3965] = "\xe7\xba", [0x3966] = "\xbd\xbc",
  [0x3967] = "\xe7\xbf", [0x3968] = "\xbd\xbe", [0x3969] = "\xe7\xc0",
  [0x396a] = "\xe7\xb0", [0x396b] = "\xe3\xd8", [0x396c] = "\xe7\xb6",
  [0x396d] = "\xe7\xaf", [0x396e] = "\xe7\xb8", [0x396f] = "\xe7\xb5",
  [0x3973] = "\xe7\xa6", [0x3974] = "\xbd\xb9", [0x3975] = "\xe7\xbd",
  [0x3976] = "\xbd\xba", [0x3977] = "\xe7\xa4", [0x3978] = "\xbd\xbd",
  [0x3979] = "\xeb\x64", [0x397a] = "\xe7\xb7", [0x397b] = "\xe7\xbc",
  [0x3981] = "\xeb\x61", [0x3982] = "\xbd\xb8", [0x3983] = "\xbf\xc0",
  [0x3984] = "\xeb\x6b", [0x3985] = "\xeb\x67", [0x3987] = "\xeb\x65",
  [0x3988] = "\xeb\x60", [0x3989] = "\xeb\x6f", [0x398d] = "\xbf\xc4",
  [0x398f] = "\xeb\x5c", [0x3990] = "\xeb\x68", [0x3991] = "\xeb\x69",
  [0x3992] = "\xeb\x5f", [0x3993] = "\xeb\x5e", [0x3994] = "\xeb\x6c",
  [0x3996] = "\xeb\x62", [0x3997] = "\xeb\x5d", [0x3998] = "\xeb\x63",
  [0x399a] = "\xeb\x6e", [0x399b] = "\xeb\x5b", [0x399c] = "\xeb\x6d",
  [0x399d] = "\xeb\x6a", [0x399e] = "\xbf\xc2", [0x399f] = "\xbf\xc1",
  [0x39a2] = "\xbf\xc3", [0x39a3] = "\xeb\x66", [0x39a4] = "\xf0\xcb",
  [0x39aa] = "\xee\x59", [0x39ab] = "\xc1\xb1", [0x39ac] = "\xee\x5d",
  [0x39ad] = "\xee\x5a", [0x39ae] = "\xee\x61", [0x39af] = "\xee\x67",
  [0x39b0] = "\xee\x5c", [0x39b2] = "\xee\x70", [0x39b3] = "\xc1\xae",
  [0x39b4] = "\xee\x6a", [0x39b5] = "\xee\x5f", [0x39b6] = "\xee\x6b",
  [0x39b7] = "\xee\x66", [0x39b8] = "\xee\x6d", [0x39b9] = "\xee\x5e",
  [0x39ba] = "\xc1\xb3", [0x39bb] = "\xc1\xb2", [0x39bc] = "\xee\x60",
  [0x39bd] = "\xee\x6e", [0x39be] = "\xee\x58", [0x39bf] = "\xee\x6c",
  [0x39c0] = "\xc1\xac", [0x39c2] = "\xee\x64", [0x39c3] = "\xee\x63",
  [0x39c4] = "\xee\x68", [0x39c5] = "\xee\x5b", [0x39c6] = "\xc1\xb0",
  [0x39c8] = "\xc1\xb4", [0x39c9] = "\xee\x62", [0x39ca] = "\xee\x69",
  [0x39cb] = "\xc1\xb5", [0x39cc] = "\xee\x65", [0x39d1] = "\xc1\xad",
  [0x39d2] = "\xc1\xaf", [0x39d3] = "\xf0\xc7", [0x39d4] = "\xf0\xc5",
  [0x39d7] = "\xf0\xcc", [0x39d8] = "\xf0\xc9", [0x39d9] = "\xf0\xcd",
  [0x39db] = "\xf0\xbe", [0x39dc] = "\xf0\xc6", [0x39dd] = "\xf0\xd1",
  [0x39de] = "\xee\x6f", [0x39df] = "\xf0\xc2", [0x39e0] = "\xc2\xcf",
  [0x39e1] = "\xe7\xa5", [0x39e2] = "\xf0\xbd", [0x39e3] = "\xf0\xca",
  [0x39e4] = "\xf0\xc4", [0x39e5] = "\xf0\xc1", [0x39e6] = "\xf0\xbc",
  [0x39e7] = "\xf0\xbb", [0x39e8] = "\xf0\xd0", [0x39ea] = "\xf0\xc0",
  [0x39eb] = "\xf0\xbf", [0x39ec] = "\xc2\xcd", [0x39ed] = "\xf0\xc8",
  [0x39ef] = "\xc2\xcc", [0x39f2] = "\xc2\xce", [0x39f3] = "\xf0\xc3",
  [0x39f4] = "\xf0\xcf", [0x39f6] = "\xf2\xde", [0x39f7] = "\xf2\xdf",
  [0x39f9] = "\xc3\xc9", [0x39fa] = "\xf2\xdc", [0x39fb] = "\xc3\xc6",
  [0x39fc] = "\xf2\xe4", [0x39fe] = "\xc3\xca", [0x39ff] = "\xf2\xe6",
  [0x3a00] = "\xf2\xdb", [0x3a01] = "\xf0\xce", [0x3a02] = "\xf2\xe8",
  [0x3a03] = "\xf2\xdd", [0x3a05] = "\xc3\xc7", [0x3a06] = "\xf2\xe3",
  [0x3a08] = "\xf2\xe5", [0x3a09] = "\xf2\xe0", [0x3a0a] = "\xf2\xe7",
  [0x3a0b] = "\xf2\xe2", [0x3a0c] = "\xf2\xe1", [0x3a0d] = "\xc3\xc8",
  [0x3a10] = "\xf4\xc5", [0x3a11] = "\xf4\xc6", [0x3a13] = "\xf4\xc8",
  [0x3a14] = "\xc4\xae", [0x3a15] = "\xc4\xaf", [0x3a16] = "\xf4\xc9",
  [0x3a17] = "\xf4\xc7", [0x3a19] = "\xf4\xc4", [0x3a1b] = "\xf6\x42",
  [0x3a1c] = "\xf6\x45", [0x3a1d] = "\xf6\x41", [0x3a1f] = "\xc4\xfa",
  [0x3a20] = "\xf6\x43", [0x3a21] = "\xc4\xf9", [0x3a22] = "\xc4\xf8",
  [0x3a23] = "\xc4\xf7", [0x3a24] = "\xf6\x44", [0x3a25] = "\xf7\x51",
  [0x3a26] = "\xf7\x4f", [0x3a28] = "\xf7\x4e", [0x3a29] = "\xf6\x40",
  [0x3a2a] = "\xf7\x50", [0x3a2b] = "\xf6\x46", [0x3a2c] = "\xf7\x4d",
  [0x3a2e] = "\xf7\xf9", [0x3a2f] = "\xf7\xd7", [0x3a30] = "\xf7\xf7",
  [0x3a31] = "\xc5\xdb", [0x3a32] = "\xf7\xf8", [0x3a33] = "\xf7\xfa",
  [0x3a35] = "\xf8\xbf", [0x3a36] = "\xc5\xfa", [0x3a37] = "\xf8\xbe",
  [0x3a38] = "\xf8\xbd", [0x3a39] = "\xc5\xfb", [0x3a3b] = "\xc6\x5a",
  [0x3a3c] = "\xf9\x6e", [0x3a3d] = "\xf9\xa7", [0x3a3e] = "\xf9\xa6",
  [0x3a3f] = "\xf9\xa8", [0x3a40] = "\xa6\xe5", [0x3a41] = "\xd0\xaa",
  [0x3a43] = "\xd3\xcf", [0x3a44] = "\xd3\xd0", [0x3a48] = "\xdb\xc0",
  [0x3a4a] = "\xf6\x47", [0x3a4b] = "\xf8\xc0", [0x3a4c] = "\xa6\xe6",
  [0x3a4d] = "\xad\x6c", [0x3a4e] = "\xd0\xab", [0x3a52] = "\xd7\xb1",
  [0x3a53] = "\xb3\x4e", [0x3a55] = "\xdb\xc2", [0x3a56] = "\xdb\xc1",
  [0x3a57] = "\xb5\xf3", [0x3a59] = "\xb8\xc5", [0x3a5a] = "\xe7\xc1",
  [0x3a5b] = "\xbd\xc3", [0x3a5d] = "\xbd\xc4", [0x3a61] = "\xbf\xc5",
  [0x3a62] = "\xc5\xfc", [0x3a63] = "\xa6\xe7", [0x3a67] = "\xd0\xac",
  [0x3a68] = "\xaa\xed", [0x3a69] = "\xd0\xae", [0x3a6a] = "\xd0\xad",
  [0x3a6b] = "\xad\x6d", [0x3a6d] = "\xd3\xd1", [0x3a6f] = "\xd3\xd8",
  [0x3a70] = "\xb0\x49", [0x3a71] = "\xd3\xd6", [0x3a72] = "\xd3\xd4",
  [0x3a74] = "\xd3\xdb", [0x3a75] = "\xd3\xd2", [0x3a76] = "\xd3\xd3",
  [0x3a77] = "\xb0\x4a", [0x3a79] = "\xb0\x4e", [0x3a7c] = "\xd3\xdc",
  [0x3a7d] = "\xb0\x4d", [0x3a7e] = "\xd3\xda", [0x3a7f] = "\xd3\xd7",
  [0x3a80] = "\xd3\xd5", [0x3a81] = "\xb0\x4b", [0x3a82] = "\xb0\x4c",
  [0x3a83] = "\xd3\xd9", [0x3a88] = "\xb3\x50", [0x3a89] = "\xd7\xb2",
  [0x3a8b] = "\xb3\x55", [0x3a8c] = "\xd7\xc2", [0x3a8d] = "\xb3\x54",
  [0x3a8e] = "\xd7\xc4", [0x3a91] = "\xd7\xb8", [0x3a92] = "\xb3\x52",
  [0x3a93] = "\xd7\xc3", [0x3a95] = "\xd7\xb3", [0x3a96] = "\xb3\x53",
  [0x3a97] = "\xd7\xbf", [0x3a98] = "\xd7\xbb", [0x3a99] = "\xd7\xbd",
  [0x3a9a] = "\xd7\xb7", [0x3a9b] = "\xd7\xbe", [0x3a9e] = "\xb3\x4f",
  [0x3a9f] = "\xd7\xba", [0x3aa1] = "\xd7\xb9", [0x3aa2] = "\xd7\xb5",
  [0x3aa4] = "\xd7\xc0", [0x3aa7] = "\xd7\xbc", [0x3aa8] = "\xd7\xb4",
  [0x3aaa] = "\xd7\xb6", [0x3aab] = "\xb3\x51", [0x3aac] = "\xd7\xc1",
  [0x3ab1] = "\xb5\xf6", [0x3ab2] = "\xdb\xcd", [0x3ab6] = "\xdb\xc9",
  [0x3ab7] = "\xdb\xcb", [0x3ab8] = "\xdb\xc6", [0x3ab9] = "\xdb\xc5",
  [0x3aba] = "\xdb\xc3", [0x3abc] = "\xdb\xca", [0x3abd] = "\xdb\xcc",
  [0x3abe] = "\xdb\xc8", [0x3ac0] = "\xdb\xc7", [0x3ac1] = "\xb5\xf4",
  [0x3ac2] = "\xb5\xf5", [0x3ac9] = "\xdb\xcf", [0x3aca] = "\xb8\xcd",
  [0x3acb] = "\xdf\xf2", [0x3acc] = "\xdf\xf8", [0x3acd] = "\xdf\xf3",
  [0x3ace] = "\xdf\xf4", [0x3acf] = "\xf9\xd8", [0x3ad0] = "\xdf\xf9",
  [0x3ad2] = "\xb8\xcf", [0x3ad4] = "\xb8\xc7", [0x3ad5] = "\xb8\xce",
  [0x3ad6] = "\xdf\xf1", [0x3ad7] = "\xdb\xc4", [0x3ad8] = "\xb8\xca",
  [0x3ad9] = "\xb8\xc8", [0x3ada] = "\xdf\xf7", [0x3adb] = "\xdf\xf6",
  [0x3adc] = "\xb8\xc9", [0x3add] = "\xb8\xcb", [0x3ade] = "\xdf\xf5",
  [0x3adf] = "\xb8\xc6", [0x3ae1] = "\xb8\xcc", [0x3ae7] = "\xe3\xf6",
  [0x3ae8] = "\xbb\x74", [0x3aeb] = "\xe4\x42", [0x3aec] = "\xe4\x41",
  [0x3aee] = "\xe3\xfb", [0x3aef] = "\xbb\x76", [0x3af0] = "\xe4\x40",
  [0x3af1] = "\xe3\xf7", [0x3af2] = "\xe3\xf8", [0x3af3] = "\xbb\x6e",
  [0x3af4] = "\xbb\x70", [0x3af6] = "\xe3\xfd", [0x3af7] = "\xe3\xf5",
  [0x3af8] = "\xbb\x72", [0x3af9] = "\xbb\x71", [0x3afa] = "\xe3\xf9",
  [0x3afb] = "\xe3\xfe", [0x3afc] = "\xe3\xfc", [0x3afd] = "\xbb\x73",
  [0x3afe] = "\xe3\xfa", [0x3b01] = "\xdb\xce", [0x3b02] = "\xbb\x6f",
  [0x3b05] = "\xe7\xc2", [0x3b06] = "\xe7\xc9", [0x3b07] = "\xbd\xc6",
  [0x3b09] = "\xe7\xcd", [0x3b0a] = "\xbd\xca", [0x3b0b] = "\xe7\xc5",
  [0x3b0c] = "\xe7\xc3", [0x3b0e] = "\xe7\xcc", [0x3b10] = "\xbd\xc5",
  [0x3b11] = "\xe7\xcb", [0x3b12] = "\xbd\xc7", [0x3b13] = "\xbd\xc8",
  [0x3b14] = "\xe7\xc4", [0x3b15] = "\xbd\xc9", [0x3b16] = "\xe7\xca",
  [0x3b17] = "\xe7\xc6", [0x3b18] = "\xe7\xc7", [0x3b19] = "\xe7\xc8",
  [0x3b1a] = "\xbb\x75", [0x3b1e] = "\xeb\x70", [0x3b1f] = "\xeb\x7c",
  [0x3b21] = "\xbf\xca", [0x3b22] = "\xeb\x77", [0x3b23] = "\xeb\x79",
  [0x3b25] = "\xbf\xc8", [0x3b26] = "\xeb\x71", [0x3b27] = "\xeb\x75",
  [0x3b29] = "\xeb\x78", [0x3b2a] = "\xbf\xc6", [0x3b2b] = "\xbf\xc9",
  [0x3b2c] = "\xeb\x7b", [0x3b2d] = "\xeb\x73", [0x3b2e] = "\xeb\x74",
  [0x3b2f] = "\xeb\x7a", [0x3b30] = "\xeb\x72", [0x3b31] = "\xeb\x76",
  [0x3b32] = "\xbf\xc7", [0x3b33] = "\xee\x72", [0x3b35] = "\xee\x71",
  [0x3b36] = "\xc1\xb7", [0x3b37] = "\xee\x77", [0x3b38] = "\xc1\xb9",
  [0x3b3b] = "\xc1\xb6", [0x3b3c] = "\xee\x73", [0x3b3d] = "\xc1\xba",
  [0x3b3e] = "\xee\x74", [0x3b41] = "\xee\x75", [0x3b42] = "\xee\x78",
  [0x3b44] = "\xc1\xb8", [0x3b46] = "\xf0\xd6", [0x3b49] = "\xf0\xd9",
  [0x3b4b] = "\xf0\xd3", [0x3b4c] = "\xf0\xd5", [0x3b4f] = "\xf0\xd4",
  [0x3b50] = "\xf0\xd7", [0x3b51] = "\xf0\xd8", [0x3b52] = "\xee\x76",
  [0x3b53] = "\xf0\xd2", [0x3b56] = "\xc3\xcd", [0x3b57] = "\xf2\xec",
  [0x3b58] = "\xf2\xef", [0x3b59] = "\xf2\xf1", [0x3b5a] = "\xf2\xea",
  [0x3b5b] = "\xf2\xeb", [0x3b5c] = "\xf2\xee", [0x3b5d] = "\xf2\xf0",
  [0x3b5e] = "\xc3\xce", [0x3b5f] = "\xc3\xcc", [0x3b60] = "\xc3\xcb",
  [0x3b61] = "\xf2\xed", [0x3b62] = "\xf2\xe9", [0x3b63] = "\xf4\xca",
  [0x3b64] = "\xc4\xb0", [0x3b66] = "\xf4\xcb", [0x3b69] = "\xf6\x49",
  [0x3b6a] = "\xc4\xfb", [0x3b6b] = "\xf6\x4b", [0x3b6c] = "\xc4\xfc",
  [0x3b6d] = "\xf6\x48", [0x3b6e] = "\xf6\x4a", [0x3b6f] = "\xc5\xa8",
  [0x3b71] = "\xf7\x52", [0x3b72] = "\xc5\xa7", [0x3b73] = "\xf7\xfd",
  [0x3b74] = "\xf7\xfc", [0x3b76] = "\xf7\xfb", [0x3b79] = "\xf9\x48",
  [0x3b7a] = "\xf9\x49", [0x3b7b] = "\xf9\x4b", [0x3b7c] = "\xf9\x4a",
  [0x3b7e] = "\xca\x50", [0x3b7f] = "\xa6\xe8", [0x3b81] = "\xad\x6e",
  [0x3b82] = "\xd7\xc5", [0x3b83] = "\xb5\xf7", [0x3b85] = "\xdf\xfa",
  [0x3b86] = "\xc2\xd0", [0x3b88] = "\xf2\xf2", [0x3b8b] = "\xa8\xa3",
  [0x3b8f] = "\xb3\x57", [0x3b93] = "\xb3\x56", [0x3b95] = "\xdb\xd0",
  [0x3b96] = "\xb5\xf8", [0x3b97] = "\xdb\xd2", [0x3b98] = "\xdb\xd1",
  [0x3b9b] = "\xdf\xfb", [0x3b9c] = "\xb8\xd0", [0x3b9d] = "\xe4\x43",
  [0x3b9e] = "\xe4\x46", [0x3b9f] = "\xe4\x45", [0x3ba1] = "\xe4\x44",
  [0x3ba2] = "\xe7\xce", [0x3ba3] = "\xe7\xd0", [0x3ba4] = "\xe7\xcf",
  [0x3ba6] = "\xbf\xcc", [0x3baa] = "\xbf\xcb", [0x3bac] = "\xc1\xbb",
  [0x3bad] = "\xee\x79", [0x3bae] = "\xee\x7b", [0x3baf] = "\xee\x7a",
  [0x3bb2] = "\xc2\xd1", [0x3bb6] = "\xf2\xf4", [0x3bb7] = "\xf2\xf3",
  [0x3bb9] = "\xf4\xcc", [0x3bba] = "\xc4\xb1", [0x3bbd] = "\xc4\xfd",
  [0x3bbe] = "\xf7\x54", [0x3bbf] = "\xf7\x53", [0x3bc0] = "\xc6\x5b",
  [0x3bd2] = "\xa8\xa4", [0x3bd3] = "\xd0\xaf", [0x3bd4] = "\xad\x6f",
  [0x3bd5] = "\xd7\xc8", [0x3bd6] = "\xd7\xc6", [0x3bd9] = "\xd7\xc7",
  [0x3bda] = "\xdb\xd4", [0x3bdb] = "\xdb\xd5", [0x3bdc] = "\xe0\x43",
  [0x3bdd] = "\xdb\xd3", [0x3bdf] = "\xdf\xfc", [0x3be0] = "\xe0\x41",
  [0x3be1] = "\xe0\x40", [0x3be2] = "\xe0\x42", [0x3be3] = "\xb8\xd1",
  [0x3be4] = "\xdf\xfe", [0x3be5] = "\xdf\xfd", [0x3be6] = "\xe0\x44",
  [0x3be8] = "\xe4\x49", [0x3be9] = "\xe4\x47", [0x3beb] = "\xe4\x48",
  [0x3bec] = "\xe7\xd3", [0x3bed] = "\xe7\xd1", [0x3bf0] = "\xe7\xd2",
  [0x3bf1] = "\xeb\x7d", [0x3bf2] = "\xee\x7c", [0x3bf3] = "\xee\x7d",
  [0x3bf4] = "\xc2\xd2", [0x3bf6] = "\xf2\xf5", [0x3bf7] = "\xf4\xcd",
  [0x3bf8] = "\xc4\xb2", [0x3bfa] = "\xf6\x4c", [0x3bfb] = "\xf7\x55",
  [0x3bfc] = "\xc5\xa9", [0x3bfe] = "\xf7\xfe", [0x3bff] = "\xf9\x4c",
  [0x3c00] = "\xa8\xa5", [0x3c02] = "\xad\x71", [0x3c03] = "\xad\x72",
  [0x3c04] = "\xd0\xb0", [0x3c07] = "\xd0\xb1", [0x3c08] = "\xad\x70",
  [0x3c0a] = "\xb0\x54", [0x3c0c] = "\xb0\x52", [0x3c0e] = "\xb0\x51",
  [0x3c0f] = "\xb0\x58", [0x3c10] = "\xb0\x50", [0x3c11] = "\xb0\x59",
  [0x3c12] = "\xd3\xdd", [0x3c13] = "\xb0\x56", [0x3c15] = "\xb0\x53",
  [0x3c16] = "\xb0\x57", [0x3c17] = "\xb0\x55", [0x3c18] = "\xb0\x4f",
  [0x3c1b] = "\xb3\x5f", [0x3c1d] = "\xb3\x59", [0x3c1e] = "\xd7\xcc",
  [0x3c1f] = "\xb3\x5e", [0x3c22] = "\xb3\x60", [0x3c23] = "\xb3\x5a",
  [0x3c25] = "\xb3\x5b", [0x3c27] = "\xd7\xca", [0x3c2a] = "\xb3\x58",
  [0x3c2c] = "\xd7\xcb", [0x3c2d] = "\xb3\x5d", [0x3c30] = "\xd7\xc9",
  [0x3c31] = "\xb3\x5c", [0x3c34] = "\xb6\x44", [0x3c36] = "\xb6\x46",
  [0x3c39] = "\xdb\xd8", [0x3c3a] = "\xb6\x45", [0x3c3b] = "\xb5\xf9",
  [0x3c3c] = "\xb5\xfd", [0x3c3e] = "\xb8\xe4", [0x3c3f] = "\xe0\x49",
  [0x3c40] = "\xdb\xda", [0x3c41] = "\xb5\xfe", [0x3c44] = "\xdb\xdd",
  [0x3c45] = "\xdb\xde", [0x3c46] = "\xb6\x43", [0x3c48] = "\xdb\xe0",
  [0x3c4a] = "\xdb\xe2", [0x3c4c] = "\xdb\xe3", [0x3c4d] = "\xdb\xd7",
  [0x3c4e] = "\xdb\xd6", [0x3c4f] = "\xdb\xe4", [0x3c50] = "\xb6\x42",
  [0x3c51] = "\xdb\xe1", [0x3c52] = "\xdb\xdf", [0x3c54] = "\xb6\x40",
  [0x3c55] = "\xb5\xfb", [0x3c56] = "\xb6\x47", [0x3c57] = "\xdb\xdb",
  [0x3c58] = "\xdb\xdc", [0x3c59] = "\xdb\xd9", [0x3c5b] = "\xb6\x41",
  [0x3c5e] = "\xb5\xfc", [0x3c60] = "\xb5\xfa", [0x3c61] = "\xe0\x48",
  [0x3c62] = "\xb8\xdf", [0x3c63] = "\xb8\xda", [0x3c66] = "\xb8\xd5",
  [0x3c68] = "\xb8\xe5", [0x3c69] = "\xb8\xd6", [0x3c6b] = "\xb8\xd2",
  [0x3c6c] = "\xb8\xe1", [0x3c6d] = "\xb8\xde", [0x3c6e] = "\xb8\xe0",
  [0x3c70] = "\xb8\xd7", [0x3c71] = "\xb8\xdc", [0x3c72] = "\xb8\xd3",
  [0x3c73] = "\xb8\xd4", [0x3c74] = "\xe0\x50", [0x3c75] = "\xe0\x4d",
  [0x3c76] = "\xe0\x45", [0x3c77] = "\xe0\x4a", [0x3c79] = "\xb8\xe2",
  [0x3c7a] = "\xe0\x51", [0x3c7b] = "\xb8\xe3", [0x3c7c] = "\xb8\xd9",
  [0x3c7f] = "\xe0\x47", [0x3c81] = "\xe0\x4f", [0x3c82] = "\xe0\x4b",
  [0x3c83] = "\xe0\x4e", [0x3c84] = "\xe0\x4c", [0x3c85] = "\xb8\xdd",
  [0x3c86] = "\xe0\x46", [0x3c87] = "\xb8\xd8", [0x3c8b] = "\xe4\x4c",
  [0x3c8c] = "\xbb\x78", [0x3c8d] = "\xbb\x7b", [0x3c8f] = "\xe4\x4e",
  [0x3c91] = "\xbb\xa5", [0x3c92] = "\xe4\x4d", [0x3c93] = "\xbb\x7d",
  [0x3c95] = "\xbd\xcf", [0x3c96] = "\xe4\x4f", [0x3c98] = "\xbb\xa4",
  [0x3c99] = "\xe4\x4b", [0x3c9a] = "\xbb\xa6", [0x3c9e] = "\xbb\x79",
  [0x3ca0] = "\xb8\xdb", [0x3ca1] = "\xbb\x7c", [0x3ca3] = "\xbb\x7a",
  [0x3ca4] = "\xbb\x7e", [0x3ca5] = "\xbb\xa2", [0x3ca6] = "\xbb\x77",
  [0x3ca7] = "\xbb\xa7", [0x3ca8] = "\xbb\xa3", [0x3caa] = "\xbb\xa1",
  [0x3cab] = "\xe4\x4a", [0x3cb0] = "\xbd\xd6", [0x3cb2] = "\xbd\xd2",
  [0x3cb6] = "\xbd\xd9", [0x3cb8] = "\xe7\xd6", [0x3cb9] = "\xbd\xda",
  [0x3cba] = "\xe7\xe2", [0x3cbb] = "\xe7\xdb", [0x3cbc] = "\xbd\xcb",
  [0x3cbd] = "\xe7\xe3", [0x3cbe] = "\xe7\xdd", [0x3cbf] = "\xbd\xd5",
  [0x3cc0] = "\xe7\xde", [0x3cc2] = "\xbd\xd4", [0x3cc3] = "\xe7\xe1",
  [0x3cc4] = "\xbd\xce", [0x3cc5] = "\xe7\xdf", [0x3cc6] = "\xe7\xd5",
  [0x3cc7] = "\xbd\xcd", [0x3cc8] = "\xeb\xaa", [0x3cc9] = "\xbd\xd3",
  [0x3ccb] = "\xbd\xd0", [0x3ccd] = "\xbd\xd8", [0x3ccf] = "\xe7\xd4",
  [0x3cd1] = "\xe7\xd8", [0x3cd2] = "\xbd\xcc", [0x3cd3] = "\xe7\xd7",
  [0x3cd4] = "\xe7\xd9", [0x3cd5] = "\xe7\xda", [0x3cd6] = "\xbd\xd7",
  [0x3cd7] = "\xe7\xdc", [0x3cd8] = "\xe7\xe0", [0x3cd9] = "\xe7\xe4",
  [0x3cdb] = "\xbd\xdb", [0x3cdc] = "\xbf\xd2", [0x3cdd] = "\xeb\xa5",
  [0x3cde] = "\xeb\xab", [0x3cdf] = "\xeb\xa8", [0x3ce0] = "\xeb\x7e",
  [0x3ce1] = "\xeb\xac", [0x3ce2] = "\xeb\xa1", [0x3ce4] = "\xeb\xa7",
  [0x3ce6] = "\xbf\xcd", [0x3ce7] = "\xbf\xd3", [0x3ce8] = "\xeb\xad",
  [0x3ceb] = "\xbf\xcf", [0x3ced] = "\xbf\xd9", [0x3cee] = "\xbf\xd4",
  [0x3cef] = "\xeb\xaf", [0x3cf0] = "\xeb\xa9", [0x3cf1] = "\xbf\xd0",
  [0x3cf2] = "\xeb\xa2", [0x3cf3] = "\xbf\xda", [0x3cf4] = "\xeb\xa3",
  [0x3cf5] = "\xeb\xa4", [0x3cf6] = "\xbf\xdb", [0x3cf7] = "\xbf\xd8",
  [0x3cf8] = "\xbd\xd1", [0x3cfa] = "\xbf\xce", [0x3cfb] = "\xeb\xb0",
  [0x3cfc] = "\xbf\xdc", [0x3cfe] = "\xbf\xd5", [0x3cff] = "\xeb\xae",
  [0x3d00] = "\xbf\xd1", [0x3d01] = "\xbf\xd6", [0x3d02] = "\xbf\xd7",
  [0x3d04] = "\xc1\xc3", [0x3d05] = "\xee\xa4", [0x3d06] = "\xee\xad",
  [0x3d07] = "\xee\xaa", [0x3d08] = "\xee\xac", [0x3d0a] = "\xc1\xc0",
  [0x3d0b] = "\xee\xa5", [0x3d0d] = "\xee\xab", [0x3d0e] = "\xc1\xbc",
  [0x3d0f] = "\xee\xa7", [0x3d10] = "\xc1\xc4", [0x3d11] = "\xee\xa3",
  [0x3d12] = "\xee\xa8", [0x3d13] = "\xee\xaf", [0x3d14] = "\xeb\xa6",
  [0x3d15] = "\xee\xa9", [0x3d16] = "\xee\xa2", [0x3d17] = "\xc1\xbd",
  [0x3d18] = "\xee\xa1", [0x3d19] = "\xc1\xbe", [0x3d1a] = "\xee\xb0",
  [0x3d1b] = "\xc1\xbf", [0x3d1c] = "\xee\xae", [0x3d1d] = "\xc1\xc2",
  [0x3d1e] = "\xee\x7e", [0x3d20] = "\xc1\xc1", [0x3d22] = "\xee\xa6",
  [0x3d23] = "\xf0\xdc", [0x3d24] = "\xf0\xea", [0x3d25] = "\xf0\xe5",
  [0x3d26] = "\xf0\xe7", [0x3d27] = "\xf0\xdb", [0x3d28] = "\xc2\xd3",
  [0x3d2a] = "\xf0\xda", [0x3d2b] = "\xc2\xd6", [0x3d2c] = "\xc2\xd5",
  [0x3d2e] = "\xf0\xe9", [0x3d2f] = "\xf0\xe1", [0x3d30] = "\xf0\xde",
  [0x3d31] = "\xf0\xe4", [0x3d33] = "\xf0\xdd", [0x3d35] = "\xf0\xdf",
  [0x3d36] = "\xf0\xe8", [0x3d37] = "\xf0\xe6", [0x3d39] = "\xc2\xd4",
  [0x3d3a] = "\xf0\xed", [0x3d3b] = "\xf0\xeb", [0x3d3c] = "\xf0\xe2",
  [0x3d3d] = "\xf0\xec", [0x3d3e] = "\xf0\xe3", [0x3d40] = "\xf2\xf9",
  [0x3d41] = "\xc3\xcf", [0x3d42] = "\xf3\x41", [0x3d45] = "\xf6\x4f",
  [0x3d46] = "\xc3\xd6", [0x3d47] = "\xf0\xe0", [0x3d48] = "\xf2\xf7",
  [0x3d49] = "\xc3\xd2", [0x3d4a] = "\xf2\xf8", [0x3d4b] = "\xf2\xfd",
  [0x3d4e] = "\xc3\xd4", [0x3d4f] = "\xc3\xd5", [0x3d50] = "\xf2\xf6",
  [0x3d51] = "\xf3\x40", [0x3d52] = "\xf3\x42", [0x3d53] = "\xf2\xfa",
  [0x3d54] = "\xf2\xfc", [0x3d55] = "\xf2\xfe", [0x3d56] = "\xf2\xfb",
  [0x3d57] = "\xf3\x43", [0x3d58] = "\xc3\xd1", [0x3d59] = "\xc3\xd7",
  [0x3d5a] = "\xc3\xd3", [0x3d5c] = "\xc3\xd0", [0x3d5d] = "\xf4\xd0",
  [0x3d5f] = "\xc4\xb7", [0x3d60] = "\xf4\xce", [0x3d63] = "\xf4\xd2",
  [0x3d65] = "\xf4\xd3", [0x3d66] = "\xc4\xb5", [0x3d67] = "\xf4\xd4",
  [0x3d68] = "\xf4\xd1", [0x3d6a] = "\xf4\xcf", [0x3d6b] = "\xc4\xb8",
  [0x3d6c] = "\xc4\xb4", [0x3d6d] = "\xf4\xd5", [0x3d6f] = "\xc4\xb6",
  [0x3d70] = "\xc4\xb3", [0x3d74] = "\xc4\xfe", [0x3d77] = "\xc5\x40",
  [0x3d78] = "\xf6\x4e", [0x3d79] = "\xf6\x4d", [0x3d7a] = "\xf6\x50",
  [0x3d7b] = "\xf6\x51", [0x3d7d] = "\xc5\x41", [0x3d7e] = "\xf7\x56",
  [0x3d7f] = "\xf7\x5b", [0x3d80] = "\xc5\xaa", [0x3d82] = "\xf7\x58",
  [0x3d84] = "\xf7\x57", [0x3d85] = "\xf7\x5a", [0x3d86] = "\xf7\x59",
  [0x3d88] = "\xf8\x43", [0x3d8a] = "\xc5\xdc", [0x3d8b] = "\xf8\x42",
  [0x3d8c] = "\xf8\x40", [0x3d8e] = "\xf8\x41", [0x3d92] = "\xc5\xfe",
  [0x3d93] = "\xc5\xfd", [0x3d94] = "\xf8\xc1", [0x3d95] = "\xf8\xc2",
  [0x3d96] = "\xc6\x40", [0x3d98] = "\xf9\x4d", [0x3d99] = "\xf9\x4e",
  [0x3d9a] = "\xc6\x67", [0x3d9c] = "\xc6\x6d", [0x3d9e] = "\xf9\xa9",
  [0x3d9f] = "\xf9\xc8", [0x3e37] = "\xa8\xa6", [0x3e39] = "\xd7\xcd",
  [0x3e3b] = "\xd7\xce", [0x3e3c] = "\xe0\x52", [0x3e3d] = "\xe4\x50",
  [0x3e3e] = "\xe7\xe5", [0x3e3f] = "\xc1\xc6", [0x3e41] = "\xc1\xc5",
  [0x3e42] = "\xf0\xee", [0x3e43] = "\xf3\x44", [0x3e45] = "\xf8\x44",
  [0x3e46] = "\xa8\xa7", [0x3e47] = "\xd3\xde", [0x3e48] = "\xb0\x5a",
  [0x3e49] = "\xb3\x61", [0x3e4a] = "\xe0\x54", [0x3e4b] = "\xe0\x53",
  [0x3e4c] = "\xbd\xdc", [0x3e4d] = "\xe7\xe6", [0x3e4e] = "\xbd\xdd",
  [0x3e4f] = "\xee\xb1", [0x3e50] = "\xc2\xd7", [0x3e54] = "\xc6\x76",
  [0x3e55] = "\xa8\xa8", [0x3e56] = "\xcd\xcb", [0x3e57] = "\xd3\xdf",
  [0x3e5a] = "\xb3\x62", [0x3e5c] = "\xd7\xcf", [0x3e5d] = "\xd7\xd0",
  [0x3e5f] = "\xdb\xe5", [0x3e61] = "\xb6\x48", [0x3e62] = "\xb8\xe6",
  [0x3e64] = "\xe0\x56", [0x3e65] = "\xe0\x55", [0x3e66] = "\xe0\x57",
  [0x3e68] = "\xe4\x51", [0x3e69] = "\xe4\x52", [0x3e6a] = "\xbb\xa8",
  [0x3e6b] = "\xbf\xdd", [0x3e6c] = "\xbd\xde", [0x3e6d] = "\xbf\xde",
  [0x3e6f] = "\xee\xb5", [0x3e70] = "\xee\xb2", [0x3e71] = "\xee\xb4",
  [0x3e72] = "\xee\xb3", [0x3e73] = "\xc1\xc7", [0x3e75] = "\xf0\xef",
  [0x3e76] = "\xf3\x46", [0x3e77] = "\xf3\x45", [0x3e78] = "\xcb\xa4",
  [0x3e79] = "\xb0\x5c", [0x3e7a] = "\xb0\x5b", [0x3e7b] = "\xd3\xe0",
  [0x3e7d] = "\xd7\xd1", [0x3e80] = "\xdb\xe7", [0x3e81] = "\xdb\xe6",
  [0x3e82] = "\xb6\x49", [0x3e84] = "\xe0\x59", [0x3e85] = "\xe0\x5a",
  [0x3e86] = "\xe0\x58", [0x3e89] = "\xb8\xe8", [0x3e8a] = "\xb8\xe7",
  [0x3e8c] = "\xbb\xaa", [0x3e8d] = "\xbb\xa9", [0x3e8f] = "\xe7\xe7",
  [0x3e90] = "\xeb\xb3", [0x3e91] = "\xeb\xb1", [0x3e92] = "\xeb\xb2",
  [0x3e93] = "\xbf\xdf", [0x3e94] = "\xee\xb7", [0x3e95] = "\xee\xb6",
  [0x3e97] = "\xf0\xf2", [0x3e98] = "\xf0\xf1", [0x3e99] = "\xf0\xf0",
  [0x3e9a] = "\xf3\x47", [0x3e9c] = "\xf9\xaa", [0x3e9d] = "\xa8\xa9",
  [0x3e9e] = "\xad\x73", [0x3ea0] = "\xad\x74", [0x3ea1] = "\xb0\x5d",
  [0x3ea2] = "\xb0\x5e", [0x3ea3] = "\xd3\xe2", [0x3ea4] = "\xd3\xe1",
  [0x3ea5] = "\xd7\xd2", [0x3ea7] = "\xb3\x68", [0x3ea8] = "\xb3\x66",
  [0x3ea9] = "\xb3\x63", [0x3eaa] = "\xb3\x67", [0x3eab] = "\xb3\x65",
  [0x3eac] = "\xb3\x64", [0x3eaf] = "\xb6\x4a", [0x3eb0] = "\xdb\xea",
  [0x3eb2] = "\xb8\xed", [0x3eb3] = "\xb6\x4c", [0x3eb4] = "\xb6\x51",
  [0x3eb5] = "\xdb\xec", [0x3eb6] = "\xb6\x53", [0x3eb7] = "\xb6\x52",
  [0x3eb8] = "\xb6\x55", [0x3eb9] = "\xdb\xeb", [0x3eba] = "\xdb\xe8",
  [0x3ebb] = "\xb6\x4f", [0x3ebc] = "\xb6\x4b", [0x3ebd] = "\xb6\x4d",
  [0x3ebe] = "\xdb\xe9", [0x3ebf] = "\xb6\x54", [0x3ec0] = "\xb6\x50",
  [0x3ec1] = "\xb6\x4e", [0x3ec2] = "\xb8\xef", [0x3ec3] = "\xb8\xee",
  [0x3ec4] = "\xb8\xec", [0x3ec5] = "\xb8\xf0", [0x3ec7] = "\xb8\xea",
  [0x3ec8] = "\xb8\xeb", [0x3eca] = "\xb8\xe9", [0x3ecc] = "\xe0\x5b",
  [0x3ecf] = "\xe4\x54", [0x3ed1] = "\xbb\xac", [0x3ed2] = "\xbb\xad",
  [0x3ed3] = "\xbb\xab", [0x3ed5] = "\xe4\x53", [0x3ed7] = "\xe4\x55",
  [0x3ed9] = "\xe7\xea", [0x3eda] = "\xe7\xec", [0x3edc] = "\xbd\xe7",
  [0x3edd] = "\xe7\xed", [0x3ede] = "\xbd\xe0", [0x3edf] = "\xe7\xe9",
  [0x3ee0] = "\xbd\xdf", [0x3ee1] = "\xbd\xe9", [0x3ee2] = "\xbd\xe5",
  [0x3ee3] = "\xbd\xe6", [0x3ee4] = "\xbd\xe2", [0x3ee5] = "\xe7\xe8",
  [0x3ee6] = "\xbd\xe1", [0x3ee7] = "\xe7\xee", [0x3ee8] = "\xe7\xeb",
  [0x3eea] = "\xbd\xe8", [0x3eec] = "\xbd\xe3", [0x3eed] = "\xbd\xe4",
  [0x3eee] = "\xeb\xb5", [0x3ef0] = "\xeb\xb7", [0x3ef1] = "\xeb\xb6",
  [0x3ef3] = "\xeb\xb8", [0x3ef4] = "\xbf\xe0", [0x3ef5] = "\xeb\xb4",
  [0x3ef8] = "\xc1\xcb", [0x3ef9] = "\xee\xb8", [0x3efa] = "\xc1\xc8",
  [0x3efb] = "\xc1\xcc", [0x3efc] = "\xc1\xca", [0x3efd] = "\xc1\xc9",
  [0x3efe] = "\xf0\xf3", [0x3f00] = "\xf0\xf6", [0x3f02] = "\xf0\xf5",
  [0x3f04] = "\xf0\xf4", [0x3f05] = "\xc2\xd8", [0x3f06] = "\xf3\x48",
  [0x3f07] = "\xf3\x49", [0x3f08] = "\xc3\xd8", [0x3f09] = "\xf3\x4a",
  [0x3f0a] = "\xc3\xd9", [0x3f0d] = "\xc4\xba", [0x3f0f] = "\xc4\xb9",
  [0x3f10] = "\xf6\x52", [0x3f13] = "\xc5\x42", [0x3f14] = "\xf6\x53",
  [0x3f15] = "\xf7\x5c", [0x3f16] = "\xc5\xab", [0x3f17] = "\xc5\xac",
  [0x3f19] = "\xf8\x45", [0x3f1b] = "\xc6\x42", [0x3f64] = "\xa8\xaa",
  [0x3f66] = "\xb3\x6a", [0x3f67] = "\xb3\x69", [0x3f68] = "\xe0\x5c",
  [0x3f69] = "\xe0\x5d", [0x3f6b] = "\xbb\xae", [0x3f6c] = "\xeb\xb9",
  [0x3f6d] = "\xbd\xea", [0x3f6e] = "\xeb\xba", [0x3f6f] = "\xee\xb9",
  [0x3f70] = "\xa8\xab", [0x3f72] = "\xd0\xb2", [0x3f73] = "\xad\x76",
  [0x3f74] = "\xad\x75", [0x3f76] = "\xd3\xe3", [0x3f77] = "\xb0\x5f",
  [0x3f78] = "\xd3\xe4", [0x3f79] = "\xd7\xd5", [0x3f7b] = "\xd7\xd4",
  [0x3f7d] = "\xd7\xd3", [0x3f80] = "\xdb\xee", [0x3f81] = "\xb6\x58",
  [0x3f84] = "\xdb\xed", [0x3f85] = "\xb6\x57", [0x3f89] = "\xdb\xef",
  [0x3f8a] = "\xb6\x56", [0x3f8c] = "\xe0\x5f", [0x3f8d] = "\xe0\x62",
  [0x3f8e] = "\xe0\x60", [0x3f8f] = "\xe0\x61", [0x3f90] = "\xe0\x65",
  [0x3f91] = "\xe0\x5e", [0x3f92] = "\xe0\x66", [0x3f93] = "\xe0\x63",
  [0x3f94] = "\xe0\x64", [0x3f95] = "\xbb\xb0", [0x3f96] = "\xe4\x56",
  [0x3f99] = "\xbb\xaf", [0x3f9b] = "\xe7\xf2", [0x3f9c] = "\xe7\xf0",
  [0x3f9f] = "\xbd\xeb", [0x3fa0] = "\xe7\xef", [0x3fa1] = "\xe7\xf1",
  [0x3fa3] = "\xbd\xec", [0x3fa5] = "\xeb\xbb", [0x3fa7] = "\xeb\xbc",
  [0x3fa8] = "\xc1\xcd", [0x3faa] = "\xf3\x4c", [0x3fab] = "\xf3\x4e",
  [0x3fac] = "\xf3\x4b", [0x3fad] = "\xf3\x4d", [0x3fae] = "\xf4\xd6",
  [0x3faf] = "\xf6\x54", [0x3fb2] = "\xf9\x6f", [0x3fb3] = "\xa8\xac",
  [0x3fb4] = "\xad\x77", [0x3fb5] = "\xd3\xe5", [0x3fb6] = "\xd3\xe7",
  [0x3fb7] = "\xd3\xe6", [0x3fb9] = "\xd7\xd8", [0x3fba] = "\xb3\x6c",
  [0x3fbc] = "\xd7\xd6", [0x3fbe] = "\xb3\x6b", [0x3fbf] = "\xd7\xd9",
  [0x3fc1] = "\xd7\xda", [0x3fc2] = "\xd7\xd7", [0x3fc5] = "\xdb\xfb",
  [0x3fc6] = "\xb6\x60", [0x3fc7] = "\xdb\xf3", [0x3fc8] = "\xdb\xf9",
  [0x3fcb] = "\xb6\x5b", [0x3fcc] = "\xb6\x5e", [0x3fcd] = "\xdb\xf2",
  [0x3fce] = "\xb6\x59", [0x3fcf] = "\xdb\xf6", [0x3fd0] = "\xe0\x6c",
  [0x3fd1] = "\xb6\x5d", [0x3fd3] = "\xdb\xf1", [0x3fd5] = "\xdb\xf7",
  [0x3fd6] = "\xdb\xf4", [0x3fd7] = "\xdb\xfa", [0x3fd8] = "\xdb\xf0",
  [0x3fd9] = "\xdb\xf8", [0x3fda] = "\xb6\x5c", [0x3fdb] = "\xb6\x5f",
  [0x3fdc] = "\xdb\xf5", [0x3fdd] = "\xb6\x5a", [0x3fdf] = "\xb8\xf2",
  [0x3fe0] = "\xe0\x68", [0x3fe1] = "\xb8\xf1", [0x3fe2] = "\xe0\x6f",
  [0x3fe3] = "\xe0\x6e", [0x3fe4] = "\xb8\xf8", [0x3fe6] = "\xb8\xf9",
  [0x3fe7] = "\xe0\x70", [0x3fe8] = "\xb8\xf3", [0x3fe9] = "\xe0\x6d",
  [0x3fea] = "\xb8\xf7", [0x3feb] = "\xe0\x72", [0x3fec] = "\xe0\x69",
  [0x3fee] = "\xe0\x6b", [0x3fef] = "\xb8\xf4", [0x3ff0] = "\xe0\x67",
  [0x3ff1] = "\xe0\x6a", [0x3ff2] = "\xe0\x71", [0x3ff3] = "\xb8\xf5",
  [0x3ff4] = "\xe0\x73", [0x3ffa] = "\xb8\xf6", [0x3ffc] = "\xbb\xb1",
  [0x3ffd] = "\xe4\x5b", [0x3ffe] = "\xe4\x61", [0x3fff] = "\xe4\x59",
  [0x4000] = "\xe4\x62", [0x4002] = "\xe4\x58", [0x4003] = "\xe4\x5d",
  [0x4004] = "\xe4\x63", [0x4005] = "\xe4\x60", [0x4006] = "\xe4\x5f",
  [0x4007] = "\xe4\x5e", [0x4009] = "\xe4\x57", [0x400a] = "\xe4\x5c",
  [0x400d] = "\xe4\x5a", [0x400f] = "\xbd\xf1", [0x4010] = "\xbd\xee",
  [0x4011] = "\xe7\xfb", [0x4012] = "\xe8\x41", [0x4013] = "\xe8\x43",
  [0x4014] = "\xe8\x40", [0x4015] = "\xe7\xf8", [0x4016] = "\xe7\xfa",
  [0x4017] = "\xe8\x45", [0x4018] = "\xe8\x42", [0x4019] = "\xe7\xfc",
  [0x401a] = "\xe8\x46", [0x401b] = "\xe7\xf9", [0x401c] = "\xe8\x44",
  [0x401d] = "\xbd\xef", [0x401e] = "\xbd\xf5", [0x401f] = "\xbd\xf3",
  [0x4020] = "\xe7\xf3", [0x4021] = "\xbd\xf4", [0x4022] = "\xbd\xf0",
  [0x4023] = "\xe7\xf4", [0x4024] = "\xe7\xf6", [0x4025] = "\xe7\xf5",
  [0x4026] = "\xe7\xfd", [0x4027] = "\xe7\xfe", [0x4029] = "\xbd\xf2",
  [0x402b] = "\xbd\xed", [0x402e] = "\xe7\xf7", [0x4030] = "\xeb\xc6",
  [0x4031] = "\xbf\xe2", [0x4033] = "\xeb\xbd", [0x4034] = "\xbf\xe3",
  [0x4035] = "\xbf\xe6", [0x4036] = "\xeb\xc2", [0x4038] = "\xeb\xbf",
  [0x4039] = "\xbf\xe5", [0x403c] = "\xeb\xc3", [0x403d] = "\xeb\xc4",
  [0x403e] = "\xeb\xbe", [0x403f] = "\xeb\xc7", [0x4040] = "\xeb\xc0",
  [0x4041] = "\xeb\xc5", [0x4042] = "\xbf\xe4", [0x4044] = "\xbf\xe1",
  [0x4045] = "\xeb\xc1", [0x4047] = "\xee\xbf", [0x4048] = "\xc1\xd0",
  [0x4049] = "\xc1\xce", [0x404a] = "\xc1\xd1", [0x404b] = "\xc1\xcf",
  [0x404c] = "\xee\xbe", [0x404d] = "\xee\xbb", [0x404e] = "\xee\xba",
  [0x4050] = "\xee\xbd", [0x4053] = "\xee\xbc", [0x4054] = "\xf1\x45",
  [0x4055] = "\xc2\xde", [0x4056] = "\xf0\xfb", [0x4057] = "\xf0\xfa",
  [0x4059] = "\xc2\xd9", [0x405a] = "\xf1\x41", [0x405b] = "\xf1\x40",
  [0x405c] = "\xf0\xf7", [0x405d] = "\xf1\x43", [0x405e] = "\xf0\xfc",
  [0x405f] = "\xc2\xdd", [0x4060] = "\xf0\xf9", [0x4061] = "\xf1\x42",
  [0x4062] = "\xf0\xf8", [0x4063] = "\xc2\xda", [0x4064] = "\xc2\xdc",
  [0x4065] = "\xf0\xfd", [0x4066] = "\xc2\xdb", [0x4067] = "\xf0\xfe",
  [0x4069] = "\xf1\x44", [0x406a] = "\xf3\x52", [0x406c] = "\xc3\xde",
  [0x406d] = "\xf3\x4f", [0x406f] = "\xf3\x53", [0x4072] = "\xc3\xdb",
  [0x4073] = "\xf3\x51", [0x4074] = "\xc3\xe0", [0x4076] = "\xc3\xdd",
  [0x4078] = "\xf3\x50", [0x407a] = "\xc3\xdf", [0x407b] = "\xf3\x54",
  [0x407c] = "\xc3\xda", [0x4081] = "\xc4\xbc", [0x4082] = "\xc4\xbe",
  [0x4084] = "\xf4\xd9", [0x4085] = "\xc4\xbd", [0x4086] = "\xf4\xd7",
  [0x4087] = "\xc3\xdc", [0x4088] = "\xf4\xd8", [0x4089] = "\xc4\xbb",
  [0x408a] = "\xc5\x43", [0x408b] = "\xc5\x45", [0x408c] = "\xf6\x56",
  [0x408d] = "\xc5\x44", [0x408e] = "\xf6\x55", [0x4090] = "\xf7\x61",
  [0x4091] = "\xc5\xad", [0x4092] = "\xf7\x60", [0x4093] = "\xc5\xae",
  [0x4094] = "\xf7\x5e", [0x4095] = "\xf7\x5d", [0x4096] = "\xf7\x62",
  [0x4097] = "\xf7\x63", [0x4098] = "\xf8\x46", [0x409a] = "\xf7\x5f",
  [0x409d] = "\xf8\xc6", [0x409e] = "\xf8\xc3", [0x409f] = "\xf8\xc4",
  [0x40a0] = "\xf8\xc5", [0x40a1] = "\xc6\x5c", [0x40a3] = "\xf9\x51",
  [0x40a4] = "\xf9\x50", [0x40a5] = "\xf9\x4f", [0x40a6] = "\xf9\x70",
  [0x40a8] = "\xf9\xbe", [0x40a9] = "\xf9\xab", [0x40aa] = "\xc6\x6e",
  [0x40ab] = "\xa8\xad", [0x40ac] = "\xb0\x60", [0x40b2] = "\xb8\xfa",
  [0x40ba] = "\xbd\xf6", [0x40bd] = "\xeb\xc8", [0x40c0] = "\xc2\xdf",
  [0x40c2] = "\xf3\x55", [0x40c9] = "\xf9\xac", [0x40ca] = "\xa8\xae",
  [0x40cb] = "\xaa\xee", [0x40cc] = "\xad\x79", [0x40cd] = "\xad\x78",
  [0x40cf] = "\xb0\x63", [0x40d1] = "\xd3\xe8", [0x40d2] = "\xb0\x61",
  [0x40d3] = "\xd3\xe9", [0x40d4] = "\xb0\x62", [0x40d7] = "\xd7\xdf",
  [0x40d8] = "\xd7\xdb", [0x40db] = "\xb3\x6d", [0x40dc] = "\xd7\xde",
  [0x40dd] = "\xd7\xdd", [0x40de] = "\xd7\xdc", [0x40df] = "\xb3\x6e",
  [0x40e0] = "\xd7\xe0", [0x40e1] = "\xd7\xe1", [0x40e5] = "\xdc\x43",
  [0x40e6] = "\xdc\x41", [0x40e7] = "\xdc\x45", [0x40e8] = "\xdc\x46",
  [0x40e9] = "\xdc\x4c", [0x40eb] = "\xdc\x48", [0x40ec] = "\xdc\x4a",
  [0x40ee] = "\xdc\x42", [0x40ef] = "\xdb\xfc", [0x40f1] = "\xdc\x49",
  [0x40f4] = "\xdc\x4b", [0x40f5] = "\xdc\x44", [0x40f6] = "\xdc\x47",
  [0x40f7] = "\xdb\xfd", [0x40f8] = "\xb6\x62", [0x40f9] = "\xdc\x40",
  [0x40fa] = "\xdb\xfe", [0x40fb] = "\xb6\x61", [0x40fc] = "\xb6\x63",
  [0x40fe] = "\xb8\xfd", [0x40ff] = "\xe0\x75", [0x4100] = "\xe0\x77",
  [0x4101] = "\xe0\x76", [0x4102] = "\xe0\x7b", [0x4103] = "\xb8\xfb",
  [0x4105] = "\xe0\x78", [0x4106] = "\xe0\x74", [0x4107] = "\xe0\x79",
  [0x4108] = "\xe0\x7a", [0x4109] = "\xb8\xfc", [0x410a] = "\xb8\xfe",
  [0x410b] = "\xe0\x7c", [0x410d] = "\xe4\x67", [0x410e] = "\xe4\x66",
  [0x4110] = "\xe4\x64", [0x4111] = "\xe4\x65", [0x4112] = "\xbb\xb3",
  [0x4113] = "\xbb\xb5", [0x4114] = "\xbb\xb2", [0x4115] = "\xbb\xb4",
  [0x4116] = "\xe8\x4d", [0x4117] = "\xe8\x4e", [0x4118] = "\xe8\x49",
  [0x411a] = "\xe8\x4a", [0x411b] = "\xbd\xf8", [0x411c] = "\xbd\xfd",
  [0x411d] = "\xbd\xf7", [0x411e] = "\xbd\xfe", [0x411f] = "\xbd\xf9",
  [0x4120] = "\xe8\x4b", [0x4123] = "\xe8\x4c", [0x4124] = "\xe8\x48",
  [0x4125] = "\xbe\x40", [0x4126] = "\xbd\xfb", [0x4129] = "\xbd\xfa",
  [0x412a] = "\xbd\xfc", [0x412c] = "\xe8\x47", [0x412e] = "\xeb\xca",
  [0x412f] = "\xbf\xe8", [0x4132] = "\xeb\xcc", [0x4133] = "\xbf\xea",
  [0x4134] = "\xeb\xcf", [0x4135] = "\xeb\xcb", [0x4136] = "\xeb\xc9",
  [0x4137] = "\xeb\xce", [0x4138] = "\xbf\xe9", [0x4139] = "\xeb\xcd",
  [0x413b] = "\xbf\xe7", [0x413e] = "\xc1\xd3", [0x413f] = "\xc1\xd6",
  [0x4140] = "\xee\xc1", [0x4142] = "\xc1\xd4", [0x4143] = "\xee\xc0",
  [0x4144] = "\xc1\xd2", [0x4145] = "\xc1\xd5", [0x4146] = "\xf1\x46",
  [0x4147] = "\xf1\x47", [0x4148] = "\xf1\x48", [0x4149] = "\xc2\xe0",
  [0x414b] = "\xf1\x49", [0x414d] = "\xc2\xe1", [0x414e] = "\xc3\xe2",
  [0x414f] = "\xf3\x58", [0x4150] = "\xf3\x59", [0x4151] = "\xf3\x57",
  [0x4152] = "\xf3\x56", [0x4153] = "\xf3\x5a", [0x4154] = "\xc3\xe1",
  [0x4155] = "\xf4\xdd", [0x4156] = "\xf4\xdb", [0x4157] = "\xf4\xdc",
  [0x4158] = "\xf4\xde", [0x4159] = "\xf4\xda", [0x415a] = "\xf4\xdf",
  [0x415b] = "\xf6\x58", [0x415d] = "\xf6\x59", [0x415e] = "\xf6\x57",
  [0x415f] = "\xc5\x46", [0x4160] = "\xf7\x64", [0x4161] = "\xc5\xaf",
  [0x4162] = "\xf7\x65", [0x4163] = "\xf8\x48", [0x4164] = "\xf8\x47",
  [0x419b] = "\xa8\xaf", [0x419c] = "\xb6\x64", [0x419f] = "\xb9\x40",
  [0x41a3] = "\xbb\xb6", [0x41a6] = "\xbf\xec", [0x41a8] = "\xbf\xeb",
  [0x41ad] = "\xc3\xe3", [0x41ae] = "\xc4\x7c", [0x41af] = "\xc5\x47",
  [0x41b0] = "\xa8\xb0", [0x41b1] = "\xb0\x64", [0x41b2] = "\xb9\x41",
  [0x41b4] = "\xf3\x5b", [0x41bf] = "\xcb\xa6", [0x41c2] = "\xa8\xb1",
  [0x41c4] = "\xa8\xb4", [0x41c5] = "\xa8\xb3", [0x41c6] = "\xa8\xb2",
  [0x41c9] = "\xcb\xa5", [0x41cb] = "\xcd\xcd", [0x41cd] = "\xcd\xcf",
  [0x41ce] = "\xaa\xef", [0x41d1] = "\xaa\xf1", [0x41d2] = "\xcd\xcc",
  [0x41d3] = "\xcd\xce", [0x41d4] = "\xaa\xf0", [0x41d5] = "\xcd\xd1",
  [0x41d6] = "\xcd\xd0", [0x41d7] = "\xcd\xd2", [0x41e0] = "\xd0\xb6",
  [0x41e1] = "\xd0\xb4", [0x41e2] = "\xad\x7c", [0x41e3] = "\xd0\xb3",
  [0x41e4] = "\xad\xa3", [0x41e5] = "\xad\x7e", [0x41e6] = "\xad\x7b",
  [0x41e8] = "\xad\xa4", [0x41ea] = "\xad\x7d", [0x41eb] = "\xad\xa2",
  [0x41ed] = "\xad\xa1", [0x41ee] = "\xd0\xb5", [0x41f0] = "\xad\x7a",
  [0x41f4] = "\xb0\x6a", [0x41f5] = "\xd3\xeb", [0x41f6] = "\xd3\xf1",
  [0x41f7] = "\xb0\x67", [0x41f8] = "\xb0\x6e", [0x41fa] = "\xb0\x69",
  [0x41fb] = "\xd3\xee", [0x41fc] = "\xd3\xf0", [0x41fd] = "\xb0\x6c",
  [0x41fe] = "\xd3\xea", [0x41ff] = "\xd3\xed", [0x4200] = "\xb0\x68",
  [0x4201] = "\xb0\x65", [0x4202] = "\xd3\xec", [0x4203] = "\xb0\x6b",
  [0x4204] = "\xd3\xef", [0x4205] = "\xb0\x6d", [0x4206] = "\xb0\x66",
  [0x420b] = "\xd7\xe3", [0x420c] = "\xd7\xe6", [0x420d] = "\xb3\x70",
  [0x420f] = "\xb3\x7a", [0x4210] = "\xb3\x76", [0x4211] = "\xd7\xe4",
  [0x4214] = "\xb3\x7e", [0x4215] = "\xb3\x77", [0x4216] = "\xb3\x7c",
  [0x4217] = "\xb3\x72", [0x4219] = "\xb3\x6f", [0x421a] = "\xb3\x71",
  [0x421b] = "\xb3\x7d", [0x421c] = "\xd7\xe5", [0x421d] = "\xb3\x75",
  [0x421e] = "\xb3\x78", [0x421f] = "\xb3\x74", [0x4220] = "\xb3\x79",
  [0x4221] = "\xd7\xe7", [0x4222] = "\xb3\x7b", [0x4223] = "\xb3\x73",
  [0x4224] = "\xd7\xe2", [0x422d] = "\xdc\x4d", [0x422e] = "\xb6\x65",
  [0x422f] = "\xdc\x4f", [0x4231] = "\xb6\x67", [0x4232] = "\xb6\x69",
  [0x4234] = "\xdc\x4e", [0x4235] = "\xb6\x66", [0x4236] = "\xb6\x6a",
  [0x4238] = "\xb6\x68", [0x423c] = "\xb9\x47", [0x423d] = "\xe0\xa3",
  [0x423e] = "\xb9\x4f", [0x423f] = "\xe0\x7e", [0x4241] = "\xb9\x50",
  [0x4242] = "\xb9\x45", [0x4244] = "\xe0\xa1", [0x4247] = "\xb9\x4a",
  [0x4249] = "\xe0\xa2", [0x424a] = "\xb9\x43", [0x424b] = "\xb9\x42",
  [0x424d] = "\xb9\x4d", [0x424e] = "\xb9\x4c", [0x424f] = "\xb9\x4b",
  [0x4250] = "\xb9\x49", [0x4251] = "\xb9\x4e", [0x4252] = "\xe0\x7d",
  [0x4253] = "\xb9\x44", [0x4254] = "\xb9\x46", [0x4255] = "\xb9\x48",
  [0x4258] = "\xbb\xb8", [0x4259] = "\xbb\xbb", [0x425b] = "\xbb\xbf",
  [0x425c] = "\xbb\xb9", [0x425d] = "\xbb\xbe", [0x425e] = "\xbb\xbc",
  [0x4260] = "\xbb\xb7", [0x4262] = "\xbb\xbd", [0x4263] = "\xbb\xba",
  [0x4267] = "\xe8\x52", [0x4268] = "\xbe\x43", [0x4269] = "\xbe\x41",
  [0x426b] = "\xe8\x53", [0x426d] = "\xbe\x44", [0x426e] = "\xbe\x42",
  [0x426f] = "\xe8\x51", [0x4270] = "\xe8\x50", [0x4272] = "\xbf\xf0",
  [0x4273] = "\xe8\x4f", [0x4274] = "\xbf\xee", [0x4275] = "\xbf\xed",
  [0x4276] = "\xeb\xd0", [0x4277] = "\xbe\x45", [0x4278] = "\xbf\xef",
  [0x4279] = "\xeb\xd1", [0x427a] = "\xbf\xf2", [0x427b] = "\xeb\xd2",
  [0x427c] = "\xbf\xf1", [0x427d] = "\xc1\xd8", [0x427e] = "\xee\xc3",
  [0x427f] = "\xc1\xd7", [0x4280] = "\xc1\xdc", [0x4281] = "\xc1\xda",
  [0x4282] = "\xc1\xdb", [0x4283] = "\xc2\xe3", [0x4284] = "\xc1\xd9",
  [0x4285] = "\xee\xc2", [0x4286] = "\xeb\xd3", [0x4287] = "\xc2\xe2",
  [0x4288] = "\xc2\xe4", [0x428a] = "\xc3\xe4", [0x428b] = "\xc3\xe5",
  [0x428d] = "\xf4\xe0", [0x428f] = "\xc5\xde", [0x4290] = "\xc5\xdd",
  [0x4291] = "\xa8\xb6", [0x4294] = "\xca\x55", [0x4295] = "\xb0\x6f",
  [0x4297] = "\xca\x52", [0x4298] = "\xca\x53", [0x4299] = "\xca\x51",
  [0x429b] = "\xca\x54", [0x429e] = "\xcb\xaa", [0x429f] = "\xcb\xa7",
  [0x42a0] = "\xcb\xac", [0x42a1] = "\xcb\xa8", [0x42a2] = "\xa8\xb7",
  [0x42a3] = "\xa8\xba", [0x42a5] = "\xcb\xa9", [0x42a6] = "\xa8\xb9",
  [0x42a7] = "\xcb\xab", [0x42aa] = "\xa8\xb8", [0x42af] = "\xcd\xd5",
  [0x42b0] = "\xcd\xd7", [0x42b1] = "\xaa\xf4", [0x42b2] = "\xcd\xd3",
  [0x42b3] = "\xcd\xd6", [0x42b4] = "\xcd\xd4", [0x42b5] = "\xaa\xf2",
  [0x42b6] = "\xaa\xf5", [0x42b8] = "\xaa\xf3", [0x42bd] = "\xd0\xb8",
  [0x42be] = "\xd0\xbc", [0x42bf] = "\xd0\xb9", [0x42c1] = "\xad\xa7",
  [0x42c3] = "\xad\xa8", [0x42c5] = "\xd0\xbb", [0x42c7] = "\xd0\xbd",
  [0x42c8] = "\xd0\xbf", [0x42ca] = "\xad\xa5", [0x42cb] = "\xd0\xbe",
  [0x42ce] = "\xad\xa6", [0x42d4] = "\xd7\xee", [0x42d5] = "\xd0\xba",
  [0x42d6] = "\xd3\xf2", [0x42d7] = "\xd3\xfb", [0x42d8] = "\xd3\xf9",
  [0x42d9] = "\xd3\xf4", [0x42da] = "\xd3\xf5", [0x42db] = "\xd3\xfa",
  [0x42dc] = "\xd3\xfc", [0x42dd] = "\xb0\x71", [0x42df] = "\xd3\xf7",
  [0x42e0] = "\xd3\xf3", [0x42e1] = "\xb0\x70", [0x42e2] = "\xb0\x72",
  [0x42e3] = "\xd3\xf6", [0x42e4] = "\xd3\xfd", [0x42e5] = "\xd3\xf8",
  [0x42e8] = "\xb3\xa1", [0x42e9] = "\xd7\xf1", [0x42ea] = "\xd7\xe9",
  [0x42eb] = "\xd7\xef", [0x42ec] = "\xd7\xf0", [0x42ed] = "\xb3\xa2",
  [0x42ef] = "\xd7\xe8", [0x42f0] = "\xd7\xea", [0x42f1] = "\xd0\xb7",
  [0x42f2] = "\xd7\xec", [0x42f3] = "\xd7\xed", [0x42f4] = "\xd7\xeb",
  [0x42f5] = "\xb6\x6c", [0x42f9] = "\xdc\x56", [0x42fa] = "\xeb\xd4",
  [0x42fb] = "\xdc\x57", [0x42fc] = "\xdc\x54", [0x42fd] = "\xb3\xa3",
  [0x42fe] = "\xb6\x6e", [0x42ff] = "\xdc\x53", [0x4300] = "\xdc\x59",
  [0x4301] = "\xdc\x58", [0x4302] = "\xb6\x6b", [0x4303] = "\xdc\x5c",
  [0x4304] = "\xdc\x52", [0x4305] = "\xdc\x5b", [0x4306] = "\xdc\x50",
  [0x4307] = "\xdc\x5a", [0x4308] = "\xdc\x55", [0x4309] = "\xb6\x6d",
  [0x430b] = "\xe0\xaa", [0x430d] = "\xe0\xa5", [0x430e] = "\xe0\xab",
  [0x430f] = "\xe0\xa6", [0x4310] = "\xe0\xa4", [0x4311] = "\xe0\xa7",
  [0x4312] = "\xb9\x51", [0x4314] = "\xe0\xa9", [0x4316] = "\xe0\xa8",
  [0x4317] = "\xb9\x52", [0x4318] = "\xbb\xc1", [0x4319] = "\xbb\xc0",
  [0x431a] = "\xe4\x6e", [0x431b] = "\xe4\x71", [0x431c] = "\xe4\x69",
  [0x431d] = "\xe4\x6d", [0x431e] = "\xbb\xc2", [0x431f] = "\xe4\x6c",
  [0x4320] = "\xe4\x6a", [0x4321] = "\xe4\x70", [0x4322] = "\xe4\x6b",
  [0x4323] = "\xe4\x68", [0x4324] = "\xe4\x6f", [0x4326] = "\xe8\x59",
  [0x4327] = "\xbe\x48", [0x4328] = "\xf1\x4a", [0x4329] = "\xe8\x56",
  [0x432a] = "\xe8\x57", [0x432b] = "\xe8\x55", [0x432c] = "\xdc\x51",
  [0x432d] = "\xbe\x47", [0x432e] = "\xe8\x5a", [0x432f] = "\xe8\x54",
  [0x4330] = "\xbe\x46", [0x4331] = "\xbe\x49", [0x4332] = "\xe8\x58",
  [0x4333] = "\xeb\xd5", [0x4334] = "\xbf\xf3", [0x4335] = "\xeb\xd6",
  [0x4336] = "\xeb\xd7", [0x4338] = "\xee\xc4", [0x4339] = "\xc1\xdd",
  [0x433a] = "\xf1\x4b", [0x433b] = "\xf1\x4c", [0x433e] = "\xf1\x4d",
  [0x433f] = "\xf3\x5d", [0x4340] = "\xf3\x5c", [0x4341] = "\xf4\xe2",
  [0x4343] = "\xf4\xe1", [0x4344] = "\xf6\x5b", [0x4345] = "\xf6\x5c",
  [0x4346] = "\xf6\x5a", [0x4347] = "\xf7\x66", [0x4348] = "\xc5\xb0",
  [0x4349] = "\xa8\xbb", [0x434a] = "\xad\xaa", [0x434b] = "\xad\xa9",
  [0x434c] = "\xb0\x75", [0x434d] = "\xb0\x74", [0x434e] = "\xd4\x40",
  [0x434f] = "\xd4\x41", [0x4350] = "\xd3\xfe", [0x4352] = "\xb0\x73",
  [0x4353] = "\xd7\xf5", [0x4355] = "\xd7\xf6", [0x4356] = "\xd7\xf2",
  [0x4357] = "\xb3\xa4", [0x4358] = "\xd7\xf3", [0x435a] = "\xd7\xf4",
  [0x435f] = "\xdc\x5f", [0x4360] = "\xdc\x61", [0x4361] = "\xdc\x5d",
  [0x4362] = "\xdc\x60", [0x4363] = "\xb6\x6f", [0x4364] = "\xdc\x5e",
  [0x4365] = "\xb6\x70", [0x4368] = "\xdd\x73", [0x4369] = "\xb9\x55",
  [0x436a] = "\xb9\x54", [0x436c] = "\xb9\x53", [0x436e] = "\xe0\xac",
  [0x436f] = "\xe0\xad", [0x4372] = "\xe4\x73", [0x4373] = "\xe4\x75",
  [0x4374] = "\xbb\xc6", [0x4375] = "\xbb\xc3", [0x4377] = "\xbb\xc5",
  [0x4378] = "\xbb\xc4", [0x4379] = "\xe4\x74", [0x437a] = "\xe4\x72",
  [0x4380] = "\xe8\x61", [0x4381] = "\xe8\x5e", [0x4382] = "\xe8\x5f",
  [0x4383] = "\xbe\x4d", [0x4384] = "\xe8\x60", [0x4385] = "\xe8\x5b",
  [0x4386] = "\xe8\x5c", [0x4387] = "\xbe\x4a", [0x4389] = "\xbe\x4b",
  [0x438a] = "\xe8\x5d", [0x438b] = "\xbe\x4c", [0x438d] = "\xeb\xdb",
  [0x438f] = "\xeb\xdc", [0x4390] = "\xeb\xd9", [0x4391] = "\xeb\xda",
  [0x4392] = "\xbf\xf4", [0x4393] = "\xeb\xd8", [0x4399] = "\xee\xc8",
  [0x439a] = "\xee\xc5", [0x439b] = "\xee\xc7", [0x439c] = "\xc1\xe0",
  [0x439d] = "\xee\xcb", [0x439e] = "\xc1\xdf", [0x439f] = "\xee\xc9",
  [0x43a0] = "\xee\xcc", [0x43a1] = "\xee\xca", [0x43a2] = "\xee\xc6",
  [0x43a3] = "\xc1\xde", [0x43a5] = "\xf1\x4f", [0x43a7] = "\xf1\x50",
  [0x43a8] = "\xf1\x4e", [0x43aa] = "\xf1\x52", [0x43ab] = "\xc2\xe5",
  [0x43ac] = "\xc2\xe6", [0x43ad] = "\xf3\x5f", [0x43ae] = "\xc3\xe7",
  [0x43af] = "\xf1\x51", [0x43b0] = "\xf3\x5e", [0x43b1] = "\xc3\xe6",
  [0x43b2] = "\xf4\xe5", [0x43b3] = "\xf4\xe6", [0x43b4] = "\xc4\xbf",
  [0x43b5] = "\xf4\xe4", [0x43b7] = "\xf4\xe3", [0x43b9] = "\xf6\x5d",
  [0x43ba] = "\xc5\x48", [0x43bc] = "\xf8\x49", [0x43bd] = "\xf8\xc8",
  [0x43be] = "\xf8\xc7", [0x43c0] = "\xc6\x43", [0x43c1] = "\xc6\x5d",
  [0x43c2] = "\xf8\xc9", [0x43c3] = "\xf9\x71", [0x43c5] = "\xc6\x6f",
  [0x43c6] = "\xa8\xbc", [0x43c7] = "\xaa\xf6", [0x43c9] = "\xb9\x56",
  [0x43cb] = "\xc4\xc0", [0x43cc] = "\xa8\xbd", [0x43cd] = "\xad\xab",
  [0x43ce] = "\xb3\xa5", [0x43cf] = "\xb6\x71", [0x43d0] = "\xc2\xe7",
  [0x43d1] = "\xaa\xf7", [0x43d3] = "\xd0\xc1", [0x43d4] = "\xd0\xc0",
  [0x43d5] = "\xd4\x42", [0x43d7] = "\xb0\x78", [0x43d8] = "\xb0\x76",
  [0x43d9] = "\xb0\x7a", [0x43da] = "\xd4\x44", [0x43dc] = "\xb0\x79",
  [0x43dd] = "\xb0\x77", [0x43e2] = "\xd4\x43", [0x43e3] = "\xb3\xa8",
  [0x43e4] = "\xd7\xfc", [0x43e6] = "\xb3\xa7", [0x43e7] = "\xb3\xa9",
  [0x43e8] = "\xd8\x42", [0x43e9] = "\xb3\xab", [0x43ea] = "\xd7\xfe",
  [0x43eb] = "\xd8\x40", [0x43ec] = "\xd7\xf7", [0x43ed] = "\xb3\xaa",
  [0x43ee] = "\xd8\x43", [0x43f1] = "\xd7\xf9", [0x43f3] = "\xd7\xfa",
  [0x43f4] = "\xd7\xf8", [0x43f5] = "\xb3\xa6", [0x43f7] = "\xd8\x41",
  [0x43f8] = "\xd7\xfb", [0x43f9] = "\xd7\xfd", [0x43fd] = "\xdc\x6d",
  [0x43ff] = "\xdc\x6c", [0x4400] = "\xdc\x6a", [0x4401] = "\xdc\x62",
  [0x4402] = "\xdc\x71", [0x4403] = "\xdc\x65", [0x4404] = "\xdc\x6f",
  [0x4405] = "\xdc\x76", [0x4406] = "\xdc\x6e", [0x4407] = "\xb6\x79",
  [0x4409] = "\xb6\x75", [0x440a] = "\xdc\x63", [0x440c] = "\xdc\x69",
  [0x440d] = "\xb6\x77", [0x440f] = "\xdc\x68", [0x4410] = "\xb6\x78",
  [0x4411] = "\xb6\x7a", [0x4412] = "\xdc\x6b", [0x4414] = "\xb6\x72",
  [0x4415] = "\xb6\x73", [0x4416] = "\xdc\x77", [0x4417] = "\xdc\x75",
  [0x4419] = "\xdc\x74", [0x441a] = "\xdc\x66", [0x441c] = "\xdc\x72",
  [0x441e] = "\xb6\x76", [0x4423] = "\xb6\x74", [0x4424] = "\xdc\x73",
  [0x4425] = "\xdc\x64", [0x4426] = "\xdc\x67", [0x4427] = "\xdc\x70",
  [0x442d] = "\xe4\xba", [0x442e] = "\xe0\xb7", [0x4430] = "\xe0\xb0",
  [0x4431] = "\xe0\xc3", [0x4432] = "\xe0\xcc", [0x4433] = "\xe0\xb3",
  [0x4434] = "\xb9\x61", [0x4436] = "\xe0\xc0", [0x4437] = "\xb9\x57",
  [0x4438] = "\xb9\x59", [0x4439] = "\xb9\x65", [0x443a] = "\xe0\xb1",
  [0x443d] = "\xb9\x5a", [0x443e] = "\xb9\x5c", [0x443f] = "\xb9\x66",
  [0x4440] = "\xb9\x5b", [0x4445] = "\xb9\x64", [0x4446] = "\xe0\xb9",
  [0x4448] = "\xe0\xae", [0x4449] = "\xb9\x62", [0x444a] = "\xe0\xb8",
  [0x444b] = "\xb9\x5e", [0x444c] = "\xe0\xca", [0x444d] = "\xb9\x63",
  [0x444e] = "\xe0\xc8", [0x444f] = "\xe0\xbc", [0x4450] = "\xe0\xc6",
  [0x4451] = "\xb9\x60", [0x4452] = "\xe0\xaf", [0x4453] = "\xe0\xc9",
  [0x4454] = "\xe0\xc4", [0x4456] = "\xe0\xcb", [0x4457] = "\xb9\x58",
  [0x445a] = "\xb9\x67", [0x445b] = "\xb9\x5d", [0x445e] = "\xe0\xb5",
  [0x4460] = "\xe0\xbd", [0x4461] = "\xe0\xc1", [0x4463] = "\xe0\xc5",
  [0x4464] = "\xb9\x5f", [0x4465] = "\xe0\xb4", [0x4466] = "\xe0\xb2",
  [0x4467] = "\xe0\xbe", [0x446c] = "\xe0\xbb", [0x446d] = "\xe0\xba",
  [0x446f] = "\xe0\xbf", [0x4470] = "\xe0\xc2", [0x4472] = "\xe0\xc7",
  [0x4476] = "\xe4\x78", [0x4478] = "\xbb\xc7", [0x4479] = "\xe4\xa4",
  [0x447a] = "\xe4\x7a", [0x447b] = "\xbb\xcc", [0x447c] = "\xbb\xd0",
  [0x447d] = "\xe4\xad", [0x447e] = "\xe4\xb5", [0x447f] = "\xe4\xa6",
  [0x4480] = "\xbb\xc8", [0x4482] = "\xe4\xaa", [0x4483] = "\xe0\xb6",
  [0x4485] = "\xbb\xc9", [0x4486] = "\xe4\xb1", [0x4487] = "\xe4\xb6",
  [0x4488] = "\xe4\xae", [0x448a] = "\xe4\xb0", [0x448b] = "\xe4\xb9",
  [0x448c] = "\xe4\xb2", [0x448d] = "\xe4\x7e", [0x448e] = "\xe4\xa9",
  [0x4491] = "\xbb\xd1", [0x4493] = "\xbb\xcd", [0x4494] = "\xe4\x7c",
  [0x4495] = "\xe4\xab", [0x4496] = "\xbb\xcb", [0x4497] = "\xe4\xa5",
  [0x4498] = "\xbb\xca", [0x4499] = "\xe4\xb3", [0x449a] = "\xe4\xa2",
  [0x449b] = "\xe4\x79", [0x449c] = "\xbb\xce", [0x449d] = "\xe4\xb8",
  [0x44a0] = "\xe4\x7b", [0x44a1] = "\xe4\xaf", [0x44a2] = "\xe4\xac",
  [0x44a3] = "\xe4\xa7", [0x44a4] = "\xe4\x77", [0x44a5] = "\xe4\x76",
  [0x44a6] = "\xe4\xa1", [0x44a7] = "\xe4\xb4", [0x44a8] = "\xbb\xcf",
  [0x44a9] = "\xe4\xb7", [0x44aa] = "\xe4\x7d", [0x44ab] = "\xe4\xa3",
  [0x44ac] = "\xbe\x52", [0x44b2] = "\xbe\x5a", [0x44b3] = "\xbe\x55",
  [0x44b4] = "\xe8\xa4", [0x44b5] = "\xe8\xa1", [0x44b6] = "\xe8\x67",
  [0x44b7] = "\xbe\x50", [0x44b9] = "\xf9\xd7", [0x44bb] = "\xbe\x4f",
  [0x44bc] = "\xbe\x56", [0x44c0] = "\xe8\x65", [0x44c1] = "\xbe\x54",
  [0x44c2] = "\xe8\x71", [0x44c3] = "\xe8\x63", [0x44c4] = "\xe8\x64",
  [0x44c5] = "\xbe\x4e", [0x44c6] = "\xe8\xa3", [0x44c7] = "\xbe\x58",
  [0x44c8] = "\xe8\x74", [0x44c9] = "\xe8\x79", [0x44ca] = "\xe8\x73",
  [0x44cb] = "\xeb\xee", [0x44cc] = "\xe8\x6f", [0x44cd] = "\xe8\x77",
  [0x44ce] = "\xe8\x75", [0x44cf] = "\xe8\x68", [0x44d0] = "\xe8\x62",
  [0x44d1] = "\xe8\x7d", [0x44d2] = "\xbe\x57", [0x44d3] = "\xe8\x7e",
  [0x44d5] = "\xe8\x78", [0x44d7] = "\xe8\x6d", [0x44d8] = "\xe8\x6b",
  [0x44d9] = "\xe8\x66", [0x44dd] = "\xe8\x6e", [0x44de] = "\xe8\x7b",
  [0x44df] = "\xe8\x6a", [0x44e0] = "\xe8\x7a", [0x44e1] = "\xe8\xa2",
  [0x44e4] = "\xbe\x53", [0x44e6] = "\xe8\x76", [0x44e7] = "\xe8\x7c",
  [0x44e8] = "\xe8\x72", [0x44e9] = "\xe8\x6c", [0x44ea] = "\xbe\x51",
  [0x44ee] = "\xe4\xa8", [0x44ef] = "\xe8\x70", [0x44f0] = "\xbe\x59",
  [0x44f1] = "\xe8\x69", [0x44f7] = "\xeb\xf4", [0x44f8] = "\xbf\xf7",
  [0x44f9] = "\xeb\xf3", [0x44fa] = "\xeb\xf0", [0x44fb] = "\xec\x44",
  [0x44fc] = "\xbf\xfb", [0x44fe] = "\xec\x41", [0x44ff] = "\xeb\xf8",
  [0x4500] = "\xec\x43", [0x4501] = "\xeb\xe9", [0x4502] = "\xeb\xf6",
  [0x4504] = "\xbf\xfd", [0x4506] = "\xeb\xe1", [0x4508] = "\xeb\xdf",
  [0x4509] = "\xec\x42", [0x450b] = "\xec\x40", [0x450c] = "\xeb\xfe",
  [0x450d] = "\xeb\xed", [0x450e] = "\xeb\xec", [0x450f] = "\xeb\xe2",
  [0x4510] = "\xc0\x40", [0x4512] = "\xeb\xe8", [0x4513] = "\xeb\xf2",
  [0x4514] = "\xeb\xfd", [0x4515] = "\xc0\x43", [0x4516] = "\xec\x45",
  [0x4518] = "\xc1\xe8", [0x4519] = "\xc0\x45", [0x451a] = "\xbf\xfe",
  [0x451b] = "\xeb\xe6", [0x451d] = "\xeb\xef", [0x451e] = "\xeb\xde",
  [0x451f] = "\xeb\xe0", [0x4520] = "\xbf\xf5", [0x4521] = "\xc0\x42",
  [0x4522] = "\xbf\xfa", [0x4523] = "\xeb\xe7", [0x4524] = "\xeb\xf7",
  [0x4525] = "\xeb\xf1", [0x4526] = "\xc0\x41", [0x4527] = "\xeb\xdd",
  [0x4528] = "\xc1\xe3", [0x4529] = "\xeb\xf9", [0x452a] = "\xeb\xfc",
  [0x452b] = "\xbf\xfc", [0x452d] = "\xeb\xeb", [0x452e] = "\xc0\x44",
  [0x452f] = "\xbf\xf9", [0x4533] = "\xbf\xf8", [0x4534] = "\xeb\xf5",
  [0x4535] = "\xeb\xfb", [0x4536] = "\xbf\xf6", [0x4538] = "\xeb\xe4",
  [0x4539] = "\xeb\xfa", [0x453c] = "\xeb\xe5", [0x4546] = "\xeb\xea",
  [0x4547] = "\xee\xd2", [0x4549] = "\xee\xd7", [0x454a] = "\xc1\xe5",
  [0x454b] = "\xc1\xe7", [0x454c] = "\xee\xdd", [0x454d] = "\xc1\xe1",
  [0x454e] = "\xee\xec", [0x454f] = "\xee\xe3", [0x4550] = "\xee\xd8",
  [0x4551] = "\xee\xd9", [0x4552] = "\xee\xe2", [0x4554] = "\xc1\xee",
  [0x4555] = "\xee\xe1", [0x4556] = "\xee\xd1", [0x4557] = "\xee\xe0",
  [0x4558] = "\xee\xd4", [0x4559] = "\xee\xed", [0x455a] = "\xc1\xed",
  [0x455b] = "\xc1\xeb", [0x455c] = "\xee\xd5", [0x455e] = "\xee\xe8",
  [0x4560] = "\xee\xda", [0x4561] = "\xee\xe7", [0x4563] = "\xee\xe9",
  [0x4564] = "\xee\xd0", [0x4565] = "\xc1\xe6", [0x4567] = "\xee\xea",
  [0x456a] = "\xee\xde", [0x456c] = "\xc1\xea", [0x456d] = "\xee\xdb",
  [0x4570] = "\xc1\xec", [0x4571] = "\xee\xe4", [0x4575] = "\xc1\xe4",
  [0x4576] = "\xee\xd6", [0x4577] = "\xee\xe5", [0x4579] = "\xee\xdf",
  [0x457a] = "\xeb\xe3", [0x457b] = "\xee\xe6", [0x457c] = "\xee\xd3",
  [0x457e] = "\xc1\xe9", [0x4580] = "\xee\xeb", [0x4582] = "\xc1\xe2",
  [0x4583] = "\xee\xce", [0x4588] = "\xf1\x60", [0x4589] = "\xf1\x59",
  [0x458a] = "\xc2\xe9", [0x458c] = "\xf1\x54", [0x458d] = "\xf1\x63",
  [0x458e] = "\xf1\x5b", [0x458f] = "\xee\xdc", [0x4591] = "\xf1\x65",
  [0x4592] = "\xf1\x55", [0x4594] = "\xc2\xe8", [0x4595] = "\xf1\x5f",
  [0x4596] = "\xc2\xea", [0x4597] = "\xc2\xf2", [0x4598] = "\xc2\xf0",
  [0x4599] = "\xf1\x61", [0x459a] = "\xc2\xf1", [0x459b] = "\xf1\x57",
  [0x459d] = "\xf1\x58", [0x459e] = "\xf1\x5d", [0x459f] = "\xf1\x62",
  [0x45a1] = "\xee\xcd", [0x45a2] = "\xc2\xeb", [0x45a3] = "\xf1\x6a",
  [0x45a4] = "\xf1\x67", [0x45a5] = "\xf1\x6b", [0x45a6] = "\xf1\x5e",
  [0x45a7] = "\xf1\x5a", [0x45a8] = "\xf1\x68", [0x45a9] = "\xf3\x6a",
  [0x45aa] = "\xf1\x5c", [0x45ac] = "\xc2\xee", [0x45ae] = "\xc2\xed",
  [0x45af] = "\xee\xcf", [0x45b0] = "\xc2\xef", [0x45b1] = "\xf1\x64",
  [0x45b2] = "\xf1\x66", [0x45b3] = "\xc2\xec", [0x45b4] = "\xf1\x69",
  [0x45b5] = "\xf1\x53", [0x45b7] = "\xf1\x56", [0x45c0] = "\xf3\x73",
  [0x45c2] = "\xf3\x63", [0x45c3] = "\xc3\xeb", [0x45c4] = "\xf3\x71",
  [0x45c7] = "\xf3\x61", [0x45c8] = "\xc3\xec", [0x45ca] = "\xf3\x6c",
  [0x45cc] = "\xf3\x68", [0x45cd] = "\xc3\xf1", [0x45ce] = "\xf3\x72",
  [0x45cf] = "\xf3\x62", [0x45d0] = "\xf3\x65", [0x45d1] = "\xc3\xe9",
  [0x45d2] = "\xf3\x74", [0x45d4] = "\xf3\x6d", [0x45d5] = "\xf3\x70",
  [0x45d6] = "\xc3\xef", [0x45d7] = "\xc3\xf4", [0x45d8] = "\xc3\xf2",
  [0x45d9] = "\xf3\x69", [0x45da] = "\xf3\x64", [0x45dc] = "\xc3\xed",
  [0x45dd] = "\xc3\xee", [0x45de] = "\xf3\x60", [0x45df] = "\xc3\xea",
  [0x45e1] = "\xc3\xe8", [0x45e2] = "\xc3\xf0", [0x45e3] = "\xf3\x6f",
  [0x45e4] = "\xc3\xf3", [0x45e6] = "\xf3\x6b", [0x45e7] = "\xf3\x75",
  [0x45e8] = "\xc3\xf5", [0x45ec] = "\xf3\x67", [0x45ee] = "\xf3\x6e",
  [0x45f5] = "\xf4\xf3", [0x45f6] = "\xf5\x42", [0x45f7] = "\xf4\xf5",
  [0x45f8] = "\xf4\xfc", [0x45f9] = "\xf3\x66", [0x45fa] = "\xf4\xfa",
  [0x45fb] = "\xf4\xe9", [0x45fc] = "\xf5\x40", [0x45fd] = "\xc4\xc3",
  [0x45fe] = "\xf4\xed", [0x45ff] = "\xf4\xfe", [0x4600] = "\xf4\xf4",
  [0x4603] = "\xc4\xc2", [0x4606] = "\xf5\x44", [0x4607] = "\xf4\xf6",
  [0x4609] = "\xf4\xfb", [0x460a] = "\xf4\xfd", [0x460b] = "\xf4\xe7",
  [0x460c] = "\xf5\x41", [0x460d] = "\xf4\xf2", [0x460e] = "\xf4\xf7",
  [0x460f] = "\xf4\xeb", [0x4610] = "\xf4\xef", [0x4611] = "\xf5\x43",
  [0x4612] = "\xf4\xf9", [0x4613] = "\xf4\xe8", [0x4614] = "\xf4\xec",
  [0x4615] = "\xf4\xee", [0x4616] = "\xf4\xf8", [0x4618] = "\xc4\xc1",
  [0x4619] = "\xf4\xf1", [0x4620] = "\xf4\xea", [0x4628] = "\xf4\xf0",
  [0x4629] = "\xf6\x61", [0x462a] = "\xf6\x66", [0x462b] = "\xc5\x4f",
  [0x462c] = "\xf6\x68", [0x462e] = "\xc5\x49", [0x4630] = "\xf6\x64",
  [0x4631] = "\xf6\x6a", [0x4632] = "\xc5\x4e", [0x4633] = "\xc5\x4a",
  [0x4635] = "\xc5\x4b", [0x4636] = "\xf6\x60", [0x4637] = "\xf6\x67",
  [0x4638] = "\xc5\x4d", [0x4639] = "\xf6\x65", [0x463a] = "\xc5\x4c",
  [0x463b] = "\xf6\x5f", [0x463c] = "\xf6\x63", [0x463d] = "\xf6\x62",
  [0x463f] = "\xf6\x5e", [0x4640] = "\xf6\x69", [0x4644] = "\xc5\xb1",
  [0x4645] = "\xf7\x6d", [0x4646] = "\xf7\x70", [0x4647] = "\xf7\x6c",
  [0x4648] = "\xf7\x6e", [0x4649] = "\xf7\x6f", [0x464a] = "\xf7\x69",
  [0x464b] = "\xf7\x6a", [0x464c] = "\xf7\x67", [0x464f] = "\xf7\x6b",
  [0x4650] = "\xf7\x68", [0x4651] = "\xc5\xb2", [0x4652] = "\xc5\xb3",
  [0x4655] = "\xf8\x4b", [0x4657] = "\xf8\x4d", [0x465d] = "\xf8\x4c",
  [0x465e] = "\xf8\x4e", [0x4660] = "\xc5\xe0", [0x4662] = "\xf8\x4a",
  [0x4663] = "\xc5\xdf", [0x4664] = "\xc5\xe1", [0x4668] = "\xf8\xcb",
  [0x4669] = "\xf8\xcc", [0x466a] = "\xc6\x44", [0x466b] = "\xf8\xca",
  [0x466d] = "\xf9\x53", [0x466e] = "\xf9\x52", [0x466f] = "\xf9\x54",
  [0x4670] = "\xc6\x5f", [0x4671] = "\xf9\x55", [0x4672] = "\xc6\x5e",
  [0x4673] = "\xf9\x56", [0x4674] = "\xf9\x72", [0x4675] = "\xf9\x75",
  [0x4676] = "\xf9\x74", [0x4677] = "\xc6\x68", [0x4678] = "\xf9\x73",
  [0x467c] = "\xc6\x72", [0x467d] = "\xc6\x70", [0x467e] = "\xc6\x71",
  [0x467f] = "\xc6\x77", [0x4680] = "\xf9\xc0", [0x4681] = "\xf9\xc1",
  [0x4682] = "\xf9\xbf", [0x4683] = "\xf9\xc9", [0x4777] = "\xaa\xf8",
  [0x477a] = "\xd8\x44", [0x477b] = "\xdc\x78", [0x477c] = "\xe8\xa5",
  [0x477d] = "\xf3\x76", [0x4780] = "\xaa\xf9", [0x4782] = "\xad\xac",
  [0x4783] = "\xb0\x7b", [0x4786] = "\xd8\x45", [0x4788] = "\xd8\x46",
  [0x4789] = "\xb3\xac", [0x478b] = "\xb6\x7d", [0x478c] = "\xdc\x7a",
  [0x478d] = "\xdc\x79", [0x478e] = "\xb6\xa3", [0x478f] = "\xb6\x7c",
  [0x4790] = "\xdc\x7b", [0x4791] = "\xb6\x7e", [0x4792] = "\xb6\xa2",
  [0x4793] = "\xb6\xa1", [0x4794] = "\xb6\x7b", [0x4798] = "\xb9\x68",
  [0x479b] = "\xe0\xd0", [0x479c] = "\xe0\xce", [0x479e] = "\xe0\xcf",
  [0x479f] = "\xe0\xcd", [0x47a1] = "\xbb\xd2", [0x47a3] = "\xbb\xd5",
  [0x47a4] = "\xbb\xd7", [0x47a5] = "\xbb\xd6", [0x47a8] = "\xbb\xd3",
  [0x47a9] = "\xbb\xd4", [0x47ab] = "\xe8\xa7", [0x47ac] = "\xe8\xa6",
  [0x47ad] = "\xbe\x5b", [0x47ae] = "\xe8\xa8", [0x47b0] = "\xe8\xa9",
  [0x47b1] = "\xbe\x5c", [0x47b5] = "\xec\x4d", [0x47b6] = "\xec\x4b",
  [0x47b7] = "\xee\xf3", [0x47b9] = "\xec\x49", [0x47ba] = "\xec\x4a",
  [0x47bb] = "\xc0\x46", [0x47bc] = "\xec\x46", [0x47bd] = "\xec\x4e",
  [0x47be] = "\xec\x48", [0x47bf] = "\xec\x4c", [0x47c0] = "\xee\xef",
  [0x47c3] = "\xee\xf1", [0x47c5] = "\xee\xf2", [0x47c6] = "\xc1\xf3",
  [0x47c7] = "\xee\xee", [0x47c8] = "\xc1\xf2", [0x47c9] = "\xee\xf0",
  [0x47ca] = "\xc1\xef", [0x47cb] = "\xc1\xf0", [0x47cc] = "\xc1\xf1",
  [0x47cd] = "\xec\x47", [0x47d0] = "\xc2\xf5", [0x47d1] = "\xf1\x6e",
  [0x47d2] = "\xf1\x6c", [0x47d3] = "\xf1\x6d", [0x47d4] = "\xc2\xf3",
  [0x47d5] = "\xc2\xf6", [0x47d6] = "\xc2\xf4", [0x47da] = "\xf3\x77",
  [0x47db] = "\xf3\x78", [0x47dc] = "\xc3\xf6", [0x47de] = "\xf5\x45",
  [0x47df] = "\xf5\x47", [0x47e0] = "\xf5\x46", [0x47e1] = "\xc4\xc4",
  [0x47e2] = "\xc5\x50", [0x47e3] = "\xf6\x6d", [0x47e4] = "\xf6\x6c",
  [0x47e5] = "\xf6\x6b", [0x481c] = "\xaa\xfa", [0x481e] = "\xc9\xaa",
  [0x4820] = "\xca\x58", [0x4821] = "\xa6\xe9", [0x4822] = "\xca\x56",
  [0x4823] = "\xca\x59", [0x4824] = "\xca\x57", [0x4828] = "\xcb\xae",
  [0x482a] = "\xa8\xc1", [0x482c] = "\xa8\xc2", [0x482d] = "\xcb\xb0",
  [0x482e] = "\xa8\xbf", [0x482f] = "\xcb\xaf", [0x4830] = "\xcb\xad",
  [0x4831] = "\xa8\xc0", [0x4832] = "\xa8\xbe", [0x4839] = "\xcd\xd8",
  [0x483a] = "\xcd\xdb", [0x483b] = "\xaa\xfd", [0x483c] = "\xcd\xda",
  [0x483d] = "\xcd\xd9", [0x483f] = "\xaa\xfc", [0x4840] = "\xaa\xfb",
  [0x4842] = "\xab\x40", [0x4843] = "\xcd\xdc", [0x4844] = "\xaa\xfe",
  [0x484a] = "\xd0\xc6", [0x484b] = "\xad\xae", [0x484c] = "\xad\xaf",
  [0x484d] = "\xad\xb0", [0x484e] = "\xd0\xc7", [0x484f] = "\xd0\xc3",
  [0x4850] = "\xad\xad", [0x4851] = "\xd0\xc4", [0x4853] = "\xd0\xc5",
  [0x4854] = "\xd0\xc2", [0x4858] = "\xb0\xa4", [0x485b] = "\xb0\xa1",
  [0x485c] = "\xd4\x45", [0x485d] = "\xb0\xa2", [0x485e] = "\xb0\xa5",
  [0x485f] = "\xd4\x46", [0x4861] = "\xb0\x7e", [0x4862] = "\xb0\x7c",
  [0x4863] = "\xb0\x7d", [0x4864] = "\xb0\xa3", [0x486a] = "\xb3\xad",
  [0x486b] = "\xd8\x49", [0x486c] = "\xb3\xb5", [0x486d] = "\xd8\x48",
  [0x486f] = "\xd8\x4b", [0x4870] = "\xb3\xb1", [0x4871] = "\xd8\x4a",
  [0x4872] = "\xb6\xab", [0x4873] = "\xb3\xaf", [0x4874] = "\xb3\xb2",
  [0x4875] = "\xb3\xae", [0x4876] = "\xb3\xb3", [0x4877] = "\xb3\xb4",
  [0x4878] = "\xb3\xb0", [0x487c] = "\xd8\x47", [0x487d] = "\xb6\xa7",
  [0x487e] = "\xdc\x7d", [0x4880] = "\xdc\xa3", [0x4883] = "\xdc\xa2",
  [0x4884] = "\xb6\xac", [0x4885] = "\xb6\xa8", [0x4886] = "\xb6\xa9",
  [0x4887] = "\xdc\x7c", [0x4888] = "\xdc\x7e", [0x4889] = "\xdc\xa1",
  [0x488a] = "\xb6\xa4", [0x488b] = "\xb6\xa6", [0x488d] = "\xb6\xaa",
  [0x488e] = "\xb6\xa5", [0x4891] = "\xe0\xd3", [0x4892] = "\xe0\xd1",
  [0x4893] = "\xe0\xd2", [0x4894] = "\xb9\x6a", [0x4895] = "\xb9\x6b",
  [0x4897] = "\xe0\xd4", [0x4898] = "\xb9\x69", [0x4899] = "\xbb\xd8",
  [0x489b] = "\xbb\xda", [0x489c] = "\xbb\xd9", [0x489e] = "\xe4\xbb",
  [0x48a1] = "\xe4\xbc", [0x48a2] = "\xe8\xab", [0x48a4] = "\xe8\xaa",
  [0x48a7] = "\xc0\x47", [0x48a8] = "\xc0\x48", [0x48a9] = "\xec\x4f",
  [0x48aa] = "\xc0\x49", [0x48ac] = "\xee\xf6", [0x48ae] = "\xee\xf4",
  [0x48b0] = "\xee\xf5", [0x48b1] = "\xc1\xf4", [0x48b3] = "\xf1\x6f",
  [0x48b4] = "\xc3\xf7", [0x48b8] = "\xc1\xf5", [0x48b9] = "\xab\x41",
  [0x48bb] = "\xb0\xa6", [0x48bc] = "\xd4\x47", [0x48bf] = "\xd8\x4c",
  [0x48c0] = "\xb3\xb6", [0x48c1] = "\xb6\xad", [0x48c2] = "\xdc\xa4",
  [0x48c3] = "\xdc\xa6", [0x48c4] = "\xb6\xaf", [0x48c5] = "\xb6\xae",
  [0x48c6] = "\xb6\xb0", [0x48c7] = "\xb6\xb1", [0x48c8] = "\xdc\xa5",
  [0x48c9] = "\xb9\x6e", [0x48ca] = "\xb9\x6f", [0x48cb] = "\xb9\x6d",
  [0x48cc] = "\xbb\xdb", [0x48cd] = "\xb9\x6c", [0x48ce] = "\xe0\xd5",
  [0x48d2] = "\xbb\xdc", [0x48d3] = "\xe8\xac", [0x48d4] = "\xec\x50",
  [0x48d5] = "\xc0\x4a", [0x48d6] = "\xc1\xf6", [0x48d7] = "\xf1\x70",
  [0x48d8] = "\xf1\x74", [0x48d9] = "\xc2\xf9", [0x48da] = "\xf1\x71",
  [0x48db] = "\xc2\xfa", [0x48dc] = "\xc2\xf8", [0x48dd] = "\xf1\x75",
  [0x48de] = "\xc2\xfb", [0x48df] = "\xf1\x73", [0x48e1] = "\xf3\x79",
  [0x48e2] = "\xc2\xf7", [0x48e3] = "\xc3\xf8", [0x48e5] = "\xf8\xcd",
  [0x48e8] = "\xab\x42", [0x48e9] = "\xb3\xb8", [0x48ea] = "\xb3\xb7",
  [0x48ef] = "\xb6\xb2", [0x48f0] = "\xdc\xa8", [0x48f1] = "\xdc\xa7",
  [0x48f2] = "\xb6\xb3", [0x48f5] = "\xe0\xd9", [0x48f6] = "\xb9\x73",
  [0x48f7] = "\xb9\x70", [0x48f8] = "\xe0\xd8", [0x48f9] = "\xb9\x72",
  [0x48fa] = "\xe0\xd6", [0x48fb] = "\xb9\x71", [0x48fd] = "\xe0\xd7",
  [0x48ff] = "\xe4\xbd", [0x4900] = "\xbb\xdd", [0x4902] = "\xe8\xaf",
  [0x4904] = "\xbe\x5d", [0x4905] = "\xe8\xad", [0x4906] = "\xbe\x5e",
  [0x4907] = "\xbe\x5f", [0x4908] = "\xe8\xae", [0x4909] = "\xbe\x60",
  [0x490b] = "\xec\x51", [0x490d] = "\xc0\x4e", [0x490e] = "\xc0\x4b",
  [0x490f] = "\xc0\x50", [0x4910] = "\xec\x53", [0x4911] = "\xc0\x4c",
  [0x4912] = "\xec\x52", [0x4913] = "\xc0\x4f", [0x4916] = "\xc0\x4d",
  [0x4918] = "\xee\xf9", [0x4919] = "\xee\xfb", [0x491c] = "\xc1\xf7",
  [0x491d] = "\xee\xfa", [0x491e] = "\xc1\xf8", [0x491f] = "\xee\xf8",
  [0x4920] = "\xee\xf7", [0x4922] = "\xf1\x77", [0x4923] = "\xf1\x76",
  [0x4924] = "\xc2\xfc", [0x4925] = "\xf1\x78", [0x4926] = "\xf3\x7e",
  [0x4927] = "\xc3\xfa", [0x4928] = "\xf3\x7d", [0x4929] = "\xf3\x7a",
  [0x492a] = "\xc3\xf9", [0x492b] = "\xf3\x7b", [0x492c] = "\xf3\x7c",
  [0x492e] = "\xf5\x48", [0x492f] = "\xf5\x49", [0x4930] = "\xc4\xc5",
  [0x4932] = "\xc5\x53", [0x4935] = "\xf6\x6e", [0x4938] = "\xc5\x51",
  [0x4939] = "\xc5\x52", [0x493a] = "\xf6\x6f", [0x493d] = "\xc5\xb4",
  [0x493e] = "\xc5\xb5", [0x493f] = "\xf7\x71", [0x4942] = "\xc6\x45",
  [0x4943] = "\xf8\xcf", [0x4944] = "\xc6\x47", [0x4946] = "\xf8\xce",
  [0x4947] = "\xf8\xd0", [0x4948] = "\xc6\x46", [0x4949] = "\xf9\x57",
  [0x494b] = "\xf9\xad", [0x4952] = "\xab\x43", [0x4956] = "\xb9\x74",
  [0x4958] = "\xe4\xbe", [0x495a] = "\xe8\xb0", [0x495b] = "\xc0\x51",
  [0x495c] = "\xc0\x52", [0x495e] = "\xab\x44", [0x4960] = "\xbe\x61",
  [0x4961] = "\xc3\xfb", [0x4962] = "\xad\xb1", [0x4966] = "\xc0\x53",
  [0x4968] = "\xc5\xe2", [0x4969] = "\xad\xb2", [0x496a] = "\xd8\x4d",
  [0x496c] = "\xdc\xa9", [0x496e] = "\xdc\xab", [0x4970] = "\xdc\xaa",
  [0x4972] = "\xe0\xdd", [0x4973] = "\xe0\xda", [0x4974] = "\xb9\x75",
  [0x4976] = "\xb9\x76", [0x4977] = "\xe0\xdb", [0x4978] = "\xe0\xdc",
  [0x497a] = "\xe4\xc0", [0x497b] = "\xe4\xc5", [0x497c] = "\xbb\xde",
  [0x497d] = "\xe4\xbf", [0x497e] = "\xe4\xc1", [0x497f] = "\xe4\xc8",
  [0x4980] = "\xe4\xc3", [0x4981] = "\xe4\xc7", [0x4982] = "\xe4\xc4",
  [0x4983] = "\xe4\xc2", [0x4984] = "\xe4\xc6", [0x4985] = "\xbb\xdf",
  [0x4988] = "\xe8\xb3", [0x498a] = "\xe8\xb1", [0x498b] = "\xbe\x63",
  [0x498d] = "\xbe\x62", [0x498e] = "\xe8\xb2", [0x498f] = "\xbe\x64",
  [0x4994] = "\xec\x56", [0x4997] = "\xec\x55", [0x4998] = "\xc0\x54",
  [0x4999] = "\xec\x54", [0x499a] = "\xee\xfc", [0x499c] = "\xee\xfe",
  [0x499d] = "\xef\x41", [0x499e] = "\xef\x40", [0x49a0] = "\xc1\xf9",
  [0x49a1] = "\xee\xfd", [0x49a2] = "\xf1\xa1", [0x49a3] = "\xc2\xfd",
  [0x49a4] = "\xf1\x7d", [0x49a5] = "\xf1\xa2", [0x49a6] = "\xc2\xfe",
  [0x49a8] = "\xf1\x7b", [0x49aa] = "\xf1\x7e", [0x49ab] = "\xf1\x7c",
  [0x49ac] = "\xf1\x79", [0x49ad] = "\xc3\x40", [0x49ae] = "\xf1\x7a",
  [0x49b3] = "\xf3\xa1", [0x49b6] = "\xf3\xa3", [0x49b7] = "\xf3\xa2",
  [0x49b9] = "\xf5\x4a", [0x49bb] = "\xf5\x4b", [0x49bf] = "\xf6\x70",
  [0x49c1] = "\xc5\xb7", [0x49c3] = "\xc5\xb6", [0x49c4] = "\xf8\x4f",
  [0x49c5] = "\xf8\x50", [0x49c6] = "\xc6\x48", [0x49c7] = "\xf8\xd1",
  [0x49c9] = "\xc6\x69", [0x49cb] = "\xad\xb3", [0x49cc] = "\xb6\xb4",
  [0x49cd] = "\xe4\xca", [0x49ce] = "\xe4\xc9", [0x49cf] = "\xe8\xb5",
  [0x49d0] = "\xe8\xb4", [0x49d3] = "\xc1\xfa", [0x49d4] = "\xef\x43",
  [0x49d5] = "\xef\x42", [0x49d6] = "\xf1\xa5", [0x49d7] = "\xf1\xa3",
  [0x49d8] = "\xf1\xa6", [0x49d9] = "\xf1\xa4", [0x49dc] = "\xc3\xfc",
  [0x49dd] = "\xf3\xa4", [0x49de] = "\xf3\xa5", [0x49df] = "\xf3\xa6",
  [0x49e1] = "\xf6\x71", [0x49e3] = "\xf7\x72", [0x49e5] = "\xf8\xd2",
  [0x49ed] = "\xad\xb4", [0x49f0] = "\xec\x57", [0x49f1] = "\xef\x44",
  [0x49f3] = "\xad\xb5", [0x49f6] = "\xbb\xe0", [0x49f8] = "\xec\x58",
  [0x49f9] = "\xc3\x41", [0x49fa] = "\xf1\xa7", [0x49fb] = "\xc3\xfd",
  [0x49fd] = "\xf5\x4c", [0x49fe] = "\xf5\x4d", [0x49ff] = "\xc5\x54",
  [0x4a00] = "\xf8\x51", [0x4a01] = "\xad\xb6", [0x4a02] = "\xb3\xbb",
  [0x4a03] = "\xb3\xbc", [0x4a04] = "\xd8\x4e", [0x4a05] = "\xb6\xb5",
  [0x4a06] = "\xb6\xb6", [0x4a07] = "\xdc\xac", [0x4a08] = "\xb6\xb7",
  [0x4a0a] = "\xb9\x7a", [0x4a0c] = "\xb9\x7c", [0x4a0d] = "\xe0\xdf",
  [0x4a0e] = "\xe0\xe0", [0x4a0f] = "\xe0\xde", [0x4a10] = "\xb9\x77",
  [0x4a11] = "\xb9\x78", [0x4a12] = "\xb9\x7b", [0x4a13] = "\xb9\x79",
  [0x4a16] = "\xe4\xcb", [0x4a17] = "\xbb\xe1", [0x4a18] = "\xbb\xe2",
  [0x4a1b] = "\xe8\xbc", [0x4a1c] = "\xbe\x67", [0x4a1d] = "\xe8\xb7",
  [0x4a1e] = "\xe8\xb6", [0x4a20] = "\xe8\xbb", [0x4a21] = "\xbe\x65",
  [0x4a24] = "\xc0\x5b", [0x4a26] = "\xe8\xb8", [0x4a27] = "\xe8\xbd",
  [0x4a28] = "\xe8\xba", [0x4a29] = "\xe8\xb9", [0x4a2b] = "\xbe\x66",
  [0x4a2d] = "\xc0\x59", [0x4a2f] = "\xec\x5a", [0x4a30] = "\xc0\x55",
  [0x4a32] = "\xec\x5b", [0x4a35] = "\xec\x59", [0x4a37] = "\xc0\x58",
  [0x4a38] = "\xc0\x56", [0x4a39] = "\xc0\x5a", [0x4a3b] = "\xc0\x57",
  [0x4a41] = "\xef\x45", [0x4a43] = "\xef\x4a", [0x4a44] = "\xef\x46",
  [0x4a45] = "\xef\x49", [0x4a46] = "\xc1\xfb", [0x4a48] = "\xed\xd4",
  [0x4a49] = "\xef\x48", [0x4a4a] = "\xef\x47", [0x4a4c] = "\xc3\x44",
  [0x4a4d] = "\xc3\x42", [0x4a4e] = "\xc3\x45", [0x4a4f] = "\xc3\x43",
  [0x4a50] = "\xf1\xa8", [0x4a51] = "\xf1\xa9", [0x4a52] = "\xf1\xaa",
  [0x4a53] = "\xc3\x46", [0x4a57] = "\xf3\xaa", [0x4a58] = "\xc4\x40",
  [0x4a59] = "\xf3\xa8", [0x4a5b] = "\xc4\x41", [0x4a5c] = "\xf3\xa7",
  [0x4a5d] = "\xf3\xa9", [0x4a5e] = "\xc3\xfe", [0x4a5f] = "\xf5\x51",
  [0x4a60] = "\xf5\x4e", [0x4a62] = "\xf5\x4f", [0x4a63] = "\xf5\x50",
  [0x4a64] = "\xf6\x72", [0x4a65] = "\xc5\x56", [0x4a67] = "\xc5\x55",
  [0x4a69] = "\xf7\x74", [0x4a6a] = "\xf7\x73", [0x4a6b] = "\xc5\xb8",
  [0x4a6f] = "\xc5\xe3", [0x4a70] = "\xc6\x49", [0x4a71] = "\xc6\x60",
  [0x4a72] = "\xf9\x58", [0x4a73] = "\xf9\xae", [0x4a74] = "\xf9\xaf",
  [0x4aa8] = "\xad\xb7", [0x4aa9] = "\xdc\xad", [0x4aac] = "\xe0\xe1",
  [0x4aad] = "\xe4\xcc", [0x4aae] = "\xe4\xcd", [0x4aaf] = "\xbb\xe3",
  [0x4ab1] = "\xbb\xe4", [0x4ab2] = "\xe8\xbe", [0x4ab3] = "\xbe\x68",
  [0x4ab6] = "\xc1\xfc", [0x4ab8] = "\xf1\xab", [0x4aba] = "\xc3\x47",
  [0x4abb] = "\xf3\xad", [0x4abc] = "\xc4\x42", [0x4abd] = "\xf3\xac",
  [0x4abe] = "\xf3\xae", [0x4abf] = "\xf3\xab", [0x4ac0] = "\xf6\x75",
  [0x4ac1] = "\xf5\x52", [0x4ac2] = "\xf5\x53", [0x4ac4] = "\xc4\xc6",
  [0x4ac6] = "\xf6\x74", [0x4ac9] = "\xf6\x73", [0x4acb] = "\xf7\x75",
  [0x4acc] = "\xf9\xb0", [0x4adb] = "\xad\xb8", [0x4adf] = "\xad\xb9",
  [0x4ae2] = "\xb0\xa7", [0x4ae3] = "\xd4\x48", [0x4ae5] = "\xd8\x4f",
  [0x4ae7] = "\xb6\xb8", [0x4ae9] = "\xb6\xbb", [0x4aea] = "\xb6\xb9",
  [0x4aeb] = "\xdc\xae", [0x4aed] = "\xb6\xbd", [0x4aef] = "\xb6\xba",
  [0x4af2] = "\xb6\xbc", [0x4af4] = "\xb9\x7e", [0x4af6] = "\xe0\xe2",
  [0x4af9] = "\xe0\xe3", [0x4afa] = "\xe8\xc0", [0x4afc] = "\xb9\x7d",
  [0x4afd] = "\xb9\xa1", [0x4afe] = "\xb9\xa2", [0x4b00] = "\xe4\xcf",
  [0x4b02] = "\xe4\xce", [0x4b03] = "\xbb\xe5", [0x4b05] = "\xbb\xe6",
  [0x4b07] = "\xe4\xd0", [0x4b08] = "\xe8\xbf", [0x4b09] = "\xbb\xe8",
  [0x4b0a] = "\xbe\x69", [0x4b0c] = "\xbb\xe7", [0x4b10] = "\xc0\x5c",
  [0x4b11] = "\xe8\xc1", [0x4b12] = "\xbe\x6b", [0x4b13] = "\xbe\x6a",
  [0x4b14] = "\xe8\xc2", [0x4b15] = "\xe8\xc5", [0x4b16] = "\xe8\xc3",
  [0x4b17] = "\xe8\xc4", [0x4b18] = "\xbe\x6c", [0x4b1a] = "\xc0\x61",
  [0x4b1b] = "\xc0\x5f", [0x4b1e] = "\xc0\x5e", [0x4b1f] = "\xec\x5d",
  [0x4b21] = "\xc0\x60", [0x4b24] = "\xec\x5c", [0x4b25] = "\xef\x4b",
  [0x4b27] = "\xec\x5e", [0x4b28] = "\xc0\x5d", [0x4b29] = "\xec\x5f",
  [0x4b2a] = "\xef\x4e", [0x4b2b] = "\xef\x4c", [0x4b2c] = "\xef\x4d",
  [0x4b2d] = "\xef\x52", [0x4b2e] = "\xc3\x4b", [0x4b2f] = "\xef\x51",
  [0x4b30] = "\xef\x54", [0x4b31] = "\xef\x53", [0x4b32] = "\xef\x50",
  [0x4b33] = "\xef\x4f", [0x4b35] = "\xc1\xfd", [0x4b3a] = "\xf1\xae",
  [0x4b3c] = "\xf1\xad", [0x4b3d] = "\xc3\x4a", [0x4b3e] = "\xc3\x48",
  [0x4b3f] = "\xc3\x49", [0x4b41] = "\xf1\xac", [0x4b43] = "\xf3\xb1",
  [0x4b45] = "\xc4\x43", [0x4b47] = "\xf3\xb0", [0x4b48] = "\xf3\xaf",
  [0x4b49] = "\xc4\x44", [0x4b4b] = "\xf5\x58", [0x4b4c] = "\xf5\x57",
  [0x4b4e] = "\xf5\x55", [0x4b50] = "\xf5\x54", [0x4b51] = "\xc4\xc8",
  [0x4b52] = "\xc4\xc7", [0x4b53] = "\xf5\x59", [0x4b54] = "\xf7\x76",
  [0x4b55] = "\xc5\xb9", [0x4b56] = "\xf6\x77", [0x4b57] = "\xc5\x57",
  [0x4b58] = "\xf6\x76", [0x4b59] = "\xf5\x56", [0x4b5b] = "\xf7\x77",
  [0x4b5c] = "\xc5\xe4", [0x4b5e] = "\xc6\x61", [0x4b5f] = "\xf9\x59",
  [0x4b61] = "\xf9\xb1", [0x4b96] = "\xad\xba", [0x4b97] = "\xd8\x50",
  [0x4b98] = "\xef\x55", [0x4b99] = "\xad\xbb", [0x4b9c] = "\xe4\xd2",
  [0x4b9d] = "\xe4\xd1", [0x4b9e] = "\xec\x60", [0x4ba1] = "\xef\x57",
  [0x4ba3] = "\xef\x56", [0x4ba5] = "\xc3\x4c", [0x4ba6] = "\xf3\xb2",
  [0x4ba7] = "\xf3\xb3", [0x4ba8] = "\xc4\xc9", [0x4bab] = "\xf9\xb2",
  [0x4bac] = "\xb0\xa8", [0x4bad] = "\xb6\xbf", [0x4bae] = "\xb6\xbe",
  [0x4baf] = "\xe0\xe4", [0x4bb0] = "\xe0\xe6", [0x4bb1] = "\xb9\xa4",
  [0x4bb2] = "\xe0\xe5", [0x4bb3] = "\xb9\xa3", [0x4bb4] = "\xb9\xa5",
  [0x4bb5] = "\xe0\xe7", [0x4bb9] = "\xe4\xd4", [0x4bba] = "\xe4\xd6",
  [0x4bbb] = "\xe4\xd5", [0x4bbd] = "\xe4\xd8", [0x4bc1] = "\xbb\xe9",
  [0x4bc2] = "\xe4\xd7", [0x4bc3] = "\xe4\xd3", [0x4bc7] = "\xe4\xd9",
  [0x4bc9] = "\xe8\xcc", [0x4bcb] = "\xe8\xcf", [0x4bcc] = "\xe8\xd1",
  [0x4bcd] = "\xe8\xc7", [0x4bce] = "\xe8\xcb", [0x4bcf] = "\xe8\xc8",
  [0x4bd0] = "\xbe\x6e", [0x4bd1] = "\xbe\x71", [0x4bd2] = "\xbe\x73",
  [0x4bd3] = "\xe8\xc9", [0x4bd4] = "\xe8\xca", [0x4bd5] = "\xbe\x72",
  [0x4bd6] = "\xe8\xcd", [0x4bd7] = "\xe8\xd0", [0x4bd8] = "\xe8\xce",
  [0x4bd9] = "\xbe\x74", [0x4bdb] = "\xbe\x70", [0x4bdc] = "\xe8\xc6",
  [0x4bdd] = "\xbe\x6d", [0x4bdf] = "\xbe\x6f", [0x4be2] = "\xc0\x63",
  [0x4be3] = "\xec\x66", [0x4be4] = "\xec\x64", [0x4be5] = "\xec\x63",
  [0x4be7] = "\xec\x69", [0x4be9] = "\xec\x68", [0x4bea] = "\xec\x67",
  [0x4bec] = "\xec\x62", [0x4bed] = "\xc0\x62", [0x4bee] = "\xec\x61",
  [0x4bf0] = "\xec\x65", [0x4bf1] = "\xc0\x64", [0x4bf4] = "\xef\x5a",
  [0x4bf6] = "\xef\x5e", [0x4bf7] = "\xef\x5b", [0x4bf8] = "\xef\x5d",
  [0x4bf9] = "\xef\x5c", [0x4bfa] = "\xef\x59", [0x4bfb] = "\xef\x5f",
  [0x4bfc] = "\xef\x62", [0x4bfd] = "\xef\x60", [0x4bfe] = "\xef\x61",
  [0x4bff] = "\xc2\x40", [0x4c01] = "\xc1\xfe", [0x4c02] = "\xef\x58",
  [0x4c03] = "\xef\x63", [0x4c04] = "\xf1\xb3", [0x4c05] = "\xf1\xb6",
  [0x4c06] = "\xf1\xb8", [0x4c07] = "\xf1\xb7", [0x4c09] = "\xf1\xb1",
  [0x4c0a] = "\xf1\xb5", [0x4c0b] = "\xf1\xb0", [0x4c0d] = "\xf1\xb2",
  [0x4c0e] = "\xc3\x4d", [0x4c0f] = "\xf1\xaf", [0x4c11] = "\xf1\xb4",
  [0x4c14] = "\xf3\xc0", [0x4c15] = "\xf3\xb5", [0x4c16] = "\xc4\x45",
  [0x4c19] = "\xc4\x46", [0x4c1a] = "\xf3\xb4", [0x4c1b] = "\xf3\xb9",
  [0x4c1c] = "\xf3\xbf", [0x4c1d] = "\xf3\xb7", [0x4c1e] = "\xf3\xbe",
  [0x4c20] = "\xf3\xbb", [0x4c22] = "\xf3\xba", [0x4c23] = "\xf3\xbd",
  [0x4c24] = "\xf3\xb8", [0x4c25] = "\xf3\xb6", [0x4c27] = "\xf3\xbc",
  [0x4c29] = "\xf5\x60", [0x4c2a] = "\xf5\x5e", [0x4c2b] = "\xc4\xca",
  [0x4c2c] = "\xf5\x5d", [0x4c2d] = "\xf5\x63", [0x4c2e] = "\xf5\x61",
  [0x4c30] = "\xc4\xcb", [0x4c31] = "\xf5\x5c", [0x4c32] = "\xf5\x5a",
  [0x4c34] = "\xf5\x5b", [0x4c35] = "\xc4\xcd", [0x4c36] = "\xf5\x5f",
  [0x4c37] = "\xc4\xcc", [0x4c38] = "\xf5\x62", [0x4c39] = "\xf6\x78",
  [0x4c3a] = "\xf6\x7e", [0x4c3d] = "\xf6\x79", [0x4c3e] = "\xc5\x5b",
  [0x4c3f] = "\xf6\xa1", [0x4c40] = "\xc5\x5a", [0x4c41] = "\xf6\x7d",
  [0x4c42] = "\xf6\x7c", [0x4c43] = "\xc5\x59", [0x4c44] = "\xf6\x7b",
  [0x4c45] = "\xc5\x58", [0x4c46] = "\xf6\x7a", [0x4c48] = "\xf7\x7d",
  [0x4c49] = "\xf7\xa1", [0x4c4a] = "\xf7\x7e", [0x4c4c] = "\xf7\x7b",
  [0x4c4d] = "\xc5\xbb", [0x4c4e] = "\xf7\x78", [0x4c4f] = "\xf7\x7c",
  [0x4c50] = "\xf7\xa3", [0x4c52] = "\xf7\xa2", [0x4c53] = "\xf7\x79",
  [0x4c54] = "\xf7\x7a", [0x4c55] = "\xc5\xba", [0x4c56] = "\xf8\x52",
  [0x4c57] = "\xc5\xe7", [0x4c59] = "\xf8\x53", [0x4c5a] = "\xc5\xe5",
  [0x4c5b] = "\xc5\xe6", [0x4c5e] = "\xf8\xd3", [0x4c5f] = "\xc6\x4a",
  [0x4c60] = "\xf9\x76", [0x4c62] = "\xc6\x6a", [0x4c64] = "\xf9\xb3",
  [0x4c65] = "\xc6\x6b", [0x4c66] = "\xf9\xb4", [0x4c67] = "\xf9\xb5",
  [0x4c68] = "\xf9\xc3", [0x4c69] = "\xf9\xc2", [0x4c6a] = "\xc6\x7a",
  [0x4c6b] = "\xf9\xcd", [0x4ca8] = "\xb0\xa9", [0x4cab] = "\xe0\xe9",
  [0x4cad] = "\xe0\xe8", [0x4caf] = "\xbb\xea", [0x4cb0] = "\xbb\xeb",
  [0x4cb1] = "\xe4\xda", [0x4cb3] = "\xe8\xd2", [0x4cb4] = "\xec\x6c",
  [0x4cb7] = "\xbe\x75", [0x4cb8] = "\xc0\x65", [0x4cb9] = "\xec\x6a",
  [0x4cbb] = "\xec\x6d", [0x4cbc] = "\xc0\x66", [0x4cbe] = "\xef\x64",
  [0x4cbf] = "\xec\x6b", [0x4cc0] = "\xf1\xb9", [0x4cc1] = "\xc3\x4e",
  [0x4cc2] = "\xf3\xc1", [0x4cc6] = "\xf5\x66", [0x4cc7] = "\xf5\x64",
  [0x4cca] = "\xf5\x65", [0x4ccd] = "\xf6\xa2", [0x4ccf] = "\xc5\x5c",
  [0x4cd0] = "\xf7\xa4", [0x4cd1] = "\xc5\xea", [0x4cd2] = "\xc5\xbc",
  [0x4cd3] = "\xc5\xe8", [0x4cd4] = "\xc5\xe9", [0x4cd5] = "\xf8\xd4",
  [0x4cd6] = "\xc6\x62", [0x4cd8] = "\xb0\xaa", [0x4cdc] = "\xf1\xba",
  [0x4cdf] = "\xd4\x49", [0x4ce1] = "\xb9\xa6", [0x4ce3] = "\xe4\xdb",
  [0x4ce6] = "\xbb\xec", [0x4ce7] = "\xe4\xdc", [0x4ceb] = "\xe8\xd4",
  [0x4cec] = "\xe8\xd3", [0x4ced] = "\xc0\x68", [0x4cee] = "\xbe\x76",
  [0x4cef] = "\xbe\x77", [0x4cf1] = "\xe8\xd7", [0x4cf2] = "\xe8\xd6",
  [0x4cf3] = "\xe8\xd5", [0x4cf6] = "\xec\x6e", [0x4cf7] = "\xec\x71",
  [0x4cf9] = "\xec\x70", [0x4cfa] = "\xec\x6f", [0x4cfb] = "\xc0\x67",
  [0x4cfc] = "\xef\x68", [0x4cfd] = "\xef\x66", [0x4cfe] = "\xef\x65",
  [0x4d01] = "\xef\x67", [0x4d03] = "\xc3\x4f", [0x4d04] = "\xf1\xbc",
  [0x4d05] = "\xf1\xbd", [0x4d06] = "\xc3\x50", [0x4d08] = "\xf1\xbb",
  [0x4d0a] = "\xf3\xc3", [0x4d0b] = "\xf3\xc2", [0x4d0c] = "\xf3\xc5",
  [0x4d0d] = "\xc4\x47", [0x4d0e] = "\xf3\xc4", [0x4d10] = "\xf5\x67",
  [0x4d11] = "\xf5\x69", [0x4d12] = "\xf5\x68", [0x4d15] = "\xf6\xa3",
  [0x4d16] = "\xf6\xa6", [0x4d17] = "\xf6\xa4", [0x4d18] = "\xf6\xa5",
  [0x4d19] = "\xf7\xa5", [0x4d1a] = "\xc5\xbd", [0x4d1e] = "\xf8\x54",
  [0x4d1f] = "\xf8\x55", [0x4d20] = "\xf8\x56", [0x4d22] = "\xc6\x4b",
  [0x4d23] = "\xc6\x63", [0x4d24] = "\xf9\xb6", [0x4d25] = "\xb0\xab",
  [0x4d27] = "\xbe\x78", [0x4d28] = "\xc0\x69", [0x4d29] = "\xf1\xbe",
  [0x4d2b] = "\xf7\xa6", [0x4d2e] = "\xf9\xc4", [0x4d2f] = "\xd4\x4a",
  [0x4d31] = "\xc6\x7b", [0x4d32] = "\xb0\xac", [0x4d33] = "\xec\x72",
  [0x4d35] = "\xf1\xbf", [0x4d37] = "\xf3\xc6", [0x4d3a] = "\xf6\xa7",
  [0x4d3b] = "\xf7\xa7", [0x4d3c] = "\xb0\xad", [0x4d3e] = "\xe4\xdd",
  [0x4d3f] = "\xe4\xde", [0x4d41] = "\xbb\xed", [0x4d42] = "\xbb\xee",
  [0x4d43] = "\xe8\xd9", [0x4d44] = "\xbe\x7a", [0x4d45] = "\xbe\x79",
  [0x4d46] = "\xe8\xd8", [0x4d48] = "\xef\x69", [0x4d4a] = "\xf1\xc0",
  [0x4d4b] = "\xf1\xc2", [0x4d4c] = "\xf1\xc1", [0x4d4d] = "\xc3\x53",
  [0x4d4e] = "\xc3\x52", [0x4d4f] = "\xc3\x51", [0x4d51] = "\xc5\x5e",
  [0x4d52] = "\xf6\xa8", [0x4d54] = "\xc5\x5d", [0x4d55] = "\xf7\xa9",
  [0x4d56] = "\xf7\xa8", [0x4d58] = "\xc6\x4c", [0x4d59] = "\xf8\xd5",
  [0x4d5a] = "\xb3\xbd", [0x4d5b] = "\xe0\xea", [0x4d5f] = "\xe4\xe1",
  [0x4d60] = "\xe4\xdf", [0x4d61] = "\xe4\xe0", [0x4d64] = "\xe8\xe2",
  [0x4d66] = "\xe8\xdd", [0x4d67] = "\xe8\xda", [0x4d68] = "\xe8\xe1",
  [0x4d6c] = "\xe8\xe3", [0x4d6f] = "\xbe\x7c", [0x4d70] = "\xe8\xe0",
  [0x4d71] = "\xe8\xdc", [0x4d74] = "\xe8\xdb", [0x4d75] = "\xe8\xdf",
  [0x4d76] = "\xe8\xde", [0x4d77] = "\xbe\x7b", [0x4d7a] = "\xec\x7d",
  [0x4d7b] = "\xec\x78", [0x4d7c] = "\xec\x76", [0x4d7d] = "\xec\xa1",
  [0x4d7e] = "\xec\x77", [0x4d80] = "\xec\x73", [0x4d82] = "\xec\x79",
  [0x4d85] = "\xec\x74", [0x4d86] = "\xef\x72", [0x4d87] = "\xec\x75",
  [0x4d88] = "\xec\xa2", [0x4d90] = "\xec\x7c", [0x4d91] = "\xc0\x6a",
  [0x4d92] = "\xec\x7b", [0x4d93] = "\xec\x7a", [0x4d95] = "\xec\x7e",
  [0x4d9a] = "\xef\x6a", [0x4d9b] = "\xef\x6d", [0x4d9e] = "\xef\x6c",
  [0x4da0] = "\xef\x74", [0x4da1] = "\xef\x6f", [0x4da2] = "\xef\x73",
  [0x4da4] = "\xef\x71", [0x4da5] = "\xef\x70", [0x4da6] = "\xef\x6e",
  [0x4da8] = "\xef\x6b", [0x4daa] = "\xc2\x43", [0x4dab] = "\xc2\x42",
  [0x4dad] = "\xc2\x44", [0x4dae] = "\xc2\x41", [0x4daf] = "\xef\x75",
  [0x4db5] = "\xf1\xc8", [0x4db6] = "\xf1\xcb", [0x4db8] = "\xf1\xc9",
  [0x4db9] = "\xf1\xcd", [0x4dbd] = "\xf1\xce", [0x4dbf] = "\xf1\xc6",
  [0x4dc0] = "\xc3\x58", [0x4dc1] = "\xf1\xc7", [0x4dc3] = "\xf1\xc5",
  [0x4dc4] = "\xf1\xcc", [0x4dc6] = "\xf1\xc4", [0x4dc7] = "\xf1\xc3",
  [0x4dc8] = "\xc3\x57", [0x4dc9] = "\xc3\x55", [0x4dca] = "\xc3\x54",
  [0x4dd3] = "\xf1\xca", [0x4dd4] = "\xf3\xcf", [0x4dd5] = "\xf3\xd5",
  [0x4dd6] = "\xc4\x4a", [0x4dd7] = "\xf3\xd0", [0x4dd9] = "\xf3\xd3",
  [0x4dda] = "\xf3\xd7", [0x4ddb] = "\xc4\x4b", [0x4ddc] = "\xf3\xd2",
  [0x4dde] = "\xf3\xca", [0x4de0] = "\xf3\xc9", [0x4de1] = "\xf3\xd6",
  [0x4de2] = "\xf3\xcd", [0x4de4] = "\xf3\xcb", [0x4de5] = "\xf3\xd4",
  [0x4de6] = "\xf3\xcc", [0x4de7] = "\xc4\x49", [0x4de8] = "\xc4\x48",
  [0x4dea] = "\xf3\xc7", [0x4deb] = "\xf3\xc8", [0x4dec] = "\xf3\xd1",
  [0x4df0] = "\xf3\xce", [0x4df7] = "\xf5\x6c", [0x4df8] = "\xf5\x6f",
  [0x4dfd] = "\xc3\x56", [0x4e05] = "\xf5\x6d", [0x4e06] = "\xf5\x73",
  [0x4e07] = "\xf5\x71", [0x4e08] = "\xf5\x6b", [0x4e09] = "\xf5\x76",
  [0x4e0b] = "\xf5\x6a", [0x4e0d] = "\xc4\xcf", [0x4e0e] = "\xf5\x72",
  [0x4e12] = "\xf5\x6e", [0x4e13] = "\xc4\xce", [0x4e14] = "\xf5\x75",
  [0x4e17] = "\xf5\x74", [0x4e1c] = "\xf6\xab", [0x4e1d] = "\xf6\xaa",
  [0x4e21] = "\xf6\xb1", [0x4e23] = "\xf6\xad", [0x4e24] = "\xf6\xb0",
  [0x4e25] = "\xc5\x60", [0x4e28] = "\xf6\xae", [0x4e29] = "\xf6\xaf",
  [0x4e2b] = "\xf6\xa9", [0x4e2c] = "\xf6\xac", [0x4e2d] = "\xc5\x5f",
  [0x4e31] = "\xc5\xbf", [0x4e32] = "\xf7\xb4", [0x4e33] = "\xf7\xaf",
  [0x4e34] = "\xf7\xb3", [0x4e36] = "\xf7\xb6", [0x4e37] = "\xf7\xb2",
  [0x4e39] = "\xf7\xae", [0x4e3b] = "\xc5\xc1", [0x4e3c] = "\xf7\xb1",
  [0x4e3d] = "\xf7\xb5", [0x4e3e] = "\xc5\xc0", [0x4e3f] = "\xf7\xac",
  [0x4e40] = "\xf5\x70", [0x4e41] = "\xf7\xb0", [0x4e44] = "\xf7\xad",
  [0x4e46] = "\xf7\xaa", [0x4e48] = "\xf7\xab", [0x4e49] = "\xc5\xbe",
  [0x4e4a] = "\xf8\x5a", [0x4e4b] = "\xf8\x5c", [0x4e4c] = "\xf8\x5f",
  [0x4e4d] = "\xf8\x5b", [0x4e4e] = "\xf8\x60", [0x4e50] = "\xf8\x59",
  [0x4e52] = "\xf8\x57", [0x4e54] = "\xc5\xeb", [0x4e55] = "\xf8\x5d",
  [0x4e56] = "\xc5\xed", [0x4e57] = "\xc5\xec", [0x4e58] = "\xf8\x58",
  [0x4e59] = "\xf8\x5e", [0x4e5e] = "\xf8\xda", [0x4e5f] = "\xc6\x4d",
  [0x4e60] = "\xf8\xdb", [0x4e62] = "\xf8\xd9", [0x4e63] = "\xf8\xd6",
  [0x4e66] = "\xf8\xd8", [0x4e67] = "\xf8\xd7", [0x4e68] = "\xf9\x5a",
  [0x4e6d] = "\xf9\x5c", [0x4e6e] = "\xf9\x5b", [0x4e71] = "\xf9\x79",
  [0x4e73] = "\xf9\x78", [0x4e74] = "\xf9\x77", [0x4e75] = "\xf9\x7a",
  [0x4e77] = "\xc6\x73", [0x4e78] = "\xc6\x74", [0x4e79] = "\xf9\xca",
  [0x4e7a] = "\xf9\xce", [0x4ee5] = "\xb3\xbe", [0x4ee6] = "\xdc\xaf",
  [0x4ee7] = "\xe0\xed", [0x4ee9] = "\xb9\xa7", [0x4eea] = "\xe0\xeb",
  [0x4eed] = "\xe0\xec", [0x4ef1] = "\xe4\xe2", [0x4ef2] = "\xe4\xe3",
  [0x4ef3] = "\xbb\xf1", [0x4ef4] = "\xbb\xef", [0x4ef5] = "\xe4\xe4",
  [0x4ef6] = "\xbb\xf0", [0x4ef7] = "\xe8\xe8", [0x4ef9] = "\xe8\xeb",
  [0x4efa] = "\xe8\xe5", [0x4efb] = "\xe8\xec", [0x4efc] = "\xe8\xe4",
  [0x4efd] = "\xe8\xe6", [0x4eff] = "\xe8\xe7", [0x4f00] = "\xe8\xea",
  [0x4f03] = "\xbe\xa1", [0x4f04] = "\xe8\xef", [0x4f05] = "\xe8\xee",
  [0x4f06] = "\xbe\x7d", [0x4f07] = "\xe8\xe9", [0x4f08] = "\xe8\xed",
  [0x4f09] = "\xbe\x7e", [0x4f10] = "\xec\xac", [0x4f12] = "\xc0\x6f",
  [0x4f14] = "\xec\xa7", [0x4f15] = "\xc0\x6b", [0x4f17] = "\xec\xa4",
  [0x4f18] = "\xec\xaa", [0x4f19] = "\xec\xad", [0x4f1b] = "\xc0\x70",
  [0x4f1d] = "\xec\xa9", [0x4f1e] = "\xec\xa6", [0x4f1f] = "\xec\xae",
  [0x4f20] = "\xec\xa5", [0x4f22] = "\xec\xab", [0x4f23] = "\xc0\x6c",
  [0x4f25] = "\xec\xa3", [0x4f26] = "\xc0\x6d", [0x4f28] = "\xc0\x6e",
  [0x4f29] = "\xec\xa8", [0x4f2d] = "\xef\xa9", [0x4f2e] = "\xef\x7a",
  [0x4f2f] = "\xef\x7b", [0x4f30] = "\xef\x7e", [0x4f31] = "\xef\x7c",
  [0x4f33] = "\xef\x76", [0x4f36] = "\xef\x79", [0x4f37] = "\xef\xa5",
  [0x4f38] = "\xef\x7d", [0x4f3b] = "\xc2\x45", [0x4f3d] = "\xef\xa7",
  [0x4f3e] = "\xef\xa4", [0x4f3f] = "\xc2\x46", [0x4f40] = "\xef\xa6",
  [0x4f41] = "\xef\x77", [0x4f42] = "\xef\xa2", [0x4f43] = "\xef\xa3",
  [0x4f45] = "\xef\xa1", [0x4f4a] = "\xf1\xd2", [0x4f4b] = "\xf1\xd4",
  [0x4f4c] = "\xf1\xd7", [0x4f4f] = "\xf1\xd1", [0x4f51] = "\xc3\x59",
  [0x4f52] = "\xf1\xd9", [0x4f53] = "\xf1\xd0", [0x4f54] = "\xf1\xda",
  [0x4f56] = "\xf1\xd6", [0x4f57] = "\xf1\xd8", [0x4f58] = "\xf1\xdc",
  [0x4f59] = "\xf1\xd5", [0x4f5a] = "\xf1\xdd", [0x4f5b] = "\xf1\xd3",
  [0x4f5c] = "\xf1\xcf", [0x4f5d] = "\xc3\x5a", [0x4f5f] = "\xf1\xdb",
  [0x4f60] = "\xc3\x5b", [0x4f61] = "\xc4\x4d", [0x4f67] = "\xef\x78",
  [0x4f68] = "\xf3\xf1", [0x4f69] = "\xf3\xe8", [0x4f6a] = "\xc4\x4f",
  [0x4f6b] = "\xf3\xe4", [0x4f6c] = "\xc4\x50", [0x4f6f] = "\xf3\xed",
  [0x4f70] = "\xf3\xe7", [0x4f71] = "\xf3\xdd", [0x4f72] = "\xc4\x4e",
  [0x4f73] = "\xf3\xea", [0x4f74] = "\xf3\xe5", [0x4f75] = "\xf3\xe6",
  [0x4f77] = "\xf3\xd8", [0x4f78] = "\xf3\xdf", [0x4f79] = "\xf3\xee",
  [0x4f7b] = "\xf3\xeb", [0x4f7d] = "\xf3\xe3", [0x4f7f] = "\xf3\xef",
  [0x4f80] = "\xf3\xde", [0x4f81] = "\xf3\xd9", [0x4f82] = "\xf3\xec",
  [0x4f84] = "\xf3\xdb", [0x4f85] = "\xf3\xe9", [0x4f86] = "\xf3\xe0",
  [0x4f87] = "\xf3\xf0", [0x4f88] = "\xf3\xdc", [0x4f89] = "\xc4\x4c",
  [0x4f8a] = "\xf3\xda", [0x4f8b] = "\xf3\xe1", [0x4f8c] = "\xf3\xe2",
  [0x4f90] = "\xf5\x7d", [0x4f92] = "\xf5\x7b", [0x4f94] = "\xf5\xa2",
  [0x4f96] = "\xf5\xae", [0x4f97] = "\xf5\xa5", [0x4f98] = "\xf5\x7c",
  [0x4f99] = "\xf5\x78", [0x4f9a] = "\xf5\xa7", [0x4f9b] = "\xf5\x7e",
  [0x4f9c] = "\xf5\xa3", [0x4f9d] = "\xf5\x7a", [0x4f9e] = "\xf5\xaa",
  [0x4f9f] = "\xf5\x77", [0x4fa0] = "\xf5\xa1", [0x4fa1] = "\xf5\xa6",
  [0x4fa2] = "\xf5\xa8", [0x4fa3] = "\xf5\xab", [0x4fa4] = "\xf5\x79",
  [0x4fa6] = "\xf5\xaf", [0x4fa7] = "\xf5\xb0", [0x4fa8] = "\xf5\xa9",
  [0x4fa9] = "\xf5\xad", [0x4faa] = "\xf5\xa4", [0x4fac] = "\xf6\xc1",
  [0x4fad] = "\xf6\xc4", [0x4faf] = "\xc5\x61", [0x4fb1] = "\xf6\xc3",
  [0x4fb2] = "\xf6\xc8", [0x4fb3] = "\xf6\xc6", [0x4fb4] = "\xc5\x62",
  [0x4fb5] = "\xf6\xbd", [0x4fb6] = "\xf6\xb3", [0x4fb7] = "\xf6\xb2",
  [0x4fb8] = "\xc5\x64", [0x4fb9] = "\xf6\xbf", [0x4fba] = "\xf6\xc0",
  [0x4fbb] = "\xf6\xbc", [0x4fbc] = "\xf6\xb4", [0x4fbe] = "\xf6\xb9",
  [0x4fbf] = "\xf5\xac", [0x4fc1] = "\xf6\xb5", [0x4fc2] = "\xc5\x63",
  [0x4fc3] = "\xf6\xbb", [0x4fc5] = "\xf6\xba", [0x4fc7] = "\xf6\xb6",
  [0x4fc8] = "\xf6\xc2", [0x4fca] = "\xf6\xb7", [0x4fcb] = "\xf7\xbb",
  [0x4fcc] = "\xf6\xc5", [0x4fcd] = "\xf6\xc7", [0x4fce] = "\xf6\xbe",
  [0x4fcf] = "\xf6\xb8", [0x4fd0] = "\xf7\xbc", [0x4fd1] = "\xf7\xbe",
  [0x4fd2] = "\xf7\xb8", [0x4fd3] = "\xc5\xc2", [0x4fd5] = "\xf7\xc5",
  [0x4fd6] = "\xf7\xc3", [0x4fd7] = "\xc5\xc3", [0x4fd8] = "\xf7\xc2",
  [0x4fd9] = "\xf7\xc1", [0x4fda] = "\xf7\xba", [0x4fdb] = "\xf7\xb7",
  [0x4fdc] = "\xf7\xbd", [0x4fdd] = "\xf7\xc6", [0x4fde] = "\xf7\xb9",
  [0x4fdf] = "\xf7\xbf", [0x4fe1] = "\xf8\x69", [0x4fe2] = "\xf8\x6e",
  [0x4fe3] = "\xf8\x64", [0x4fe4] = "\xf8\x67", [0x4fe5] = "\xc5\xee",
  [0x4fe6] = "\xf8\x6b", [0x4fe8] = "\xf8\x72", [0x4fe9] = "\xf7\xc0",
  [0x4feb] = "\xf8\x65", [0x4fec] = "\xf8\x6f", [0x4fed] = "\xf8\x73",
  [0x4fee] = "\xf8\x6a", [0x4fef] = "\xf8\x63", [0x4ff0] = "\xf8\x6d",
  [0x4ff2] = "\xf8\x6c", [0x4ff3] = "\xf8\x71", [0x4ff4] = "\xf8\x70",
  [0x4ff5] = "\xf7\xc4", [0x4ff6] = "\xf8\x68", [0x4ff7] = "\xf8\x62",
  [0x4ff8] = "\xf8\x66", [0x4ff9] = "\xc6\x4e", [0x4ffa] = "\xc6\x4f",
  [0x4ffb] = "\xf8\x61", [0x4ffd] = "\xf8\xe6", [0x4ffe] = "\xf8\xdd",
  [0x4fff] = "\xf8\xe5", [0x5000] = "\xf8\xe2", [0x5001] = "\xf8\xe3",
  [0x5002] = "\xf8\xdc", [0x5003] = "\xf8\xdf", [0x5004] = "\xf8\xe7",
  [0x5005] = "\xf8\xe1", [0x5006] = "\xf8\xe0", [0x5007] = "\xf8\xde",
  [0x5009] = "\xf8\xe4", [0x500b] = "\xf9\x5d", [0x500d] = "\xf9\x5e",
  [0x500f] = "\xf9\x60", [0x5010] = "\xf9\x5f", [0x5011] = "\xf9\x62",
  [0x5012] = "\xf9\x61", [0x5013] = "\xf9\x7c", [0x5014] = "\xf9\x7b",
  [0x5015] = "\xf9\xb7", [0x5017] = "\xf9\xb8", [0x5019] = "\xf9\xc5",
  [0x501a] = "\xc6\x78", [0x501b] = "\xc6\x7c", [0x501d] = "\xf9\xcf",
  [0x501e] = "\xc6\x7d", [0x5075] = "\xb3\xbf", [0x5079] = "\xc4\xd0",
  [0x507a] = "\xf6\xc9", [0x507c] = "\xc6\x50", [0x507d] = "\xc6\x51",
  [0x507f] = "\xb3\xc0", [0x5080] = "\xe0\xee", [0x5082] = "\xb9\xa8",
  [0x5083] = "\xe8\xf0", [0x5086] = "\xec\xb0", [0x5087] = "\xec\xb1",
  [0x5088] = "\xec\xaf", [0x5089] = "\xef\xab", [0x508a] = "\xef\xaa",
  [0x508b] = "\xc2\x47", [0x508c] = "\xf1\xdf", [0x508d] = "\xef\xac",
  [0x508e] = "\xf1\xde", [0x5091] = "\xf3\xf3", [0x5092] = "\xc4\x51",
  [0x5093] = "\xc4\x53", [0x5094] = "\xf3\xf2", [0x5097] = "\xc4\x52",
  [0x5099] = "\xf5\xb1", [0x509a] = "\xf5\xb3", [0x509b] = "\xf5\xb2",
  [0x509c] = "\xf6\xca", [0x509d] = "\xc5\x65", [0x509f] = "\xc5\xef",
  [0x50a0] = "\xf8\xe8", [0x50a1] = "\xf9\x63", [0x50a4] = "\xf9\xd2",
  [0x50a5] = "\xb3\xc1", [0x50a7] = "\xe4\xe5", [0x50a9] = "\xbe\xa2",
  [0x50ad] = "\xec\xb3", [0x50ae] = "\xec\xb2", [0x50b0] = "\xef\xad",
  [0x50b4] = "\xc4\x54", [0x50b5] = "\xc4\xd1", [0x50b6] = "\xf7\xc7",
  [0x50b7] = "\xf9\xcb", [0x50bb] = "\xb3\xc2", [0x50bc] = "\xbb\xf2",
  [0x50be] = "\xbe\xa3", [0x50c0] = "\xf3\xf4", [0x50c2] = "\xf8\x74",
  [0x50c3] = "\xb6\xc0", [0x50c8] = "\xef\xae", [0x50cc] = "\xc6\x64",
  [0x50cd] = "\xb6\xc1", [0x50ce] = "\xbe\xa4", [0x50cf] = "\xc2\x48",
  [0x50d0] = "\xf8\x75", [0x50d1] = "\xb6\xc2", [0x50d3] = "\xe8\xf1",
  [0x50d4] = "\xc0\x72", [0x50d5] = "\xec\xb4", [0x50d6] = "\xec\xb5",
  [0x50d8] = "\xc0\x71", [0x50da] = "\xef\xaf", [0x50db] = "\xc2\x4c",
  [0x50dc] = "\xc2\x4a", [0x50dd] = "\xc2\x4b", [0x50de] = "\xc2\x49",
  [0x50df] = "\xf1\xe0", [0x50e0] = "\xc3\x5c", [0x50e4] = "\xf5\xb5",
  [0x50e5] = "\xf5\xb4", [0x50e6] = "\xf5\xb7", [0x50e7] = "\xf5\xb6",
  [0x50e8] = "\xc4\xd2", [0x50eb] = "\xf6\xcb", [0x50ed] = "\xf6\xcd",
  [0x50ee] = "\xf6\xcc", [0x50ef] = "\xc5\x66", [0x50f0] = "\xf7\xc8",
  [0x50f2] = "\xf8\x76", [0x50f3] = "\xf8\x77", [0x50f4] = "\xc5\xf0",
  [0x50f5] = "\xf9\x64", [0x50f6] = "\xf9\x7d", [0x50f7] = "\xc6\x75",
  [0x50f9] = "\xdc\xb0", [0x50fa] = "\xec\xb6", [0x50fb] = "\xef\xb0",
  [0x50fc] = "\xf3\xf5", [0x50fd] = "\xe0\xef", [0x50ff] = "\xef\xb1",
  [0x5100] = "\xf1\xe2", [0x5101] = "\xf1\xe1", [0x5106] = "\xf8\x78",
  [0x5107] = "\xc6\x52", [0x5109] = "\xf9\x65", [0x510a] = "\xf9\x7e",
  [0x510e] = "\xb9\xa9", [0x510f] = "\xe8\xf2", [0x5110] = "\xe8\xf3",
  [0x5112] = "\xec\xb7", [0x5113] = "\xb9\xaa", [0x5115] = "\xc3\x5d",
  [0x5116] = "\xf1\xe3", [0x5118] = "\xf6\xcf", [0x5119] = "\xc5\x67",
  [0x511a] = "\xf6\xd0", [0x511b] = "\xf6\xce", [0x511c] = "\xf8\x79",
  [0x511e] = "\xf8\xe9", [0x5120] = "\xb9\xab", [0x5122] = "\xef\xb4",
  [0x5123] = "\xef\xb3", [0x5124] = "\xef\xb2", [0x5125] = "\xf1\xe4",
  [0x5128] = "\xf1\xe8", [0x5129] = "\xf1\xe7", [0x512a] = "\xf1\xe6",
  [0x512b] = "\xf1\xe5", [0x512c] = "\xc3\x5e", [0x512d] = "\xf3\xf6",
  [0x512e] = "\xf5\xb9", [0x512f] = "\xc4\xd3", [0x5130] = "\xf5\xb8",
  [0x5131] = "\xf6\xd1", [0x5132] = "\xf7\xcb", [0x5133] = "\xf7\xca",
  [0x5134] = "\xc5\xc4", [0x5135] = "\xf7\xc9", [0x5136] = "\xf8\x7c",
  [0x5137] = "\xf8\x7b", [0x5138] = "\xf8\x7a", [0x513b] = "\xbb\xf3",
  [0x513d] = "\xec\xb8", [0x513e] = "\xc2\x4d", [0x5140] = "\xf3\xf7",
  [0x5141] = "\xf3\xf8", [0x5142] = "\xf7\xcc", [0x5143] = "\xf8\x7d",
  [0x5146] = "\xf8\xea", [0x5147] = "\xf9\x66", [0x5148] = "\xf9\xb9",
  [0x5149] = "\xf9\xd4", [0x514a] = "\xbb\xf4", [0x514b] = "\xc2\x4e",
  [0x514c] = "\xf1\xe9", [0x514d] = "\xf3\xf9", [0x514e] = "\xf6\xd2",
  [0x514f] = "\xf8\x7e", [0x5152] = "\xbe\xa6", [0x5154] = "\xef\xb5",
  [0x5155] = "\xf1\xea", [0x5156] = "\xf3\xfa", [0x5157] = "\xf3\xfb",
  [0x5158] = "\xf3\xfc", [0x5159] = "\xf5\xbe", [0x515b] = "\xf5\xba",
  [0x515c] = "\xc5\x68", [0x515d] = "\xf5\xbd", [0x515e] = "\xf5\xbc",
  [0x515f] = "\xc4\xd4", [0x5160] = "\xf5\xbb", [0x5161] = "\xc4\xd6",
  [0x5163] = "\xc4\xd5", [0x5164] = "\xf6\xd4", [0x5165] = "\xf6\xd3",
  [0x5166] = "\xc5\x69", [0x5167] = "\xc5\x6a", [0x516a] = "\xc5\xc6",
  [0x516b] = "\xf7\xcd", [0x516c] = "\xc5\xc5", [0x516e] = "\xf8\xa3",
  [0x516f] = "\xf8\xa4", [0x5170] = "\xf8\xa2", [0x5171] = "\xf8\xa1",
  [0x5172] = "\xc6\x54", [0x5174] = "\xf8\xeb", [0x5175] = "\xf8\xec",
  [0x5176] = "\xf8\xed", [0x5177] = "\xc6\x53", [0x5178] = "\xf9\x67",
  [0x5179] = "\xf9\x6a", [0x517a] = "\xf9\x69", [0x517b] = "\xf9\x68",
  [0x517e] = "\xf9\xd3", [0x518d] = "\xc0\x73", [0x5190] = "\xc3\x65",
  [0x5191] = "\xf5\xbf", [0x5192] = "\xf6\xd5", [0x5194] = "\xc5\xc7",
  [0x5195] = "\xf7\xce", [0x5198] = "\xf9\xd5", [0x519c] = "\xc0\x74",
  [0x51a0] = "\xef\xb6", [0x51a2] = "\xf7\xcf", [0x51a4] = "\xf9\xa1",
};

/* Table of range U+F6B1 - U+F848. */
static const char from_ucs4_tab13[][2] =
{
  [0x0000] = "\xc6\xa1", [0x0001] = "\xc6\xa2", [0x0002] = "\xc6\xa3",
  [0x0003] = "\xc6\xa4", [0x0004] = "\xc6\xa5", [0x0005] = "\xc6\xa6",
  [0x0006] = "\xc6\xa7", [0x0007] = "\xc6\xa8", [0x0008] = "\xc6\xa9",
  [0x0009] = "\xc6\xaa", [0x000a] = "\xc6\xab", [0x000b] = "\xc6\xac",
  [0x000c] = "\xc6\xad", [0x000d] = "\xc6\xae", [0x000e] = "\xc6\xaf",
  [0x000f] = "\xc6\xb0", [0x0010] = "\xc6\xb1", [0x0011] = "\xc6\xb2",
  [0x0012] = "\xc6\xb3", [0x0013] = "\xc6\xb4", [0x0014] = "\xc6\xb5",
  [0x0015] = "\xc6\xb6", [0x0016] = "\xc6\xb7", [0x0017] = "\xc6\xb8",
  [0x0018] = "\xc6\xb9", [0x0019] = "\xc6\xba", [0x001a] = "\xc6\xbb",
  [0x001b] = "\xc6\xbc", [0x001c] = "\xc6\xbd", [0x001d] = "\xc6\xbe",
  [0x001e] = "\xc6\xbf", [0x001f] = "\xc6\xc0", [0x0020] = "\xc6\xc1",
  [0x0021] = "\xc6\xc2", [0x0022] = "\xc6\xc3", [0x0023] = "\xc6\xc4",
  [0x0024] = "\xc6\xc5", [0x0025] = "\xc6\xc6", [0x0026] = "\xc6\xc7",
  [0x0027] = "\xc6\xc8", [0x0028] = "\xc6\xc9", [0x0029] = "\xc6\xca",
  [0x002a] = "\xc6\xcb", [0x002b] = "\xc6\xcc", [0x002c] = "\xc6\xcd",
  [0x002d] = "\xc6\xce", [0x002e] = "\xc6\xcf", [0x002f] = "\xc6\xd0",
  [0x0030] = "\xc6\xd1", [0x0031] = "\xc6\xd2", [0x0032] = "\xc6\xd3",
  [0x0033] = "\xc6\xd4", [0x0034] = "\xc6\xd5", [0x0035] = "\xc6\xd6",
  [0x0036] = "\xc6\xd7", [0x0037] = "\xc6\xd8", [0x0038] = "\xc6\xd9",
  [0x0039] = "\xc6\xda", [0x003a] = "\xc6\xdb", [0x003b] = "\xc6\xdc",
  [0x003c] = "\xc6\xdd", [0x003d] = "\xc6\xde", [0x003e] = "\xc6\xdf",
  [0x003f] = "\xc6\xe0", [0x0040] = "\xc6\xe1", [0x0041] = "\xc6\xe2",
  [0x0042] = "\xc6\xe3", [0x0043] = "\xc6\xe4", [0x0044] = "\xc6\xe5",
  [0x0045] = "\xc6\xe6", [0x0046] = "\xc6\xe7", [0x0047] = "\xc6\xe8",
  [0x0048] = "\xc6\xe9", [0x0049] = "\xc6\xea", [0x004a] = "\xc6\xeb",
  [0x004b] = "\xc6\xec", [0x004c] = "\xc6\xed", [0x004d] = "\xc6\xee",
  [0x004e] = "\xc6\xef", [0x004f] = "\xc6\xf0", [0x0050] = "\xc6\xf1",
  [0x0051] = "\xc6\xf2", [0x0052] = "\xc6\xf3", [0x0053] = "\xc6\xf4",
  [0x0054] = "\xc6\xf5", [0x0055] = "\xc6\xf6", [0x0056] = "\xc6\xf7",
  [0x0057] = "\xc6\xf8", [0x0058] = "\xc6\xf9", [0x0059] = "\xc6\xfa",
  [0x005a] = "\xc6\xfb", [0x005b] = "\xc6\xfc", [0x005c] = "\xc6\xfd",
  [0x005d] = "\xc6\xfe", [0x005e] = "\xc7\x40", [0x005f] = "\xc7\x41",
  [0x0060] = "\xc7\x42", [0x0061] = "\xc7\x43", [0x0062] = "\xc7\x44",
  [0x0063] = "\xc7\x45", [0x0064] = "\xc7\x46", [0x0065] = "\xc7\x47",
  [0x0066] = "\xc7\x48", [0x0067] = "\xc7\x49", [0x0068] = "\xc7\x4a",
  [0x0069] = "\xc7\x4b", [0x006a] = "\xc7\x4c", [0x006b] = "\xc7\x4d",
  [0x006c] = "\xc7\x4e", [0x006d] = "\xc7\x4f", [0x006e] = "\xc7\x50",
  [0x006f] = "\xc7\x51", [0x0070] = "\xc7\x52", [0x0071] = "\xc7\x53",
  [0x0072] = "\xc7\x54", [0x0073] = "\xc7\x55", [0x0074] = "\xc7\x56",
  [0x0075] = "\xc7\x57", [0x0076] = "\xc7\x58", [0x0077] = "\xc7\x59",
  [0x0078] = "\xc7\x5a", [0x0079] = "\xc7\x5b", [0x007a] = "\xc7\x5c",
  [0x007b] = "\xc7\x5d", [0x007c] = "\xc7\x5e", [0x007d] = "\xc7\x5f",
  [0x007e] = "\xc7\x60", [0x007f] = "\xc7\x61", [0x0080] = "\xc7\x62",
  [0x0081] = "\xc7\x63", [0x0082] = "\xc7\x64", [0x0083] = "\xc7\x65",
  [0x0084] = "\xc7\x66", [0x0085] = "\xc7\x67", [0x0086] = "\xc7\x68",
  [0x0087] = "\xc7\x69", [0x0088] = "\xc7\x6a", [0x0089] = "\xc7\x6b",
  [0x008a] = "\xc7\x6c", [0x008b] = "\xc7\x6d", [0x008c] = "\xc7\x6e",
  [0x008d] = "\xc7\x6f", [0x008e] = "\xc7\x70", [0x008f] = "\xc7\x71",
  [0x0090] = "\xc7\x72", [0x0091] = "\xc7\x73", [0x0092] = "\xc7\x74",
  [0x0093] = "\xc7\x75", [0x0094] = "\xc7\x76", [0x0095] = "\xc7\x77",
  [0x0096] = "\xc7\x78", [0x0097] = "\xc7\x79", [0x0098] = "\xc7\x7a",
  [0x0099] = "\xc7\x7b", [0x009a] = "\xc7\x7c", [0x009b] = "\xc7\x7d",
  [0x009c] = "\xc7\x7e", [0x009d] = "\xc7\xa1", [0x009e] = "\xc7\xa2",
  [0x009f] = "\xc7\xa3", [0x00a0] = "\xc7\xa4", [0x00a1] = "\xc7\xa5",
  [0x00a2] = "\xc7\xa6", [0x00a3] = "\xc7\xa7", [0x00a4] = "\xc7\xa8",
  [0x00a5] = "\xc7\xa9", [0x00a6] = "\xc7\xaa", [0x00a7] = "\xc7\xab",
  [0x00a8] = "\xc7\xac", [0x00a9] = "\xc7\xad", [0x00aa] = "\xc7\xae",
  [0x00ab] = "\xc7\xaf", [0x00ac] = "\xc7\xb0", [0x00ad] = "\xc7\xb1",
  [0x00ae] = "\xc7\xb2", [0x00af] = "\xc7\xb3", [0x00b0] = "\xc7\xb4",
  [0x00b1] = "\xc7\xb5", [0x00b2] = "\xc7\xb6", [0x00b3] = "\xc7\xb7",
  [0x00b4] = "\xc7\xb8", [0x00b5] = "\xc7\xb9", [0x00b6] = "\xc7\xba",
  [0x00b7] = "\xc7\xbb", [0x00b8] = "\xc7\xbc", [0x00b9] = "\xc7\xbd",
  [0x00ba] = "\xc7\xbe", [0x00bb] = "\xc7\xbf", [0x00bc] = "\xc7\xc0",
  [0x00bd] = "\xc7\xc1", [0x00be] = "\xc7\xc2", [0x00bf] = "\xc7\xc3",
  [0x00c0] = "\xc7\xc4", [0x00c1] = "\xc7\xc5", [0x00c2] = "\xc7\xc6",
  [0x00c3] = "\xc7\xc7", [0x00c4] = "\xc7\xc8", [0x00c5] = "\xc7\xc9",
  [0x00c6] = "\xc7\xca", [0x00c7] = "\xc7\xcb", [0x00c8] = "\xc7\xcc",
  [0x00c9] = "\xc7\xcd", [0x00ca] = "\xc7\xce", [0x00cb] = "\xc7\xcf",
  [0x00cc] = "\xc7\xd0", [0x00cd] = "\xc7\xd1", [0x00ce] = "\xc7\xd2",
  [0x00cf] = "\xc7\xd3", [0x00d0] = "\xc7\xd4", [0x00d1] = "\xc7\xd5",
  [0x00d2] = "\xc7\xd6", [0x00d3] = "\xc7\xd7", [0x00d4] = "\xc7\xd8",
  [0x00d5] = "\xc7\xd9", [0x00d6] = "\xc7\xda", [0x00d7] = "\xc7\xdb",
  [0x00d8] = "\xc7\xdc", [0x00d9] = "\xc7\xdd", [0x00da] = "\xc7\xde",
  [0x00db] = "\xc7\xdf", [0x00dc] = "\xc7\xe0", [0x00dd] = "\xc7\xe1",
  [0x00de] = "\xc7\xe2", [0x00df] = "\xc7\xe3", [0x00e0] = "\xc7\xe4",
  [0x00e1] = "\xc7\xe5", [0x00e2] = "\xc7\xe6", [0x00e3] = "\xc7\xe7",
  [0x00e4] = "\xc7\xe8", [0x00e5] = "\xc7\xe9", [0x00e6] = "\xc7\xea",
  [0x00e7] = "\xc7\xeb", [0x00e8] = "\xc7\xec", [0x00e9] = "\xc7\xed",
  [0x00ea] = "\xc7\xee", [0x00eb] = "\xc7\xef", [0x00ec] = "\xc7\xf0",
  [0x00ed] = "\xc7\xf1", [0x00ee] = "\xc7\xf2", [0x00ef] = "\xc7\xf3",
  [0x00f0] = "\xc7\xf4", [0x00f1] = "\xc7\xf5", [0x00f2] = "\xc7\xf6",
  [0x00f3] = "\xc7\xf7", [0x00f4] = "\xc7\xf8", [0x00f5] = "\xc7\xf9",
  [0x00f6] = "\xc7\xfa", [0x00f7] = "\xc7\xfb", [0x00f8] = "\xc7\xfc",
  [0x00f9] = "\xc7\xfd", [0x00fa] = "\xc7\xfe", [0x00fb] = "\xc8\x40",
  [0x00fc] = "\xc8\x41", [0x00fd] = "\xc8\x42", [0x00fe] = "\xc8\x43",
  [0x00ff] = "\xc8\x44", [0x0100] = "\xc8\x45", [0x0101] = "\xc8\x46",
  [0x0102] = "\xc8\x47", [0x0103] = "\xc8\x48", [0x0104] = "\xc8\x49",
  [0x0105] = "\xc8\x4a", [0x0106] = "\xc8\x4b", [0x0107] = "\xc8\x4c",
  [0x0108] = "\xc8\x4d", [0x0109] = "\xc8\x4e", [0x010a] = "\xc8\x4f",
  [0x010b] = "\xc8\x50", [0x010c] = "\xc8\x51", [0x010d] = "\xc8\x52",
  [0x010e] = "\xc8\x53", [0x010f] = "\xc8\x54", [0x0110] = "\xc8\x55",
  [0x0111] = "\xc8\x56", [0x0112] = "\xc8\x57", [0x0113] = "\xc8\x58",
  [0x0114] = "\xc8\x59", [0x0115] = "\xc8\x5a", [0x0116] = "\xc8\x5b",
  [0x0117] = "\xc8\x5c", [0x0118] = "\xc8\x5d", [0x0119] = "\xc8\x5e",
  [0x011a] = "\xc8\x5f", [0x011b] = "\xc8\x60", [0x011c] = "\xc8\x61",
  [0x011d] = "\xc8\x62", [0x011e] = "\xc8\x63", [0x011f] = "\xc8\x64",
  [0x0120] = "\xc8\x65", [0x0121] = "\xc8\x66", [0x0122] = "\xc8\x67",
  [0x0123] = "\xc8\x68", [0x0124] = "\xc8\x69", [0x0125] = "\xc8\x6a",
  [0x0126] = "\xc8\x6b", [0x0127] = "\xc8\x6c", [0x0128] = "\xc8\x6d",
  [0x0129] = "\xc8\x6e", [0x012a] = "\xc8\x6f", [0x012b] = "\xc8\x70",
  [0x012c] = "\xc8\x71", [0x012d] = "\xc8\x72", [0x012e] = "\xc8\x73",
  [0x012f] = "\xc8\x74", [0x0130] = "\xc8\x75", [0x0131] = "\xc8\x76",
  [0x0132] = "\xc8\x77", [0x0133] = "\xc8\x78", [0x0134] = "\xc8\x79",
  [0x0135] = "\xc8\x7a", [0x0136] = "\xc8\x7b", [0x0137] = "\xc8\x7c",
  [0x0138] = "\xc8\x7d", [0x0139] = "\xc8\x7e", [0x013a] = "\xc8\xa1",
  [0x013b] = "\xc8\xa2", [0x013c] = "\xc8\xa3", [0x013d] = "\xc8\xa4",
  [0x013e] = "\xc8\xa5", [0x013f] = "\xc8\xa6", [0x0140] = "\xc8\xa7",
  [0x0141] = "\xc8\xa8", [0x0142] = "\xc8\xa9", [0x0143] = "\xc8\xaa",
  [0x0144] = "\xc8\xab", [0x0145] = "\xc8\xac", [0x0146] = "\xc8\xad",
  [0x0147] = "\xc8\xae", [0x0148] = "\xc8\xaf", [0x0149] = "\xc8\xb0",
  [0x014a] = "\xc8\xb1", [0x014b] = "\xc8\xb2", [0x014c] = "\xc8\xb3",
  [0x014d] = "\xc8\xb4", [0x014e] = "\xc8\xb5", [0x014f] = "\xc8\xb6",
  [0x0150] = "\xc8\xb7", [0x0151] = "\xc8\xb8", [0x0152] = "\xc8\xb9",
  [0x0153] = "\xc8\xba", [0x0154] = "\xc8\xbb", [0x0155] = "\xc8\xbc",
  [0x0156] = "\xc8\xbd", [0x0157] = "\xc8\xbe", [0x0158] = "\xc8\xbf",
  [0x0159] = "\xc8\xc0", [0x015a] = "\xc8\xc1", [0x015b] = "\xc8\xc2",
  [0x015c] = "\xc8\xc3", [0x015d] = "\xc8\xc4", [0x015e] = "\xc8\xc5",
  [0x015f] = "\xc8\xc6", [0x0160] = "\xc8\xc7", [0x0161] = "\xc8\xc8",
  [0x0162] = "\xc8\xc9", [0x0163] = "\xc8\xca", [0x0164] = "\xc8\xcb",
  [0x0165] = "\xc8\xcc", [0x0166] = "\xc8\xcd", [0x0167] = "\xc8\xce",
  [0x0168] = "\xc8\xcf", [0x0169] = "\xc8\xd0", [0x016a] = "\xc8\xd1",
  [0x016b] = "\xc8\xd2", [0x016c] = "\xc8\xd3", [0x016d] = "\xc8\xd4",
  [0x016e] = "\xc8\xd5", [0x016f] = "\xc8\xd6", [0x0170] = "\xc8\xd7",
  [0x0171] = "\xc8\xd8", [0x0172] = "\xc8\xd9", [0x0173] = "\xc8\xda",
  [0x0174] = "\xc8\xdb", [0x0175] = "\xc8\xdc", [0x0176] = "\xc8\xdd",
  [0x0177] = "\xc8\xde", [0x0178] = "\xc8\xdf", [0x0179] = "\xc8\xe0",
  [0x017a] = "\xc8\xe1", [0x017b] = "\xc8\xe2", [0x017c] = "\xc8\xe3",
  [0x017d] = "\xc8\xe4", [0x017e] = "\xc8\xe5", [0x017f] = "\xc8\xe6",
  [0x0180] = "\xc8\xe7", [0x0181] = "\xc8\xe8", [0x0182] = "\xc8\xe9",
  [0x0183] = "\xc8\xea", [0x0184] = "\xc8\xeb", [0x0185] = "\xc8\xec",
  [0x0186] = "\xc8\xed", [0x0187] = "\xc8\xee", [0x0188] = "\xc8\xef",
  [0x0189] = "\xc8\xf0", [0x018a] = "\xc8\xf1", [0x018b] = "\xc8\xf2",
  [0x018c] = "\xc8\xf3", [0x018d] = "\xc8\xf4", [0x018e] = "\xc8\xf5",
  [0x018f] = "\xc8\xf6", [0x0190] = "\xc8\xf7", [0x0191] = "\xc8\xf8",
  [0x0192] = "\xc8\xf9", [0x0193] = "\xc8\xfa", [0x0194] = "\xc8\xfb",
  [0x0195] = "\xc8\xfc", [0x0196] = "\xc8\xfd", [0x0197] = "\xc8\xfe",
};

/* Table of range U+FE30 - U+FE6B. */
static const char from_ucs4_tab14[][2] =
{
  [0x0000] = "\xa1\x4a", [0x0001] = "\xa1\x57", [0x0003] = "\xa1\x59",
  [0x0004] = "\xa1\x5b", [0x0005] = "\xa1\x5f", [0x0006] = "\xa1\x60",
  [0x0007] = "\xa1\x63", [0x0008] = "\xa1\x64", [0x0009] = "\xa1\x67",
  [0x000a] = "\xa1\x68", [0x000b] = "\xa1\x6b", [0x000c] = "\xa1\x6c",
  [0x000d] = "\xa1\x6f", [0x000e] = "\xa1\x70", [0x000f] = "\xa1\x73",
  [0x0010] = "\xa1\x74", [0x0011] = "\xa1\x77", [0x0012] = "\xa1\x78",
  [0x0013] = "\xa1\x7b", [0x0014] = "\xa1\x7c", [0x0019] = "\xa1\xc6",
  [0x001a] = "\xa1\xc7", [0x001b] = "\xa1\xca", [0x001c] = "\xa1\xcb",
  [0x001d] = "\xa1\xc8", [0x001e] = "\xa1\xc9", [0x001f] = "\xa1\x5c",
  [0x0020] = "\xa1\x4d", [0x0021] = "\xa1\x4e", [0x0022] = "\xa1\x4f",
  [0x0024] = "\xa1\x51", [0x0025] = "\xa1\x52", [0x0026] = "\xa1\x53",
  [0x0027] = "\xa1\x54", [0x0029] = "\xa1\x7d", [0x002a] = "\xa1\x7e",
  [0x002b] = "\xa1\xa1", [0x002c] = "\xa1\xa2", [0x002d] = "\xa1\xa3",
  [0x002e] = "\xa1\xa4", [0x002f] = "\xa1\xcc", [0x0030] = "\xa1\xcd",
  [0x0031] = "\xa1\xce", [0x0032] = "\xa1\xde", [0x0033] = "\xa1\xdf",
  [0x0034] = "\xa1\xe0", [0x0035] = "\xa1\xe1", [0x0036] = "\xa1\xe2",
  [0x0038] = "\xa2\x42", [0x0039] = "\xa2\x4c", [0x003a] = "\xa2\x4d",
  [0x003b] = "\xa2\x4e",
};

/* Table of range U+FF01 - U+FFE5. */
static const char from_ucs4_tab15[][2] =
{
  [0x0000] = "\xa1\x49", [0x0002] = "\xa1\xad", [0x0003] = "\xa2\x43",
  [0x0004] = "\xa2\x48", [0x0005] = "\xa1\xae", [0x0007] = "\xa1\x5d",
  [0x0008] = "\xa1\x5e", [0x0009] = "\xa1\xaf", [0x000a] = "\xa1\xcf",
  [0x000b] = "\xa1\x41", [0x000c] = "\xa1\xd0", [0x000d] = "\xa1\x44",
  [0x000e] = "\xa1\xfe", [0x000f] = "\xa2\xaf", [0x0010] = "\xa2\xb0",
  [0x0011] = "\xa2\xb1", [0x0012] = "\xa2\xb2", [0x0013] = "\xa2\xb3",
  [0x0014] = "\xa2\xb4", [0x0015] = "\xa2\xb5", [0x0016] = "\xa2\xb6",
  [0x0017] = "\xa2\xb7", [0x0018] = "\xa2\xb8", [0x0019] = "\xa1\x47",
  [0x001a] = "\xa1\x46", [0x001b] = "\xa1\xd5", [0x001c] = "\xa1\xd7",
  [0x001d] = "\xa1\xd6", [0x001e] = "\xa1\x48", [0x001f] = "\xa2\x49",
  [0x0020] = "\xa2\xcf", [0x0021] = "\xa2\xd0", [0x0022] = "\xa2\xd1",
  [0x0023] = "\xa2\xd2", [0x0024] = "\xa2\xd3", [0x0025] = "\xa2\xd4",
  [0x0026] = "\xa2\xd5", [0x0027] = "\xa2\xd6", [0x0028] = "\xa2\xd7",
  [0x0029] = "\xa2\xd8", [0x002a] = "\xa2\xd9", [0x002b] = "\xa2\xda",
  [0x002c] = "\xa2\xdb", [0x002d] = "\xa2\xdc", [0x002e] = "\xa2\xdd",
  [0x002f] = "\xa2\xde", [0x0030] = "\xa2\xdf", [0x0031] = "\xa2\xe0",
  [0x0032] = "\xa2\xe1", [0x0033] = "\xa2\xe2", [0x0034] = "\xa2\xe3",
  [0x0035] = "\xa2\xe4", [0x0036] = "\xa2\xe5", [0x0037] = "\xa2\xe6",
  [0x0038] = "\xa2\xe7", [0x0039] = "\xa2\xe8", [0x003b] = "\xa2\x40",
  [0x003e] = "\xa1\xc4", [0x0040] = "\xa2\xe9", [0x0041] = "\xa2\xea",
  [0x0042] = "\xa2\xeb", [0x0043] = "\xa2\xec", [0x0044] = "\xa2\xed",
  [0x0045] = "\xa2\xee", [0x0046] = "\xa2\xef", [0x0047] = "\xa2\xf0",
  [0x0048] = "\xa2\xf1", [0x0049] = "\xa2\xf2", [0x004a] = "\xa2\xf3",
  [0x004b] = "\xa2\xf4", [0x004c] = "\xa2\xf5", [0x004d] = "\xa2\xf6",
  [0x004e] = "\xa2\xf7", [0x004f] = "\xa2\xf8", [0x0050] = "\xa2\xf9",
  [0x0051] = "\xa2\xfa", [0x0052] = "\xa2\xfb", [0x0053] = "\xa2\xfc",
  [0x0054] = "\xa2\xfd", [0x0055] = "\xa2\xfe", [0x0056] = "\xa3\x40",
  [0x0057] = "\xa3\x41", [0x0058] = "\xa3\x42", [0x0059] = "\xa3\x43",
  [0x005a] = "\xa1\x61", [0x005b] = "\xa1\x55", [0x005c] = "\xa1\x62",
  [0x005d] = "\xa1\xe3", [0x00df] = "\xa2\x46", [0x00e0] = "\xa2\x47",
  [0x00e2] = "\xa1\xc3", [0x00e4] = "\xa2\x44",
};




/* Definitions used in the body of the `gconv' function.  */
#define CHARSET_NAME		"BIG5//"
#define FROM_LOOP		from_big5
#define TO_LOOP			to_big5
#define DEFINE_INIT		1
#define DEFINE_FINI		1
#define MIN_NEEDED_FROM		1
#define MAX_NEEDED_FROM		2
#define MIN_NEEDED_TO		4


/* First define the conversion function from Big5 to UCS4.  */
#define MIN_NEEDED_INPUT	MIN_NEEDED_FROM
#define MAX_NEEDED_INPUT	MAX_NEEDED_FROM
#define MIN_NEEDED_OUTPUT	MIN_NEEDED_TO
#define LOOPFCT			FROM_LOOP
#define BODY \
  {									      \
    uint32_t ch = *inptr;						      \
									      \
    if (ch >= 0xa1 && ch <= 0xf9)					      \
      {									      \
	/* Two-byte character.  First test whether the next character	      \
	   is also available.  */					      \
	uint32_t ch2;							      \
	int idx;							      \
									      \
	if (__builtin_expect (inptr + 1 >= inend, 0))			      \
	  {								      \
	    /* The second character is not available.  */		      \
	    result = __GCONV_INCOMPLETE_INPUT;				      \
	    break;							      \
	  }								      \
									      \
	idx = (ch - 0xa1) * 157;					      \
	ch2 = inptr[1];							      \
	/* See whether the second byte is in the correct range.  */	      \
	if (ch2 >= 0x40 && ch2 <= 0x7e)					      \
	  idx += ch2 - 0x40;						      \
	else if (__builtin_expect (ch2 >= 0xa1, 1)			      \
		 && __builtin_expect (ch2 <= 0xfe, 1))			      \
	  idx += 0x3f + (ch2 - 0xa1);					      \
	else								      \
	  {								      \
	    /* This is illegal.  */					      \
	    STANDARD_FROM_LOOP_ERR_HANDLER (1);				      \
	  }								      \
									      \
	/* Get the value from the table.  */				      \
	ch = big5_to_ucs[idx];						      \
									      \
	/* Is this character defined?  */				      \
	if (__builtin_expect (ch == 0, 0))				      \
	  {								      \
	    /* This is an illegal character.  */			      \
	    STANDARD_FROM_LOOP_ERR_HANDLER (2);				      \
	  }								      \
									      \
	inptr += 2;							      \
      }									      \
    else if (ch <= 0x80)						      \
      ++inptr;								      \
    else								      \
      {									      \
	/* This is illegal.  */						      \
	STANDARD_FROM_LOOP_ERR_HANDLER (1);				      \
      }									      \
									      \
    put32 (outptr, ch);							      \
    outptr += 4;							      \
  }
#define LOOP_NEED_FLAGS
#include <iconv/loop.c>


/* Next, define the other direction.  */
#define MIN_NEEDED_INPUT	MIN_NEEDED_TO
#define MIN_NEEDED_OUTPUT	MIN_NEEDED_FROM
#define MAX_NEEDED_OUTPUT	MAX_NEEDED_FROM
#define LOOPFCT			TO_LOOP
#define BODY \
  {									      \
    uint32_t ch = get32 (inptr);					      \
    const char *cp;							      \
									      \
    if (__builtin_expect (ch >= (sizeof (from_ucs4_tab1)		      \
				 / sizeof (from_ucs4_tab1[0])), 0))	      \
      switch (ch)							      \
	{								      \
        case 0x2c7 ... 0x2d9:						      \
	  cp = from_ucs4_tab2[ch - 0x2c7];				      \
	  break;							      \
	case 0x391 ... 0x3c9:						      \
	  cp = from_ucs4_tab3[ch - 0x391];				      \
	  break;							      \
	case 0x2013 ... 0x203b:						      \
	  cp = from_ucs4_tab4[ch - 0x2013];				      \
	  break;							      \
	case 0x20ac:							      \
	  cp = "\xa3\xe1";						      \
	  break;							      \
	case 0x2103:							      \
	  cp = "\xa2\x4a";						      \
	  break;							      \
	case 0x2105:							      \
	  cp = "\xa1\xc1";						      \
	  break;							      \
	case 0x2109:							      \
	  cp = "\xa2\x4b";						      \
	  break;							      \
	case 0x2160 ... 0x2169:						      \
	  cp = from_ucs4_tab5[ch - 0x2160];				      \
	  break;							      \
	case 0x2190 ... 0x2199:						      \
	  cp = from_ucs4_tab6[ch - 0x2190];				      \
	  break;							      \
	case 0x2215 ...	0x2267:						      \
	  cp = from_ucs4_tab7[ch - 0x2215];				      \
	  break;							      \
	case 0x2295:							      \
	  cp = "\xa1\xf2";						      \
	  break;							      \
	case 0x2299:							      \
	  cp = "\xa1\xf3";						      \
	  break;							      \
	case 0x22a5:							      \
	  cp = "\xa1\xe6";						      \
	  break;							      \
	case 0x22bf:							      \
	  cp = "\xa1\xe9";						      \
	  break;							      \
	case 0x2500 ... 0x2642:						      \
	  cp = from_ucs4_tab8[ch - 0x2500];				      \
	  break;							      \
	case 0x3000 ... 0x3029:						      \
	  cp = from_ucs4_tab9[ch - 0x3000];				      \
	  break;							      \
	case 0x3105 ... 0x3129:						      \
	  cp = from_ucs4_tab10[ch - 0x3105];				      \
	  break;							      \
	case 0x32a3:							      \
	  cp = "\xa1\xc0";						      \
	  break;							      \
	case 0x338e ... 0x33d5:						      \
	  cp = from_ucs4_tab11[ch - 0x338e];				      \
	  break;							      \
	case 0x4e00 ... 0x9fa4:						      \
	  cp = from_ucs4_tab12[ch - 0x4e00];				      \
	  break;							      \
	case 0xf6b1 ... 0xf848:						      \
	  cp = from_ucs4_tab13[ch - 0xf6b1];				      \
	  break;							      \
	case 0xfa0c:							      \
	  cp = "\xc9\x4a";						      \
	  break;							      \
	case 0xfa0d:							      \
	  cp = "\xdd\xfc";						      \
	  break;							      \
	case 0xfe30 ... 0xfe6b:						      \
	  cp = from_ucs4_tab14[ch - 0xfe30];				      \
	  break;							      \
	case 0xff01 ... 0xffe5:						      \
	  cp = from_ucs4_tab15[ch - 0xff01];				      \
	  break;							      \
	default:							      \
	  UNICODE_TAG_HANDLER (ch, 4);					      \
	  /* Illegal character.  */					      \
	  cp = "";							      \
	  break;							      \
	}								      \
    else								      \
      cp = from_ucs4_tab1[ch];						      \
									      \
    if (__builtin_expect (cp[0], '\1') == '\0' && ch != 0)		      \
      {									      \
	/* Illegal character.  */					      \
	STANDARD_TO_LOOP_ERR_HANDLER (4);				      \
      }									      \
    else								      \
      {									      \
	/* See whether there is enough room for the second byte we write.  */ \
	if (__builtin_expect (cp[1], '\1') != '\0'			      \
	    && __builtin_expect (outptr + 1 >= outend, 0))		      \
	  {								      \
	    /* We have not enough room.  */				      \
	    result = __GCONV_FULL_OUTPUT;				      \
	    break;							      \
	  }								      \
									      \
	*outptr++ = cp[0];						      \
	if (cp[1] != '\0')						      \
	  *outptr++ = cp[1];						      \
      }									      \
									      \
    inptr += 4;								      \
  }
#define LOOP_NEED_FLAGS
#include <iconv/loop.c>


/* Now define the toplevel functions.  */
#include <iconv/skeleton.c>
