/*
  This file is part of TALER
  (C) 2019, 2020, 2021, 2024, 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-merchant-httpd_private-get-products.c
 * @brief implement GET /products
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-get-products.h"


/**
 * Add product details to our JSON array.
 *
 * @param cls a `json_t *` JSON array to build
 * @param product_serial serial (row) number of the product in the database
 * @param product_id ID of the product
 */
static void
add_product (void *cls,
             uint64_t product_serial,
             const char *product_id)
{
  json_t *pa = cls;

  GNUNET_assert (0 ==
                 json_array_append_new (
                   pa,
                   GNUNET_JSON_PACK (
                     GNUNET_JSON_pack_uint64 ("product_serial",
                                              product_serial),
                     GNUNET_JSON_pack_string ("product_id",
                                              product_id))));
}


/**
 * Transforms an (untrusted) input filter into a Postgresql LIKE filter.
 * Escapes "%" and "_" in the @a input and adds "%" at the beginning
 * and the end to turn the @a input into a suitable Postgresql argument.
 *
 * @param input text to turn into a substring match expression, or NULL
 * @return NULL if @a input was NULL, otherwise transformed @a input
 */
static char *
tr (const char *input)
{
  char *out;
  size_t slen;
  size_t wpos;

  if (NULL == input)
    return NULL;
  slen = strlen (input);
  out = GNUNET_malloc (slen * 2 + 3);
  wpos = 0;
  out[wpos++] = '%';
  for (size_t i = 0; i<slen; i++)
  {
    char c = input[i];

    if ( (c == '%') ||
         (c == '_') )
      out[wpos++] = '\\';
    out[wpos++] = c;
  }
  out[wpos++] = '%';
  GNUNET_assert (wpos < slen * 2 + 3);
  return out;
}


MHD_RESULT
TMH_private_get_products (const struct TMH_RequestHandler *rh,
                          struct MHD_Connection *connection,
                          struct TMH_HandlerContext *hc)
{
  json_t *pa;
  enum GNUNET_DB_QueryStatus qs;
  char *category_filter;
  char *name_filter;
  char *description_filter;
  int64_t limit;
  uint64_t offset;

  limit = 20; /* default */
  TALER_MHD_parse_request_snumber (connection,
                                   "limit",
                                   &limit);
  if (limit < 0)
    offset = INT64_MAX;
  else
    offset = 0;
  TALER_MHD_parse_request_number (connection,
                                  "offset",
                                  &offset);
  category_filter = tr (MHD_lookup_connection_value (connection,
                                                     MHD_GET_ARGUMENT_KIND,
                                                     "category_filter"));
  name_filter = tr (MHD_lookup_connection_value (connection,
                                                 MHD_GET_ARGUMENT_KIND,
                                                 "name_filter"));
  description_filter = tr (MHD_lookup_connection_value (connection,
                                                        MHD_GET_ARGUMENT_KIND,
                                                        "description_filter"));
  pa = json_array ();
  GNUNET_assert (NULL != pa);
  qs = TMH_db->lookup_products (TMH_db->cls,
                                hc->instance->settings.id,
                                offset,
                                limit,
                                category_filter,
                                name_filter,
                                description_filter,
                                &add_product,
                                pa);
  GNUNET_free (category_filter);
  GNUNET_free (name_filter);
  GNUNET_free (description_filter);
  if (0 > qs)
  {
    GNUNET_break (0);
    json_decref (pa);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       NULL);
  }
  return TALER_MHD_REPLY_JSON_PACK (connection,
                                    MHD_HTTP_OK,
                                    GNUNET_JSON_pack_array_steal ("products",
                                                                  pa));
}


/* end of taler-merchant-httpd_private-get-products.c */
