package gnu.crypto.key.srp6;

// ----------------------------------------------------------------------------
// $Id: $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import gnu.crypto.Registry;
import gnu.crypto.hash.IMessageDigest;
import gnu.crypto.key.KeyAgreementException;
import gnu.crypto.key.IncomingMessage;
import gnu.crypto.key.OutgoingMessage;
import gnu.crypto.sasl.srp.SRPAuthInfoProvider;
import gnu.crypto.sasl.srp.SRPRegistry;
import gnu.crypto.util.Util;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>A variation of the SRP-6 protocol as used in the SASL-SRP mechanism, for
 * the Host (server side).</p>
 *
 * <p>In this alternative, the exchange goes as follows:</p>
 * <pre>
 *    C -> S:  I                      (identifies self)
 *    S -> C:  N, g, s, B = 3v + g^b  (sends salt, b = random number)
 *    C -> S:  A = g^a                (a = random number)
 * </pre>
 *
 * <p>All elements are computed the same way as in the standard version.</p>
 *
 * <p>Reference:</p>
 * <ol>
 *    <li><a href="http://www.ietf.org/internet-drafts/draft-burdis-cat-srp-sasl-09.txt">
 *    Secure Remote Password Authentication Mechanism</a><br>
 *    K. Burdis, R. Naffah.</li>
 *    <li><a href="http://srp.stanford.edu/design.html">SRP Protocol Design</a><br>
 *    Thomas J. Wu.</li>
 * </ol>
 *
 * @version $Revision: $
 */
public class SRP6SaslServer extends SRP6KeyAgreement {

   // Constants and variables
   // -------------------------------------------------------------------------

   /** The user's ephemeral key pair. */
   private KeyPair hostKeyPair;

   /** The SRP password database. */
   private SRPAuthInfoProvider passwordDB;

   // Constructor(s)
   // -------------------------------------------------------------------------

   // default 0-arguments constructor

   // Class methods
   // -------------------------------------------------------------------------

   // Instance methods
   // -------------------------------------------------------------------------

   // implementation of abstract methods in base class ------------------------

   protected void engineInit(Map attributes) throws KeyAgreementException {
      rnd = (SecureRandom) attributes.get(SOURCE_OF_RANDOMNESS);

      mda = (IMessageDigest) attributes.get(HASH_FUNCTION);
      if (mda == null) {
         throw new KeyAgreementException("missing hash function");
      }

      passwordDB = (SRPAuthInfoProvider) attributes.get(HOST_PASSWORD_DB);
      if (passwordDB == null) {
         throw new KeyAgreementException("missing SRP password database");
      }
   }

   protected OutgoingMessage engineProcessMessage(IncomingMessage in)
   throws KeyAgreementException {
      switch (step) {
      case 0:  return sendParameters(in);
      case 1:  return computeSharedSecret(in);
      default: throw new IllegalStateException("unexpected state");
      }
   }

   protected void engineReset() {
      hostKeyPair = null;
      super.engineReset();
   }

   // own methods -------------------------------------------------------------

   private OutgoingMessage sendParameters(IncomingMessage in)
   throws KeyAgreementException {
      String I = in.readString();

      // get s and v for user identified by I
      // ----------------------------------------------------------------------
      Map credentials = null;
      try {
         Map userID = new HashMap();
         userID.put(Registry.SASL_USERNAME, I);
         userID.put(SRPRegistry.MD_NAME_FIELD, mda.name());
         credentials = passwordDB.lookup(userID);
      } catch (IOException x) {
         throw new KeyAgreementException("computeSharedSecret()", x);
      }

      BigInteger s = new BigInteger(1, Util.fromBase64(
            (String) credentials.get(SRPRegistry.SALT_FIELD)));
      BigInteger v = new BigInteger(1, Util.fromBase64(
            (String) credentials.get(SRPRegistry.USER_VERIFIER_FIELD)));

      Map configuration = null;
      try {
         String mode = (String) credentials.get(SRPRegistry.CONFIG_NDX_FIELD);
         configuration = passwordDB.getConfiguration(mode);
      } catch (IOException x) {
         throw new KeyAgreementException("computeSharedSecret()", x);
      }

      N = new BigInteger(1, Util.fromBase64(
            (String) configuration.get(SRPRegistry.SHARED_MODULUS)));
      g = new BigInteger(1, Util.fromBase64(
            (String) configuration.get(SRPRegistry.FIELD_GENERATOR)));
      // ----------------------------------------------------------------------

      // generate an ephemeral keypair
      SRPKeyPairGenerator kpg = new SRPKeyPairGenerator();
      Map attributes = new HashMap();
      if (rnd != null) {
         attributes.put(SRPKeyPairGenerator.SOURCE_OF_RANDOMNESS, rnd);
      }
      attributes.put(SRPKeyPairGenerator.SHARED_MODULUS, N);
      attributes.put(SRPKeyPairGenerator.GENERATOR,      g);
      attributes.put(SRPKeyPairGenerator.USER_VERIFIER,  v);
      kpg.setup(attributes);
      hostKeyPair = kpg.generate();

      BigInteger B = ((SRPPublicKey) hostKeyPair.getPublic()).getY();

      OutgoingMessage result = new OutgoingMessage();
      result.writeMPI(N);
      result.writeMPI(g);
      result.writeMPI(s);
      result.writeMPI(B);

      return result;
   }

   private OutgoingMessage computeSharedSecret(IncomingMessage in)
   throws KeyAgreementException {
      BigInteger A = in.readMPI();

      BigInteger B = ((SRPPublicKey) hostKeyPair.getPublic()).getY();
      BigInteger u = uValue(A, B); // u = H(A | B)

      // compute S = (Av^u) ^ b
      BigInteger b = ((SRPPrivateKey) hostKeyPair.getPrivate()).getX();
      BigInteger v = ((SRPPrivateKey) hostKeyPair.getPrivate()).getV();
      BigInteger S = A.multiply(v.modPow(u, N)).modPow(b, N);

      byte[] sBytes = Util.trim(S);
      mda.update(sBytes, 0, sBytes.length);
      K = new BigInteger(1, mda.digest());

      complete = true;
      return null;
   }
}
