/* iobuf.c  -  file handling
 *	Copyright (C) 1998, 1999, 2000, 2001 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h> 
#include <io.h>
#include <windows.h>

#include "openpgp.h"

#define my_fileno(a)  (a)
#define my_fopen_ro(a,b) fd_cache_open ((a),(b)) 
#define my_fopen(a,b) direct_open ((a),(b))  
typedef HANDLE FILEP_OR_FD;
#define INVALID_FP  ((HANDLE)-1)
#define FILEP_OR_FD_FOR_STDIN  (GetStdHandle (STD_INPUT_HANDLE))
#define FILEP_OR_FD_FOR_STDOUT (GetStdHandle (STD_OUTPUT_HANDLE))
#undef USE_SETMODE 
typedef struct {
    FILEP_OR_FD  fp;	   /* open file handle */
    int keep_open;
    int no_cache;
    int eof_seen;
    int  print_only_name; /* flags indicating that fname is not a real file*/
    char fname[1]; /* name of the file */
} file_filter_ctx_t ;

struct close_cache_s { 
    struct close_cache_s *next;
    FILEP_OR_FD fp;
    char fname[1];
};
typedef struct close_cache_s *CLOSE_CACHE;
static CLOSE_CACHE close_cache;

/* The first partial length header block must be of size 512
 * to make it easier (and efficienter) we use a min. block size of 512
 * for all chunks (but the last one) */
#define OP_MIN_PARTIAL_CHUNK	  512
#define OP_MIN_PARTIAL_CHUNK_2POW 9

typedef struct {
    int use;
    size_t size;
    size_t count;
    int partial;  /* 1 = partial header, 2 in last partial packet */
    char *buffer;    /* used for partial header */
    size_t buflen;   /* used size of buffer */
    int first_c;     /* of partial header (which is > 0)*/
    int eof;
} block_filter_ctx_t;

static int special_names_enabled;

static int underflow(gpg_iobuf_t a);
static int translate_file_handle ( int fd, int for_write );

/*
 * Invalidate (i.e. close) a cached iobuf
 */
static void
fd_cache_invalidate (const char *fname)
{
    CLOSE_CACHE cc;
    
    assert (fname);
    
    for (cc=close_cache; cc; cc = cc->next ) {
        if ( cc->fp != INVALID_FP && !strcmp (cc->fname, fname) ) {          
            CloseHandle (cc->fp);
            cc->fp = INVALID_FP;
        }
    }
}

static FILEP_OR_FD
direct_open (const char *fname, const char *mode)
{
    unsigned long da, cd, sm;
    HANDLE hfile;
    
    /* Note, that we do not handle all mode combinations */
    
    /* According to the ReactOS source it seems that open() of the
     * standard MSW32 crt does open the file in share mode which is
     * something new for MS applications ;-)
     */
    if ( strchr (mode, '+') ) {
        fd_cache_invalidate (fname);
        da = GENERIC_READ|GENERIC_WRITE;
        cd = OPEN_EXISTING;
        sm = FILE_SHARE_READ | FILE_SHARE_WRITE;
    }
    else if ( strchr (mode, 'w') ) {
        fd_cache_invalidate (fname);
        da = GENERIC_WRITE;
        cd = CREATE_ALWAYS;
        sm = FILE_SHARE_WRITE;
    }
    else {
        da = GENERIC_READ;
        cd = OPEN_EXISTING;
        sm = FILE_SHARE_READ;
    }
    
    hfile = CreateFile (fname, da, sm, NULL, cd, FILE_ATTRIBUTE_NORMAL, NULL);
    return hfile;
}


/*
 * Instead of closing an FD we keep it open and cache it for later reuse 
 * Note that this caching strategy only works if the process does not chdir.
 */
static void
fd_cache_close (const char *fname, FILEP_OR_FD fp)
{
    CLOSE_CACHE cc;
    
    assert (fp);
    if ( !fname || !*fname ) {
        CloseHandle (fp);
	return;
    }
    /* try to reuse a slot */
    for (cc=close_cache; cc; cc = cc->next ) {
        if ( cc->fp == INVALID_FP && !strcmp (cc->fname, fname) ) {
            cc->fp = fp;            
            return;
        }
    }
    /* add a new one */    
    cc = calloc (1, sizeof *cc + strlen (fname));
    strcpy (cc->fname, fname);
    cc->fp = fp;
    cc->next = close_cache;
    close_cache = cc;
}

/*
 * Do an direct_open on FNAME but first try to reuse one from the fd_cache
 */
static FILEP_OR_FD
fd_cache_open (const char *fname, const char *mode)
{
    CLOSE_CACHE cc;
    
    assert (fname);
    for (cc=close_cache; cc; cc = cc->next ) {
        if ( cc->fp != INVALID_FP && !strcmp (cc->fname, fname) ) {
            FILEP_OR_FD fp = cc->fp;
            cc->fp = INVALID_FP;                      
            if (SetFilePointer (fp, 0, NULL, FILE_BEGIN) == 0xffffffff ) {
                printf ("rewind file failed on handle %p: ec=%d\n",
                        fp, (int)GetLastError () );
                fp = INVALID_FP;
            }          
            return fp;
        }
    }    
    return direct_open (fname, mode);
}


/****************
 * Read data from a file into buf which has an allocated length of *LEN.
 * return the number of read bytes in *LEN. OPAQUE is the FILE * of
 * the stream. A is not used.
 * control may be:
 * IOBUFCTRL_INIT: called just before the function is linked into the
 *		   list of function. This can be used to prepare internal
 *		   data structures of the function.
 * IOBUFCTRL_FREE: called just before the function is removed from the
 *		    list of functions and can be used to release internal
 *		    data structures or close a file etc.
 * IOBUFCTRL_UNDERFLOW: called by iobuf_underflow to fill the buffer
 *		    with new stuff. *RET_LEN is the available size of the
 *		    buffer, and should be set to the number of bytes
 *		    which were put into the buffer. The function
 *		    returns 0 to indicate success, -1 on EOF and
 *		    G10ERR_xxxxx for other errors.
 *
 * IOBUFCTRL_FLUSH: called by iobuf_flush() to write out the collected stuff.
 *		    *RET_LAN is the number of bytes in BUF.
 *
 * IOBUFCTRL_CANCEL: send to all filters on behalf of iobuf_cancel.  The
 *		    filter may take appropriate action on this message.
 */
static int
file_filter(void *opaque, int control, gpg_iobuf_t chain, byte *buf, size_t *ret_len)
{
    file_filter_ctx_t *a = opaque;
    FILEP_OR_FD f = a->fp;
    size_t size = *ret_len;
    size_t nbytes = 0;
    int rc = 0;
    
    if( control == IOBUFCTRL_UNDERFLOW ) {
	assert( size ); /* need a buffer */
	if ( a->eof_seen) {
	    rc = -1;		
	    *ret_len = 0;	
	}
	else {          
            unsigned long nread;
            
            nbytes = 0;
            if ( !ReadFile ( f, buf, size, &nread, NULL ) ) {
                int ec = (int)GetLastError ();
                if ( ec != ERROR_BROKEN_PIPE ) {
                    printf("%s: read error: ec=%d\n", a->fname, ec);
                    rc = G10ERR_READ_FILE;
                }
            }
            else if ( !nread ) {
                a->eof_seen = 1;
                rc = -1;
            }
            else {
                nbytes = nread;
            }          
	    *ret_len = nbytes;
	}
    }
    else if( control == IOBUFCTRL_FLUSH ) {
	if( size ) {          
            byte *p = buf;
            unsigned long n;
            
            nbytes = size;
            do {
                if ( size && !WriteFile ( f,  p, nbytes, &n, NULL) ) {
                    int ec = (int)GetLastError ();
                    printf("%s: write error: ec=%d\n", a->fname, ec);
                    rc = G10ERR_WRITE_FILE;
                    break;
                }
                p += n;
                nbytes -= n;
            } while ( nbytes );
            nbytes = p - buf;          
	}
	*ret_len = nbytes;
    }
    else if ( control == IOBUFCTRL_INIT ) {
        a->eof_seen = 0;
        a->keep_open = 0;
        a->no_cache = 0;
    }
    else if ( control == IOBUFCTRL_DESC ) {
	*(char**)buf = "file_filter(fd)";
    }
    else if ( control == IOBUFCTRL_FREE ) {      
        if ( f != FILEP_OR_FD_FOR_STDIN && f != FILEP_OR_FD_FOR_STDOUT ) {	    
            if (!a->keep_open)
                fd_cache_close (a->no_cache?NULL:a->fname, f);
        }      
	safe_free (a); /* we can free our context now */
    }
    return rc;
}

/****************
 * This is used to implement the block write mode.
 * Block reading is done on a byte by byte basis in readbyte(),
 * without a filter
 */
static int
block_filter(void *opaque, int control, gpg_iobuf_t chain, byte *buf, size_t *ret_len)
{
    block_filter_ctx_t *a = opaque;
    size_t size = *ret_len;
    int c, needed, rc = 0;
    char *p;
    
    if( control == IOBUFCTRL_UNDERFLOW ) {
	size_t n=0;
        
	p = buf;
	assert( size ); /* need a buffer */
	if( a->eof ) /* don't read any further */
	    rc = -1;
	while( !rc && size ) {
	    if( !a->size ) { /* get the length bytes */
		if( a->partial == 2 ) {
		    a->eof = 1;
		    if( !n )
			rc = -1;
		    break;
		}
		else if( a->partial ) {
		    /* These OpenPGP introduced huffman like encoded length
		     * bytes are really a mess :-( */
		    if( a->first_c ) {
			c = a->first_c;
			a->first_c = 0;
		    }
		    else if( (c = gpg_iobuf_get(chain)) == -1 ) {
			printf("block_filter: 1st length byte missing\n");
			rc = G10ERR_READ_FILE;
			break;
		    }
		    if( c < 192 ) {
			a->size = c;
			a->partial = 2;
			if( !a->size ) {
			    a->eof = 1;
			    if( !n )
				rc = -1;
			    break;
			}
		    }
		    else if( c < 224 ) {
			a->size = (c - 192) * 256;
			if( (c = gpg_iobuf_get(chain)) == -1 ) {
			    printf("block_filter: 2nd length byte missing\n");
			    rc = G10ERR_READ_FILE;
			    break;
			}
			a->size += c + 192;
			a->partial = 2;
			if( !a->size ) {
			    a->eof = 1;
			    if( !n )
				rc = -1;
			    break;
			}
		    }
		    else if( c == 255 ) {
			a->size  = gpg_iobuf_get(chain) << 24;
			a->size |= gpg_iobuf_get(chain) << 16;
			a->size |= gpg_iobuf_get(chain) << 8;
			if( (c = gpg_iobuf_get(chain)) == -1 ) {
			    printf("block_filter: invalid 4 byte length\n");
			    rc = G10ERR_READ_FILE;
			    break;
			}
			a->size |= c;
		    }
		    else { /* next partial body length */
			a->size = 1 << (c & 0x1f);
		    }
                    /*	printf("partial: ctx=%p c=%02x size=%u\n", a, c, a->size);*/
		}
		else { /* the gnupg partial length scheme - much better :-) */
		    c = gpg_iobuf_get(chain);
		    a->size = c << 8;
		    c = gpg_iobuf_get(chain);
		    a->size |= c;
		    if( c == -1 ) {
			printf("block_filter: error reading length info\n");
			rc = G10ERR_READ_FILE;
		    }
		    if( !a->size ) {
			a->eof = 1;
			if( !n )
			    rc = -1;
			break;
		    }
		}
	    }
            
	    while( !rc && size && a->size ) {
		needed = size < a->size ? size : a->size;
		c = gpg_iobuf_read( chain, p, needed );
		if( c < needed ) {
		    if( c == -1 ) c = 0;
		    printf("block_filter %p: read error (size=%lu,a->size=%lu)\n",
                           a,  (ulong)size+c, (ulong)a->size+c);
		    rc = G10ERR_READ_FILE;
		}
		else {
		    size -= c;
		    a->size -= c;
		    p += c;
		    n += c;
		}
	    }
	}
	*ret_len = n;
    }
    else if( control == IOBUFCTRL_FLUSH ) {
	if( a->partial ) { /* the complicated openpgp scheme */
	    size_t blen, n, nbytes = size + a->buflen;
            
	    assert( a->buflen <= OP_MIN_PARTIAL_CHUNK );
	    if( nbytes < OP_MIN_PARTIAL_CHUNK ) {
		/* not enough to write a partial block out; so we store it*/
		if( !a->buffer )
		    a->buffer = malloc( OP_MIN_PARTIAL_CHUNK );
		memcpy( a->buffer + a->buflen, buf, size );
		a->buflen += size;
	    }
	    else { /* okay, we can write out something */
		/* do this in a loop to use the most efficient block lengths */
		p = buf;
		do {
		    /* find the best matching block length - this is limited
		     * by the size of the internal buffering */
		    for( blen=OP_MIN_PARTIAL_CHUNK*2,
                             c=OP_MIN_PARTIAL_CHUNK_2POW+1; blen <= nbytes;
                         blen *=2, c++ )
			;
		    blen /= 2; c--;
		    /* write the partial length header */
		    assert( c <= 0x1f ); /*;-)*/
		    c |= 0xe0;
		    gpg_iobuf_put( chain, c );
		    if( (n=a->buflen) ) { /* write stuff from the buffer */
			assert( n == OP_MIN_PARTIAL_CHUNK);
			if( gpg_iobuf_write(chain, a->buffer, n ) )
			    rc = G10ERR_WRITE_FILE;
			a->buflen = 0;
			nbytes -= n;
		    }
		    if( (n = nbytes) > blen )
			n = blen;
		    if( n && gpg_iobuf_write(chain, p, n ) )
			rc = G10ERR_WRITE_FILE;
		    p += n;
		    nbytes -= n;
		} while( !rc && nbytes >= OP_MIN_PARTIAL_CHUNK );
		/* store the rest in the buffer */
		if( !rc && nbytes ) {
		    assert( !a->buflen );
		    assert( nbytes < OP_MIN_PARTIAL_CHUNK );
		    if( !a->buffer )
			a->buffer = malloc( OP_MIN_PARTIAL_CHUNK );
		    memcpy( a->buffer, p, nbytes );
		    a->buflen = nbytes;
		}
	    }
	}
	else { /* the gnupg scheme (which is not openpgp compliant) */
	    size_t avail, n;
            
	    for(p=buf; !rc && size; ) {
		n = size;
		avail = a->size - a->count;
		if( !avail ) {
		    if( n > a->size ) {
			gpg_iobuf_put( chain, (a->size >> 8) & 0xff );
			gpg_iobuf_put( chain, a->size & 0xff );
			avail = a->size;
			a->count = 0;
		    }
		    else {
			gpg_iobuf_put( chain, (n >> 8) & 0xff );
			gpg_iobuf_put( chain, n & 0xff );
			avail = n;
			a->count = a->size - n;
		    }
		}
		if( n > avail )
		    n = avail;
		if( gpg_iobuf_write(chain, p, n ) )
		    rc = G10ERR_WRITE_FILE;
		a->count += n;
		p += n;
		size -= n;
	    }
	}
    }
    else if( control == IOBUFCTRL_INIT ) {	
	if( a->partial )
	    a->count = 0;
	else if( a->use == 1 )
	    a->count = a->size = 0;
	else
	    a->count = a->size; /* force first length bytes */
	a->eof = 0;
	a->buffer = NULL;
	a->buflen = 0;
    }
    else if( control == IOBUFCTRL_DESC ) {
	*(char**)buf = "block_filter";
    }
    else if( control == IOBUFCTRL_FREE ) {
	if( a->use == 2 ) { /* write the end markers */
	    if( a->partial ) {
		u32 len;
		/* write out the remaining bytes without a partial header
		 * the length of this header may be 0 - but if it is
		 * the first block we are not allowed to use a partial header
		 * and frankly we can't do so, because this length must be
		 * a power of 2. This is _really_ complicated because we
		 * have to check the possible length of a packet prior
		 * to it's creation: a chain of filters becomes complicated
		 * and we need a lot of code to handle compressed packets etc.
		 *   :-(((((((
		 */
		/* construct header */
		len = a->buflen;
		/*printf("partial: remaining length=%u\n", len );*/
		if( len < 192 )
		    rc = gpg_iobuf_put(chain, len );
		else if( len < 8384 ) {
		    if( !(rc=gpg_iobuf_put( chain, ((len-192) / 256) + 192)) )
			rc = gpg_iobuf_put( chain, ((len-192) % 256));
		}
		else { /* use a 4 byte header */
		    if( !(rc=gpg_iobuf_put( chain, 0xff )) )
			if( !(rc=gpg_iobuf_put( chain, (len >> 24)&0xff )) )
			    if( !(rc=gpg_iobuf_put( chain, (len >> 16)&0xff )) )
				if( !(rc=gpg_iobuf_put( chain, (len >> 8)&0xff )))
				    rc=gpg_iobuf_put( chain, len & 0xff );
		}
		if( !rc && len )
		    rc = gpg_iobuf_write(chain, a->buffer, len );
		if( rc ) {
		    printf("block_filter: write error: %s\n",strerror(errno));
		    rc = G10ERR_WRITE_FILE;
		}
		safe_free( a->buffer ); a->buffer = NULL; a->buflen = 0;
	    }
	    else {
		gpg_iobuf_writebyte(chain, 0);
		gpg_iobuf_writebyte(chain, 0);
	    }
	}
	else if( a->size ) {
	    printf("block_filter: pending bytes!\n");
	}	
	safe_free(a); /* we can free our context now */
    }
    
    return rc;
}

/****************
 * Allocate a new io buffer, with no function assigned.
 * Use is the desired usage: 1 for input, 2 for output, 3 for temp buffer
 * BUFSIZE is a suggested buffer size.
 */
gpg_iobuf_t
gpg_iobuf_alloc(int use, size_t bufsize)
{
    gpg_iobuf_t a;
    static int number=0;
    
    a = calloc(1, sizeof *a);
    a->use = use;
    a->d.buf = malloc( bufsize );
    a->d.size = bufsize;
    a->no = ++number;
    a->subno = 0;
    a->opaque = NULL;
    a->real_fname = NULL;
    return a;
}

int
gpg_iobuf_close( gpg_iobuf_t a )
{
    gpg_iobuf_t a2;
    size_t dummy_len=0;    
    int rc=0;
    
    if( a && a->directfp ) {
	fclose( a->directfp );
	safe_free( a->real_fname );	
	return 0;
    }
    
    for( ; a && !rc ; a = a2 ) {
	a2 = a->chain;
	if( a->use == 2 && (rc=gpg_iobuf_flush(a)) )
	    printf("iobuf_flush failed on close\n");
	if( a->filter && (rc = a->filter(a->filter_ov, IOBUFCTRL_FREE,
					 a->chain, NULL, &dummy_len)) )
	    printf("IOBUFCTRL_FREE failed on close\n");
	safe_free(a->real_fname);	
        if (a->d.buf) {
            memset (a->d.buf, 0, a->d.size); /* erase the buffer */
            safe_free(a->d.buf);
        }
	safe_free(a);
    }
    
    return rc;
}

int
gpg_iobuf_cancel( gpg_iobuf_t a )
{
    const char *s;
    gpg_iobuf_t a2;
    int rc;  
    char *remove_name = NULL;  
    
    if( a && a->use == 2 ) {
	s = gpg_iobuf_get_real_fname(a);
	if( s && *s ) {	  
	    remove_name = strdup ( s );
	}
    }
    
    /* send a cancel message to all filters */
    for( a2 = a; a2 ; a2 = a2->chain ) {
	size_t dummy;
	if( a2->filter )
	    a2->filter( a2->filter_ov, IOBUFCTRL_CANCEL, a2->chain,
                        NULL, &dummy );
    }
    
    rc = gpg_iobuf_close(a);  
    if ( remove_name ) {
	/* Argg, MSDOS does not allow to remove open files.  So
	 * we have to do it here */
	remove ( remove_name );
	safe_free ( remove_name );
    }  
    return rc;
}


/****************
 * create a temporary iobuf, which can be used to collect stuff
 * in an iobuf and later be written by iobuf_write_temp() to another
 * iobuf.
 */
gpg_iobuf_t
gpg_iobuf_temp( void )
{
    gpg_iobuf_t a;
    
    a = gpg_iobuf_alloc(3, 8192 );
    
    return a;
}

gpg_iobuf_t
gpg_iobuf_temp_with_content( const char *buffer, size_t length )
{
    gpg_iobuf_t a;

    a = gpg_iobuf_alloc(3, length );
    memcpy( a->d.buf, buffer, length );
    a->d.len = length;

    return a;
}

void
gpg_iobuf_enable_special_filenames ( int yes )
{
    special_names_enabled = yes;
}

/*
 * see whether the filename has the for "-&nnnn", where n is a
 * non-zero number.
 * Returns this number or -1 if it is not the case.
 */
static int
check_special_filename ( const char *fname )
{
    if ( special_names_enabled
         && fname && *fname == '-' && fname[1] == '&' ) {
        int i;
        
        fname += 2;
        for (i=0; isdigit (fname[i]); i++ )
            ;
        if ( !fname[i] ) 
            return atoi (fname);
    }
    return -1;
}

/****************
 * Create a head iobuf for reading from a file
 * returns: NULL if an error occures and sets errno
 */
gpg_iobuf_t
gpg_iobuf_open( const char *fname )
{
    gpg_iobuf_t a;
    FILEP_OR_FD fp;
    file_filter_ctx_t *fcx;
    size_t len;
    int print_only = 0;
    int fd;
    
    if( !fname || (*fname=='-' && !fname[1])  ) {
	fp = FILEP_OR_FD_FOR_STDIN;      
	//setmode ( my_fileno(fp) , O_BINARY );      
	fname = "[stdin]";
	print_only = 1;
    }
    else if ( (fd = check_special_filename ( fname )) != -1 )
        return gpg_iobuf_fdopen ( translate_file_handle (fd,0), "rb" );
    else if( (fp = my_fopen_ro(fname, "rb")) == INVALID_FP )
	return NULL;
    a = gpg_iobuf_alloc(1, 8192 );
    fcx = malloc( sizeof *fcx + strlen(fname) );
    fcx->fp = fp;
    fcx->print_only_name = print_only;
    strcpy(fcx->fname, fname );
    if( !print_only )
	a->real_fname = strdup( fname );
    a->filter = file_filter;
    a->filter_ov = fcx;
    file_filter( fcx, IOBUFCTRL_DESC, NULL, (byte*)&a->desc, &len );
    file_filter( fcx, IOBUFCTRL_INIT, NULL, NULL, &len );    
    
    return a;
}

/****************
 * Create a head iobuf for reading from a file
 * returns: NULL if an error occures and sets errno
 */
gpg_iobuf_t
gpg_iobuf_fdopen( int fd, const char *mode )
{
    gpg_iobuf_t a;
    FILEP_OR_FD fp;
    file_filter_ctx_t *fcx;
    size_t len;
    
    fp = (FILEP_OR_FD)fd;
    a = gpg_iobuf_alloc( strchr( mode, 'w')? 2:1, 8192 );
    fcx = malloc( sizeof *fcx + 20 );
    fcx->fp = fp;
    fcx->print_only_name = 1;
    sprintf(fcx->fname, "[fd %d]", fd );
    a->filter = file_filter;
    a->filter_ov = fcx;
    file_filter( fcx, IOBUFCTRL_DESC, NULL, (byte*)&a->desc, &len );
    file_filter( fcx, IOBUFCTRL_INIT, NULL, NULL, &len );    
    gpg_iobuf_ioctl (a,3,1,NULL); /* disable fd caching */
    return a;
}

/****************
 * create an iobuf for writing to a file; the file will be created.
 */
gpg_iobuf_t
gpg_iobuf_create( const char *fname )
{
    gpg_iobuf_t a;
    FILEP_OR_FD fp;
    file_filter_ctx_t *fcx;
    size_t len;
    int print_only = 0;
    int fd;
    
    if( !fname || (*fname=='-' && !fname[1]) ) {
	fp = FILEP_OR_FD_FOR_STDOUT;      
	fname = "[stdout]";
	print_only = 1;
    }
    else if ( (fd = check_special_filename ( fname )) != -1 )
        return gpg_iobuf_fdopen ( translate_file_handle (fd, 1), "wb" );
    else if( (fp = my_fopen(fname, "wb")) == INVALID_FP )
	return NULL;
    a = gpg_iobuf_alloc(2, 8192 );
    fcx = malloc( sizeof *fcx + strlen(fname) );
    fcx->fp = fp;
    fcx->print_only_name = print_only;
    strcpy(fcx->fname, fname );
    if( !print_only )
	a->real_fname = strdup( fname );
    a->filter = file_filter;
    a->filter_ov = fcx;
    file_filter( fcx, IOBUFCTRL_DESC, NULL, (byte*)&a->desc, &len );
    file_filter( fcx, IOBUFCTRL_INIT, NULL, NULL, &len );    
    
    return a;
}

gpg_iobuf_t
gpg_iobuf_openrw( const char *fname )
{
    gpg_iobuf_t a;
    FILEP_OR_FD fp;
    file_filter_ctx_t *fcx;
    size_t len;
    
    if( !fname )
	return NULL;
    else if( (fp = my_fopen(fname, "r+b")) == INVALID_FP )
	return NULL;
    a = gpg_iobuf_alloc(2, 8192 );
    fcx = malloc( sizeof *fcx + strlen(fname) );
    fcx->fp = fp;
    strcpy(fcx->fname, fname );
    a->real_fname = strdup( fname );
    a->filter = file_filter;
    a->filter_ov = fcx;
    file_filter( fcx, IOBUFCTRL_DESC, NULL, (byte*)&a->desc, &len );
    file_filter( fcx, IOBUFCTRL_INIT, NULL, NULL, &len );    
    
    return a;
}


int
gpg_iobuf_ioctl ( gpg_iobuf_t a, int cmd, int intval, void *ptrval )
{
    if ( cmd == 1 ) {  /* keep system filepointer/descriptor open */        
        for( ; a; a = a->chain )
            if( !a->chain && a->filter == file_filter ) {
                file_filter_ctx_t *b = a->filter_ov;
                b->keep_open = intval;
                return 0;
            }
    }
    else if ( cmd == 2 ) {  /* invalidate cache */
        if ( !a && !intval && ptrval ) {          
            fd_cache_invalidate (ptrval);          
            return 0;
        }
    }
    else if ( cmd == 3 ) {  /* disallow/allow caching */        
        for( ; a; a = a->chain )
            if( !a->chain && a->filter == file_filter ) {
                file_filter_ctx_t *b = a->filter_ov;
                b->no_cache = intval;
                return 0;
            }
    }
    
    return -1;
}


/****************
 * Register an i/o filter.
 */
int
gpg_iobuf_push_filter( gpg_iobuf_t a,
		   int (*f)(void *opaque, int control,
		   gpg_iobuf_t chain, byte *buf, size_t *len), void *ov )
{
    return gpg_iobuf_push_filter2( a, f, ov, 0 );
}

int
gpg_iobuf_push_filter2( gpg_iobuf_t a,
		    int (*f)(void *opaque, int control,
		    gpg_iobuf_t chain, byte *buf, size_t *len),
		    void *ov, int rel_ov )
{
    gpg_iobuf_t b;
    size_t dummy_len=0;
    int rc=0;
    
    if( a->directfp ) {
	printf( "This is a BUG.\n%s:%d\n", __FILE__, __LINE__ );
	exit( -1 );
    }
    
    if( a->use == 2 && (rc=gpg_iobuf_flush(a)) )
	return rc;
    /* make a copy of the current stream, so that
     * A is the new stream and B the original one.
     * The contents of the buffers are transferred to the
     * new stream.
     */
    b = malloc(sizeof *b);
    memcpy(b, a, sizeof *b );
    /* fixme: it is stupid to keep a copy of the name at every level
     * but we need the name somewhere because the name known by file_filter
     * may have been released when we need the name of the file */
    b->real_fname = a->real_fname? strdup(a->real_fname):NULL;
    /* remove the filter stuff from the new stream */
    a->filter = NULL;
    a->filter_ov = NULL;
    a->filter_ov_owner = 0;
    a->filter_eof = 0;
    if( a->use == 3 )
	a->use = 2;  /* make a write stream from a temp stream */
    
    if( a->use == 2 ) { /* allocate a fresh buffer for the original stream */
	b->d.buf = malloc( a->d.size );
	b->d.len = 0;
	b->d.start = 0;
    }
    else { /* allocate a fresh buffer for the new stream */
	a->d.buf = malloc( a->d.size );
	a->d.len = 0;
	a->d.start = 0;
    }
    /* disable nlimit for the new stream */
    a->ntotal = b->ntotal + b->nbytes;
    a->nlimit = a->nbytes = 0;
    a->nofast &= ~1;
    /* make a link from the new stream to the original stream */
    a->chain = b;
    a->opaque = b->opaque;
    
    /* setup the function on the new stream */
    a->filter = f;
    a->filter_ov = ov;
    a->filter_ov_owner = rel_ov;
    
    a->subno = b->subno + 1;
    f( ov, IOBUFCTRL_DESC, NULL, (byte*)&a->desc, &dummy_len );    
    
    /* now we can initialize the new function if we have one */
    if( a->filter && (rc = a->filter(a->filter_ov, IOBUFCTRL_INIT, a->chain,
                                     NULL, &dummy_len)) )
	printf("IOBUFCTRL_INIT failed\n");
    return rc;
}

/****************
 * Remove an i/o filter.
 */
int
pop_filter( gpg_iobuf_t a, int (*f)(void *opaque, int control,
		      gpg_iobuf_t chain, byte *buf, size_t *len), void *ov )
{
    gpg_iobuf_t b;
    size_t dummy_len=0;
    int rc=0;
    
    if( a->directfp ) {
	printf( "This is a BUG.\n%s:%d\n", __FILE__, __LINE__ );
	exit( -1 );
    }
    
    if( !a->filter ) { /* this is simple */
	b = a->chain;
	assert(b);
	safe_free(a->d.buf);
	safe_free(a->real_fname);
	memcpy(a,b, sizeof *a);
	safe_free(b);
	return 0;
    }
    for(b=a ; b; b = b->chain )
	if( b->filter == f && (!ov || b->filter_ov == ov) )
	    break;
    if( !b )
	printf("pop_filter(): filter function not found\n");
    
    /* flush this stream if it is an output stream */
    if( a->use == 2 && (rc=gpg_iobuf_flush(b)) ) {
	printf("iobuf_flush failed in pop_filter\n");
	return rc;
    }
    /* and tell the filter to free it self */
    if( b->filter && (rc = b->filter(b->filter_ov, IOBUFCTRL_FREE, b->chain,
                                     NULL, &dummy_len)) ) {
	printf("IOBUFCTRL_FREE failed\n");
	return rc;
    }
    if( b->filter_ov && b->filter_ov_owner ) {
	safe_free( b->filter_ov );
	b->filter_ov = NULL;
    }
    
    
    /* and see how to remove it */
    if( a == b && !b->chain )
	printf("can't remove the last filter from the chain\n");
    else if( a == b ) { /* remove the first iobuf from the chain */
	/* everything from b is copied to a. This is save because
	 * a flush has been done on the to be removed entry
	 */
	b = a->chain;
	safe_free(a->d.buf);
	safe_free(a->real_fname);
	memcpy(a,b, sizeof *a);
	safe_free(b);	
    }
    else if( !b->chain ) { /* remove the last iobuf from the chain */
	printf("Ohh jeee, trying to remove a head filter\n");
    }
    else {  /* remove an intermediate iobuf from the chain */
	printf("Ohh jeee, trying to remove an intermediate filter\n");
    }
    
    return rc;
}


/****************
 * read underflow: read more bytes into the buffer and return
 * the first byte or -1 on EOF.
 */
static int
underflow(gpg_iobuf_t a)
{
    size_t len;
    int rc;
    
    assert( a->d.start == a->d.len );
    if( a->use == 3 )
	return -1; /* EOF because a temp buffer can't do an underflow */
    
    if( a->filter_eof ) {
	if( a->chain ) {
	    gpg_iobuf_t b = a->chain;	    
	    safe_free(a->d.buf);
	    safe_free(a->real_fname);
	    memcpy(a, b, sizeof *a);
	    safe_free(b);
	}
	else
            a->filter_eof = 0;  /* for the top level filter */	
	return -1; /* return one(!) EOF */
    }
    if( a->error ) {	
	return -1;
    }
    
    if( a->directfp ) {
	FILE *fp = a->directfp;
        
	len = fread( a->d.buf, 1, a->d.size, fp);
	if( len < a->d.size ) {
	    if( ferror(fp) )
		a->error = 1;
	}
	a->d.len = len;
	a->d.start = 0;
	return len? a->d.buf[a->d.start++] : -1;
    }
    
    
    if( a->filter ) {
	len = a->d.size;	
	rc = a->filter( a->filter_ov, IOBUFCTRL_UNDERFLOW, a->chain,
			a->d.buf, &len );	
	if( a->use == 1 && rc == -1 ) { /* EOF: we can remove the filter */
	    size_t dummy_len=0;
            
	    /* and tell the filter to free itself */
	    if( (rc = a->filter(a->filter_ov, IOBUFCTRL_FREE, a->chain,
                                NULL, &dummy_len)) )
		printf("IOBUFCTRL_FREE failed\n");
	    if( a->filter_ov && a->filter_ov_owner ) {
		safe_free( a->filter_ov );
		a->filter_ov = NULL;
	    }
	    a->filter = NULL;
	    a->desc = NULL;
	    a->filter_ov = NULL;
	    a->filter_eof = 1;
	    if( !len && a->chain ) {
		gpg_iobuf_t b = a->chain;		
		safe_free(a->d.buf);
		safe_free(a->real_fname);
		memcpy(a,b, sizeof *a);
		safe_free(b);
	    }
	}
	else if( rc )
	    a->error = 1;
        
	if( !len ) {
	    return -1;
	}
	a->d.len = len;
	a->d.start = 0;
	return a->d.buf[a->d.start++];
    }
    else {	
	return -1;  /* no filter; return EOF */
    }
}


int
gpg_iobuf_flush(gpg_iobuf_t a)
{
    size_t len;
    int rc;
    
    if( a->directfp )
	return 0;
    
    if( a->use == 3 ) { /* increase the temp buffer */
	char *newbuf;
	size_t newsize = a->d.size + 8192;
        
	printf("increasing temp iobuf from %lu to %lu\n",
               (ulong)a->d.size, (ulong)newsize );
	newbuf = malloc( newsize );
	memcpy( newbuf, a->d.buf, a->d.len );
	safe_free(a->d.buf);
	a->d.buf = newbuf;
	a->d.size = newsize;
	return 0;
    }
    else if( a->use != 2 )
	printf("flush on non-output iobuf\n");
    else if( !a->filter )
	printf("iobuf_flush: no filter\n");
    len = a->d.len;
    rc = a->filter( a->filter_ov, IOBUFCTRL_FLUSH, a->chain, a->d.buf, &len );
    if( !rc && len != a->d.len ) {
	printf("iobuf_flush did not write all!\n");
	rc = G10ERR_WRITE_FILE;
    }
    else if( rc )
	a->error = 1;
    a->d.len = 0;
    
    return rc;
}


/****************
 * Read a byte from the iobuf; returns -1 on EOF
 */
int
gpg_iobuf_readbyte(gpg_iobuf_t a)
{
    int c;
    
    /* nlimit does not work together with unget */
    /* nbytes is also not valid! */
    if( a->unget.buf ) {
	if( a->unget.start < a->unget.len )
	    return a->unget.buf[a->unget.start++];
	safe_free(a->unget.buf);
	a->unget.buf = NULL;
	a->nofast &= ~2;
    }
    
    if( a->nlimit && a->nbytes >= a->nlimit )
	return -1; /* forced EOF */
    
    if( a->d.start < a->d.len ) {
	c = a->d.buf[a->d.start++];
    }
    else if( (c=underflow(a)) == -1 )
	return -1; /* EOF */
    
    a->nbytes++;
    return c;
}


int
gpg_iobuf_read(gpg_iobuf_t a, byte *buf, unsigned buflen )
{
    int c, n;
    
    if( a->unget.buf || a->nlimit ) {
	/* handle special cases */
	for(n=0 ; n < buflen; n++ ) {
	    if( (c = gpg_iobuf_readbyte(a)) == -1 ) {
		if( !n )
		    return -1; /* eof */
		break;
	    }
	    else
		if( buf ) *buf = c;
	    if( buf ) buf++;
	}
	return n;
    }
    
    n = 0;
    do {
	if( n < buflen && a->d.start < a->d.len ) {
	    unsigned size = a->d.len - a->d.start;
	    if( size > buflen - n )
		size = buflen - n;
	    if( buf )
		memcpy( buf, a->d.buf + a->d.start, size );
	    n += size;
	    a->d.start += size;
	    if( buf )
		buf += size;
	}
	if( n < buflen ) {
	    if( (c=underflow(a)) == -1 ) {
		a->nbytes += n;
		return n? n : -1/*EOF*/;
	    }
	    if( buf )
		*buf++ = c;
	    n++;
	}
    } while( n < buflen );
    a->nbytes += n;
    return n;
}


/****************
 * Have a look at the iobuf.
 * NOTE: This only works in special cases.
 */
int
gpg_iobuf_peek(gpg_iobuf_t a, byte *buf, unsigned buflen )
{
    int n=0;
    
    if( a->filter_eof )
	return -1;
    
    if( !(a->d.start < a->d.len) ) {
	if( underflow(a) == -1 )
	    return -1;
	/* and unget this character */
	assert(a->d.start == 1);
	a->d.start = 0;
    }
    
    for(n=0 ; n < buflen && (a->d.start+n) < a->d.len ; n++, buf++ )
	*buf = a->d.buf[n];
    return n;
}




int
gpg_iobuf_writebyte(gpg_iobuf_t a, unsigned c)
{
    
    if( a->directfp ) {
	printf( "This is a BUG.\n%s:%d\n", __FILE__, __LINE__ );
	exit( -1 );
    }
    
    if( a->d.len == a->d.size )
	if( gpg_iobuf_flush(a) )
	    return -1;
    
    assert( a->d.len < a->d.size );
    a->d.buf[a->d.len++] = c;
    return 0;
}


int
gpg_iobuf_write(gpg_iobuf_t a, byte *buf, unsigned buflen )
{
    
    if( a->directfp ) {
	printf( "This is a BUG.\n%s:%d\n", __FILE__, __LINE__ );
	exit( -1 );
    }
    
    do {
	if( buflen && a->d.len < a->d.size ) {
	    unsigned size = a->d.size - a->d.len;
	    if( size > buflen ) size = buflen;
	    memcpy( a->d.buf + a->d.len, buf, size );
	    buflen -= size;
	    buf += size;
	    a->d.len += size;
	}
	if( buflen ) {
	    if( gpg_iobuf_flush(a) )
		return -1;
	}
    } while( buflen );
    return 0;
}


int
gpg_iobuf_writestr(gpg_iobuf_t a, const char *buf )
{
    for( ; *buf; buf++ )
	if( gpg_iobuf_writebyte(a, *buf) )
	    return -1;
    return 0;
}



/****************
 * copy the contents of TEMP to A.
 */
int
gpg_iobuf_write_temp( gpg_iobuf_t a, gpg_iobuf_t temp )
{
    while( temp->chain )
	pop_filter( temp, temp->filter, NULL );
    return gpg_iobuf_write(a, temp->d.buf, temp->d.len );
}

/****************
 * copy the contents of the temp io stream to BUFFER.
 */
size_t
gpg_iobuf_temp_to_buffer( gpg_iobuf_t a, byte *buffer, size_t buflen )
{
    size_t n = a->d.len;
    
    if( n > buflen )
	n = buflen;
    memcpy( buffer, a->d.buf, n );
    return n;
}


/****************
 * Call this function to terminate processing of the temp stream
 * without closing it.	This removes all filters from the stream
 * makes sure that iobuf_get_temp_{buffer,length}() returns correct
 * values.
 */
void
gpg_iobuf_flush_temp( gpg_iobuf_t temp )
{
    while( temp->chain )
	pop_filter( temp, temp->filter, NULL );
}


/****************
 * Set a limit on how many bytes may be read from the input stream A.
 * Setting the limit to 0 disables this feature.
 */
void
gpg_iobuf_set_limit( gpg_iobuf_t a, _off_t nlimit )
{
    if( nlimit )
	a->nofast |= 1;
    else
	a->nofast &= ~1;
    a->nlimit = nlimit;
    a->ntotal += a->nbytes;
    a->nbytes = 0;
}



/****************
 * Return the length of an open file
 */
_off_t
gpg_iobuf_get_filelength( gpg_iobuf_t a )
{
    struct stat st;
    
    if( a->directfp )  {
	FILE *fp = a->directfp;
        
        if( !fstat(fileno(fp), &st) )
            return st.st_size;
	printf("fstat() failed: %s\n", strerror(errno) );
	return 0;
    }
    
    /* Hmmm: file_filter may have already been removed */
    for( ; a; a = a->chain )
	if( !a->chain && a->filter == file_filter ) {
	    file_filter_ctx_t *b = a->filter_ov;
	    FILEP_OR_FD fp = b->fp;
            
            ulong size;
            
            if ( (size=GetFileSize (fp, NULL)) != 0xffffffff ) 
                return size;
            printf ("GetFileSize for handle %p failed: ec=%d\n",
                    fp, (int)GetLastError () );
	    break;
	}
    
    return 0;
}

/****************
 * Tell the file position, where the next read will take place
 */
_off_t
gpg_iobuf_tell( gpg_iobuf_t a )
{
    return a->ntotal + a->nbytes;
}


#if !defined(HAVE_FSEEKO) && !defined(fseeko)

#ifdef HAVE_LIMITS_H
# include <limits.h>
#endif
#ifndef LONG_MAX
# define LONG_MAX ((long) ((unsigned long) -1 >> 1))
#endif
#ifndef LONG_MIN
# define LONG_MIN (-1 - LONG_MAX)
#endif

/****************
 * A substitute for fseeko, for hosts that don't have it.
 */
static int
fseeko( FILE *stream, _off_t newpos, int whence )
{
    while( newpos != (long) newpos ) {
        long pos = newpos < 0 ? LONG_MIN : LONG_MAX;
        if( fseek( stream, pos, whence ) != 0 )
            return -1;
        newpos -= pos;
        whence = SEEK_CUR;
    }
    return fseek( stream, (long)newpos, whence );
}
#endif

/****************
 * This is a very limited implementation. It simply discards all internal
 * buffering and removes all filters but the first one.
 */
int
gpg_iobuf_seek( gpg_iobuf_t a, _off_t newpos )
{
    file_filter_ctx_t *b = NULL;
    
    if( a->directfp ) {
	FILE *fp = a->directfp;
        if( fseeko( fp, newpos, SEEK_SET ) ) {
            printf("can't seek: %s\n", strerror(errno) );
	    return -1;
	}
	clearerr(fp);
    }
    else {
	for( ; a; a = a->chain ) {
	    if( !a->chain && a->filter == file_filter ) {
		b = a->filter_ov;
		break;
	    }
	}
	if( !a )
	    return -1;
        if (SetFilePointer (b->fp, newpos, NULL, FILE_BEGIN) == 0xffffffff ) {
            printf ("SetFilePointer failed on handle %p: ec=%d\n",
                    b->fp, (int)GetLastError () );
            return -1;
        }    
    }
    a->d.len = 0;   /* discard buffer */
    a->d.start = 0;
    a->nbytes = 0;
    a->nlimit = 0;
    a->nofast &= ~1;
    a->ntotal = newpos;
    a->error = 0;
    /* remove filters, but the last */
    if( a->chain )
	printf("pop_filter called in iobuf_seek - please report\n");
    while( a->chain )
        pop_filter( a, a->filter, NULL );
    
    return 0;
}






/****************
 * Retrieve the real filename
 */
const char *
gpg_iobuf_get_real_fname( gpg_iobuf_t a )
{
    if( a->real_fname )
	return a->real_fname;
    
    /* the old solution */
    for( ; a; a = a->chain )
	if( !a->chain && a->filter == file_filter ) {
	    file_filter_ctx_t *b = a->filter_ov;
	    return b->print_only_name? NULL : b->fname;
	}
    
    return NULL;
}


/****************
 * Retrieve the filename
 */
const char *
gpg_iobuf_get_fname( gpg_iobuf_t a )
{
    for( ; a; a = a->chain )
	if( !a->chain && a->filter == file_filter ) {
	    file_filter_ctx_t *b = a->filter_ov;
	    return b->fname;
	}

    return NULL;
}

/****************
 * Start the block write mode, see rfc1991.new for details.
 * A value of 0 for N stops this mode (flushes and writes
 * the end marker)
 */
void
gpg_iobuf_set_block_mode( gpg_iobuf_t a, size_t n )
{
    block_filter_ctx_t *ctx = calloc( 1, sizeof *ctx );
    
    assert( a->use == 1 || a->use == 2 );
    ctx->use = a->use;
    if( !n ) {
	if( a->use == 1 )
	    printf("pop_filter called in set_block_mode - please report\n");
	pop_filter(a, block_filter, NULL );
    }
    else {
	ctx->size = n; /* only needed for use 2 */
	gpg_iobuf_push_filter(a, block_filter, ctx );
    }
}

/****************
 * enable partial block mode as described in the OpenPGP draft.
 * LEN is the first length byte on read, but ignored on writes.
 */
void
gpg_iobuf_set_partial_block_mode( gpg_iobuf_t a, size_t len )
{
    block_filter_ctx_t *ctx = calloc( 1, sizeof *ctx );
    
    assert( a->use == 1 || a->use == 2 );
    ctx->use = a->use;
    if( !len ) {
	if( a->use == 1 )
	    printf("pop_filter called in set_partial_block_mode"
                   " - please report\n");
	pop_filter(a, block_filter, NULL );
    }
    else {
	ctx->partial = 1;
	ctx->size = 0;
	ctx->first_c = len;
	gpg_iobuf_push_filter(a, block_filter, ctx );
    }
}


/****************
 * Checks whether the stream is in block mode
 * Note: This does not work if other filters are pushed on the stream.
 */
int
gpg_iobuf_in_block_mode( gpg_iobuf_t a )
{
    if( a && a->filter == block_filter )
	return 1; /* yes */
    return 0; /* no */
}


/****************
 * Same as fgets() but if the buffer is too short a larger one will
 * be allocated up to some limit *max_length.
 * A line is considered a byte stream ending in a LF.
 * Returns the length of the line. EOF is indicated by a line of
 * length zero. The last LF may be missing due to an EOF.
 * is max_length is zero on return, the line has been truncated.
 *
 * Note: The buffer is allocated with enough space to append a CR,LF,EOL
 */
unsigned
gpg_iobuf_read_line( gpg_iobuf_t a, byte **addr_of_buffer,
			  unsigned *length_of_buffer, unsigned *max_length )
{
    int c;
    char *buffer = *addr_of_buffer;
    unsigned length = *length_of_buffer;
    unsigned nbytes = 0;
    unsigned maxlen = *max_length;
    char *p;
    
    if( !buffer ) { /* must allocate a new buffer */
	length = 256;
	buffer = (char *)malloc( length );
	*addr_of_buffer = buffer;
	*length_of_buffer = length;
    }
    
    length -= 3; /* reserve 3 bytes (cr,lf,eol) */
    p = buffer;
    while( (c=gpg_iobuf_get(a)) != -1 ) {
	if( nbytes == length ) { /* increase the buffer */
	    if( length > maxlen  ) { /* this is out limit */
		/* skip the rest of the line */
		while( c != '\n' && (c=gpg_iobuf_get(a)) != -1 )
		    ;
		*p++ = '\n'; /* always append a LF (we have reserved space) */
		nbytes++;
		*max_length = 0; /* indicate truncation */
		break;
	    }
	    length += 3; /* correct for the reserved byte */
	    length += length < 1024? 256 : 1024;
	    buffer = realloc( buffer, length );
	    *addr_of_buffer = buffer;
	    *length_of_buffer = length;
	    length -= 3; /* and reserve again */
	    p = buffer + nbytes;
	}
	*p++ = c;
	nbytes++;
	if( c == '\n' )
	    break;
    }
    *p = 0; /* make sure the line is a string */
    
    return nbytes;
}

/* This is the non iobuf specific function */
int
gpg_iobuf_translate_file_handle ( int fd, int for_write )
{  
    {
        int x;
        
        if  ( fd <= 2 )
            return fd; /* do not do this for error, stdin, stdout, stderr */
        
        x = _open_osfhandle ( fd, for_write? 1:0 );
        if (x==-1 )
            printf ("failed to translate osfhandle %p\n", (void*)fd );
        else {
            /*log_info ("_open_osfhandle %p yields %d%s\n",
              (void*)fd, x, for_write? " for writing":"" );*/
            fd = x;
        }
    }  
    return fd;
}

static int
translate_file_handle ( int fd, int for_write )
{  
    {
        int x;
        
        if  ( fd == 0 ) 
            x = (int)GetStdHandle (STD_INPUT_HANDLE);
        else if (fd == 1)    
            x = (int)GetStdHandle (STD_OUTPUT_HANDLE);
        else if (fd == 2)    
            x = (int)GetStdHandle (STD_ERROR_HANDLE);
        else
            x = fd;
        
        if (x == -1)
            printf ("GetStdHandle(%d) failed: ec=%d\n",
                    fd, (int)GetLastError () );
        
        fd = x;
    }
    return fd;
}


