/* ownertrust.c - Ownertrust handling
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "context.h"
#include "ops.h"
#include "key.h"
#include "util.h"


static gpgme_error_t
key_trust_export_start( gpgme_ctx_t c, gpgme_data_t out )
{
    gpgme_error_t rc;
    
    fail_on_pending_request( c );
    c->pending = 1;
    
    _gpgme_gpg_release( &c->gpg );
    rc = _gpgme_gpg_new( &c->gpg );
    if( rc )
        goto leave;
    
    _gpgme_data_set_mode( out, GPGME_DATA_MODE_IN );
    _gpgme_gpg_add_arg( c->gpg, "--export-ownertrust" );
    _gpgme_gpg_add_arg( c->gpg, "--output" );
    _gpgme_gpg_add_arg( c->gpg, "-" );
    _gpgme_gpg_add_data( c->gpg, out, 1 );
    rc = _gpgme_gpg_spawn( c->gpg, c );
    
leave:
    if( rc ) {
        c->pending = 0;
        _gpgme_gpg_release( &c->gpg ); 
    }

    return rc;
} /* gpgme_op_key_trust_export_start */


gpgme_error_t
gpgme_op_key_trust_export( gpgme_ctx_t c, gpgme_data_t out )
{
    gpgme_error_t rc = key_trust_export_start( c, out );
    if( !rc ) {
        gpgme_wait( c, 1 );
        c->pending = 0;
    }

    return rc;
} /* gpgme_op_key_trust_export */


static gpgme_error_t
key_trust_import_start( gpgme_ctx_t c, gpgme_data_t in )
{
    gpgme_error_t rc;
    
    fail_on_pending_request( c );
    c->pending = 1;
    
    _gpgme_gpg_release( &c->gpg );
    rc = _gpgme_gpg_new( &c->gpg );
    if( rc )
        goto leave;
    
    _gpgme_data_set_mode( in, GPGME_DATA_MODE_OUT );
    _gpgme_gpg_add_arg( c->gpg, "--import-ownertrust" );
    _gpgme_gpg_add_arg( c->gpg, "--" );
    _gpgme_gpg_add_data( c->gpg, in, 0 );
    
    rc = _gpgme_gpg_spawn( c->gpg, c );
    
leave:
    if( rc ) {
        c->pending = 0;
        _gpgme_gpg_release( &c->gpg );
    }
    return rc;
} /* key_trust_import_start */


gpgme_error_t
gpgme_op_key_trust_import( gpgme_ctx_t ctx, gpgme_data_t in )
{
    gpgme_error_t err;
    
    err = key_trust_import_start( ctx, in );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
    }

    return err;
} /* gpgme_op_key_trust_import */


gpgme_error_t
gpgme_op_ownertrust_export_file( const char *output )
{
    gpgme_error_t err;
    gpgme_ctx_t ctx;
    gpgme_data_t otrust;
    
    err = gpgme_new( &ctx );
    if( !err )        
	err = gpgme_data_new( &otrust );
    if( !err )        
	err = gpgme_op_key_trust_export( ctx, otrust );
    if( !err )        
	err = gpgme_data_release_and_set_file( otrust, output );
    
    gpgme_release( ctx );
    return err;
} /* gpgme_op_ownertrust_export_file */


gpgme_error_t
gpgme_op_ownertrust_import_file( const char *input )
{
    gpgme_error_t err;
    gpgme_ctx_t ctx;
    gpgme_data_t otrust = NULL;
    
    err = gpgme_new( &ctx );
    if( !err )
	err = gpgme_data_new_from_file (&otrust, input);
    if( !err )
	err = gpgme_op_key_trust_import( ctx, otrust );
    
    gpgme_data_release( otrust );
    gpgme_release( ctx );
    return err;
} /* gpgme_op_ownertrust_import_file */


static gpgme_error_t
rebuild_cache_start( gpgme_ctx_t c )
{
    gpgme_error_t rc = 0;
    
    _gpgme_gpg_release( &c->gpg );
    rc = _gpgme_gpg_new( &c->gpg );
    if( rc )
        goto leave;
    
    _gpgme_gpg_add_arg( c->gpg, "--rebuild-keydb-cache" );
    rc = _gpgme_gpg_spawn( c->gpg, c );
    
leave:
    if( rc ) {
        c->pending = 0;
        _gpgme_gpg_release( &c->gpg );
    }
    return rc;
} /* rebuild_cache_start */


gpgme_error_t
gpgme_op_rebuild_cache( void )
{
    gpgme_error_t err;
    gpgme_ctx_t ctx;
    
    err = gpgme_new( &ctx );
    if( err )
        return err;
    err = rebuild_cache_start( ctx );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
    }
    gpgme_release( ctx );

    return err;
} /* gpgme_op_rebuild_cache */

