/* wptErrors.cpp - Error management
 *	Copyright (C) 2000-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <stdio.h>

#include "wptErrors.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptGPG.h"
#include "wptNLS.h"


static char *
gpg_strerror( int errid )
{
    static char buf[512];
    char * path = NULL;

    path = get_gnupg_path();
    if( !path )
	BUG( NULL );
    switch( errid ) {
    case WPTERR_GPG_EXEFILE:
	_snprintf( buf, sizeof buf - 1, _("Could not locate GPG.exe in %s."), path );	
        break;
    case WPTERR_GPG_OPT_KEYRINGS:
    case WPTERR_GPG_KEYRINGS:
	_snprintf( buf, sizeof buf - 1, 
                   _("Could not find keyring entries into the options file in %s "
                     "or the file does NOT exist."), path );
	break;
    }
    free_if_alloc( path );
    return buf;
} /* gpg_strerror */


static void 
w32_fmt_msg (char *buf, size_t buflen)
{
    FormatMessage (FORMAT_MESSAGE_FROM_SYSTEM, NULL, GetLastError (), 
		   MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT), 
		   buf, buflen, NULL);
}

/* Return an error specific message.
   If there is a translation available, it returns the translated text. */
const char*
winpt_strerror (int errid)
{	
    static char buf[256];
	
    switch( errid ) {
    case WPTERR_GENERAL: return _("General error occured");
    case WPTERR_FILE_OPEN: return _("Could not open file");
    case WPTERR_FILE_CREAT: return _("Could not create file");
    case WPTERR_FILE_READ: return _("Could not read file");
    case WPTERR_FILE_WRITE: return _("Could not write file");
    case WPTERR_FILE_CLOSE: return _("Could not close file");
    case WPTERR_FILE_EXIST: return _("File does not exist");
    case WPTERR_FILE_REMOVE: return _("Could not delete file");
    case WPTERR_CLIP_OPEN: return _("Could not open Clipboard");
    case WPTERR_CLIP_CLOSE: return _("Could not close Clipboard");
    case WPTERR_CLIP_EMPTY: return _("Could not empty Clipboard");
    case WPTERR_CLIP_SET: return _("Could not set Clipboard data");
    case WPTERR_CLIP_GET: return _("Could not get Clipboard data");
    case WPTERR_CLIP_ISEMPTY: return _("There is no text in the Clipboard");
    case WPTERR_CLIP_SECURED: return _("The Clipboard already contains GPG data");
    case WPTERR_CLIP: return _("General Clipboard error");
    case WPTERR_REGISTRY: sprintf (buf, "%s", _("Registry error: "));
			  w32_fmt_msg (buf + strlen (buf), 128);
			  break;
    case WPTERR_WINSOCK_INIT: return _("Could not startup Winsock 2 interface");
    case WPTERR_WINSOCK_RESOLVE: return _("Could not resolve hostname");
    case WPTERR_WINSOCK_SOCKET: return _("Could not create new socket");
    case WPTERR_WINSOCK_CONNECT: return _("Could not connect to the host");
    case WPTERR_WINSOCK_SENDKEY: return _("Could not send the key to the keyserver");
    case WPTERR_WINSOCK_RECVKEY: return _("Could not receive the key from the keyserver");
    case WPTERR_WINSOCK_TIMEOUT: return _("Socket timed out, no data");
    case WPTERR_KEYSERVER_NOTFOUND: return _("Keyserver returned: no matching keys in database");
    case WPTERR_HOTKEY: sprintf (buf, "%s", _("Could not register hotkey: "));
			w32_fmt_msg (buf + strlen (buf), 128);
			break;
    case WPTERR_DIR_OPEN: return _("Could not open directory");
    case WPTERR_DIR_CREAT: return _("Could not create directory");
    case WPTERR_CURR_WND: return _("Could not extract data from the current window");
    case WPTERR_CONFIG_FILE: return _("Could not load config file");	
    case WPTERR_GPG_EXEFILE:	
    case WPTERR_GPG_OPT_KEYRINGS:
    case WPTERR_GPG_KEYRINGS: return gpg_strerror (errid);
    case WPTERR_NOCARD: return _("There is no card in the reader");
    case WPTERR_NOREADER: return _("There was no reader found");
    case WPTERR_NOPGPCARD: return _("This is not an OpenPGP card");
    case WPTERR_CDISK_LOCK: return _("Could not lock or unlock volume");
    case WPTERR_CDISK_MOUNT: return _("Could not mount volume");
    case WPTERR_CDISK_UMOUNT: return _("Could not unmount volume");
    case WPTERR_CDISK_OPEN: return _("Could not  open volume");
    case WPTERR_CDISK_BUSY: return _("Drive that belongs to the volume is busy");
    case WPTERR_CDISK_QUERY: return _("Could not query volume information");
    default: _snprintf (buf, sizeof (buf) - 1, _("Unknown error=%d"), errid); return buf;
    }

    return NULL;
} /* winpt_strerror */


void
winpt_errmsg (const char * name, int is_file)
{	
    TCHAR buf[500], head[500];
    DWORD last_err;

    memset (head, 0, sizeof head);
    last_err = GetLastError ();
    if (!name)
	strcpy (head, _("WinPT Error"));
    else if (is_file)
	_snprintf (head, DIM (head)-1, "%s: %d", name, last_err);
    else
	strncpy (head, name, DIM (head)-1);
    FormatMessage (FORMAT_MESSAGE_FROM_SYSTEM, NULL, last_err, 
		   MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT), 
		   buf, DIM (buf)-1, NULL);
    MessageBox (NULL, buf, head, MB_OK);
} /* winpt_errmsg */


void
winpt_debug_msg( void )
{	
    char output[768], temp[384];
	
    GetTempPath( sizeof temp -1, temp );
    _snprintf( output, sizeof output - 1, 
	"The GPGME output file is gpgme.dbg\n"
	"The WinPT output file is %swinpt.log\n", temp );
    MessageBox( NULL, output, "WinPT now runs in DEBUG MODE", MB_INFO );
} /* winpt_debug_msg */


void
debug_box( const char *msg, const char *file, int line )
{
#if _DEBUG
    char output[4096];

    memset( output, 0, sizeof output );	
    _snprintf( output, sizeof output -1,
	"WinPT DBG:\n \"%s\", file %s, line %d\n", msg, file, line );	
    MessageBox( NULL, output, PGM_NAME, MB_OK | MB_ICONERROR );
#endif
} /* debug_box */


void
debug_f( const char *msg, const char *file, int line )
{
#if _DEBUG
    FILE *fp;
    char temp_path[ 1024 ];
    char output[ 4096 ];

    GetTempPath( sizeof temp_path - 1, temp_path );	
    strcat( temp_path, "\\WinPT.DEBUG" );	
    fp = fopen( temp_path, "a+b" );	
    if( !fp )
	return;
    memset( output, 0, sizeof output );
    _snprintf( output, sizeof output -1,
	       "WinPT DBG:\n \"%s\", file %s, line %d\n", msg, file, line );
    fwrite( output, 1, strlen( output ), fp );
    fflush( fp );
    fclose( fp );
#endif
} /* debug_f */


void
log_f( const char *format, ... )
{
    FILE *fp;
    char temp_path[384];
    va_list arg_ptr;

    GetTempPath( sizeof temp_path - 1, temp_path );
    strcat( temp_path, "\\WinPT.LOG" );
    fp = fopen( temp_path, "a+b" );
    if( !fp )
	return;
    va_start( arg_ptr, format );
    vfprintf( fp, format, arg_ptr );
    va_end( arg_ptr );
    fflush( fp );
    fclose( fp );
} /* log_f */


int
log_box (const char *title, int style, const char *format, ...)
{
    va_list arg_ptr;
    char log[8192];
    int id;

    va_start (arg_ptr, format);
    _vsnprintf (log, sizeof (log)-1, format, arg_ptr);    
    id = msg_box (NULL, log, title? title : "Log Output", style);
    va_end( arg_ptr );

    return id;
} /* log_box */