/* wptHTTP.c - HTTP support
 *	Copyright (C) 2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
#include <windows.h>
#else
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/time.h>
#endif
#include <ctype.h>
#include <errno.h>


#include "wptHTTP.h"


static int parse_headers (int fd, http_head_t * r_head);


static int
http_head_new (http_head_t * head, const char * val)
{
    http_head_t h;
    
    h = calloc (1, sizeof * h + strlen (val) + 1);
    if (h == NULL)
	abort ();
    strcpy (h->d, val);
    *head = h;
    return 0;
}


static void
http_head_free (http_head_t head)
{
    http_head_t h;
    
    while (head)
    {
	h = head->next;
	free (head);
	head = h;
    }
}


static void
http_head_add (http_head_t root, http_head_t node)
{
    http_head_t n;
    
    for (n = root; n->next; n=n->next)
	;
    n->next = node;
}


static http_head_t
http_head_find (http_head_t root, const char * name,
                const char ** val)
{
    http_head_t n;
    
    for (n = root; n; n = n->next)
    {
	if (strlen (n->d) >= strlen (name) &&
	    !strncmp (n->d, name, strlen (name)))
        {
            *val = n->d + strlen (name);
	    return n;
        }
    }
    return NULL;
}


static int
http_connect (const char * host, int port, int * r_fd)
{
    struct hostent * hp;
    struct sockaddr_in srv;
    unsigned long addr = 0;
    int i = 1;
    int fd;
    
    if (r_fd)
	*r_fd = 0;
    memset (&srv, 0, sizeof srv);
    srv.sin_port = htons ((unsigned short)port);
    srv.sin_family = AF_INET;
    
    if (isalpha (*host))
    {
	hp = gethostbyname (host);
	if (hp == NULL)
        {
            fprintf (stderr, "** gethostbyname(): %s\n", strerror (errno));
	    return HTTP_ERR_CONNECT;
        }
	memcpy (&srv.sin_addr, hp->h_addr, hp->h_length);
    }
    else
    {
	addr = inet_addr (host);
	if (addr == -1)
        {
            fprintf (stderr, "** inet_addr(): %s\n", strerror (errno));
	    return HTTP_ERR_INVADDR;
        }
	memcpy (&srv.sin_addr, &addr, sizeof addr);
    }
	    
    fd = socket (AF_INET, SOCK_STREAM, 0);
    if (fd < 0)
    {
        fprintf (stderr, "** socket(): %s\n", strerror (errno));
	return HTTP_ERR_SOCK;
    }
    
    if (setsockopt (fd, SOL_SOCKET, SO_REUSEADDR, (const char*)&i, sizeof i))
	return HTTP_ERR_SOCK;
    
    if (connect (fd, (struct sockaddr *)&srv, sizeof srv))
    {
	s_close (fd);
        fprintf (stderr, "** connect(): %s\n", strerror (errno));
	return HTTP_ERR_CONNECT;
    }
    if (r_fd)
	*r_fd = fd;
    else
	s_close (fd);
    return 0;
}


static int
http_close (int fd)
{
    s_close (fd);
    return 0;
}


static int
http_data_avail (int fd)
{
    struct timeval tv;
    fd_set inp;
    int n;

    FD_ZERO (&inp);
    FD_SET (fd, &inp);
    tv.tv_sec = 1;
    tv.tv_usec = 0;

    n = select (fd+1, &inp, NULL, NULL, &tv);
    if (n && FD_ISSET (fd, &inp))
        return n;
    return 0;
}


static int
http_read_line (int fd, char * buf, size_t nbuf,
                int nonblock, int * nn, int * eof)
{
    char c;
    int n, i;
    
    if (nn)
	*nn = 0;
    if (eof)
        *eof = 0;
    i = 0;
    do
    {
        if (nonblock == 1 && http_data_avail (fd) == 0)
        {
	    fprintf (stderr, "** fd timeout.\n");
            buf[i++] = '\0';
            i = -1;
            break;
        }
	n = recv (fd, &c, 1, 0);
	if (n == -1)
	    break;
	if (n == 0 || nbuf == 0 || c == '\n')
	{
	    if (n == 0)
	    {
	    	/*fprintf (stderr, "** fd eof (%d).\n", i);*/
                if (eof)
                    *eof = 1;
                buf[i++] = '\0';
	    }
            else
            {
                buf[i++] = c;
                buf[i] = '\0';
            }
            break;
	}
        else
        {
            buf[i++] = c;
            nbuf--;
        }
    } while (n > 0 && i != -1);
    
    if (nn)
	*nn = i;
    return 0;
}


static int
http_send_req (int fd, const char * host, const char * url)
{
    char * p;
    int n;
    
    if (*url == '/')
    	url++;
    if (url == NULL)
    	url = "";
    p = malloc (strlen (url) + strlen (host) + 64);
    if (p == NULL)
	abort ();
    sprintf (p, "GET /%s HTTP/1.1\r\n"
	        "Host: %s\r\n"
	        "\r\n"
	        "\r\n",
	     url, host);
    n = send (fd, p, strlen (p), 0);
    free (p);
    if (n == -1)
    {
        fprintf (stderr, "** send(): %s\n", strerror (errno));
	return HTTP_ERR_IO;
    }
    return 0;
}


int
http_send_request2 (const char * url, http_hd_t * r_hd)
{
    http_hd_t hd;
    char * host = NULL, * p;
    char tmpbuf[512];
    int rc;

    memset (tmpbuf, 0, 512);
    strncpy (tmpbuf, url, 512);
    
    if (strlen (url) < 10 || strncmp (url, "http://", 7))
        return HTTP_ERR_PROTO;
    url += 7;
    p = strtok (tmpbuf+7, "/");
    if (p == NULL)
        return HTTP_ERR_PROTO;
    host = strdup (p);
    if (host == NULL)
        abort ();
    p = strchr (url, '/');
    if (p == NULL)
    {
        free (host);
        return HTTP_ERR_PROTO;
    }
    rc = http_send_request (host, 80, p+1, &hd);
    if (r_hd)
        *r_hd = hd;
    else
        http_hd_free (hd);
    free (host);
    return rc;
}


int
http_send_request (const char * host, int port, const char * url,
                   http_hd_t * r_hd)
{
    http_hd_t hd;
    int rc;

    http_hd_new (&hd);
    rc = http_connect (host, port, &hd->fd);
    if (!rc)
        rc = http_send_req (hd->fd, host, url);
    if (r_hd)
        *r_hd = hd;
    else
        http_hd_free (hd);
    return rc;
}


int
http_req_new (http_req_t * ctx)
{
    http_req_t c;
    c = calloc (1, sizeof * c);
    if (c == NULL)
        abort ();
    *ctx = c;
    return 0;
}


void
http_req_free (http_req_t ctx)
{
    if (!ctx)
        return;
    http_head_free (ctx->head);
    if (ctx->url)
        free (ctx->url);
    free (ctx);
}


static int
parse_reqline (const char * buf, http_req_t * ctx)
{
    http_req_t c;
    char * p;
    int pos = 0;

    *ctx = NULL;
    http_req_new (&c);
    if (strlen (buf) < 4)
        return HTTP_ERR_PROTO;
    if (!strncmp (buf, "GET", 3))
    {
        c->type = HTTP_GET;
        buf += 3;
    }
    else if (!strncmp (buf, "POST", 4))
    {
        c->type = HTTP_POST;
        buf += 4;
    }
    else
    {
        http_req_free (c);
        return HTTP_ERR_PROTO;
    }
    buf++;
    p = strchr (buf, ' ');
    if (p == NULL)
    {
        http_req_free (c);
        return HTTP_ERR_PROTO;
    }
    c->url = malloc (p-buf+2);
    if (c->url == NULL)
        abort ();
    while (*buf != ' ')
        c->url[pos++] = *buf++;
    c->url[pos] = '\0';
    buf++;
    if (strncmp (buf, "HTTP/1.", 7))
    {
        http_req_free (c);
        return HTTP_ERR_PROTO;
    }
    *ctx = c;
    return 0;
}


int
http_parse_request (http_hd_t hd, http_req_t * r_req)
{
    http_req_t req;
    char buf[1024+1];
    int rc, n;

    rc = http_read_line (hd->fd, buf, 1024, 0, &n, NULL);
    if (rc)
        return rc;
    rc = parse_reqline (buf, &req);
    if (rc)
        return rc;
    rc = parse_headers (hd->fd, &hd->head);
    if (rc)
    {
        http_req_free (req);
        return rc;
    }
    *r_req = req;
    return 0;
}


static int
    parse_statline (const char * buf, int * code)
{
    int tmp = 0;
    
    if (strlen (buf) < 8 ||
        (strncmp (buf, "HTTP/1.", 7)))
    {
        fprintf (stderr, "** invalid proto: '%s'\n", buf);
	return HTTP_ERR_PROTO;
    }
    buf += 8; /* skip HTTP/1.x */
    buf++;    /* white space */
    tmp = atoi (buf);
    if (tmp == 0)
    {
        fprintf (stderr, "** invalid statcode: %s\n", buf);
	return HTTP_ERR_PROTO;
    }
    *code = tmp;
    buf += 3;
    buf++; /* whitespace */
    return 0;
}


static int
parse_headers (int fd, http_head_t * r_head)
{
    http_head_t h, head = NULL;
    char buf[300];
    int nn;
    int rc;
    
    do 
    {
	rc = http_read_line (fd, buf, 299, 1, &nn, NULL);
	if (rc)
	    return rc;
	if (nn == 2)
	    break; /* reach empty line */
	http_head_new (&h, buf);
	if (head == NULL)
	    head = h;
	else
	    http_head_add (head, h);
    } while (rc == 0);
    if (r_head)
        *r_head = head;
    else
        http_head_free (head);
    return 0;
}


static int
     read_binary (int fd, int nlen, FILE * out)
{
    char buf[1024+1];
    int n;

    do
    {
        n = recv (fd, buf, nlen > 1024? 1024 : nlen, 0);
        if (n == -1)
        {
            fprintf (stderr, "** recv(): %s\n", strerror (errno));
            return HTTP_ERR_IO;
        }
        if (n == 0)
            break;
        nlen -= n;
        fwrite (buf, 1, n, out);
    } while (nlen > 0);
    return 0;
}


static int
    parse_data (int fd, http_head_t head, FILE * out)
{
    http_head_t h;
    const char * s = "Content-Length: ", * val;
    char buf[1024+1];
    int nlen = 0, nn = 0, n, eof=0;
    int rc;
    
    h = http_head_find (head, s, &val);
    if (h)
	nlen = atoi (val);
    else
    {
        h = http_head_find (head, "Connection: ", &val);
        if (h == NULL || strncmp (val, "close", 4))
            return HTTP_ERR_PROTO;
    }

    h = http_head_find (head, "Content-Type: ", &val);
    if (h == NULL)
        return HTTP_ERR_PROTO;
    if (strncmp (val, "text", 4))
        return read_binary (fd, nlen, out);
    
    do
    {
	rc = http_read_line (fd, buf, 1024, 1, &n, &eof);
	if (rc)
	    return rc;
	if (n > 0)
	    fwrite (buf, 1, n, out);
        if (nlen > 0)
        {
            nn += n;
            if (nlen == nn)
                break;
        }
    } while (eof == 0);
    return 0;
}


static int
     check_status (int code, int * statcode)
{
    if (code != HTTP_STAT_200)
    {
        fprintf (stderr, "** problem with response status=%d\n", code);
        if (statcode)
            *statcode = code;
        if (code == HTTP_STAT_400 ||
            code == HTTP_STAT_403 ||
            code == HTTP_STAT_404 ||
            code == HTTP_STAT_405)
            return HTTP_ERR_STAT;
    }
    return 0;
}


int
    http_parse_response (http_hd_t hd, int * statcode)
{
    char buf[300];
    int rc;
    int code = 0, nn = 0;

    if (statcode)
        *statcode = 0;
    rc = http_read_line (hd->fd, buf, 299, 1, &nn, NULL);
    if (rc)
	return rc;
    rc = parse_statline (buf, &code);
    if (rc)
	return rc;
    rc = check_status (code, statcode);
    if (rc)
        return rc;
    rc = parse_headers (hd->fd, &hd->head);
    if (rc)
	return rc;
    {
        http_head_t t;
        fprintf (stderr, "HTTP headers: \n");
        for (t = hd->head; t; t=t->next)
            fprintf (stderr, "%s", t->d);
        fprintf (stderr, "\n");
    }
    return 0;
}


int
     http_parse_data (http_hd_t hd, FILE * out)
{
    int rc = 0;

    rc = parse_data (hd->fd, hd->head, out);

    http_close (hd->fd);
    
    return rc;
}


const char *
     http_strerror (int rc)
{
    switch (rc)
    {
      case HTTP_ERR_SUCCESS: return "operation successfull";
      case HTTP_ERR_CONNECT: return "could not connect to host";
      case HTTP_ERR_INVADDR: return "invalid address";
      case HTTP_ERR_SOCK:    return "socket error";
      case HTTP_ERR_IO:      return "input/output error";
      case HTTP_ERR_PROTO:   return "protocol violation";
      case HTTP_ERR_STAT:    return "status code indicates an error";
      default:               return "unknown error";
    }
    return NULL;
}


int
http_hd_new (http_hd_t * r_hd)
{
    http_hd_t hd;

    if (!r_hd)
        return -1;
    hd = calloc (1, sizeof * hd);
    if (hd == NULL)
        abort ();
    *r_hd = hd;
    return 0;
}


void
http_hd_free (http_hd_t hd)
{
    if (!hd)
        return;
    http_head_free (hd->head);
    http_close (hd->fd);
    free (hd);
}