/* wptKeyPropsDlg.cpp - WinPT key properties dialog
 *	Copyright (C) 2000, 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"
#include "wptErrors.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptNLS.h"
#include "wptDlgs.h" 
#include "wptTypes.h"
#include "wptKeylist.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptKeyEdit.h"

int keyedit_ownertrust_get_val (void);


static void
do_change_ownertrust( winpt_key_t k, const char *s )
{
    char ot[64], *p;
    
    if( strstr( s, "ultimate" ) ) {
	listview_get_item_text( k->callback.ctl, k->callback.idx, 5, ot, sizeof ot-1 );
	if( (p = strrchr( ot, ']' )) )
	    strcpy( ot + (p - ot) + 2, "Ultimate" );
	else
	    strcpy( ot, "Ultimate" );
	listview_add_sub_item( k->callback.ctl, k->callback.idx, 5, ot );
	/* fixme: If we switch back from Ultimate to a lower level */
    }
} /* do_change_ownertrust */


static int
do_check_key( gpgme_key_t key )
{
    int okay = 0;
    okay = gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_EXPIRED, NULL, 0 );
    if( !okay )
	okay = gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_REVOKED, NULL, 0 );
    return okay;
} /* do_check_key */


static const char*
ownertrust_to_string (gpgme_editkey_t ctx, int val)
{
    const char * inf;
    int id;
    
    if (ctx)
        id = keyedit_ownertrust_get_val ();
    else if (!ctx && val)
        id = val;
    switch (id) {
    case 1: inf = _("Don't know");         break;
    case 2: inf = _("I do NOT trust");     break;
    case 3: inf = _("I trust marginally"); break;
    case 4: inf = _("I trust fully");      break;
    case 5:
    case 6: inf = _("I trust ultimately"); break;
    default:inf = _("Unknown");            break;
    }
    
    return inf;
} /* ownertrust_to_string */


static int
keyprops_show_photo (HWND dlg, gpgme_key_t key)
{
    RECT r;	
    POINT p;
    HWND h;
    const BYTE *img;
    DWORD imglen = 0;
    int pos=0;

    img = (const byte *)gpgme_key_get_string_attr( key, GPGME_ATTR_PHOTO, 
						    (void **)&imglen, 0 );
    if (!img || !imglen)
	return -1;
    FILE * f = fopen ("temp.jpg", "wb");
    if (f) {
	for (pos = 0; img[pos] != 0x10; pos++)
		;
	pos += 16;
	fwrite (img + pos, 1, imglen - pos, f);
	fwrite (img, 1, imglen, f);
	fclose (f);
    }

    h = GetDlgItem (dlg, IDC_KEYPROPS_IMG);
    GetWindowRect (h, &r);
    p.x = r.left + 5;
    p.y = r.top - 2;
    memset (&p, 0, sizeof p);
    PTD_jpg_show (h, &p, "temp.jpg");
    unlink ("temp.jpg");

    return 0;
}


BOOL CALLBACK
keyprops_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static winpt_key_t k;
    static gpgme_key_t key, sk;
    static int photo_done = 0;
    gpgme_editkey_t ek;
    gpgme_ctx_t ctx;
    gpgme_error_t ec;    
    refresh_cache_s rcs = {0};
    const char *inf;
    const char * sym_prefs;
    char info[2048], dummy_symprefs[] = {0x01, 0x00}; 
    u32 created, expires;
    int valid, ot, cancel = 0;
    int rc;
    
    switch (msg) {
    case WM_INITDIALOG:
        if (!lparam)
            dlg_fatal_error( dlg, "Could not get dialog param!" );
        k = (winpt_key_t)lparam;
	#ifndef LANG_DE
        SetWindowText( dlg, _("Key Properties") );
	SetDlgItemText (dlg, IDC_KEYPROPS_OT_CHANGE, _("&Change"));
	SetDlgItemText (dlg, IDC_KEYPROPS_REVOKERS, _("&Revokers"));
	SetDlgItemText (dlg, IDC_KEYPROPS_CHANGE_PWD, _("Change &Passwd"));
	#endif	

	photo_done = 0;
	get_seckey (k->keyid, &sk);
	if (sk)
	    k->is_protected = gpgme_key_get_ulong_attr( sk, GPGME_ATTR_IS_PROTECTED, NULL, 0 );
	if (get_pubkey (k->keyid, &key))
	    BUG (0);
        created = gpgme_key_get_ulong_attr( key, GPGME_ATTR_CREATED, NULL, 0 );
	expires = gpgme_key_get_ulong_attr( key, GPGME_ATTR_EXPIRES, NULL, 0 );
	sym_prefs = gpgme_key_get_string_attr( key, GPGME_ATTR_KEY_SYMPREFS, NULL, 0 );
	if (!sym_prefs)
	    sym_prefs = dummy_symprefs;
        valid = gpgme_key_get_ulong_attr (key, GPGME_ATTR_VALIDITY, NULL, 0);
        _snprintf (info, sizeof info -1,
                  "Type: %s\r\n"
                  "Key ID: %s\r\n"
                  "Algorithm: %s\r\n"
                  "Size: %s\r\n"
                  "Created: %s\r\n"
		  "Expires: %s\r\n"
                  "Validity: %s\r\n"
		  "Cipher: %s\r\n",
                  get_key_type( key ),
                  k->keyid, 
                  get_key_algo( key, 0 ),
                  get_key_size( key, 0 ),
                  get_key_created( created ),
		  get_key_expire_date( expires ),
                  gpgme_key_expand_attr( GPGME_ATTR_VALIDITY, valid ),
		  gpgme_key_expand_attr( GPGME_ATTR_KEY_SYMPREFS, *sym_prefs ) );
        SetDlgItemText( dlg, IDC_KEYPROPS_INFO, info );
        SetDlgItemText( dlg, IDC_KEYPROPS_FPR, get_key_fpr( key ) );
        ot = gpgme_key_get_ulong_attr( key, GPGME_ATTR_OTRUST, NULL, 0 );		
        inf = ownertrust_to_string( NULL, ot );
        SetDlgItemText( dlg, IDC_KEYPROPS_OT, inf );
        if( k->key_pair )
            EnableWindow( GetDlgItem( dlg, IDC_KEYPROPS_CHANGE_PWD ), TRUE );
	if( gpgme_key_count_items( key, GPGME_ATTR_REVKEY_FPR ) )
	    EnableWindow( GetDlgItem( dlg, IDC_KEYPROPS_REVOKERS ), TRUE );
	center_window (dlg);
        SetForegroundWindow (dlg);	
        return TRUE;
      
    case WM_PAINT:	
	if (photo_done == 0) {
	    photo_done = 1;
	    keyprops_show_photo (dlg, key);
	}
	break;

    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDOK:
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDC_KEYPROPS_OT_CHANGE:
	    if( do_check_key( key ) ) {
		msg_box( dlg, _("The status of this key is 'revoked' or 'expired'.\n"
				"You cannot change the ownertrust of such keys."), 
				_("WinPT Warning"), MB_ERR );
		return FALSE;
	    }
	    if( !k->key_pair && gpgme_key_get_ulong_attr( key, GPGME_ATTR_VALIDITY, NULL, 0 ) < 3 ) {
		rc = msg_box( dlg, _("This is a non-valid key.\n"
				     "Modifying the ownertrust has no effect on such keys.\n"
			             "Do you really want to continue?"), 
				     _("WinPT Warning"), MB_ICONWARNING|MB_YESNO );
		if (rc == IDNO)
		    return FALSE;
	    }
            GetDlgItemText( dlg, IDC_KEYPROPS_OT, info, sizeof info -1 );
            gpgme_editkey_new( &ek );
            gpgme_editkey_is_secret( ek, k->key_pair );
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYEDIT_OWNERTRUST,
                             dlg, (DLGPROC)keyedit_ownertrust_dlg_proc,
                             (LPARAM)ek, _("Change Ownertrust"), 
			     IDS_WINPT_KEYEDIT_OWNERTRUST );
            if (!gpgme_editkey_is_valid (ek))
                return FALSE;
            ec = gpgme_new (&ctx);
	    if (ec)
		BUG (0);
            gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_TRUST );
            ec = gpgme_op_editkey( ctx, k->keyid );
	    gpgme_editkey_release( ek );
                gpgme_release( ctx );
            if( ec ) {
                msg_box( dlg, gpgme_strerror( ec ), _("Ownertrust"), MB_ERR );
                return FALSE;
            }
            inf = ownertrust_to_string (ek, 0);
	    k->callback.new_val = keyedit_ownertrust_get_val ();
	    do_change_ownertrust (k, inf);
            SetDlgItemText (dlg, IDC_KEYPROPS_OT, inf);
            msg_box (dlg, _("Ownertrust successfully changed."), _("GnuPG Status"), MB_OK);
            
            /* reload only the keylist */
	    rcs.kr_reload = 1; rcs.kr_update = 1;
	    rcs.tr_update = 0;
	    DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_KEYCACHE, dlg,
			    keycache_dlg_proc, (LPARAM)&rcs );
	    get_seckey (k->keyid, &sk);
	    if (get_pubkey (k->keyid, &key))
		BUG (0);
            return TRUE;
            
        case IDC_KEYPROPS_CHANGE_PWD:
            keyedit_change_passwd (k, dlg);	    
            return TRUE;

	case IDC_KEYPROPS_REVOKERS:	    
	    dialog_box_param( glob_hinst, (LPCTSTR)IDD_WINPT_KEYREVOKERS, dlg,
			      key_revokers_dlg_proc, (LPARAM)key, _("Key Revokers"), 
			      IDS_WINPT_KEY_REVOKERS );
	    break;
        }
    }
    
    return FALSE;
} /* keyprops_dlg_proc */
