/* wptKeyRevokeDlg.cpp - Key revocation dialog
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"
#include "wptErrors.h"
#include "wptGPG.h"
#include "wptW32API.h"
#include "wptTypes.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptNLS.h"
#include "wptUTF8.h"


static const char *
mk_cert_fname( const char * keyid )
{
    static char fname[128];

    if( strlen( keyid ) > 32 )
	return NULL;
    _snprintf( fname, sizeof fname-1, "%s-revcert.asc", keyid );
    return fname;
} /* mk_cert_fname */


BOOL CALLBACK
key_revoke_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{	
    static winpt_key_t k;
    HWND list;	
    int idx, use_desc, chk = 0, pgpmode = 0;
    char desc[256], file[256], * p = NULL;
    char pwd[256];
    gpgme_editkey_t rev;
    gpgme_data_t rev_cert;
    gpgme_ctx_t c;
    gpgme_error_t err;
    
    switch( msg ) {
    case WM_INITDIALOG:
        if( !lparam )
            dlg_fatal_error(dlg, "Could not get dialog param!");
        k = (winpt_key_t )lparam;
#ifndef LANG_DE
        SetWindowText( dlg, _("Key Revocation") );
#endif
        SetDlgItemText( dlg, IDC_KEYREVOKE_HINT,
                       _("Please move this certificate to a medium where it can be "
                         "stored in a safe place (floppy, CDR, etc..). "
                         "If an attacker gets access to this certificate he can use it to "
                         "render your key unusable!") );
        list = GetDlgItem( dlg, IDC_KEYREVOKE_REASON );
	listbox_add_string( list, _("0. No reason specified") );
        listbox_add_string( list, _("1. Key has been compromised") );
        listbox_add_string( list, _("2. Key is superseded") );
        listbox_add_string( list, _("3. Key is no longer used") );
	/* we set the PGP revoke mode by default because it does not do any 
	 * harm and makes sure the revocation certificate is compatible with PGP.
	 */
	CheckDlgButton (dlg, IDC_KEYREVOKE_PGPMODE, BST_CHECKED);
	SetDlgItemText (dlg, IDC_KEYREVOKE_PGPMODE, _("Make output &PGP compatible"));
        SetForegroundWindow (dlg);
	center_window (dlg);
        return TRUE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE ) {
            SetDlgItemText( dlg, IDC_KEYREVOKE_PWD, "" );
            EndDialog( dlg, TRUE );
        }
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
	case IDC_KEYREVOKE_CHOOSE:
	    const char *s, * name;
	    name = mk_cert_fname( k->keyid );
	    s = get_filename_dlg( dlg, 1, _("Choose File to save the Certificate"), NULL, name );
	    if( s && *s )
		SetDlgItemText( dlg, IDC_KEYREVOKE_FILE, s );
	    return TRUE;

        case IDOK:
            list = GetDlgItem( dlg, IDC_KEYREVOKE_REASON );
            idx = SendMessage( list, LB_GETCURSEL, NULL, NULL );
            if( idx < 0 || idx > 3 ) {
		msg_box( dlg, _("Please select a reason."), _("Key Revocation"), MB_ERR );
                return FALSE;
            }
            if( !GetDlgItemText(dlg, IDC_KEYREVOKE_FILE, file, sizeof file-1 ) ) {
		msg_box( dlg, _("Please enter a filename."), _("Key Revocation"), MB_ERR );
                return FALSE;
            }
            use_desc = 1;
            if( !GetDlgItemText( dlg, IDC_KEYREVOKE_TEXT, desc, sizeof desc-1 ) )
                use_desc = 0;
            if( !GetDlgItemText( dlg, IDC_KEYREVOKE_PWD, pwd, sizeof pwd-1 ) ) {
		msg_box( dlg, _("Please enter the passphrase."), _("Key Revocation"), MB_ERR );
                return FALSE;
            }
            err = gpgme_editkey_new( &rev );
	    if( err )
		BUG( dlg );
	    
	    if( use_desc )
		p = wincp_to_utf8 (desc, strlen (desc));
	    /* we use the keyid to avoid charset problems and UTF8 encodings.*/
	    if( IsDlgButtonChecked( dlg, IDC_KEYREVOKE_PGPMODE ) )
		pgpmode = 1;
	    gpgme_revoke_set( rev, k->keyid, use_desc? p : NULL, idx, pgpmode, pwd );
            err = gpgme_data_new( &rev_cert );
	    if( !err )		
		err = gpgme_new( &c );
	    if( err )
		BUG( dlg );
            
            err = gpgme_op_revoke( c, rev, rev_cert );
            memset( &pwd, 0, sizeof pwd );
            if( err ) {
                msg_box( dlg, gpgme_strerror( err ), _("Key Revocation"), MB_ERR );
                gpgme_data_release( rev_cert );
                gpgme_editkey_release( rev );
                gpgme_release( c );
		free_if_alloc( p );
                return FALSE;
            }
            
            msg_box( dlg, _("Revocation certificate generated."), _("GnuPG Status"), MB_OK );
	    chk = file_exist_check( file );
	    if( !chk )
		log_box( _("Key Revocation"), MB_YESNO|MB_INFO, 
		         _("\"%s\" already exists.\nOverwrite the file?"), file );
	    if( idx == IDYES || chk )
		gpgme_data_release_and_set_file( rev_cert, file );
	    else
		gpgme_data_release( rev_cert );
            gpgme_editkey_release( rev );
            gpgme_release( c );
	    free_if_alloc( p );
	    EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDCANCEL:
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* key_revoke_dlg_proc */
