/* wptRegistry.cpp - W32 Registry access
 *	Copyright (C) 2000-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <stdio.h>

#include "wptErrors.h"
#include "wptW32API.h"
#include "wptGPG.h"
#include "wptRegistry.h"
#include "wptKeyserver.h"
#include "wptTypes.h"
#include "wptNLS.h"
#include "wptVersion.h"

#define rc_ok(rc) ((rc) == ERROR_SUCCESS)

struct reg_hotkey_s reg_hotkeys[] = {
    {"ClipEncrypt", NULL, 0},
    {"ClipDecrypt", NULL, 0},
    {"ClipSign",    NULL, 0},
    {"ClipSignEnc", NULL, 0},
    {"CwsEncrypt",  NULL, 0},
    {"CwsDecrypt",  NULL, 0},
    {"CwsSign",     NULL, 0},
    {"CwsSignEnc",  NULL, 0},
    {0}
};

winpt_reg_prefs_s reg_prefs;

#define WINPT_REG "Software\\WinPT"


void
free_reg_prefs (void)
{
    free_if_alloc (reg_prefs.backup.path);
    free_if_alloc (reg_prefs.kserv_conf);
    memset (&reg_prefs, 0, sizeof reg_prefs);
} /* free_reg_prefs */


/*
 * Register the given WinPT filetype.
 */
static int
regist_single_filetype (gpg_filetype *gfile)
{
    char icon[256], prog[256];
    
    memset (&icon, 0, sizeof (icon));
    GetModuleFileName (glob_hinst, prog, sizeof (prog)-1);
    _snprintf (icon, sizeof (icon) -1, "%s,%d", prog, gfile->nicon);
    return create_file_type (prog, gfile->ext, gfile->descr, icon);
} /* regist_single_filetype */


/*
 * Install the GPG related into the W32 resgistry, if the entry already
 * exists< the function returns immediately.
 */
int
regist_inst_gnupg( int create_mokey )
{
    int rc;
    HKEY reg;

    rc = RegOpenKeyEx( HKEY_CURRENT_USER, "Software\\GNU\\GnuPG", 0, KEY_READ, &reg );	
    if( rc_ok( rc ) ) {
	RegCloseKey( reg );
	return 0;	
    }
    rc = RegCreateKey( HKEY_CURRENT_USER, "Software\\GNU\\GnuPG", &reg );
    if( !rc_ok( rc ) )		
	return WPTERR_REGISTRY;	
    RegCloseKey( reg );
    if( create_mokey ) {
	rc = RegOpenKeyEx( HKEY_CURRENT_USER,  "Control Panel\\MingW32\\NLS", 0, KEY_READ, &reg );
	if( rc_ok( rc ) ) {
	    RegCloseKey( reg );
	    return 0;	
	}
	rc = RegCreateKey( HKEY_CURRENT_USER, "Control Panel\\MingW32\\NLS", &reg );
	if( !rc_ok( rc ) )
	    return WPTERR_REGISTRY;	
	RegCloseKey( reg );
    }

    return 0;
} /* regist_inst_gpg */


/*
 * Install WinPT into the W32 registry, if the entry already 
 * exists the function returns immediately.
 */
int
regist_inst_winpt( int with_ext, int * created )
{
    HKEY reg;
    char * p = NULL;
    int rc, i, id, n = 0;

    if( created )
	*created = 0;

    p = get_reg_entry( HKEY_CURRENT_USER, WINPT_REG, "Extensions" );
    if( p && *p == '1' )
	with_ext = 0;
    free_if_alloc( p );

    if( with_ext ) {
	id = msg_box( NULL, _("WinPT can register some GPG file types for you so they can "
	                      "be processed with a double click in the explorer.\n"
			      "Do you want to continue?"), _("WinPT"), MB_YESNO|MB_INFO );
	if( id == IDNO ) {
	    set_reg_entry( HKEY_CURRENT_USER, WINPT_REG, "Extensions", "1" );
	    goto start;
	}
	for( i = 0; gpg_filetypes[i].ext; i++ ) {
	    rc = RegOpenKeyEx( HKEY_CLASSES_ROOT, gpg_filetypes[i].ext, 0, KEY_READ, &reg );
	    if( rc_ok( rc ) ) {
		RegCloseKey( reg );
		id = log_box( _("WinPT WARNING"), MB_YESNO|MB_INFO, 
			      _("It seems there was already a '%s' file type registered by another application.\n"
			        "Do you want to overwrite it?"), gpg_filetypes[i].ext );
		if( id == IDNO )
		    continue;
	    }
	    regist_single_filetype( &gpg_filetypes[i] );
	    n++;
	}
    }
	
start:
    rc = RegOpenKeyEx( HKEY_CURRENT_USER, WINPT_REG, 0, KEY_READ, &reg );
    if( rc_ok( rc ) ) {	
	RegCloseKey( reg );
	rc = RegOpenKeyEx( HKEY_CURRENT_USER, WINPT_REG"\\Keyserver", 0, KEY_READ, &reg );
	if( !rc_ok( rc ) ) {
	    RegCreateKey( HKEY_CURRENT_USER, WINPT_REG"\\Keyserver", &reg );
	    RegCloseKey( reg );
	}
	p = get_reg_entry_keyserver ("Default");
	if (!p) {
	    char buf[16];
	    sprintf (buf, "%d", HKP_PORT);
	    set_reg_entry_keyserver ("Default_Port", buf);
	    set_reg_entry_keyserver ("Default", DEF_HKP_KEYSERVER);
	}
	free_if_alloc( p );
	if( n )
	    set_reg_entry( HKEY_CURRENT_USER, WINPT_REG, "Extensions", "1" );
	return 0;
    }
    rc = RegCreateKey( HKEY_CURRENT_USER, WINPT_REG, &reg );
    if( !rc_ok( rc ) )
	return WPTERR_REGISTRY;
    if( created )
	*created = 1;
    RegCloseKey( reg );
    if( n )
	set_reg_entry( HKEY_CURRENT_USER, WINPT_REG, "Extensions", "1" );
    return 0;
} /* regist_inst_winpt */


/*
 * Create a new filetype in the W32 registry.
 * We should really care of errors! Otherwise we can damage the registry!
 */
int
create_file_type( const char *exefile, const char *ext, const char *extname, char *iconfile )
{
    int rc;
    HKEY reg = NULL;
    char deficon[256], defexec[256], p_exefile[256];
    
    
    rc = RegCreateKey( HKEY_CLASSES_ROOT, ext, &reg );
    if( rc_ok( rc ) ) 
	rc = RegSetValueEx( reg, NULL, 0, REG_SZ, (byte *)extname, strlen( extname ) );
    if( rc_ok( rc ) )
	rc = RegCloseKey( reg );
    if( rc_ok( rc ) )    
	rc = RegCreateKey( HKEY_CLASSES_ROOT, extname, &reg );
    if( rc_ok( rc ) )
	rc = RegSetValueEx( reg, NULL, 0, REG_SZ, (byte *) extname, strlen( extname ) );
    if( rc_ok( rc ) )
	rc = RegCloseKey( reg );
    if( !rc_ok( rc ) ) {
	rc = WPTERR_REGISTRY;
	goto leave;
    }
    
    memset( &deficon, 0, sizeof deficon );
    _snprintf( deficon, sizeof deficon - 1, "%s\\DefaultIcon", extname );
    memset( &defexec, 0, sizeof defexec );
    _snprintf( defexec, sizeof defexec - 1, "%s\\shell\\open\\command", extname );	
    memset( &p_exefile, 0, sizeof p_exefile );
    _snprintf( p_exefile, sizeof p_exefile - 1, "%s %%1", exefile );
    
    rc = RegCreateKey( HKEY_CLASSES_ROOT, deficon, &reg );
    if( rc_ok( rc ) )
	rc = RegSetValueEx(reg, NULL, 0, REG_SZ, (byte *)iconfile, strlen( iconfile ) );
    if( rc_ok( rc ) )
	rc = RegCloseKey( reg );
    if( rc_ok( rc ) )
	rc = RegCreateKey( HKEY_CLASSES_ROOT, defexec, &reg );
    if( rc_ok( rc ) )
	rc = RegSetValueEx( reg, NULL, 0, REG_SZ, (byte *)p_exefile, strlen( exefile ) );
    if( rc_ok( rc ) )
	rc = RegCloseKey( reg );
    if( !rc_ok( rc ) ) {
	rc = WPTERR_REGISTRY;
	goto leave;	
    }
    
leave:
    if( reg )
	RegCloseKey( reg );
    return rc;
} /* create_file_type */


static char *
expand_path( const char *path )
{
    DWORD len;
    char *p;

    len = ExpandEnvironmentStrings( path, NULL, 0 );
    if( !len )
	return NULL;
    len += 1;
    p = new char[len];    
    if( !p )
	return NULL;
    len = ExpandEnvironmentStrings( path, p, len );
    if( !len ) {
	free_if_alloc( p );
	return NULL;
    }
    return p; 
} /* expand_path */


char *
get_reg_entry( HKEY root_key, const char * dir, const char * key )
{
    int rc;
    char text[384] = {0};
    DWORD nbytes, type, n1 = 0;
    HKEY reg_key = NULL;
    char * p = NULL, * tmp = NULL;
    
    rc = RegOpenKeyEx( root_key, dir, 0, KEY_QUERY_VALUE, &reg_key );
    if( !rc_ok( rc ) )
	goto leave;
    nbytes = sizeof text - 1;
    type = REG_SZ;
    
    rc = RegQueryValueEx (reg_key, key, 0, &type, (BYTE *)&text, &nbytes);
    if (!rc_ok (rc) || !nbytes)
	goto leave;
    
    if (type == REG_EXPAND_SZ && strchr (text, '%'))
	p = expand_path (text);
    else {
	p = new char[nbytes + 1];
	if (!p)
	    BUG (0);
	memcpy (p, text, nbytes);
	p[nbytes] = '\0';
    }
    
leave:
    if (reg_key)
	RegCloseKey (reg_key);
    return p;
} /* get_reg_entry */


int
set_reg_entry( HKEY root_key, const char * dir, const char * key,
	       const char * value )
{
    int rc = 0;
    HKEY reg_key;

    rc = RegOpenKeyEx( root_key, dir, 0, KEY_WRITE, &reg_key );
    if( !rc_ok( rc ) )
	return WPTERR_REGISTRY;
    rc = RegSetValueEx( reg_key, key, 0, REG_SZ, (BYTE *)value, strlen( value ) );
    if( !rc_ok( rc ) )
	rc = WPTERR_REGISTRY;	
    RegCloseKey( reg_key );
    return rc;
} /* set_reg_entry */


int
set_reg_key( HKEY root_key, const char * dir, const char * key,
	     const char * value )
{
    int rc = 0;
    HKEY reg_key;
    
    rc = RegOpenKeyEx( root_key, dir, 0, KEY_WRITE, &reg_key );
    if( !rc_ok( rc ) )
	return WPTERR_REGISTRY;
    
    rc = RegSetValueEx( reg_key, key, 0, REG_SZ, (BYTE *)value, strlen( value ) );	
    if( !rc_ok( rc ) ) {	
	if ( RegCreateKey( root_key, key, &reg_key ) != ERROR_SUCCESS ) {
	    rc = WPTERR_REGISTRY; 
	    goto leave;
	}
	rc = RegSetValueEx( reg_key, key, 0, REG_SZ, (BYTE *)value, strlen( value ) );	    	
	if ( !rc_ok( rc ) )
	    rc = WPTERR_REGISTRY;
    }
    
leave:
    RegCloseKey( reg_key );
    return rc;
} /* set_reg_key */


int
set_reg_entry_gpg( const char * key, const char * value )
{	
    return set_reg_entry( HKEY_CURRENT_USER, "Software\\GNU\\GnuPG", key, value );
} /* set_reg_entry_gpg */


int
set_reg_entry_mo( const char * value )
{	
    return set_reg_entry( HKEY_CURRENT_USER, "Control Panel\\Mingw32\\NLS", "MODir", value );
} /* set_reg_entry_mo */


char *
get_reg_entry_gpg( const char *key )
{
    return get_reg_entry( HKEY_CURRENT_USER, "Software\\GNU\\GnuPG", key );
} /* get_reg_entry_gpg */


char *
get_reg_entry_mo( void )
{	
    return get_reg_entry( HKEY_CURRENT_USER, "Control Panel\\Mingw32\\NLS", "MODir" );
} /* get_reg_entry_mo */


static const char * cfg [] = {
	NULL,
	"CacheTime",
	"WordWrap",
	"FastMode",
	"Viewer",
	"KeylistMode",
	"WipeMode",
	"AlwaysTrust",
	"AutoBackup",
	"BackupMode",
	"DisableHotkeys",	
	"NoCompressMultiMedia",	
	"Expert",
    };


int
set_reg_winpt_single (int id, int val)
{
    char buf[64];
    int rc;

    sprintf (buf, "%d", val);
    rc = set_reg_entry (HKEY_CURRENT_USER, WINPT_REG, cfg[id], buf);
    return rc;
}


int
get_reg_winpt_single (int id)
{
    char * buf = NULL;
    int val = 0;

    buf = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG, cfg[id]);
    if (buf && *buf != ' ')
	val = 1;
    else if (!buf)
	val = -1;
    free_if_alloc (buf);
    return val;
}


int
set_reg_winpt_prefs (winpt_reg_prefs_s * opt)
{
    char buf[128];
    int rc = 0, i;

    for (i=1; i < DIM (cfg); i++) {
	switch (i) {
	case CFG_CACHETIME:
	    sprintf (buf, "%d", opt->cache_time);
	    break;
	case CFG_WORDWRAP:
	    sprintf (buf, "%d", opt->word_wrap);
	    break;
	case CFG_WIPEMODE:
	    sprintf (buf, "%d", opt->wipe_mode);
	    break;
	case CFG_FASTMODE:
	    sprintf (buf, "%d", opt->use_tmpfiles);
	    break;
	case CFG_NOZIP_MMEDIA:
	    sprintf (buf, "%d", opt->no_zip_mmedia);
	    break;
	case CFG_VIEWER:
	    sprintf (buf, "%d", opt->use_viewer);
	    break;
	case CFG_KEYLISTMODE:
	    sprintf (buf, "%d", opt->keylist_mode);
	    break;
	case CFG_ALWAYSTRUST:
	    sprintf (buf, "%d", opt->always_trust);
	    break;
	case CFG_AUTOBACKUP:
	    sprintf (buf, "%d", opt->auto_backup);
	    break;
	case CFG_AUTOBAKMODE:
	    sprintf (buf, "%d", opt->backup.mode);
	    break;
	case CFG_DISHOTKEYS:
	    sprintf (buf, "%d", opt->no_hotkeys);
	    break;
	case CFG_EXPERT:
	    sprintf (buf, "%d", opt->expert);
	    break;
	}
	rc = set_reg_entry (HKEY_CURRENT_USER, WINPT_REG, cfg[i], buf);
	if (rc)
	    goto leave;
    }

    if (opt->backup.path) {
	rc = set_reg_entry (HKEY_CURRENT_USER, WINPT_REG, "BackupPath",
			    opt->backup.path);
	if (rc)
	    goto leave;
    }
    if (opt->kserv_conf) {
	rc = set_reg_entry (HKEY_CURRENT_USER, WINPT_REG, "KeyserverConfig",
			    opt->kserv_conf);
	if (rc)
	    goto leave;
    }

    for (i=0; reg_hotkeys[i].reg_entry; i++) {
	strcpy (buf, " ");
	if (reg_hotkeys[i].enabled)
	    strcpy (buf, reg_hotkeys[i].key);
	rc = set_reg_key (HKEY_CURRENT_USER, WINPT_REG, 
			  reg_hotkeys[i].reg_entry, buf);
	if (rc)
	    break;
    }
    
leave:
    if (rc) {
	msg_box (NULL, _("Could not write to Registry."), _("Preferences"), MB_ERR);
	return rc;
    }
    return 0;
} /* set_reg_winpt_prefs */


int
set_reg_winpt_flag (const char * name, int val)
{
    return set_reg_entry (HKEY_CURRENT_USER, WINPT_REG, name, val? "1" : "0");
} /* set_reg_winpt_flag */


int
get_reg_winpt_flag (const char * name)
{
    char * buf;
    int flag = 0;

    buf = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG, name);
    if (buf && buf[0] == '1')
	flag = 1;
    else if (!buf || buf && buf[0] != '0')
	flag = -1;
    free_if_alloc (buf);
    return flag;
} /* get_reg_winpt_flag */


int
get_reg_winpt_prefs (winpt_reg_prefs_s * opt)
{
    char * val = NULL;
    int i;

    for (i=1; i < DIM (cfg); i++) {
	val = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG, cfg[i]);
	if (!val || *val == ' ') {
	    free_if_alloc (val);
	    continue;
	}
	switch (i) {
	case CFG_CACHETIME:
	    opt->cache_time = atol (val);
	    /* We do NOT support passphrase caching for more than an hour.
	     * Perhaps we should allow it, but for now we silently drop this.
	     */
	    if (opt->cache_time > 3600)
		opt->cache_time = 3600;
	    break;
	case CFG_WORDWRAP:
	    opt->word_wrap = atol (val);
	    break;
	case CFG_FASTMODE:
	    opt->use_tmpfiles = atol (val);
	    break;
	case CFG_NOZIP_MMEDIA:
	    opt->no_zip_mmedia = atol (val);
	    break;
	case CFG_VIEWER:
	    opt->use_viewer = atol (val);
	    break;
	case CFG_KEYLISTMODE:
	    opt->keylist_mode = atol (val);
	    break;
	case CFG_WIPEMODE:
	    opt->wipe_mode = atol (val);
	    break;
	case CFG_DISHOTKEYS:
	    opt->no_hotkeys = atol (val);
	    break;
	case CFG_ALWAYSTRUST:
	    opt->always_trust = atol (val);
	    break;
	case CFG_AUTOBACKUP:
	    opt->auto_backup = atol (val);
	    break;
	case CFG_AUTOBAKMODE:
	    opt->backup.mode = atol (val);
	    break;
	case CFG_EXPERT:
	    opt->expert = atol (val);
	    break;
	}
	free_if_alloc (val);
    }    

    val = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG, "BackupPath");
    if (val && val[0] != ' ')
	opt->backup.path = m_strdup (val);
    free_if_alloc (val);

    val = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG, "KeyserverConfig");
    if (val && val[0] != ' ')
	opt->kserv_conf = m_strdup (val);
    free_if_alloc (val);
    
    for (i=0; reg_hotkeys[i].reg_entry; i++) {
	val = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG, reg_hotkeys[i].reg_entry);
	if (val && val[0] != ' ') {
	    reg_hotkeys[i].key[0] = *val;
	    reg_hotkeys[i].enabled = 1;
	}
	else
	    reg_hotkeys[i].enabled = 0;
	free_if_alloc (val);
    }
    return 0;
} /* get_reg_winpt_prefs */


char *
get_reg_entry_keyserver (const char *name)
{
    char * p = get_reg_entry (HKEY_CURRENT_USER, WINPT_REG"\\Keyserver", name);
    if (p && !strcmp (p, "")) {
	free_if_alloc (p);
	return NULL;
    }
    return p;
} /* get_reg_entry_keyserver */


void
get_reg_proxy_prefs (char ** host, int * port, char ** user, char ** pass)
{
    if (host)
	*host = get_reg_entry_keyserver ("Host");
    if (user)
	*user = get_reg_entry_keyserver ("User");
    if (pass)
	*pass = get_reg_entry_keyserver ("Pass");
    if (port) {
	char * p = get_reg_entry_keyserver ("Port");
	if (p) {
	    *port = atol (p);
	    free_if_alloc (p);
	}
    }
} /* get_reg_proxy_prefs */


int
set_reg_entry_keyserver( const char * name, const char * val )
{
    return set_reg_entry( HKEY_CURRENT_USER, WINPT_REG"\\Keyserver", name, val );
} /* set_reg_entry_keyserver */


int
set_reg_proxy_prefs( const char * host, int port, const char * user, const char * pass )
{
    int rc;
    
    rc = set_reg_entry_keyserver( "Host", host? host : "" );
    if( !rc ) {
	char buf[32];
	sprintf( buf, "%d", port );
	rc = set_reg_entry_keyserver( "Port", buf );
    }
    if( !rc )
	rc = set_reg_entry_keyserver( "User", user? user : "" );
    if( !rc )
	rc = set_reg_entry_keyserver( "Pass", pass? pass : "" );
    return rc;
} /* set_reg_proxy_prefs */
