// ---------------------------------------------------------------------------
//  FM Sound Generator - Core Unit
//  Copyright (C) cisc 1998, 1999.
// ---------------------------------------------------------------------------
//  $Id: fmgen.cpp,v 1.10 1999/07/22 15:55:44 cisc Exp $
// ---------------------------------------------------------------------------
//  Ql:
//  FM sound generator for M.A.M.E., written by Tatsuyuki Satoh.
//
//  :
//      SSG-Type Envelop
//
//  :
//      TvO[g̐ݒ̓O[o
//
#include "headers.h"
#include "misc.h"
#include "fmgen.h"

#define LOGNAME "fmgen"
#include "diag.h"

#if !FM_LIGHT && !defined(_DEBUG) && !defined(__OS2__)
    #define FM_USE_X86_CODE         // LFO  x86 ł̂ݗL
#endif

// ---------------------------------------------------------------------------
//  Global Table/etc
//
namespace FM
{
#ifdef __OS2__
inline int Min(int x, int y) { return (x < y) ? x : y; }
#endif
    const uint8 notetable[128] =
    {
         0,  0,  0,  0,  0,  0,  0,  1,  2,  3,  3,  3,  3,  3,  3,  3,
         4,  4,  4,  4,  4,  4,  4,  5,  6,  7,  7,  7,  7,  7,  7,  7,
         8,  8,  8,  8,  8,  8,  8,  9, 10, 11, 11, 11, 11, 11, 11, 11,
        12, 12, 12, 12, 12, 12, 12, 13, 14, 15, 15, 15, 15, 15, 15, 15,
        16, 16, 16, 16, 16, 16, 16, 17, 18, 19, 19, 19, 19, 19, 19, 19,
        20, 20, 20, 20, 20, 20, 20, 21, 22, 23, 23, 23, 23, 23, 23, 23,
        24, 24, 24, 24, 24, 24, 24, 25, 26, 27, 27, 27, 27, 27, 27, 27,
        28, 28, 28, 28, 28, 28, 28, 29, 30, 31, 31, 31, 31, 31, 31, 31,
    };

    uint32 tltable[FM_TLPOS + FM_TLENTS];
    #if FM_LIGHT
        uint32 cltable[FM_CLENTS*2];
    #endif

    uint32 dltable[3][FM_EGENTS];
    int32 sinetable[FM_OPSINENTS];
    uint32 pmtable[8][FM_LFOENTS];
    uint32 amtable[4][FM_LFOENTS];
    uint32 lfotable[8];

    static bool tablemade = false;
}

#include "fmgeninl.h"

namespace FM
{

// ---------------------------------------------------------------------------
//  e[u쐬
//
void MakeTable()
{
    if (!tablemade)
    {
        int i;
        Operator::MakeTable();
        tablemade = true;
        for (i=-FM_TLPOS; i<FM_TLENTS; i++)
            tltable[FM_TLPOS + i] = uint(65535.0 * pow(2.0, i * -16. / FM_TLENTS));
#if FM_LIGHT
        for (i=0; i<FM_CLENTS; i++)
        {
            int c = int(((1 << (16 + FM_ISHIFT)) - 1) * pow(2.0, i * -16. / FM_CLENTS));
            cltable[i*2  ] = c;
            cltable[i*2+1] = -c;
            LOG2("cltable[%4d*2] = %6d\n", i, cltable[i*2]);
        }
#endif
    }

    static const double pms[8] = { 0, 3.4, 6.7, 10, 14, 20, 40, 80 };
    static const double ams[4] = { 0, 1.4, 5.9, 11.8 };

    int i;
    for (i=0; i<8; i++)
    {
        double pmb = pms[i] / 2400.0;
        for (int j=0; j<FM_LFOENTS; j++)
        {
            pmtable[i][j] =
                uint32(0x10000 * pow(2.0, pmb * sin(2 * FM_PI * j / FM_LFOENTS)));
//          LOG3("PM[%d][%3d] = %8x\n", i, j, pmtable[i][j]);
        }
    }
    for (i=0; i<4; i++)
    {
        double amb = -ams[i] / 40.0;
        for (int j=0; j<FM_LFOENTS; j++)
        {
            amtable[i][j] =
                uint32(0x10000 * pow(10.0, amb * (1 + sin(2 * FM_PI * j / FM_LFOENTS))));
//          LOG3("AM[%d][%3d] = %8x\n", i, j, amtable[i][j]);
        }
    }
}

//  `bṽTvO[gƐ鉹̃TvO[gƂ̔ݒ
void MakeTimeTable(uint ratio)
{
    Operator::MakeTimeTable(ratio);

    static const int table[8] =
    {
        109,  78,  72,  68,  63,  45,  9,  6
    };

    for (int i=0; i<8; i++)
    {
        lfotable[i] = uint(78.0 / 10000.0 / table[i] * ratio * (1 << (FM_LFOBITS + FM_LFOCBITS - FM_RATIOBITS)));
        LOG3("lfotable[%d] = %8x(%f)\n", i, lfotable[i], lfotable[i]*1.0/(1<<(FM_LFOBITS + FM_LFOCBITS)) );
    }
}

// ---------------------------------------------------------------------------
//  Envelope Generator
//
//  n = .75dB step
#define EGC(n)              (n << (FM_EGCBITS + FM_EGBITS - 7))
#define EGINDEX(n)          ((n) << (FM_EGCBITS + FM_EGBITS))

#if FM_LIGHT
    #define TOTLL(l)        (l << (FM_CLBITS-7+1))
#else
    #define TOTLL(l)        tltable[FM_TLPOS + (l << (FM_TLBITS-7))]
#endif

//  
void Operator::Prepare()
{
    //  PG Part
    pgdcount = (dp + dttable[detune + bn]) * multable[multiple];
    LOG1("%.8x ", pgdcount);

    // EG Part
    tllinear = TOTLL(tl);
    ksr = bn >> (3-ks);

    switch (phase)
    {
    case attack:
        egdcount = ar ? attacktable[Min(63, ar+ksr)] : 0;
        break;
    case decay:
        egdcount = dr ? decaytable [Min(63, dr+ksr)] : 0;
        eglimit = EGINDEX(dlnormal) + EGC(sl);
        break;
    case sustain:
        egdcount = sr ? decaytable [Min(63, sr+ksr)] : 0;
        break;
    case release:
        egdcount =      decaytable [Min(63, rr+ksr)];
        break;
    }
}

//  envelope  phase ύX
void Operator::ShiftPhase(EGPhase nextphase)
{
    if (nextphase == next)
        nextphase = EGPhase(phase + 1);

    switch (nextphase)
    {
    case attack:        // Attack Phase
        tl = tll;
        tllinear = TOTLL(tl);
        if ((ar+ksr) < 62)
        {
            if (phase == release)
                egcount = d2atable[(egcount >> (FM_EGCBITS+FM_EGBITS-FM_EGSBITS)) & (FM_EGSENTS-1)];
            else
                egcount = EGINDEX(dlattack);

            eglimit = EGINDEX(dlattack) + EGC(128);
            egdcount = ar ? attacktable[Min(63, ar+ksr)] : 0;
            phase = attack;
            LOG3("ar=%2d  dcount=%8d (%8d)\n", ar+ksr, egdcount, dltable[(egcount >> FM_EGCBITS) & (FM_EGENTS-1)]);
            break;
        }
    case decay:         // Decay Phase
        if (sl)
        {
            egcount = EGINDEX(dlnormal);
            eglimit = EGINDEX(dlnormal) + EGC(sl);
            egdcount = dr ? decaytable[Min(63, dr+ksr)] : 0;
            phase = decay;
            break;
        }
    case sustain:       // Sustain Phase
        egcount = EGINDEX(dlnormal) + EGC(sl);
        eglimit = EGINDEX(dlnormal) + EGC(128);
        egdcount = sr ? decaytable[Min(63, sr+ksr)] : 0;
        phase = sustain;
        break;

    case release:       // Release Phase
        if (phase == attack || (egcount < EGC(128) && phase != off))
        {
            if (phase == attack)
                egcount = a2dtable[(egcount >> (FM_EGCBITS+FM_EGBITS-FM_EGSBITS)) & (FM_EGSENTS-1)];
            egdcount = rr ? decaytable[Min(63, rr+ksr)] : 0;
            eglimit = EGINDEX(dlnormal) + EGC(128);
            phase = release;
            break;
        }
    case off:           // off
    default:
        egcount = EGINDEX(dlnormal) + EGC(128) - 1;
        eglimit = EGINDEX(dlnormal) + EGC(128);
        egdcount = 0;
        phase = off;
        break;
    }
}

#if FM_LIGHT
    #define ToTLE(a)    (Counter(a * (FM_CLENTS*2 - 1)) & ~1)
#else
    #define ToTLE(a)    Counter(65535.0 * pow(10.0, a * (-96. / 20.0)))
#endif

//  e[u쐬
void Operator::MakeTable()
{
    //  EG Part
    int i;
    const double c = 0.0125;            // K(^^;
    const double logcinv = 1.0 / log(c);
    const double m  = 1.0 / (1.0 - c);
    const double minv = 1.0 - c;

    // make dB->linear conversion table
    for (i=0; i<FM_EGENTS; i++)
    {
        double x = double(i) / FM_EGENTS;
        dltable[dlnormal][i] = ToTLE(x);
        dltable[dlattack][i] = ToTLE((1 - m * (1 - pow(c, x))));
        LOG2("dlnormal[%d] = %d\n", i, dltable[dlnormal][i]);
    }

    for (i=0; i<FM_EGSENTS; i++)
    {
        double x, y;

        x = double(i) / FM_EGSENTS;
        y = 1.0 - m * (1 - pow(c, x));
        a2dtable[i] = EGINDEX(dlnormal  )     + Counter(((1 << (FM_EGCBITS + FM_EGBITS))-1) * y);
        y = 1.0 - log(1.0 - (1.0 - x) * minv) * logcinv;
        d2atable[i] = EGINDEX(dlattack+1) - 1 - Counter(((1 << (FM_EGCBITS + FM_EGBITS))  ) * y);

//      LOG3("%4d  a2d=%8x  d2a=%8x\n", i, a2dtable[i], d2atable[i]);
    }

    //  PG Part
    static const uint8 table[128] =
    {
         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, //FD=0
         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
         0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  2,  2,  2,  2, //FD=1
         2,  3,  3,  3,  4,  4,  4,  5,  5,  6,  6,  7,  8,  8,  8,  8,
         1,  1,  1,  1,  2,  2,  2,  2,  2,  3,  3,  3,  4,  4,  4,  5, //FD=2
         5,  6,  6,  7,  8,  8,  9, 10, 11, 12, 13, 14, 16, 16, 16, 16,
         2,  2,  2,  2,  2,  3,  3,  3,  4,  4,  4,  5,  5,  6,  6,  7, //FD=3
         8,  8,  9, 10, 11, 12, 13, 14, 16, 17, 19, 20, 22, 22, 22, 22,
    };

    for (i=0; i<128; i++)
    {
        int val = table[i] << 1;
        dttable[i    ] =  val;
        dttable[i+128] = -val;
    }

    //  OP Part
#if FM_LIGHT
    for (i=1; i<FM_OPSINENTS/2; i++)
    {
        double s = -log10(sin(2. * FM_PI * i / FM_OPSINENTS)) * 20. / 96.;
        int x = Min(int32(FM_CLENTS * s), FM_CLENTS-1);
        sinetable[i             ] = x * 2;
        sinetable[FM_OPSINENTS-i] = x * 2 + 1;

        LOG2("sin[%4d] = %8d\n", i, sinetable[i]);
    }
    sinetable[0] = sinetable[FM_OPSINENTS / 2] = (FM_CLENTS - 1) * 2;
#else
    for (i=0; i<FM_OPSINENTS; i++)
    {
        sinetable[i] = int32((1 << FM_ISHIFT) * sin(2. * FM_PI * i / FM_OPSINENTS));
    }
#endif
}

//  NbNETvO[gɈˑe[u쐬
void Operator::MakeTimeTable(uint ratio)
{
    int i;

    //  EG Part
    static const int16 table[16] =
    {
            0,     2,     4,     8,    16,    32,    64,   128,
          256,   512,  1024,  2048,  4096,  7475, 14238, 29900
    };

    double ra = double(ratio) / (1 << FM_RATIOBITS)
              * (1 << FM_EGCBITS) * FM_EGENTS / FM_ARRATE;
    double rd = double(ratio) / (1 << FM_RATIOBITS)
              * (1 << FM_EGCBITS) * FM_EGENTS / FM_DRRATE;

    Counter* pa = attacktable;
    Counter* pd = decaytable;

    for (i=0; i<16; i++)
    {
        for (int j=4; j<8; j++)
        {
            *pa++ = Counter(ra * (table[i] * j / 4));
            *pd++ = Counter(rd * Min(32767, i ? (1 << i) * j / 4 : 0));
            LOG4("%2d-%d  %10d  %10d\n", i, j-4, pa[-1], pd[-1]);
        }
    }
    attacktable[62] = attacktable[63] = EGC(128);

    // PG Part
    for (i=0; i<16; i++)
    {
        int mul = i ? i * 2 : 1;
#if ((2 + FM_RATIOBITS) > FM_PGBITS)
        multable[i] = (mul * ratio) >> (2 + FM_RATIOBITS - FM_PGBITS);
#else
        multable[i] = (mul * ratio) << (FM_PGBITS - FM_RATIOBITS - 2));
#endif
    }
}

//  Block/F-Num
void Operator::SetFNum(uint f)
{
    dp = (f & 2047) << ((f >> 11) & 7);
    bn = notetable[(f >> 7) & 127];
    LOG1("dp = %.8x\n", dp);
}

//  static tables
uint32 Operator::multable[16];
int32  Operator::dttable[256];
Operator::Counter Operator::attacktable[64];
Operator::Counter Operator::decaytable[64];
Operator::Counter Operator::a2dtable[FM_EGSENTS];
Operator::Counter Operator::d2atable[FM_EGSENTS];


// ---------------------------------------------------------------------------
//  4-op Channel
//
const uint8 Channel4::fbtable[8] = { 31, 7, 6, 5, 4, 3, 2, 1 };

// Zbg
void Channel4::Reset()
{
    op[0].Reset();
    op[1].Reset();
    op[2].Reset();
    op[3].Reset();
}

//  Calc ̗p
int Channel4::Prepare()
{
    op[0].Prepare();
    op[1].Prepare();
    op[2].Prepare();
    op[3].Prepare();
    LOG0("\n");
    return op[0].IsOn() | op[1].IsOn()
         | op[2].IsOn() | op[3].IsOn();
}

//  F-Number/BLOCK ݒ
void Channel4::SetFNum(uint f)
{
    uint dp = (f & 2047) << ((f >> 11) & 7);
    uint bn = notetable[(f >> 7) & 127];
    op[0].SetDPBN(dp, bn);
    op[1].SetDPBN(dp, bn);
    op[2].SetDPBN(dp, bn);
    op[3].SetDPBN(dp, bn);
}

//  L[
void Channel4::KeyControl(uint key)
{
    if (key & 0x10) op[0].KeyOn(); else op[0].KeyOff();
    if (key & 0x20) op[1].KeyOn(); else op[1].KeyOff();
    if (key & 0x40) op[2].KeyOn(); else op[2].KeyOff();
    if (key & 0x80) op[3].KeyOn(); else op[3].KeyOff();
}

//  ASYݒ
void Channel4::SetAlgorithm(uint algo)
{
    static const uint8 table1[8][6] =
    {
        { 0, 1, 1, 2, 2, 3 },   { 1, 0, 0, 1, 1, 2 },
        { 1, 1, 1, 0, 0, 2 },   { 0, 1, 2, 1, 1, 2 },
        { 0, 1, 2, 2, 2, 1 },   { 0, 1, 0, 1, 0, 1 },
        { 0, 1, 2, 1, 2, 1 },   { 1, 0, 1, 0, 1, 0 },
    };

    in [0] = &buf[table1[algo][0]];
    out[0] = &buf[table1[algo][1]];
    in [1] = &buf[table1[algo][2]];
    out[1] = &buf[table1[algo][3]];
    in [2] = &buf[table1[algo][4]];
    out[2] = &buf[table1[algo][5]];
    op[0].ResetFB();
}

// ---------------------------------------------------------------------------
//  4-op Channel
//  PTv

//  ISample  envelop count (2) ɕϊVtg
#define IS2EC_SHIFT     ((20 + FM_PGBITS) - (16 + FM_ISHIFT))

#ifdef FM_USE_X86_CODE
    #include "fmgenx86.h"
#else

// : s = 20+FM_PGBITS = 29
#define Sine(s) sinetable[((s) >> (20+FM_PGBITS-FM_OPSINBITS))&(FM_OPSINENTS-1)]

#if FM_LIGHT
    #define CMUL(a, b)          (a + b)
    static inline FM::ISample OPMUL(uint32 a, uint32 b)
    {
        return ((a + b) < FM_CLENTS * 2) ? FM::cltable[a + b] : 0;
    }
#else
    #define CMUL(a, b)          ((a * b) >> 16)
    #define OPMUL(a, b)         (a * b)
#endif

//  EG vZ
inline int32 FM::Operator::EGCalc()
{
    egcount += egdcount;
    if (egcount >= eglimit)
        ShiftPhase(next);
    return CMUL(tllinear, dltable[0][egcount >> FM_EGCBITS]);
}

//  PG vZ
//  ret:2^(20+PGBITS) / cycle
inline uint32 FM::Operator::PGCalc(/* int32 lfoa */)
{
    return pgcount += pgdcount;     // + dpms * lfoa;
}

//  OP vZ
//  in: ISample (ő 8)
inline FM::ISample FM::Operator::Calc(ISample in)
{
    return OPMUL(EGCalc(), Sine(PGCalc() + (in << (2 + IS2EC_SHIFT))));
}

//  OP (FB) vZ
//  Self Feedback ̕ϒő = 4
inline FM::ISample FM::Operator::CalcFB(uint fb)
{
    ISample in = out + out2;
    out2 = out;
    out = OPMUL(EGCalc(), Sine(PGCalc() + ((in << (1 + IS2EC_SHIFT)) >> fb)));
    return out;
}

#undef Sine

//  
ISample Channel4::Calc()
{
    buf[1] = buf[2] = buf[3] = 0;

    buf[0] = op[0].CalcFB(fb);
    *out[0] += op[1].Calc(*in[0]);
    *out[1] += op[2].Calc(*in[1]);
    return *out[2] + op[3].Calc(*in[2]);
}

//  C++ łł LFO T|[ĝ Calc Ɠ
ISample Channel4::Calc2(uint)
{
    buf[1] = buf[2] = buf[3] = 0;

    buf[0] = op[0].CalcFB(fb);
    *out[0] += op[1].Calc(*in[0]);
    *out[1] += op[2].Calc(*in[1]);
    return *out[2] + op[3].Calc(*in[2]);
}
#endif // FM_USE_X86_CODE

}   // namespace FM
