// ---------------------------------------------------------------------------
//  PC-8801 emulator
//  Copyright (C) cisc 1999.
// ---------------------------------------------------------------------------
//  foCXƐisǗ
// ---------------------------------------------------------------------------
//  $Id: pc88.cpp,v 1.25 1999/07/04 08:48:56 cisc Exp $

//  Memory Bus Banksize <= 0x400

#include "headers.h"
#include "pc88/pc88.h"
#include "pc88/memory.h"
#include "pc88/config.h"
#include "draw.h"
#include "file.h"
#include "status.h"
#include "device_i.h"

#define LOGNAME "pc88"
#include "diag.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//  \zEj
//
PC88::PC88()
  : base('BASE'), cpu1('CPU1'), mem1('MEM1'), dmac('DMAC'),
    knj1('KNJ1'), knj2('KNJ2'), scrn('SCRN'), intc('INTC'),
    crtc('CRTC'), cpu2('CPU2'), fdc ('FDC '), subsys('SUB '),
    sio ('SIO '), cdif('CDIF'), opn1('OPN1', 0x80), opn2('OPN2', 0x80)
{
//    assert((1 << MemoryBus::pagebits) <= 0x400);
    clock = 100;
    DIAGINIT(&cpu1);
    dexc = 0;
}

PC88::~PC88()
{
    DumpMemory();
}

// ---------------------------------------------------------------------------
//  
//
bool PC88::Init(Draw* _draw)
{
    draw = _draw;
    if (!Scheduler::Init())
        return false;
    if (!diskmgr.Init())
        return false;
    if (!draw->Init(640, 400, 8))
        return false;
    if (!bus1.Init(portend, 0x10000 >> MemoryBus::pagebits, cpu1.GetPages()))
        return false;
    if (!bus2.Init(portend2, 0x10000 >> MemoryBus::pagebits, cpu2.GetPages()))
        return false;

    if (!ConnectDevices())
        return false;
    if (!ConnectDevices2())
        return false;

    Reset();
    region.Reset();
    palchanged = true;
    clock = 1;
    refreshcount = 0;
    return true;
}

// ---------------------------------------------------------------------------
//  s
//  1 tick = 10s
//
int PC88::Proceed(uint ticks, uint clk)
{
    clock = Max(1, clk);
    return Scheduler::Proceed(ticks);
}

// ---------------------------------------------------------------------------
//  s
//
int PC88::Execute(int ticks)
{
    int exc = ticks * clock;
    if (!(cpumode & stopwhenidle) || subsys.IsBusy() || fdc.IsBusy())
    {
        if ((cpumode & 1) == ms11)
            exc = Z80::ExecDual(&cpu1, &cpu2, exc);
        else
            exc = Z80::ExecDual2(&cpu1, &cpu2, exc);
    }
    else
    {
        exc = Z80::ExecSingle(&cpu1, &cpu2, exc);
    }
    exc += dexc;
    dexc = exc % clock;
    return exc / clock;
}

// ---------------------------------------------------------------------------
//  sNbNύX
//
void PC88::Shorten(int ticks)
{
    Z80::StopDual(ticks * clock);
}

int PC88::GetTicks()
{
    int tick = Z80::GetCCount() + dexc;
//  int id = Z80::GetCurrentCPUID() & 0xff;
//  LOG3("%c: t:%.8x  a+t:%.8x\n", id, tick, tick / clock);
    return tick / clock;
}

// ---------------------------------------------------------------------------
//  ʍXV
//
void PC88::UpdateScreen()
{
    int bpl;
    uint8* image;

    statusdisplay.UpdateDisplay();
//  statusdisplay.Show(10, 0, "PC = %.4x", cpu1.GetPC());

    if (++refreshcount >= refreshtiming)
    {
        refreshcount = 0;
        if (draw->Lock(&image, &bpl))
        {
            LOG2("(%d -> %d) ", region.top, region.bottom);
            crtc.UpdateScreen(image, bpl, region);
            LOG2("(%d -> %d) ", region.top, region.bottom);
            scrn.UpdateScreen(image, bpl, region);
            LOG2("(%d -> %d)\n", region.top, region.bottom);
            if (scrn.UpdatePalette(draw))
                palchanged = true;
            draw->Unlock();
            if ((palchanged || region.Valid()) && draw->ReadyToDraw() )
            {
                draw->DrawScreen(region);
                region.Reset();
                palchanged = false;
            }
        }
/*
    static int count = 0;
    {
        char buf[16];
        sprintf(buf, "scrn%.4d.bin", count++);
        FileIO fio(buf, FileIO::create);
        draw->Lock(&image, &bpl);
        fio.Write(image+640*16, 640*16);
        draw->Unlock();
    }
/**/
    }
}

// ---------------------------------------------------------------------------
//  Zbg
//
void PC88::Reset()
{
    base.SetFDBoot(diskmgr.GetCurrentDisk(0) >= 0);
    base.Reset();       // Switch ֌W̍XV

    bool isv2 = (bus1.In(0x31) & 0x40) != 0;

    if (isv2)
        dmac.Connect(mem1.GetTVRAM(), 0xf000, 0x1000);
    else
        dmac.Connect(mem1.GetRAM(), 0, 0x10000);

    opn1.SetOPNMode((cfgflags & Config::enableopna) != 0);
    opn1.Enable(isv2 || !(cfgflag2 & Config::disableopn44));
    opn2.SetOPNMode((cfgflags & Config::opnaona8) != 0);
    opn2.Enable((cfgflags & (Config::opnaona8 | Config::opnona8)) != 0);

    bus1.Out(pres, 0);
    bus1.Out(0x30, 1);
    bus1.Out(0x30, 0);
    bus1.Out(0x31, 0);
    bus1.Out(0x32, 0x80);
    bus1.Out(0x34, 0);
    bus1.Out(0x35, 0);
    bus1.Out(0x40, 0);
    bus1.Out(0x53, 0);
    bus1.Out(0x5f, 0);
    bus1.Out(0x70, 0);
    bus1.Out(0x99, 0x10);
    bus1.Out(0xe2, 0);
    bus1.Out(0xe4, 0);
    bus1.Out(0xe6, 0);
    bus1.Out(0xf1, 1);
    bus2.Out(pres2, 0);

//  statusdisplay.Show(10, 1000, "CPUMode = %d", cpumode);
}

// ---------------------------------------------------------------------------
//  foCXڑ
//
bool PC88::ConnectDevices()
{
    const static Bus::Connector c_cpu1[] =
    {
        { pres, Bus::portout, Z80::reset },
        { pirq, Bus::portout, Z80::irq },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&cpu1, c_cpu1)) return false;
    if (!cpu1.Init(&bus1, piack)) return false;

    const static Bus::Connector c_base[] =
    {
        { pres, Bus::portout, Base::reset },
        { vrtc, Bus::portout, Base::vrtc },
        { 0x30, Bus::portin,  Base::in30 },
        { 0x31, Bus::portin,  Base::in31 },
        { 0x40, Bus::portin,  Base::in40 },
        { 0x6e, Bus::portin,  Base::in6e },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&base, c_base)) return false;
    if (!base.Init(this)) return false;

    const static Bus::Connector c_mem1[] =
    {
        { pres, Bus::portout, Memory::reset },
        { 0x31, Bus::portout, Memory::out31 },
        { 0x32, Bus::portout, Memory::out32 },
        { 0x34, Bus::portout, Memory::out34 },
        { 0x35, Bus::portout, Memory::out35 },
        { 0x5c, Bus::portout, Memory::out5x },
        { 0x5d, Bus::portout, Memory::out5x },
        { 0x5e, Bus::portout, Memory::out5x },
        { 0x5f, Bus::portout, Memory::out5x },
        { 0x70, Bus::portout, Memory::out70 },
        { 0x71, Bus::portout, Memory::out71 },
        { 0x78, Bus::portout, Memory::out78 },
        { 0x99, Bus::portout, Memory::out99 },
        { 0xe2, Bus::portout, Memory::oute2 },
        { 0xe3, Bus::portout, Memory::oute3 },
        { 0xf0, Bus::portout, Memory::outf0 },
        { 0xf1, Bus::portout, Memory::outf1 },
        { vrtc, Bus::portout, Memory::vrtc  },
        { 0x32, Bus::portin,  Memory::in32  },
        { 0x5c, Bus::portin,  Memory::in5c  },
        { 0x70, Bus::portin,  Memory::in70  },
        { 0x71, Bus::portin,  Memory::in71  },
        { 0xe2, Bus::portin,  Memory::ine2  },
        { 0xe3, Bus::portin,  Memory::ine3  },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&mem1, c_mem1)) return false;
    if (!mem1.Init(&bus1, &crtc)) return false;

    const static Bus::Connector c_dmac[] =
    {
        { pres, Bus::portout, PD8257::reset },
        { 0x60, Bus::portout, PD8257::setaddr  },
        { 0x61, Bus::portout, PD8257::setcount },
        { 0x62, Bus::portout, PD8257::setaddr  },
        { 0x63, Bus::portout, PD8257::setcount },
        { 0x64, Bus::portout, PD8257::setaddr  },
        { 0x65, Bus::portout, PD8257::setcount },
        { 0x66, Bus::portout, PD8257::setaddr  },
        { 0x67, Bus::portout, PD8257::setcount },
        { 0x68, Bus::portout, PD8257::setmode  },
        { 0x60, Bus::portin,  PD8257::getaddr  },
        { 0x61, Bus::portin,  PD8257::getcount },
        { 0x62, Bus::portin,  PD8257::getaddr  },
        { 0x63, Bus::portin,  PD8257::getcount },
        { 0x64, Bus::portin,  PD8257::getaddr  },
        { 0x65, Bus::portin,  PD8257::getcount },
        { 0x66, Bus::portin,  PD8257::getaddr  },
        { 0x67, Bus::portin,  PD8257::getcount },
        { 0x68, Bus::portin,  PD8257::getstat  },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&dmac, c_dmac)) return false;

    const static Bus::Connector c_knj1[] =
    {
        { 0xe8, Bus::portout, KanjiROM::setl  },
        { 0xe9, Bus::portout, KanjiROM::seth  },
        { 0xe8, Bus::portin,  KanjiROM::readl },
        { 0xe9, Bus::portin,  KanjiROM::readh },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&knj1, c_knj1)) return false;
    if (!knj1.Init("kanji1.rom")) return false;

    const static Bus::Connector c_knj2[] =
    {
        { 0xec, Bus::portout, KanjiROM::setl  },
        { 0xed, Bus::portout, KanjiROM::seth  },
        { 0xec, Bus::portin,  KanjiROM::readl },
        { 0xed, Bus::portin,  KanjiROM::readh },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&knj2, c_knj2)) return false;
    if (!knj2.Init("kanji2.rom")) return false;

    const static Bus::Connector c_scrn[] =
    {
        { pres, Bus::portout, Screen::reset },
        { 0x30, Bus::portout, Screen::out30 },
        { 0x31, Bus::portout, Screen::out31 },
        { 0x32, Bus::portout, Screen::out32 },
        { 0x52, Bus::portout, Screen::out52 },
        { 0x53, Bus::portout, Screen::out53 },
        { 0x54, Bus::portout, Screen::out54 },
        { 0x55, Bus::portout, Screen::out55to5b },
        { 0x56, Bus::portout, Screen::out55to5b },
        { 0x57, Bus::portout, Screen::out55to5b },
        { 0x58, Bus::portout, Screen::out55to5b },
        { 0x59, Bus::portout, Screen::out55to5b },
        { 0x5a, Bus::portout, Screen::out55to5b },
        { 0x5b, Bus::portout, Screen::out55to5b },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&scrn, c_scrn)) return false;
    if (!scrn.Init(&bus1, &mem1, &crtc)) return false;

    const static Bus::Connector c_intc[] =
    {
        { pres,  Bus::portout, INTC::reset },
        { pint0, Bus::portout, INTC::request },
        { pint1, Bus::portout, INTC::request },
        { pint2, Bus::portout, INTC::request },
        { pint3, Bus::portout, INTC::request },
        { pint4, Bus::portout, INTC::request },
        { pint5, Bus::portout, INTC::request },
        { pint6, Bus::portout, INTC::request },
        { pint7, Bus::portout, INTC::request },
        { 0xe4,  Bus::portout, INTC::setreg },
        { 0xe6,  Bus::portout, INTC::setmask },
        { piack, Bus::portin,  INTC::intack },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&intc, c_intc)) return false;
    if (!intc.Init(&bus1, pirq)) return false;

    const static Bus::Connector c_crtc[] =
    {
        { pres,  Bus::portout, CRTC::reset },
        { 0x50,  Bus::portout, CRTC::out },
        { 0x51,  Bus::portout, CRTC::out },
        { 0x50,  Bus::portin,  CRTC::getstatus },
        { 0x51,  Bus::portin,  CRTC::in },
        { 0x00,  Bus::portout, CRTC::pcgout },
        { 0x01,  Bus::portout, CRTC::pcgout },
        { 0x02,  Bus::portout, CRTC::pcgout },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&crtc, c_crtc)) return false;
    if (!crtc.Init(this, draw)) return false;

    const static Bus::Connector c_subsys[] =
    {
        { pres,  Bus::portout, SubSystem::reset },
        { 0xfc,  Bus::portout | Bus::sync, SubSystem::m_set0 },
        { 0xfd,  Bus::portout | Bus::sync, SubSystem::m_set1 },
        { 0xfe,  Bus::portout | Bus::sync, SubSystem::m_set2 },
        { 0xff,  Bus::portout | Bus::sync, SubSystem::m_setcw },
        { 0xfc,  Bus::portin  | Bus::sync, SubSystem::m_read0 },
        { 0xfd,  Bus::portin  | Bus::sync, SubSystem::m_read1 },
        { 0xfe,  Bus::portin  | Bus::sync, SubSystem::m_read2 },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&subsys, c_subsys)) return false;

    const static Bus::Connector c_sio[] =
    {
        { pres,  Bus::portout, SIO::reset },
        { 0x20,  Bus::portout, SIO::setdata },
        { 0x21,  Bus::portout, SIO::setcontrol },
        { 0x20,  Bus::portin,  SIO::getdata },
        { 0x21,  Bus::portin,  SIO::getstatus },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&sio, c_sio)) return false;
    if (!sio.Init(&bus1, pint0, this)) return false;

    const static Bus::Connector c_cdif[] =
    {
        { 0x90,  Bus::portout, CDIF::out90 },
        { 0x91,  Bus::portout, CDIF::out91 },
        { 0x94,  Bus::portout, CDIF::out94 },
        { 0x97,  Bus::portout, CDIF::out97 },
        { 0x98,  Bus::portout, CDIF::out98 },
        { 0x99,  Bus::portout, CDIF::out99 },
        { 0x9f,  Bus::portout, CDIF::out9f },
        { 0x90,  Bus::portin,  CDIF::in90 },
        { 0x91,  Bus::portin,  CDIF::in91 },
        { 0x92,  Bus::portin,  CDIF::in92 },
        { 0x93,  Bus::portin,  CDIF::in93 },
        { 0x96,  Bus::portin,  CDIF::in96 },
        { 0x98,  Bus::portin,  CDIF::in98 },
        { 0x99,  Bus::portin,  CDIF::in99 },
        { 0x9b,  Bus::portin,  CDIF::in9b },
        { 0x9d,  Bus::portin,  CDIF::in9d },
        { 0, 0, 0 }
    };
//  if (!bus1.Connect(&cdif, c_cdif)) return false;
    if (!cdif.Init(&dmac)) return false;

    const static Bus::Connector c_opn1[] =
    {
        { pres, Bus::portout, OPNIF::reset },
        { 0x32, Bus::portout, OPNIF::setintrmask },
        { 0x44, Bus::portout, OPNIF::setindex0 },
        { 0x45, Bus::portout, OPNIF::writedata0 },
        { 0x46, Bus::portout, OPNIF::setindex1 },
        { 0x47, Bus::portout, OPNIF::writedata1 },
        { 0x44, Bus::portin,  OPNIF::readstatus },
        { 0x45, Bus::portin,  OPNIF::readdata0 },
        { 0x46, Bus::portin,  OPNIF::readstatusex },
        { 0x47, Bus::portin,  OPNIF::readdata1 },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&opn1, c_opn1)) return false;
    if (!opn1.Init(&bus1, pint4, popnio, this, 3993600, 8000)) return false;

    const static Bus::Connector c_opn2[] =
    {
        { pres, Bus::portout, OPNIF::reset },
        { 0xaa, Bus::portout, OPNIF::setintrmask },
        { 0xa8, Bus::portout, OPNIF::setindex0 },
        { 0xa9, Bus::portout, OPNIF::writedata0 },
        { 0xac, Bus::portout, OPNIF::setindex1 },
        { 0xad, Bus::portout, OPNIF::writedata1 },
        { 0xa8, Bus::portin,  OPNIF::readstatus },
        { 0xa9, Bus::portin,  OPNIF::readdata0 },
        { 0xac, Bus::portin,  OPNIF::readstatusex },
        { 0xad, Bus::portin,  OPNIF::readdata1 },
        { 0, 0, 0 }
    };
    if (!bus1.Connect(&opn2, c_opn2)) return false;
    if (!opn2.Init(&bus1, pint4, popnio, this, 3993600, 8000)) return false;

    return true;
}

// ---------------------------------------------------------------------------
//  foCXڑ(TuCPU)
//
bool PC88::ConnectDevices2()
{
    const static Bus::Connector c_cpu2[] =
    {
        { pres2, Bus::portout, Z80::reset },
        { pirq2, Bus::portout, Z80::irq },
        { 0, 0, 0 }
    };
    if (!bus2.Connect(&cpu2, c_cpu2)) return false;
    if (!cpu2.Init(&bus2, piac2)) return false;

    const static Bus::Connector c_mem2[] =
    {
        { piac2, Bus::portin, SubSystem::intack },
        { 0xfc,  Bus::portout | Bus::sync, SubSystem::s_set0 },
        { 0xfd,  Bus::portout | Bus::sync, SubSystem::s_set1 },
        { 0xfe,  Bus::portout | Bus::sync, SubSystem::s_set2 },
        { 0xff,  Bus::portout | Bus::sync, SubSystem::s_setcw },
        { 0xfc,  Bus::portin  | Bus::sync, SubSystem::s_read0 },
        { 0xfd,  Bus::portin  | Bus::sync, SubSystem::s_read1 },
        { 0xfe,  Bus::portin  | Bus::sync, SubSystem::s_read2 },
        { 0, 0, 0 }
    };
    if (!bus2.Connect(&subsys, c_mem2)) return false;
    if (!subsys.Init(&bus2)) return false;

    const static Bus::Connector c_fdc[] =
    {
        { pres2, Bus::portout, FDC::reset },
        { 0xfb,  Bus::portout, FDC::setdata },
        { 0xf4,  Bus::portout, FDC::drivecontrol },
        { 0xf8,  Bus::portout, FDC::motorcontrol },
        { 0xf8,  Bus::portin,  FDC::tcin },
        { 0xfa,  Bus::portin,  FDC::getstatus },
        { 0xfb,  Bus::portin,  FDC::getdata },
        { 0, 0, 0 }
    };
    if (!bus2.Connect(&fdc, c_fdc)) return false;
    if (!fdc.Init(&diskmgr, this, &bus2, pirq2)) return false;

    return true;
}

// ---------------------------------------------------------------------------
//  fobOp_v
//
inline void PC88::DumpMemory()
{
#ifdef _DEBUG
    FileIO fio("memimage.bin", FileIO::create);

    uint8 buf[0x10000];
    int i;

    for (i=0; i<0x10000; i++)
        buf[i] = bus1.Read8(i);
    fio.Write(buf, 0x10000);

    for (i=0; i<0x8000; i++)
        buf[i] = bus2.Read8(i);
    fio.Write(buf, 0x8000);
#endif
}

// ---------------------------------------------------------------------------
//  ݒ蔽f
//
void PC88::ApplyConfig(Config* cfg)
{
    cfgflags = cfg->flags;
    cfgflag2 = cfg->flag2;

    refreshtiming = cfg->refreshtiming;
    base.SetSwitch(cfg);
    scrn.ApplyConfig(cfg);
    fdc.ShowStatus((cfg->flags & Config::showfdcstatus) != 0);
    mem1.ApplyConfig(cfg);
    crtc.ApplyConfig(cfg);

    cpumode = (cfg->cpumode == Config::msauto)
        ? (cfg->mainsubratio > 1 ? ms21 : ms11)
        :  (cfg->cpumode & 1);
    if ((cfg->flags & Config::subcpucontrol) != 0)
        cpumode |= stopwhenidle;
}

