/*******************************************************************************
* FILE NAME: icvhdr.cpp                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in icvhdr.hpp.                                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_GPILOGCOLORTABLE  // GpiCreateLogColorTable, etc.
#define INCL_WINBUTTONS        // BS_xxx, BM_xxx
#define INCL_WINDIALOGS        // WinEnumDlgItem, etc.
#define INCL_WINFRAMEMGR       // WM_QUERYFOCUSCHAIN, SC_CLOSE
#define INCL_WININPUT          // WM_CHAR, WinGetKeyState, WinQueryFocus
#define INCL_WINMESSAGEMGR     // WM_COMMAND, etc., CMDSRC_OTHER
#define INCL_WINSTATICS        // SS_BKGNDRECT, SS_FGNDRECT
#define INCL_WINWINDOWMGR      // WinDefWindowProc, WinQueryWindowULong, etc.

#include <iwindefs.h>
#include <icvhdr.hpp>
#include <icanvas.hpp>
#ifdef IC_PMWIN
#include <icanvprv.hpp>
#endif
#include <icconst.h>
#include <icmdevt.hpp>
#include <icmnfun.hpp>
#include <icoordsy.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <iframe.hpp>
#include <igbundle.hpp>
#include <igrafctx.hpp>
#include <igrect.hpp>
#include <igregion.hpp>
#include <ikeyevt.hpp>
#include <ipainevt.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICVHDR_CPP_
  #include <ipagetun.h>
#endif

IWindowHandle origDefaultButtonHandle( const IWindow* canvas );
//------------------------------------------------------------------
// DEFECT 20847 : new local function 
//------------------------------------------------------------------
void setDefaultPushButton ( const IWindow* pcvCanvas,
                            IWindowHandle  pushbutton,
                            IBase::Boolean setAsDefault );
//------------------------------------------------------------------

ICanvasHandler*     ICanvasStatics :: pClCanvasHandler = 0;
ICVKeyboardHandler* ICanvasStatics :: fgcvKeyboardHandler = 0;

/*------------------------------------------------------------------------------
| ICanvasStatics::~ICanvasStatics                                              |
|                                                                              |
| Destructor to clean up.                                                      |
------------------------------------------------------------------------------*/
ICanvasStatics :: ~ICanvasStatics ( )
{
  if (pClCanvasHandler)
  {
     delete pClCanvasHandler;
  }
  if (fgcvKeyboardHandler)
  {
     delete fgcvKeyboardHandler;
  }
}

/*------------------------------------------------------------------------------
| ICanvasStatics::canvasHandler                                                |
|                                                                              |
| Returns the common canvas handler.                                           |
------------------------------------------------------------------------------*/
ICanvasHandler& ICanvasStatics :: canvasHandler ( )
{
  if (pClCanvasHandler == 0)
  {
     pClCanvasHandler = new ICanvasHandler;
  }
  return *pClCanvasHandler;
}

/*------------------------------------------------------------------------------
| ICanvasStatics::keyboardHandler                                              |
|                                                                              |
| Returns the common keyboard handler.                                         |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICanvasStatics :: keyboardHandler ( )
{
  if (fgcvKeyboardHandler == 0)
  {
     fgcvKeyboardHandler = new ICVKeyboardHandler;
  }
  return *fgcvKeyboardHandler;
}


/*------------------------------------------------------------------------------
| ICanvasHandler::ICanvasHandler                                               |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
ICanvasHandler :: ICanvasHandler ( )
  : ICanvasHandler::Inherited ( )
{ }


/*------------------------------------------------------------------------------
| ICanvasHandler::~ICanvasHandler                                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ICanvasHandler :: ~ICanvasHandler ( )
{ }


/*------------------------------------------------------------------------------
| ICanvasHandler::dispatchHandlerEvent                                         |
|                                                                              |
| Router and default event processing for events.                              |
| Notes: Event processing includes:                                            |
|    -- WM_COMMAND, WM_CONTROLPOINTER, WM_SYSCOMMAND                           |
|       (sends these messages on to the owner window, unless doing so would    |
|       cause an infinite loop between the frame and client area canvas).      |
|    -- WM_WINDOWPOSCHANGED                                                    |
|       (calls WinDefWindowProc to avoid the handling done by the WC_STATIC    |
|       window procedure)                                                      |
|    -- WM_HITTEST                                                             |
|       (calls WinDefWindowProc to avoid the handling done by the WC_STATIC    |
|       window procedure)                                                      |
|    -- WM_BUTTON1DOWN, WM_BUTTON1DBLCLK, WM_BUTTON2DOWN                       |
|       (don't pass event to owner window if canvas is a notebook page window) |
|    -- WM_WINQUERYDLGCODE                                                     |
|       (avoids the handling done by the WC_STATIC window procedure)           |
|    -- WM_PAINT                                                               |
|       (with IC_UM_CANVAS_PAINT, calls the layout routine to provide          |
|       specialized sizing and positioning of child controls)                  |
|    -- IC_UM_CANVAS_PAINT                                                     |
|       (with WM_PAINT, calls the layout routine to provide specialized sizing |
|       and positioning of child controls)                                     |
|    -- WM_SHOW                                                                |
|       (with IC_UM_CANVAS_PAINT, calls the layout routine to provide          |
|       specialized sizing and positioning of child controls)                  |
------------------------------------------------------------------------------*/
Boolean ICanvasHandler :: dispatchHandlerEvent ( IEvent & event )
{
   IMODTRACE_ALL("ICanvasHandler::dispatchHandlerEvent");
   Boolean bProcessed = false;

   switch (event.eventId())
   {
      case WM_COMMAND:
      case WM_SYSCOMMAND:
      {                        // These aren't normally sent to owner.
         /**********************************************************/
         /* Must ensure that WM_COMMAND and WM_SYSCOMMAND are not  */
         /* sent to the owner window if this canvas is the client  */
         /* area, in the cases where doing so would cause an       */
         /* endless loop between the frame and canvas handlers.    */
         /* The underlying problem is that the frame handler sends */
         /* WM_COMMAND and WM_SYSCOMMAND/SC_CLOSE events, and PM's */
         /* default frame procedure sends WM_COMMAND messages back */
         /* to the client area (FID_CLIENT), which the canvas      */
         /* would then send back to the owner frame, ad infinitum. */
         /* To remedy this problem, the canvas handler sets a bit  */
         /* reserved by the class library in mp2 of the event      */
         /* before sending it on to the owner frame, but doesn't   */
         /* send on such events if the bit is already set (the bit */
         /* is used to signal whether the event has already been   */
         /* routed to the frame, and doesn't need to be sent again */
         /* since it would just come back from the frame).         */
         /**********************************************************/
         ICanvas* pcvCanvas = (ICanvas*)event.window();
         const IWindow* pwndOwner = pcvCanvas->owner();

         if ( pwndOwner )
         {          // Have an owner window to route the event to.

            ICommandEvent cmdevt(event);

            Boolean bClient = (pcvCanvas->id() == FID_CLIENT  &&
                               pwndOwner->isFrameWindow());

            if (!bClient  ||  cmdevt.isFromFrame() == false)
            {       // Canvas is not client or event never sent to frame.
               IWindowHandle whOwner = pwndOwner->handle();

               if (whOwner != IWindow::desktopWindow()->handle()  &&
                   whOwner != IWindow::objectWindow()->handle())
               {                          // Valid owner window.
#ifndef IC_WIN
                  unsigned long ul2 = (unsigned long)event.parameter2();

                  if (bClient)
                  {                       // Canvas is client window.
                     ul2 |= 0x8000;       // Set reserved bit in source value.
                  }
                  event.setResult(pwndOwner->handle().
                                    sendEvent(event.eventId(),
                                              event.parameter1(),
                                              IEventParameter2(ul2)));
                             // Forward command event to owner window.
#endif
#ifdef IC_WIN
                  unsigned long ul1 = (unsigned long)event.parameter1();

                  if (bClient)
                  {                       // Canvas is client window.
                     ul1 |= 0x80000000ul;   // Set reserved bit in source value.
                  }
                  event.setResult(pwndOwner->handle().
                                     sendEvent(event.eventId(),
                                               IEventParameter1(ul1),
                                               event.parameter2() ));
                             // Forward command event to owner window.
#endif
                  bProcessed = true;
               }
            } /* endif */
         } /* endif */
         break;
      } /* endcase */

      case WM_CONTROLPOINTER:
      {                          // This isn't normally sent to owner.
         /**********************************************************/
         /* Pass the WM_CONTROLPOINTER to the owner.  This message */
         /* will make its way to the frame handler (if another     */
         /* handler does not process the message) where the mouse  */
         /* pointer can be set for a frame window and all of its   */
         /* child windows.  Even if the canvas is the client, the  */
         /* control pointer message will not be rerouted by the    */
         /* frame back to the client (the default PM behavior)     */
         /* since the frame handler will always return true for    */
         /* this message.                                          */
         /**********************************************************/
         if (event.window() == event.dispatchingWindow())
         {
           ICanvas* pcvCanvas = (ICanvas*)event.window();
           const IWindow* pwndOwner = pcvCanvas->owner();

           if (pwndOwner != 0)
           {
              IWindowHandle whOwner = pwndOwner->handle();
              if (whOwner != IWindow::desktopWindow()->handle()  &&
                  whOwner != IWindow::objectWindow()->handle())
              {                          // Valid owner window.
                 event.setResult(pwndOwner->handle().
                                          sendEvent(event.eventId(),
                                                    event.parameter1(),
                                                    event.parameter2()));
                 bProcessed = true;
              }
           } /* endif */
         }
         else
         {
           bProcessed = false;
         } /* endif */

         break;
      } /* endcase */

      case WM_WINDOWPOSCHANGED:
      {
         /**********************************************************/
         /* Must ensure that WM_WINDOWPOSCHANGED is sent to the    */
         /* default window procedure instead of the WC_STATIC      */
         /* window procedure, since otherwise ISplitCanvas and     */
         /* IMultiCellCanvas will never receive the WM_SIZE        */
         /* message they are dependent on.                         */
         /**********************************************************/
         MRESULT mr = IDEFWINDOWPROC( event.handle(),
                                      event.eventId(),
                                      event.parameter1(),
                                      event.parameter2() );
         event.setResult( mr );
         bProcessed = true;
         break;
      } /* endcase */

#ifdef   IC_PM
      case WM_HITTEST:                 // Allow button presses on children.
#endif   //IC_PM
#ifdef   IC_WIN
      case WM_NCHITTEST:               // Allow button presses on children.
#endif   //IC_WIN
      {        // Avoid HT_TRANSPARENT from WC_STATIC window procedure.
         MRESULT mr = IDEFWINDOWPROC( event.handle(),
                                      event.eventId(),
                                      event.parameter1(),
                                      event.parameter2() );
         event.setResult( mr );
         bProcessed = true;
         break;
      } /* endcase */

      case WM_BUTTON1DOWN:
      case WM_BUTTON1DBLCLK:
      case WM_BUTTON2DOWN:
      {                                // Notebook page window case?
         /**********************************************************/
         /* These mouse events are processed to provide special    */
         /* processing in the case where the canvas is a notebook  */
         /* page.  Without this, clicking on an entry field child  */
         /* window of a canvas, or in white space of the canvas    */
         /* itself will result in the notebook eventually getting  */
         /* the button event and changing the input focus back to  */
         /* itself.                                                */
         /**********************************************************/
         HWND parent = IPARENTOF( event.handle() );
         HWND grandParent = IPARENTOF( parent );
                // (Can't use parent(), since notebook's intermediate
                // window isn't known to the class library.)
         if (grandParent)
         {                             // Notebook was orig parent?
            IWindowClassName className( grandParent );

            if (className == WC_NOTEBOOK)
            {                          // Parent was notebook.
               IWindowHandle canvasWindow = event.handle();
               IWindowHandle focusWindow =
                  IQUERYFOCUS(IWindow::desktopWindow()->handle());
               IEventResult mr =
                  focusWindow.sendEvent(WM_QUERYFOCUSCHAIN,
                                        IEventParameter1(QFC_PARTOFCHAIN),
                                        IEventParameter2(canvasWindow));
               if (mr.asUnsignedLong() == false)
               {                       // Focus not on canvas child
                  event.window()->setFocus();
               }
               bProcessed = true;      // Don't let notebook get event
                                       //   since it will only steal
                                       //   away the input focus.
            }                          // End parent was notebook.
         }                             // End know who grand parent is.
         break;
      } /* endcase */

      case WM_QUERYDLGCODE:
      {
#ifdef IC_WIN
         // we want keys
         event.setResult( DLGC_WANTALLKEYS );
#else
         // Don't return DLGC_TABONCLICK, DLGC_STATIC.
         event.setResult(0);
#endif
         bProcessed = true;
         break;
      } /* endcase */


      case WM_PAINT:
      {
         IMODTRACE_DEVELOP("ICanvasHandler::dispatchHandlerEvent(WM_PAINT)");
         ICanvas* pcvCanvas = (ICanvas*)event.window();
         unsigned long ulStaticStyle = (pcvCanvas->style() & 0x000F);

         if (pcvCanvas->isLayoutDistorted(IWindow::layoutChanged)  &&
             pcvCanvas->isVisible())
         {              // Use layout to show child changes on screen.
            ITRACE_DEVELOP("Send IC_UM_CANVAS_PAINT message");
            pcvCanvas->sendEvent(IC_UM_CANVAS_PAINT, 0, 0);
         }                             // Pass processing on to PM.

         /**********************************************************/
         /* Paint background of canvas.                            */
         /**********************************************************/
#ifdef   IC_PM
         _RECTL rectlInvalid;
#endif   //IC_PM
#ifdef   IC_WIN
         PAINTSTRUCT rectlInvalid;
#endif   //IC_WIN
         IPresSpaceHandle hps =
            IBEGINPAINT(pcvCanvas->handle(), 0, &rectlInvalid);
#ifdef   IC_PM
         IRectangle rectPaint(rectlInvalid);
#endif   //IC_PM
#ifdef   IC_WIN
         IRectangle rectPaint(rectlInvalid.rcPaint);
#endif   //IC_WIN

         { //Scope graphic context so that it is destructed before EndPaint
         IGraphicContext gc(hps);
         IGRectangle box(rectPaint);
         IColor clrBgnd = pcvCanvas->backgroundColor();

         gc.setDrawOperation( IGraphicBundle::fillAndFrame );

         Boolean clipChildren = true;
         IGRegion clipRegion( gc, rectPaint );
#ifdef IC_PMWIN
         #define MAX_CHILDREN_FOR_CLIPPING  6
         if ( pcvCanvas->fCanvasData->fShowWindow  ||
              pcvCanvas->fCanvasData->fChildCount > MAX_CHILDREN_FOR_CLIPPING  ||
              pcvCanvas->style() & WS_CLIPCHILDREN )
         {  // Don't clip on the initial display, or if building the
            // clip region looks too expensive because of the number
            // of child windows, or if the canvas is using child
            // clipping support provided by the presentation system.
            clipChildren = false;
            pcvCanvas->fCanvasData->fShowWindow = false;
         }
#endif

         if ( clipChildren )
         {
            // Set up clip regions so that child windows are not overpainted.
            IWindow::ChildCursor cursor(*pcvCanvas);
            for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
            {
              IWindowHandle hwndChild = pcvCanvas->childAt(cursor);
              if ( hwndChild )
              {
                IWindow* pChild = IWindow::windowWithHandle(hwndChild);
                if ( pChild && pChild->isVisible() )
                {
                  // The rectangle of the child region has to be in native
                  // coordinates.
                  IRectangle visibleRect = pChild->visibleRectangle();
                  if ( ICoordinateSystem::isConversionNeeded() )
                  {
                     visibleRect =
                        ICoordinateSystem::convertToNative( visibleRect,
                                                            pcvCanvas->size() );
                  }
                  IGRegion childRegion( gc, visibleRect );
                  clipRegion -= childRegion;
                }
              }
            }

            gc.setClipRegion(clipRegion);
         }
         gc.setPenColor(clrBgnd);
         gc.setFillColor(clrBgnd);
         gc.draw(box);
#ifndef IC_WIN
         if ( clipChildren )
         {
            gc.clearClipRegion();
         }
#endif

         ITRACE_DEVELOP(IString("Paint rectangle = ") +
                        rectPaint.asDebugInfo());
         ITRACE_DEVELOP(IString("Window rectangle = ") +
                        pcvCanvas->nativeRect().asDebugInfo());

#ifdef IC_WIN
         // We need to paint the group box if present
         if ((ulStaticStyle != SS_BKGNDRECT)  &&
             (ulStaticStyle != SS_FGNDRECT ) )
         {
            // draw the outline box
            RECT rc;
            GetClientRect( pcvCanvas->handle(), &rc);
            IGRectangle outlineBox( IRectangle(0,0, rc.right, rc.bottom) );
            gc.setDrawOperation( IGraphicBundle::frame );
            gc.setPenColor( pcvCanvas->foregroundColor() );
            gc.draw(outlineBox);
            // draw the text
            unsigned long strLength =
               (unsigned long)IQUERYWINDOWTEXTLENGTH( pcvCanvas->handle() );
            if ( strLength != 0 )
            {
               char* str = new char[strLength + 1];
               IQUERYWINDOWTEXT( pcvCanvas->handle(),
                                 strLength,
                                 str );

               SIZE  lpSize;
               if (!GetTextExtentPoint(gc.handle(),
                                       str,
                                       strLength,
                                       &lpSize ) )
               {                                    // error
                  delete [] str;                    // don't leak
                  ITHROWGUIERROR("GetTextExtentPoint");
               }
               // adjust rectangle to include only text height and to
               // be 4 pixels in from the left edge
               rc.bottom = rc.top + lpSize.cy;
               rc.left  += 4;
               DrawText( gc.handle(),
                         str,
                         strLength,
                         &rc,
                         DT_LEFT | DT_NOPREFIX );
               delete [] str;
            }  // if text present
         }  // if group box
         bProcessed = true;      // all painting done

         if ( clipChildren )
         {
            gc.clearClipRegion();
         }
#endif

         } //Force graphic context to be destructed here (before EndPaint)
         IENDPAINT(pcvCanvas->handle(),hps,&rectlInvalid);

#ifdef   IC_PM
         if (ulStaticStyle == SS_BKGNDRECT  ||
             ulStaticStyle == SS_FGNDRECT)
         {                        // Canvas is a fill box.
            bProcessed = true;    // Nothing more needed.
         }
         else                     // Else let PM draw group box.
         {                        // Canvas is a group box.
            // To avoid an infinite loop condition, always be sure that the
            // syncPaint style is OFF before invalidating the rectangle.
            unsigned long oldStyle = ISTYLEOF(pcvCanvas->handle());
            if ( oldStyle & WS_SYNCPAINT )
              ISETWINDOWSTYLE( pcvCanvas->handle(),
                               oldStyle & (unsigned long)~WS_SYNCPAINT );

            IINVALIDATERECT(pcvCanvas->handle(),
                            &rectlInvalid, false);

            // Restore original style (if needed).
            if ( oldStyle & WS_SYNCPAINT )
              ISETWINDOWSTYLE(pcvCanvas->handle(), oldStyle);
         }                        // Let PM draw group box now.
#endif   //IC_PM
         break;
      } /* endcase */

      case IC_UM_CANVAS_PAINT:
      {
         ICanvas* pcvCanvas = (ICanvas*)event.window();
         pcvCanvas->setLayoutDistorted(0, IWindow::fontPropogated);
         bProcessed = layoutCanvas(event);
         break;
      } /* endcase */

#ifdef   IC_PM
      case WM_SHOW:
#endif   //IC_PM
#ifdef   IC_WIN
      case WM_SHOWWINDOW:
#endif   //IC_WIN
      {
         ICanvas* pcvCanvas = (ICanvas*)event.window();
         if ( event.parameter1().number1() )
         {              // Canvas is being shown.
#ifdef IC_PMWIN
            pcvCanvas->fCanvasData->fShowWindow = true;
#endif
            if ( pcvCanvas->isLayoutDistorted( IWindow::layoutChanged ) )
            {                     // Window being shown and need new layout.
               IMODTRACE_DEVELOP("ICanvasHandler::dispatchHandlerEvent(WM_SHOW)");
               pcvCanvas->sendEvent(IC_UM_CANVAS_PAINT, 0, 0);
            }                     // Pass processing on to PM.
         }
         break;
      } /* endcase */

#ifdef   IC_PM
      case 0x0BD0: /* WM_SETBIDIATTR - Note: Not in toolkit header files. */
      {
         ICanvas* pcvCanvas = (ICanvas*)event.window();
         pcvCanvas->setLayoutDistorted(IWindow::layoutChanged,0);
         pcvCanvas->postEvent(IC_UM_CANVAS_PAINT);
         break;
      }
#endif   //IC_PM

      default:
         break;
   }  /* end switch */

   return bProcessed;
}


/*------------------------------------------------------------------------------
| ICanvasHandler::layoutCanvas                                                 |
|                                                                              |
| Process a paint event by calling the canvas's layout() routine.              |
------------------------------------------------------------------------------*/
Boolean ICanvasHandler :: layoutCanvas ( IEvent& event )
{
   IMODTRACE_DEVELOP("ICanvasHandler::layoutCanvas");
   ICanvas* pcvCanvas = (ICanvas*)event.window();
   if (pcvCanvas->isLayoutDistorted(IWindow::layoutChanged))
   {
      pcvCanvas->layout();
   }
   return true;
}

/*------------------------------------------------------------------------------
| ICanvasHandler::handleEventsFor                                              |
|                                                                              |
| Attach the handler to the canvas.                                            |
------------------------------------------------------------------------------*/
ICanvasHandler& ICanvasHandler :: handleEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::handleEventsFor(canvas);
   return *this;
}


/*------------------------------------------------------------------------------
| ICanvasHandler::stopHandlingEventsFor                                        |
|                                                                              |
| Detach the handler from the canvas.                                          |
------------------------------------------------------------------------------*/
ICanvasHandler& ICanvasHandler :: stopHandlingEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::stopHandlingEventsFor(canvas);
   return *this;
}


/*------------------------------------------------------------------------------
| ICanvasHandler::handleEventsFor                                              |
------------------------------------------------------------------------------*/
IHandler& ICanvasHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}


/*------------------------------------------------------------------------------
| ICanvasHandler::stopHandlingEventsFor                                        |
------------------------------------------------------------------------------*/
IHandler& ICanvasHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::ICVKeyboardHandler                                               |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
ICVKeyboardHandler :: ICVKeyboardHandler ( )
  : ICVKeyboardHandler::Inherited ( )
{ }


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::~ICVKeyboardHandler                                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ICVKeyboardHandler :: ~ICVKeyboardHandler ( )
{ }

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::dispatchHandlerEvent                                         |
|                                                                              |
| Router and default event processing for events.                              |
| Notes: Event processing includes:                                            |
|    -- WM_SETFOCUS                                                            |
|       (with IC_UM_CANVAS_SETFOCUS, sets input focus to a tab-able child      |
|       control)                                                               |
|    -- IC_UM_CANVAS_SETFOCUS                                                  |
|       (with WM_SETFOCUS, sets input focus to a tab-able child control)       |
|    -- WM_CHAR                                                                |
|       (processes tab and cursor movement keys in a similar manner as is done |
|       for frame windows by PM)                                               |
------------------------------------------------------------------------------*/
Boolean ICVKeyboardHandler :: dispatchHandlerEvent ( IEvent & event )
{
   IMODTRACE_ALL("ICVKeyboardHandler::dispatchHandlerEvent");
   Boolean bProcessed = false;

   switch (event.eventId())
   {
      case WM_CHAR:
#ifdef IC_WIN
      case WM_KEYDOWN:
      case WM_SYSKEYDOWN:
#endif
      {
         IKeyboardEvent keyevt(event);
         bProcessed = key(keyevt);
         if (bProcessed)
         {                             // Key processed by canvas.
            event.setResult(keyevt.result());
         }
         break;
      } /* endcase */


      case WM_SETFOCUS:
      {                   // Assign to first/last tab-able control.
         ITRACE_ALL( IString( "WM_SETFOCUS hwnd=" ) +
                     IString( event.handle().asUnsigned() ).d2x() +
                     IString( " wParam=" ) +
                     IString( event.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( event.parameter2().asUnsignedLong() ).d2x() );
#ifdef IC_PM     // In Windows loss of focus is separate message
         if (event.parameter2().number1())
#endif
         {                             // Canvas window getting focus.
            /*******************************************************/
            /* Post the IC_UM_CANVAS_SETFOCUS message to avoid     */
            /* changing the window with the input focus during     */
            /* processing of a WM_SETFOCUS message.                */
            /*******************************************************/
            event.handle().postEvent(IC_UM_CANVAS_SETFOCUS, 0, 0);
            bProcessed = true;
         }
         break;
      } /* endcase */

      case IC_UM_CANVAS_SETFOCUS:
      {                                // From WM_SETFOCUS processing.
         ITRACE_ALL( IString( "IC_UM_CANVAS_SETFOCUS hwnd=" ) +
                     IString( event.handle().asUnsigned() ).d2x() +
                     IString( " wParam=" ) +
                     IString( event.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( event.parameter2().asUnsignedLong() ).d2x() );
         bProcessed = true;
         IWindow* pwndParent;
         IWindowHandle hwndFocus =
                   IQUERYFOCUS(IWindow::desktopWindow()->handle());
         Boolean bClientFocus = false;
         if (!(event.window()->isFrameWindow()))
            {
            bClientFocus =
              (event.window()->id() == FID_CLIENT  &&
               (pwndParent = event.window()->parent())  &&
               hwndFocus == pwndParent->handle()  &&
               pwndParent->isFrameWindow() );
            }
         if (hwndFocus == event.window()->handle()  ||
             bClientFocus )
         {                             // Canvas has input focus.
            /*******************************************************/
            /* Canvas still has the focus from when WM_SETFOCUS    */
            /* was detected.  We should ignore this                */
            /* IC_UM_CANVAS_SETFOCUS message if PM has already     */
            /* changed the window with the input focus since we    */
            /* posted this message--otherwise we will be changing  */
            /* the input focus incorrectly back to ourselves.      */
            /*                                                     */
            /* The one exception to the above is if the frame      */
            /* gained the input focus, but is letting the client   */
            /* process the notification (the frame is still the    */
            /* focus window in this case, but the canvas is sent a */
            /* WM_SETFOCUS notification).  Presumably this means   */
            /* the frame is not providing any special processing,  */
            /* and does not care if we take the focus away (so     */
            /* we will go ahead and take the focus away).          */
            /*******************************************************/
            bProcessed = gotFocus(event);
         }
         break;
      } /* endcase */
      default:
         break;
   }  /* end switch */

   return bProcessed;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::gotFocus                                                     |
|                                                                              |
| Gives input focus to the appropriate child window.                           |
------------------------------------------------------------------------------*/
Boolean ICVKeyboardHandler :: gotFocus ( IEvent& event )
{
#ifdef   IC_PM
   unsigned long ulKeyState =                // Check for a back-tab key.
      IGETKEYSTATE(IWindow::desktopWindow()->handle(), VK_BACKTAB);
#endif   //IC_PM
#ifdef   IC_WIN
   unsigned long ulKeyState =                // Check for a back-tab key.
      IGETKEYSTATE(IWindow::desktopWindow()->handle(), VK_TAB) &
      IGETKEYSTATE(IWindow::desktopWindow()->handle(), VK_SHIFT);
#endif   //IC_WIN
   unsigned long ulTabCode = (ulKeyState & 0x8000) ?      // Key pressed?
                                EDI_LASTTABITEM : EDI_FIRSTTABITEM;
   /****************************************************************/
   /* No check is made for cursor movement keys (e.g. VK_UP),      */
   /* since the canvas is not assumed to belong to a group and no  */
   /* children in the canvas are assumed to belong to a group      */
   /* outside the canvas.                                          */
   /*                                                              */
   /* The use of WinEnumDlgItem here assumes that either all child */
   /* windows have been created with HWND_BOTTOM positioning, or   */
   /* the z-order of the child windows has been reversed by the    */
   /* canvas, so that the order that child windows are returned by */
   /* WinEnumDlgItem matches the expected order that the windows   */
   /* are layed out and tabbed through.                            */
   /****************************************************************/
   IWindow* pcvCanvas = event.window();
   IWindowHandle hwndNextFocus =
                          IENUMDLGITEM(event.handle(), 0, ulTabCode);
   if (hwndNextFocus == 0)        // No window to tab to.
      {                              // Let next sibling try.
#ifdef IC_WIN
      if ( pcvCanvas->isFrameWindow() )
         {
         // Windows only case.  There are no tabbable children of the
         // frame.  If there is a client, set focus to it.  If there
         // is no client, set focus to the first child. (as Windows dialogs
         // do)  Otherwise, punt.
         hwndNextFocus = IWINDOWFROMID( pcvCanvas->handle(), FID_CLIENT );
         if (hwndNextFocus == 0)
            {           // no client
            hwndNextFocus = GetWindow( pcvCanvas->handle(), GW_CHILD);
            }
         }
      else
#endif
         {
         /*************************************************************/
         /* We will probably fall into here only for the case of a    */
         /* canvas with no child windows.  Try to find a sibling      */
         /* window that can take the input focus.                     */
         /*************************************************************/
         IWindow* pwndParent = pcvCanvas->parent();
         if (pwndParent)
            {                                // Has parent window.
            hwndNextFocus =
               IENUMDLGITEM(pwndParent->handle(),
                            event.handle(),
                            (ulTabCode == EDI_LASTTABITEM) ?
                                EDI_PREVTABITEM : EDI_NEXTTABITEM);
            }
         }
      }                              // End no window to tab to.

   if (hwndNextFocus)
      {                              // Have window to give focus to.
      ICVKeyboardHandler::buttonSetup(pcvCanvas, hwndNextFocus,
                                      ICVKeyboardHandler::enteringGroup);
      if (hwndNextFocus != event.handle())
         {                           // Giving the focus to someone else.
         ISETFOCUS(IWindow::desktopWindow()->handle(),
                   hwndNextFocus);
#ifdef IC_PM
         // Note: On OS/2 there is a bug that entry fields do not highlight
         //       selected text when they get focus from a window outside
         //       of the canvas.  To fix this, we need to check for entry
         //       fields and force them to repaint
         IWindowClassName className( hwndNextFocus );
         if ( className == WC_ENTRYFIELD )
            {
            IINVALIDATERECT( hwndNextFocus, 0, true );
            }
#endif
         }
      }

   return true;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::key                                                      |
|                                                                              |
| Process tabs and cursor movement keys.                                       |
| Notes: Process tab and cursor movement since PM's frame window procedure     |
|          will only set focus to a direct child window (e.g. a canvas window  |
|          instead of a control on the canvas, which is what is needed).       |
------------------------------------------------------------------------------*/
Boolean ICVKeyboardHandler :: key ( IKeyboardEvent& keyevt )
{
   Boolean bProcessed = false;

   if (!(keyevt.isUpTransition()))
   {                                   // Key press.
      if ( keyevt.isCharacter() ||     // Character key (mnemonic?)
           (( keyevt.isAltDown()  ||  keyevt.isCtrlDown() )  &&
#ifdef IC_PM
            keyevt.parameter2().number1() ))
#endif
#ifdef IC_WIN
            MapVirtualKey( keyevt.parameter1(), 2 ) ))
#endif
      {                                // Data key or Alt+key or Ctrl+key.
         ITRACE_ALL( IString( "key - Mnemonic hwnd=" ) +
                     IString( keyevt.handle().asUnsigned() ).d2x() +
                     IString( " msg=" ) +
                     IString( keyevt.eventId() ).d2x() +
                     IString( " wParam=" ) +
                     IString( keyevt.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( keyevt.parameter2().asUnsignedLong() ).d2x() );
         /**********************************************************/
         /* Emulate PM's handling of accelerators (mnemonics for   */
         /* buttons, for example).                                 */
         /**********************************************************/
         IWindowHandle hwndMnemonic(0);
         IWindow* pcvCanvas = keyevt.window();
         unsigned short usMnemonic =
#ifdef IC_PM
                       (unsigned short)(keyevt.parameter2().number1());
#endif
#ifdef IC_WIN
                       (unsigned short) MapVirtualKey( keyevt.parameter1(), 2);
#endif

         if (usMnemonic)
         {                             // Have a possible mnemonic.
            IWindow* pwndOwner;
            hwndMnemonic = pcvCanvas->matchForMnemonic(usMnemonic);
            if (hwndMnemonic == 0  &&
                (pwndOwner = pcvCanvas->owner())  &&
                pwndOwner->isFrameWindow())
            {                          // Try frame extensions also.
               IWindow::ChildCursor cursor(*pwndOwner);
               IWindowHandle hwndCanvas = keyevt.handle();
               IWindow* pwndSibling;
               for (cursor.setToFirst();
                    bProcessed == false  &&  cursor.isValid();
                    cursor.setToNext())
               {                       // Get children of frame.
                  IWindowHandle hwndSibling = pwndOwner->childAt(cursor);
                  if (hwndSibling != hwndCanvas  &&
                      (pwndSibling =
                         IWindow::windowWithHandle(hwndSibling))  &&
                      (hwndMnemonic =
                         pwndSibling->matchForMnemonic(usMnemonic)))
                  {                    // Button mnemonic matched.
                     bProcessed = true;  // Flag as done.
                  }
               }                       // End for all children of frame.
            }                          // End frame extension check.
         }                             // End character key.

         if ( hwndMnemonic  &&
              ( hwndMnemonic.sendEvent( WM_QUERYDLGCODE, 0, 0 )
                            .asUnsignedLong() & DLGC_STATIC ) )
         {  // The control is a static text.
            // Find the next sibling window with a tab stop.  If this
            // canvas/frame does not have a window with a tab stop following
            // it, then treat the mnemonic as invalid (rather than trying to
            // deal with finding a next window--either by wrapping back to
            // the first sibling or crossing canvas boundaries).
            IWindowHandle
              mnemonicParent( IPARENTOF( hwndMnemonic ) );
            IWindow
             *parentWindow = IWindow::windowWithHandle( mnemonicParent );
            if ( parentWindow )
            {
               Boolean found = false;
               IWindow::ChildCursor
                 cursor( *parentWindow );
               for ( cursor.setToFirst();
                     !found  &&  cursor.isValid();
                     cursor.setToNext() )
               {  // First find the static text control.
                  if ( parentWindow->childAt( cursor ) == hwndMnemonic )
                  {
                     // Now start the search for a tabbable sibling.
                     // Put the result into hwndMnemonic.
                     hwndMnemonic = 0;
                     for ( cursor.setToNext();
                           !found  &&  cursor.isValid();
                           cursor.setToNext() )
                     {
                        IWindowHandle
                          hwndChild( parentWindow->childAt( cursor ) );
                        if ( ( ISTYLEOF( hwndChild )
                                & ( WS_TABSTOP | WS_DISABLED | WS_VISIBLE ) )
                                     == ( WS_TABSTOP | WS_VISIBLE ) )
                        {  // Found a good sibling window.
                           hwndMnemonic = hwndChild;
                           found = true;
                        }
                     } // End inner loop.
                  }
               } // End outer loop.
            }
            else
            {  // Don't bother with this case.
               hwndMnemonic = 0;
            }
         }

         if (hwndMnemonic)
         {                             // Found a window for it.
            /********************************************************/
            /* Note: Don't worry about returning default emphasis   */
            /* to the original default push button for now for the  */
            /* case where you're going from a push button to a      */
            /* radio button or check box.                           */
            /********************************************************/
            //---------------------------------------------------
            // DEFECT 8015 : call setfocus() AFTER buttonSetup()
            //---------------------------------------------------
            buttonSetup(pcvCanvas, hwndMnemonic, enteringGroup);
            ISETFOCUS(IWindow::desktopWindow()->handle(),
                                              hwndMnemonic);
            //---------------------------------------------------
            // DEFECT 8015 : For non radio buttons, click the button.   
            //               buttonSetup() handles radio buttons.
            //---------------------------------------------------
            unsigned long ulCode = hwndMnemonic.sendEvent(WM_QUERYDLGCODE, 0, 0);
            if ( !(ulCode & DLGC_RADIOBUTTON) )
            {
#ifdef   IC_PM
               hwndMnemonic.sendEvent(BM_CLICK, true, 0);
#endif   //IC_PM
#ifdef   IC_WIN
               ICLICKBUTTON(hwndMnemonic);
#endif   //IC_WIN
            }
            bProcessed = true;         // Try selecting the window.
         }
      }

      if (!bProcessed  &&  keyevt.isVirtual())
      {                                // Virtual key press/repeat-press.
         ITRACE_ALL( IString( "key - virtual hwnd=" ) +
                     IString( keyevt.handle().asUnsigned() ).d2x() +
                     IString( " msg=" ) +
                     IString( keyevt.eventId() ).d2x() +
                     IString( " wParam=" ) +
                     IString( keyevt.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( keyevt.parameter2().asUnsignedLong() ).d2x() );
         IKeyboardEvent::VirtualKey vkKey = keyevt.virtualKey();
         switch (vkKey)
         {
            case (IKeyboardEvent::tab):
            case (IKeyboardEvent::backTab):
               bProcessed = handleCursorMovement(keyevt);
               break;

            case (IKeyboardEvent::left):
            case (IKeyboardEvent::right):
            case (IKeyboardEvent::up):
            case (IKeyboardEvent::down):
               // (IKeyboardEvent::virtualKey screens out NumLock case.)
               bProcessed = handleCursorMovement(keyevt);
               break;

            case (IKeyboardEvent::enter):
            case (IKeyboardEvent::newLine):
               bProcessed = handleEnter(keyevt);
               break;

#ifdef IC_WIN
            // Standard Windows dialogs process Esc as if the cancel
            // button was pressed.
            case (IKeyboardEvent::esc):
               if (keyevt.window()->isFrameWindow())
                  {
                  // lParam is the ID of the Cancel button if there
                  // is one, 0 if not.
                  IWindowHandle hwndCancel = IWINDOWFROMID(
                                                keyevt.handle(), IDCANCEL);
                  keyevt.handle().postEvent( WM_COMMAND,
                                            IDCANCEL,
                                            (HWND)hwndCancel);
                  bProcessed = true;
                  }
               break;
#endif

            default:
               break;
         }  /* end switch */
      }
   }                                   // End key press.

   return bProcessed;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleCursorMovement                                     |
|                                                                              |
| Handles tabbing between groups and canvases and cursor movement within a     |
| group.                                                                       |
| Notes: This processing is necessary since PM's frame window procedure only   |
|          gives tabbing and cursoring between immediate child windows of the  |
|          frame, and we need to support tabbing and cursoring to child        |
|          windows of a canvas (which could be a child, or grandchild, or      |
|          great grandchildren, etc., of the frame).  However, where we are    |
|          tabbing from the last child or back tabbing from the first child of |
|          a canvas, we will let the owner window determine if there is a      |
|          siblign window that should get the input focus.                     |
|        All controls in a group are assumed to be contained in (children of)  |
|          the canvas.                                                         |
------------------------------------------------------------------------------*/
Boolean ICVKeyboardHandler :: handleCursorMovement ( IKeyboardEvent& keyevt )
{
   Boolean bProcessed = false;

   IWindow* pcvCanvas = keyevt.window();
   IWindowHandle hwndCanvas = keyevt.handle();
   IWindowHandle hwndFocus =
                   IQUERYFOCUS(IWindow::desktopWindow()->handle());
   if (hwndFocus  &&  IISCHILD(hwndFocus, hwndCanvas))
   {                              // Child with input focus.
      unsigned long ulNextControlType;
      IKeyboardEvent::VirtualKey vkKey = keyevt.virtualKey();
      switch (vkKey)
      {
         /**********************************************************/
         /* The use of WinEnumDlgItem here assumes that either all */
         /* child windows have been created with HWND_BOTTOM       */
         /* positioning, or the z-order of the child windows has   */
         /* been reversed by the canvas, so that the order that    */
         /* child windows are returned by WinEnumDlgItem matches   */
         /* the expected order that the windows are layed out and  */
         /* tabbed through.                                        */
         /**********************************************************/
         case (IKeyboardEvent::tab):
            ulNextControlType = EDI_NEXTTABITEM;
            break;

         case (IKeyboardEvent::backTab):
            ulNextControlType = EDI_PREVTABITEM;
            break;

         case (IKeyboardEvent::right):
         case (IKeyboardEvent::down):
            ulNextControlType = EDI_NEXTGROUPITEM;
            break;

         case (IKeyboardEvent::left):
         case (IKeyboardEvent::up):
            ulNextControlType = EDI_PREVGROUPITEM;
            break;

         default:                 // Code error.
            ITHROWGUIERROR2("IKeyboardEvent::VirtualKey",
                            IErrorInfo::invalidParameter,
                            IException::recoverable);
            break;
      }  /* end switch */

                                  // Find next control in this canvas.
      IWindowHandle hwndNextControl =
            IENUMDLGITEM(hwndCanvas, hwndFocus, ulNextControlType);
      if (hwndNextControl)
      {                    // Need to change control with input focus.
         bProcessed = true;       // Assume staying in canvas.
         /**********************************************************/
         /*  Let the owner (another canvas or PM's frame window    */
         /*  procedure) handle the case of tabbing and back-       */
         /*  tabbing from this canvas (if passed to                */
         /*  WinDefWindowProc, the WM_CHAR will be passed up the   */
         /*  chain to the next owner).  This class must process    */
         /*  the WM_SETFOCUS message to set the input focus from   */
         /*  the canvas window to the appropriate child control    */
         /*  when the canvas window is tabbed to.                  */
         /**********************************************************/
         if (ulNextControlType == EDI_NEXTTABITEM  &&
             hwndNextControl ==
                     IENUMDLGITEM(hwndCanvas, 0, EDI_FIRSTTABITEM))
         {            // Really tabbing off the canvas (wrapped?).
            IWindowHandle hwndChild;
            IWindowHandle hwndParent = hwndFocus;

            do
            {                     // Find immediate child of canvas.
               hwndChild = hwndParent;
               hwndParent = IPARENTOF(hwndChild);
            }
            while (hwndParent != hwndCanvas);

            Boolean bDone = false;
            IWindow::ChildCursor cursor(*pcvCanvas);
            for (cursor.setToFirst();
                 bDone == false  &&  cursor.isValid();
                 cursor.setToNext())
            {
               IWindowHandle hwndTest = pcvCanvas->childAt(cursor);
               if (hwndTest == hwndNextControl)
               {        // Next tab item precedes control tabbed from.
                  bProcessed = false;  // Let owner handle this.
#ifdef IC_WIN
                  // Below is a Windows only exception.
                  // In Windows, this handler is used for the frame
                  // window as will.  If we are handling the frame then we
                  // can't punt.  Set focus to the child.
                  if (pcvCanvas->isFrameWindow())
                  {
                     bProcessed = true;
                  }
                  else
#endif
                  {
                     /*************************************************/
                     /*  Below is an exception.  The Tab will not be  */
                     /*  allowed to proceed to the owner window if    */
                     /*  the canvas is a notebook page window.  If    */
                     /*  the notebook is the owner, Tab causes the    */
                     /*  input cursor to disappear.  If the owner of  */
                     /*  the notebook is the owner, then the Tab will */
                     /*  change the input focus to the notebook.  No  */
                     /*  other owner window gives any better results, */
                     /*  so the Tab is handled here instead.          */
                     /*************************************************/
                     HWND parent = IPARENTOF(hwndCanvas);
                     HWND grandParent = IPARENTOF(parent);
                                       // Notebook has intermediate window.
                     if (grandParent)
                     {                    // Notebook was orig parent?
                        IWindowClassName className( grandParent );
                        if (className == WC_NOTEBOOK)
                        {                 // Parent was notebook.
                           bProcessed = true;
                        }
                     }                    // End know who grand parent is.
                  }     // else for notebook
                  bDone = true;        // Allow tab off the canvas.
               }
               else if (hwndTest == hwndChild)
               {          // Control tabbed from precedes next tab item.
                  bDone = true;        // Tabbing to hwndNextControl.
               }
            }
         }
         else if (ulNextControlType == EDI_PREVTABITEM  &&
                  hwndNextControl ==
                     IENUMDLGITEM(hwndCanvas, 0, EDI_LASTTABITEM))
         {                        // Back-tabbing off the canvas?
            IWindowHandle hwndChild;
            IWindowHandle hwndParent = hwndFocus;
            do
            {                     // Find immediate child of canvas.
               hwndChild = hwndParent;
               hwndParent = IPARENTOF(hwndChild);
            }
            while (hwndParent != hwndCanvas);

            IWindow::ChildCursor cursor(*pcvCanvas);
            Boolean bDone = false;

            for (cursor.setToFirst();
                 bDone == false  &&  cursor.isValid();
                 cursor.setToNext())
            {
               IWindowHandle hwndTest = pcvCanvas->childAt(cursor);
               if (hwndTest == hwndChild)
               {       // Control tabbed from precedes first tab item.
                  bProcessed = false;  // Let owner handle.
#ifdef IC_WIN
                  // Below is a Windows only exception.
                  // In Windows, this handler is used for the frame
                  // window as will.  If we are handling the frame then we
                  // can't punt.  Set focus to the child.
                  if (pcvCanvas->isFrameWindow())
                  {
                     bProcessed = true;
                  }
                  else
#endif
                  {
                     /*************************************************/
                     /*  Below is an exception.  The backtab will not */
                     /*  be allowed to proceed to the owner window if */
                     /*  the canvas is a notebook page window.  If    */
                     /*  the notebook is the owner, backtab causes    */
                     /*  the input cursor to disappear.  If the owner */
                     /*  of the notebook is the owner, then the       */
                     /*  backtab will change the input focus to the   */
                     /*  notebook.  No other owner window gives any   */
                     /*  better results, so the backtab is handled    */
                     /*  here instead.                                */
                     /*************************************************/
                     HWND parent = IPARENTOF(hwndCanvas);
                     HWND grandParent = IPARENTOF(parent);
                                       // Notebook has intermediate window.
                     if (grandParent)
                     {                    // Notebook was orig parent?
                        IWindowClassName className( grandParent );
                        if (className == WC_NOTEBOOK)
                        {                 // Parent was notebook.
                           bProcessed = true;
                        }
                     }                    // End know who grand parent is.
                  }     // else for notebook
                  bDone = true;        // Allow back-tab off canvas.
               }
               else if (hwndTest == hwndNextControl)
               {          // Next tab item precedes control tabbed from.
                  bDone = true;        // Back-tabbing to hwndNextControl.
               }
            }
         }

         /**********************************************************/
         /* Check for any fix-ups for buttons.                     */
         /**********************************************************/
         if (bProcessed == false)
         {                  // Leaving group (owner window to handle).
            buttonSetup(pcvCanvas, hwndFocus, leavingGroup);
         }
         else if (hwndNextControl != hwndFocus)
         {                  // New child is group getting input focus.
            //-------------------------------------------------------------------
            // DEFECT 8015 : only force a click of radiobutton if moving within
            //       an existing radiobutton group, not tabbing into the group.
            //-------------------------------------------------------------------
            Boolean bClick(FALSE);
            if ( ulNextControlType == EDI_NEXTGROUPITEM ||
                 ulNextControlType == EDI_PREVGROUPITEM )
               bClick = TRUE;
            //-------------------------------------------------------------------
            buttonSetup(pcvCanvas, hwndNextControl, withinGroup, bClick); // 8015
            ISETFOCUS(IWindow::desktopWindow()->handle(),
                      hwndNextControl);
         }
      } /* endif have a tabbable child */
#ifdef IC_WIN
      else if (pcvCanvas->isFrameWindow())
      {
         // There is no tabbable child, and this is a frame.  Set focus
         // to the frame.  This will cause gotFocus to eventually
         // run for this window and set focus to the client or first
         // child.
         buttonSetup(pcvCanvas, hwndFocus, leavingGroup);
         ISETFOCUS(IWindow::desktopWindow()->handle(),
                   pcvCanvas->handle() );
         bProcessed = true;
      }
#endif //IC_WIN

   } /* endif */

   return bProcessed;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEnter                                              |
|                                                                              |
| Process the Enter key by causing the default push button to be selected.     |
| Notes: The push button selected is the button current hilited as the default |
|          rather than the original default push button.                       |
------------------------------------------------------------------------------*/
Boolean ICVKeyboardHandler :: handleEnter ( IKeyboardEvent& keyevt )
{
   IMODTRACE_ALL("ICVKeyboardHandler::handleEnter");
   Boolean bProcessed = false;
   IWindow *pwndOwner;
   IWindow *pwndCanvas = keyevt.window();
   IWindowHandle hwndDefault = pwndCanvas->defaultPushButton();

   if (hwndDefault == 0  &&
       (pwndOwner = pwndCanvas->owner())  &&
       (pwndOwner->isFrameWindow()))
   {                                   // Try frame extensions also.
      IWindow::ChildCursor cursor(*pwndOwner);
      IWindowHandle hwndCanvas = keyevt.handle();
      IWindow* pwndSibling;
      for (cursor.setToFirst();
           bProcessed == false  &&  cursor.isValid();
           cursor.setToNext())
      {
         IWindowHandle hwndSibling = pwndOwner->childAt(cursor);
         if (hwndSibling != hwndCanvas  &&
             (pwndSibling = IWindow::windowWithHandle(hwndSibling))  &&
             (hwndDefault = pwndSibling->defaultPushButton()))
         {                             // A default push button found.
            bProcessed = true;         // Flag as done.
         }
      }                                // End for all children of frame.
   }                                   // End frame extension check.

   if (hwndDefault)
   {                                   // A default push button found.
      ICLICKBUTTON(hwndDefault);
      bProcessed = true;               // Flag as done.
   }
   return bProcessed;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::buttonSetup                                                  |
|                                                                              |
| Handles cursor selection of radio buttons and default push button hiliting   |
| with cursor movement.                                                        |
------------------------------------------------------------------------------*/
void ICVKeyboardHandler :: buttonSetup (
                                        const IWindow* pcvCanvas,
                                        const IWindowHandle& hwndChild,
                                              State state,
                                              Boolean fClickRadio/*= TRUE*/ )
{
   IASSERTPARM(pcvCanvas != 0);

   /****************************************************************/
   /* Select the specified control if it is a radio button without */
   /* the BS_NOCURSORSELECT style.                                 */
   /****************************************************************/
   unsigned long ulCode = hwndChild.sendEvent(WM_QUERYDLGCODE, 0, 0);

   //-------------------------------------------------------------------
   // DEFECT 8015 : add a new flag argument, and only force a click of 
   //               the radiobutton if new arg value allows same.
   //-------------------------------------------------------------------
   if (fClickRadio && (ulCode & DLGC_RADIOBUTTON)  &&
       state != ICVKeyboardHandler::leavingGroup)
   {                              // Child window is a radio button.
#ifdef   IC_PM
      if (!(IQUERYWINDOWULONG(hwndChild, QWL_STYLE) &
                                       BS_NOCURSORSELECT))
      {               // Processing for cursor selected radio buttons.
         hwndChild.sendEvent(BM_CLICK, true, 0);
      }
#endif   //IC_PM
#ifdef   IC_WIN
      IWindow* rbWindow = IWindow::windowWithHandle( hwndChild, false );
      if ( rbWindow && ((IRadioButton*)rbWindow)->allowsMouseClickFocus() )
      {
         ICLICKBUTTON( hwndChild );
      }
      // IC_NOTYET    BS_NOCURSORSELECT not supported in Windows.
//      hwndChild.sendEvent(BM_SETCHECK, true, 0);
      // find and uncheck all others in the group
//      IWindowHandle hwndGroup = GetNextDlgGroupItem(
//                                   pcvCanvas->handle(),
//                                   hwndChild,
//                                   false );
//      while (hwndGroup != hwndChild)
//         {
//         hwndGroup.sendEvent(BM_SETCHECK, false, 0);
//         hwndGroup = GetNextDlgGroupItem(
//                        pcvCanvas->handle(),
//                        hwndGroup,
//                        false );
//         }
#endif   //IC_WIN
   }

   /****************************************************************/
   /* Emulate PM's behavior with default push button hilighting as */
   /* the cursor is moved:                                         */
   /*   - change the default push button as you cursor through a   */
   /*     group of push buttons                                    */
   /*   - restore the default button when tabbing from a group     */
   /*   - ignore the default push button when tabbing to a group   */
   /*   - ignore the default button when selecting a push button   */
   /*     using the mouse                                          */
   /****************************************************************/
   switch (state)
   {
      case (ICVKeyboardHandler::enteringGroup):
      {                    // Find and return the default push button.
         if (ulCode & DLGC_PUSHBUTTON)
         {                 // Make current push button look like default.
            //----------------------------------------------------
            // DEFECT 20847 : prev called ISETDEFAULTPUSHBUTTON()
            //----------------------------------------------------
            setDefaultPushButton( pcvCanvas, hwndChild, true );
            //----------------------------------------------------
         }
         break;
      } /* end case */

      case (ICVKeyboardHandler::leavingGroup):
      {                    // Need to restore original default push button.
         IWindowHandle hwndDefault = origDefaultButtonHandle(pcvCanvas);
         if (hwndDefault == 0)
         {                 // No application-defined default push button.
            if (ulCode & DLGC_DEFAULT)
            {                     // Current control is a push button.
               //----------------------------------------------------
               // DEFECT 20847 : prev called ISETDEFAULTPUSHBUTTON()
               //----------------------------------------------------
               setDefaultPushButton( pcvCanvas, hwndChild, false );
               //----------------------------------------------------
            }                     // Be sure to de-emphasize it now.
         }
         else
         {                        // Restore original default button.
            //----------------------------------------------------
            // DEFECT 20847 : prev called ISETDEFAULTPUSHBUTTON()
            //----------------------------------------------------
            setDefaultPushButton( pcvCanvas, hwndDefault, true );
            //----------------------------------------------------
         }
         break;
      } /* end case */

      case (ICVKeyboardHandler::withinGroup):
      {                           // Use specified child over default.
         if (ulCode & DLGC_PUSHBUTTON)
         {                        // Make push button look like default.
            //----------------------------------------------------
            // DEFECT 20847 : prev called ISETDEFAULTPUSHBUTTON()
            //----------------------------------------------------
            setDefaultPushButton( pcvCanvas, hwndChild, true );
            //----------------------------------------------------
         }
         else if (!(ulCode & DLGC_DEFAULT))
         {                        // Not a push button (and not default one).
            IWindowHandle hwndDefault = origDefaultButtonHandle(pcvCanvas);
            if (hwndDefault)
            {                     // Have a default push button.
               //----------------------------------------------------
               // DEFECT 20847 : prev called ISETDEFAULTPUSHBUTTON()
               //----------------------------------------------------
               setDefaultPushButton( pcvCanvas, hwndDefault, true );
               //----------------------------------------------------
            }                     // Ensure it looks like the default.
         }
         break;
      } /* end case */

      default:                    // Uninitialized enum.
         ITHROWLIBRARYERROR(IC_INVALIDENUMVALUE,
                            IErrorInfo::invalidParameter,
                            IException::unrecoverable);
         break;
   } /* end switch */

   return;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEventsFor                                              |
|                                                                              |
| Attach the handler to the canvas.                                            |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: handleEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::handleEventsFor(canvas);
   return *this;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::stopHandlingEventsFor                                        |
|                                                                              |
| Detach the handler from the canvas.                                          |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: stopHandlingEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::stopHandlingEventsFor(canvas);
   return *this;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEventsFor                                              |
|                                                                              |
| Attach the handler to the frame window.                                      |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: handleEventsFor ( IFrameWindow* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::handleEventsFor(canvas);
   return *this;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::stopHandlingEventsFor                                        |
|                                                                              |
| Detach the handler from the frame window.                                    |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: stopHandlingEventsFor ( IFrameWindow* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::stopHandlingEventsFor(canvas);
   return *this;
}
#endif


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEventsFor                                              |
------------------------------------------------------------------------------*/
IHandler& ICVKeyboardHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::stopHandlingEventsFor                                        |
------------------------------------------------------------------------------*/
IHandler& ICVKeyboardHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}

/*------------------------------------------------------------------------------
| origDefaultButtonHandle                                                      |
------------------------------------------------------------------------------*/
IWindowHandle origDefaultButtonHandle(const IWindow* canvas )
{
  IASSERTPARM( canvas != 0 );
  IWindowHandle hwndDefault = 0;

  // Find the frame window.
  IEventResult mr = canvas->sendEvent( WM_QUERYFOCUSCHAIN,
                                     IEventParameter1( QFC_FRAME ),
                                     IEventParameter2( 0 ));
  IWindowHandle hwndFrame( mr );

  if ( hwndFrame )
  {                // Found the frame window.
#ifdef IC_PM
     // Get the default button the frame knows of.
     // (Can't use IWindow::windowULong, because it's protected.)
     hwndDefault = IQUERYWINDOWULONG( hwndFrame, QWL_DEFBUTTON );
#endif
#ifdef IC_WIN
     // IC_NOTYET ... may want to store in an extra window word.
     unsigned long result = hwndFrame.sendEvent( DM_GETDEFID );
     if (result)
        {
        hwndDefault = IWINDOWFROMID(hwndFrame, LOWORD(result) );
        }
#endif
  }

  return hwndDefault;
}

//------------------------------------------------------------------
// DEFECT 20847 : new local function
//    Set the DEFAULT pushbutton flag for the control, and
//    clear the default flag from any sibling controls.
//------------------------------------------------------------------
void setDefaultPushButton ( const IWindow* pcvCanvas,
                            IWindowHandle  pushbutton,
                            IBase::Boolean setAsDefault )
{
#ifdef IC_PM
   unsigned long ulPbCode = pushbutton.sendEvent(WM_QUERYDLGCODE, 0, 0);

   if ( setAsDefault )                          // make it the default
   {
      if (ulPbCode & DLGC_PUSHBUTTON )          // if not already the default
      {
         // Calls BM_SETDEFAULT for the pushbutton
         ISETDEFAULTPUSHBUTTON( pushbutton, setAsDefault );

         // Need to clear default button if set on any other pushbuttons
         IWindowHandle hwndChild;
         unsigned long ulChildCode;
         IWindow* pCanvas = (IWindow*)pcvCanvas;
         IWindow::ChildCursor cursor(*pCanvas);
         for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
         {
            hwndChild = pcvCanvas->childAt(cursor);
            ulChildCode = hwndChild.sendEvent(WM_QUERYDLGCODE, 0, 0);
            if (ulChildCode & DLGC_DEFAULT )       // if already the default
               {
                  // Calls BM_SETDEFAULT for the pushbutton
                  ISETDEFAULTPUSHBUTTON( hwndChild, FALSE );
               }
         }
      }
   }
   else                                         // make it non-default
   {
      if ( ulPbCode & DLGC_DEFAULT )
      {
         // Calls BM_SETDEFAULT for the pushbutton
         ISETDEFAULTPUSHBUTTON( pushbutton, setAsDefault );
      }
   }
#else
   // Macro calls isetDefaultPushButton() to process DEFAULT style
   // settings for Windows in a manner similar to the code above.
   ISETDEFAULTPUSHBUTTON( hwndChild, true );
#endif
}
//------------------------------------------------------------------

