/*******************************************************************************
* FILE NAME: icanvas.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in icanvas.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512  (from iwindow.cpp)
#pragma priority( -2147482112 )

#ifndef _IBASE_
  #include <ibase.hpp>
#endif

extern "C" {
  #ifdef IC_PM
  #define INCL_WINDIALOGS           // DLGC_xxx
  #define INCL_WINERRORS            // PMERR_xxx
  #define INCL_WINFRAMEMGR          // WM_QUERYFOCUSCHAIN
  #define INCL_WINSTATICS           // SS_BKGNDRECT
  #define INCL_WINSYS               // PP_BACKGROUNDCOLOR
  #define INCL_WINWINDOWMGR         // QWL_xxx
  #endif
  #include <iwindefs.h>
}

#include <icanvas.hpp>

#ifdef IC_PMWIN
#include <icanvprv.hpp>
#endif
#include <icconst.h>
#include <icmnfun.hpp>
#include <icolor.hpp>
#include <icvhdr.hpp>
#include <iexcept.hpp>
#include <ipoint.hpp>
#include <irecohdr.hpp>
#include <irect.hpp>
#include <itrace.hpp>
#include <iwposbuf.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ICANVAS_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| Public canvas styles.                                                        |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLStaticConst)
const ICanvas::Style ICanvas::classDefaultStyle = WS_VISIBLE;
#pragma data_seg()

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
ICanvas::Style ICanvas::currentDefaultStyle     = WS_VISIBLE;

/*------------------------------------------------------------------------------
| The static object that ICanvas and all of its derived classes use to obtain  |
| the common canvas handler.                                                   |
------------------------------------------------------------------------------*/
static ICanvasStatics CanvasStatics;
#pragma data_seg()


#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ICanvasData::ICanvasData                                                     |
|                                                                              |
| Constructor for the private canvas data class.                               |
------------------------------------------------------------------------------*/
ICanvasData::ICanvasData ( )
  : ICanvasData::Inherited(),
    fShowWindow( true ),
    fChildCount( 0 )
{ }
#endif

/*------------------------------------------------------------------------------
| ICanvas::ICanvas                                                             |
|                                                                              |
| Constructor to create a canvas window.                                       |
| Notes: The canvas window is created as a WC_STATIC, SS_BGNDRECT window       |
|        (fill box control).                                                   |
------------------------------------------------------------------------------*/
ICanvas::ICanvas ( unsigned long windowIdentifier,
                   IWindow* parent, IWindow* owner,
                   const IRectangle& initial, const Style& style )
  : sizClLayout(), hwndClDefaultButton( 0 ),
    bClChildrenReversed( false ), bClDefaultHandlerAdded( false ),
#ifdef IC_PMWIN
    fCanvasData( new ICanvasData() )
#endif
#ifndef IC_PMWIN
    fCanvasData( 0 )
#endif
{
  // Save the extended style to make sure we have a copy of it stored.
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  // Create fill box control.
  this->initialize( windowIdentifier,
                    parent,
                    owner,
                    initial,
                    this->convertToGUIStyle( style ),
                    0 );

  /*****************************************************************/
  /* Implementation notes:                                         */
  /* - The WS_TABSTOP style is added in later by layout() only if  */
  /*   the canvas has a tab-able child window.  This window style  */
  /*   is needed to allow a frame or canvas parent to identify the */
  /*   canvas as a tab-able child window with WinEnumDlgItem       */
  /*   during Tab key processing (PM does this for a frame window, */
  /*   and the ICanvasHandler class does this for a parent         */
  /*   canvas), and for the ICanvasHandler to give the input focus */
  /*   to the appropriate child window during processing of a      */
  /*   WM_SETFOCUS message.                                        */
  /* - Like all other controls, the initial rectangle is only used */
  /*   if the canvas is not a frame control and not a child of     */
  /*   another canvas window (if it is a child of another canvas,  */
  /*   it will size itself to be able to hold all of its child     */
  /*   windows in calcMinimumSize()).                              */
  /*****************************************************************/
  CanvasStatics.canvasHandler().handleEventsFor( this );
  CanvasStatics.keyboardHandler().handleEventsFor( this );
                                            // Add private handler.
  bClDefaultHandlerAdded = true;            // Flag as added.

  // Add the default recoordination handler.  This handler is needed
  // for the canvas classes that do not do their own positioning of
  // child windows.   Those that do child window positioning need to
  // later remove the handler.
  IRecoordHandler::defaultHandler()->handleEventsFor( this );

  if ( IWindow::defaultOrdering() == IWindow::behindSiblings )
  {                                    // No need to reverse children.
     bClChildrenReversed = true;
  }
}

/*------------------------------------------------------------------------------
| ICanvas::ICanvas                                                             |
------------------------------------------------------------------------------*/
ICanvas::ICanvas ( )
  : sizClLayout(), hwndClDefaultButton( 0 ),
    bClChildrenReversed( false ), bClDefaultHandlerAdded( false ),
#ifdef IC_PMWIN
    fCanvasData( new ICanvasData() )
#endif
#ifndef IC_PMWIN
    fCanvasData( 0 )
#endif
{
  CanvasStatics.canvasHandler().handleEventsFor( this );
  CanvasStatics.keyboardHandler().handleEventsFor( this );
                                            // Add private handler.
  bClDefaultHandlerAdded = true;            // Flag as added.

  // Assume that derived classes that need the recoordination handler
  // will add one themselves (canvases that do not do their own
  // positioning of child windows).

  if ( IWindow::defaultOrdering() == IWindow::behindSiblings )
  {                                    // No need to reverse children.
     bClChildrenReversed = true;
  }
}

/*------------------------------------------------------------------------------
| ICanvas::initialize                                                          |
|                                                                              |
| Create the presentation system window.                                       |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::initialize ( unsigned long     windowIdentifier,
                               IWindow*          parent,
                               IWindow*          owner,
                               const IRectangle& initialRect,
                               unsigned long     style,
                               unsigned long     extendedStyle )
{
  IASSERTPARM( parent != 0 );

  IWindowHandle canvasHandle =
                   this->create( windowIdentifier,
                                 0,
                                 style,
                                 WC_STATIC,
                                 parent->handle(),
                                 owner ?
                                   owner->handle() : IWindowHandle( 0 ),
                                 initialRect,
                                 0,
                                 0 );
  this->startHandlingEventsFor( canvasHandle );

  // Remove styles that were not specified by the caller, but which
  // get added via class styles.
  unsigned long
    currentStyle = this->style(),
    newStyle     = currentStyle;
  // Prevent painting over parent.
  if ( ! ( style & IWindow::clipToParent.asUnsignedLong() ) )
  {           // Don't let canvas paint over its parent.
     newStyle &= ~IWindow::clipToParent.asUnsignedLong();
  }
  if ( ! ( style & IWindow::synchPaint.asUnsignedLong() ) )
  {           // Disable synchronous painting.
     newStyle &= ~IWindow::synchPaint.asUnsignedLong();
  }
  if ( newStyle != currentStyle )
  {
     this->setStyle( newStyle );
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::~ICanvas                                                            |
------------------------------------------------------------------------------*/
ICanvas::~ICanvas ( )
{
  // Remove the recoordination handler.  May be a noop if never added,
  // or a derived class already removed it in its ctor.
  IRecoordHandler::defaultHandler()->stopHandlingEventsFor( this );

  if ( bClDefaultHandlerAdded )
  {                               // Remove the private handler.
     CanvasStatics.keyboardHandler().stopHandlingEventsFor( this );
     CanvasStatics.canvasHandler().stopHandlingEventsFor( this );
     bClDefaultHandlerAdded = false;        // Flag as removed.
  }

#ifdef IC_PMWIN
  delete fCanvasData;
#endif
#ifndef IC_PMWIN
  // delete fCanvasData;
#endif
}

/*------------------------------------------------------------------------------
| ICanvas::defaultStyle                                                        |
|                                                                              |
| Return the current default style for the class.                              |
------------------------------------------------------------------------------*/
ICanvas::Style ICanvas::defaultStyle ( )
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| ICanvas::convertToGUIStyle                                                   |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long ICanvas::convertToGUIStyle ( const IBitFlag& guiStyle,
                                           Boolean bExtOnly ) const
{
  // Obtain the style from the class (IControl) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if ( bExtOnly )
  {
    // Use mask to only return extended styles in the user defined range.
    ulStyle |= extendedStyle() & IS_EXTMASK;
  }
  else
  {
    // No styles are defined for the base canvas, so we do not mask.
    ulStyle |= guiStyle.asUnsignedLong();

    // Add the static background rectangle and group styles.
    ulStyle |= SS_BKGNDRECT | IControl::group.asUnsignedLong()
#ifdef IC_WIN
                            | WS_CHILD
#endif
       ;        // That's all.
  }

  return ulStyle;
}

/*------------------------------------------------------------------------------
| ICanvas::backgroundColor                                                     |
|                                                                              |
| Returns the background color of the canvas.                                  |
------------------------------------------------------------------------------*/
IColor ICanvas::backgroundColor ( ) const
{
#ifdef   IC_PM
   return Inherited::color( PP_BACKGROUNDCOLOR,
                            defaultbackgroundcolor( handle() ));
              // The above accounts for dialog vs window background.
#endif   //IC_PM
#ifdef   IC_WIN
   // IC_NOTYET ... color model
   return Inherited::backgroundColor( );
#endif   //IC_WIN
}

/*------------------------------------------------------------------------------
| ICanvas::isTabStop                                                           |
|                                                                              |
| Returns true if the canvas can be tabbed to based on whether any of its      |
| child windows can be tabbed to.                                              |
------------------------------------------------------------------------------*/
Boolean ICanvas::isTabStop ( ) const
{
  Boolean bTabStop = false;       // Assume no child windows are tab stops.
  IWindowHandle hwndChild = 0;
  IWindow::ChildCursor cursor( *(IWindow*)this );

  for ( cursor.setToFirst();
        cursor.isValid()  &&  bTabStop == false;
        cursor.setToNext() )
  {                               // Enumerate all child windows.
     Boolean bNewChild = false;
     IWindowHandle hwndChild = this->childAt( cursor );
     IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );
     if ( !pwndChild )
     {                            // Child window object found.
        pwndChild = new IWindow( hwndChild );
        bNewChild = true;
     }
     bTabStop = pwndChild->isTabStop();
              // Note: Can't just check style directly, since if the
              // child is another canvas, its isTabStop() function
              // needs to be called.
     if ( bNewChild )
     {                                 // Memory clean-up needed.
        delete pwndChild;
     }
  }                               // End for all child windows.

  ((ICanvas*)this)->enableTabStop( bTabStop );
                 // Make non-const to store appropriate tabStop style.
  return bTabStop;
}

/*------------------------------------------------------------------------------
| ICanvas::origDefaultButtonHandle                                             |
|                                                                              |
| Return the original default push button for the frame window.                |
------------------------------------------------------------------------------*/
IWindowHandle ICanvas::origDefaultButtonHandle ( ) const
{
  IWindowHandle hwndDefault = 0;

  // Find the frame window.
  IEventResult mr = this->sendEvent( WM_QUERYFOCUSCHAIN,
                                     IEventParameter1( QFC_FRAME ),
                                     IEventParameter2( 0 ));
  IWindowHandle hwndFrame( mr );

  if ( hwndFrame )
  {                // Found the frame window.
#ifdef IC_PM
     // Get the default button the frame knows of.
     // (Can't use IWindow::windowULong, because it's protected.)
     hwndDefault = IQUERYWINDOWULONG( hwndFrame, QWL_DEFBUTTON );
#endif
#ifdef IC_WIN
     unsigned long result = hwndFrame.sendEvent( DM_GETDEFID );
     if (result)
        {
        hwndDefault = IWINDOWFROMID(hwndFrame, LOWORD(result) );
        }
#endif
  }

  return hwndDefault;
}

/*------------------------------------------------------------------------------
| ICanvas::defaultPushButton                                                   |
|                                                                              |
| Return the first child window that is a default push button, if one exists.  |
| Notes: This information is used by the canvas handler to provide dialog-like |
|          behavior.  The behavior in this case is selection of the default    |
|          push button when the Enter key is pressed.  As a result, this       |
|          function should return the push button currently hilighted as the   |
|          default, rather than the push button returned by the                |
|          origDefaultButtonHandle funtion.                                    |
|        Performance is a consideration for this routine when used on a window |
|          with lots of nested canvases, each with lots of child windows, and  |
|          input focus on one that does not have the default push button       |
|          (since the branch with the input focus will get called each time    |
|          processing of the Enter key is passed one level up the owner chain).|
------------------------------------------------------------------------------*/
IWindowHandle ICanvas::defaultPushButton ( ) const
{
  IWindowHandle hwndDefault = this->origDefaultButtonHandle();

  if ( !(hwndDefault  &&
         (hwndDefault.sendEvent( WM_QUERYDLGCODE, 0, 0 ).asUnsignedLong() &
                                                              DLGC_DEFAULT)))
  {                                  // No default, or not current default.
     hwndDefault = 0;
     IWindow::ChildCursor cursor( *(IWindow*)this );

     for ( cursor.setToFirst();
           !hwndDefault  &&  cursor.isValid();
           cursor.setToNext() )
     {                               // Check all child windows.
        IWindowHandle hwndChild = this->childAt( cursor );
        if ( hwndChild.sendEvent( WM_QUERYDLGCODE, 0, 0 ).asUnsignedLong() &
             DLGC_DEFAULT )
        {                            // A default push button found.
           hwndDefault = hwndChild;  // Return this child window.
        }
     }                               // End for all child windows.
  }

  if ( hwndDefault == 0 )
  {                         // No default push buttons in this canvas.
     IWindow::ChildCursor cursor( *(IWindow*)this );

     for ( cursor.setToFirst();
           !hwndDefault  &&  cursor.isValid();
           cursor.setToNext() )
     {                               // Check child canvases next.
        IWindow *pwndChild = windowWithHandle( this->childAt( cursor ));
        if ( pwndChild )
        {               // Child canvas has a default push button?
           hwndDefault = pwndChild->defaultPushButton();
        }
     }                               // End for all child windows.
  }

  return hwndDefault;
}

/*------------------------------------------------------------------------------
| ICanvas::matchForMnemonic                                                    |
|                                                                              |
| Returns the first child window that uses the specified character as a        |
| mnemonic.                                                                    |
------------------------------------------------------------------------------*/
IWindowHandle ICanvas::matchForMnemonic ( unsigned short character ) const
{
  IWindowHandle hwndMnemonic( 0 );     // Assume no mnemonic match.

  Boolean bDone = false;               // Flag for exiting loop.
  IWindow::ChildCursor cursor( *(IWindow*)this );

  for ( cursor.setToFirst();
        bDone == false  &&  cursor.isValid();
        cursor.setToNext() )
  {                                    // Check all child windows.
     IWindowHandle hwndChild = this->childAt( cursor );

#ifndef   IC_WIN
   // IC_NOTYET ... may have to do WM_MATCHMNEMONIC ourselves.
     if (
#ifdef IC_PM
        ( (ISTYLEOF(hwndChild) & WS_DISABLED) == 0) &&
#endif
        ( hwndChild.sendEvent( WM_MATCHMNEMONIC, character, 0 ) == true ) )
     {                                 // A match for the mnemonic found.
        hwndMnemonic = hwndChild;      // Return this child window.
        bDone = true;                  // Can exit loop now.
     }
#endif   //IC_WIN
  }                                    // End for all child windows.

  if ( hwndMnemonic == 0 )
  {                          // No match among children of the canvas.
     for ( cursor.setToFirst();
           bDone == false  &&  cursor.isValid();
           cursor.setToNext() )
     {                                 // Check child canvases next.
        IWindow *pwndChild = windowWithHandle( this->childAt( cursor ));

        if ( pwndChild  &&
             (hwndMnemonic = pwndChild->matchForMnemonic( character )))
        {                    // Found a canvas with a mnemonic match.
           bDone = true;               // Can exit loop now.
        } /* endif */
     }                                 // End for all child windows.
  }

  return hwndMnemonic;
}

/*------------------------------------------------------------------------------
| ICanvas::calcMinimumSize                                                     |
|                                                                              |
| Calculate the minimum screen size needed by the canvas.  This is the size    |
| needed to display all child windows (as determined by the layout routine).   |
------------------------------------------------------------------------------*/
ISize ICanvas::calcMinimumSize ( ) const
{
  ((ICanvas*)this)->layout();        // Calculate minimum size.
  ISize sizMin = layoutSize();       // Use size from layout().
  return sizMin;
}

/*------------------------------------------------------------------------------
| ICanvas::setLayoutDistorted                                                  |
|                                                                              |
| Provide common canvas processing for various distortion notifications.       |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setLayoutDistorted( unsigned long layoutAttributeOn,
                                      unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = layoutAttributeOn;

  if ( layoutAttributeOn & IWindow::childWindowCreated )
  {                                    // New child window.
     ulFlagsOn &= (unsigned long)(~IWindow::childWindowCreated);
     ulFlagsOn |= IWindow::layoutChanged;
  }                                    // Need new layout.
  if ( layoutAttributeOn & IWindow::childWindowDestroyed )
  {                                    // Lost a child window.
     ulFlagsOn &= (unsigned long)(~IWindow::childWindowDestroyed);
     ulFlagsOn |= IWindow::layoutChanged;
  }                                    // Need new layout.

  Inherited::setLayoutDistorted( ulFlagsOn, layoutAttributeOff );

  if ( layoutAttributeOn  &&
       this->isLayoutDistorted( IWindow::immediateUpdate )  &&
       this->isVisible() )
  {                                    // Need to update canvas appearance.
     Inherited::setLayoutDistorted( 0, IWindow::immediateUpdate );
     if ( this->size() > ISize() )
     {                                 // Truly visible window.
        this->refresh();               // Don't bypass paint handlers.
     }
     else                              // Height or width is 0.
     {                                 // Will never get a paint.
        this->layout();                // Circumvent WM_PAINT.
     }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::layout                                                              |
|                                                                              |
| This method should be overridden by derived classes to position/size child   |
| windows.                                                                     |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::layout ( )
{
  IMODTRACE_DEVELOP( "ICanvas::layout()" );

  if ( this->isLayoutDistorted( IWindow::layoutChanged ))
  {
     ITRACE_DEVELOP( "isLayoutDistorted(layoutChanged)==true" );
     /**************************************************************/
     /* The layout is distorted so see if this canvas needs        */
     /* updating.                                                  */
     /*                                                            */
     /* The user is responsible for positioning child windows and  */
     /* sizing the canvas large enough so that all child windows   */
     /* are contained by the canvas.                               */
     /*                                                            */
     /* Some of this responsibility will be easied if the canvas   */
     /* is a child of a specialized canvas (one that sizes and     */
     /* positions children according to minimumSize()).            */
     /* minimumSize() calls ICanvas::calcMinimumSize() if the user */
     /* has not specified a minimum size himself.  This function   */
     /* will return the size stored by the layout() routine, one   */
     /* large enough to contain all the child windows.             */
     /**************************************************************/
     ISize sizLayout = this->layoutSize(); // Save original size for children.

     IWindowPosBuffer wposbuf = this->fixupChildren();
     wposbuf.apply( false );           // Reorder z-order of children.

     this->setLayoutDistorted( 0, IWindow::layoutChanged );
     if ( sizLayout != this->layoutSize() )  // New minimum size for children.
     {                                 // Cause layout in parent canvas.
        this->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
     }
  }   /* end isDistorted */

  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::fixupChildren                                                       |
|                                                                              |
| Generates a list of all child windows, determines whether any can be tabbed  |
| to, and allows for reversing the order in which they are iterated.           |
------------------------------------------------------------------------------*/
IWindowPosBuffer ICanvas::fixupChildren ( )
{
  IMODTRACE_DEVELOP( "ICanvas::fixupChildren()" );
  /*****************************************************************/
  /* Dynamically set the WS_TABSTOP style based on whether the     */
  /* canvas has any tab-able children (WS_TABSTOP is needed so     */
  /* PM's frame window procedure and the ICanvasHandler will       */
  /* consider giving the input focus to the canvas during Tab key  */
  /* processing, and so the canvas can receive WM_SETFOCUS and     */
  /* give the input focus to an appropriate child window).         */
  /*                                                               */
  /* Additionally, reverse the z-order of the canvas' child        */
  /* windows, since the order that windows are constructed and     */
  /* expected to be layed out is typically the opposite of the     */
  /* z-order.  Because of complications involving tabbing and      */
  /* cursor movement with autoselect radio buttons, it became far  */
  /* easier to reverse the z-order of the child windows than to    */
  /* compensate in the tab and cursor movement code to fix this.   */
  /*****************************************************************/
  IWindowPosBuffer wposbuf( this );
  unsigned long childCount = 0;
  Boolean bTabControl = false;         // Assume no tab-able children.
  IPoint ptMax;
  IWindowHandle hwndChild = 0;
  IWindowHandle hwndLastChild = 0;
  Boolean
    defaultPBFound = ( this->origDefaultButtonHandle() == 0 ) ?
                          false : true;

  IWindow::ChildCursor cursor( *(IWindow*)this );
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {                                    // Enumerate all child controls.
     IPoint ptChild;
     Boolean bNewChild = false;
     childCount++;
     hwndChild = this->childAt( cursor );
     IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );
     if ( ! pwndChild )
     {                                 // A non-wrappered window.
        pwndChild = new IWindow( hwndChild );
        bNewChild = true;
     }

     ptChild = pwndChild->rect().topRight();
     if ( bTabControl == false )
     {                                 // No tab-able children found yet.
        if ( pwndChild->isTabStop() )
        {                         // Child window that can be tabbed to.
           bTabControl = true;         // Flag as such.
        }
        /*******************************************************/
        /* Note: if child is a canvas, isTabStop() will allow  */
        /* it to check all its children for a tab-able control)*/
        /*******************************************************/
     }
     if ( bNewChild )
     {                                 // Memory clean-up needed.
        delete pwndChild;
     }
     ptMax = ptMax.maximum( ptChild );

     if ( bClChildrenReversed == false )
     {                              // Need to reverse z-order of children.
        if ( hwndLastChild )
        {                           // Have a preceding child.
           wposbuf.orderAfter( hwndLastChild, hwndChild );
        }                           // Push preceding towards bottom.
        hwndLastChild = hwndChild;
     }                              // End reverse children.

     if ( ! defaultPBFound )
     {             // Frame doesn't have a default push button yet.
        if ( hwndChild.sendEvent( WM_QUERYDLGCODE, 0, 0 ).asUnsignedLong() &
             DLGC_DEFAULT)
        {                           // Found a default push button.
#ifdef IC_PM
           // Save it away with the frame.
           IEventResult mr =        // Find the frame window.
                      hwndChild.sendEvent( WM_QUERYFOCUSCHAIN,
                                           IEventParameter1( QFC_FRAME ),
                                           IEventParameter2( 0 ));
           IWindowHandle hwndFrame( mr.asUnsignedLong() );
           if ( hwndFrame )
           {                        // Store the default button.
              ISETWINDOWULONG( hwndFrame, QWL_DEFBUTTON, hwndChild );
           }  // (Can't use IWindow::setWindowData, because it's protected.)
#else   // !IC_PM
           hwndClDefaultButton = hwndChild;  // Save it away.
#endif  // !IC_PM
        }                           // End found a default push button.
     }                              // End default push button search.
  }                                 // End enumerate all children.

  fCanvasData->fChildCount = childCount;
  if ( bClChildrenReversed == false  &&  hwndLastChild )
  {                                 // Need to reverse z-order of children.
     wposbuf.orderAsFirst( hwndLastChild );  // Re-order last one.
     bClChildrenReversed = true;
  }                                 // Make topmost child.
  this->enableTabStop( bTabControl );
  this->setLayoutSize( ISize( ptMax ));  // Store size used by layout.

  return wposbuf;
}
