/*******************************************************************************
* FILE NAME: iwindow2.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iwindow.hpp that are common to OS/2 and Windows.                        *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

/*
  Currently OS/2 only:
    dispatch             ==> needs change for WM_PRESPARAM
                             font change notification seems to be a bit
                             of a problem
    setStyle             ==> WinSetWindowULong for GWL_STYLE

*/

#define INCL_WINACCELERATORS
#define INCL_WINERRORS
#define INCL_WINWINDOWMGR
#define INCL_WINMESSAGEMGR
#define INCL_WININPUT
#define INCL_WINSYS
#define INCL_GPIPRIMITIVES
#define INCL_GPIBITMAPS
#define INCL_WINFRAMEMGR
#define INCL_WINPOINTERS
#define INCL_WINDIALOGS
extern "C" {
  #include <iwindefs.h>
}


#ifdef IC_PMWIN  // Ensure clean compile in AIX

#include <iwindow.hpp>
#include <iacceltb.hpp>
#include <icconst.h>
#include <icoordsy.hpp>
#include <icritsec.hpp>
#include <ievent.hpp>
#include <ievtpriv.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <iframe.hpp>
#include <ihandler.hpp>
#include <inotifev.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <ireslock.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>
#include <iwinlsts.hpp>
#include <iwinnhdr.hpp>
#include <iwinpriv.hpp>

#pragma info(none)
  #include <ikeyset.h>
  #include <iseq.h>
#pragma info(restore)

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IWINDOW2_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_PM
  #define ISETACCELTABLE(haccel,hwnd) \
      (WinSetAccelTable( IThread::current().anchorBlock(), haccel, hwnd ))
  #define IQUERYACCELTABLE(hwnd) \
      (WinQueryAccelTable( IThread::current().anchorBlock(), hwnd ))
  #define IQUERYUSERDATA(hwnd) \
        (WinQueryWindowULong( hwnd, QWL_USER ))
  #define ISETUSERDATA(hwnd,valu) \
        (WinSetWindowULong( hwnd, QWL_USER, (unsigned long)valu))
  #define IQUERYWINDOWPROCEDURE(hwnd) \
        (WinQueryWindowPtr( hwnd, QWP_PFNWP ))
#endif

// A pointer to this structure is placed into the user window word.
// The data IOC uses from the structure is the window member.  The cbSignature
// member contains a constant that the code checks to verify that the
// block is indeed this structure.
const unsigned long kUserWordSignature = 0x10c1da7a; //iocldata
typedef struct _IUserWindowWord {
   unsigned long  fcbSignature;
   IWindow*       fwindow;
//   unsigned long  fuserData;
} IUserWindowWord;

/***************************************************************/
/* Public window styles.                                       */
/***************************************************************/
#pragma data_seg(ICLStaticConst)
const IWindow::Style
  IWindow::noStyle      = 0,
  IWindow::visible      = WS_VISIBLE,           // 0x80000000
  IWindow::disabled     = WS_DISABLED,          // 0x40000000
  IWindow::clipChildren = WS_CLIPCHILDREN,      // 0x20000000
  IWindow::clipSiblings = WS_CLIPSIBLINGS,      // 0x10000000
  IWindow::clipToParent = WS_PARENTCLIP,        // 0x08000000
  IWindow::saveBits     = WS_SAVEBITS,          // 0x04000000
  IWindow::synchPaint   = WS_SYNCPAINT;         // 0x02000000
#pragma data_seg()

/*------------------------------------------------------------------------------
| Window cache data                                                            |
|                                                                              |
| This data is scoped to this module so that the IWindow destructor can clear  |
| the cache.                                                                   |
------------------------------------------------------------------------------*/
static IWindow* pwinSaved=0;
static IWindow* pwinOwnerSaved=0;
static
#ifdef IC_WIN
       void *
#else
       unsigned long
#endif
                     hwndSaved=0;
static unsigned long ulControlIdSaved=0;


/*------------------------------------------------------------------------------
|ITheDesktopWindow::ITheDesktopWindow                                          |
------------------------------------------------------------------------------*/
ITheDesktopWindow::ITheDesktopWindow()
     : hwnd(IQUERYDESKTOPWINDOW(IThread::current().anchorBlock(), 0))
{ }


/*------------------------------------------------------------------------------
| IWindow::desktopWindow                                                       |
|                                                                              |
| Return static ITheDesktopWindow pointer.                                     |
------------------------------------------------------------------------------*/
IWindow* IWindow::desktopWindow()
{
   if (IWindowData::desktopWindow == 0) {
      IWindowData::desktopWindow = new ITheDesktopWindow();
   } /* endif */
   return IWindowData::desktopWindow;
}

/*------------------------------------------------------------------------------
| IWindow::objectWindow                                                        |
|                                                                              |
| Return static ITheObjectWindow pointer.                                      |
------------------------------------------------------------------------------*/
IWindow* IWindow::objectWindow()
{
   if (IWindowData::objectWindow == 0) {
      IWindowData::objectWindow = new ITheObjectWindow();
   } /* endif */
   return IWindowData::objectWindow;
}


/*------------------------------------------------------------------------------
| IWindow::IWindow                                                             |
|                                                                              |
| Constructor to create an instance for template window.                       |
------------------------------------------------------------------------------*/
IWindow::IWindow ( unsigned long id,
                   IWindow* parentDialog )
  : pWindowData( new IWindowData( (IWinProc*)IDEFWINDOWPROC,
                                  IWindow::cacheMinimumSize )),
    hwnd( 0 )
{
  this->startHandlingEventsFor( id, parentDialog );
}

/*------------------------------------------------------------------------------
| IWindow::IWindow                                                             |
|                                                                              |
| Constructor to instantiate from an existing window handle.                   |
------------------------------------------------------------------------------*/
IWindow::IWindow ( const IWindowHandle& handle )
  : pWindowData( new IWindowData( (IWinProc*)IDEFWINDOWPROC,
                                  IWindow::cacheMinimumSize )),
    hwnd( 0 )
{
  // Assertions on input parameters.
  IASSERTPARM( handle!=0 );
  this->startHandlingEventsFor( handle );
}

/*------------------------------------------------------------------------------
| IWindow::IWindow                                                             |
------------------------------------------------------------------------------*/
IWindow::IWindow ( )
  : pWindowData( new IWindowData( (IWinProc*)IDEFWINDOWPROC,
                                  IWindow::autoDestroyWindow
                                    | IWindow::cacheMinimumSize )),
    hwnd( 0 )
{
  IMODTRACE_ALL( "Win::ctor" );

  #ifdef IC_DEVELOP
    IWindowList::incrementWindowAddedCount();
  #endif
}

/*------------------------------------------------------------------------------
| IWindow::~IWindow                                                            |
------------------------------------------------------------------------------*/
IWindow::~IWindow()
{
   IMODTRACE_ALL("Win::dtor");
//   ITRACE_ALL(asDebugInfo());

   notifyObservers(INotificationEvent(IWindow::deleteId, *this));

  #ifdef IC_DEVELOP
    IWindowList::incrementWindowDeletedCount();
  #endif

  // If the parent of this window is a frame window, attempt
  // to remove this window from the parents frame extension
  // list.  If this window is not of frame extension,
  // the call to removeExtension will have no affect.
  if ( this->isValid() && this->isAutoDestroyWindow() )
  {
    IFrameWindow* fParent((IFrameWindow*)(this->parent()));
    if ( fParent && fParent->isFrameWindow() )
    {
      fParent -> removeExtension( this, false );
    }
  }

  // We are going to destroy the PM window, so get rid of any
  // handlers
  if (pWindowData->handlerList != 0) {
     IResourceLock windowLock(IWindowList::libraryKey());
     pWindowData->handlerList->removeAll();
     delete pWindowData->handlerList;
     pWindowData->handlerList = 0;
     pWindowData->fHandlerRemovePending = false;
  } /* endif */

  if ( this->isValid() )
  {  // window not destroyed yet

     // Canvas can adjust itself for the destroyed IWindow
     // We want to do this only if this->hwnd is still valid.
     // For ISubmenu, IFileDialog, and IFontDialog objects
     // this->hwnd may be invalid since these don't do startHandlingEventsFor
     setLayoutDistorted(windowDestroyed, 0);  // Flag for canvases

     // Destroy the window handle if autoDestroy except if
     // a delete is in process which would indicate that
     // we are already processing a WM_DESTROY
     ITRACE_ALL("Valid window");
     if (isAutoDestroyWindow()) {
       if (!deleteIsInProcess())
       {
          setDeleteInProcess();
          ITRACE_ALL(IString("WinDestroyWin ")+IString(this->hwnd).d2x());
          IDESTROYWINDOW(this->hwnd);
       }
     }
     else
     {
       // Reset the window procedure to its original value.
       // If we don't do this and this same HWND is wrappered again
       // we would set pWindowData->defaultProc to our own ICLUI window procedure
       // and cause infinite loops.
       if ( pWindowData->defaultProc != 0  &&
            pWindowData->defaultProc != _pfnwpICWinProc )
       {
          ITRACE_ALL( "Resetting winproc" );
#ifdef IC_WIN
          long
#else
          PFNWP
#endif
             pfnwpDefault = ISUBCLASSWINDOW( this->hwnd,
                                             pWindowData->defaultProc );
          if ( pfnwpDefault == 0 )
          {
             ITHROWGUIERROR( "WinSubclassWindow" );
          }
          pWindowData->defaultProc = 0;
       }
     } /* endif */

     this->cleanUp();
  }
  else
  {
    // Cleanup is normally handled in WM_DESTROY but we need to check if
    // the GUI has been terminated and handle that case as well
    if ( !IThread::current().isGUIInitialized() )
    {
       // The GUI is not active so we need to be sure and clean up.
       this->cleanUp();
    }
  }

  delete pWindowData->_observerData;
  delete pWindowData;

  // Clear the window handle cache used by the subclass window proc in case
  // this window is stored in the cache (we do not want to get a cache hit
  // for an IWindow object that has been deleted)
  {
    ICritSec critSec;
    pwinSaved=0;
    pwinOwnerSaved=0;
    hwndSaved=0;
    ulControlIdSaved=0;
  }

}

/*------------------------------------------------------------------------------
| IWindow::dispatch                                                            |
|                                                                              |
| Dispatch the event. First to the handler in the list,                        |
| then to handleWindow event and finally to defaultProcedure.                  |
------------------------------------------------------------------------------*/
Boolean IWindow::dispatch(IEvent& evt)
{
   IMODTRACE_ALL("Win::dispatch");

   /*---------------------------------------------------------*/
   /* We need to keep track of the top level entry to this    */
   /* function so that we can be sure we have left all nested */
   /* calls prior to deleted the object on WM_DESTROY.        */
   /*---------------------------------------------------------*/
   Boolean fNested = (pWindowData->state & dispatchInProcess);
   pWindowData->state |= dispatchInProcess;

   /********************************************************/
   /* Dispatch event to Handler.                           */
   /********************************************************/
   Boolean bStop = false;
   if (pWindowData->handlerList != 0 &&
       pWindowData->handlerList->numberOfElements()!=0)
      {
      IHandlerList::Cursor cursor(*pWindowData->handlerList);

      ITRACE_ALL(IString("Win=")+IString((unsigned long)this).d2x()+
                 IString(" HWND=")+IString(this->hwnd).d2x());
      ITRACE_ALL(IString("HWND=") + IString(evt.handle().asUnsigned()).d2x()+
                 IString(", Msg=") + IString(evt.eventId()).d2x()+
                 IString(", MP1=") + IString((unsigned long)evt.parameter1()).d2x()+
                 IString(", MP2=") + IString((unsigned long)evt.parameter2()).d2x());
      ITRACE_ALL(IString("Handler count=")+
                 IString(pWindowData->handlerList->numberOfElements()));

      //*************************************************************
      // Store the handler list in the event.  This is done so that
      // if dispatchRemainingHandlers is called, it will have access
      // to the handler list in which to iterate.
      //*************************************************************
      evt.pData->setHandlerList(pWindowData->handlerList);

      for (cursor.setToFirst();
           bStop == false && cursor.isValid();
           cursor.setToNext())
         {
         IHandlerListObject& listObj(pWindowData->handlerList->elementAt(cursor));
         IHandler* phandler = listObj.fHandler;
         if ( listObj.fRemovePending )
         {
           // The handler is being removed, so do not dispatch to it
           phandler = 0L;
         }
         if (phandler)
            {
            /*--------------------------------------------------*/
            /* Note: we have added a bit of trickery here since */
            /* folks like to delete handler but leave them in   */
            /* the list. We need to have the handler destructor */
            /* remove them from the list but in the interim we  */
            /* will rely on the value of the is Enabled field to*/
            /* try detect the majority of these cases.  The     */
            /* handler destructor sets this field to a large    */
            /* positive value (as opposed to the Boolean true(1)*/
            /* for enabled or the Boolean false (0) for disabled*/
            /*--------------------------------------------------*/
            if (phandler->isEnabled()==true)
               {
               ITRACE_ALL(IString("Handler is: ")+
                         IString((unsigned long)phandler).d2x());

               /********************************************************/
               /* Store the current cursor postion in the event.  This */
               /* is stored for use by dispatchRemainingHandlers() so  */
               /* it will know where to start from.                    */
               /********************************************************/
               evt.pData->setHandlerCursor(&cursor);

               // This is where we try to catch exceptions at the
               // top of the dispatch chain.
               try
                  {
                  bStop = phandler->dispatchHandlerEvent(evt);
                  }
               catch (IException& exc)
                  {
                  if (!handleException(exc, evt))
                     IRETHROW(exc);
                  }
               }  // if handler enabled
            else if (phandler->isEnabled())
               {  // possibly invalid handler
               IInvalidRequest exc(IBase::messageText(
                             IC_DISPATCHTO_INVALID_HANDLER),
                             IC_DISPATCHTO_INVALID_HANDLER,
                             recoverable);
               if (!handleException(exc, evt))
                    ITHROW(exc);
               }
            else
               ITRACE_ALL("Handler is disabled");
            }  // endif phandler
         }  // end for
      } // endif

/////// DAB - Need to add code here when we design common color scheme handling
///////       between PM and Windows NT  IC_NOTYET
#ifdef IC_PM
   //---------------------------------------------------------
   // If color or font change, notify the window through the
   // setLayoutDistorted function.
   //---------------------------------------------------------
   if (evt.eventId() == WM_PRESPARAMCHANGED)
      {
      if(evt.parameter1() == PP_FONTNAMESIZE ||
         evt.parameter1() == PP_FONTHANDLE)
         setLayoutDistorted(IWindow::fontChanged, 0);
      else if(evt.parameter1() != PP_USER && evt.parameter1() != PP_RESERVED )
         setLayoutDistorted(IWindow::colorChanged, 0);
      }
   else if ( evt.eventId() == WM_SYSCOLORCHANGE )
      {               // System-wide scheme change.
      setLayoutDistorted( IWindow::colorChanged, 0 );
      }
#endif

////// RDL - We might want to move the following code so
//////       it occurs after default procedure.
   if ((evt.eventId() != WM_MOUSEMOVE) && (pWindowData->_observerData))
      {
      if (isEnabledForNotification())
         {
         pWindowData->_observerData->notifyHandler->dispatchHandlerEvent(evt);
         }
      }

   //---------------------------------------------------------
   // If destroy is seen we want to:
   //  1) call cleanUp to remove handlers and such
   //  2) set the "needsDelete" state if isAutoDeleteObject
   //  3) call the default window procedure so that children
   //     get deleted as well.
   //---------------------------------------------------------
   if (evt.eventId() == WM_DESTROY)
      {
      IWinProc* pwinProc = pWindowData->defaultProc;
      IMODTRACE_ALL("Win::dispatch(WM_DESTROY)");
      ITRACE_ALL(IString("Handle: ")+IString(this->hwnd).d2x());

      // Remove the ICLUI window procedure, because it seems there
      // is a slight delay between receiving this message and when
      // the GUI window is actually destroyed.  During the wait, we
      // don't want any handlers getting called, since they would
      // probably trap because event.window() will return 0.
      if ( pwinProc != 0  &&  pwinProc != _pfnwpICWinProc )
         {
         ITRACE_ALL( "Resetting winproc" );
#ifdef IC_WIN
         long
#else
         PFNWP
#endif
            pfnwpDefault = ISUBCLASSWINDOW( this->hwnd, pwinProc );
         if ( pfnwpDefault == 0 )
            {
            ITHROWGUIERROR( "WinSubclassWindow" );
            }
         pWindowData->defaultProc = 0;
         }  // endif reset winproc

      // Remove ourselves from the list.
      this->cleanUp();

      // Delete this pointer if client set autoDeleteObject.
      if (!deleteIsInProcess() )
         {
         if (isAutoDeleteObject() )
            {
            setDeleteInProcess();
            pWindowData->state |= needsDelete;
            }  // if autoDelete
         }  // if not delete in process
      void* result = (void*)pwinProc(evt.handle(), evt.eventId(),
                            evt.parameter1(), evt.parameter2());
      evt.setResult(result);
      // return true;
      bStop = true;
      } // endif WM_DESTROY

   //---------------------------------------------------------
   // Remove any handlers that are "pending" because they
   // could not safely be removed during message dispatch
   //---------------------------------------------------------
   if (!fNested && pWindowData->fHandlerRemovePending)
   {
      IResourceLock windowLock(IWindowList::libraryKey());
      IHandlerList::Cursor cursor(*pWindowData->handlerList);
      cursor.setToFirst();
      // Loop until we manage to read the end of the list
      while (cursor.isValid())
      {
        if ( pWindowData->handlerList->elementAt(cursor).fRemovePending )
        {
          // Ok, we want to remove this handler
          pWindowData->handlerList->removeAt(cursor);
          // Since our cursor is now in limbo, reset it to the beginning
          // of the collection and start searching for more handlers to
          // be removed
          cursor.setToFirst();
        }
        else
        {
          // We do not want to remove this handler, so just move to
          // the next handler in the list
          cursor.setToNext();
        }
      }
      // All "pending" handlers should now be removed
      pWindowData->fHandlerRemovePending = false;
   }

   //---------------------------------------------------------
   // If we have rolled back up through any nested calling
   // and object deletion is necessary because a WM_DESTROY
   // was seen with AutoDelete=true, delete the object.
   //---------------------------------------------------------
   if (!fNested)
      {
      pWindowData->state &= (unsigned long)~dispatchInProcess;
      if (pWindowData->state & needsDelete)
         {
         ITRACE_ALL(IString("Delete this: ")+IString(this->hwnd).d2x());
         pWindowData->state &= (unsigned long)~needsDelete;
         this->setAutoDeleteObject( false );
         delete this;
         }  // if needs delete
      }  // if not nested
   return bStop;
}

/*------------------------------------------------------------------------------
| IWindow::dispatchRemainingHandlers                                           |
|                                                                              |
| Dispatch the event to all of the remaining handlers in the list that do not  |
| process the event(ie. return false).                                         |
|                                                                              |
| NOTE: This function assumes that the event passed to it is either the same   |
|       event passed to the dispatch() function, or a derived event constructed|
|       from the event passed to the dispatch() function.                      |
------------------------------------------------------------------------------*/
Boolean IWindow::dispatchRemainingHandlers(IEvent& evt, Boolean callDefProc)
{
  IMODTRACE_ALL("Win::dispatchRemainingHandlers");

  /********************************************************/
  /* Dispatch event to the remaining handlers.            */
  /********************************************************/
  Boolean bStop = false;
  IHandlerList* remainingHandlerList = evt.pData->handlerList();
  IHandlerList::Cursor* pCursor = evt.pData->handlerCursor();

  if (remainingHandlerList &&
      remainingHandlerList->numberOfElements() && pCursor)
  {
    ITRACE_ALL(IString("Win=")+IString((unsigned long)this).d2x()+
               IString(" HWND=")+IString(this->hwnd).d2x());
    ITRACE_ALL(IString("HWND=") + IString(evt.handle().asUnsigned()).d2x()+
               IString(", Msg=") + IString(evt.eventId()).d2x()+
               IString(", MP1=") + IString((unsigned long)evt.parameter1()).d2x()+
               IString(", MP2=") + IString((unsigned long)evt.parameter2()).d2x());
    ITRACE_ALL(IString("Handler count=")+
               IString(remainingHandlerList->numberOfElements()));

    IHandlerList::Cursor cursor = *pCursor;

    /*******************************************************************/
    /* Since the current cursor is stored, we need to proceed with the */
    /* next one.  Bump the cursor to the next one and dispatch from    */
    /* there.                                                          */
    /*******************************************************************/
    for(cursor.setToNext();
        bStop == false && cursor.isValid();
        cursor.setToNext())
    {
      IHandlerListObject& listObj(remainingHandlerList->elementAt(cursor));
      IHandler* phandler = listObj.fHandler;
      if ( listObj.fRemovePending )
      {
        // The handler is being removed, so do not dispatch to it
        phandler = 0L;
      }
      if (phandler)
      {
        /*--------------------------------------------------*/
        /* Note: we have added a bit of trickery here since */
        /* folks like to delete handler but leave them in   */
        /* the list. We need to have the handler destructor */
        /* remove them from the list but in the interim we  */
        /* will rely on the value of the is Enabled field to*/
        /* try detect the majority of these cases.  The     */
        /* handler destructor sets this field to a large    */
        /* positive value (as opposed to the Boolean true(1)*/
        /* for enabled or the Boolean false (0) for disabled*/
        /*--------------------------------------------------*/
        if (phandler->isEnabled()==true)
        {
          ITRACE_ALL(IString("Handler is: ")+
                    IString((unsigned long)phandler).d2x());

          /********************************************************/
          /* Store the current cursor postion in the event.  This */
          /* is stored for use by dispatchRemainingHandlers() so  */
          /* it will know where to start from if it is called     */
          /* again.                                               */
          /********************************************************/
          evt.pData->setHandlerCursor(&cursor);

          /***************************************************/
          /* This is where we try to catch exceptions at the */
          /* top of the dispatch chain.                      */
          /***************************************************/
          try
          {
            bStop = phandler->dispatchHandlerEvent(evt);
          }
          catch (IException& exc)
          {
            if (!handleException(exc, evt))
            {
              IRETHROW(exc);
            }
          }
        }
        else if (phandler->isEnabled())
        {
          IInvalidRequest exc(IBase::messageText(
                                     IC_DISPATCHTO_INVALID_HANDLER),
                              IC_DISPATCHTO_INVALID_HANDLER,
                              recoverable);

          if (!handleException(exc, evt))
          {
            ITHROW(exc);
          }
        }
        else
        {
          ITRACE_ALL("Handler is disabled");
        }
      } /* endif phandler */
    } /* end for */
  } /* endif */

  /******************************************************************/
  /* If no handler returned true and we have been asked to call the */
  /* default procedure before exiting, do so.                       */
  /******************************************************************/
  if (!bStop && callDefProc)
  {
#ifdef IC_WIN
    /*--------------------------------------------------------------*/
    /* Convert control message back to command before default proc. */
    /*--------------------------------------------------------------*/
    if (evt.eventId() == WM_CONTROL)
    {
      evt = IEvent( evt.handle(), WM_COMMAND, evt.parameter1(),
                    evt.parameter2() );
    }
#endif
    /***************************************************************/
    /* Call defaultProcedure().  NOTE: We cannot just simply call  */
    /* this->defaultProcedure().  This event could have been       */
    /* generated via a WM_CONTROL, WM_CONTROLPOINTER, WM_DRAWITEM, */
    /* or WM_MEASUREITEM message and we need to call               */
    /* defaultProcedure() on the owner of a control, not the       */
    /* control itself.                                             */
    /***************************************************************/
    evt.window()->defaultProcedure(evt);
    bStop = true;
  }

  return bStop;
}

/*------------------------------------------------------------------------------
| _pfnwpICWinProc                                                              |
|                                                                              |
| Common window procedure used in ICLUI.                                       |
|                                                                              |
| The purpose of this routine is to identify the IWindow                       |
| dispatcher to be used to process a message. For certain                      |
| messages (WM_CONTROL, WM_DRAWITEM, WM_MEASUREITEM), an                       |
| attempt is made to identify the control that generated the                   |
| message sent to it's owner and route the message back to                     |
| the control. If the control can't be found or chooses not                    |
| to process the message, it is then routed to the owner (the                  |
| normal PM behavior).                                                         |
|                                                                              |
| This function is also attempting to optimize some of the                     |
| window look up by caching the last recipient of a message                    |
| so that it doesn't need to be rediscovered. The success                      |
| of this strategy has yet to be measured.                                     |
------------------------------------------------------------------------------*/
#ifdef IC_WIN
void* __stdcall _pfnwpICWinProc(
                              void*
#else
void* _System _pfnwpICWinProc(
                              unsigned long
#endif
                                        hwnd,
                              unsigned long ulMsg,
                              void* mp1, void* mp2)
{
   IMODTRACE_ALL("_pfnwpICWinProc");

#ifdef IC_WIN
   //-----------------------------------------------------------------
   // Convert notification command messages in Windows to control
   //  before event construction
   //-----------------------------------------------------------------
   if (ulMsg == WM_COMMAND && (unsigned long)(mp2) != 0)
      {
      ulMsg = WM_CONTROL;
      }
#endif
   IEvent evt(IWindowHandle(hwnd),
              ulMsg,
              IEventParameter1(mp1),
              IEventParameter2(mp2));
   IWindow* pwin = 0;
   Boolean fCacheHit=false;

   #ifdef IC_DEVELOP
      IWindowList::incrementEventCount();
   #endif

   ITRACE_ALL(IString("WinP            HWND=") + IString((unsigned long)hwnd).d2x()+
              IString(", Msg=") + IString(ulMsg).d2x()+
              IString(", MP1=") + IString((unsigned long)mp1).d2x()+
              IString(", MP2=") + IString((unsigned long)mp2).d2x());

   //---------------------------------------------------------
   // Route control notification to the control first.
   // If no action taken or not a window, message is
   // routed to the owner. Note that in NT/Windows control
   // notification messages are combined with WM_COMMAND
   // messages. In our code, we redefine WM_CONTROL to be
   // WM_COMMAND.  To ensure that we only route the control
   // notification messages on, we check the control hwnd
   // stored in lParam; if not 0, then its a notification
   // message
   //---------------------------------------------------------
   if (ulMsg == WM_DRAWITEM || ulMsg == WM_MEASUREITEM ||
       ulMsg == WM_CONTROL  || ulMsg == WM_CONTROLPOINTER )
      {
      unsigned long controlID = evt.parameter1().number1();
#ifdef IC_WIN
      // WM_SETCURSOR had handle of control, not id.
      if (ulMsg == WM_CONTROLPOINTER)
         controlID = IIDOF( evt.parameter1() );
#endif
      IWindow* pwinOwner = 0;
         // Lets see if we can get the window from static cache.
         // We can only recover from the cache for this hwnd if
         // it is redispatched to the same control as before.
         {
         ICritSec critSec;
         if (hwnd==hwndSaved &&
             ulControlIdSaved &&
             ulControlIdSaved==controlID )
            {
            pwin=pwinSaved;
            pwinOwner=pwinOwnerSaved;
            fCacheHit=true;
            #ifdef IC_DEVELOP
               IWindowList::incrementCacheCount();
            #endif
            }  // endif
         }  // ICritSec scope

      //--------------------------------------------------------
      // If we didn't get a cache hit, we need to find the
      // control so we can dispatch to it first.
      //--------------------------------------------------------
      if (!fCacheHit)
         {  // The HWND was not in the cache
         ITRACE_ALL("Not in cache");
         HWND hwndCtl = IWindow::handleWithParent( controlID,
                                                   hwnd );
         pwinOwner = IWindow::windowWithHandle(hwnd,false);
         if (hwndCtl)
            {
            pwin = IWindow::windowWithHandle(hwndCtl,false);
            }
         else
            {
            if(pwinOwner)
               pwin = IWindow::windowWithOwner(controlID,
                                            pwinOwner,false);
            }
         } /* endif */
      #ifdef IC_DEVELOP
      else
         {
         ITRACE_ALL( IString( "restored from cache: pwin= " ) +
                     IString( (unsigned long)pwin ).d2x() );
         }
      #endif
      //---------------------------------------------------------------
      // Save the window in the cache if not already gotten from there
      //---------------------------------------------------------------
      if (!fCacheHit && controlID)
         {      // Store the window in the cache
            {     // Scope the ICritSec to not include IString ctor.
            ICritSec critSec;
            if(pwin)
              hwndSaved=hwnd;
            else
              hwndSaved=0;
            pwinSaved=pwin;
            pwinOwnerSaved=pwinOwner;
            ulControlIdSaved=controlID;
            }
         ITRACE_ALL(IString("saved to cache: pwin= ")+
                    IString((unsigned long)pwin).d2x());
         }

      //---------------------------------------------------------------
      // If we found a control, dispatch the message to it. If the
      // control does not process the message or we couldn't find
      // a control, then try to route it to the owner.
      //---------------------------------------------------------------
      if (pwin)
        evt.setDispatchingHandle(pwin->handle());
      if (pwin != 0 && (pwin->dispatch(evt) ) )
         {
         return evt.result();
         }
      else
         {
         pwin = pwinOwner;
         if (pwin)
            evt.setDispatchingHandle(pwin->handle());
         if ( (pwinOwner!=0) && (pwinOwner->dispatch(evt) ) )
            return evt.result();    // event handled ... return
         }
      } // endif routing to control first
   else
      {
      //---------------------------------------------------------
      // Route event to the window.
      //---------------------------------------------------------
      // Lets see if we can get the window from static cache.
      // We can only recover from the cache for this hwnd if
      // the cache hwnd was Not a redispatch to the control.
      //---------------------------------------------------------
         {  // scope critical section to check for cache hit
         ICritSec critSec;
         if (hwnd==hwndSaved && ulControlIdSaved==0)
            {
            pwin=pwinSaved;
            fCacheHit=true;
            #ifdef IC_DEVELOP
               IWindowList::incrementCacheCount();
            #endif
            }
         }  // end of critical section
      if ( !fCacheHit )
         {
         pwin = IWindow::windowWithHandle(hwnd,false);
            {  // scope of critical section to set cache
            ICritSec critSec;
            if (pwin)
               hwndSaved=hwnd;
            else
               hwndSaved=0;
            pwinSaved=pwin;
            ulControlIdSaved=0;
            }  // end of critical section
         }
      #ifdef IC_DEVELOP
      else
         {
         ITRACE_ALL( IString( "restored from cache: pwin= " ) +
                     IString( (unsigned long)pwin ).d2x() );
         }
      #endif

      // Call the dispatcher with the message
      if (pwin != 0 && (pwin->dispatch(evt) ) )
         {
         // event handled ... return
         return evt.result();
         } // endif
      } // else not routing to control first

   ITRACE_ALL(IString("WinP pWindow=") + IString((unsigned long)pwin).d2x());

#ifdef IC_WIN
   /*-----------------------------------------------------------------*/
   /* Convert control message back to command before default proc.    */
   /*-----------------------------------------------------------------*/
   if (ulMsg == WM_CONTROL)
      evt = IEvent( evt.handle(),
                    WM_COMMAND,
                    evt.parameter1(),
                    evt.parameter2() );
#endif
   // route to the default winproc
   if (pwin != 0)
      {
      pwin->defaultProcedure(evt);

#ifdef IC_WIN
      // We implement WM_QUERYFOCUSCHAIN partially as needed by the library
      // We handle it after the default proc since the frame window handler
      // handles the message.
      if (ulMsg == WM_QUERYFOCUSCHAIN)
         {
         if (!evt.result().asUnsignedLong())
            {
            unsigned long fsCmd = LOWORD( (unsigned long)mp1 );
            switch (fsCmd)
               {
               case QFC_NEXTINCHAIN  :
                   return IEventResult( hwnd );
               case QFC_PARTOFCHAIN  :
                   return IEventResult( (hwnd == (HWND)mp2) );
//          Handled in frame handler
//             case QFC_FRAME :
//          These are not supported and not used by the Class Library
//               case QFC_ACTIVE       :
//               case QFC_SELECTACTIVE :
               default:
                  break;
               }
            IWindowHandle hwndParent = IPARENTOF(hwnd);
            if (hwndParent)
               return hwndParent.sendEvent( ulMsg, mp1, mp2 );
            }  // switch
         }  // end WM_QUERYFOCUSCHAIN

      // In Windows, we need to route certain events to the owner if they
      // are not processed by the default procedure.
      if ( ( (ulMsg >= WM_KEYFIRST) && (ulMsg <= WM_KEYLAST) ) ||
           ( (ulMsg >= WM_MOUSEFIRST) && (ulMsg <= WM_MOUSELAST) &&
             (ulMsg != WM_MOUSEMOVE)  ) )
         {
         ITRACE_DEVELOP(
            IString("Not processed  HWND=") + IString((unsigned long)hwnd).d2x()+
            IString(", Msg=") + IString(ulMsg).d2x()+
            IString(", MP1=") + IString((unsigned long)mp1).d2x()+
            IString(", MP2=") + IString((unsigned long)mp2).d2x()+
            IString(" Result=") + IString(evt.result().asUnsignedLong() ) );
         //
         // Dispatch the event to the owner until one of the
         // following becomes true:
         //   1) The owner window is 0 (not found)
         //   2) A handler returns true
         //   3) The owner is the desktop window
         IWindow* pwinOwner = pwin->owner();
         if (pwinOwner == IWindow::desktopWindow())
            pwinOwner = 0;
         while ( (pwinOwner != 0) && (!evt.result().asUnsignedLong()) )
            {
            ITRACE_DEVELOP(
               IString("To Owner    HWND=") +
               IString(pwinOwner->handle().asUnsigned()).d2x()+
               IString(", Msg=") + IString(ulMsg).d2x()+
               IString(", MP1=") + IString((unsigned long)mp1).d2x()+
               IString(", MP2=") + IString((unsigned long)mp2).d2x()+
               IString(" Result=") + IString(evt.result().asUnsignedLong() ) );

            // dispatch to owner window
            evt.setDispatchingHandle(pwinOwner->handle());
            evt.setHandle(pwinOwner->handle());
            if ( pwinOwner->dispatch(evt) )
               {
               return evt.result();       // Handled, get out
               }

            // still not handled ... on to next owner
            pwinOwner = pwinOwner->owner();
            if (pwinOwner == IWindow::desktopWindow())
               pwinOwner = 0;
            }  // while owner and evt.result() == 0
         }  // case
#endif
      // return the result
      ITRACE_ALL(IString("Not processed ... returning ") +
                 IString( evt.result().asUnsignedLong() ).d2x()  );
      return evt.result();
      }
   else
      {
      /* No window found so call the default window procedure */
#ifdef IC_WIN
      return (void*)(IDEFWINDOWPROC(hwnd, (unsigned)ulMsg, (unsigned)mp1, (long)mp2));
#else
      return IDEFWINDOWPROC(hwnd, ulMsg, mp1, mp2);
#endif
      } // else no window found

} /* end win proc */

/*------------------------------------------------------------------------------
| IWindow::convertToGUIStyle                                                   |
|                                                                              |
| Returns base/extended style for the underlying GUI.                          |
------------------------------------------------------------------------------*/
unsigned long IWindow::convertToGUIStyle(const IBitFlag& guiStyle,
                                         Boolean bExtOnly) const
{
  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    return( (extendedStyle() & IS_EXTMASK) );
  }

  // WS_ styles have a one-to-one correspondence to our style bits, and
  // inhabit the upper word of the base GUI style.  Therefore, return
  // that portion asis, masking out the lower word.
  return( (guiStyle.asUnsignedLong() & IWS_MASK) );
}

/*------------------------------------------------------------------------------
| IWindow::isValid                                                             |
|                                                                              |
| Does the IWindow still represent a valid presentation system window?         |
------------------------------------------------------------------------------*/
Boolean IWindow::isValid ( ) const
{
#ifdef IC_PM
  return IQUERYANCHOR( this->hwnd );
#endif
#ifdef IC_WIN
  return IISWINDOW( 1, this->hwnd );
#endif
#ifndef IC_PMWIN
  return IISWINDOW( IThread::current().anchorBlock(), this->hwnd );
#endif
}

/*------------------------------------------------------------------------------
| IWindow::isFrameWindow                                                       |
|                                                                              |
| Is window a frame window.                                                    |
------------------------------------------------------------------------------*/
Boolean IWindow::isFrameWindow() const
{
#ifdef IC_WIN
// IC_NOTYET - Simple logic for now is if not a child or popup window,
// assume it is a top level window (i.e. Frame Window)
   if (!(style() & (WS_CHILD | WS_POPUP)))
#else
   IWindowClassName className( handle() );
   if (className == WC_FRAME)
#endif
      return true;

   return false;
}

/*------------------------------------------------------------------------------
| IWindow::id                                                                  |
|                                                                              |
| Get the object identity.                                                     |
------------------------------------------------------------------------------*/
unsigned long IWindow::id() const
{
  unsigned long ulId = IIDOF( this->handle() );

  // no exception required b'cos handle() check for valid
  // window handle and QWS_ID is defined in toolkit.

  return ulId;
}

/*------------------------------------------------------------------------------
| IWindow::setId                                                                  |
|                                                                              |
| Set the object identifier                                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::setId( unsigned long ulId )
{
  ISETIDOF( this->handle(), ulId );

  // no exception required b'cos handle() check for valid
  // window handle and QWS_ID is defined in toolkit.

  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::presSpace                                                           |
|                                                                              |
| Grab the window cache presentation space.                                    |
------------------------------------------------------------------------------*/
IPresSpaceHandle IWindow::presSpace() const
{
  // no exception required b'cos handle() check for valid
  // window handle

  IMODTRACE_ALL("Win::presSpace");
  // Note: PS should be released for performance.

#ifdef IC_WIN
  HDC hps =
#else
  HPS hps =
#endif
            IGETPS(handle());

  if (!hps)
     ITHROWGUIERROR("WinGetPS");

  if ( IBidiSettings::isBidiSupported() )
  {
    IPresSpaceHandle ps(hps);
    IBidiSettings bidi(*this);
    bidi.apply(ps);
    return ps;
  }
  else
    return IPresSpaceHandle(hps);
}

/*------------------------------------------------------------------------------
| IWindow::releasePresSpace                                                    |
|                                                                              |
| Release the window cache presentation space.                                 |
------------------------------------------------------------------------------*/
void IWindow::releasePresSpace(const IPresSpaceHandle& hps) const
{
   IMODTRACE_ALL("Win::releasePresSpace");

   // Need to verify that handle() call is NOT made in OS/2 (since not needed)
   if (!IRELEASEPS(handle(),hps) ) {
      ITHROWLIBRARYERROR(IC_RELEASEPSFAIL,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */
}



/*------------------------------------------------------------------------------
| IWindow::show                                                                |
|                                                                              |
| Make the window visible.                                                     |
------------------------------------------------------------------------------*/
IWindow& IWindow::show(Boolean bShow)
{
#ifdef IC_WIN
   // IC_NOTYET ... may not be needed
//   if ( bShow )
//     {
//     // Call moveSizeTo for recoordination
//     moveSizeTo( rect() );
//     }
#endif
   ISHOWWINDOW(handle(),bShow);
   return *this;
   // no exception required because handle() check for valid
   // window handle
}

/*------------------------------------------------------------------------------
| IWindow::enableUpdate                                                        |
|                                                                              |
| Cause further updates to the window to be painted to the display.            |
------------------------------------------------------------------------------*/
IWindow& IWindow::enableUpdate(Boolean enableWindow)
{
   if(enableWindow)
      IENABLEWINDOWUPDATE(handle(), true);
   else
      IENABLEWINDOWUPDATE(handle(), false);

   // No exception, handle() checks validity of handle.

   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::isVisible                                                           |
------------------------------------------------------------------------------*/
Boolean IWindow::isVisible() const
{
   return IISWINDOWVISIBLE(handle());

   // no exception required because handle() check for valid
   // window handle
}


/*------------------------------------------------------------------------------
| IWindow::setFocus                                                            |
|                                                                              |
| Make the window the focus window.                                            |
------------------------------------------------------------------------------*/
IWindow& IWindow::setFocus()
{
   ISETFOCUS(HWND_DESKTOP, handle());
   return *this;
   // no exception required because handle() check for valid
   // window handle
}

/*------------------------------------------------------------------------------
| IWindow::hasFocus                                                            |
|                                                                              |
| Is the window the focus window?                                              |
------------------------------------------------------------------------------*/
Boolean IWindow::hasFocus() const
{
   HWND hwndFocus = IQUERYFOCUS(HWND_DESKTOP);
   return (hwndFocus == handle());

   // no exception required because HWND_DESKTOP is default
   // parameter.
}

/*------------------------------------------------------------------------------
| IWindow::enable                                                              |
|                                                                              |
| Enable the object to accept input.                                           |
------------------------------------------------------------------------------*/
IWindow& IWindow::enable(Boolean bEnable)
{
   //--------------------------------------------------------
   // DEFECT 3717 :
   // check if the window to be disabled currently has focus
   //--------------------------------------------------------
   Boolean bResetFocus(false);
   if ( !bEnable && hasFocus() )
        bResetFocus = true;
   //--------------------------------------------------------
   
   IENABLEWINDOW(handle(),bEnable);
   // no exception required b'cos handle() check for valid
   // window handle
   
   //--------------------------------------------------------
   // DEFECT 3717 :
   // must attempt to locate a sibling that can have focus
   //--------------------------------------------------------
   if ( bResetFocus ) {
      if ( owner() && owner()->isValid() ) {
         owner()->setFocus();
      }
      else if ( parent() && parent()->isValid() ) {
         parent()->setFocus();
      }
   }
   
   return *this;
}

/*-----------------------------------------------------------------------------|
| IWindow::isEnabled                                                           |
|                                                                              |
| Is the object enabled?                                                       |
------------------------------------------------------------------------------*/
Boolean IWindow::isEnabled() const
{
   return IISWINDOWENABLED(handle());
   // no exception required because handle() checks for valid
   // window handle
}

/*------------------------------------------------------------------------------
| IWindow::refresh                                                             |
|                                                                              |
| Refresh the window.  If RefreshType calls for an immediate update, call      |
| IUPDATEWINDOW to force the window to paint synchronously (i.e. painting is   |
| completed before exiting this function and returning to the caller.          |
| If type calls for painting the whole window (instead of just the invalidated |
| region), then call IINVALIDATERECT to add the whole window to the            |
| area to be updated on the next paint.                                        |
------------------------------------------------------------------------------*/
IWindow& IWindow::refresh( RefreshType type )
{
  IMODTRACE_ALL("Win::refresh");

  if (( type == IWindow::paintAll ) || ( type == IWindow::paintAllImmediate ))
  {
    IINVALIDATERECT( handle(), 0, true );
    ITRACE_ALL("Invalidate the whole window");
  }

  if (( type == IWindow::immediate ) || ( type == IWindow::paintAllImmediate ))
  {
    IUPDATEWINDOW(handle());
    ITRACE_ALL("Paint immediately");
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::refresh                                                             |
|                                                                              |
| Redraw a specific window rect. If immediate update is true, call             |
| IUPDATEWINDOW to force the window to paint synchronously...ie painting is    |
| completed before exiting this function and returning to the caller.          |
------------------------------------------------------------------------------*/
IWindow& IWindow::refresh(const IRectangle& rectArea, Boolean immediate)
{
  IMODTRACE_ALL("Win::refresh");

  IRectangle nativeRect = ICoordinateSystem::convertToNative(
                             rectArea,
                             size() );
#ifdef IC_PM
  RECTL rectl;
  rectl.xLeft   = nativeRect.minX();
  rectl.yBottom = nativeRect.minY();
  rectl.xRight  = nativeRect.maxX();
  rectl.yTop    = nativeRect.maxY();
#else
  RECT rectl;
  rectl.left   = nativeRect.minX();
  rectl.top    = nativeRect.minY();
  rectl.right  = nativeRect.maxX();
  rectl.bottom = nativeRect.maxY();
#endif

  IINVALIDATERECT(handle(), &rectl, true);
  if (immediate)
  {
    IUPDATEWINDOW(handle());
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setParent                                                           |
|                                                                              |
| Change the window's parent                                                   |
------------------------------------------------------------------------------*/
IWindow& IWindow::setParent(const IWindow* newParent)
{
   IASSERTPARM(newParent != 0);
   ISETPARENT(handle(), newParent->handle(), TRUE);

#ifdef IC_PM
   HWND hwndOwner = IOWNEROF(handle());
#endif
#ifdef IC_WIN
   IWindow*   winOwner = owner();
   HWND hwndOwner = IWindowHandle(0);
   if (winOwner)
      hwndOwner = winOwner->handle();
#endif
   if (newParent->handle() == desktopWindow()->handle() &&
        (hwndOwner == 0 || hwndOwner == desktopWindow()->handle())) {
       pWindowData->state |= primaryWindow;
    }
    else
       pWindowData->state &= (unsigned long)~primaryWindow;
   return *this;

   // no exception required b'cos handle() check for valid
   // window handle
}

/*------------------------------------------------------------------------------
| IWindow::parent                                                              |
|                                                                              |
| Return the window's parent.                                                  |
------------------------------------------------------------------------------*/
IWindow* IWindow::parent() const
{
   IMODTRACE_ALL("Win::parent");
   HWND hwndParent = IPARENTOF(handle());
   // no exception required b'cos handle() check for valid
   // window handle

   ITRACE_ALL(IString("hwndParent = ") +
                  IString((unsigned long)hwndParent));
   ITRACE_ALL(IString("desktopWindow handle = ") +
                  IString(desktopWindow()->handle().asUnsigned()));
   ITRACE_ALL(IString("objectWindow handle = ") +
                  IString(objectWindow()->handle().asUnsigned()));

   if (hwndParent != 0) {
      if (hwndParent == desktopWindow()->handle())
         return desktopWindow();
      else if (hwndParent == objectWindow()->handle())
         return objectWindow();
       else
         return IWindow::windowWithHandle(hwndParent);
      } /* endif */

   return 0;
}


/*------------------------------------------------------------------------------
| IWindow::owner                                                               |
|                                                                              |
| Return the window's parent. Null if no IBMCLASS parent is                    |
| assigned.                                                                    |
------------------------------------------------------------------------------*/
IWindow* IWindow::owner() const
{
   IMODTRACE_ALL("Win::owner");
#ifdef IC_PM
   HWND hwndOwner = IOWNEROF(handle());
#endif
#ifdef IC_WIN
   HWND hwndOwner = pWindowData->fhwndOwner;
   if ( ( ISTYLEOF( handle() ) & WS_CHILD ) == 0 )
      {
      hwndOwner = IOWNEROF( handle() );
      }
#endif
   // no exception required b'cos handle() check for valid
   // window handle

   ITRACE_ALL(IString("hwndOwner = ") +
                  IString((unsigned long)hwndOwner));
   ITRACE_ALL(IString("desktopWindow handle = ") +
                  IString(desktopWindow()->handle().asUnsigned()));
   ITRACE_ALL(IString("objectWindow handle = ") +
                  IString(objectWindow()->handle().asUnsigned()));

#ifndef IC_WIN
   // HWND_DESKTOP is 0 in Windows
   if (hwndOwner != 0) {
#endif
      if (hwndOwner == desktopWindow()->handle())
         return desktopWindow();
      else if (hwndOwner == objectWindow()->handle())
         return objectWindow();
      else
         return IWindow::windowWithHandle(hwndOwner);
#ifndef IC_WIN
   } /* endif */

   return 0;
#endif
}

/*------------------------------------------------------------------------------
| IWindow::style                                                               |
|                                                                              |
| Return unsigned long style flag.                                             |
------------------------------------------------------------------------------*/
unsigned long IWindow::style() const
{
   return (ISTYLEOF(handle()));
}

/*------------------------------------------------------------------------------
| IWindow::extendedStyle                                                       |
|                                                                              |
| Return unsigned long extended GUI style flag.                                |
------------------------------------------------------------------------------*/
unsigned long IWindow::extendedStyle() const
{
  return ( this->pWindowData->ulExtGUIStyle );
}

/*------------------------------------------------------------------------------
| IWindow::setStyle                                                            |
------------------------------------------------------------------------------*/
IWindow& IWindow::setStyle(unsigned long flStyle)
{
   ISETWINDOWSTYLE(handle(), flStyle) ;
    // not exception throw because handle() is validated.
   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setExtendedStyle                                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::setExtendedStyle(unsigned long flExtStyle)
{
   this->pWindowData->ulExtGUIStyle = flExtStyle;
   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::startHandlingEventsFor                                              |
|                                                                              |
| This function subclasses the PM window procedure to                          |
| begin dispatch events.                                                       |
------------------------------------------------------------------------------*/
IWindow& IWindow::startHandlingEventsFor(const IWindowHandle& windowHandle)
{
   IMODTRACE_ALL("Win::startHandlingEventsFor");
   ITRACE_ALL(IString("Handle: ")+IString(windowHandle.asUnsigned()).d2x());

   // Ensure this handle nonzero.  The addToWindowSet function detects
   // the case where the handle is already in the window list collection.
   IASSERTPARM(windowHandle!=0);

   IWindow::addToWindowSet( this, windowHandle );

   // See if we have a primary window.
   HWND hwndParent = IPARENTOF(windowHandle);
   HWND hwndOwner = IOWNEROF(windowHandle);
#ifdef IC_WIN
   if ((hwndParent == 0 || hwndParent == desktopWindow()->handle()) &&
        (hwndOwner == 0 || hwndOwner == desktopWindow()->handle()) )
#else
   if (hwndParent == desktopWindow()->handle() &&
       (hwndOwner == 0 || hwndOwner == desktopWindow()->handle()) )
#endif
   {
      pWindowData->state |= primaryWindow;
   }

   // Ensure that the window's distorted flag is set.
   setLayoutDistorted(windowCreated | colorChanged | fontChanged, 0);

   // Subclass the window.
#ifdef IC_WIN
   void * pfnwpDefault = (void*)   // Cast the long returned as a void* for comparisons.
#else
   PFNWP pfnwpDefault =
#endif
                     ISUBCLASSWINDOW(windowHandle,
                                      _pfnwpICWinProc);

   // Ensure we don't get ourself back - looping will result.
   IASSERTSTATE(pfnwpDefault && pfnwpDefault!=_pfnwpICWinProc);

#ifdef IC_WIN
   pWindowData->defaultProc = (IWinProc*) pfnwpDefault;
#else
   pWindowData->defaultProc = pfnwpDefault;
#endif

   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::addToWindowSet                                                      |
|                                                                              |
| Add a window to a threads collection of windows.                             |
------------------------------------------------------------------------------*/
void IWindow::addToWindowSet ( IWindow* window,
                               const IWindowHandle& windowHandle )
{
   IASSERTPARM(window!=0 && windowHandle.isValid());

   // Save the window handle
   window->hwnd = windowHandle.asUnsigned();

   IHandle::Value
     threadId,
     processId;
   IQUERYWINDOWPROCESS( windowHandle, &processId, &threadId );
   IWindowList* pwinlist = IWindowList::listForThread( threadId );
   {
     IResourceLock windowLock(IWindowList::libraryKey());
     // Throw an exception if the window already is in the collection.
     // This indicates an attempt to create 2 IWindow objects for the same
     // windowHandle.  The locateOrAddElementWithKey function returns false
     // if the new element was added, true if it already existed.
     IASSERTSTATE(pwinlist->locateOrAddElementWithKey(window) == false);
     window->pWindowData->state |= addedToList;
   }
   if ( IWindow::isFastWindowWithHandleEnabled() )
      {
      // In PM, The File and font dialog use the word for "convenience" to
      // store a dialog specific structure.  This is inconvenient for this
      // design, but is not too much of a problem because these dialogs do not
      // use the IOC subclass procedure. Help instances also use the user word
      // to store a structure.
      // Therefore, we do not overwrite a non-0 value already in the user
      // window word.  We use the fcbSignature member of the user data
      // structure to detect the presence of the IUserWindowWord structure.
      unsigned long oldUserData = IQUERYUSERDATA( windowHandle );
      if ( oldUserData == 0 )
         {
         IUserWindowWord* userData = new IUserWindowWord;
         userData->fcbSignature = kUserWordSignature;
         userData->fwindow      = window;
         ISETUSERDATA( windowHandle, userData);
         }
      }
}

/*------------------------------------------------------------------------------
| IWindow::removeFromWindowSet                                                 |
|                                                                              |
| Remove a window from a threads collection of windows.                        |
------------------------------------------------------------------------------*/
void IWindow::removeFromWindowSet ( IWindow* window )
{
   IASSERTPARM(window!=0);

   if ( window->pWindowData->state & addedToList )
   {               // In a window list.
      IWindowList* pwinlist;
      if ( window->isValid() )
      {            // Window handle is valid.
         IHandle::Value
           threadId,
           processId;
         IQUERYWINDOWPROCESS( window->handle(), &processId, &threadId );
         pwinlist = IWindowList::listForThread( threadId );
         if ( IWindow::isFastWindowWithHandleEnabled() )
            {
            // Remove pointer to user data structure from window word and
            // delete the user data structure we allocated in addToWindowSet.
            IUserWindowWord* userData = (IUserWindowWord*)
                IQUERYUSERDATA( window->handle() );
            if (userData && (userData->fcbSignature == kUserWordSignature))
               {
               ISETUSERDATA( window->handle(), 0);
               delete userData;
               }
            }
      }
      else         // GUI window already destroyed.
      {            // Clean-up should have already been done.
         pwinlist = IWindowList::current();
      }
      {
        IResourceLock windowLock( IWindowList::libraryKey() );
        pwinlist->removeElementWithKey( window->hwnd );
      }
      window->pWindowData->state &= (unsigned long)~addedToList;
   }
}


/*------------------------------------------------------------------------------
| IWindow::create                                                              |
|                                                                              |
| Call the presentation system to create the window.                           |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::create( unsigned long        id,
                               const char*          text,
                               unsigned long        style,
                               const char*          windowClass,
                               const IWindowHandle& parent,
                               const IWindowHandle& owner,
                               const IRectangle&    initRect,
                               const void*          ctlData,
                               const void*          presParams,
                               IWindow::SiblingOrder order )
{
  HWND behind = HWND_TOP;
  if (order == IWindow::behindSiblings)
    behind = HWND_BOTTOM;

  ITRACE_ALL( IString("create initRect=") + initRect.asString() );
  unsigned long width = initRect.width();
  unsigned long height = initRect.height();

  // need to obtain the parent size inline here since we have't created
  // the window yet to use the ICoordinateSystem functions.
#ifdef IC_WIN
  RECT parentRect;
  if (!parent)
     {
     GetClientRect( IWindow::desktopWindow()->handle(),
                    &parentRect );
     }
  else
     {
     GetClientRect(parent, &parentRect);
     }
  ISize parentSize = ISize(parentRect.right - parentRect.left,
                           parentRect.bottom - parentRect.top);
#endif
#ifdef   IC_PM
   // IC_NOTYET ... does this work correctly for frames ??
  RECTL parentRect;
  if (!parent)
     {
     WinQueryWindowRect( IWindow::desktopWindow()->handle(),
                         &parentRect );
     }
  else
     {
     WinQueryWindowRect(parent, &parentRect);
     }
  ISize parentSize = ISize(parentRect.xRight - parentRect.xLeft,
                           parentRect.yTop - parentRect.yBottom);
#endif   //IC_PM

  IPoint   position =
     ICoordinateSystem::convertToNative( initRect,
                                         parentSize).minXMinY();
#ifdef IC_WIN
  // check for any pass thru special flags
  if ((initRect.minX() == CW_USEDEFAULT) &&
      (initRect.minY() == CW_USEDEFAULT) &&
      (initRect.width() == (0x7FFFFFFF)) &&
      (initRect.height() == (0x7FFFFFFF)))
     {
     width =  (unsigned long)CW_USEDEFAULT;
     height = (unsigned long)CW_USEDEFAULT;
     }

  // store the owner information for PM msg routing simulation.  Child windows
  // cant be owned in base Windows
  IWindowHandle realParent = parent;
  if ( style & WS_CHILD )
     {
     // owner is simulated.
     pWindowData->fhwndOwner = owner;
     }
  else
     {
     // owner is real.  The ICREATEWINDOW macro ignores the owner parameter
     // but Windows interprets parent as owner if WS_CHILD is not set.
     // Note also that if the owner window has WS_CHILD set,  Windows will
     // assign ownership to the top level parent of the child, since a
     // child cant be an owner.
     pWindowData->fhwndOwner = 0;
     realParent = owner;
     }


  HINSTANCE    hinst=0;
  if (presParams)
  {
     hinst= (HINSTANCE)presParams;
  }
  ITRACE_ALL( IString("ICREATEWINDOW style=") + IString(style).d2x() +
              IString(" parent=") + IString(realParent.asUnsigned()).d2x() +
              IString(" position=") + position.asString() +
              IString(" size=") + ISize(width, height).asString() +
              IString(" id=") + IString(id) );
  IWindowHandle
    hwnd = ICREATEWINDOW(
               realParent,
               windowClass,
               text,
               style,
               position.x(),
               position.y(),
               width,
               height,
               owner,
               behind,
               id,
               (void*)ctlData,
               NULL,
               hinst );
#endif   //IC_WIN
#ifdef IC_PM
  IWindowHandle
    hwnd = ICREATEWINDOW(
               parent,
               windowClass,
               text,
               style,
               position.x(),
               position.y(),
               width,
               height,
               owner,
               behind,
               id,
               (void*)ctlData,
               (void*)presParams,
               0 );
#endif

  if ( hwnd == 0) { // Error: throw exception.
    ITHROWGUIERROR(IString("WinCreateWindow: Id=")+IString(id)+
           IString(" Class=")+IString((unsigned short)(unsigned long)windowClass));
  }

  return hwnd;

}

/*------------------------------------------------------------------------------
| IWindow::addHandler                                                          |
|                                                                              |
| Add a Handler to the top of the list.                                        |
------------------------------------------------------------------------------*/
IWindow& IWindow::addHandler(IHandler* pshNew)
{
   IMODTRACE_DEVELOP("Win::addHandler");
   IASSERTPARM(pshNew != 0);
   ITRACE_DEVELOP(IString("Win is: ")+IString((unsigned long)this).d2x() +
              IString(" Handler is: ")+IString((unsigned long)pshNew).d2x());

   IResourceLock windowLock(IWindowList::libraryKey());
   if (pWindowData->handlerList == 0)
      pWindowData->handlerList = new IHandlerList();

   Boolean
     found = false;

   // If handler is handling another window...
   if ( pshNew->numWindows || pWindowData->fHandlerRemovePending )
     {
     IHandlerList::Cursor cursor( *pWindowData->handlerList );
     // Examine all handlers for this window...
     for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
       {
       // If one of them is this handler, don't add it again...
       if ( pshNew == pWindowData->handlerList->elementAt(cursor).fHandler )
         {
         found = true;
         if ( pWindowData->handlerList->elementAt(cursor).fRemovePending )
           {
           pWindowData->handlerList->elementAt(cursor).fRemovePending = false;
           pshNew->numWindows++;
           }
         break;
         }
       }
     }

   if ( !found )
     {
     pWindowData->handlerList->addAsFirst(pshNew);
     pshNew->numWindows++;

     #ifdef IC_DEVELOP
     pWindowData->handlerList->totalHandlers++;
     unsigned long ulCount=pWindowData->handlerList->numberOfElements();
     if(ulCount>pWindowData->handlerList->maxHandlers)
        pWindowData->handlerList->maxHandlers = ulCount;
     #endif
     }

   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::removeHandler                                                       |
------------------------------------------------------------------------------*/
IWindow& IWindow::removeHandler(IHandler* pshOld)
{
  IMODTRACE_DEVELOP("Win::removeHandler");

  ITRACE_DEVELOP(IString("Win is: ")+IString((unsigned long)this).d2x() +
             IString(" Handler is: ")+IString((unsigned long)pshOld).d2x());

   if (pWindowData->handlerList != 0) {
      IResourceLock windowLock(IWindowList::libraryKey());
      IHandlerList::Cursor cursor(*pWindowData->handlerList);
      unsigned long index = 1;
      for(cursor.setToFirst();
          cursor.isValid() && index <= pWindowData->handlerList->numberOfElements();
          cursor.setToNext(), index++) {
             if (pWindowData->handlerList->elementAt(cursor).fHandler == pshOld)
             {
                // If we are currently dispatching messages to this handler
                // then we cannot safely delete it, so set remove pending flag
                // to handler this case and we will clean it up later
                if ( pWindowData->state & dispatchInProcess )
                {
                  pWindowData->handlerList->elementAt(cursor).fRemovePending = true;
                  pWindowData->fHandlerRemovePending = true;
                }
                else
                {
                  // We are not dispatching, so remove the handler from the list
                  pWindowData->handlerList->removeAt(cursor);
                }
                pshOld->numWindows--;
                break;
             } /* endif */
      }/* end for */
   } /* endif */
   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::windowWithHandle                                                    |
|                                                                              |
| Search the internal collection for the IWindow with the passed handle.       |
------------------------------------------------------------------------------*/
IWindow* IWindow::windowWithHandle ( const IWindowHandle& wndh,
                                     Boolean allThreads )
{
   // Attempt to retrieve the IWindow pointer from the user data window
   // word of the control.  Because we use the user data as a pointer,
   // we also test for the window procedure being the IOC subclass procedure.
   // This reduces the risk of using an invalid pointer.  Some IOC
   // window classes, such as the file and font dialogs, do not use the
   // IOC subclass procedure.  In these cases we fall back to the collection
   // search to find the IWindow pointer.  We also validate that the
   // pointer points to our structure, again in an attempt to validate
   // that the user data is the pointer we placed there.

   // Test is the same as if ( IWindow::isFastWindowWithHandleEnabled() )
   // but we do the test directly here to avoid function call overhead.
   if (IWindowData::staticFlags & IWindowData::fastWindowWithHandle)
      {
      #ifdef IC_DEVELOP
         static unsigned long hitCount = 0;
      #endif
      void* winProc = IQUERYWINDOWPROCEDURE( wndh );
      if (winProc == _pfnwpICWinProc)
         {
         IUserWindowWord* userData = (IUserWindowWord*) IQUERYUSERDATA( wndh );
         if (userData && (userData->fcbSignature == kUserWordSignature))
            {
            #ifdef IC_DEVELOP
            hitCount++;
            #endif
            return userData->fwindow;
            }
         }
   #ifdef IC_DEVELOP
      else
         {
         ITRACE_DEVELOP(
           IString("IWindow::windowWithHandle - not IOC window procedure, hwnd=")+
           wndh.asString().d2x() );
         }

      ITRACE_DEVELOP(
          IString("IWindow::windowWithHandle - not in window word, hwnd=") +
          wndh.asString().d2x() +
          IString(" hitCount=") + IString(hitCount) );
      hitCount = 0;
   #endif //IC_DEVELOP
      }

   // If the use of the window word is disabled or we could not find the
   // IWindow pointer there, search the window list collection for
   // a key matching the given handle.
   unsigned long ulHandle = wndh.asUnsigned();
   IWindowList* pwinlist = 0;

   if ( allThreads )
   {
      // An IWindow is always put into the window list of the thread that
      // created the underlying presentation system window.  It is faster
      // to get the thread from the presentation system than it is to
      // cursor through the list of threads (even if the list has only one
      // entry in it) and potentially the window lists of each of those
      // threads.  Not cursoring the list also avoids the problem of the
      // cursor becoming invalid by the addition/removal of a thread.
      IHandle::Value
        threadId,
        processId;
      if ( IQUERYWINDOWPROCESS( ulHandle, &processId, &threadId ) )
      {            // Valid window handle.
         pwinlist = IWindowList::listForThread( threadId );
      }
   }
   else
   {
      pwinlist = IWindowList::current();
   }

   if ( pwinlist  &&  pwinlist->numberOfElements() != 0 )
   {
      IWindowList::Cursor cursor( *pwinlist );
      if ( pwinlist->locateElementWithKey( ulHandle, cursor ) )
      {
         return pwinlist->elementAt( cursor );
      }
   }

   ITRACE_DEVELOP( "IWindow::windowWithHandle - hwnd " +
                   wndh.asString() + " not found" );
   return 0;
}

/*------------------------------------------------------------------------------
| IWindow::handleWithParent                                                    |
|                                                                              |
| Return the window handle with the passed id and parent.                      |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::handleWithParent( unsigned long ulId,
                                         const IWindowHandle& parent)
{
   IASSERTPARM(parent!=0);
   IWindowHandle hwnd = IWINDOWFROMID(parent, ulId);
   return hwnd;
}

/*------------------------------------------------------------------------------
| IWindow::handleWithPointerCaptured                                           |
|                                                                              |
| Return the window handle that currently has the mouse capture.               |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::handleWithPointerCaptured( )
{
   return IWindowHandle(IQUERYCAPTURE);
}

/*------------------------------------------------------------------------------
| IWindow::postEvent                                                           |
|                                                                              |
| Construct an IEvent from the arguments and post it to this window.           |
------------------------------------------------------------------------------*/
const IWindow& IWindow::postEvent ( EventType           eventType,
                                    const IEventParameter1 &parm1,
                                    const IEventParameter2 &parm2 ) const
{
  unsigned long eventId;
  switch (eventType) {
  case command:
     eventId = WM_COMMAND;
     break;
  case systemCommand:
     eventId = WM_SYSCOMMAND;
     break;
  case control:
     eventId = WM_CONTROL;
     break;
#ifdef IC_PM   // IC_NOTYET
  case help:
     eventId = WM_HELP;    // help is totally different in Windows
     break;
#endif
  case character:
     eventId = WM_CHAR;
     break;
  } /* endswitch */

  handle().postEvent(eventId, parm1, parm2);
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::sendEvent                                                           |
|                                                                              |
| Construct an IEvent from the arguments and send it to this window.  Return   |
| the event result.                                                            |
------------------------------------------------------------------------------*/
IEventResult IWindow::sendEvent ( EventType          eventType,
                                  const IEventParameter1 &parm1,
                                  const IEventParameter2 &parm2 ) const
{
  unsigned long eventId;
  switch (eventType) {
  case command:
     eventId = WM_COMMAND;
     break;
  case systemCommand:
     eventId = WM_SYSCOMMAND;
     break;
  case control:
     eventId = WM_CONTROL;
     break;
#ifdef IC_PM   // IC_NOTYET
  case help:
     eventId = WM_HELP;
     break;
#endif
  case character:
     eventId = WM_CHAR;
     break;
  } /* endswitch */

  return handle().sendEvent(eventId, parm1, parm2);
}

/*------------------------------------------------------------------------------
| IWindow::cleanUp                                                             |
|                                                                              |
| Remove the window from the window list.  If this is a primary window and     |
| there are no remaining primary windows in the window list, post a WM_QUIT    |
| to the message queue.                                                        |
------------------------------------------------------------------------------*/
IWindow& IWindow::cleanUp()
{
  IMODTRACE_ALL("Win::cleanUp");
//  ITRACE_ALL(asString());
  IWindow::removeFromWindowSet( this );

  // Find the window list for the window's thread.
  IHandle::Value threadId;
  if ( this->isValid() )
  {           // Window handle is valid.
     IHandle::Value processId;
     IQUERYWINDOWPROCESS( this->hwnd, &processId, &threadId );
  }
  else        // GUI window already destroyed.
  {           // Clean-up should have already been done.
     threadId = IThread::currentId();
  }
  IThread windowThread( threadId );
  IWindowList* pwinlist = IWindowList::listForThread( threadId );

  // See if the last primary window was closed.
  if ( windowThread.inMsgLoop()  &&  this->isPrimaryWindow() )
  {
     IWindowList::Cursor cursor( *pwinlist );
     Boolean fFound = false;
     unsigned long windowsRemaining;
     do
     {     // Repeat if not able to iterate through all windows
           // (cursor may have been invalidated by add/remove
           // done from another thread).
        windowsRemaining = pwinlist->numberOfElements();
        for ( cursor.setToFirst();
              cursor.isValid() && fFound==false;
              cursor.setToNext() )
        {
           IWindow* pwin = pwinlist->elementAt( cursor );
           if ( pwin->isPrimaryWindow() )
           {
              fFound = true;
           }
           windowsRemaining--;
        }
     } /* enddo */
     while ( !fFound  &&  windowsRemaining );

     if ( !fFound )
     {
        ITRACE_DEVELOP("No primary windows, posting quit");
#ifdef IC_PM
        IMessageQueueHandle::postEvent( windowThread.messageQueue(),
                                        WM_QUIT );
#endif
#ifdef IC_WIN
        // IC_NOTYET - This code is temporary until we implement
        // a portable version of postEvent for Threads - DAB
        PostQuitMessage(0);
//      PostThreadMessage( windowThread.id(), WM_QUIT,
//                         (WPARAM)0, (LPARAM)0 );
#endif

     } // end !found
  } // end isPrimaryWindow

  return *this;
} // end cleanUp

/*------------------------------------------------------------------------------
| IWindow::setLayoutDistorted                                                  |
|                                                                              |
| Virtual function used in canvas classes. If a window has been created,       |
| or its font has been changed, or its minimum size requirements have          |
| changed, notify its parent by calling setLayoutDistorted.                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::setLayoutDistorted( unsigned long layoutAttributeOn,
                                      unsigned long layoutAttributeOff )
{
  pWindowData->layoutChange &= ~layoutAttributeOff;  // Remove deleted flags
  pWindowData->layoutChange |= layoutAttributeOn;  // Add in new flags

  unsigned long ulParentFlags = 0;
  if (layoutAttributeOn & windowCreated)
  {                               // Notify parent window
     ulParentFlags |= childWindowCreated;
  }
  if (layoutAttributeOn & windowDestroyed)
  {                               // Notify parent window
     ulParentFlags |= childWindowDestroyed;
  }

  Boolean parentInitialized = false;
  IWindow* pwinParent = 0;
  if ((layoutAttributeOn & fontChanged)  &&
      !(layoutAttributeOn & windowCreated))
  {                     // Font change (allow parent to buffer these)
     pwinParent = this->parent();
     parentInitialized = true;
     if (pwinParent  &&
         pwinParent != desktopWindow()  &&
         pwinParent != objectWindow())
     {                                 // Someone to pass change to
        if (pwinParent == owner())
        {                    // WM_PRESPARAMCHANGED from owner window
           char achChildFont[16];      // Buffer size doesn't matter
           unsigned long ulChildFontLen =   // Has own font?
#ifdef IC_PM
              WinQueryPresParam(handle(), PP_FONTNAMESIZE, 0, 0,
                                sizeof(achChildFont), &achChildFont,
                                QPF_NOINHERIT);
#else  // IC_NOTYET
           0;  ////// DAB - Need to update this later, default for now
#endif
           if (ulChildFontLen == 0)    // No font presentation parameter
           {            // Font must have been changed further up chain
              ulParentFlags |= fontPropogated;
           }
        }
     }                                 // End have a good parent
     pWindowData->layoutChange &= (unsigned long)~fontPropogated;
                         // Done waiting for font change to bubble up
  }
  if (layoutAttributeOn & minimumSizeChanged)
  {                               // Notify parent canvas
     // If a minimum size has explicity been set, do not tell the
     // parent that a child minimum size changed
     if ( pWindowData->fSetMinSizeInProgress ||
       (( pWindowData->minimumSize.width()  == -1 ) &&
        ( pWindowData->minimumSize.height() == -1 )))
     {
        ulParentFlags |= childMinimumSizeChanged;
     }
  }

  /*****************************************************************/
  /* Send layout distortion notification to parent, if applicable. */
  /*****************************************************************/
  if (ulParentFlags)
  {                               // Notify parent now
     if ( ! parentInitialized )
     {             // Get parent window now if not gotten already.
        pwinParent = this->parent();
     }

     if (pwinParent  &&
         pwinParent != desktopWindow()  &&
         pwinParent != objectWindow())
     {                            // Someone to pass change to
        pwinParent->setLayoutDistorted(ulParentFlags, 0);
     }                            // End have a good parent
  }                               // End have notify flags for parent

  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::defaultPushButton                                                   |
|                                                                              |
| let canvas do the work                                                       |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::defaultPushButton ( ) const
{
   // IC_NOTYET  ... we should override this in IFrameWindow.  Need to
   // add virtual function to interface.  This function should just return
   // IWindowHandle(0) in IWindow.
   IWindowHandle hwndDefault = 0;
   if ( this->isFrameWindow() )
      {
      IWindow::ChildCursor cursor( *(IWindow*)this );

      for ( cursor.setToFirst();
            !hwndDefault  &&  cursor.isValid();
            cursor.setToNext() )
         {                               // Check all child windows.
         IWindowHandle hwndChild = this->childAt( cursor );
         if ( hwndChild.sendEvent( WM_QUERYDLGCODE, 0, 0 ).asUnsignedLong() &
              DLGC_DEFAULT )
            {                            // A default push button found.
            hwndDefault = hwndChild;  // Return this child window.
            }
         }                               // End for all child windows.
      }  // End if frame window
   return hwndDefault;
}

/*------------------------------------------------------------------------------
| IWindow::mapPoint                                                            |
|                                                                              |
| Remap points from one coordinate space to another.                           |
------------------------------------------------------------------------------*/
IPoint IWindow::mapPoint ( const IPoint        &aPoint,
                           const IWindowHandle &from,
                           const IWindowHandle &to )
{
   POINTL pt = aPoint.asPOINTL();
   if ( ICoordinateSystem::isConversionNeeded() )
      {
#ifdef   IC_WIN
      RECT rect;
      GetClientRect(from, &rect);
      ISize size(rect.right - rect.left, rect.bottom - rect.top );
#endif   //IC_WIN
#ifdef   IC_PM
      SWP swp;
      WinQueryWindowPos(from, &swp);
      ISize size(swp.cx,swp.cy);
#endif   //IC_PM
      pt = ICoordinateSystem::convertToNative( aPoint, size ).asPOINTL() ;
      }
   IMAPWINDOWPOINTS(from, to, &pt, 1);
   if ( ICoordinateSystem::isConversionNeeded() )
      {
#ifdef   IC_WIN
      RECT rect;
      GetClientRect(to, &rect);
      ISize size(rect.right - rect.left, rect.bottom - rect.top );
#endif   //IC_WIN
#ifdef   IC_PM
      SWP swp;
      WinQueryWindowPos(to, &swp);
      ISize size(swp.cx,swp.cy);
#endif   //IC_PM
      pt = ICoordinateSystem::convertToApplication(
                                 IPoint(pt), size ).asPOINTL();
      }
   return pt;
}


/*------------------------------------------------------------------------------
| IWindow::matchForMnemonic                                                    |
|                                                                              |
| let canvas do the work                                                       |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::matchForMnemonic ( unsigned short character ) const
{
  return 0;
}


/*------------------------------------------------------------------------------
| IWindow::windowUShort                                                        |
|                                                                              |
| Fetch the window ushort at given index.                                      |
------------------------------------------------------------------------------*/
unsigned short IWindow::windowUShort ( long index ) const
{
  unsigned short
    result = IQUERYWINDOWUSHORT( this->handle(), index );
  return result;
}


/*------------------------------------------------------------------------------
| IWindow::windowULong                                                         |
|                                                                              |
| Fetch the window ulong at given index.                                       |
------------------------------------------------------------------------------*/
unsigned long IWindow::windowULong ( long index ) const
{
  unsigned long
    result = IQUERYWINDOWULONG( this->handle(), index );
  return result;
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IWindow::messageQueue                                                        |
|                                                                              |
| Return the message queue handle extracted from QWL_HMQ window word.          |
------------------------------------------------------------------------------*/
IMessageQueueHandle IWindow::messageQueue ( ) const
{
  return IMessageQueueHandle( this->windowULong( QWL_HMQ ) );
}
#endif

/*------------------------------------------------------------------------------
| IWindow::setWindowData                                                       |
|                                                                              |
| Set the appropriate unsigned short window word.                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::setWindowData ( long index, unsigned short ushort )
{
  if ( !ISETWINDOWUSHORT( this->handle(), index, ushort ) )
  {
    // Error, throw exception...
    ITHROWGUIERROR("WinSetWindowUShort");
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setWindowData                                                       |
|                                                                              |
| Set the appropriate unsigned long window word.                               |
------------------------------------------------------------------------------*/
IWindow& IWindow::setWindowData ( long index, unsigned long ulong )
{
  if ( !ISETWINDOWULONG( this->handle(), index, ulong ) )
  {
    // Error, throw exception...
    ITHROWGUIERROR("WinSetWindowULong");
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setFont                                                             |
|                                                                              |
| Sets the font for the window.                                                |
------------------------------------------------------------------------------*/
IWindow& IWindow::setFont (const IFont& fm)
{
#ifndef IC_WIN
  fm.setWindowFont(this);
  pWindowData->sizClChar.setWidth(fm.avgCharWidth());
  pWindowData->sizClChar.setHeight(fm.maxCharHeight());
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::resetFont                                                           |
|                                                                              |
| Causes the window to use the default font.                                   |
------------------------------------------------------------------------------*/
IWindow& IWindow::resetFont ( )
{
#ifndef IC_WIN
  WinRemovePresParam( this->handle(), PP_FONTNAMESIZE );
      // Don't care if window has no such presentation parameter.

  // Update current character size information.
  IFont defaultFont = this->font();
  this->pWindowData->sizClChar.setWidth( defaultFont.avgCharWidth() );
  this->pWindowData->sizClChar.setHeight( defaultFont.maxCharHeight() );
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::font                                                                |
|                                                                              |
| Returns the font used for the window.                                        |
------------------------------------------------------------------------------*/
#ifndef IC_WIN
IFont IWindow::font () const
{
  return IFont(this);
}
#endif

/*------------------------------------------------------------------------------
| IWindow::characterSize                                                       |
|                                                                              |
| Return the average character width and maximum character height for the      |
| window.                                                                      |
------------------------------------------------------------------------------*/
ISize IWindow::characterSize () const
{
#ifndef IC_WIN
  if ((pWindowData->sizClChar.width() == 0) ||
      (isLayoutDistorted(IWindow::fontChanged)))
  {                               // needs initializing
     IFont fontmgr(this);         // get current font information
     pWindowData->sizClChar.setWidth(fontmgr.avgCharWidth());
     pWindowData->sizClChar.setHeight(fontmgr.maxCharHeight());
     ((IWindow *)this)->setLayoutDistorted(0,IWindow::fontChanged);
  }
  return pWindowData->sizClChar;
#else
  // IC_NOTYET  ... return a bogus but non-0 value
  return ISize(10,10);
#endif
}

/*------------------------------------------------------------------------------
| IWindow::isTabStop                                                           |
|                                                                              |
| only meaningful for controls                                                 |
------------------------------------------------------------------------------*/
IBase::Boolean IWindow::isTabStop ( ) const
{
  unsigned long ulStyle = style();
  if (ulStyle & WS_TABSTOP)
     return true;
  else
     return false;

}

/*------------------------------------------------------------------------------
| IWindow::isGroup                                                             |
|                                                                              |
| Does this have the WS_GROUP Style.                                           |
------------------------------------------------------------------------------*/
IBase::Boolean IWindow::isGroup ( ) const
{
  unsigned long ulStyle = style();
  if (ulStyle & WS_GROUP)
     return true;
  else
     return false;
}

/*------------------------------------------------------------------------------
| IWindow::capturePointer                                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::capturePointer( Boolean capture )
{
  if (capture)
  {
    if (IQUERYCAPTURE == 0)
    {
      Boolean rc;
      ISETCAPTURE(rc, this->handle());
    }
    else
    {
      ITHROWGUIERROR("WinSetCapture");
    }
  }
  else
  {
    if (IQUERYCAPTURE == this->handle())
    {
      IRELEASECAPTURE;
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::releasePointer                                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::releasePointer( )
{
   return (capturePointer( false ));
}

/*------------------------------------------------------------------------------
| IWindow::hasPointerCaptured                                                  |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
Boolean IWindow::hasPointerCaptured( ) const
{
  return (IQUERYCAPTURE == this->handle());
}

/*------------------------------------------------------------------------------
| IWindow::movePointerTo                                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IWindow::movePointerTo ( const IPoint& position )
{
  IPoint nativePoint = ICoordinateSystem::convertToNative(
                          position,
                          desktopWindow()->size() );
  long x = nativePoint.x();
  long y = nativePoint.y();

  ISETPOINTERPOSITION(x,y);
}

/*------------------------------------------------------------------------------
| IWindow::pointerPosition                                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IPoint IWindow::pointerPosition ( )
{
  long x,y;
  IQUERYPOINTERPOSITION(x,y);
  return ICoordinateSystem::convertToApplication(
                               IPoint(x,y),
                               desktopWindow()->size() );
}

/*------------------------------------------------------------------------------
| IWindow::setAcceleratorTable                                                 |
|                                                                              |
| Store the accelerator table for a window.                                    |
------------------------------------------------------------------------------*/
IWindow&
  IWindow::setAcceleratorTable ( const IAcceleratorTable* acceleratorTable )
{
  IAccelTblHandle handle( 0 );
  if ( acceleratorTable )
  {
     handle = acceleratorTable->handle();
  }
  return this->setAcceleratorHandle( handle );
}

/*------------------------------------------------------------------------------
| IWindow::acceleratorTable                                                    |
|                                                                              |
| Return the accelerator table for the window.                                 |
------------------------------------------------------------------------------*/
IAcceleratorTable IWindow::acceleratorTable ( ) const
{
  // Construct the IAcceleratorTable using the IWindow* constructor,
  // so that the object will support immediate updates on changes to
  // the contents of the table.
  IAcceleratorTable table( this );
  return table;
}

/*------------------------------------------------------------------------------
| IWindow::setAcceleratorHandle                                                |
|                                                                              |
| Store the accelerator table for a window.                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::setAcceleratorHandle ( const IAccelTblHandle& handle )
{
  // Clean-up the last one.
  if ( pWindowData->fAccelHandle )
  {  // Ignore any errors, since the application could have already
     // found and destroyed this using system APIs.
     IAcceleratorTable::destroyHandle( pWindowData->fAccelHandle );
  }

  // Store the new accelerator table handle now.
  pWindowData->fAccelHandle = handle;

#ifdef IC_PM
  // Keep the accelerator table handles maintained by IWindow and PM
  // in synch.
  if ( ! ISETACCELTABLE( handle, this->handle() ) )
  {
     ITHROWGUIERROR( "ISETACCELTABLE" );
  }
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::acceleratorHandle                                                   |
|                                                                              |
| Return the accelerator table for a window.                                   |
------------------------------------------------------------------------------*/
IAccelTblHandle IWindow::acceleratorHandle ( ) const
{
  IAccelTblHandle accelTbl = pWindowData->fAccelHandle;
#ifdef IC_PM
  IAccelTblHandle presAccelTbl = IQUERYACCELTABLE( this->handle() );
  if ( presAccelTbl != accelTbl )
  {  // The IWindow and PM accelerator tables are not the same.
     // Assume the one maintained by PM is more current (it must have been
     // updated through a PM API later than any calls to setAcceleratorTable
     // or setAcceleratorHandle), so replace the one being maintained by
     // IWindow.
     IAcceleratorTable::destroyHandle( accelTbl );
     accelTbl = presAccelTbl;
     ((IWindow* const)this)->setAcceleratorHandle( accelTbl );
  }
#endif
  return accelTbl;
}

#endif // IC_PMWIN
