implementation module Tetris;


import StdEnv, deltaPicture;

::	RandomSeed	:== Int;     

	YLength	:==	22;
	XLength	:== 10;
	ScorePos	:== (67, 20);
	LevelPos :== (167, 20);


	BS	  	:== 12; 	// block size
	Hoff  :== 20;	// horizontal offset from pixel coordinates to block coordinates
	Voff  :== 284;	// ditto vertical


    

::	State	:== (!Board,!Pos,!RandomSeed);
:: Coord :== (!Int, !Int);								// coordinate of one cell
:: Block :== (!Coord, !Coord, !Coord, !Coord);	// a tetris block contains four cells

:: Row	:== [Bool];		// a position in a row is FALSE if there is a cell 
:: Board :== [Row];		//	the tetris board

:: Pos :== (!Int, !Int, !Int);	// position of a block

:: Mode =  ToLeft | Turn | ToRight | Tick | Drop ; // indicates in which direction a block should go


    

/*	SetUp InitialBoard, no blocks, all board cells made TRUE:
*/

InitialBoard	::    Board;
InitialBoard = InitBoard 1 YLength EmptyRow;

EmptyRow ::    Row;
EmptyRow =	InitBoard 1 XLength True;

InitBoard	:: Int Int x	-> [x];
InitBoard low high v	| (low > high)		=  [];
InitBoard low high v =  [v : InitBoard (inc low) high v];

FullRow ::   	Row;
FullRow =	InitBoard 1 XLength False;


/*	SetUp StartPosition:
*/

StartPos :: Int -> Pos;
StartPos x	=  (5, 20, Elem [1,3,7,11,13,16,19] (inc (x mod 7)));

// RWS, I know: this is not the best random generator
Random :: RandomSeed -> (Int, RandomSeed);
Random seed
	=	(newSeed, newSeed)
	where {
		newSeed	= (seed * 75) mod 65537;
	}



/* Commands:
*/

DrawBlockMove :: Mode State -> (State, Bool, Bool, [DrawFunction]);
DrawBlockMove Drop state=:(board,pos, seed)
	= 	DoDrop state pos;
DrawBlockMove mode (board,pos, seed)
	| not legal						= 	((board,  pos, newSeed  ), False, False, []);
	| free	= 	((board,  pos`, newSeed ), False, False, DrawMove pos pos`);		// accept move
							| free`= 	((board``,pos``, newSeed), False, True,  pic``); 	// block landed, start new one
	= 	((board`, pos`, newSeed ), True,  True,  pic);										// no more room, game over
	where {
	pos`				=: NewPos pos mode;
	(board``, pic`)=: CheckFullRows 1 board` 0 [];
	pos``				=: StartPos (random >> 5);
	(random, newSeed)
		=	Random seed;
	pic``				=: Concat pic pic`;
	board`			=: AddBlock pos board;
	pic				=: DrawPos` BlackColour pos;
	(free`, legal`)=: Fits board pos`` mode;
	(free, legal)	=: Fits board pos`  mode;
	};

DoDrop	:: State Pos -> (State, Bool, Bool, [DrawFunction]);
DoDrop state oldpos
	| not landed	=  DoDrop state` oldpos;
	= 	(state`, gameover, landed, Concat (DrawDisappear oldpos) pic);
		where {
		(state`, gameover, landed, pic)=: DrawBlockMove Tick state;
		};


/*	Calculate what new position of block would be:
*/

NewPos :: Pos Mode -> Pos;
NewPos (x, y, tet) ToLeft	=  (dec x,	y, tet);
NewPos (x, y, tet) ToRight	=  (inc x,	y, tet);
NewPos (x, y, tet) Turn		=  (x, 		y, Rotate tet);
NewPos (x, y, tet) Tick		=  (x,	dec y, tet);


/*	Check if there is room in the board at that new position (TRUE = not landed).
	Check also on boundaries (TRUE = legal).
*/

Fits :: Board Pos Mode -> (Bool,Bool);
Fits board pos mode
	= 	(b1,b2);
   	where {
   	b1=:				 r1 && r2  && (r3 && r4);
		b2=:				 l1 && l2  && (l3 && l4);
		(r1,l1)=:			CheckFree board c1 mode;
		(r2,l2)=:			CheckFree board c2 mode;
		(r3,l3)=:			CheckFree board c3 mode;
		(r4,l4)=:			CheckFree board c4 mode;
		(c1,c2,c3,c4)=: Pos2Block pos;
   	};

CheckFree :: Board Coord Mode -> (Bool, Bool);
CheckFree board (x, y) Tick
   | (Within 1 XLength x) && (Within 1 YLength y) 	= 	(Elem (Elem board y) x, True);
CheckFree board (x, y) else
   | (Within 1 XLength x) && (Within 1 YLength y) 	= 	(True,  Elem (Elem board y) x);
   | not (Within 1 XLength x) 						= 	(True,  False);
   = 	(False, True);


/*	If there is no room, the tetris block is landed and has to be administrated in the board:
*/

AddBlock :: Pos Board -> Board;
AddBlock pos board =  For 1 YLength (AddBlock1 (Pos2Block pos) board);

AddBlock1 :: Block Board Int -> Row;
AddBlock1 block board y   =  For 1 XLength (AddBlock2 block board y);

AddBlock2 :: Block Board Int Int -> Bool;
AddBlock2 (c1, c2, c3, c4) board y x
	= 	And (Elem (Elem board y) x)
			 (And (And (And (sm__gr_Co c c1) (sm__gr_Co c c2)) (sm__gr_Co c c3)) (sm__gr_Co c c4));
	   where {
	   c=: (x,y);
	   };


/*	If the tetris block has landed, possibly filled rows have to be removed from the board:
*/

CheckFullRows :: Int Board Int Board -> (Board, [DrawFunction]);
CheckFullRows cnt [] dcnt droab
   =  (newboard, [SetPenNormal]);
   	where {
   	newboard=: WindupBlock (YLength - dcnt) droab [];
   	};
CheckFullRows cnt [row : rows] dcnt draob
	| eq_Row row FullRow = 	(newboard, Concat (ShiftRows (inc dcnt) row rows) pic);
	= 	(newboard`, pic`);								 
		where {
		(newboard`, pic`)=: CheckFullRows (inc cnt) rows (inc dcnt) [row : draob];
		(newboard,  pic) =: CheckFullRows (inc cnt) rows dcnt draob;
		};

WindupBlock :: Int Board Board -> Board;
WindupBlock cnt draob board | cnt <= 0 =  Reverse2 draob board;
 										 =  WindupBlock (dec cnt) draob [EmptyRow : board];

ShiftRows :: Int Row Board -> [DrawFunction];
ShiftRows cnt row rows =  Concat (RemoveRow cnt row) (Shift cnt rows); 
   
RemoveRow	:: Int Row	-> [DrawFunction];
RemoveRow cnt row	= 	PaintRow EmptyRow 1 cnt;

Shift	:: Int Board			-> [DrawFunction];
Shift cnt []				= 	PaintRow EmptyRow 1 cnt;
Shift cnt [row : rows]	= 	Concat (PaintRow row 1 cnt) (Shift (inc cnt) rows);

PaintRow :: Row Int Int		-> [DrawFunction];
PaintRow [r : rr] x y	| not r =  [PaintCell BlackColour (x,y) : PaintRow rr (inc x) y];
   								=  [PaintCell WhiteColour (x,y) : PaintRow rr (inc x) y];
PaintRow [] x y			=  [];


/*	Draw move of tetris block in window:
*/

DrawMove :: Pos Pos -> [DrawFunction];
DrawMove old new =  Concat (DrawDisappear old) (DrawReappear new);

DrawDisappear :: Pos -> [DrawFunction];
DrawDisappear pos =  DrawPos` WhiteColour pos;

DrawReappear :: Pos -> [DrawFunction];
DrawReappear pos=:(x,y,tet) =  DrawPos` (ColourTable tet) pos;

DrawPos :: Pos -> [DrawFunction];
DrawPos pos=:(x,y,tet) =  DrawBlock (ColourTable tet) (Pos2Block pos);

DrawPos` :: Colour Pos -> [DrawFunction];
DrawPos` color pos =  DrawBlock color (Pos2Block pos);
   
DrawBlock	:: Colour Block -> [DrawFunction];
DrawBlock color (c1, c2, c3, c4)
	= 	[PaintCell color c1, PaintCell color c2, PaintCell color c3, PaintCell color c4];

PaintCell	:: Colour Coord Picture -> Picture;
PaintCell WhiteColour coord pic =  Paintbox WhiteColour 0 coord pic;
PaintCell color coord pic
   = 	Paintbox color 2 coord
   		(Paintbox WhiteColour 1 coord
   			(Paintbox BlackColour 0 coord pic));

Paintbox	:: Colour Int Coord Picture -> Picture;
Paintbox color i (x,y) pic
	= 	FillRectangle ((h + i,v + i), ( BS - i  + h, BS - i  + v)) (SetPenColour color pic);
		where {
		h=: Hoff +  BS * x ;
		v=: Voff -  BS * y ;
		};


/*	Draw border of tetris game:
*/

DrawBorder	::    [DrawFunction];
DrawBorder
	=	Concat pic1 (Concat pic2 pic3);
   	where {
   	pic1=: Map (Paintbox RedColour 0) (For 0 11 DrawBorderH ); 
   	pic2=: Map (Paintbox RedColour 0) (For 1 20 DrawBorderVL); 
   	pic3=: Map (Paintbox RedColour 0) (For 1 20 DrawBorderVR);
   	};

DrawBorderH :: Int 	->	Coord;
DrawBorderH	 i  	= 	(i,0);
DrawBorderVL :: Int 	->	Coord;
DrawBorderVL i  	= 	(0,i);
DrawBorderVR :: Int 	->	Coord;
DrawBorderVR i   	= 	(11,i);


/*	Draw the score and the level (speed):
*/

DrawFirstScore_and_Level	:: Int Int Int -> [DrawFunction];
DrawFirstScore_and_Level maxl level points
	= 	[SetPenColour MagentaColour,
	    MovePenTo (lx - 51, ly), 
		 DrawString "Level: ", 
		 MovePenTo (px - 57, py), 
		 DrawString "Points: " : DrawScore_and_Level maxl level points];
		where {
		(px,py)=: ScorePos;
		(lx,ly)=: LevelPos;
		};

DrawScore_and_Level	:: Int Int Int -> [DrawFunction];
DrawScore_and_Level maxl level points
	= 	[SetPenColour BlackColour,
		 EraseRectangle ((dec px, py - 13), (px + 40, py + 4)),
		 MovePenTo ScorePos,
		 DrawString (toString points),
		 EraseRectangle ((dec lx, ly - 13), (lx + 40, ly + 4)),
		 MovePenTo LevelPos,
		 DrawString (toString (inc ((maxl - level) >> 1)))];
		where {
		(px,py)=: ScorePos;
		(lx,ly)=: LevelPos;
		};


/*	Draw contents board with frozen blocks:
*/

DrawBoard	:: Board -> [DrawFunction];
DrawBoard board
	= 	[SetPenColour BlackColour : Map (PaintCell BlackColour) (ConvBoard board)]; 


/*	Convert boolean board to list of coordinates to be drawn
*/

ConvBoard :: Board -> [Coord];
ConvBoard board =  Foldr Concat [] (ScanRow board 1);

ScanRow :: Board Int	-> [[Coord]];
ScanRow [r] y		=  [ScanElem r 1 y];
ScanRow [r:rr] y	=  [ScanElem r 1 y : ScanRow rr (inc y)];

ScanElem :: Row Int Int	-> [Coord];
ScanElem [r:rr] x y	| not r =  [(x,y) : ScanElem rr (inc x) y];
   							=  ScanElem rr (inc x) y;
ScanElem [] x y		=  [];


/*	Fetch block coordinates from table:
*/

Pos2Block :: Pos  -> Block;
Pos2Block (x, y, tet) =  Rel2abs (x,y) (Elem Table tet);

Rel2abs :: Coord Block  -> Block;
Rel2abs (x,y) ((x1,y1), (x2,y2), (x3,y3), (x4,y4))
   = 	((x + x1, y + y1), (x + x2, y + y2), (x + x3, y + y3), (x + x4, y + y4));


/*	Fetch rotated block coordinates:
*/

Rotate :: Int -> Int;
Rotate tet =  Elem [2,1,4,5,6,3,8,9,10,7,12,11,14,13,16,17,18,15,19] tet;


/*	ColourTable for moving Tetris blocks:
*/

ColourTable	:: Int -> Colour;
ColourTable n	| (n == 1) || (n == 2)	= 	YellowColour;
						|	(n > 2) && (n < 7)		= 	BlueColour;
						|	(n > 6) && (n < 11)		= 	CyanColour;
						| (n > 10) && (n < 13)		= 	RedColour;
						| (n > 12) && (n < 15)	= 	MagentaColour;
						| (n > 14) && (n < 19)	= 	GreenColour;
						= 	YellowColour;


/*	This table contains the pieces in the following order:

		X
		X		X		X		X		 X		X
		X		X		X		XX		XX		XX		XX
		X		XX	  XX		 X		X		X		XX
*/

Table ::    [Block];
Table = [
				((-1, 0), ( 0, 0), ( 1, 0), ( 2, 0)),		//	 1
				(( 0,-1), ( 0, 0), ( 0, 1), ( 0, 2)),		//	 2

				((-1,-1), (-1, 0), ( 0, 0), ( 1, 0)),		//	 3
				(( 0,-1), ( 1,-1), ( 0, 0), ( 0, 1)),		//	 4
				((-1, 0), ( 0, 0), ( 1, 0), ( 1, 1)),		//	 5
				(( 0,-1), ( 0, 0), ( 0, 1), (-1, 1)),		//	 6

				((-1, 0), ( 0, 0), ( 1, 0), ( 1,-1)),		//	 7
				(( 0,-1), ( 0, 0), ( 0, 1), ( 1, 1)),		//	 8
				((-1, 0), ( 0, 0), ( 1, 0), (-1, 1)),		//	 9
				((-1,-1), ( 0,-1), ( 0, 0), ( 0, 1)),		//	10

				((-1, 0), ( 0, 0), ( 0, 1), ( 1, 1)),		//	11
				(( 0, 1), ( 0, 0), ( 1, 0), ( 1,-1)),		//	12

				((-1, 1), ( 0, 1), ( 0, 0), ( 1, 0)),		//	13
				(( 0,-1), ( 0, 0), ( 1, 0), ( 1, 1)),		//	14

				(( 0, 0), ( 0, 1), (-1, 0), ( 0,-1)),		//	15
				(( 0, 0), (-1, 0), ( 0,-1), ( 1, 0)),		//	16
				(( 0,-1), ( 0, 0), ( 1, 0), ( 0, 1)),		//	17
				(( 0, 0), ( 1, 0), ( 0, 1), (-1, 0)),		//	18

				(( 0, 0), ( 0, 1), ( 1, 1), ( 1, 0))		//	19
			];


/*	Auxilary functions:
*/

Map :: (x -> y) [x] -> [y];
Map f []		=  [];
Map f [h:t]	=  [f h : Map f t];

Foldr	:: (x ->  y -> y ) y [x] -> y;
Foldr f x [] =  x;
Foldr f x [y:ys] =  f y (Foldr f x ys);
 

Concat	:: [x] [x] -> [x];
Concat [] ys	=  ys;
Concat [x:xs] ys =  [x: Concat xs ys];

For :: Int Int (Int -> x) -> [x];
For low high fun	 | (low > high)							=  [];
							 =  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	strict1=fun low;
		strict2=For (inc low) high fun;
		
	};

Elem :: [x]  Int -> x;
Elem [h : t] 1 =  h;
Elem [h : t] n =  Elem t (dec n);

Within :: Int Int Int -> Bool;
Within low high	n	=  And (low <= n) (high >= n);


/*	Lazy conjunction and disjunction:
*/

And :: Bool Bool -> Bool;
And True bool =  bool;
And False dum =  False;
   
Or :: Bool Bool	-> Bool;
Or True dum		=  True;
Or False bool	=  bool;


/*	Inequality for coordinates:
*/

sm__gr_Co :: Coord	Coord	-> Bool;
sm__gr_Co (x1,y1) (x2,y2) =  Or (x1 <> x2) (y1 <> y2);


/*	Optimised reverse:
*/

Reverse2 :: [x] [x] -> [x];
Reverse2 []			l =  l;
Reverse2 [h : t]	l =  Reverse2 t [h : l];


/*	Equality for lists of bools:
*/

eq_Row :: [Bool] [Bool] -> Bool;
eq_Row []			[]			=  True;
eq_Row []			[x]		=  False;
eq_Row [x]			[]			=  False;
eq_Row [x : xs]	[y : ys]	=  And (x == y) (eq_Row xs ys);   
