MODULE SVDTest;

        (********************************************************)
        (*                                                      *)
        (*     Tests related to singular value decomposition    *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        17 April 2016                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


IMPORT Vec;

FROM LongMath IMPORT
    (* proc *)  sqrt;

FROM LongComplexMath IMPORT
    (* proc *)  abs, conj, scalarMult;

FROM MatExtra IMPORT
    (* proc *)  ReQRFactor, ReLQFactor, CxQRFactor, CxLQFactor,
                RePLQFactor, CxPLQFactor,
                ReHouseholder, CxHouseholder, ReColHouseholder, CxColHouseholder,
                ReToBidiag, CxToBidiag, ReSVD, CxSVD;

FROM Mat IMPORT
    (* type *)  ArrayPtr, CxArrayPtr,
    (* proc *)  Write, CxWrite, NewArray, NewCxArray, DisposeArray,
                DisposeCxArray,
                ReCopy, CxCopy, Unit, CxUnit, Zero, Transpose, Adjoint,
                Sub, CxSub, Mul, CxMul,
                Random, CxRandom;

FROM MiscM2 IMPORT
    (* proc *)  SelectWindow, WriteChar, WriteString, WriteCard,
                WriteLJCard, WriteLn,
                WriteLongReal, WriteLongComplex, PressAnyKey;

(************************************************************************)

TYPE
    Re1x4 = ARRAY [1..1], [1..4] OF LONGREAL;
    Re2x2 = ARRAY [1..2], [1..2] OF LONGREAL;
    Re3x3 = ARRAY [1..3], [1..3] OF LONGREAL;
    Cx1x2 = ARRAY [1..1], [1..2] OF LONGCOMPLEX;

CONST
    small = 1.0E-15;
    CxZero = CMPLX (0.0, 0.0);

(************************************************************************)
(*                      Repeated PLQ FACTORISATION                      *)
(************************************************************************)

PROCEDURE MaxElt (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL): LONGREAL;

    (* Maximum absolute value in the array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO rows-1 DO
            FOR j := 0 TO cols-1 DO
                val := ABS(A[i,j]);
                IF val > max THEN
                    max := val;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END MaxElt;

(************************************************************************)

PROCEDURE CxMaxElt (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL): LONGREAL;

    (* Maximum absolute value in the array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO rows-1 DO
            FOR j := 0 TO cols-1 DO
                val := abs(A[i,j]);
                IF val > max THEN
                    max := val;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END CxMaxElt;

(************************************************************************)
(*                 TESTING SINGULAR VALUE DECOMPOSITION                 *)
(************************************************************************)

PROCEDURE OneReSVDtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of singular value decomposition. *)

    CONST
        fieldsize = 10;

    VAR P, L, Q, PL, PLQ, Diff: ArrayPtr;
        q0, rank: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewArray (rows, q0);
        L := NewArray (q0, q0);
        Q := NewArray (q0, cols);
        WriteString ("------------- SVD test -------------");
        WriteLn;

        rank := ReSVD (A, rows, cols, P^, L^, Q^);

        PL := NewArray (rows, rank);
        Mul (P^, L^, rows, rank, rank, PL^);
        PLQ := NewArray (rows, cols);
        Mul (PL^, Q^, rows, rank, cols, PLQ^);
        Diff := NewArray (rows, cols);
        Sub (A, PLQ^, rows, cols, Diff^);
        (*
        WriteString ("PL =");  WriteLn;
        Write (PL^, rows, rank, fieldsize);
        WriteString ("PLQ =");  WriteLn;
        Write (PLQ^, rows, cols, fieldsize);
        WriteString ("A - PLQ =");  WriteLn;
        Write (Diff^, rows, cols, fieldsize);
        *)
        WriteString ("Maximum element of A-PLQ = ");
        WriteLongReal (MaxElt(Diff^,rows,cols), fieldsize);
        WriteLn;

        DisposeArray (Diff, rows, cols);
        DisposeArray (PLQ, rows, cols);
        DisposeArray (PL, rows, rank);

        DisposeArray (Q, q0, cols);
        DisposeArray (L, q0, q0);
        DisposeArray (P, rows, q0);

    END OneReSVDtest;

(************************************************************************)

PROCEDURE ReSVDtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of singular value decomposition. *)

    CONST
        fieldsize = 10;

    BEGIN
        WriteString (" --------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        Write (A, rows, cols, fieldsize);
        OneReSVDtest (A, rows, cols);
    END ReSVDtest;

(************************************************************************)

PROCEDURE OneCxSVDtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of singular value decomposition. *)

    CONST
        fieldsize = 6;

    VAR P, L, Q, PL, PLQ, Diff: CxArrayPtr;
        q0, rank: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewCxArray (rows, q0);
        L := NewCxArray (q0, q0);
        Q := NewCxArray (q0, cols);
        WriteString ("------------- SVD test --------------");
        WriteLn;

        rank := CxSVD (A, rows, cols, P^, L^, Q^);

        PL := NewCxArray (rows, rank);
        CxMul (P^, L^, rows, rank, rank, PL^);
        PLQ := NewCxArray (rows, cols);
        CxMul (PL^, Q^, rows, rank, cols, PLQ^);
        Diff := NewCxArray (rows, cols);
        CxSub (A, PLQ^, rows, cols, Diff^);
        (*
        WriteString ("PL =");  WriteLn;
        Write (PL^, rows, rank, fieldsize);
        WriteString ("PLQ =");  WriteLn;
        CxWrite (PLQ^, rows, cols, fieldsize);
        WriteString ("A - PLQ =");  WriteLn;
        CxWrite (Diff^, rows, cols, fieldsize);
        *)
        WriteString ("Maximum element of A-PLQ = ");
        WriteLongReal (CxMaxElt(Diff^,rows,cols), fieldsize);
        WriteLn;

        DisposeCxArray (Diff, rows, cols);
        DisposeCxArray (PLQ, rows, cols);
        DisposeCxArray (PL, rows, rank);

        DisposeCxArray (Q, q0, cols);
        DisposeCxArray (L, q0, q0);
        DisposeCxArray (P, rows, q0);

    END OneCxSVDtest;

(************************************************************************)

PROCEDURE CxSVDtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of singular value decomposition. *)

    CONST
        fieldsize = 6;

    BEGIN
        WriteString (" --------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        OneCxSVDtest (A, rows, cols);
    END CxSVDtest;

(************************************************************************)

PROCEDURE DoSVDtests;

    (* Tests of singular value decomposition. *)

    VAR A: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);
        (**)
        ReSVDtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4);
        PressAnyKey;
        ReSVDtest (Re2x2 {{1.0, 0.0},
                         {0.0, 1.0}}, 2, 2);
        PressAnyKey;
        (**)
        ReSVDtest (Re2x2 {{1.0, 1.0},
                         {1.0, 1.0}}, 2, 2);
        PressAnyKey;
        (**)
        ReSVDtest (Re3x3 {{0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0}}, 3, 3);
        PressAnyKey;
        (**)
        r := 6;  c := 6;
        A := NewArray (r, c);
        Random (A^, r, c);
        ReSVDtest (A^, r, c);
        DisposeArray (A, r, c);
        PressAnyKey;

        r := 9;  c := 16;
        A := NewArray (r, c);
        Random (A^, r, c);
        ReSVDtest (A^, r, c);
        DisposeArray (A, r, c);
        PressAnyKey;
        (**)

        (* The complex case.  *)

        (**)
        r := 2;  c := 2;
        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxSVDtest (B^, r, c);
        DisposeCxArray (B, r, c);
        PressAnyKey;
        (**)

        r := 8;  c := 6;
        B := NewCxArray (r, c);
        CxRandom (B^, r, c);
        CxSVDtest (B^, r, c);
        DisposeCxArray (B, r, c);

    END DoSVDtests;

(************************************************************************)
(*                 REDUCING A MATRIX TO BIDIAGONAL FORM                 *)
(************************************************************************)

PROCEDURE ReBidiagTest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of reducing a matrix to bidiagonal form. *)

    CONST fieldsize = 8;

    (********************************************************************)

    PROCEDURE CheckOrthog (swap: BOOLEAN;  M: ARRAY OF ARRAY OF LONGREAL;
                              rows, cols: CARDINAL;  name: ARRAY OF CHAR);

        (* Check that M*M = I, if swap is FALSE, or MM* = I otherwise. *)

        VAR Mstar, Prod, I, Diff: ArrayPtr;
            r, c: CARDINAL;

        BEGIN
            Mstar := NewArray (cols, rows);
            Transpose (M, rows, cols, Mstar^);
            IF swap THEN
                r := rows;  c := cols;
            ELSE
                c := rows;  r := cols;
            END (*IF*);
            Prod := NewArray (r, r);
            I := NewArray (r, r);
            Unit (I^, r);
            Diff := NewArray (r, r);
            IF swap THEN
                Mul (M, Mstar^, r, c, r, Prod^);
            ELSE
                Mul (Mstar^, M, r, c, r, Prod^);
            END (*IF*);
            Sub (I^, Prod^, r, r, Diff^);
            WriteString ("Largest element of I - ");  WriteString (name);
            IF NOT swap THEN WriteChar ("*") END (*IF*);
            WriteString (" x ");
            WriteString (name);
            IF swap THEN WriteChar ("*") END (*IF*);
            WriteString (" = ");
            WriteLongReal (MaxElt(Diff^, r, r), fieldsize);  WriteLn;
            DisposeArray (Mstar, r, r);
            DisposeArray (Prod, r, r);
            DisposeArray (I, r, r);
            DisposeArray (Diff, r, r);
        END CheckOrthog;

    (********************************************************************)

    CONST showall = TRUE;

    VAR oldrank, rank: CARDINAL;
        Q, B, P, Prod0, Prod, Diff: ArrayPtr;

    BEGIN
        WriteString ("-------- Bidiagonal conversion ----------");  WriteLn;
        WriteString ("Original matrix A =");  WriteLn;
        Write (A, rows, cols, fieldsize);
        PressAnyKey;

        (* Work out required array sizes. *)

        oldrank := rows;
        IF cols < oldrank THEN oldrank := cols END(*IF*);

        (* Do the factorisation. *)

        Q := NewArray (rows, oldrank);
        B := NewArray (oldrank, oldrank);
        P := NewArray (oldrank, cols);
        rank := ReToBidiag (A, rows, cols, Q^, B^, P^);

        IF showall THEN
            WriteString ("Q =");  WriteLn;
            Write (Q^, rows, rank, fieldsize);
        END (*IF*);
        WriteString ("B =");  WriteLn;
        Write (B^, rank, rank, fieldsize);
        IF showall THEN
            WriteString ("P =");  WriteLn;
            Write (P^, rank, cols, fieldsize);
        END (*IF*);

        Prod0 := NewArray (rows, rank);
        Mul (Q^, B^, rows, rank, rank, Prod0^);
        Prod := NewArray (rows, cols);
        Mul (Prod0^, P^, rows, rank, cols, Prod^);
        DisposeArray (Prod0, rows, rank);

        Diff := NewArray (rows, cols);
        Sub (A, Prod^, rows, cols, Diff^);
        CheckOrthog (FALSE, Q^, rows, rank, "Q");
        CheckOrthog (TRUE, P^, rank, cols, "P");
        WriteString ("Largest element of A - Q x B x P = ");
        WriteLongReal (MaxElt(Diff^, rows, cols), fieldsize);  WriteLn;
        DisposeArray (Diff, rows, cols);
        DisposeArray (Prod, rows, cols);

        DisposeArray (P, oldrank, cols);
        DisposeArray (B, oldrank, oldrank);
        DisposeArray (Q, rows, oldrank);

    END ReBidiagTest;

(************************************************************************)
(*        REDUCING A MATRIX TO BIDIAGONAL FORM - THE COMPLEX CASE       *)
(************************************************************************)

PROCEDURE CxBidiagTest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of reducing a matrix to bidiagonal form. *)

    CONST fieldsize = 8;

    (********************************************************************)

    PROCEDURE CheckOrthog (swap: BOOLEAN;  M: ARRAY OF ARRAY OF LONGCOMPLEX;
                              rows, cols: CARDINAL;  name: ARRAY OF CHAR);

        (* Check that M*M = I, if swap is FALSE, or MM* = I otherwise. *)

        VAR Mstar, Prod, I, Diff: CxArrayPtr;
            r, c: CARDINAL;

        BEGIN
            Mstar := NewCxArray (cols, rows);
            Adjoint (M, rows, cols, Mstar^);
            IF swap THEN
                r := rows;  c := cols;
            ELSE
                c := rows;  r := cols;
            END (*IF*);
            Prod := NewCxArray (r, r);
            I := NewCxArray (r, r);
            CxUnit (I^, r);
            Diff := NewCxArray (r, r);
            IF swap THEN
                CxMul (M, Mstar^, r, c, r, Prod^);
            ELSE
                CxMul (Mstar^, M, r, c, r, Prod^);
            END (*IF*);
            CxSub (I^, Prod^, r, r, Diff^);
            WriteString ("Largest element of I - ");  WriteString (name);
            IF NOT swap THEN WriteChar ("*") END (*IF*);
            WriteString (" x ");
            WriteString (name);
            IF swap THEN WriteChar ("*") END (*IF*);
            WriteString (" = ");
            WriteLongReal (CxMaxElt(Diff^, r, r), fieldsize);  WriteLn;
            DisposeCxArray (Mstar, r, r);
            DisposeCxArray (Prod, r, r);
            DisposeCxArray (I, r, r);
            DisposeCxArray (Diff, r, r);
        END CheckOrthog;

    (********************************************************************)

    CONST showall = TRUE;

    VAR oldrank, rank: CARDINAL;
        Q, B, P, Prod0, Prod, Diff: CxArrayPtr;

    BEGIN
        WriteString ("-------- Bidiagonal conversion ----------");  WriteLn;
        WriteString ("Original matrix A =");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        PressAnyKey;

        (* Work out required array sizes. *)

        oldrank := rows;
        IF cols < oldrank THEN oldrank := cols END(*IF*);

        (* Do the factorisation. *)

        Q := NewCxArray (rows, oldrank);
        B := NewCxArray (oldrank, oldrank);
        P := NewCxArray (oldrank, cols);
        rank := CxToBidiag (A, rows, cols, Q^, B^, P^);

        IF showall THEN
            WriteString ("Q =");  WriteLn;
            CxWrite (Q^, rows, rank, fieldsize);
        END (*IF*);
        WriteString ("B =");  WriteLn;
        CxWrite (B^, rank, rank, fieldsize);
        IF showall THEN
            WriteString ("P =");  WriteLn;
            CxWrite (P^, rank, cols, fieldsize);
        END (*IF*);

        Prod0 := NewCxArray (rows, rank);
        CxMul (Q^, B^, rows, rank, rank, Prod0^);
        Prod := NewCxArray (rows, cols);
        CxMul (Prod0^, P^, rows, rank, cols, Prod^);
        DisposeCxArray (Prod0, rows, rank);

        Diff := NewCxArray (rows, cols);
        CxSub (A, Prod^, rows, cols, Diff^);
        CheckOrthog (FALSE, Q^, rows, rank, "Q");
        CheckOrthog (TRUE, P^, rank, cols, "P");
        WriteString ("Largest element of A - Q x B x P = ");
        WriteLongReal (CxMaxElt(Diff^, rows, cols), fieldsize);  WriteLn;
        DisposeCxArray (Diff, rows, cols);
        DisposeCxArray (Prod, rows, cols);

        DisposeCxArray (P, oldrank, cols);
        DisposeCxArray (B, oldrank, oldrank);
        DisposeCxArray (Q, rows, oldrank);

    END CxBidiagTest;

(************************************************************************)
(*                          BIDIAGONAL TESTS                            *)
(************************************************************************)

PROCEDURE DoBidiagTests;

    (* Tests of bidiagonal reduction. *)

    VAR A: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);

        (**)
        r := 2;  c := 4;
        A := NewArray (r, c);
        Zero (A^, r, c);
        ReBidiagTest (A^, r, c);
        DisposeArray (A, r, c);
        PressAnyKey;

        r := 8;  c := 6;
        A := NewArray (r, c);
        Random (A^, r, c);
        ReBidiagTest (A^, r, c);
        DisposeArray (A, r, c);
        (**)

        (* The complex case.  *)

        (**)
        r := 2;  c := 2;
        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxBidiagTest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)

        r := 50;  c := 3;
        B := NewCxArray (r, c);
        CxRandom (B^, r, c);
        CxBidiagTest (B^, r, c);
        DisposeCxArray (B, r, c);

    END DoBidiagTests;

(************************************************************************)
(*                             MAIN PROGRAM                             *)
(************************************************************************)

BEGIN
    DoBidiagTests;
    DoSVDtests;
END SVDTest.

