/* 
 * tclOS2Chan.c
 *
 *	Channel drivers for OS/2 channels based on files, command
 *	pipes and TCP sockets.
 *
 * Copyright (c) 1995-1996 Sun Microsystems, Inc.
 * Copyright (c) 1996-1998 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tclOS2Int.h"

/*
 * This structure describes per-instance state of a pipe based channel.
 *
 * IMPORTANT NOTE: If you modify this structure, make sure that the
 * "asynch" field remains the first field - PipeBlockMode depends
 * on this.
 */

typedef struct PipeState {
    int asynch;			/* 1 if channel is in asynch mode. */
    Tcl_File readFile;		/* Output from pipe. */
    Tcl_File writeFile;		/* Input from pipe. */
    Tcl_File errorFile;		/* Error output from pipe. */
    int numPids;		/* Number of processes attached to pipe. */
    int *pidPtr;		/* Pids of attached processes. */
} PipeState;

/*
 * This structure describes per-instance state of a tcp based channel.
 */

typedef struct TcpState {
    int flags;                          /* ORed combination of the
                                         * bitfields defined below. */
    Tcl_File sock;                      /* The socket itself. */
    Tcl_TcpAcceptProc *acceptProc;      /* Proc to call on accept. */
    ClientData acceptProcData;          /* The data for the accept proc. */
} TcpState;

/*
 * These bits may be ORed together into the "flags" field of a TcpState
 * structure.
 */

#define TCP_ASYNC_SOCKET        (1<<0)  /* Asynchronous socket. */
#define TCP_ASYNC_CONNECT       (1<<1)  /* Async connect in progress. */

/*
 * The following defines the maximum length of the listen queue. This is
 * the number of outstanding yet-to-be-serviced requests for a connection
 * on a server socket, more than this number of outstanding requests and
 * the connection request will fail.
 */

#ifndef SOMAXCONN
#define SOMAXCONN       100
#endif

#if     (SOMAXCONN < 100)
#undef  SOMAXCONN
#define SOMAXCONN       100
#endif

/*
 * The following defines how much buffer space the kernel should maintain
 * for a socket.
 */

#define SOCKET_BUFSIZE  4096

/*
 * Static routines for this file:
 */

static int              FileBlockModeProc _ANSI_ARGS_((
                            ClientData instanceData, int mode));
static int              FileCloseProc _ANSI_ARGS_((ClientData instanceData,
                            Tcl_Interp *interp));
static int              FileSeekProc _ANSI_ARGS_((ClientData instanceData,
                            long offset, int mode, int *errorCode));
static int              FileInputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toRead, int *errorCode));
static int              FileOutputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toWrite, int *errorCode));
static int              FileType _ANSI_ARGS_((HFILE h));
static void             FileWatchProc _ANSI_ARGS_((ClientData instanceData,
                            int mask));
static int              FileReadyProc _ANSI_ARGS_((ClientData instanceData,
                            int mask));
static Tcl_File         FileGetProc _ANSI_ARGS_((ClientData instanceData,
                            int direction));

static int              PipeBlockModeProc _ANSI_ARGS_((
                            ClientData instanceData, int mode));
static int              PipeCloseProc _ANSI_ARGS_((ClientData instanceData,
                            Tcl_Interp *interp));
static int              PipeInputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toRead, int *errorCode));
static int              PipeOutputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toWrite, int *errorCode));
static void             PipeWatchProc _ANSI_ARGS_((ClientData instanceData,
                            int mask));
static int              PipeReadyProc _ANSI_ARGS_((ClientData instanceData,
                            int mask));
static Tcl_File         PipeGetProc _ANSI_ARGS_((ClientData instanceData,
                            int direction));

static TcpState *       CreateSocket _ANSI_ARGS_((Tcl_Interp *interp,
                            int port, char *host, int server,
                            char *myaddr, int myport, int async));
static int              CreateSocketAddress _ANSI_ARGS_(
                            (struct sockaddr_in *sockaddrPtr,
                            char *host, int port));
static void             TcpAccept _ANSI_ARGS_((ClientData data, int mask));
static int              TcpBlockModeProc _ANSI_ARGS_((ClientData data,
                            int mode));
static int              TcpCloseProc _ANSI_ARGS_((ClientData instanceData,
                            Tcl_Interp *interp));
static Tcl_File         TcpGetProc _ANSI_ARGS_((ClientData instanceData,
                            int direction));
static int              TcpGetOptionProc _ANSI_ARGS_((ClientData instanceData,
                            char *optionName, Tcl_DString *dsPtr));
static int              TcpInputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toRead, int *errorCode));
static int              TcpOutputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toWrite, int *errorCode));
static int              TcpReadyProc _ANSI_ARGS_((ClientData instanceData,
                            int mask));
static void             TcpWatchProc _ANSI_ARGS_((ClientData instanceData,
                            int mask));
static int              WaitForConnect _ANSI_ARGS_((TcpState *statePtr,
                            int *errorCodePtr));

/*
 * This structure describes the channel type structure for file based IO.
 */

static Tcl_ChannelType fileChannelType = {
    "file",			/* Type name. */
    FileBlockModeProc,		/* Set blocking or non-blocking mode.*/
    FileCloseProc,		/* Close proc. */
    FileInputProc,		/* Input proc. */
    FileOutputProc,		/* Output proc. */
    FileSeekProc,		/* Seek proc. */
    NULL,			/* Set option proc. */
    NULL,			/* Get option proc. */
    FileWatchProc,              /* Set up the notifier to watch the channel. */
    FileReadyProc,              /* Are events present? */
    FileGetProc,                /* Get a Tcl_File from channel. */
};

/*
 * This structure describes the channel type structure for command pipe
 * based IO.
 */

static Tcl_ChannelType pipeChannelType = {
    "pipe",			/* Type name. */
    PipeBlockModeProc,		/* Set blocking or non-blocking mode.*/
    PipeCloseProc,		/* Close proc. */
    PipeInputProc,		/* Input proc. */
    PipeOutputProc,		/* Output proc. */
    NULL,			/* Seek proc. */
    NULL,			/* Set option proc. */
    NULL,			/* Get option proc. */
    PipeWatchProc,              /* Set up notifier to watch the channel. */
    PipeReadyProc,              /* Are events present? */
    PipeGetProc,                /* Get a Tcl_File from channel. */
};

/*
 * This structure describes the channel type structure for TCP socket
 * based IO:
 */

static Tcl_ChannelType tcpChannelType = {
    "tcp",                              /* Type name. */
    TcpBlockModeProc,                   /* Set blocking/nonblocking mode.*/
    TcpCloseProc,                       /* Close proc. */
    TcpInputProc,                       /* Input proc. */
    TcpOutputProc,                      /* Output proc. */
    NULL,                               /* Seek proc. */
    NULL,                               /* Set option proc. */
    TcpGetOptionProc,                   /* Get option proc. */
    TcpWatchProc,                       /* Initialize notifier. */
    TcpReadyProc,                       /* Are there events? */
    TcpGetProc,                         /* Get Tcl_Files out of channel. */
};

/*
 * This is the size of the channel name for File based channels
 */

#define CHANNEL_NAME_SIZE	64
static char channelName[CHANNEL_NAME_SIZE+1];

/*
 * Structure describing per-instance state for file based channels.
 *
 * IMPORTANT NOTE: If you modify this structure, make sure that the
 * "asynch" field remains the first field - FileBlockMode depends
 * on this.
 */

typedef struct FileState {
    int asynch;			/* 1 if channel is in asynch mode. */
    int append;			/* 1 if channel is in append mode. */
    Tcl_File inFile;            /* Input file. */
    Tcl_File outFile;           /* Output file. */
} FileState;

/*
 *----------------------------------------------------------------------
 *
 * FileBlockModeProc --
 *
 *      Set blocking or non-blocking mode on channel.
 *
 * Results:
 *      0 if successful, errno when failed.
 *
 * Side effects:
 *      Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

static int
FileBlockModeProc(instanceData, mode)
    ClientData instanceData;            /* Instance state for channel. */
    int mode;                           /* The mode to set. */
{
    FileState *fsPtr = (FileState *) instanceData;

    /*
     * Files on OS/2 can not be switched between blocking and nonblocking,
     * hence we have to emulate the behavior. This is done in the input
     * function by checking against a bit in the state. We set or unset the
     * bit here to cause the input function to emulate the correct behavior.
     */

    fsPtr->asynch = (mode == TCL_MODE_BLOCKING) ? 0 : 1;
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * FileCloseProc --
 *
 *	Closes the IO channel.
 *
 * Results:
 *	0 if successful, the value of errno if failed.
 *
 * Side effects:
 *	Closes the physical channel
 *
 *----------------------------------------------------------------------
 */

static int
FileCloseProc(instanceData, interp)
    ClientData instanceData;	/* Pointer to FileState structure. */
    Tcl_Interp *interp;		/* Not used. */
{
    FileState *fsPtr = (FileState *) instanceData;
    HFILE handle;
    int type, errorCode = 0;

    if (fsPtr->inFile != NULL) {
        handle = (HFILE) Tcl_GetFileInfo(fsPtr->inFile, &type);

	/*
	 * Check for read/write file so we only close it once.
	 */

	if (fsPtr->inFile == fsPtr->outFile) {
	    fsPtr->outFile = NULL;
	}
        /*
         * Close the OS/2 handle first, since Tcl_FreeFile can call DosDelete
         * via the notifier data in case of a temp file.
         */

	rc = DosClose(handle);
	if (rc != NO_ERROR) {
	    TclOS2ConvertError(rc);
	    errorCode = errno;
	}

        Tcl_FreeFile(fsPtr->inFile);

    }
    if (fsPtr->outFile != NULL) {
        handle = (HFILE) Tcl_GetFileInfo(fsPtr->outFile, &type);
        /*
         * Close the OS/2 handle first, since Tcl_FreeFile can call DosDelete
         * via the notifier data in case of a temp file.
         */


	rc = DosClose(handle);
	if (rc != NO_ERROR && errorCode == 0 ) {
	    TclOS2ConvertError(rc);
            errorCode = errno;
	}

	Tcl_FreeFile(fsPtr->outFile);

    }
    ckfree((char *) instanceData);
    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * FileSeekProc --
 *
 *	Seeks on a file-based channel. Returns the new position.
 *
 * Results:
 *	-1 if failed, the new position if successful. If failed, it
 *	also sets *errorCodePtr to the error code.
 *
 * Side effects:
 *	Moves the location at which the channel will be accessed in
 *	future operations.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
FileSeekProc(instanceData, offset, mode, errorCodePtr)
    ClientData instanceData;			/* Unused. */
    long offset;				/* Offset to seek to. */
    int mode;					/* Relative to where
                                                 * should we seek? */
    int *errorCodePtr;				/* To store error code. */
{
    FileState *fsPtr = (FileState *) instanceData;
    ULONG moveMethod;
    ULONG newPos;
    HFILE handle;
    int type;

    *errorCodePtr = 0;
    if (fsPtr->inFile != (Tcl_File) NULL) {
        handle = (HFILE) Tcl_GetFileInfo(fsPtr->inFile, &type);
    } else if (fsPtr->outFile != (Tcl_File) NULL) {
        handle = (HFILE) Tcl_GetFileInfo(fsPtr->outFile, &type);
    } else {
        *errorCodePtr = EFAULT;
        return -1;
    }
    
    if (mode == SEEK_SET) {
        moveMethod = FILE_BEGIN;
    } else if (mode == SEEK_CUR) {
        moveMethod = FILE_CURRENT;
    } else {
        moveMethod = FILE_END;
    }

    rc = DosSetFilePtr(handle, offset, moveMethod, &newPos);
    if (rc != NO_ERROR) {
        return -1;
    }
    return newPos;
}

/*
 *----------------------------------------------------------------------
 *
 * FileInputProc --
 *
 *	Reads input from the IO channel into the buffer given. Returns
 *	count of how many bytes were actually read, and an error indication.
 *
 * Results:
 *	A count of how many bytes were read is returned and an error
 *	indication is returned in an output argument.
 *
 * Side effects:
 *	Reads input from the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileInputProc(instanceData, buf, bufSize, errorCode)
    ClientData instanceData;		/* File state. */
    char *buf;				/* Where to store data read. */
    int bufSize;			/* How much space is available
                                         * in the buffer? */
    int *errorCode;			/* Where to store error code. */
{
    FileState *statePtr;
    HFILE handle;
    ULONG bytesRead;
    int type;

    *errorCode = 0;
    statePtr = (FileState *) instanceData;
    handle = (HFILE) Tcl_GetFileInfo(statePtr->inFile, &type);

    /*
     * Note that we will block on reads from a console buffer until a
     * full line has been entered.  The only way I know of to get
     * around this is to write a console driver.  We should probably
     * do this at some point, but for now, we just block.
     */

    rc = DosRead(handle, (PVOID) buf, (ULONG) bufSize, &bytesRead);
    if (rc != NO_ERROR) {
	goto error;
    }
    
    return bytesRead;

error:
    TclOS2ConvertError(rc);
    *errorCode = errno;
    if (errno == EPIPE) {
	return 0;
    }
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * FileOutputProc --
 *
 *	Writes the given output on the IO channel. Returns count of how
 *	many characters were actually written, and an error indication.
 *
 * Results:
 *	A count of how many characters were written is returned and an
 *	error indication is returned in an output argument.
 *
 * Side effects:
 *	Writes output on the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileOutputProc(instanceData, buf, toWrite, errorCode)
    ClientData instanceData;		/* Unused. */
    char *buf;				/* The data buffer. */
    int toWrite;			/* How many bytes to write? */
    int *errorCode;			/* Where to store error code. */
{
    FileState *statePtr = (FileState *) instanceData;
    int type;
    ULONG bytesWritten;
    HFILE handle;
    ULONG newPos;
    
    *errorCode = 0;
    handle = (HFILE) Tcl_GetFileInfo(statePtr->outFile, &type);

    /*
     * If we are writing to a file that was opened with O_APPEND, we need to
     * seek to the end of the file before writing the current buffer.
     */

    if (statePtr->append) {
        DosSetFilePtr(handle, 0, FILE_END, &newPos);
    }

    rc = DosWrite(handle, (PVOID) buf, (ULONG) toWrite, &bytesWritten);
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        if (errno == EPIPE) {
            return 0;
        }
        *errorCode = errno;
        return -1;
    }
    DosResetBuffer(handle);
    return bytesWritten;
}

/*
 *----------------------------------------------------------------------
 *
 * FileWatchProc --
 *
 *      Called by the notifier to set up to watch for events on this
 *      channel.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static void
FileWatchProc(instanceData, mask)
    ClientData instanceData;            /* File state. */
    int mask;                           /* What events to watch for; OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    FileState *fsPtr = (FileState *) instanceData;

    if ((mask & TCL_READABLE) && (fsPtr->inFile != (Tcl_File) NULL)) {
        Tcl_WatchFile(fsPtr->inFile, TCL_READABLE);
    }
    if ((mask & TCL_WRITABLE) && (fsPtr->outFile != (Tcl_File) NULL)) {
        Tcl_WatchFile(fsPtr->outFile, TCL_WRITABLE);
    }

    if (mask & TCL_EXCEPTION) {
        if (fsPtr->inFile != (Tcl_File) NULL) {
            Tcl_WatchFile(fsPtr->inFile, TCL_EXCEPTION);
        }
        if (fsPtr->outFile != (Tcl_File) NULL) {
            Tcl_WatchFile(fsPtr->outFile, TCL_EXCEPTION);
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * FileReadyProc --
 *
 *      Called by the notifier to check whether events of interest are
 *      present on the channel.
 *
 * Results:
 *      Returns OR-ed combination of TCL_READABLE, TCL_WRITABLE and
 *      TCL_EXCEPTION to indicate which events of interest are present.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
FileReadyProc(instanceData, mask)
    ClientData instanceData;            /* The file state. */
    int mask;                           /* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    FileState *fsPtr = (FileState *) instanceData;
    int present = 0;

    if ((mask & TCL_READABLE) && (fsPtr->inFile != (Tcl_File) NULL)) {
        present |= Tcl_FileReady(fsPtr->inFile, TCL_READABLE);
    }
    if ((mask & TCL_WRITABLE) && (fsPtr->outFile != (Tcl_File) NULL)) {
        present |= Tcl_FileReady(fsPtr->outFile, TCL_WRITABLE);
    }
    if (mask & TCL_EXCEPTION) {
        if (fsPtr->inFile != (Tcl_File) NULL) {
            present |= Tcl_FileReady(fsPtr->inFile, TCL_EXCEPTION);
        }
        if (fsPtr->outFile != (Tcl_File) NULL) {
            present |= Tcl_FileReady(fsPtr->outFile, TCL_EXCEPTION);
        }
    }
    return present;
}

/*
 *----------------------------------------------------------------------
 *
 * FileGetProc --
 *
 *      Called from Tcl_GetChannelFile to retrieve Tcl_Files from inside
 *      a file based channel.
 *
 * Results:
 *      The appropriate Tcl_File or NULL if not present.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static Tcl_File
FileGetProc(instanceData, direction)
    ClientData instanceData;            /* The file state. */
    int direction;                      /* Which Tcl_File to retrieve? */
{
    FileState *fsPtr = (FileState *) instanceData;

    if (direction == TCL_READABLE) {
        return fsPtr->inFile;
    }
    if (direction == TCL_WRITABLE) {
        return fsPtr->outFile;
    }
    return (Tcl_File) NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeBlockModeProc --
 *
 *	Set blocking or non-blocking mode on channel.
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

static int
PipeBlockModeProc(instanceData, mode)
    ClientData instanceData;		/* Instance state for channel. */
    int mode;				/* The mode to set. */
{
    PipeState *statePtr = (PipeState *) instanceData;
    /*
     * Files on OS/2 can not be switched between blocking and nonblocking,
     * hence we have to emulate the behavior. This is done in the input
     * function by checking against a bit in the state. We set or unset the
     * bit here to cause the input function to emulate the correct behavior.
     */

    statePtr->asynch = (mode == TCL_MODE_BLOCKING) ? 0 : 1;

    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeCloseProc --
 *
 *	Closes a pipe based IO channel.
 *
 * Results:
 *	0 on success, errno otherwise.
 *
 * Side effects:
 *	Closes the physical channel.
 *
 *----------------------------------------------------------------------
 */

static int
PipeCloseProc(instanceData, interp)
    ClientData instanceData;	/* Pointer to PipeState structure. */
    Tcl_Interp *interp;		/* For error reporting. */
{
    PipeState *pipePtr = (PipeState *) instanceData;
    FileState *fsPtr;
    HFILE handle;
    Tcl_Channel errChan;
    int errorCode, result;

    errorCode = 0;
    if (pipePtr->readFile != NULL) {
        handle = (HFILE) Tcl_GetFileInfo(pipePtr->readFile, NULL);
        /*
         * Close the OS/2 handle first, since Tcl_FreeFile can call DosDelete
         * via the notifier data in case of a temp file.
         */

	rc = DosClose(handle);
	if (rc != NO_ERROR) {
	    TclOS2ConvertError(rc);
	    errorCode = errno;
	}

        Tcl_FreeFile(pipePtr->readFile);

    }
    if (pipePtr->writeFile != NULL) {
        handle = (HFILE) Tcl_GetFileInfo(pipePtr->writeFile, NULL);
        /*
         * Close the OS/2 handle first, since Tcl_FreeFile can call DosDelete
         * via the notifier data in case of a temp file.
         */

	rc = DosClose(handle);
	if (rc != NO_ERROR && errorCode == 0 ) {
            TclOS2ConvertError(rc);
            errorCode = errno;
	}

	Tcl_FreeFile(pipePtr->writeFile);
    }
    
    /*
     * Wrap the error file into a channel and give it to the cleanup
     * routine.
     */

    if (pipePtr->errorFile != NULL) {
        fsPtr = (FileState *) ckalloc((unsigned) sizeof(FileState));

        fsPtr->inFile = pipePtr->errorFile;
        fsPtr->outFile = (Tcl_File) NULL;
        fsPtr->asynch = 0;
        fsPtr->append = 0;

	errChan = Tcl_CreateChannel(&fileChannelType, "pipeError",
                                    (ClientData) fsPtr, TCL_READABLE);
        if (Tcl_SetChannelOption(interp, errChan, "-translation", "auto") ==
                TCL_ERROR) {
            Tcl_Close((Tcl_Interp *) NULL, errChan);
            errChan = (Tcl_Channel) NULL;
        }
        if ((errChan != (Tcl_Channel) NULL) &&
                (Tcl_SetChannelOption(NULL, errChan, "-eofchar", "\032") ==
                        TCL_ERROR)) {
            Tcl_Close((Tcl_Interp *) NULL, errChan);
            errChan = (Tcl_Channel) NULL;
        }
    } else {
        errChan = NULL;
    }
    result = TclCleanupChildren(interp, pipePtr->numPids, pipePtr->pidPtr,
            errChan);
    if (pipePtr->numPids > 0) {
        ckfree((char *) pipePtr->pidPtr);
    }
    ckfree((char *) pipePtr);
    if (errorCode == 0) {
        return result;
    }
    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeInputProc --
 *
 *	Reads input from the IO channel into the buffer given. Returns
 *	count of how many bytes were actually read, and an error indication.
 *
 * Results:
 *	A count of how many bytes were read is returned and an error
 *	indication is returned in an output argument.
 *
 * Side effects:
 *	Reads input from the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
PipeInputProc(instanceData, buf, bufSize, errorCode)
    ClientData instanceData;		/* Pipe state. */
    char *buf;				/* Where to store data read. */
    int bufSize;			/* How much space is available
                                         * in the buffer? */
    int *errorCode;			/* Where to store error code. */
{
    PipeState *statePtr;
    HFILE handle;
    ULONG bytesRead;
    int type;

    *errorCode = 0;
    statePtr = (PipeState *) instanceData;
    handle = (HFILE) Tcl_GetFileInfo(statePtr->readFile, &type);

    /*
     * Pipes will block until the requested number of bytes has been
     * read.  To avoid blocking unnecessarily, we look ahead and only
     * read as much as is available.
     */

    /*
     * Note that we will block on reads from a console buffer until a
     * full line has been entered.  The only way I know of to get
     * around this is to write a console driver.  We should probably
     * do this at some point, but for now, we just block.
     */

    rc = DosRead(handle, (PVOID) buf, (ULONG) bufSize, &bytesRead);
    if (rc != NO_ERROR) {
	goto error;
    }
    
    return bytesRead;

error:
    TclOS2ConvertError(rc);
    *errorCode = errno;
    if (errno == EPIPE) {
	return 0;
    }
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeOutputProc --
 *
 *	Writes the given output on the IO channel. Returns count of how
 *	many characters were actually written, and an error indication.
 *
 * Results:
 *	A count of how many characters were written is returned and an
 *	error indication is returned in an output argument.
 *
 * Side effects:
 *	Writes output on the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
PipeOutputProc(instanceData, buf, toWrite, errorCode)
    ClientData instanceData;		/* Unused. */
    char *buf;				/* The data buffer. */
    int toWrite;			/* How many bytes to write? */
    int *errorCode;			/* Where to store error code. */
{
    PipeState *statePtr = (PipeState *) instanceData;
    int type;
    ULONG bytesWritten;
    HFILE handle;
    
    *errorCode = 0;
    handle = (HFILE) Tcl_GetFileInfo(statePtr->writeFile, &type);

    rc = DosWrite(handle, (PVOID) buf, (ULONG) toWrite, &bytesWritten);
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        if (errno == EPIPE) {
            return 0;
        }
        *errorCode = errno;
        return -1;
    }
    return bytesWritten;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeWatchProc --
 *
 *      Called by the notifier to set up to watch for events on this
 *      channel.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static void
PipeWatchProc(instanceData, mask)
    ClientData instanceData;            /* Pipe state. */
    int mask;                           /* What events to watch for; OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    PipeState *psPtr = (PipeState *) instanceData;

    if ((mask & TCL_READABLE) && (psPtr->readFile != (Tcl_File) NULL)) {
        Tcl_WatchFile(psPtr->readFile, TCL_READABLE);
    }
    if ((mask & TCL_WRITABLE) && (psPtr->writeFile != (Tcl_File) NULL)) {
        Tcl_WatchFile(psPtr->writeFile, TCL_WRITABLE);
    }

    if (mask & TCL_EXCEPTION) {
        if (psPtr->readFile != (Tcl_File) NULL) {
            Tcl_WatchFile(psPtr->readFile, TCL_EXCEPTION);
        }
        if (psPtr->writeFile != (Tcl_File) NULL) {
            Tcl_WatchFile(psPtr->writeFile, TCL_EXCEPTION);
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PipeReadyProc --
 *
 *      Called by the notifier to check whether events of interest are
 *      present on the channel.
 *
 * Results:
 *      Returns OR-ed combination of TCL_READABLE, TCL_WRITABLE and
 *      TCL_EXCEPTION to indicate which events of interest are present.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
PipeReadyProc(instanceData, mask)
    ClientData instanceData;            /* The file state. */
    int mask;                           /* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    PipeState *psPtr = (PipeState *) instanceData;
    int present = 0;

    if ((mask & TCL_READABLE) && (psPtr->readFile != (Tcl_File) NULL)) {
        present |= Tcl_FileReady(psPtr->readFile, TCL_READABLE);
    }
    if ((mask & TCL_WRITABLE) && (psPtr->writeFile != (Tcl_File) NULL)) {
        present |= Tcl_FileReady(psPtr->writeFile, TCL_WRITABLE);
    }
    if (mask & TCL_EXCEPTION) {
        if (psPtr->readFile != (Tcl_File) NULL) {
            present |= Tcl_FileReady(psPtr->readFile, TCL_EXCEPTION);
        }
        if (psPtr->writeFile != (Tcl_File) NULL) {
            present |= Tcl_FileReady(psPtr->writeFile, TCL_EXCEPTION);
        }
    }
    return present;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeGetProc --
 *
 *      Called from Tcl_GetChannelFile to retrieve Tcl_Files from inside
 *      a file based channel.
 *
 * Results:
 *      The appropriate Tcl_File or NULL if not present.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static Tcl_File
PipeGetProc(instanceData, direction)
    ClientData instanceData;            /* The file state. */
    int direction;                      /* Which Tcl_File to retrieve? */
{
    PipeState *psPtr = (PipeState *) instanceData;

    if (direction == TCL_READABLE) {
        return psPtr->readFile;
    }
    if (direction == TCL_WRITABLE) {
        return psPtr->writeFile;
    }
    return (Tcl_File) NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenFileChannel --
 *
 *	Open an File based channel.
 *
 * Results:
 *	The new channel or NULL. If NULL, the output argument
 *	errorCodePtr is set to a POSIX error.
 *
 * Side effects:
 *	May open the channel and may cause creation of a file on the
 *	file system.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenFileChannel(interp, fileName, modeString, permissions)
    Tcl_Interp *interp;			/* Interpreter for error reporting;
                                         * can be NULL. */
    char *fileName;			/* Name of file to open. */
    char *modeString;			/* A list of POSIX open modes or
                                         * a string such as "rw". */
    int permissions;			/* If the open involves creating a
                                         * file, with what modes to create
                                         * it? */
{
    Tcl_File file;
    Tcl_Channel chan;
    FileState *sPtr;
    int seekFlag, mode, readWriteMode;
    HFILE handle;
    ULONG accessMode = 0, createMode, flags, exist;
    BOOL readonly = FALSE;
    char *nativeName;
    Tcl_DString buffer;

    mode = TclGetOpenMode(interp, modeString, &seekFlag);
    if (mode == -1) {
        return NULL;
    }
    switch (mode & (O_RDONLY | O_WRONLY | O_RDWR)) {
	case O_RDONLY:
	    accessMode = OPEN_ACCESS_READONLY;
	    readonly = TRUE; /* Needed because O_A_R is 0 */
	    break;
	case O_WRONLY:
	    accessMode = OPEN_ACCESS_WRITEONLY;
	    break;
	case O_RDWR:
	    accessMode = OPEN_ACCESS_READWRITE;
	    break;
	default:
	    panic("Tcl_OpenFileChannel: invalid mode value");
	    break;
    }

    /*
     * Map the creation flags to the OS/2 open mode.
     */

    switch (mode & (O_CREAT | O_EXCL | O_TRUNC)) {
	case (O_CREAT | O_EXCL | O_TRUNC):
	    createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_FAIL_IF_EXISTS;
	    break;
	case (O_CREAT | O_EXCL):
	    createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_FAIL_IF_EXISTS;
	    break;
	case (O_CREAT | O_TRUNC):
	    createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS;
	    break;
	case O_CREAT:
	    createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS;
	    break;
	case O_TRUNC:
	case (O_TRUNC | O_EXCL):
	    createMode = OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS;
	    break;
	default:
	    createMode = OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS;
	    break;
    }

    /*
     * If the file is being created, get the file attributes from the
     * permissions argument, else use the existing file attributes.
     */

    if (mode & O_CREAT) {
        if (permissions & S_IWRITE) {
            flags = FILE_NORMAL;
        } else {
            flags = FILE_READONLY;
        }
    } else {
        FILESTATUS3 infoBuf;

	if (DosQueryPathInfo(fileName, FIL_STANDARD, &infoBuf, sizeof(infoBuf))
	        == NO_ERROR) {
	    flags = infoBuf.attrFile;
        } else {
	    flags = 0;
	}
    }

    /*
     * Set up the attributes so this file is not inherited by child processes.
     */

    accessMode |= OPEN_FLAGS_NOINHERIT;

    /*
     * Set up the file sharing mode.  We want to allow simultaneous access.
     */

    accessMode |= OPEN_SHARE_DENYNONE;

    /*
     * Now we get to create the file.
     */

    nativeName = Tcl_TranslateFileName(interp, fileName, &buffer);
    if (nativeName == NULL) {
	return NULL;
    }
    rc = DosOpen(nativeName, &handle, &exist, 0, flags, createMode,
                  accessMode, (PEAOP2)NULL);
    Tcl_DStringFree(&buffer);

    if (rc != NO_ERROR) {
	ULONG err = ERROR_SIGNAL_REFUSED;

	switch (rc) {
	    case ERROR_FILE_NOT_FOUND:
	    case ERROR_PATH_NOT_FOUND:
	        err = ERROR_FILE_NOT_FOUND;
	        break;
	    case ERROR_ACCESS_DENIED:
	    case ERROR_INVALID_ACCESS:
	    case ERROR_SHARING_VIOLATION:
	    case ERROR_CANNOT_MAKE:
	    case ERROR_OPEN_FAILED:
	        err = (mode & O_CREAT) ? ERROR_FILE_EXISTS
	                               : ERROR_FILE_NOT_FOUND;
	        break;
	}
        TclOS2ConvertError(err);
	if (interp != (Tcl_Interp *) NULL) {
            Tcl_AppendResult(interp, "couldn't open \"", fileName, "\": ",
                    Tcl_PosixError(interp), (char *) NULL);
        }
        return NULL;
    }

    file = Tcl_GetFile((ClientData) handle, TCL_OS2_FILE);

    sPtr = (FileState *) ckalloc((unsigned) sizeof(FileState));
    sPtr->asynch = 0;
    sPtr->append = (mode & O_APPEND) ? 1 : 0;
    readWriteMode = 0;
    /* WATCH IT!!!!
     * The symbol OPEN_ACCESS_READONLY is 0, so the & operator ALWAYS gives
     * 0, so the channel NEVER gets the file as argument => use Boolean
     * variable "readonly" that is set when O_A_R is applicable.
     */
    if (readonly) {
        readWriteMode |= TCL_READABLE;
        sPtr->inFile = file;
    } else {
        sPtr->inFile = (Tcl_File) NULL;
    }
    if (accessMode & OPEN_ACCESS_WRITEONLY) {
        readWriteMode |= TCL_WRITABLE;
        sPtr->outFile = file;
    } else {
        sPtr->outFile = (Tcl_File) NULL;
    }
    sprintf(channelName, "file%d", (int) Tcl_GetFileInfo(file, NULL));

    chan = Tcl_CreateChannel(&fileChannelType, channelName,
            (ClientData) sPtr, readWriteMode);
    if (chan == (Tcl_Channel) NULL) {
        if (interp != (Tcl_Interp *) NULL) {
            Tcl_AppendResult(interp, "could not open channel \"",
                    channelName, "\": ", Tcl_PosixError(interp),
                    (char *) NULL);
        }
        Tcl_FreeFile(file);
        rc = DosClose(handle);
        ckfree((char *) sPtr);
        return NULL;
    }

    if (seekFlag) {
        if (Tcl_Seek(chan, 0, SEEK_END) < 0) {
            if (interp != (Tcl_Interp *) NULL) {
                Tcl_AppendResult(interp, "could not seek to end of file on \"",
                        channelName, "\": ", Tcl_PosixError(interp),
                        (char *) NULL);
            }
            Tcl_Close((Tcl_Interp *) NULL, chan);
            return NULL;
        }
    }

    /*
     * Files have default translation of AUTO and ^Z eof char, which
     * means that a ^Z will be appended to them at close.
     */
    
    if (Tcl_SetChannelOption(interp, chan, "-translation", "auto") ==
            TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return (Tcl_Channel) NULL;
    }
    if (Tcl_SetChannelOption(NULL, chan, "-eofchar", "\032 {}") ==
            TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return (Tcl_Channel) NULL;
    }
    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * FileType --
 *
 *	Converts an OS/2 handle type to a Tcl file type
 *
 * Results:
 *	The Tcl file type corresponding to the given OS/2 handle type
 *	or -1 on error.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
FileType(h)
    HFILE h;		/* Convert the type of this handle to
                         * a Tcl file type. */
{
    ULONG type, attr;

    rc = DosQueryHType(h, &type, &attr);
    if (rc != NO_ERROR) return -1;
    switch (type & (HANDTYPE_FILE | HANDTYPE_DEVICE | HANDTYPE_PIPE)) {
    case HANDTYPE_DEVICE:
        return TCL_OS2_CONSOLE;
    case HANDTYPE_FILE:
        return TCL_OS2_FILE;
    case HANDTYPE_PIPE:
        return TCL_OS2_PIPE;
    default:
        return -1;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_MakeFileChannel --
 *
 *	Creates a Tcl_Channel from an existing platform specific file
 *	handle.
 *
 * Results:
 *	The Tcl_Channel created around the preexisting file.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_MakeFileChannel(inFile, outFile, mode)
    ClientData inFile;		/* OS level handle used for input. */
    ClientData outFile;		/* OS level handle used for output. */
    int mode;			/* ORed combination of TCL_READABLE and
                                 * TCL_WRITABLE to indicate whether inFile
                                 * and/or outFile are valid. */
{
    int fileUsed;
    Tcl_File inFd, outFd;
    char channelName[20];
    FileState *sPtr;
    Tcl_Channel chan;

    if (mode & TCL_READABLE) {
        sprintf(channelName, "file%d", (int) inFile);
        inFd = Tcl_GetFile(inFile, FileType((HFILE) inFile));
    } else {
        inFd = (Tcl_File) NULL;
    }
    
    if (mode & TCL_WRITABLE) {
        sprintf(channelName, "file%d", (int) outFile);
        outFd = Tcl_GetFile(outFile, FileType((HFILE) outFile));
    } else {
        outFd = (Tcl_File) NULL;
    }

    /*
     * See if a channel with the right Tcl_Files in it already exists. If
     * so, return it.
     */

    chan = TclFindFileChannel(inFd, outFd, &fileUsed);
    if (chan != (Tcl_Channel) NULL) {
        return chan;
    }

    /*
     * If one of the Tcl_Files is already used by another channel, do not
     * create a new channel containing it. This will avoid core dumps later
     * when the Tcl_File would be freed twice.
     */

    if (fileUsed) {
        return (Tcl_Channel) NULL;
    }

    sPtr = (FileState *) ckalloc((unsigned) sizeof(FileState));
    sPtr->asynch = 0;
    sPtr->append = 0;
    sPtr->inFile = inFd;
    sPtr->outFile = outFd;

    chan = Tcl_CreateChannel(&fileChannelType, channelName, (ClientData) sPtr,
                             mode);
    if (chan == (Tcl_Channel) NULL) {
        ckfree((char *) sPtr);
        return NULL;
    }

    /*
     * OS/2 files have AUTO translation mode and ^Z eof char, which
     * means that ^Z will be appended on close and accepted as EOF.
     */
    
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, chan, "-translation",
            "auto") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return (Tcl_Channel) NULL;
    }
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, chan, "-eofchar",
            "\032 {}") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return (Tcl_Channel) NULL;
    }
    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * TclCreateCommandChannel --
 *
 *	This function is called by Tcl_OpenCommandChannel to perform
 *	the platform specific channel initialization for a command
 *	channel.
 *
 * Results:
 *	Returns a new channel or NULL on failure.
 *
 * Side effects:
 *	Allocates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
TclCreateCommandChannel(readFile, writeFile, errorFile, numPids, pidPtr)
    Tcl_File readFile;		/* If non-null, gives the file for reading. */
    Tcl_File writeFile;		/* If non-null, gives the file for writing. */
    Tcl_File errorFile;		/* If non-null, gives the file where errors
				 * can be read. */
    int numPids;		/* The number of pids in the pid array. */
    int *pidPtr;		/* An array of process identifiers. */
{
    Tcl_Channel channel;
    char channelName[20];
    int channelId;
    int permissions;
    PipeState *statePtr = (PipeState *) ckalloc((unsigned) sizeof(PipeState));

    statePtr->asynch = 0;
    statePtr->readFile = readFile;
    statePtr->writeFile = writeFile;
    statePtr->errorFile = errorFile;
    statePtr->numPids = numPids;
    statePtr->pidPtr = pidPtr;

    /*
     * Use one of the fds associated with the channel as the
     * channel id.
     */

    if (readFile) {
	channelId = (int) Tcl_GetFileInfo(readFile, NULL);
    } else if (writeFile) {
	channelId = (int) Tcl_GetFileInfo(writeFile, NULL);
    } else if (errorFile) {
	channelId = (int) Tcl_GetFileInfo(errorFile, NULL);
    } else {
	channelId = 0;
    }

    permissions = 0;
    if (readFile != (Tcl_File) NULL) {
        permissions |= TCL_READABLE;
    }
    if (writeFile != (Tcl_File) NULL) {
        permissions |= TCL_WRITABLE;
    }

    /*
     * For backward compatibility with previous versions of Tcl, we
     * use "file%d" as the base name for pipes even though it would
     * be more natural to use "pipe%d".
     */

    sprintf(channelName, "file%d", channelId);
    channel = Tcl_CreateChannel(&pipeChannelType, channelName,
	                        (ClientData) statePtr, permissions);

    if (channel == NULL) {
	ckfree((char *)statePtr);
        return NULL;
    }

    /*
     * Pipes have AUTO translation mode on OS/2 and ^Z eof char, which
     * means that a ^Z will be appended to them at close. This is needed
     * for OS/2 programs that expect a ^Z at EOF.
     */

    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-translation",
            "auto") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, channel);
        return (Tcl_Channel) NULL;
    }
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-eofchar",
            "\032 {}") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, channel);
        return (Tcl_Channel) NULL;
    }
    return channel;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_PidCmd --
 *
 *	This procedure is invoked to process the "pid" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Tcl_PidCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Tcl_Channel chan;			/* The channel to get pids for. */
    Tcl_ChannelType *typePtr;
    PipeState *pipePtr;			/* The pipe state. */
    int i;				/* Loops over PIDs attached to the
                                         * pipe. */
    char string[50];			/* Temp buffer for string rep. of
                                         * PIDs attached to the pipe. */

    if (argc > 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " ?channelId?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (argc == 1) {
	sprintf(interp->result, "%ld", (long) getpid());
    } else {
        chan = Tcl_GetChannel(interp, argv[1], NULL);
        if (chan == (Tcl_Channel) NULL) {
	    return TCL_ERROR;
	}
	typePtr = Tcl_GetChannelType(chan);
	if (typePtr != &pipeChannelType) {
            return TCL_OK;
        }
        pipePtr = (PipeState *) Tcl_GetChannelInstanceData(chan);
        for (i = 0; i < pipePtr->numPids; i++) {
	    sprintf(string, "%d", pipePtr->pidPtr[i]);
	    Tcl_AppendElement(interp, string);
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TclGetDefaultStdChannel --
 *
 *	Constructs a channel for the specified standard OS handle.
 *
 * Results:
 *	Returns the specified default standard channel, or NULL.
 *
 * Side effects:
 *	May cause the creation of a standard channel and the underlying
 *	file.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
TclGetDefaultStdChannel(type)
    int type;			/* One of TCL_STDIN, TCL_STDOUT, TCL_STDERR. */
{
    Tcl_Channel channel;
    HFILE handle = 0;
    int mode = TCL_READABLE;
    char *bufMode = "line";

    switch (type) {
	case TCL_STDIN:
	    handle = (HFILE)0;
	    mode = TCL_READABLE;
	    bufMode = "line";
	    break;
	case TCL_STDOUT:
	    handle = (HFILE)1;
	    mode = TCL_WRITABLE;
	    bufMode = "line";
	    break;
	case TCL_STDERR:
	    handle = (HFILE)2;
	    mode = TCL_WRITABLE;
	    bufMode = "none";
	    break;
	default:
	    panic("TclGetDefaultStdChannel: Unexpected channel type");
	    break;
    }

    channel = Tcl_MakeFileChannel((ClientData)handle, (ClientData)handle, mode);

    /*
     * Set up the normal channel options for stdio handles.
     */

    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-translation",
            "auto") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, channel);
        return (Tcl_Channel) NULL;
    }
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-eofchar",
            "\032 {}") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, channel);
        return (Tcl_Channel) NULL;
    }
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-buffering",
            bufMode) == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, channel);
        return (Tcl_Channel) NULL;
    }
    return channel;
}

/*
 *----------------------------------------------------------------------
 *
 * TclGetAndDetachPids --
 *
 *	Stores a list of the command PIDs for a command channel in
 *	interp->result.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Modifies interp->result.
 *
 *----------------------------------------------------------------------
 */

void
TclGetAndDetachPids(interp, chan)
    Tcl_Interp *interp;
    Tcl_Channel chan;
{
    PipeState *pipePtr;
    Tcl_ChannelType *chanTypePtr;
    int i;
    char buf[20];

    /*
     * Punt if the channel is not a command channel.
     */

    chanTypePtr = Tcl_GetChannelType(chan);
    if (chanTypePtr != &pipeChannelType) {
        return;
    }

    pipePtr = (PipeState *) Tcl_GetChannelInstanceData(chan);
    for (i = 0; i < pipePtr->numPids; i++) {
        sprintf(buf, "%d", pipePtr->pidPtr[i]);
        Tcl_AppendElement(interp, buf);
        Tcl_DetachPids(1, &(pipePtr->pidPtr[i]));
    }
    if (pipePtr->numPids > 0) {
        ckfree((char *) pipePtr->pidPtr);
        pipePtr->numPids = 0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TclClosePipeFile --
 *
 *	This function is a simple wrapper for close on a file or
 *	pipe handle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Closes the HFILE and frees the Tcl_File.
 *
 *----------------------------------------------------------------------
 */

void
TclClosePipeFile(file)
    Tcl_File file;
{
    int type;
    HFILE handle = (HFILE) Tcl_GetFileInfo(file, &type);
    /*
     * Close the OS/2 handle first, since Tcl_FreeFile can call DosDelete
     * via the notifier data in case of a temp file.
     */

    switch (type) {
	case TCL_OS2_FILE:
	case TCL_OS2_PIPE:
	    rc = DosClose(handle);
	    break;
	default:
	    break;
    }
    Tcl_FreeFile(file);
}

/*
 *----------------------------------------------------------------------
 *
 * TcpBlockModeProc --
 *
 *      This procedure is invoked by the generic IO level to set blocking
 *      and nonblocking mode on a TCP socket based channel.
 *
 * Results:
 *      0 if successful, errno when failed.
 *
 * Side effects:
 *      Sets the device into blocking or nonblocking mode.
 *
 *----------------------------------------------------------------------
 */

        /* ARGSUSED */
static int
TcpBlockModeProc(instanceData, mode)
    ClientData instanceData;            /* Socket state. */
    int mode;                           /* The mode to set. Can be one of
                                         * TCL_MODE_BLOCKING or
                                         * TCL_MODE_NONBLOCKING. */
{
    TcpState *statePtr;
    int sock;
    int setting;

    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
#ifndef USE_FIONBIO
    setting = fcntl(sock, F_GETFL);
    if (mode == TCL_MODE_BLOCKING) {
        statePtr->flags &= (~(TCP_ASYNC_SOCKET));
        setting &= (~(O_NONBLOCK));
    } else {
        statePtr->flags |= TCP_ASYNC_SOCKET;
        setting |= O_NONBLOCK;
    }
    if (fcntl(sock, F_SETFL, setting) < 0) {
        return errno;
    }
#endif

#ifdef  USE_FIONBIO
    if (mode == TCL_MODE_BLOCKING) {
        statePtr->flags &= (~(TCP_ASYNC_SOCKET));
        setting = 0;
        if (ioctl(sock, (int) FIONBIO, &setting) == -1) {
            return errno;
        }
    } else {
        statePtr->flags |= TCP_ASYNC_SOCKET;
        setting = 1;
        if (ioctl(sock, (int) FIONBIO, &setting) == -1) {
            return errno;
        }
    }
#endif

    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * WaitForConnect --
 *
 *      Waits for a connection on an asynchronously opened socket to
 *      be completed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The socket is connected after this function returns.
 *
 *----------------------------------------------------------------------
 */

static int
WaitForConnect(statePtr, errorCodePtr)
    TcpState *statePtr;         /* State of the socket. */
    int *errorCodePtr;          /* Where to store errors? */
{
    int sock;                   /* The socket itself. */
    int timeOut;                /* How long to wait. */
    int state;                  /* Of calling TclWaitForFile. */
    int flags;                  /* fcntl flags for the socket. */

    /*
     * If an asynchronous connect is in progress, attempt to wait for it
     * to complete before reading.
     */

    if (statePtr->flags & TCP_ASYNC_CONNECT) {
        if (statePtr->flags & TCP_ASYNC_SOCKET) {
            timeOut = 0;
        } else {
            timeOut = -1;
        }
        errno = 0;
        state = TclWaitForFile(statePtr->sock, TCL_WRITABLE | TCL_EXCEPTION,
                timeOut);
        if (!(statePtr->flags & TCP_ASYNC_SOCKET)) {
            sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
#ifndef USE_FIONBIO
            flags = fcntl(sock, F_GETFL);
            flags &= (~(O_NONBLOCK));
            (void) fcntl(sock, F_SETFL, flags);
#endif

#ifdef  USE_FIONBIO
            flags = 0;
            (void) ioctl(sock, FIONBIO, &flags);
#endif
        }
        if (state & TCL_EXCEPTION) {
            return -1;
        }
        if (state & TCL_WRITABLE) {
            statePtr->flags &= (~(TCP_ASYNC_CONNECT));
        } else if (timeOut == 0) {
            *errorCodePtr = errno = EWOULDBLOCK;
            return -1;
        }
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpInputProc --
 *
 *      This procedure is invoked by the generic IO level to read input
 *      from a TCP socket based channel.
 *
 *      NOTE: We cannot share code with FilePipeInputProc because here
 *      we must use recv to obtain the input from the channel, not read.
 *
 * Results:
 *      The number of bytes read is returned or -1 on error. An output
 *      argument contains the POSIX error code on error, or zero if no
 *      error occurred.
 *
 * Side effects:
 *      Reads input from the input device of the channel.
 *
 *----------------------------------------------------------------------
 */

        /* ARGSUSED */
static int
TcpInputProc(instanceData, buf, bufSize, errorCodePtr)
    ClientData instanceData;            /* Socket state. */
    char *buf;                          /* Where to store data read. */
    int bufSize;                        /* How much space is available
                                         * in the buffer? */
    int *errorCodePtr;                  /* Where to store error code. */
{
    TcpState *statePtr;                 /* The state of the socket. */
    int sock;                           /* The OS handle. */
    int bytesRead;                      /* How many bytes were read? */
    int state;                          /* Of waiting for connection. */

    *errorCodePtr = 0;
    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);

    state = WaitForConnect(statePtr, errorCodePtr);
    if (state != 0) {
        return -1;
    }
    bytesRead = recv(sock, buf, bufSize, 0);
    if (bytesRead > -1) {
        return bytesRead;
    }
    if (errno == ECONNRESET) {

        /*
         * Turn ECONNRESET into a soft EOF condition.
         */

        return 0;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpOutputProc --
 *
 *      This procedure is invoked by the generic IO level to write output
 *      to a TCP socket based channel.
 *
 *      NOTE: We cannot share code with FilePipeOutputProc because here
 *      we must use send, not write, to get reliable error reporting.
 *
 * Results:
 *      The number of bytes written is returned. An output argument is
 *      set to a POSIX error code if an error occurred, or zero.
 *
 * Side effects:
 *      Writes output on the output device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
TcpOutputProc(instanceData, buf, toWrite, errorCodePtr)
    ClientData instanceData;            /* Socket state. */
    char *buf;                          /* The data buffer. */
    int toWrite;                        /* How many bytes to write? */
    int *errorCodePtr;                  /* Where to store error code. */
{
    TcpState *statePtr;
    int written;
    int sock;                           /* OS level socket. */
    int state;                          /* Of waiting for connection. */

    *errorCodePtr = 0;
    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
    state = WaitForConnect(statePtr, errorCodePtr);
    if (state != 0) {
        return -1;
    }
    written = send(sock, buf, toWrite, 0);
    if (written > -1) {
        return written;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpCloseProc --
 *
 *      This procedure is invoked by the generic IO level to perform
 *      channel-type-specific cleanup when a TCP socket based channel
 *      is closed.
 *
 * Results:
 *      0 if successful, the value of errno if failed.
 *
 * Side effects:
 *      Closes the socket of the channel.
 *
 *----------------------------------------------------------------------
 */

        /* ARGSUSED */
static int
TcpCloseProc(instanceData, interp)
    ClientData instanceData;    /* The socket to close. */
    Tcl_Interp *interp;         /* For error reporting - unused. */
{
    TcpState *statePtr;
    Tcl_File sockFile;
    int sock;
    int errorCode = 0;

    statePtr = (TcpState *) instanceData;
    sockFile = statePtr->sock;
    sock = (int) Tcl_GetFileInfo(sockFile, NULL);

    /*
     * Delete a file handler that may be active for this socket if this
     * is a server socket - the file handler was created automatically
     * by Tcl as part of the mechanism to accept new client connections.
     * Channel handlers are already deleted in the generic IO channel
     * closing code that called this function, so we do not have to
     * delete them here.
     */

    Tcl_DeleteFileHandler(sockFile);

    ckfree((char *) statePtr);

    /*
     * We assume that inFile==outFile==sockFile and so
     * we only clean up sockFile.
     */

    Tcl_FreeFile(sockFile);

    if (close(sock) < 0) {
        errorCode = errno;
    }

    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpGetOptionProc --
 *
 *      Computes an option value for a TCP socket based channel, or a
 *      list of all options and their values.
 *
 *      Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *      A standard Tcl result. The value of the specified option or a
 *      list of all options and their values is returned in the
 *      supplied DString.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
TcpGetOptionProc(instanceData, optionName, dsPtr)
    ClientData instanceData;            /* Socket state. */
    char *optionName;                   /* Name of the option to
                                         * retrieve the value for, or
                                         * NULL to get all options and
                                         * their values. */
    Tcl_DString *dsPtr;                 /* Where to store the computed
                                         * value; initialized by caller. */
{
    TcpState *statePtr;
    struct sockaddr_in sockname;
    struct sockaddr_in peername;
    struct hostent *hostEntPtr;
    int sock;
    int size = sizeof(struct sockaddr_in);
    size_t len = 0;
    char buf[128];

    statePtr = (TcpState *) instanceData;
    sock = (int) Tcl_GetFileInfo(statePtr->sock, NULL);
    if (optionName != (char *) NULL) {
        len = strlen(optionName);
    }

    if ((len == 0) ||
            ((len > 1) && (optionName[1] == 'p') &&
                    (strncmp(optionName, "-peername", len) == 0))) {
        if (getpeername(sock, (struct sockaddr *) &peername, &size) >= 0) {
            if (len == 0) {
                Tcl_DStringAppendElement(dsPtr, "-peername");
                Tcl_DStringStartSublist(dsPtr);
            }
            Tcl_DStringAppendElement(dsPtr, inet_ntoa(peername.sin_addr));
            hostEntPtr = gethostbyaddr((char *) &(peername.sin_addr),
                    sizeof(peername.sin_addr), AF_INET);
            if (hostEntPtr != (struct hostent *) NULL) {
                Tcl_DStringAppendElement(dsPtr, hostEntPtr->h_name);
            } else {
                Tcl_DStringAppendElement(dsPtr, inet_ntoa(peername.sin_addr));
            }
            sprintf(buf, "%d", ntohs(peername.sin_port));
            Tcl_DStringAppendElement(dsPtr, buf);
            if (len == 0) {
                Tcl_DStringEndSublist(dsPtr);
            } else {
                return TCL_OK;
            }
        }
    }

    if ((len == 0) ||
            ((len > 1) && (optionName[1] == 's') &&
                    (strncmp(optionName, "-sockname", len) == 0))) {
        if (getsockname(sock, (struct sockaddr *) &sockname, &size) >= 0) {
            if (len == 0) {
                Tcl_DStringAppendElement(dsPtr, "-sockname");
                Tcl_DStringStartSublist(dsPtr);
            }
            Tcl_DStringAppendElement(dsPtr, inet_ntoa(sockname.sin_addr));
            hostEntPtr = gethostbyaddr((char *) &(sockname.sin_addr),
                    sizeof(peername.sin_addr), AF_INET);
            if (hostEntPtr != (struct hostent *) NULL) {
                Tcl_DStringAppendElement(dsPtr, hostEntPtr->h_name);
            } else {
                Tcl_DStringAppendElement(dsPtr, inet_ntoa(sockname.sin_addr));
            }
            sprintf(buf, "%d", ntohs(sockname.sin_port));
            Tcl_DStringAppendElement(dsPtr, buf);
            if (len == 0) {
                Tcl_DStringEndSublist(dsPtr);
            } else {
                return TCL_OK;
            }
        }
    }

    if (len > 0) {
        Tcl_SetErrno(EINVAL);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpWatchProc --
 *
 *      Initialize the notifier to watch Tcl_Files from this channel.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Sets up the notifier so that a future event on the channel will
 *      be seen by Tcl.
 *
 *----------------------------------------------------------------------
 */

static void
TcpWatchProc(instanceData, mask)
    ClientData instanceData;            /* The socket state. */
    int mask;                           /* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABEL and TCL_EXCEPTION. */
{
    TcpState *statePtr = (TcpState *) instanceData;

    Tcl_WatchFile(statePtr->sock, mask);
}

/*
 *----------------------------------------------------------------------
 *
 * TcpReadyProc --
 *
 *      Called by the notifier to check whether events of interest are
 *      present on the channel.
 *
 * Results:
 *      Returns OR-ed combination of TCL_READABLE, TCL_WRITABLE and
 *      TCL_EXCEPTION to indicate which events of interest are present.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
TcpReadyProc(instanceData, mask)
    ClientData instanceData;            /* The socket state. */
    int mask;                           /* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    TcpState *statePtr = (TcpState *) instanceData;

    return Tcl_FileReady(statePtr->sock, mask);
}

/*
 *----------------------------------------------------------------------
 *
 * TcpGetProc --
 *
 *      Called from Tcl_GetChannelFile to retrieve Tcl_Files from inside
 *      a TCP socket based channel.
 *
 * Results:
 *      The appropriate Tcl_File or NULL if not present.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

        /* ARGSUSED */
static Tcl_File
TcpGetProc(instanceData, direction)
    ClientData instanceData;            /* The socket state. */
    int direction;                      /* Which Tcl_File to retrieve? */
{
    TcpState *statePtr = (TcpState *) instanceData;

    return statePtr->sock;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateSocket --
 *
 *      This function opens a new socket in client or server mode
 *      and initializes the TcpState structure.
 *
 * Results:
 *      Returns a new TcpState, or NULL with an error in interp->result,
 *      if interp is not NULL.
 *
 * Side effects:
 *      Opens a socket.
 *
 *----------------------------------------------------------------------
 */

static TcpState *
CreateSocket(interp, port, host, server, myaddr, myport, async)
    Tcl_Interp *interp;         /* For error reporting; can be NULL. */
    int port;                   /* Port number to open. */
    char *host;                 /* Name of host on which to open port.
                                 * NULL implies INADDR_ANY */
    int server;                 /* 1 if socket should be a server socket,
                                 * else 0 for a client socket. */
    char *myaddr;               /* Optional client-side address */
    int myport;                 /* Optional client-side port */
    int async;                  /* If nonzero and creating a client socket,
                                 * attempt to do an async connect. Otherwise
                                 * do a synchronous connect or bind. */
{
    int status, sock, asyncConnect, curState, origState;
    struct sockaddr_in sockaddr;        /* socket address */
    struct sockaddr_in mysockaddr;      /* Socket address for client */
    TcpState *statePtr;

    sock = -1;
    origState = 0;
    if (! CreateSocketAddress(&sockaddr, host, port)) {
        goto addressError;
    }
    if ((myaddr != NULL || myport != 0) &&
            ! CreateSocketAddress(&mysockaddr, myaddr, myport)) {
        goto addressError;
    }

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
        goto addressError;
    }

    /*
     * Set the close-on-exec flag so that the socket will not get
     * inherited by child processes.
     */

    fcntl(sock, F_SETFD, FD_CLOEXEC);

    /*
     * Set kernel space buffering
     */

    TclSockMinimumBuffers(sock, SOCKET_BUFSIZE);

    asyncConnect = 0;
    status = 0;
    if (server) {

        /*
         * Set up to reuse server addresses automatically and bind to the
         * specified port.
         */

        status = 1;
        (void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &status,
                sizeof(status));
        status = bind(sock, (struct sockaddr *) &sockaddr,
                sizeof(struct sockaddr));
        if (status != -1) {
            status = listen(sock, SOMAXCONN);
        }
    } else {
        if (myaddr != NULL || myport != 0) {
            status = 1;
            (void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
                    (char *) &curState, sizeof(status));
            status = bind(sock, (struct sockaddr *) &mysockaddr,
                    sizeof(struct sockaddr));
            if (status < 0) {
                goto bindError;
            }
        }

        /*
         * Attempt to connect. The connect may fail at present with an
         * EINPROGRESS but at a later time it will complete. The caller
         * will set up a file handler on the socket if she is interested in
         * being informed when the connect completes.
         */

        if (async) {
#ifndef USE_FIONBIO
            origState = fcntl(sock, F_GETFL);
            curState = origState | O_NONBLOCK;
            status = fcntl(sock, F_SETFL, curState);
#endif

#ifdef  USE_FIONBIO
            curState = 1;
            status = ioctl(sock, FIONBIO, &curState);
#endif
        } else {
            status = 0;
        }
        if (status > -1) {
            status = connect(sock, (struct sockaddr *) &sockaddr,
                    sizeof(sockaddr));
            if (status < 0) {
                if (errno == EINPROGRESS) {
                    asyncConnect = 1;
                    status = 0;
                }
            }
        }
    }

bindError:
    if (status < 0) {
        if (interp != NULL) {
            Tcl_AppendResult(interp, "couldn't open socket: ",
                    Tcl_PosixError(interp), (char *) NULL);
        }
        if (sock != -1) {
            close(sock);
        }
        return NULL;
    }

    /*
     * Allocate a new TcpState for this socket.
     */

    statePtr = (TcpState *) ckalloc((unsigned) sizeof(TcpState));
    statePtr->flags = 0;
    if (asyncConnect) {
        statePtr->flags = TCP_ASYNC_CONNECT;
    }
    statePtr->sock = Tcl_GetFile((ClientData) sock, TCL_OS2_SOCKET);

    return statePtr;

addressError:
    if (sock != -1) {
        close(sock);
    }
    if (interp != NULL) {
        Tcl_AppendResult(interp, "couldn't open socket: ",
                Tcl_PosixError(interp), (char *) NULL);
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateSocketAddress --
 *
 *      This function initializes a sockaddr structure for a host and port.
 *
 * Results:
 *      1 if the host was valid, 0 if the host could not be converted to
 *      an IP address.
 *
 * Side effects:
 *      Fills in the *sockaddrPtr structure.
 *
 *----------------------------------------------------------------------
 */

static int
CreateSocketAddress(sockaddrPtr, host, port)
    struct sockaddr_in *sockaddrPtr;    /* Socket address */
    char *host;                         /* Host.  NULL implies INADDR_ANY */
    int port;                           /* Port number */
{
    struct hostent *hostent;            /* Host database entry */
    struct in_addr addr;                /* For 64/32 bit madness */

    (void) memset((VOID *) sockaddrPtr, '\0', sizeof(struct sockaddr_in));
    sockaddrPtr->sin_family = AF_INET;
    sockaddrPtr->sin_port = htons((unsigned short) (port & 0xFFFF));
    if (host == NULL) {
        addr.s_addr = INADDR_ANY;
    } else {
        addr.s_addr = inet_addr(host);
        if (addr.s_addr == (unsigned long) -1) {
            hostent = gethostbyname(host);
            if (hostent != NULL) {
                memcpy((VOID *) &addr,
                        (VOID *) hostent->h_addr_list[0],
                        (size_t) hostent->h_length);
            } else {
#ifdef  EHOSTUNREACH
                errno = EHOSTUNREACH;
#else
#ifdef ENXIO
                errno = ENXIO;
#endif
#endif
                return 0;       /* error */
            }
        }
    }

    /*
     * NOTE: On 64 bit machines the assignment below is rumored to not
     * do the right thing. Please report errors related to this if you
     * observe incorrect behavior on 64 bit machines such as DEC Alphas.
     * Should we modify this code to do an explicit memcpy?
     */

    sockaddrPtr->sin_addr.s_addr = addr.s_addr;
    return 1;   /* Success. */
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenTcpClient --
 *
 *      Opens a TCP client socket and creates a channel around it.
 *
 * Results:
 *      The channel or NULL if failed.  An error message is returned
 *      in the interpreter on failure.
 *
 * Side effects:
 *      Opens a client socket and creates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenTcpClient(interp, port, host, myaddr, myport, async)
    Tcl_Interp *interp;                 /* For error reporting; can be NULL. */
    int port;                           /* Port number to open. */
    char *host;                         /* Host on which to open port. */
    char *myaddr;                       /* Client-side address */
    int myport;                         /* Client-side port */
    int async;                          /* If nonzero, attempt to do an
                                         * asynchronous connect. Otherwise
                                         * we do a blocking connect. */
{
    Tcl_Channel chan;
    TcpState *statePtr;
    char channelName[20];

    /*
     * Create a new client socket and wrap it in a channel.
     */

    statePtr = CreateSocket(interp, port, host, 0, myaddr, myport, async);
    if (statePtr == NULL) {
        return NULL;
    }

    statePtr->acceptProc = NULL;
    statePtr->acceptProcData = (ClientData) NULL;

    sprintf(channelName, "sock%d",
            (int) Tcl_GetFileInfo(statePtr->sock, NULL));

    chan = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));
    if (Tcl_SetChannelOption(interp, chan, "-translation", "auto crlf") ==
            TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return NULL;
    }
    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_MakeTcpClientChannel --
 *
 *      Creates a Tcl_Channel from an existing client TCP socket.
 *
 * Results:
 *      The Tcl_Channel wrapped around the preexisting TCP socket.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_MakeTcpClientChannel(sock)
    ClientData sock;            /* The socket to wrap up into a channel. */
{
    TcpState *statePtr;
    Tcl_File sockFile;
    char channelName[20];
    Tcl_Channel chan;

    sockFile = Tcl_GetFile(sock, TCL_OS2_SOCKET);
    statePtr = (TcpState *) ckalloc((unsigned) sizeof(TcpState));
    statePtr->sock = sockFile;
    statePtr->acceptProc = NULL;
    statePtr->acceptProcData = (ClientData) NULL;

    sprintf(channelName, "sock%d", (int) sock);

    chan = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, chan, "-translation",
            "auto crlf") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return NULL;
    }
    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenTcpServer --
 *
 *      Opens a TCP server socket and creates a channel around it.
 *
 * Results:
 *      The channel or NULL if failed. If an error occurred, an
 *      error message is left in interp->result if interp is
 *      not NULL.
 *
 * Side effects:
 *      Opens a server socket and creates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenTcpServer(interp, port, myHost, acceptProc, acceptProcData)
    Tcl_Interp *interp;                 /* For error reporting - may be
                                         * NULL. */
    int port;                           /* Port number to open. */
    char *myHost;                       /* Name of local host. */
    Tcl_TcpAcceptProc *acceptProc;      /* Callback for accepting connections
                                         * from new clients. */
    ClientData acceptProcData;          /* Data for the callback. */
{
    Tcl_Channel chan;
    TcpState *statePtr;
    char channelName[20];

    /*
     * Create a new client socket and wrap it in a channel.
     */

    statePtr = CreateSocket(interp, port, myHost, 1, NULL, 0, 0);
    if (statePtr == NULL) {
        return NULL;
    }

    statePtr->acceptProc = acceptProc;
    statePtr->acceptProcData = acceptProcData;

    /*
     * Set up the callback mechanism for accepting connections
     * from new clients.
     */

    Tcl_CreateFileHandler(statePtr->sock, TCL_READABLE, TcpAccept,
            (ClientData) statePtr);
    sprintf(channelName, "sock%d",
            (int) Tcl_GetFileInfo(statePtr->sock, NULL));
    chan = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, 0);
    return chan;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpAccept --
 *      Accept a TCP socket connection.  This is called by the event loop.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Creates a new connection socket. Calls the registered callback
 *      for the connection acceptance mechanism.
 *
 *----------------------------------------------------------------------
 */

        /* ARGSUSED */
static void
TcpAccept(data, mask)
    ClientData data;                    /* Callback token. */
    int mask;                           /* Not used. */
{
    TcpState *sockState;                /* Client data of server socket. */
    int newsock;                        /* The new client socket */
    Tcl_File newFile;                   /* Its file. */
    TcpState *newSockState;             /* State for new socket. */
    struct sockaddr_in addr;            /* The remote address */
    int len;                            /* For accept interface */
    Tcl_Channel chan;                   /* Channel instance created. */
    char channelName[20];

    sockState = (TcpState *) data;

    len = sizeof(struct sockaddr_in);
    newsock = accept((int) Tcl_GetFileInfo(sockState->sock, NULL),
            (struct sockaddr *)&addr, &len);
    if (newsock < 0) {
        return;
    }

    /*
     * Set close-on-exec flag to prevent the newly accepted socket from
     * being inherited by child processes.
     */

    (void) fcntl(newsock, F_SETFD, FD_CLOEXEC);

    newFile = Tcl_GetFile((ClientData) newsock, TCL_OS2_SOCKET);
    if (newFile) {
        newSockState = (TcpState *) ckalloc((unsigned) sizeof(TcpState));

        newSockState->flags = 0;
        newSockState->sock = newFile;
        newSockState->acceptProc = (Tcl_TcpAcceptProc *) NULL;
        newSockState->acceptProcData = (ClientData) NULL;

        sprintf(channelName, "sock%d", (int) newsock);
        chan = Tcl_CreateChannel(&tcpChannelType, channelName,
                (ClientData) newSockState, (TCL_READABLE | TCL_WRITABLE));
        if (chan == (Tcl_Channel) NULL) {
            ckfree((char *) newSockState);
            close(newsock);
            Tcl_FreeFile(newFile);
        } else {
            if (Tcl_SetChannelOption((Tcl_Interp *) NULL, chan, "-translation",
                    "auto crlf") == TCL_ERROR) {
                Tcl_Close((Tcl_Interp *) NULL, chan);
            }
            if (sockState->acceptProc != (Tcl_TcpAcceptProc *) NULL) {
                (sockState->acceptProc) (sockState->acceptProcData, chan,
                        inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
            }
        }
    }
}
