/*
 * glyphs.c - Opreations for Character Glyphs
 */

#include <stdio.h>
#include <stdlib.h>

#define INCL_PM
#include <os2.h>

#include "xfont.h"
#include "xfontres.h"

/*
 * bitmapFromGlyph - create bitmap from Glyph Data
 */

static  HBITMAP bitmapFromGlyph(HWND hwnd, CharInfoPtr pci)
{
    HAB     hab  ;
    HDC     hdc  ;
    HPS     hps  ;
    SIZEL   size ;
    HBITMAP hbm  ;
    PBYTE   ptr  ;
    int     scan, len ;
    PBITMAPINFO2  pbm ;

    if (hwnd == NULLHANDLE || pci == NULL) {
        return NULLHANDLE ;
    }
    
    len = sizeof(BITMAPINFO2) + sizeof(RGB) ;
    if ((pbm = (PBITMAPINFO2) malloc(len)) == NULL) {
        return NULLHANDLE ;
    }
    memset(pbm, 0, len) ;

    size.cx = pci->metrics.rightSideBearing - pci->metrics.leftSideBearing ;
    size.cy = pci->metrics.ascent + pci->metrics.descent ;

    pbm->cbFix = sizeof(BITMAPINFOHEADER2) ;
    pbm->cx = size.cx  ;
    pbm->cy = size.cy  ;
    pbm->cPlanes = 1   ;
    pbm->cBitCount = 1 ;
    
    pbm->argbColor[0].bBlue  = 0x00 ;
    pbm->argbColor[0].bGreen = 0x00 ;
    pbm->argbColor[0].bRed   = 0x00 ;
    pbm->argbColor[1].bBlue  = 0xff ;
    pbm->argbColor[1].bGreen = 0xff ;
    pbm->argbColor[1].bRed   = 0xff ;

    hab = WinQueryAnchorBlock(hwnd) ;
    hdc = DevOpenDC(hab, OD_MEMORY, "*", 0, NULL, NULLHANDLE) ;
    hps = GpiCreatePS(hab, hdc, &size,
            PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC) ;

    hbm = GpiCreateBitmap(hps, (PBITMAPINFOHEADER2) pbm, 0, NULL, NULL) ;
    GpiSetBitmap(hps, hbm) ;

    len = (size.cx + 7) >> 3 ;
    ptr = pci->glyph ;
    for (scan = size.cy - 1 ; scan >= 0 ; scan--) {
        GpiSetBitmapBits(hps, scan, 1, ptr, pbm) ;
	ptr += len ;
    }

    GpiDestroyPS(hps) ;
    DevCloseDC(hdc) ;
    free(pbm) ;
    
    return hbm ;
}

/*
 * bitmapWithReverse - create reverse bitmap from Glyph Data
 */

static  HBITMAP bitmapWithReverse(HWND hwnd, CharInfoPtr pci)
{
    HAB     hab  ;
    HDC     hdc  ;
    HPS     hps  ;
    SIZEL   size ;
    HBITMAP hbm  ;
    PBYTE   ptr  ;
    int     scan, len ;
    PBITMAPINFO2  pbm ;

    if (hwnd == NULLHANDLE || pci == NULL) {
        return NULLHANDLE ;
    }
    
    len = sizeof(BITMAPINFO2) + sizeof(RGB) ;
    if ((pbm = (PBITMAPINFO2) malloc(len)) == NULL) {
        return NULLHANDLE ;
    }
    memset(pbm, 0, len) ;

    size.cx = pci->metrics.rightSideBearing - pci->metrics.leftSideBearing ;
    size.cy = pci->metrics.ascent + pci->metrics.descent ;

    pbm->cbFix = sizeof(BITMAPINFOHEADER2) ;
    pbm->cx = size.cx  ;
    pbm->cy = size.cy  ;
    pbm->cPlanes = 1   ;
    pbm->cBitCount = 1 ;
    
    pbm->argbColor[0].bBlue  = 0xff ;
    pbm->argbColor[0].bGreen = 0xff ;
    pbm->argbColor[0].bRed   = 0xff ;
    pbm->argbColor[1].bBlue  = 0x00 ;
    pbm->argbColor[1].bGreen = 0x00 ;
    pbm->argbColor[1].bRed   = 0x00 ;

    hab = WinQueryAnchorBlock(hwnd) ;
    hdc = DevOpenDC(hab, OD_MEMORY, "*", 0, NULL, NULLHANDLE) ;
    hps = GpiCreatePS(hab, hdc, &size,
            PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC) ;

    hbm = GpiCreateBitmap(hps, (PBITMAPINFOHEADER2) pbm, 0, NULL, NULL) ;
    GpiSetBitmap(hps, hbm) ;

    len = (size.cx + 7) >> 3 ;
    ptr = pci->glyph ;
    for (scan = size.cy - 1 ; scan >= 0 ; scan--) {
        GpiSetBitmapBits(hps, scan, 1, ptr, pbm) ;
	ptr += len ;
    }

    GpiDestroyPS(hps) ;
    DevCloseDC(hdc) ;
    free(pbm) ;
    
    return hbm ;
}

/*
 * bitmapIntoClipboard
 */

static  void    bitmapIntoClipboard(HWND hwnd, CharInfoPtr pci, BOOL revs)
{
    HAB     hab ;
    HBITMAP hbm ;
    
    if (hwnd == NULLHANDLE || pci == NULL) {
        return ;
    }
    if (revs == FALSE) {
        hbm = bitmapFromGlyph(hwnd, pci) ;
    } else {
        hbm = bitmapWithReverse(hwnd, pci) ;
    }
    if (hbm == NULLHANDLE) {
        return ;
    }

    hab = WinQueryAnchorBlock(hwnd) ;

    WinOpenClipbrd(hab)  ;
    WinEmptyClipbrd(hab) ;
    WinSetClipbrdData(hab, (ULONG) hbm, CF_BITMAP, CFI_HANDLE) ;
    WinCloseClipbrd(hab) ;
}
 
/*
 * Magnify Control Block
 */

typedef struct _MAGREC {
    CharInfoPtr pci    ;
    xCharInfo   max    ;
    xCharInfo   chr    ;
    HBITMAP     hbm    ;
    HWND        frame  ;
    HWND        client ;
    HWND        menu   ;
} MAGREC, *MAGPTR ;

static  HWND    hwndMenu  = NULLHANDLE ;

/*
 * magnifyDraw - Draw Magnified Glyph
 */

static  void    magnifyDraw(HWND hwnd, MAGPTR mag, HPS hps)
{
    RECTL   rct, cbox ;
    LONG    mx, my, cx, cy, ox, oy ;
    double  x, y, xscale, yscale, sizex, sizey, stepx, stepy ;
    POINTL  pts, pte ;
    
    WinQueryWindowRect(hwnd, &rct) ;
    WinFillRect(hps, &rct, CLR_BACKGROUND) ;

    if (mag == NULL) {
        return ;
    }

    /*
     * Calc. Scaling Factor 
     */
     
    mx = mag->max.rightSideBearing - mag->max.leftSideBearing ;
    my = mag->max.ascent + mag->max.descent ;
    xscale = (double) (rct.xRight - rct.xLeft) / (double) mx ;
    yscale = (double) (rct.yTop - rct.yBottom) / (double) my ;

    /*
     * Scale Current Glyph
     */
    
    cx = mag->chr.rightSideBearing - mag->chr.leftSideBearing ;
    cy = mag->chr.ascent + mag->chr.descent ;
    cbox.xLeft   = 0 ;
    cbox.yBottom = 0 ;
    cbox.xRight  = (LONG) ((double) cx * xscale + 0.5) ;
    cbox.yTop    = (LONG) ((double) cy * yscale + 0.5) ;

    /*
     * Adjust Base Position
     */
     
    ox = mag->chr.leftSideBearing - mag->max.leftSideBearing ;
    oy = mag->max.descent - mag->chr.descent ;
    cbox.xLeft   += (LONG) ((double) ox * xscale + 0.5) ;
    cbox.xRight  += (LONG) ((double) ox * xscale + 0.5) ;
    cbox.yTop    += (LONG) ((double) oy * yscale + 0.5) ;
    cbox.yBottom += (LONG) ((double) oy * yscale + 0.5) ;

    /*
     * Draw Bitmap with Scaling
     */
     
    if (mag->hbm) {
        WinDrawBitmap(hps, mag->hbm, NULL, (PPOINTL) &cbox,
	    CLR_NEUTRAL, CLR_BACKGROUND, DBM_STRETCH) ;
    }

    /*
     * Draw Glids
     */

    sizex = (double) (rct.xRight - rct.xLeft) ;
    sizey = (double) (rct.yTop - rct.yBottom) ;
    stepx = sizex / (double) mx ;
    stepy = sizey / (double) my ;
    
    pts.y = 0 ; pte.y = (LONG) sizey ;
    for (x = 0, ox = -mag->max.leftSideBearing ; x < sizex ; x += stepx, ox--) {
        pts.x = pte.x = (LONG) x ;
	if (ox != 0) {
	    GpiSetColor(hps, CLR_PALEGRAY) ;
	} else {
	    GpiSetColor(hps, CLR_DARKGRAY) ;
	}
	GpiMove(hps, &pts) ;
	GpiLine(hps, &pte) ;
    }
    
    pts.x = 0 ; pte.x = (LONG) sizex ;
    for (y = 0, oy = mag->max.descent ; y < sizey ; y += stepy, oy--) {
        pts.y = pte.y = (LONG) y ;
	if (oy != 0) {
	    GpiSetColor(hps, CLR_PALEGRAY) ;
	} else {
	    GpiSetColor(hps, CLR_DARKGRAY) ;
	}
	GpiMove(hps, &pts) ;
	GpiLine(hps, &pte) ;
    }
}

/*
 * magnifySelect - Select Request on Magnify Window
 */

static  void    magnifySelect(HWND hwnd, MAGPTR mag, MPARAM mp1, MPARAM mp2)
{
    WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(IDM_MAGN_MOVE), NULL) ;
}

/*
 * magnifyAction - Action Request on Magnify Window
 */

static  void    magnifyAction(HWND hwnd, MAGPTR mag, MPARAM mp1, MPARAM mp2)
{
    WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(IDM_MAGN_EXIT), NULL) ;
}

/*
 * magnifyPopup - Popup Menu
 */

static  void    magnifyPopup(HWND hwnd, MAGPTR mag, MPARAM mp1, MPARAM mp2)
{
    int     x, y ;
    ULONG   opts ;

    if (mag == NULL || mag->menu == NULLHANDLE) {
        return ;
    }
    
    x = SHORT1FROMMP(mp1) ;
    y = SHORT2FROMMP(mp1) ;
    opts = PU_POSITIONONITEM| PU_HCONSTRAIN | PU_VCONSTRAIN |
        PU_MOUSEBUTTON2DOWN |
	PU_KEYBOARD | PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2 ;

    WinPopupMenu(hwnd, hwnd, mag->menu, x, y, IDM_MAGN_MOVE, opts) ;
}

/*
 * magnifyClip - Copy to Clipboard
 */
 
static  void    magnifyClip(HWND hwnd, MAGPTR mag, BOOL revs)
{
    HBITMAP hbm ;
    
    if (mag == NULL || mag->pci == NULL) {
        return ;
    }
    bitmapIntoClipboard(hwnd, mag->pci, revs) ;
}

/*
 * magnifyMove - Move Window Around
 */
 
static  void    magnifyMove(HWND hwnd, MAGPTR mag)
{
    WinSendMsg(WinQueryWindow(hwnd, QW_PARENT),
            WM_TRACKFRAME, MPFROMLONG(TF_MOVE), NULL) ;
}

/*
 * magnifyExit - Exit
 */
 
static  void    magnifyExit(HWND hwnd, MAGPTR mag)
{
    WinDestroyWindow(mag->frame) ;
}

/*
 * MagnifyWndProc - Window Procedure for Glyph Magnifier
 */
 
MRESULT EXPENTRY MagnifyWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MAGPTR  mag ;
    HPS     hps ;
    
    mag = (MAGPTR) WinQueryWindowPtr(hwnd, 0) ;
    
    switch (msg) {

    case WM_MAGN_INIT :
        mag = (MAGPTR) PVOIDFROMMP(mp1) ;
	mag->menu = WinLoadMenu(hwnd, NULLHANDLE, ID_POPMAGN) ;
	WinSetWindowPtr(hwnd, 0, (PVOID) mag) ;
	return (MRESULT) 0 ;

    case WM_DESTROY :
        if (mag->hbm) GpiDeleteBitmap(mag->hbm) ;
        free(mag) ;
	return (MRESULT) 0 ;
	
    case WM_SIZE :
        WinInvalidateRect(hwnd, NULL, FALSE) ;
	return (MRESULT) 0 ;

    case WM_PAINT :
        hps = WinBeginPaint(hwnd, NULLHANDLE, NULL) ;
	magnifyDraw(hwnd, mag, hps) ;
	WinEndPaint(hps) ;
	return (MRESULT) 0 ;
	
    case WM_BUTTON1DOWN :
        magnifySelect(hwnd, mag, mp1, mp2) ;
        return WinDefWindowProc(hwnd, msg, mp1, mp2) ;        

    case WM_BUTTON2DOWN :
        magnifyPopup(hwnd, mag, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_BUTTON1DBLCLK :
    case WM_BUTTON2DBLCLK :
        magnifyAction(hwnd, mag, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_COMMAND :

        switch (SHORT1FROMMP(mp1)) {
	case IDM_MAGN_CLIP :
	    magnifyClip(hwnd, mag, FALSE) ;
	    return (MRESULT) 0 ;
        case IDM_MAGN_REVS :
	    magnifyClip(hwnd, mag, TRUE) ;
	    return (MRESULT) 0 ;
        case IDM_MAGN_MOVE :
	    magnifyMove(hwnd, mag) ;
	    return (MRESULT) 0 ;
	case IDM_MAGN_EXIT :
	    magnifyExit(hwnd, mag) ;
	    return (MRESULT) 0 ;
        }
    }
    return WinDefWindowProc(hwnd, msg, mp1, mp2) ;
}

/*
 * glyphMagnify - Maginfy Character Glyph
 */

static  BOOL    MagnRegist = FALSE ;
static  char    MagnName[] = "xfont.magnify" ;

#define CX_MIN  128
#define CY_MIN  128

static  nMagnify(int cx, int cy)
{
    int     m ;

    for (m = 1 ; m < CX_MIN && m < CY_MIN ; m++) {
        if ((cx * m) >= CX_MIN && (cy * m) >= CY_MIN) {
	    break ;
	}
    }
    return m ;
}
 
void    glyphMagnify(HWND hwnd, CharInfoPtr pci)
{
    MAGPTR  mag ;
    ULONG   flStyle      ;
    ULONG   flFrameFlags ;
    LONG    fx, fy, cx, cy, m ;
    POINTL  pt ;
    
    if (hwnd == NULLHANDLE || curFont == NULL || pci == NULL) {
        return ;
    }
    
    if (! MagnRegist) {
        WinRegisterClass(WinQueryAnchorBlock(hwnd), 
	    MagnName, MagnifyWndProc, CS_SIZEREDRAW, sizeof(MAGPTR)) ;
        MagnRegist = TRUE ;
    }

    if ((mag = (MAGPTR) malloc(sizeof(MAGREC))) == NULL) {
        return ;
    }
    mag->pci = pci               ;
    mag->max = curFont->maxbound ;
    mag->chr = pci->metrics      ;
    mag->hbm = bitmapFromGlyph(hwnd, pci) ;
    mag->max.leftSideBearing = curFont->minbound.leftSideBearing ;
    
    flStyle      = 0              ;
    flFrameFlags = FCF_SIZEBORDER ;
    
    mag->frame = WinCreateStdWindow(HWND_DESKTOP, flStyle,
            &flFrameFlags, MagnName, NULL, 0L, 0, ID_XFONT, &mag->client) ;

    if (mag->frame == NULLHANDLE || mag->client == NULLHANDLE) {
        if (mag->hbm) GpiDeleteBitmap(mag->hbm) ;
	free(mag) ;
        return ;
    }
    
    fx = mag->max.rightSideBearing - mag->max.leftSideBearing ;
    fy = mag->max.ascent + mag->max.descent ;
    m = nMagnify(fx, fy) ;
    cx = fx * m ;
    cy = fy * m ;

    WinSendMsg(mag->client, WM_MAGN_INIT, MPFROMP(mag), NULL) ;

    WinQueryPointerPos(HWND_DESKTOP, &pt) ;
    pt.x -= (cx / 2) ;
    pt.y -= (cy / 2) ;
    
    WinSetWindowPos(mag->frame, HWND_TOP, pt.x, pt.y, cx, cy,
            SWP_ACTIVATE | SWP_SHOW | SWP_ZORDER | SWP_MOVE | SWP_SIZE) ;
}

/*
 * glyphClipon - Copy Glyph Bitmap into Clipboard
 */
 
void    glyphClipon(HWND hwnd, CharInfoPtr pci, BOOL revs)
{
    HBITMAP hbm ;
    
    if (hwnd == NULLHANDLE || pci == NULL) {
        return ;
    }
    bitmapIntoClipboard(hwnd, pci, revs) ;
}
