#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "neuro_d.h"
#include "neuro_e.h"
#include "neuro_k.h"



static CHAR MODULE[] =	"NeuroKernel";



static BOOLEAN initialized = FALSE;


static VOID NeuronInit(VOID);

static Weight* AllocWeight(LONG number);
static VOID FreeWeight(Weight* weight);
static Neuron* AllocNeuron(LONG number);
static VOID FreeNeuron(Neuron* neuron);
static NeuronLayer* AllocNeuronLayer(LONG number);
static VOID FreeNeuronLayer(NeuronLayer* layer);
static NeuronNet* AllocNeuronNet(VOID);
static VOID FreeNeuronNet(NeuronNet* net);

static LONG CalcNeuronError(NeuronNet* net, InOutput* out, FLOAT* error);
static LONG SetInput(NeuronNet* net, InOutput in[], LONG nrin);
static LONG SetNeuronInput(NeuronNet* net, InOutput* in);
static LONG CalcForward(NeuronNet* net);
static FLOAT       NeuronCalcActivation(NeuronNet* net, Neuron* neuron);
static LONG SetDelta(NeuronNet* net, InOutput out[], LONG nrout);
static LONG SetNeuronDelta(NeuronNet* net, InOutput* out);
static VOID        AddNeuronDelta(NeuronNet* net, LONG l, LONG n, FLOAT value);
static LONG CalcBackward(NeuronNet* net);
static LONG CalcWeights(NeuronNet* net, BOOLEAN update, FLOAT lambda, FLOAT mu);
static FLOAT       ClipWeight(FLOAT value);
static LONG GetOutput(NeuronNet* net, InOutput out[], LONG nrout);
static LONG GetNeuronOutput(NeuronNet* net, InOutput* out);
static FLOAT       GetNeuronDerivOutput2(NeuronNet* net, LONG l, LONG n);
static FLOAT       GetNeuronOutput2(NeuronNet* net, LONG l, LONG n);

static VOID PrintWeight(Weight* weight, LONG w, LONG n, LONG l);
static VOID PrintNeuron(Neuron* neuron, LONG n, LONG l);
static VOID PrintNeuronLayer(NeuronLayer* layer, LONG l);
static VOID PrintNeuronNet(NeuronNet* net);
	     
static FLOAT GetWeight(NeuronNet* net, LONG l, LONG n, LONG w);
static VOID SetWeight(NeuronNet* net, LONG l, LONG n, LONG w, FLOAT value);

static FLOAT logistic(Neuron* neuron);
static FLOAT dlogistic(Neuron* neuron);
static FLOAT step(Neuron* neuron);
static FLOAT dstep(Neuron* neuron);

static FLOAT sqr(FLOAT x)  { return x*x; }



NeuronNet* NewNeuronNet(UNSIGNED seed, LONG nrlayers, ...)
{
NeuronNet* net;
NeuronLayer* layers;
Neuron* neurons;
Weight* weights;
LONG i, j, l, n, w, nrneurons;
va_list nrneurons_list;

	if (!initialized)
		NeuronInit();
				   
    if (seed > 1)
		srand(seed);
    else
		srand((unsigned) time(NULL));
		
        if (nrlayers < 2)
        {
                ERROR(MODULE, NEURO_SMALL);
                return NULL;
        }

        va_start(nrneurons_list, nrlayers);

        if ((net = AllocNeuronNet()) == NULL)
        {
                va_end(nrneurons_list);
                return NULL;
        }

        if ((layers = AllocNeuronLayer(nrlayers)) == NULL)
        {
                FreeNeuronNet(net);
                va_end(nrneurons_list);
                return NULL;
        }

        net->nrlayers = nrlayers;
        net->layers = layers;

        for (l = 0; l < nrlayers; l++)
        {
                nrneurons = va_arg(nrneurons_list, LONG);
                if (nrneurons == 0 || (neurons = AllocNeuron(nrneurons)) == NULL)
                        break;
                else
                {
                        layers[l].nrneurons = nrneurons;
                        layers[l].neurons = neurons;
                        if (l == 0)     /*  at input layer?  */
                                for (n = 0; n < nrneurons; n++)
				{
					layers[l].neurons[n].bias = (FLOAT)(RANDOM(1000)-500)/1000;
					layers[l].neurons[n].dbias = 0;
					layers[l].neurons[n].prvdbias = 0;
                                        layers[l].neurons[n].func = step;
                                        layers[l].neurons[n].dfunc = dstep;
				}
                        else
                                for (n = 0; n < nrneurons; n++)
				{
					layers[l].neurons[n].bias = (FLOAT)(RANDOM(1000)-500)/1000;
					layers[l].neurons[n].dbias = 0;
					layers[l].neurons[n].prvdbias = 0;
                                        layers[l].neurons[n].func = logistic;
                                        layers[l].neurons[n].dfunc = dlogistic;
				}
                }
        }

        if (l < nrlayers)       /*  clean up if neuron allocation failed  */
        {
                for (i = 0; i < l; i++)
                        FreeNeuron(layers[i].neurons);
                FreeNeuronLayer(layers);
                FreeNeuronNet(net);
                va_end(nrneurons_list);
                return NULL;
        }

        for (l = 1; l < nrlayers; l++)  /*  first layer doesn't have weights to a lower layer  */
        {
                for (n = 0; n < layers[l].nrneurons; n++)
                {
                        if ((weights = AllocWeight(layers[l-1].nrneurons)) == NULL)
                                break;  /*  the number of weights equals the number of neurons in the lower layer  */
                        else
                        {
                                layers[l].neurons[n].nrweights = layers[l-1].nrneurons;
                                layers[l].neurons[n].weights = weights;
                                for (w = 0; w < layers[l].neurons[n].nrweights; w++)
                                {
                                        layers[l].neurons[n].weights[w].fromlayer = l-1;
                                        layers[l].neurons[n].weights[w].fromneuron = w;
					layers[l].neurons[n].weights[w].value = (FLOAT)(RANDOM(1000)-500)/1000;
					layers[l].neurons[n].weights[w].delta = 0;
					layers[l].neurons[n].weights[w].prvdelta = 0;
                                }
                        }
                }

                if (n < layers[l].nrneurons)
                {
                        for (i = 0; i < n; i++)
                                FreeWeight(layers[l].neurons[i].weights);
                        for (j = 1; j < l; j++)
                                for (i = 0; i < layers[j].nrneurons; i++)
                                        FreeWeight(layers[j].neurons[i].weights);
                        for (i = 0; i < nrlayers; i++)
                                FreeNeuron(layers[i].neurons);
                        FreeNeuronLayer(layers);
                        FreeNeuronNet(net);
                        va_end(nrneurons_list);
                        return NULL;
                }
        }

        va_end(nrneurons_list);

        return net;
}


VOID DelNeuronNet(NeuronNet* net)
{
LONG l, n;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return;
        }
        for (l = 0; l < net->nrlayers; l++)
        {
                for (n = 0; n < net->layers[l].nrneurons; n++)
                        FreeWeight(net->layers[l].neurons[n].weights);
                FreeNeuron(net->layers[l].neurons);
        }
        FreeNeuronLayer(net->layers);
        FreeNeuronNet(net);
}


InOutput* NewNeuronInOutput(LONG size)
{
	return MALLOC(size*sizeof(InOutput));
}


VOID DelNeuronInOutput(InOutput* inout)
{
	FREE(inout);
}


LONG CalcNeuronNet(NeuronNet* net, InOutput in[], LONG nrin,
                          InOutput out[], LONG nrout, BOOLEAN train, BOOLEAN update,
			  FLOAT lambda, FLOAT mu)
{
LONG rc;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        if ((rc = SetInput(net, in, nrin)) != NEURO_OK)
        {
                ERROR(MODULE, rc);
                return rc;
        }

        if ((rc = CalcForward(net)) != NEURO_OK)
        {
                ERROR(MODULE, rc);
                return rc;
        }

        if (train)
        {
                if ((rc = SetDelta(net, out, nrout)) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
                if ((rc = CalcBackward(net)) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
                if ((rc = CalcWeights(net, update, lambda, mu)) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
        }

        if ((rc = GetOutput(net, out, nrout)) != NEURO_OK)
        {
                ERROR(MODULE, rc);
                return rc;
        }

        return NEURO_OK;
}


LONG CalcNeuronNetError(NeuronNet* net, InOutput out[], LONG nrout, FLOAT* error)
{
LONG rc;
LONG i;
FLOAT nerror;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }

	*error = 0;
        for (i = 0; i < nrout; i++)
        {
                if ((rc = CalcNeuronError(net, &out[i], &nerror)) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
		
		*error += nerror;
        }	 
	
        return NEURO_OK;
}
 

LONG CalcNeuronError(NeuronNet* net, InOutput* out, FLOAT* error)
{
LONG l, n;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }

        if (!((n = out->inoutnr) >= 0 && n < net->layers[l = net->nrlayers-1].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_NINDEX;
        }

	*error = 0.5*sqr(out->value - net->layers[l].neurons[n].output);

        return NEURO_OK;
}				       


LONG SetInput(NeuronNet* net, InOutput in[], LONG nrin)
{
LONG rc;
LONG i;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        for (i = 0; i < nrin; i++)
        {
                if ((rc = SetNeuronInput(net, &in[i])) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
        }
        return NEURO_OK;
}


LONG SetNeuronInput(NeuronNet* net, InOutput* in)
{
LONG n;
Neuron* neuron;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }

        if (!((n = in->inoutnr) >= 0 && n < net->layers[0].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_NINDEX;
        }

        neuron = &net->layers[0].neurons[n];
        neuron->activation = in->value;
        neuron->output = neuron->func(neuron);

        return NEURO_OK;
}


LONG CalcForward(NeuronNet* net)
{
LONG l, n;
Neuron* neuron;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        for (l = 1; l < net->nrlayers; l++)
        {
                for (n = 0; n < net->layers[l].nrneurons; n++)
                {
                        neuron = &net->layers[l].neurons[n];
                        neuron->activation = NeuronCalcActivation(net, neuron);
                        neuron->output = neuron->func(neuron);
			neuron->delta = 0;
                }
        }
        return NEURO_OK;
}


FLOAT NeuronCalcActivation(NeuronNet* net, Neuron* neuron)
{
FLOAT activation, input;
LONG w;

        activation = 0;
        for (w = 0; w < neuron->nrweights; w++)
        {
                input = GetNeuronOutput2(net, neuron->weights[w].fromlayer,
                                              neuron->weights[w].fromneuron);
                activation += neuron->weights[w].value*input;
        }
        return activation;
}


LONG SetDelta(NeuronNet* net, InOutput out[], LONG nrout)
{
LONG rc;
LONG i;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        for (i = 0; i < nrout; i++)
        {
                if ((rc = SetNeuronDelta(net, &out[i])) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
        }
        return NEURO_OK;
}


LONG SetNeuronDelta(NeuronNet* net, InOutput* out)
{
LONG l, n;
Neuron* neuron;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }

        if (!((n = out->inoutnr) >= 0 && n < net->layers[l = net->nrlayers-1].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_NINDEX;
        }

        neuron = &net->layers[l].neurons[n];
        neuron->delta = neuron->dfunc(neuron)*(out->value-neuron->output);

        return NEURO_OK;
}


LONG CalcBackward(NeuronNet* net)
{
LONG l, n, w;
Neuron* neuron;
Weight* weight;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        for (l = net->nrlayers-1; l > 1; l--)
        {
                for (n = 0; n < net->layers[l].nrneurons; n++)
                {
                        neuron = &net->layers[l].neurons[n];

                        for (w = 0; w < neuron->nrweights; w++)
                        {
                                weight = &net->layers[l].neurons[n].weights[w];
                                AddNeuronDelta(net, weight->fromlayer, weight->fromneuron,
                                               GetNeuronDerivOutput2(net, weight->fromlayer, weight->fromneuron)*weight->value*neuron->delta);
                        }
                }
        }
        return NEURO_OK;
}


VOID AddNeuronDelta(NeuronNet* net, LONG l, LONG n, FLOAT value)
{
        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return;
        }

        if (!(l >= 0 && l < net->nrlayers))
        {
                FATAL(MODULE, NEURO_LINDEX);
                return;
        }

        if (!(n >= 0 && n < net->layers[l].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return;
        }

	net->layers[l].neurons[n].delta += value;

        return;
}


LONG CalcWeights(NeuronNet* net, BOOLEAN update, FLOAT lambda, FLOAT mu)
{	
LONG l, n, w;
FLOAT input;
Neuron* neuron;
Weight* weight;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        for (l = 1; l < net->nrlayers; l++)
        {
                for (n = 0; n < net->layers[l].nrneurons; n++)
                {
                        neuron = &net->layers[l].neurons[n];
			neuron->dbias += lambda*neuron->delta;
			if (update)
			{
				neuron->bias += (1-mu)*neuron->dbias+mu*neuron->prvdbias;
				neuron->prvdbias = neuron->dbias;
				neuron->dbias = 0;	/*  reset delta bias after update  */
				neuron->bias = ClipWeight(neuron->bias);
			}

                        for (w = 0; w < neuron->nrweights; w++)
                        {
                                weight = &net->layers[l].neurons[n].weights[w];
                		input = GetNeuronOutput2(net, neuron->weights[w].fromlayer,
	                                                 neuron->weights[w].fromneuron);
				weight->delta += lambda*neuron->delta*input;
				if (update)
				{
					weight->value += (1-mu)*weight->delta+mu*weight->prvdelta;
					weight->prvdelta = weight->delta;
					weight->delta = 0;	/*  reset delta weight after update  */
					weight->value = ClipWeight(weight->value);
				}
                        }
                }
        }
        return NEURO_OK;
}


FLOAT ClipWeight(FLOAT value)
{
	if (value > NEURO_MAXWEIGHT)
		value = NEURO_MAXWEIGHT;
	else
	if (value < NEURO_MINWEIGHT)
		value = NEURO_MINWEIGHT;
	return value;
}


LONG GetOutput(NeuronNet* net, InOutput out[], LONG nrout)
{
LONG rc;
LONG i;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }
        for (i = 0; i < nrout; i++)
        {
                if ((rc = GetNeuronOutput(net, &out[i])) != NEURO_OK)
                {
                        ERROR(MODULE, rc);
                        return rc;
                }
        }
        return NEURO_OK;
}


LONG GetNeuronOutput(NeuronNet* net, InOutput* out)
{
LONG l, n;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_NETNULL;
        }

        if (!((n = out->inoutnr) >= 0 && n < net->layers[l = net->nrlayers-1].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_NINDEX;
        }

        out->value = net->layers[l].neurons[n].output;

        return NEURO_OK;
}


FLOAT GetNeuronDerivOutput2(NeuronNet* net, LONG l, LONG n)
{
Neuron* neuron;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_MINOUTPUT;
        }

        if (!(l >= 0 && l < net->nrlayers))
        {
                FATAL(MODULE, NEURO_LINDEX);
                return NEURO_MINOUTPUT;
        }

        if (!(n >= 0 && n < net->layers[l].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_MINOUTPUT;
        }

	neuron = &net->layers[l].neurons[n];
        return neuron->dfunc(neuron);
}


FLOAT GetNeuronOutput2(NeuronNet* net, LONG l, LONG n)
{
        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_MINOUTPUT;
        }

        if (!(l >= 0 && l < net->nrlayers))
        {
                FATAL(MODULE, NEURO_LINDEX);
                return NEURO_MINOUTPUT;
        }

        if (!(n >= 0 && n < net->layers[l].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_MINOUTPUT;
        }

        return net->layers[l].neurons[n].output;
}


VOID PrintNeuronNet(NeuronNet* net)
{
LONG l, n, w;

        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return;
        }
        printf("The net has %d layers\n", net->nrlayers);
        for (l = 0; l < net->nrlayers; l++)
        {
                PrintNeuronLayer(&(net->layers[l]), l);
                for (n = 0; n < net->layers[l].nrneurons; n++)
                {
                        PrintNeuron(&(net->layers[l].neurons[n]), n, l);
                        for (w = 0; w < net->layers[l].neurons[n].nrweights; w++)
                                PrintWeight(&(net->layers[l].neurons[n].weights[w]), w, n, l);
                }
        }
}


Weight* AllocWeight(LONG number)
{
Weight* weights;

        weights = MALLOC(number*sizeof(Weight));
        DEBUG(weights == NULL);
        return weights;
}


VOID FreeWeight(Weight* weight)
{
        FREE(weight);
}


VOID PrintWeight(Weight* weight, LONG w, LONG n, LONG l)
{
        printf("weight %d from neuron %d at layer %d to neuron %d at layer %d has value %f\n",
                w, weight->fromneuron, weight->fromlayer, n, l, weight->value);
}


Neuron* AllocNeuron(LONG number)
{
Neuron* neurons;

        neurons = MALLOC(number*sizeof(Neuron));
        DEBUG(neurons == NULL);
        return neurons;
}


VOID FreeNeuron(Neuron* neuron)
{
        FREE(neuron);
}


VOID PrintNeuron(Neuron* neuron, LONG n, LONG l)
{
        printf("neuron %d at layer %d has %d weights and its bias is %f\n", n, l, neuron->nrweights, neuron->bias);
}


NeuronLayer* AllocNeuronLayer(LONG number)
{
NeuronLayer* layers;

        layers = MALLOC(number*sizeof(NeuronLayer));
        DEBUG(layers == NULL);
        return layers;
}


VOID FreeNeuronLayer(NeuronLayer* layer)
{
        FREE(layer);
}


VOID PrintNeuronLayer(NeuronLayer* layer, LONG l)
{
        printf("layer %d has %d neurons\n", l, layer->nrneurons);
}


NeuronNet* AllocNeuronNet(VOID)
{
NeuronNet* net;

        net = MALLOC(sizeof(NeuronNet));
        DEBUG(net == NULL);
        return net;
}


VOID FreeNeuronNet(NeuronNet* net)
{
        FREE(net);
}


FLOAT GetWeight(NeuronNet* net, LONG l, LONG n, LONG w)
{
        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return NEURO_MINWEIGHT;
        }

        if (!(l >= 0 && l < net->nrlayers))
        {
                FATAL(MODULE, NEURO_LINDEX);
                return NEURO_MINWEIGHT;
        }

        if (!(n >= 0 && n < net->layers[l].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return NEURO_MINWEIGHT;
        }

        if (!(w >= 0 && w < net->layers[l].neurons[n].nrweights))
        {
                FATAL(MODULE, NEURO_WINDEX);
                return NEURO_MINWEIGHT;
        }

        return net->layers[l].neurons[n].weights[w].value;
}


VOID SetWeight(NeuronNet* net, LONG l, LONG n, LONG w, FLOAT value)
{
        if (net == NULL)
        {
                ERROR(MODULE, NEURO_NETNULL);
                return;
        }

        if (!(l >= 0 && l < net->nrlayers))
        {
                FATAL(MODULE, NEURO_LINDEX);
                return;
        }

        if (!(n >= 0 && n < net->layers[l].nrneurons))
        {
                FATAL(MODULE, NEURO_NINDEX);
                return;
        }

        if (!(w >= 0 && w < net->layers[l].neurons[n].nrweights))
        {
                FATAL(MODULE, NEURO_WINDEX);
                return;
        }

        net->layers[l].neurons[n].weights[w].value = value;
}

 
FLOAT logistic(Neuron* neuron)
{
        return NEURO_MAXMINOUTPUT/(1.0+exp(-(neuron->activation+neuron->bias)))+NEURO_MINOUTPUT;
}


FLOAT dlogistic(Neuron* neuron)
{
	return 0.25*NEURO_MAXMINOUTPUT*(1+neuron->output)*(1-neuron->output);
}


FLOAT step(Neuron* neuron)
{
        return neuron->activation > 0 ? NEURO_MAXOUTPUT : NEURO_MINOUTPUT;
}


FLOAT dstep(Neuron* neuron)
{
	return 0;
}


VOID NeuronInit(VOID)
{
	initialized = TRUE;
}



#ifdef TEST

int main(int argc, char* argv[])
{
NeuronNet* net;
NeuronNet* net2;
#define NRIN	8
#define NROUT	4
InOutput in[NRIN];
InOutput tout[NROUT];
InOutput out;
FLOAT lambda, mu, error, perror;
LONG p;

        net = NewNeuronNet(333, 3, 8, 4, 2);
        if (net == NULL)
                return -1;
        PrintNeuronNet(net);
        DelNeuronNet(net);

	net2 = NewNeuronNet(0, 3, 2, 2, 1);
        if (net2 == NULL)
                return -1;
        PrintNeuronNet(net2);

	/*  the following statements fill 'in' and 'out' with  */
	/*  the values for the famous XOR test                 */
	in[0].inoutnr = 0;
	in[0].value = NEURO_MINOUTPUT;
	in[1].inoutnr = 1;
	in[1].value = NEURO_MINOUTPUT;
	in[2].inoutnr = 0;
	in[2].value = NEURO_MINOUTPUT;
	in[3].inoutnr = 1;
	in[3].value = NEURO_MAXOUTPUT;
	in[4].inoutnr = 0;
	in[4].value = NEURO_MAXOUTPUT;
	in[5].inoutnr = 1;
	in[5].value = NEURO_MINOUTPUT;
	in[6].inoutnr = 0;
	in[6].value = NEURO_MAXOUTPUT;
	in[7].inoutnr = 1;
	in[7].value = NEURO_MAXOUTPUT;

	tout[0].inoutnr = 0;
	tout[0].value = NEURO_MINOUTPUT;
	tout[1].inoutnr = 0;
	tout[1].value = NEURO_MAXOUTPUT;
	tout[2].inoutnr = 0;
	tout[2].value = NEURO_MAXOUTPUT;
	tout[3].inoutnr = 0;
	tout[3].value = NEURO_MINOUTPUT;

	lambda = 1.0;
	mu = 0.25;

	do
	{
		error = 0;
		for (p = 0; p < NRIN; p += 2)
		{
			out.inoutnr = tout[p/2].inoutnr;
			out.value = tout[p/2].value;
			CalcNeuronNet(net2, &in[p], 2, &out, 1, 
				      TRUE, p == NRIN-2 ? TRUE : FALSE, 
				      lambda, mu);
			CalcNeuronNetError(net2, &tout[p/2], 1, &perror);

			printf("neuron output for pattern %d = %f, error = %f\n", p/2, out.value, perror);
			error += perror;
		}
		printf("error = %f\n", error);

	} while (error > 0.1);

        DelNeuronNet(net2);

        return NEURO_OK;
}

#endif


