#include <dnpap.h>
#include <message.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <mibdf.h>
#include <config.h>
#include <string.h>


#include "defs.h"
#include "funcs.h"


		    
#define MAXNAME		256


enum INI2CFG_ERRORS
{		      
	NO_ERR,
	USAGE_ERR,
	OPTION_ERR,
    ARGUMENT_ERR,
    MIB_ERR,
    INIT_ERR,
    NOCONFIG_ERR,
	LOAD_ERR,
	LEVEL_ERR,
	NAME_ERR,
	TYPE_ERR,
	BOOLEAN_ERR,
	SHORT_ERR,
	LONG_ERR,
	IPADDR_ERR,
	DOUBLE_ERR,
	STRING_ERR,
	BYTES_ERR,
    LONGBUF_ERR
};


static BOOLEAN extension, current;


static VOID PrintLevel(USHORT level);
static VOID PrintName(CHAR *name);
static VOID PrintAssign(VOID);
static VOID PrintType(USHORT type);
static VOID PrintBooleanValue(BOOLEAN b);
static VOID PrintShortValue(SHORT s);
static VOID PrintLongValue(LONG l);
static VOID PrintIPAddrValue(IPADDR a);
static VOID PrintDoubleValue(DOUBLE d);
static VOID PrintStringValue(CHAR *c);
static VOID PrintBytesValue(BYTE *b, USHORT n);
static VOID PrintLongBufValue(LONG *b, USHORT n);
static VOID PrintMibIDValue(LONG *b, USHORT n);
static VOID PrintNullValue(VOID);
static VOID PrintNewLine(VOID);
static BOOLEAN MibID(LONG *b, USHORT n);
static BOOLEAN Printable(BYTE *b, SHORT n);
static VOID Usage(VOID);



int main(int argc, char *argv[])
{					   
INT argi, i;
CHAR iname[MAXNAME];
CONFIG_ENTRY *e;
BYTE *name;
USHORT level, type;
BOOLEAN t;
SHORT s;
LONG l;
IPADDR a;
DOUBLE d;
CHAR *c;
BYTE *b;
LONG *lb;
USHORT n;
CHAR *grepword = NULL;
BOOLEAN configread = FALSE, skiparg = FALSE;

extern BYTE configBase[];


	MessageInit("stdio:err", MSGFILE, MSGSTDIO, NULL);
	MessageConfig(NO_ERR, "Cfg2Ini");
	
	if (argc < 2)
	{
		MESSAGE(DMC_FATAL, USAGE_ERR, "argument missing");
		Usage();
		DnpapExit(1);
	}
		     
	for (argi = 1; argi < argc; argi++)
	{				 
		if (argv[argi][0] == '-')
        {
			for (i = 1; i < STRLEN(argv[argi]); i++)
				switch (argv[argi][i])
				{
				case 'h':
					Usage();
					DnpapExit(0);
                case 's':
                    if (argi+1 < argc)
                    {
                        grepword = argv[argi+1];
                        skiparg = TRUE;
                    }
                    else
                    {
                        MESSAGE(DMC_FATAL, ARGUMENT_ERR, "argument for -s is missing");
                        Usage();
                        DnpapExit(1);
                    }
                    break;
				default:
					MESSAGE(DMC_FATAL, OPTION_ERR, "unrecognized option");
					Usage();
					DnpapExit(1);
				}
            if (skiparg == TRUE)
            {
                argi++;
                skiparg = FALSE;
            }
        }
		else
		{
			extension = FALSE;
			current = FALSE;
			
			strcpy(iname, argv[argi]);
			
			if (STRLEN(iname) >= 4 && strncasecmp(iname+STRLEN(iname)-4, ".ini", 4) == 0)
			{
				extension = TRUE;
				current = TRUE;
			}
            else
            {
                Basename(argv[argi], iname, sizeof(iname));
            }
						       					 
			if (current == TRUE)
			{		
				if (ConfigRead(iname, 0) == FALSE)
				{
					MESSAGE(DMC_FATAL, LOAD_ERR, "reading of INI file %s failed", iname);
					DnpapExit(1);
				}
			}
			else
			{
				strcpy(configBase, iname);
                if (ConfigInit(argc, argv) == FALSE)
				{
					MESSAGE(DMC_FATAL, INIT_ERR, "initialization failed, base %s", configBase);
					DnpapExit(1);
				}
				strcpy(configBase, iname);

				if (ConfigLoad() == FALSE)
				{
					MESSAGE(DMC_FATAL, LOAD_ERR, "loading of INI files failed, base %s", configBase);
					DnpapExit(1);
				}
			}

            configread = TRUE;
		}
	}
	
    if (configread == FALSE)
    {
		MESSAGE(DMC_FATAL, NOCONFIG_ERR, "no config file specified");
        Usage();
		DnpapExit(1);
    }

    if (MibInit() != 0 || MibFullMib() != 0)
    {
            MESSAGE(DMC_FATAL, MIB_ERR, "MIB initialization failed");
            DnpapExit(1);
    }

	e = ConfigFirst();
	
	while (e != NULL)
	{
		if (!ConfigGetName(e, &name))
		{
			MESSAGE(DMC_FATAL, NAME_ERR, "name identification failed");
			DnpapExit(1);
		}
		
        if (grepword != NULL && strstr(name, grepword) == NULL)
        {
    	    e = ConfigNext(e);
            continue;
        }

		if (!ConfigGetLevel(e, &level))
		{
			MESSAGE(DMC_FATAL, LEVEL_ERR, "level identification failed");
			DnpapExit(1);
		}

		if (!ConfigGetType(e, &type))
		{
			MESSAGE(DMC_FATAL, TYPE_ERR, "type identification failed");
			DnpapExit(1);
		}
		  
		switch (type)
		{
		case CONFIG_TYPE_BOOLEAN:
			if (!ConfigGetValueBoolean(e, &t))
			{
				MESSAGE(DMC_FATAL, BOOLEAN_ERR, "boolean value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_SHORT:
			if (!ConfigGetValueShort(e, &s))
			{
				MESSAGE(DMC_FATAL, SHORT_ERR, "short value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_LONG:
			if (!ConfigGetValueLong(e, &l))
			{
				MESSAGE(DMC_FATAL, LONG_ERR, "long value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_IPADDR:
			if (!ConfigGetValueIPAddr(e, &a))
			{
				MESSAGE(DMC_FATAL, IPADDR_ERR, "long value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_DOUBLE:
			if (!ConfigGetValueDouble(e, &d))
			{
				MESSAGE(DMC_FATAL, DOUBLE_ERR, "double value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_STRING:
			if (!ConfigGetValueString(e, &c))
			{
				MESSAGE(DMC_FATAL, STRING_ERR, "string value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_BYTES:
			if (!ConfigGetValueBytes(e, &b, &n))
			{
				MESSAGE(DMC_FATAL, BYTES_ERR, "bytes value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_LONGBUF:
			if (!ConfigGetValueLongBuf(e, &lb, &n))
			{
				MESSAGE(DMC_FATAL, LONGBUF_ERR, "longbuf value resolve failed");
				DnpapExit(1);
			}
			break;
		case CONFIG_TYPE_NULL:
		default:
			break;
		}
		
		PrintLevel(level);
		PrintName(name);
		PrintAssign();
		if (type != CONFIG_TYPE_LONGBUF && type != CONFIG_TYPE_MIBID)
			PrintType(type);
			
		switch (type)
		{
		case CONFIG_TYPE_BOOLEAN:
			PrintBooleanValue(t);
			break;
		case CONFIG_TYPE_SHORT:
			PrintShortValue(s);
			break;
		case CONFIG_TYPE_LONG:
			PrintLongValue(l);
			break;
		case CONFIG_TYPE_IPADDR:
			PrintIPAddrValue(a);
			break;
		case CONFIG_TYPE_DOUBLE:
			PrintDoubleValue(d);
			break;
		case CONFIG_TYPE_STRING:
			PrintStringValue(c);
			break;
		case CONFIG_TYPE_BYTES:
			PrintBytesValue(b, n);
			break;
		case CONFIG_TYPE_LONGBUF:
			if (MibID(lb, n) == FALSE)
			{
				PrintType(type);
				PrintLongBufValue(lb, n);
				break;
			}
			type = CONFIG_TYPE_MIBID;	/*  fall through in case 'lb' holds a representable Mib ID  */
		case CONFIG_TYPE_MIBID:
			PrintType(type);
			PrintMibIDValue(lb, n);
			break;
		case CONFIG_TYPE_NULL:
		default:		   
			PrintNullValue();
			break;
		}

		PrintNewLine();
				
    	e = ConfigNext(e);
	}

    return 0;
}

					 
VOID PrintLevel(USHORT level)
{
    if (current == FALSE && level == 0)
        printf("            ");
    else
    if (level > 0)
        printf("(Level %2u)  ", level);
}


VOID PrintName(CHAR *name)
{		    
#define SPACING		21
CHAR spacing[SPACING] = "                    ";     /* SPACING-1 spaces  */
			 
	if (SPACING-STRLEN(name) >= 0)
		spacing[SPACING-STRLEN(name)] = '\0';
    else
        spacing[0] = '\0';
	printf("%s%s ", name, spacing);
}


VOID PrintAssign(VOID)
{
	printf("= ");
}


VOID PrintType(USHORT type)
{
	switch (type)
	{
	case CONFIG_TYPE_BOOLEAN:
		printf("(Boolean) ");
		break;
	case CONFIG_TYPE_SHORT:
		printf("(Short) ");
		break;
	case CONFIG_TYPE_LONG:
		printf("(Long) ");
		break;
	case CONFIG_TYPE_IPADDR:
		printf("(IPAddr) ");
		break;
	case CONFIG_TYPE_DOUBLE:
		printf("(Double) ");
		break;
	case CONFIG_TYPE_STRING:
		printf("(String) ");
		break;
	case CONFIG_TYPE_BYTES:
		printf("(Bytes) ");
		break;
	case CONFIG_TYPE_LONGBUF:
		printf("(LongBuf) ");
		break;
	case CONFIG_TYPE_MIBID:
		printf("(MibID) ");
		break;
	case CONFIG_TYPE_NULL:
	default:		   
		printf("(Null)");
		break;
	}
}


VOID PrintBooleanValue(BOOLEAN b)
{
	printf("%s", b != FALSE ? "TRUE" : "FALSE");
}


VOID PrintShortValue(SHORT s)
{
	printf("%d", s);
}


VOID PrintLongValue(LONG l)
{
	printf("%ld", l);
}


VOID PrintIPAddrValue(IPADDR a)
{
	printf("%u.%u.%u.%u", (BYTE)(a >> 24), (BYTE)(a >> 16), (BYTE)(a >> 8), (BYTE)(a));
}


VOID PrintDoubleValue(DOUBLE d)
{
	printf("%f", d);
}


VOID PrintStringValue(CHAR *s)
{		      
INT n;

    n = strlen(s)-1;
    if (n < 0)
        n = 0;
    if (isspace(s[0]) || isspace(s[n]))
    	printf("\"%s\"", s);
    else
    	printf("%s", s);
}


VOID PrintBytesValue(BYTE *b, USHORT n)
{			
USHORT i;

    if (Printable(b, n) == TRUE)
        printf("\"%.*s\"", n, b);
    else
	    for (i = 0; i < n; i++)	      
	    {
            if (b[i] == 0)
                printf("0x");
		    printf("%#2.2x", b[i]);
		    if (i < n-1)
			    printf(":");
	    }
}


VOID PrintLongBufValue(LONG *b, USHORT n)
{			
USHORT i;

	for (i = 0; i < n; i++)	      
	{
		printf("%ld", b[i]);
		if (i < n-1)
			printf(".");
	}
}


VOID PrintMibIDValue(LONG *b, USHORT n)
{			
USHORT i;
CONST CHAR *name;
INT m = 0;

	if ((name = MibID2Name2(b, n, &m)) != NULL)
	{
		printf("%s", name);
		if (m < n)
			printf(".");
	}
	for (i = m; i < n; i++)	      
	{
		printf("%ld", b[i]);
		if (i < n-1)
			printf(".");
	}
}


VOID PrintNullValue(VOID)
{		
	/*  exactly, here is nothing, zilch, nada  */
}
			
			
VOID PrintNewLine(VOID)
{
	printf("\n");
}


BOOLEAN MibID(LONG *b, USHORT n)
{
CONST CHAR *name;
INT m;

	if ((name = MibID2Name2(b, n, &m)) != NULL && m > 3)
		return TRUE;
	return FALSE;
}


BOOLEAN Printable(BYTE *b, SHORT n)
{
INT i;

    for (i = 0; i < n; i++)
        if (!isprint(b[i]))
            return FALSE;
    return TRUE;
}


VOID Usage(VOID)
{
	MESSAGE(DMC_MESSAGE, USAGE_ERR, "usage: ini2cfg [-s substring] filename\n"
				        "\n"
				        "       filename with extension '.ini' will cause this file to be read only\n"
				        "       filename without extension will cause the whole tree to be read\n"
                        "       substring is used to select just those variables that match it");
}
