#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2018 Mopi
# SPDX-FileCopyrightText: © 2023 Hoël Bézier <hoelbezier@riseup.net>
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Wonder Boy: The Dragon's Trap
# send your bug reports to bugs@dotslashplay.it
###

script_version=20250407.1

PLAYIT_COMPATIBILITY_LEVEL='2.32'

GAME_ID='wonder-boy'
GAME_NAME='Wonder Boy: The Dragon’s Trap'

ARCHIVE_BASE_0_NAME='wonder_boy_the_dragon_s_trap_en_1_03f_02_20817.sh'
ARCHIVE_BASE_0_MD5='775ac16ded3d2aca7ebcb27c7a2caf48'
ARCHIVE_BASE_0_SIZE='1100000'
ARCHIVE_BASE_0_VERSION='1.03f.02-gog20817'
ARCHIVE_BASE_0_URL='https://www.gog.com/game/wonder_boy_the_dragons_trap'

CONTENT_PATH_DEFAULT='data/noarch/game'
CONTENT_GAME_BIN64_FILES='
x86_64'
CONTENT_GAME_BIN32_FILES='
x86'
CONTENT_GAME_DATA_FILES='
bin_pc
gamecontrollerdb.txt
WonderBoy.png'
CONTENT_DOC_DATA_FILES='
Linux.README'

CONTENT_GAME_BIN_DEPENDENCIES_LIBRARIES='
libc.so.6
libstdc++.so.6
libGL.so.1'

APP_MAIN_EXE_BIN32='x86/wb.x86'
APP_MAIN_EXE_BIN64='x86_64/wb.x86_64'
APP_MAIN_ICON='WonderBoy.png'

# Load common functions

if [ -z "$PLAYIT_LIB2" ]; then
	: "${XDG_DATA_HOME:="$HOME/.local/share"}"
	for path in\
		"$PWD"\
		"$XDG_DATA_HOME/play.it"\
		'/usr/local/share/games/play.it'\
		'/usr/local/share/play.it'\
		'/usr/share/games/play.it'\
		'/usr/share/play.it'
	do
		if [ -e "$path/libplayit2.sh" ]; then
			PLAYIT_LIB2="$path/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

pkg_setup() {
	option_package=$(option_value 'package')
	case "$option_package" in
		('arch'|'gentoo')
			PKG_BIN32_ARCH='32'
			PKG_BIN32_DEPENDENCIES_LIBRARIES="$CONTENT_GAME_BIN_DEPENDENCIES_LIBRARIES"
			PKG_BIN64_ARCH='64'
			PKG_BIN64_DEPENDENCIES_LIBRARIES="$CONTENT_GAME_BIN_DEPENDENCIES_LIBRARIES"
		;;
		('deb')
			PKG_DATA_ID="${GAME_ID}-data"
			PKG_DATA_DESCRIPTION='data'
			PKG_BIN64_ARCH='64'
			PKG_BIN64_DEPENDENCIES_LIBRARIES="$CONTENT_GAME_BIN_DEPENDENCIES_LIBRARIES"
			PKG_BIN32_ARCH='32'
			PKG_BIN32_DEPENDENCIES_LIBRARIES="$CONTENT_GAME_BIN_DEPENDENCIES_LIBRARIES"
			PKG_BIN_DEPENDENCIES_SIBLINGS='
			PKG_DATA'
			PKG_BIN64_DEPENDENCIES_SIBLINGS="$PKG_BIN_DEPENDENCIES_SIBLINGS"
			PKG_BIN32_DEPENDENCIES_SIBLINGS="$PKG_BIN_DEPENDENCIES_SIBLINGS"
		;;
		('egentoo')
			PKG_BIN32_ARCH='32'
			PKG_BIN64_ARCH='64'
			PKG_DEPENDENCIES_LIBRARIES="$CONTENT_GAME_BIN_DEPENDENCIES_LIBRARIES"
			PKG_BIN64_DEPENDENCIES_LIBRARIES="$PKG_DEPENDENCIES_LIBRARIES" # play.it 2.20 uses 64-bit binary packages dependencies
		;;
		(*)
			error_option_invalid 'package' "$option_package"
			return 1
		;;
	esac
}

src_extract() {
	archive_extraction_default
}

src_prepare() {
	:
}

pkg_prepare() {
	option_package=$(option_value 'package')
	case "$option_package" in
		('deb'|'egentoo')
			content_inclusion_default
		;;
		('arch'|'gentoo')
			content_inclusion_default
			# Copy data files into both binary packages
			local GAME_DATA_PATH target_path_bin32 target_path_bin64
			GAME_DATA_PATH="${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME_DATA')"
			target_path_bin32="$(realpath --canonicalize-missing "$(package_path 'PKG_BIN32')$(path_game_data)")"
			target_path_bin64="$(realpath --canonicalize-missing "$(package_path 'PKG_BIN64')$(path_game_data)")"

			(
				cd "$GAME_DATA_PATH"
				local file_pattern file_path
				while read -r file_pattern; do
					pwd
					if [ -z "$file_pattern" ]; then
						continue
					fi
					if [ -e "$file_pattern" ]; then
						mkdir --parents "$target_path_bin32"
						mkdir --parents "$target_path_bin64"
						cp \
							--force \
							--link \
							--recursive \
							--no-dereference \
							--parents \
							--preserve=links \
							"$file_pattern" "$target_path_bin32"
						cp \
							--force \
							--link \
							--recursive \
							--no-dereference \
							--parents \
							--preserve=links \
							"$file_pattern" "$target_path_bin64"
						rm --force --recursive "$file_pattern"
					else
						while read -r file_path; do
							mkdir --parents "$target_path_bin32"
							mkdir --parents "$target_path_bin64"
							cp \
								--force \
								--link \
								--recursive \
								--no-dereference \
								--parents \
								--preserve=links \
								"$file_pattern" "$target_path_bin32"
							cp \
								--force \
								--link \
								--recursive \
								--no-dereference \
								--parents \
								--preserve=links \
								"$file_pattern" "$target_path_bin64"
							rm --force --recursive "$file_pattern"
						done <<- EOF
						$(find . -path "./${file_pattern#./}")
						EOF
					fi
				done <<- EOF
				$(content_files 'GAME_DATA')
				EOF
			)
		;;
		(*)
			error_option_invalid 'package' "$option_package"
			return 1
		;;
	esac
}

src_clean() {
	# Nothing to do here
	return 0
}

pkg_icons() {
	option_package=$(option_value 'package')
	case "$option_package" in
		('deb'|'egentoo')
			content_inclusion_icons 'PKG_DATA'
		;;
		('arch'|'gentoo')
			content_inclusion_icons 'PKG_BIN64'
			content_inclusion_icons 'PKG_BIN32'
		;;
		(*)
			error_option_invalid 'package' "$option_package"
			return 1
		;;
	esac
}

pkg_launchers() {
	launchers_generation 'PKG_BIN64'
	launchers_generation 'PKG_BIN32'
}

pkg_build() {
	packages_generation
}

pkg_clean() {
	working_directory_cleanup
}

pkg_instructions() {
	print_instructions
}

## Ensure that the default package is correctly set,
## despite PACKAGES_LIST not being set early
init_package() {
	option_package=$(option_value 'package')
	case "$option_package" in
		('arch'|'gentoo')
			PACKAGES_LIST='
			PKG_BIN64
			PKG_BIN32'
		;;
		('deb')
			PACKAGES_LIST='
			PKG_BIN64
			PKG_BIN32
			PKG_DATA'
		;;
		('egentoo')
			PACKAGES_LIST='
			PKG_BIN64
			PKG_BIN32
			PKG_DATA'
		;;
		(*)
			error_option_invalid 'package' "$option_package"
			return 1
		;;
	esac
	set_default_package 'PKG_BIN64'
}

## Run the default initialization actions
initialization_default "$@"

pkg_setup
src_extract
src_prepare
pkg_icons ## must be called before pkg_prepare
pkg_prepare
src_clean
pkg_launchers
pkg_build
pkg_instructions
pkg_clean ## must be called after pkg_instructions

exit 0
